#ifndef FrameCPP_VERSION_8_FrHistory_HH
#define FrameCPP_VERSION_8_FrHistory_HH

#include "framecpp/FrameCPP.hh"

#if ! defined(SWIGIMPORTED)
#include "framecpp/Version7/FrHistory.hh"
#endif /* ! defined(SWIGIMPORTED) */

#if ! defined(SWIGIMPORTED)
#include "framecpp/Version8/FrameSpec.hh"
#include "framecpp/Version8/STRING.hh"
#endif /* ! defined(SWIGIMPORTED) */

namespace FrameCPP
{
  namespace Version_8
  {
    //===================================================================
    /// \brief History Structure Definition
    //===================================================================
    class FrHistory
#if ! defined(SWIG)
      : public ObjectWithChecksum< Common::FrameSpec::Object >
#endif /* ! defined(SWIG) */
    {
    public:
      //-----------------------------------------------------------------
      /// \brief Type for name of history record
      //-----------------------------------------------------------------
      typedef STRING name_type;
      //-----------------------------------------------------------------
      /// \brief Type for comment field.
      ///
      /// Program name and relevant comments needed to
      /// define post-processing.
      //-----------------------------------------------------------------
      typedef STRING comment_type;
      //-----------------------------------------------------------------
      /// \brief Type for time field.
      ///
      /// Time of post-processing, GPS time in integer seconds
      /// since GPS standard epoch.
      //-----------------------------------------------------------------
      typedef INT_4U time_type;

      //-----------------------------------------------------------------
      /// \brief Default constructor
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrHistory( );

      //-----------------------------------------------------------------
      /// \brief Copy Constructor
      ///
      /// \param[in] Source
      ///     The object from which to copy the information.
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrHistory( const FrHistory& Source );

      //-----------------------------------------------------------------
      /// \brief Constructor
      ///
      /// \param[in] Name
      ///     Name of the history record
      /// \param[in] Time
      ///     Time of post-processing
      ///     (GPS seconds since GPS standard epoch)
      /// \param[in] Comment
      ///     Program name and relevant comments needed to define
      ///     post-processing
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrHistory( const std::string& Name,
		 time_type Time,
		 const std::string& Comment );

      //-----------------------------------------------------------------
      /// \brief Promotion Constructor
      ///
      /// \param[in] Source
      ///     An FrHistory structure from a previous frame specification.
      /// \param[in] Stream
      ///     The stream from which the earliest version of the
      ///     FrAdcData structure was read.
      ///
      /// \return
      ///     A new instance of an FrHistory object
      //-----------------------------------------------------------------
      FrHistory( Previous::FrHistory& Source, istream_type* Stream );

      //-----------------------------------------------------------------
      /// \brief The name structure name of this object.
      ///
      /// \return
      ///     The name of the structure as specified by the frame
      ///     specification.
      //-----------------------------------------------------------------
      static const char* StructName( );

      //-----------------------------------------------------------------
      /// \brief The description of structure
      ///
      /// \return
      ///     A Description object which describes this structure as
      ///     specified by the frame specification.
      //-----------------------------------------------------------------
      static const Common::Description* StructDescription( );

      //-----------------------------------------------------------------
      /// \brief The name structure name of this object.
      ///
      /// \return
      ///     The name of the structure as specified by the frame
      ///     specification.
      //-----------------------------------------------------------------
      virtual const char* ObjectStructName( ) const;

      virtual std::string ErrorInfo( ) const;

      //-----------------------------------------------------------------
      /// \brief Return the name associate with the FrHistory structure.
      ///
      /// \return
      ///     The name associated with the FrHistory structure
      //-----------------------------------------------------------------
      const std::string& GetName() const;

      //-----------------------------------------------------------------
      /// \brief Return the time of the post-processing.
      ///
      /// \return
      ///     The time of the post-processing.
      //-----------------------------------------------------------------
      time_type GetTime( ) const
      {
         return m_data.time;
      }   

      //-----------------------------------------------------------------
      /// \brief Return the description of the post-porcessing.
      ///
      /// \return
      ///     The description of the post-processing
      //-----------------------------------------------------------------
      const std::string& GetComment( ) const
      {
 	 return m_data.comment;
      }   

      //-----------------------------------------------------------------
      /// \brief Merge with another FrHistory object
      ///
      /// \param[in] RHS
      ///     The source of the information to append to this FrHistory
      ///     structure.
      ///
      /// \return
      ///     A reference to this object
      //-----------------------------------------------------------------
      FrHistory& Merge( const FrHistory& RHS );

      //-----------------------------------------------------------------
      /// \brief Promote previous version of object to this version
      ///
      /// \param[in] Source
      ///     The frame specification version of the object.
      /// \param[in] Obj
      ///     The previous version of the object.
      /// \param[in] Stream
      ///     Stream from which the origional object was created.
      ///
      /// \return
      ///     A new instance of FrameH based on Obj
      //-----------------------------------------------------------------
      static promote_ret_type
      Promote( INT_2U Source,
	       promote_arg_type Obj,
	       istream_type* Stream );

      //-----------------------------------------------------------------
      /// \brief equality operator
      ///
      /// \param[in] RHS
      ///     The FrHistory object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      bool operator==( const FrHistory& RHS ) const;

      //-----------------------------------------------------------------
      /// \brief equality operator for abstract data type
      ///
      /// \param[in] RHS
      ///     The object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      virtual bool operator==( const Common::FrameSpec::Object& RHS ) const;

    protected:
      //-----------------------------------------------------------------
      /// \brief Demotes object to previous version of the frame spec
      ///
      /// \param[in] Target
      ///     The version of the frame specification to demote too.
      /// \param[in] Obj
      ///     The version of the object to demote.
      /// \param[in] Stream
      ///     The input stream from which the original object was read.
      ///
      /// \return
      ///     An object of the previous generation.
      //-----------------------------------------------------------------
      virtual demote_ret_type
      demote( INT_2U Target,
	      demote_arg_type Obj,
	      istream_type* Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Promotes object to another version of the frame spec
      ///
      /// \param[in] Source
      ///     The version of the promoted frame specification.
      /// \param[in] Obj
      ///     The object to be promoted.
      /// \param[in] Stream
      ///     The input stream from which the original object was read.
      ///
      /// \return
      ///     An object promoted to the next generation.
      //-----------------------------------------------------------------
      virtual promote_ret_type
      promote( INT_2U Source,
	       promote_arg_type Obj,
	       istream_type* Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Number of bytes needed to write this structure
      ///
      /// \param[in] Stream
      ///     The stream from which to the object is being read or
      ///     written.
      ///
      /// \return
      ///     The number of bytes need to read or write this object.
      //-----------------------------------------------------------------
      cmn_streamsize_type pBytes( const Common::StreamBase& Stream) const;

      //-----------------------------------------------------------------
      /// \brief Virtual constructor
      ///
      /// \param[in] Stream
      ///     The input stream from where the object is being read.
      //-----------------------------------------------------------------
      virtual FrHistory* pCreate( istream_type& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Write the structure to the stream
      ///
      /// \param[in] Stream
      ///     The output stream where the object is to be written.
      //-----------------------------------------------------------------
      virtual void pWrite( ostream_type& Stream ) const;

    private:
      //-----------------------------------------------------------------
      //-----------------------------------------------------------------
      typedef ObjectWithChecksum< Common::FrameSpec::Object > object_type;

      using Common::FrameSpec::Object::Create;

      struct fr_history_data_type {
	//---------------------------------------------------------------
	/// Name of the history record.
        /// \note
	///     When an FrHistory is linked to an FrProcData, its name
	///     variable must be the FrProcData channel name.
	//---------------------------------------------------------------
	name_type	name;
	//---------------------------------------------------------------
	/// Time of post-processing,
	/// GPS time in integer seconds since GPS standard epoch.
	//---------------------------------------------------------------
	time_type	time;
	//---------------------------------------------------------------
	/// Program name and relevant comments needed to define
	/// post-processing.
	//---------------------------------------------------------------
	comment_type	comment;

	//---------------------------------------------------------------
	/// \brief equality operator
	///
	/// \param[in] RHS
	///     The fr_history_data_type object to be compared.
	///
	/// \return
	///     The value true is returned if this object is equivelent
	///     to the RHS instance; false otherwise.
	//---------------------------------------------------------------
	bool operator==( const fr_history_data_type& RHS ) const;
      };

      fr_history_data_type	m_data;

      //-----------------------------------------------------------------
      /// \brief Stream Constructor
      ///
      /// \param[in] Stream
      ///     The stream from which the object is being read.
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrHistory( istream_type& Stream );
    };

    inline const char* FrHistory::
    StructName( )
    {
      static const char* name = "FrHistory";
      return name;
    }

    inline bool FrHistory::
    operator==( const FrHistory& RHS ) const
    {
      return ( m_data == RHS.m_data );
    }
  } // namespace - Version_8
} // namespace FrameCPP
#endif /* FrameCPP_VERSION_8_FrHistory_HH */
