#include "framecpp/config.h"

#include "framecpp/Common/Description.hh"
#include "framecpp/Common/FrameStream.hh"
#include "framecpp/Common/FrameSpec.tcc"

#include "framecpp/Version8/FrameSpec.hh"
#include "framecpp/Version8/FrHistory.hh"
#include "framecpp/Version8/FrSE.hh"
#include "framecpp/Version8/FrSH.hh"
#include "framecpp/Version8/PTR_STRUCT.hh"

using FrameCPP::Common::Description;
using FrameCPP::Common::FrameSpec;

//=======================================================================
// Static
//=======================================================================
static const FrameSpec::Info::frame_object_types s_object_id
= FrameSpec::Info::FSI_FR_HISTORY;

namespace FrameCPP
{
  namespace Version_8
  {
    //=======================================================================
    // FrHistory::fr_history_data_type
    //=======================================================================

    bool FrHistory::fr_history_data_type::
    operator==( const fr_history_data_type& RHS ) const
    {
#define CMP__(X) ( X == RHS.X )

      return ( ( &RHS == this )
	       || ( CMP__( name )
		    && CMP__( time )
		    && CMP__( comment )
		    ) )
	;
#undef CMP__
    }

    //=======================================================================
    // FrHistory
    /// The constuction of the default FrHistory.
    //=======================================================================
    FrHistory::
    FrHistory( )
      : object_type( s_object_id, StructDescription( ) )
    {
    }

    //-----------------------------------------------------------------------
    /// Constructs a new FrHistory based on a previously initialized
	     /// FrHistory instance.
	     //-----------------------------------------------------------------------
	     FrHistory::
	     FrHistory( const FrHistory& Source )
	       : object_type( s_object_id, StructDescription( ) )
	     {
	       m_data.name = Source.m_data.name;
	       m_data.time = Source.m_data.time;
	       m_data.comment = Source.m_data.comment;
	     }

    //-----------------------------------------------------------------------
    /// Construct a new instance with values for all important 
		      //-----------------------------------------------------------------------
    FrHistory::
    FrHistory( const std::string& Name,
	       time_type Time,
	       const std::string& Comment )
      : object_type( s_object_id, StructDescription( ) )
    {
      m_data.name = Name;
      m_data.time = Time;
      m_data.comment = Comment;
    }

    //-----------------------------------------------------------------------
    /// Construct a new instance based upon reading a stream.
	     //-----------------------------------------------------------------------
    FrHistory::
    FrHistory( istream_type& Stream )
      : object_type( s_object_id, StructDescription( ) )
    {
      Stream >> m_data.name
	     >> m_data.time
	     >> m_data.comment
	;
      Stream.Next( this );
    }

    //-----------------------------------------------------------------------
    /// Construct a new instance based upon reading a stream which
	     /// implements an earlier version of the fame specification.
	     //-----------------------------------------------------------------------
	     FrHistory::
	     FrHistory( Previous::FrHistory& Source, istream_type* Stream )
	       : object_type( s_object_id, StructDescription( ) )
	     {
	       m_data.name = Source.GetName( );
	       m_data.time = Source.GetTime( );
	       m_data.comment = Source.GetComment( );
	     }

    std::string FrHistory::
    ErrorInfo( ) const
    {
      std::ostringstream	result;

      result << "FrameCPP::Version8::FrHistory: {Name: " << m_data.name
	     << " Time: " << m_data.time
	     << "} "
	;
      return result.str( );
    }

    const std::string& FrHistory::GetName() const
    {
      return m_data.name;   
    }

    FrHistory& FrHistory::
    Merge( const FrHistory& RHS )
    {
      //:TODO: Need to implement Merge routine
      std::string msg( "Merge currently not implemented for " );
      msg += StructName( );

      throw std::domain_error( msg );
      return *this;
    }

    FrHistory::promote_ret_type FrHistory::
    Promote( INT_2U Source,
	     promote_arg_type Obj,
	     istream_type* Stream )
    {
      return Object::PromoteObject< Previous::FrHistory, FrHistory >
	( DATA_FORMAT_VERSION, Source, Obj, Stream );
    }

    const char* FrHistory::
    ObjectStructName( ) const
    {
      return StructName( );
    }

    const Description* FrHistory::
    StructDescription( )
    {
      static Description ret;

      if ( ret.size( ) == 0 )
      {
	ret( FrSH( FrHistory::StructName( ), s_object_id,
		   "History Structure" ) );
	ret( FrSE( "name", "STRING",
		   "Name of history record" ) );
	ret( FrSE( "time", "INT_4U",
		   "Time of post-processing." ) );
	ret( FrSE( "comment", "STRING",
		   "Program name and relevant comments needed to"
		   " define post-processing" ) );

	ret( FrSE( "next", PTR_STRUCT::Desc( FrHistory::StructName( ) ),
		   "Identifier for next history structure in the linked list") );

	ret( FrSE( "chkSum", CheckSumDataClass( ), CheckSumDataComment( ) ) );
      }
 
      return &ret;
    }

    bool FrHistory::
    operator==( const Common::FrameSpec::Object& RHS ) const
    {
      return ( *this == RHS );
    }

    FrHistory::demote_ret_type FrHistory::
    demote( INT_2U Target,
	    demote_arg_type Obj,
	    istream_type* Stream ) const
    { 
      if ( Target >= DATA_FORMAT_VERSION )
      {
	return Obj;
      }
      try
      {
	//-------------------------------------------------------------------
	// Copy non-reference information
	//-------------------------------------------------------------------

	// Do actual down conversion
	General::SharedPtr< Previous::FrHistory >
	  retval( new Previous::FrHistory( GetName( ),
					   GetTime( ),
					   GetComment( ) )
		  )
	  ;
	//-------------------------------------------------------------------
	// Return demoted object
	//-------------------------------------------------------------------
	return retval;
      }
      catch( ... )
      {
      }
      throw
	Unimplemented( "Object* FrHistory::demote( Object* Obj ) const",
		       DATA_FORMAT_VERSION, __FILE__, __LINE__ );
    }

    FrHistory::promote_ret_type FrHistory::
    promote( INT_2U Target,
	     promote_arg_type Obj,
	     istream_type* Stream ) const
    {
      return Promote( Target, Obj, Stream );
    }

    FrameCPP::cmn_streamsize_type FrHistory::
    pBytes( const Common::StreamBase& Stream ) const
    {
      return
	m_data.name.Bytes( )
	+ sizeof( m_data.time )
	+ m_data.comment.Bytes( )
	+ Stream.PtrStructBytes( )	// next
	;
    }

    FrHistory* FrHistory::
    pCreate( istream_type& Stream ) const
    {
      return new FrHistory( Stream );
    }

    void FrHistory::
    pWrite( ostream_type& Stream ) const
    {
      try {
	Stream << m_data.name
	       << m_data.time
	       << m_data.comment
	  ;
	WriteNext( Stream );
      }
      catch( const std::exception& Exception )
      {
	std::cerr << "DEBUG: Error Writing FrHistory: " << m_data.name
		  << " exception: " << Exception.what( )
		  << std::endl;
	throw;
      }
    }
  } // namespace - Version_8
} // namespace - FrameCPP
