#include "framecpp/config.h"

#include <iostream>

#include "framecpp/Common/IOStream.hh"
#include "framecpp/Common/FrameStream.hh"
#include "framecpp/Common/CheckSumFilter.hh"
#include "framecpp/Common/Description.hh"
#include "framecpp/Common/Verify.hh"
#include "framecpp/Common/FrameSpec.tcc"

#include "framecpp/Version8/FrEndOfFile.hh"
#include "framecpp/Version8/FrHeader.hh"
#include "framecpp/Version8/FrSE.hh"
#include "framecpp/Version8/FrSH.hh"

#include "framecpp/Version8/FrameSpec.hh"

using FrameCPP::Common::CheckSum;
using FrameCPP::Common::CheckSumFilter;
using FrameCPP::Common::Description;
using FrameCPP::Common::FrameSpec;
using FrameCPP::Common::OFrameStream;
using FrameCPP::Common::VerifyException;



namespace FrameCPP
{
  namespace Version_8
  {

    FrEndOfFile::
    FrEndOfFile( )
      : Common::FrEndOfFile( StructDescription( ) ),
	m_nFrames( 0 ),
	m_nBytes( 0 ),
	m_seekTOC( 0 ),
	m_chkSumFrHeader( 0 ),
	m_chkSum( 0 ),
	m_chkSumFile( 0 )
    {
    }

    FrEndOfFile::
    FrEndOfFile( istream_type& Stream )
      : Common::FrEndOfFile( StructDescription( ) )
    {
      //---------------------------------------------------------------------
      // Read data stream
      //---------------------------------------------------------------------
      Stream >> m_nFrames
	     >> m_nBytes
	     >> m_seekTOC
	     >> m_chkSumFrHeader
	;
      //---------------------------------------------------------------------
      // FrHeader checksum
      //---------------------------------------------------------------------
      try
      {
	using FrameCPP::Common::IFrameStream;

#if LM_DEBUG_INPUT
	std::cerr << "DEBUG:"
		  << " [ " << __LINE__ << ": " __FILE__ << " ]"
		  << std::endl
	  ;
#endif /* LM_DEBUG_INPUT */
	const FrHeader& hdr( dynamic_cast< const FrHeader& >( Stream.GetFrHeader( ) ) );

	Common::CheckSumFilter
	  checksum( CheckSum::CRC );
#if LM_DEBUG_INPUT
	std::cerr << "DEBUG:"
		  << " [ " << __LINE__ << ": " __FILE__ << " ]"
		  << std::endl
	  ;
#endif /* LM_DEBUG_INPUT */
	hdr.Filter( checksum );

#if LM_DEBUG_INPUT
	std::cerr << "DEBUG:"
		  << " [ " << __LINE__ << ": " __FILE__ << " ]"
		  << std::endl
	  ;
#endif /* LM_DEBUG_INPUT */
	chkSumFrHeader_type
	  calc_chkSumFrHeader( checksum.Value( ) );

#if LM_DEBUG_INPUT
	std::cerr << "DEBUG:"
		  << " m_chkSumFrHeader: " << m_chkSumFrHeader
		  << " calc_chkSumFrHeader: " << calc_chkSumFrHeader
		  << " [ " << __LINE__ << ": " __FILE__ << " ]"
		  << std::endl
	  ;
#endif /* LM_DEBUG_INPUT */
	if ( m_chkSumFrHeader
	     && calc_chkSumFrHeader
	     && ( m_chkSumFrHeader != calc_chkSumFrHeader ) )
	{
	  std::ostringstream	msg;

	  msg << "FrHeader (via FrEndOfFile): "
	      << CheckSum::FormatError( m_chkSumFrHeader,
					calc_chkSumFrHeader )
	    ;

	  throw VerifyException( VerifyException::CHECKSUM_ERROR, msg.str( ) );
	}
      }
      catch( const VerifyException& Exception )
      {
	throw;
      }
      catch( ... )
      {
#if LM_DEBUG_INPUT
	std::cerr << "DEBUG:"
		  << " exception ..."
		  << " [ " << __LINE__ << ": " __FILE__ << " ]"
		  << std::endl
	  ;
#endif /* LM_DEBUG_INPUT */
      }
      //---------------------------------------------------------------------
      // Structure checksum
      //---------------------------------------------------------------------
      {
	chkSum_type	checksum = 0;

	if ( Stream.GetCheckSumObject( )
	     && Stream.GetCheckSumObject( )->GetChecksum( ) )
	{
	  checksum = Stream.GetCheckSumObject( )->GetChecksum( )->value( );
	}

	Stream
	  >> m_chkSum
	  ;
#if LM_DEBUG_INPUT
	std::cerr << "DEBUG:"
		  << " m_chkSum: " << m_chkSum
		  << " checksum: " << checksum
		  << " [ " << __LINE__ << ": " __FILE__ << " ]"
		  << std::endl
	  ;
#endif /* LM_DEBUG_INPUT */
	if ( m_chkSum
	     && checksum
	     && ( m_chkSum != checksum ) )
	{
	  std::ostringstream	msg;
	  
	  msg << "FrEndOfFile Object: "
	      << CheckSum::FormatError( m_chkSum,
					checksum )
	    ;

	  throw VerifyException( VerifyException::CHECKSUM_ERROR, msg.str( ) );
	}
      }
      //---------------------------------------------------------------------
      // File checksum
      //---------------------------------------------------------------------
      {
	chkSumFile_type	checksum = 0;

	CheckSumFilter* csf( Stream.GetCheckSumFile( ) );
	if ( csf )
	{
	  //-----------------------------------------------------------------
	  // Calculate the value to end checksum calculation.
	  //-----------------------------------------------------------------
	  checksum = csf->Value( );
	}
	Stream
	  >> m_chkSumFile
	  ;
#if LM_DEBUG_INPUT
	std::cerr << "DEBUG:"
		  << " m_chkSumFile: " << m_chkSumFile
		  << " checksum: " << checksum
		  << " [ " << __LINE__ << ": " __FILE__ << " ]"
		  << std::endl
	  ;
#endif /* LM_DEBUG_INPUT */
	if ( checksum
	     && m_chkSumFile
	     && ( checksum != m_chkSumFile ) )
	{
	  std::ostringstream	msg;
	  
	  msg << "File: "
	      << CheckSum::FormatError( m_chkSumFile,
					checksum )
	    ;

	  throw VerifyException( VerifyException::CHECKSUM_ERROR, msg.str( ) );
	}
      }
    }

    FrEndOfFile::
    ~FrEndOfFile( )
    {
    }

    FrameSpec::size_type FrEndOfFile::
    Bytes( const Common::StreamBase& Stream ) const
    {
      return
	sizeof( m_nFrames )
	+ sizeof( m_nBytes )
	+ sizeof( m_seekTOC )
	+ sizeof( m_chkSumFrHeader )
	+ sizeof( m_chkSum )
	+ sizeof( m_chkSumFile )
	;
    }

    FrEndOfFile* FrEndOfFile::
    Clone( ) const
    {
      return new FrEndOfFile( *this );
    }

    FrEndOfFile* FrEndOfFile::
    Create( istream_type& Stream ) const
    {
      return new FrEndOfFile( Stream );
    }

    FrEndOfFile::chkSum_cmn_type FrEndOfFile::
    Filter( const istream_type& Stream,
	    Common::CheckSum& Filt,
	    chkType_cmn_type& Type,
	    void* Buffer,
	    INT_8U Size ) const
    {
      chkSum_type	value;

      if ( Bytes( Stream ) != Size )
      {
	std::ostringstream	msg;
	msg << "FrEndOfFile::Filter: "
	    << " Expected a buffer of size: " << Bytes( Stream )
	    << " but received a buffer of size: " << Size
	  ;
	throw std::length_error( msg.str( ) );
      }
      const size_t	chkSumOffset( Size - sizeof( chkSumFile_type ) );

      Filt.calc( Buffer, chkSumOffset );

      std::copy( (const char*)Buffer
		 + chkSumOffset,
		 (const char*)Buffer
		 + chkSumOffset +
		 sizeof( value ),
		 (char*)&value );

      if ( Stream.ByteSwapping( ) )
      {
	reverse< sizeof( value ) >( &value, 1 );
      }

      Type = FrameCPP::Common::CheckSum::UNSET;

      return value;
    }

    FrEndOfFile::nBytes_cmn_type FrEndOfFile::
    NBytes( ) const
    {
      return m_nBytes;
    }

    const char* FrEndOfFile::
    ObjectStructName( ) const
    {
      return StructName( );
    }

    const Description* FrEndOfFile::
    StructDescription( )
    {
      static Description ret;

      if ( ret.size( ) == 0 )
      {
	ret( FrSH( "FrEndOfFile", FrEndOfFile::s_object_id,
		   "End of File Data Structure" ) );
	ret( FrSE( "nFrames", "INT_4U",
		   "Number of frames in this file") );
	ret( FrSE( "nBytes", "INT_8U",
		   "Total number of bytes in this file"
		   " ( 0 if NOT computed )") );
	ret( FrSE ( "seekTOC", "INT_8U",
		    "Byes to back up to the beginning of the table of"
		    " contents structure. If seekTOC == 0, then there"
		    " is no TOC for this file." ) );
	ret( FrSE ( "chkSumFrHeader", "INT_4U", "FrHeader checksum." ) );
	ret( FrSE ( "chkSum", "INT_4U", "Structure checksum." ) );
	ret( FrSE ( "chkSumFile", "INT_4U", "File checksum." ) );
      }

      return &ret;
    }

    void FrEndOfFile::
    VerifyObject( Common::Verify& Verifier,
		  Common::IFrameStream& Stream ) const
    {
      Verifier.SeenFrEndOfFile( true );
      Verifier.ExamineFrHeaderChecksum( m_chkSumFrHeader );
      Verifier.ExamineFrEndOfFileChecksum( Stream, Verifier.ChecksumScheme( ), m_chkSumFile );
    }
		  
    void FrEndOfFile::
    Write( ostream_type& Stream ) const
    {
      //---------------------------------------------------------------------
      // Get local information
      //---------------------------------------------------------------------
 
      const nFrames_type	nFrames( Stream.GetNumberOfFrames( ) );
      const nBytes_type	nBytes( Stream.tellp( )
				+ std::streampos( Bytes( Stream ) ) );
      const seekTOC_type	seekTOC( ( Stream.GetTOCOffset( ) > 0 )
					 ? nBytes - Stream.GetTOCOffset( )
					 : 0 );

      chkSumFrHeader_type	chkSumFrHeader = 0;
      chkSum_type		chkSum = 0;
      chkSumFile_type		chkSumFile = 0;

      const FrHeader* hdr
	= Stream.GetFrHeader< FrHeader >( );

      if ( hdr )
      {
	chkSumFrHeader = hdr->Checksum( );
      }

      //---------------------------------------------------------------------
      // Write out to the stream
      //---------------------------------------------------------------------
      Stream << nFrames
	     << nBytes
	     << seekTOC
	     << chkSumFrHeader
	;
      if ( Stream.GetCheckSumObject( )
	   && Stream.GetCheckSumObject( )->GetChecksum( ) )
      {
	chkSum = Stream.GetCheckSumObject( )->Value( );
      }
      Stream << chkSum;
#if LM_DEBUG_OUTPUT
      std::cerr << "DEBUG: FrEndOfFile:"
		<< " Writing object checksum: " << chkSum
		<< " nFrames: " << nFrames
		<< " nBytes: " << nBytes
		<< " seekTOC: " << seekTOC
		<< " chkSumFrHeader: " << chkSumFrHeader
		<< std::endl
	;
#endif /* LM_DEBUG_OUTPUT */

      if ( Stream.GetCheckSumFile( )
	   && Stream.GetCheckSumFile( )->GetChecksum( ) )
      {
	chkSumFile = Stream.GetCheckSumFile( )->Value( );
      }
      Stream << chkSumFile;
    }

    bool FrEndOfFile::
    operator==( const FrEndOfFile& RHS ) const
    {
      return ( ( m_nFrames == RHS.m_nFrames )
	       && ( m_nBytes == RHS.m_nBytes )
	       && ( m_seekTOC == RHS.m_seekTOC )
	       && ( m_chkSumFrHeader == RHS.m_chkSumFrHeader )
	       && ( m_chkSum == RHS.m_chkSum )
	       && ( m_chkSumFile == RHS.m_chkSumFile )
	       )
	;
    }

    bool FrEndOfFile::
    operator==( const Common::FrameSpec::Object& RHS ) const
    {
      return ( *this == RHS );
    }

    void FrEndOfFile::
    assign( assign_stream_type& Stream )
    {
      //-----------------------------------------------------------------
      // Read the data from the stream
      //-----------------------------------------------------------------
      Stream >> m_nFrames
	     >> m_nBytes
	     >> m_seekTOC
	     >> m_chkSumFrHeader
	     >> m_chkSum
	     >> m_chkSumFile
	;
    }

    FrEndOfFile::demote_ret_type FrEndOfFile::
    demote( INT_2U Target,
	    demote_arg_type Obj,
	    istream_type* Stream ) const
    {
      throw
	Unimplemented( "Object* FrEndOfFile::demote( Object* Obj ) const",
		       DATA_FORMAT_VERSION, __FILE__, __LINE__ );
    }

    FrEndOfFile::promote_ret_type FrEndOfFile::
    promote( INT_2U Target,
	     promote_arg_type Obj,
	     istream_type* Stream ) const
    {
      throw
	Unimplemented( "Object* FrEndOfFile::promote( Object* Obj ) const",
		       DATA_FORMAT_VERSION, __FILE__, __LINE__ );
    }
  } // namespace - Version_8
} // namespace - FrameCPP
