#ifndef FrameCPP_VERSION_8_Dimension_HH
#define FrameCPP_VERSION_8_Dimension_HH


// System Includes
#if ! defined(SWIGIMPORTED)
#include <memory>
#include <string>
#endif /* ! defined(SWIGIMPORTED) */

#if ! defined(SWIGIMPORTED)
#include "general/types.hh"

#include "framecpp/config.h"

#include "framecpp/Version7/Dimension.hh"

#include "framecpp/Version8/FrameSpec.hh"
#include "framecpp/Version8/STRING.hh"
#endif /* ! defined(SWIGIMPORTED) */

#undef DIMENSION_NDIM_TYPE
#undef DIMENSION_NX_TYPE
#undef DIMENSION_DX_TYPE
#undef DIMENSION_STARTX_TYPE

#define DIMENSION_NDIM_TYPE	INT_4U
#define DIMENSION_NX_TYPE	INT_8U
#define DIMENSION_DX_TYPE	REAL_8
#define DIMENSION_STARTX_TYPE	REAL_8

namespace FrameCPP {

  namespace Version_8 {

    //-------------------------------------------------------------------
    //
    //: Dimension information for a Vect.
    //
    // This is a simple class storing dimension information for a Vect object.
    // It stores the following:
    // <p><ol>
    //     <li> The dimension length. </li>
    //     <li> Scale factor </li>
    //     <li> The units (unit per step size along the coordintate). </li>
    // </ol>
    //
    class Dimension
    {
    public:
      typedef DIMENSION_NDIM_TYPE	nDim_type;
      typedef DIMENSION_NX_TYPE		nx_type;
      typedef DIMENSION_DX_TYPE		dx_type;
      typedef DIMENSION_STARTX_TYPE	startX_type;
      typedef STRING unitX_type;

      /* Constructors/Destructor */
      //!exc: None.
      Dimension();
    
      //!exc: std::bad_alloc - Memory allocation failed.
      Dimension( const Dimension& dim );
   
      //!exc: std::bad_alloc - Memory allocation failed.   
      explicit Dimension( nx_type nx,
			  dx_type dx = 1.0,
			  const std::string& unitX = "",
			  startX_type startX = 0.0 );

      Dimension( const Previous::Dimension& Source );
    
      /* Operator Overloads */
#if ! defined(SWIG)
      //-----------------------------------------------------------------
      //
      //: Assignment operator.
      //
      //!param: const Dimension& dim - The object to assign from.
      //
      //!return: const Dimension& -- This object.
      //
      //!exc: std::bad_alloc - Memory allocation failed.
      //
      const Dimension& operator=( const Dimension& dim );
#endif /* ! defined(SWIG) */

      //----------------------------------------------------------------
      //
      //: Equal comparison.
      //
      //!param: const Dimension& dim - The object to compare with.
      //
      //!return: bool -- true if the objects are equal, false otherwise.
      //
      //!exc: None.
      //
      bool operator==( const Dimension& dim ) const;

      //----------------------------------------------------------------
      //
      //: Not equal comparison.
      //
      //!param: const Dimension& dim - The object to compare with.
      //
      //!return: bool -- false if the objects are equal, true otherwise.
      //
      //!exc: None.
      //
      bool operator!=( const Dimension& dim ) const;
    
      /* Accessors */
      //!exc: None.
      nx_type GetNx( ) const;
   
      //!exc: None.   
      dx_type GetDx( ) const;
      
      //!exc: None.   
      startX_type GetStartX( ) const;
   
      //!exc: None.   
      const unitX_type& GetUnitX( ) const;
    
      //-----------------------------------------------------------------
      /// \brief Set sample spacing
      ///
      /// \param[in] Dx
      ///     The new sample rate for the coordinate.
      //-----------------------------------------------------------------
      void SetDx( dx_type Dx );
    
      void SetNx( nx_type Nx );
    
      void SetStartX( startX_type StartX );

      //-----------------------------------------------------------------
      /// \brief Set scale factor in ASCII
      ///
      /// \param[in] UnitX
      ///     The new scale factor description.
      //-----------------------------------------------------------------
      void SetUnitX( const unitX_type& UnitX );

      cmn_streamsize_type Bytes( ) const;

      static nx_type CalcNData( nDim_type NDims, const Dimension* Dims );

    private:
      //: Dimension length.
      nx_type mNx;
      //: Scale factor.
      dx_type mDx;
      //: Scale factor in ASCII.
      unitX_type mUnitX;
      //: Origin.
      startX_type mStartX;
    };




    /******************/
    /* Inline Methods */
    /******************/
    inline Dimension::
    Dimension( const Previous::Dimension& Source )
      : mNx( Source.GetNx( ) ),
	mDx( Source.GetDx( ) ),
	mUnitX( Source.GetUnitX( ) ),
	mStartX( Source.GetStartX( ) )
    {
    }


    inline Dimension::nx_type Dimension::
    CalcNData( nDim_type NDims, const Dimension* Dims )
    {
      if ( NDims > 0 )
      {
	nx_type retval = 1;
	
	for ( nDim_type cur = 0,
		last = NDims;
	      cur != last;
	      ++cur )
	{
	  retval *= Dims[ cur ].GetNx( );
	}
	return retval;
      }
      return 0;
    }
    
    //------------------------------------------------------------------
    // Accessors
    //------------------------------------------------------------------


    //------------------------------------------------------------------
    //
    //: Get the dimension length.
    //
    //!return: INT_8U -- The dimension length.
    //
    //!exc: None.   
    //      
    inline Dimension::nx_type Dimension::
    GetNx() const
    {
      return mNx;
    }


    //------------------------------------------------------------------
    //
    //: Get the scale factor.
    //
    //!return: REAL_8 -- The scale factor.
    //
    //!exc: None.   
    //      
    inline Dimension::dx_type Dimension::
    GetDx( ) const
    {
      return mDx;
    }

    //------------------------------------------------------------------
    //
    //: Get data set origin.
    //
    //!return: REAL_8 -- Data set origin.
    //
    //!exc: None.   
    //      
    inline Dimension::startX_type Dimension::
    GetStartX() const
    {
      return mStartX;
    }


    //------------------------------------------------------------------
    //
    //: Get the units.
    //
    //!return: const std::string& -- The units.
    //
    //!exc: None.   
    //      
    inline const Dimension::unitX_type& Dimension::
    GetUnitX() const
    {
      return mUnitX;
    }

    //------------------------------------------------------------------
    /// Establish a new value for sample spacing along the coordinate.
    //------------------------------------------------------------------
    inline void Dimension::
    SetDx( dx_type Dx )
    {
      mDx = Dx;
    }

    inline void Dimension::
    SetNx( nx_type Nx )
    {
      mNx = Nx;
    }
 
    inline void Dimension::
    SetStartX( startX_type StartX )
    {
      mStartX = StartX;
    }

    //------------------------------------------------------------------
    /// Establish a new value for sample spacing along the coordinate.
    //------------------------------------------------------------------
    inline void Dimension::
    SetUnitX( const unitX_type& UnitX )
    {
      mUnitX = UnitX;
    }

    //------------------------------------------------------------------
    // Operator Overloads
    //------------------------------------------------------------------


    //------------------------------------------------------------------
    //
    //: Assignment operator.
    //
    //!param: const Dimension& dim - The object to assign from.
    //
    //!return: const Dimension& - This object.
    //
    //!exc: std::bad_alloc - Memory allocation failed.
    //
    //------------------------------------------------------------------
    inline const Dimension& Dimension::
    operator=( const Dimension& dim )
    {
      if ( this != &dim )
      {
        mNx = dim.mNx;
        mDx = dim.mDx;
        mStartX = dim.mStartX;
        mUnitX = dim.mUnitX;
      }

      return *this;
    }


    //------------------------------------------------------------------
    //
    //: Equal comparison.
    //
    //!param: const Dimension& dim - The object to compare with.
    //
    //!return: bool - true if the objects are equal, false otherwise.
    //
    //!exc: None.   
    //      
    //------------------------------------------------------------------
    inline bool Dimension::
    operator==( const Dimension& dim ) const
    {
      return ( ( mNx    == dim.mNx    ) &&
	       ( mDx    == dim.mDx    ) &&
	       ( mStartX== dim.mStartX) &&
	       ( mUnitX == dim.mUnitX ) );
    }


    //------------------------------------------------------------------
    //
    //: Not equal comparison.
    //
    //!param: const Dimension& dim - The object to compare with.
    //
    //!return: bool - false if the objects are equal, true otherwise.
    //
    //!exc: None.   
    //      
    //------------------------------------------------------------------
    inline bool Dimension::
    operator!=( const Dimension& dim ) const
    {
      return !(dim == *this);
    }


    inline cmn_streamsize_type Dimension::
    Bytes( ) const
    {
      return
	sizeof( mNx )
	+ sizeof( mDx )
	+ mUnitX.Bytes( )
	+ sizeof( mStartX )
	;
    }
  } // namespace - Version_8

} // namespace - FrameCPP

#endif /* FrameCPP_VERSION_8_Dimension_HH */
