#ifndef FrameCPP_VERSION_4_Dimension_HH
#define FrameCPP_VERSION_4_Dimension_HH


// System Includes
#if ! defined(SWIGIMPORTED)
#include <memory>
#include <string>
#endif /* ! defined(SWIGIMPORTED) */

#include "general/types.hh"

#include "framecpp/config.h"

#include "framecpp/Version3/Dimension.hh"

#include "framecpp/Version4/FrameSpec.hh"
#include "framecpp/Version4/STRING.hh"

namespace FrameCPP {

  namespace Version_4 {

    //-------------------------------------------------------------------
    /// \brief Dimension information for a Vect.
    ///
    /// This is a simple class storing dimension information
    /// for a Vect object.
    /// It stores the following:
    ///
    /// <ol>
    ///   <li> The dimension length. </li>
    ///   <li> Scale factor </li>
    ///   <li> The units (unit per step size along the coordintate). </li>
    /// </ol>
    //-------------------------------------------------------------------
    class Dimension
    {
    public:
      typedef INT_4U nx_type;
      typedef REAL_8 dx_type;
      typedef REAL_8 startX_type;

      /* Constructors/Destructor */
      //!exc: None.
      Dimension();
    
      //-----------------------------------------------------------------
      /// \brief Copy constructor
      ///
      /// \param[in] dim
      ///     The source from which to construct the new object.
      ///
      /// \exception std::bad_alloc
      ///     Memory allocation failed.
      //-----------------------------------------------------------------
      Dimension( const Dimension& dim );
   
      //!exc: std::bad_alloc - Memory allocation failed.
      Dimension( const Previous::Dimension& dim );

      //-----------------------------------------------------------------
      /// \brief Constructor.
      ///
      /// \param[in] nx
      ///     The dimension length.
      /// \param[in] dx
      ///     The Scale Factor.
      /// \param[in] unitX
      ///     The units (unit per step size).
      /// \param[in] startX
      ///     The origin of the data set.
      ///
      /// \exception std::bad_alloc
      ///     Memory allocation failed.
      //-----------------------------------------------------------------
      explicit Dimension( nx_type nx,
			  dx_type dx = 1.0,
			  const std::string& unitX = "",
			  startX_type startX = 0.0 );
    
      /* Operator Overloads */
#if ! defined(SWIG)
      //-----------------------------------------------------------------
      ///
      /// \brief Assignment operator.
      ///
      /// \param[in] dim
      ///     The object to assign from.
      ///
      /// \return
      ///     This object.
      ///
      /// \exception std::bad_alloc
      ///     Memory allocation failed.
      //-----------------------------------------------------------------
      const Dimension& operator=( const Dimension& dim );
#endif /* ! defined(SWIG) */

      //----------------------------------------------------------------
      //
      //: Equal comparison.
      //
      //!param: const Dimension& dim - The object to compare with.
      //
      //!return: bool -- true if the objects are equal, false otherwise.
      //
      //!exc: None.
      //
      bool operator==( const Dimension& dim ) const;

      //----------------------------------------------------------------
      //
      //: Not equal comparison.
      //
      //!param: const Dimension& dim - The object to compare with.
      //
      //!return: bool -- false if the objects are equal, true otherwise.
      //
      //!exc: None.
      //
      bool operator!=( const Dimension& dim ) const;
    
      /* Accessors */
      //!exc: None.
      nx_type GetNx() const;
   
      //!exc: None.   
      dx_type GetDx() const;
      
      //!exc: None.   
      startX_type GetStartX() const;
   
      //!exc: None.   
      const STRING& GetUnitX() const;
    
      void SetNx( nx_type Nx );
    
      cmn_streamsize_type Bytes( ) const;

    private:
      //: Dimension length.
      nx_type mNx;
      //: Scale factor.
      dx_type mDx;
      //: Scale factor in ASCII.
      STRING mUnitX;
      //: Origin.
      startX_type mStartX;
    };




    /******************/
    /* Inline Methods */
    /******************/

    inline Dimension::
    Dimension( const Previous::Dimension& Source )
      : mNx( Source.GetNx( ) ),
	mDx( Source.GetDx( ) ),
	mUnitX( Source.GetUnitX( ) ),
	mStartX( 0.0 )
    {
    }

    //------------------------------------------------------------------
    // Accessors
    //------------------------------------------------------------------


    //------------------------------------------------------------------
    //
    //: Get the dimension length.
    //
    //!return: INT_8U -- The dimension length.
    //
    //!exc: None.   
    //      
    inline Dimension::nx_type Dimension::
    GetNx() const
    {
      return mNx;
    }


    //------------------------------------------------------------------
    //
    //: Get the scale factor.
    //
    //!return: REAL_8 -- The scale factor.
    //
    //!exc: None.   
    //      
    inline Dimension::dx_type Dimension::
    GetDx( ) const
    {
      return mDx;
    }

    //------------------------------------------------------------------
    //
    //: Get data set origin.
    //
    //!return: REAL_8 -- Data set origin.
    //
    //!exc: None.   
    //      
    inline Dimension::startX_type Dimension::
    GetStartX() const
    {
      return mStartX;
    }


    //------------------------------------------------------------------
    //
    //: Get the units.
    //
    //!return: const std::string& -- The units.
    //
    //!exc: None.   
    //      
    inline const STRING& Dimension::
    GetUnitX() const
    {
      return mUnitX;
    }

    inline void Dimension::
    SetNx( nx_type Nx )
    {
      mNx = Nx;
    }

 
    //------------------------------------------------------------------
    // Operator Overloads
    //------------------------------------------------------------------


    //------------------------------------------------------------------
    //
    //: Assignment operator.
    //
    //!param: const Dimension& dim - The object to assign from.
    //
    //!return: const Dimension& - This object.
    //
    //!exc: std::bad_alloc - Memory allocation failed.
    //
    //------------------------------------------------------------------
    inline const Dimension& Dimension::
    operator=( const Dimension& dim )
    {
      if ( this != &dim )
      {
        mNx = dim.mNx;
        mDx = dim.mDx;
        mStartX = dim.mStartX;
        mUnitX = dim.mUnitX;
      }

      return *this;
    }


    //------------------------------------------------------------------
    //
    //: Equal comparison.
    //
    //!param: const Dimension& dim - The object to compare with.
    //
    //!return: bool - true if the objects are equal, false otherwise.
    //
    //!exc: None.   
    //      
    //------------------------------------------------------------------
    inline bool Dimension::
    operator==( const Dimension& dim ) const
    {
      return ( ( mNx    == dim.mNx    ) &&
	       ( mDx    == dim.mDx    ) &&
	       ( mStartX== dim.mStartX) &&
	       ( mUnitX == dim.mUnitX ) );
    }


    //------------------------------------------------------------------
    //
    //: Not equal comparison.
    //
    //!param: const Dimension& dim - The object to compare with.
    //
    //!return: bool - false if the objects are equal, true otherwise.
    //
    //!exc: None.   
    //      
    //------------------------------------------------------------------
    inline bool Dimension::
    operator!=( const Dimension& dim ) const
    {
      return !(dim == *this);
    }


    inline cmn_streamsize_type Dimension::
    Bytes( ) const
    {
      return
	sizeof( mNx )
	+ sizeof( mDx )
	+ mUnitX.Bytes( )
	+ sizeof( mStartX )
	;
    }
  } // namespace - Version_4

} // namespace - FrameCPP

#endif /* FrameCPP_VERSION_4_Dimension_HH */
