#ifndef FrameCPP_VERSION_3_Dimension_HH
#define FrameCPP_VERSION_3_Dimension_HH


// System Includes
#if ! defined(SWIGIMPORTED)
#include <memory>
#include <string>
#endif /* ! defined(SWIGIMPORTED) */

#include "general/types.hh"

#include "framecpp/config.h"

#include "framecpp/Version3/STRING.hh"

namespace FrameCPP {

  namespace Version_3 {

    //-------------------------------------------------------------------
    //
    //: Dimension information for a Vect.
    //
    // This is a simple class storing dimension information for a Vect object.
    // It stores the following:
    // <p><ol>
    //     <li> The dimension length. </li>
    //     <li> Scale factor </li>
    //     <li> The units (unit per step size along the coordintate). </li>
    // </ol>
    //
    class Dimension
    {
    public:
      typedef INT_4U nx_type;
      typedef REAL_8 dx_type;

      /* Constructors/Destructor */
      //!exc: None.
      Dimension();
    
      //!exc: std::bad_alloc - Memory allocation failed.
      Dimension( const Dimension& dim );
   
      //!exc: std::bad_alloc - Memory allocation failed.   
      explicit Dimension( nx_type nx,
			  dx_type dx = 1.0,
			  const std::string& unitX = "" );
    
      /* Operator Overloads */
#if ! defined(SWIG)
      //-----------------------------------------------------------------
      //
      //: Assignment operator.
      //
      //!param: const Dimension& dim - The object to assign from.
      //
      //!return: const Dimension& -- This object.
      //
      //!exc: std::bad_alloc - Memory allocation failed.
      //
      const Dimension& operator=( const Dimension& dim );
#endif /* ! defined(SWIG) */

      //----------------------------------------------------------------
      //
      //: Equal comparison.
      //
      //!param: const Dimension& dim - The object to compare with.
      //
      //!return: bool -- true if the objects are equal, false otherwise.
      //
      //!exc: None.
      //
      bool operator==( const Dimension& dim ) const;

      //----------------------------------------------------------------
      //
      //: Not equal comparison.
      //
      //!param: const Dimension& dim - The object to compare with.
      //
      //!return: bool -- false if the objects are equal, true otherwise.
      //
      //!exc: None.
      //
      bool operator!=( const Dimension& dim ) const;
    
      /* Accessors */
      //!exc: None.
      nx_type GetNx() const;
   
      //!exc: None.   
      dx_type GetDx() const;
      
      //!exc: None.   
      const STRING& GetUnitX() const;
    
      //-----------------------------------------------------------------
      /// \brief Set sample spacing
      ///
      /// \param[in] Dx
      ///     The new sample rate for the coordinate.
      //-----------------------------------------------------------------
      void SetDx( dx_type Dx );
    
      void SetNx( nx_type Nx );

      //-----------------------------------------------------------------
      /// \brief Set scale factor in ASCII
      ///
      /// \param[in] UnitX
      ///     The new scale factor description.
      //-----------------------------------------------------------------
      void SetUnitX( const STRING& UnitX );

      cmn_streamsize_type Bytes( ) const;

    private:
      //: Dimension length.
      nx_type mNx;
      //: Scale factor.
      dx_type mDx;
      //: Scale factor in ASCII.
      STRING mUnitX;
    };




    /******************/
    /* Inline Methods */
    /******************/


    //------------------------------------------------------------------
    // Accessors
    //------------------------------------------------------------------


    //------------------------------------------------------------------
    //
    //: Get the dimension length.
    //
    //!return: INT_8U -- The dimension length.
    //
    //!exc: None.   
    //      
    inline Dimension::nx_type Dimension::
    GetNx() const
    {
      return mNx;
    }


    //------------------------------------------------------------------
    //
    //: Get the scale factor.
    //
    //!return: REAL_8 -- The scale factor.
    //
    //!exc: None.   
    //      
    inline Dimension::dx_type Dimension::
    GetDx( ) const
    {
      return mDx;
    }

    //------------------------------------------------------------------
    //
    //: Get the units.
    //
    //!return: const std::string& -- The units.
    //
    //!exc: None.   
    //      
    inline const STRING& Dimension::
    GetUnitX() const
    {
      return mUnitX;
    }

    //------------------------------------------------------------------
    /// Establish a new value for sample spacing along the coordinate.
    //------------------------------------------------------------------
    inline void Dimension::
    SetDx( dx_type Dx )
    {
      mDx = Dx;
    }

    inline void Dimension::
    SetNx( nx_type Nx )
    {
      mNx = Nx;
    }

    //------------------------------------------------------------------
    /// Establish a new value for sample spacing along the coordinate.
    //------------------------------------------------------------------
    inline void Dimension::
    SetUnitX( const STRING& UnitX )
    {
      mUnitX = UnitX;
    }
 
    //------------------------------------------------------------------
    // Operator Overloads
    //------------------------------------------------------------------


    //------------------------------------------------------------------
    //
    //: Assignment operator.
    //
    //!param: const Dimension& dim - The object to assign from.
    //
    //!return: const Dimension& - This object.
    //
    //!exc: std::bad_alloc - Memory allocation failed.
    //
    //------------------------------------------------------------------
    inline const Dimension& Dimension::
    operator=( const Dimension& dim )
    {
      if ( this != &dim )
      {
        mNx = dim.mNx;
        mDx = dim.mDx;
        mUnitX = dim.mUnitX;
      }

      return *this;
    }


    //------------------------------------------------------------------
    //
    //: Equal comparison.
    //
    //!param: const Dimension& dim - The object to compare with.
    //
    //!return: bool - true if the objects are equal, false otherwise.
    //
    //!exc: None.   
    //      
    //------------------------------------------------------------------
    inline bool Dimension::
    operator==( const Dimension& dim ) const
    {
      return ( ( mNx    == dim.mNx    ) &&
	       ( mDx    == dim.mDx    ) &&
	       ( mUnitX == dim.mUnitX ) );
    }


    //------------------------------------------------------------------
    //
    //: Not equal comparison.
    //
    //!param: const Dimension& dim - The object to compare with.
    //
    //!return: bool - false if the objects are equal, true otherwise.
    //
    //!exc: None.   
    //      
    //------------------------------------------------------------------
    inline bool Dimension::
    operator!=( const Dimension& dim ) const
    {
      return !(dim == *this);
    }


    inline cmn_streamsize_type Dimension::
    Bytes( ) const
    {
      return
	sizeof( mNx )
	+ sizeof( mDx )
	+ mUnitX.Bytes( )
	;
    }
  } // namespace - Version_3

} // namespace - FrameCPP

#endif /* FrameCPP_VERSION_3_Dimension_HH */
