#ifndef FRAME_CPP_INTERFACE__IFrameStream_HH
#define FRAME_CPP_INTERFACE__IFrameStream_HH

#if !defined(SWIG)
#include "framecpp/Common/FrameBuffer.hh"
#include "framecpp/Common/FrameStream.hh"
#include "framecpp/FrameCPP.hh"
#include "framecpp/FrameH.hh"
#include "framecpp/FrAdcData.hh"
#include "framecpp/FrDetector.hh"
#include "framecpp/FrEvent.hh"
#include "framecpp/FrProcData.hh"
#include "framecpp/FrTOC.hh"

#if FRAME_SPEC_CURRENT == 3
#include "framecpp/Version3/FrameStream.hh"
#elif FRAME_SPEC_CURRENT == 4
#include "framecpp/Version4/FrameStream.hh"
#elif FRAME_SPEC_CURRENT == 6
#include "framecpp/Version6/FrameStream.hh"
#elif FRAME_SPEC_CURRENT == 7
#include "framecpp/Version7/FrameStream.hh"
#elif FRAME_SPEC_CURRENT == 8
#include "framecpp/Version8/FrameStream.hh"
#else
#error "Unsupported frame spec: FRAME_SPEC_CURRENT"
#endif
#endif /* !defined(SWIG) */

namespace FrameCPP
{
  typedef Version::IFrameStream IFrameStream;

  class IFrameFStream
  {
  public:
    typedef Common::IFrameStream::frame_library_type	frame_library_type;
    typedef Common::IFrameStream::library_revision_type	library_revision_type;
    typedef Common::IFrameStream::size_type	size_type;
    typedef Common::IFrameStream::version_type	version_type;
    typedef General::SharedPtr< FrameH >	frame_h_type;
    typedef General::SharedPtr< FrDetector >	fr_detector_type;
    typedef General::SharedPtr< FrEvent >	fr_event_type;
    typedef General::SharedPtr< FrAdcData >	fr_adc_data_type;
    typedef General::SharedPtr< FrProcData >	fr_proc_data_type;

    IFrameFStream( const char* Filename );

    frame_library_type FrameLibrary( ) const;

    std::string FrameLibraryName( ) const;

    size_type GetNumberOfFrames( ) const;

    const FrTOC& GetTOC( ) const;

    library_revision_type LibraryRevision( ) const;

    fr_detector_type ReadDetector( const std::string& Name );

    fr_event_type ReadFrEvent( INT_4U Frame, const std::string& Channel );

    fr_adc_data_type ReadFrAdcData( INT_4U Frame, const std::string& Channel );

    fr_proc_data_type ReadFrProcData( INT_4U Frame, const std::string& Channel );

    frame_h_type ReadFrameN( INT_4U Index );

    frame_h_type ReadNextFrame( );

    version_type Version( ) const;

  private:
    typedef General::filebuf			filebuf_type;
    typedef Common::FrameBuffer< filebuf_type >	frame_buffer_type;

    static const int			BUFFER_SIZE = 1024 * 1024;
    static const bool			MEMORY_MAPPED_IO = false;

    General::AutoArray< CHAR >		m_buffer;
    std::auto_ptr< frame_buffer_type >	m_frame_buffer;
    std::auto_ptr< IFrameStream >	m_frame_stream;
    mutable std::auto_ptr< FrTOC >	m_toc;
  };

  inline IFrameFStream::
  IFrameFStream( const char* Filename )
  {
    m_buffer.reset( new CHAR[ BUFFER_SIZE ] );

    m_frame_buffer.reset( new frame_buffer_type( std::ios::in ) );
    m_frame_buffer->pubsetbuf( m_buffer.get( ), BUFFER_SIZE );
    m_frame_buffer->UseMemoryMappedIO( MEMORY_MAPPED_IO );
    m_frame_buffer->open( Filename,
			  std::ios::in | std::ios::binary );

    m_frame_stream.reset( new IFrameStream( m_frame_buffer.get( ) ) );
    m_frame_stream->SetAutoDeleteBuffer( false );
  }

  inline IFrameFStream::frame_library_type IFrameFStream::
  FrameLibrary( ) const
  {
    return m_frame_stream->FrameLibrary( );
  }

  inline std::string IFrameFStream::
  FrameLibraryName( ) const
  {
    return m_frame_stream->FrameLibraryName( );
  }

  inline IFrameFStream::size_type IFrameFStream::
  GetNumberOfFrames( ) const
  {
    return m_frame_stream->GetNumberOfFrames( );
  }
  
  inline const FrTOC& IFrameFStream::
  GetTOC( ) const
  {
    if ( m_toc.get( ) == NULL )
    {
      m_toc.reset( new FrTOC( m_frame_stream->GetTOC( ) ) );
    }
    return *m_toc;
  }

  inline IFrameFStream::library_revision_type IFrameFStream::
  LibraryRevision( ) const
  {
    return m_frame_stream->LibraryRevision( );
  }

  inline IFrameFStream::fr_detector_type IFrameFStream::
  ReadDetector( const std::string& Name )
  {

    Common::FrameSpec::ObjectInterface::object_type
      od( m_frame_stream->ReadDetector( Name ) );

    fr_detector_type
      detector( General::DynamicPointerCast< FrDetector >( od ) );
		    
    if ( ! detector )
    {
      throw std::range_error( "Unable to read detector" );
    }
    return detector;
  }

  inline IFrameFStream::fr_event_type IFrameFStream::
  ReadFrEvent( INT_4U Frame,
		 const std::string& Channel )
  {
    typedef fr_event_type retval_type;

    Common::FrameSpec::ObjectInterface::object_type
      o( m_frame_stream->ReadFrEvent( Frame, Channel ) );

    retval_type
      retval( General::DynamicPointerCast< retval_type::element_type >
	      ( o ) );

    if ( retval )
    {
      return retval;
    }
    throw std::range_error( "Unable to read FrEvent" );
  }

  inline IFrameFStream::fr_adc_data_type IFrameFStream::
  ReadFrAdcData( INT_4U Frame,
		 const std::string& Channel )
  {
    typedef fr_adc_data_type retval_type;

    Common::FrameSpec::ObjectInterface::object_type
      o( m_frame_stream->ReadFrAdcData( Frame, Channel ) );

    retval_type
      retval( General::DynamicPointerCast< retval_type::element_type >
	      ( o ) );

    if ( retval )
    {
      return retval;
    }
    throw std::range_error( "Unable to read FrAdcData" );
  }

  inline IFrameFStream::fr_proc_data_type IFrameFStream::
  ReadFrProcData( INT_4U Frame,
		 const std::string& Channel )
  {
    typedef fr_proc_data_type retval_type;

    Common::FrameSpec::ObjectInterface::object_type
      o( m_frame_stream->ReadFrProcData( Frame, Channel ) );

    retval_type
      retval( General::DynamicPointerCast< retval_type::element_type >
	      ( o ) );

    if ( retval )
    {
      return retval;
    }
    throw std::range_error( "Unable to read FrAdcData" );
  }

  inline IFrameFStream::frame_h_type IFrameFStream::
  ReadFrameN( INT_4U Index )
  {
    typedef frame_h_type retval_type;

    Common::FrameSpec::ObjectInterface::object_type
      o( m_frame_stream->ReadFrameN( Index ) );

    retval_type
      retval( General::DynamicPointerCast< retval_type::element_type >
	      ( o ) );

    if ( retval )
    {
      return retval;
    }
    throw std::range_error( "Unable to read next frame" );
  }

  inline IFrameFStream::frame_h_type IFrameFStream::
  ReadNextFrame( )
  {
    typedef frame_h_type retval_type;

    Common::FrameSpec::ObjectInterface::object_type
      o( m_frame_stream->ReadNextFrame( ) );

    retval_type
      retval( General::DynamicPointerCast< retval_type::element_type >
	      ( o ) );

    if ( retval )
    {
      return retval;
    }
    throw std::range_error( "Unable to read next frame" );
  }

  inline IFrameFStream::version_type IFrameFStream::
  Version( ) const
  {
    return m_frame_stream->Version( );
  }
}

#endif /* FRAME_CPP_INTERFACE__IFrameStream_HH */
