/* -*- mode: c++ -*- */

%module frameCPP

%{
#include "general/types.hh"
#include "framecpp/Detectors.hh"
#include "framecpp/FrDetector.hh"

using namespace FrameCPP;
%}

%import "general/SharedPtr.hh"
%import "framecpp/FrDetector.hh"

%SharedPtr(FrDetector)

%feature("autodoc",
"""
FrDetector (Detector Data) implementation of the frame specification.

  Attributes:

    name          Instrument name as described in the appendix
                  of the current frame specification document.
    prefix        Channel prefix for this detector as described in
                  the frame specification appendix.
    longitude     Detector vertex longitude, geographical coordinates:
                  radians; Value > 0 => E of Greenwich
    lattiude      Detector vertex latitude, geographical coordinates:
                  radians; Value > 0 => N of Equator
    elevation     Vertex elevation, meters, relative to WGS84 elipsoid.
    armXazimuth   Orientation of X arm, measured in radians Eas of North
                  (0<= Arm X azimuth < 2pi)
    armYazimuth   Orientation of Y arm, measured in radians Eas of North
                  (0<= Arm Y azimuth < 2pi)
    armXaltitude  Altitude (pitch) angle of X arm, measured in radians
                  above horizon (local tangent to WGS84 ellipsoid).
    armYaltitude  Altitude (pitch) angle of Y arm, measured in radians
                  above horizon (local tangent to WGS84 ellipsoid).
    armXmidpoint  Distance between the detector vertex and the middle
                  of the X cavity (meters) (should be zero for bars)
    armYmidpoint  Distance between the detector vertex and the middle
                  of the Y cavity (meters) (should be zero for bars)
    localTime     Local seasonal time - UTC in seconds. If
                  localTime % 1800 != 0, then localTime is undefined.
""" ) FrDetector;

class FrDetector
{
public:
  //---------------------------------------------------------------------
  /// \brief Retrieve the name of the instrument.
  ///
  /// \return
  ///     The name of the instrument.
  //---------------------------------------------------------------------
  const std::string& GetName( ) const;

  //---------------------------------------------------------------------
  /// \brief Retrieve the prefix of the instrument.
  ///
  /// \return
  ///     The prefix for the instrument.
  //---------------------------------------------------------------------
  const CHAR* GetPrefix( ) const;

  //---------------------------------------------------------------------
  /// \brief Retrieve the longitude of the detector vertex.
  ///
  /// \return
  ///     The longitude of the detector vertex.
  //---------------------------------------------------------------------
  REAL_8 GetLongitude( ) const;

  //---------------------------------------------------------------------
  /// \brief Retrieve the latitude of the detector vertex.
  ///
  /// \return
  ///     The latitude of the detector vertex.
  //---------------------------------------------------------------------
  REAL_8 GetLatitude( ) const;

  //---------------------------------------------------------------------
  /// \brief Retrieve the vertex elevation of the detector.
  ///
  /// \return
  ///     The vertex elevation of the detector.
  //---------------------------------------------------------------------
  REAL_4 GetElevation( ) const;

  //---------------------------------------------------------------------
  /// \brief Retrieve the orientation of X arm of the detector.
  ///
  /// \return
  ///     The orientation of the X arm of the detector.
  //---------------------------------------------------------------------
  REAL_4 GetArmXazimuth( ) const;

  //---------------------------------------------------------------------
  /// \brief Retrieve the orientation of Y arm of the detector.
  ///
  /// \return
  ///     The orientation of the Y arm of the detector.
  //---------------------------------------------------------------------
  REAL_4 GetArmYazimuth( ) const;

  //---------------------------------------------------------------------
  /// \brief Retrieve the altitude angle of X arm of the detector.
  ///
  /// \return
  ///     The altitude angle of the X arm of the detector.
  //---------------------------------------------------------------------
  REAL_4 GetArmXaltitude( ) const;

  //---------------------------------------------------------------------
  /// \brief Retrieve the altitude angle of Y arm of the detector.
  ///
  /// \return
  ///     The altitude angle of the Y arm of the detector.
  //---------------------------------------------------------------------
  REAL_4 GetArmYaltitude( ) const;

  //---------------------------------------------------------------------
  /// \brief Retrieve the midpoint of the X arm of the detector.
  ///
  /// \return
  ///     The midpoint of the X arm of the detector.
  //---------------------------------------------------------------------
  REAL_4 GetArmXmidpoint( ) const;

  //---------------------------------------------------------------------
  /// \brief Retrieve the midpoint of the Y arm of the detector.
  ///
  /// \return
  ///     The midpoint of the Y arm of the detector.
  //---------------------------------------------------------------------
  REAL_4 GetArmYmidpoint( ) const;

  //---------------------------------------------------------------------
  /// \brief Retrieve the local seasonal time.
  ///
  /// \return
  ///     The local seasonal time.
  //---------------------------------------------------------------------
  INT_4S GetLocalTime( ) const;

  LDAS_PROPERTY_READ_ONLY("name",GetName)
  LDAS_PROPERTY_READ_ONLY("prefix",GetPrefix)
  LDAS_PROPERTY_READ_ONLY("longitude",GetLongitude)
  LDAS_PROPERTY_READ_ONLY("latitude",GetLatitude)
  LDAS_PROPERTY_READ_ONLY("elevation",GetElevation)
  LDAS_PROPERTY_READ_ONLY("armXazimuth",GetArmXazimuth)
  LDAS_PROPERTY_READ_ONLY("armYazimuth",GetArmYazimuth)
  LDAS_PROPERTY_READ_ONLY("armXaltitude",GetArmXaltitude)
  LDAS_PROPERTY_READ_ONLY("armYaltitude",GetArmYaltitude)
  LDAS_PROPERTY_READ_ONLY("armXmidpoint",GetArmXmidpoint)
  LDAS_PROPERTY_READ_ONLY("armYmidpoint",GetArmYmidpoint)
  LDAS_PROPERTY_READ_ONLY("localTime",GetLocalTime)
};
