/* -*- mode: c++ -*- */
#ifndef FR_ADC_DATA_I
#define FR_ADC_DATA_I

/** @cond IGNORE_BY_DOXYGEN */
%module frameCPP

%{
#include "general/types.hh"
#include "framecpp/FrAdcData.hh"
%}

%import "framecpp/FrameCPP.hh"
%import "framecpp/FrAdcData.hh"

%include "general/SharedPtr.i"
%import "Container.i"

%SharedPtr(FrAdcData)

%feature("autodoc",
"""
FrAdcData (ADC Data) implementation of the frame specification.

  Attributes:

    name           Channel name -- must be unique with the frame
    comment        Comment
    channelNumber  Channel number
    channelGroup   Channel grouping number containing ADC
    nBits          Number of bits in A/D output
    bias           DC bias on channel (Units @ ADC counts = 0)
    slope          ADC calibration: input units/ct
    sampleRate     Data Acquisition rate, samples/s.
    timeOffset     Offset of 1st sample relative to the frame start
                   time (seconds). Must be positive and smaller
                   than the frame length.
    fShift         fShift is the frequency (in Hz) in the origional
                   data that corresponds to 0 Hz in the heterodyned
                   series.
    phase          Phase (in radian) of heterodyning signal at
                   start of dataset.
    dataValid      Data valid flag: dataValid = 0 > ADC data valid;
                   dataValid != 0 > ADC data suspect/not valid.
    data           Identifier of rverctor of sampled data.
""" ) FrAdcData;

class FrAdcData
{
public:
  typedef FR_ADC_DATA_CHANNEL_GROUP_TYPE channelGroup_type;
  typedef FR_ADC_DATA_CHANNEL_NUMBER_TYPE channelNumber_type;
  typedef FR_ADC_DATA_N_BITS_TYPE nBits_type;
  typedef FR_ADC_DATA_BIAS_TYPE bias_type;
  typedef FR_ADC_DATA_SLOPE_TYPE slope_type;
  typedef FR_ADC_DATA_SAMPLERATE_TYPE sampleRate_type;
  typedef FR_ADC_DATA_TIMEOFFSET_TYPE timeOffset_type;
  typedef FR_ADC_DATA_FSHIFT_TYPE fShift_type;
  typedef FR_ADC_DATA_PHASE_TYPE phase_type;
  typedef FR_ADC_DATA_DATAVALID_TYPE dataValid_type;

#ifdef SWIGPYTHON
  typedef Container< FrVect > data_type;
#else
  using FrameCPP::FrAdcData::data_type;
#endif

  FrAdcData( );
  FrAdcData( const std::string& Name,
	     channelGroup_type Group, channelNumber_type Channel,
	     nBits_type NBits,
	     sampleRate_type SampleRate );
	     
  const std::string& GetName() const; 
  const std::string& GetComment( ) const;
  channelNumber_type GetChannelNumber( ) const;
  channelGroup_type GetChannelGroup( ) const;
  nBits_type GetNBits( ) const;
  bias_type GetBias( ) const;
  slope_type GetSlope( ) const;
  // STRING GetUnits( ) const;
  sampleRate_type GetSampleRate( ) const;
  timeOffset_type GetTimeOffset( ) const;
  fShift_type GetFShift( ) const;
  phase_type GetPhase( ) const;
  dataValid_type GetDataValid( ) const;

  void SetComment( const std::string& Value );
  void SetChannelGroup( channelGroup_type Value );
  void SetChannelNumber( channelNumber_type Value );
  void SetDataValid( dataValid_type Value );
  void SetSampleRate( sampleRate_type Value );
  void SetTimeOffset( timeOffset_type Value );

  %extend {
    
    void
    AppendData( FrVect& Data )
    {
      self->RefData( ).append( Data );
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    size_t RefDataSize( );

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    data_type& RefData( );
  }

  LDAS_PROPERTY_READ_ONLY("name",GetName)
  LDAS_PROPERTY_READ_WRITE("comment",GetComment,SetComment)
  LDAS_PROPERTY_READ_WRITE("channelNumber",GetChannelNumber,SetChannelNumber)
  LDAS_PROPERTY_READ_WRITE("channelGroup",GetChannelGroup,SetChannelGroup)
  LDAS_PROPERTY_READ_ONLY("nBits",GetNBits)
  LDAS_PROPERTY_READ_ONLY("bias",GetBias)
  LDAS_PROPERTY_READ_ONLY("slope",GetSlope)
  LDAS_PROPERTY_READ_WRITE("sampleRate",GetSampleRate,SetSampleRate)
  LDAS_PROPERTY_READ_WRITE("timeOffset",GetTimeOffset,SetTimeOffset)
  LDAS_PROPERTY_READ_ONLY("fShift",GetFShift)
  LDAS_PROPERTY_READ_ONLY("phase",GetPhase)
  LDAS_PROPERTY_READ_WRITE("dataValid",GetDataValid,SetDataValid)
  LDAS_PROPERTY_READ_ONLY("data",RefData)

};

#if ! SWIGIMPORTED
%{
  FrAdcData::data_type&
  FrAdcData_RefData( FrAdcData* Adc )
  {
    return Adc->RefData( );
  }

  size_t
  FrAdcData_RefDataSize( FrAdcData* Adc )
  {
    return Adc->RefData( ).size( );
  }
%}

CONTAINER_WRAP(FrAdcData,FrAdcData)
#endif /* SWIGIMPORTED */

/** @endcond */ /* IGNORE_BY_DOXYGEN */

#endif /* FR_ADC_DATA_I */
