#ifndef FRAMECPP__FRAME_FILENAME_HH
#define FRAMECPP__FRAME_FILENAME_HH

#include "framecpp/config.h"

#include <string>

#include "general/ldasexception.hh"

namespace FrameCPP
{
  namespace Common
  {
    //: Class to parse frame filename
    //
    // This implemnetation is based on the document LIGO-T010150
    class FrameFilename
    {
    public:
      typedef unsigned int	gps_seconds_type;
      typedef unsigned int	total_time_interval_type;

      //: Exception describing a corrupted frame file name.
      class InvalidFrameFilename: public LdasException
      {
      public:
	InvalidFrameFilename( const std::string& Filename,
			      const std::string& Spec,
			      const char* File, size_t Line );
      private:
	static std::string format( const std::string& Filename,
				   const std::string& Spec );
      };

      //: Constructor
      FrameFilename( const std::string& Filename );
      //: Return the base filename
      const std::string& Base( ) const;
      //: Return the directory component of the filename
      const std::string& Dir( ) const;
      //: Return the description of the contents of a file
      const std::string& D( ) const;
      //: Return the GPS start time in seconds of the first frame in the file
      gps_seconds_type G( ) const;
      //: Return the source of the data
      const std::string& S( ) const;
      //: Return the total time interval covered by the file, in seconds
      total_time_interval_type T( ) const;
      //: Return the extension component of the filename
      const std::string& Ext( ) const;

    private:
      //: Name of document specifying LDAS' naming convetion for frame files
      static const std::string m_spec;

      //: Origional text of filename
      std::string			m_filename;
      //: S component of filename
      std::string			m_s;
      //: D component of filename
      std::string			m_d;
      //: G component of filename
      gps_seconds_type		m_g;
      //: T component of filename
      total_time_interval_type	m_tt;
      std::string			m_ext;
      //: Base component of the filename
      std::string			m_base;
      //: Directory component of the filename
      std::string			m_directory;
      //: state indicating that the name comforms to the spec
      bool			m_good;
    }; // class FrameFilename

    inline const std::string& FrameFilename::
    Base( ) const
    {
      if ( !m_good )
      {
	throw InvalidFrameFilename( m_filename, m_spec, __FILE__, __LINE__ );
      }
      return m_base;
    }

    inline const std::string& FrameFilename::
    Dir( ) const
    {
      if ( !m_good )
      {
	throw InvalidFrameFilename( m_filename, m_spec, __FILE__, __LINE__ );
      }
      return m_directory;
    }

    inline const std::string& FrameFilename::
    D( ) const
    {
      if ( !m_good )
      {
	throw InvalidFrameFilename( m_filename, m_spec, __FILE__, __LINE__ );
      }
      return m_d;
    }

    inline FrameFilename::gps_seconds_type FrameFilename::
    G( ) const
    {
      if ( !m_good )
      {
	throw InvalidFrameFilename( m_filename, m_spec, __FILE__, __LINE__ );
      }
      return m_g;
    }

    inline const std::string& FrameFilename::
    S( ) const
    {
      if ( !m_good )
      {
	throw InvalidFrameFilename( m_filename, m_spec, __FILE__, __LINE__ );
      }
      return m_s;
    }

    inline FrameFilename::total_time_interval_type FrameFilename::
    T( ) const
    {
      if ( !m_good )
      {
	throw InvalidFrameFilename( m_filename, m_spec, __FILE__, __LINE__ );
      }
      return m_tt;
    }

    inline const std::string& FrameFilename::
    Ext ( ) const
    {
      if ( !m_good )
      {
	throw InvalidFrameFilename( m_filename, m_spec, __FILE__, __LINE__ );
      }
      return m_ext;
    }
  } // namespace - Common
} // namespace - FrameCPP

#endif /* FRAMECPP__FRAME_FILENAME_HH */
