/* -*- mode: c++; c-basic-offset: 2; -*- */
#ifndef VALARRAY_UTILS_HH
#define	VALARRAY_UTILS_HH

#include <complex>
#include <typeinfo>
#include <valarray>

namespace Filters {

  namespace Private {
    template <class OutType, class InType>
    inline
    void
    valarray_copy_assign( std::valarray<OutType>& out,
			  const std::valarray<InType>& in )
    {
      // Not the same type, need to do translations
      std::valarray<InType>& inRef = const_cast<std::valarray<InType>&>(in);

      std::copy(&inRef[0], &inRef[inRef.size()], &out[0]);
    }
    
    template <>
    inline
    void
    valarray_copy_assign( std::valarray< int >& out,
			  const std::valarray< int >& in )
    {
      // Since they are the same type, just do a copy
      out = in;
    }

    template <>
    inline
    void
    valarray_copy_assign( std::valarray< float >& out,
			  const std::valarray< float >& in )
    {
      // Since they are the same type, just do a copy
      out = in;
    }

    template <>
    inline
    void
    valarray_copy_assign( std::valarray< double >& out,
			  const std::valarray< double >& in )
    {
      // Since they are the same type, just do a copy
      out = in;
    }
    template <>
    inline
    void
    valarray_copy_assign( std::valarray< std::complex< float > >& out,
			  const std::valarray< std::complex< float > >& in )
    {
      // Since they are the same type, just do a copy
      out = in;
    }

    template <>
    inline
    void
    valarray_copy_assign( std::valarray< std::complex< double > >& out,
			  const std::valarray< std::complex< double > >& in )
    {
      // Since they are the same type, just do a copy
      out = in;
    }

  } // namespace - Filters::Private

  //: retrieve the real component of a valarray
  //
  //!param: std::valarray< Type >& Data - Data set from which to extract
  //+       the real compoenent.
  //
  //!return: std::valarray< Type > - real component of Data
  //
  template < class Type >
  std::valarray< Type >
  real(const std::valarray< Type >& Data );

  //: retrieve the real component of a valarray
  //
  //!param: std::valarray< Type >& Data - Data set from which to extract
  //+       the real compoenent.
  //
  //!return: std::valarray< Type > - real component of Data
  //
  template < class Type >
  std::valarray< Type >
  real(const std::valarray< std::complex< Type > >& Data );

  //: retrieve the imaginary component of a valarray
  //
  //!param: std::valarray< Type >& Data - Data set from which to extract
  //+       the imaginary compoenent.
  //
  //!return: std::valarray< Type > - imaginary component of Data
  //
  template < class Type >
  std::valarray< Type >
  imag(const std::valarray< Type >& Data );

  //: retrieve the imaginary component of a valarray
  //
  //!param: std::valarray< Type >& Data - Data set from which to extract
  //+       the imaginary compoenent.
  //
  //!return: std::valarray< Type > - imaginary component of Data
  //
  template < class Type >
  std::valarray< Type >
  imag(const std::valarray< std::complex< Type > >& Data );

  //: calculate the absolute value of a valarray
  template<class T>
  std::valarray<T>
  abs(const std::valarray<std::complex<T> >& in);

  //: calculate the arg of a valarray
  template<class T>
  std::valarray<T>
  arg(const std::valarray<std::complex<T> >& in);

  //: retrieve the conjugate of a valarray
  template < class Type >
  std::valarray< Type >
  conj(const std::valarray< Type >& Data );

  //: retrieve the conjugate of a valarray
  template < class Type >
  std::valarray< std::complex< Type > >
  conj(const std::valarray< std::complex< Type > >& Data );

  //: copy from an arbitrary valarray to an arbitrary valarray
  //
  //!param: std::valarray<OutType>& out - target
  //!param: std::valarray<InType>& in - source
  //
  template <class OutType, class InType>
  inline
  void
  valarray_copy(      std::valarray<OutType>& out,
		const std::valarray<InType>& in)
  {
    if (out.size() != in.size())
    {
      out.resize(in.size());
    }

    Private::valarray_copy_assign( out, in );
  }

  //
  //: Specialisation to deal with complex<double> to complex<float>
  //
  template <>
  inline
  void
  valarray_copy(      std::valarray<std::complex<float> >& out,
		const std::valarray<std::complex<double> >& in)
  {
    if (out.size() != in.size())
    {
      out.resize(in.size());
    }

    for (size_t k = 0; k < in.size(); ++k)
    {
      out[k] = std::complex<float>(in[k].real(), in[k].imag());
    }
  }

  template< class Type >
  inline const std::valarray< Type >&
  valarray_copy_slice( const std::slice& Slice,
		       const std::valarray< Type >& In,
		       std::valarray< Type >& Out )
  {
    for ( size_t
	    x = Slice.start( ),
	    x_stride = Slice.stride( ),
	    y = 0,
	    y_end = Slice.size( );
	  y < y_end;
	  x += x_stride,
	    ++y)
    {
      Out[ y ] = In[x];
    }
    return Out;
  }
}	// namespace - Filters

#endif	/* VALARRAY_UTILS_HH */
