#include "genericAPI/config.h"

#include <sys/types.h>
#include <unistd.h>

#include <sstream>

#include "general/AtExit.hh"
#include "general/unordered_map.hh"

#include "genericAPI/Logging.hh"
#include "genericAPI/Symbols.hh"

typedef GenericAPI::LogHTML log_type;

static log_type& find_log( const std::string& Filename );

typedef General::unordered_map< std::string, log_type::message_type> group_map_type;

static group_map_type& group_map_init( );

static group_map_type& group_map = group_map_init( );

static MutexLock::lock_type	logs_baton = MutexLock::Initialize( );

namespace GenericAPI
{
  void
  queueLogEntry( const std::string& Message,
		 const std::string& Group,
		 const int Level,
		 const std::string& Caller,
		 const std::string& JobInfo,
		 const unsigned int Time,
		 const std::string& Filename )
  {
    log_type::message_type mt = group_map[ std::string( Group ) ];

    queueLogEntry( Message, mt, Level, Caller, JobInfo, Time, Filename );
  }

  void
  queueLogEntry( const std::string& Message,
		 const int Group,
		 const int Level,
		 const std::string& Caller,
		 const std::string& JobInfo,
		 const unsigned int Time,
		 const std::string& Filename )
  {
    if ( General::AtExit::IsExiting( ) == true )
    {
      return;
    }
    log_type&	log = find_log( std::string( Filename ) );
    std::ostringstream	job_info;

    try
    {
      //-----------------------------------------------------------------
      // Create hyperlinks to job files.
      //-----------------------------------------------------------------
      Symbols::RUN_CODE::value_type	run_code;
    
      Symbols::RUN_CODE::Get( run_code );
      if ( ( run_code.length( ) > 0 )
	   && ( JobInfo.compare( 0, run_code.length( ), run_code ) == 0 ) )
      {
	int			jobid;
	std::istringstream
	  jobid_stream( JobInfo.substr( run_code.length( ) ) );

	jobid_stream >> jobid;
	if ( jobid_stream.eof( ) )
	{
	  //-------------------------------------------------------------
	  // It matches the regex ^{run_code}\d+$
	  //-------------------------------------------------------------
	  Symbols::HTTP_URL::value_type	http_url;
    
	  Symbols::HTTP_URL::Get( http_url );
	  job_info << "<a href=\"" << http_url
		   << "/" << run_code << "_" << ( jobid / 10000 )
		   << "/" << JobInfo
		   << "/user_command.html\">"
		   << "<b><i>" << JobInfo
		   << "</i></b></a>"
	    ;
	  
	}
      }
    }
    catch( ... )
    {
      job_info.str( "" );
    }
    if ( job_info.str( ).length( ) == 0 )
    {
      job_info << "<b><i><font color=\"green\">" << JobInfo << "</font></i></b>";
    }

    //-------------------------------------------------------------------
    // Queue the message.
    //-------------------------------------------------------------------
    {
      MutexLock	l( logs_baton );

      if ( General::AtExit::IsExiting( ) == false )
      {
	log.Message( log_type::message_type( Group ),
		     log_type::level_type( Level ),
		     Caller, job_info.str( ), Message );
      }
    }
  }

  void
  setLogDebugLevel( const int Level,
		    const std::string& Filename )
  {
    log_type&	log = find_log( std::string( Filename ) );
    
    {
      MutexLock	l( logs_baton );

      if ( General::AtExit::IsExiting( ) == false )
      {
	log.Verbosity( log_type::MT_DEBUG, Level );
      }
    }
  }
}

//-----------------------------------------------------------------------
// Local variables
//-----------------------------------------------------------------------
typedef General::unordered_map< std::string,
				log_type* > log_container_type;

static log_container_type	logs;

static void
unregister_logs( )
{
  MutexLock	l( logs_baton );

  for( log_container_type::iterator
	 cur = logs.begin( ),
	 last = logs.end( );
       cur != last;
       ++cur )
  {
    delete cur->second;
  }
  logs.erase( logs.begin( ), logs.end( ) );
}

static GenericAPI::LogHTML&
find_log( const std::string& Filename )
{
  std::ostringstream logname( Filename );

  //---------------------------------------------------------------------
  // 
  //---------------------------------------------------------------------
  if ( logname.str( ).length( ) == 0 )
  {
    //-------------------------------------------------------------------
    // :TODO: Fix this once TCL variables can be accessed.
    // :TODO:   The current workaround is to just use the pid.
    //-------------------------------------------------------------------
    logname << "LDAS"
	    << ::getpid( )
	    << ".log.html";
  }

  //---------------------------------------------------------------------
  // Mutex lock to prevent multiple creations
  //---------------------------------------------------------------------
  MutexLock	l( logs_baton );

  log_container_type::iterator i = logs.find( logname.str( ) );
  if ( i != logs.end( ) )
  {
    //-------------------------------------------------------------------
    // Have a log associated with the file.
    //-------------------------------------------------------------------
    return *(i->second);
  }
  //---------------------------------------------------------------------
  // Check to see if the list is empty
  //---------------------------------------------------------------------
  if ( logs.empty( ) )
  {
    //-------------------------------------------------------------------
    // Since it is, register the AtExit handler to clean up the memory.
    //-------------------------------------------------------------------
    General::AtExit::Append( unregister_logs,
			     "unregister_logs@api/genericAPI/Logging.cc",
			     150 );
  }

  //---------------------------------------------------------------------
  // Does not already exist so create a new instance and then add to the
  //   list.
  //---------------------------------------------------------------------
  log_type* n( new log_type( logname.str( ) ) );
  
  
  if ( n )
  {
    log_type& retval( *n );

    logs[ logname.str( ) ] = n;
    n->Spawn( );

    return retval;
  }
  return *n;
}

static group_map_type&
group_map_init( )
{
  static group_map_type gm;

  if ( gm.size( ) == 0 )
  {
    gm[ "0" ] = log_type::MT_OK;
    gm[ "1" ] = log_type::MT_WARN;
    gm[ "2" ] = log_type::MT_ERROR;
    gm[ "3" ] = log_type::MT_EMAIL;
    gm[ "4" ] = log_type::MT_PHONE;
    gm[ "5" ] = log_type::MT_DEBUG;
    gm[ "6" ] = log_type::MT_NOTE;
    gm[ "orange" ] = log_type::MT_ORANGE;
    gm[ "green" ] = log_type::MT_GREEN;
    gm[ "yellow" ] = log_type::MT_YELLOW;
    gm[ "red" ] = log_type::MT_RED;
    gm[ "blue" ] = log_type::MT_BLUE;
    gm[ "purple" ] = log_type::MT_PURPLE;
    gm[ "phone" ] = log_type::MT_PHONE;
    gm[ "pager" ] = log_type::MT_PHONE;
    gm[ "mail" ] = log_type::MT_MAIL;
    gm[ "email" ] = log_type::MT_EMAIL;
    gm[ "certmail" ] = log_type::MT_CERTMAIL;
  }
  return gm;
}

