/* -*- mode: c++; c-basic-offset: 3 -*- */
// System Header Files   
#include "config.h"

#include <cassert>
#include <cstring>
#include <unistd.h>
#include <cerrno>

#include <algorithm>   
#include <cstdlib>   
#include <sstream>   
#include <cmath>   
#include <memory>   
   
// General Header Files   
#include "general/gpstime.hh"
#include "general/types.hh"
   
#include "framecpp/Common/CheckSum.hh"
#include "framecpp/Common/MD5Sum.hh"

#include "framecpp/FrameH.hh"
#include "framecpp/FrAdcData.hh"
#include "framecpp/FrHistory.hh"
#include "framecpp/FrProcData.hh"
#include "framecpp/FrRawData.hh"
#include "framecpp/FrVect.hh"

// Filters Header Files   
#include "filters/Resample.hh"
   
#include "filereader.hh"
#include "ResampleCmd.hh"
#include "rdsresample.hh"
#include "RDSStream.hh"
#include "rdsutil.hh"   
#include "util.hh"
   
using namespace std;
using namespace FrameCPP;

using General::GPSTime;   
using FrameCPP::Common::CheckSum;
using FrameCPP::Common::MD5Sum;
using FrameAPI::LogMD5Sum;

   
namespace
{
   static void adjustData( FrameCPP::FrVect* data, 
			   FrameCPP::FrVect* next_data,
			   const INT_4U delay );

   template< class T >
   static void adjustVector( FrameCPP::FrVect* vec, 
			     FrameCPP::FrVect* next_vec,
			     const INT_4U delay );

   inline void
   validate_resample_factor( INT_4U Value )
   {
      switch( Value )
      {
      case 1:
      case 2:
      case 4:
      case 8:
      case 16:
	 //--------------------------------------------------------------
	 // Take no action for all valid values
	 //--------------------------------------------------------------
	 break;
      default:
	 //--------------------------------------------------------------
	 // Throw exception for non-valid values
	 //--------------------------------------------------------------
	 std::ostringstream err;
	 err << "Invalid resample factor: " << Value
	     << ". Allowed resample factor: 1, 2, 4, 8, 16.";
         throw SWIGEXCEPTION( err.str( ) );
	 break;
      } // switch
   } // validate_resample_factor
}  // namespace - anonymous

//#define DEBUG_RDS   

//-----------------------------------------------------------------------------
//   
//: Constructor.
//
//!param: const char* frame_files - A list of frame file names.
//!param: const char* channels - A list of channels to extract from 
//+       original frames (only channel names are allowed).
//!param: const char* resample - A list of resample factors for
//+       each channel.   
//!param: const char* dir - Directory to write result frames to.   
//!param: const char* compression_method - method to use for compression of data
//!param: const level - level
//   
ResampleRawFrame::
ResampleRawFrame( const char* frame_files, 
		  const char* channels,
		  const char* resample,
		  const RDSFrame::Options& UserOptions )
   : RDSFrame( frame_files, channels, UserOptions ),
#if 0
     mChannelOffset( 0 ),
#endif
     mCurrentDt( 0.0f ),
#if 0
     mSeedState( true ),
#endif
     m_should_write( false )
{
   if( resample == 0 || strlen( resample ) == 0 )
   {
      throw SWIGEXCEPTION( "Resample factor must be specified." );      
   }

   // At least three files must be specified
   if( GetNumberOfFrameFiles( ) < 3 )
   {
      throw SWIGEXCEPTION( "At least three frame files must be specified." );
   }

   mResampleRecord = resample;         
   initResampleFactor();
   
   // Make sure there're as many resample factors as channels.
   if( GetNumberOfChannels( ) != mResampleFactor.size() )
   {
      throw SWIGEXCEPTION( "Inconsistent number of channels and resample factors." );
   }
}   
   
ResampleRawFrame::
ResampleRawFrame( const frame_file_container_type& frame_files, 
		  const channel_container_type& channels,
		  const resample_container_type& resampling,
		  const RDSFrame::Options& CommandOptions )
   : RDSFrame( frame_files, channels, CommandOptions ),
     mCurrentDt( 0.0f ),
     m_should_write( false )
{
   //--------------------------------------------------------------------
   // At least three files must be specified
   //--------------------------------------------------------------------
   if( GetNumberOfFrameFiles( ) < 3 )
   {
      throw SWIGEXCEPTION( "At least three frame files must be specified." );
   }

   {
      std::ostringstream	msg;
      resample_container_type::iterator	cur_rf;

      mResampleFactor.resize( resampling.size( ) );
      cur_rf = mResampleFactor.begin( );
      msg << "{ ";
      for ( resample_container_type::const_iterator
	       first = resampling.begin( ),
	       cur = resampling.begin( ),
	       last = resampling.end( );
	    cur != last;
	    ++cur, ++cur_rf )
      {
	 validate_resample_factor( *cur );
	 *cur_rf = *cur;
	 if ( cur != first )
	 {
	    msg << ",";
	 }
	 msg << *cur;
      }
      msg << " }";
      mResampleRecord = msg.str( );
   }
   
   // Make sure there're as many resample factors as channels.
   if( GetNumberOfChannels( ) != mResampleFactor.size() )
   {
      throw SWIGEXCEPTION( "Inconsistent number of channels and resample factors." );
   }
}

//-----------------------------------------------------------------------------
//   
//: Destructor.
//
ResampleRawFrame::~ResampleRawFrame()
{
   for( state_iterator iter = mResampleBase.begin(), 
                       end_iter = mResampleBase.end(); 
        iter != end_iter; ++iter )
   {
      if( *iter )
      {
         destructResampleState( *iter );
      }
   }
}

void ResampleRawFrame::
processChannel( INT_4U ChannelOffset,
		fr_adc_data_type Adc )
{
   //--------------------------------------------------------------------
   // initialize frequently accessed quantities
   //--------------------------------------------------------------------
   INT_4U		resample_factor( mResampleFactor[ ChannelOffset ] );
   //--------------------------------------------------------------------
   // Determine if things need to be initialized or not
   //--------------------------------------------------------------------
   if ( mResampleBase.size( ) <= ChannelOffset )
   {
      //-----------------------------------------------------------------
      // Seed the state
      //-----------------------------------------------------------------
      mResampleBase.push_back( ( resample_factor == 1 )
			       ? (Filters::ResampleBase*)NULL
			       : createResampleState( resample_factor,
						      Adc.get( ) ) );
      
   }
   //--------------------------------------------------------------------
   // Resample the channel
   //--------------------------------------------------------------------

   Filters::ResampleBase*	state( mResampleBase[ ChannelOffset ] );

   if ( state )
   {
      General::SharedPtr< FrProcData >	oproc;

      //-----------------------------------------------------------------
      // Perform resampling on the data
      //-----------------------------------------------------------------
      oproc.reset( resampleAdcData( Adc.get( ), state ) );
      //-----------------------------------------------------------------
      // Unregister result and put into result frame
      //-----------------------------------------------------------------
      unregisterProcData( oproc.get( ) );
      if ( mResultFrame )
      {
	 getResultProcData( )->append( oproc );
      }
   }
   else
   {
      if ( mResultFrame.get( ) )
      {
	 //--------------------------------------------------------------
	 // Append origional channel: no resampling required
	 //--------------------------------------------------------------
	 mResultFrame->GetRawData( )->RefFirstAdc( ).append( Adc );
      }
   }
}

void ResampleRawFrame::
processChannel( INT_4U ChannelOffset,
		fr_proc_data_type Proc )
{
   //--------------------------------------------------------------------
   // initialize frequently accessed quantities
   //--------------------------------------------------------------------
   INT_4U		resample_factor( mResampleFactor[ ChannelOffset ] );
   //--------------------------------------------------------------------
   // Determine if things need to be initialized or not
   //--------------------------------------------------------------------
   if ( mResampleBase.size( ) <= ChannelOffset )
   {
      //-----------------------------------------------------------------
      // Seed the state
      //-----------------------------------------------------------------
      mResampleBase.push_back( ( resample_factor == 1 )
			       ? ( Filters::ResampleBase*)NULL
			       : createResampleState( resample_factor,
						      Proc.get( ) ) );
   }
   //--------------------------------------------------------------------
   // Resample the channel
   //--------------------------------------------------------------------

   Filters::ResampleBase*	state( mResampleBase[ ChannelOffset ] );
   fr_proc_data_type		oproc;

   if ( state )
   {
      //-----------------------------------------------------------------
      // Perform resampling on the data
      //-----------------------------------------------------------------
      oproc.reset( resampleProcData( Proc.get( ), state ) );
      //-----------------------------------------------------------------
      // Put into result frame. This object was not registered
      //-----------------------------------------------------------------
      getResultProcData( )->append( oproc );
   }
   else
   {
      getResultProcData( )->append( Proc );
   }
}

//-----------------------------------------------------------------------------
//   
//: Create history record.
//      
void ResampleRawFrame::
createHistory( )
{
   if ( ( mResultFrame.get( ) == (FrameH*)NULL )
	|| ( m_options.HistoryRecord( ) == false ) )
   {
      return;
   }

   RDSFrame::createHistory( );

   GPSTime gps_time;
   gps_time.Now();
   
   ostringstream record;
   size_t len;

   record << "frameAPI: resampled RDS (resample factors: ";

   len = record.str( ).length( );

   for ( resample_container_type::const_iterator
	    cur = mResampleFactor.begin( ),
	    last = mResampleFactor.end( );
	 cur != last;
	 ++cur )
   {
      ostringstream	r;

      r << *cur;
      if ( ( r.str( ).length( ) + len + 2 )
	   >= FrameCPP::Version::STRING::MAX_STRING_LENGTH )
      {
	 //--------------------------------------------------------------
	 // Need to output the record since it is full
	 //--------------------------------------------------------------
	 record <<  ")";

	 FrameCPP::FrameH::history_type::value_type
	    h( new FrHistory( getHistoryName( ),
			      gps_time.GetSeconds(),
			      record.str( ) ) );
	 mResultFrame->RefHistory().append( h );
	 record.str( "frameAPI: RDS (resample factors (cont):" );
	 len = record.str( ).length( );
      }
      record << " " << r.str( );
      len += r.str( ).length( ) + 1;
   }
   //--------------------------------------------------------------------
   // Write the final record of resample factors information
   //--------------------------------------------------------------------
   record << ")";
   FrameCPP::FrameH::history_type::value_type
      h( new FrHistory( getHistoryName(),
			gps_time.GetSeconds(),
			record.str() ) );
   mResultFrame->RefHistory().append( h );
   
   return;
}

//-----------------------------------------------------------------------------
//   
//: Parse out numeric values of resample factors.
//
//!return: const char* resample - A list of resample factors.
//   
void ResampleRawFrame::
initResampleFactor()
{
   ListParser parser( mResampleRecord.c_str() );
   vector< string > resample_vec( parser.getTokenList() );     
   
   // Convert to numeric values
   INT_4U tmp;
   
   for( stringv_const_iterator iter = resample_vec.begin(), 
           end_iter = resample_vec.end(); iter != end_iter; ++iter )
   {
      // Convert to int:
      char* endptr( 0 );      
      tmp = strtoul( ( *iter ).c_str(), &endptr, 0 );
      if( *endptr != '\0' )
      {
         string msg( "Resample factor must be numeric: " );
         msg += *iter;
         throw SWIGEXCEPTION( msg );
      }
      validate_resample_factor( tmp );
      mResampleFactor.push_back( tmp );
   }
   
   return;
}   
   
//-----------------------------------------------------------------------------
//   
//: Adjust data for the current result frame.
// 
// Current frame object is represented by mCurrentFrame, frame object
// used for data source is represented by mResultFrame.
//           
// In a case if the channel was resampled, this method will do the 
// following data adjustment:   
// 1. Trim first data points at the beginning of the channel
// 2. Append missing data points at the end of the channel using data
//    from the next frame (mResultFrame)   
//   
void ResampleRawFrame::
adjustChannelData( )
{ 
   // Rehash container to create channel map
   mCurrentFrame->RefProcData( ).rehash( );
   
   // Rehash container to create channel map   
   getResultProcData( )->rehash();   

   INT_4U	offset( 0 );
   // Step through all channels in result frame:
   for ( state_const_iterator
	    state_iter( mResampleBase.begin( ) ),
	    end_state_iter( mResampleBase.end( ) );
	 state_iter != end_state_iter;
	 state_iter++, offset++ )
   {
      adjustResultChannel( getChannelName( offset ), *state_iter );
   }
   
   return;
}   
   
//-----------------------------------------------------------------------------
//   
//: Adjust data of resampled channel.
//   
// Caller must call 'rehash' on all passed FrAdcData containers to 
// guarantee existence of channel map for the frame.
//   
void ResampleRawFrame::adjustResultChannel( 
   const std::string& name, const Filters::ResampleBase* state )
{
   if( state == 0 )
   {
      // There's no resampling for the channel ===> nothing needs to be done
      return;
   }

   
   // Get state delay
   REAL_8 state_delay( state->getDelay() );
   if( ceil( state_delay ) != state_delay )
   {
      ostringstream msg;
      msg << "Resample delay must be an integer: ";
      msg << state_delay;
   
      throw SWIGEXCEPTION( msg.str() );
   }
   
   const INT_4U delay( static_cast< const INT_4U >( state_delay ) );
   if( delay == 0 )
   { 
      // There're no samples to adjust for the channel
      return;
   }
   

#ifdef DEBUG_RDS
   cout << "Adjusting data for channel: " << name << endl;
#endif   

   // If channel was resampled, it's FrProc   
   FrProcData* proc( getProcChannel( name,
				     &( mCurrentFrame->RefProcData( ) ) ) );
   
#ifdef DEBUG_RDS
   cout << "Got current proc channel: " << name << endl;
#endif      
   
   FrProcData* next_proc( getProcChannel( name, getResultProcData( ) ) );
   
#ifdef DEBUG_RDS
   cout << "Got next channel: " << name << endl;
#endif            
   
   
   // Channel data
   General::SharedPtr< FrameCPP::FrVect > vect( proc->RefData()[ 0 ] );
   if( ! vect )
   {
      string msg( "FrProcData of resampled frame is missing data: " );
      msg += name;
      throw SWIGEXCEPTION( msg );
   }
   
   General::SharedPtr< FrameCPP::FrVect > next_vect( next_proc->RefData()[ 0 ] );
   if( ! next_vect )
   {
      string msg( "FrProcData of next resampled frame is missing data: " );
      msg += name;
      throw SWIGEXCEPTION( msg );
   }   

   if( vect->GetNData() < delay || next_vect->GetNData() < delay )
   {
      ostringstream msg;
      msg << "Number of points for shift (currentVector="
          << vect->GetNData() << ", nextVector=" 
          << next_vect->GetNData() << ") is less than ResampleBase::delay ("
          << delay << ")";
      throw SWIGEXCEPTION( msg.str() );
   }
   
   // Get rid of 'delay' samples at the beginning of the data,
   // and append 'delay' samples from next frame channel at the end
   adjustData( vect.get( ), next_vect.get( ), delay );

   return;
}

#if 0
void ResampleRawFrame::
openFrameFile( const std::string& Filename )
{
#if 1
   assert( 0 );
#else
   RDSFrame::openFrameFile( Filename );
#endif /* 0 */
}
#endif /* 0 */

void ResampleRawFrame::
writeFrameToFile( )
{
   std::cerr << "DEBUG:"
	     << " ResampleRawFrame::writeFrameToFile:"
	     << " m_should_write: " << m_should_write
	     << " mCurrentFrame: " << (void*)(mCurrentFrame.get( ) )
	     << " mCurrentFrame->GetGTime(): "
	     << ( ( mCurrentFrame.get( ) )
		  ? mCurrentFrame->GetGTime( )
		  : GPSTime( 0, 0 ) )
	     << " Should Start: " << GPSTime( m_options.OutputTimeStart( ), 0 )
	     << std::endl
      ;
   if ( ( m_should_write == false ) &&
	( mCurrentFrame.get( ) ) &&
	( mCurrentFrame->GetGTime( ) >=
	  GPSTime( m_options.OutputTimeStart( ), 0 ) ) )
   {
      std::cerr << "DEBUG:"
		<< " __LINE__: " << __LINE__
		<< " __FILE__: " << __FILE__
		<< std::endl
	 ;

      m_should_write = true;
      m_is_last_frame_of_file = m_options.FramesPerFile( );
   }
   std::cerr << "DEBUG:"
	     << " ResampleRawFrame::writeFrameToFile:"
	     << " m_should_write: " << m_should_write
	     << " mCurrentFrame: " << (void*)( mCurrentFrame.get( ) )
	     << std::endl
      ;
   if ( m_should_write &&
	mCurrentFrame.get( ) )
   {
      //-----------------------------------------------------------------
      // Adjust the current frame data
      //-----------------------------------------------------------------

      std::cerr << "DEBUG:"
		<< " __LINE__: " << __LINE__
		<< " __FILE__: " << __FILE__
		<< std::endl
	 ;

      adjustChannelData( );

      std::cerr << "DEBUG:"
		<< " __LINE__: " << __LINE__
		<< " __FILE__: " << __FILE__
		<< std::endl
	 ;

      mCurrentFrame->SetDt( mCurrentDt );

      //-----------------------------------------------------------------
      // Write Frame to the stream
      //-----------------------------------------------------------------
      std::cerr << "DEBUG:"
		<< " __LINE__: " << __LINE__
		<< " __FILE__: " << __FILE__
		<< std::endl
	 ;

      m_stream->Write( mCurrentFrame,
		       m_options.CompressionMethod( ),
		       m_options.CompressionLevel( ),
		       ( ( m_options.CreateChecksumPerFrame( ) )
			 ? CheckSum::CRC
			 : CheckSum::NONE ) );
      //-----------------------------------------------------------------
      // Check to see if the file has been stuffed with all its frames.
      //-----------------------------------------------------------------
      std::cerr << "DEBUG:"
		<< " __LINE__: " << __LINE__
		<< " __FILE__: " << __FILE__
		<< std::endl
	 ;

      if ( ( --m_is_last_frame_of_file ) == 0 )
      {
	 m_stream->Close( false );
      }
   }
   //--------------------------------------------------------------------
   // Setup for the next frame
   //--------------------------------------------------------------------
   std::cerr << "DEBUG:"
	     << " __LINE__: " << __LINE__
	     << " __FILE__: " << __FILE__
	     << std::endl
      ;

   mCurrentFrame = mResultFrame;
   mCurrentDt = mCurrentFrame->GetDt( );
}

namespace FrameAPI
{
   namespace RDS
   {
      class CollectNames
	 : public FrameCPP::Common::FrTOC::FunctionString,
	   public channel_container_type
      {
      public:
	 virtual ~CollectNames( )
	 {
	 }

	 virtual void operator()( const std::string& ChannelName )
	 {
	    push_back( ChannelName );
	 }
      };

      void
      ExpandChannelList( channel_container_type& Channels,
			 resample_container_type& ResampleRates,
			 const std::string& SampleFrame )
      {
	 channel_container_type				retval_chan;
	 resample_container_type			retval_rates;
	 resample_container_type::const_iterator	cur_rate;
	 resample_container_type::const_iterator	last_rate;
	 channel_container_type				sample_frame_channel_names;

	 cur_rate = ResampleRates.begin( );
	 last_rate = ResampleRates.end( );
	 for( channel_container_type::const_iterator
		 cur_chan = Channels.begin( ),
		 last_chan = Channels.end( );
	      cur_chan != last_chan;
	      ++cur_chan )
	 {
	    size_t	pos( cur_chan->find( '/' ) );
	    size_t	posend( cur_chan->rfind( '/' ) );

	    if ( ( posend > pos )
		 && ( ( posend - pos ) > 1 ) )
	    {
	       //--------------------------------------------------------
	       // This is the case were a regular expression has been
	       //  listed as a channel and it needs to be expanded
	       //--------------------------------------------------------
	       if ( sample_frame_channel_names.size( ) <= 0 )
	       {
		  //-----------------------------------------------------
		  // Load the list of channel names from the specified
		  //   sample file.
		  // This is done only once for efficency purposes.
		  //-----------------------------------------------------
		  using FrameCPP::Common::FrameBuffer;
		  using FrameCPP::Common::IFrameStream;
		  using FrameCPP::Common::FrTOC;

		  FrameBuffer< General::filebuf >	ifb( std::ios::in );

		  ifb.open( SampleFrame.c_str( ), std::ios::in );

		  IFrameStream	ifs( false, &ifb );
		  const FrTOC*	toc( ifs.GetTOC( ) );
		  CollectNames	n;

		  if ( toc == (FrTOC*)NULL )
		  {
		     /// \todo Throw exception for wild cards are only supported
		     /// for frame sets with table of contents
		  }
		  toc->ForEach( FrameCPP::Common::FrTOC::TOC_CHANNEL_NAMES, n );
		  sample_frame_channel_names.swap( n );
	       }
	       //--------------------------------------------------------
	       // Need to handle matching of channels using regex.
	       //--------------------------------------------------------
	       size_t		l( ( posend > ( pos + 1 ) )
				   ? ( ( posend - pos ) - 2 )
				   : 0 );
	       std::string	suffix;
			   
	       Regex 		pattern( cur_chan->substr( pos + 1, l ) );
	       RegexMatch	matches;
	 
	       if ( posend < ( cur_chan->length( ) - 1 ) )
	       {
		  suffix = cur_chan->substr( ( posend + 1 ) );
		  std::cerr << "WARNING:"
			    << " Extra text beyond expression is being ignored: " << suffix
			    << std::endl
		     ;
	       }

	       //--------------------------------------------------------
	       // loop over channels
	       //--------------------------------------------------------
	       for( channel_container_type::const_iterator
		       cur = sample_frame_channel_names.begin( ),
		       last = sample_frame_channel_names.end( );
		    cur != last;
		    ++cur )
	       {
		  if ( matches.match( pattern, cur->c_str( ) ) )
		  {
		     //--------------------------------------------------
		     // add to list of channels
		     //--------------------------------------------------
		     retval_chan.push_back( *cur );
		     retval_rates.push_back( ( cur_rate != last_rate )
					     ? *cur_rate
					     : 1 );
		  }
	       }
	    }
	    else if ( pos != std::string::npos )
	    {
	       //--------------------------------------------------------
	       // Case where an expression has been specified, but is
	       // malformed
	       //--------------------------------------------------------
	    }
	    else
	    {
	       //--------------------------------------------------------
	       // Case where the channel name is to be placed in the
	       // output
	       //--------------------------------------------------------
	       retval_chan.push_back( *cur_chan );
	       retval_rates.push_back( ( cur_rate != last_rate )
				       ? *cur_rate
				       : 1 );
	    }
	    if ( cur_rate != last_rate )
	    {
	       ++cur_rate;
	    }
	 }
	 Channels.swap( retval_chan );
	 ResampleRates.swap( retval_rates );
      } // Function - ExpandChannelList
   } // namespace - RDS
} // namespace - FrameAPI

namespace {
   //--------------------------------------------------------------------
   //   
   //: Do actual data adjustment on resampled channel.
   //      
   void adjustData( FrameCPP::FrVect* data, 
		    FrameCPP::FrVect* next_data,
		    const INT_4U delay )
   {
      FrVect::data_types_type id
	 = FrVect::data_types_type( data->GetType() );

      switch( id )
      {
      case( FrVect::FR_VECT_2S ):
	 return adjustVector< INT_2S >( data, next_data, delay );
      case( FrVect::FR_VECT_2U ):
         return adjustVector< INT_2U >( data, next_data, delay );
      case( FrVect::FR_VECT_4S ):
         return adjustVector< INT_4S >( data, next_data, delay );
      case( FrVect::FR_VECT_4U ):
         return adjustVector< INT_4U >( data, next_data, delay );
      case( FrVect::FR_VECT_8S ):
         return adjustVector< INT_8S >( data, next_data, delay );
      case( FrVect::FR_VECT_8U ):
         return adjustVector< INT_8U >( data, next_data, delay );
      case( FrVect::FR_VECT_8C ):
         return adjustVector< COMPLEX_8 >( data, next_data, delay );
      case( FrVect::FR_VECT_16C ):
         return adjustVector< COMPLEX_16 >( data, next_data, delay );
      case( FrVect::FR_VECT_4R ):
         return adjustVector< REAL_4 >( data, next_data, delay );
      case( FrVect::FR_VECT_8R ):
         return adjustVector< REAL_8 >( data, next_data, delay );
      case( FrVect::FR_VECT_C ):
         return adjustVector< CHAR >( data, next_data, delay );
      case( FrVect::FR_VECT_1U ):
         return adjustVector< CHAR_U >( data, next_data, delay );
      default:
         throw SWIGEXCEPTION( "Unsupported vector type." );
      }
   
      return;
   }
   
   template< class T >
   void adjustVector( FrameCPP::FrVect* v,
		      FrameCPP::FrVect* next_v,
		      const INT_4U delay )
   {
      FrameCPP::FrVect::data_type	data_src( v->GetDataUncompressed( ) );
      FrameCPP::FrVect::data_type	next_data_src( next_v->GetDataUncompressed( ) );

      T* data( reinterpret_cast< T* >( data_src.get( ) ) );
      const T* next_data( reinterpret_cast< const T* >( next_data_src.get( ) ) );
      
      if( data == 0 || next_data == 0 )
      {
	 throw SWIGEXCEPTION( "Channel has no data." );   
      }
      
      // Get rid of first 'delay' samples
      const INT_4U numData( v->GetNData() );
      copy( data + delay, data + numData, data );
      
      // Append 'delay' samples at the end
      copy( next_data, next_data + delay, data + numData - delay );
      
      return;
   }
   
   // Instantiate templates   
   template void adjustVector< CHAR >( FrameCPP::FrVect* v,
				       FrameCPP::FrVect* next_v,
				       const INT_4U delay );
   template void adjustVector< CHAR_U >( FrameCPP::FrVect* v,
					 FrameCPP::FrVect* next_v,
					 const INT_4U delay );   
   template void adjustVector< INT_2S >( FrameCPP::FrVect* v
					 , FrameCPP::FrVect* next_v,
					 const INT_4U delay );   
   template void adjustVector< INT_2U >( FrameCPP::FrVect* v,
					 FrameCPP::FrVect* next_v,
					 const INT_4U delay );   
   template void adjustVector< INT_4S >( FrameCPP::FrVect* v,
					 FrameCPP::FrVect* next_v,
					 const INT_4U delay );      
   template void adjustVector< INT_4U >( FrameCPP::FrVect* v,
					 FrameCPP::FrVect* next_v,
					 const INT_4U delay );         
   template void adjustVector< INT_8S >( FrameCPP::FrVect* v,
					 FrameCPP::FrVect* next_v,
					 const INT_4U delay );         
   template void adjustVector< INT_8U >( FrameCPP::FrVect* v,
					 FrameCPP::FrVect* next_v,
					 const INT_4U delay );            
   template void adjustVector< REAL_4 >( FrameCPP::FrVect* v,
					 FrameCPP::FrVect* next_v,
					 const INT_4U delay );            
   template void adjustVector< REAL_8 >( FrameCPP::FrVect* v,
					 FrameCPP::FrVect* next_v,
					 const INT_4U delay );               
   template void adjustVector< COMPLEX_8 >( FrameCPP::FrVect* v,
					    FrameCPP::FrVect* next_v,
					    const INT_4U delay );
   template void adjustVector< COMPLEX_16 >( FrameCPP::FrVect* v,
					     FrameCPP::FrVect* next_v,
					     const INT_4U delay );
} // namespace - anonymous
