#ifndef DiskCacheCmdHH
#define DiskCacheCmdHH   

// #define DEBUG_TIME

// System Header Files   
#include <string>
#include <iostream>
   
#include "general/types.hh"
#include "genericAPI/threaddecl.hh"

#if FOR_TCL
#include "genericAPI/tid.hh"
#endif /* FOR_TCL */

#include "diskcacheAPI/Cache/ExcludedDirectoriesSingleton.hh"
#include "diskcacheAPI/Cache/QueryAnswer.hh"
#include "diskcacheAPI/Cache/SDGTx.hh"

#include "Streams/StreamsInterface.hh"

#include "MountPointManagerSingleton.hh"
#include "MountPointScanner.hh"

namespace diskCache
{
  namespace Cache
  {
    class QueryAnswer;
  } // namespace - diskCache::Cache
} // namespace - diskCache

using diskCache::Cache::ExcludedDirectoriesSingleton;
using diskCache::MountPointManagerSingleton;

//------------------------------------------------------------------------------
//
//: Invoke "slow" scan (recursive collection of subdirectories and 
//: frame data files) on directory listed in MOUNT_PT list (rsc file variable).
//   
// This function gets recursive content of the passed to it directory. It
// should be called only for directories that appear in the MOUNT_PT list.   
// The same function is used to introduce new MOUNT_PT directory or to rescan
// already existing directory in the global hash for new data.
//   
//!usage: set string [ getDirEnt dirpath ]
//   
//!param: const char* dirpath - A full path to the directory.
//   
//!return: const string - A list of all identified and scanned subdirectories
//+        that have frame data.   
//      
std::string getDirEnt( const std::string dirpath );

#if defined( CREATE_THREADED1_DECL )
CREATE_THREADED1_DECL( getDirEnt, std::string, const std::string );   
#endif /* defined( CREATE_THREADED1_DECL ) */

   
#if ! FOR_PYTHON
//------------------------------------------------------------------------------
//  
//: Get TCL formatted lists for each mount point with name or mount point, 
//: number of directories and number of files for data matching the specified
//: ifo and type.  Return lists are of the form:
//:   {mountpoint_name number_of_dirs number_of_files } 
//
//!usage: set string [ getHashNumbers ]
//
//!param: const char* ifo - An ifo to look up. Default is "all".
//!param: const char* type - A type to look up. Default is "all".
//
//!return: string - A list for each mount point with name, number of
//+        directories and number of files under that mount point:
//+        {mountpoint_name number_of_dirs number_of files }
// 
void
getHashNumbers( diskCache::Cache::QueryAnswer& Answer,
		const char* ifo = "all",
		const char* type = "all" );
#endif /* ! FOR_PYTHON */


//------------------------------------------------------------------------------
//   
//: Write content of global frame data hash to binary file.   
//
//!param: const char* filename - Name of the file to write frame hash to.
//+       Default is an empty string (C++ will use default file name).      
//
//!return: Nothing.
//
void writeDirCache( const char* filename = "",
		    diskCache::Streams::Interface::version_type Version
		      = diskCache::Streams::Interface::VERSION_NONE );   
#if ! FOR_PYTHON
CREATE_THREADED2V_DECL( writeDirCache, const char*, diskCache::Streams::Interface::version_type );
#endif /* ! FOR_PYTHON */

//------------------------------------------------------------------------------
//   
//: Write content of global frame data hash to binary file.   
//
//!param: const char* filename - Name of the file to write frame hash to.
//+       Default is an empty string (C++ will use default file name).      
//
//!return: Nothing.
//
void writeDirCacheAscii( const char* filename = "",
			 diskCache::Streams::Interface::version_type Version
			   = diskCache::Streams::Interface::VERSION_NONE );   
#if ! FOR_PYTHON
CREATE_THREADED2V_DECL( writeDirCacheAscii, const char*, diskCache::Streams::Interface::version_type );
#endif /* ! FOR_PYTHON */

   
//------------------------------------------------------------------------------
//   
//: Write content of global frame data hash to binary file.   
//
//!param: const char* bfilename - Name of the file to write binary frame hash to.
//+       Default is an empty string (C++ will use default file name).      
//
//!param: const char* afilename - Name of the file to write ascii frame hash to.
//+       Default is an empty string (C++ will use default file name).      
//
//!return: Nothing.
//
void writeDirCacheFiles( const char* bfilename = "",
			 const char* afilename = "" );
#if ! FOR_PYTHON
CREATE_THREADED2V_DECL( writeDirCacheFiles, const char*, const char* );
#endif /* ! FOR_PYTHON */

//------------------------------------------------------------------------------
//   
//: Read content of global frame data hash from binary file.   
//
// Tcl layer can specify different files for read and write operations. 
// This function forces all read and write operations to be sequencial.
//   
// ATTN: This function destroys existing hash before reading a new one from the
//       given file. Caller must assure there are no running threads that might
//       access global frame data hash at the time "readDirCache" is called.
//    
//!param: const char* filename - Name of the file to read frame hash from.
//+       Default is an empty string (C++ will use default file name).      
//
//!return: Nothing.
//
void CacheRead( const char* filename = "" );   

#if ! FOR_PYTHON
CREATE_THREADED1V_DECL( CacheRead, const char* );         
#endif /* ! FOR_PYTHON */
#if 0
//------------------------------------------------------------------------------
//   
//: Read content of global frame data hash from binary file.   
//
// Tcl layer can specify different files for read and write operations. 
// This function forces all read and write operations to be sequencial.
//   
// ATTN: This function destroys existing hash before reading a new one from the
//       given file. Caller must assure there are no running threads that might
//       access global frame data hash at the time "readDirCache" is called.
//    
//!param: const char* filename - Name of the file to read frame hash from.
//+       Default is an empty string (C++ will use default file name).      
//
//!return: Nothing.
//
void readDirCache( const char* filename = "" );   

#endif /* 0 */
   

//------------------------------------------------------------------------------
//   
//: Update list of excluded directories (as they appear in
//: the resource file): check if existing data hash relies on such directories
//:                     already, remove those directories recursively from
//:                     global hash.   
//
//!usage: set dir_list [ excludedDirList dirs_to_exclude ]
//   
//!param: const CHAR* dir_list - A list of directories to exclude 
//+                              (as they appear in API resource file variable).
//   
//!return: string - Sorted Tcl list of all removed subdirectories, followed
//+        by error messages if any:
//+        {Directories excluded by pattern 'dir_list': dir1 dir2 ...} 
//+        {error1 error2 ...}
//         
std::string excludedDirList( const ExcludedDirectoriesSingleton::directory_container_type&
			     dir_list );   

//------------------------------------------------------------------------------
//   
//: Delete global frame data hash.
//
// ATTN: This function destroys existing hash in memory. 
//       Caller must assure there are no running threads that might
//       access global frame data hash at the time "deleteDirCache" is called.
//    
//!return: Nothing.
//
void deleteDirCache();         
   
#if ! FOR_PYTHON
CREATE_THREADED0V_DECL( deleteDirCache );
#endif /* !FOR_PYTHON */

#if FOR_TCL
bool bindToCheckForMountPTConflictVariable( Tcl_Interp* Interp, char* Variable );
#endif /* FOR_TCL */


//-----------------------------------------------------------------------
//-----------------------------------------------------------------------
void DumpCacheDaemonStart( std::ostream* Stream = NULL );

//------------------------------------------------------------------------------
/// \brief The number of mount point directories to scan concurrently
//------------------------------------------------------------------------------
void ScanConcurrency( INT_4U Concurrency );

//------------------------------------------------------------------------------
/// \brief Scan the list of mount points.
//------------------------------------------------------------------------------
void ScanMountPointList( diskCache::MountPointScanner::ScanResults& Answer );

//-----------------------------------------------------------------------
/// \brief Continuously scan the list of mount points
//-----------------------------------------------------------------------
void ScanMountPointListContinuously( std::ostream* Stream = NULL );

#endif /* define DiskCacheCmdHH */
