/* -*- mode: c++ -*- */
#ifndef FR_PROC_DATA_I
#define FR_PROC_DATA_I

%module frameCPP

%{
#include "ldastoolsal/types.hh"
#include "framecpp/FrProcData.hh"

using FrameCPP::FrProcData;
typedef FrameCPP::FrProcData::AuxParamList_type AuxParamList_type;
typedef FrameCPP::FrProcData::AuxParam_type AuxParam_type;

%}

%include "std_string.i"
%include "std_map.i"

%import "framecpp/FrameCPP.hh"
%import "framecpp/FrProcData.hh"
%import "framecpp/FrVect.hh"

%import "Container.i"

%include "ldastoolsal/SharedPtr.i"
%SharedPtr(FrProcData)

%feature("autodoc",
"""
FrProcData (Post-processed Data) implementation of the frame specification.

  Attributes:

    name        Data or channel name.
    comment     Comment.
    type        Type of data object.
    subType     Subtye for f-Series (TBD for other types).
    timeOffset  Offset of 1st sample relative to the frame start
                time (seconds). Must be positive and smaller than
                the frame length.
    tRange      Duration of sampled data (tStop-tStart).
    fShift      fShift is the frequency in the origional data
                that corresponds to 0 Hz in the heterodyned series.
                In multidimensional objects this applies to the first
                frequency dimension.
    phase       Phase of hetrodyning signal at start of dataset
                (radians, 0 if unknown).
    fRange      Frequency range (= fMax - fMin, 0 if unknown).
    BW          Resolution bandwidth.
    auxParam    Array of auxiliary parameters.
    data        Data vector.

""" ) FrProcData;

class FrProcData
{
public:
  typedef FR_PROC_DATA_TIME_OFFSET_TYPE	timeOffset_type;
  typedef FR_PROC_DATA_T_RANGE_TYPE	tRange_type;
  typedef FR_PROC_DATA_F_SHIFT_TYPE	fShift_type;
  typedef FR_PROC_DATA_PHASE_TYPE	phase_type;
  typedef FR_PROC_DATA_F_RANGE_TYPE	fRange_type;
  typedef FR_PROC_DATA_BW_TYPE		BW_type;

#ifdef SWIGPYTHON
  typedef Container< FrVect >		data_type;
#else
  using FrameCPP::FrProcData::data_type;
#endif

  enum type_type {
    UNKNOWN_TYPE = FrameCPP::FrProcData::UNKNOWN_TYPE,
    TIME_SERIES = FrameCPP::FrProcData::TIME_SERIES,
    FREQUENCY_SERIES = FrameCPP::FrProcData::FREQUENCY_SERIES,
    OTHER_1D_SERIES_DATA = FrameCPP::FrProcData::OTHER_1D_SERIES_DATA,
    TIME_FREQUENCY = FrameCPP::FrProcData::TIME_FREQUENCY,
    WAVELETS = FrameCPP::FrProcData::WAVELETS,
    MULTI_DIMENSIONAL = FrameCPP::FrProcData::MULTI_DIMENSIONAL
  };

  enum subType_type {
    UNKNOWN_SUB_TYPE = FrameCPP::FrProcData::UNKNOWN_SUB_TYPE,
    //---------------------------------------------------------------
    // Subtype for fSeries
    //---------------------------------------------------------------
    DFT = FrameCPP::FrProcData::DFT,
    AMPLITUDE_SPECTRAL_DENSITY = FrameCPP::FrProcData::AMPLITUDE_SPECTRAL_DENSITY,
    POWER_SPECTRAL_DENSITY = FrameCPP::FrProcData::POWER_SPECTRAL_DENSITY,
    CROSS_SPECTRAL_DENSITY = FrameCPP::FrProcData::CROSS_SPECTRAL_DENSITY,
    COHERENCE = FrameCPP::FrProcData::COHERENCE,
    TRANSFER_FUNCTION = FrameCPP::FrProcData::TRANSFER_FUNCTION
  };

  FrProcData( );
  FrProcData( const std::string& Name,
	      const std::string& Comment,
	      type_type Type,
	      subType_type SubType,
	      timeOffset_type TimeOffset,
	      tRange_type TRange,
	      fShift_type FShift,
	      phase_type Phase,
	      fRange_type FRange,
	      BW_type BW );

  const std::string& GetName() const; 
  const std::string& GetComment( ) const;
  type_type GetType( ) const;
  subType_type GetSubType( ) const;
  timeOffset_type GetTimeOffset( ) const;
  tRange_type GetTRange( ) const;
  fShift_type GetFShift( ) const;
  phase_type GetPhase( ) const;
  fRange_type GetFRange( ) const;
  BW_type GetBW( ) const;
  const AuxParamList_type& GetAuxParam( ) const;

  void SetComment( const std::string& Value );
  void SetTimeOffset( timeOffset_type Value );
  void SetTRange( tRange_type Value );
  void SetAuxParam( const AuxParamList_type& Value );

  %extend {
    void
    AppendData( FrVect& Data )
    {
      self->RefData( ).append( Data );
    }

    void
    AppendAuxParam( const char* Name, double Value )
    {
      self->AppendAuxParam( FrProcData::AuxParam_type( Name, Value ) );
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    size_t RefDataSize( );
    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    data_type& RefData( );
  }

  LDAS_PROPERTY_READ_ONLY("name",GetName)
  LDAS_PROPERTY_READ_WRITE("comment",GetComment,SetComment)
  LDAS_PROPERTY_READ_ONLY("type",GetType)
  LDAS_PROPERTY_READ_ONLY("subType",GetSubType)
  LDAS_PROPERTY_READ_WRITE("timeOffset",GetTimeOffset,SetTimeOffset)
  LDAS_PROPERTY_READ_WRITE("tRange",GetTRange,SetTRange)
  LDAS_PROPERTY_READ_ONLY("fShift",GetFShift)
  LDAS_PROPERTY_READ_ONLY("phase",GetPhase)
  LDAS_PROPERTY_READ_ONLY("fRange",GetFRange)
  LDAS_PROPERTY_READ_ONLY("BW",GetBW)
  LDAS_PROPERTY_READ_ONLY("data",RefData)
  LDAS_PROPERTY_READ_WRITE("auxParam",GetAuxParam,SetAuxParam)
};

CONTAINER_WRAP(FrProcData,FrProcData)

%{
  FrProcData::data_type&
  FrProcData_RefData( FrProcData* Proc )
  {
    if ( Proc->RefData( ).size( ) <= 0 )
    {
      throw std::length_error( "FrProcData no data available" );
    }
    return Proc->RefData( );
  }

  size_t
  FrProcData_RefDataSize( FrProcData* Proc )
  {
    return Proc->RefData( ).size( );
  }
%}

#endif /* FR_PROC_DATA_I */
