#include <framecpp_config.h>

#include "ldastoolsal/mutexlock.hh"
#include "ldastoolsal/Singleton.hh"
#include "ldastoolsal/gpstime.hh"

#include "framecpp/Common/Description.hh"
#include "framecpp/Common/FrameStream.hh"

#include "framecpp/Version4/FrProcData.hh"
#include "framecpp/Version4/FrSE.hh"
#include "framecpp/Version4/FrSH.hh"

#include "framecpp/Version4/PTR_STRUCT.hh"

#include "Common/ComparePrivate.hh"

using std::ceil;
using std::floor;

using FrameCPP::Common::Description;
using FrameCPP::Common::FrameSpec;

//=======================================================================
// Static
//=======================================================================

static const FrameSpec::Info::frame_object_types s_object_id
= FrameSpec::Info::FSI_FR_PROC_DATA;

//=======================================================================
// FrProcData::fr_proc_data_ref_type
//=======================================================================
static INT_2U MAX_REF = 3;

namespace FrameCPP
{
  namespace Version_4
  {
    Common::IStream& FrProcData::fr_proc_data_ref_type::
    operator( )( Common::IStream& Stream )
    {
      Stream >> data
	     >> aux
	     >> table
	;
      return Stream;
    }

    FrameCPP::cmn_streamsize_type FrProcData::fr_proc_data_ref_type::
    Bytes( const Common::StreamBase& Stream ) const
    {
      return
	Stream.PtrStructBytes( )	// data
	+ Stream.PtrStructBytes( )	// aux
	+ Stream.PtrStructBytes( )	// table
	;
    }

    void FrProcData::fr_proc_data_ref_type::
    Write( ostream_type& Stream ) const
    {
      Stream << data
	     << aux
	     << table
	;
    }

    //=======================================================================
    // FrProcData
    //=======================================================================
    FrProcData::
    FrProcData( )
      : FrameSpec::Object( s_object_id, StructDescription( ) ),
	m_synced_with_vector( false )
    {
    }

    FrProcData::
    FrProcData( const FrProcData& ProcData )
      : FrameSpec::Object( s_object_id, StructDescription( ) ),
	FrProcDataData( ProcData ),
	TOCInfo( ProcData ),
	m_refs( ProcData.m_refs ),
	m_synced_with_vector( false )
    {
    }

    FrProcData::
    FrProcData( const std::string& Name,
		const std::string& Comment,
		const REAL_8 SampleRate,
		const GPSTime& TimeOffset,
		const REAL_8 FShift )
      : FrameSpec::Object( s_object_id, StructDescription( ) ),
	FrProcDataData( Name, Comment, SampleRate, TimeOffset, FShift ),
	m_synced_with_vector( false )
    {
    }

    FrProcData::
    FrProcData( Previous::FrProcData& Source, istream_type* Stream )
      : FrameSpec::Object( s_object_id, StructDescription( ) ),
	FrProcDataData( Source ),
	m_synced_with_vector( false )
    {
      if ( Stream )
      {
	//-------------------------------------------------------------------
	// Modify references
	//-------------------------------------------------------------------
	Stream->ReplaceRef( RefData( ), Source.RefData( ), MAX_REF );
	Stream->ReplaceRef( RefAux( ), Source.RefAux( ), MAX_REF );
      }
    }

    FrProcData::
    FrProcData( istream_type& Stream )
      : FrameSpec::Object( s_object_id, StructDescription( ) ),
	FrProcDataData( Stream ),
	m_synced_with_vector( false )
    {
      m_refs( Stream );
      Stream.Next( this );
    }

    FrProcData::
    ~FrProcData( )
    {
    }

    FrameCPP::cmn_streamsize_type FrProcData::
    Bytes( const Common::StreamBase& Stream ) const
    {
      return
	m_data.Bytes( Stream )
	+ m_refs.Bytes( Stream )
	+ Stream.PtrStructBytes( ) // next
	;
    }

    FrProcData* FrProcData::
    Create( istream_type& Stream ) const
    {
      return new FrProcData( Stream );
    }

    const char* FrProcData::
    ObjectStructName( ) const
    {
      return StructName( );
    }

    const std::string& FrProcData::
    GetNameSlow( ) const
    {
      return GetName( );
    }

    const Description* FrProcData::
    StructDescription( )
    {
      static Description ret;

      if ( ret.size( ) == 0 )
      {
	ret( FrSH( FrProcData::StructName(), s_object_id,
		   "Post-processed Data Structure" ) );

	ret( FrSE( "name", "STRING",
		   "" ) );
	ret( FrSE( "comment", "STRING",
		   "" ) );
	ret( FrSE( "sampleRate", "REAL_8",
		   "" ) );
	ret( FrSE( "timeOffsetS", "INT_4U",
		   "" ) );
	ret( FrSE( "timeOffsetN", "INT_4U",
		   "" ) );
	ret( FrSE( "fShift", "REAL_8",
		   "" ) );

	ret( FrSE( "data",	PTR_STRUCT::Desc( FrVect::StructName( ) ) ) );
	ret( FrSE( "aux", PTR_STRUCT::Desc( FrVect::StructName( ) ) ) );
	ret( FrSE( "table", PTR_STRUCT::Desc( FrTable::StructName( ) ) ) );

	ret( FrSE( "next", PTR_STRUCT::Desc( FrProcData::StructName( ) ) ) );
      }

      return &ret;
    }

    void FrProcData::
    Write( ostream_type& Stream ) const
    {
      m_data.Write( Stream );
      m_refs.Write( Stream );

      WriteNext( Stream );
    }

    void FrProcData::
#if WORKING_VIRTUAL_TOCQUERY
    TOCQuery( int InfoClass, ... ) const
#else /*  WORKING_VIRTUAL_TOCQUERY */
    vTOCQuery( int InfoClass, va_list vl ) const
#endif /*  WORKING_VIRTUAL_TOCQUERY */
    {
      using Common::TOCInfo;

#if WORKING_VIRTUAL_TOCQUERY
      va_list	vl;
      va_start( vl, InfoClass );
#endif /* WORKING_VIRTUAL_TOCQUERY */

      while ( InfoClass != TOCInfo::IC_EOQ )
      {
	int data_type = va_arg( vl, int );
	switch( data_type )
	{
	case TOCInfo::DT_STRING_2:
	  {
	    STRING* data = va_arg( vl, STRING* );
	    switch( InfoClass )
	    {
	    case TOCInfo::IC_NAME:
	      *data = GetName( );
	      break;
	    default:
	      goto cleanup;
	      break;
	    }
	  }
	  break;
	default:
	  // Stop processing
	  goto cleanup;
	}
	InfoClass = va_arg( vl, int );
      }
    cleanup:
#if WORKING_VIRTUAL_TOCQUERY
      va_end( vl )
#endif /* WORKING_VIRTUAL_TOCQUERY */
	;
    }

    FrProcData& FrProcData::
    Merge( const FrProcData& RHS )
    {
      //:TODO: Need to implement Merge routine
      std::string msg( "Merge currently not implemented for " );
      msg += StructName( );

      throw std::domain_error( msg );
      return *this;
    }

#if WORKING
    std::auto_ptr< FrProcData > FrProcData::
    SubFrProcData( REAL_8 Offset, REAL_8 Dt ) const
    {
#if WORKING || 1
      std::auto_ptr< FrProcData >	retval;

      //---------------------------------------------------------------------
      // Process the types that are understood
      //---------------------------------------------------------------------
      switch( GetType( ) )
      {
      case TIME_SERIES:
	retval = sub_time_series( Offset, Dt );
	break;
      default:
	{
	  //-----------------------------------------------------------------
	  // This type is not supported so throw an exception
	  //-----------------------------------------------------------------
	  std::ostringstream	msg;
	  msg << "Currently cannot take a subsection of type: "
	      << IDTypeToString( GetType( ) );
	  const std::string& subtype( IDSubTypeToString( GetType( ),
							 GetSubType( ) ) );
	  if ( subtype.length( ) > 0 )
	  {
	    msg << " subType: " << subtype;
	  }
	  throw std::runtime_error( msg.str( ) );
	}
	break;
      }

      return retval;
#else /* WORKING */
      throw
	Unimplemented( "FrProcData::SubFrAdcData( READL_8 Offset, REAL_8 Dt ) const",
		       DATA_FORMAT_VERSION,
		       __FILE__, __LINE__ );

#endif /* WORKING */
    }
#endif /* 0 */

    bool FrProcData::
    operator==( const Common::FrameSpec::Object& Obj ) const
    {
      return Common::Compare( *this, Obj );
    }

    void FrProcData::
    copy_core( const FrProcData& Source )
    {
      static_cast< FrProcDataData& >( *this ) = Source;
      m_refs.copy_core( Source.m_refs );
    }

    FrProcData::demote_ret_type FrProcData::
    demote( INT_2U Target,
	    demote_arg_type Obj,
	    istream_type* Stream ) const
    {
      if ( Target >= DATA_FORMAT_VERSION )
      {
	return Obj;
      }
      try
      {
	//-------------------------------------------------------------------
	// Copy non-reference information
	//-------------------------------------------------------------------
	// Do actual down conversion
	/// \todo See if the actual sample rate can be computed
	demote_ret_type
	  retval( new Previous::FrProcData( GetName( ),
					    GetComment( ),
					    GetSampleRate( ),
					    GetTimeOffset( ),
					    GetFShift( )
					    ) )
	  ;
	if ( Stream )
	{
#if WORKING
	  //-----------------------------------------------------------------
	  // Modify references
	  //-----------------------------------------------------------------
	  Stream->ReplaceRef( retval->RefData( ), RefData( ), MAX_REF );
	  Stream->ReplaceRef( retval->RefAux( ), RefAux( ), MAX_REF );
	  Stream->RemoveResolver( RefTable( ), MAX_REF );
#else
	  assert( 0 );
#endif
	}
	//-------------------------------------------------------------------
	// Return demoted object
	//-------------------------------------------------------------------
	return retval;
      }
      catch( ... )
      {
      }
      throw
	Unimplemented( "Object* FrProcData::demote( Object* Obj ) const",
		       DATA_FORMAT_VERSION, __FILE__, __LINE__ );
    }

    FrProcData::promote_ret_type FrProcData::
    promote( INT_2U Target,
	     promote_arg_type Obj,
	     istream_type* Stream ) const
    {
      return Promote( Target, Obj, Stream );
    }

#if WORKING
    // Routine to take a subsection when the data type is TimeSeries data
    std::auto_ptr< FrProcData > FrProcData::
    sub_time_series( REAL_8 Offset, REAL_8 Dt ) const
    {
      //---------------------------------------------------------------------
      // Sanity checks
      //---------------------------------------------------------------------
      // Verify the existance of only a single FrVect component
      //---------------------------------------------------------------------
      if ( m_refs.data.size( ) != 1 )
      {
	std::ostringstream	msg;

	msg << "METHOD: frameAPI::FrProcData::SubFrProcData: "
	    << "data vectors of length " << m_refs.data.size( )
	    << " currently are not supported";
	throw std::logic_error( msg.str( ) );
      }
      INT_4U	num_elements = m_refs.data[0]->GetDim( 0 ).GetNx( );
      //---------------------------------------------------------------------
      //:TODO: Ensure the dimension of the FrVect is 1
      //---------------------------------------------------------------------
      //---------------------------------------------------------------------
      //:TODO: Ensure the requested range is valid
      //---------------------------------------------------------------------
      REAL_8	sample_rate = 1.0 / m_refs.data[ 0 ]->GetDim( 0 ).GetDx( );
      if ( Offset > ( m_data.timeOffset +
		      ( num_elements / sample_rate ) ) )
      {
	std::ostringstream msg;
	msg << "METHOD: frameAPI::FrProcData::SubFrProcData: "
	    << "The value for the parameter Offset (" << Offset
	    << ") is out of range ([0-"
	    << ( m_data.timeOffset + ( num_elements / sample_rate ) )
	    << "))";
	throw std::range_error( msg.str( ) );
      }
      //---------------------------------------------------------------------
      // Start moving of data
      //---------------------------------------------------------------------
      std::auto_ptr< FrProcData >	retval( new FrProcData );
      // Copy in the header information
      retval->copy_core( *this );

      //:TODO: Get the slice of FrVect
      REAL_8	startSampleReal =
	( Offset * sample_rate ) -
	( retval->m_data.timeOffset * sample_rate );

      INT_4U	endSample =
	INT_4U( ceil ( startSampleReal +
		       ( Dt * sample_rate ) ) );
      INT_4U	startSample = ( startSampleReal > REAL_8( 0 ) )
	? INT_4U( floor( startSampleReal ) )
	: INT_4U( 0 );

      retval->m_refs.data.append( this->m_refs.data[ 0 ]->SubFrVect( startSample, endSample ).release( ),
				  false, /* allocate */
				  true /* own */ );
      //---------------------------------------------------------------------
      // Shift the offset
      //---------------------------------------------------------------------
      if ( startSampleReal > 0 )
      {
	// There is no gap
	retval->m_data.timeOffset = REAL_8( 0 );
      }
      else
      {
	retval->m_data.timeOffset += Offset;
      }
      //---------------------------------------------------------------------
      // :TODO: Recalculate tRange
      //---------------------------------------------------------------------
      retval->SetTRange( retval->RefData( )[ 0 ]->GetDim( 0 ).GetNx( ) * 
			 retval->RefData( )[ 0 ]->GetDim( 0 ).GetDx( ) );
      //---------------------------------------------------------------------
      // Return the new FrProcData
      //---------------------------------------------------------------------
      return retval;
    }
#endif /* WORKING */
  } // namespace - Version_4
} // namespace - FrameCPP
