/*
 * Copyright (C) 2015-2017  Leo Singer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with with program; see the file COPYING. If not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301  USA
 */


#include <lal/cubic_interp.h>
#include <gsl/gsl_test.h>
#include <gsl/gsl_math.h>

#include <lal/XLALError.h>

#ifdef __GNUC__
#define UNUSED __attribute__ ((unused))
#else
#define UNUSED
#endif


int main(int UNUSED argc, char UNUSED **argv)
{
    {
        static const double data[] = {0, 0, 0, 0};
        cubic_interp *interp = cubic_interp_init(data, 4, -1, 1);
        XLAL_CHECK_EXIT(interp);
        for (double t = 0; t <= 1; t += 0.01)
        {
            const double result = cubic_interp_eval(interp, t);
            const double expected = 0;
            gsl_test_abs(result, expected, 0,
                "testing cubic interpolant for zero input");
        }
        cubic_interp_free(interp);
    }

    {
        static const double data[] = {1, 1, 1, 1};
        cubic_interp *interp = cubic_interp_init(data, 4, -1, 1);
        XLAL_CHECK_EXIT(interp);
        for (double t = 0; t <= 1; t += 0.01)
        {
            const double result = cubic_interp_eval(interp, t);
            const double expected = 1;
            gsl_test_abs(result, expected, 0,
                "testing cubic interpolant for unit input");
        }
        cubic_interp_free(interp);
    }

    {
        static const double data[] = {1, 0, 1, 4};
        cubic_interp *interp = cubic_interp_init(data, 4, -1, 1);
        XLAL_CHECK_EXIT(interp);
        for (double t = 0; t <= 1; t += 0.01)
        {
            const double result = cubic_interp_eval(interp, t);
            const double expected = gsl_pow_2(t);
            gsl_test_abs(result, expected, 10 * GSL_DBL_EPSILON,
                "testing cubic interpolant for quadratic input");
        }
        cubic_interp_free(interp);
    }

    {
        static const double data[] = {
            GSL_POSINF, GSL_POSINF, GSL_POSINF, GSL_POSINF};
        cubic_interp *interp = cubic_interp_init(data, 4, -1, 1);
        XLAL_CHECK_EXIT(interp);
        for (double t = 0; t <= 1; t += 0.01)
        {
            const double result = cubic_interp_eval(interp, t);
            const double expected = GSL_POSINF;
            gsl_test_abs(result, expected, 0,
                "testing cubic interpolant for +inf input");
        }
        cubic_interp_free(interp);
    }

    {
        static const double data[] = {
            0, GSL_POSINF, GSL_POSINF, GSL_POSINF};
        cubic_interp *interp = cubic_interp_init(data, 4, -1, 1);
        XLAL_CHECK_EXIT(interp);
        for (double t = 0; t <= 1; t += 0.01)
        {
            const double result = cubic_interp_eval(interp, t);
            const double expected = GSL_POSINF;
            gsl_test_abs(result, expected, 0,
                "testing cubic interpolant for +inf input");
        }
        cubic_interp_free(interp);
    }

    {
        static const double data[] = {
            GSL_POSINF, GSL_POSINF, GSL_POSINF, 0};
        cubic_interp *interp = cubic_interp_init(data, 4, -1, 1);
        XLAL_CHECK_EXIT(interp);
        for (double t = 0; t <= 1; t += 0.01)
        {
            const double result = cubic_interp_eval(interp, t);
            const double expected = GSL_POSINF;
            gsl_test_abs(result, expected, 0,
                "testing cubic interpolant for +inf input");
        }
        cubic_interp_free(interp);
    }

    {
        static const double data[] = {
            0, GSL_POSINF, GSL_POSINF, 0};
        cubic_interp *interp = cubic_interp_init(data, 4, -1, 1);
        XLAL_CHECK_EXIT(interp);
        for (double t = 0; t <= 1; t += 0.01)
        {
            const double result = cubic_interp_eval(interp, t);
            const double expected = GSL_POSINF;
            gsl_test_abs(result, expected, 0,
                "testing cubic interpolant for +inf input");
        }
        cubic_interp_free(interp);
    }

    {
        static const double data[] = {
            0, 0, GSL_POSINF, 0};
        cubic_interp *interp = cubic_interp_init(data, 4, -1, 1);
        XLAL_CHECK_EXIT(interp);
        for (double t = 0.01; t <= 1; t += 0.01)
        {
            const double result = cubic_interp_eval(interp, t);
            const double expected = 0;
            gsl_test_abs(result, expected, 0,
                "testing cubic interpolant for +inf input");
        }
        cubic_interp_free(interp);
    }

    {
        static const double data[] = {
            0, GSL_NEGINF, GSL_POSINF, 0};
        cubic_interp *interp = cubic_interp_init(data, 4, -1, 1);
        XLAL_CHECK_EXIT(interp);
        const double result = cubic_interp_eval(interp, 1);
        cubic_interp_free(interp);
        const double expected = GSL_POSINF;
        gsl_test_abs(result, expected, 0,
            "testing cubic interpolant for +inf input");
    }

    {
        static const double data[] = {
            0, GSL_POSINF, GSL_NEGINF, 0};
        cubic_interp *interp = cubic_interp_init(data, 4, -1, 1);
        XLAL_CHECK_EXIT(interp);
        const double result = cubic_interp_eval(interp, 0);
        cubic_interp_free(interp);
        const double expected = GSL_POSINF;
        gsl_test_abs(result, expected, 0,
            "testing cubic interpolant for +inf input");
    }

    {
        static const double data[] = {
            0, GSL_NEGINF, GSL_NEGINF, GSL_NEGINF};
        cubic_interp *interp = cubic_interp_init(data, 4, -1, 1);
        XLAL_CHECK_EXIT(interp);
        for (double t = 0; t <= 1; t += 0.01)
        {
            const double result = cubic_interp_eval(interp, t);
            const double expected = GSL_NEGINF;
            gsl_test_abs(result, expected, 0,
                "testing cubic interpolant for -inf input");
        }
        cubic_interp_free(interp);
    }

    {
        static const double data[] = {
            GSL_NEGINF, GSL_NEGINF, GSL_NEGINF, 0};
        cubic_interp *interp = cubic_interp_init(data, 4, -1, 1);
        XLAL_CHECK_EXIT(interp);
        for (double t = 0; t <= 1; t += 0.01)
        {
            const double result = cubic_interp_eval(interp, t);
            const double expected = GSL_NEGINF;
            gsl_test_abs(result, expected, 0,
                "testing cubic interpolant for -inf input");
        }
        cubic_interp_free(interp);
    }

    {
        static const double data[] = {
            0, GSL_NEGINF, GSL_NEGINF, 0};
        cubic_interp *interp = cubic_interp_init(data, 4, -1, 1);
        XLAL_CHECK_EXIT(interp);
        for (double t = 0; t <= 1; t += 0.01)
        {
            const double result = cubic_interp_eval(interp, t);
            const double expected = GSL_NEGINF;
            gsl_test_abs(result, expected, 0,
                "testing cubic interpolant for -inf input");
        }
        cubic_interp_free(interp);
    }

    {
        static const double data[] = {
            0, 0, GSL_NEGINF, 0};
        cubic_interp *interp = cubic_interp_init(data, 4, -1, 1);
        XLAL_CHECK_EXIT(interp);
        for (double t = 0.01; t <= 1; t += 0.01)
        {
            const double result = cubic_interp_eval(interp, t);
            const double expected = 0;
            gsl_test_abs(result, expected, 0,
                "testing cubic interpolant for -inf input");
        }
        cubic_interp_free(interp);
    }

    {
        static const double data[] = {
            0, GSL_NEGINF, GSL_POSINF, 0};
        cubic_interp *interp = cubic_interp_init(data, 4, -1, 1);
        XLAL_CHECK_EXIT(interp);
        const double result = cubic_interp_eval(interp, 0);
        cubic_interp_free(interp);
        const double expected = GSL_NEGINF;
        gsl_test_abs(result, expected, 0,
            "testing cubic interpolant for -inf input");
    }

    {
        static const double data[] = {
            0, GSL_POSINF, GSL_NEGINF, 0};
        cubic_interp *interp = cubic_interp_init(data, 4, -1, 1);
        XLAL_CHECK_EXIT(interp);
        const double result = cubic_interp_eval(interp, 1);
        cubic_interp_free(interp);
        const double expected = GSL_NEGINF;
        gsl_test_abs(result, expected, 0,
            "testing cubic interpolant for -inf input");
    }

    {
        static const double data[] = {
            0, GSL_NEGINF, GSL_POSINF, 0};
        cubic_interp *interp = cubic_interp_init(data, 4, -1, 1);
        XLAL_CHECK_EXIT(interp);
        for (double t = 0.01; t < 1; t += 0.01)
        {
            const double result = cubic_interp_eval(interp, t);
            const double expected = GSL_NEGINF;
            gsl_test_abs(result, expected, 0,
                "testing cubic interpolant for indeterminate input");
        }
        cubic_interp_free(interp);
    }


    {
        static const double constants[] = {
            0, 1, GSL_POSINF, GSL_NEGINF, GSL_NAN};
        for (unsigned k = 0; k < sizeof(constants) / sizeof(*constants); k ++)
        {
            double data[4][4];
            for (int i = 0; i < 4; i ++)
                for (int j = 0; j < 4; j ++)
                    data[i][j] = constants[k];
            bicubic_interp *interp = bicubic_interp_init(
                *data, 4, 4, -1, -1, 1, 1);
            for (double s = -5; s <= 2; s += 0.1)
            {
                for (double t = -5; t <= 1; t += 0.1)
                {
                    const double result = bicubic_interp_eval(interp, s, t);
                    const double expected = constants[k];
                    gsl_test_abs(result, expected, 0,
                        "testing bicubic interpolant for constant %g input",
                        constants[k]);
                }
            }
            XLAL_CHECK_EXIT(interp);
            bicubic_interp_free(interp);
        }
    }

    for (int k = 1; k < 3; k ++)
    {
        {
            double data[4][4];
            for (int i = 0; i < 4; i ++)
                for (int j = 0; j < 4; j ++)
                    data[i][j] = gsl_pow_int(i - 1, k);
            bicubic_interp *interp = bicubic_interp_init(
                *data, 4, 4, -1, -1, 1, 1);
            for (double s = 0; s <= 1; s += 0.1)
            {
                for (double t = 0; t <= 1; t += 0.1)
                {
                    const double result = bicubic_interp_eval(interp, s, t);
                    const double expected = gsl_pow_int(s, k);
                    gsl_test_abs(result, expected, 10 * GSL_DBL_EPSILON,
                        "testing bicubic interpolant for s^%d input", k);
                }
            }
            XLAL_CHECK_EXIT(interp);
            bicubic_interp_free(interp);
        }

        {
            double data[4][4];
            for (int i = 0; i < 4; i ++)
                for (int j = 0; j < 4; j ++)
                    data[i][j] = gsl_pow_int(j - 1, k);
            bicubic_interp *interp = bicubic_interp_init(
                *data, 4, 4, -1, -1, 1, 1);
            for (double s = 0; s <= 1; s += 0.1)
            {
                for (double t = 0; t <= 1; t += 0.1)
                {
                    const double result = bicubic_interp_eval(interp, s, t);
                    const double expected = gsl_pow_int(t, k);
                    gsl_test_abs(result, expected, 10 * GSL_DBL_EPSILON,
                        "testing bicubic interpolant for t^%d input", k);
                }
            }
            XLAL_CHECK_EXIT(interp);
            bicubic_interp_free(interp);
        }

        {
            double data[4][4];
            for (int i = 0; i < 4; i ++)
                for (int j = 0; j < 4; j ++)
                    data[i][j] = gsl_pow_int(i - 1, k) + gsl_pow_int(j - 1, k);
            bicubic_interp *interp = bicubic_interp_init(
                *data, 4, 4, -1, -1, 1, 1);
            for (double s = 0; s <= 1; s += 0.1)
            {
                for (double t = 0; t <= 1; t += 0.1)
                {
                    const double result = bicubic_interp_eval(interp, s, t);
                    const double expected = gsl_pow_int(s, k)
                                          + gsl_pow_int(t, k);
                    gsl_test_abs(result, expected, 10 * GSL_DBL_EPSILON,
                        "testing bicubic interpolant for s^%d + t^%d input",
                        k, k);
                }
            }
            XLAL_CHECK_EXIT(interp);
            bicubic_interp_free(interp);
        }
    }

    return gsl_test_summary();
}
