## Copyright (C) 2014 Reed Essick
##
## This program is free software; you can redistribute it and/or modify it
## under the terms of the GNU General Public License as published by the
## Free Software Foundation; either version 2 of the License, or (at your
## option) any later version.
##
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
## Public License for more details.
##
## You should have received a copy of the GNU General Public License along
## with this program; if not, write to the Free Software Foundation, Inc.,
## 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
#

description = \
    """
This code checks the calibration of the iDQ pipeline using the FAP time-series data.
We expect that segments generated by thresholding on the FAP time-series at FAP=0.XYZ should correspond to an amount of time DURATION=0.XYZ*LIVETIME
"""

# =================================================

import numpy as np
from laldetchar.idq import event
from laldetchar.idq import idq

from laldetchar.idq import idq_gdb_utils

from laldetchar import git_version

__author__ = 'Reed Essick <reed.essick@ligo.org>'
__version__ = git_version.id
__date__ = git_version.date


# =================================================

def timeseries_to_segments(t, ts, thr):
    """
....computes segements from t  = time stamps
....                        ts = time series (values)
....                        thr=threshold on time series
....so that t \in segments iff ts(t) >= thr

....pad is added to the end of the time-series points when generating segments
...."""

    segs = []
    in_seg = False
    min_TS = np.infty
    for (T, TS) in zip(t, ts):
        if TS < thr:
            if in_seg:
                segs.append([start, T])
                in_seg = False
            else:
                pass
        else:

              # TS >= thr

            if min_TS > TS:
                min_TS = TS
            if not in_seg:
                start = T
                in_seg = True
    if in_seg:
        dt = t[1] - t[0]
        segs.append([start, T])

    if len(segs):
        return (segs, min_TS)
    else:
        return ([], None)


# ========================

def check_calibartion(
    realtimedir,
    start,
    end,
    classifier,
    FAPthrs,
    verbose=False,
    ):
    """
....checks the pipeline's calibration at each "FAPthr in FAPThrs"
        """

        # =================================================
        # grab idq_segments so we compute meaningful livetimes
        # =================================================

    if verbose:
        print 'getting idq_segements'

    idq_segs = idq.get_idq_segments(realtimedir, start, end,
                                    suffix='.npy.gz')
    idq_livetime = event.livetime(idq_segs)

    # =================================================
    # grab relevant data
    # =================================================

    if verbose:
        print 'looking for *_fap_*.npy.gz files in', opts.realtimedir

    fapfilenames = [filename for filename in
                    idq.get_all_files_in_range(realtimedir, start, end,
                    pad=0, suffix='.npy.gz') if '_fap_' in filename
                    and classifier in filename]

    if opts.verbose:
        print 'discovered %d files' % len(fapfilenames)
        print 'building time-series'

    (times, timeseries) = idq_gdb_utils.combine_ts(fapfilenames)

        # =================================================
        # check calibration
        # =================================================

    segments = []
    deadtimes = []
    statedFAPs = []
    for FAPthr in FAPthrs:
        if verbose:
            print 'computing segements for FAPthr =', FAPthr

        segs = []
        max_statedFAP = None
        for (t, ts) in zip(times, timeseries):
            (_segs, _min_ts) = timeseries_to_segments(t, -ts, -FAPthr)  # we want FAP <= FAPthr <--> -FAP >= FAPthr
            segs += _segs
            if _min_ts != None:
                statedFAP = -_min_ts
                if max_statedFAP < statedFAP:
                    max_statedFAP = statedFAP

        segs = event.andsegments([segs, idq_segs])
        segments.append(segs)
        deadtimes.append(1.0 * event.livetime(segs) / idq_livetime)
        statedFAPs.append(statedFAP)

    return (idq_segs, segments, deadtimes, statedFAPs)


# =================================================
#
#                  MAIN
#
# =================================================

if __name__ == '__main__':
    from optparse import OptionParser

    parser = OptionParser(version='Name: %%prog\n%s'
                          % git_version.verbose_msg,
                          usage='%prog [options]',
                          description=description)

    parser.add_option('-v', '--verbose', default=False,
                      action='store_true')
    parser.add_option('-r', '--realtimedir', default='./', type='string'
                      ,
                      help='the iDQ realtimedir in which FAP time-series files are stored'
                      )
    parser.add_option('-s', '--start', default=-np.infty, type='float'
                      ,
                      help='gps start time'
                      )
    parser.add_option('-e', '--end', default=np.infty, type='float'
                      ,
                      help='gps end time'
                      )
    parser.add_option('-c', '--classifier', default='ovl', type='string'
                      ,
                      help='classifier for which we test the calibration'
                      )
    parser.add_option('', '--FAPthrs', default='0.01', type='string',
                      help='check calibration at these FAP values (must be supplied as a string. eg: "0.001 0.005 0.01"). DEFAULT="0.01"'
                      )

    (opts, args) = parser.parse_args()

    FAPthrs = [float(l) for l in opts.FAPthrs.split()]

    (idq_segs, segments, deadtimes, statedFAPs) = check_calibartion(
        opts.realtimedir,
        opts.start,
        opts.end,
        opts.classifier,
        FAPthrs,
        verbose=opts.verbose,
        )

    # ## Report!

    print 'idq_livetime = %.3f' % event.livetime(idq_segs)

    report_str = \
        """
    FAPthr   = %.5f
  stated FAP = %.5f
   deadtime  = %.5f
%s difference = %.3f%s"""

    for (FAPthr, deadtime, statedFAP) in zip(FAPthrs, deadtimes,
            statedFAPs):

                # ## Report!

        print report_str % (
            FAPthr,
            statedFAP,
            deadtime,
            '%',
            100 * (deadtime / FAPthr - 1),
            '%',
            )

