/*
*  Copyright (C) 2007 Badri Krishnan, Reinhard Prix, Alicia Sintes Olives
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with with program; see the file COPYING. If not, write to the
*  Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
*  MA  02111-1307  USA
*/

/**
 * \file
 * \ingroup lalapps_pulsar_Hough
 * \author Alicia Sintes, Badri Krishnan
 * \brief
 * Monte Carlo signal injections for several h_0 values and
 * compute the Hough transform for only one point in parameter space each time
 */

/* 
 The idea is that we would like to analize a 300 Hz band on a cluster of
 machines. Each process should analyze 1 Hz band  (or whatever).
 
 	- Read the  band to be analized and the wings needed to read the originals SFTs. 
	-Read the h_0 values to be analyzed in one go
	- Read the file  containing the times and velocities generated by
	DriveHoughColor or compute them 
	-loop over the MC injections:
		+ Generate random parameters (f, f', alpha, delata, i...)
		+ generate h(t), produce its FFT
		+ Add h(f) to SFT for a given h_o value (and all of them)
		+ get number count
		+ wite to file
	(note if one loop fails, should print error , but continue with the next
	value)
	
Input shoud be from
             SFT files 
	     band, wings, nh_0, h_01, h_02....
	     ephemeris info
             (it should also read the times and velocities used in
	     DriveHoughColor)
	     
   This code will output files containing the MC results and info about injected
   signals. 
*/

#include "./MCInjectHoughS2.h" /* proper path*/



#define EARTHEPHEMERIS "./earth00-04.dat"
#define SUNEPHEMERIS "./sun00-04.dat"

#define ACCURACY 0.00000001 /* of the velocity calculation */
#define MAXFILES 3000 /* maximum number of files to read in a directory */
#define MAXFILENAMELENGTH 256 /* maximum # of characters  of a SFT filename */

#define IFO 2         /*  detector, 1:GEO, 2:LLO, 3:LHO */
#define THRESHOLD 1.6 /* thresold for peak selection, with respect to the
                              the averaged power in the search band */
#define F0 250.0          /*  frequency to build the LUT and start search */
#define FBAND 2.0          /* search frequency band  (in Hz) */
#define ALPHA 0.0		/* center of the sky patch (in radians) */
#define DELTA  (-LAL_PI_2)
#define PATCHSIZEX (LAL_PI*0.99) /* patch size */
#define PATCHSIZEY (LAL_PI*0.99)
#define NFSIZE  21 /* n-freq. span of the cylinder, to account for spin-down
                          search */
#define BLOCKSRNGMED 101 /* Running median window size */
#define NH0 2 /* number of h0 values to be analyzed */
#define H0MIN 1.0e-23
#define H0MAX 1.0e-22
#define NMCLOOP 2 /* number of Monte-Carlos */
#define NTEMPLATES 16 /* number templates for each Monte-Carlo */

#define SFTDIRECTORY "/home/badkri/L1sfts"
#define FILEOUT "./HoughMC"      /* prefix file output */
#define HARMONICSFILE "./harmonicsS2LLO4K_200_400.txt"

#define TRUE (1==1)
#define FALSE (1==0)

/* vvvvvvvvvvvvvvvvvvvvvvvvvvvvvv------------------------------------ */
int main(int argc, char *argv[]){

  static LineNoiseInfo   lines, lines2;
  static LineHarmonicsInfo harmonics; 

  static LALStatus            status;  
  static LALDetector          detector;
  static LIGOTimeGPSVector    *timeV=NULL;
  static REAL8Cart3CoorVector velV;
  static REAL8Vector          timeDiffV;
  static REAL8Vector          foft;
  static REAL8Vector          foftV[NTEMPLATES];
  static REAL8Vector          h0V;
 
  static HoughInjectParams    injectPar;
  static PulsarData           pulsarInject;
  static HoughTemplate        pulsarTemplate;
  static HoughNearTemplates   closeTemplates;

  INT4 nLines=0, nHarmonicSets, count1;

  SFTVector    *inputSFTs  = NULL;  
  SFTVector    *outputSFTs = NULL;
  REAL4TimeSeries   *signalTseries = NULL;
  
  static PulsarSignalParams  params;
  static SFTParams           sftParams;

  EphemerisData   *edat = NULL;

  static COMPLEX8SFTData1  sft1;
  static REAL8PeriodoPSD   periPSD;
  static UCHARPeakGram     pg1;
    
  UINT4  msp; /*number of spin-down parameters */

  
  UINT4  numberCount,maxNumberCount;
  INT4   nTemplates, controlN, controlNN, controlNH;
  UINT4  numberCountV[NTEMPLATES];
   
  INT4   mObsCoh, nfSizeCylinder;
  INT8   f0Bin, fLastBin;           /* freq. bin to perform search */
  REAL8  normalizeThr;
  REAL8  timeBase, deltaF;

  REAL8  threshold, h0scale;

  UINT4  sftlength; 
  INT4   fWings;

  INT4   sftFminBin;
  REAL8  fHeterodyne;
  REAL8  tSamplingRate;
      
  CHAR *fnamelog=NULL;
 
  INT4 MCloopId;
  INT4 h0loop;
  
  FILE  *fpPar = NULL;
  FILE  *fpH0 = NULL;
  FILE  *fpNc = NULL;
  FILE  *fpLog = NULL;
  CHAR   *logstr=NULL; 

  /* user input variables */
  INT4 uvar_blocksRngMed, uvar_nh0, uvar_nMCloop, uvar_AllSkyFlag;
  REAL8 uvar_f0, uvar_fSearchBand, uvar_peakThreshold, uvar_h0Min, uvar_h0Max;
  REAL8 uvar_alpha, uvar_delta, uvar_patchSizeAlpha, uvar_patchSizeDelta;
  CHAR *uvar_earthEphemeris=NULL;
  CHAR *uvar_sunEphemeris=NULL;
  CHAR *uvar_sftDir=NULL;
  CHAR *uvar_fnameout=NULL;
  CHAR *uvar_harmonicsfile=NULL;  
  CHAR *uvar_ifo=NULL;

#ifdef TIMING
  unsigned long long start, stop;
#endif


#ifdef TIMING
   start = realcc();
#endif

  /*  set up the default parameters  */

  uvar_AllSkyFlag = 1;
  
  uvar_nh0 = NH0;
  uvar_h0Min = H0MIN;
  uvar_h0Max = H0MAX;

  uvar_nMCloop = NMCLOOP;
  nTemplates = NTEMPLATES;  
  uvar_alpha = ALPHA;
  uvar_delta = DELTA;
  uvar_f0 =  F0;
  uvar_fSearchBand = FBAND;
  uvar_peakThreshold = THRESHOLD;
  nfSizeCylinder = NFSIZE;
  uvar_patchSizeAlpha = PATCHSIZEX;
  uvar_patchSizeDelta = PATCHSIZEY; 

  uvar_earthEphemeris = (CHAR *)LALMalloc(512*sizeof(CHAR));
  strcpy(uvar_earthEphemeris,EARTHEPHEMERIS);

  uvar_sunEphemeris = (CHAR *)LALMalloc(512*sizeof(CHAR));
  strcpy(uvar_sunEphemeris,SUNEPHEMERIS);

  uvar_sftDir = (CHAR *)LALMalloc(512*sizeof(CHAR));
  strcpy(uvar_sftDir,SFTDIRECTORY);

  uvar_harmonicsfile = (CHAR *)LALMalloc(512*sizeof(CHAR));
  strcpy(uvar_harmonicsfile,HARMONICSFILE);  

  uvar_fnameout = (CHAR *)LALMalloc(512*sizeof(CHAR));
  strcpy(uvar_fnameout, FILEOUT);

  uvar_blocksRngMed = BLOCKSRNGMED;


  /* register user input variables */
  LAL_CALL( LALRegisterSTRINGUserVar( &status, "ifo",             'i', UVAR_OPTIONAL, "Detector L1, H1, H2, G1",       &uvar_ifo ),            &status);
  LAL_CALL( LALRegisterINTUserVar(    &status, "blocksRngMed",    'w', UVAR_OPTIONAL, "RngMed block size",             &uvar_blocksRngMed),    &status);
  LAL_CALL( LALRegisterREALUserVar(   &status, "f0",              'f', UVAR_OPTIONAL, "Start search frequency",        &uvar_f0),              &status);
  LAL_CALL( LALRegisterREALUserVar(   &status, "fSearchBand",     'b', UVAR_OPTIONAL, "Search frequency band",         &uvar_fSearchBand),     &status);
  LAL_CALL( LALRegisterREALUserVar(   &status, "peakThreshold",   't', UVAR_OPTIONAL, "Peak selection threshold",      &uvar_peakThreshold),   &status);
  LAL_CALL( LALRegisterSTRINGUserVar( &status, "earthEphemeris",  'E', UVAR_OPTIONAL, "Earth Ephemeris file",          &uvar_earthEphemeris),  &status);
  LAL_CALL( LALRegisterSTRINGUserVar( &status, "sunEphemeris",    'S', UVAR_OPTIONAL, "Sun Ephemeris file",            &uvar_sunEphemeris),    &status);
  LAL_CALL( LALRegisterSTRINGUserVar( &status, "sftDir",          'D', UVAR_OPTIONAL, "SFT Directory",                 &uvar_sftDir),          &status);
  LAL_CALL( LALRegisterSTRINGUserVar( &status, "fnameout",        'o', UVAR_OPTIONAL, "Output file prefix",            &uvar_fnameout),        &status);
  LAL_CALL( LALRegisterREALUserVar(   &status, "alpha",           'r', UVAR_OPTIONAL, "Right ascension",               &uvar_alpha),           &status);
  LAL_CALL( LALRegisterREALUserVar(   &status, "delta",           'l', UVAR_OPTIONAL, "Declination",                   &uvar_delta),           &status);
  LAL_CALL( LALRegisterREALUserVar(   &status, "patchSizeAlpha",  'R', UVAR_OPTIONAL, "Patch size in right ascension", &uvar_patchSizeAlpha),  &status);
  LAL_CALL( LALRegisterREALUserVar(   &status, "patchSizeDelta",  'L', UVAR_OPTIONAL, "Patch size in declination",     &uvar_patchSizeDelta),  &status);
  LAL_CALL( LALRegisterINTUserVar(    &status, "patch",           'P', UVAR_OPTIONAL, "Inject in patch if 0",          &uvar_AllSkyFlag),      &status);  
  LAL_CALL( LALRegisterINTUserVar(    &status, "nMCloop",         'N', UVAR_OPTIONAL, "Number of MC injections",       &uvar_nMCloop),         &status);
  LAL_CALL( LALRegisterREALUserVar(   &status, "h0Min",           'm', UVAR_OPTIONAL, "Smallest h0 to inject",         &uvar_h0Min),           &status);
  LAL_CALL( LALRegisterREALUserVar(   &status, "h0Max",           'M', UVAR_OPTIONAL, "Largest h0 to inject",          &uvar_h0Max),           &status);
  LAL_CALL( LALRegisterINTUserVar(    &status, "nh0",             'n', UVAR_OPTIONAL, "Number of h0 values to inject", &uvar_nh0),             &status);  
  LAL_CALL( LALRegisterSTRINGUserVar( &status, "harmonicsfile",   'H', UVAR_OPTIONAL, "List of known lines",           &uvar_harmonicsfile),   &status);



  /* read all command line variables */
  BOOLEAN should_exit = 0;
  LAL_CALL( LALUserVarReadAllInput(&status, &should_exit, argc, argv), &status);
  if (should_exit)
    exit(1);
  

  /* write the log file */
  fnamelog = (CHAR *)LALMalloc( 512*sizeof(CHAR));
  strcpy(fnamelog, uvar_fnameout);
  strcat(fnamelog, "_log");
  /* open the log file for writing */
  if ((fpLog = fopen(fnamelog, "w")) == NULL) {
    fprintf(stderr, "Unable to open file %s for writing\n", fnamelog);
    LALFree(fnamelog);
    exit(1);
  }

  /* get the log string */
  LAL_CALL( LALUserVarGetLog(&status, &logstr, UVAR_LOGFMT_CFGFILE), &status);  


  fprintf( fpLog, "## Log file for MCInjectHoughS2\n\n");
  fprintf( fpLog, "# User Input:\n");
  fprintf( fpLog, "#-------------------------------------------\n");
  fprintf( fpLog, "%s", logstr);
  LALFree(logstr);

  /* copy contents of harmonics file into logfile */
  fprintf(fpLog, "\n\n# Contents of harmonics file:\n");
  fclose(fpLog);
  {
    CHAR command[1024] = "";
    sprintf(command, "cat %s >> %s", uvar_harmonicsfile, fnamelog);
    if ( system(command) ) fprintf (stderr, "\nsystem('%s') returned non-zero status!\n\n", command );
  }

  /* append an ident-string defining the exact CVS-version of the code used */
  fpLog = fopen(fnamelog, "a");
  {
    CHAR command[1024] = "";
    fprintf (fpLog, "\n\n# CVS-versions of executable:\n");
    fprintf (fpLog, "# -----------------------------------------\n");
    fclose (fpLog);
    
    sprintf (command, "ident %s | sort -u >> %s", argv[0], fnamelog);
    /* we don't check this. If it fails, we assume that */
    /* one of the system-commands was not available, and */
    /* therefore the CVS-versions will not be logged */
    if ( system(command) ) fprintf (stderr, "\nsystem('%s') returned non-zero status!\n\n", command );

    LALFree(fnamelog); 
  }


  /* real line noise information */
  /* find number of harmonics */
  LAL_CALL( LALFindNumberHarmonics (&status, &harmonics, uvar_harmonicsfile), &status); 
  nHarmonicSets = harmonics.nHarmonicSets; 
  
  /* convert harmonics to explicit lines */
  nLines = 0;
  if (nHarmonicSets > 0)
    {
      REAL8 dopplerFreq;
      harmonics.startFreq = (REAL8 *)LALMalloc(harmonics.nHarmonicSets * sizeof(REAL8));
      harmonics.gapFreq = (REAL8 *)LALMalloc(harmonics.nHarmonicSets * sizeof(REAL8));
      harmonics.numHarmonics = (INT4 *)LALMalloc(harmonics.nHarmonicSets * sizeof(INT4));
      harmonics.leftWing = (REAL8 *)LALMalloc(harmonics.nHarmonicSets * sizeof(REAL8));
      harmonics.rightWing = (REAL8 *)LALMalloc(harmonics.nHarmonicSets * sizeof(REAL8));
    

      LAL_CALL( LALReadHarmonicsInfo( &status, &harmonics, uvar_harmonicsfile ), &status);
      
      for (count1=0; count1 < nHarmonicSets; count1++)
	{
	  nLines += *(harmonics.numHarmonics + count1);
	}

      
      lines2.nLines = nLines;
      lines2.lineFreq = (REAL8 *)LALMalloc(nLines * sizeof(REAL8));
      lines2.leftWing = (REAL8 *)LALMalloc(nLines * sizeof(REAL8));
      lines2.rightWing = (REAL8 *)LALMalloc(nLines * sizeof(REAL8));
      
      lines.nLines = nLines;
      lines.lineFreq = (REAL8 *)LALMalloc(nLines * sizeof(REAL8));
      lines.leftWing = (REAL8 *)LALMalloc(nLines * sizeof(REAL8));
      lines.rightWing = (REAL8 *)LALMalloc(nLines * sizeof(REAL8));
      
      LAL_CALL( LALHarmonics2Lines( &status, &lines2, &harmonics), &status);
      
      dopplerFreq = (uvar_f0 + uvar_fSearchBand)*VTOT;
      LAL_CALL( LALChooseLines (&status, &lines, &lines2, uvar_f0 - dopplerFreq, 
			uvar_f0 + uvar_fSearchBand + dopplerFreq), &status); 
      nLines = lines.nLines;
      
      LALFree(lines2.lineFreq);
      LALFree(lines2.leftWing);
      LALFree(lines2.rightWing);

      LALFree(harmonics.startFreq);
      LALFree(harmonics.gapFreq);
      LALFree(harmonics.numHarmonics);
      LALFree(harmonics.leftWing);
      LALFree(harmonics.rightWing);
      
    }  /* done with reading line noise info */


  /* check that the band is not covered by lines */
  {
    INT4  counter=0, j, flag;
    REAL8 sampFreq, stepFreq, dopplerWing;

    dopplerWing = (uvar_f0 + uvar_fSearchBand)*VTOT;
    stepFreq = (uvar_fSearchBand + 2*dopplerWing)/100.0;

    for (j=0; j<100; j++)
      {
	sampFreq = uvar_f0 - dopplerWing + j*stepFreq;
	LAL_CALL( LALCheckLines (&status, &flag, &lines, sampFreq), &status); 
	if ( flag>0 ) counter++;
      }
    /* exit if more than 90% is covered by lines */
    if (counter > 90 )
      {
	fprintf(stdout, "Too many lines in this band...nothing to do! \n");
	/* deallocate memory and exit */
	if (nLines > 0)
	  {
	    LALFree(lines.lineFreq);
	    LALFree(lines.leftWing);
	    LALFree(lines.rightWing);
	  }
	exit(0);
      }
  }
  

  /* set fullsky flag */
  injectPar.fullSky = 1;
  if (uvar_AllSkyFlag == 0)
    injectPar.fullSky= 0;  /* patch case */

  LAL_CALL( LALRngMedBias( &status, &normalizeThr, uvar_blocksRngMed ), &status ); 
  
  msp = 1; /*only one spin-down */



  /* computing h0 values and preparing  output files */
  h0V.length=uvar_nh0;
  h0V.data = NULL;
  h0V.data = (REAL8 *)LALMalloc(uvar_nh0*sizeof(REAL8));
  h0V.data[0] = uvar_h0Min;
  
  if(uvar_nh0 >1){
    INT4 k;
    REAL8 steph0;   
    steph0 = (uvar_h0Max-uvar_h0Min)/(uvar_nh0-1.);
    for(k=1; k<uvar_nh0; ++k) h0V.data[k]= h0V.data[k-1]+steph0;
  }
  
   /*fp = LALMalloc(nh0*sizeof(FILE *)); */
  {
    INT4 k;
    CHAR filename[MAXFILENAMELENGTH];
    
    /* the paramerter file */
    strcpy( filename, uvar_fnameout);
    strcat( filename, "_par");
    fpPar= fopen(filename, "w"); /* where to write the parameters */
    /*setlinebuf(fpPar);*/  /* line buffered on */
    setvbuf(fpPar, (char *)NULL, _IOLBF, 0);
    
    /* the  file  with the h0 values */
    strcpy( filename, uvar_fnameout);
    strcat( filename, "_h0");
    fpH0= fopen(filename, "w"); /* where to write the parameters */
    /*setlinebuf(fpH0); */ /* line buffered on */
    setvbuf(fpH0, (char *)NULL, _IOLBF, 0); 
   
    /* the  file  with the the number-counts for different h0 values */
    strcpy( filename, uvar_fnameout);
    strcat( filename, "_nc");
    fpNc= fopen(filename, "w"); /* where to write the parameters */
    /*setlinebuf(fpNc);*/  /* line buffered on */
    setvbuf(fpNc, (char *)NULL, _IOLBF, 0);

    for (k=0; k<uvar_nh0; ++k){ fprintf(fpH0, "%g \n",  h0V.data[k] ); }  
    fclose(fpH0);
    
    /*
     *     for (k=0; k<uvar_nh0; ++k){
     *       sprintf(filename, "%s_%03d.m",uvar_fnameout, k); 
     *       fp[k] = fopen(filename, "w");
     *       setlinebuf(fp[k]);  
     *       fprintf(fp[k], " h0 = %g; \n",  h0V.data[k] );
     *       fprintf(fp[k], " Ncount= [ \n" );
     *     } 
     */
    
  }


  /* sft reading */
  {
    /* new SFT I/O data types */
    SFTCatalog *catalog = NULL;
    static SFTConstraints constraints;

    CHAR *tempDir;
    REAL8 doppWings, f_min, f_max;
    INT4 length;

    /* set detector constraint */
    constraints.detector = NULL;
    if ( LALUserVarWasSet( &uvar_ifo ) )    
      constraints.detector = XLALGetChannelPrefix ( uvar_ifo );

    /* get sft catalog */
    tempDir = (CHAR *)LALCalloc( MAXFILENAMELENGTH , sizeof(CHAR));
    strcpy(tempDir, uvar_sftDir);
    strcat(tempDir, "/*SFT*.*");
    LAL_CALL( LALSFTdataFind( &status, &catalog, tempDir, &constraints), &status);


    /* get some sft parameters */
    mObsCoh = catalog->length; /* number of sfts */
    deltaF = catalog->data->header.deltaF;  /* frequency resolution */
    timeBase= 1.0/deltaF; /* coherent integration time */
    f0Bin = floor( uvar_f0 * timeBase + 0.5); /* initial search frequency */
    length =  uvar_fSearchBand * timeBase; /* total number of search bins - 1 */
    fLastBin = f0Bin + length;   /* final frequency bin to be analyzed */
    fWings =  floor( fLastBin * VTOT + 0.5) + nfSizeCylinder + uvar_blocksRngMed;

    /* some more sft parameetrs */
    sftlength = 1 + length + 2*fWings;
    sftFminBin= f0Bin - fWings;
    fHeterodyne = sftFminBin*deltaF;
    tSamplingRate = 2.0*deltaF*(sftlength -1.);

    /* get SFT timestamps */
    LAL_CALL( LALSFTtimestampsFromCatalog(  &status, &timeV, catalog ), &status);  	

    /* add wings for Doppler modulation and running median block size*/
    doppWings = (uvar_f0 + uvar_fSearchBand) * VTOT;    
    f_min = uvar_f0 - doppWings - (uvar_blocksRngMed + nfSizeCylinder) * deltaF;
    f_max = uvar_f0 + uvar_fSearchBand + doppWings + (uvar_blocksRngMed + nfSizeCylinder) * deltaF;

    /* read sfts */
    /* read sft files making sure to add extra bins for running median */
    LAL_CALL( LALLoadSFTs ( &status, &inputSFTs, catalog, f_min, f_max), &status);


    /* calculation of weights comes here */

    /* normalize sfts */
    LAL_CALL( LALNormalizeSFTVect (&status, inputSFTs, uvar_blocksRngMed), &status);

        
    /* free memory */
    if ( LALUserVarWasSet( &uvar_ifo ) )    
      LALFree( constraints.detector );
    LALFree( tempDir);
    LAL_CALL( LALDestroySFTCatalog( &status, &catalog ), &status);  	 

  } 
   


  /* compute the time difference relative to startTime for all SFT */
  timeDiffV.length = mObsCoh;
  timeDiffV.data = NULL; 
  timeDiffV.data = (REAL8 *)LALCalloc(mObsCoh, sizeof(REAL8));
  
  {
    INT4   j; 
    
    for(j=0; j < mObsCoh; ++j)
      timeDiffV.data[j] = XLALGPSDiff( timeV->data + j, timeV->data ) + 0.5*timeBase;
  }



  /******************************************************************/ 
  /*   setting of ephemeris info */ 
  /******************************************************************/ 
  edat = (EphemerisData *)LALMalloc(sizeof(EphemerisData));
  (*edat).ephiles.earthEphemeris = uvar_earthEphemeris;
  (*edat).ephiles.sunEphemeris = uvar_sunEphemeris;
  
  /******************************************************************/
  /* compute detector velocity for those time stamps  
     if it is too slow , we can read it from the file genrated from the driver*/
  /******************************************************************/
  velV.length = mObsCoh;
  velV.data = NULL;
  velV.data = (REAL8Cart3Coor *)LALMalloc(mObsCoh*sizeof(REAL8Cart3Coor));
  
  {  
    VelocityPar   velPar;
    REAL8     vel[3]; 
    UINT4     j; 

    velPar.detector = detector;
    velPar.tBase = timeBase;
    velPar.vTol = ACCURACY;
    velPar.edat = NULL;

    /* read in ephemeris data */
    LAL_CALL( LALInitBarycenter( &status, edat), &status);
    velPar.edat = edat;
    
    for(j=0; j< velV.length; ++j){
      velPar.startTime.gpsSeconds     = timeV->data[j].gpsSeconds;
      velPar.startTime.gpsNanoSeconds = timeV->data[j].gpsNanoSeconds;
      
      LAL_CALL( LALAvgDetectorVel ( &status, vel, &velPar), &status );
      velV.data[j].x= vel[0];
      velV.data[j].y= vel[1];
      velV.data[j].z= vel[2];   
    }  
  }
 
  /******************************************************************/ 
  /*   setting of parameters */ 
  /******************************************************************/ 
  injectPar.h0   = uvar_h0Min;
  injectPar.fmin = uvar_f0;
  injectPar.fSearchBand = uvar_fSearchBand;
  injectPar.deltaF = deltaF;
  injectPar.alpha = uvar_alpha;  /* patch center if not full sky */
  injectPar.delta = uvar_delta;
  injectPar.patchSizeAlpha = uvar_patchSizeAlpha; /* patch size if not full sky */
  injectPar.patchSizeDelta = uvar_patchSizeDelta; 
  injectPar.pixelFactor = PIXELFACTOR;
  injectPar.vTotC = VTOT;
  injectPar.timeObs =timeDiffV.data[mObsCoh-1] + 0.5 * timeBase;
  
  injectPar.spnFmax.data = NULL; 
  injectPar.spnFmax.length=msp;   /*only 1 spin */
  injectPar.spnFmax.data = (REAL8 *)LALMalloc(msp*sizeof(REAL8));
  injectPar.spnFmax.data[0] = -(nfSizeCylinder/2) *deltaF/timeDiffV.data[mObsCoh-1];
  
  pulsarInject.spindown.length = msp;
  pulsarTemplate.spindown.length = msp;
  
  pulsarInject.spindown.data = NULL;
  pulsarTemplate.spindown.data = NULL;
  
  pulsarInject.spindown.data = (REAL8 *)LALMalloc(msp*sizeof(REAL8));
  pulsarTemplate.spindown.data = (REAL8 *)LALMalloc(msp*sizeof(REAL8));
 
  /******************************************************************/  
  sftParams.Tsft = timeBase;
  sftParams.timestamps = timeV;
  sftParams.noiseSFTs = NULL;       /* or =inputSFTs; */
  
  /* ****************************************************************/
  params.orbit.asini = 0 /* isolated pulsar */;
  /* params.transferFunction = NULL; */
  params.site = &(detector);
  params.ephemerides = edat;
  params.startTimeGPS.gpsSeconds = timeV->data[0].gpsSeconds;   /* start time of output time series */
  params.startTimeGPS.gpsNanoSeconds = timeV->data[0].gpsNanoSeconds;   /* start time of output time series */
  params.duration = injectPar.timeObs; /* length of time series in seconds */
  params.samplingRate = tSamplingRate;
  params.fHeterodyne = fHeterodyne;
  
  params.pulsar.refTime.gpsSeconds = timeV->data[0].gpsSeconds; 
  params.pulsar.refTime.gpsNanoSeconds = timeV->data[0].gpsNanoSeconds; 
  /* ****************************************************************/
  
  /* WE SHOULD LOOP OVER MC SIGNAL INJECTION HERE
     BEFORE THAT :
         -initialize ephemeris data
	 -create the set of h0 to be studied
	 -for each different h0 value create a file containing the h0
	 value
     LOOP over xxx Monte-Carlo signal Injections:
		- Generate signal injections parameters (using uvar_h0Min values) and
		random numbers....and also generate the corresponding template
		parameters (position , frequency spin-down) allowing some
		mismatch
		- Compute the frequency path for the template parameters
		-Generate the time series for injected signals and the
		corresponding SFTs with no added noise (for all times).
		 (free memory)
		
		LOOP over the different h0 values:
		   number count  = 0
		   compute normalization factor (for different h0)
		- for j=0; j<mObsCoh:
		     -Add SFT with the signal normalized to the SFT original noise
		     -COMPLEX8SFT2Periodogram1
		     -Periodo2PSDrng
		     -SelectPeackColorNoise
		     - check corresponding index
		     - increase or not numbercount
		 print final number count to the correesponding file
		 END LOOP for h0
	END LOOP for MC
 		(free memory)   */
		
  /* ****************************************************************/
  
  /* sft1  stores the sum of one  signal + noise SFT */
  sft1.length = sftlength;
  sft1.fminBinIndex = sftFminBin;
  sft1.epoch.gpsSeconds = timeV->data[0].gpsSeconds;
  sft1.epoch.gpsNanoSeconds = timeV->data[0].gpsNanoSeconds;
  sft1.timeBase = timeBase;
  sft1.data = NULL;
  sft1.data = (COMPLEX8 *)LALMalloc(sftlength* sizeof(COMPLEX8));
  
  periPSD.periodogram.length = sftlength;
  periPSD.periodogram.data = NULL;
  periPSD.periodogram.data = (REAL8 *)LALMalloc(sftlength* sizeof(REAL8));
  periPSD.psd.length = sftlength;
  periPSD.psd.data = NULL;
  periPSD.psd.data = (REAL8 *)LALMalloc(sftlength* sizeof(REAL8));
  
  threshold = uvar_peakThreshold/normalizeThr; 
  
  pg1.length = sftlength;
  pg1.data = NULL;
  pg1.data = (UCHAR *)LALMalloc(sftlength* sizeof(UCHAR));
  
  foft.length = mObsCoh;
  foft.data = NULL;
  foft.data = (REAL8 *)LALMalloc(mObsCoh*sizeof(REAL8));
  {
    INT4 j;
    for (j=0;j<nTemplates;++j) {
      foftV[j].length = mObsCoh;
      foftV[j].data = NULL;
      foftV[j].data = (REAL8 *)LALMalloc(mObsCoh*sizeof(REAL8));
    }
  }


  /* ****************************************************************/
  /*  HERE SHOULD START THE MONTE-CARLO */
    
  for(MCloopId=0; MCloopId < uvar_nMCloop; ++MCloopId){

    controlN=uvar_nh0; /* checks if near template corresponds to max number count*/
    controlNN=0;/* checks if near template corresponds to max number count*/
    controlNH=1;  /* checks if near template corresponds to max 
    		number count for the highest h0 value */
    
    LAL_CALL( GenerateInjectParams(&status, &pulsarInject, &pulsarTemplate,
			&closeTemplates, &injectPar, &lines), &status );
    
    /*  params.pulsar.TRefSSB=  ? ; */
    params.pulsar.position.longitude = pulsarInject.longitude;
    params.pulsar.position.latitude =  pulsarInject.latitude ;
    params.pulsar.position.system= COORDINATESYSTEM_EQUATORIAL; 
    params.pulsar.psi=    pulsarInject.psi;
    params.pulsar.aPlus=  pulsarInject.aPlus;
    params.pulsar.aCross= pulsarInject.aCross;
    params.pulsar.phi0=   pulsarInject.phi0;
    params.pulsar.f0=     pulsarInject.f0;
    params.pulsar.spindown=  &pulsarInject.spindown ;
    
    LAL_CALL( LALGeneratePulsarSignal(&status, &signalTseries, &params ), &status);
    LAL_CALL( LALSignalToSFTs(&status, &outputSFTs, signalTseries, &sftParams), 
	 &status);
	   
   /* ****************************************************************/
    /* writing the parameters into fpPar, following the format
       MCloopId  I.f0 H.f0 I.f1 H.f1 I.alpha H.alpha I.delta H.delta I.phi0  I.psi
       (not cos iota)  */
    /* ****************************************************************/   
    fprintf(fpPar," %d %f %f %g %g %f %f %f %f %f %f ", 
	    MCloopId, pulsarInject.f0, pulsarTemplate.f0,
	    pulsarInject.spindown.data[0], pulsarTemplate.spindown.data[0],
	    pulsarInject.longitude, pulsarTemplate.longitude,
	    pulsarInject.latitude, pulsarTemplate.latitude,
	    pulsarInject.phi0, pulsarInject.psi
	    );
    /* ****************************************************************/
    /* Computing the frequency path f(t) = f0(t)* (1+v/c.n)  for */
    /*  all the different templates */
    /* ****************************************************************/   

    /* the geometrically nearest template */
    LAL_CALL( ComputeFoft(&status, &foft,&pulsarTemplate,&timeDiffV,&velV, timeBase), &status);
    
    /* for all the 16 near templates */
    {
      UINT4 j,i,k, itemplate;
      
      itemplate =0;
      for(j=0;j<2;++j){
        pulsarTemplate.f0 = closeTemplates.f0[j];
        for(i=0;i<2;++i){
          pulsarTemplate.spindown.data[0] = closeTemplates.f1[i];
	  for(k=0;k<4;++k){
	    pulsarTemplate.latitude = closeTemplates.skytemp[k].delta;
	    pulsarTemplate.longitude = closeTemplates.skytemp[k].alpha;
            LAL_CALL( ComputeFoft(&status, &(foftV[itemplate]),
	                   &pulsarTemplate,&timeDiffV,&velV, timeBase), &status);
            ++itemplate;
	  }
	}
      }
    }
     
    /*  HERE THE LOOP FOR DIFFERENT H0 VALUES */
    
    fprintf(fpNc, " %d ",  MCloopId);
    
    for(h0loop=0; h0loop <uvar_nh0; ++h0loop){
      
      INT4  j, i, ind, itemplate; 
      COMPLEX8 *noise1SFT;
      COMPLEX8 *signal1SFT;
      COMPLEX8 *sumSFT;
      
      controlNN=0; 
      
      numberCount=0;
      for(itemplate=0; itemplate<nTemplates; ++itemplate){
        numberCountV[itemplate]=0;
      }
      h0scale =h0V.data[h0loop]/h0V.data[0]; /* different for different h0 values */
      
      /* ****************************************************************/
      /* adding signal+ noise SFT,  generating peakgrams  and producing the number-count*/
      /* ****************************************************************/      
      for (j=0; j < mObsCoh; j++)  {
	sumSFT = sft1.data;
	signal1SFT = outputSFTs->data[j].data->data;
	noise1SFT  =  inputSFTs->data[j].data->data;
	
	for (i=0; (UINT4)i < sftlength; i++)  {
	  /* sumSFT->re = noise1SFT->re + h0scale *signal1SFT->re; */
	  /* sumSFT->im = noise1SFT->im + h0scale *signal1SFT->im; */
	  *(sumSFT) = crectf( crealf(*noise1SFT) + h0scale *crealf(*signal1SFT), cimagf(*noise1SFT) + h0scale *cimagf(*signal1SFT) );
	  ++noise1SFT;
	  ++signal1SFT;
	  ++sumSFT;
	}
	
	LAL_CALL( COMPLEX8SFT2Periodogram1(&status, &periPSD.periodogram, &sft1), &status );	
	/* for color noise */    
	LAL_CALL( LALPeriodo2PSDrng( &status, 
			     &periPSD.psd, &periPSD.periodogram, &uvar_blocksRngMed), &status );	
	/* LAL_CALL( Periodo2PSDrng( &status, 
                     &periPSD.psd, &periPSD.periodogram, &uvar_blocksRngMed),  &status ); */	
	LAL_CALL( LALSelectPeakColorNoise(&status,&pg1,&threshold,&periPSD), &status); 	

	ind = floor( foft.data[j]*timeBase -sftFminBin+0.5); 
	numberCount+=pg1.data[ind]; /* adds 0 or 1 to the counter*/

        for (itemplate=0; itemplate<nTemplates; ++itemplate) {
	  ind = floor( foftV[itemplate].data[j]*timeBase -sftFminBin+0.5); 
          numberCountV[itemplate]+=pg1.data[ind];
        }
      }
      
      /*check the max number count */
      maxNumberCount = numberCount;
      for (itemplate=0; itemplate<nTemplates; ++itemplate) {
         if( numberCountV[itemplate] > maxNumberCount ) {
	   maxNumberCount = numberCountV[itemplate];
	   controlNN=1;
	   if (h0loop == (uvar_nh0-1)) controlNH=0;
	 }
      }
      controlN-=controlNN; /* substracts 1 every the near template was not the
      best*/
      /******************************************************************/
      /* printing result in the proper file */
      /******************************************************************/
      fprintf(fpNc, " %d ", maxNumberCount);
     
    } /* closing loop for different h0 values */
    fprintf(fpNc, " \n");
    
    

    /* ****************************************************************/
    /* writing the parameters into fpPar, following the format
       MCloopId  I.f0 H.f0 I.f1 H.f1 I.alpha H.alpha I.delta H.delta I.phi0  I.psi
       (not cos iota) and now adding the 2 control */
    /* ****************************************************************/   
    fprintf(fpPar,"  %d %d \n",  controlN, controlNH );
    
    LALFree(signalTseries->data->data);
    LALFree(signalTseries->data);
    LALFree(signalTseries);
    signalTseries =NULL;
    LAL_CALL(LALDestroySFTVector(&status, &outputSFTs),&status );
    outputSFTs = NULL;
    
  } /* Closing MC loop */
  
  /******************************************************************/
  /* Closing files */
  /******************************************************************/  
  fclose(fpPar); 
  fclose(fpNc); 

  
  /******************************************************************/
  /* Free memory and exit */
  /******************************************************************/
  
  /* LALFree(fp); */
  LALFree(sft1.data);
  LALFree(periPSD.periodogram.data);
  LALFree(periPSD.psd.data);
  LALFree(pg1.data);

  LAL_CALL(LALDestroyTimestampVector ( &status, &timeV), &status); 
  
  LALFree(timeDiffV.data);
  LALFree(velV.data);
  LALFree(foft.data);
  LALFree(h0V.data);
  {
     INT4 j;
     for (j=0;j<nTemplates;++j) {
        LALFree(foftV[j].data);
     }
   }

  
  LALFree(injectPar.spnFmax.data);
  LALFree(pulsarInject.spindown.data);
  LALFree(pulsarTemplate.spindown.data);
   
  LALFree(edat->ephemE);
  LALFree(edat->ephemS);
  LALFree(edat);
  
  LAL_CALL(LALDestroySFTVector(&status, &inputSFTs),&status );

  if (nLines > 0)
    {
      LALFree(lines.lineFreq);
      LALFree(lines.leftWing);
      LALFree(lines.rightWing);
    }

  LAL_CALL (LALDestroyUserVars(&status), &status);  
  LALCheckMemoryLeaks();
  
  
#ifdef TIMING
  stop = realcc();
  printf(" All: %llu\n", stop-start);
#endif
  
  INFO( DRIVEHOUGHCOLOR_MSGENORM );
  return DRIVEHOUGHCOLOR_ENORM;
}


 
/***************************************************************************/
void GenerateInjectParams(LALStatus   *status,
                        PulsarData           *injectPulsar,
                        HoughTemplate        *templatePulsar,
			HoughNearTemplates   *closeTemplates,
                        HoughInjectParams    *params,
			LineNoiseInfo        *lines  ){
			
  INT4          seed=0; /* seed generated using current time */
  REAL4         randval;
  RandomParams  *randPar=NULL;
  FILE     *fpRandom;
  INT4     count;
  
  REAL4    cosiota, h0;
  REAL8    f0, deltaF, deltaX;
  REAL8    latitude, longitude;  /* of the source in radians */
  INT8    f0bin;
  UINT4    msp;
  
  /* --------------------------------------------- */
  INITSTATUS(status);
  ATTATCHSTATUSPTR (status);
  
  /*   Make sure the arguments are not NULL: */
  ASSERT (injectPulsar,  status, DRIVEHOUGHCOLOR_ENULL, DRIVEHOUGHCOLOR_MSGENULL);
  ASSERT (templatePulsar, status, DRIVEHOUGHCOLOR_ENULL, DRIVEHOUGHCOLOR_MSGENULL);
  ASSERT (params, status, DRIVEHOUGHCOLOR_ENULL, DRIVEHOUGHCOLOR_MSGENULL);
  ASSERT (lines, status, DRIVEHOUGHCOLOR_ENULL, DRIVEHOUGHCOLOR_MSGENULL);
  
  /*  ++++++++++++++++++from makefakedata
   * Modified so as to not create random number parameters with seed
   * drawn from clock.  Seconds don't change fast enough and sft's
   * look alike.  We open /dev/urandom and read a 4 byte integer from
   * it and use that as our seed.  Note: /dev/random is slow after the
   * first, few accesses.
   */

  fpRandom = fopen("/dev/urandom","r");
  ASSERT (fpRandom, status, DRIVEHOUGHCOLOR_EFILE,  DRIVEHOUGHCOLOR_MSGEFILE); 
  
  count = fread(&seed, sizeof(INT4),1, fpRandom);
  if ( count == 0 ) ABORT ( status, DRIVEHOUGHCOLOR_EARG,  DRIVEHOUGHCOLOR_MSGEARG);
  
  fclose(fpRandom);
  
  TRY( LALCreateRandomParams(status->statusPtr, &randPar, seed), status);
  
 /*
  *   to create a single random deviate distributed uniforly between zero and unity		     
  *   TRY( LALUniformDeviate(status->statusPtr, &randval, randPar), status);
  */
  
  
  /* get random value phi0 [0, 2 pi] */ 
  TRY( LALUniformDeviate(status->statusPtr, &randval, randPar), status);
  injectPulsar->phi0 = randval * LAL_TWOPI;
  
  /* get random value cos iota [-1,1] */ 
  TRY( LALUniformDeviate(status->statusPtr, &randval, randPar), status);
  cosiota = 2.0* randval -1.0;
  
  h0=params->h0;
  injectPulsar->aCross = h0*cosiota;
  injectPulsar->aPlus  = 0.5*h0*(1.0 + cosiota*cosiota);
  
  /* get random value psi [0, 2 pi] */ 
  TRY( LALUniformDeviate(status->statusPtr, &randval, randPar), status);
  injectPulsar->psi = randval * LAL_TWOPI;

  /* getting random number for the frequency (and mismatch)*/
  TRY( LALUniformDeviate(status->statusPtr, &randval, randPar), status);
  f0 = params->fmin + (params->fSearchBand) * randval;
  
  /* veto the frequency if it is affected by a line */
  {
    INT4 veto=1;
    while( veto > 0 ){

      TRY( LALCheckLines (status->statusPtr, &veto, lines, f0 ), status); 
      if ( veto > 0 )
	{
	  TRY( LALUniformDeviate(status->statusPtr, &randval, randPar), status);
	  f0 = params->fmin + (params->fSearchBand) * randval;
	}
    } /* end of while loop */
  }
   
  injectPulsar->f0 = f0;
  deltaF = params->deltaF;
  f0bin  = floor(f0/deltaF +0.5);
  templatePulsar->f0 = f0bin*deltaF;
  closeTemplates->f0[0] = floor(f0/deltaF)*deltaF;
  closeTemplates->f0[1] = ceil(f0/deltaF)*deltaF;
 
  /* sky location, depending if  full sky or small patch is analyzed */
  deltaX = deltaF/(params->vTotC * params->pixelFactor *
 	           (params->fmin + params->fSearchBand) );
  
  
  if (params->fullSky){ /*full sky*/   
    REAL8 kkcos;
    
    TRY( LALUniformDeviate(status->statusPtr, &randval, randPar), status);
    longitude = randval * LAL_TWOPI;
    TRY( LALUniformDeviate(status->statusPtr, &randval, randPar), status);
    kkcos = 2.0* randval -1.0;
    latitude = acos(kkcos) -LAL_PI_2;
  }
  else {  /*small patch */  
    TRY( LALUniformDeviate(status->statusPtr, &randval, randPar), status);
    longitude = params->alpha + (params->patchSizeAlpha) *(randval-0.5);
    TRY( LALUniformDeviate(status->statusPtr, &randval, randPar), status);
    latitude = params->delta + (params->patchSizeDelta) *(randval-0.5);    
  }
  
  injectPulsar->longitude = longitude;
  injectPulsar->latitude  = latitude;   
  
  {
    REAL8UnitPolarCoor    template, par; 
    REAL8UnitPolarCoor    templRotated;
    REAL8Cart2Coor        templProjected;
    REAL8      dX1[2], dX2[2];
    INT4      ii,jj,kk;
    
    par.alpha = injectPulsar->longitude;
    par.delta = injectPulsar->latitude; 

    /* mismatch with the template in stereographic plane */
    TRY( LALUniformDeviate(status->statusPtr, &randval, randPar), status);
    templProjected.x = dX1[0] = deltaX*(randval-0.5);
    TRY( LALUniformDeviate(status->statusPtr, &randval, randPar), status);
    templProjected.y = dX2[0] = deltaX*(randval-0.5);

    if (dX1[0]<0.0) { 
      dX1[1]= dX1[0]+deltaX;
    } else {
      dX1[1]= dX1[0]-deltaX;
    }
    
    if (dX2[0]<0.0) { 
      dX2[1]= dX2[0]+deltaX;
    } else {
      dX2[1]= dX2[0]-deltaX;
    }
    
    /* invert the stereographic projection for a point on the projected plane */
    TRY( LALStereoInvProjectCart( status->statusPtr,
                                &templRotated, &templProjected ), status );
    /* inverse rotate the mismatch from the south pole to desired location */
    TRY( LALInvRotatePolarU( status->statusPtr, &template, &templRotated, &par), status);
    templatePulsar->longitude = template.alpha; 
    templatePulsar->latitude = template.delta; 
     
    kk=0;
    for (ii=0; ii<2; ii++){
      for (jj=0; jj<2; jj++) {
      templProjected.x = dX1[ii];
      templProjected.y = dX2[jj];
      TRY( LALStereoInvProjectCart( status->statusPtr,
                                &templRotated, &templProjected ), status );
      TRY( LALInvRotatePolarU( status->statusPtr, &(closeTemplates->skytemp[kk]), &templRotated, 
                               &par), status);
      ++kk;
      }
    }
    
  }

  /* now the spindown if any */
  msp = params->spnFmax.length ;
  closeTemplates->f1[0] = 0.0;
  closeTemplates->f1[1] = 0.0;

  ASSERT (templatePulsar->spindown.length == msp, status, DRIVEHOUGHCOLOR_EBAD,
	  DRIVEHOUGHCOLOR_MSGEBAD);
  ASSERT (injectPulsar->spindown.length == msp, status, DRIVEHOUGHCOLOR_EBAD,
	  DRIVEHOUGHCOLOR_MSGEBAD);
  
  if(msp){ /*if there are spin-down values */
    REAL8 deltaFk, spink;
    REAL8 timeObsInv;
    UINT4   i;
    ASSERT (injectPulsar->spindown.data,  status, DRIVEHOUGHCOLOR_ENULL, 
	    DRIVEHOUGHCOLOR_MSGENULL);
    ASSERT (templatePulsar->spindown.data,  status, DRIVEHOUGHCOLOR_ENULL, 
	    DRIVEHOUGHCOLOR_MSGENULL);
    ASSERT (params->spnFmax.data,  status, DRIVEHOUGHCOLOR_ENULL, 
	    DRIVEHOUGHCOLOR_MSGENULL);
    
    /* delta f_k = k! deltaF/ [T_Obs}^k  spd grid resolution*/
    timeObsInv= 1.0/params->timeObs;
    deltaFk= deltaF*timeObsInv;
    
    /* first spin-down parameter, (only spin-down) */	    
    TRY( LALUniformDeviate(status->statusPtr, &randval, randPar), status);
    spink=params->spnFmax.data[0]* randval;
    
    injectPulsar->spindown.data[0]= spink;
    templatePulsar->spindown.data[0] = floor(spink/deltaFk +0.5)*deltaFk;
    
    closeTemplates->f1[0] = floor(spink/deltaFk)*deltaFk;
    closeTemplates->f1[1] = ceil( spink/deltaFk)*deltaFk;

    /* the rest of the spin orders */
    for (i=1; i< msp; ++i) {
      TRY( LALUniformDeviate(status->statusPtr, &randval, randPar), status);
      spink=params->spnFmax.data[i]* (2.0* randval-1.0);
      injectPulsar->spindown.data[i]= spink;   
      deltaFk= deltaFk*timeObsInv*(i+1.0);
      templatePulsar->spindown.data[i] = floor(spink/deltaFk +0.5)*deltaFk;
    }
  }
  /* free memory */
  TRY( LALDestroyRandomParams(status->statusPtr, &randPar), status);
  
  DETATCHSTATUSPTR (status);
  /* normal exit */
  RETURN (status);
}


/* ****************************************************************/
/* Computing the frequency path f(t) = f0(t)* (1+v/c.n)   */
/* ****************************************************************/   
/******************************************************************/
void ComputeFoft(LALStatus   *status,
		 REAL8Vector          *foft,
                 HoughTemplate        *pulsarTemplate,
		 REAL8Vector          *timeDiffV,
		 REAL8Cart3CoorVector *velV,
                 REAL8                 timeBase){
  
  INT4   mObsCoh;
  REAL8   f0new, vcProdn, timeDiffN;
  INT4    f0newBin;
  REAL8   sourceDelta, sourceAlpha, cosDelta;
  INT4    j,i, nspin, factorialN; 
  REAL8Cart3Coor  sourceLocation;
  
  /* --------------------------------------------- */
  INITSTATUS(status);
  ATTATCHSTATUSPTR (status);
  
  /*   Make sure the arguments are not NULL: */
  ASSERT (foft,  status, DRIVEHOUGHCOLOR_ENULL, DRIVEHOUGHCOLOR_MSGENULL);
  ASSERT (pulsarTemplate,  status, DRIVEHOUGHCOLOR_ENULL, DRIVEHOUGHCOLOR_MSGENULL);
  ASSERT (timeDiffV,  status, DRIVEHOUGHCOLOR_ENULL, DRIVEHOUGHCOLOR_MSGENULL);
  ASSERT (velV,  status, DRIVEHOUGHCOLOR_ENULL, DRIVEHOUGHCOLOR_MSGENULL);
  
  ASSERT (foft->data,  status, DRIVEHOUGHCOLOR_ENULL, DRIVEHOUGHCOLOR_MSGENULL);
  ASSERT (timeDiffV->data,  status, DRIVEHOUGHCOLOR_ENULL, DRIVEHOUGHCOLOR_MSGENULL);
  ASSERT (velV->data,  status, DRIVEHOUGHCOLOR_ENULL, DRIVEHOUGHCOLOR_MSGENULL);
  
  sourceDelta = pulsarTemplate->latitude;
  sourceAlpha = pulsarTemplate->longitude;
  cosDelta = cos(sourceDelta);
  
  sourceLocation.x = cosDelta* cos(sourceAlpha);
  sourceLocation.y = cosDelta* sin(sourceAlpha);
  sourceLocation.z = sin(sourceDelta);
    
  mObsCoh = foft->length;    
  nspin = pulsarTemplate->spindown.length;
  
  for (j=0; j<mObsCoh; ++j){  /* loop for all different time stamps */
    vcProdn = velV->data[j].x * sourceLocation.x
      + velV->data[j].y * sourceLocation.y
      + velV->data[j].z * sourceLocation.z;
    f0new = pulsarTemplate->f0;
    factorialN = 1;
    timeDiffN = timeDiffV->data[j];
    
    for (i=0; i<nspin;++i){ /* loop for spin-down values */
      factorialN *=(i+1);
      f0new += pulsarTemplate->spindown.data[i]* timeDiffN / factorialN;
      timeDiffN *= timeDiffN;
    }
    f0newBin = floor( f0new * timeBase + 0.5);
    foft->data[j] = f0newBin * (1.0 +vcProdn) / timeBase;
  }    
    
  DETATCHSTATUSPTR (status);
  /* normal exit */
  RETURN (status);
}			
