/*
 *  Copyright (C) 2012 Karl Wette
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with with program; see the file COPYING. If not, write to the
 *  Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 */

#ifndef _FACTORIAL_H
#define _FACTORIAL_H

#include <lal/LALAtomicDatatypes.h>

/**
 * \addtogroup Factorial_h
 * \author Karl Wette
 * \brief Static tables of the factorial and double-factorial functions, and derived quantities.
 *
 * ### Description ###
 *
 * The following tables of the factorial function are provided:
 * <ul>
 * <li>UINT8 LAL_FACT[\f$i\f$] = \f$i!\f$
 * <li>REAL8 LAL_FACT_INV[\f$i\f$] = \f$1/i!\f$
 * <li>REAL8 LAL_FACT_LOG[\f$i\f$] = \f$\log i!\f$
 * </ul>
 * The maximum allowable index \f$i\f$ of these tables is LAL_FACT_MAX.
 * This value is determines by the maximum value of \f$i!\f$ that can be stored in an UINT8.
 *
 * The following tables of the double-factorial function are provided:
 * <ul>
 * <li>UINT8 LAL_FACT2[\f$i\f$] = \f$i!!\f$
 * <li>REAL8 LAL_FACT2_INV[\f$i\f$] = \f$1/i!!\f$
 * <li>REAL8 LAL_FACT2_LOG[\f$i\f$] = \f$\log i!!\f$
 * </ul>
 * The maximum allowable index \f$i\f$ of these tables is LAL_FACT2_MAX.
 * This value is determines by the maximum value of \f$i!!\f$ that can be stored in an UINT8.
 *
 * All values were computed to 35 significant figures with Mathematica v8.0.1.0 on 64-bit Linux.
 */
/*@{*/

#ifndef SWIG /* exclude from SWIG wrappings */

#define LAL_FACT_MAX 20

static const UINT8 LAL_FACT[] = { \
  1, \
  1, \
  2, \
  6, \
  24, \
  120, \
  720, \
  5040, \
  40320, \
  362880, \
  3628800, \
  39916800, \
  479001600, \
  6227020800, \
  87178291200, \
  1307674368000, \
  20922789888000, \
  355687428096000, \
  6402373705728000, \
  121645100408832000, \
  2432902008176640000 \
};

static const REAL8 LAL_FACT_INV[] = { \
  1., \
  1., \
  0.5, \
  0.16666666666666666666666666666666667, \
  0.041666666666666666666666666666666667, \
  0.0083333333333333333333333333333333333, \
  0.0013888888888888888888888888888888889, \
  0.00019841269841269841269841269841269841, \
  0.000024801587301587301587301587301587302, \
  2.7557319223985890652557319223985891e-6, \
  2.7557319223985890652557319223985891e-7, \
  2.5052108385441718775052108385441719e-8, \
  2.0876756987868098979210090321201432e-9, \
  1.6059043836821614599392377170154948e-10, \
  1.1470745597729724713851697978682106e-11, \
  7.6471637318198164759011319857880704e-13, \
  4.779477332387385297438207491117544e-14, \
  2.81145725434552076319894558301032e-15, \
  1.5619206968586226462216364350057333e-16, \
  8.2206352466243297169559812368722807e-18, \
  4.1103176233121648584779906184361404e-19 \
};

static const REAL8 LAL_FACT_LOG[] = { \
  0., \
  0., \
  0.6931471805599453094172321214581766, \
  1.7917594692280550008124773583807023, \
  3.1780538303479456196469416012970554, \
  4.787491742782045994247700934523243, \
  6.5792512120101009950601782929039453, \
  8.5251613610654143001655310363471251, \
  10.6046029027452502284172274007216548, \
  12.8018274800814696112077178745667062, \
  15.1044125730755152952257093292510704, \
  17.5023078458738858392876529072161997, \
  19.9872144956618861495173623870550785, \
  22.5521638531234228855708498286203971, \
  25.1912211827386815000934346935217534, \
  27.8992713838408915660894392636704668, \
  30.671860106080672803758367749503173, \
  33.5050734501368888840079023673762996, \
  36.3954452080330535762156249626795275, \
  39.3398841871994940362246523945673811, \
  42.3356164607534850296598759707099219 \
};

#define LAL_FACT2_MAX 33

static const UINT8 LAL_FACT2[] = { \
  1, \
  1, \
  2, \
  3, \
  8, \
  15, \
  48, \
  105, \
  384, \
  945, \
  3840, \
  10395, \
  46080, \
  135135, \
  645120, \
  2027025, \
  10321920, \
  34459425, \
  185794560, \
  654729075, \
  3715891200, \
  13749310575, \
  81749606400, \
  316234143225, \
  1961990553600, \
  7905853580625, \
  51011754393600, \
  213458046676875, \
  1428329123020800, \
  6190283353629375, \
  42849873690624000, \
  191898783962510625, \
  1371195958099968000, \
  6332659870762850625 \
};

static const REAL8 LAL_FACT2_INV[] = { \
  1., \
  1., \
  0.5, \
  0.33333333333333333333333333333333333, \
  0.125, \
  0.066666666666666666666666666666666667, \
  0.020833333333333333333333333333333333, \
  0.0095238095238095238095238095238095238, \
  0.0026041666666666666666666666666666667, \
  0.0010582010582010582010582010582010582, \
  0.00026041666666666666666666666666666667, \
  0.0000962000962000962000962000962000962, \
  0.000021701388888888888888888888888888889, \
  7.4000074000074000074000074000074e-6, \
  1.5500992063492063492063492063492063e-6, \
  4.9333382666716000049333382666716e-7, \
  9.6881200396825396825396825396825397e-8, \
  2.9019636862774117676078460392185882e-8, \
  5.3822889109347442680776014109347443e-9, \
  1.5273493085670588250567610732729412e-9, \
  2.6911444554673721340388007054673721e-10, \
  7.2730919455574229764607670155854342e-11, \
  1.2232474797578964245630912297578964e-11, \
  3.1622138893727925984612030502545366e-12, \
  5.0968644989912351023462134573245684e-13, \
  1.2648855557491170393844812201018146e-13, \
  1.9603324996120135009023897912786802e-14, \
  4.6847613175893223680906711855622764e-15, \
  7.0011874986143339317942492545667149e-16, \
  1.6154349370997663338243693743318195e-16, \
  2.3337291662047779772647497515222383e-17, \
  5.2110804422573107542721592720381273e-18, \
  7.2929036443899311789523429735069947e-19, \
  1.5791152855325184103855028097085234e-19 \
};

static const REAL8 LAL_FACT2_LOG[] = { \
  0., \
  0., \
  0.6931471805599453094172321214581766, \
  1.0986122886681096913952452369225257, \
  2.0794415416798359282516963643745297, \
  2.7080502011022100659960045701487133, \
  3.871201010907890929064173722755232, \
  4.6539603501575233711013573135918931, \
  5.9506425525877268573158700871297617, \
  6.8511849274937427538918477874369445, \
  8.2532276455817725413338615418141259, \
  9.2490802002921132979537913654020738, \
  10.7381342953697728515635710216530047, \
  11.8140295577536500340072788069673924, \
  13.377191624985031466086155886554361, \
  14.5220797588558601000032833771161057, \
  16.1497803472248127037550843723870673, \
  17.3552931029120761802528179949892323, \
  19.0401521051209773959628069676902953, \
  20.2997320820785166402618454268770858, \
  22.0358843786749683893980305438328361, \
  23.3442545198019396367624434072427912, \
  25.1269268320332842428772062432561419, \
  26.4797487357310893275691962390529874, \
  28.3049806623812298625241478445531973, \
  29.6986245605992900767707149055053626, \
  31.5630772004027119079948674075766925, \
  32.9944614266036191509564506162729398, \
  34.8952817105779158319346843939362254, \
  36.3617572565900931781397226486348514, \
  38.2964790922400712073479210855431153, \
  39.7957444610752394240688869731772086, \
  41.7622149950397977544340816928339981, \
  43.2922520225417196595260757880648636 \
};

#endif /* SWIG */

/*@}*/

#endif /*_FACTORIAL_H*/
