/***********************************************************************
*                                                                      *
*               This software is part of the ast package               *
*          Copyright (c) 2003-2013 AT&T Intellectual Property          *
*                      and is licensed under the                       *
*                 Eclipse Public License, Version 1.0                  *
*                    by AT&T Intellectual Property                     *
*                                                                      *
*                A copy of the License is available at                 *
*          http://www.eclipse.org/org/documents/epl-v10.html           *
*         (with md5 checksum b35adb5213ca9657e911e9befb180842)         *
*                                                                      *
*              Information and Software Systems Research               *
*                            AT&T Research                             *
*                           Florham Park NJ                            *
*                                                                      *
*                     Phong Vo <phongvo@gmail.com>                     *
*                                                                      *
***********************************************************************/
#include	"vchdr.h"

/*	Map bytes from one to another (e.g., the rot13 program).
**	If no mapping is specified, the identity map is used.
**	Various mappings between different types of EBCDIC and ASCII
**	are provided.
**
**	Written by Kiem-Phong Vo
*/

#if _PACKAGE_ast

#include	<ccode.h>

#define A2E	(void*)CCOP(CC_ASCII,CC_EBCDIC)
#define E2A	(void*)CCOP(CC_EBCDIC,CC_ASCII)
#define A2I	(void*)CCOP(CC_ASCII,CC_EBCDIC_I)
#define I2A	(void*)CCOP(CC_EBCDIC_I,CC_ASCII)
#define A2O	(void*)CCOP(CC_ASCII,CC_EBCDIC_O)
#define O2A	(void*)CCOP(CC_EBCDIC_O,CC_ASCII)
#define A2S	(void*)CCOP(CC_ASCII,CC_EBCDIC_S)
#define S2A	(void*)CCOP(CC_EBCDIC_S,CC_ASCII)
#define A2H	(void*)CCOP(CC_ASCII,CC_EBCDIC_H)
#define H2A	(void*)CCOP(CC_EBCDIC_H,CC_ASCII)
#define A2M	(void*)CCOP(CC_ASCII,CC_EBCDIC_M)
#define M2A	(void*)CCOP(CC_EBCDIC_M,CC_ASCII)
#define A2U	(void*)CCOP(CC_ASCII,CC_EBCDIC_U)
#define U2A	(void*)CCOP(CC_EBCDIC_U,CC_ASCII)

#else

static Vcchar_t A2E[] =
{	/* CC_ASCII=>CC_EBCDIC_E */
	0000,0001,0002,0003,0067,0055,0056,0057, 0026,0005,0045,0013,0014,0015,0016,0017,
	0020,0021,0022,0023,0074,0075,0062,0046, 0030,0031,0077,0047,0034,0035,0036,0037,
	0100,0132,0177,0173,0133,0154,0120,0175, 0115,0135,0134,0116,0153,0140,0113,0141,
	0360,0361,0362,0363,0364,0365,0366,0367, 0370,0371,0172,0136,0114,0176,0156,0157,
	0174,0301,0302,0303,0304,0305,0306,0307, 0310,0311,0321,0322,0323,0324,0325,0326,
	0327,0330,0331,0342,0343,0344,0345,0346, 0347,0350,0351,0255,0340,0275,0232,0155,
	0171,0201,0202,0203,0204,0205,0206,0207, 0210,0211,0221,0222,0223,0224,0225,0226,
	0227,0230,0231,0242,0243,0244,0245,0246, 0247,0250,0251,0300,0117,0320,0137,0007,
	0040,0041,0042,0043,0044,0025,0006,0027, 0050,0051,0052,0053,0054,0011,0012,0033,
	0060,0061,0032,0063,0064,0065,0066,0010, 0070,0071,0072,0073,0004,0024,0076,0341,
	0101,0102,0103,0104,0105,0106,0107,0110, 0111,0121,0122,0123,0124,0125,0126,0127,
	0130,0131,0142,0143,0144,0145,0146,0147, 0150,0151,0160,0161,0162,0163,0164,0165,
	0166,0167,0170,0200,0212,0213,0214,0215, 0216,0217,0220,0152,0233,0234,0235,0236,
	0237,0240,0252,0253,0254,0112,0256,0257, 0260,0261,0262,0263,0264,0265,0266,0267,
	0270,0271,0272,0273,0274,0241,0276,0277, 0312,0313,0314,0315,0316,0317,0332,0333,
	0334,0335,0336,0337,0352,0353,0354,0355, 0356,0357,0372,0373,0374,0375,0376,0377
};

static Vcchar_t E2A[] =
{	/* CC_EBCDIC_E=>CC_ASCII */
	0000,0001,0002,0003,0234,0011,0206,0177, 0227,0215,0216,0013,0014,0015,0016,0017,
	0020,0021,0022,0023,0235,0205,0010,0207, 0030,0031,0222,0217,0034,0035,0036,0037,
	0200,0201,0202,0203,0204,0012,0027,0033, 0210,0211,0212,0213,0214,0005,0006,0007,
	0220,0221,0026,0223,0224,0225,0226,0004, 0230,0231,0232,0233,0024,0025,0236,0032,
	0040,0240,0241,0242,0243,0244,0245,0246, 0247,0250,0325,0056,0074,0050,0053,0174,
	0046,0251,0252,0253,0254,0255,0256,0257, 0260,0261,0041,0044,0052,0051,0073,0176,
	0055,0057,0262,0263,0264,0265,0266,0267, 0270,0271,0313,0054,0045,0137,0076,0077,
	0272,0273,0274,0275,0276,0277,0300,0301, 0302,0140,0072,0043,0100,0047,0075,0042,
	0303,0141,0142,0143,0144,0145,0146,0147, 0150,0151,0304,0305,0306,0307,0310,0311,
	0312,0152,0153,0154,0155,0156,0157,0160, 0161,0162,0136,0314,0315,0316,0317,0320,
	0321,0345,0163,0164,0165,0166,0167,0170, 0171,0172,0322,0323,0324,0133,0326,0327,
	0330,0331,0332,0333,0334,0335,0336,0337, 0340,0341,0342,0343,0344,0135,0346,0347,
	0173,0101,0102,0103,0104,0105,0106,0107, 0110,0111,0350,0351,0352,0353,0354,0355,
	0175,0112,0113,0114,0115,0116,0117,0120, 0121,0122,0356,0357,0360,0361,0362,0363,
	0134,0237,0123,0124,0125,0126,0127,0130, 0131,0132,0364,0365,0366,0367,0370,0371,
	0060,0061,0062,0063,0064,0065,0066,0067, 0070,0071,0372,0373,0374,0375,0376,0377
};

static Vcchar_t	A2I[] =
{	/* CC_ASCII=>CC_EBCDIC_I */
	0000,0001,0002,0003,0067,0055,0056,0057, 0026,0005,0045,0013,0014,0015,0016,0017,
	0020,0021,0022,0023,0074,0075,0062,0046, 0030,0031,0077,0047,0034,0035,0036,0037,
	0100,0132,0177,0173,0133,0154,0120,0175, 0115,0135,0134,0116,0153,0140,0113,0141,
	0360,0361,0362,0363,0364,0365,0366,0367, 0370,0371,0172,0136,0114,0176,0156,0157,
	0174,0301,0302,0303,0304,0305,0306,0307, 0310,0311,0321,0322,0323,0324,0325,0326,
	0327,0330,0331,0342,0343,0344,0345,0346, 0347,0350,0351,0255,0340,0275,0137,0155,
	0171,0201,0202,0203,0204,0205,0206,0207, 0210,0211,0221,0222,0223,0224,0225,0226,
	0227,0230,0231,0242,0243,0244,0245,0246, 0247,0250,0251,0300,0117,0320,0241,0007,
	0040,0041,0042,0043,0044,0025,0006,0027, 0050,0051,0052,0053,0054,0011,0012,0033,
	0060,0061,0032,0063,0064,0065,0066,0010, 0070,0071,0072,0073,0004,0024,0076,0341,
	0101,0102,0103,0104,0105,0106,0107,0110, 0111,0121,0122,0123,0124,0125,0126,0127,
	0130,0131,0142,0143,0144,0145,0146,0147, 0150,0151,0160,0161,0162,0163,0164,0165,
	0166,0167,0170,0200,0212,0213,0214,0215, 0216,0217,0220,0232,0233,0234,0235,0236,
	0237,0240,0252,0253,0254,0255,0256,0257, 0260,0261,0262,0263,0264,0265,0266,0267,
	0270,0271,0272,0273,0274,0275,0276,0277, 0312,0313,0314,0315,0316,0317,0332,0333,
	0334,0335,0336,0337,0352,0353,0354,0355, 0356,0357,0372,0373,0374,0375,0376,0377
};

static Vcchar_t I2A[] =
{	/* CC_EBCDIC_I=>CC_ASCII */
	0000,0001,0002,0003,0234,0011,0206,0177, 0227,0215,0216,0013,0014,0015,0016,0017,
	0020,0021,0022,0023,0235,0205,0010,0207, 0030,0031,0222,0217,0034,0035,0036,0037,
	0200,0201,0202,0203,0204,0012,0027,0033, 0210,0211,0212,0213,0214,0005,0006,0007,
	0220,0221,0026,0223,0224,0225,0226,0004, 0230,0231,0232,0233,0024,0025,0236,0032,
	0040,0240,0241,0242,0243,0244,0245,0246, 0247,0250,0112,0056,0074,0050,0053,0174,
	0046,0251,0252,0253,0254,0255,0256,0257, 0260,0261,0041,0044,0052,0051,0073,0136,
	0055,0057,0262,0263,0264,0265,0266,0267, 0270,0271,0152,0054,0045,0137,0076,0077,
	0272,0273,0274,0275,0276,0277,0300,0301, 0302,0140,0072,0043,0100,0047,0075,0042,
	0303,0141,0142,0143,0144,0145,0146,0147, 0150,0151,0304,0305,0306,0307,0310,0311,
	0312,0152,0153,0154,0155,0156,0157,0160, 0161,0162,0313,0314,0315,0316,0317,0320,
	0321,0176,0163,0164,0165,0166,0167,0170, 0171,0172,0322,0323,0324,0133,0326,0327,
	0330,0331,0332,0333,0334,0335,0336,0337, 0340,0341,0342,0343,0344,0135,0346,0347,
	0173,0101,0102,0103,0104,0105,0106,0107, 0110,0111,0350,0351,0352,0353,0354,0355,
	0175,0112,0113,0114,0115,0116,0117,0120, 0121,0122,0356,0357,0360,0361,0362,0363,
	0134,0237,0123,0124,0125,0126,0127,0130, 0131,0132,0364,0365,0366,0367,0370,0371,
	0060,0061,0062,0063,0064,0065,0066,0067, 0070,0071,0372,0373,0374,0375,0376,0377
};

static Vcchar_t A2O[] =
{	/* CC_ASCII=>CC_EBDCID_O */
	0000,0001,0002,0003,0067,0055,0056,0057, 0026,0005,0025,0013,0014,0015,0016,0017,
	0020,0021,0022,0023,0074,0075,0062,0046, 0030,0031,0077,0047,0034,0035,0036,0037,
	0100,0132,0177,0173,0133,0154,0120,0175, 0115,0135,0134,0116,0153,0140,0113,0141,
	0360,0361,0362,0363,0364,0365,0366,0367, 0370,0371,0172,0136,0114,0176,0156,0157,
	0174,0301,0302,0303,0304,0305,0306,0307, 0310,0311,0321,0322,0323,0324,0325,0326,
	0327,0330,0331,0342,0343,0344,0345,0346, 0347,0350,0351,0255,0340,0275,0137,0155,
	0171,0201,0202,0203,0204,0205,0206,0207, 0210,0211,0221,0222,0223,0224,0225,0226,
	0227,0230,0231,0242,0243,0244,0245,0246, 0247,0250,0251,0300,0117,0320,0241,0007,
	0040,0041,0042,0043,0044,0045,0006,0027, 0050,0051,0052,0053,0054,0011,0012,0033,
	0060,0061,0032,0063,0064,0065,0066,0010, 0070,0071,0072,0073,0004,0024,0076,0377,
	0101,0252,0112,0261,0237,0262,0152,0265, 0273,0264,0232,0212,0260,0312,0257,0274,
	0220,0217,0352,0372,0276,0240,0266,0263, 0235,0332,0233,0213,0267,0270,0271,0253,
	0144,0145,0142,0146,0143,0147,0236,0150, 0164,0161,0162,0163,0170,0165,0166,0167,
	0254,0151,0355,0356,0353,0357,0354,0277, 0200,0375,0376,0373,0374,0272,0256,0131,
	0104,0105,0102,0106,0103,0107,0234,0110, 0124,0121,0122,0123,0130,0125,0126,0127,
	0214,0111,0315,0316,0313,0317,0314,0341, 0160,0335,0336,0333,0334,0215,0216,0337
};

static Vcchar_t O2A[] =
{	/* CC_EBDCID_O=>CC_ASCII */
	0000,0001,0002,0003,0234,0011,0206,0177, 0227,0215,0216,0013,0014,0015,0016,0017,
	0020,0021,0022,0023,0235,0012,0010,0207, 0030,0031,0222,0217,0034,0035,0036,0037,
	0200,0201,0202,0203,0204,0205,0027,0033, 0210,0211,0212,0213,0214,0005,0006,0007,
	0220,0221,0026,0223,0224,0225,0226,0004, 0230,0231,0232,0233,0024,0025,0236,0032,
	0040,0240,0342,0344,0340,0341,0343,0345, 0347,0361,0242,0056,0074,0050,0053,0174,
	0046,0351,0352,0353,0350,0355,0356,0357, 0354,0337,0041,0044,0052,0051,0073,0136,
	0055,0057,0302,0304,0300,0301,0303,0305, 0307,0321,0246,0054,0045,0137,0076,0077,
	0370,0311,0312,0313,0310,0315,0316,0317, 0314,0140,0072,0043,0100,0047,0075,0042,
	0330,0141,0142,0143,0144,0145,0146,0147, 0150,0151,0253,0273,0360,0375,0376,0261,
	0260,0152,0153,0154,0155,0156,0157,0160, 0161,0162,0252,0272,0346,0270,0306,0244,
	0265,0176,0163,0164,0165,0166,0167,0170, 0171,0172,0241,0277,0320,0133,0336,0256,
	0254,0243,0245,0267,0251,0247,0266,0274, 0275,0276,0335,0250,0257,0135,0264,0327,
	0173,0101,0102,0103,0104,0105,0106,0107, 0110,0111,0255,0364,0366,0362,0363,0365,
	0175,0112,0113,0114,0115,0116,0117,0120, 0121,0122,0271,0373,0374,0371,0372,0377,
	0134,0367,0123,0124,0125,0126,0127,0130, 0131,0132,0262,0324,0326,0322,0323,0325,
	0060,0061,0062,0063,0064,0065,0066,0067, 0070,0071,0263,0333,0334,0331,0332,0237
};

static Vcchar_t A2S[] =
{	/* CC_ASCII=>CC_EBCDIC_S */
	0000,0001,0002,0003,0067,0055,0056,0057, 0026,0005,0025,0013,0014,0015,0016,0017,
	0020,0021,0022,0023,0074,0075,0062,0046, 0030,0031,0077,0047,0034,0035,0036,0037,
	0100,0132,0177,0173,0133,0154,0120,0175, 0115,0135,0134,0116,0153,0140,0113,0141,
	0360,0361,0362,0363,0364,0365,0366,0367, 0370,0371,0172,0136,0114,0176,0156,0157,
	0174,0301,0302,0303,0304,0305,0306,0307, 0310,0311,0321,0322,0323,0324,0325,0326,
	0327,0330,0331,0342,0343,0344,0345,0346, 0347,0350,0351,0273,0274,0275,0152,0155,
	0112,0201,0202,0203,0204,0205,0206,0207, 0210,0211,0221,0222,0223,0224,0225,0226,
	0227,0230,0231,0242,0243,0244,0245,0246, 0247,0250,0251,0373,0117,0375,0377,0007,
	0040,0041,0042,0043,0044,0045,0006,0027, 0050,0051,0052,0053,0054,0011,0012,0033,
	0060,0061,0032,0063,0064,0065,0066,0010, 0070,0071,0072,0073,0004,0024,0076,0137,
	0101,0252,0260,0261,0237,0262,0320,0265, 0171,0264,0232,0212,0272,0312,0257,0241,
	0220,0217,0352,0372,0276,0240,0266,0263, 0235,0332,0233,0213,0267,0270,0271,0253,
	0144,0145,0142,0146,0143,0147,0236,0150, 0164,0161,0162,0163,0170,0165,0166,0167,
	0254,0151,0355,0356,0353,0357,0354,0277, 0200,0340,0376,0335,0374,0255,0256,0131,
	0104,0105,0102,0106,0103,0107,0234,0110, 0124,0121,0122,0123,0130,0125,0126,0127,
	0214,0111,0315,0316,0313,0317,0314,0341, 0160,0300,0336,0333,0334,0215,0216,0337
};

static Vcchar_t	S2A[] =
{	/* CC_EBCDIC_S=>CC_ASCII */
	0000,0001,0002,0003,0234,0011,0206,0177, 0227,0215,0216,0013,0014,0015,0016,0017,
	0020,0021,0022,0023,0235,0012,0010,0207, 0030,0031,0222,0217,0034,0035,0036,0037,
	0200,0201,0202,0203,0204,0205,0027,0033, 0210,0211,0212,0213,0214,0005,0006,0007,
	0220,0221,0026,0223,0224,0225,0226,0004, 0230,0231,0232,0233,0024,0025,0236,0032,
	0040,0240,0342,0344,0340,0341,0343,0345, 0347,0361,0140,0056,0074,0050,0053,0174,
	0046,0351,0352,0353,0350,0355,0356,0357, 0354,0337,0041,0044,0052,0051,0073,0237,
	0055,0057,0302,0304,0300,0301,0303,0305, 0307,0321,0136,0054,0045,0137,0076,0077,
	0370,0311,0312,0313,0310,0315,0316,0317, 0314,0250,0072,0043,0100,0047,0075,0042,
	0330,0141,0142,0143,0144,0145,0146,0147, 0150,0151,0253,0273,0360,0375,0376,0261,
	0260,0152,0153,0154,0155,0156,0157,0160, 0161,0162,0252,0272,0346,0270,0306,0244,
	0265,0257,0163,0164,0165,0166,0167,0170, 0171,0172,0241,0277,0320,0335,0336,0256,
	0242,0243,0245,0267,0251,0247,0266,0274, 0275,0276,0254,0133,0134,0135,0264,0327,
	0371,0101,0102,0103,0104,0105,0106,0107, 0110,0111,0255,0364,0366,0362,0363,0365,
	0246,0112,0113,0114,0115,0116,0117,0120, 0121,0122,0271,0373,0374,0333,0372,0377,
	0331,0367,0123,0124,0125,0126,0127,0130, 0131,0132,0262,0324,0326,0322,0323,0325,
	0060,0061,0062,0063,0064,0065,0066,0067, 0070,0071,0263,0173,0334,0175,0332,0176
};

static Vcchar_t	A2H[] =
{	/* CC_ASCII=>CC_EBCDIC_H */
	0000,0001,0002,0003,0067,0055,0056,0057, 0026,0005,0045,0013,0014,0015,0016,0017,
	0020,0021,0022,0023,0074,0075,0062,0046, 0030,0031,0077,0047,0034,0035,0036,0037,
	0100,0132,0177,0173,0133,0154,0120,0175, 0115,0135,0134,0116,0153,0140,0113,0141,
	0360,0361,0362,0363,0364,0365,0366,0367, 0370,0371,0172,0136,0114,0176,0156,0157,
	0174,0301,0302,0303,0304,0305,0306,0307, 0310,0311,0321,0322,0323,0324,0325,0326,
	0327,0330,0331,0342,0343,0344,0345,0346, 0347,0350,0351,0272,0340,0273,0260,0155,
	0171,0201,0202,0203,0204,0205,0206,0207, 0210,0211,0221,0222,0223,0224,0225,0226,
	0227,0230,0231,0242,0243,0244,0245,0246, 0247,0250,0251,0300,0117,0320,0241,0007,
	0040,0041,0042,0043,0044,0025,0006,0027, 0050,0051,0052,0053,0054,0011,0012,0033,
	0060,0061,0032,0063,0064,0065,0066,0010, 0070,0071,0072,0073,0004,0024,0076,0377,
	0101,0252,0112,0261,0237,0262,0152,0265, 0275,0264,0232,0212,0137,0312,0257,0274,
	0220,0217,0352,0372,0276,0240,0266,0263, 0235,0332,0233,0213,0267,0270,0271,0253,
	0144,0145,0142,0146,0143,0147,0236,0150, 0164,0161,0162,0163,0170,0165,0166,0167,
	0254,0151,0355,0356,0353,0357,0354,0277, 0200,0375,0376,0373,0374,0255,0256,0131,
	0104,0105,0102,0106,0103,0107,0234,0110, 0124,0121,0122,0123,0130,0125,0126,0127,
	0214,0111,0315,0316,0313,0317,0314,0341, 0160,0335,0336,0333,0334,0215,0216,0337
};

static Vcchar_t	H2A[] =
{	/* CC_EBCDIC_H=>CC_ASCII */
	0000,0001,0002,0003,0234,0011,0206,0177, 0227,0215,0216,0013,0014,0015,0016,0017,
	0020,0021,0022,0023,0235,0205,0010,0207, 0030,0031,0222,0217,0034,0035,0036,0037,
	0200,0201,0202,0203,0204,0012,0027,0033, 0210,0211,0212,0213,0214,0005,0006,0007,
	0220,0221,0026,0223,0224,0225,0226,0004, 0230,0231,0232,0233,0024,0025,0236,0032,
	0040,0240,0342,0344,0340,0341,0343,0345, 0347,0361,0242,0056,0074,0050,0053,0174,
	0046,0351,0352,0353,0350,0355,0356,0357, 0354,0337,0041,0044,0052,0051,0073,0254,
	0055,0057,0302,0304,0300,0301,0303,0305, 0307,0321,0246,0054,0045,0137,0076,0077,
	0370,0311,0312,0313,0310,0315,0316,0317, 0314,0140,0072,0043,0100,0047,0075,0042,
	0330,0141,0142,0143,0144,0145,0146,0147, 0150,0151,0253,0273,0360,0375,0376,0261,
	0260,0152,0153,0154,0155,0156,0157,0160, 0161,0162,0252,0272,0346,0270,0306,0244,
	0265,0176,0163,0164,0165,0166,0167,0170, 0171,0172,0241,0277,0320,0335,0336,0256,
	0136,0243,0245,0267,0251,0247,0266,0274, 0275,0276,0133,0135,0257,0250,0264,0327,
	0173,0101,0102,0103,0104,0105,0106,0107, 0110,0111,0255,0364,0366,0362,0363,0365,
	0175,0112,0113,0114,0115,0116,0117,0120, 0121,0122,0271,0373,0374,0371,0372,0377,
	0134,0367,0123,0124,0125,0126,0127,0130, 0131,0132,0262,0324,0326,0322,0323,0325,
	0060,0061,0062,0063,0064,0065,0066,0067, 0070,0071,0263,0333,0334,0331,0332,0237
};

static Vcchar_t	A2M[] =
{	/* CC_ASCII=>CC_EBCDIC_M */
 0x00, 0x01, 0x02, 0x03, 0x37, 0x2d, 0x2e, 0x2f,
 0x16, 0x05, 0x15, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
 0x10, 0x11, 0x12, 0x13, 0x3c, 0x3d, 0x32, 0x26,
 0x18, 0x19, 0x3f, 0x27, 0x1c, 0x1d, 0x1e, 0x1f,
 0x40, 0x5a, 0x7f, 0x7b, 0x5b, 0x6c, 0x50, 0x7d,
 0x4d, 0x5d, 0x5c, 0x4e, 0x6b, 0x60, 0x4b, 0x61,
 0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
 0xf8, 0xf9, 0x7a, 0x5e, 0x4c, 0x7e, 0x6e, 0x6f,
 0x7c, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
 0xc8, 0xc9, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6,
 0xd7, 0xd8, 0xd9, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6,
 0xe7, 0xe8, 0xe9, 0xad, 0xe0, 0xbd, 0x5f, 0x6d,
 0x79, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
 0x88, 0x89, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96,
 0x97, 0x98, 0x99, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6,
 0xa7, 0xa8, 0xa9, 0xc0, 0x4f, 0xd0, 0xa1, 0x07,
 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x06, 0x17,
 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x09, 0x0a, 0x1b,
 0x30, 0x31, 0x1a, 0x33, 0x34, 0x35, 0x36, 0x08,
 0x38, 0x39, 0x3a, 0x3b, 0x04, 0x14, 0x3e, 0xdf,
 0x41, 0xaa, 0x4a, 0xb1, 0x9f, 0xb2, 0x6a, 0xb5,
 0xbb, 0xb4, 0x9a, 0x8a, 0xb0, 0xca, 0xaf, 0xbc,
 0x90, 0x8f, 0xea, 0xfa, 0xbe, 0xa0, 0xb6, 0xb3,
 0x9d, 0xda, 0x9b, 0x8b, 0xb7, 0xb8, 0xb9, 0xab,
 0x64, 0x65, 0x62, 0x66, 0x63, 0x67, 0x9e, 0x68,
 0x74, 0x71, 0x72, 0x73, 0x78, 0x75, 0x76, 0x77,
 0xac, 0x69, 0xed, 0xee, 0xeb, 0xef, 0xec, 0xbf,
 0x80, 0xfd, 0xfe, 0xfb, 0xfc, 0xba, 0xae, 0x59,
 0x44, 0x45, 0x42, 0x46, 0x43, 0x47, 0x9c, 0x48,
 0x54, 0x51, 0x52, 0x53, 0x58, 0x55, 0x56, 0x57,
 0x8c, 0x49, 0xcd, 0xce, 0xcb, 0xcf, 0xcc, 0xe1,
 0x70, 0xdd, 0xde, 0xdb, 0xdc, 0x8d, 0x8e, 0xff,
};

static Vcchar_t	M2A[] =
{	/* CC_EBCDIC_M=>CC_ASCII */
 0x00, 0x01, 0x02, 0x03, 0x9c, 0x09, 0x86, 0x7f,
 0x97, 0x8d, 0x8e, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
 0x10, 0x11, 0x12, 0x13, 0x9d, 0x0a, 0x08, 0x87,
 0x18, 0x19, 0x92, 0x8f, 0x1c, 0x1d, 0x1e, 0x1f,
 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x17, 0x1b,
 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x05, 0x06, 0x07,
 0x90, 0x91, 0x16, 0x93, 0x94, 0x95, 0x96, 0x04,
 0x98, 0x99, 0x9a, 0x9b, 0x14, 0x15, 0x9e, 0x1a,
 0x20, 0xa0, 0xe2, 0xe4, 0xe0, 0xe1, 0xe3, 0xe5,
 0xe7, 0xf1, 0xa2, 0x2e, 0x3c, 0x28, 0x2b, 0x7c,
 0x26, 0xe9, 0xea, 0xeb, 0xe8, 0xed, 0xee, 0xef,
 0xec, 0xdf, 0x21, 0x24, 0x2a, 0x29, 0x3b, 0x5e,
 0x2d, 0x2f, 0xc2, 0xc4, 0xc0, 0xc1, 0xc3, 0xc5,
 0xc7, 0xd1, 0xa6, 0x2c, 0x25, 0x5f, 0x3e, 0x3f,
 0xf8, 0xc9, 0xca, 0xcb, 0xc8, 0xcd, 0xce, 0xcf,
 0xcc, 0x60, 0x3a, 0x23, 0x40, 0x27, 0x3d, 0x22,
 0xd8, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
 0x68, 0x69, 0xab, 0xbb, 0xf0, 0xfd, 0xfe, 0xb1,
 0xb0, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70,
 0x71, 0x72, 0xaa, 0xba, 0xe6, 0xb8, 0xc6, 0xa4,
 0xb5, 0x7e, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
 0x79, 0x7a, 0xa1, 0xbf, 0xd0, 0x5b, 0xde, 0xae,
 0xac, 0xa3, 0xa5, 0xb7, 0xa9, 0xa7, 0xb6, 0xbc,
 0xbd, 0xbe, 0xdd, 0xa8, 0xaf, 0x5d, 0xb4, 0xd7,
 0x7b, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
 0x48, 0x49, 0xad, 0xf4, 0xf6, 0xf2, 0xf3, 0xf5,
 0x7d, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50,
 0x51, 0x52, 0xb9, 0xfb, 0xfc, 0xf9, 0xfa, 0x9f,
 0x5c, 0xf7, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
 0x59, 0x5a, 0xb2, 0xd4, 0xd6, 0xd2, 0xd3, 0xd5,
 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
 0x38, 0x39, 0xb3, 0xdb, 0xdc, 0xd9, 0xda, 0xff,
};

static Vcchar_t	A2U[] =
{	/* CC_ASCII=>CC_EBCDIC_U */
 0x00, 0x01, 0x02, 0x03, 0x37, 0x2d, 0x2e, 0x2f,
 0x16, 0x05, 0x25, 0x0b, 0x0c, 0x0d, 0x0e, 0x9f,
 0x10, 0x11, 0x12, 0x13, 0xb6, 0xb5, 0x32, 0x26,
 0x18, 0x19, 0x3f, 0x27, 0x1c, 0x1d, 0x1e, 0x1f,
 0x40, 0x5a, 0x7f, 0x7b, 0x5b, 0x6c, 0x50, 0x7d,
 0x4d, 0x5d, 0x5c, 0x4e, 0x6b, 0x60, 0x4b, 0x61,
 0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
 0xf8, 0xf9, 0x7a, 0x5e, 0x4c, 0x7e, 0x6e, 0x6f,
 0x7c, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
 0xc8, 0xc9, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6,
 0xd7, 0xd8, 0xd9, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6,
 0xe7, 0xe8, 0xe9, 0xad, 0xe0, 0xbd, 0x5f, 0x6d,
 0x79, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
 0x88, 0x89, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96,
 0x97, 0x98, 0x99, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6,
 0xa7, 0xa8, 0xa9, 0xc0, 0x4f, 0xd0, 0xa1, 0x07,
 0x68, 0xdc, 0x51, 0x42, 0x43, 0x44, 0x47, 0x48,
 0x52, 0x53, 0x54, 0x57, 0x56, 0x58, 0x63, 0x67,
 0x71, 0x9c, 0x9e, 0xcb, 0xcc, 0xcd, 0xdb, 0xdd,
 0xdf, 0xec, 0xfc, 0xb0, 0xb1, 0xb2, 0x3e, 0xb4,
 0x45, 0x55, 0xce, 0xde, 0x49, 0x69, 0x9a, 0x9b,
 0xab, 0x0f, 0xba, 0xb8, 0xb7, 0xaa, 0x8a, 0x8b,
 0x3c, 0x3d, 0x62, 0x6a, 0x64, 0x65, 0x66, 0x20,
 0x21, 0x22, 0x70, 0x23, 0x72, 0x73, 0x74, 0xbe,
 0x76, 0x77, 0x78, 0x80, 0x24, 0x15, 0x8c, 0x8d,
 0x8e, 0x41, 0x06, 0x17, 0x28, 0x29, 0x9d, 0x2a,
 0x2b, 0x2c, 0x09, 0x0a, 0xac, 0x4a, 0xae, 0xaf,
 0x1b, 0x30, 0x31, 0xfa, 0x1a, 0x33, 0x34, 0x35,
 0x36, 0x59, 0x08, 0x38, 0xbc, 0x39, 0xa0, 0xbf,
 0xca, 0x3a, 0xfe, 0x3b, 0x04, 0xcf, 0xda, 0x14,
 0xe1, 0x8f, 0x46, 0x75, 0xfd, 0xeb, 0xee, 0xed,
 0x90, 0xef, 0xb3, 0xfb, 0xb9, 0xea, 0xbb, 0xff,
};

static Vcchar_t	U2A[] =
{	/* CC_EBCDIC_U=>CC_ASCII */
 0x00, 0x01, 0x02, 0x03, 0xec, 0x09, 0xca, 0x7f,
 0xe2, 0xd2, 0xd3, 0x0b, 0x0c, 0x0d, 0x0e, 0xa9,
 0x10, 0x11, 0x12, 0x13, 0xef, 0xc5, 0x08, 0xcb,
 0x18, 0x19, 0xdc, 0xd8, 0x1c, 0x1d, 0x1e, 0x1f,
 0xb7, 0xb8, 0xb9, 0xbb, 0xc4, 0x0a, 0x17, 0x1b,
 0xcc, 0xcd, 0xcf, 0xd0, 0xd1, 0x05, 0x06, 0x07,
 0xd9, 0xda, 0x16, 0xdd, 0xde, 0xdf, 0xe0, 0x04,
 0xe3, 0xe5, 0xe9, 0xeb, 0xb0, 0xb1, 0x9e, 0x1a,
 0x20, 0xc9, 0x83, 0x84, 0x85, 0xa0, 0xf2, 0x86,
 0x87, 0xa4, 0xd5, 0x2e, 0x3c, 0x28, 0x2b, 0x7c,
 0x26, 0x82, 0x88, 0x89, 0x8a, 0xa1, 0x8c, 0x8b,
 0x8d, 0xe1, 0x21, 0x24, 0x2a, 0x29, 0x3b, 0x5e,
 0x2d, 0x2f, 0xb2, 0x8e, 0xb4, 0xb5, 0xb6, 0x8f,
 0x80, 0xa5, 0xb3, 0x2c, 0x25, 0x5f, 0x3e, 0x3f,
 0xba, 0x90, 0xbc, 0xbd, 0xbe, 0xf3, 0xc0, 0xc1,
 0xc2, 0x60, 0x3a, 0x23, 0x40, 0x27, 0x3d, 0x22,
 0xc3, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
 0x68, 0x69, 0xae, 0xaf, 0xc6, 0xc7, 0xc8, 0xf1,
 0xf8, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70,
 0x71, 0x72, 0xa6, 0xa7, 0x91, 0xce, 0x92, 0x0f,
 0xe6, 0x7e, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
 0x79, 0x7a, 0xad, 0xa8, 0xd4, 0x5b, 0xd6, 0xd7,
 0x9b, 0x9c, 0x9d, 0xfa, 0x9f, 0x15, 0x14, 0xac,
 0xab, 0xfc, 0xaa, 0xfe, 0xe4, 0x5d, 0xbf, 0xe7,
 0x7b, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
 0x48, 0x49, 0xe8, 0x93, 0x94, 0x95, 0xa2, 0xed,
 0x7d, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50,
 0x51, 0x52, 0xee, 0x96, 0x81, 0x97, 0xa3, 0x98,
 0x5c, 0xf0, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
 0x59, 0x5a, 0xfd, 0xf5, 0x99, 0xf7, 0xf6, 0xf9,
 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
 0x38, 0x39, 0xdb, 0xfb, 0x9a, 0xf4, 0xea, 0xff,
};

#endif

#define MAP_INPLACE	((Void_t*)1)

/* shared with Vcetoa */
Vcmtarg_t _Mapargs[] =
{	{ "a2e", "ASCII -> Xopen dd(1) EBCDIC", A2E },
	{ "e2a", "Xopen dd(1) EBCDIC -> ASCII", E2A },
	{ "a2i", "ASCII -> Xopen dd(1) IBM EBCDIC", A2I },
	{ "i2a", "Xopen dd(1) IBM EBCDIC -> ASCII", I2A },
	{ "a2o", "ASCII -> IBM OpenEdition EBCDIC", A2O },
	{ "o2a", "IBM OpenEdition EBCDIC -> ASCII", O2A },
	{ "a2s", "ASCII -> Siemens Posix-bc EBCDIC", A2S },
	{ "s2a", "Siemens Posix-bc EBCDIC -> ASCII", S2A },
	{ "a2h", "ASCII -> IBM-37 AS/400 EBCDIC", A2H },
	{ "h2a", "IBM-37 AS/400 EBCDIC -> ASCII", H2A },
	{ "a2m", "ASCII -> IBM mvs cobol EBCDIC", A2M },
	{ "m2a", "IBM mvs cobol EBCDIC -> ASCII", M2A },
	{ "a2u", "ASCII -> microfocus cobol EBCDIC", A2U },
	{ "u2a", "microfocus cobol EBCDIC -> ASCII", U2A },
	{   0  , "Identity mapping.", 0 }
};

typedef struct _vcmap_s
{	Vcchar_t*	map;	/* character map	*/
	int		inplace; /* inplace mapping	*/
} Vcmap_t;

#if __STD_C
static ssize_t vcmap(Vcodex_t* vc, const Void_t* data, size_t size, Void_t** out)
#else
static ssize_t vcmap(vc, data, size, out)
Vcodex_t*	vc;
Void_t*		data;
size_t		size;
Void_t**	out;
#endif
{
	ssize_t		sz;
	Vcchar_t	*dt, *enddt, *output, *o;
	Vcchar_t	*map;
	Vcmap_t		*vcm = vcgetmtdata(vc, Vcmap_t*);
	Vcdisc_t	*disc = vcgetdisc(vc);

	if((sz = (ssize_t)size) == 0)
		return 0;
	if(!(dt = (Vcchar_t*)data) )
		return -1;

	if(!(map = vcm->map) )
		if(disc && disc->data)
#if _PACKAGE_ast
			map = CCMAP(integralof(disc->data), 0);
#else
			map = disc->data;
#endif

	if(vc->flags&VC_DECODE)
		if(vcrecode(vc, &dt, &sz, 0, 0) < 0)
			return -1;

	if((vc->flags&VC_ENCODE) && vcm->inplace)
		output = (Vcchar_t*)data;
	else if(!(output = vcbuffer(vc, NIL(Vcchar_t*), sz, 0)) )
		return -1;

	if(map) /* non-trivial mapping */
	{	for(o = output, enddt = dt+sz; dt < enddt; )
			*o++ = map[*dt++];
	}
	else /* identity mapping */
	{	if(!vcm->inplace)
			memcpy(output, dt, sz);
	}

	if((vc->flags&VC_DECODE) && vc->coder)
		vcbuffer(vc, dt-sz, -1, -1);

	if(vc->flags&VC_ENCODE)
	{	dt = output;
		if(vcrecode(vc, &output, &sz, 0, 0) < 0 )
			return -1;
		if(dt != output)
			vcbuffer(vc, dt, -1, -1);
	}

	if(out)
		*out = output;
	return sz;
}

#if __STD_C
static int mapevent(Vcodex_t* vc, int type, Void_t* params)
#else
static int mapevent(vc, type, params)
Vcodex_t*	vc;
int		type;
Void_t*		params;
#endif
{
	Vcmap_t		*vcm;
	char		*data;
	Vcmtarg_t	*arg;

	if(type == VC_OPENING)
	{	if(!(vcm = (Vcmap_t*)calloc(1, sizeof(Vcmap_t))) )
			return -1;
		vcsetmtdata(vc, vcm);
		goto vc_setarg;
	}
	else if(type == VC_SETMTARG)
	{	if(!(vcm = vcgetmtdata(vc, Vcmap_t*)) )
			return -1;
	vc_setarg:
		for(data = (char*)params; data; )
		{	data = vcgetmtarg(data, 0, 0, _Mapargs, &arg);
			if(arg)
			{	if(arg->data != MAP_INPLACE)
					vcm->map = arg->data;
				else if(vc->flags&VC_ENCODE)
					vcm->inplace = 1;
			}
		}
	}
	else if(type == VC_CLOSING)
	{	if((vcm = vcgetmtdata(vc, Vcmap_t*)) )
			free(vcm);
		vcsetmtdata(vc, NIL(Vcmap_t*));
	}

	return 0;
}

Vcmethod_t _Vcmap =
{	vcmap,
	vcmap,
	mapevent,
	"map", "Mapping bytes from codeset to codeset.",
	"[-version?map (AT&T Research) 2003-01-01]" USAGE_LICENSE,
	_Mapargs,
	1024*1024,
	0
};

VCLIB(Vcmap)
