/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008-2021 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/
#ifndef YAML_CONFIG_H
#define YAML_CONFIG_H 1
#include <deque>
#include <glibmm/i18n.h>
#include <glibmm/ustring.h>
#include <iostream>
#include <string>
#include <config.h>

/// The default page title for printing
const std::string DEFAULT_PAGE_TITLE = _("Kitlist");

/// The maximum number of recent files to maintain
const gint DEFAULT_MAX_RECENT_FILES = 4;

#ifndef GCONF

/**
 * \brief Maintains the application's configuration parameters in a
 * YAML formatted file.
 *
 * See https://yaml.org/ for information on the YAML file format.
 *
 * The application uses the C++ implementation of a YAML parser from
 * https://github.com/jbeder/yaml-cpp/
 *
 * The location of the configuration file follows the XDG Base
 * Directory Specification.  By default this is `~/.config/kitlist`
 * but can be overridden by specifying the name of the configuration
 * directory by defining the `XDG_CONFIG_HOME` environment variable.
 *
 * See: https://specifications.freedesktop.org/basedir-spec/basedir-spec-latest.html
 */
class YamlConfig {
private:

    /// The list of recently used files for the recent files menu.
    gint m_max_recent_files;

    /// \brief The current filename.
    ///
    /// Used to reload the same file on the next occasion the
    /// application is launched.
    Glib::ustring m_current_filename;

    /// \brief The name of the log file to write debug information to.
    ///
    /// Note: This is only used if the application is compiled
    /// with KITLIST_DEBUG.  See the distribution README for more
    /// information.
    Glib::ustring m_debug_log_filename;

    /// The title to use when printing a page or creating a PDF.
    Glib::ustring m_page_title;

    Glib::ustring get_config_filename();

    /// The history list of most recently used filenames.
    std::deque<Glib::ustring> m_mru_file_history;
public:
    YamlConfig();

    /// Also saves the configuration on program exit.
    ~YamlConfig() { save(); };

    void load();

    void save();

    /// \brief The current filename
    /// \return the current filename
    Glib::ustring get_current_filename() { return m_current_filename; };

    /// \brief Sets the current filename.
    ///
    /// \param filename the name of the file to set.
    void set_current_filename(Glib::ustring filename) { m_current_filename = filename; };

    /// \brief The page title to use for printing or creating a PDF.
    /// \return the page title.
    Glib::ustring get_page_title() { return m_page_title; };

    /// \brief Set the page title.
    ///
    /// Sets the page title used for printing or creating a PDF.
    ///
    /// \param title the page title to set.
    void set_page_title(Glib::ustring title) { m_page_title = title; };

    void add_recent_filename(Glib::ustring filename);

    /// \brief The history list of recent filenames.
    ///
    /// \return a list of the most recently used filenames.
    std::deque<Glib::ustring> get_recent_filenames() { return m_mru_file_history; };

    /// \brief the name of the debug log file.
    /// \return the filename.
    Glib::ustring get_debug_log_filename() { return m_debug_log_filename; };
};

#endif // GCONF
#endif // YAML_CONFIG_H
