/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008-2021 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "yamlconfig.hpp"
#include <config.h>

#ifndef GCONF
#include <glibmm/miscutils.h>
#include <fstream>
#include <iostream>
#include <yaml-cpp/yaml.h>


/// The filename used to store the application's configuration
const std::string CONFIG_FILENAME = "/kitlist";

/// The key used to store the value of the page title in the configuration file
const std::string PAGE_TITLE_CONFIG_KEY = "Printed page title";

/// The key used to store the value of the current filename in the configuration file
const std::string CURRENT_FILENAME_CONFIG_KEY = "current filename";

/// The key used to store the value of the current filename in the configuration file
const std::string RECENT_FILES_CONFIG_KEY = "file history list";

/// The key for the maximum number of files to maintain in the recent files menu
const std::string MAX_RECENT_FILES_CONFIG_KEY = "max recent files";

/// GConf entry for the page title
const std::string DEBUG_LOG_FILENAME_CONFIG_KEY = "debug_log_filename";

YamlConfig::YamlConfig() :
    m_max_recent_files(DEFAULT_MAX_RECENT_FILES),
    m_current_filename(""),
    m_page_title(""),
    m_debug_log_filename("/tmp/kitlist.log") {
    YamlConfig::load();
};

/// Loads the configuration file.
void YamlConfig::load() {
    // Try getting the value from XDG config file
    try {
        YAML::Node config = YAML::LoadFile(get_config_filename().c_str());
        try {
            m_current_filename = config[CURRENT_FILENAME_CONFIG_KEY].as<std::string>();
        } catch (std::exception) {
            g_info("Current filename not found in config");
        }
        try {
            YAML::Node files = config[RECENT_FILES_CONFIG_KEY];
            for (YAML::const_iterator it = files.begin(); it != files.end(); ++it) {
                Glib::ustring filename = it->as<std::string>();
                m_mru_file_history.push_back(filename);
            }
        } catch (std::exception) {
            g_info("MRU file list not found in config");
        }
        try {
            m_page_title = config[PAGE_TITLE_CONFIG_KEY].as<std::string>();
        } catch (std::exception) {
            g_info("Page title not found in config");
            m_page_title = DEFAULT_PAGE_TITLE;
        }
        try {
            m_max_recent_files = config[MAX_RECENT_FILES_CONFIG_KEY].as<int>();
        } catch (std::exception) {
            g_info("Max recent filename history count not found in config");
        }
        try {
            m_debug_log_filename = config[DEBUG_LOG_FILENAME_CONFIG_KEY].as<std::string>();
        } catch (std::exception) {
            g_info("Debug log filename not found in config");
        }
    } catch (std::exception ex) {
        g_info("Error loading config file");
        m_page_title = DEFAULT_PAGE_TITLE;
        // m_yaml_config[PAGE_TITLE_CONFIG_KEY] = m_page_title.c_str();
    }
}

/**
 * \brief Saves the current configuration.
 */
void YamlConfig::save() {
    YAML::Node config;
    if (m_current_filename.empty()) {
        config.remove(CURRENT_FILENAME_CONFIG_KEY);
    } else {
        config[CURRENT_FILENAME_CONFIG_KEY] = m_current_filename.c_str();
    }
    if (m_page_title.empty()) {
        config.remove(PAGE_TITLE_CONFIG_KEY);
    } else {
        config[PAGE_TITLE_CONFIG_KEY] = m_page_title.c_str();
    }
    config[MAX_RECENT_FILES_CONFIG_KEY] = m_max_recent_files;
    config[DEBUG_LOG_FILENAME_CONFIG_KEY] = m_debug_log_filename.c_str();
    for (std::deque<Glib::ustring>::const_iterator it = m_mru_file_history.begin(); it != m_mru_file_history.end(); ++it) {
        config[RECENT_FILES_CONFIG_KEY].push_back(it->c_str());
    }
    std::ofstream fout (get_config_filename().c_str());
    fout << config;
}

/**
 * Returns the path to the user's configuration file for this application.
 * e.g. "/home/user/.config/kitlist"
 */
Glib::ustring YamlConfig::get_config_filename() {
    std::ostringstream os;
    os << Glib::get_user_config_dir() << CONFIG_FILENAME;
    return os.str();
}


/**
 * \brief Adds a filename to the list of recently used filenames.
 *
 * \param filename the filename to set.
 */
void YamlConfig::add_recent_filename(Glib::ustring filename) {
    // Remove any existing entry with the same filename
    for (auto it = m_mru_file_history.begin(); it != m_mru_file_history.end(); ) {
        if (*it == filename) {
            it = m_mru_file_history.erase(it);
        } else {
            ++it;
        }
    }
    m_mru_file_history.push_front(filename);
    while (m_mru_file_history.size() > m_max_recent_files) {
        m_mru_file_history.pop_back();
    }
};

#endif // GCONF
