/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008,2009 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "kitparser.hpp"

#include <algorithm>
#include <iostream>

/// Does nothing.  Called at the start of a document
void KitParser::on_start_document() {
//     std::cout << "on_start_document()" << std::endl;
}


/// Does nothing.  Called at the end of a document
void KitParser::on_end_document() {
//     std::cout << "on_end_document()" << std::endl;
}


/**
 * \brief Reads an item's attributes from an 'item' element.
 *
 * \param attributes The list of attributes for the element.
 */
void KitParser::process_item(const AttributeList& attributes) {
    m_item = new ModelItem();
    xmlpp::SaxParser::AttributeList::const_iterator attribute =
        std::find_if(attributes.begin(), attributes.end(), AttributeHasName("id"));
    if (attribute != attributes.end()) {
        m_item->set_id(atol(attribute->value.c_str()));
    } else {
        std::cout << "WARNING: Missing id attribute for a item element " << std::endl;
    }
    attribute =
        std::find_if(attributes.begin(), attributes.end(), AttributeHasName("checked"));
    if (attribute != attributes.end()) {
        m_item->set_checked(attribute->value.casefold().compare("false") != 0);
    }
    m_model.add_item(m_item);
}


/**
 * \brief Reads a category's attributes from a 'category' element.
 *
 * \param attributes The list of attributes for the element.
 */
void KitParser::process_category(const AttributeList& attributes) {
    xmlpp::SaxParser::AttributeList::const_iterator attribute =
        std::find_if(attributes.begin(), attributes.end(), AttributeHasName("id"));
    if (attribute != attributes.end()) {
        m_category = new ModelCategory();
        m_category->set_id(atol(attribute->value.c_str()));
        m_model.add_category(m_category);
    } else {
        std::cout << "WARNING: Missing id attribute for a category element " << std::endl;
    }
}


/**
 * \brief Reads details of an item association with a category from a
 * 'category-item' element.
 *
 * \param attributes The list of attributes for the element.
 */
void KitParser::process_category_item(const AttributeList& attributes) {
    xmlpp::SaxParser::AttributeList::const_iterator attribute =
        std::find_if(attributes.begin(), attributes.end(), AttributeHasName("id"));
    if (attribute != attributes.end()) {
        if (m_category != NULL) {
            ModelItem* item = m_model.find_item(atol(attribute->value.c_str()));
            if (item) {
                m_category->add_item(item);
            } else {
                std::cout << "Unable to find item id: " 
                          << attribute->value
                          << "belonging to category id: "
                          << m_category->get_id()
                          << std::endl;
            }
        }
    } else {
        std::cout << "WARNING: Missing id attribute for a category-item element " << std::endl;
    }
}


/**
 * \brief Called for each element.
 *
 * Determines which element is being processed and calls appropriate
 * method to process the corresponding attributes.
 */
void KitParser::on_start_element(const Glib::ustring& name, const AttributeList& attributes) {
    // reset the CDATA store at the start of a new element
    m_cdata.clear();
//     std::cout << "on_start_element(): " << name << std::endl;
    if (name.compare("item")  == 0) {
        process_item(attributes);
    } else if (name.compare("category") == 0) {
        process_category(attributes);
    } else if (name.compare("category-item") == 0) {
        process_category_item(attributes);
    }
    // Attributes
//     for (xmlpp::SaxParser::AttributeList::const_iterator
//              iter = attributes.begin(); iter != attributes.end(); ++iter) {
//         std::cout << "    Attribute: " << iter->name << " = \"" << iter->value << "\"" << std::endl;
//     }
}


/**
 * \brief Called at the end of each element.
 *
 * If any CDATA existed in the element body, sets the text belonging
 * to the appropriate element object with the previously captured
 * CDATA.
 */
void KitParser::on_end_element(const Glib::ustring& name) {
//     std::cout << "on_end_element(): " << name << std::endl;
    if (name.compare("item")  == 0 && m_item != NULL && m_cdata.size() > 0) {
        m_item->set_description(m_cdata.c_str());
    } else if (name.compare("category-name") == 0 && m_category != NULL  && m_cdata.size() > 0) {
//         std::cout << "Setting category name to: " << name << std::endl;
        m_category->set_name(m_cdata.c_str());
    }
}


/**
 * \brief Called for each piece of CDATA belonging to an element.
 *
 * This may be called a number of times, each time passing a bit more
 * of the parsed CDATA.  The processing of the CDATA is split up by
 * any embedded entities.
 *
 * The CDATA is held in a private member variable.  The CDATA variable
 * is cleared at the start of each new element and appended to at each
 * call to this method.  The captured text is then set on the
 * appropriate object during KitParser::on_end_element().
 */
void KitParser::on_characters(const Glib::ustring& text) {
//     std::cout << "on_characters(): " << text << std::endl;
    m_cdata += text;
}


/// Does nothing.  Called for each comment.
void KitParser::on_comment(const Glib::ustring& text) {
//     std::cout << "on_comment(): " << text << std::endl;
}


/// Outputs any warnings to STDOUT.
void KitParser::on_warning(const Glib::ustring& text) {
    std::cout << "on_warning(): " << text << std::endl;
}


/// Outputs any errors to STDOUT.
void KitParser::on_error(const Glib::ustring& text) {
    std::cout << "on_error(): " << text << std::endl;
}


/// Outputs any fatal errors to STDOUT.
void KitParser::on_fatal_error(const Glib::ustring& text) {
    std::cout << "on_fatal_error: " << text << std::endl;
}
