/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008,2009 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef ITEM_H
#define ITEM_H 1

#include <iostream>
#include <list>
#include <string>
#include <vector>
#include <sigc++/signal.h>

class ItemCompareName;

/**
 * \brief Represents an Item.
 */
class Item {
    long m_id; ///< Unique ID
    std::string m_desc; ///< The item's description
    bool m_checked; ///< Whether checked/ticked or not
 public:
    Item() : m_checked(true), m_id(-1) {}
    /// Creates a copy of this item based on the passed item.
    Item(const Item& i) : m_id(i.m_id), m_desc(i.m_desc), m_checked(i.m_checked) {}
    void set_id(long id) { m_id = id; }
    long  get_id() { return m_id; }
    void set_description(const std::string description) { m_desc = description; }
    std::string get_description() { return m_desc; }
    virtual void set_checked(bool checked) { m_checked = checked; }
    bool get_checked() { return m_checked; }
    friend class ItemCompareName;
    friend class ItemCompareId;
    friend std::ostream& operator<<(std::ostream& os, const Item& i) { return os << i.m_id; }
};


/**
 * \brief Comparator used for sorting Items by name.
 * \see Item
 */
class ItemCompareName {
 public:
    ItemCompareName() {}
    int operator()(Item* i1, Item* i2) { return (i1->m_desc < i2->m_desc); }
};


/**
 * \brief Comparator used for comparing Items by id.
 * \see Item
 */
class ItemCompareId {
 public:
    ItemCompareId() {}
    int operator()(Item* i1, Item* i2) { return (i1->m_id < i2->m_id); }
};


/**
 * \brief Functor for processing items.
 *
 * Intended to have the operator() overriden by an actual
 * implementation.
 */
class ItemFunctor {
 public:
    virtual bool operator()(Item& item) = 0;
};


typedef std::vector<Item*> ItemContainer;
typedef ItemContainer::iterator ItemIter;

typedef std::list<Item> ItemList;
typedef ItemList::iterator ItemListIter;

typedef sigc::slot<bool, Item&> SlotForeachItem;

#endif // ITEM_H
