/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008,2009 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef XMLDAO_H
#define XMLDAO_H 1

#include <cassert>
#include <libxml++/libxml++.h>
#include "kitlistdao.hpp"
#include "kitmodel.hpp"

#define NYI assert(false == "Method not implemented")


/**
 * \brief Implementation of a KitListDao using XML as the persistence
 * store.
 */
class XmlDao : public KitListDao {
private:

    bool add_item_to_dom(ModelItem& item);

    bool add_category_item_to_dom(Item& item);

    bool add_category_to_dom(ModelCategory& item);


protected:
    /// The filename to load or save the XML document from.
    Glib::ustring m_filename;
    /// Temporary reference to the items' node
    xmlpp::Element* m_items_node;
    /// Temporary reference to the categories' node
    xmlpp::Element* m_categories_node;
    /// Temporary reference to the categories' items' node
    xmlpp::Element* m_cat_items_node;
    /// The last used ID for items
    long m_max_item_id;
    /// The last used ID for categories
    long m_max_category_id;

public :
    XmlDao(int verbose = 0) : KitListDao(verbose),
        m_filename(""),
        m_max_item_id(-1),
        m_max_category_id(-1),
        m_categories_node(NULL),
        m_items_node(NULL) {}

    KitModel* get_model();

    KitModel* get_model(Glib::ustring filename) { set_filename(filename); return get_model(); }

    void save_model(KitModel* model);

    /**
     * \brief Saves the model as an XML document.
     *
     * \param model The model to save.
     * \param filename The name of the file to save to.
     */
    void save_model(KitModel* model, Glib::ustring filename) { set_filename(filename); save_model(model); }

    Category* get_category(long cat_id, item_choice choice) { NYI; }

    ItemContainer* get_all_items(item_choice choice) { NYI; }

    long add_item(const std::string name) { NYI; }

    long add_item(const std::string name, long cat_id) { NYI; }

    void append_items_to_category(long to_cat_id, long from_cat_id, item_choice choice) { NYI; }

    void associate_item_with_category(long id, long cat_id) { NYI; }

    CategoryContainer get_categories() { NYI; }

    long new_category(const std::string name) { NYI; }

    void delete_item(long id) { NYI; }

    void update_item_checked_state(ItemContainer& items) { NYI; }

    void remove_item_from_category(long id, long cat_id) { NYI; }

    long get_next_item_id();

    long get_next_category_id();

    void delete_category(long id) { NYI; }

    void set_item_flag(long id) { NYI; }

    void unset_item_flag(long id) { NYI; }

    void set_category_flag(long id) { NYI; }

    void unset_category_flag(long id) { NYI; }

    void set_all_flags() { NYI; }

    void unset_all_flags() { NYI; }

    void set_filename(Glib::ustring filename) { m_filename = filename; }

    /**
     * \brief Indicates that this implementation requires a filename.
     *
     * This persistence model requires a filename to be
     * chosen before the data can be persisted.
     *
     * \return Always returns true.
     */
    virtual bool require_filename() { return true; }

};

#endif // XMLDAO_H
