/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008,2009 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef SERVICE_H
#define SERVICE_H 1

#include "kitlistdao.hpp"
#include "kitmodel.hpp"
#include "xmldao.hpp"
#include <glibmm/ustring.h>


/**
 * \brief Business/service layer implementation.
 *
 * Implements the service layer of the application, such that a
 * different front-end can re-use the provided business methods.
 */
class Service {
 protected:
    KitListDao& m_dao; ///< Reference to the perisitence data access object.
    KitModel* m_model; ///< The application's data model.
    KitModel* load_model();
    long get_next_item_id();
    long get_next_category_id();
 public:
    Service(KitListDao& dao);
    ~Service();
    ModelItem* find_item(long id);
    ModelCategory* find_category(long cat_id);
    void copy_items(const ModelItemContainer& items, long cat_id);
    Item* create_item(long cat_id);
    bool delete_item(long id);
    bool delete_category(long cat_id);
    Category* create_category();
    bool is_model_dirty() { return m_model ? m_model->is_dirty() : false; }
    void set_model_dirty(bool flag = true);
    /**
     * \brief Applies the current filter.
     *
     * \param checked The checked/ticked state of the item being filtered.
     * \return true if the item should be included.
     */
    virtual bool filter(bool checked) { return m_model->filter(checked); }
    void create_default_model();
    void open_as_xml(const Glib::ustring& filename);
    void save();
    void save_as_xml(const Glib::ustring& filename);
    bool update_item(long id, const std::string description, bool checked);
    ItemContainer* get_items(long cat_id = -1);
    ItemContainer* get_filtered_items(long cat_id = -1);
    CategoryContainer* get_categories();
    /// Removes filter.  All items are shown.
    virtual void show_all() { m_model->show_all(); }
    /// Sets the filter to show only checked items.
    virtual void show_checked_only() { m_model->show_checked_only(); }
    /// Sets the filter to show only unchecked items.
    virtual void show_unchecked_only() { m_model->show_unchecked_only(); }
    virtual void select_items(ModelItemContainer* items, bool checked = true);
    virtual void toggle_selected_items(ModelItemContainer* items);
    virtual bool require_filename() { return m_dao.require_filename(); }

};


#endif // SERVICE_H
