/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008,2009 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "printing.hpp"
#include <sstream>
#include <glibmm/i18n.h>
#include <config.h>

/// Seems calculating page body height is inaccurate
const int PAGE_TOLERANCE = 40;

/// The spacing between the header and the body
const int HEADER_SPACING = 10;

/// The spacing between the footer and the body
const int FOOTER_SPACING = 10;

/// The space above and below the horizontal lines at the top and bottom of each page
const int BORDER_SPACING = 10;

/// Text used to print page number in footer - Page x of n
const Glib::ustring FOOTER_TEXT = _("Page %1 of %2");

/// Destructor
KitPrintOperation::~KitPrintOperation() {
//     g_debug("~KitPrintOperation()");
    if (m_items)
        delete m_items;
}

/// Factory to create instances
Glib::RefPtr<KitPrintOperation> KitPrintOperation::create() {
    return Glib::RefPtr<KitPrintOperation>(new KitPrintOperation());
}

/**
 * Creates a new header element for a page
 *
 * \param context the print context
 */
layout_refptr KitPrintOperation::new_header(const Glib::RefPtr<Gtk::PrintContext>& context) {
    Pango::FontDescription header_font("sans 16");
    layout_refptr retval = context->create_pango_layout();
    retval->set_font_description(header_font);
#ifdef MAEMO
    std::ostringstream os;
    os << "<b>" << m_page_title << "</b>";
    retval->set_markup(os.str());
#else
    retval->set_markup(Glib::ustring::compose("<b>%1</b>", m_page_title));
#endif
    retval->set_alignment(Pango::ALIGN_CENTER);
    retval->set_width(static_cast<int>(context->get_width() * Pango::SCALE));
    m_ref_headers.push_back(retval);
    return retval;
}

/**
 * Creates a new footer element for a page
 *
 * \param context the print context
 */
layout_refptr KitPrintOperation::new_footer(const Glib::RefPtr<Gtk::PrintContext>& context) {
    Pango::FontDescription font_desc("sans 12");
    layout_refptr retval = context->create_pango_layout();
    retval->set_font_description(font_desc);
    Glib::ustring footer = FOOTER_TEXT;
    retval->set_text(footer);
    retval->set_alignment(Pango::ALIGN_CENTER);
    retval->set_width(static_cast<int>(context->get_width() * Pango::SCALE));
    m_ref_footers.push_back(retval);
    return retval;
}

/**
 * Called prior to pages being printed.  Calculates what is printed on each page.
 *
 * \param context the print context
 */
void KitPrintOperation::on_begin_print(const Glib::RefPtr<Gtk::PrintContext>& context) {
    Pango::FontDescription font_desc("sans 12");
    layout_refptr ref_footer = new_footer(context);
    layout_refptr ref_header = new_header(context);
    m_ref_layout = context->create_pango_layout();
    const double width = context->get_width();
    double height = context->get_height();
    m_ref_layout->set_width(static_cast<int>(width * Pango::SCALE));

    // Decrement remaining height to allow for header and footer height");
    int w;
    int h;
    ref_header->get_size(w, h);
    height -= static_cast<int>(h / Pango::SCALE);
    ref_footer->get_size(w, h);
    height -= static_cast<int>(h / Pango::SCALE);
    // Knock off a little more to allow for spacing between the body, header and footer
    height -= (HEADER_SPACING + FOOTER_SPACING + PAGE_TOLERANCE + BORDER_SPACING * 2);

    // Build up all the text to be printed on the page
    Glib::ustring page_text;
    for (ItemIter i = m_items->begin(); i != m_items->end(); ++i) {
        Item* item = *i;
        page_text += item->get_description() + "\n";
    }
    m_ref_layout->set_text(page_text);
    const int line_count = m_ref_layout->get_line_count();

    Glib::RefPtr<Pango::LayoutLine> layout_line;
    double page_height = 0;
    for (int line = 0; line < line_count; ++line) {
        Pango::Rectangle ink_rect, logical_rect;
        layout_line = m_ref_layout->get_line(line);
        layout_line->get_extents(ink_rect, logical_rect);
        const double line_height = logical_rect.get_height() / Pango::SCALE;

        if (page_height + line_height > height) {
            // Start a new page
            page_height = 0;
            // Save the current line number and header and footer elements
            m_page_breaks.push_back(line);
            ref_header = new_header(context);
            ref_footer = new_footer(context);
        }
        page_height += line_height;
    }

    const int n_pages = m_page_breaks.size() + 1;
    set_n_pages(n_pages);

    // rewrite page numbers in the footers
    for (int page_number = 0; page_number < n_pages; page_number++) {
        // page number x of n pages
#ifdef MAEMO
        std::ostringstream os;
        os << "Page " << page_number + 1 << " of " << n_pages;
        Glib::ustring footer = os.str();
#else
        Glib::ustring footer = Glib::ustring::compose(FOOTER_TEXT,
                                                      page_number + 1,
                                                      n_pages);
#endif
        m_ref_footers[page_number]->set_text(footer);
    }
}

/**
 * Prints a specified page.
 *
 * \param context the print context
 * \param page_number the number of the page to be printed
 */
void KitPrintOperation::on_draw_page(const Glib::RefPtr<Gtk::PrintContext>& context, int page_number) {
    int start_page_line = 0;
    int end_page_line = 0;

    if (page_number == 0) {
        start_page_line = 0;
    } else {
        start_page_line = m_page_breaks[page_number - 1];
    }

    if (page_number < static_cast<int>(m_page_breaks.size())) {
        end_page_line = m_page_breaks[page_number];
    } else {
        end_page_line = m_ref_layout->get_line_count();
    }

//     g_debug("Printing lines from %d to %d", start_page_line, end_page_line);
    Cairo::RefPtr<Cairo::Context> cairo_ctx = context->get_cairo_context();
    cairo_ctx->set_source_rgb(0, 0, 0);

    Pango::LayoutIter iter;
    double start_pos = 0;
    int line_index = 0;
    double header_height = 0;

    // Header
    layout_refptr ref_header = m_ref_headers[page_number];
    ref_header->get_iter(iter);
    do {
        Glib::RefPtr<Pango::LayoutLine> header_line = iter.get_line();
        Pango::Rectangle hr = iter.get_line_logical_extents();
        int bl = iter.get_baseline();
        double hx = hr.get_x() / Pango::SCALE;
        double hy = bl / Pango::SCALE;
        header_height = hy;
        cairo_ctx->move_to(hx, hy);
        header_line->show_in_cairo_context(cairo_ctx);
    } while (iter.next_line());

    // Horizontal line between header and body
    cairo_ctx->set_line_width(0.5);
    cairo_ctx->set_line_cap(Cairo::LINE_CAP_SQUARE);
    cairo_ctx->set_line_join(Cairo::LINE_JOIN_MITER);
    header_height += BORDER_SPACING;
    cairo_ctx->move_to(0.0, header_height);
    cairo_ctx->line_to(context->get_width(), header_height);
    cairo_ctx->stroke();
    header_height += HEADER_SPACING;

    // Page body
    m_ref_layout->get_iter(iter);
    do {
        if (line_index >= start_page_line) {
            Glib::RefPtr<Pango::LayoutLine> layout_line = iter.get_line();
            Pango::Rectangle logical_rect = iter.get_line_logical_extents();
            int baseline = iter.get_baseline();
            if (line_index == start_page_line) {
                start_pos = logical_rect.get_y() / Pango::SCALE;
            }
            double x = logical_rect.get_x() / Pango::SCALE;
            double y = baseline / Pango::SCALE - start_pos + header_height;
            cairo_ctx->move_to(x, y);
            layout_line->show_in_cairo_context(cairo_ctx);
        }
        line_index++;
    } while (line_index < end_page_line && iter.next_line());

    // Footer
    layout_refptr ref_footer = m_ref_footers[page_number];
    int w;
    int h;
    ref_footer->get_size(w, h);
    start_pos = context->get_height() - h / Pango::SCALE - BORDER_SPACING;

    // Horizontal line between body and footer
    cairo_ctx->set_source_rgb(0, 0, 0);
    cairo_ctx->set_line_width(0.5);
    cairo_ctx->set_line_cap(Cairo::LINE_CAP_SQUARE);
    cairo_ctx->set_line_join(Cairo::LINE_JOIN_MITER);
    cairo_ctx->move_to(0.0, start_pos);
    cairo_ctx->line_to(context->get_width(), start_pos);
    cairo_ctx->stroke();
    start_pos += BORDER_SPACING;

    ref_footer->get_iter(iter);
    do {
        Glib::RefPtr<Pango::LayoutLine> footer_line = iter.get_line();
        Pango::Rectangle fr = iter.get_line_logical_extents();
        int bl = iter.get_baseline();
        double fx = fr.get_x() / Pango::SCALE;
        double fy = bl / Pango::SCALE + start_pos;
        cairo_ctx->move_to(fx, fy);
        footer_line->show_in_cairo_context(cairo_ctx);
    } while (iter.next_line());

}
