/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008,2009 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef KIT_MODEL_H
#define KIT_MODEL_H 1

#include "item.hpp"
#include "category.hpp"
#include <map>
#include <sigc++/signal.h>


/**
 * \brief Class encapsulating state of an object in the data model.
 *
 * Provides additional flags to identityf whether the object is dirty,
 * has been deleted or is new.
 */
class GuiState {
 protected:
    bool m_dirty;
    bool m_deleted;
    bool m_new;
 public:
    GuiState() : m_dirty(false), m_deleted(false), m_new(false) {}
    bool is_dirty() { return m_dirty; }
    void set_dirty(bool dirty = true) { m_dirty = dirty; }
    bool is_deleted() { return m_deleted; }
    void set_deleted(bool deleted) { m_deleted = deleted; }
    void set_new_flag(bool flag) { m_new = flag; }
    bool is_new() { return m_new; }
    virtual void reset();
};

class ItemCompareId;


/**
 * \brief Represents an Item combined with GuiState attributes.
 */
class ModelItem : public Item, public GuiState {
 public:
    ModelItem() : Item(), GuiState() {}
    friend class ModelItemCompareId;
    virtual void set_checked(bool checked) { set_dirty(); Item::set_checked(checked); }
};


/**
 * \brief Comparator for comparing items by their unique ID.
 * \see Item
 */
class ModelItemCompareId {
 public:
    ModelItemCompareId() {}
    int operator()(ModelItem* i1, ModelItem* i2) {
        return (i1->get_id() < i2->get_id()); 
    }
};


typedef std::vector<ModelItem*> ModelItemContainer;
typedef ModelItemContainer::iterator ModelItemIter;

typedef std::map<long, ModelItem*> ItemMap;
typedef ItemMap::iterator ItemMapIter;

typedef sigc::slot<bool, const ItemMap::iterator&> SlotForeachModelItemIter;
typedef sigc::slot<bool, ModelItem&> SlotForeachModelItem;


/**
 * \brief Represents a Category combined with GuiState attributes.
 */
class ModelCategory : public Category, public GuiState {
 protected:
    /// List of items removed from the Category
    ItemMap* m_removed_children;
    /// List of items added to the Category
    ItemMap* m_added_children;
 public:
    ModelCategory();
    ~ModelCategory();
    virtual ModelItemContainer* get_model_items();
    virtual ItemContainer* get_items();
    virtual ItemContainer* get_items(ItemFunctor& functor);
    /// Returns a list of items that have been removed from the Category
    virtual ItemMap* get_removed_children() { return m_removed_children; }
    /// Returns a list of items that have been added to the Category
    virtual ItemMap* get_added_children() { return m_added_children; }
    virtual void add_item(Item*);
    virtual void remove_item(Item* item);
    virtual void remove_items(ModelItemContainer* items);
    virtual void reset();
    virtual void purge();
    friend class KitModel;
};


typedef std::vector<ModelCategory*> ModelCategoryContainer;
typedef ModelCategoryContainer::iterator ModelCategoryIter;

typedef std::map<long, ModelCategory*> CategoryMap;
typedef CategoryMap::iterator CategoryMapIter;

typedef sigc::slot<bool, const CategoryMap::iterator&> SlotForeachCategoryIter;
typedef sigc::slot<bool, ModelCategory&> SlotForeachCategory;

enum item_filter_types {ALL, CHECKED, UNCHECKED};


/**
 * \brief Holds a rich graph of objects representing the application's
 * data model.
 */
class KitModel {
 protected:
    /// Indicates whether the model needs saving. I.e it's state has changed.
    bool m_dirty;
    /**
     * \brief Map allowing categories to be located by ID.
     *
     * The map's key is the category ID, with the second field of the pair
     * containing a pointer to the ModelCategory instance.
     */
    CategoryMap* m_category_map;
    /**
     * \brief Map allowing items to be located by ID.
     *
     * The map's key is the item ID, with the second field of the pair
     * containing a pointer to the ModelItem instance.
     */
    ItemMap* m_item_map;
    /**
     * \brief Indicates whether and how items are currently filtered.
     *
     * One of ALL, CHECKED or UNCHECKED.
     */
    enum item_filter_types m_item_filter;
 public:
    KitModel();
    ~KitModel();
    void foreach_item_iter(const SlotForeachModelItemIter& slot);
    void foreach_item(const SlotForeachModelItem& slot);
    void foreach_category_iter(const SlotForeachCategoryIter& slot);
    void foreach_category(const SlotForeachCategory& slot);
    virtual ModelCategory* find_category(long id);
    virtual ModelItem* find_item(long id);
    virtual CategoryContainer* get_categories();
    virtual ItemContainer* get_all_items();
    virtual ItemContainer* get_all_items(ItemFunctor& functor);
    virtual void add_category(ModelCategory* category);
    virtual void add_item(ModelItem* item);
    virtual void add_item(ModelItem* item, long cat_id);
    virtual void copy_items(const ModelItemContainer& items, long cat_id = -1);
    virtual bool filter(bool checked);
    virtual void set_dirty(bool dirty = true) { m_dirty=dirty; }
    virtual bool is_dirty() { return m_dirty; }
    /// Removes filter.  All items are shown.
    virtual void show_all() { m_item_filter = ALL; }
    /// Sets the filter to show only checked items.
    virtual void show_checked_only() { m_item_filter = CHECKED; }
    /// Sets the filter to show only unchecked items.
    virtual void show_unchecked_only() { m_item_filter = UNCHECKED; }
    virtual void reset();
    virtual void purge();
};


#endif // KIT_MODEL_H
