/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008,2009 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/

#include <algorithm>
// #include <iterator>
#include "category.hpp"


/// Associates the passed item with this Category.
void Category::add_item(Item* item) {
    m_items.push_back(item);
}


/**
 * \brief Removes the association of the passed item from this Category.
 *
 * The passed item is not deleted.
 * \param item the item to un-associate.
 */
void Category::remove_item(Item* item) {
    ItemIter i = std::find(m_items.begin(), m_items.end(), item);
    if (i != m_items.end()) {
        m_items.erase(i);
    }
}


/**
 * \brief Executes a callback function for each associated item.
 *
 * The callback function will be passed a reference to the current
 * item being iterated.
 * \param slot the callback function.
 */
void Category::foreach_item(const SlotForeachItem& slot) {
    for (ItemIter i = m_items.begin(); i != m_items.end(); ++i) {
        if (slot(*(*i)))
            break;
    }
}


/**
 * \brief Executes the passed ItemFunctor.
 *
 * The ItemFunctor's override operator() method is called, passing a
 * reference to the item being iterated over.  If the called method
 * returns true, the iteration stops and no more calls will be made to
 * the functor.
 */
void Category::execute(ItemFunctor& functor) {
    for (ItemIter i = m_items.begin(); i != m_items.end(); ++i) {
        if (functor(**i))
            break;
    }    
}
