"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.configureClient = void 0;
exports.getErrorMessage = getErrorMessage;
exports.getRequestDebugMeta = getRequestDebugMeta;

var _buffer = require("buffer");

var _querystring = require("querystring");

var _elasticsearch = require("@elastic/elasticsearch");

var _client_config = require("./client_config");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
const noop = () => undefined;

const configureClient = (config, {
  logger,
  type,
  scoped = false,
  getExecutionContext = noop
}) => {
  const clientOptions = (0, _client_config.parseClientOptions)(config, scoped);

  class KibanaTransport extends _elasticsearch.Transport {
    request(params, options) {
      const opts = options || {};
      const opaqueId = getExecutionContext();

      if (opaqueId && !opts.opaqueId) {
        // rewrites headers['x-opaque-id'] if it presents
        opts.opaqueId = opaqueId;
      }

      return super.request(params, opts);
    }

  }

  const client = new _elasticsearch.Client({ ...clientOptions,
    Transport: KibanaTransport
  });
  addLogging(client, logger.get('query', type)); // --------------------------------------------------------------------------------- //
  // Hack to disable the "Product check" only in the scoped clients while we           //
  // come up with a better approach in https://github.com/elastic/kibana/issues/110675 //

  if (scoped) skipProductCheck(client); // --------------------------------------------------------------------------------- //

  return client;
};

exports.configureClient = configureClient;

const convertQueryString = qs => {
  if (qs === undefined || typeof qs === 'string') {
    return qs !== null && qs !== void 0 ? qs : '';
  }

  return (0, _querystring.stringify)(qs);
};

function ensureString(body) {
  if (typeof body === 'string') return body;
  if (_buffer.Buffer.isBuffer(body)) return '[buffer]';
  if ('readable' in body && body.readable && typeof body._read === 'function') return '[stream]';
  return JSON.stringify(body);
}
/**
 * Returns a debug message from an Elasticsearch error in the following format:
 * [error type] error reason
 */


function getErrorMessage(error) {
  if (error instanceof _elasticsearch.errors.ResponseError) {
    var _error$meta$body, _error$meta$body$erro, _error$meta$body$erro2, _error$meta$body2, _error$meta$body2$err;

    return `[${(_error$meta$body = error.meta.body) === null || _error$meta$body === void 0 ? void 0 : (_error$meta$body$erro = _error$meta$body.error) === null || _error$meta$body$erro === void 0 ? void 0 : _error$meta$body$erro.type}]: ${(_error$meta$body$erro2 = (_error$meta$body2 = error.meta.body) === null || _error$meta$body2 === void 0 ? void 0 : (_error$meta$body2$err = _error$meta$body2.error) === null || _error$meta$body2$err === void 0 ? void 0 : _error$meta$body2$err.reason) !== null && _error$meta$body$erro2 !== void 0 ? _error$meta$body$erro2 : error.message}`;
  }

  return `[${error.name}]: ${error.message}`;
}
/**
 * returns a string in format:
 *
 * status code
 * method URL
 * request body
 *
 * so it could be copy-pasted into the Dev console
 */


function getResponseMessage(event) {
  const errorMeta = getRequestDebugMeta(event);
  const body = errorMeta.body ? `\n${errorMeta.body}` : '';
  return `${errorMeta.statusCode}\n${errorMeta.method} ${errorMeta.url}${body}`;
}
/**
 * Returns stringified debug information from an Elasticsearch request event
 * useful for logging in case of an unexpected failure.
 */


function getRequestDebugMeta(event) {
  const params = event.meta.request.params; // definition is wrong, `params.querystring` can be either a string or an object

  const querystring = convertQueryString(params.querystring);
  return {
    url: `${params.path}${querystring ? `?${querystring}` : ''}`,
    body: params.body ? `${ensureString(params.body)}` : '',
    method: params.method,
    statusCode: event.statusCode
  };
}

const addLogging = (client, logger) => {
  client.on('response', (error, event) => {
    if (event) {
      const opaqueId = event.meta.request.options.opaqueId;
      const meta = opaqueId ? {
        http: {
          request: {
            id: event.meta.request.options.opaqueId
          }
        }
      } : undefined; // do not clutter logs if opaqueId is not present

      if (error) {
        if (error instanceof _elasticsearch.errors.ResponseError) {
          logger.debug(`${getResponseMessage(event)} ${getErrorMessage(error)}`, meta);
        } else {
          logger.debug(getErrorMessage(error), meta);
        }
      } else {
        logger.debug(getResponseMessage(event), meta);
      }
    }
  });
};
/**
 * Hack to skip the Product Check performed by the Elasticsearch-js client.
 * We noticed that the scoped clients are always performing this check because
 * of the way we initialize the clients. We'll discuss changing this in the issue
 * https://github.com/elastic/kibana/issues/110675. In the meanwhile, let's skip
 * it for the scoped clients.
 *
 * The hack is copied from the test/utils in the elasticsearch-js repo
 * (https://github.com/elastic/elasticsearch-js/blob/master/test/utils/index.js#L45-L56)
 */


function skipProductCheck(client) {
  const tSymbol = Object.getOwnPropertySymbols(client.transport || client).filter(symbol => symbol.description === 'product check')[0]; // @ts-expect-error `tSymbol` is missing in the index signature of Transport

  (client.transport || client)[tSymbol] = 2;
}