(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/parser/cssScanner',["require", "exports"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.Scanner = exports.MultiLineStream = exports.TokenType = void 0;
    var TokenType;
    (function (TokenType) {
        TokenType[TokenType["Ident"] = 0] = "Ident";
        TokenType[TokenType["AtKeyword"] = 1] = "AtKeyword";
        TokenType[TokenType["String"] = 2] = "String";
        TokenType[TokenType["BadString"] = 3] = "BadString";
        TokenType[TokenType["UnquotedString"] = 4] = "UnquotedString";
        TokenType[TokenType["Hash"] = 5] = "Hash";
        TokenType[TokenType["Num"] = 6] = "Num";
        TokenType[TokenType["Percentage"] = 7] = "Percentage";
        TokenType[TokenType["Dimension"] = 8] = "Dimension";
        TokenType[TokenType["UnicodeRange"] = 9] = "UnicodeRange";
        TokenType[TokenType["CDO"] = 10] = "CDO";
        TokenType[TokenType["CDC"] = 11] = "CDC";
        TokenType[TokenType["Colon"] = 12] = "Colon";
        TokenType[TokenType["SemiColon"] = 13] = "SemiColon";
        TokenType[TokenType["CurlyL"] = 14] = "CurlyL";
        TokenType[TokenType["CurlyR"] = 15] = "CurlyR";
        TokenType[TokenType["ParenthesisL"] = 16] = "ParenthesisL";
        TokenType[TokenType["ParenthesisR"] = 17] = "ParenthesisR";
        TokenType[TokenType["BracketL"] = 18] = "BracketL";
        TokenType[TokenType["BracketR"] = 19] = "BracketR";
        TokenType[TokenType["Whitespace"] = 20] = "Whitespace";
        TokenType[TokenType["Includes"] = 21] = "Includes";
        TokenType[TokenType["Dashmatch"] = 22] = "Dashmatch";
        TokenType[TokenType["SubstringOperator"] = 23] = "SubstringOperator";
        TokenType[TokenType["PrefixOperator"] = 24] = "PrefixOperator";
        TokenType[TokenType["SuffixOperator"] = 25] = "SuffixOperator";
        TokenType[TokenType["Delim"] = 26] = "Delim";
        TokenType[TokenType["EMS"] = 27] = "EMS";
        TokenType[TokenType["EXS"] = 28] = "EXS";
        TokenType[TokenType["Length"] = 29] = "Length";
        TokenType[TokenType["Angle"] = 30] = "Angle";
        TokenType[TokenType["Time"] = 31] = "Time";
        TokenType[TokenType["Freq"] = 32] = "Freq";
        TokenType[TokenType["Exclamation"] = 33] = "Exclamation";
        TokenType[TokenType["Resolution"] = 34] = "Resolution";
        TokenType[TokenType["Comma"] = 35] = "Comma";
        TokenType[TokenType["Charset"] = 36] = "Charset";
        TokenType[TokenType["EscapedJavaScript"] = 37] = "EscapedJavaScript";
        TokenType[TokenType["BadEscapedJavaScript"] = 38] = "BadEscapedJavaScript";
        TokenType[TokenType["Comment"] = 39] = "Comment";
        TokenType[TokenType["SingleLineComment"] = 40] = "SingleLineComment";
        TokenType[TokenType["EOF"] = 41] = "EOF";
        TokenType[TokenType["CustomToken"] = 42] = "CustomToken";
    })(TokenType = exports.TokenType || (exports.TokenType = {}));
    var MultiLineStream = /** @class */ (function () {
        function MultiLineStream(source) {
            this.source = source;
            this.len = source.length;
            this.position = 0;
        }
        MultiLineStream.prototype.substring = function (from, to) {
            if (to === void 0) { to = this.position; }
            return this.source.substring(from, to);
        };
        MultiLineStream.prototype.eos = function () {
            return this.len <= this.position;
        };
        MultiLineStream.prototype.pos = function () {
            return this.position;
        };
        MultiLineStream.prototype.goBackTo = function (pos) {
            this.position = pos;
        };
        MultiLineStream.prototype.goBack = function (n) {
            this.position -= n;
        };
        MultiLineStream.prototype.advance = function (n) {
            this.position += n;
        };
        MultiLineStream.prototype.nextChar = function () {
            return this.source.charCodeAt(this.position++) || 0;
        };
        MultiLineStream.prototype.peekChar = function (n) {
            if (n === void 0) { n = 0; }
            return this.source.charCodeAt(this.position + n) || 0;
        };
        MultiLineStream.prototype.lookbackChar = function (n) {
            if (n === void 0) { n = 0; }
            return this.source.charCodeAt(this.position - n) || 0;
        };
        MultiLineStream.prototype.advanceIfChar = function (ch) {
            if (ch === this.source.charCodeAt(this.position)) {
                this.position++;
                return true;
            }
            return false;
        };
        MultiLineStream.prototype.advanceIfChars = function (ch) {
            if (this.position + ch.length > this.source.length) {
                return false;
            }
            var i = 0;
            for (; i < ch.length; i++) {
                if (this.source.charCodeAt(this.position + i) !== ch[i]) {
                    return false;
                }
            }
            this.advance(i);
            return true;
        };
        MultiLineStream.prototype.advanceWhileChar = function (condition) {
            var posNow = this.position;
            while (this.position < this.len && condition(this.source.charCodeAt(this.position))) {
                this.position++;
            }
            return this.position - posNow;
        };
        return MultiLineStream;
    }());
    exports.MultiLineStream = MultiLineStream;
    var _a = 'a'.charCodeAt(0);
    var _f = 'f'.charCodeAt(0);
    var _z = 'z'.charCodeAt(0);
    var _A = 'A'.charCodeAt(0);
    var _F = 'F'.charCodeAt(0);
    var _Z = 'Z'.charCodeAt(0);
    var _0 = '0'.charCodeAt(0);
    var _9 = '9'.charCodeAt(0);
    var _TLD = '~'.charCodeAt(0);
    var _HAT = '^'.charCodeAt(0);
    var _EQS = '='.charCodeAt(0);
    var _PIP = '|'.charCodeAt(0);
    var _MIN = '-'.charCodeAt(0);
    var _USC = '_'.charCodeAt(0);
    var _PRC = '%'.charCodeAt(0);
    var _MUL = '*'.charCodeAt(0);
    var _LPA = '('.charCodeAt(0);
    var _RPA = ')'.charCodeAt(0);
    var _LAN = '<'.charCodeAt(0);
    var _RAN = '>'.charCodeAt(0);
    var _ATS = '@'.charCodeAt(0);
    var _HSH = '#'.charCodeAt(0);
    var _DLR = '$'.charCodeAt(0);
    var _BSL = '\\'.charCodeAt(0);
    var _FSL = '/'.charCodeAt(0);
    var _NWL = '\n'.charCodeAt(0);
    var _CAR = '\r'.charCodeAt(0);
    var _LFD = '\f'.charCodeAt(0);
    var _DQO = '"'.charCodeAt(0);
    var _SQO = '\''.charCodeAt(0);
    var _WSP = ' '.charCodeAt(0);
    var _TAB = '\t'.charCodeAt(0);
    var _SEM = ';'.charCodeAt(0);
    var _COL = ':'.charCodeAt(0);
    var _CUL = '{'.charCodeAt(0);
    var _CUR = '}'.charCodeAt(0);
    var _BRL = '['.charCodeAt(0);
    var _BRR = ']'.charCodeAt(0);
    var _CMA = ','.charCodeAt(0);
    var _DOT = '.'.charCodeAt(0);
    var _BNG = '!'.charCodeAt(0);
    var staticTokenTable = {};
    staticTokenTable[_SEM] = TokenType.SemiColon;
    staticTokenTable[_COL] = TokenType.Colon;
    staticTokenTable[_CUL] = TokenType.CurlyL;
    staticTokenTable[_CUR] = TokenType.CurlyR;
    staticTokenTable[_BRR] = TokenType.BracketR;
    staticTokenTable[_BRL] = TokenType.BracketL;
    staticTokenTable[_LPA] = TokenType.ParenthesisL;
    staticTokenTable[_RPA] = TokenType.ParenthesisR;
    staticTokenTable[_CMA] = TokenType.Comma;
    var staticUnitTable = {};
    staticUnitTable['em'] = TokenType.EMS;
    staticUnitTable['ex'] = TokenType.EXS;
    staticUnitTable['px'] = TokenType.Length;
    staticUnitTable['cm'] = TokenType.Length;
    staticUnitTable['mm'] = TokenType.Length;
    staticUnitTable['in'] = TokenType.Length;
    staticUnitTable['pt'] = TokenType.Length;
    staticUnitTable['pc'] = TokenType.Length;
    staticUnitTable['deg'] = TokenType.Angle;
    staticUnitTable['rad'] = TokenType.Angle;
    staticUnitTable['grad'] = TokenType.Angle;
    staticUnitTable['ms'] = TokenType.Time;
    staticUnitTable['s'] = TokenType.Time;
    staticUnitTable['hz'] = TokenType.Freq;
    staticUnitTable['khz'] = TokenType.Freq;
    staticUnitTable['%'] = TokenType.Percentage;
    staticUnitTable['fr'] = TokenType.Percentage;
    staticUnitTable['dpi'] = TokenType.Resolution;
    staticUnitTable['dpcm'] = TokenType.Resolution;
    var Scanner = /** @class */ (function () {
        function Scanner() {
            this.stream = new MultiLineStream('');
            this.ignoreComment = true;
            this.ignoreWhitespace = true;
            this.inURL = false;
        }
        Scanner.prototype.setSource = function (input) {
            this.stream = new MultiLineStream(input);
        };
        Scanner.prototype.finishToken = function (offset, type, text) {
            return {
                offset: offset,
                len: this.stream.pos() - offset,
                type: type,
                text: text || this.stream.substring(offset)
            };
        };
        Scanner.prototype.substring = function (offset, len) {
            return this.stream.substring(offset, offset + len);
        };
        Scanner.prototype.pos = function () {
            return this.stream.pos();
        };
        Scanner.prototype.goBackTo = function (pos) {
            this.stream.goBackTo(pos);
        };
        Scanner.prototype.scanUnquotedString = function () {
            var offset = this.stream.pos();
            var content = [];
            if (this._unquotedString(content)) {
                return this.finishToken(offset, TokenType.UnquotedString, content.join(''));
            }
            return null;
        };
        Scanner.prototype.scan = function () {
            // processes all whitespaces and comments
            var triviaToken = this.trivia();
            if (triviaToken !== null) {
                return triviaToken;
            }
            var offset = this.stream.pos();
            // End of file/input
            if (this.stream.eos()) {
                return this.finishToken(offset, TokenType.EOF);
            }
            return this.scanNext(offset);
        };
        Scanner.prototype.scanNext = function (offset) {
            // CDO <!--
            if (this.stream.advanceIfChars([_LAN, _BNG, _MIN, _MIN])) {
                return this.finishToken(offset, TokenType.CDO);
            }
            // CDC -->
            if (this.stream.advanceIfChars([_MIN, _MIN, _RAN])) {
                return this.finishToken(offset, TokenType.CDC);
            }
            var content = [];
            if (this.ident(content)) {
                return this.finishToken(offset, TokenType.Ident, content.join(''));
            }
            // at-keyword
            if (this.stream.advanceIfChar(_ATS)) {
                content = ['@'];
                if (this._name(content)) {
                    var keywordText = content.join('');
                    if (keywordText === '@charset') {
                        return this.finishToken(offset, TokenType.Charset, keywordText);
                    }
                    return this.finishToken(offset, TokenType.AtKeyword, keywordText);
                }
                else {
                    return this.finishToken(offset, TokenType.Delim);
                }
            }
            // hash
            if (this.stream.advanceIfChar(_HSH)) {
                content = ['#'];
                if (this._name(content)) {
                    return this.finishToken(offset, TokenType.Hash, content.join(''));
                }
                else {
                    return this.finishToken(offset, TokenType.Delim);
                }
            }
            // Important
            if (this.stream.advanceIfChar(_BNG)) {
                return this.finishToken(offset, TokenType.Exclamation);
            }
            // Numbers
            if (this._number()) {
                var pos = this.stream.pos();
                content = [this.stream.substring(offset, pos)];
                if (this.stream.advanceIfChar(_PRC)) {
                    // Percentage 43%
                    return this.finishToken(offset, TokenType.Percentage);
                }
                else if (this.ident(content)) {
                    var dim = this.stream.substring(pos).toLowerCase();
                    var tokenType_1 = staticUnitTable[dim];
                    if (typeof tokenType_1 !== 'undefined') {
                        // Known dimension 43px
                        return this.finishToken(offset, tokenType_1, content.join(''));
                    }
                    else {
                        // Unknown dimension 43ft
                        return this.finishToken(offset, TokenType.Dimension, content.join(''));
                    }
                }
                return this.finishToken(offset, TokenType.Num);
            }
            // String, BadString
            content = [];
            var tokenType = this._string(content);
            if (tokenType !== null) {
                return this.finishToken(offset, tokenType, content.join(''));
            }
            // single character tokens
            tokenType = staticTokenTable[this.stream.peekChar()];
            if (typeof tokenType !== 'undefined') {
                this.stream.advance(1);
                return this.finishToken(offset, tokenType);
            }
            // includes ~=
            if (this.stream.peekChar(0) === _TLD && this.stream.peekChar(1) === _EQS) {
                this.stream.advance(2);
                return this.finishToken(offset, TokenType.Includes);
            }
            // DashMatch |=
            if (this.stream.peekChar(0) === _PIP && this.stream.peekChar(1) === _EQS) {
                this.stream.advance(2);
                return this.finishToken(offset, TokenType.Dashmatch);
            }
            // Substring operator *=
            if (this.stream.peekChar(0) === _MUL && this.stream.peekChar(1) === _EQS) {
                this.stream.advance(2);
                return this.finishToken(offset, TokenType.SubstringOperator);
            }
            // Substring operator ^=
            if (this.stream.peekChar(0) === _HAT && this.stream.peekChar(1) === _EQS) {
                this.stream.advance(2);
                return this.finishToken(offset, TokenType.PrefixOperator);
            }
            // Substring operator $=
            if (this.stream.peekChar(0) === _DLR && this.stream.peekChar(1) === _EQS) {
                this.stream.advance(2);
                return this.finishToken(offset, TokenType.SuffixOperator);
            }
            // Delim
            this.stream.nextChar();
            return this.finishToken(offset, TokenType.Delim);
        };
        Scanner.prototype.trivia = function () {
            while (true) {
                var offset = this.stream.pos();
                if (this._whitespace()) {
                    if (!this.ignoreWhitespace) {
                        return this.finishToken(offset, TokenType.Whitespace);
                    }
                }
                else if (this.comment()) {
                    if (!this.ignoreComment) {
                        return this.finishToken(offset, TokenType.Comment);
                    }
                }
                else {
                    return null;
                }
            }
        };
        Scanner.prototype.comment = function () {
            if (this.stream.advanceIfChars([_FSL, _MUL])) {
                var success_1 = false, hot_1 = false;
                this.stream.advanceWhileChar(function (ch) {
                    if (hot_1 && ch === _FSL) {
                        success_1 = true;
                        return false;
                    }
                    hot_1 = ch === _MUL;
                    return true;
                });
                if (success_1) {
                    this.stream.advance(1);
                }
                return true;
            }
            return false;
        };
        Scanner.prototype._number = function () {
            var npeek = 0, ch;
            if (this.stream.peekChar() === _DOT) {
                npeek = 1;
            }
            ch = this.stream.peekChar(npeek);
            if (ch >= _0 && ch <= _9) {
                this.stream.advance(npeek + 1);
                this.stream.advanceWhileChar(function (ch) {
                    return ch >= _0 && ch <= _9 || npeek === 0 && ch === _DOT;
                });
                return true;
            }
            return false;
        };
        Scanner.prototype._newline = function (result) {
            var ch = this.stream.peekChar();
            switch (ch) {
                case _CAR:
                case _LFD:
                case _NWL:
                    this.stream.advance(1);
                    result.push(String.fromCharCode(ch));
                    if (ch === _CAR && this.stream.advanceIfChar(_NWL)) {
                        result.push('\n');
                    }
                    return true;
            }
            return false;
        };
        Scanner.prototype._escape = function (result, includeNewLines) {
            var ch = this.stream.peekChar();
            if (ch === _BSL) {
                this.stream.advance(1);
                ch = this.stream.peekChar();
                var hexNumCount = 0;
                while (hexNumCount < 6 && (ch >= _0 && ch <= _9 || ch >= _a && ch <= _f || ch >= _A && ch <= _F)) {
                    this.stream.advance(1);
                    ch = this.stream.peekChar();
                    hexNumCount++;
                }
                if (hexNumCount > 0) {
                    try {
                        var hexVal = parseInt(this.stream.substring(this.stream.pos() - hexNumCount), 16);
                        if (hexVal) {
                            result.push(String.fromCharCode(hexVal));
                        }
                    }
                    catch (e) {
                        // ignore
                    }
                    // optional whitespace or new line, not part of result text
                    if (ch === _WSP || ch === _TAB) {
                        this.stream.advance(1);
                    }
                    else {
                        this._newline([]);
                    }
                    return true;
                }
                if (ch !== _CAR && ch !== _LFD && ch !== _NWL) {
                    this.stream.advance(1);
                    result.push(String.fromCharCode(ch));
                    return true;
                }
                else if (includeNewLines) {
                    return this._newline(result);
                }
            }
            return false;
        };
        Scanner.prototype._stringChar = function (closeQuote, result) {
            // not closeQuote, not backslash, not newline
            var ch = this.stream.peekChar();
            if (ch !== 0 && ch !== closeQuote && ch !== _BSL && ch !== _CAR && ch !== _LFD && ch !== _NWL) {
                this.stream.advance(1);
                result.push(String.fromCharCode(ch));
                return true;
            }
            return false;
        };
        Scanner.prototype._string = function (result) {
            if (this.stream.peekChar() === _SQO || this.stream.peekChar() === _DQO) {
                var closeQuote = this.stream.nextChar();
                result.push(String.fromCharCode(closeQuote));
                while (this._stringChar(closeQuote, result) || this._escape(result, true)) {
                    // loop
                }
                if (this.stream.peekChar() === closeQuote) {
                    this.stream.nextChar();
                    result.push(String.fromCharCode(closeQuote));
                    return TokenType.String;
                }
                else {
                    return TokenType.BadString;
                }
            }
            return null;
        };
        Scanner.prototype._unquotedChar = function (result) {
            // not closeQuote, not backslash, not newline
            var ch = this.stream.peekChar();
            if (ch !== 0 && ch !== _BSL && ch !== _SQO && ch !== _DQO && ch !== _LPA && ch !== _RPA && ch !== _WSP && ch !== _TAB && ch !== _NWL && ch !== _LFD && ch !== _CAR) {
                this.stream.advance(1);
                result.push(String.fromCharCode(ch));
                return true;
            }
            return false;
        };
        Scanner.prototype._unquotedString = function (result) {
            var hasContent = false;
            while (this._unquotedChar(result) || this._escape(result)) {
                hasContent = true;
            }
            return hasContent;
        };
        Scanner.prototype._whitespace = function () {
            var n = this.stream.advanceWhileChar(function (ch) {
                return ch === _WSP || ch === _TAB || ch === _NWL || ch === _LFD || ch === _CAR;
            });
            return n > 0;
        };
        Scanner.prototype._name = function (result) {
            var matched = false;
            while (this._identChar(result) || this._escape(result)) {
                matched = true;
            }
            return matched;
        };
        Scanner.prototype.ident = function (result) {
            var pos = this.stream.pos();
            var hasMinus = this._minus(result);
            if (hasMinus && this._minus(result) /* -- */) {
                if (this._identFirstChar(result) || this._escape(result)) {
                    while (this._identChar(result) || this._escape(result)) {
                        // loop
                    }
                    return true;
                }
            }
            else if (this._identFirstChar(result) || this._escape(result)) {
                while (this._identChar(result) || this._escape(result)) {
                    // loop
                }
                return true;
            }
            this.stream.goBackTo(pos);
            return false;
        };
        Scanner.prototype._identFirstChar = function (result) {
            var ch = this.stream.peekChar();
            if (ch === _USC || // _
                ch >= _a && ch <= _z || // a-z
                ch >= _A && ch <= _Z || // A-Z
                ch >= 0x80 && ch <= 0xFFFF) { // nonascii
                this.stream.advance(1);
                result.push(String.fromCharCode(ch));
                return true;
            }
            return false;
        };
        Scanner.prototype._minus = function (result) {
            var ch = this.stream.peekChar();
            if (ch === _MIN) {
                this.stream.advance(1);
                result.push(String.fromCharCode(ch));
                return true;
            }
            return false;
        };
        Scanner.prototype._identChar = function (result) {
            var ch = this.stream.peekChar();
            if (ch === _USC || // _
                ch === _MIN || // -
                ch >= _a && ch <= _z || // a-z
                ch >= _A && ch <= _Z || // A-Z
                ch >= _0 && ch <= _9 || // 0/9
                ch >= 0x80 && ch <= 0xFFFF) { // nonascii
                this.stream.advance(1);
                result.push(String.fromCharCode(ch));
                return true;
            }
            return false;
        };
        return Scanner;
    }());
    exports.Scanner = Scanner;
});

(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/utils/strings',["require", "exports"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.trim = exports.getLimitedString = exports.difference = exports.endsWith = exports.startsWith = void 0;
    function startsWith(haystack, needle) {
        if (haystack.length < needle.length) {
            return false;
        }
        for (var i = 0; i < needle.length; i++) {
            if (haystack[i] !== needle[i]) {
                return false;
            }
        }
        return true;
    }
    exports.startsWith = startsWith;
    /**
     * Determines if haystack ends with needle.
     */
    function endsWith(haystack, needle) {
        var diff = haystack.length - needle.length;
        if (diff > 0) {
            return haystack.lastIndexOf(needle) === diff;
        }
        else if (diff === 0) {
            return haystack === needle;
        }
        else {
            return false;
        }
    }
    exports.endsWith = endsWith;
    /**
     * Computes the difference score for two strings. More similar strings have a higher score.
     * We use largest common subsequence dynamic programming approach but penalize in the end for length differences.
     * Strings that have a large length difference will get a bad default score 0.
     * Complexity - both time and space O(first.length * second.length)
     * Dynamic programming LCS computation http://en.wikipedia.org/wiki/Longest_common_subsequence_problem
     *
     * @param first a string
     * @param second a string
     */
    function difference(first, second, maxLenDelta) {
        if (maxLenDelta === void 0) { maxLenDelta = 4; }
        var lengthDifference = Math.abs(first.length - second.length);
        // We only compute score if length of the currentWord and length of entry.name are similar.
        if (lengthDifference > maxLenDelta) {
            return 0;
        }
        // Initialize LCS (largest common subsequence) matrix.
        var LCS = [];
        var zeroArray = [];
        var i, j;
        for (i = 0; i < second.length + 1; ++i) {
            zeroArray.push(0);
        }
        for (i = 0; i < first.length + 1; ++i) {
            LCS.push(zeroArray);
        }
        for (i = 1; i < first.length + 1; ++i) {
            for (j = 1; j < second.length + 1; ++j) {
                if (first[i - 1] === second[j - 1]) {
                    LCS[i][j] = LCS[i - 1][j - 1] + 1;
                }
                else {
                    LCS[i][j] = Math.max(LCS[i - 1][j], LCS[i][j - 1]);
                }
            }
        }
        return LCS[first.length][second.length] - Math.sqrt(lengthDifference);
    }
    exports.difference = difference;
    /**
     * Limit of string length.
     */
    function getLimitedString(str, ellipsis) {
        if (ellipsis === void 0) { ellipsis = true; }
        if (!str) {
            return '';
        }
        if (str.length < 140) {
            return str;
        }
        return str.slice(0, 140) + (ellipsis ? '\u2026' : '');
    }
    exports.getLimitedString = getLimitedString;
    /**
     * Limit of string length.
     */
    function trim(str, regexp) {
        var m = regexp.exec(str);
        if (m && m[0].length) {
            return str.substr(0, str.length - m[0].length);
        }
        return str;
    }
    exports.trim = trim;
});

var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/parser/cssNodes',["require", "exports", "../utils/strings"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.ParseErrorCollector = exports.Marker = exports.Level = exports.Module = exports.GuardCondition = exports.LessGuard = exports.ListEntry = exports.UnknownAtRule = exports.MixinDeclaration = exports.MixinReference = exports.MixinContentDeclaration = exports.MixinContentReference = exports.ExtendsReference = exports.Variable = exports.Interpolation = exports.VariableDeclaration = exports.NumericValue = exports.HexColorValue = exports.Operator = exports.AttributeSelector = exports.Term = exports.BinaryExpression = exports.Expression = exports.PageBoxMarginBox = exports.Page = exports.SupportsCondition = exports.MediaQuery = exports.Medialist = exports.Document = exports.Supports = exports.Media = exports.Namespace = exports.ForwardVisibility = exports.Forward = exports.ModuleConfiguration = exports.Use = exports.Import = exports.KeyframeSelector = exports.Keyframe = exports.NestedProperties = exports.FontFace = exports.ViewPort = exports.FunctionDeclaration = exports.ElseStatement = exports.WhileStatement = exports.EachStatement = exports.ForStatement = exports.IfStatement = exports.FunctionArgument = exports.FunctionParameter = exports.Function = exports.Invocation = exports.Property = exports.CustomPropertyDeclaration = exports.Declaration = exports.CustomPropertySet = exports.AbstractDeclaration = exports.AtApplyRule = exports.SimpleSelector = exports.Selector = exports.RuleSet = exports.BodyDeclaration = exports.Declarations = exports.Stylesheet = exports.Identifier = exports.Nodelist = exports.Node = exports.getParentDeclaration = exports.getNodePath = exports.getNodeAtOffset = exports.ReferenceType = exports.NodeType = void 0;
    var strings_1 = require("../utils/strings");
    /// <summary>
    /// Nodes for the css 2.1 specification. See for reference:
    /// http://www.w3.org/TR/CSS21/grammar.html#grammar
    /// </summary>
    var NodeType;
    (function (NodeType) {
        NodeType[NodeType["Undefined"] = 0] = "Undefined";
        NodeType[NodeType["Identifier"] = 1] = "Identifier";
        NodeType[NodeType["Stylesheet"] = 2] = "Stylesheet";
        NodeType[NodeType["Ruleset"] = 3] = "Ruleset";
        NodeType[NodeType["Selector"] = 4] = "Selector";
        NodeType[NodeType["SimpleSelector"] = 5] = "SimpleSelector";
        NodeType[NodeType["SelectorInterpolation"] = 6] = "SelectorInterpolation";
        NodeType[NodeType["SelectorCombinator"] = 7] = "SelectorCombinator";
        NodeType[NodeType["SelectorCombinatorParent"] = 8] = "SelectorCombinatorParent";
        NodeType[NodeType["SelectorCombinatorSibling"] = 9] = "SelectorCombinatorSibling";
        NodeType[NodeType["SelectorCombinatorAllSiblings"] = 10] = "SelectorCombinatorAllSiblings";
        NodeType[NodeType["SelectorCombinatorShadowPiercingDescendant"] = 11] = "SelectorCombinatorShadowPiercingDescendant";
        NodeType[NodeType["Page"] = 12] = "Page";
        NodeType[NodeType["PageBoxMarginBox"] = 13] = "PageBoxMarginBox";
        NodeType[NodeType["ClassSelector"] = 14] = "ClassSelector";
        NodeType[NodeType["IdentifierSelector"] = 15] = "IdentifierSelector";
        NodeType[NodeType["ElementNameSelector"] = 16] = "ElementNameSelector";
        NodeType[NodeType["PseudoSelector"] = 17] = "PseudoSelector";
        NodeType[NodeType["AttributeSelector"] = 18] = "AttributeSelector";
        NodeType[NodeType["Declaration"] = 19] = "Declaration";
        NodeType[NodeType["Declarations"] = 20] = "Declarations";
        NodeType[NodeType["Property"] = 21] = "Property";
        NodeType[NodeType["Expression"] = 22] = "Expression";
        NodeType[NodeType["BinaryExpression"] = 23] = "BinaryExpression";
        NodeType[NodeType["Term"] = 24] = "Term";
        NodeType[NodeType["Operator"] = 25] = "Operator";
        NodeType[NodeType["Value"] = 26] = "Value";
        NodeType[NodeType["StringLiteral"] = 27] = "StringLiteral";
        NodeType[NodeType["URILiteral"] = 28] = "URILiteral";
        NodeType[NodeType["EscapedValue"] = 29] = "EscapedValue";
        NodeType[NodeType["Function"] = 30] = "Function";
        NodeType[NodeType["NumericValue"] = 31] = "NumericValue";
        NodeType[NodeType["HexColorValue"] = 32] = "HexColorValue";
        NodeType[NodeType["MixinDeclaration"] = 33] = "MixinDeclaration";
        NodeType[NodeType["MixinReference"] = 34] = "MixinReference";
        NodeType[NodeType["VariableName"] = 35] = "VariableName";
        NodeType[NodeType["VariableDeclaration"] = 36] = "VariableDeclaration";
        NodeType[NodeType["Prio"] = 37] = "Prio";
        NodeType[NodeType["Interpolation"] = 38] = "Interpolation";
        NodeType[NodeType["NestedProperties"] = 39] = "NestedProperties";
        NodeType[NodeType["ExtendsReference"] = 40] = "ExtendsReference";
        NodeType[NodeType["SelectorPlaceholder"] = 41] = "SelectorPlaceholder";
        NodeType[NodeType["Debug"] = 42] = "Debug";
        NodeType[NodeType["If"] = 43] = "If";
        NodeType[NodeType["Else"] = 44] = "Else";
        NodeType[NodeType["For"] = 45] = "For";
        NodeType[NodeType["Each"] = 46] = "Each";
        NodeType[NodeType["While"] = 47] = "While";
        NodeType[NodeType["MixinContentReference"] = 48] = "MixinContentReference";
        NodeType[NodeType["MixinContentDeclaration"] = 49] = "MixinContentDeclaration";
        NodeType[NodeType["Media"] = 50] = "Media";
        NodeType[NodeType["Keyframe"] = 51] = "Keyframe";
        NodeType[NodeType["FontFace"] = 52] = "FontFace";
        NodeType[NodeType["Import"] = 53] = "Import";
        NodeType[NodeType["Namespace"] = 54] = "Namespace";
        NodeType[NodeType["Invocation"] = 55] = "Invocation";
        NodeType[NodeType["FunctionDeclaration"] = 56] = "FunctionDeclaration";
        NodeType[NodeType["ReturnStatement"] = 57] = "ReturnStatement";
        NodeType[NodeType["MediaQuery"] = 58] = "MediaQuery";
        NodeType[NodeType["FunctionParameter"] = 59] = "FunctionParameter";
        NodeType[NodeType["FunctionArgument"] = 60] = "FunctionArgument";
        NodeType[NodeType["KeyframeSelector"] = 61] = "KeyframeSelector";
        NodeType[NodeType["ViewPort"] = 62] = "ViewPort";
        NodeType[NodeType["Document"] = 63] = "Document";
        NodeType[NodeType["AtApplyRule"] = 64] = "AtApplyRule";
        NodeType[NodeType["CustomPropertyDeclaration"] = 65] = "CustomPropertyDeclaration";
        NodeType[NodeType["CustomPropertySet"] = 66] = "CustomPropertySet";
        NodeType[NodeType["ListEntry"] = 67] = "ListEntry";
        NodeType[NodeType["Supports"] = 68] = "Supports";
        NodeType[NodeType["SupportsCondition"] = 69] = "SupportsCondition";
        NodeType[NodeType["NamespacePrefix"] = 70] = "NamespacePrefix";
        NodeType[NodeType["GridLine"] = 71] = "GridLine";
        NodeType[NodeType["Plugin"] = 72] = "Plugin";
        NodeType[NodeType["UnknownAtRule"] = 73] = "UnknownAtRule";
        NodeType[NodeType["Use"] = 74] = "Use";
        NodeType[NodeType["ModuleConfiguration"] = 75] = "ModuleConfiguration";
        NodeType[NodeType["Forward"] = 76] = "Forward";
        NodeType[NodeType["ForwardVisibility"] = 77] = "ForwardVisibility";
        NodeType[NodeType["Module"] = 78] = "Module";
    })(NodeType = exports.NodeType || (exports.NodeType = {}));
    var ReferenceType;
    (function (ReferenceType) {
        ReferenceType[ReferenceType["Mixin"] = 0] = "Mixin";
        ReferenceType[ReferenceType["Rule"] = 1] = "Rule";
        ReferenceType[ReferenceType["Variable"] = 2] = "Variable";
        ReferenceType[ReferenceType["Function"] = 3] = "Function";
        ReferenceType[ReferenceType["Keyframe"] = 4] = "Keyframe";
        ReferenceType[ReferenceType["Unknown"] = 5] = "Unknown";
        ReferenceType[ReferenceType["Module"] = 6] = "Module";
        ReferenceType[ReferenceType["Forward"] = 7] = "Forward";
        ReferenceType[ReferenceType["ForwardVisibility"] = 8] = "ForwardVisibility";
    })(ReferenceType = exports.ReferenceType || (exports.ReferenceType = {}));
    function getNodeAtOffset(node, offset) {
        var candidate = null;
        if (!node || offset < node.offset || offset > node.end) {
            return null;
        }
        // Find the shortest node at the position
        node.accept(function (node) {
            if (node.offset === -1 && node.length === -1) {
                return true;
            }
            if (node.offset <= offset && node.end >= offset) {
                if (!candidate) {
                    candidate = node;
                }
                else if (node.length <= candidate.length) {
                    candidate = node;
                }
                return true;
            }
            return false;
        });
        return candidate;
    }
    exports.getNodeAtOffset = getNodeAtOffset;
    function getNodePath(node, offset) {
        var candidate = getNodeAtOffset(node, offset);
        var path = [];
        while (candidate) {
            path.unshift(candidate);
            candidate = candidate.parent;
        }
        return path;
    }
    exports.getNodePath = getNodePath;
    function getParentDeclaration(node) {
        var decl = node.findParent(NodeType.Declaration);
        var value = decl && decl.getValue();
        if (value && value.encloses(node)) {
            return decl;
        }
        return null;
    }
    exports.getParentDeclaration = getParentDeclaration;
    var Node = /** @class */ (function () {
        function Node(offset, len, nodeType) {
            if (offset === void 0) { offset = -1; }
            if (len === void 0) { len = -1; }
            this.parent = null;
            this.offset = offset;
            this.length = len;
            if (nodeType) {
                this.nodeType = nodeType;
            }
        }
        Object.defineProperty(Node.prototype, "end", {
            get: function () { return this.offset + this.length; },
            enumerable: false,
            configurable: true
        });
        Object.defineProperty(Node.prototype, "type", {
            get: function () {
                return this.nodeType || NodeType.Undefined;
            },
            set: function (type) {
                this.nodeType = type;
            },
            enumerable: false,
            configurable: true
        });
        Node.prototype.getTextProvider = function () {
            var node = this;
            while (node && !node.textProvider) {
                node = node.parent;
            }
            if (node) {
                return node.textProvider;
            }
            return function () { return 'unknown'; };
        };
        Node.prototype.getText = function () {
            return this.getTextProvider()(this.offset, this.length);
        };
        Node.prototype.matches = function (str) {
            return this.length === str.length && this.getTextProvider()(this.offset, this.length) === str;
        };
        Node.prototype.startsWith = function (str) {
            return this.length >= str.length && this.getTextProvider()(this.offset, str.length) === str;
        };
        Node.prototype.endsWith = function (str) {
            return this.length >= str.length && this.getTextProvider()(this.end - str.length, str.length) === str;
        };
        Node.prototype.accept = function (visitor) {
            if (visitor(this) && this.children) {
                for (var _i = 0, _a = this.children; _i < _a.length; _i++) {
                    var child = _a[_i];
                    child.accept(visitor);
                }
            }
        };
        Node.prototype.acceptVisitor = function (visitor) {
            this.accept(visitor.visitNode.bind(visitor));
        };
        Node.prototype.adoptChild = function (node, index) {
            if (index === void 0) { index = -1; }
            if (node.parent && node.parent.children) {
                var idx = node.parent.children.indexOf(node);
                if (idx >= 0) {
                    node.parent.children.splice(idx, 1);
                }
            }
            node.parent = this;
            var children = this.children;
            if (!children) {
                children = this.children = [];
            }
            if (index !== -1) {
                children.splice(index, 0, node);
            }
            else {
                children.push(node);
            }
            return node;
        };
        Node.prototype.attachTo = function (parent, index) {
            if (index === void 0) { index = -1; }
            if (parent) {
                parent.adoptChild(this, index);
            }
            return this;
        };
        Node.prototype.collectIssues = function (results) {
            if (this.issues) {
                results.push.apply(results, this.issues);
            }
        };
        Node.prototype.addIssue = function (issue) {
            if (!this.issues) {
                this.issues = [];
            }
            this.issues.push(issue);
        };
        Node.prototype.hasIssue = function (rule) {
            return Array.isArray(this.issues) && this.issues.some(function (i) { return i.getRule() === rule; });
        };
        Node.prototype.isErroneous = function (recursive) {
            if (recursive === void 0) { recursive = false; }
            if (this.issues && this.issues.length > 0) {
                return true;
            }
            return recursive && Array.isArray(this.children) && this.children.some(function (c) { return c.isErroneous(true); });
        };
        Node.prototype.setNode = function (field, node, index) {
            if (index === void 0) { index = -1; }
            if (node) {
                node.attachTo(this, index);
                this[field] = node;
                return true;
            }
            return false;
        };
        Node.prototype.addChild = function (node) {
            if (node) {
                if (!this.children) {
                    this.children = [];
                }
                node.attachTo(this);
                this.updateOffsetAndLength(node);
                return true;
            }
            return false;
        };
        Node.prototype.updateOffsetAndLength = function (node) {
            if (node.offset < this.offset || this.offset === -1) {
                this.offset = node.offset;
            }
            var nodeEnd = node.end;
            if ((nodeEnd > this.end) || this.length === -1) {
                this.length = nodeEnd - this.offset;
            }
        };
        Node.prototype.hasChildren = function () {
            return !!this.children && this.children.length > 0;
        };
        Node.prototype.getChildren = function () {
            return this.children ? this.children.slice(0) : [];
        };
        Node.prototype.getChild = function (index) {
            if (this.children && index < this.children.length) {
                return this.children[index];
            }
            return null;
        };
        Node.prototype.addChildren = function (nodes) {
            for (var _i = 0, nodes_1 = nodes; _i < nodes_1.length; _i++) {
                var node = nodes_1[_i];
                this.addChild(node);
            }
        };
        Node.prototype.findFirstChildBeforeOffset = function (offset) {
            if (this.children) {
                var current = null;
                for (var i = this.children.length - 1; i >= 0; i--) {
                    // iterate until we find a child that has a start offset smaller than the input offset
                    current = this.children[i];
                    if (current.offset <= offset) {
                        return current;
                    }
                }
            }
            return null;
        };
        Node.prototype.findChildAtOffset = function (offset, goDeep) {
            var current = this.findFirstChildBeforeOffset(offset);
            if (current && current.end >= offset) {
                if (goDeep) {
                    return current.findChildAtOffset(offset, true) || current;
                }
                return current;
            }
            return null;
        };
        Node.prototype.encloses = function (candidate) {
            return this.offset <= candidate.offset && this.offset + this.length >= candidate.offset + candidate.length;
        };
        Node.prototype.getParent = function () {
            var result = this.parent;
            while (result instanceof Nodelist) {
                result = result.parent;
            }
            return result;
        };
        Node.prototype.findParent = function (type) {
            var result = this;
            while (result && result.type !== type) {
                result = result.parent;
            }
            return result;
        };
        Node.prototype.findAParent = function () {
            var types = [];
            for (var _i = 0; _i < arguments.length; _i++) {
                types[_i] = arguments[_i];
            }
            var result = this;
            while (result && !types.some(function (t) { return result.type === t; })) {
                result = result.parent;
            }
            return result;
        };
        Node.prototype.setData = function (key, value) {
            if (!this.options) {
                this.options = {};
            }
            this.options[key] = value;
        };
        Node.prototype.getData = function (key) {
            if (!this.options || !this.options.hasOwnProperty(key)) {
                return null;
            }
            return this.options[key];
        };
        return Node;
    }());
    exports.Node = Node;
    var Nodelist = /** @class */ (function (_super) {
        __extends(Nodelist, _super);
        function Nodelist(parent, index) {
            if (index === void 0) { index = -1; }
            var _this = _super.call(this, -1, -1) || this;
            _this.attachTo(parent, index);
            _this.offset = -1;
            _this.length = -1;
            return _this;
        }
        return Nodelist;
    }(Node));
    exports.Nodelist = Nodelist;
    var Identifier = /** @class */ (function (_super) {
        __extends(Identifier, _super);
        function Identifier(offset, length) {
            var _this = _super.call(this, offset, length) || this;
            _this.isCustomProperty = false;
            return _this;
        }
        Object.defineProperty(Identifier.prototype, "type", {
            get: function () {
                return NodeType.Identifier;
            },
            enumerable: false,
            configurable: true
        });
        Identifier.prototype.containsInterpolation = function () {
            return this.hasChildren();
        };
        return Identifier;
    }(Node));
    exports.Identifier = Identifier;
    var Stylesheet = /** @class */ (function (_super) {
        __extends(Stylesheet, _super);
        function Stylesheet(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Stylesheet.prototype, "type", {
            get: function () {
                return NodeType.Stylesheet;
            },
            enumerable: false,
            configurable: true
        });
        return Stylesheet;
    }(Node));
    exports.Stylesheet = Stylesheet;
    var Declarations = /** @class */ (function (_super) {
        __extends(Declarations, _super);
        function Declarations(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Declarations.prototype, "type", {
            get: function () {
                return NodeType.Declarations;
            },
            enumerable: false,
            configurable: true
        });
        return Declarations;
    }(Node));
    exports.Declarations = Declarations;
    var BodyDeclaration = /** @class */ (function (_super) {
        __extends(BodyDeclaration, _super);
        function BodyDeclaration(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        BodyDeclaration.prototype.getDeclarations = function () {
            return this.declarations;
        };
        BodyDeclaration.prototype.setDeclarations = function (decls) {
            return this.setNode('declarations', decls);
        };
        return BodyDeclaration;
    }(Node));
    exports.BodyDeclaration = BodyDeclaration;
    var RuleSet = /** @class */ (function (_super) {
        __extends(RuleSet, _super);
        function RuleSet(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(RuleSet.prototype, "type", {
            get: function () {
                return NodeType.Ruleset;
            },
            enumerable: false,
            configurable: true
        });
        RuleSet.prototype.getSelectors = function () {
            if (!this.selectors) {
                this.selectors = new Nodelist(this);
            }
            return this.selectors;
        };
        RuleSet.prototype.isNested = function () {
            return !!this.parent && this.parent.findParent(NodeType.Declarations) !== null;
        };
        return RuleSet;
    }(BodyDeclaration));
    exports.RuleSet = RuleSet;
    var Selector = /** @class */ (function (_super) {
        __extends(Selector, _super);
        function Selector(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Selector.prototype, "type", {
            get: function () {
                return NodeType.Selector;
            },
            enumerable: false,
            configurable: true
        });
        return Selector;
    }(Node));
    exports.Selector = Selector;
    var SimpleSelector = /** @class */ (function (_super) {
        __extends(SimpleSelector, _super);
        function SimpleSelector(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(SimpleSelector.prototype, "type", {
            get: function () {
                return NodeType.SimpleSelector;
            },
            enumerable: false,
            configurable: true
        });
        return SimpleSelector;
    }(Node));
    exports.SimpleSelector = SimpleSelector;
    var AtApplyRule = /** @class */ (function (_super) {
        __extends(AtApplyRule, _super);
        function AtApplyRule(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(AtApplyRule.prototype, "type", {
            get: function () {
                return NodeType.AtApplyRule;
            },
            enumerable: false,
            configurable: true
        });
        AtApplyRule.prototype.setIdentifier = function (node) {
            return this.setNode('identifier', node, 0);
        };
        AtApplyRule.prototype.getIdentifier = function () {
            return this.identifier;
        };
        AtApplyRule.prototype.getName = function () {
            return this.identifier ? this.identifier.getText() : '';
        };
        return AtApplyRule;
    }(Node));
    exports.AtApplyRule = AtApplyRule;
    var AbstractDeclaration = /** @class */ (function (_super) {
        __extends(AbstractDeclaration, _super);
        function AbstractDeclaration(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        return AbstractDeclaration;
    }(Node));
    exports.AbstractDeclaration = AbstractDeclaration;
    var CustomPropertySet = /** @class */ (function (_super) {
        __extends(CustomPropertySet, _super);
        function CustomPropertySet(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(CustomPropertySet.prototype, "type", {
            get: function () {
                return NodeType.CustomPropertySet;
            },
            enumerable: false,
            configurable: true
        });
        return CustomPropertySet;
    }(BodyDeclaration));
    exports.CustomPropertySet = CustomPropertySet;
    var Declaration = /** @class */ (function (_super) {
        __extends(Declaration, _super);
        function Declaration(offset, length) {
            var _this = _super.call(this, offset, length) || this;
            _this.property = null;
            return _this;
        }
        Object.defineProperty(Declaration.prototype, "type", {
            get: function () {
                return NodeType.Declaration;
            },
            enumerable: false,
            configurable: true
        });
        Declaration.prototype.setProperty = function (node) {
            return this.setNode('property', node);
        };
        Declaration.prototype.getProperty = function () {
            return this.property;
        };
        Declaration.prototype.getFullPropertyName = function () {
            var propertyName = this.property ? this.property.getName() : 'unknown';
            if (this.parent instanceof Declarations && this.parent.getParent() instanceof NestedProperties) {
                var parentDecl = this.parent.getParent().getParent();
                if (parentDecl instanceof Declaration) {
                    return parentDecl.getFullPropertyName() + propertyName;
                }
            }
            return propertyName;
        };
        Declaration.prototype.getNonPrefixedPropertyName = function () {
            var propertyName = this.getFullPropertyName();
            if (propertyName && propertyName.charAt(0) === '-') {
                var vendorPrefixEnd = propertyName.indexOf('-', 1);
                if (vendorPrefixEnd !== -1) {
                    return propertyName.substring(vendorPrefixEnd + 1);
                }
            }
            return propertyName;
        };
        Declaration.prototype.setValue = function (value) {
            return this.setNode('value', value);
        };
        Declaration.prototype.getValue = function () {
            return this.value;
        };
        Declaration.prototype.setNestedProperties = function (value) {
            return this.setNode('nestedProperties', value);
        };
        Declaration.prototype.getNestedProperties = function () {
            return this.nestedProperties;
        };
        return Declaration;
    }(AbstractDeclaration));
    exports.Declaration = Declaration;
    var CustomPropertyDeclaration = /** @class */ (function (_super) {
        __extends(CustomPropertyDeclaration, _super);
        function CustomPropertyDeclaration(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(CustomPropertyDeclaration.prototype, "type", {
            get: function () {
                return NodeType.CustomPropertyDeclaration;
            },
            enumerable: false,
            configurable: true
        });
        CustomPropertyDeclaration.prototype.setPropertySet = function (value) {
            return this.setNode('propertySet', value);
        };
        CustomPropertyDeclaration.prototype.getPropertySet = function () {
            return this.propertySet;
        };
        return CustomPropertyDeclaration;
    }(Declaration));
    exports.CustomPropertyDeclaration = CustomPropertyDeclaration;
    var Property = /** @class */ (function (_super) {
        __extends(Property, _super);
        function Property(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Property.prototype, "type", {
            get: function () {
                return NodeType.Property;
            },
            enumerable: false,
            configurable: true
        });
        Property.prototype.setIdentifier = function (value) {
            return this.setNode('identifier', value);
        };
        Property.prototype.getIdentifier = function () {
            return this.identifier;
        };
        Property.prototype.getName = function () {
            return strings_1.trim(this.getText(), /[_\+]+$/); /* +_: less merge */
        };
        Property.prototype.isCustomProperty = function () {
            return !!this.identifier && this.identifier.isCustomProperty;
        };
        return Property;
    }(Node));
    exports.Property = Property;
    var Invocation = /** @class */ (function (_super) {
        __extends(Invocation, _super);
        function Invocation(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Invocation.prototype, "type", {
            get: function () {
                return NodeType.Invocation;
            },
            enumerable: false,
            configurable: true
        });
        Invocation.prototype.getArguments = function () {
            if (!this.arguments) {
                this.arguments = new Nodelist(this);
            }
            return this.arguments;
        };
        return Invocation;
    }(Node));
    exports.Invocation = Invocation;
    var Function = /** @class */ (function (_super) {
        __extends(Function, _super);
        function Function(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Function.prototype, "type", {
            get: function () {
                return NodeType.Function;
            },
            enumerable: false,
            configurable: true
        });
        Function.prototype.setIdentifier = function (node) {
            return this.setNode('identifier', node, 0);
        };
        Function.prototype.getIdentifier = function () {
            return this.identifier;
        };
        Function.prototype.getName = function () {
            return this.identifier ? this.identifier.getText() : '';
        };
        return Function;
    }(Invocation));
    exports.Function = Function;
    var FunctionParameter = /** @class */ (function (_super) {
        __extends(FunctionParameter, _super);
        function FunctionParameter(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(FunctionParameter.prototype, "type", {
            get: function () {
                return NodeType.FunctionParameter;
            },
            enumerable: false,
            configurable: true
        });
        FunctionParameter.prototype.setIdentifier = function (node) {
            return this.setNode('identifier', node, 0);
        };
        FunctionParameter.prototype.getIdentifier = function () {
            return this.identifier;
        };
        FunctionParameter.prototype.getName = function () {
            return this.identifier ? this.identifier.getText() : '';
        };
        FunctionParameter.prototype.setDefaultValue = function (node) {
            return this.setNode('defaultValue', node, 0);
        };
        FunctionParameter.prototype.getDefaultValue = function () {
            return this.defaultValue;
        };
        return FunctionParameter;
    }(Node));
    exports.FunctionParameter = FunctionParameter;
    var FunctionArgument = /** @class */ (function (_super) {
        __extends(FunctionArgument, _super);
        function FunctionArgument(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(FunctionArgument.prototype, "type", {
            get: function () {
                return NodeType.FunctionArgument;
            },
            enumerable: false,
            configurable: true
        });
        FunctionArgument.prototype.setIdentifier = function (node) {
            return this.setNode('identifier', node, 0);
        };
        FunctionArgument.prototype.getIdentifier = function () {
            return this.identifier;
        };
        FunctionArgument.prototype.getName = function () {
            return this.identifier ? this.identifier.getText() : '';
        };
        FunctionArgument.prototype.setValue = function (node) {
            return this.setNode('value', node, 0);
        };
        FunctionArgument.prototype.getValue = function () {
            return this.value;
        };
        return FunctionArgument;
    }(Node));
    exports.FunctionArgument = FunctionArgument;
    var IfStatement = /** @class */ (function (_super) {
        __extends(IfStatement, _super);
        function IfStatement(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(IfStatement.prototype, "type", {
            get: function () {
                return NodeType.If;
            },
            enumerable: false,
            configurable: true
        });
        IfStatement.prototype.setExpression = function (node) {
            return this.setNode('expression', node, 0);
        };
        IfStatement.prototype.setElseClause = function (elseClause) {
            return this.setNode('elseClause', elseClause);
        };
        return IfStatement;
    }(BodyDeclaration));
    exports.IfStatement = IfStatement;
    var ForStatement = /** @class */ (function (_super) {
        __extends(ForStatement, _super);
        function ForStatement(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(ForStatement.prototype, "type", {
            get: function () {
                return NodeType.For;
            },
            enumerable: false,
            configurable: true
        });
        ForStatement.prototype.setVariable = function (node) {
            return this.setNode('variable', node, 0);
        };
        return ForStatement;
    }(BodyDeclaration));
    exports.ForStatement = ForStatement;
    var EachStatement = /** @class */ (function (_super) {
        __extends(EachStatement, _super);
        function EachStatement(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(EachStatement.prototype, "type", {
            get: function () {
                return NodeType.Each;
            },
            enumerable: false,
            configurable: true
        });
        EachStatement.prototype.getVariables = function () {
            if (!this.variables) {
                this.variables = new Nodelist(this);
            }
            return this.variables;
        };
        return EachStatement;
    }(BodyDeclaration));
    exports.EachStatement = EachStatement;
    var WhileStatement = /** @class */ (function (_super) {
        __extends(WhileStatement, _super);
        function WhileStatement(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(WhileStatement.prototype, "type", {
            get: function () {
                return NodeType.While;
            },
            enumerable: false,
            configurable: true
        });
        return WhileStatement;
    }(BodyDeclaration));
    exports.WhileStatement = WhileStatement;
    var ElseStatement = /** @class */ (function (_super) {
        __extends(ElseStatement, _super);
        function ElseStatement(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(ElseStatement.prototype, "type", {
            get: function () {
                return NodeType.Else;
            },
            enumerable: false,
            configurable: true
        });
        return ElseStatement;
    }(BodyDeclaration));
    exports.ElseStatement = ElseStatement;
    var FunctionDeclaration = /** @class */ (function (_super) {
        __extends(FunctionDeclaration, _super);
        function FunctionDeclaration(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(FunctionDeclaration.prototype, "type", {
            get: function () {
                return NodeType.FunctionDeclaration;
            },
            enumerable: false,
            configurable: true
        });
        FunctionDeclaration.prototype.setIdentifier = function (node) {
            return this.setNode('identifier', node, 0);
        };
        FunctionDeclaration.prototype.getIdentifier = function () {
            return this.identifier;
        };
        FunctionDeclaration.prototype.getName = function () {
            return this.identifier ? this.identifier.getText() : '';
        };
        FunctionDeclaration.prototype.getParameters = function () {
            if (!this.parameters) {
                this.parameters = new Nodelist(this);
            }
            return this.parameters;
        };
        return FunctionDeclaration;
    }(BodyDeclaration));
    exports.FunctionDeclaration = FunctionDeclaration;
    var ViewPort = /** @class */ (function (_super) {
        __extends(ViewPort, _super);
        function ViewPort(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(ViewPort.prototype, "type", {
            get: function () {
                return NodeType.ViewPort;
            },
            enumerable: false,
            configurable: true
        });
        return ViewPort;
    }(BodyDeclaration));
    exports.ViewPort = ViewPort;
    var FontFace = /** @class */ (function (_super) {
        __extends(FontFace, _super);
        function FontFace(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(FontFace.prototype, "type", {
            get: function () {
                return NodeType.FontFace;
            },
            enumerable: false,
            configurable: true
        });
        return FontFace;
    }(BodyDeclaration));
    exports.FontFace = FontFace;
    var NestedProperties = /** @class */ (function (_super) {
        __extends(NestedProperties, _super);
        function NestedProperties(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(NestedProperties.prototype, "type", {
            get: function () {
                return NodeType.NestedProperties;
            },
            enumerable: false,
            configurable: true
        });
        return NestedProperties;
    }(BodyDeclaration));
    exports.NestedProperties = NestedProperties;
    var Keyframe = /** @class */ (function (_super) {
        __extends(Keyframe, _super);
        function Keyframe(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Keyframe.prototype, "type", {
            get: function () {
                return NodeType.Keyframe;
            },
            enumerable: false,
            configurable: true
        });
        Keyframe.prototype.setKeyword = function (keyword) {
            return this.setNode('keyword', keyword, 0);
        };
        Keyframe.prototype.getKeyword = function () {
            return this.keyword;
        };
        Keyframe.prototype.setIdentifier = function (node) {
            return this.setNode('identifier', node, 0);
        };
        Keyframe.prototype.getIdentifier = function () {
            return this.identifier;
        };
        Keyframe.prototype.getName = function () {
            return this.identifier ? this.identifier.getText() : '';
        };
        return Keyframe;
    }(BodyDeclaration));
    exports.Keyframe = Keyframe;
    var KeyframeSelector = /** @class */ (function (_super) {
        __extends(KeyframeSelector, _super);
        function KeyframeSelector(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(KeyframeSelector.prototype, "type", {
            get: function () {
                return NodeType.KeyframeSelector;
            },
            enumerable: false,
            configurable: true
        });
        return KeyframeSelector;
    }(BodyDeclaration));
    exports.KeyframeSelector = KeyframeSelector;
    var Import = /** @class */ (function (_super) {
        __extends(Import, _super);
        function Import(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Import.prototype, "type", {
            get: function () {
                return NodeType.Import;
            },
            enumerable: false,
            configurable: true
        });
        Import.prototype.setMedialist = function (node) {
            if (node) {
                node.attachTo(this);
                return true;
            }
            return false;
        };
        return Import;
    }(Node));
    exports.Import = Import;
    var Use = /** @class */ (function (_super) {
        __extends(Use, _super);
        function Use() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        Object.defineProperty(Use.prototype, "type", {
            get: function () {
                return NodeType.Use;
            },
            enumerable: false,
            configurable: true
        });
        Use.prototype.getParameters = function () {
            if (!this.parameters) {
                this.parameters = new Nodelist(this);
            }
            return this.parameters;
        };
        Use.prototype.setIdentifier = function (node) {
            return this.setNode('identifier', node, 0);
        };
        Use.prototype.getIdentifier = function () {
            return this.identifier;
        };
        return Use;
    }(Node));
    exports.Use = Use;
    var ModuleConfiguration = /** @class */ (function (_super) {
        __extends(ModuleConfiguration, _super);
        function ModuleConfiguration() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        Object.defineProperty(ModuleConfiguration.prototype, "type", {
            get: function () {
                return NodeType.ModuleConfiguration;
            },
            enumerable: false,
            configurable: true
        });
        ModuleConfiguration.prototype.setIdentifier = function (node) {
            return this.setNode('identifier', node, 0);
        };
        ModuleConfiguration.prototype.getIdentifier = function () {
            return this.identifier;
        };
        ModuleConfiguration.prototype.getName = function () {
            return this.identifier ? this.identifier.getText() : '';
        };
        ModuleConfiguration.prototype.setValue = function (node) {
            return this.setNode('value', node, 0);
        };
        ModuleConfiguration.prototype.getValue = function () {
            return this.value;
        };
        return ModuleConfiguration;
    }(Node));
    exports.ModuleConfiguration = ModuleConfiguration;
    var Forward = /** @class */ (function (_super) {
        __extends(Forward, _super);
        function Forward() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        Object.defineProperty(Forward.prototype, "type", {
            get: function () {
                return NodeType.Forward;
            },
            enumerable: false,
            configurable: true
        });
        Forward.prototype.setIdentifier = function (node) {
            return this.setNode('identifier', node, 0);
        };
        Forward.prototype.getIdentifier = function () {
            return this.identifier;
        };
        Forward.prototype.getMembers = function () {
            if (!this.members) {
                this.members = new Nodelist(this);
            }
            return this.members;
        };
        Forward.prototype.getParameters = function () {
            if (!this.parameters) {
                this.parameters = new Nodelist(this);
            }
            return this.parameters;
        };
        return Forward;
    }(Node));
    exports.Forward = Forward;
    var ForwardVisibility = /** @class */ (function (_super) {
        __extends(ForwardVisibility, _super);
        function ForwardVisibility() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        Object.defineProperty(ForwardVisibility.prototype, "type", {
            get: function () {
                return NodeType.ForwardVisibility;
            },
            enumerable: false,
            configurable: true
        });
        ForwardVisibility.prototype.setIdentifier = function (node) {
            return this.setNode('identifier', node, 0);
        };
        ForwardVisibility.prototype.getIdentifier = function () {
            return this.identifier;
        };
        return ForwardVisibility;
    }(Node));
    exports.ForwardVisibility = ForwardVisibility;
    var Namespace = /** @class */ (function (_super) {
        __extends(Namespace, _super);
        function Namespace(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Namespace.prototype, "type", {
            get: function () {
                return NodeType.Namespace;
            },
            enumerable: false,
            configurable: true
        });
        return Namespace;
    }(Node));
    exports.Namespace = Namespace;
    var Media = /** @class */ (function (_super) {
        __extends(Media, _super);
        function Media(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Media.prototype, "type", {
            get: function () {
                return NodeType.Media;
            },
            enumerable: false,
            configurable: true
        });
        return Media;
    }(BodyDeclaration));
    exports.Media = Media;
    var Supports = /** @class */ (function (_super) {
        __extends(Supports, _super);
        function Supports(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Supports.prototype, "type", {
            get: function () {
                return NodeType.Supports;
            },
            enumerable: false,
            configurable: true
        });
        return Supports;
    }(BodyDeclaration));
    exports.Supports = Supports;
    var Document = /** @class */ (function (_super) {
        __extends(Document, _super);
        function Document(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Document.prototype, "type", {
            get: function () {
                return NodeType.Document;
            },
            enumerable: false,
            configurable: true
        });
        return Document;
    }(BodyDeclaration));
    exports.Document = Document;
    var Medialist = /** @class */ (function (_super) {
        __extends(Medialist, _super);
        function Medialist(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Medialist.prototype.getMediums = function () {
            if (!this.mediums) {
                this.mediums = new Nodelist(this);
            }
            return this.mediums;
        };
        return Medialist;
    }(Node));
    exports.Medialist = Medialist;
    var MediaQuery = /** @class */ (function (_super) {
        __extends(MediaQuery, _super);
        function MediaQuery(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(MediaQuery.prototype, "type", {
            get: function () {
                return NodeType.MediaQuery;
            },
            enumerable: false,
            configurable: true
        });
        return MediaQuery;
    }(Node));
    exports.MediaQuery = MediaQuery;
    var SupportsCondition = /** @class */ (function (_super) {
        __extends(SupportsCondition, _super);
        function SupportsCondition(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(SupportsCondition.prototype, "type", {
            get: function () {
                return NodeType.SupportsCondition;
            },
            enumerable: false,
            configurable: true
        });
        return SupportsCondition;
    }(Node));
    exports.SupportsCondition = SupportsCondition;
    var Page = /** @class */ (function (_super) {
        __extends(Page, _super);
        function Page(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Page.prototype, "type", {
            get: function () {
                return NodeType.Page;
            },
            enumerable: false,
            configurable: true
        });
        return Page;
    }(BodyDeclaration));
    exports.Page = Page;
    var PageBoxMarginBox = /** @class */ (function (_super) {
        __extends(PageBoxMarginBox, _super);
        function PageBoxMarginBox(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(PageBoxMarginBox.prototype, "type", {
            get: function () {
                return NodeType.PageBoxMarginBox;
            },
            enumerable: false,
            configurable: true
        });
        return PageBoxMarginBox;
    }(BodyDeclaration));
    exports.PageBoxMarginBox = PageBoxMarginBox;
    var Expression = /** @class */ (function (_super) {
        __extends(Expression, _super);
        function Expression(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Expression.prototype, "type", {
            get: function () {
                return NodeType.Expression;
            },
            enumerable: false,
            configurable: true
        });
        return Expression;
    }(Node));
    exports.Expression = Expression;
    var BinaryExpression = /** @class */ (function (_super) {
        __extends(BinaryExpression, _super);
        function BinaryExpression(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(BinaryExpression.prototype, "type", {
            get: function () {
                return NodeType.BinaryExpression;
            },
            enumerable: false,
            configurable: true
        });
        BinaryExpression.prototype.setLeft = function (left) {
            return this.setNode('left', left);
        };
        BinaryExpression.prototype.getLeft = function () {
            return this.left;
        };
        BinaryExpression.prototype.setRight = function (right) {
            return this.setNode('right', right);
        };
        BinaryExpression.prototype.getRight = function () {
            return this.right;
        };
        BinaryExpression.prototype.setOperator = function (value) {
            return this.setNode('operator', value);
        };
        BinaryExpression.prototype.getOperator = function () {
            return this.operator;
        };
        return BinaryExpression;
    }(Node));
    exports.BinaryExpression = BinaryExpression;
    var Term = /** @class */ (function (_super) {
        __extends(Term, _super);
        function Term(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Term.prototype, "type", {
            get: function () {
                return NodeType.Term;
            },
            enumerable: false,
            configurable: true
        });
        Term.prototype.setOperator = function (value) {
            return this.setNode('operator', value);
        };
        Term.prototype.getOperator = function () {
            return this.operator;
        };
        Term.prototype.setExpression = function (value) {
            return this.setNode('expression', value);
        };
        Term.prototype.getExpression = function () {
            return this.expression;
        };
        return Term;
    }(Node));
    exports.Term = Term;
    var AttributeSelector = /** @class */ (function (_super) {
        __extends(AttributeSelector, _super);
        function AttributeSelector(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(AttributeSelector.prototype, "type", {
            get: function () {
                return NodeType.AttributeSelector;
            },
            enumerable: false,
            configurable: true
        });
        AttributeSelector.prototype.setNamespacePrefix = function (value) {
            return this.setNode('namespacePrefix', value);
        };
        AttributeSelector.prototype.getNamespacePrefix = function () {
            return this.namespacePrefix;
        };
        AttributeSelector.prototype.setIdentifier = function (value) {
            return this.setNode('identifier', value);
        };
        AttributeSelector.prototype.getIdentifier = function () {
            return this.identifier;
        };
        AttributeSelector.prototype.setOperator = function (operator) {
            return this.setNode('operator', operator);
        };
        AttributeSelector.prototype.getOperator = function () {
            return this.operator;
        };
        AttributeSelector.prototype.setValue = function (value) {
            return this.setNode('value', value);
        };
        AttributeSelector.prototype.getValue = function () {
            return this.value;
        };
        return AttributeSelector;
    }(Node));
    exports.AttributeSelector = AttributeSelector;
    var Operator = /** @class */ (function (_super) {
        __extends(Operator, _super);
        function Operator(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Operator.prototype, "type", {
            get: function () {
                return NodeType.Operator;
            },
            enumerable: false,
            configurable: true
        });
        return Operator;
    }(Node));
    exports.Operator = Operator;
    var HexColorValue = /** @class */ (function (_super) {
        __extends(HexColorValue, _super);
        function HexColorValue(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(HexColorValue.prototype, "type", {
            get: function () {
                return NodeType.HexColorValue;
            },
            enumerable: false,
            configurable: true
        });
        return HexColorValue;
    }(Node));
    exports.HexColorValue = HexColorValue;
    var _dot = '.'.charCodeAt(0), _0 = '0'.charCodeAt(0), _9 = '9'.charCodeAt(0);
    var NumericValue = /** @class */ (function (_super) {
        __extends(NumericValue, _super);
        function NumericValue(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(NumericValue.prototype, "type", {
            get: function () {
                return NodeType.NumericValue;
            },
            enumerable: false,
            configurable: true
        });
        NumericValue.prototype.getValue = function () {
            var raw = this.getText();
            var unitIdx = 0;
            var code;
            for (var i = 0, len = raw.length; i < len; i++) {
                code = raw.charCodeAt(i);
                if (!(_0 <= code && code <= _9 || code === _dot)) {
                    break;
                }
                unitIdx += 1;
            }
            return {
                value: raw.substring(0, unitIdx),
                unit: unitIdx < raw.length ? raw.substring(unitIdx) : undefined
            };
        };
        return NumericValue;
    }(Node));
    exports.NumericValue = NumericValue;
    var VariableDeclaration = /** @class */ (function (_super) {
        __extends(VariableDeclaration, _super);
        function VariableDeclaration(offset, length) {
            var _this = _super.call(this, offset, length) || this;
            _this.variable = null;
            _this.value = null;
            _this.needsSemicolon = true;
            return _this;
        }
        Object.defineProperty(VariableDeclaration.prototype, "type", {
            get: function () {
                return NodeType.VariableDeclaration;
            },
            enumerable: false,
            configurable: true
        });
        VariableDeclaration.prototype.setVariable = function (node) {
            if (node) {
                node.attachTo(this);
                this.variable = node;
                return true;
            }
            return false;
        };
        VariableDeclaration.prototype.getVariable = function () {
            return this.variable;
        };
        VariableDeclaration.prototype.getName = function () {
            return this.variable ? this.variable.getName() : '';
        };
        VariableDeclaration.prototype.setValue = function (node) {
            if (node) {
                node.attachTo(this);
                this.value = node;
                return true;
            }
            return false;
        };
        VariableDeclaration.prototype.getValue = function () {
            return this.value;
        };
        return VariableDeclaration;
    }(AbstractDeclaration));
    exports.VariableDeclaration = VariableDeclaration;
    var Interpolation = /** @class */ (function (_super) {
        __extends(Interpolation, _super);
        // private _interpolations: void; // workaround for https://github.com/Microsoft/TypeScript/issues/18276
        function Interpolation(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Interpolation.prototype, "type", {
            get: function () {
                return NodeType.Interpolation;
            },
            enumerable: false,
            configurable: true
        });
        return Interpolation;
    }(Node));
    exports.Interpolation = Interpolation;
    var Variable = /** @class */ (function (_super) {
        __extends(Variable, _super);
        function Variable(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(Variable.prototype, "type", {
            get: function () {
                return NodeType.VariableName;
            },
            enumerable: false,
            configurable: true
        });
        Variable.prototype.getName = function () {
            return this.getText();
        };
        return Variable;
    }(Node));
    exports.Variable = Variable;
    var ExtendsReference = /** @class */ (function (_super) {
        __extends(ExtendsReference, _super);
        function ExtendsReference(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(ExtendsReference.prototype, "type", {
            get: function () {
                return NodeType.ExtendsReference;
            },
            enumerable: false,
            configurable: true
        });
        ExtendsReference.prototype.getSelectors = function () {
            if (!this.selectors) {
                this.selectors = new Nodelist(this);
            }
            return this.selectors;
        };
        return ExtendsReference;
    }(Node));
    exports.ExtendsReference = ExtendsReference;
    var MixinContentReference = /** @class */ (function (_super) {
        __extends(MixinContentReference, _super);
        function MixinContentReference(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(MixinContentReference.prototype, "type", {
            get: function () {
                return NodeType.MixinContentReference;
            },
            enumerable: false,
            configurable: true
        });
        MixinContentReference.prototype.getArguments = function () {
            if (!this.arguments) {
                this.arguments = new Nodelist(this);
            }
            return this.arguments;
        };
        return MixinContentReference;
    }(Node));
    exports.MixinContentReference = MixinContentReference;
    var MixinContentDeclaration = /** @class */ (function (_super) {
        __extends(MixinContentDeclaration, _super);
        function MixinContentDeclaration(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(MixinContentDeclaration.prototype, "type", {
            get: function () {
                return NodeType.MixinContentReference;
            },
            enumerable: false,
            configurable: true
        });
        MixinContentDeclaration.prototype.getParameters = function () {
            if (!this.parameters) {
                this.parameters = new Nodelist(this);
            }
            return this.parameters;
        };
        return MixinContentDeclaration;
    }(BodyDeclaration));
    exports.MixinContentDeclaration = MixinContentDeclaration;
    var MixinReference = /** @class */ (function (_super) {
        __extends(MixinReference, _super);
        function MixinReference(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(MixinReference.prototype, "type", {
            get: function () {
                return NodeType.MixinReference;
            },
            enumerable: false,
            configurable: true
        });
        MixinReference.prototype.getNamespaces = function () {
            if (!this.namespaces) {
                this.namespaces = new Nodelist(this);
            }
            return this.namespaces;
        };
        MixinReference.prototype.setIdentifier = function (node) {
            return this.setNode('identifier', node, 0);
        };
        MixinReference.prototype.getIdentifier = function () {
            return this.identifier;
        };
        MixinReference.prototype.getName = function () {
            return this.identifier ? this.identifier.getText() : '';
        };
        MixinReference.prototype.getArguments = function () {
            if (!this.arguments) {
                this.arguments = new Nodelist(this);
            }
            return this.arguments;
        };
        MixinReference.prototype.setContent = function (node) {
            return this.setNode('content', node);
        };
        MixinReference.prototype.getContent = function () {
            return this.content;
        };
        return MixinReference;
    }(Node));
    exports.MixinReference = MixinReference;
    var MixinDeclaration = /** @class */ (function (_super) {
        __extends(MixinDeclaration, _super);
        function MixinDeclaration(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(MixinDeclaration.prototype, "type", {
            get: function () {
                return NodeType.MixinDeclaration;
            },
            enumerable: false,
            configurable: true
        });
        MixinDeclaration.prototype.setIdentifier = function (node) {
            return this.setNode('identifier', node, 0);
        };
        MixinDeclaration.prototype.getIdentifier = function () {
            return this.identifier;
        };
        MixinDeclaration.prototype.getName = function () {
            return this.identifier ? this.identifier.getText() : '';
        };
        MixinDeclaration.prototype.getParameters = function () {
            if (!this.parameters) {
                this.parameters = new Nodelist(this);
            }
            return this.parameters;
        };
        MixinDeclaration.prototype.setGuard = function (node) {
            if (node) {
                node.attachTo(this);
                this.guard = node;
            }
            return false;
        };
        return MixinDeclaration;
    }(BodyDeclaration));
    exports.MixinDeclaration = MixinDeclaration;
    var UnknownAtRule = /** @class */ (function (_super) {
        __extends(UnknownAtRule, _super);
        function UnknownAtRule(offset, length) {
            return _super.call(this, offset, length) || this;
        }
        Object.defineProperty(UnknownAtRule.prototype, "type", {
            get: function () {
                return NodeType.UnknownAtRule;
            },
            enumerable: false,
            configurable: true
        });
        UnknownAtRule.prototype.setAtRuleName = function (atRuleName) {
            this.atRuleName = atRuleName;
        };
        UnknownAtRule.prototype.getAtRuleName = function () {
            return this.atRuleName;
        };
        return UnknownAtRule;
    }(BodyDeclaration));
    exports.UnknownAtRule = UnknownAtRule;
    var ListEntry = /** @class */ (function (_super) {
        __extends(ListEntry, _super);
        function ListEntry() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        Object.defineProperty(ListEntry.prototype, "type", {
            get: function () {
                return NodeType.ListEntry;
            },
            enumerable: false,
            configurable: true
        });
        ListEntry.prototype.setKey = function (node) {
            return this.setNode('key', node, 0);
        };
        ListEntry.prototype.setValue = function (node) {
            return this.setNode('value', node, 1);
        };
        return ListEntry;
    }(Node));
    exports.ListEntry = ListEntry;
    var LessGuard = /** @class */ (function (_super) {
        __extends(LessGuard, _super);
        function LessGuard() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        LessGuard.prototype.getConditions = function () {
            if (!this.conditions) {
                this.conditions = new Nodelist(this);
            }
            return this.conditions;
        };
        return LessGuard;
    }(Node));
    exports.LessGuard = LessGuard;
    var GuardCondition = /** @class */ (function (_super) {
        __extends(GuardCondition, _super);
        function GuardCondition() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        GuardCondition.prototype.setVariable = function (node) {
            return this.setNode('variable', node);
        };
        return GuardCondition;
    }(Node));
    exports.GuardCondition = GuardCondition;
    var Module = /** @class */ (function (_super) {
        __extends(Module, _super);
        function Module() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        Object.defineProperty(Module.prototype, "type", {
            get: function () {
                return NodeType.Module;
            },
            enumerable: false,
            configurable: true
        });
        Module.prototype.setIdentifier = function (node) {
            return this.setNode('identifier', node, 0);
        };
        Module.prototype.getIdentifier = function () {
            return this.identifier;
        };
        return Module;
    }(Node));
    exports.Module = Module;
    var Level;
    (function (Level) {
        Level[Level["Ignore"] = 1] = "Ignore";
        Level[Level["Warning"] = 2] = "Warning";
        Level[Level["Error"] = 4] = "Error";
    })(Level = exports.Level || (exports.Level = {}));
    var Marker = /** @class */ (function () {
        function Marker(node, rule, level, message, offset, length) {
            if (offset === void 0) { offset = node.offset; }
            if (length === void 0) { length = node.length; }
            this.node = node;
            this.rule = rule;
            this.level = level;
            this.message = message || rule.message;
            this.offset = offset;
            this.length = length;
        }
        Marker.prototype.getRule = function () {
            return this.rule;
        };
        Marker.prototype.getLevel = function () {
            return this.level;
        };
        Marker.prototype.getOffset = function () {
            return this.offset;
        };
        Marker.prototype.getLength = function () {
            return this.length;
        };
        Marker.prototype.getNode = function () {
            return this.node;
        };
        Marker.prototype.getMessage = function () {
            return this.message;
        };
        return Marker;
    }());
    exports.Marker = Marker;
    /*
    export class DefaultVisitor implements IVisitor {
    
        public visitNode(node:Node):boolean {
            switch (node.type) {
                case NodeType.Stylesheet:
                    return this.visitStylesheet(<Stylesheet> node);
                case NodeType.FontFace:
                    return this.visitFontFace(<FontFace> node);
                case NodeType.Ruleset:
                    return this.visitRuleSet(<RuleSet> node);
                case NodeType.Selector:
                    return this.visitSelector(<Selector> node);
                case NodeType.SimpleSelector:
                    return this.visitSimpleSelector(<SimpleSelector> node);
                case NodeType.Declaration:
                    return this.visitDeclaration(<Declaration> node);
                case NodeType.Function:
                    return this.visitFunction(<Function> node);
                case NodeType.FunctionDeclaration:
                    return this.visitFunctionDeclaration(<FunctionDeclaration> node);
                case NodeType.FunctionParameter:
                    return this.visitFunctionParameter(<FunctionParameter> node);
                case NodeType.FunctionArgument:
                    return this.visitFunctionArgument(<FunctionArgument> node);
                case NodeType.Term:
                    return this.visitTerm(<Term> node);
                case NodeType.Declaration:
                    return this.visitExpression(<Expression> node);
                case NodeType.NumericValue:
                    return this.visitNumericValue(<NumericValue> node);
                case NodeType.Page:
                    return this.visitPage(<Page> node);
                case NodeType.PageBoxMarginBox:
                    return this.visitPageBoxMarginBox(<PageBoxMarginBox> node);
                case NodeType.Property:
                    return this.visitProperty(<Property> node);
                case NodeType.NumericValue:
                    return this.visitNodelist(<Nodelist> node);
                case NodeType.Import:
                    return this.visitImport(<Import> node);
                case NodeType.Namespace:
                    return this.visitNamespace(<Namespace> node);
                case NodeType.Keyframe:
                    return this.visitKeyframe(<Keyframe> node);
                case NodeType.KeyframeSelector:
                    return this.visitKeyframeSelector(<KeyframeSelector> node);
                case NodeType.MixinDeclaration:
                    return this.visitMixinDeclaration(<MixinDeclaration> node);
                case NodeType.MixinReference:
                    return this.visitMixinReference(<MixinReference> node);
                case NodeType.Variable:
                    return this.visitVariable(<Variable> node);
                case NodeType.VariableDeclaration:
                    return this.visitVariableDeclaration(<VariableDeclaration> node);
            }
            return this.visitUnknownNode(node);
        }
    
        public visitFontFace(node:FontFace):boolean {
            return true;
        }
    
        public visitKeyframe(node:Keyframe):boolean {
            return true;
        }
    
        public visitKeyframeSelector(node:KeyframeSelector):boolean {
            return true;
        }
    
        public visitStylesheet(node:Stylesheet):boolean {
            return true;
        }
    
        public visitProperty(Node:Property):boolean {
            return true;
        }
    
        public visitRuleSet(node:RuleSet):boolean {
            return true;
        }
    
        public visitSelector(node:Selector):boolean {
            return true;
        }
    
        public visitSimpleSelector(node:SimpleSelector):boolean {
            return true;
        }
    
        public visitDeclaration(node:Declaration):boolean {
            return true;
        }
    
        public visitFunction(node:Function):boolean {
            return true;
        }
    
        public visitFunctionDeclaration(node:FunctionDeclaration):boolean {
            return true;
        }
    
        public visitInvocation(node:Invocation):boolean {
            return true;
        }
    
        public visitTerm(node:Term):boolean {
            return true;
        }
    
        public visitImport(node:Import):boolean {
            return true;
        }
    
        public visitNamespace(node:Namespace):boolean {
            return true;
        }
    
        public visitExpression(node:Expression):boolean {
            return true;
        }
    
        public visitNumericValue(node:NumericValue):boolean {
            return true;
        }
    
        public visitPage(node:Page):boolean {
            return true;
        }
    
        public visitPageBoxMarginBox(node:PageBoxMarginBox):boolean {
            return true;
        }
    
        public visitNodelist(node:Nodelist):boolean {
            return true;
        }
    
        public visitVariableDeclaration(node:VariableDeclaration):boolean {
            return true;
        }
    
        public visitVariable(node:Variable):boolean {
            return true;
        }
    
        public visitMixinDeclaration(node:MixinDeclaration):boolean {
            return true;
        }
    
        public visitMixinReference(node:MixinReference):boolean {
            return true;
        }
    
        public visitUnknownNode(node:Node):boolean {
            return true;
        }
    }
    */
    var ParseErrorCollector = /** @class */ (function () {
        function ParseErrorCollector() {
            this.entries = [];
        }
        ParseErrorCollector.entries = function (node) {
            var visitor = new ParseErrorCollector();
            node.acceptVisitor(visitor);
            return visitor.entries;
        };
        ParseErrorCollector.prototype.visitNode = function (node) {
            if (node.isErroneous()) {
                node.collectIssues(this.entries);
            }
            return true;
        };
        return ParseErrorCollector;
    }());
    exports.ParseErrorCollector = ParseErrorCollector;
});

/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
define('vscode-nls/vscode-nls',["require", "exports"], function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.config = exports.loadMessageBundle = void 0;
    function format(message, args) {
        var result;
        if (args.length === 0) {
            result = message;
        }
        else {
            result = message.replace(/\{(\d+)\}/g, function (match, rest) {
                var index = rest[0];
                return typeof args[index] !== 'undefined' ? args[index] : match;
            });
        }
        return result;
    }
    function localize(key, message) {
        var args = [];
        for (var _i = 2; _i < arguments.length; _i++) {
            args[_i - 2] = arguments[_i];
        }
        return format(message, args);
    }
    function loadMessageBundle(file) {
        return localize;
    }
    exports.loadMessageBundle = loadMessageBundle;
    function config(opt) {
        return loadMessageBundle;
    }
    exports.config = config;
});

define('vscode-nls', ['vscode-nls/vscode-nls'], function (main) { return main; });

(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/parser/cssErrors',["require", "exports", "vscode-nls"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.ParseError = exports.CSSIssueType = void 0;
    var nls = require("vscode-nls");
    var localize = nls.loadMessageBundle();
    var CSSIssueType = /** @class */ (function () {
        function CSSIssueType(id, message) {
            this.id = id;
            this.message = message;
        }
        return CSSIssueType;
    }());
    exports.CSSIssueType = CSSIssueType;
    exports.ParseError = {
        NumberExpected: new CSSIssueType('css-numberexpected', localize('expected.number', "number expected")),
        ConditionExpected: new CSSIssueType('css-conditionexpected', localize('expected.condt', "condition expected")),
        RuleOrSelectorExpected: new CSSIssueType('css-ruleorselectorexpected', localize('expected.ruleorselector', "at-rule or selector expected")),
        DotExpected: new CSSIssueType('css-dotexpected', localize('expected.dot', "dot expected")),
        ColonExpected: new CSSIssueType('css-colonexpected', localize('expected.colon', "colon expected")),
        SemiColonExpected: new CSSIssueType('css-semicolonexpected', localize('expected.semicolon', "semi-colon expected")),
        TermExpected: new CSSIssueType('css-termexpected', localize('expected.term', "term expected")),
        ExpressionExpected: new CSSIssueType('css-expressionexpected', localize('expected.expression', "expression expected")),
        OperatorExpected: new CSSIssueType('css-operatorexpected', localize('expected.operator', "operator expected")),
        IdentifierExpected: new CSSIssueType('css-identifierexpected', localize('expected.ident', "identifier expected")),
        PercentageExpected: new CSSIssueType('css-percentageexpected', localize('expected.percentage', "percentage expected")),
        URIOrStringExpected: new CSSIssueType('css-uriorstringexpected', localize('expected.uriorstring', "uri or string expected")),
        URIExpected: new CSSIssueType('css-uriexpected', localize('expected.uri', "URI expected")),
        VariableNameExpected: new CSSIssueType('css-varnameexpected', localize('expected.varname', "variable name expected")),
        VariableValueExpected: new CSSIssueType('css-varvalueexpected', localize('expected.varvalue', "variable value expected")),
        PropertyValueExpected: new CSSIssueType('css-propertyvalueexpected', localize('expected.propvalue', "property value expected")),
        LeftCurlyExpected: new CSSIssueType('css-lcurlyexpected', localize('expected.lcurly', "{ expected")),
        RightCurlyExpected: new CSSIssueType('css-rcurlyexpected', localize('expected.rcurly', "} expected")),
        LeftSquareBracketExpected: new CSSIssueType('css-rbracketexpected', localize('expected.lsquare', "[ expected")),
        RightSquareBracketExpected: new CSSIssueType('css-lbracketexpected', localize('expected.rsquare', "] expected")),
        LeftParenthesisExpected: new CSSIssueType('css-lparentexpected', localize('expected.lparen', "( expected")),
        RightParenthesisExpected: new CSSIssueType('css-rparentexpected', localize('expected.rparent', ") expected")),
        CommaExpected: new CSSIssueType('css-commaexpected', localize('expected.comma', "comma expected")),
        PageDirectiveOrDeclarationExpected: new CSSIssueType('css-pagedirordeclexpected', localize('expected.pagedirordecl', "page directive or declaraton expected")),
        UnknownAtRule: new CSSIssueType('css-unknownatrule', localize('unknown.atrule', "at-rule unknown")),
        UnknownKeyword: new CSSIssueType('css-unknownkeyword', localize('unknown.keyword', "unknown keyword")),
        SelectorExpected: new CSSIssueType('css-selectorexpected', localize('expected.selector', "selector expected")),
        StringLiteralExpected: new CSSIssueType('css-stringliteralexpected', localize('expected.stringliteral', "string literal expected")),
        WhitespaceExpected: new CSSIssueType('css-whitespaceexpected', localize('expected.whitespace', "whitespace expected")),
        MediaQueryExpected: new CSSIssueType('css-mediaqueryexpected', localize('expected.mediaquery', "media query expected")),
        IdentifierOrWildcardExpected: new CSSIssueType('css-idorwildcardexpected', localize('expected.idorwildcard', "identifier or wildcard expected")),
        WildcardExpected: new CSSIssueType('css-wildcardexpected', localize('expected.wildcard', "wildcard expected")),
        IdentifierOrVariableExpected: new CSSIssueType('css-idorvarexpected', localize('expected.idorvar', "identifier or variable expected")),
    };
});

(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/languageFacts/entry',["require", "exports"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.getBrowserLabel = exports.textToMarkedString = exports.getEntryDescription = exports.browserNames = void 0;
    exports.browserNames = {
        E: 'Edge',
        FF: 'Firefox',
        S: 'Safari',
        C: 'Chrome',
        IE: 'IE',
        O: 'Opera'
    };
    function getEntryStatus(status) {
        switch (status) {
            case 'experimental':
                return '⚠️ Property is experimental. Be cautious when using it.️\n\n';
            case 'nonstandard':
                return '🚨️ Property is nonstandard. Avoid using it.\n\n';
            case 'obsolete':
                return '🚨️️️ Property is obsolete. Avoid using it.\n\n';
            default:
                return '';
        }
    }
    function getEntryDescription(entry, doesSupportMarkdown, settings) {
        var result;
        if (doesSupportMarkdown) {
            result = {
                kind: 'markdown',
                value: getEntryMarkdownDescription(entry, settings)
            };
        }
        else {
            result = {
                kind: 'plaintext',
                value: getEntryStringDescription(entry, settings)
            };
        }
        if (result.value === '') {
            return undefined;
        }
        return result;
    }
    exports.getEntryDescription = getEntryDescription;
    function textToMarkedString(text) {
        text = text.replace(/[\\`*_{}[\]()#+\-.!]/g, '\\$&'); // escape markdown syntax tokens: http://daringfireball.net/projects/markdown/syntax#backslash
        return text.replace(/</g, '&lt;').replace(/>/g, '&gt;');
    }
    exports.textToMarkedString = textToMarkedString;
    function getEntryStringDescription(entry, settings) {
        if (!entry.description || entry.description === '') {
            return '';
        }
        if (typeof entry.description !== 'string') {
            return entry.description.value;
        }
        var result = '';
        if ((settings === null || settings === void 0 ? void 0 : settings.documentation) !== false) {
            if (entry.status) {
                result += getEntryStatus(entry.status);
            }
            result += entry.description;
            var browserLabel = getBrowserLabel(entry.browsers);
            if (browserLabel) {
                result += '\n(' + browserLabel + ')';
            }
            if ('syntax' in entry) {
                result += "\n\nSyntax: " + entry.syntax;
            }
        }
        if (entry.references && entry.references.length > 0 && (settings === null || settings === void 0 ? void 0 : settings.references) !== false) {
            if (result.length > 0) {
                result += '\n\n';
            }
            result += entry.references.map(function (r) {
                return r.name + ": " + r.url;
            }).join(' | ');
        }
        return result;
    }
    function getEntryMarkdownDescription(entry, settings) {
        if (!entry.description || entry.description === '') {
            return '';
        }
        var result = '';
        if ((settings === null || settings === void 0 ? void 0 : settings.documentation) !== false) {
            if (entry.status) {
                result += getEntryStatus(entry.status);
            }
            var description = typeof entry.description === 'string' ? entry.description : entry.description.value;
            result += textToMarkedString(description);
            var browserLabel = getBrowserLabel(entry.browsers);
            if (browserLabel) {
                result += '\n\n(' + textToMarkedString(browserLabel) + ')';
            }
            if ('syntax' in entry && entry.syntax) {
                result += "\n\nSyntax: " + textToMarkedString(entry.syntax);
            }
        }
        if (entry.references && entry.references.length > 0 && (settings === null || settings === void 0 ? void 0 : settings.references) !== false) {
            if (result.length > 0) {
                result += '\n\n';
            }
            result += entry.references.map(function (r) {
                return "[" + r.name + "](" + r.url + ")";
            }).join(' | ');
        }
        return result;
    }
    /**
     * Input is like `["E12","FF49","C47","IE","O"]`
     * Output is like `Edge 12, Firefox 49, Chrome 47, IE, Opera`
     */
    function getBrowserLabel(browsers) {
        if (browsers === void 0) { browsers = []; }
        if (browsers.length === 0) {
            return null;
        }
        return browsers
            .map(function (b) {
            var result = '';
            var matches = b.match(/([A-Z]+)(\d+)?/);
            var name = matches[1];
            var version = matches[2];
            if (name in exports.browserNames) {
                result += exports.browserNames[name];
            }
            if (version) {
                result += ' ' + version;
            }
            return result;
        })
            .join(', ');
    }
    exports.getBrowserLabel = getBrowserLabel;
});

/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/languageFacts/colors',["require", "exports", "../parser/cssNodes", "vscode-nls"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.getColorValue = exports.hslFromColor = exports.colorFromHSL = exports.colorFrom256RGB = exports.colorFromHex = exports.hexDigit = exports.isColorValue = exports.isColorConstructor = exports.colorKeywords = exports.colors = exports.colorFunctions = void 0;
    var nodes = require("../parser/cssNodes");
    var nls = require("vscode-nls");
    var localize = nls.loadMessageBundle();
    exports.colorFunctions = [
        { func: 'rgb($red, $green, $blue)', desc: localize('css.builtin.rgb', 'Creates a Color from red, green, and blue values.') },
        { func: 'rgba($red, $green, $blue, $alpha)', desc: localize('css.builtin.rgba', 'Creates a Color from red, green, blue, and alpha values.') },
        { func: 'hsl($hue, $saturation, $lightness)', desc: localize('css.builtin.hsl', 'Creates a Color from hue, saturation, and lightness values.') },
        { func: 'hsla($hue, $saturation, $lightness, $alpha)', desc: localize('css.builtin.hsla', 'Creates a Color from hue, saturation, lightness, and alpha values.') }
    ];
    exports.colors = {
        aliceblue: '#f0f8ff',
        antiquewhite: '#faebd7',
        aqua: '#00ffff',
        aquamarine: '#7fffd4',
        azure: '#f0ffff',
        beige: '#f5f5dc',
        bisque: '#ffe4c4',
        black: '#000000',
        blanchedalmond: '#ffebcd',
        blue: '#0000ff',
        blueviolet: '#8a2be2',
        brown: '#a52a2a',
        burlywood: '#deb887',
        cadetblue: '#5f9ea0',
        chartreuse: '#7fff00',
        chocolate: '#d2691e',
        coral: '#ff7f50',
        cornflowerblue: '#6495ed',
        cornsilk: '#fff8dc',
        crimson: '#dc143c',
        cyan: '#00ffff',
        darkblue: '#00008b',
        darkcyan: '#008b8b',
        darkgoldenrod: '#b8860b',
        darkgray: '#a9a9a9',
        darkgrey: '#a9a9a9',
        darkgreen: '#006400',
        darkkhaki: '#bdb76b',
        darkmagenta: '#8b008b',
        darkolivegreen: '#556b2f',
        darkorange: '#ff8c00',
        darkorchid: '#9932cc',
        darkred: '#8b0000',
        darksalmon: '#e9967a',
        darkseagreen: '#8fbc8f',
        darkslateblue: '#483d8b',
        darkslategray: '#2f4f4f',
        darkslategrey: '#2f4f4f',
        darkturquoise: '#00ced1',
        darkviolet: '#9400d3',
        deeppink: '#ff1493',
        deepskyblue: '#00bfff',
        dimgray: '#696969',
        dimgrey: '#696969',
        dodgerblue: '#1e90ff',
        firebrick: '#b22222',
        floralwhite: '#fffaf0',
        forestgreen: '#228b22',
        fuchsia: '#ff00ff',
        gainsboro: '#dcdcdc',
        ghostwhite: '#f8f8ff',
        gold: '#ffd700',
        goldenrod: '#daa520',
        gray: '#808080',
        grey: '#808080',
        green: '#008000',
        greenyellow: '#adff2f',
        honeydew: '#f0fff0',
        hotpink: '#ff69b4',
        indianred: '#cd5c5c',
        indigo: '#4b0082',
        ivory: '#fffff0',
        khaki: '#f0e68c',
        lavender: '#e6e6fa',
        lavenderblush: '#fff0f5',
        lawngreen: '#7cfc00',
        lemonchiffon: '#fffacd',
        lightblue: '#add8e6',
        lightcoral: '#f08080',
        lightcyan: '#e0ffff',
        lightgoldenrodyellow: '#fafad2',
        lightgray: '#d3d3d3',
        lightgrey: '#d3d3d3',
        lightgreen: '#90ee90',
        lightpink: '#ffb6c1',
        lightsalmon: '#ffa07a',
        lightseagreen: '#20b2aa',
        lightskyblue: '#87cefa',
        lightslategray: '#778899',
        lightslategrey: '#778899',
        lightsteelblue: '#b0c4de',
        lightyellow: '#ffffe0',
        lime: '#00ff00',
        limegreen: '#32cd32',
        linen: '#faf0e6',
        magenta: '#ff00ff',
        maroon: '#800000',
        mediumaquamarine: '#66cdaa',
        mediumblue: '#0000cd',
        mediumorchid: '#ba55d3',
        mediumpurple: '#9370d8',
        mediumseagreen: '#3cb371',
        mediumslateblue: '#7b68ee',
        mediumspringgreen: '#00fa9a',
        mediumturquoise: '#48d1cc',
        mediumvioletred: '#c71585',
        midnightblue: '#191970',
        mintcream: '#f5fffa',
        mistyrose: '#ffe4e1',
        moccasin: '#ffe4b5',
        navajowhite: '#ffdead',
        navy: '#000080',
        oldlace: '#fdf5e6',
        olive: '#808000',
        olivedrab: '#6b8e23',
        orange: '#ffa500',
        orangered: '#ff4500',
        orchid: '#da70d6',
        palegoldenrod: '#eee8aa',
        palegreen: '#98fb98',
        paleturquoise: '#afeeee',
        palevioletred: '#d87093',
        papayawhip: '#ffefd5',
        peachpuff: '#ffdab9',
        peru: '#cd853f',
        pink: '#ffc0cb',
        plum: '#dda0dd',
        powderblue: '#b0e0e6',
        purple: '#800080',
        red: '#ff0000',
        rebeccapurple: '#663399',
        rosybrown: '#bc8f8f',
        royalblue: '#4169e1',
        saddlebrown: '#8b4513',
        salmon: '#fa8072',
        sandybrown: '#f4a460',
        seagreen: '#2e8b57',
        seashell: '#fff5ee',
        sienna: '#a0522d',
        silver: '#c0c0c0',
        skyblue: '#87ceeb',
        slateblue: '#6a5acd',
        slategray: '#708090',
        slategrey: '#708090',
        snow: '#fffafa',
        springgreen: '#00ff7f',
        steelblue: '#4682b4',
        tan: '#d2b48c',
        teal: '#008080',
        thistle: '#d8bfd8',
        tomato: '#ff6347',
        turquoise: '#40e0d0',
        violet: '#ee82ee',
        wheat: '#f5deb3',
        white: '#ffffff',
        whitesmoke: '#f5f5f5',
        yellow: '#ffff00',
        yellowgreen: '#9acd32'
    };
    exports.colorKeywords = {
        'currentColor': 'The value of the \'color\' property. The computed value of the \'currentColor\' keyword is the computed value of the \'color\' property. If the \'currentColor\' keyword is set on the \'color\' property itself, it is treated as \'color:inherit\' at parse time.',
        'transparent': 'Fully transparent. This keyword can be considered a shorthand for rgba(0,0,0,0) which is its computed value.',
    };
    function getNumericValue(node, factor) {
        var val = node.getText();
        var m = val.match(/^([-+]?[0-9]*\.?[0-9]+)(%?)$/);
        if (m) {
            if (m[2]) {
                factor = 100.0;
            }
            var result = parseFloat(m[1]) / factor;
            if (result >= 0 && result <= 1) {
                return result;
            }
        }
        throw new Error();
    }
    function getAngle(node) {
        var val = node.getText();
        var m = val.match(/^([-+]?[0-9]*\.?[0-9]+)(deg)?$/);
        if (m) {
            return parseFloat(val) % 360;
        }
        throw new Error();
    }
    function isColorConstructor(node) {
        var name = node.getName();
        if (!name) {
            return false;
        }
        return /^(rgb|rgba|hsl|hsla)$/gi.test(name);
    }
    exports.isColorConstructor = isColorConstructor;
    /**
     * Returns true if the node is a color value - either
     * defined a hex number, as rgb or rgba function, or
     * as color name.
     */
    function isColorValue(node) {
        if (node.type === nodes.NodeType.HexColorValue) {
            return true;
        }
        else if (node.type === nodes.NodeType.Function) {
            return isColorConstructor(node);
        }
        else if (node.type === nodes.NodeType.Identifier) {
            if (node.parent && node.parent.type !== nodes.NodeType.Term) {
                return false;
            }
            var candidateColor = node.getText().toLowerCase();
            if (candidateColor === 'none') {
                return false;
            }
            if (exports.colors[candidateColor]) {
                return true;
            }
        }
        return false;
    }
    exports.isColorValue = isColorValue;
    var Digit0 = 48;
    var Digit9 = 57;
    var A = 65;
    var F = 70;
    var a = 97;
    var f = 102;
    function hexDigit(charCode) {
        if (charCode < Digit0) {
            return 0;
        }
        if (charCode <= Digit9) {
            return charCode - Digit0;
        }
        if (charCode < a) {
            charCode += (a - A);
        }
        if (charCode >= a && charCode <= f) {
            return charCode - a + 10;
        }
        return 0;
    }
    exports.hexDigit = hexDigit;
    function colorFromHex(text) {
        if (text[0] !== '#') {
            return null;
        }
        switch (text.length) {
            case 4:
                return {
                    red: (hexDigit(text.charCodeAt(1)) * 0x11) / 255.0,
                    green: (hexDigit(text.charCodeAt(2)) * 0x11) / 255.0,
                    blue: (hexDigit(text.charCodeAt(3)) * 0x11) / 255.0,
                    alpha: 1
                };
            case 5:
                return {
                    red: (hexDigit(text.charCodeAt(1)) * 0x11) / 255.0,
                    green: (hexDigit(text.charCodeAt(2)) * 0x11) / 255.0,
                    blue: (hexDigit(text.charCodeAt(3)) * 0x11) / 255.0,
                    alpha: (hexDigit(text.charCodeAt(4)) * 0x11) / 255.0,
                };
            case 7:
                return {
                    red: (hexDigit(text.charCodeAt(1)) * 0x10 + hexDigit(text.charCodeAt(2))) / 255.0,
                    green: (hexDigit(text.charCodeAt(3)) * 0x10 + hexDigit(text.charCodeAt(4))) / 255.0,
                    blue: (hexDigit(text.charCodeAt(5)) * 0x10 + hexDigit(text.charCodeAt(6))) / 255.0,
                    alpha: 1
                };
            case 9:
                return {
                    red: (hexDigit(text.charCodeAt(1)) * 0x10 + hexDigit(text.charCodeAt(2))) / 255.0,
                    green: (hexDigit(text.charCodeAt(3)) * 0x10 + hexDigit(text.charCodeAt(4))) / 255.0,
                    blue: (hexDigit(text.charCodeAt(5)) * 0x10 + hexDigit(text.charCodeAt(6))) / 255.0,
                    alpha: (hexDigit(text.charCodeAt(7)) * 0x10 + hexDigit(text.charCodeAt(8))) / 255.0
                };
        }
        return null;
    }
    exports.colorFromHex = colorFromHex;
    function colorFrom256RGB(red, green, blue, alpha) {
        if (alpha === void 0) { alpha = 1.0; }
        return {
            red: red / 255.0,
            green: green / 255.0,
            blue: blue / 255.0,
            alpha: alpha
        };
    }
    exports.colorFrom256RGB = colorFrom256RGB;
    function colorFromHSL(hue, sat, light, alpha) {
        if (alpha === void 0) { alpha = 1.0; }
        hue = hue / 60.0;
        if (sat === 0) {
            return { red: light, green: light, blue: light, alpha: alpha };
        }
        else {
            var hueToRgb = function (t1, t2, hue) {
                while (hue < 0) {
                    hue += 6;
                }
                while (hue >= 6) {
                    hue -= 6;
                }
                if (hue < 1) {
                    return (t2 - t1) * hue + t1;
                }
                if (hue < 3) {
                    return t2;
                }
                if (hue < 4) {
                    return (t2 - t1) * (4 - hue) + t1;
                }
                return t1;
            };
            var t2 = light <= 0.5 ? (light * (sat + 1)) : (light + sat - (light * sat));
            var t1 = light * 2 - t2;
            return { red: hueToRgb(t1, t2, hue + 2), green: hueToRgb(t1, t2, hue), blue: hueToRgb(t1, t2, hue - 2), alpha: alpha };
        }
    }
    exports.colorFromHSL = colorFromHSL;
    function hslFromColor(rgba) {
        var r = rgba.red;
        var g = rgba.green;
        var b = rgba.blue;
        var a = rgba.alpha;
        var max = Math.max(r, g, b);
        var min = Math.min(r, g, b);
        var h = 0;
        var s = 0;
        var l = (min + max) / 2;
        var chroma = max - min;
        if (chroma > 0) {
            s = Math.min((l <= 0.5 ? chroma / (2 * l) : chroma / (2 - (2 * l))), 1);
            switch (max) {
                case r:
                    h = (g - b) / chroma + (g < b ? 6 : 0);
                    break;
                case g:
                    h = (b - r) / chroma + 2;
                    break;
                case b:
                    h = (r - g) / chroma + 4;
                    break;
            }
            h *= 60;
            h = Math.round(h);
        }
        return { h: h, s: s, l: l, a: a };
    }
    exports.hslFromColor = hslFromColor;
    function getColorValue(node) {
        if (node.type === nodes.NodeType.HexColorValue) {
            var text = node.getText();
            return colorFromHex(text);
        }
        else if (node.type === nodes.NodeType.Function) {
            var functionNode = node;
            var name = functionNode.getName();
            var colorValues = functionNode.getArguments().getChildren();
            if (!name || colorValues.length < 3 || colorValues.length > 4) {
                return null;
            }
            try {
                var alpha = colorValues.length === 4 ? getNumericValue(colorValues[3], 1) : 1;
                if (name === 'rgb' || name === 'rgba') {
                    return {
                        red: getNumericValue(colorValues[0], 255.0),
                        green: getNumericValue(colorValues[1], 255.0),
                        blue: getNumericValue(colorValues[2], 255.0),
                        alpha: alpha
                    };
                }
                else if (name === 'hsl' || name === 'hsla') {
                    var h = getAngle(colorValues[0]);
                    var s = getNumericValue(colorValues[1], 100.0);
                    var l = getNumericValue(colorValues[2], 100.0);
                    return colorFromHSL(h, s, l, alpha);
                }
            }
            catch (e) {
                // parse error on numeric value
                return null;
            }
        }
        else if (node.type === nodes.NodeType.Identifier) {
            if (node.parent && node.parent.type !== nodes.NodeType.Term) {
                return null;
            }
            var term = node.parent;
            if (term && term.parent && term.parent.type === nodes.NodeType.BinaryExpression) {
                var expression = term.parent;
                if (expression.parent && expression.parent.type === nodes.NodeType.ListEntry && expression.parent.key === expression) {
                    return null;
                }
            }
            var candidateColor = node.getText().toLowerCase();
            if (candidateColor === 'none') {
                return null;
            }
            var colorHex = exports.colors[candidateColor];
            if (colorHex) {
                return colorFromHex(colorHex);
            }
        }
        return null;
    }
    exports.getColorValue = getColorValue;
});

(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/languageFacts/builtinData',["require", "exports"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.pageBoxDirectives = exports.svgElements = exports.html5Tags = exports.units = exports.basicShapeFunctions = exports.transitionTimingFunctions = exports.imageFunctions = exports.cssWideKeywords = exports.geometryBoxKeywords = exports.boxKeywords = exports.lineWidthKeywords = exports.lineStyleKeywords = exports.repeatStyleKeywords = exports.positionKeywords = void 0;
    exports.positionKeywords = {
        'bottom': 'Computes to ‘100%’ for the vertical position if one or two values are given, otherwise specifies the bottom edge as the origin for the next offset.',
        'center': 'Computes to ‘50%’ (‘left 50%’) for the horizontal position if the horizontal position is not otherwise specified, or ‘50%’ (‘top 50%’) for the vertical position if it is.',
        'left': 'Computes to ‘0%’ for the horizontal position if one or two values are given, otherwise specifies the left edge as the origin for the next offset.',
        'right': 'Computes to ‘100%’ for the horizontal position if one or two values are given, otherwise specifies the right edge as the origin for the next offset.',
        'top': 'Computes to ‘0%’ for the vertical position if one or two values are given, otherwise specifies the top edge as the origin for the next offset.'
    };
    exports.repeatStyleKeywords = {
        'no-repeat': 'Placed once and not repeated in this direction.',
        'repeat': 'Repeated in this direction as often as needed to cover the background painting area.',
        'repeat-x': 'Computes to ‘repeat no-repeat’.',
        'repeat-y': 'Computes to ‘no-repeat repeat’.',
        'round': 'Repeated as often as will fit within the background positioning area. If it doesn’t fit a whole number of times, it is rescaled so that it does.',
        'space': 'Repeated as often as will fit within the background positioning area without being clipped and then the images are spaced out to fill the area.'
    };
    exports.lineStyleKeywords = {
        'dashed': 'A series of square-ended dashes.',
        'dotted': 'A series of round dots.',
        'double': 'Two parallel solid lines with some space between them.',
        'groove': 'Looks as if it were carved in the canvas.',
        'hidden': 'Same as ‘none’, but has different behavior in the border conflict resolution rules for border-collapsed tables.',
        'inset': 'Looks as if the content on the inside of the border is sunken into the canvas.',
        'none': 'No border. Color and width are ignored.',
        'outset': 'Looks as if the content on the inside of the border is coming out of the canvas.',
        'ridge': 'Looks as if it were coming out of the canvas.',
        'solid': 'A single line segment.'
    };
    exports.lineWidthKeywords = ['medium', 'thick', 'thin'];
    exports.boxKeywords = {
        'border-box': 'The background is painted within (clipped to) the border box.',
        'content-box': 'The background is painted within (clipped to) the content box.',
        'padding-box': 'The background is painted within (clipped to) the padding box.'
    };
    exports.geometryBoxKeywords = {
        'margin-box': 'Uses the margin box as reference box.',
        'fill-box': 'Uses the object bounding box as reference box.',
        'stroke-box': 'Uses the stroke bounding box as reference box.',
        'view-box': 'Uses the nearest SVG viewport as reference box.'
    };
    exports.cssWideKeywords = {
        'initial': 'Represents the value specified as the property’s initial value.',
        'inherit': 'Represents the computed value of the property on the element’s parent.',
        'unset': 'Acts as either `inherit` or `initial`, depending on whether the property is inherited or not.'
    };
    exports.imageFunctions = {
        'url()': 'Reference an image file by URL',
        'image()': 'Provide image fallbacks and annotations.',
        '-webkit-image-set()': 'Provide multiple resolutions. Remember to use unprefixed image-set() in addition.',
        'image-set()': 'Provide multiple resolutions of an image and const the UA decide which is most appropriate in a given situation.',
        '-moz-element()': 'Use an element in the document as an image. Remember to use unprefixed element() in addition.',
        'element()': 'Use an element in the document as an image.',
        'cross-fade()': 'Indicates the two images to be combined and how far along in the transition the combination is.',
        '-webkit-gradient()': 'Deprecated. Use modern linear-gradient() or radial-gradient() instead.',
        '-webkit-linear-gradient()': 'Linear gradient. Remember to use unprefixed version in addition.',
        '-moz-linear-gradient()': 'Linear gradient. Remember to use unprefixed version in addition.',
        '-o-linear-gradient()': 'Linear gradient. Remember to use unprefixed version in addition.',
        'linear-gradient()': 'A linear gradient is created by specifying a straight gradient line, and then several colors placed along that line.',
        '-webkit-repeating-linear-gradient()': 'Repeating Linear gradient. Remember to use unprefixed version in addition.',
        '-moz-repeating-linear-gradient()': 'Repeating Linear gradient. Remember to use unprefixed version in addition.',
        '-o-repeating-linear-gradient()': 'Repeating Linear gradient. Remember to use unprefixed version in addition.',
        'repeating-linear-gradient()': 'Same as linear-gradient, except the color-stops are repeated infinitely in both directions, with their positions shifted by multiples of the difference between the last specified color-stop’s position and the first specified color-stop’s position.',
        '-webkit-radial-gradient()': 'Radial gradient. Remember to use unprefixed version in addition.',
        '-moz-radial-gradient()': 'Radial gradient. Remember to use unprefixed version in addition.',
        'radial-gradient()': 'Colors emerge from a single point and smoothly spread outward in a circular or elliptical shape.',
        '-webkit-repeating-radial-gradient()': 'Repeating radial gradient. Remember to use unprefixed version in addition.',
        '-moz-repeating-radial-gradient()': 'Repeating radial gradient. Remember to use unprefixed version in addition.',
        'repeating-radial-gradient()': 'Same as radial-gradient, except the color-stops are repeated infinitely in both directions, with their positions shifted by multiples of the difference between the last specified color-stop’s position and the first specified color-stop’s position.'
    };
    exports.transitionTimingFunctions = {
        'ease': 'Equivalent to cubic-bezier(0.25, 0.1, 0.25, 1.0).',
        'ease-in': 'Equivalent to cubic-bezier(0.42, 0, 1.0, 1.0).',
        'ease-in-out': 'Equivalent to cubic-bezier(0.42, 0, 0.58, 1.0).',
        'ease-out': 'Equivalent to cubic-bezier(0, 0, 0.58, 1.0).',
        'linear': 'Equivalent to cubic-bezier(0.0, 0.0, 1.0, 1.0).',
        'step-end': 'Equivalent to steps(1, end).',
        'step-start': 'Equivalent to steps(1, start).',
        'steps()': 'The first parameter specifies the number of intervals in the function. The second parameter, which is optional, is either the value “start” or “end”.',
        'cubic-bezier()': 'Specifies a cubic-bezier curve. The four values specify points P1 and P2  of the curve as (x1, y1, x2, y2).',
        'cubic-bezier(0.6, -0.28, 0.735, 0.045)': 'Ease-in Back. Overshoots.',
        'cubic-bezier(0.68, -0.55, 0.265, 1.55)': 'Ease-in-out Back. Overshoots.',
        'cubic-bezier(0.175, 0.885, 0.32, 1.275)': 'Ease-out Back. Overshoots.',
        'cubic-bezier(0.6, 0.04, 0.98, 0.335)': 'Ease-in Circular. Based on half circle.',
        'cubic-bezier(0.785, 0.135, 0.15, 0.86)': 'Ease-in-out Circular. Based on half circle.',
        'cubic-bezier(0.075, 0.82, 0.165, 1)': 'Ease-out Circular. Based on half circle.',
        'cubic-bezier(0.55, 0.055, 0.675, 0.19)': 'Ease-in Cubic. Based on power of three.',
        'cubic-bezier(0.645, 0.045, 0.355, 1)': 'Ease-in-out Cubic. Based on power of three.',
        'cubic-bezier(0.215, 0.610, 0.355, 1)': 'Ease-out Cubic. Based on power of three.',
        'cubic-bezier(0.95, 0.05, 0.795, 0.035)': 'Ease-in Exponential. Based on two to the power ten.',
        'cubic-bezier(1, 0, 0, 1)': 'Ease-in-out Exponential. Based on two to the power ten.',
        'cubic-bezier(0.19, 1, 0.22, 1)': 'Ease-out Exponential. Based on two to the power ten.',
        'cubic-bezier(0.47, 0, 0.745, 0.715)': 'Ease-in Sine.',
        'cubic-bezier(0.445, 0.05, 0.55, 0.95)': 'Ease-in-out Sine.',
        'cubic-bezier(0.39, 0.575, 0.565, 1)': 'Ease-out Sine.',
        'cubic-bezier(0.55, 0.085, 0.68, 0.53)': 'Ease-in Quadratic. Based on power of two.',
        'cubic-bezier(0.455, 0.03, 0.515, 0.955)': 'Ease-in-out Quadratic. Based on power of two.',
        'cubic-bezier(0.25, 0.46, 0.45, 0.94)': 'Ease-out Quadratic. Based on power of two.',
        'cubic-bezier(0.895, 0.03, 0.685, 0.22)': 'Ease-in Quartic. Based on power of four.',
        'cubic-bezier(0.77, 0, 0.175, 1)': 'Ease-in-out Quartic. Based on power of four.',
        'cubic-bezier(0.165, 0.84, 0.44, 1)': 'Ease-out Quartic. Based on power of four.',
        'cubic-bezier(0.755, 0.05, 0.855, 0.06)': 'Ease-in Quintic. Based on power of five.',
        'cubic-bezier(0.86, 0, 0.07, 1)': 'Ease-in-out Quintic. Based on power of five.',
        'cubic-bezier(0.23, 1, 0.320, 1)': 'Ease-out Quintic. Based on power of five.'
    };
    exports.basicShapeFunctions = {
        'circle()': 'Defines a circle.',
        'ellipse()': 'Defines an ellipse.',
        'inset()': 'Defines an inset rectangle.',
        'polygon()': 'Defines a polygon.'
    };
    exports.units = {
        'length': ['em', 'rem', 'ex', 'px', 'cm', 'mm', 'in', 'pt', 'pc', 'ch', 'vw', 'vh', 'vmin', 'vmax'],
        'angle': ['deg', 'rad', 'grad', 'turn'],
        'time': ['ms', 's'],
        'frequency': ['Hz', 'kHz'],
        'resolution': ['dpi', 'dpcm', 'dppx'],
        'percentage': ['%', 'fr']
    };
    exports.html5Tags = ['a', 'abbr', 'address', 'area', 'article', 'aside', 'audio', 'b', 'base', 'bdi', 'bdo', 'blockquote', 'body', 'br', 'button', 'canvas', 'caption',
        'cite', 'code', 'col', 'colgroup', 'data', 'datalist', 'dd', 'del', 'details', 'dfn', 'dialog', 'div', 'dl', 'dt', 'em', 'embed', 'fieldset', 'figcaption', 'figure', 'footer',
        'form', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'head', 'header', 'hgroup', 'hr', 'html', 'i', 'iframe', 'img', 'input', 'ins', 'kbd', 'keygen', 'label', 'legend', 'li', 'link',
        'main', 'map', 'mark', 'menu', 'menuitem', 'meta', 'meter', 'nav', 'noscript', 'object', 'ol', 'optgroup', 'option', 'output', 'p', 'param', 'picture', 'pre', 'progress', 'q',
        'rb', 'rp', 'rt', 'rtc', 'ruby', 's', 'samp', 'script', 'section', 'select', 'small', 'source', 'span', 'strong', 'style', 'sub', 'summary', 'sup', 'table', 'tbody', 'td',
        'template', 'textarea', 'tfoot', 'th', 'thead', 'time', 'title', 'tr', 'track', 'u', 'ul', 'const', 'video', 'wbr'];
    exports.svgElements = ['circle', 'clipPath', 'cursor', 'defs', 'desc', 'ellipse', 'feBlend', 'feColorMatrix', 'feComponentTransfer', 'feComposite', 'feConvolveMatrix', 'feDiffuseLighting',
        'feDisplacementMap', 'feDistantLight', 'feDropShadow', 'feFlood', 'feFuncA', 'feFuncB', 'feFuncG', 'feFuncR', 'feGaussianBlur', 'feImage', 'feMerge', 'feMergeNode', 'feMorphology',
        'feOffset', 'fePointLight', 'feSpecularLighting', 'feSpotLight', 'feTile', 'feTurbulence', 'filter', 'foreignObject', 'g', 'hatch', 'hatchpath', 'image', 'line', 'linearGradient',
        'marker', 'mask', 'mesh', 'meshpatch', 'meshrow', 'metadata', 'mpath', 'path', 'pattern', 'polygon', 'polyline', 'radialGradient', 'rect', 'set', 'solidcolor', 'stop', 'svg', 'switch',
        'symbol', 'text', 'textPath', 'tspan', 'use', 'view'];
    exports.pageBoxDirectives = [
        '@bottom-center', '@bottom-left', '@bottom-left-corner', '@bottom-right', '@bottom-right-corner',
        '@left-bottom', '@left-middle', '@left-top', '@right-bottom', '@right-middle', '@right-top',
        '@top-center', '@top-left', '@top-left-corner', '@top-right', '@top-right-corner'
    ];
});

var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __exportStar = (this && this.__exportStar) || function(m, exports) {
    for (var p in m) if (p !== "default" && !Object.prototype.hasOwnProperty.call(exports, p)) __createBinding(exports, m, p);
};
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/languageFacts/facts',["require", "exports", "./entry", "./colors", "./builtinData"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    __exportStar(require("./entry"), exports);
    __exportStar(require("./colors"), exports);
    __exportStar(require("./builtinData"), exports);
});

(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/utils/objects',["require", "exports"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.isDefined = exports.values = void 0;
    function values(obj) {
        return Object.keys(obj).map(function (key) { return obj[key]; });
    }
    exports.values = values;
    function isDefined(obj) {
        return typeof obj !== 'undefined';
    }
    exports.isDefined = isDefined;
});

(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/parser/cssParser',["require", "exports", "./cssScanner", "./cssNodes", "./cssErrors", "../languageFacts/facts", "../utils/objects"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.Parser = void 0;
    var cssScanner_1 = require("./cssScanner");
    var nodes = require("./cssNodes");
    var cssErrors_1 = require("./cssErrors");
    var languageFacts = require("../languageFacts/facts");
    var objects_1 = require("../utils/objects");
    /// <summary>
    /// A parser for the css core specification. See for reference:
    /// https://www.w3.org/TR/CSS21/grammar.html
    /// http://www.w3.org/TR/CSS21/syndata.html#tokenization
    /// </summary>
    var Parser = /** @class */ (function () {
        function Parser(scnr) {
            if (scnr === void 0) { scnr = new cssScanner_1.Scanner(); }
            this.keyframeRegex = /^@(\-(webkit|ms|moz|o)\-)?keyframes$/i;
            this.scanner = scnr;
            this.token = { type: cssScanner_1.TokenType.EOF, offset: -1, len: 0, text: '' };
            this.prevToken = undefined;
        }
        Parser.prototype.peekIdent = function (text) {
            return cssScanner_1.TokenType.Ident === this.token.type && text.length === this.token.text.length && text === this.token.text.toLowerCase();
        };
        Parser.prototype.peekKeyword = function (text) {
            return cssScanner_1.TokenType.AtKeyword === this.token.type && text.length === this.token.text.length && text === this.token.text.toLowerCase();
        };
        Parser.prototype.peekDelim = function (text) {
            return cssScanner_1.TokenType.Delim === this.token.type && text === this.token.text;
        };
        Parser.prototype.peek = function (type) {
            return type === this.token.type;
        };
        Parser.prototype.peekOne = function (types) {
            return types.indexOf(this.token.type) !== -1;
        };
        Parser.prototype.peekRegExp = function (type, regEx) {
            if (type !== this.token.type) {
                return false;
            }
            return regEx.test(this.token.text);
        };
        Parser.prototype.hasWhitespace = function () {
            return !!this.prevToken && (this.prevToken.offset + this.prevToken.len !== this.token.offset);
        };
        Parser.prototype.consumeToken = function () {
            this.prevToken = this.token;
            this.token = this.scanner.scan();
        };
        Parser.prototype.mark = function () {
            return {
                prev: this.prevToken,
                curr: this.token,
                pos: this.scanner.pos()
            };
        };
        Parser.prototype.restoreAtMark = function (mark) {
            this.prevToken = mark.prev;
            this.token = mark.curr;
            this.scanner.goBackTo(mark.pos);
        };
        Parser.prototype.try = function (func) {
            var pos = this.mark();
            var node = func();
            if (!node) {
                this.restoreAtMark(pos);
                return null;
            }
            return node;
        };
        Parser.prototype.acceptOneKeyword = function (keywords) {
            if (cssScanner_1.TokenType.AtKeyword === this.token.type) {
                for (var _i = 0, keywords_1 = keywords; _i < keywords_1.length; _i++) {
                    var keyword = keywords_1[_i];
                    if (keyword.length === this.token.text.length && keyword === this.token.text.toLowerCase()) {
                        this.consumeToken();
                        return true;
                    }
                }
            }
            return false;
        };
        Parser.prototype.accept = function (type) {
            if (type === this.token.type) {
                this.consumeToken();
                return true;
            }
            return false;
        };
        Parser.prototype.acceptIdent = function (text) {
            if (this.peekIdent(text)) {
                this.consumeToken();
                return true;
            }
            return false;
        };
        Parser.prototype.acceptKeyword = function (text) {
            if (this.peekKeyword(text)) {
                this.consumeToken();
                return true;
            }
            return false;
        };
        Parser.prototype.acceptDelim = function (text) {
            if (this.peekDelim(text)) {
                this.consumeToken();
                return true;
            }
            return false;
        };
        Parser.prototype.acceptRegexp = function (regEx) {
            if (regEx.test(this.token.text)) {
                this.consumeToken();
                return true;
            }
            return false;
        };
        Parser.prototype._parseRegexp = function (regEx) {
            var node = this.createNode(nodes.NodeType.Identifier);
            do { } while (this.acceptRegexp(regEx));
            return this.finish(node);
        };
        Parser.prototype.acceptUnquotedString = function () {
            var pos = this.scanner.pos();
            this.scanner.goBackTo(this.token.offset);
            var unquoted = this.scanner.scanUnquotedString();
            if (unquoted) {
                this.token = unquoted;
                this.consumeToken();
                return true;
            }
            this.scanner.goBackTo(pos);
            return false;
        };
        Parser.prototype.resync = function (resyncTokens, resyncStopTokens) {
            while (true) {
                if (resyncTokens && resyncTokens.indexOf(this.token.type) !== -1) {
                    this.consumeToken();
                    return true;
                }
                else if (resyncStopTokens && resyncStopTokens.indexOf(this.token.type) !== -1) {
                    return true;
                }
                else {
                    if (this.token.type === cssScanner_1.TokenType.EOF) {
                        return false;
                    }
                    this.token = this.scanner.scan();
                }
            }
        };
        Parser.prototype.createNode = function (nodeType) {
            return new nodes.Node(this.token.offset, this.token.len, nodeType);
        };
        Parser.prototype.create = function (ctor) {
            return new ctor(this.token.offset, this.token.len);
        };
        Parser.prototype.finish = function (node, error, resyncTokens, resyncStopTokens) {
            // parseNumeric misuses error for boolean flagging (however the real error mustn't be a false)
            // + nodelist offsets mustn't be modified, because there is a offset hack in rulesets for smartselection
            if (!(node instanceof nodes.Nodelist)) {
                if (error) {
                    this.markError(node, error, resyncTokens, resyncStopTokens);
                }
                // set the node end position
                if (this.prevToken) {
                    // length with more elements belonging together
                    var prevEnd = this.prevToken.offset + this.prevToken.len;
                    node.length = prevEnd > node.offset ? prevEnd - node.offset : 0; // offset is taken from current token, end from previous: Use 0 for empty nodes
                }
            }
            return node;
        };
        Parser.prototype.markError = function (node, error, resyncTokens, resyncStopTokens) {
            if (this.token !== this.lastErrorToken) { // do not report twice on the same token
                node.addIssue(new nodes.Marker(node, error, nodes.Level.Error, undefined, this.token.offset, this.token.len));
                this.lastErrorToken = this.token;
            }
            if (resyncTokens || resyncStopTokens) {
                this.resync(resyncTokens, resyncStopTokens);
            }
        };
        Parser.prototype.parseStylesheet = function (textDocument) {
            var versionId = textDocument.version;
            var text = textDocument.getText();
            var textProvider = function (offset, length) {
                if (textDocument.version !== versionId) {
                    throw new Error('Underlying model has changed, AST is no longer valid');
                }
                return text.substr(offset, length);
            };
            return this.internalParse(text, this._parseStylesheet, textProvider);
        };
        Parser.prototype.internalParse = function (input, parseFunc, textProvider) {
            this.scanner.setSource(input);
            this.token = this.scanner.scan();
            var node = parseFunc.bind(this)();
            if (node) {
                if (textProvider) {
                    node.textProvider = textProvider;
                }
                else {
                    node.textProvider = function (offset, length) { return input.substr(offset, length); };
                }
            }
            return node;
        };
        Parser.prototype._parseStylesheet = function () {
            var node = this.create(nodes.Stylesheet);
            while (node.addChild(this._parseStylesheetStart())) {
                // Parse statements only valid at the beginning of stylesheets.
            }
            var inRecovery = false;
            do {
                var hasMatch = false;
                do {
                    hasMatch = false;
                    var statement = this._parseStylesheetStatement();
                    if (statement) {
                        node.addChild(statement);
                        hasMatch = true;
                        inRecovery = false;
                        if (!this.peek(cssScanner_1.TokenType.EOF) && this._needsSemicolonAfter(statement) && !this.accept(cssScanner_1.TokenType.SemiColon)) {
                            this.markError(node, cssErrors_1.ParseError.SemiColonExpected);
                        }
                    }
                    while (this.accept(cssScanner_1.TokenType.SemiColon) || this.accept(cssScanner_1.TokenType.CDO) || this.accept(cssScanner_1.TokenType.CDC)) {
                        // accept empty statements
                        hasMatch = true;
                        inRecovery = false;
                    }
                } while (hasMatch);
                if (this.peek(cssScanner_1.TokenType.EOF)) {
                    break;
                }
                if (!inRecovery) {
                    if (this.peek(cssScanner_1.TokenType.AtKeyword)) {
                        this.markError(node, cssErrors_1.ParseError.UnknownAtRule);
                    }
                    else {
                        this.markError(node, cssErrors_1.ParseError.RuleOrSelectorExpected);
                    }
                    inRecovery = true;
                }
                this.consumeToken();
            } while (!this.peek(cssScanner_1.TokenType.EOF));
            return this.finish(node);
        };
        Parser.prototype._parseStylesheetStart = function () {
            return this._parseCharset();
        };
        Parser.prototype._parseStylesheetStatement = function (isNested) {
            if (isNested === void 0) { isNested = false; }
            if (this.peek(cssScanner_1.TokenType.AtKeyword)) {
                return this._parseStylesheetAtStatement(isNested);
            }
            return this._parseRuleset(isNested);
        };
        Parser.prototype._parseStylesheetAtStatement = function (isNested) {
            if (isNested === void 0) { isNested = false; }
            return this._parseImport()
                || this._parseMedia(isNested)
                || this._parsePage()
                || this._parseFontFace()
                || this._parseKeyframe()
                || this._parseSupports(isNested)
                || this._parseViewPort()
                || this._parseNamespace()
                || this._parseDocument()
                || this._parseUnknownAtRule();
        };
        Parser.prototype._tryParseRuleset = function (isNested) {
            var mark = this.mark();
            if (this._parseSelector(isNested)) {
                while (this.accept(cssScanner_1.TokenType.Comma) && this._parseSelector(isNested)) {
                    // loop
                }
                if (this.accept(cssScanner_1.TokenType.CurlyL)) {
                    this.restoreAtMark(mark);
                    return this._parseRuleset(isNested);
                }
            }
            this.restoreAtMark(mark);
            return null;
        };
        Parser.prototype._parseRuleset = function (isNested) {
            if (isNested === void 0) { isNested = false; }
            var node = this.create(nodes.RuleSet);
            var selectors = node.getSelectors();
            if (!selectors.addChild(this._parseSelector(isNested))) {
                return null;
            }
            while (this.accept(cssScanner_1.TokenType.Comma)) {
                if (!selectors.addChild(this._parseSelector(isNested))) {
                    return this.finish(node, cssErrors_1.ParseError.SelectorExpected);
                }
            }
            return this._parseBody(node, this._parseRuleSetDeclaration.bind(this));
        };
        Parser.prototype._parseRuleSetDeclarationAtStatement = function () {
            return this._parseAtApply()
                || this._parseUnknownAtRule();
        };
        Parser.prototype._parseRuleSetDeclaration = function () {
            // https://www.w3.org/TR/css-syntax-3/#consume-a-list-of-declarations0
            if (this.peek(cssScanner_1.TokenType.AtKeyword)) {
                return this._parseRuleSetDeclarationAtStatement();
            }
            return this._parseDeclaration();
        };
        /**
         * Parses declarations like:
         *   @apply --my-theme;
         *
         * Follows https://tabatkins.github.io/specs/css-apply-rule/#using
         */
        Parser.prototype._parseAtApply = function () {
            if (!this.peekKeyword('@apply')) {
                return null;
            }
            var node = this.create(nodes.AtApplyRule);
            this.consumeToken();
            if (!node.setIdentifier(this._parseIdent([nodes.ReferenceType.Variable]))) {
                return this.finish(node, cssErrors_1.ParseError.IdentifierExpected);
            }
            return this.finish(node);
        };
        Parser.prototype._needsSemicolonAfter = function (node) {
            switch (node.type) {
                case nodes.NodeType.Keyframe:
                case nodes.NodeType.ViewPort:
                case nodes.NodeType.Media:
                case nodes.NodeType.Ruleset:
                case nodes.NodeType.Namespace:
                case nodes.NodeType.If:
                case nodes.NodeType.For:
                case nodes.NodeType.Each:
                case nodes.NodeType.While:
                case nodes.NodeType.MixinDeclaration:
                case nodes.NodeType.FunctionDeclaration:
                case nodes.NodeType.MixinContentDeclaration:
                    return false;
                case nodes.NodeType.ExtendsReference:
                case nodes.NodeType.MixinContentReference:
                case nodes.NodeType.ReturnStatement:
                case nodes.NodeType.MediaQuery:
                case nodes.NodeType.Debug:
                case nodes.NodeType.Import:
                case nodes.NodeType.AtApplyRule:
                case nodes.NodeType.CustomPropertyDeclaration:
                    return true;
                case nodes.NodeType.VariableDeclaration:
                    return node.needsSemicolon;
                case nodes.NodeType.MixinReference:
                    return !node.getContent();
                case nodes.NodeType.Declaration:
                    return !node.getNestedProperties();
            }
            return false;
        };
        Parser.prototype._parseDeclarations = function (parseDeclaration) {
            var node = this.create(nodes.Declarations);
            if (!this.accept(cssScanner_1.TokenType.CurlyL)) {
                return null;
            }
            var decl = parseDeclaration();
            while (node.addChild(decl)) {
                if (this.peek(cssScanner_1.TokenType.CurlyR)) {
                    break;
                }
                if (this._needsSemicolonAfter(decl) && !this.accept(cssScanner_1.TokenType.SemiColon)) {
                    return this.finish(node, cssErrors_1.ParseError.SemiColonExpected, [cssScanner_1.TokenType.SemiColon, cssScanner_1.TokenType.CurlyR]);
                }
                // We accepted semicolon token. Link it to declaration.
                if (decl && this.prevToken && this.prevToken.type === cssScanner_1.TokenType.SemiColon) {
                    decl.semicolonPosition = this.prevToken.offset;
                }
                while (this.accept(cssScanner_1.TokenType.SemiColon)) {
                    // accept empty statements
                }
                decl = parseDeclaration();
            }
            if (!this.accept(cssScanner_1.TokenType.CurlyR)) {
                return this.finish(node, cssErrors_1.ParseError.RightCurlyExpected, [cssScanner_1.TokenType.CurlyR, cssScanner_1.TokenType.SemiColon]);
            }
            return this.finish(node);
        };
        Parser.prototype._parseBody = function (node, parseDeclaration) {
            if (!node.setDeclarations(this._parseDeclarations(parseDeclaration))) {
                return this.finish(node, cssErrors_1.ParseError.LeftCurlyExpected, [cssScanner_1.TokenType.CurlyR, cssScanner_1.TokenType.SemiColon]);
            }
            return this.finish(node);
        };
        Parser.prototype._parseSelector = function (isNested) {
            var node = this.create(nodes.Selector);
            var hasContent = false;
            if (isNested) {
                // nested selectors can start with a combinator
                hasContent = node.addChild(this._parseCombinator());
            }
            while (node.addChild(this._parseSimpleSelector())) {
                hasContent = true;
                node.addChild(this._parseCombinator()); // optional
            }
            return hasContent ? this.finish(node) : null;
        };
        Parser.prototype._parseDeclaration = function (stopTokens) {
            var custonProperty = this._tryParseCustomPropertyDeclaration(stopTokens);
            if (custonProperty) {
                return custonProperty;
            }
            var node = this.create(nodes.Declaration);
            if (!node.setProperty(this._parseProperty())) {
                return null;
            }
            if (!this.accept(cssScanner_1.TokenType.Colon)) {
                return this.finish(node, cssErrors_1.ParseError.ColonExpected, [cssScanner_1.TokenType.Colon], stopTokens || [cssScanner_1.TokenType.SemiColon]);
            }
            if (this.prevToken) {
                node.colonPosition = this.prevToken.offset;
            }
            if (!node.setValue(this._parseExpr())) {
                return this.finish(node, cssErrors_1.ParseError.PropertyValueExpected);
            }
            node.addChild(this._parsePrio());
            if (this.peek(cssScanner_1.TokenType.SemiColon)) {
                node.semicolonPosition = this.token.offset; // not part of the declaration, but useful information for code assist
            }
            return this.finish(node);
        };
        Parser.prototype._tryParseCustomPropertyDeclaration = function (stopTokens) {
            if (!this.peekRegExp(cssScanner_1.TokenType.Ident, /^--/)) {
                return null;
            }
            var node = this.create(nodes.CustomPropertyDeclaration);
            if (!node.setProperty(this._parseProperty())) {
                return null;
            }
            if (!this.accept(cssScanner_1.TokenType.Colon)) {
                return this.finish(node, cssErrors_1.ParseError.ColonExpected, [cssScanner_1.TokenType.Colon]);
            }
            if (this.prevToken) {
                node.colonPosition = this.prevToken.offset;
            }
            var mark = this.mark();
            if (this.peek(cssScanner_1.TokenType.CurlyL)) {
                // try to parse it as nested declaration
                var propertySet = this.create(nodes.CustomPropertySet);
                var declarations = this._parseDeclarations(this._parseRuleSetDeclaration.bind(this));
                if (propertySet.setDeclarations(declarations) && !declarations.isErroneous(true)) {
                    propertySet.addChild(this._parsePrio());
                    if (this.peek(cssScanner_1.TokenType.SemiColon)) {
                        this.finish(propertySet);
                        node.setPropertySet(propertySet);
                        node.semicolonPosition = this.token.offset; // not part of the declaration, but useful information for code assist
                        return this.finish(node);
                    }
                }
                this.restoreAtMark(mark);
            }
            // try tp parse as expression
            var expression = this._parseExpr();
            if (expression && !expression.isErroneous(true)) {
                this._parsePrio();
                if (this.peekOne(stopTokens || [cssScanner_1.TokenType.SemiColon])) {
                    node.setValue(expression);
                    node.semicolonPosition = this.token.offset; // not part of the declaration, but useful information for code assist
                    return this.finish(node);
                }
            }
            this.restoreAtMark(mark);
            node.addChild(this._parseCustomPropertyValue(stopTokens));
            node.addChild(this._parsePrio());
            if (objects_1.isDefined(node.colonPosition) && this.token.offset === node.colonPosition + 1) {
                return this.finish(node, cssErrors_1.ParseError.PropertyValueExpected);
            }
            return this.finish(node);
        };
        /**
         * Parse custom property values.
         *
         * Based on https://www.w3.org/TR/css-variables/#syntax
         *
         * This code is somewhat unusual, as the allowed syntax is incredibly broad,
         * parsing almost any sequence of tokens, save for a small set of exceptions.
         * Unbalanced delimitors, invalid tokens, and declaration
         * terminators like semicolons and !important directives (when not inside
         * of delimitors).
         */
        Parser.prototype._parseCustomPropertyValue = function (stopTokens) {
            var _this = this;
            if (stopTokens === void 0) { stopTokens = [cssScanner_1.TokenType.CurlyR]; }
            var node = this.create(nodes.Node);
            var isTopLevel = function () { return curlyDepth === 0 && parensDepth === 0 && bracketsDepth === 0; };
            var onStopToken = function () { return stopTokens.indexOf(_this.token.type) !== -1; };
            var curlyDepth = 0;
            var parensDepth = 0;
            var bracketsDepth = 0;
            done: while (true) {
                switch (this.token.type) {
                    case cssScanner_1.TokenType.SemiColon:
                        // A semicolon only ends things if we're not inside a delimitor.
                        if (isTopLevel()) {
                            break done;
                        }
                        break;
                    case cssScanner_1.TokenType.Exclamation:
                        // An exclamation ends the value if we're not inside delims.
                        if (isTopLevel()) {
                            break done;
                        }
                        break;
                    case cssScanner_1.TokenType.CurlyL:
                        curlyDepth++;
                        break;
                    case cssScanner_1.TokenType.CurlyR:
                        curlyDepth--;
                        if (curlyDepth < 0) {
                            // The property value has been terminated without a semicolon, and
                            // this is the last declaration in the ruleset.
                            if (onStopToken() && parensDepth === 0 && bracketsDepth === 0) {
                                break done;
                            }
                            return this.finish(node, cssErrors_1.ParseError.LeftCurlyExpected);
                        }
                        break;
                    case cssScanner_1.TokenType.ParenthesisL:
                        parensDepth++;
                        break;
                    case cssScanner_1.TokenType.ParenthesisR:
                        parensDepth--;
                        if (parensDepth < 0) {
                            if (onStopToken() && bracketsDepth === 0 && curlyDepth === 0) {
                                break done;
                            }
                            return this.finish(node, cssErrors_1.ParseError.LeftParenthesisExpected);
                        }
                        break;
                    case cssScanner_1.TokenType.BracketL:
                        bracketsDepth++;
                        break;
                    case cssScanner_1.TokenType.BracketR:
                        bracketsDepth--;
                        if (bracketsDepth < 0) {
                            return this.finish(node, cssErrors_1.ParseError.LeftSquareBracketExpected);
                        }
                        break;
                    case cssScanner_1.TokenType.BadString: // fall through
                        break done;
                    case cssScanner_1.TokenType.EOF:
                        // We shouldn't have reached the end of input, something is
                        // unterminated.
                        var error = cssErrors_1.ParseError.RightCurlyExpected;
                        if (bracketsDepth > 0) {
                            error = cssErrors_1.ParseError.RightSquareBracketExpected;
                        }
                        else if (parensDepth > 0) {
                            error = cssErrors_1.ParseError.RightParenthesisExpected;
                        }
                        return this.finish(node, error);
                }
                this.consumeToken();
            }
            return this.finish(node);
        };
        Parser.prototype._tryToParseDeclaration = function (stopTokens) {
            var mark = this.mark();
            if (this._parseProperty() && this.accept(cssScanner_1.TokenType.Colon)) {
                // looks like a declaration, go ahead
                this.restoreAtMark(mark);
                return this._parseDeclaration(stopTokens);
            }
            this.restoreAtMark(mark);
            return null;
        };
        Parser.prototype._parseProperty = function () {
            var node = this.create(nodes.Property);
            var mark = this.mark();
            if (this.acceptDelim('*') || this.acceptDelim('_')) {
                // support for  IE 5.x, 6 and 7 star hack: see http://en.wikipedia.org/wiki/CSS_filter#Star_hack
                if (this.hasWhitespace()) {
                    this.restoreAtMark(mark);
                    return null;
                }
            }
            if (node.setIdentifier(this._parsePropertyIdentifier())) {
                return this.finish(node);
            }
            return null;
        };
        Parser.prototype._parsePropertyIdentifier = function () {
            return this._parseIdent();
        };
        Parser.prototype._parseCharset = function () {
            if (!this.peek(cssScanner_1.TokenType.Charset)) {
                return null;
            }
            var node = this.create(nodes.Node);
            this.consumeToken(); // charset
            if (!this.accept(cssScanner_1.TokenType.String)) {
                return this.finish(node, cssErrors_1.ParseError.IdentifierExpected);
            }
            if (!this.accept(cssScanner_1.TokenType.SemiColon)) {
                return this.finish(node, cssErrors_1.ParseError.SemiColonExpected);
            }
            return this.finish(node);
        };
        Parser.prototype._parseImport = function () {
            if (!this.peekKeyword('@import')) {
                return null;
            }
            var node = this.create(nodes.Import);
            this.consumeToken(); // @import
            if (!node.addChild(this._parseURILiteral()) && !node.addChild(this._parseStringLiteral())) {
                return this.finish(node, cssErrors_1.ParseError.URIOrStringExpected);
            }
            if (!this.peek(cssScanner_1.TokenType.SemiColon) && !this.peek(cssScanner_1.TokenType.EOF)) {
                node.setMedialist(this._parseMediaQueryList());
            }
            return this.finish(node);
        };
        Parser.prototype._parseNamespace = function () {
            // http://www.w3.org/TR/css3-namespace/
            // namespace  : NAMESPACE_SYM S* [IDENT S*]? [STRING|URI] S* ';' S*
            if (!this.peekKeyword('@namespace')) {
                return null;
            }
            var node = this.create(nodes.Namespace);
            this.consumeToken(); // @namespace
            if (!node.addChild(this._parseURILiteral())) { // url literal also starts with ident
                node.addChild(this._parseIdent()); // optional prefix
                if (!node.addChild(this._parseURILiteral()) && !node.addChild(this._parseStringLiteral())) {
                    return this.finish(node, cssErrors_1.ParseError.URIExpected, [cssScanner_1.TokenType.SemiColon]);
                }
            }
            if (!this.accept(cssScanner_1.TokenType.SemiColon)) {
                return this.finish(node, cssErrors_1.ParseError.SemiColonExpected);
            }
            return this.finish(node);
        };
        Parser.prototype._parseFontFace = function () {
            if (!this.peekKeyword('@font-face')) {
                return null;
            }
            var node = this.create(nodes.FontFace);
            this.consumeToken(); // @font-face
            return this._parseBody(node, this._parseRuleSetDeclaration.bind(this));
        };
        Parser.prototype._parseViewPort = function () {
            if (!this.peekKeyword('@-ms-viewport') &&
                !this.peekKeyword('@-o-viewport') &&
                !this.peekKeyword('@viewport')) {
                return null;
            }
            var node = this.create(nodes.ViewPort);
            this.consumeToken(); // @-ms-viewport
            return this._parseBody(node, this._parseRuleSetDeclaration.bind(this));
        };
        Parser.prototype._parseKeyframe = function () {
            if (!this.peekRegExp(cssScanner_1.TokenType.AtKeyword, this.keyframeRegex)) {
                return null;
            }
            var node = this.create(nodes.Keyframe);
            var atNode = this.create(nodes.Node);
            this.consumeToken(); // atkeyword
            node.setKeyword(this.finish(atNode));
            if (atNode.matches('@-ms-keyframes')) { // -ms-keyframes never existed
                this.markError(atNode, cssErrors_1.ParseError.UnknownKeyword);
            }
            if (!node.setIdentifier(this._parseKeyframeIdent())) {
                return this.finish(node, cssErrors_1.ParseError.IdentifierExpected, [cssScanner_1.TokenType.CurlyR]);
            }
            return this._parseBody(node, this._parseKeyframeSelector.bind(this));
        };
        Parser.prototype._parseKeyframeIdent = function () {
            return this._parseIdent([nodes.ReferenceType.Keyframe]);
        };
        Parser.prototype._parseKeyframeSelector = function () {
            var node = this.create(nodes.KeyframeSelector);
            if (!node.addChild(this._parseIdent()) && !this.accept(cssScanner_1.TokenType.Percentage)) {
                return null;
            }
            while (this.accept(cssScanner_1.TokenType.Comma)) {
                if (!node.addChild(this._parseIdent()) && !this.accept(cssScanner_1.TokenType.Percentage)) {
                    return this.finish(node, cssErrors_1.ParseError.PercentageExpected);
                }
            }
            return this._parseBody(node, this._parseRuleSetDeclaration.bind(this));
        };
        Parser.prototype._tryParseKeyframeSelector = function () {
            var node = this.create(nodes.KeyframeSelector);
            var pos = this.mark();
            if (!node.addChild(this._parseIdent()) && !this.accept(cssScanner_1.TokenType.Percentage)) {
                return null;
            }
            while (this.accept(cssScanner_1.TokenType.Comma)) {
                if (!node.addChild(this._parseIdent()) && !this.accept(cssScanner_1.TokenType.Percentage)) {
                    this.restoreAtMark(pos);
                    return null;
                }
            }
            if (!this.peek(cssScanner_1.TokenType.CurlyL)) {
                this.restoreAtMark(pos);
                return null;
            }
            return this._parseBody(node, this._parseRuleSetDeclaration.bind(this));
        };
        Parser.prototype._parseSupports = function (isNested) {
            if (isNested === void 0) { isNested = false; }
            // SUPPORTS_SYM S* supports_condition '{' S* ruleset* '}' S*
            if (!this.peekKeyword('@supports')) {
                return null;
            }
            var node = this.create(nodes.Supports);
            this.consumeToken(); // @supports
            node.addChild(this._parseSupportsCondition());
            return this._parseBody(node, this._parseSupportsDeclaration.bind(this, isNested));
        };
        Parser.prototype._parseSupportsDeclaration = function (isNested) {
            if (isNested === void 0) { isNested = false; }
            if (isNested) {
                // if nested, the body can contain rulesets, but also declarations
                return this._tryParseRuleset(true)
                    || this._tryToParseDeclaration()
                    || this._parseStylesheetStatement(true);
            }
            return this._parseStylesheetStatement(false);
        };
        Parser.prototype._parseSupportsCondition = function () {
            // supports_condition : supports_negation | supports_conjunction | supports_disjunction | supports_condition_in_parens ;
            // supports_condition_in_parens: ( '(' S* supports_condition S* ')' ) | supports_declaration_condition | general_enclosed ;
            // supports_negation: NOT S+ supports_condition_in_parens ;
            // supports_conjunction: supports_condition_in_parens ( S+ AND S+ supports_condition_in_parens )+;
            // supports_disjunction: supports_condition_in_parens ( S+ OR S+ supports_condition_in_parens )+;
            // supports_declaration_condition: '(' S* declaration ')';
            // general_enclosed: ( FUNCTION | '(' ) ( any | unused )* ')' ;
            var node = this.create(nodes.SupportsCondition);
            if (this.acceptIdent('not')) {
                node.addChild(this._parseSupportsConditionInParens());
            }
            else {
                node.addChild(this._parseSupportsConditionInParens());
                if (this.peekRegExp(cssScanner_1.TokenType.Ident, /^(and|or)$/i)) {
                    var text = this.token.text.toLowerCase();
                    while (this.acceptIdent(text)) {
                        node.addChild(this._parseSupportsConditionInParens());
                    }
                }
            }
            return this.finish(node);
        };
        Parser.prototype._parseSupportsConditionInParens = function () {
            var node = this.create(nodes.SupportsCondition);
            if (this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                if (this.prevToken) {
                    node.lParent = this.prevToken.offset;
                }
                if (!node.addChild(this._tryToParseDeclaration([cssScanner_1.TokenType.ParenthesisR]))) {
                    if (!this._parseSupportsCondition()) {
                        return this.finish(node, cssErrors_1.ParseError.ConditionExpected);
                    }
                }
                if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                    return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected, [cssScanner_1.TokenType.ParenthesisR], []);
                }
                if (this.prevToken) {
                    node.rParent = this.prevToken.offset;
                }
                return this.finish(node);
            }
            else if (this.peek(cssScanner_1.TokenType.Ident)) {
                var pos = this.mark();
                this.consumeToken();
                if (!this.hasWhitespace() && this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                    var openParentCount = 1;
                    while (this.token.type !== cssScanner_1.TokenType.EOF && openParentCount !== 0) {
                        if (this.token.type === cssScanner_1.TokenType.ParenthesisL) {
                            openParentCount++;
                        }
                        else if (this.token.type === cssScanner_1.TokenType.ParenthesisR) {
                            openParentCount--;
                        }
                        this.consumeToken();
                    }
                    return this.finish(node);
                }
                else {
                    this.restoreAtMark(pos);
                }
            }
            return this.finish(node, cssErrors_1.ParseError.LeftParenthesisExpected, [], [cssScanner_1.TokenType.ParenthesisL]);
        };
        Parser.prototype._parseMediaDeclaration = function (isNested) {
            if (isNested === void 0) { isNested = false; }
            if (isNested) {
                // if nested, the body can contain rulesets, but also declarations
                return this._tryParseRuleset(true)
                    || this._tryToParseDeclaration()
                    || this._parseStylesheetStatement(true);
            }
            return this._parseStylesheetStatement(false);
        };
        Parser.prototype._parseMedia = function (isNested) {
            if (isNested === void 0) { isNested = false; }
            // MEDIA_SYM S* media_query_list '{' S* ruleset* '}' S*
            // media_query_list : S* [media_query [ ',' S* media_query ]* ]?
            if (!this.peekKeyword('@media')) {
                return null;
            }
            var node = this.create(nodes.Media);
            this.consumeToken(); // @media
            if (!node.addChild(this._parseMediaQueryList())) {
                return this.finish(node, cssErrors_1.ParseError.MediaQueryExpected);
            }
            return this._parseBody(node, this._parseMediaDeclaration.bind(this, isNested));
        };
        Parser.prototype._parseMediaQueryList = function () {
            var node = this.create(nodes.Medialist);
            if (!node.addChild(this._parseMediaQuery([cssScanner_1.TokenType.CurlyL]))) {
                return this.finish(node, cssErrors_1.ParseError.MediaQueryExpected);
            }
            while (this.accept(cssScanner_1.TokenType.Comma)) {
                if (!node.addChild(this._parseMediaQuery([cssScanner_1.TokenType.CurlyL]))) {
                    return this.finish(node, cssErrors_1.ParseError.MediaQueryExpected);
                }
            }
            return this.finish(node);
        };
        Parser.prototype._parseMediaQuery = function (resyncStopToken) {
            // http://www.w3.org/TR/css3-mediaqueries/
            // media_query : [ONLY | NOT]? S* IDENT S* [ AND S* expression ]* | expression [ AND S* expression ]*
            // expression : '(' S* IDENT S* [ ':' S* expr ]? ')' S*
            var node = this.create(nodes.MediaQuery);
            var parseExpression = true;
            var hasContent = false;
            if (!this.peek(cssScanner_1.TokenType.ParenthesisL)) {
                if (this.acceptIdent('only') || this.acceptIdent('not')) {
                    // optional
                }
                if (!node.addChild(this._parseIdent())) {
                    return null;
                }
                hasContent = true;
                parseExpression = this.acceptIdent('and');
            }
            while (parseExpression) {
                // Allow short-circuting for other language constructs.
                if (node.addChild(this._parseMediaContentStart())) {
                    parseExpression = this.acceptIdent('and');
                    continue;
                }
                if (!this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                    if (hasContent) {
                        return this.finish(node, cssErrors_1.ParseError.LeftParenthesisExpected, [], resyncStopToken);
                    }
                    return null;
                }
                if (!node.addChild(this._parseMediaFeatureName())) {
                    return this.finish(node, cssErrors_1.ParseError.IdentifierExpected, [], resyncStopToken);
                }
                if (this.accept(cssScanner_1.TokenType.Colon)) {
                    if (!node.addChild(this._parseExpr())) {
                        return this.finish(node, cssErrors_1.ParseError.TermExpected, [], resyncStopToken);
                    }
                }
                if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                    return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected, [], resyncStopToken);
                }
                parseExpression = this.acceptIdent('and');
            }
            return this.finish(node);
        };
        Parser.prototype._parseMediaContentStart = function () {
            return null;
        };
        Parser.prototype._parseMediaFeatureName = function () {
            return this._parseIdent();
        };
        Parser.prototype._parseMedium = function () {
            var node = this.create(nodes.Node);
            if (node.addChild(this._parseIdent())) {
                return this.finish(node);
            }
            else {
                return null;
            }
        };
        Parser.prototype._parsePageDeclaration = function () {
            return this._parsePageMarginBox() || this._parseRuleSetDeclaration();
        };
        Parser.prototype._parsePage = function () {
            // http://www.w3.org/TR/css3-page/
            // page_rule : PAGE_SYM S* page_selector_list '{' S* page_body '}' S*
            // page_body :  /* Can be empty */ declaration? [ ';' S* page_body ]? | page_margin_box page_body
            if (!this.peekKeyword('@page')) {
                return null;
            }
            var node = this.create(nodes.Page);
            this.consumeToken();
            if (node.addChild(this._parsePageSelector())) {
                while (this.accept(cssScanner_1.TokenType.Comma)) {
                    if (!node.addChild(this._parsePageSelector())) {
                        return this.finish(node, cssErrors_1.ParseError.IdentifierExpected);
                    }
                }
            }
            return this._parseBody(node, this._parsePageDeclaration.bind(this));
        };
        Parser.prototype._parsePageMarginBox = function () {
            // page_margin_box :  margin_sym S* '{' S* declaration? [ ';' S* declaration? ]* '}' S*
            if (!this.peek(cssScanner_1.TokenType.AtKeyword)) {
                return null;
            }
            var node = this.create(nodes.PageBoxMarginBox);
            if (!this.acceptOneKeyword(languageFacts.pageBoxDirectives)) {
                this.markError(node, cssErrors_1.ParseError.UnknownAtRule, [], [cssScanner_1.TokenType.CurlyL]);
            }
            return this._parseBody(node, this._parseRuleSetDeclaration.bind(this));
        };
        Parser.prototype._parsePageSelector = function () {
            // page_selector : pseudo_page+ | IDENT pseudo_page*
            // pseudo_page :  ':' [ "left" | "right" | "first" | "blank" ];
            if (!this.peek(cssScanner_1.TokenType.Ident) && !this.peek(cssScanner_1.TokenType.Colon)) {
                return null;
            }
            var node = this.create(nodes.Node);
            node.addChild(this._parseIdent()); // optional ident
            if (this.accept(cssScanner_1.TokenType.Colon)) {
                if (!node.addChild(this._parseIdent())) { // optional ident
                    return this.finish(node, cssErrors_1.ParseError.IdentifierExpected);
                }
            }
            return this.finish(node);
        };
        Parser.prototype._parseDocument = function () {
            // -moz-document is experimental but has been pushed to css4
            if (!this.peekKeyword('@-moz-document')) {
                return null;
            }
            var node = this.create(nodes.Document);
            this.consumeToken(); // @-moz-document
            this.resync([], [cssScanner_1.TokenType.CurlyL]); // ignore all the rules
            return this._parseBody(node, this._parseStylesheetStatement.bind(this));
        };
        // https://www.w3.org/TR/css-syntax-3/#consume-an-at-rule
        Parser.prototype._parseUnknownAtRule = function () {
            if (!this.peek(cssScanner_1.TokenType.AtKeyword)) {
                return null;
            }
            var node = this.create(nodes.UnknownAtRule);
            node.addChild(this._parseUnknownAtRuleName());
            var isTopLevel = function () { return curlyDepth === 0 && parensDepth === 0 && bracketsDepth === 0; };
            var curlyLCount = 0;
            var curlyDepth = 0;
            var parensDepth = 0;
            var bracketsDepth = 0;
            done: while (true) {
                switch (this.token.type) {
                    case cssScanner_1.TokenType.SemiColon:
                        if (isTopLevel()) {
                            break done;
                        }
                        break;
                    case cssScanner_1.TokenType.EOF:
                        if (curlyDepth > 0) {
                            return this.finish(node, cssErrors_1.ParseError.RightCurlyExpected);
                        }
                        else if (bracketsDepth > 0) {
                            return this.finish(node, cssErrors_1.ParseError.RightSquareBracketExpected);
                        }
                        else if (parensDepth > 0) {
                            return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected);
                        }
                        else {
                            return this.finish(node);
                        }
                    case cssScanner_1.TokenType.CurlyL:
                        curlyLCount++;
                        curlyDepth++;
                        break;
                    case cssScanner_1.TokenType.CurlyR:
                        curlyDepth--;
                        // End of at-rule, consume CurlyR and return node
                        if (curlyLCount > 0 && curlyDepth === 0) {
                            this.consumeToken();
                            if (bracketsDepth > 0) {
                                return this.finish(node, cssErrors_1.ParseError.RightSquareBracketExpected);
                            }
                            else if (parensDepth > 0) {
                                return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected);
                            }
                            break done;
                        }
                        if (curlyDepth < 0) {
                            // The property value has been terminated without a semicolon, and
                            // this is the last declaration in the ruleset.
                            if (parensDepth === 0 && bracketsDepth === 0) {
                                break done;
                            }
                            return this.finish(node, cssErrors_1.ParseError.LeftCurlyExpected);
                        }
                        break;
                    case cssScanner_1.TokenType.ParenthesisL:
                        parensDepth++;
                        break;
                    case cssScanner_1.TokenType.ParenthesisR:
                        parensDepth--;
                        if (parensDepth < 0) {
                            return this.finish(node, cssErrors_1.ParseError.LeftParenthesisExpected);
                        }
                        break;
                    case cssScanner_1.TokenType.BracketL:
                        bracketsDepth++;
                        break;
                    case cssScanner_1.TokenType.BracketR:
                        bracketsDepth--;
                        if (bracketsDepth < 0) {
                            return this.finish(node, cssErrors_1.ParseError.LeftSquareBracketExpected);
                        }
                        break;
                }
                this.consumeToken();
            }
            return node;
        };
        Parser.prototype._parseUnknownAtRuleName = function () {
            var node = this.create(nodes.Node);
            if (this.accept(cssScanner_1.TokenType.AtKeyword)) {
                return this.finish(node);
            }
            return node;
        };
        Parser.prototype._parseOperator = function () {
            // these are operators for binary expressions
            if (this.peekDelim('/') ||
                this.peekDelim('*') ||
                this.peekDelim('+') ||
                this.peekDelim('-') ||
                this.peek(cssScanner_1.TokenType.Dashmatch) ||
                this.peek(cssScanner_1.TokenType.Includes) ||
                this.peek(cssScanner_1.TokenType.SubstringOperator) ||
                this.peek(cssScanner_1.TokenType.PrefixOperator) ||
                this.peek(cssScanner_1.TokenType.SuffixOperator) ||
                this.peekDelim('=')) { // doesn't stick to the standard here
                var node = this.createNode(nodes.NodeType.Operator);
                this.consumeToken();
                return this.finish(node);
            }
            else {
                return null;
            }
        };
        Parser.prototype._parseUnaryOperator = function () {
            if (!this.peekDelim('+') && !this.peekDelim('-')) {
                return null;
            }
            var node = this.create(nodes.Node);
            this.consumeToken();
            return this.finish(node);
        };
        Parser.prototype._parseCombinator = function () {
            if (this.peekDelim('>')) {
                var node = this.create(nodes.Node);
                this.consumeToken();
                var mark = this.mark();
                if (!this.hasWhitespace() && this.acceptDelim('>')) {
                    if (!this.hasWhitespace() && this.acceptDelim('>')) {
                        node.type = nodes.NodeType.SelectorCombinatorShadowPiercingDescendant;
                        return this.finish(node);
                    }
                    this.restoreAtMark(mark);
                }
                node.type = nodes.NodeType.SelectorCombinatorParent;
                return this.finish(node);
            }
            else if (this.peekDelim('+')) {
                var node = this.create(nodes.Node);
                this.consumeToken();
                node.type = nodes.NodeType.SelectorCombinatorSibling;
                return this.finish(node);
            }
            else if (this.peekDelim('~')) {
                var node = this.create(nodes.Node);
                this.consumeToken();
                node.type = nodes.NodeType.SelectorCombinatorAllSiblings;
                return this.finish(node);
            }
            else if (this.peekDelim('/')) {
                var node = this.create(nodes.Node);
                this.consumeToken();
                var mark = this.mark();
                if (!this.hasWhitespace() && this.acceptIdent('deep') && !this.hasWhitespace() && this.acceptDelim('/')) {
                    node.type = nodes.NodeType.SelectorCombinatorShadowPiercingDescendant;
                    return this.finish(node);
                }
                this.restoreAtMark(mark);
            }
            return null;
        };
        Parser.prototype._parseSimpleSelector = function () {
            // simple_selector
            //  : element_name [ HASH | class | attrib | pseudo ]* | [ HASH | class | attrib | pseudo ]+ ;
            var node = this.create(nodes.SimpleSelector);
            var c = 0;
            if (node.addChild(this._parseElementName())) {
                c++;
            }
            while ((c === 0 || !this.hasWhitespace()) && node.addChild(this._parseSimpleSelectorBody())) {
                c++;
            }
            return c > 0 ? this.finish(node) : null;
        };
        Parser.prototype._parseSimpleSelectorBody = function () {
            return this._parsePseudo() || this._parseHash() || this._parseClass() || this._parseAttrib();
        };
        Parser.prototype._parseSelectorIdent = function () {
            return this._parseIdent();
        };
        Parser.prototype._parseHash = function () {
            if (!this.peek(cssScanner_1.TokenType.Hash) && !this.peekDelim('#')) {
                return null;
            }
            var node = this.createNode(nodes.NodeType.IdentifierSelector);
            if (this.acceptDelim('#')) {
                if (this.hasWhitespace() || !node.addChild(this._parseSelectorIdent())) {
                    return this.finish(node, cssErrors_1.ParseError.IdentifierExpected);
                }
            }
            else {
                this.consumeToken(); // TokenType.Hash
            }
            return this.finish(node);
        };
        Parser.prototype._parseClass = function () {
            // class: '.' IDENT ;
            if (!this.peekDelim('.')) {
                return null;
            }
            var node = this.createNode(nodes.NodeType.ClassSelector);
            this.consumeToken(); // '.'
            if (this.hasWhitespace() || !node.addChild(this._parseSelectorIdent())) {
                return this.finish(node, cssErrors_1.ParseError.IdentifierExpected);
            }
            return this.finish(node);
        };
        Parser.prototype._parseElementName = function () {
            // element_name: (ns? '|')? IDENT | '*';
            var pos = this.mark();
            var node = this.createNode(nodes.NodeType.ElementNameSelector);
            node.addChild(this._parseNamespacePrefix());
            if (!node.addChild(this._parseSelectorIdent()) && !this.acceptDelim('*')) {
                this.restoreAtMark(pos);
                return null;
            }
            return this.finish(node);
        };
        Parser.prototype._parseNamespacePrefix = function () {
            var pos = this.mark();
            var node = this.createNode(nodes.NodeType.NamespacePrefix);
            if (!node.addChild(this._parseIdent()) && !this.acceptDelim('*')) {
                // ns is optional
            }
            if (!this.acceptDelim('|')) {
                this.restoreAtMark(pos);
                return null;
            }
            return this.finish(node);
        };
        Parser.prototype._parseAttrib = function () {
            // attrib : '[' S* IDENT S* [ [ '=' | INCLUDES | DASHMATCH ] S*   [ IDENT | STRING ] S* ]? ']'
            if (!this.peek(cssScanner_1.TokenType.BracketL)) {
                return null;
            }
            var node = this.create(nodes.AttributeSelector);
            this.consumeToken(); // BracketL
            // Optional attrib namespace
            node.setNamespacePrefix(this._parseNamespacePrefix());
            if (!node.setIdentifier(this._parseIdent())) {
                return this.finish(node, cssErrors_1.ParseError.IdentifierExpected);
            }
            if (node.setOperator(this._parseOperator())) {
                node.setValue(this._parseBinaryExpr());
                this.acceptIdent('i'); // case insensitive matching
            }
            if (!this.accept(cssScanner_1.TokenType.BracketR)) {
                return this.finish(node, cssErrors_1.ParseError.RightSquareBracketExpected);
            }
            return this.finish(node);
        };
        Parser.prototype._parsePseudo = function () {
            var _this = this;
            // pseudo: ':' [ IDENT | FUNCTION S* [IDENT S*]? ')' ]
            var node = this._tryParsePseudoIdentifier();
            if (node) {
                if (!this.hasWhitespace() && this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                    var tryAsSelector = function () {
                        var selectors = _this.create(nodes.Node);
                        if (!selectors.addChild(_this._parseSelector(false))) {
                            return null;
                        }
                        while (_this.accept(cssScanner_1.TokenType.Comma) && selectors.addChild(_this._parseSelector(false))) {
                            // loop
                        }
                        if (_this.peek(cssScanner_1.TokenType.ParenthesisR)) {
                            return _this.finish(selectors);
                        }
                        return null;
                    };
                    node.addChild(this.try(tryAsSelector) || this._parseBinaryExpr());
                    if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                        return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected);
                    }
                }
                return this.finish(node);
            }
            return null;
        };
        Parser.prototype._tryParsePseudoIdentifier = function () {
            if (!this.peek(cssScanner_1.TokenType.Colon)) {
                return null;
            }
            var pos = this.mark();
            var node = this.createNode(nodes.NodeType.PseudoSelector);
            this.consumeToken(); // Colon
            if (this.hasWhitespace()) {
                this.restoreAtMark(pos);
                return null;
            }
            // optional, support ::
            this.accept(cssScanner_1.TokenType.Colon);
            if (this.hasWhitespace() || !node.addChild(this._parseIdent())) {
                return this.finish(node, cssErrors_1.ParseError.IdentifierExpected);
            }
            return this.finish(node);
        };
        Parser.prototype._tryParsePrio = function () {
            var mark = this.mark();
            var prio = this._parsePrio();
            if (prio) {
                return prio;
            }
            this.restoreAtMark(mark);
            return null;
        };
        Parser.prototype._parsePrio = function () {
            if (!this.peek(cssScanner_1.TokenType.Exclamation)) {
                return null;
            }
            var node = this.createNode(nodes.NodeType.Prio);
            if (this.accept(cssScanner_1.TokenType.Exclamation) && this.acceptIdent('important')) {
                return this.finish(node);
            }
            return null;
        };
        Parser.prototype._parseExpr = function (stopOnComma) {
            if (stopOnComma === void 0) { stopOnComma = false; }
            var node = this.create(nodes.Expression);
            if (!node.addChild(this._parseBinaryExpr())) {
                return null;
            }
            while (true) {
                if (this.peek(cssScanner_1.TokenType.Comma)) { // optional
                    if (stopOnComma) {
                        return this.finish(node);
                    }
                    this.consumeToken();
                }
                if (!node.addChild(this._parseBinaryExpr())) {
                    break;
                }
            }
            return this.finish(node);
        };
        Parser.prototype._parseNamedLine = function () {
            // https://www.w3.org/TR/css-grid-1/#named-lines
            if (!this.peek(cssScanner_1.TokenType.BracketL)) {
                return null;
            }
            var node = this.createNode(nodes.NodeType.GridLine);
            this.consumeToken();
            while (node.addChild(this._parseIdent())) {
                // repeat
            }
            if (!this.accept(cssScanner_1.TokenType.BracketR)) {
                return this.finish(node, cssErrors_1.ParseError.RightSquareBracketExpected);
            }
            return this.finish(node);
        };
        Parser.prototype._parseBinaryExpr = function (preparsedLeft, preparsedOper) {
            var node = this.create(nodes.BinaryExpression);
            if (!node.setLeft((preparsedLeft || this._parseTerm()))) {
                return null;
            }
            if (!node.setOperator(preparsedOper || this._parseOperator())) {
                return this.finish(node);
            }
            if (!node.setRight(this._parseTerm())) {
                return this.finish(node, cssErrors_1.ParseError.TermExpected);
            }
            // things needed for multiple binary expressions
            node = this.finish(node);
            var operator = this._parseOperator();
            if (operator) {
                node = this._parseBinaryExpr(node, operator);
            }
            return this.finish(node);
        };
        Parser.prototype._parseTerm = function () {
            var node = this.create(nodes.Term);
            node.setOperator(this._parseUnaryOperator()); // optional
            if (node.setExpression(this._parseTermExpression())) {
                return this.finish(node);
            }
            return null;
        };
        Parser.prototype._parseTermExpression = function () {
            return this._parseURILiteral() || // url before function
                this._parseFunction() || // function before ident
                this._parseIdent() ||
                this._parseStringLiteral() ||
                this._parseNumeric() ||
                this._parseHexColor() ||
                this._parseOperation() ||
                this._parseNamedLine();
        };
        Parser.prototype._parseOperation = function () {
            if (!this.peek(cssScanner_1.TokenType.ParenthesisL)) {
                return null;
            }
            var node = this.create(nodes.Node);
            this.consumeToken(); // ParenthesisL
            node.addChild(this._parseExpr());
            if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected);
            }
            return this.finish(node);
        };
        Parser.prototype._parseNumeric = function () {
            if (this.peek(cssScanner_1.TokenType.Num) ||
                this.peek(cssScanner_1.TokenType.Percentage) ||
                this.peek(cssScanner_1.TokenType.Resolution) ||
                this.peek(cssScanner_1.TokenType.Length) ||
                this.peek(cssScanner_1.TokenType.EMS) ||
                this.peek(cssScanner_1.TokenType.EXS) ||
                this.peek(cssScanner_1.TokenType.Angle) ||
                this.peek(cssScanner_1.TokenType.Time) ||
                this.peek(cssScanner_1.TokenType.Dimension) ||
                this.peek(cssScanner_1.TokenType.Freq)) {
                var node = this.create(nodes.NumericValue);
                this.consumeToken();
                return this.finish(node);
            }
            return null;
        };
        Parser.prototype._parseStringLiteral = function () {
            if (!this.peek(cssScanner_1.TokenType.String) && !this.peek(cssScanner_1.TokenType.BadString)) {
                return null;
            }
            var node = this.createNode(nodes.NodeType.StringLiteral);
            this.consumeToken();
            return this.finish(node);
        };
        Parser.prototype._parseURILiteral = function () {
            if (!this.peekRegExp(cssScanner_1.TokenType.Ident, /^url(-prefix)?$/i)) {
                return null;
            }
            var pos = this.mark();
            var node = this.createNode(nodes.NodeType.URILiteral);
            this.accept(cssScanner_1.TokenType.Ident);
            if (this.hasWhitespace() || !this.peek(cssScanner_1.TokenType.ParenthesisL)) {
                this.restoreAtMark(pos);
                return null;
            }
            this.scanner.inURL = true;
            this.consumeToken(); // consume ()
            node.addChild(this._parseURLArgument()); // argument is optional
            this.scanner.inURL = false;
            if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected);
            }
            return this.finish(node);
        };
        Parser.prototype._parseURLArgument = function () {
            var node = this.create(nodes.Node);
            if (!this.accept(cssScanner_1.TokenType.String) && !this.accept(cssScanner_1.TokenType.BadString) && !this.acceptUnquotedString()) {
                return null;
            }
            return this.finish(node);
        };
        Parser.prototype._parseIdent = function (referenceTypes) {
            if (!this.peek(cssScanner_1.TokenType.Ident)) {
                return null;
            }
            var node = this.create(nodes.Identifier);
            if (referenceTypes) {
                node.referenceTypes = referenceTypes;
            }
            node.isCustomProperty = this.peekRegExp(cssScanner_1.TokenType.Ident, /^--/);
            this.consumeToken();
            return this.finish(node);
        };
        Parser.prototype._parseFunction = function () {
            var pos = this.mark();
            var node = this.create(nodes.Function);
            if (!node.setIdentifier(this._parseFunctionIdentifier())) {
                return null;
            }
            if (this.hasWhitespace() || !this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                this.restoreAtMark(pos);
                return null;
            }
            if (node.getArguments().addChild(this._parseFunctionArgument())) {
                while (this.accept(cssScanner_1.TokenType.Comma)) {
                    if (this.peek(cssScanner_1.TokenType.ParenthesisR)) {
                        break;
                    }
                    if (!node.getArguments().addChild(this._parseFunctionArgument())) {
                        this.markError(node, cssErrors_1.ParseError.ExpressionExpected);
                    }
                }
            }
            if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected);
            }
            return this.finish(node);
        };
        Parser.prototype._parseFunctionIdentifier = function () {
            if (!this.peek(cssScanner_1.TokenType.Ident)) {
                return null;
            }
            var node = this.create(nodes.Identifier);
            node.referenceTypes = [nodes.ReferenceType.Function];
            if (this.acceptIdent('progid')) {
                // support for IE7 specific filters: 'progid:DXImageTransform.Microsoft.MotionBlur(strength=13, direction=310)'
                if (this.accept(cssScanner_1.TokenType.Colon)) {
                    while (this.accept(cssScanner_1.TokenType.Ident) && this.acceptDelim('.')) {
                        // loop
                    }
                }
                return this.finish(node);
            }
            this.consumeToken();
            return this.finish(node);
        };
        Parser.prototype._parseFunctionArgument = function () {
            var node = this.create(nodes.FunctionArgument);
            if (node.setValue(this._parseExpr(true))) {
                return this.finish(node);
            }
            return null;
        };
        Parser.prototype._parseHexColor = function () {
            if (this.peekRegExp(cssScanner_1.TokenType.Hash, /^#([A-Fa-f0-9]{3}|[A-Fa-f0-9]{4}|[A-Fa-f0-9]{6}|[A-Fa-f0-9]{8})$/g)) {
                var node = this.create(nodes.HexColorValue);
                this.consumeToken();
                return this.finish(node);
            }
            else {
                return null;
            }
        };
        return Parser;
    }());
    exports.Parser = Parser;
});

(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/utils/arrays',["require", "exports"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.union = exports.includes = exports.findFirst = void 0;
    /**
     * Takes a sorted array and a function p. The array is sorted in such a way that all elements where p(x) is false
     * are located before all elements where p(x) is true.
     * @returns the least x for which p(x) is true or array.length if no element fullfills the given function.
     */
    function findFirst(array, p) {
        var low = 0, high = array.length;
        if (high === 0) {
            return 0; // no children
        }
        while (low < high) {
            var mid = Math.floor((low + high) / 2);
            if (p(array[mid])) {
                high = mid;
            }
            else {
                low = mid + 1;
            }
        }
        return low;
    }
    exports.findFirst = findFirst;
    function includes(array, item) {
        return array.indexOf(item) !== -1;
    }
    exports.includes = includes;
    function union() {
        var arrays = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            arrays[_i] = arguments[_i];
        }
        var result = [];
        for (var _a = 0, arrays_1 = arrays; _a < arrays_1.length; _a++) {
            var array = arrays_1[_a];
            for (var _b = 0, array_1 = array; _b < array_1.length; _b++) {
                var item = array_1[_b];
                if (!includes(result, item)) {
                    result.push(item);
                }
            }
        }
        return result;
    }
    exports.union = union;
});

var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/parser/cssSymbolScope',["require", "exports", "./cssNodes", "../utils/arrays"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.Symbols = exports.ScopeBuilder = exports.Symbol = exports.GlobalScope = exports.Scope = void 0;
    var nodes = require("./cssNodes");
    var arrays_1 = require("../utils/arrays");
    var Scope = /** @class */ (function () {
        function Scope(offset, length) {
            this.offset = offset;
            this.length = length;
            this.symbols = [];
            this.parent = null;
            this.children = [];
        }
        Scope.prototype.addChild = function (scope) {
            this.children.push(scope);
            scope.setParent(this);
        };
        Scope.prototype.setParent = function (scope) {
            this.parent = scope;
        };
        Scope.prototype.findScope = function (offset, length) {
            if (length === void 0) { length = 0; }
            if (this.offset <= offset && this.offset + this.length > offset + length || this.offset === offset && this.length === length) {
                return this.findInScope(offset, length);
            }
            return null;
        };
        Scope.prototype.findInScope = function (offset, length) {
            if (length === void 0) { length = 0; }
            // find the first scope child that has an offset larger than offset + length
            var end = offset + length;
            var idx = arrays_1.findFirst(this.children, function (s) { return s.offset > end; });
            if (idx === 0) {
                // all scopes have offsets larger than our end
                return this;
            }
            var res = this.children[idx - 1];
            if (res.offset <= offset && res.offset + res.length >= offset + length) {
                return res.findInScope(offset, length);
            }
            return this;
        };
        Scope.prototype.addSymbol = function (symbol) {
            this.symbols.push(symbol);
        };
        Scope.prototype.getSymbol = function (name, type) {
            for (var index = 0; index < this.symbols.length; index++) {
                var symbol = this.symbols[index];
                if (symbol.name === name && symbol.type === type) {
                    return symbol;
                }
            }
            return null;
        };
        Scope.prototype.getSymbols = function () {
            return this.symbols;
        };
        return Scope;
    }());
    exports.Scope = Scope;
    var GlobalScope = /** @class */ (function (_super) {
        __extends(GlobalScope, _super);
        function GlobalScope() {
            return _super.call(this, 0, Number.MAX_VALUE) || this;
        }
        return GlobalScope;
    }(Scope));
    exports.GlobalScope = GlobalScope;
    var Symbol = /** @class */ (function () {
        function Symbol(name, value, node, type) {
            this.name = name;
            this.value = value;
            this.node = node;
            this.type = type;
        }
        return Symbol;
    }());
    exports.Symbol = Symbol;
    var ScopeBuilder = /** @class */ (function () {
        function ScopeBuilder(scope) {
            this.scope = scope;
        }
        ScopeBuilder.prototype.addSymbol = function (node, name, value, type) {
            if (node.offset !== -1) {
                var current = this.scope.findScope(node.offset, node.length);
                if (current) {
                    current.addSymbol(new Symbol(name, value, node, type));
                }
            }
        };
        ScopeBuilder.prototype.addScope = function (node) {
            if (node.offset !== -1) {
                var current = this.scope.findScope(node.offset, node.length);
                if (current && (current.offset !== node.offset || current.length !== node.length)) { // scope already known?
                    var newScope = new Scope(node.offset, node.length);
                    current.addChild(newScope);
                    return newScope;
                }
                return current;
            }
            return null;
        };
        ScopeBuilder.prototype.addSymbolToChildScope = function (scopeNode, node, name, value, type) {
            if (scopeNode && scopeNode.offset !== -1) {
                var current = this.addScope(scopeNode); // create the scope or gets the existing one
                if (current) {
                    current.addSymbol(new Symbol(name, value, node, type));
                }
            }
        };
        ScopeBuilder.prototype.visitNode = function (node) {
            switch (node.type) {
                case nodes.NodeType.Keyframe:
                    this.addSymbol(node, node.getName(), void 0, nodes.ReferenceType.Keyframe);
                    return true;
                case nodes.NodeType.CustomPropertyDeclaration:
                    return this.visitCustomPropertyDeclarationNode(node);
                case nodes.NodeType.VariableDeclaration:
                    return this.visitVariableDeclarationNode(node);
                case nodes.NodeType.Ruleset:
                    return this.visitRuleSet(node);
                case nodes.NodeType.MixinDeclaration:
                    this.addSymbol(node, node.getName(), void 0, nodes.ReferenceType.Mixin);
                    return true;
                case nodes.NodeType.FunctionDeclaration:
                    this.addSymbol(node, node.getName(), void 0, nodes.ReferenceType.Function);
                    return true;
                case nodes.NodeType.FunctionParameter: {
                    return this.visitFunctionParameterNode(node);
                }
                case nodes.NodeType.Declarations:
                    this.addScope(node);
                    return true;
                case nodes.NodeType.For:
                    var forNode = node;
                    var scopeNode = forNode.getDeclarations();
                    if (scopeNode && forNode.variable) {
                        this.addSymbolToChildScope(scopeNode, forNode.variable, forNode.variable.getName(), void 0, nodes.ReferenceType.Variable);
                    }
                    return true;
                case nodes.NodeType.Each: {
                    var eachNode = node;
                    var scopeNode_1 = eachNode.getDeclarations();
                    if (scopeNode_1) {
                        var variables = eachNode.getVariables().getChildren();
                        for (var _i = 0, variables_1 = variables; _i < variables_1.length; _i++) {
                            var variable = variables_1[_i];
                            this.addSymbolToChildScope(scopeNode_1, variable, variable.getName(), void 0, nodes.ReferenceType.Variable);
                        }
                    }
                    return true;
                }
            }
            return true;
        };
        ScopeBuilder.prototype.visitRuleSet = function (node) {
            var current = this.scope.findScope(node.offset, node.length);
            if (current) {
                for (var _i = 0, _a = node.getSelectors().getChildren(); _i < _a.length; _i++) {
                    var child = _a[_i];
                    if (child instanceof nodes.Selector) {
                        if (child.getChildren().length === 1) { // only selectors with a single element can be extended
                            current.addSymbol(new Symbol(child.getChild(0).getText(), void 0, child, nodes.ReferenceType.Rule));
                        }
                    }
                }
            }
            return true;
        };
        ScopeBuilder.prototype.visitVariableDeclarationNode = function (node) {
            var value = node.getValue() ? node.getValue().getText() : void 0;
            this.addSymbol(node, node.getName(), value, nodes.ReferenceType.Variable);
            return true;
        };
        ScopeBuilder.prototype.visitFunctionParameterNode = function (node) {
            // parameters are part of the body scope
            var scopeNode = node.getParent().getDeclarations();
            if (scopeNode) {
                var valueNode = node.getDefaultValue();
                var value = valueNode ? valueNode.getText() : void 0;
                this.addSymbolToChildScope(scopeNode, node, node.getName(), value, nodes.ReferenceType.Variable);
            }
            return true;
        };
        ScopeBuilder.prototype.visitCustomPropertyDeclarationNode = function (node) {
            var value = node.getValue() ? node.getValue().getText() : '';
            this.addCSSVariable(node.getProperty(), node.getProperty().getName(), value, nodes.ReferenceType.Variable);
            return true;
        };
        ScopeBuilder.prototype.addCSSVariable = function (node, name, value, type) {
            if (node.offset !== -1) {
                this.scope.addSymbol(new Symbol(name, value, node, type));
            }
        };
        return ScopeBuilder;
    }());
    exports.ScopeBuilder = ScopeBuilder;
    var Symbols = /** @class */ (function () {
        function Symbols(node) {
            this.global = new GlobalScope();
            node.acceptVisitor(new ScopeBuilder(this.global));
        }
        Symbols.prototype.findSymbolsAtOffset = function (offset, referenceType) {
            var scope = this.global.findScope(offset, 0);
            var result = [];
            var names = {};
            while (scope) {
                var symbols = scope.getSymbols();
                for (var i = 0; i < symbols.length; i++) {
                    var symbol = symbols[i];
                    if (symbol.type === referenceType && !names[symbol.name]) {
                        result.push(symbol);
                        names[symbol.name] = true;
                    }
                }
                scope = scope.parent;
            }
            return result;
        };
        Symbols.prototype.internalFindSymbol = function (node, referenceTypes) {
            var scopeNode = node;
            if (node.parent instanceof nodes.FunctionParameter && node.parent.getParent() instanceof nodes.BodyDeclaration) {
                scopeNode = node.parent.getParent().getDeclarations();
            }
            if (node.parent instanceof nodes.FunctionArgument && node.parent.getParent() instanceof nodes.Function) {
                var funcId = node.parent.getParent().getIdentifier();
                if (funcId) {
                    var functionSymbol = this.internalFindSymbol(funcId, [nodes.ReferenceType.Function]);
                    if (functionSymbol) {
                        scopeNode = functionSymbol.node.getDeclarations();
                    }
                }
            }
            if (!scopeNode) {
                return null;
            }
            var name = node.getText();
            var scope = this.global.findScope(scopeNode.offset, scopeNode.length);
            while (scope) {
                for (var index = 0; index < referenceTypes.length; index++) {
                    var type = referenceTypes[index];
                    var symbol = scope.getSymbol(name, type);
                    if (symbol) {
                        return symbol;
                    }
                }
                scope = scope.parent;
            }
            return null;
        };
        Symbols.prototype.evaluateReferenceTypes = function (node) {
            if (node instanceof nodes.Identifier) {
                var referenceTypes = node.referenceTypes;
                if (referenceTypes) {
                    return referenceTypes;
                }
                else {
                    if (node.isCustomProperty) {
                        return [nodes.ReferenceType.Variable];
                    }
                    // are a reference to a keyframe?
                    var decl = nodes.getParentDeclaration(node);
                    if (decl) {
                        var propertyName = decl.getNonPrefixedPropertyName();
                        if ((propertyName === 'animation' || propertyName === 'animation-name')
                            && decl.getValue() && decl.getValue().offset === node.offset) {
                            return [nodes.ReferenceType.Keyframe];
                        }
                    }
                }
            }
            else if (node instanceof nodes.Variable) {
                return [nodes.ReferenceType.Variable];
            }
            var selector = node.findAParent(nodes.NodeType.Selector, nodes.NodeType.ExtendsReference);
            if (selector) {
                return [nodes.ReferenceType.Rule];
            }
            return null;
        };
        Symbols.prototype.findSymbolFromNode = function (node) {
            if (!node) {
                return null;
            }
            while (node.type === nodes.NodeType.Interpolation) {
                node = node.getParent();
            }
            var referenceTypes = this.evaluateReferenceTypes(node);
            if (referenceTypes) {
                return this.internalFindSymbol(node, referenceTypes);
            }
            return null;
        };
        Symbols.prototype.matchesSymbol = function (node, symbol) {
            if (!node) {
                return false;
            }
            while (node.type === nodes.NodeType.Interpolation) {
                node = node.getParent();
            }
            if (!node.matches(symbol.name)) {
                return false;
            }
            var referenceTypes = this.evaluateReferenceTypes(node);
            if (!referenceTypes || referenceTypes.indexOf(symbol.type) === -1) {
                return false;
            }
            var nodeSymbol = this.internalFindSymbol(node, referenceTypes);
            return nodeSymbol === symbol;
        };
        Symbols.prototype.findSymbol = function (name, type, offset) {
            var scope = this.global.findScope(offset);
            while (scope) {
                var symbol = scope.getSymbol(name, type);
                if (symbol) {
                    return symbol;
                }
                scope = scope.parent;
            }
            return null;
        };
        return Symbols;
    }());
    exports.Symbols = Symbols;
});

(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-languageserver-types/main',["require", "exports"], factory);
    }
})(function (require, exports) {
    /* --------------------------------------------------------------------------------------------
     * Copyright (c) Microsoft Corporation. All rights reserved.
     * Licensed under the MIT License. See License.txt in the project root for license information.
     * ------------------------------------------------------------------------------------------ */
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.TextDocument = exports.EOL = exports.SelectionRange = exports.DocumentLink = exports.FormattingOptions = exports.CodeLens = exports.CodeAction = exports.CodeActionContext = exports.CodeActionKind = exports.DocumentSymbol = exports.SymbolInformation = exports.SymbolTag = exports.SymbolKind = exports.DocumentHighlight = exports.DocumentHighlightKind = exports.SignatureInformation = exports.ParameterInformation = exports.Hover = exports.MarkedString = exports.CompletionList = exports.CompletionItem = exports.InsertTextMode = exports.InsertReplaceEdit = exports.CompletionItemTag = exports.InsertTextFormat = exports.CompletionItemKind = exports.MarkupContent = exports.MarkupKind = exports.TextDocumentItem = exports.OptionalVersionedTextDocumentIdentifier = exports.VersionedTextDocumentIdentifier = exports.TextDocumentIdentifier = exports.WorkspaceChange = exports.WorkspaceEdit = exports.DeleteFile = exports.RenameFile = exports.CreateFile = exports.TextDocumentEdit = exports.AnnotatedTextEdit = exports.ChangeAnnotationIdentifier = exports.ChangeAnnotation = exports.TextEdit = exports.Command = exports.Diagnostic = exports.CodeDescription = exports.DiagnosticTag = exports.DiagnosticSeverity = exports.DiagnosticRelatedInformation = exports.FoldingRange = exports.FoldingRangeKind = exports.ColorPresentation = exports.ColorInformation = exports.Color = exports.LocationLink = exports.Location = exports.Range = exports.Position = exports.uinteger = exports.integer = void 0;
    var integer;
    (function (integer) {
        integer.MIN_VALUE = -2147483648;
        integer.MAX_VALUE = 2147483647;
    })(integer = exports.integer || (exports.integer = {}));
    var uinteger;
    (function (uinteger) {
        uinteger.MIN_VALUE = 0;
        uinteger.MAX_VALUE = 2147483647;
    })(uinteger = exports.uinteger || (exports.uinteger = {}));
    /**
     * The Position namespace provides helper functions to work with
     * [Position](#Position) literals.
     */
    var Position;
    (function (Position) {
        /**
         * Creates a new Position literal from the given line and character.
         * @param line The position's line.
         * @param character The position's character.
         */
        function create(line, character) {
            if (line === Number.MAX_VALUE) {
                line = uinteger.MAX_VALUE;
            }
            if (character === Number.MAX_VALUE) {
                character = uinteger.MAX_VALUE;
            }
            return { line: line, character: character };
        }
        Position.create = create;
        /**
         * Checks whether the given literal conforms to the [Position](#Position) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.objectLiteral(candidate) && Is.uinteger(candidate.line) && Is.uinteger(candidate.character);
        }
        Position.is = is;
    })(Position = exports.Position || (exports.Position = {}));
    /**
     * The Range namespace provides helper functions to work with
     * [Range](#Range) literals.
     */
    var Range;
    (function (Range) {
        function create(one, two, three, four) {
            if (Is.uinteger(one) && Is.uinteger(two) && Is.uinteger(three) && Is.uinteger(four)) {
                return { start: Position.create(one, two), end: Position.create(three, four) };
            }
            else if (Position.is(one) && Position.is(two)) {
                return { start: one, end: two };
            }
            else {
                throw new Error("Range#create called with invalid arguments[" + one + ", " + two + ", " + three + ", " + four + "]");
            }
        }
        Range.create = create;
        /**
         * Checks whether the given literal conforms to the [Range](#Range) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.objectLiteral(candidate) && Position.is(candidate.start) && Position.is(candidate.end);
        }
        Range.is = is;
    })(Range = exports.Range || (exports.Range = {}));
    /**
     * The Location namespace provides helper functions to work with
     * [Location](#Location) literals.
     */
    var Location;
    (function (Location) {
        /**
         * Creates a Location literal.
         * @param uri The location's uri.
         * @param range The location's range.
         */
        function create(uri, range) {
            return { uri: uri, range: range };
        }
        Location.create = create;
        /**
         * Checks whether the given literal conforms to the [Location](#Location) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.defined(candidate) && Range.is(candidate.range) && (Is.string(candidate.uri) || Is.undefined(candidate.uri));
        }
        Location.is = is;
    })(Location = exports.Location || (exports.Location = {}));
    /**
     * The LocationLink namespace provides helper functions to work with
     * [LocationLink](#LocationLink) literals.
     */
    var LocationLink;
    (function (LocationLink) {
        /**
         * Creates a LocationLink literal.
         * @param targetUri The definition's uri.
         * @param targetRange The full range of the definition.
         * @param targetSelectionRange The span of the symbol definition at the target.
         * @param originSelectionRange The span of the symbol being defined in the originating source file.
         */
        function create(targetUri, targetRange, targetSelectionRange, originSelectionRange) {
            return { targetUri: targetUri, targetRange: targetRange, targetSelectionRange: targetSelectionRange, originSelectionRange: originSelectionRange };
        }
        LocationLink.create = create;
        /**
         * Checks whether the given literal conforms to the [LocationLink](#LocationLink) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.defined(candidate) && Range.is(candidate.targetRange) && Is.string(candidate.targetUri)
                && (Range.is(candidate.targetSelectionRange) || Is.undefined(candidate.targetSelectionRange))
                && (Range.is(candidate.originSelectionRange) || Is.undefined(candidate.originSelectionRange));
        }
        LocationLink.is = is;
    })(LocationLink = exports.LocationLink || (exports.LocationLink = {}));
    /**
     * The Color namespace provides helper functions to work with
     * [Color](#Color) literals.
     */
    var Color;
    (function (Color) {
        /**
         * Creates a new Color literal.
         */
        function create(red, green, blue, alpha) {
            return {
                red: red,
                green: green,
                blue: blue,
                alpha: alpha,
            };
        }
        Color.create = create;
        /**
         * Checks whether the given literal conforms to the [Color](#Color) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.numberRange(candidate.red, 0, 1)
                && Is.numberRange(candidate.green, 0, 1)
                && Is.numberRange(candidate.blue, 0, 1)
                && Is.numberRange(candidate.alpha, 0, 1);
        }
        Color.is = is;
    })(Color = exports.Color || (exports.Color = {}));
    /**
     * The ColorInformation namespace provides helper functions to work with
     * [ColorInformation](#ColorInformation) literals.
     */
    var ColorInformation;
    (function (ColorInformation) {
        /**
         * Creates a new ColorInformation literal.
         */
        function create(range, color) {
            return {
                range: range,
                color: color,
            };
        }
        ColorInformation.create = create;
        /**
         * Checks whether the given literal conforms to the [ColorInformation](#ColorInformation) interface.
         */
        function is(value) {
            var candidate = value;
            return Range.is(candidate.range) && Color.is(candidate.color);
        }
        ColorInformation.is = is;
    })(ColorInformation = exports.ColorInformation || (exports.ColorInformation = {}));
    /**
     * The Color namespace provides helper functions to work with
     * [ColorPresentation](#ColorPresentation) literals.
     */
    var ColorPresentation;
    (function (ColorPresentation) {
        /**
         * Creates a new ColorInformation literal.
         */
        function create(label, textEdit, additionalTextEdits) {
            return {
                label: label,
                textEdit: textEdit,
                additionalTextEdits: additionalTextEdits,
            };
        }
        ColorPresentation.create = create;
        /**
         * Checks whether the given literal conforms to the [ColorInformation](#ColorInformation) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.string(candidate.label)
                && (Is.undefined(candidate.textEdit) || TextEdit.is(candidate))
                && (Is.undefined(candidate.additionalTextEdits) || Is.typedArray(candidate.additionalTextEdits, TextEdit.is));
        }
        ColorPresentation.is = is;
    })(ColorPresentation = exports.ColorPresentation || (exports.ColorPresentation = {}));
    /**
     * Enum of known range kinds
     */
    var FoldingRangeKind;
    (function (FoldingRangeKind) {
        /**
         * Folding range for a comment
         */
        FoldingRangeKind["Comment"] = "comment";
        /**
         * Folding range for a imports or includes
         */
        FoldingRangeKind["Imports"] = "imports";
        /**
         * Folding range for a region (e.g. `#region`)
         */
        FoldingRangeKind["Region"] = "region";
    })(FoldingRangeKind = exports.FoldingRangeKind || (exports.FoldingRangeKind = {}));
    /**
     * The folding range namespace provides helper functions to work with
     * [FoldingRange](#FoldingRange) literals.
     */
    var FoldingRange;
    (function (FoldingRange) {
        /**
         * Creates a new FoldingRange literal.
         */
        function create(startLine, endLine, startCharacter, endCharacter, kind) {
            var result = {
                startLine: startLine,
                endLine: endLine
            };
            if (Is.defined(startCharacter)) {
                result.startCharacter = startCharacter;
            }
            if (Is.defined(endCharacter)) {
                result.endCharacter = endCharacter;
            }
            if (Is.defined(kind)) {
                result.kind = kind;
            }
            return result;
        }
        FoldingRange.create = create;
        /**
         * Checks whether the given literal conforms to the [FoldingRange](#FoldingRange) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.uinteger(candidate.startLine) && Is.uinteger(candidate.startLine)
                && (Is.undefined(candidate.startCharacter) || Is.uinteger(candidate.startCharacter))
                && (Is.undefined(candidate.endCharacter) || Is.uinteger(candidate.endCharacter))
                && (Is.undefined(candidate.kind) || Is.string(candidate.kind));
        }
        FoldingRange.is = is;
    })(FoldingRange = exports.FoldingRange || (exports.FoldingRange = {}));
    /**
     * The DiagnosticRelatedInformation namespace provides helper functions to work with
     * [DiagnosticRelatedInformation](#DiagnosticRelatedInformation) literals.
     */
    var DiagnosticRelatedInformation;
    (function (DiagnosticRelatedInformation) {
        /**
         * Creates a new DiagnosticRelatedInformation literal.
         */
        function create(location, message) {
            return {
                location: location,
                message: message
            };
        }
        DiagnosticRelatedInformation.create = create;
        /**
         * Checks whether the given literal conforms to the [DiagnosticRelatedInformation](#DiagnosticRelatedInformation) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.defined(candidate) && Location.is(candidate.location) && Is.string(candidate.message);
        }
        DiagnosticRelatedInformation.is = is;
    })(DiagnosticRelatedInformation = exports.DiagnosticRelatedInformation || (exports.DiagnosticRelatedInformation = {}));
    /**
     * The diagnostic's severity.
     */
    var DiagnosticSeverity;
    (function (DiagnosticSeverity) {
        /**
         * Reports an error.
         */
        DiagnosticSeverity.Error = 1;
        /**
         * Reports a warning.
         */
        DiagnosticSeverity.Warning = 2;
        /**
         * Reports an information.
         */
        DiagnosticSeverity.Information = 3;
        /**
         * Reports a hint.
         */
        DiagnosticSeverity.Hint = 4;
    })(DiagnosticSeverity = exports.DiagnosticSeverity || (exports.DiagnosticSeverity = {}));
    /**
     * The diagnostic tags.
     *
     * @since 3.15.0
     */
    var DiagnosticTag;
    (function (DiagnosticTag) {
        /**
         * Unused or unnecessary code.
         *
         * Clients are allowed to render diagnostics with this tag faded out instead of having
         * an error squiggle.
         */
        DiagnosticTag.Unnecessary = 1;
        /**
         * Deprecated or obsolete code.
         *
         * Clients are allowed to rendered diagnostics with this tag strike through.
         */
        DiagnosticTag.Deprecated = 2;
    })(DiagnosticTag = exports.DiagnosticTag || (exports.DiagnosticTag = {}));
    /**
     * The CodeDescription namespace provides functions to deal with descriptions for diagnostic codes.
     *
     * @since 3.16.0
     */
    var CodeDescription;
    (function (CodeDescription) {
        function is(value) {
            var candidate = value;
            return candidate !== undefined && candidate !== null && Is.string(candidate.href);
        }
        CodeDescription.is = is;
    })(CodeDescription = exports.CodeDescription || (exports.CodeDescription = {}));
    /**
     * The Diagnostic namespace provides helper functions to work with
     * [Diagnostic](#Diagnostic) literals.
     */
    var Diagnostic;
    (function (Diagnostic) {
        /**
         * Creates a new Diagnostic literal.
         */
        function create(range, message, severity, code, source, relatedInformation) {
            var result = { range: range, message: message };
            if (Is.defined(severity)) {
                result.severity = severity;
            }
            if (Is.defined(code)) {
                result.code = code;
            }
            if (Is.defined(source)) {
                result.source = source;
            }
            if (Is.defined(relatedInformation)) {
                result.relatedInformation = relatedInformation;
            }
            return result;
        }
        Diagnostic.create = create;
        /**
         * Checks whether the given literal conforms to the [Diagnostic](#Diagnostic) interface.
         */
        function is(value) {
            var _a;
            var candidate = value;
            return Is.defined(candidate)
                && Range.is(candidate.range)
                && Is.string(candidate.message)
                && (Is.number(candidate.severity) || Is.undefined(candidate.severity))
                && (Is.integer(candidate.code) || Is.string(candidate.code) || Is.undefined(candidate.code))
                && (Is.undefined(candidate.codeDescription) || (Is.string((_a = candidate.codeDescription) === null || _a === void 0 ? void 0 : _a.href)))
                && (Is.string(candidate.source) || Is.undefined(candidate.source))
                && (Is.undefined(candidate.relatedInformation) || Is.typedArray(candidate.relatedInformation, DiagnosticRelatedInformation.is));
        }
        Diagnostic.is = is;
    })(Diagnostic = exports.Diagnostic || (exports.Diagnostic = {}));
    /**
     * The Command namespace provides helper functions to work with
     * [Command](#Command) literals.
     */
    var Command;
    (function (Command) {
        /**
         * Creates a new Command literal.
         */
        function create(title, command) {
            var args = [];
            for (var _i = 2; _i < arguments.length; _i++) {
                args[_i - 2] = arguments[_i];
            }
            var result = { title: title, command: command };
            if (Is.defined(args) && args.length > 0) {
                result.arguments = args;
            }
            return result;
        }
        Command.create = create;
        /**
         * Checks whether the given literal conforms to the [Command](#Command) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.defined(candidate) && Is.string(candidate.title) && Is.string(candidate.command);
        }
        Command.is = is;
    })(Command = exports.Command || (exports.Command = {}));
    /**
     * The TextEdit namespace provides helper function to create replace,
     * insert and delete edits more easily.
     */
    var TextEdit;
    (function (TextEdit) {
        /**
         * Creates a replace text edit.
         * @param range The range of text to be replaced.
         * @param newText The new text.
         */
        function replace(range, newText) {
            return { range: range, newText: newText };
        }
        TextEdit.replace = replace;
        /**
         * Creates a insert text edit.
         * @param position The position to insert the text at.
         * @param newText The text to be inserted.
         */
        function insert(position, newText) {
            return { range: { start: position, end: position }, newText: newText };
        }
        TextEdit.insert = insert;
        /**
         * Creates a delete text edit.
         * @param range The range of text to be deleted.
         */
        function del(range) {
            return { range: range, newText: '' };
        }
        TextEdit.del = del;
        function is(value) {
            var candidate = value;
            return Is.objectLiteral(candidate)
                && Is.string(candidate.newText)
                && Range.is(candidate.range);
        }
        TextEdit.is = is;
    })(TextEdit = exports.TextEdit || (exports.TextEdit = {}));
    var ChangeAnnotation;
    (function (ChangeAnnotation) {
        function create(label, needsConfirmation, description) {
            var result = { label: label };
            if (needsConfirmation !== undefined) {
                result.needsConfirmation = needsConfirmation;
            }
            if (description !== undefined) {
                result.description = description;
            }
            return result;
        }
        ChangeAnnotation.create = create;
        function is(value) {
            var candidate = value;
            return candidate !== undefined && Is.objectLiteral(candidate) && Is.string(candidate.label) &&
                (Is.boolean(candidate.needsConfirmation) || candidate.needsConfirmation === undefined) &&
                (Is.string(candidate.description) || candidate.description === undefined);
        }
        ChangeAnnotation.is = is;
    })(ChangeAnnotation = exports.ChangeAnnotation || (exports.ChangeAnnotation = {}));
    var ChangeAnnotationIdentifier;
    (function (ChangeAnnotationIdentifier) {
        function is(value) {
            var candidate = value;
            return typeof candidate === 'string';
        }
        ChangeAnnotationIdentifier.is = is;
    })(ChangeAnnotationIdentifier = exports.ChangeAnnotationIdentifier || (exports.ChangeAnnotationIdentifier = {}));
    var AnnotatedTextEdit;
    (function (AnnotatedTextEdit) {
        /**
         * Creates an annotated replace text edit.
         *
         * @param range The range of text to be replaced.
         * @param newText The new text.
         * @param annotation The annotation.
         */
        function replace(range, newText, annotation) {
            return { range: range, newText: newText, annotationId: annotation };
        }
        AnnotatedTextEdit.replace = replace;
        /**
         * Creates an annotated insert text edit.
         *
         * @param position The position to insert the text at.
         * @param newText The text to be inserted.
         * @param annotation The annotation.
         */
        function insert(position, newText, annotation) {
            return { range: { start: position, end: position }, newText: newText, annotationId: annotation };
        }
        AnnotatedTextEdit.insert = insert;
        /**
         * Creates an annotated delete text edit.
         *
         * @param range The range of text to be deleted.
         * @param annotation The annotation.
         */
        function del(range, annotation) {
            return { range: range, newText: '', annotationId: annotation };
        }
        AnnotatedTextEdit.del = del;
        function is(value) {
            var candidate = value;
            return TextEdit.is(candidate) && (ChangeAnnotation.is(candidate.annotationId) || ChangeAnnotationIdentifier.is(candidate.annotationId));
        }
        AnnotatedTextEdit.is = is;
    })(AnnotatedTextEdit = exports.AnnotatedTextEdit || (exports.AnnotatedTextEdit = {}));
    /**
     * The TextDocumentEdit namespace provides helper function to create
     * an edit that manipulates a text document.
     */
    var TextDocumentEdit;
    (function (TextDocumentEdit) {
        /**
         * Creates a new `TextDocumentEdit`
         */
        function create(textDocument, edits) {
            return { textDocument: textDocument, edits: edits };
        }
        TextDocumentEdit.create = create;
        function is(value) {
            var candidate = value;
            return Is.defined(candidate)
                && OptionalVersionedTextDocumentIdentifier.is(candidate.textDocument)
                && Array.isArray(candidate.edits);
        }
        TextDocumentEdit.is = is;
    })(TextDocumentEdit = exports.TextDocumentEdit || (exports.TextDocumentEdit = {}));
    var CreateFile;
    (function (CreateFile) {
        function create(uri, options, annotation) {
            var result = {
                kind: 'create',
                uri: uri
            };
            if (options !== undefined && (options.overwrite !== undefined || options.ignoreIfExists !== undefined)) {
                result.options = options;
            }
            if (annotation !== undefined) {
                result.annotationId = annotation;
            }
            return result;
        }
        CreateFile.create = create;
        function is(value) {
            var candidate = value;
            return candidate && candidate.kind === 'create' && Is.string(candidate.uri) && (candidate.options === undefined ||
                ((candidate.options.overwrite === undefined || Is.boolean(candidate.options.overwrite)) && (candidate.options.ignoreIfExists === undefined || Is.boolean(candidate.options.ignoreIfExists)))) && (candidate.annotationId === undefined || ChangeAnnotationIdentifier.is(candidate.annotationId));
        }
        CreateFile.is = is;
    })(CreateFile = exports.CreateFile || (exports.CreateFile = {}));
    var RenameFile;
    (function (RenameFile) {
        function create(oldUri, newUri, options, annotation) {
            var result = {
                kind: 'rename',
                oldUri: oldUri,
                newUri: newUri
            };
            if (options !== undefined && (options.overwrite !== undefined || options.ignoreIfExists !== undefined)) {
                result.options = options;
            }
            if (annotation !== undefined) {
                result.annotationId = annotation;
            }
            return result;
        }
        RenameFile.create = create;
        function is(value) {
            var candidate = value;
            return candidate && candidate.kind === 'rename' && Is.string(candidate.oldUri) && Is.string(candidate.newUri) && (candidate.options === undefined ||
                ((candidate.options.overwrite === undefined || Is.boolean(candidate.options.overwrite)) && (candidate.options.ignoreIfExists === undefined || Is.boolean(candidate.options.ignoreIfExists)))) && (candidate.annotationId === undefined || ChangeAnnotationIdentifier.is(candidate.annotationId));
        }
        RenameFile.is = is;
    })(RenameFile = exports.RenameFile || (exports.RenameFile = {}));
    var DeleteFile;
    (function (DeleteFile) {
        function create(uri, options, annotation) {
            var result = {
                kind: 'delete',
                uri: uri
            };
            if (options !== undefined && (options.recursive !== undefined || options.ignoreIfNotExists !== undefined)) {
                result.options = options;
            }
            if (annotation !== undefined) {
                result.annotationId = annotation;
            }
            return result;
        }
        DeleteFile.create = create;
        function is(value) {
            var candidate = value;
            return candidate && candidate.kind === 'delete' && Is.string(candidate.uri) && (candidate.options === undefined ||
                ((candidate.options.recursive === undefined || Is.boolean(candidate.options.recursive)) && (candidate.options.ignoreIfNotExists === undefined || Is.boolean(candidate.options.ignoreIfNotExists)))) && (candidate.annotationId === undefined || ChangeAnnotationIdentifier.is(candidate.annotationId));
        }
        DeleteFile.is = is;
    })(DeleteFile = exports.DeleteFile || (exports.DeleteFile = {}));
    var WorkspaceEdit;
    (function (WorkspaceEdit) {
        function is(value) {
            var candidate = value;
            return candidate &&
                (candidate.changes !== undefined || candidate.documentChanges !== undefined) &&
                (candidate.documentChanges === undefined || candidate.documentChanges.every(function (change) {
                    if (Is.string(change.kind)) {
                        return CreateFile.is(change) || RenameFile.is(change) || DeleteFile.is(change);
                    }
                    else {
                        return TextDocumentEdit.is(change);
                    }
                }));
        }
        WorkspaceEdit.is = is;
    })(WorkspaceEdit = exports.WorkspaceEdit || (exports.WorkspaceEdit = {}));
    var TextEditChangeImpl = /** @class */ (function () {
        function TextEditChangeImpl(edits, changeAnnotations) {
            this.edits = edits;
            this.changeAnnotations = changeAnnotations;
        }
        TextEditChangeImpl.prototype.insert = function (position, newText, annotation) {
            var edit;
            var id;
            if (annotation === undefined) {
                edit = TextEdit.insert(position, newText);
            }
            else if (ChangeAnnotationIdentifier.is(annotation)) {
                id = annotation;
                edit = AnnotatedTextEdit.insert(position, newText, annotation);
            }
            else {
                this.assertChangeAnnotations(this.changeAnnotations);
                id = this.changeAnnotations.manage(annotation);
                edit = AnnotatedTextEdit.insert(position, newText, id);
            }
            this.edits.push(edit);
            if (id !== undefined) {
                return id;
            }
        };
        TextEditChangeImpl.prototype.replace = function (range, newText, annotation) {
            var edit;
            var id;
            if (annotation === undefined) {
                edit = TextEdit.replace(range, newText);
            }
            else if (ChangeAnnotationIdentifier.is(annotation)) {
                id = annotation;
                edit = AnnotatedTextEdit.replace(range, newText, annotation);
            }
            else {
                this.assertChangeAnnotations(this.changeAnnotations);
                id = this.changeAnnotations.manage(annotation);
                edit = AnnotatedTextEdit.replace(range, newText, id);
            }
            this.edits.push(edit);
            if (id !== undefined) {
                return id;
            }
        };
        TextEditChangeImpl.prototype.delete = function (range, annotation) {
            var edit;
            var id;
            if (annotation === undefined) {
                edit = TextEdit.del(range);
            }
            else if (ChangeAnnotationIdentifier.is(annotation)) {
                id = annotation;
                edit = AnnotatedTextEdit.del(range, annotation);
            }
            else {
                this.assertChangeAnnotations(this.changeAnnotations);
                id = this.changeAnnotations.manage(annotation);
                edit = AnnotatedTextEdit.del(range, id);
            }
            this.edits.push(edit);
            if (id !== undefined) {
                return id;
            }
        };
        TextEditChangeImpl.prototype.add = function (edit) {
            this.edits.push(edit);
        };
        TextEditChangeImpl.prototype.all = function () {
            return this.edits;
        };
        TextEditChangeImpl.prototype.clear = function () {
            this.edits.splice(0, this.edits.length);
        };
        TextEditChangeImpl.prototype.assertChangeAnnotations = function (value) {
            if (value === undefined) {
                throw new Error("Text edit change is not configured to manage change annotations.");
            }
        };
        return TextEditChangeImpl;
    }());
    /**
     * A helper class
     */
    var ChangeAnnotations = /** @class */ (function () {
        function ChangeAnnotations(annotations) {
            this._annotations = annotations === undefined ? Object.create(null) : annotations;
            this._counter = 0;
            this._size = 0;
        }
        ChangeAnnotations.prototype.all = function () {
            return this._annotations;
        };
        Object.defineProperty(ChangeAnnotations.prototype, "size", {
            get: function () {
                return this._size;
            },
            enumerable: false,
            configurable: true
        });
        ChangeAnnotations.prototype.manage = function (idOrAnnotation, annotation) {
            var id;
            if (ChangeAnnotationIdentifier.is(idOrAnnotation)) {
                id = idOrAnnotation;
            }
            else {
                id = this.nextId();
                annotation = idOrAnnotation;
            }
            if (this._annotations[id] !== undefined) {
                throw new Error("Id " + id + " is already in use.");
            }
            if (annotation === undefined) {
                throw new Error("No annotation provided for id " + id);
            }
            this._annotations[id] = annotation;
            this._size++;
            return id;
        };
        ChangeAnnotations.prototype.nextId = function () {
            this._counter++;
            return this._counter.toString();
        };
        return ChangeAnnotations;
    }());
    /**
     * A workspace change helps constructing changes to a workspace.
     */
    var WorkspaceChange = /** @class */ (function () {
        function WorkspaceChange(workspaceEdit) {
            var _this = this;
            this._textEditChanges = Object.create(null);
            if (workspaceEdit !== undefined) {
                this._workspaceEdit = workspaceEdit;
                if (workspaceEdit.documentChanges) {
                    this._changeAnnotations = new ChangeAnnotations(workspaceEdit.changeAnnotations);
                    workspaceEdit.changeAnnotations = this._changeAnnotations.all();
                    workspaceEdit.documentChanges.forEach(function (change) {
                        if (TextDocumentEdit.is(change)) {
                            var textEditChange = new TextEditChangeImpl(change.edits, _this._changeAnnotations);
                            _this._textEditChanges[change.textDocument.uri] = textEditChange;
                        }
                    });
                }
                else if (workspaceEdit.changes) {
                    Object.keys(workspaceEdit.changes).forEach(function (key) {
                        var textEditChange = new TextEditChangeImpl(workspaceEdit.changes[key]);
                        _this._textEditChanges[key] = textEditChange;
                    });
                }
            }
            else {
                this._workspaceEdit = {};
            }
        }
        Object.defineProperty(WorkspaceChange.prototype, "edit", {
            /**
             * Returns the underlying [WorkspaceEdit](#WorkspaceEdit) literal
             * use to be returned from a workspace edit operation like rename.
             */
            get: function () {
                this.initDocumentChanges();
                if (this._changeAnnotations !== undefined) {
                    if (this._changeAnnotations.size === 0) {
                        this._workspaceEdit.changeAnnotations = undefined;
                    }
                    else {
                        this._workspaceEdit.changeAnnotations = this._changeAnnotations.all();
                    }
                }
                return this._workspaceEdit;
            },
            enumerable: false,
            configurable: true
        });
        WorkspaceChange.prototype.getTextEditChange = function (key) {
            if (OptionalVersionedTextDocumentIdentifier.is(key)) {
                this.initDocumentChanges();
                if (this._workspaceEdit.documentChanges === undefined) {
                    throw new Error('Workspace edit is not configured for document changes.');
                }
                var textDocument = { uri: key.uri, version: key.version };
                var result = this._textEditChanges[textDocument.uri];
                if (!result) {
                    var edits = [];
                    var textDocumentEdit = {
                        textDocument: textDocument,
                        edits: edits
                    };
                    this._workspaceEdit.documentChanges.push(textDocumentEdit);
                    result = new TextEditChangeImpl(edits, this._changeAnnotations);
                    this._textEditChanges[textDocument.uri] = result;
                }
                return result;
            }
            else {
                this.initChanges();
                if (this._workspaceEdit.changes === undefined) {
                    throw new Error('Workspace edit is not configured for normal text edit changes.');
                }
                var result = this._textEditChanges[key];
                if (!result) {
                    var edits = [];
                    this._workspaceEdit.changes[key] = edits;
                    result = new TextEditChangeImpl(edits);
                    this._textEditChanges[key] = result;
                }
                return result;
            }
        };
        WorkspaceChange.prototype.initDocumentChanges = function () {
            if (this._workspaceEdit.documentChanges === undefined && this._workspaceEdit.changes === undefined) {
                this._changeAnnotations = new ChangeAnnotations();
                this._workspaceEdit.documentChanges = [];
                this._workspaceEdit.changeAnnotations = this._changeAnnotations.all();
            }
        };
        WorkspaceChange.prototype.initChanges = function () {
            if (this._workspaceEdit.documentChanges === undefined && this._workspaceEdit.changes === undefined) {
                this._workspaceEdit.changes = Object.create(null);
            }
        };
        WorkspaceChange.prototype.createFile = function (uri, optionsOrAnnotation, options) {
            this.initDocumentChanges();
            if (this._workspaceEdit.documentChanges === undefined) {
                throw new Error('Workspace edit is not configured for document changes.');
            }
            var annotation;
            if (ChangeAnnotation.is(optionsOrAnnotation) || ChangeAnnotationIdentifier.is(optionsOrAnnotation)) {
                annotation = optionsOrAnnotation;
            }
            else {
                options = optionsOrAnnotation;
            }
            var operation;
            var id;
            if (annotation === undefined) {
                operation = CreateFile.create(uri, options);
            }
            else {
                id = ChangeAnnotationIdentifier.is(annotation) ? annotation : this._changeAnnotations.manage(annotation);
                operation = CreateFile.create(uri, options, id);
            }
            this._workspaceEdit.documentChanges.push(operation);
            if (id !== undefined) {
                return id;
            }
        };
        WorkspaceChange.prototype.renameFile = function (oldUri, newUri, optionsOrAnnotation, options) {
            this.initDocumentChanges();
            if (this._workspaceEdit.documentChanges === undefined) {
                throw new Error('Workspace edit is not configured for document changes.');
            }
            var annotation;
            if (ChangeAnnotation.is(optionsOrAnnotation) || ChangeAnnotationIdentifier.is(optionsOrAnnotation)) {
                annotation = optionsOrAnnotation;
            }
            else {
                options = optionsOrAnnotation;
            }
            var operation;
            var id;
            if (annotation === undefined) {
                operation = RenameFile.create(oldUri, newUri, options);
            }
            else {
                id = ChangeAnnotationIdentifier.is(annotation) ? annotation : this._changeAnnotations.manage(annotation);
                operation = RenameFile.create(oldUri, newUri, options, id);
            }
            this._workspaceEdit.documentChanges.push(operation);
            if (id !== undefined) {
                return id;
            }
        };
        WorkspaceChange.prototype.deleteFile = function (uri, optionsOrAnnotation, options) {
            this.initDocumentChanges();
            if (this._workspaceEdit.documentChanges === undefined) {
                throw new Error('Workspace edit is not configured for document changes.');
            }
            var annotation;
            if (ChangeAnnotation.is(optionsOrAnnotation) || ChangeAnnotationIdentifier.is(optionsOrAnnotation)) {
                annotation = optionsOrAnnotation;
            }
            else {
                options = optionsOrAnnotation;
            }
            var operation;
            var id;
            if (annotation === undefined) {
                operation = DeleteFile.create(uri, options);
            }
            else {
                id = ChangeAnnotationIdentifier.is(annotation) ? annotation : this._changeAnnotations.manage(annotation);
                operation = DeleteFile.create(uri, options, id);
            }
            this._workspaceEdit.documentChanges.push(operation);
            if (id !== undefined) {
                return id;
            }
        };
        return WorkspaceChange;
    }());
    exports.WorkspaceChange = WorkspaceChange;
    /**
     * The TextDocumentIdentifier namespace provides helper functions to work with
     * [TextDocumentIdentifier](#TextDocumentIdentifier) literals.
     */
    var TextDocumentIdentifier;
    (function (TextDocumentIdentifier) {
        /**
         * Creates a new TextDocumentIdentifier literal.
         * @param uri The document's uri.
         */
        function create(uri) {
            return { uri: uri };
        }
        TextDocumentIdentifier.create = create;
        /**
         * Checks whether the given literal conforms to the [TextDocumentIdentifier](#TextDocumentIdentifier) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.defined(candidate) && Is.string(candidate.uri);
        }
        TextDocumentIdentifier.is = is;
    })(TextDocumentIdentifier = exports.TextDocumentIdentifier || (exports.TextDocumentIdentifier = {}));
    /**
     * The VersionedTextDocumentIdentifier namespace provides helper functions to work with
     * [VersionedTextDocumentIdentifier](#VersionedTextDocumentIdentifier) literals.
     */
    var VersionedTextDocumentIdentifier;
    (function (VersionedTextDocumentIdentifier) {
        /**
         * Creates a new VersionedTextDocumentIdentifier literal.
         * @param uri The document's uri.
         * @param uri The document's text.
         */
        function create(uri, version) {
            return { uri: uri, version: version };
        }
        VersionedTextDocumentIdentifier.create = create;
        /**
         * Checks whether the given literal conforms to the [VersionedTextDocumentIdentifier](#VersionedTextDocumentIdentifier) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.defined(candidate) && Is.string(candidate.uri) && Is.integer(candidate.version);
        }
        VersionedTextDocumentIdentifier.is = is;
    })(VersionedTextDocumentIdentifier = exports.VersionedTextDocumentIdentifier || (exports.VersionedTextDocumentIdentifier = {}));
    /**
     * The OptionalVersionedTextDocumentIdentifier namespace provides helper functions to work with
     * [OptionalVersionedTextDocumentIdentifier](#OptionalVersionedTextDocumentIdentifier) literals.
     */
    var OptionalVersionedTextDocumentIdentifier;
    (function (OptionalVersionedTextDocumentIdentifier) {
        /**
         * Creates a new OptionalVersionedTextDocumentIdentifier literal.
         * @param uri The document's uri.
         * @param uri The document's text.
         */
        function create(uri, version) {
            return { uri: uri, version: version };
        }
        OptionalVersionedTextDocumentIdentifier.create = create;
        /**
         * Checks whether the given literal conforms to the [OptionalVersionedTextDocumentIdentifier](#OptionalVersionedTextDocumentIdentifier) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.defined(candidate) && Is.string(candidate.uri) && (candidate.version === null || Is.integer(candidate.version));
        }
        OptionalVersionedTextDocumentIdentifier.is = is;
    })(OptionalVersionedTextDocumentIdentifier = exports.OptionalVersionedTextDocumentIdentifier || (exports.OptionalVersionedTextDocumentIdentifier = {}));
    /**
     * The TextDocumentItem namespace provides helper functions to work with
     * [TextDocumentItem](#TextDocumentItem) literals.
     */
    var TextDocumentItem;
    (function (TextDocumentItem) {
        /**
         * Creates a new TextDocumentItem literal.
         * @param uri The document's uri.
         * @param languageId The document's language identifier.
         * @param version The document's version number.
         * @param text The document's text.
         */
        function create(uri, languageId, version, text) {
            return { uri: uri, languageId: languageId, version: version, text: text };
        }
        TextDocumentItem.create = create;
        /**
         * Checks whether the given literal conforms to the [TextDocumentItem](#TextDocumentItem) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.defined(candidate) && Is.string(candidate.uri) && Is.string(candidate.languageId) && Is.integer(candidate.version) && Is.string(candidate.text);
        }
        TextDocumentItem.is = is;
    })(TextDocumentItem = exports.TextDocumentItem || (exports.TextDocumentItem = {}));
    /**
     * Describes the content type that a client supports in various
     * result literals like `Hover`, `ParameterInfo` or `CompletionItem`.
     *
     * Please note that `MarkupKinds` must not start with a `$`. This kinds
     * are reserved for internal usage.
     */
    var MarkupKind;
    (function (MarkupKind) {
        /**
         * Plain text is supported as a content format
         */
        MarkupKind.PlainText = 'plaintext';
        /**
         * Markdown is supported as a content format
         */
        MarkupKind.Markdown = 'markdown';
    })(MarkupKind = exports.MarkupKind || (exports.MarkupKind = {}));
    (function (MarkupKind) {
        /**
         * Checks whether the given value is a value of the [MarkupKind](#MarkupKind) type.
         */
        function is(value) {
            var candidate = value;
            return candidate === MarkupKind.PlainText || candidate === MarkupKind.Markdown;
        }
        MarkupKind.is = is;
    })(MarkupKind = exports.MarkupKind || (exports.MarkupKind = {}));
    var MarkupContent;
    (function (MarkupContent) {
        /**
         * Checks whether the given value conforms to the [MarkupContent](#MarkupContent) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.objectLiteral(value) && MarkupKind.is(candidate.kind) && Is.string(candidate.value);
        }
        MarkupContent.is = is;
    })(MarkupContent = exports.MarkupContent || (exports.MarkupContent = {}));
    /**
     * The kind of a completion entry.
     */
    var CompletionItemKind;
    (function (CompletionItemKind) {
        CompletionItemKind.Text = 1;
        CompletionItemKind.Method = 2;
        CompletionItemKind.Function = 3;
        CompletionItemKind.Constructor = 4;
        CompletionItemKind.Field = 5;
        CompletionItemKind.Variable = 6;
        CompletionItemKind.Class = 7;
        CompletionItemKind.Interface = 8;
        CompletionItemKind.Module = 9;
        CompletionItemKind.Property = 10;
        CompletionItemKind.Unit = 11;
        CompletionItemKind.Value = 12;
        CompletionItemKind.Enum = 13;
        CompletionItemKind.Keyword = 14;
        CompletionItemKind.Snippet = 15;
        CompletionItemKind.Color = 16;
        CompletionItemKind.File = 17;
        CompletionItemKind.Reference = 18;
        CompletionItemKind.Folder = 19;
        CompletionItemKind.EnumMember = 20;
        CompletionItemKind.Constant = 21;
        CompletionItemKind.Struct = 22;
        CompletionItemKind.Event = 23;
        CompletionItemKind.Operator = 24;
        CompletionItemKind.TypeParameter = 25;
    })(CompletionItemKind = exports.CompletionItemKind || (exports.CompletionItemKind = {}));
    /**
     * Defines whether the insert text in a completion item should be interpreted as
     * plain text or a snippet.
     */
    var InsertTextFormat;
    (function (InsertTextFormat) {
        /**
         * The primary text to be inserted is treated as a plain string.
         */
        InsertTextFormat.PlainText = 1;
        /**
         * The primary text to be inserted is treated as a snippet.
         *
         * A snippet can define tab stops and placeholders with `$1`, `$2`
         * and `${3:foo}`. `$0` defines the final tab stop, it defaults to
         * the end of the snippet. Placeholders with equal identifiers are linked,
         * that is typing in one will update others too.
         *
         * See also: https://microsoft.github.io/language-server-protocol/specifications/specification-current/#snippet_syntax
         */
        InsertTextFormat.Snippet = 2;
    })(InsertTextFormat = exports.InsertTextFormat || (exports.InsertTextFormat = {}));
    /**
     * Completion item tags are extra annotations that tweak the rendering of a completion
     * item.
     *
     * @since 3.15.0
     */
    var CompletionItemTag;
    (function (CompletionItemTag) {
        /**
         * Render a completion as obsolete, usually using a strike-out.
         */
        CompletionItemTag.Deprecated = 1;
    })(CompletionItemTag = exports.CompletionItemTag || (exports.CompletionItemTag = {}));
    /**
     * The InsertReplaceEdit namespace provides functions to deal with insert / replace edits.
     *
     * @since 3.16.0
     */
    var InsertReplaceEdit;
    (function (InsertReplaceEdit) {
        /**
         * Creates a new insert / replace edit
         */
        function create(newText, insert, replace) {
            return { newText: newText, insert: insert, replace: replace };
        }
        InsertReplaceEdit.create = create;
        /**
         * Checks whether the given literal conforms to the [InsertReplaceEdit](#InsertReplaceEdit) interface.
         */
        function is(value) {
            var candidate = value;
            return candidate && Is.string(candidate.newText) && Range.is(candidate.insert) && Range.is(candidate.replace);
        }
        InsertReplaceEdit.is = is;
    })(InsertReplaceEdit = exports.InsertReplaceEdit || (exports.InsertReplaceEdit = {}));
    /**
     * How whitespace and indentation is handled during completion
     * item insertion.
     *
     * @since 3.16.0
     */
    var InsertTextMode;
    (function (InsertTextMode) {
        /**
         * The insertion or replace strings is taken as it is. If the
         * value is multi line the lines below the cursor will be
         * inserted using the indentation defined in the string value.
         * The client will not apply any kind of adjustments to the
         * string.
         */
        InsertTextMode.asIs = 1;
        /**
         * The editor adjusts leading whitespace of new lines so that
         * they match the indentation up to the cursor of the line for
         * which the item is accepted.
         *
         * Consider a line like this: <2tabs><cursor><3tabs>foo. Accepting a
         * multi line completion item is indented using 2 tabs and all
         * following lines inserted will be indented using 2 tabs as well.
         */
        InsertTextMode.adjustIndentation = 2;
    })(InsertTextMode = exports.InsertTextMode || (exports.InsertTextMode = {}));
    /**
     * The CompletionItem namespace provides functions to deal with
     * completion items.
     */
    var CompletionItem;
    (function (CompletionItem) {
        /**
         * Create a completion item and seed it with a label.
         * @param label The completion item's label
         */
        function create(label) {
            return { label: label };
        }
        CompletionItem.create = create;
    })(CompletionItem = exports.CompletionItem || (exports.CompletionItem = {}));
    /**
     * The CompletionList namespace provides functions to deal with
     * completion lists.
     */
    var CompletionList;
    (function (CompletionList) {
        /**
         * Creates a new completion list.
         *
         * @param items The completion items.
         * @param isIncomplete The list is not complete.
         */
        function create(items, isIncomplete) {
            return { items: items ? items : [], isIncomplete: !!isIncomplete };
        }
        CompletionList.create = create;
    })(CompletionList = exports.CompletionList || (exports.CompletionList = {}));
    var MarkedString;
    (function (MarkedString) {
        /**
         * Creates a marked string from plain text.
         *
         * @param plainText The plain text.
         */
        function fromPlainText(plainText) {
            return plainText.replace(/[\\`*_{}[\]()#+\-.!]/g, '\\$&'); // escape markdown syntax tokens: http://daringfireball.net/projects/markdown/syntax#backslash
        }
        MarkedString.fromPlainText = fromPlainText;
        /**
         * Checks whether the given value conforms to the [MarkedString](#MarkedString) type.
         */
        function is(value) {
            var candidate = value;
            return Is.string(candidate) || (Is.objectLiteral(candidate) && Is.string(candidate.language) && Is.string(candidate.value));
        }
        MarkedString.is = is;
    })(MarkedString = exports.MarkedString || (exports.MarkedString = {}));
    var Hover;
    (function (Hover) {
        /**
         * Checks whether the given value conforms to the [Hover](#Hover) interface.
         */
        function is(value) {
            var candidate = value;
            return !!candidate && Is.objectLiteral(candidate) && (MarkupContent.is(candidate.contents) ||
                MarkedString.is(candidate.contents) ||
                Is.typedArray(candidate.contents, MarkedString.is)) && (value.range === undefined || Range.is(value.range));
        }
        Hover.is = is;
    })(Hover = exports.Hover || (exports.Hover = {}));
    /**
     * The ParameterInformation namespace provides helper functions to work with
     * [ParameterInformation](#ParameterInformation) literals.
     */
    var ParameterInformation;
    (function (ParameterInformation) {
        /**
         * Creates a new parameter information literal.
         *
         * @param label A label string.
         * @param documentation A doc string.
         */
        function create(label, documentation) {
            return documentation ? { label: label, documentation: documentation } : { label: label };
        }
        ParameterInformation.create = create;
    })(ParameterInformation = exports.ParameterInformation || (exports.ParameterInformation = {}));
    /**
     * The SignatureInformation namespace provides helper functions to work with
     * [SignatureInformation](#SignatureInformation) literals.
     */
    var SignatureInformation;
    (function (SignatureInformation) {
        function create(label, documentation) {
            var parameters = [];
            for (var _i = 2; _i < arguments.length; _i++) {
                parameters[_i - 2] = arguments[_i];
            }
            var result = { label: label };
            if (Is.defined(documentation)) {
                result.documentation = documentation;
            }
            if (Is.defined(parameters)) {
                result.parameters = parameters;
            }
            else {
                result.parameters = [];
            }
            return result;
        }
        SignatureInformation.create = create;
    })(SignatureInformation = exports.SignatureInformation || (exports.SignatureInformation = {}));
    /**
     * A document highlight kind.
     */
    var DocumentHighlightKind;
    (function (DocumentHighlightKind) {
        /**
         * A textual occurrence.
         */
        DocumentHighlightKind.Text = 1;
        /**
         * Read-access of a symbol, like reading a variable.
         */
        DocumentHighlightKind.Read = 2;
        /**
         * Write-access of a symbol, like writing to a variable.
         */
        DocumentHighlightKind.Write = 3;
    })(DocumentHighlightKind = exports.DocumentHighlightKind || (exports.DocumentHighlightKind = {}));
    /**
     * DocumentHighlight namespace to provide helper functions to work with
     * [DocumentHighlight](#DocumentHighlight) literals.
     */
    var DocumentHighlight;
    (function (DocumentHighlight) {
        /**
         * Create a DocumentHighlight object.
         * @param range The range the highlight applies to.
         */
        function create(range, kind) {
            var result = { range: range };
            if (Is.number(kind)) {
                result.kind = kind;
            }
            return result;
        }
        DocumentHighlight.create = create;
    })(DocumentHighlight = exports.DocumentHighlight || (exports.DocumentHighlight = {}));
    /**
     * A symbol kind.
     */
    var SymbolKind;
    (function (SymbolKind) {
        SymbolKind.File = 1;
        SymbolKind.Module = 2;
        SymbolKind.Namespace = 3;
        SymbolKind.Package = 4;
        SymbolKind.Class = 5;
        SymbolKind.Method = 6;
        SymbolKind.Property = 7;
        SymbolKind.Field = 8;
        SymbolKind.Constructor = 9;
        SymbolKind.Enum = 10;
        SymbolKind.Interface = 11;
        SymbolKind.Function = 12;
        SymbolKind.Variable = 13;
        SymbolKind.Constant = 14;
        SymbolKind.String = 15;
        SymbolKind.Number = 16;
        SymbolKind.Boolean = 17;
        SymbolKind.Array = 18;
        SymbolKind.Object = 19;
        SymbolKind.Key = 20;
        SymbolKind.Null = 21;
        SymbolKind.EnumMember = 22;
        SymbolKind.Struct = 23;
        SymbolKind.Event = 24;
        SymbolKind.Operator = 25;
        SymbolKind.TypeParameter = 26;
    })(SymbolKind = exports.SymbolKind || (exports.SymbolKind = {}));
    /**
     * Symbol tags are extra annotations that tweak the rendering of a symbol.
     * @since 3.16
     */
    var SymbolTag;
    (function (SymbolTag) {
        /**
         * Render a symbol as obsolete, usually using a strike-out.
         */
        SymbolTag.Deprecated = 1;
    })(SymbolTag = exports.SymbolTag || (exports.SymbolTag = {}));
    var SymbolInformation;
    (function (SymbolInformation) {
        /**
         * Creates a new symbol information literal.
         *
         * @param name The name of the symbol.
         * @param kind The kind of the symbol.
         * @param range The range of the location of the symbol.
         * @param uri The resource of the location of symbol, defaults to the current document.
         * @param containerName The name of the symbol containing the symbol.
         */
        function create(name, kind, range, uri, containerName) {
            var result = {
                name: name,
                kind: kind,
                location: { uri: uri, range: range }
            };
            if (containerName) {
                result.containerName = containerName;
            }
            return result;
        }
        SymbolInformation.create = create;
    })(SymbolInformation = exports.SymbolInformation || (exports.SymbolInformation = {}));
    var DocumentSymbol;
    (function (DocumentSymbol) {
        /**
         * Creates a new symbol information literal.
         *
         * @param name The name of the symbol.
         * @param detail The detail of the symbol.
         * @param kind The kind of the symbol.
         * @param range The range of the symbol.
         * @param selectionRange The selectionRange of the symbol.
         * @param children Children of the symbol.
         */
        function create(name, detail, kind, range, selectionRange, children) {
            var result = {
                name: name,
                detail: detail,
                kind: kind,
                range: range,
                selectionRange: selectionRange
            };
            if (children !== undefined) {
                result.children = children;
            }
            return result;
        }
        DocumentSymbol.create = create;
        /**
         * Checks whether the given literal conforms to the [DocumentSymbol](#DocumentSymbol) interface.
         */
        function is(value) {
            var candidate = value;
            return candidate &&
                Is.string(candidate.name) && Is.number(candidate.kind) &&
                Range.is(candidate.range) && Range.is(candidate.selectionRange) &&
                (candidate.detail === undefined || Is.string(candidate.detail)) &&
                (candidate.deprecated === undefined || Is.boolean(candidate.deprecated)) &&
                (candidate.children === undefined || Array.isArray(candidate.children)) &&
                (candidate.tags === undefined || Array.isArray(candidate.tags));
        }
        DocumentSymbol.is = is;
    })(DocumentSymbol = exports.DocumentSymbol || (exports.DocumentSymbol = {}));
    /**
     * A set of predefined code action kinds
     */
    var CodeActionKind;
    (function (CodeActionKind) {
        /**
         * Empty kind.
         */
        CodeActionKind.Empty = '';
        /**
         * Base kind for quickfix actions: 'quickfix'
         */
        CodeActionKind.QuickFix = 'quickfix';
        /**
         * Base kind for refactoring actions: 'refactor'
         */
        CodeActionKind.Refactor = 'refactor';
        /**
         * Base kind for refactoring extraction actions: 'refactor.extract'
         *
         * Example extract actions:
         *
         * - Extract method
         * - Extract function
         * - Extract variable
         * - Extract interface from class
         * - ...
         */
        CodeActionKind.RefactorExtract = 'refactor.extract';
        /**
         * Base kind for refactoring inline actions: 'refactor.inline'
         *
         * Example inline actions:
         *
         * - Inline function
         * - Inline variable
         * - Inline constant
         * - ...
         */
        CodeActionKind.RefactorInline = 'refactor.inline';
        /**
         * Base kind for refactoring rewrite actions: 'refactor.rewrite'
         *
         * Example rewrite actions:
         *
         * - Convert JavaScript function to class
         * - Add or remove parameter
         * - Encapsulate field
         * - Make method static
         * - Move method to base class
         * - ...
         */
        CodeActionKind.RefactorRewrite = 'refactor.rewrite';
        /**
         * Base kind for source actions: `source`
         *
         * Source code actions apply to the entire file.
         */
        CodeActionKind.Source = 'source';
        /**
         * Base kind for an organize imports source action: `source.organizeImports`
         */
        CodeActionKind.SourceOrganizeImports = 'source.organizeImports';
        /**
         * Base kind for auto-fix source actions: `source.fixAll`.
         *
         * Fix all actions automatically fix errors that have a clear fix that do not require user input.
         * They should not suppress errors or perform unsafe fixes such as generating new types or classes.
         *
         * @since 3.15.0
         */
        CodeActionKind.SourceFixAll = 'source.fixAll';
    })(CodeActionKind = exports.CodeActionKind || (exports.CodeActionKind = {}));
    /**
     * The CodeActionContext namespace provides helper functions to work with
     * [CodeActionContext](#CodeActionContext) literals.
     */
    var CodeActionContext;
    (function (CodeActionContext) {
        /**
         * Creates a new CodeActionContext literal.
         */
        function create(diagnostics, only) {
            var result = { diagnostics: diagnostics };
            if (only !== undefined && only !== null) {
                result.only = only;
            }
            return result;
        }
        CodeActionContext.create = create;
        /**
         * Checks whether the given literal conforms to the [CodeActionContext](#CodeActionContext) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.defined(candidate) && Is.typedArray(candidate.diagnostics, Diagnostic.is) && (candidate.only === undefined || Is.typedArray(candidate.only, Is.string));
        }
        CodeActionContext.is = is;
    })(CodeActionContext = exports.CodeActionContext || (exports.CodeActionContext = {}));
    var CodeAction;
    (function (CodeAction) {
        function create(title, kindOrCommandOrEdit, kind) {
            var result = { title: title };
            var checkKind = true;
            if (typeof kindOrCommandOrEdit === 'string') {
                checkKind = false;
                result.kind = kindOrCommandOrEdit;
            }
            else if (Command.is(kindOrCommandOrEdit)) {
                result.command = kindOrCommandOrEdit;
            }
            else {
                result.edit = kindOrCommandOrEdit;
            }
            if (checkKind && kind !== undefined) {
                result.kind = kind;
            }
            return result;
        }
        CodeAction.create = create;
        function is(value) {
            var candidate = value;
            return candidate && Is.string(candidate.title) &&
                (candidate.diagnostics === undefined || Is.typedArray(candidate.diagnostics, Diagnostic.is)) &&
                (candidate.kind === undefined || Is.string(candidate.kind)) &&
                (candidate.edit !== undefined || candidate.command !== undefined) &&
                (candidate.command === undefined || Command.is(candidate.command)) &&
                (candidate.isPreferred === undefined || Is.boolean(candidate.isPreferred)) &&
                (candidate.edit === undefined || WorkspaceEdit.is(candidate.edit));
        }
        CodeAction.is = is;
    })(CodeAction = exports.CodeAction || (exports.CodeAction = {}));
    /**
     * The CodeLens namespace provides helper functions to work with
     * [CodeLens](#CodeLens) literals.
     */
    var CodeLens;
    (function (CodeLens) {
        /**
         * Creates a new CodeLens literal.
         */
        function create(range, data) {
            var result = { range: range };
            if (Is.defined(data)) {
                result.data = data;
            }
            return result;
        }
        CodeLens.create = create;
        /**
         * Checks whether the given literal conforms to the [CodeLens](#CodeLens) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.defined(candidate) && Range.is(candidate.range) && (Is.undefined(candidate.command) || Command.is(candidate.command));
        }
        CodeLens.is = is;
    })(CodeLens = exports.CodeLens || (exports.CodeLens = {}));
    /**
     * The FormattingOptions namespace provides helper functions to work with
     * [FormattingOptions](#FormattingOptions) literals.
     */
    var FormattingOptions;
    (function (FormattingOptions) {
        /**
         * Creates a new FormattingOptions literal.
         */
        function create(tabSize, insertSpaces) {
            return { tabSize: tabSize, insertSpaces: insertSpaces };
        }
        FormattingOptions.create = create;
        /**
         * Checks whether the given literal conforms to the [FormattingOptions](#FormattingOptions) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.defined(candidate) && Is.uinteger(candidate.tabSize) && Is.boolean(candidate.insertSpaces);
        }
        FormattingOptions.is = is;
    })(FormattingOptions = exports.FormattingOptions || (exports.FormattingOptions = {}));
    /**
     * The DocumentLink namespace provides helper functions to work with
     * [DocumentLink](#DocumentLink) literals.
     */
    var DocumentLink;
    (function (DocumentLink) {
        /**
         * Creates a new DocumentLink literal.
         */
        function create(range, target, data) {
            return { range: range, target: target, data: data };
        }
        DocumentLink.create = create;
        /**
         * Checks whether the given literal conforms to the [DocumentLink](#DocumentLink) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.defined(candidate) && Range.is(candidate.range) && (Is.undefined(candidate.target) || Is.string(candidate.target));
        }
        DocumentLink.is = is;
    })(DocumentLink = exports.DocumentLink || (exports.DocumentLink = {}));
    /**
     * The SelectionRange namespace provides helper function to work with
     * SelectionRange literals.
     */
    var SelectionRange;
    (function (SelectionRange) {
        /**
         * Creates a new SelectionRange
         * @param range the range.
         * @param parent an optional parent.
         */
        function create(range, parent) {
            return { range: range, parent: parent };
        }
        SelectionRange.create = create;
        function is(value) {
            var candidate = value;
            return candidate !== undefined && Range.is(candidate.range) && (candidate.parent === undefined || SelectionRange.is(candidate.parent));
        }
        SelectionRange.is = is;
    })(SelectionRange = exports.SelectionRange || (exports.SelectionRange = {}));
    exports.EOL = ['\n', '\r\n', '\r'];
    /**
     * @deprecated Use the text document from the new vscode-languageserver-textdocument package.
     */
    var TextDocument;
    (function (TextDocument) {
        /**
         * Creates a new ITextDocument literal from the given uri and content.
         * @param uri The document's uri.
         * @param languageId  The document's language Id.
         * @param content The document's content.
         */
        function create(uri, languageId, version, content) {
            return new FullTextDocument(uri, languageId, version, content);
        }
        TextDocument.create = create;
        /**
         * Checks whether the given literal conforms to the [ITextDocument](#ITextDocument) interface.
         */
        function is(value) {
            var candidate = value;
            return Is.defined(candidate) && Is.string(candidate.uri) && (Is.undefined(candidate.languageId) || Is.string(candidate.languageId)) && Is.uinteger(candidate.lineCount)
                && Is.func(candidate.getText) && Is.func(candidate.positionAt) && Is.func(candidate.offsetAt) ? true : false;
        }
        TextDocument.is = is;
        function applyEdits(document, edits) {
            var text = document.getText();
            var sortedEdits = mergeSort(edits, function (a, b) {
                var diff = a.range.start.line - b.range.start.line;
                if (diff === 0) {
                    return a.range.start.character - b.range.start.character;
                }
                return diff;
            });
            var lastModifiedOffset = text.length;
            for (var i = sortedEdits.length - 1; i >= 0; i--) {
                var e = sortedEdits[i];
                var startOffset = document.offsetAt(e.range.start);
                var endOffset = document.offsetAt(e.range.end);
                if (endOffset <= lastModifiedOffset) {
                    text = text.substring(0, startOffset) + e.newText + text.substring(endOffset, text.length);
                }
                else {
                    throw new Error('Overlapping edit');
                }
                lastModifiedOffset = startOffset;
            }
            return text;
        }
        TextDocument.applyEdits = applyEdits;
        function mergeSort(data, compare) {
            if (data.length <= 1) {
                // sorted
                return data;
            }
            var p = (data.length / 2) | 0;
            var left = data.slice(0, p);
            var right = data.slice(p);
            mergeSort(left, compare);
            mergeSort(right, compare);
            var leftIdx = 0;
            var rightIdx = 0;
            var i = 0;
            while (leftIdx < left.length && rightIdx < right.length) {
                var ret = compare(left[leftIdx], right[rightIdx]);
                if (ret <= 0) {
                    // smaller_equal -> take left to preserve order
                    data[i++] = left[leftIdx++];
                }
                else {
                    // greater -> take right
                    data[i++] = right[rightIdx++];
                }
            }
            while (leftIdx < left.length) {
                data[i++] = left[leftIdx++];
            }
            while (rightIdx < right.length) {
                data[i++] = right[rightIdx++];
            }
            return data;
        }
    })(TextDocument = exports.TextDocument || (exports.TextDocument = {}));
    /**
     * @deprecated Use the text document from the new vscode-languageserver-textdocument package.
     */
    var FullTextDocument = /** @class */ (function () {
        function FullTextDocument(uri, languageId, version, content) {
            this._uri = uri;
            this._languageId = languageId;
            this._version = version;
            this._content = content;
            this._lineOffsets = undefined;
        }
        Object.defineProperty(FullTextDocument.prototype, "uri", {
            get: function () {
                return this._uri;
            },
            enumerable: false,
            configurable: true
        });
        Object.defineProperty(FullTextDocument.prototype, "languageId", {
            get: function () {
                return this._languageId;
            },
            enumerable: false,
            configurable: true
        });
        Object.defineProperty(FullTextDocument.prototype, "version", {
            get: function () {
                return this._version;
            },
            enumerable: false,
            configurable: true
        });
        FullTextDocument.prototype.getText = function (range) {
            if (range) {
                var start = this.offsetAt(range.start);
                var end = this.offsetAt(range.end);
                return this._content.substring(start, end);
            }
            return this._content;
        };
        FullTextDocument.prototype.update = function (event, version) {
            this._content = event.text;
            this._version = version;
            this._lineOffsets = undefined;
        };
        FullTextDocument.prototype.getLineOffsets = function () {
            if (this._lineOffsets === undefined) {
                var lineOffsets = [];
                var text = this._content;
                var isLineStart = true;
                for (var i = 0; i < text.length; i++) {
                    if (isLineStart) {
                        lineOffsets.push(i);
                        isLineStart = false;
                    }
                    var ch = text.charAt(i);
                    isLineStart = (ch === '\r' || ch === '\n');
                    if (ch === '\r' && i + 1 < text.length && text.charAt(i + 1) === '\n') {
                        i++;
                    }
                }
                if (isLineStart && text.length > 0) {
                    lineOffsets.push(text.length);
                }
                this._lineOffsets = lineOffsets;
            }
            return this._lineOffsets;
        };
        FullTextDocument.prototype.positionAt = function (offset) {
            offset = Math.max(Math.min(offset, this._content.length), 0);
            var lineOffsets = this.getLineOffsets();
            var low = 0, high = lineOffsets.length;
            if (high === 0) {
                return Position.create(0, offset);
            }
            while (low < high) {
                var mid = Math.floor((low + high) / 2);
                if (lineOffsets[mid] > offset) {
                    high = mid;
                }
                else {
                    low = mid + 1;
                }
            }
            // low is the least x for which the line offset is larger than the current offset
            // or array.length if no line offset is larger than the current offset
            var line = low - 1;
            return Position.create(line, offset - lineOffsets[line]);
        };
        FullTextDocument.prototype.offsetAt = function (position) {
            var lineOffsets = this.getLineOffsets();
            if (position.line >= lineOffsets.length) {
                return this._content.length;
            }
            else if (position.line < 0) {
                return 0;
            }
            var lineOffset = lineOffsets[position.line];
            var nextLineOffset = (position.line + 1 < lineOffsets.length) ? lineOffsets[position.line + 1] : this._content.length;
            return Math.max(Math.min(lineOffset + position.character, nextLineOffset), lineOffset);
        };
        Object.defineProperty(FullTextDocument.prototype, "lineCount", {
            get: function () {
                return this.getLineOffsets().length;
            },
            enumerable: false,
            configurable: true
        });
        return FullTextDocument;
    }());
    var Is;
    (function (Is) {
        var toString = Object.prototype.toString;
        function defined(value) {
            return typeof value !== 'undefined';
        }
        Is.defined = defined;
        function undefined(value) {
            return typeof value === 'undefined';
        }
        Is.undefined = undefined;
        function boolean(value) {
            return value === true || value === false;
        }
        Is.boolean = boolean;
        function string(value) {
            return toString.call(value) === '[object String]';
        }
        Is.string = string;
        function number(value) {
            return toString.call(value) === '[object Number]';
        }
        Is.number = number;
        function numberRange(value, min, max) {
            return toString.call(value) === '[object Number]' && min <= value && value <= max;
        }
        Is.numberRange = numberRange;
        function integer(value) {
            return toString.call(value) === '[object Number]' && -2147483648 <= value && value <= 2147483647;
        }
        Is.integer = integer;
        function uinteger(value) {
            return toString.call(value) === '[object Number]' && 0 <= value && value <= 2147483647;
        }
        Is.uinteger = uinteger;
        function func(value) {
            return toString.call(value) === '[object Function]';
        }
        Is.func = func;
        function objectLiteral(value) {
            // Strictly speaking class instances pass this check as well. Since the LSP
            // doesn't use classes we ignore this for now. If we do we need to add something
            // like this: `Object.getPrototypeOf(Object.getPrototypeOf(x)) === null`
            return value !== null && typeof value === 'object';
        }
        Is.objectLiteral = objectLiteral;
        function typedArray(value, check) {
            return Array.isArray(value) && value.every(check);
        }
        Is.typedArray = typedArray;
    })(Is || (Is = {}));
});
//# sourceMappingURL=main.js.map;
define('vscode-languageserver-types', ['vscode-languageserver-types/main'], function (main) { return main; });

(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-languageserver-textdocument/main',["require", "exports"], factory);
    }
})(function (require, exports) {
    /* --------------------------------------------------------------------------------------------
     * Copyright (c) Microsoft Corporation. All rights reserved.
     * Licensed under the MIT License. See License.txt in the project root for license information.
     * ------------------------------------------------------------------------------------------ */
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    var FullTextDocument = /** @class */ (function () {
        function FullTextDocument(uri, languageId, version, content) {
            this._uri = uri;
            this._languageId = languageId;
            this._version = version;
            this._content = content;
            this._lineOffsets = undefined;
        }
        Object.defineProperty(FullTextDocument.prototype, "uri", {
            get: function () {
                return this._uri;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(FullTextDocument.prototype, "languageId", {
            get: function () {
                return this._languageId;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(FullTextDocument.prototype, "version", {
            get: function () {
                return this._version;
            },
            enumerable: true,
            configurable: true
        });
        FullTextDocument.prototype.getText = function (range) {
            if (range) {
                var start = this.offsetAt(range.start);
                var end = this.offsetAt(range.end);
                return this._content.substring(start, end);
            }
            return this._content;
        };
        FullTextDocument.prototype.update = function (changes, version) {
            for (var _i = 0, changes_1 = changes; _i < changes_1.length; _i++) {
                var change = changes_1[_i];
                if (FullTextDocument.isIncremental(change)) {
                    // makes sure start is before end
                    var range = getWellformedRange(change.range);
                    // update content
                    var startOffset = this.offsetAt(range.start);
                    var endOffset = this.offsetAt(range.end);
                    this._content = this._content.substring(0, startOffset) + change.text + this._content.substring(endOffset, this._content.length);
                    // update the offsets
                    var startLine = Math.max(range.start.line, 0);
                    var endLine = Math.max(range.end.line, 0);
                    var lineOffsets = this._lineOffsets;
                    var addedLineOffsets = computeLineOffsets(change.text, false, startOffset);
                    if (endLine - startLine === addedLineOffsets.length) {
                        for (var i = 0, len = addedLineOffsets.length; i < len; i++) {
                            lineOffsets[i + startLine + 1] = addedLineOffsets[i];
                        }
                    }
                    else {
                        if (addedLineOffsets.length < 10000) {
                            lineOffsets.splice.apply(lineOffsets, [startLine + 1, endLine - startLine].concat(addedLineOffsets));
                        }
                        else { // avoid too many arguments for splice
                            this._lineOffsets = lineOffsets = lineOffsets.slice(0, startLine + 1).concat(addedLineOffsets, lineOffsets.slice(endLine + 1));
                        }
                    }
                    var diff = change.text.length - (endOffset - startOffset);
                    if (diff !== 0) {
                        for (var i = startLine + 1 + addedLineOffsets.length, len = lineOffsets.length; i < len; i++) {
                            lineOffsets[i] = lineOffsets[i] + diff;
                        }
                    }
                }
                else if (FullTextDocument.isFull(change)) {
                    this._content = change.text;
                    this._lineOffsets = undefined;
                }
                else {
                    throw new Error('Unknown change event received');
                }
            }
            this._version = version;
        };
        FullTextDocument.prototype.getLineOffsets = function () {
            if (this._lineOffsets === undefined) {
                this._lineOffsets = computeLineOffsets(this._content, true);
            }
            return this._lineOffsets;
        };
        FullTextDocument.prototype.positionAt = function (offset) {
            offset = Math.max(Math.min(offset, this._content.length), 0);
            var lineOffsets = this.getLineOffsets();
            var low = 0, high = lineOffsets.length;
            if (high === 0) {
                return { line: 0, character: offset };
            }
            while (low < high) {
                var mid = Math.floor((low + high) / 2);
                if (lineOffsets[mid] > offset) {
                    high = mid;
                }
                else {
                    low = mid + 1;
                }
            }
            // low is the least x for which the line offset is larger than the current offset
            // or array.length if no line offset is larger than the current offset
            var line = low - 1;
            return { line: line, character: offset - lineOffsets[line] };
        };
        FullTextDocument.prototype.offsetAt = function (position) {
            var lineOffsets = this.getLineOffsets();
            if (position.line >= lineOffsets.length) {
                return this._content.length;
            }
            else if (position.line < 0) {
                return 0;
            }
            var lineOffset = lineOffsets[position.line];
            var nextLineOffset = (position.line + 1 < lineOffsets.length) ? lineOffsets[position.line + 1] : this._content.length;
            return Math.max(Math.min(lineOffset + position.character, nextLineOffset), lineOffset);
        };
        Object.defineProperty(FullTextDocument.prototype, "lineCount", {
            get: function () {
                return this.getLineOffsets().length;
            },
            enumerable: true,
            configurable: true
        });
        FullTextDocument.isIncremental = function (event) {
            var candidate = event;
            return candidate !== undefined && candidate !== null &&
                typeof candidate.text === 'string' && candidate.range !== undefined &&
                (candidate.rangeLength === undefined || typeof candidate.rangeLength === 'number');
        };
        FullTextDocument.isFull = function (event) {
            var candidate = event;
            return candidate !== undefined && candidate !== null &&
                typeof candidate.text === 'string' && candidate.range === undefined && candidate.rangeLength === undefined;
        };
        return FullTextDocument;
    }());
    var TextDocument;
    (function (TextDocument) {
        /**
         * Creates a new text document.
         *
         * @param uri The document's uri.
         * @param languageId  The document's language Id.
         * @param version The document's initial version number.
         * @param content The document's content.
         */
        function create(uri, languageId, version, content) {
            return new FullTextDocument(uri, languageId, version, content);
        }
        TextDocument.create = create;
        /**
         * Updates a TextDocument by modifing its content.
         *
         * @param document the document to update. Only documents created by TextDocument.create are valid inputs.
         * @param changes the changes to apply to the document.
         * @returns The updated TextDocument. Note: That's the same document instance passed in as first parameter.
         *
         */
        function update(document, changes, version) {
            if (document instanceof FullTextDocument) {
                document.update(changes, version);
                return document;
            }
            else {
                throw new Error('TextDocument.update: document must be created by TextDocument.create');
            }
        }
        TextDocument.update = update;
        function applyEdits(document, edits) {
            var text = document.getText();
            var sortedEdits = mergeSort(edits.map(getWellformedEdit), function (a, b) {
                var diff = a.range.start.line - b.range.start.line;
                if (diff === 0) {
                    return a.range.start.character - b.range.start.character;
                }
                return diff;
            });
            var lastModifiedOffset = 0;
            var spans = [];
            for (var _i = 0, sortedEdits_1 = sortedEdits; _i < sortedEdits_1.length; _i++) {
                var e = sortedEdits_1[_i];
                var startOffset = document.offsetAt(e.range.start);
                if (startOffset < lastModifiedOffset) {
                    throw new Error('Overlapping edit');
                }
                else if (startOffset > lastModifiedOffset) {
                    spans.push(text.substring(lastModifiedOffset, startOffset));
                }
                if (e.newText.length) {
                    spans.push(e.newText);
                }
                lastModifiedOffset = document.offsetAt(e.range.end);
            }
            spans.push(text.substr(lastModifiedOffset));
            return spans.join('');
        }
        TextDocument.applyEdits = applyEdits;
    })(TextDocument = exports.TextDocument || (exports.TextDocument = {}));
    function mergeSort(data, compare) {
        if (data.length <= 1) {
            // sorted
            return data;
        }
        var p = (data.length / 2) | 0;
        var left = data.slice(0, p);
        var right = data.slice(p);
        mergeSort(left, compare);
        mergeSort(right, compare);
        var leftIdx = 0;
        var rightIdx = 0;
        var i = 0;
        while (leftIdx < left.length && rightIdx < right.length) {
            var ret = compare(left[leftIdx], right[rightIdx]);
            if (ret <= 0) {
                // smaller_equal -> take left to preserve order
                data[i++] = left[leftIdx++];
            }
            else {
                // greater -> take right
                data[i++] = right[rightIdx++];
            }
        }
        while (leftIdx < left.length) {
            data[i++] = left[leftIdx++];
        }
        while (rightIdx < right.length) {
            data[i++] = right[rightIdx++];
        }
        return data;
    }
    function computeLineOffsets(text, isAtLineStart, textOffset) {
        if (textOffset === void 0) { textOffset = 0; }
        var result = isAtLineStart ? [textOffset] : [];
        for (var i = 0; i < text.length; i++) {
            var ch = text.charCodeAt(i);
            if (ch === 13 /* CarriageReturn */ || ch === 10 /* LineFeed */) {
                if (ch === 13 /* CarriageReturn */ && i + 1 < text.length && text.charCodeAt(i + 1) === 10 /* LineFeed */) {
                    i++;
                }
                result.push(textOffset + i + 1);
            }
        }
        return result;
    }
    function getWellformedRange(range) {
        var start = range.start;
        var end = range.end;
        if (start.line > end.line || (start.line === end.line && start.character > end.character)) {
            return { start: end, end: start };
        }
        return range;
    }
    function getWellformedEdit(textEdit) {
        var range = getWellformedRange(textEdit.range);
        if (range !== textEdit.range) {
            return { newText: textEdit.newText, range: range };
        }
        return textEdit;
    }
});

define('vscode-languageserver-textdocument', ['vscode-languageserver-textdocument/main'], function (main) { return main; });

var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __exportStar = (this && this.__exportStar) || function(m, exports) {
    for (var p in m) if (p !== "default" && !Object.prototype.hasOwnProperty.call(exports, p)) __createBinding(exports, m, p);
};
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/cssLanguageTypes',["require", "exports", "vscode-languageserver-types", "vscode-languageserver-textdocument", "vscode-languageserver-types"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.FileType = exports.ClientCapabilities = exports.TextDocument = void 0;
    var vscode_languageserver_types_1 = require("vscode-languageserver-types");
    var vscode_languageserver_textdocument_1 = require("vscode-languageserver-textdocument");
    Object.defineProperty(exports, "TextDocument", { enumerable: true, get: function () { return vscode_languageserver_textdocument_1.TextDocument; } });
    __exportStar(require("vscode-languageserver-types"), exports);
    var ClientCapabilities;
    (function (ClientCapabilities) {
        ClientCapabilities.LATEST = {
            textDocument: {
                completion: {
                    completionItem: {
                        documentationFormat: [vscode_languageserver_types_1.MarkupKind.Markdown, vscode_languageserver_types_1.MarkupKind.PlainText]
                    }
                },
                hover: {
                    contentFormat: [vscode_languageserver_types_1.MarkupKind.Markdown, vscode_languageserver_types_1.MarkupKind.PlainText]
                }
            }
        };
    })(ClientCapabilities = exports.ClientCapabilities || (exports.ClientCapabilities = {}));
    var FileType;
    (function (FileType) {
        /**
         * The file type is unknown.
         */
        FileType[FileType["Unknown"] = 0] = "Unknown";
        /**
         * A regular file.
         */
        FileType[FileType["File"] = 1] = "File";
        /**
         * A directory.
         */
        FileType[FileType["Directory"] = 2] = "Directory";
        /**
         * A symbolic link to a file.
         */
        FileType[FileType["SymbolicLink"] = 64] = "SymbolicLink";
    })(FileType = exports.FileType || (exports.FileType = {}));
});

!function(t,e){if("object"==typeof exports&&"object"==typeof module)module.exports=e();else if("function"==typeof define&&define.amd)define('vscode-uri/index',[],e);else{var r=e();for(var n in r)("object"==typeof exports?exports:t)[n]=r[n]}}(this,(function(){return(()=>{"use strict";var t={470:t=>{function e(t){if("string"!=typeof t)throw new TypeError("Path must be a string. Received "+JSON.stringify(t))}function r(t,e){for(var r,n="",i=0,o=-1,a=0,h=0;h<=t.length;++h){if(h<t.length)r=t.charCodeAt(h);else{if(47===r)break;r=47}if(47===r){if(o===h-1||1===a);else if(o!==h-1&&2===a){if(n.length<2||2!==i||46!==n.charCodeAt(n.length-1)||46!==n.charCodeAt(n.length-2))if(n.length>2){var s=n.lastIndexOf("/");if(s!==n.length-1){-1===s?(n="",i=0):i=(n=n.slice(0,s)).length-1-n.lastIndexOf("/"),o=h,a=0;continue}}else if(2===n.length||1===n.length){n="",i=0,o=h,a=0;continue}e&&(n.length>0?n+="/..":n="..",i=2)}else n.length>0?n+="/"+t.slice(o+1,h):n=t.slice(o+1,h),i=h-o-1;o=h,a=0}else 46===r&&-1!==a?++a:a=-1}return n}var n={resolve:function(){for(var t,n="",i=!1,o=arguments.length-1;o>=-1&&!i;o--){var a;o>=0?a=arguments[o]:(void 0===t&&(t=process.cwd()),a=t),e(a),0!==a.length&&(n=a+"/"+n,i=47===a.charCodeAt(0))}return n=r(n,!i),i?n.length>0?"/"+n:"/":n.length>0?n:"."},normalize:function(t){if(e(t),0===t.length)return".";var n=47===t.charCodeAt(0),i=47===t.charCodeAt(t.length-1);return 0!==(t=r(t,!n)).length||n||(t="."),t.length>0&&i&&(t+="/"),n?"/"+t:t},isAbsolute:function(t){return e(t),t.length>0&&47===t.charCodeAt(0)},join:function(){if(0===arguments.length)return".";for(var t,r=0;r<arguments.length;++r){var i=arguments[r];e(i),i.length>0&&(void 0===t?t=i:t+="/"+i)}return void 0===t?".":n.normalize(t)},relative:function(t,r){if(e(t),e(r),t===r)return"";if((t=n.resolve(t))===(r=n.resolve(r)))return"";for(var i=1;i<t.length&&47===t.charCodeAt(i);++i);for(var o=t.length,a=o-i,h=1;h<r.length&&47===r.charCodeAt(h);++h);for(var s=r.length-h,f=a<s?a:s,u=-1,c=0;c<=f;++c){if(c===f){if(s>f){if(47===r.charCodeAt(h+c))return r.slice(h+c+1);if(0===c)return r.slice(h+c)}else a>f&&(47===t.charCodeAt(i+c)?u=c:0===c&&(u=0));break}var l=t.charCodeAt(i+c);if(l!==r.charCodeAt(h+c))break;47===l&&(u=c)}var p="";for(c=i+u+1;c<=o;++c)c!==o&&47!==t.charCodeAt(c)||(0===p.length?p+="..":p+="/..");return p.length>0?p+r.slice(h+u):(h+=u,47===r.charCodeAt(h)&&++h,r.slice(h))},_makeLong:function(t){return t},dirname:function(t){if(e(t),0===t.length)return".";for(var r=t.charCodeAt(0),n=47===r,i=-1,o=!0,a=t.length-1;a>=1;--a)if(47===(r=t.charCodeAt(a))){if(!o){i=a;break}}else o=!1;return-1===i?n?"/":".":n&&1===i?"//":t.slice(0,i)},basename:function(t,r){if(void 0!==r&&"string"!=typeof r)throw new TypeError('"ext" argument must be a string');e(t);var n,i=0,o=-1,a=!0;if(void 0!==r&&r.length>0&&r.length<=t.length){if(r.length===t.length&&r===t)return"";var h=r.length-1,s=-1;for(n=t.length-1;n>=0;--n){var f=t.charCodeAt(n);if(47===f){if(!a){i=n+1;break}}else-1===s&&(a=!1,s=n+1),h>=0&&(f===r.charCodeAt(h)?-1==--h&&(o=n):(h=-1,o=s))}return i===o?o=s:-1===o&&(o=t.length),t.slice(i,o)}for(n=t.length-1;n>=0;--n)if(47===t.charCodeAt(n)){if(!a){i=n+1;break}}else-1===o&&(a=!1,o=n+1);return-1===o?"":t.slice(i,o)},extname:function(t){e(t);for(var r=-1,n=0,i=-1,o=!0,a=0,h=t.length-1;h>=0;--h){var s=t.charCodeAt(h);if(47!==s)-1===i&&(o=!1,i=h+1),46===s?-1===r?r=h:1!==a&&(a=1):-1!==r&&(a=-1);else if(!o){n=h+1;break}}return-1===r||-1===i||0===a||1===a&&r===i-1&&r===n+1?"":t.slice(r,i)},format:function(t){if(null===t||"object"!=typeof t)throw new TypeError('The "pathObject" argument must be of type Object. Received type '+typeof t);return function(t,e){var r=e.dir||e.root,n=e.base||(e.name||"")+(e.ext||"");return r?r===e.root?r+n:r+"/"+n:n}(0,t)},parse:function(t){e(t);var r={root:"",dir:"",base:"",ext:"",name:""};if(0===t.length)return r;var n,i=t.charCodeAt(0),o=47===i;o?(r.root="/",n=1):n=0;for(var a=-1,h=0,s=-1,f=!0,u=t.length-1,c=0;u>=n;--u)if(47!==(i=t.charCodeAt(u)))-1===s&&(f=!1,s=u+1),46===i?-1===a?a=u:1!==c&&(c=1):-1!==a&&(c=-1);else if(!f){h=u+1;break}return-1===a||-1===s||0===c||1===c&&a===s-1&&a===h+1?-1!==s&&(r.base=r.name=0===h&&o?t.slice(1,s):t.slice(h,s)):(0===h&&o?(r.name=t.slice(1,a),r.base=t.slice(1,s)):(r.name=t.slice(h,a),r.base=t.slice(h,s)),r.ext=t.slice(a,s)),h>0?r.dir=t.slice(0,h-1):o&&(r.dir="/"),r},sep:"/",delimiter:":",win32:null,posix:null};n.posix=n,t.exports=n},465:(t,e,r)=>{Object.defineProperty(e,"__esModule",{value:!0}),e.Utils=e.URI=void 0;var n=r(796);Object.defineProperty(e,"URI",{enumerable:!0,get:function(){return n.URI}});var i=r(679);Object.defineProperty(e,"Utils",{enumerable:!0,get:function(){return i.Utils}})},674:(t,e)=>{if(Object.defineProperty(e,"__esModule",{value:!0}),e.isWindows=void 0,"object"==typeof process)e.isWindows="win32"===process.platform;else if("object"==typeof navigator){var r=navigator.userAgent;e.isWindows=r.indexOf("Windows")>=0}},796:function(t,e,r){var n,i,o=this&&this.__extends||(n=function(t,e){return(n=Object.setPrototypeOf||{__proto__:[]}instanceof Array&&function(t,e){t.__proto__=e}||function(t,e){for(var r in e)Object.prototype.hasOwnProperty.call(e,r)&&(t[r]=e[r])})(t,e)},function(t,e){function r(){this.constructor=t}n(t,e),t.prototype=null===e?Object.create(e):(r.prototype=e.prototype,new r)});Object.defineProperty(e,"__esModule",{value:!0}),e.uriToFsPath=e.URI=void 0;var a=r(674),h=/^\w[\w\d+.-]*$/,s=/^\//,f=/^\/\//,u="",c="/",l=/^(([^:/?#]+?):)?(\/\/([^/?#]*))?([^?#]*)(\?([^#]*))?(#(.*))?/,p=function(){function t(t,e,r,n,i,o){void 0===o&&(o=!1),"object"==typeof t?(this.scheme=t.scheme||u,this.authority=t.authority||u,this.path=t.path||u,this.query=t.query||u,this.fragment=t.fragment||u):(this.scheme=function(t,e){return t||e?t:"file"}(t,o),this.authority=e||u,this.path=function(t,e){switch(t){case"https":case"http":case"file":e?e[0]!==c&&(e=c+e):e=c}return e}(this.scheme,r||u),this.query=n||u,this.fragment=i||u,function(t,e){if(!t.scheme&&e)throw new Error('[UriError]: Scheme is missing: {scheme: "", authority: "'+t.authority+'", path: "'+t.path+'", query: "'+t.query+'", fragment: "'+t.fragment+'"}');if(t.scheme&&!h.test(t.scheme))throw new Error("[UriError]: Scheme contains illegal characters.");if(t.path)if(t.authority){if(!s.test(t.path))throw new Error('[UriError]: If a URI contains an authority component, then the path component must either be empty or begin with a slash ("/") character')}else if(f.test(t.path))throw new Error('[UriError]: If a URI does not contain an authority component, then the path cannot begin with two slash characters ("//")')}(this,o))}return t.isUri=function(e){return e instanceof t||!!e&&"string"==typeof e.authority&&"string"==typeof e.fragment&&"string"==typeof e.path&&"string"==typeof e.query&&"string"==typeof e.scheme&&"function"==typeof e.fsPath&&"function"==typeof e.with&&"function"==typeof e.toString},Object.defineProperty(t.prototype,"fsPath",{get:function(){return b(this,!1)},enumerable:!1,configurable:!0}),t.prototype.with=function(t){if(!t)return this;var e=t.scheme,r=t.authority,n=t.path,i=t.query,o=t.fragment;return void 0===e?e=this.scheme:null===e&&(e=u),void 0===r?r=this.authority:null===r&&(r=u),void 0===n?n=this.path:null===n&&(n=u),void 0===i?i=this.query:null===i&&(i=u),void 0===o?o=this.fragment:null===o&&(o=u),e===this.scheme&&r===this.authority&&n===this.path&&i===this.query&&o===this.fragment?this:new g(e,r,n,i,o)},t.parse=function(t,e){void 0===e&&(e=!1);var r=l.exec(t);return r?new g(r[2]||u,_(r[4]||u),_(r[5]||u),_(r[7]||u),_(r[9]||u),e):new g(u,u,u,u,u)},t.file=function(t){var e=u;if(a.isWindows&&(t=t.replace(/\\/g,c)),t[0]===c&&t[1]===c){var r=t.indexOf(c,2);-1===r?(e=t.substring(2),t=c):(e=t.substring(2,r),t=t.substring(r)||c)}return new g("file",e,t,u,u)},t.from=function(t){return new g(t.scheme,t.authority,t.path,t.query,t.fragment)},t.prototype.toString=function(t){return void 0===t&&(t=!1),C(this,t)},t.prototype.toJSON=function(){return this},t.revive=function(e){if(e){if(e instanceof t)return e;var r=new g(e);return r._formatted=e.external,r._fsPath=e._sep===d?e.fsPath:null,r}return e},t}();e.URI=p;var d=a.isWindows?1:void 0,g=function(t){function e(){var e=null!==t&&t.apply(this,arguments)||this;return e._formatted=null,e._fsPath=null,e}return o(e,t),Object.defineProperty(e.prototype,"fsPath",{get:function(){return this._fsPath||(this._fsPath=b(this,!1)),this._fsPath},enumerable:!1,configurable:!0}),e.prototype.toString=function(t){return void 0===t&&(t=!1),t?C(this,!0):(this._formatted||(this._formatted=C(this,!1)),this._formatted)},e.prototype.toJSON=function(){var t={$mid:1};return this._fsPath&&(t.fsPath=this._fsPath,t._sep=d),this._formatted&&(t.external=this._formatted),this.path&&(t.path=this.path),this.scheme&&(t.scheme=this.scheme),this.authority&&(t.authority=this.authority),this.query&&(t.query=this.query),this.fragment&&(t.fragment=this.fragment),t},e}(p),v=((i={})[58]="%3A",i[47]="%2F",i[63]="%3F",i[35]="%23",i[91]="%5B",i[93]="%5D",i[64]="%40",i[33]="%21",i[36]="%24",i[38]="%26",i[39]="%27",i[40]="%28",i[41]="%29",i[42]="%2A",i[43]="%2B",i[44]="%2C",i[59]="%3B",i[61]="%3D",i[32]="%20",i);function m(t,e){for(var r=void 0,n=-1,i=0;i<t.length;i++){var o=t.charCodeAt(i);if(o>=97&&o<=122||o>=65&&o<=90||o>=48&&o<=57||45===o||46===o||95===o||126===o||e&&47===o)-1!==n&&(r+=encodeURIComponent(t.substring(n,i)),n=-1),void 0!==r&&(r+=t.charAt(i));else{void 0===r&&(r=t.substr(0,i));var a=v[o];void 0!==a?(-1!==n&&(r+=encodeURIComponent(t.substring(n,i)),n=-1),r+=a):-1===n&&(n=i)}}return-1!==n&&(r+=encodeURIComponent(t.substring(n))),void 0!==r?r:t}function y(t){for(var e=void 0,r=0;r<t.length;r++){var n=t.charCodeAt(r);35===n||63===n?(void 0===e&&(e=t.substr(0,r)),e+=v[n]):void 0!==e&&(e+=t[r])}return void 0!==e?e:t}function b(t,e){var r;return r=t.authority&&t.path.length>1&&"file"===t.scheme?"//"+t.authority+t.path:47===t.path.charCodeAt(0)&&(t.path.charCodeAt(1)>=65&&t.path.charCodeAt(1)<=90||t.path.charCodeAt(1)>=97&&t.path.charCodeAt(1)<=122)&&58===t.path.charCodeAt(2)?e?t.path.substr(1):t.path[1].toLowerCase()+t.path.substr(2):t.path,a.isWindows&&(r=r.replace(/\//g,"\\")),r}function C(t,e){var r=e?y:m,n="",i=t.scheme,o=t.authority,a=t.path,h=t.query,s=t.fragment;if(i&&(n+=i,n+=":"),(o||"file"===i)&&(n+=c,n+=c),o){var f=o.indexOf("@");if(-1!==f){var u=o.substr(0,f);o=o.substr(f+1),-1===(f=u.indexOf(":"))?n+=r(u,!1):(n+=r(u.substr(0,f),!1),n+=":",n+=r(u.substr(f+1),!1)),n+="@"}-1===(f=(o=o.toLowerCase()).indexOf(":"))?n+=r(o,!1):(n+=r(o.substr(0,f),!1),n+=o.substr(f))}if(a){if(a.length>=3&&47===a.charCodeAt(0)&&58===a.charCodeAt(2))(l=a.charCodeAt(1))>=65&&l<=90&&(a="/"+String.fromCharCode(l+32)+":"+a.substr(3));else if(a.length>=2&&58===a.charCodeAt(1)){var l;(l=a.charCodeAt(0))>=65&&l<=90&&(a=String.fromCharCode(l+32)+":"+a.substr(2))}n+=r(a,!0)}return h&&(n+="?",n+=r(h,!1)),s&&(n+="#",n+=e?s:m(s,!1)),n}function A(t){try{return decodeURIComponent(t)}catch(e){return t.length>3?t.substr(0,3)+A(t.substr(3)):t}}e.uriToFsPath=b;var w=/(%[0-9A-Za-z][0-9A-Za-z])+/g;function _(t){return t.match(w)?t.replace(w,(function(t){return A(t)})):t}},679:function(t,e,r){var n=this&&this.__spreadArrays||function(){for(var t=0,e=0,r=arguments.length;e<r;e++)t+=arguments[e].length;var n=Array(t),i=0;for(e=0;e<r;e++)for(var o=arguments[e],a=0,h=o.length;a<h;a++,i++)n[i]=o[a];return n};Object.defineProperty(e,"__esModule",{value:!0}),e.Utils=void 0;var i,o=r(470),a=o.posix||o;(i=e.Utils||(e.Utils={})).joinPath=function(t){for(var e=[],r=1;r<arguments.length;r++)e[r-1]=arguments[r];return t.with({path:a.join.apply(a,n([t.path],e))})},i.resolvePath=function(t){for(var e=[],r=1;r<arguments.length;r++)e[r-1]=arguments[r];var i=t.path||"/";return t.with({path:a.resolve.apply(a,n([i],e))})},i.dirname=function(t){var e=a.dirname(t.path);return 1===e.length&&46===e.charCodeAt(0)?t:t.with({path:e})},i.basename=function(t){return a.basename(t.path)},i.extname=function(t){return a.extname(t.path)}}},e={};return function r(n){if(e[n])return e[n].exports;var i=e[n]={exports:{}};return t[n].call(i.exports,i,i.exports,r),i.exports}(465)})()}));
//# sourceMappingURL=index.js.map;
define('vscode-uri', ['vscode-uri/index'], function (main) { return main; });

/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __spreadArrays = (this && this.__spreadArrays) || function () {
    for (var s = 0, i = 0, il = arguments.length; i < il; i++) s += arguments[i].length;
    for (var r = Array(s), k = 0, i = 0; i < il; i++)
        for (var a = arguments[i], j = 0, jl = a.length; j < jl; j++, k++)
            r[k] = a[j];
    return r;
};
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/utils/resources',["require", "exports", "vscode-uri"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.joinPath = exports.dirname = void 0;
    var vscode_uri_1 = require("vscode-uri");
    function dirname(uriString) {
        return vscode_uri_1.Utils.dirname(vscode_uri_1.URI.parse(uriString)).toString();
    }
    exports.dirname = dirname;
    function joinPath(uriString) {
        var paths = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            paths[_i - 1] = arguments[_i];
        }
        return vscode_uri_1.Utils.joinPath.apply(vscode_uri_1.Utils, __spreadArrays([vscode_uri_1.URI.parse(uriString)], paths)).toString();
    }
    exports.joinPath = joinPath;
});

/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __generator = (this && this.__generator) || function (thisArg, body) {
    var _ = { label: 0, sent: function() { if (t[0] & 1) throw t[1]; return t[1]; }, trys: [], ops: [] }, f, y, t, g;
    return g = { next: verb(0), "throw": verb(1), "return": verb(2) }, typeof Symbol === "function" && (g[Symbol.iterator] = function() { return this; }), g;
    function verb(n) { return function (v) { return step([n, v]); }; }
    function step(op) {
        if (f) throw new TypeError("Generator is already executing.");
        while (_) try {
            if (f = 1, y && (t = op[0] & 2 ? y["return"] : op[0] ? y["throw"] || ((t = y["return"]) && t.call(y), 0) : y.next) && !(t = t.call(y, op[1])).done) return t;
            if (y = 0, t) op = [op[0] & 2, t.value];
            switch (op[0]) {
                case 0: case 1: t = op; break;
                case 4: _.label++; return { value: op[1], done: false };
                case 5: _.label++; y = op[1]; op = [0]; continue;
                case 7: op = _.ops.pop(); _.trys.pop(); continue;
                default:
                    if (!(t = _.trys, t = t.length > 0 && t[t.length - 1]) && (op[0] === 6 || op[0] === 2)) { _ = 0; continue; }
                    if (op[0] === 3 && (!t || (op[1] > t[0] && op[1] < t[3]))) { _.label = op[1]; break; }
                    if (op[0] === 6 && _.label < t[1]) { _.label = t[1]; t = op; break; }
                    if (t && _.label < t[2]) { _.label = t[2]; _.ops.push(op); break; }
                    if (t[2]) _.ops.pop();
                    _.trys.pop(); continue;
            }
            op = body.call(thisArg, _);
        } catch (e) { op = [6, e]; y = 0; } finally { f = t = 0; }
        if (op[0] & 5) throw op[1]; return { value: op[0] ? op[1] : void 0, done: true };
    }
};
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/services/pathCompletion',["require", "exports", "../cssLanguageTypes", "../utils/strings", "../utils/resources"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.PathCompletionParticipant = void 0;
    var cssLanguageTypes_1 = require("../cssLanguageTypes");
    var strings_1 = require("../utils/strings");
    var resources_1 = require("../utils/resources");
    var PathCompletionParticipant = /** @class */ (function () {
        function PathCompletionParticipant(readDirectory) {
            this.readDirectory = readDirectory;
            this.literalCompletions = [];
            this.importCompletions = [];
        }
        PathCompletionParticipant.prototype.onCssURILiteralValue = function (context) {
            this.literalCompletions.push(context);
        };
        PathCompletionParticipant.prototype.onCssImportPath = function (context) {
            this.importCompletions.push(context);
        };
        PathCompletionParticipant.prototype.computeCompletions = function (document, documentContext) {
            return __awaiter(this, void 0, void 0, function () {
                var result, _i, _a, literalCompletion, uriValue, fullValue, items, _b, items_1, item, _c, _d, importCompletion, pathValue, fullValue, suggestions, _e, suggestions_1, item;
                return __generator(this, function (_f) {
                    switch (_f.label) {
                        case 0:
                            result = { items: [], isIncomplete: false };
                            _i = 0, _a = this.literalCompletions;
                            _f.label = 1;
                        case 1:
                            if (!(_i < _a.length)) return [3 /*break*/, 5];
                            literalCompletion = _a[_i];
                            uriValue = literalCompletion.uriValue;
                            fullValue = stripQuotes(uriValue);
                            if (!(fullValue === '.' || fullValue === '..')) return [3 /*break*/, 2];
                            result.isIncomplete = true;
                            return [3 /*break*/, 4];
                        case 2: return [4 /*yield*/, this.providePathSuggestions(uriValue, literalCompletion.position, literalCompletion.range, document, documentContext)];
                        case 3:
                            items = _f.sent();
                            for (_b = 0, items_1 = items; _b < items_1.length; _b++) {
                                item = items_1[_b];
                                result.items.push(item);
                            }
                            _f.label = 4;
                        case 4:
                            _i++;
                            return [3 /*break*/, 1];
                        case 5:
                            _c = 0, _d = this.importCompletions;
                            _f.label = 6;
                        case 6:
                            if (!(_c < _d.length)) return [3 /*break*/, 10];
                            importCompletion = _d[_c];
                            pathValue = importCompletion.pathValue;
                            fullValue = stripQuotes(pathValue);
                            if (!(fullValue === '.' || fullValue === '..')) return [3 /*break*/, 7];
                            result.isIncomplete = true;
                            return [3 /*break*/, 9];
                        case 7: return [4 /*yield*/, this.providePathSuggestions(pathValue, importCompletion.position, importCompletion.range, document, documentContext)];
                        case 8:
                            suggestions = _f.sent();
                            if (document.languageId === 'scss') {
                                suggestions.forEach(function (s) {
                                    if (strings_1.startsWith(s.label, '_') && strings_1.endsWith(s.label, '.scss')) {
                                        if (s.textEdit) {
                                            s.textEdit.newText = s.label.slice(1, -5);
                                        }
                                        else {
                                            s.label = s.label.slice(1, -5);
                                        }
                                    }
                                });
                            }
                            for (_e = 0, suggestions_1 = suggestions; _e < suggestions_1.length; _e++) {
                                item = suggestions_1[_e];
                                result.items.push(item);
                            }
                            _f.label = 9;
                        case 9:
                            _c++;
                            return [3 /*break*/, 6];
                        case 10: return [2 /*return*/, result];
                    }
                });
            });
        };
        PathCompletionParticipant.prototype.providePathSuggestions = function (pathValue, position, range, document, documentContext) {
            return __awaiter(this, void 0, void 0, function () {
                var fullValue, isValueQuoted, valueBeforeCursor, currentDocUri, fullValueRange, replaceRange, valueBeforeLastSlash, parentDir, result, infos, _i, infos_1, _a, name, type, e_1;
                return __generator(this, function (_b) {
                    switch (_b.label) {
                        case 0:
                            fullValue = stripQuotes(pathValue);
                            isValueQuoted = strings_1.startsWith(pathValue, "'") || strings_1.startsWith(pathValue, "\"");
                            valueBeforeCursor = isValueQuoted
                                ? fullValue.slice(0, position.character - (range.start.character + 1))
                                : fullValue.slice(0, position.character - range.start.character);
                            currentDocUri = document.uri;
                            fullValueRange = isValueQuoted ? shiftRange(range, 1, -1) : range;
                            replaceRange = pathToReplaceRange(valueBeforeCursor, fullValue, fullValueRange);
                            valueBeforeLastSlash = valueBeforeCursor.substring(0, valueBeforeCursor.lastIndexOf('/') + 1);
                            parentDir = documentContext.resolveReference(valueBeforeLastSlash || '.', currentDocUri);
                            if (!parentDir) return [3 /*break*/, 4];
                            _b.label = 1;
                        case 1:
                            _b.trys.push([1, 3, , 4]);
                            result = [];
                            return [4 /*yield*/, this.readDirectory(parentDir)];
                        case 2:
                            infos = _b.sent();
                            for (_i = 0, infos_1 = infos; _i < infos_1.length; _i++) {
                                _a = infos_1[_i], name = _a[0], type = _a[1];
                                // Exclude paths that start with `.`
                                if (name.charCodeAt(0) !== CharCode_dot && (type === cssLanguageTypes_1.FileType.Directory || resources_1.joinPath(parentDir, name) !== currentDocUri)) {
                                    result.push(createCompletionItem(name, type === cssLanguageTypes_1.FileType.Directory, replaceRange));
                                }
                            }
                            return [2 /*return*/, result];
                        case 3:
                            e_1 = _b.sent();
                            return [3 /*break*/, 4];
                        case 4: return [2 /*return*/, []];
                    }
                });
            });
        };
        return PathCompletionParticipant;
    }());
    exports.PathCompletionParticipant = PathCompletionParticipant;
    var CharCode_dot = '.'.charCodeAt(0);
    function stripQuotes(fullValue) {
        if (strings_1.startsWith(fullValue, "'") || strings_1.startsWith(fullValue, "\"")) {
            return fullValue.slice(1, -1);
        }
        else {
            return fullValue;
        }
    }
    function pathToReplaceRange(valueBeforeCursor, fullValue, fullValueRange) {
        var replaceRange;
        var lastIndexOfSlash = valueBeforeCursor.lastIndexOf('/');
        if (lastIndexOfSlash === -1) {
            replaceRange = fullValueRange;
        }
        else {
            // For cases where cursor is in the middle of attribute value, like <script src="./s|rc/test.js">
            // Find the last slash before cursor, and calculate the start of replace range from there
            var valueAfterLastSlash = fullValue.slice(lastIndexOfSlash + 1);
            var startPos = shiftPosition(fullValueRange.end, -valueAfterLastSlash.length);
            // If whitespace exists, replace until it
            var whitespaceIndex = valueAfterLastSlash.indexOf(' ');
            var endPos = void 0;
            if (whitespaceIndex !== -1) {
                endPos = shiftPosition(startPos, whitespaceIndex);
            }
            else {
                endPos = fullValueRange.end;
            }
            replaceRange = cssLanguageTypes_1.Range.create(startPos, endPos);
        }
        return replaceRange;
    }
    function createCompletionItem(name, isDir, replaceRange) {
        if (isDir) {
            name = name + '/';
            return {
                label: escapePath(name),
                kind: cssLanguageTypes_1.CompletionItemKind.Folder,
                textEdit: cssLanguageTypes_1.TextEdit.replace(replaceRange, escapePath(name)),
                command: {
                    title: 'Suggest',
                    command: 'editor.action.triggerSuggest'
                }
            };
        }
        else {
            return {
                label: escapePath(name),
                kind: cssLanguageTypes_1.CompletionItemKind.File,
                textEdit: cssLanguageTypes_1.TextEdit.replace(replaceRange, escapePath(name))
            };
        }
    }
    // Escape https://www.w3.org/TR/CSS1/#url
    function escapePath(p) {
        return p.replace(/(\s|\(|\)|,|"|')/g, '\\$1');
    }
    function shiftPosition(pos, offset) {
        return cssLanguageTypes_1.Position.create(pos.line, pos.character + offset);
    }
    function shiftRange(range, startOffset, endOffset) {
        var start = shiftPosition(range.start, startOffset);
        var end = shiftPosition(range.end, endOffset);
        return cssLanguageTypes_1.Range.create(start, end);
    }
});

var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __generator = (this && this.__generator) || function (thisArg, body) {
    var _ = { label: 0, sent: function() { if (t[0] & 1) throw t[1]; return t[1]; }, trys: [], ops: [] }, f, y, t, g;
    return g = { next: verb(0), "throw": verb(1), "return": verb(2) }, typeof Symbol === "function" && (g[Symbol.iterator] = function() { return this; }), g;
    function verb(n) { return function (v) { return step([n, v]); }; }
    function step(op) {
        if (f) throw new TypeError("Generator is already executing.");
        while (_) try {
            if (f = 1, y && (t = op[0] & 2 ? y["return"] : op[0] ? y["throw"] || ((t = y["return"]) && t.call(y), 0) : y.next) && !(t = t.call(y, op[1])).done) return t;
            if (y = 0, t) op = [op[0] & 2, t.value];
            switch (op[0]) {
                case 0: case 1: t = op; break;
                case 4: _.label++; return { value: op[1], done: false };
                case 5: _.label++; y = op[1]; op = [0]; continue;
                case 7: op = _.ops.pop(); _.trys.pop(); continue;
                default:
                    if (!(t = _.trys, t = t.length > 0 && t[t.length - 1]) && (op[0] === 6 || op[0] === 2)) { _ = 0; continue; }
                    if (op[0] === 3 && (!t || (op[1] > t[0] && op[1] < t[3]))) { _.label = op[1]; break; }
                    if (op[0] === 6 && _.label < t[1]) { _.label = t[1]; t = op; break; }
                    if (t && _.label < t[2]) { _.label = t[2]; _.ops.push(op); break; }
                    if (t[2]) _.ops.pop();
                    _.trys.pop(); continue;
            }
            op = body.call(thisArg, _);
        } catch (e) { op = [6, e]; y = 0; } finally { f = t = 0; }
        if (op[0] & 5) throw op[1]; return { value: op[0] ? op[1] : void 0, done: true };
    }
};
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/services/cssCompletion',["require", "exports", "../parser/cssNodes", "../parser/cssSymbolScope", "../languageFacts/facts", "../utils/strings", "../cssLanguageTypes", "vscode-nls", "../utils/objects", "./pathCompletion"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.CSSCompletion = void 0;
    var nodes = require("../parser/cssNodes");
    var cssSymbolScope_1 = require("../parser/cssSymbolScope");
    var languageFacts = require("../languageFacts/facts");
    var strings = require("../utils/strings");
    var cssLanguageTypes_1 = require("../cssLanguageTypes");
    var nls = require("vscode-nls");
    var objects_1 = require("../utils/objects");
    var pathCompletion_1 = require("./pathCompletion");
    var localize = nls.loadMessageBundle();
    var SnippetFormat = cssLanguageTypes_1.InsertTextFormat.Snippet;
    var SortTexts;
    (function (SortTexts) {
        // char code 32, comes before everything
        SortTexts["Enums"] = " ";
        SortTexts["Normal"] = "d";
        SortTexts["VendorPrefixed"] = "x";
        SortTexts["Term"] = "y";
        SortTexts["Variable"] = "z";
    })(SortTexts || (SortTexts = {}));
    var CSSCompletion = /** @class */ (function () {
        function CSSCompletion(variablePrefix, lsOptions, cssDataManager) {
            if (variablePrefix === void 0) { variablePrefix = null; }
            this.variablePrefix = variablePrefix;
            this.lsOptions = lsOptions;
            this.cssDataManager = cssDataManager;
            this.completionParticipants = [];
        }
        CSSCompletion.prototype.configure = function (settings) {
            this.settings = settings;
        };
        CSSCompletion.prototype.getSymbolContext = function () {
            if (!this.symbolContext) {
                this.symbolContext = new cssSymbolScope_1.Symbols(this.styleSheet);
            }
            return this.symbolContext;
        };
        CSSCompletion.prototype.setCompletionParticipants = function (registeredCompletionParticipants) {
            this.completionParticipants = registeredCompletionParticipants || [];
        };
        CSSCompletion.prototype.doComplete2 = function (document, position, styleSheet, documentContext) {
            return __awaiter(this, void 0, void 0, function () {
                var participant, contributedParticipants, result, pathCompletionResult;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            if (!this.lsOptions.fileSystemProvider || !this.lsOptions.fileSystemProvider.readDirectory) {
                                return [2 /*return*/, this.doComplete(document, position, styleSheet)];
                            }
                            participant = new pathCompletion_1.PathCompletionParticipant(this.lsOptions.fileSystemProvider.readDirectory);
                            contributedParticipants = this.completionParticipants;
                            this.completionParticipants = [participant].concat(contributedParticipants);
                            result = this.doComplete(document, position, styleSheet);
                            _a.label = 1;
                        case 1:
                            _a.trys.push([1, , 3, 4]);
                            return [4 /*yield*/, participant.computeCompletions(document, documentContext)];
                        case 2:
                            pathCompletionResult = _a.sent();
                            return [2 /*return*/, {
                                    isIncomplete: result.isIncomplete || pathCompletionResult.isIncomplete,
                                    items: pathCompletionResult.items.concat(result.items)
                                }];
                        case 3:
                            this.completionParticipants = contributedParticipants;
                            return [7 /*endfinally*/];
                        case 4: return [2 /*return*/];
                    }
                });
            });
        };
        CSSCompletion.prototype.doComplete = function (document, position, styleSheet) {
            this.offset = document.offsetAt(position);
            this.position = position;
            this.currentWord = getCurrentWord(document, this.offset);
            this.defaultReplaceRange = cssLanguageTypes_1.Range.create(cssLanguageTypes_1.Position.create(this.position.line, this.position.character - this.currentWord.length), this.position);
            this.textDocument = document;
            this.styleSheet = styleSheet;
            try {
                var result = { isIncomplete: false, items: [] };
                this.nodePath = nodes.getNodePath(this.styleSheet, this.offset);
                for (var i = this.nodePath.length - 1; i >= 0; i--) {
                    var node = this.nodePath[i];
                    if (node instanceof nodes.Property) {
                        this.getCompletionsForDeclarationProperty(node.getParent(), result);
                    }
                    else if (node instanceof nodes.Expression) {
                        if (node.parent instanceof nodes.Interpolation) {
                            this.getVariableProposals(null, result);
                        }
                        else {
                            this.getCompletionsForExpression(node, result);
                        }
                    }
                    else if (node instanceof nodes.SimpleSelector) {
                        var parentRef = node.findAParent(nodes.NodeType.ExtendsReference, nodes.NodeType.Ruleset);
                        if (parentRef) {
                            if (parentRef.type === nodes.NodeType.ExtendsReference) {
                                this.getCompletionsForExtendsReference(parentRef, node, result);
                            }
                            else {
                                var parentRuleSet = parentRef;
                                this.getCompletionsForSelector(parentRuleSet, parentRuleSet && parentRuleSet.isNested(), result);
                            }
                        }
                    }
                    else if (node instanceof nodes.FunctionArgument) {
                        this.getCompletionsForFunctionArgument(node, node.getParent(), result);
                    }
                    else if (node instanceof nodes.Declarations) {
                        this.getCompletionsForDeclarations(node, result);
                    }
                    else if (node instanceof nodes.VariableDeclaration) {
                        this.getCompletionsForVariableDeclaration(node, result);
                    }
                    else if (node instanceof nodes.RuleSet) {
                        this.getCompletionsForRuleSet(node, result);
                    }
                    else if (node instanceof nodes.Interpolation) {
                        this.getCompletionsForInterpolation(node, result);
                    }
                    else if (node instanceof nodes.FunctionDeclaration) {
                        this.getCompletionsForFunctionDeclaration(node, result);
                    }
                    else if (node instanceof nodes.MixinReference) {
                        this.getCompletionsForMixinReference(node, result);
                    }
                    else if (node instanceof nodes.Function) {
                        this.getCompletionsForFunctionArgument(null, node, result);
                    }
                    else if (node instanceof nodes.Supports) {
                        this.getCompletionsForSupports(node, result);
                    }
                    else if (node instanceof nodes.SupportsCondition) {
                        this.getCompletionsForSupportsCondition(node, result);
                    }
                    else if (node instanceof nodes.ExtendsReference) {
                        this.getCompletionsForExtendsReference(node, null, result);
                    }
                    else if (node.type === nodes.NodeType.URILiteral) {
                        this.getCompletionForUriLiteralValue(node, result);
                    }
                    else if (node.parent === null) {
                        this.getCompletionForTopLevel(result);
                    }
                    else if (node.type === nodes.NodeType.StringLiteral && this.isImportPathParent(node.parent.type)) {
                        this.getCompletionForImportPath(node, result);
                        // } else if (node instanceof nodes.Variable) {
                        // this.getCompletionsForVariableDeclaration()
                    }
                    else {
                        continue;
                    }
                    if (result.items.length > 0 || this.offset > node.offset) {
                        return this.finalize(result);
                    }
                }
                this.getCompletionsForStylesheet(result);
                if (result.items.length === 0) {
                    if (this.variablePrefix && this.currentWord.indexOf(this.variablePrefix) === 0) {
                        this.getVariableProposals(null, result);
                    }
                }
                return this.finalize(result);
            }
            finally {
                // don't hold on any state, clear symbolContext
                this.position = null;
                this.currentWord = null;
                this.textDocument = null;
                this.styleSheet = null;
                this.symbolContext = null;
                this.defaultReplaceRange = null;
                this.nodePath = null;
            }
        };
        CSSCompletion.prototype.isImportPathParent = function (type) {
            return type === nodes.NodeType.Import;
        };
        CSSCompletion.prototype.finalize = function (result) {
            return result;
        };
        CSSCompletion.prototype.findInNodePath = function () {
            var types = [];
            for (var _i = 0; _i < arguments.length; _i++) {
                types[_i] = arguments[_i];
            }
            for (var i = this.nodePath.length - 1; i >= 0; i--) {
                var node = this.nodePath[i];
                if (types.indexOf(node.type) !== -1) {
                    return node;
                }
            }
            return null;
        };
        CSSCompletion.prototype.getCompletionsForDeclarationProperty = function (declaration, result) {
            return this.getPropertyProposals(declaration, result);
        };
        CSSCompletion.prototype.getPropertyProposals = function (declaration, result) {
            var _this = this;
            var triggerPropertyValueCompletion = this.isTriggerPropertyValueCompletionEnabled;
            var completePropertyWithSemicolon = this.isCompletePropertyWithSemicolonEnabled;
            var properties = this.cssDataManager.getProperties();
            properties.forEach(function (entry) {
                var range;
                var insertText;
                var retrigger = false;
                if (declaration) {
                    range = _this.getCompletionRange(declaration.getProperty());
                    insertText = entry.name;
                    if (!objects_1.isDefined(declaration.colonPosition)) {
                        insertText += ': ';
                        retrigger = true;
                    }
                }
                else {
                    range = _this.getCompletionRange(null);
                    insertText = entry.name + ': ';
                    retrigger = true;
                }
                // Empty .selector { | } case
                if (!declaration && completePropertyWithSemicolon) {
                    insertText += '$0;';
                }
                // Cases such as .selector { p; } or .selector { p:; }
                if (declaration && !declaration.semicolonPosition) {
                    if (completePropertyWithSemicolon && _this.offset >= _this.textDocument.offsetAt(range.end)) {
                        insertText += '$0;';
                    }
                }
                var item = {
                    label: entry.name,
                    documentation: languageFacts.getEntryDescription(entry, _this.doesSupportMarkdown()),
                    tags: isDeprecated(entry) ? [cssLanguageTypes_1.CompletionItemTag.Deprecated] : [],
                    textEdit: cssLanguageTypes_1.TextEdit.replace(range, insertText),
                    insertTextFormat: cssLanguageTypes_1.InsertTextFormat.Snippet,
                    kind: cssLanguageTypes_1.CompletionItemKind.Property
                };
                if (!entry.restrictions) {
                    retrigger = false;
                }
                if (triggerPropertyValueCompletion && retrigger) {
                    item.command = {
                        title: 'Suggest',
                        command: 'editor.action.triggerSuggest'
                    };
                }
                var relevance = typeof entry.relevance === 'number' ? Math.min(Math.max(entry.relevance, 0), 99) : 50;
                var sortTextSuffix = (255 - relevance).toString(16);
                var sortTextPrefix = strings.startsWith(entry.name, '-') ? SortTexts.VendorPrefixed : SortTexts.Normal;
                item.sortText = sortTextPrefix + '_' + sortTextSuffix;
                result.items.push(item);
            });
            this.completionParticipants.forEach(function (participant) {
                if (participant.onCssProperty) {
                    participant.onCssProperty({
                        propertyName: _this.currentWord,
                        range: _this.defaultReplaceRange
                    });
                }
            });
            return result;
        };
        Object.defineProperty(CSSCompletion.prototype, "isTriggerPropertyValueCompletionEnabled", {
            get: function () {
                if (!this.settings ||
                    !this.settings.completion ||
                    this.settings.completion.triggerPropertyValueCompletion === undefined) {
                    return true;
                }
                return this.settings.completion.triggerPropertyValueCompletion;
            },
            enumerable: false,
            configurable: true
        });
        Object.defineProperty(CSSCompletion.prototype, "isCompletePropertyWithSemicolonEnabled", {
            get: function () {
                if (!this.settings ||
                    !this.settings.completion ||
                    this.settings.completion.completePropertyWithSemicolon === undefined) {
                    return true;
                }
                return this.settings.completion.completePropertyWithSemicolon;
            },
            enumerable: false,
            configurable: true
        });
        CSSCompletion.prototype.getCompletionsForDeclarationValue = function (node, result) {
            var _this = this;
            var propertyName = node.getFullPropertyName();
            var entry = this.cssDataManager.getProperty(propertyName);
            var existingNode = node.getValue() || null;
            while (existingNode && existingNode.hasChildren()) {
                existingNode = existingNode.findChildAtOffset(this.offset, false);
            }
            this.completionParticipants.forEach(function (participant) {
                if (participant.onCssPropertyValue) {
                    participant.onCssPropertyValue({
                        propertyName: propertyName,
                        propertyValue: _this.currentWord,
                        range: _this.getCompletionRange(existingNode)
                    });
                }
            });
            if (entry) {
                if (entry.restrictions) {
                    for (var _i = 0, _a = entry.restrictions; _i < _a.length; _i++) {
                        var restriction = _a[_i];
                        switch (restriction) {
                            case 'color':
                                this.getColorProposals(entry, existingNode, result);
                                break;
                            case 'position':
                                this.getPositionProposals(entry, existingNode, result);
                                break;
                            case 'repeat':
                                this.getRepeatStyleProposals(entry, existingNode, result);
                                break;
                            case 'line-style':
                                this.getLineStyleProposals(entry, existingNode, result);
                                break;
                            case 'line-width':
                                this.getLineWidthProposals(entry, existingNode, result);
                                break;
                            case 'geometry-box':
                                this.getGeometryBoxProposals(entry, existingNode, result);
                                break;
                            case 'box':
                                this.getBoxProposals(entry, existingNode, result);
                                break;
                            case 'image':
                                this.getImageProposals(entry, existingNode, result);
                                break;
                            case 'timing-function':
                                this.getTimingFunctionProposals(entry, existingNode, result);
                                break;
                            case 'shape':
                                this.getBasicShapeProposals(entry, existingNode, result);
                                break;
                        }
                    }
                }
                this.getValueEnumProposals(entry, existingNode, result);
                this.getCSSWideKeywordProposals(entry, existingNode, result);
                this.getUnitProposals(entry, existingNode, result);
            }
            else {
                var existingValues = collectValues(this.styleSheet, node);
                for (var _b = 0, _c = existingValues.getEntries(); _b < _c.length; _b++) {
                    var existingValue = _c[_b];
                    result.items.push({
                        label: existingValue,
                        textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), existingValue),
                        kind: cssLanguageTypes_1.CompletionItemKind.Value
                    });
                }
            }
            this.getVariableProposals(existingNode, result);
            this.getTermProposals(entry, existingNode, result);
            return result;
        };
        CSSCompletion.prototype.getValueEnumProposals = function (entry, existingNode, result) {
            if (entry.values) {
                for (var _i = 0, _a = entry.values; _i < _a.length; _i++) {
                    var value = _a[_i];
                    var insertString = value.name;
                    var insertTextFormat = void 0;
                    if (strings.endsWith(insertString, ')')) {
                        var from = insertString.lastIndexOf('(');
                        if (from !== -1) {
                            insertString = insertString.substr(0, from) + '($1)';
                            insertTextFormat = SnippetFormat;
                        }
                    }
                    var sortText = SortTexts.Enums;
                    if (strings.startsWith(value.name, '-')) {
                        sortText += SortTexts.VendorPrefixed;
                    }
                    var item = {
                        label: value.name,
                        documentation: languageFacts.getEntryDescription(value, this.doesSupportMarkdown()),
                        tags: isDeprecated(entry) ? [cssLanguageTypes_1.CompletionItemTag.Deprecated] : [],
                        textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), insertString),
                        sortText: sortText,
                        kind: cssLanguageTypes_1.CompletionItemKind.Value,
                        insertTextFormat: insertTextFormat
                    };
                    result.items.push(item);
                }
            }
            return result;
        };
        CSSCompletion.prototype.getCSSWideKeywordProposals = function (entry, existingNode, result) {
            for (var keywords in languageFacts.cssWideKeywords) {
                result.items.push({
                    label: keywords,
                    documentation: languageFacts.cssWideKeywords[keywords],
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), keywords),
                    kind: cssLanguageTypes_1.CompletionItemKind.Value
                });
            }
            return result;
        };
        CSSCompletion.prototype.getCompletionsForInterpolation = function (node, result) {
            if (this.offset >= node.offset + 2) {
                this.getVariableProposals(null, result);
            }
            return result;
        };
        CSSCompletion.prototype.getVariableProposals = function (existingNode, result) {
            var symbols = this.getSymbolContext().findSymbolsAtOffset(this.offset, nodes.ReferenceType.Variable);
            for (var _i = 0, symbols_1 = symbols; _i < symbols_1.length; _i++) {
                var symbol = symbols_1[_i];
                var insertText = strings.startsWith(symbol.name, '--') ? "var(" + symbol.name + ")" : symbol.name;
                var completionItem = {
                    label: symbol.name,
                    documentation: symbol.value ? strings.getLimitedString(symbol.value) : symbol.value,
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), insertText),
                    kind: cssLanguageTypes_1.CompletionItemKind.Variable,
                    sortText: SortTexts.Variable
                };
                if (typeof completionItem.documentation === 'string' && isColorString(completionItem.documentation)) {
                    completionItem.kind = cssLanguageTypes_1.CompletionItemKind.Color;
                }
                if (symbol.node.type === nodes.NodeType.FunctionParameter) {
                    var mixinNode = (symbol.node.getParent());
                    if (mixinNode.type === nodes.NodeType.MixinDeclaration) {
                        completionItem.detail = localize('completion.argument', 'argument from \'{0}\'', mixinNode.getName());
                    }
                }
                result.items.push(completionItem);
            }
            return result;
        };
        CSSCompletion.prototype.getVariableProposalsForCSSVarFunction = function (result) {
            var symbols = this.getSymbolContext().findSymbolsAtOffset(this.offset, nodes.ReferenceType.Variable);
            symbols = symbols.filter(function (symbol) {
                return strings.startsWith(symbol.name, '--');
            });
            for (var _i = 0, symbols_2 = symbols; _i < symbols_2.length; _i++) {
                var symbol = symbols_2[_i];
                var completionItem = {
                    label: symbol.name,
                    documentation: symbol.value ? strings.getLimitedString(symbol.value) : symbol.value,
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(null), symbol.name),
                    kind: cssLanguageTypes_1.CompletionItemKind.Variable
                };
                if (typeof completionItem.documentation === 'string' && isColorString(completionItem.documentation)) {
                    completionItem.kind = cssLanguageTypes_1.CompletionItemKind.Color;
                }
                result.items.push(completionItem);
            }
            return result;
        };
        CSSCompletion.prototype.getUnitProposals = function (entry, existingNode, result) {
            var currentWord = '0';
            if (this.currentWord.length > 0) {
                var numMatch = this.currentWord.match(/^-?\d[\.\d+]*/);
                if (numMatch) {
                    currentWord = numMatch[0];
                    result.isIncomplete = currentWord.length === this.currentWord.length;
                }
            }
            else if (this.currentWord.length === 0) {
                result.isIncomplete = true;
            }
            if (existingNode && existingNode.parent && existingNode.parent.type === nodes.NodeType.Term) {
                existingNode = existingNode.getParent(); // include the unary operator
            }
            if (entry.restrictions) {
                for (var _i = 0, _a = entry.restrictions; _i < _a.length; _i++) {
                    var restriction = _a[_i];
                    var units = languageFacts.units[restriction];
                    if (units) {
                        for (var _b = 0, units_1 = units; _b < units_1.length; _b++) {
                            var unit = units_1[_b];
                            var insertText = currentWord + unit;
                            result.items.push({
                                label: insertText,
                                textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), insertText),
                                kind: cssLanguageTypes_1.CompletionItemKind.Unit
                            });
                        }
                    }
                }
            }
            return result;
        };
        CSSCompletion.prototype.getCompletionRange = function (existingNode) {
            if (existingNode && existingNode.offset <= this.offset && this.offset <= existingNode.end) {
                var end = existingNode.end !== -1 ? this.textDocument.positionAt(existingNode.end) : this.position;
                var start = this.textDocument.positionAt(existingNode.offset);
                if (start.line === end.line) {
                    return cssLanguageTypes_1.Range.create(start, end); // multi line edits are not allowed
                }
            }
            return this.defaultReplaceRange;
        };
        CSSCompletion.prototype.getColorProposals = function (entry, existingNode, result) {
            for (var color in languageFacts.colors) {
                result.items.push({
                    label: color,
                    documentation: languageFacts.colors[color],
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), color),
                    kind: cssLanguageTypes_1.CompletionItemKind.Color
                });
            }
            for (var color in languageFacts.colorKeywords) {
                result.items.push({
                    label: color,
                    documentation: languageFacts.colorKeywords[color],
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), color),
                    kind: cssLanguageTypes_1.CompletionItemKind.Value
                });
            }
            var colorValues = new Set();
            this.styleSheet.acceptVisitor(new ColorValueCollector(colorValues, this.offset));
            for (var _i = 0, _a = colorValues.getEntries(); _i < _a.length; _i++) {
                var color = _a[_i];
                result.items.push({
                    label: color,
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), color),
                    kind: cssLanguageTypes_1.CompletionItemKind.Color
                });
            }
            var _loop_1 = function (p) {
                var tabStop = 1;
                var replaceFunction = function (_match, p1) { return '${' + tabStop++ + ':' + p1 + '}'; };
                var insertText = p.func.replace(/\[?\$(\w+)\]?/g, replaceFunction);
                result.items.push({
                    label: p.func.substr(0, p.func.indexOf('(')),
                    detail: p.func,
                    documentation: p.desc,
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this_1.getCompletionRange(existingNode), insertText),
                    insertTextFormat: SnippetFormat,
                    kind: cssLanguageTypes_1.CompletionItemKind.Function
                });
            };
            var this_1 = this;
            for (var _b = 0, _c = languageFacts.colorFunctions; _b < _c.length; _b++) {
                var p = _c[_b];
                _loop_1(p);
            }
            return result;
        };
        CSSCompletion.prototype.getPositionProposals = function (entry, existingNode, result) {
            for (var position in languageFacts.positionKeywords) {
                result.items.push({
                    label: position,
                    documentation: languageFacts.positionKeywords[position],
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), position),
                    kind: cssLanguageTypes_1.CompletionItemKind.Value
                });
            }
            return result;
        };
        CSSCompletion.prototype.getRepeatStyleProposals = function (entry, existingNode, result) {
            for (var repeat in languageFacts.repeatStyleKeywords) {
                result.items.push({
                    label: repeat,
                    documentation: languageFacts.repeatStyleKeywords[repeat],
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), repeat),
                    kind: cssLanguageTypes_1.CompletionItemKind.Value
                });
            }
            return result;
        };
        CSSCompletion.prototype.getLineStyleProposals = function (entry, existingNode, result) {
            for (var lineStyle in languageFacts.lineStyleKeywords) {
                result.items.push({
                    label: lineStyle,
                    documentation: languageFacts.lineStyleKeywords[lineStyle],
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), lineStyle),
                    kind: cssLanguageTypes_1.CompletionItemKind.Value
                });
            }
            return result;
        };
        CSSCompletion.prototype.getLineWidthProposals = function (entry, existingNode, result) {
            for (var _i = 0, _a = languageFacts.lineWidthKeywords; _i < _a.length; _i++) {
                var lineWidth = _a[_i];
                result.items.push({
                    label: lineWidth,
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), lineWidth),
                    kind: cssLanguageTypes_1.CompletionItemKind.Value
                });
            }
            return result;
        };
        CSSCompletion.prototype.getGeometryBoxProposals = function (entry, existingNode, result) {
            for (var box in languageFacts.geometryBoxKeywords) {
                result.items.push({
                    label: box,
                    documentation: languageFacts.geometryBoxKeywords[box],
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), box),
                    kind: cssLanguageTypes_1.CompletionItemKind.Value
                });
            }
            return result;
        };
        CSSCompletion.prototype.getBoxProposals = function (entry, existingNode, result) {
            for (var box in languageFacts.boxKeywords) {
                result.items.push({
                    label: box,
                    documentation: languageFacts.boxKeywords[box],
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), box),
                    kind: cssLanguageTypes_1.CompletionItemKind.Value
                });
            }
            return result;
        };
        CSSCompletion.prototype.getImageProposals = function (entry, existingNode, result) {
            for (var image in languageFacts.imageFunctions) {
                var insertText = moveCursorInsideParenthesis(image);
                result.items.push({
                    label: image,
                    documentation: languageFacts.imageFunctions[image],
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), insertText),
                    kind: cssLanguageTypes_1.CompletionItemKind.Function,
                    insertTextFormat: image !== insertText ? SnippetFormat : void 0
                });
            }
            return result;
        };
        CSSCompletion.prototype.getTimingFunctionProposals = function (entry, existingNode, result) {
            for (var timing in languageFacts.transitionTimingFunctions) {
                var insertText = moveCursorInsideParenthesis(timing);
                result.items.push({
                    label: timing,
                    documentation: languageFacts.transitionTimingFunctions[timing],
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), insertText),
                    kind: cssLanguageTypes_1.CompletionItemKind.Function,
                    insertTextFormat: timing !== insertText ? SnippetFormat : void 0
                });
            }
            return result;
        };
        CSSCompletion.prototype.getBasicShapeProposals = function (entry, existingNode, result) {
            for (var shape in languageFacts.basicShapeFunctions) {
                var insertText = moveCursorInsideParenthesis(shape);
                result.items.push({
                    label: shape,
                    documentation: languageFacts.basicShapeFunctions[shape],
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), insertText),
                    kind: cssLanguageTypes_1.CompletionItemKind.Function,
                    insertTextFormat: shape !== insertText ? SnippetFormat : void 0
                });
            }
            return result;
        };
        CSSCompletion.prototype.getCompletionsForStylesheet = function (result) {
            var node = this.styleSheet.findFirstChildBeforeOffset(this.offset);
            if (!node) {
                return this.getCompletionForTopLevel(result);
            }
            if (node instanceof nodes.RuleSet) {
                return this.getCompletionsForRuleSet(node, result);
            }
            if (node instanceof nodes.Supports) {
                return this.getCompletionsForSupports(node, result);
            }
            return result;
        };
        CSSCompletion.prototype.getCompletionForTopLevel = function (result) {
            var _this = this;
            this.cssDataManager.getAtDirectives().forEach(function (entry) {
                result.items.push({
                    label: entry.name,
                    textEdit: cssLanguageTypes_1.TextEdit.replace(_this.getCompletionRange(null), entry.name),
                    documentation: languageFacts.getEntryDescription(entry, _this.doesSupportMarkdown()),
                    tags: isDeprecated(entry) ? [cssLanguageTypes_1.CompletionItemTag.Deprecated] : [],
                    kind: cssLanguageTypes_1.CompletionItemKind.Keyword
                });
            });
            this.getCompletionsForSelector(null, false, result);
            return result;
        };
        CSSCompletion.prototype.getCompletionsForRuleSet = function (ruleSet, result) {
            var declarations = ruleSet.getDeclarations();
            var isAfter = declarations && declarations.endsWith('}') && this.offset >= declarations.end;
            if (isAfter) {
                return this.getCompletionForTopLevel(result);
            }
            var isInSelectors = !declarations || this.offset <= declarations.offset;
            if (isInSelectors) {
                return this.getCompletionsForSelector(ruleSet, ruleSet.isNested(), result);
            }
            return this.getCompletionsForDeclarations(ruleSet.getDeclarations(), result);
        };
        CSSCompletion.prototype.getCompletionsForSelector = function (ruleSet, isNested, result) {
            var _this = this;
            var existingNode = this.findInNodePath(nodes.NodeType.PseudoSelector, nodes.NodeType.IdentifierSelector, nodes.NodeType.ClassSelector, nodes.NodeType.ElementNameSelector);
            if (!existingNode && this.hasCharacterAtPosition(this.offset - this.currentWord.length - 1, ':')) {
                // after the ':' of a pseudo selector, no node generated for just ':'
                this.currentWord = ':' + this.currentWord;
                if (this.hasCharacterAtPosition(this.offset - this.currentWord.length - 1, ':')) {
                    this.currentWord = ':' + this.currentWord; // for '::'
                }
                this.defaultReplaceRange = cssLanguageTypes_1.Range.create(cssLanguageTypes_1.Position.create(this.position.line, this.position.character - this.currentWord.length), this.position);
            }
            var pseudoClasses = this.cssDataManager.getPseudoClasses();
            pseudoClasses.forEach(function (entry) {
                var insertText = moveCursorInsideParenthesis(entry.name);
                var item = {
                    label: entry.name,
                    textEdit: cssLanguageTypes_1.TextEdit.replace(_this.getCompletionRange(existingNode), insertText),
                    documentation: languageFacts.getEntryDescription(entry, _this.doesSupportMarkdown()),
                    tags: isDeprecated(entry) ? [cssLanguageTypes_1.CompletionItemTag.Deprecated] : [],
                    kind: cssLanguageTypes_1.CompletionItemKind.Function,
                    insertTextFormat: entry.name !== insertText ? SnippetFormat : void 0
                };
                if (strings.startsWith(entry.name, ':-')) {
                    item.sortText = SortTexts.VendorPrefixed;
                }
                result.items.push(item);
            });
            var pseudoElements = this.cssDataManager.getPseudoElements();
            pseudoElements.forEach(function (entry) {
                var insertText = moveCursorInsideParenthesis(entry.name);
                var item = {
                    label: entry.name,
                    textEdit: cssLanguageTypes_1.TextEdit.replace(_this.getCompletionRange(existingNode), insertText),
                    documentation: languageFacts.getEntryDescription(entry, _this.doesSupportMarkdown()),
                    tags: isDeprecated(entry) ? [cssLanguageTypes_1.CompletionItemTag.Deprecated] : [],
                    kind: cssLanguageTypes_1.CompletionItemKind.Function,
                    insertTextFormat: entry.name !== insertText ? SnippetFormat : void 0
                };
                if (strings.startsWith(entry.name, '::-')) {
                    item.sortText = SortTexts.VendorPrefixed;
                }
                result.items.push(item);
            });
            if (!isNested) { // show html tags only for top level
                for (var _i = 0, _a = languageFacts.html5Tags; _i < _a.length; _i++) {
                    var entry = _a[_i];
                    result.items.push({
                        label: entry,
                        textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), entry),
                        kind: cssLanguageTypes_1.CompletionItemKind.Keyword
                    });
                }
                for (var _b = 0, _c = languageFacts.svgElements; _b < _c.length; _b++) {
                    var entry = _c[_b];
                    result.items.push({
                        label: entry,
                        textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), entry),
                        kind: cssLanguageTypes_1.CompletionItemKind.Keyword
                    });
                }
            }
            var visited = {};
            visited[this.currentWord] = true;
            var docText = this.textDocument.getText();
            this.styleSheet.accept(function (n) {
                if (n.type === nodes.NodeType.SimpleSelector && n.length > 0) {
                    var selector = docText.substr(n.offset, n.length);
                    if (selector.charAt(0) === '.' && !visited[selector]) {
                        visited[selector] = true;
                        result.items.push({
                            label: selector,
                            textEdit: cssLanguageTypes_1.TextEdit.replace(_this.getCompletionRange(existingNode), selector),
                            kind: cssLanguageTypes_1.CompletionItemKind.Keyword
                        });
                    }
                    return false;
                }
                return true;
            });
            if (ruleSet && ruleSet.isNested()) {
                var selector = ruleSet.getSelectors().findFirstChildBeforeOffset(this.offset);
                if (selector && ruleSet.getSelectors().getChildren().indexOf(selector) === 0) {
                    this.getPropertyProposals(null, result);
                }
            }
            return result;
        };
        CSSCompletion.prototype.getCompletionsForDeclarations = function (declarations, result) {
            if (!declarations || this.offset === declarations.offset) { // incomplete nodes
                return result;
            }
            var node = declarations.findFirstChildBeforeOffset(this.offset);
            if (!node) {
                return this.getCompletionsForDeclarationProperty(null, result);
            }
            if (node instanceof nodes.AbstractDeclaration) {
                var declaration = node;
                if (!objects_1.isDefined(declaration.colonPosition) || this.offset <= declaration.colonPosition) {
                    // complete property
                    return this.getCompletionsForDeclarationProperty(declaration, result);
                }
                else if ((objects_1.isDefined(declaration.semicolonPosition) && declaration.semicolonPosition < this.offset)) {
                    if (this.offset === declaration.semicolonPosition + 1) {
                        return result; // don't show new properties right after semicolon (see Bug 15421:[intellisense] [css] Be less aggressive when manually typing CSS)
                    }
                    // complete next property
                    return this.getCompletionsForDeclarationProperty(null, result);
                }
                if (declaration instanceof nodes.Declaration) {
                    // complete value
                    return this.getCompletionsForDeclarationValue(declaration, result);
                }
            }
            else if (node instanceof nodes.ExtendsReference) {
                this.getCompletionsForExtendsReference(node, null, result);
            }
            else if (this.currentWord && this.currentWord[0] === '@') {
                this.getCompletionsForDeclarationProperty(null, result);
            }
            else if (node instanceof nodes.RuleSet) {
                this.getCompletionsForDeclarationProperty(null, result);
            }
            return result;
        };
        CSSCompletion.prototype.getCompletionsForVariableDeclaration = function (declaration, result) {
            if (this.offset && objects_1.isDefined(declaration.colonPosition) && this.offset > declaration.colonPosition) {
                this.getVariableProposals(declaration.getValue(), result);
            }
            return result;
        };
        CSSCompletion.prototype.getCompletionsForExpression = function (expression, result) {
            var parent = expression.getParent();
            if (parent instanceof nodes.FunctionArgument) {
                this.getCompletionsForFunctionArgument(parent, parent.getParent(), result);
                return result;
            }
            var declaration = expression.findParent(nodes.NodeType.Declaration);
            if (!declaration) {
                this.getTermProposals(undefined, null, result);
                return result;
            }
            var node = expression.findChildAtOffset(this.offset, true);
            if (!node) {
                return this.getCompletionsForDeclarationValue(declaration, result);
            }
            if (node instanceof nodes.NumericValue || node instanceof nodes.Identifier) {
                return this.getCompletionsForDeclarationValue(declaration, result);
            }
            return result;
        };
        CSSCompletion.prototype.getCompletionsForFunctionArgument = function (arg, func, result) {
            var identifier = func.getIdentifier();
            if (identifier && identifier.matches('var')) {
                if (!func.getArguments().hasChildren() || func.getArguments().getChild(0) === arg) {
                    this.getVariableProposalsForCSSVarFunction(result);
                }
            }
            return result;
        };
        CSSCompletion.prototype.getCompletionsForFunctionDeclaration = function (decl, result) {
            var declarations = decl.getDeclarations();
            if (declarations && this.offset > declarations.offset && this.offset < declarations.end) {
                this.getTermProposals(undefined, null, result);
            }
            return result;
        };
        CSSCompletion.prototype.getCompletionsForMixinReference = function (ref, result) {
            var _this = this;
            var allMixins = this.getSymbolContext().findSymbolsAtOffset(this.offset, nodes.ReferenceType.Mixin);
            for (var _i = 0, allMixins_1 = allMixins; _i < allMixins_1.length; _i++) {
                var mixinSymbol = allMixins_1[_i];
                if (mixinSymbol.node instanceof nodes.MixinDeclaration) {
                    result.items.push(this.makeTermProposal(mixinSymbol, mixinSymbol.node.getParameters(), null));
                }
            }
            var identifierNode = ref.getIdentifier() || null;
            this.completionParticipants.forEach(function (participant) {
                if (participant.onCssMixinReference) {
                    participant.onCssMixinReference({
                        mixinName: _this.currentWord,
                        range: _this.getCompletionRange(identifierNode)
                    });
                }
            });
            return result;
        };
        CSSCompletion.prototype.getTermProposals = function (entry, existingNode, result) {
            var allFunctions = this.getSymbolContext().findSymbolsAtOffset(this.offset, nodes.ReferenceType.Function);
            for (var _i = 0, allFunctions_1 = allFunctions; _i < allFunctions_1.length; _i++) {
                var functionSymbol = allFunctions_1[_i];
                if (functionSymbol.node instanceof nodes.FunctionDeclaration) {
                    result.items.push(this.makeTermProposal(functionSymbol, functionSymbol.node.getParameters(), existingNode));
                }
            }
            return result;
        };
        CSSCompletion.prototype.makeTermProposal = function (symbol, parameters, existingNode) {
            var decl = symbol.node;
            var params = parameters.getChildren().map(function (c) {
                return (c instanceof nodes.FunctionParameter) ? c.getName() : c.getText();
            });
            var insertText = symbol.name + '(' + params.map(function (p, index) { return '${' + (index + 1) + ':' + p + '}'; }).join(', ') + ')';
            return {
                label: symbol.name,
                detail: symbol.name + '(' + params.join(', ') + ')',
                textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), insertText),
                insertTextFormat: SnippetFormat,
                kind: cssLanguageTypes_1.CompletionItemKind.Function,
                sortText: SortTexts.Term
            };
        };
        CSSCompletion.prototype.getCompletionsForSupportsCondition = function (supportsCondition, result) {
            var child = supportsCondition.findFirstChildBeforeOffset(this.offset);
            if (child) {
                if (child instanceof nodes.Declaration) {
                    if (!objects_1.isDefined(child.colonPosition) || this.offset <= child.colonPosition) {
                        return this.getCompletionsForDeclarationProperty(child, result);
                    }
                    else {
                        return this.getCompletionsForDeclarationValue(child, result);
                    }
                }
                else if (child instanceof nodes.SupportsCondition) {
                    return this.getCompletionsForSupportsCondition(child, result);
                }
            }
            if (objects_1.isDefined(supportsCondition.lParent) && this.offset > supportsCondition.lParent && (!objects_1.isDefined(supportsCondition.rParent) || this.offset <= supportsCondition.rParent)) {
                return this.getCompletionsForDeclarationProperty(null, result);
            }
            return result;
        };
        CSSCompletion.prototype.getCompletionsForSupports = function (supports, result) {
            var declarations = supports.getDeclarations();
            var inInCondition = !declarations || this.offset <= declarations.offset;
            if (inInCondition) {
                var child = supports.findFirstChildBeforeOffset(this.offset);
                if (child instanceof nodes.SupportsCondition) {
                    return this.getCompletionsForSupportsCondition(child, result);
                }
                return result;
            }
            return this.getCompletionForTopLevel(result);
        };
        CSSCompletion.prototype.getCompletionsForExtendsReference = function (extendsRef, existingNode, result) {
            return result;
        };
        CSSCompletion.prototype.getCompletionForUriLiteralValue = function (uriLiteralNode, result) {
            var uriValue;
            var position;
            var range;
            // No children, empty value
            if (!uriLiteralNode.hasChildren()) {
                uriValue = '';
                position = this.position;
                var emptyURIValuePosition = this.textDocument.positionAt(uriLiteralNode.offset + 'url('.length);
                range = cssLanguageTypes_1.Range.create(emptyURIValuePosition, emptyURIValuePosition);
            }
            else {
                var uriValueNode = uriLiteralNode.getChild(0);
                uriValue = uriValueNode.getText();
                position = this.position;
                range = this.getCompletionRange(uriValueNode);
            }
            this.completionParticipants.forEach(function (participant) {
                if (participant.onCssURILiteralValue) {
                    participant.onCssURILiteralValue({
                        uriValue: uriValue,
                        position: position,
                        range: range
                    });
                }
            });
            return result;
        };
        CSSCompletion.prototype.getCompletionForImportPath = function (importPathNode, result) {
            var _this = this;
            this.completionParticipants.forEach(function (participant) {
                if (participant.onCssImportPath) {
                    participant.onCssImportPath({
                        pathValue: importPathNode.getText(),
                        position: _this.position,
                        range: _this.getCompletionRange(importPathNode)
                    });
                }
            });
            return result;
        };
        CSSCompletion.prototype.hasCharacterAtPosition = function (offset, char) {
            var text = this.textDocument.getText();
            return (offset >= 0 && offset < text.length) && text.charAt(offset) === char;
        };
        CSSCompletion.prototype.doesSupportMarkdown = function () {
            var _a, _b, _c;
            if (!objects_1.isDefined(this.supportsMarkdown)) {
                if (!objects_1.isDefined(this.lsOptions.clientCapabilities)) {
                    this.supportsMarkdown = true;
                    return this.supportsMarkdown;
                }
                var documentationFormat = (_c = (_b = (_a = this.lsOptions.clientCapabilities.textDocument) === null || _a === void 0 ? void 0 : _a.completion) === null || _b === void 0 ? void 0 : _b.completionItem) === null || _c === void 0 ? void 0 : _c.documentationFormat;
                this.supportsMarkdown = Array.isArray(documentationFormat) && documentationFormat.indexOf(cssLanguageTypes_1.MarkupKind.Markdown) !== -1;
            }
            return this.supportsMarkdown;
        };
        return CSSCompletion;
    }());
    exports.CSSCompletion = CSSCompletion;
    function isDeprecated(entry) {
        if (entry.status && (entry.status === 'nonstandard' || entry.status === 'obsolete')) {
            return true;
        }
        return false;
    }
    /**
     * Rank number should all be same length strings
     */
    function computeRankNumber(n) {
        var nstr = n.toString();
        switch (nstr.length) {
            case 4:
                return nstr;
            case 3:
                return '0' + nstr;
            case 2:
                return '00' + nstr;
            case 1:
                return '000' + nstr;
            default:
                return '0000';
        }
    }
    var Set = /** @class */ (function () {
        function Set() {
            this.entries = {};
        }
        Set.prototype.add = function (entry) {
            this.entries[entry] = true;
        };
        Set.prototype.getEntries = function () {
            return Object.keys(this.entries);
        };
        return Set;
    }());
    function moveCursorInsideParenthesis(text) {
        return text.replace(/\(\)$/, "($1)");
    }
    function collectValues(styleSheet, declaration) {
        var fullPropertyName = declaration.getFullPropertyName();
        var entries = new Set();
        function visitValue(node) {
            if (node instanceof nodes.Identifier || node instanceof nodes.NumericValue || node instanceof nodes.HexColorValue) {
                entries.add(node.getText());
            }
            return true;
        }
        function matchesProperty(decl) {
            var propertyName = decl.getFullPropertyName();
            return fullPropertyName === propertyName;
        }
        function vistNode(node) {
            if (node instanceof nodes.Declaration && node !== declaration) {
                if (matchesProperty(node)) {
                    var value = node.getValue();
                    if (value) {
                        value.accept(visitValue);
                    }
                }
            }
            return true;
        }
        styleSheet.accept(vistNode);
        return entries;
    }
    var ColorValueCollector = /** @class */ (function () {
        function ColorValueCollector(entries, currentOffset) {
            this.entries = entries;
            this.currentOffset = currentOffset;
            // nothing to do
        }
        ColorValueCollector.prototype.visitNode = function (node) {
            if (node instanceof nodes.HexColorValue || (node instanceof nodes.Function && languageFacts.isColorConstructor(node))) {
                if (this.currentOffset < node.offset || node.end < this.currentOffset) {
                    this.entries.add(node.getText());
                }
            }
            return true;
        };
        return ColorValueCollector;
    }());
    function getCurrentWord(document, offset) {
        var i = offset - 1;
        var text = document.getText();
        while (i >= 0 && ' \t\n\r":{[()]},*>+'.indexOf(text.charAt(i)) === -1) {
            i--;
        }
        return text.substring(i + 1, offset);
    }
    function isColorString(s) {
        // From https://stackoverflow.com/questions/8027423/how-to-check-if-a-string-is-a-valid-hex-color-representation/8027444
        return (s.toLowerCase() in languageFacts.colors) || /(^#[0-9A-F]{6}$)|(^#[0-9A-F]{3}$)/i.test(s);
    }
});

var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/services/selectorPrinting',["require", "exports", "../parser/cssNodes", "../parser/cssScanner", "vscode-nls"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.selectorToElement = exports.SelectorPrinting = exports.toElement = exports.LabelElement = exports.RootElement = exports.Element = void 0;
    var nodes = require("../parser/cssNodes");
    var cssScanner_1 = require("../parser/cssScanner");
    var nls = require("vscode-nls");
    var localize = nls.loadMessageBundle();
    var Element = /** @class */ (function () {
        function Element() {
            this.parent = null;
            this.children = null;
            this.attributes = null;
        }
        Element.prototype.findAttribute = function (name) {
            if (this.attributes) {
                for (var _i = 0, _a = this.attributes; _i < _a.length; _i++) {
                    var attribute = _a[_i];
                    if (attribute.name === name) {
                        return attribute.value;
                    }
                }
            }
            return null;
        };
        Element.prototype.addChild = function (child) {
            if (child instanceof Element) {
                child.parent = this;
            }
            if (!this.children) {
                this.children = [];
            }
            this.children.push(child);
        };
        Element.prototype.append = function (text) {
            if (this.attributes) {
                var last = this.attributes[this.attributes.length - 1];
                last.value = last.value + text;
            }
        };
        Element.prototype.prepend = function (text) {
            if (this.attributes) {
                var first = this.attributes[0];
                first.value = text + first.value;
            }
        };
        Element.prototype.findRoot = function () {
            var curr = this;
            while (curr.parent && !(curr.parent instanceof RootElement)) {
                curr = curr.parent;
            }
            return curr;
        };
        Element.prototype.removeChild = function (child) {
            if (this.children) {
                var index = this.children.indexOf(child);
                if (index !== -1) {
                    this.children.splice(index, 1);
                    return true;
                }
            }
            return false;
        };
        Element.prototype.addAttr = function (name, value) {
            if (!this.attributes) {
                this.attributes = [];
            }
            for (var _i = 0, _a = this.attributes; _i < _a.length; _i++) {
                var attribute = _a[_i];
                if (attribute.name === name) {
                    attribute.value += ' ' + value;
                    return;
                }
            }
            this.attributes.push({ name: name, value: value });
        };
        Element.prototype.clone = function (cloneChildren) {
            if (cloneChildren === void 0) { cloneChildren = true; }
            var elem = new Element();
            if (this.attributes) {
                elem.attributes = [];
                for (var _i = 0, _a = this.attributes; _i < _a.length; _i++) {
                    var attribute = _a[_i];
                    elem.addAttr(attribute.name, attribute.value);
                }
            }
            if (cloneChildren && this.children) {
                elem.children = [];
                for (var index = 0; index < this.children.length; index++) {
                    elem.addChild(this.children[index].clone());
                }
            }
            return elem;
        };
        Element.prototype.cloneWithParent = function () {
            var clone = this.clone(false);
            if (this.parent && !(this.parent instanceof RootElement)) {
                var parentClone = this.parent.cloneWithParent();
                parentClone.addChild(clone);
            }
            return clone;
        };
        return Element;
    }());
    exports.Element = Element;
    var RootElement = /** @class */ (function (_super) {
        __extends(RootElement, _super);
        function RootElement() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        return RootElement;
    }(Element));
    exports.RootElement = RootElement;
    var LabelElement = /** @class */ (function (_super) {
        __extends(LabelElement, _super);
        function LabelElement(label) {
            var _this = _super.call(this) || this;
            _this.addAttr('name', label);
            return _this;
        }
        return LabelElement;
    }(Element));
    exports.LabelElement = LabelElement;
    var MarkedStringPrinter = /** @class */ (function () {
        function MarkedStringPrinter(quote) {
            this.quote = quote;
            this.result = [];
            // empty
        }
        MarkedStringPrinter.prototype.print = function (element) {
            this.result = [];
            if (element instanceof RootElement) {
                if (element.children) {
                    this.doPrint(element.children, 0);
                }
            }
            else {
                this.doPrint([element], 0);
            }
            var value = this.result.join('\n');
            return [{ language: 'html', value: value }];
        };
        MarkedStringPrinter.prototype.doPrint = function (elements, indent) {
            for (var _i = 0, elements_1 = elements; _i < elements_1.length; _i++) {
                var element = elements_1[_i];
                this.doPrintElement(element, indent);
                if (element.children) {
                    this.doPrint(element.children, indent + 1);
                }
            }
        };
        MarkedStringPrinter.prototype.writeLine = function (level, content) {
            var indent = new Array(level + 1).join('  ');
            this.result.push(indent + content);
        };
        MarkedStringPrinter.prototype.doPrintElement = function (element, indent) {
            var name = element.findAttribute('name');
            // special case: a simple label
            if (element instanceof LabelElement || name === '\u2026') {
                this.writeLine(indent, name);
                return;
            }
            // the real deal
            var content = ['<'];
            // element name
            if (name) {
                content.push(name);
            }
            else {
                content.push('element');
            }
            // attributes
            if (element.attributes) {
                for (var _i = 0, _a = element.attributes; _i < _a.length; _i++) {
                    var attr = _a[_i];
                    if (attr.name !== 'name') {
                        content.push(' ');
                        content.push(attr.name);
                        var value = attr.value;
                        if (value) {
                            content.push('=');
                            content.push(quotes.ensure(value, this.quote));
                        }
                    }
                }
            }
            content.push('>');
            this.writeLine(indent, content.join(''));
        };
        return MarkedStringPrinter;
    }());
    var quotes;
    (function (quotes) {
        function ensure(value, which) {
            return which + remove(value) + which;
        }
        quotes.ensure = ensure;
        function remove(value) {
            var match = value.match(/^['"](.*)["']$/);
            if (match) {
                return match[1];
            }
            return value;
        }
        quotes.remove = remove;
    })(quotes || (quotes = {}));
    var Specificity = /** @class */ (function () {
        function Specificity() {
            /** Count of identifiers (e.g., `#app`) */
            this.id = 0;
            /** Count of attributes (`[type="number"]`), classes (`.container-fluid`), and pseudo-classes (`:hover`) */
            this.attr = 0;
            /** Count of tag names (`div`), and pseudo-elements (`::before`) */
            this.tag = 0;
        }
        return Specificity;
    }());
    function toElement(node, parentElement) {
        var result = new Element();
        for (var _i = 0, _a = node.getChildren(); _i < _a.length; _i++) {
            var child = _a[_i];
            switch (child.type) {
                case nodes.NodeType.SelectorCombinator:
                    if (parentElement) {
                        var segments = child.getText().split('&');
                        if (segments.length === 1) {
                            // should not happen
                            result.addAttr('name', segments[0]);
                            break;
                        }
                        result = parentElement.cloneWithParent();
                        if (segments[0]) {
                            var root = result.findRoot();
                            root.prepend(segments[0]);
                        }
                        for (var i = 1; i < segments.length; i++) {
                            if (i > 1) {
                                var clone = parentElement.cloneWithParent();
                                result.addChild(clone.findRoot());
                                result = clone;
                            }
                            result.append(segments[i]);
                        }
                    }
                    break;
                case nodes.NodeType.SelectorPlaceholder:
                    if (child.matches('@at-root')) {
                        return result;
                    }
                // fall through
                case nodes.NodeType.ElementNameSelector:
                    var text = child.getText();
                    result.addAttr('name', text === '*' ? 'element' : unescape(text));
                    break;
                case nodes.NodeType.ClassSelector:
                    result.addAttr('class', unescape(child.getText().substring(1)));
                    break;
                case nodes.NodeType.IdentifierSelector:
                    result.addAttr('id', unescape(child.getText().substring(1)));
                    break;
                case nodes.NodeType.MixinDeclaration:
                    result.addAttr('class', child.getName());
                    break;
                case nodes.NodeType.PseudoSelector:
                    result.addAttr(unescape(child.getText()), '');
                    break;
                case nodes.NodeType.AttributeSelector:
                    var selector = child;
                    var identifier = selector.getIdentifier();
                    if (identifier) {
                        var expression = selector.getValue();
                        var operator = selector.getOperator();
                        var value = void 0;
                        if (expression && operator) {
                            switch (unescape(operator.getText())) {
                                case '|=':
                                    // excatly or followed by -words
                                    value = quotes.remove(unescape(expression.getText())) + "-\u2026";
                                    break;
                                case '^=':
                                    // prefix
                                    value = quotes.remove(unescape(expression.getText())) + "\u2026";
                                    break;
                                case '$=':
                                    // suffix
                                    value = "\u2026" + quotes.remove(unescape(expression.getText()));
                                    break;
                                case '~=':
                                    // one of a list of words
                                    value = " \u2026 " + quotes.remove(unescape(expression.getText())) + " \u2026 ";
                                    break;
                                case '*=':
                                    // substring
                                    value = "\u2026" + quotes.remove(unescape(expression.getText())) + "\u2026";
                                    break;
                                default:
                                    value = quotes.remove(unescape(expression.getText()));
                                    break;
                            }
                        }
                        result.addAttr(unescape(identifier.getText()), value);
                    }
                    break;
            }
        }
        return result;
    }
    exports.toElement = toElement;
    function unescape(content) {
        var scanner = new cssScanner_1.Scanner();
        scanner.setSource(content);
        var token = scanner.scanUnquotedString();
        if (token) {
            return token.text;
        }
        return content;
    }
    var SelectorPrinting = /** @class */ (function () {
        function SelectorPrinting(cssDataManager) {
            this.cssDataManager = cssDataManager;
        }
        SelectorPrinting.prototype.selectorToMarkedString = function (node) {
            var root = selectorToElement(node);
            if (root) {
                var markedStrings = new MarkedStringPrinter('"').print(root);
                markedStrings.push(this.selectorToSpecificityMarkedString(node));
                return markedStrings;
            }
            else {
                return [];
            }
        };
        SelectorPrinting.prototype.simpleSelectorToMarkedString = function (node) {
            var element = toElement(node);
            var markedStrings = new MarkedStringPrinter('"').print(element);
            markedStrings.push(this.selectorToSpecificityMarkedString(node));
            return markedStrings;
        };
        SelectorPrinting.prototype.isPseudoElementIdentifier = function (text) {
            var match = text.match(/^::?([\w-]+)/);
            if (!match) {
                return false;
            }
            return !!this.cssDataManager.getPseudoElement("::" + match[1]);
        };
        SelectorPrinting.prototype.selectorToSpecificityMarkedString = function (node) {
            var _this = this;
            //https://www.w3.org/TR/selectors-3/#specificity
            var calculateScore = function (node) {
                for (var _i = 0, _a = node.getChildren(); _i < _a.length; _i++) {
                    var element = _a[_i];
                    switch (element.type) {
                        case nodes.NodeType.IdentifierSelector:
                            specificity.id++;
                            break;
                        case nodes.NodeType.ClassSelector:
                        case nodes.NodeType.AttributeSelector:
                            specificity.attr++;
                            break;
                        case nodes.NodeType.ElementNameSelector:
                            //ignore universal selector
                            if (element.matches("*")) {
                                break;
                            }
                            specificity.tag++;
                            break;
                        case nodes.NodeType.PseudoSelector:
                            var text = element.getText();
                            if (_this.isPseudoElementIdentifier(text)) {
                                specificity.tag++; // pseudo element
                            }
                            else {
                                //ignore psuedo class NOT
                                if (text.match(/^:not/i)) {
                                    break;
                                }
                                specificity.attr++; //pseudo class
                            }
                            break;
                    }
                    if (element.getChildren().length > 0) {
                        calculateScore(element);
                    }
                }
            };
            var specificity = new Specificity();
            calculateScore(node);
            return localize('specificity', "[Selector Specificity](https://developer.mozilla.org/en-US/docs/Web/CSS/Specificity): ({0}, {1}, {2})", specificity.id, specificity.attr, specificity.tag);
        };
        return SelectorPrinting;
    }());
    exports.SelectorPrinting = SelectorPrinting;
    var SelectorElementBuilder = /** @class */ (function () {
        function SelectorElementBuilder(element) {
            this.prev = null;
            this.element = element;
        }
        SelectorElementBuilder.prototype.processSelector = function (selector) {
            var parentElement = null;
            if (!(this.element instanceof RootElement)) {
                if (selector.getChildren().some(function (c) { return c.hasChildren() && c.getChild(0).type === nodes.NodeType.SelectorCombinator; })) {
                    var curr = this.element.findRoot();
                    if (curr.parent instanceof RootElement) {
                        parentElement = this.element;
                        this.element = curr.parent;
                        this.element.removeChild(curr);
                        this.prev = null;
                    }
                }
            }
            for (var _i = 0, _a = selector.getChildren(); _i < _a.length; _i++) {
                var selectorChild = _a[_i];
                if (selectorChild instanceof nodes.SimpleSelector) {
                    if (this.prev instanceof nodes.SimpleSelector) {
                        var labelElement = new LabelElement('\u2026');
                        this.element.addChild(labelElement);
                        this.element = labelElement;
                    }
                    else if (this.prev && (this.prev.matches('+') || this.prev.matches('~')) && this.element.parent) {
                        this.element = this.element.parent;
                    }
                    if (this.prev && this.prev.matches('~')) {
                        this.element.addChild(new LabelElement('\u22EE'));
                    }
                    var thisElement = toElement(selectorChild, parentElement);
                    var root = thisElement.findRoot();
                    this.element.addChild(root);
                    this.element = thisElement;
                }
                if (selectorChild instanceof nodes.SimpleSelector ||
                    selectorChild.type === nodes.NodeType.SelectorCombinatorParent ||
                    selectorChild.type === nodes.NodeType.SelectorCombinatorShadowPiercingDescendant ||
                    selectorChild.type === nodes.NodeType.SelectorCombinatorSibling ||
                    selectorChild.type === nodes.NodeType.SelectorCombinatorAllSiblings) {
                    this.prev = selectorChild;
                }
            }
        };
        return SelectorElementBuilder;
    }());
    function isNewSelectorContext(node) {
        switch (node.type) {
            case nodes.NodeType.MixinDeclaration:
            case nodes.NodeType.Stylesheet:
                return true;
        }
        return false;
    }
    function selectorToElement(node) {
        if (node.matches('@at-root')) {
            return null;
        }
        var root = new RootElement();
        var parentRuleSets = [];
        var ruleSet = node.getParent();
        if (ruleSet instanceof nodes.RuleSet) {
            var parent = ruleSet.getParent(); // parent of the selector's ruleset
            while (parent && !isNewSelectorContext(parent)) {
                if (parent instanceof nodes.RuleSet) {
                    if (parent.getSelectors().matches('@at-root')) {
                        break;
                    }
                    parentRuleSets.push(parent);
                }
                parent = parent.getParent();
            }
        }
        var builder = new SelectorElementBuilder(root);
        for (var i = parentRuleSets.length - 1; i >= 0; i--) {
            var selector = parentRuleSets[i].getSelectors().getChild(0);
            if (selector) {
                builder.processSelector(selector);
            }
        }
        builder.processSelector(node);
        return root;
    }
    exports.selectorToElement = selectorToElement;
});

(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/services/cssHover',["require", "exports", "../parser/cssNodes", "../languageFacts/facts", "./selectorPrinting", "../utils/strings", "../cssLanguageTypes", "../utils/objects"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.CSSHover = void 0;
    var nodes = require("../parser/cssNodes");
    var languageFacts = require("../languageFacts/facts");
    var selectorPrinting_1 = require("./selectorPrinting");
    var strings_1 = require("../utils/strings");
    var cssLanguageTypes_1 = require("../cssLanguageTypes");
    var objects_1 = require("../utils/objects");
    var CSSHover = /** @class */ (function () {
        function CSSHover(clientCapabilities, cssDataManager) {
            this.clientCapabilities = clientCapabilities;
            this.cssDataManager = cssDataManager;
            this.selectorPrinting = new selectorPrinting_1.SelectorPrinting(cssDataManager);
        }
        CSSHover.prototype.doHover = function (document, position, stylesheet, settings) {
            function getRange(node) {
                return cssLanguageTypes_1.Range.create(document.positionAt(node.offset), document.positionAt(node.end));
            }
            var offset = document.offsetAt(position);
            var nodepath = nodes.getNodePath(stylesheet, offset);
            /**
             * nodepath is top-down
             * Build up the hover by appending inner node's information
             */
            var hover = null;
            for (var i = 0; i < nodepath.length; i++) {
                var node = nodepath[i];
                if (node instanceof nodes.Selector) {
                    hover = {
                        contents: this.selectorPrinting.selectorToMarkedString(node),
                        range: getRange(node)
                    };
                    break;
                }
                if (node instanceof nodes.SimpleSelector) {
                    /**
                     * Some sass specific at rules such as `@at-root` are parsed as `SimpleSelector`
                     */
                    if (!strings_1.startsWith(node.getText(), '@')) {
                        hover = {
                            contents: this.selectorPrinting.simpleSelectorToMarkedString(node),
                            range: getRange(node)
                        };
                    }
                    break;
                }
                if (node instanceof nodes.Declaration) {
                    var propertyName = node.getFullPropertyName();
                    var entry = this.cssDataManager.getProperty(propertyName);
                    if (entry) {
                        var contents = languageFacts.getEntryDescription(entry, this.doesSupportMarkdown(), settings);
                        if (contents) {
                            hover = {
                                contents: contents,
                                range: getRange(node)
                            };
                        }
                        else {
                            hover = null;
                        }
                    }
                    continue;
                }
                if (node instanceof nodes.UnknownAtRule) {
                    var atRuleName = node.getText();
                    var entry = this.cssDataManager.getAtDirective(atRuleName);
                    if (entry) {
                        var contents = languageFacts.getEntryDescription(entry, this.doesSupportMarkdown(), settings);
                        if (contents) {
                            hover = {
                                contents: contents,
                                range: getRange(node)
                            };
                        }
                        else {
                            hover = null;
                        }
                    }
                    continue;
                }
                if (node instanceof nodes.Node && node.type === nodes.NodeType.PseudoSelector) {
                    var selectorName = node.getText();
                    var entry = selectorName.slice(0, 2) === '::'
                        ? this.cssDataManager.getPseudoElement(selectorName)
                        : this.cssDataManager.getPseudoClass(selectorName);
                    if (entry) {
                        var contents = languageFacts.getEntryDescription(entry, this.doesSupportMarkdown(), settings);
                        if (contents) {
                            hover = {
                                contents: contents,
                                range: getRange(node)
                            };
                        }
                        else {
                            hover = null;
                        }
                    }
                    continue;
                }
            }
            if (hover) {
                hover.contents = this.convertContents(hover.contents);
            }
            return hover;
        };
        CSSHover.prototype.convertContents = function (contents) {
            if (!this.doesSupportMarkdown()) {
                if (typeof contents === 'string') {
                    return contents;
                }
                // MarkupContent
                else if ('kind' in contents) {
                    return {
                        kind: 'plaintext',
                        value: contents.value
                    };
                }
                // MarkedString[]
                else if (Array.isArray(contents)) {
                    return contents.map(function (c) {
                        return typeof c === 'string' ? c : c.value;
                    });
                }
                // MarkedString
                else {
                    return contents.value;
                }
            }
            return contents;
        };
        CSSHover.prototype.doesSupportMarkdown = function () {
            if (!objects_1.isDefined(this.supportsMarkdown)) {
                if (!objects_1.isDefined(this.clientCapabilities)) {
                    this.supportsMarkdown = true;
                    return this.supportsMarkdown;
                }
                var hover = this.clientCapabilities.textDocument && this.clientCapabilities.textDocument.hover;
                this.supportsMarkdown = hover && hover.contentFormat && Array.isArray(hover.contentFormat) && hover.contentFormat.indexOf(cssLanguageTypes_1.MarkupKind.Markdown) !== -1;
            }
            return this.supportsMarkdown;
        };
        return CSSHover;
    }());
    exports.CSSHover = CSSHover;
});

var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __generator = (this && this.__generator) || function (thisArg, body) {
    var _ = { label: 0, sent: function() { if (t[0] & 1) throw t[1]; return t[1]; }, trys: [], ops: [] }, f, y, t, g;
    return g = { next: verb(0), "throw": verb(1), "return": verb(2) }, typeof Symbol === "function" && (g[Symbol.iterator] = function() { return this; }), g;
    function verb(n) { return function (v) { return step([n, v]); }; }
    function step(op) {
        if (f) throw new TypeError("Generator is already executing.");
        while (_) try {
            if (f = 1, y && (t = op[0] & 2 ? y["return"] : op[0] ? y["throw"] || ((t = y["return"]) && t.call(y), 0) : y.next) && !(t = t.call(y, op[1])).done) return t;
            if (y = 0, t) op = [op[0] & 2, t.value];
            switch (op[0]) {
                case 0: case 1: t = op; break;
                case 4: _.label++; return { value: op[1], done: false };
                case 5: _.label++; y = op[1]; op = [0]; continue;
                case 7: op = _.ops.pop(); _.trys.pop(); continue;
                default:
                    if (!(t = _.trys, t = t.length > 0 && t[t.length - 1]) && (op[0] === 6 || op[0] === 2)) { _ = 0; continue; }
                    if (op[0] === 3 && (!t || (op[1] > t[0] && op[1] < t[3]))) { _.label = op[1]; break; }
                    if (op[0] === 6 && _.label < t[1]) { _.label = t[1]; t = op; break; }
                    if (t && _.label < t[2]) { _.label = t[2]; _.ops.push(op); break; }
                    if (t[2]) _.ops.pop();
                    _.trys.pop(); continue;
            }
            op = body.call(thisArg, _);
        } catch (e) { op = [6, e]; y = 0; } finally { f = t = 0; }
        if (op[0] & 5) throw op[1]; return { value: op[0] ? op[1] : void 0, done: true };
    }
};
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/services/cssNavigation',["require", "exports", "../cssLanguageTypes", "vscode-nls", "../parser/cssNodes", "../parser/cssSymbolScope", "../languageFacts/facts", "../utils/strings", "../utils/resources"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.CSSNavigation = void 0;
    var cssLanguageTypes_1 = require("../cssLanguageTypes");
    var nls = require("vscode-nls");
    var nodes = require("../parser/cssNodes");
    var cssSymbolScope_1 = require("../parser/cssSymbolScope");
    var facts_1 = require("../languageFacts/facts");
    var strings_1 = require("../utils/strings");
    var resources_1 = require("../utils/resources");
    var localize = nls.loadMessageBundle();
    var CSSNavigation = /** @class */ (function () {
        function CSSNavigation(fileSystemProvider) {
            this.fileSystemProvider = fileSystemProvider;
        }
        CSSNavigation.prototype.findDefinition = function (document, position, stylesheet) {
            var symbols = new cssSymbolScope_1.Symbols(stylesheet);
            var offset = document.offsetAt(position);
            var node = nodes.getNodeAtOffset(stylesheet, offset);
            if (!node) {
                return null;
            }
            var symbol = symbols.findSymbolFromNode(node);
            if (!symbol) {
                return null;
            }
            return {
                uri: document.uri,
                range: getRange(symbol.node, document)
            };
        };
        CSSNavigation.prototype.findReferences = function (document, position, stylesheet) {
            var highlights = this.findDocumentHighlights(document, position, stylesheet);
            return highlights.map(function (h) {
                return {
                    uri: document.uri,
                    range: h.range
                };
            });
        };
        CSSNavigation.prototype.findDocumentHighlights = function (document, position, stylesheet) {
            var result = [];
            var offset = document.offsetAt(position);
            var node = nodes.getNodeAtOffset(stylesheet, offset);
            if (!node || node.type === nodes.NodeType.Stylesheet || node.type === nodes.NodeType.Declarations) {
                return result;
            }
            if (node.type === nodes.NodeType.Identifier && node.parent && node.parent.type === nodes.NodeType.ClassSelector) {
                node = node.parent;
            }
            var symbols = new cssSymbolScope_1.Symbols(stylesheet);
            var symbol = symbols.findSymbolFromNode(node);
            var name = node.getText();
            stylesheet.accept(function (candidate) {
                if (symbol) {
                    if (symbols.matchesSymbol(candidate, symbol)) {
                        result.push({
                            kind: getHighlightKind(candidate),
                            range: getRange(candidate, document)
                        });
                        return false;
                    }
                }
                else if (node && node.type === candidate.type && candidate.matches(name)) {
                    // Same node type and data
                    result.push({
                        kind: getHighlightKind(candidate),
                        range: getRange(candidate, document)
                    });
                }
                return true;
            });
            return result;
        };
        CSSNavigation.prototype.isRawStringDocumentLinkNode = function (node) {
            return node.type === nodes.NodeType.Import;
        };
        CSSNavigation.prototype.findDocumentLinks = function (document, stylesheet, documentContext) {
            var links = this.findUnresolvedLinks(document, stylesheet);
            for (var i = 0; i < links.length; i++) {
                var target = links[i].target;
                if (target && !(/^\w+:\/\//g.test(target))) {
                    var resolved = documentContext.resolveReference(target, document.uri);
                    if (resolved) {
                        links[i].target = resolved;
                    }
                }
            }
            return links;
        };
        CSSNavigation.prototype.findDocumentLinks2 = function (document, stylesheet, documentContext) {
            return __awaiter(this, void 0, void 0, function () {
                var links, resolvedLinks, _i, links_1, link, target, resolvedTarget;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            links = this.findUnresolvedLinks(document, stylesheet);
                            resolvedLinks = [];
                            _i = 0, links_1 = links;
                            _a.label = 1;
                        case 1:
                            if (!(_i < links_1.length)) return [3 /*break*/, 5];
                            link = links_1[_i];
                            target = link.target;
                            if (!(target && !(/^\w+:\/\//g.test(target)))) return [3 /*break*/, 3];
                            return [4 /*yield*/, this.resolveRelativeReference(target, document.uri, documentContext)];
                        case 2:
                            resolvedTarget = _a.sent();
                            if (resolvedTarget !== undefined) {
                                link.target = resolvedTarget;
                                resolvedLinks.push(link);
                            }
                            return [3 /*break*/, 4];
                        case 3:
                            resolvedLinks.push(link);
                            _a.label = 4;
                        case 4:
                            _i++;
                            return [3 /*break*/, 1];
                        case 5: return [2 /*return*/, resolvedLinks];
                    }
                });
            });
        };
        CSSNavigation.prototype.findUnresolvedLinks = function (document, stylesheet) {
            var _this = this;
            var result = [];
            var collect = function (uriStringNode) {
                var rawUri = uriStringNode.getText();
                var range = getRange(uriStringNode, document);
                // Make sure the range is not empty
                if (range.start.line === range.end.line && range.start.character === range.end.character) {
                    return;
                }
                if (strings_1.startsWith(rawUri, "'") || strings_1.startsWith(rawUri, "\"")) {
                    rawUri = rawUri.slice(1, -1);
                }
                result.push({ target: rawUri, range: range });
            };
            stylesheet.accept(function (candidate) {
                if (candidate.type === nodes.NodeType.URILiteral) {
                    var first = candidate.getChild(0);
                    if (first) {
                        collect(first);
                    }
                    return false;
                }
                /**
                 * In @import, it is possible to include links that do not use `url()`
                 * For example, `@import 'foo.css';`
                 */
                if (candidate.parent && _this.isRawStringDocumentLinkNode(candidate.parent)) {
                    var rawText = candidate.getText();
                    if (strings_1.startsWith(rawText, "'") || strings_1.startsWith(rawText, "\"")) {
                        collect(candidate);
                    }
                    return false;
                }
                return true;
            });
            return result;
        };
        CSSNavigation.prototype.findDocumentSymbols = function (document, stylesheet) {
            var result = [];
            stylesheet.accept(function (node) {
                var entry = {
                    name: null,
                    kind: cssLanguageTypes_1.SymbolKind.Class,
                    location: null
                };
                var locationNode = node;
                if (node instanceof nodes.Selector) {
                    entry.name = node.getText();
                    locationNode = node.findAParent(nodes.NodeType.Ruleset, nodes.NodeType.ExtendsReference);
                    if (locationNode) {
                        entry.location = cssLanguageTypes_1.Location.create(document.uri, getRange(locationNode, document));
                        result.push(entry);
                    }
                    return false;
                }
                else if (node instanceof nodes.VariableDeclaration) {
                    entry.name = node.getName();
                    entry.kind = cssLanguageTypes_1.SymbolKind.Variable;
                }
                else if (node instanceof nodes.MixinDeclaration) {
                    entry.name = node.getName();
                    entry.kind = cssLanguageTypes_1.SymbolKind.Method;
                }
                else if (node instanceof nodes.FunctionDeclaration) {
                    entry.name = node.getName();
                    entry.kind = cssLanguageTypes_1.SymbolKind.Function;
                }
                else if (node instanceof nodes.Keyframe) {
                    entry.name = localize('literal.keyframes', "@keyframes {0}", node.getName());
                }
                else if (node instanceof nodes.FontFace) {
                    entry.name = localize('literal.fontface', "@font-face");
                }
                else if (node instanceof nodes.Media) {
                    var mediaList = node.getChild(0);
                    if (mediaList instanceof nodes.Medialist) {
                        entry.name = '@media ' + mediaList.getText();
                        entry.kind = cssLanguageTypes_1.SymbolKind.Module;
                    }
                }
                if (entry.name) {
                    entry.location = cssLanguageTypes_1.Location.create(document.uri, getRange(locationNode, document));
                    result.push(entry);
                }
                return true;
            });
            return result;
        };
        CSSNavigation.prototype.findDocumentColors = function (document, stylesheet) {
            var result = [];
            stylesheet.accept(function (node) {
                var colorInfo = getColorInformation(node, document);
                if (colorInfo) {
                    result.push(colorInfo);
                }
                return true;
            });
            return result;
        };
        CSSNavigation.prototype.getColorPresentations = function (document, stylesheet, color, range) {
            var result = [];
            var red256 = Math.round(color.red * 255), green256 = Math.round(color.green * 255), blue256 = Math.round(color.blue * 255);
            var label;
            if (color.alpha === 1) {
                label = "rgb(" + red256 + ", " + green256 + ", " + blue256 + ")";
            }
            else {
                label = "rgba(" + red256 + ", " + green256 + ", " + blue256 + ", " + color.alpha + ")";
            }
            result.push({ label: label, textEdit: cssLanguageTypes_1.TextEdit.replace(range, label) });
            if (color.alpha === 1) {
                label = "#" + toTwoDigitHex(red256) + toTwoDigitHex(green256) + toTwoDigitHex(blue256);
            }
            else {
                label = "#" + toTwoDigitHex(red256) + toTwoDigitHex(green256) + toTwoDigitHex(blue256) + toTwoDigitHex(Math.round(color.alpha * 255));
            }
            result.push({ label: label, textEdit: cssLanguageTypes_1.TextEdit.replace(range, label) });
            var hsl = facts_1.hslFromColor(color);
            if (hsl.a === 1) {
                label = "hsl(" + hsl.h + ", " + Math.round(hsl.s * 100) + "%, " + Math.round(hsl.l * 100) + "%)";
            }
            else {
                label = "hsla(" + hsl.h + ", " + Math.round(hsl.s * 100) + "%, " + Math.round(hsl.l * 100) + "%, " + hsl.a + ")";
            }
            result.push({ label: label, textEdit: cssLanguageTypes_1.TextEdit.replace(range, label) });
            return result;
        };
        CSSNavigation.prototype.doRename = function (document, position, newName, stylesheet) {
            var _a;
            var highlights = this.findDocumentHighlights(document, position, stylesheet);
            var edits = highlights.map(function (h) { return cssLanguageTypes_1.TextEdit.replace(h.range, newName); });
            return {
                changes: (_a = {}, _a[document.uri] = edits, _a)
            };
        };
        CSSNavigation.prototype.resolveRelativeReference = function (ref, documentUri, documentContext) {
            return __awaiter(this, void 0, void 0, function () {
                var moduleName, rootFolderUri, documentFolderUri, modulePath, pathWithinModule;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            if (!(ref[0] === '~' && ref[1] !== '/' && this.fileSystemProvider)) return [3 /*break*/, 3];
                            ref = ref.substring(1);
                            if (!strings_1.startsWith(documentUri, 'file://')) return [3 /*break*/, 2];
                            moduleName = getModuleNameFromPath(ref);
                            rootFolderUri = documentContext.resolveReference('/', documentUri);
                            documentFolderUri = resources_1.dirname(documentUri);
                            return [4 /*yield*/, this.resolvePathToModule(moduleName, documentFolderUri, rootFolderUri)];
                        case 1:
                            modulePath = _a.sent();
                            if (modulePath) {
                                pathWithinModule = ref.substring(moduleName.length + 1);
                                return [2 /*return*/, resources_1.joinPath(modulePath, pathWithinModule)];
                            }
                            _a.label = 2;
                        case 2: return [2 /*return*/, documentContext.resolveReference(ref, documentUri)];
                        case 3: return [2 /*return*/, documentContext.resolveReference(ref, documentUri)];
                    }
                });
            });
        };
        CSSNavigation.prototype.resolvePathToModule = function (_moduleName, documentFolderUri, rootFolderUri) {
            return __awaiter(this, void 0, void 0, function () {
                var packPath;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            packPath = resources_1.joinPath(documentFolderUri, 'node_modules', _moduleName, 'package.json');
                            return [4 /*yield*/, this.fileExists(packPath)];
                        case 1:
                            if (_a.sent()) {
                                return [2 /*return*/, resources_1.dirname(packPath)];
                            }
                            else if (rootFolderUri && documentFolderUri.startsWith(rootFolderUri) && (documentFolderUri.length !== rootFolderUri.length)) {
                                return [2 /*return*/, this.resolvePathToModule(_moduleName, resources_1.dirname(documentFolderUri), rootFolderUri)];
                            }
                            return [2 /*return*/, undefined];
                    }
                });
            });
        };
        CSSNavigation.prototype.fileExists = function (uri) {
            return __awaiter(this, void 0, void 0, function () {
                var stat, err_1;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            if (!this.fileSystemProvider) {
                                return [2 /*return*/, false];
                            }
                            _a.label = 1;
                        case 1:
                            _a.trys.push([1, 3, , 4]);
                            return [4 /*yield*/, this.fileSystemProvider.stat(uri)];
                        case 2:
                            stat = _a.sent();
                            if (stat.type === cssLanguageTypes_1.FileType.Unknown && stat.size === -1) {
                                return [2 /*return*/, false];
                            }
                            return [2 /*return*/, true];
                        case 3:
                            err_1 = _a.sent();
                            return [2 /*return*/, false];
                        case 4: return [2 /*return*/];
                    }
                });
            });
        };
        return CSSNavigation;
    }());
    exports.CSSNavigation = CSSNavigation;
    function getColorInformation(node, document) {
        var color = facts_1.getColorValue(node);
        if (color) {
            var range = getRange(node, document);
            return { color: color, range: range };
        }
        return null;
    }
    function getRange(node, document) {
        return cssLanguageTypes_1.Range.create(document.positionAt(node.offset), document.positionAt(node.end));
    }
    function getHighlightKind(node) {
        if (node.type === nodes.NodeType.Selector) {
            return cssLanguageTypes_1.DocumentHighlightKind.Write;
        }
        if (node instanceof nodes.Identifier) {
            if (node.parent && node.parent instanceof nodes.Property) {
                if (node.isCustomProperty) {
                    return cssLanguageTypes_1.DocumentHighlightKind.Write;
                }
            }
        }
        if (node.parent) {
            switch (node.parent.type) {
                case nodes.NodeType.FunctionDeclaration:
                case nodes.NodeType.MixinDeclaration:
                case nodes.NodeType.Keyframe:
                case nodes.NodeType.VariableDeclaration:
                case nodes.NodeType.FunctionParameter:
                    return cssLanguageTypes_1.DocumentHighlightKind.Write;
            }
        }
        return cssLanguageTypes_1.DocumentHighlightKind.Read;
    }
    function toTwoDigitHex(n) {
        var r = n.toString(16);
        return r.length !== 2 ? '0' + r : r;
    }
    function getModuleNameFromPath(path) {
        // If a scoped module (starts with @) then get up until second instance of '/', otherwise get until first instance of '/'
        if (path[0] === '@') {
            return path.substring(0, path.indexOf('/', path.indexOf('/') + 1));
        }
        return path.substring(0, path.indexOf('/'));
    }
});

(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/services/lintRules',["require", "exports", "../parser/cssNodes", "vscode-nls"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.LintConfigurationSettings = exports.Settings = exports.Rules = exports.Setting = exports.Rule = void 0;
    var nodes = require("../parser/cssNodes");
    var nls = require("vscode-nls");
    var localize = nls.loadMessageBundle();
    var Warning = nodes.Level.Warning;
    var Error = nodes.Level.Error;
    var Ignore = nodes.Level.Ignore;
    var Rule = /** @class */ (function () {
        function Rule(id, message, defaultValue) {
            this.id = id;
            this.message = message;
            this.defaultValue = defaultValue;
            // nothing to do
        }
        return Rule;
    }());
    exports.Rule = Rule;
    var Setting = /** @class */ (function () {
        function Setting(id, message, defaultValue) {
            this.id = id;
            this.message = message;
            this.defaultValue = defaultValue;
            // nothing to do
        }
        return Setting;
    }());
    exports.Setting = Setting;
    exports.Rules = {
        AllVendorPrefixes: new Rule('compatibleVendorPrefixes', localize('rule.vendorprefixes.all', "When using a vendor-specific prefix make sure to also include all other vendor-specific properties"), Ignore),
        IncludeStandardPropertyWhenUsingVendorPrefix: new Rule('vendorPrefix', localize('rule.standardvendorprefix.all', "When using a vendor-specific prefix also include the standard property"), Warning),
        DuplicateDeclarations: new Rule('duplicateProperties', localize('rule.duplicateDeclarations', "Do not use duplicate style definitions"), Ignore),
        EmptyRuleSet: new Rule('emptyRules', localize('rule.emptyRuleSets', "Do not use empty rulesets"), Warning),
        ImportStatemement: new Rule('importStatement', localize('rule.importDirective', "Import statements do not load in parallel"), Ignore),
        BewareOfBoxModelSize: new Rule('boxModel', localize('rule.bewareOfBoxModelSize', "Do not use width or height when using padding or border"), Ignore),
        UniversalSelector: new Rule('universalSelector', localize('rule.universalSelector', "The universal selector (*) is known to be slow"), Ignore),
        ZeroWithUnit: new Rule('zeroUnits', localize('rule.zeroWidthUnit', "No unit for zero needed"), Ignore),
        RequiredPropertiesForFontFace: new Rule('fontFaceProperties', localize('rule.fontFaceProperties', "@font-face rule must define 'src' and 'font-family' properties"), Warning),
        HexColorLength: new Rule('hexColorLength', localize('rule.hexColor', "Hex colors must consist of three, four, six or eight hex numbers"), Error),
        ArgsInColorFunction: new Rule('argumentsInColorFunction', localize('rule.colorFunction', "Invalid number of parameters"), Error),
        UnknownProperty: new Rule('unknownProperties', localize('rule.unknownProperty', "Unknown property."), Warning),
        UnknownAtRules: new Rule('unknownAtRules', localize('rule.unknownAtRules', "Unknown at-rule."), Warning),
        IEStarHack: new Rule('ieHack', localize('rule.ieHack', "IE hacks are only necessary when supporting IE7 and older"), Ignore),
        UnknownVendorSpecificProperty: new Rule('unknownVendorSpecificProperties', localize('rule.unknownVendorSpecificProperty', "Unknown vendor specific property."), Ignore),
        PropertyIgnoredDueToDisplay: new Rule('propertyIgnoredDueToDisplay', localize('rule.propertyIgnoredDueToDisplay', "Property is ignored due to the display."), Warning),
        AvoidImportant: new Rule('important', localize('rule.avoidImportant', "Avoid using !important. It is an indication that the specificity of the entire CSS has gotten out of control and needs to be refactored."), Ignore),
        AvoidFloat: new Rule('float', localize('rule.avoidFloat', "Avoid using 'float'. Floats lead to fragile CSS that is easy to break if one aspect of the layout changes."), Ignore),
        AvoidIdSelector: new Rule('idSelector', localize('rule.avoidIdSelector', "Selectors should not contain IDs because these rules are too tightly coupled with the HTML."), Ignore),
    };
    exports.Settings = {
        ValidProperties: new Setting('validProperties', localize('rule.validProperties', "A list of properties that are not validated against the `unknownProperties` rule."), [])
    };
    var LintConfigurationSettings = /** @class */ (function () {
        function LintConfigurationSettings(conf) {
            if (conf === void 0) { conf = {}; }
            this.conf = conf;
        }
        LintConfigurationSettings.prototype.getRule = function (rule) {
            if (this.conf.hasOwnProperty(rule.id)) {
                var level = toLevel(this.conf[rule.id]);
                if (level) {
                    return level;
                }
            }
            return rule.defaultValue;
        };
        LintConfigurationSettings.prototype.getSetting = function (setting) {
            return this.conf[setting.id];
        };
        return LintConfigurationSettings;
    }());
    exports.LintConfigurationSettings = LintConfigurationSettings;
    function toLevel(level) {
        switch (level) {
            case 'ignore': return nodes.Level.Ignore;
            case 'warning': return nodes.Level.Warning;
            case 'error': return nodes.Level.Error;
        }
        return null;
    }
});

(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/services/cssCodeActions',["require", "exports", "../parser/cssNodes", "../utils/strings", "../services/lintRules", "../cssLanguageTypes", "vscode-nls"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.CSSCodeActions = void 0;
    var nodes = require("../parser/cssNodes");
    var strings_1 = require("../utils/strings");
    var lintRules_1 = require("../services/lintRules");
    var cssLanguageTypes_1 = require("../cssLanguageTypes");
    var nls = require("vscode-nls");
    var localize = nls.loadMessageBundle();
    var CSSCodeActions = /** @class */ (function () {
        function CSSCodeActions(cssDataManager) {
            this.cssDataManager = cssDataManager;
        }
        CSSCodeActions.prototype.doCodeActions = function (document, range, context, stylesheet) {
            return this.doCodeActions2(document, range, context, stylesheet).map(function (ca) {
                var textDocumentEdit = ca.edit && ca.edit.documentChanges && ca.edit.documentChanges[0];
                return cssLanguageTypes_1.Command.create(ca.title, '_css.applyCodeAction', document.uri, document.version, textDocumentEdit && textDocumentEdit.edits);
            });
        };
        CSSCodeActions.prototype.doCodeActions2 = function (document, range, context, stylesheet) {
            var result = [];
            if (context.diagnostics) {
                for (var _i = 0, _a = context.diagnostics; _i < _a.length; _i++) {
                    var diagnostic = _a[_i];
                    this.appendFixesForMarker(document, stylesheet, diagnostic, result);
                }
            }
            return result;
        };
        CSSCodeActions.prototype.getFixesForUnknownProperty = function (document, property, marker, result) {
            var propertyName = property.getName();
            var candidates = [];
            this.cssDataManager.getProperties().forEach(function (p) {
                var score = strings_1.difference(propertyName, p.name);
                if (score >= propertyName.length / 2 /*score_lim*/) {
                    candidates.push({ property: p.name, score: score });
                }
            });
            // Sort in descending order.
            candidates.sort(function (a, b) {
                return b.score - a.score || a.property.localeCompare(b.property);
            });
            var maxActions = 3;
            for (var _i = 0, candidates_1 = candidates; _i < candidates_1.length; _i++) {
                var candidate = candidates_1[_i];
                var propertyName_1 = candidate.property;
                var title = localize('css.codeaction.rename', "Rename to '{0}'", propertyName_1);
                var edit = cssLanguageTypes_1.TextEdit.replace(marker.range, propertyName_1);
                var documentIdentifier = cssLanguageTypes_1.VersionedTextDocumentIdentifier.create(document.uri, document.version);
                var workspaceEdit = { documentChanges: [cssLanguageTypes_1.TextDocumentEdit.create(documentIdentifier, [edit])] };
                var codeAction = cssLanguageTypes_1.CodeAction.create(title, workspaceEdit, cssLanguageTypes_1.CodeActionKind.QuickFix);
                codeAction.diagnostics = [marker];
                result.push(codeAction);
                if (--maxActions <= 0) {
                    return;
                }
            }
        };
        CSSCodeActions.prototype.appendFixesForMarker = function (document, stylesheet, marker, result) {
            if (marker.code !== lintRules_1.Rules.UnknownProperty.id) {
                return;
            }
            var offset = document.offsetAt(marker.range.start);
            var end = document.offsetAt(marker.range.end);
            var nodepath = nodes.getNodePath(stylesheet, offset);
            for (var i = nodepath.length - 1; i >= 0; i--) {
                var node = nodepath[i];
                if (node instanceof nodes.Declaration) {
                    var property = node.getProperty();
                    if (property && property.offset === offset && property.end === end) {
                        this.getFixesForUnknownProperty(document, property, marker, result);
                        return;
                    }
                }
            }
        };
        return CSSCodeActions;
    }());
    exports.CSSCodeActions = CSSCodeActions;
});

(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/services/lintUtil',["require", "exports", "../utils/arrays"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.Element = void 0;
    var arrays_1 = require("../utils/arrays");
    var Element = /** @class */ (function () {
        function Element(decl) {
            this.fullPropertyName = decl.getFullPropertyName().toLowerCase();
            this.node = decl;
        }
        return Element;
    }());
    exports.Element = Element;
    function setSide(model, side, value, property) {
        var state = model[side];
        state.value = value;
        if (value) {
            if (!arrays_1.includes(state.properties, property)) {
                state.properties.push(property);
            }
        }
    }
    function setAllSides(model, value, property) {
        setSide(model, 'top', value, property);
        setSide(model, 'right', value, property);
        setSide(model, 'bottom', value, property);
        setSide(model, 'left', value, property);
    }
    function updateModelWithValue(model, side, value, property) {
        if (side === 'top' || side === 'right' ||
            side === 'bottom' || side === 'left') {
            setSide(model, side, value, property);
        }
        else {
            setAllSides(model, value, property);
        }
    }
    function updateModelWithList(model, values, property) {
        switch (values.length) {
            case 1:
                updateModelWithValue(model, undefined, values[0], property);
                break;
            case 2:
                updateModelWithValue(model, 'top', values[0], property);
                updateModelWithValue(model, 'bottom', values[0], property);
                updateModelWithValue(model, 'right', values[1], property);
                updateModelWithValue(model, 'left', values[1], property);
                break;
            case 3:
                updateModelWithValue(model, 'top', values[0], property);
                updateModelWithValue(model, 'right', values[1], property);
                updateModelWithValue(model, 'left', values[1], property);
                updateModelWithValue(model, 'bottom', values[2], property);
                break;
            case 4:
                updateModelWithValue(model, 'top', values[0], property);
                updateModelWithValue(model, 'right', values[1], property);
                updateModelWithValue(model, 'bottom', values[2], property);
                updateModelWithValue(model, 'left', values[3], property);
                break;
        }
    }
    function matches(value, candidates) {
        for (var _i = 0, candidates_1 = candidates; _i < candidates_1.length; _i++) {
            var candidate = candidates_1[_i];
            if (value.matches(candidate)) {
                return true;
            }
        }
        return false;
    }
    /**
     * @param allowsKeywords whether the initial value of property is zero, so keywords `initial` and `unset` count as zero
     * @return `true` if this node represents a non-zero border; otherwise, `false`
     */
    function checkLineWidth(value, allowsKeywords) {
        if (allowsKeywords === void 0) { allowsKeywords = true; }
        if (allowsKeywords && matches(value, ['initial', 'unset'])) {
            return false;
        }
        // a <length> is a value and a unit
        // so use `parseFloat` to strip the unit
        return parseFloat(value.getText()) !== 0;
    }
    function checkLineWidthList(nodes, allowsKeywords) {
        if (allowsKeywords === void 0) { allowsKeywords = true; }
        return nodes.map(function (node) { return checkLineWidth(node, allowsKeywords); });
    }
    /**
     * @param allowsKeywords whether keywords `initial` and `unset` count as zero
     * @return `true` if this node represents a non-zero border; otherwise, `false`
     */
    function checkLineStyle(valueNode, allowsKeywords) {
        if (allowsKeywords === void 0) { allowsKeywords = true; }
        if (matches(valueNode, ['none', 'hidden'])) {
            return false;
        }
        if (allowsKeywords && matches(valueNode, ['initial', 'unset'])) {
            return false;
        }
        return true;
    }
    function checkLineStyleList(nodes, allowsKeywords) {
        if (allowsKeywords === void 0) { allowsKeywords = true; }
        return nodes.map(function (node) { return checkLineStyle(node, allowsKeywords); });
    }
    function checkBorderShorthand(node) {
        var children = node.getChildren();
        // the only child can be a keyword, a <line-width>, or a <line-style>
        // if either check returns false, the result is no border
        if (children.length === 1) {
            var value = children[0];
            return checkLineWidth(value) && checkLineStyle(value);
        }
        // multiple children can't contain keywords
        // if any child means no border, the result is no border
        for (var _i = 0, children_1 = children; _i < children_1.length; _i++) {
            var child = children_1[_i];
            var value = child;
            if (!checkLineWidth(value, /* allowsKeywords: */ false) ||
                !checkLineStyle(value, /* allowsKeywords: */ false)) {
                return false;
            }
        }
        return true;
    }
    function calculateBoxModel(propertyTable) {
        var model = {
            top: { value: false, properties: [] },
            right: { value: false, properties: [] },
            bottom: { value: false, properties: [] },
            left: { value: false, properties: [] },
        };
        for (var _i = 0, propertyTable_1 = propertyTable; _i < propertyTable_1.length; _i++) {
            var property = propertyTable_1[_i];
            var value = property.node.value;
            if (typeof value === 'undefined') {
                continue;
            }
            switch (property.fullPropertyName) {
                case 'box-sizing':
                    // has `box-sizing`, bail out
                    return {
                        top: { value: false, properties: [] },
                        right: { value: false, properties: [] },
                        bottom: { value: false, properties: [] },
                        left: { value: false, properties: [] },
                    };
                case 'width':
                    model.width = property;
                    break;
                case 'height':
                    model.height = property;
                    break;
                default:
                    var segments = property.fullPropertyName.split('-');
                    switch (segments[0]) {
                        case 'border':
                            switch (segments[1]) {
                                case undefined:
                                case 'top':
                                case 'right':
                                case 'bottom':
                                case 'left':
                                    switch (segments[2]) {
                                        case undefined:
                                            updateModelWithValue(model, segments[1], checkBorderShorthand(value), property);
                                            break;
                                        case 'width':
                                            // the initial value of `border-width` is `medium`, not zero
                                            updateModelWithValue(model, segments[1], checkLineWidth(value, false), property);
                                            break;
                                        case 'style':
                                            // the initial value of `border-style` is `none`
                                            updateModelWithValue(model, segments[1], checkLineStyle(value, true), property);
                                            break;
                                    }
                                    break;
                                case 'width':
                                    // the initial value of `border-width` is `medium`, not zero
                                    updateModelWithList(model, checkLineWidthList(value.getChildren(), false), property);
                                    break;
                                case 'style':
                                    // the initial value of `border-style` is `none`
                                    updateModelWithList(model, checkLineStyleList(value.getChildren(), true), property);
                                    break;
                            }
                            break;
                        case 'padding':
                            if (segments.length === 1) {
                                // the initial value of `padding` is zero
                                updateModelWithList(model, checkLineWidthList(value.getChildren(), true), property);
                            }
                            else {
                                // the initial value of `padding` is zero
                                updateModelWithValue(model, segments[1], checkLineWidth(value, true), property);
                            }
                            break;
                    }
                    break;
            }
        }
        return model;
    }
    exports.default = calculateBoxModel;
});

(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/services/lint',["require", "exports", "vscode-nls", "../languageFacts/facts", "../parser/cssNodes", "../utils/arrays", "./lintRules", "./lintUtil"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.LintVisitor = void 0;
    var nls = require("vscode-nls");
    var languageFacts = require("../languageFacts/facts");
    var nodes = require("../parser/cssNodes");
    var arrays_1 = require("../utils/arrays");
    var lintRules_1 = require("./lintRules");
    var lintUtil_1 = require("./lintUtil");
    var localize = nls.loadMessageBundle();
    var NodesByRootMap = /** @class */ (function () {
        function NodesByRootMap() {
            this.data = {};
        }
        NodesByRootMap.prototype.add = function (root, name, node) {
            var entry = this.data[root];
            if (!entry) {
                entry = { nodes: [], names: [] };
                this.data[root] = entry;
            }
            entry.names.push(name);
            if (node) {
                entry.nodes.push(node);
            }
        };
        return NodesByRootMap;
    }());
    var LintVisitor = /** @class */ (function () {
        function LintVisitor(document, settings, cssDataManager) {
            var _this = this;
            this.cssDataManager = cssDataManager;
            this.warnings = [];
            this.settings = settings;
            this.documentText = document.getText();
            this.keyframes = new NodesByRootMap();
            this.validProperties = {};
            var properties = settings.getSetting(lintRules_1.Settings.ValidProperties);
            if (Array.isArray(properties)) {
                properties.forEach(function (p) {
                    if (typeof p === 'string') {
                        var name = p.trim().toLowerCase();
                        if (name.length) {
                            _this.validProperties[name] = true;
                        }
                    }
                });
            }
        }
        LintVisitor.entries = function (node, document, settings, cssDataManager, entryFilter) {
            var visitor = new LintVisitor(document, settings, cssDataManager);
            node.acceptVisitor(visitor);
            visitor.completeValidations();
            return visitor.getEntries(entryFilter);
        };
        LintVisitor.prototype.isValidPropertyDeclaration = function (element) {
            var propertyName = element.fullPropertyName;
            return this.validProperties[propertyName];
        };
        LintVisitor.prototype.fetch = function (input, s) {
            var elements = [];
            for (var _i = 0, input_1 = input; _i < input_1.length; _i++) {
                var curr = input_1[_i];
                if (curr.fullPropertyName === s) {
                    elements.push(curr);
                }
            }
            return elements;
        };
        LintVisitor.prototype.fetchWithValue = function (input, s, v) {
            var elements = [];
            for (var _i = 0, input_2 = input; _i < input_2.length; _i++) {
                var inputElement = input_2[_i];
                if (inputElement.fullPropertyName === s) {
                    var expression = inputElement.node.getValue();
                    if (expression && this.findValueInExpression(expression, v)) {
                        elements.push(inputElement);
                    }
                }
            }
            return elements;
        };
        LintVisitor.prototype.findValueInExpression = function (expression, v) {
            var found = false;
            expression.accept(function (node) {
                if (node.type === nodes.NodeType.Identifier && node.matches(v)) {
                    found = true;
                }
                return !found;
            });
            return found;
        };
        LintVisitor.prototype.getEntries = function (filter) {
            if (filter === void 0) { filter = (nodes.Level.Warning | nodes.Level.Error); }
            return this.warnings.filter(function (entry) {
                return (entry.getLevel() & filter) !== 0;
            });
        };
        LintVisitor.prototype.addEntry = function (node, rule, details) {
            var entry = new nodes.Marker(node, rule, this.settings.getRule(rule), details);
            this.warnings.push(entry);
        };
        LintVisitor.prototype.getMissingNames = function (expected, actual) {
            var expectedClone = expected.slice(0); // clone
            for (var i = 0; i < actual.length; i++) {
                var k = expectedClone.indexOf(actual[i]);
                if (k !== -1) {
                    expectedClone[k] = null;
                }
            }
            var result = null;
            for (var i = 0; i < expectedClone.length; i++) {
                var curr = expectedClone[i];
                if (curr) {
                    if (result === null) {
                        result = localize('namelist.single', "'{0}'", curr);
                    }
                    else {
                        result = localize('namelist.concatenated', "{0}, '{1}'", result, curr);
                    }
                }
            }
            return result;
        };
        LintVisitor.prototype.visitNode = function (node) {
            switch (node.type) {
                case nodes.NodeType.UnknownAtRule:
                    return this.visitUnknownAtRule(node);
                case nodes.NodeType.Keyframe:
                    return this.visitKeyframe(node);
                case nodes.NodeType.FontFace:
                    return this.visitFontFace(node);
                case nodes.NodeType.Ruleset:
                    return this.visitRuleSet(node);
                case nodes.NodeType.SimpleSelector:
                    return this.visitSimpleSelector(node);
                case nodes.NodeType.Function:
                    return this.visitFunction(node);
                case nodes.NodeType.NumericValue:
                    return this.visitNumericValue(node);
                case nodes.NodeType.Import:
                    return this.visitImport(node);
                case nodes.NodeType.HexColorValue:
                    return this.visitHexColorValue(node);
                case nodes.NodeType.Prio:
                    return this.visitPrio(node);
                case nodes.NodeType.IdentifierSelector:
                    return this.visitIdentifierSelector(node);
            }
            return true;
        };
        LintVisitor.prototype.completeValidations = function () {
            this.validateKeyframes();
        };
        LintVisitor.prototype.visitUnknownAtRule = function (node) {
            var atRuleName = node.getChild(0);
            if (!atRuleName) {
                return false;
            }
            var atDirective = this.cssDataManager.getAtDirective(atRuleName.getText());
            if (atDirective) {
                return false;
            }
            this.addEntry(atRuleName, lintRules_1.Rules.UnknownAtRules, "Unknown at rule " + atRuleName.getText());
            return true;
        };
        LintVisitor.prototype.visitKeyframe = function (node) {
            var keyword = node.getKeyword();
            if (!keyword) {
                return false;
            }
            var text = keyword.getText();
            this.keyframes.add(node.getName(), text, (text !== '@keyframes') ? keyword : null);
            return true;
        };
        LintVisitor.prototype.validateKeyframes = function () {
            // @keyframe and it's vendor specific alternatives
            // @keyframe should be included
            var expected = ['@-webkit-keyframes', '@-moz-keyframes', '@-o-keyframes'];
            for (var name in this.keyframes.data) {
                var actual = this.keyframes.data[name].names;
                var needsStandard = (actual.indexOf('@keyframes') === -1);
                if (!needsStandard && actual.length === 1) {
                    continue; // only the non-vendor specific keyword is used, that's fine, no warning
                }
                var missingVendorSpecific = this.getMissingNames(expected, actual);
                if (missingVendorSpecific || needsStandard) {
                    for (var _i = 0, _a = this.keyframes.data[name].nodes; _i < _a.length; _i++) {
                        var node = _a[_i];
                        if (needsStandard) {
                            var message = localize('keyframes.standardrule.missing', "Always define standard rule '@keyframes' when defining keyframes.");
                            this.addEntry(node, lintRules_1.Rules.IncludeStandardPropertyWhenUsingVendorPrefix, message);
                        }
                        if (missingVendorSpecific) {
                            var message = localize('keyframes.vendorspecific.missing', "Always include all vendor specific rules: Missing: {0}", missingVendorSpecific);
                            this.addEntry(node, lintRules_1.Rules.AllVendorPrefixes, message);
                        }
                    }
                }
            }
            return true;
        };
        LintVisitor.prototype.visitSimpleSelector = function (node) {
            /////////////////////////////////////////////////////////////
            //	Lint - The universal selector (*) is known to be slow.
            /////////////////////////////////////////////////////////////
            var firstChar = this.documentText.charAt(node.offset);
            if (node.length === 1 && firstChar === '*') {
                this.addEntry(node, lintRules_1.Rules.UniversalSelector);
            }
            return true;
        };
        LintVisitor.prototype.visitIdentifierSelector = function (node) {
            /////////////////////////////////////////////////////////////
            //	Lint - Avoid id selectors
            /////////////////////////////////////////////////////////////
            this.addEntry(node, lintRules_1.Rules.AvoidIdSelector);
            return true;
        };
        LintVisitor.prototype.visitImport = function (node) {
            /////////////////////////////////////////////////////////////
            //	Lint - Import statements shouldn't be used, because they aren't offering parallel downloads.
            /////////////////////////////////////////////////////////////
            this.addEntry(node, lintRules_1.Rules.ImportStatemement);
            return true;
        };
        LintVisitor.prototype.visitRuleSet = function (node) {
            /////////////////////////////////////////////////////////////
            //	Lint - Don't use empty rulesets.
            /////////////////////////////////////////////////////////////
            var declarations = node.getDeclarations();
            if (!declarations) {
                // syntax error
                return false;
            }
            if (!declarations.hasChildren()) {
                this.addEntry(node.getSelectors(), lintRules_1.Rules.EmptyRuleSet);
            }
            var propertyTable = [];
            for (var _i = 0, _a = declarations.getChildren(); _i < _a.length; _i++) {
                var element = _a[_i];
                if (element instanceof nodes.Declaration) {
                    propertyTable.push(new lintUtil_1.Element(element));
                }
            }
            /////////////////////////////////////////////////////////////
            // the rule warns when it finds:
            // width being used with border, border-left, border-right, padding, padding-left, or padding-right
            // height being used with border, border-top, border-bottom, padding, padding-top, or padding-bottom
            // No error when box-sizing property is specified, as it assumes the user knows what he's doing.
            // see https://github.com/CSSLint/csslint/wiki/Beware-of-box-model-size
            /////////////////////////////////////////////////////////////
            var boxModel = lintUtil_1.default(propertyTable);
            if (boxModel.width) {
                var properties = [];
                if (boxModel.right.value) {
                    properties = arrays_1.union(properties, boxModel.right.properties);
                }
                if (boxModel.left.value) {
                    properties = arrays_1.union(properties, boxModel.left.properties);
                }
                if (properties.length !== 0) {
                    for (var _b = 0, properties_1 = properties; _b < properties_1.length; _b++) {
                        var item = properties_1[_b];
                        this.addEntry(item.node, lintRules_1.Rules.BewareOfBoxModelSize);
                    }
                    this.addEntry(boxModel.width.node, lintRules_1.Rules.BewareOfBoxModelSize);
                }
            }
            if (boxModel.height) {
                var properties = [];
                if (boxModel.top.value) {
                    properties = arrays_1.union(properties, boxModel.top.properties);
                }
                if (boxModel.bottom.value) {
                    properties = arrays_1.union(properties, boxModel.bottom.properties);
                }
                if (properties.length !== 0) {
                    for (var _c = 0, properties_2 = properties; _c < properties_2.length; _c++) {
                        var item = properties_2[_c];
                        this.addEntry(item.node, lintRules_1.Rules.BewareOfBoxModelSize);
                    }
                    this.addEntry(boxModel.height.node, lintRules_1.Rules.BewareOfBoxModelSize);
                }
            }
            /////////////////////////////////////////////////////////////
            //	Properties ignored due to display
            /////////////////////////////////////////////////////////////
            // With 'display: inline-block', 'float' has no effect
            var displayElems = this.fetchWithValue(propertyTable, 'display', 'inline-block');
            if (displayElems.length > 0) {
                var elem = this.fetch(propertyTable, 'float');
                for (var index = 0; index < elem.length; index++) {
                    var node_1 = elem[index].node;
                    var value = node_1.getValue();
                    if (value && !value.matches('none')) {
                        this.addEntry(node_1, lintRules_1.Rules.PropertyIgnoredDueToDisplay, localize('rule.propertyIgnoredDueToDisplayInlineBlock', "inline-block is ignored due to the float. If 'float' has a value other than 'none', the box is floated and 'display' is treated as 'block'"));
                    }
                }
            }
            // With 'display: block', 'vertical-align' has no effect
            displayElems = this.fetchWithValue(propertyTable, 'display', 'block');
            if (displayElems.length > 0) {
                var elem = this.fetch(propertyTable, 'vertical-align');
                for (var index = 0; index < elem.length; index++) {
                    this.addEntry(elem[index].node, lintRules_1.Rules.PropertyIgnoredDueToDisplay, localize('rule.propertyIgnoredDueToDisplayBlock', "Property is ignored due to the display. With 'display: block', vertical-align should not be used."));
                }
            }
            /////////////////////////////////////////////////////////////
            //	Avoid 'float'
            /////////////////////////////////////////////////////////////
            var elements = this.fetch(propertyTable, 'float');
            for (var index = 0; index < elements.length; index++) {
                var element = elements[index];
                if (!this.isValidPropertyDeclaration(element)) {
                    this.addEntry(element.node, lintRules_1.Rules.AvoidFloat);
                }
            }
            /////////////////////////////////////////////////////////////
            //	Don't use duplicate declarations.
            /////////////////////////////////////////////////////////////
            for (var i = 0; i < propertyTable.length; i++) {
                var element = propertyTable[i];
                if (element.fullPropertyName !== 'background' && !this.validProperties[element.fullPropertyName]) {
                    var value = element.node.getValue();
                    if (value && this.documentText.charAt(value.offset) !== '-') {
                        var elements_1 = this.fetch(propertyTable, element.fullPropertyName);
                        if (elements_1.length > 1) {
                            for (var k = 0; k < elements_1.length; k++) {
                                var value_1 = elements_1[k].node.getValue();
                                if (value_1 && this.documentText.charAt(value_1.offset) !== '-' && elements_1[k] !== element) {
                                    this.addEntry(element.node, lintRules_1.Rules.DuplicateDeclarations);
                                }
                            }
                        }
                    }
                }
            }
            /////////////////////////////////////////////////////////////
            //	Unknown propery & When using a vendor-prefixed gradient, make sure to use them all.
            /////////////////////////////////////////////////////////////
            var isExportBlock = node.getSelectors().matches(":export");
            if (!isExportBlock) {
                var propertiesBySuffix = new NodesByRootMap();
                var containsUnknowns = false;
                for (var _d = 0, propertyTable_1 = propertyTable; _d < propertyTable_1.length; _d++) {
                    var element = propertyTable_1[_d];
                    var decl = element.node;
                    if (this.isCSSDeclaration(decl)) {
                        var name = element.fullPropertyName;
                        var firstChar = name.charAt(0);
                        if (firstChar === '-') {
                            if (name.charAt(1) !== '-') { // avoid css variables
                                if (!this.cssDataManager.isKnownProperty(name) && !this.validProperties[name]) {
                                    this.addEntry(decl.getProperty(), lintRules_1.Rules.UnknownVendorSpecificProperty);
                                }
                                var nonPrefixedName = decl.getNonPrefixedPropertyName();
                                propertiesBySuffix.add(nonPrefixedName, name, decl.getProperty());
                            }
                        }
                        else {
                            var fullName = name;
                            if (firstChar === '*' || firstChar === '_') {
                                this.addEntry(decl.getProperty(), lintRules_1.Rules.IEStarHack);
                                name = name.substr(1);
                            }
                            // _property and *property might be contributed via custom data
                            if (!this.cssDataManager.isKnownProperty(fullName) && !this.cssDataManager.isKnownProperty(name)) {
                                if (!this.validProperties[name]) {
                                    this.addEntry(decl.getProperty(), lintRules_1.Rules.UnknownProperty, localize('property.unknownproperty.detailed', "Unknown property: '{0}'", decl.getFullPropertyName()));
                                }
                            }
                            propertiesBySuffix.add(name, name, null); // don't pass the node as we don't show errors on the standard
                        }
                    }
                    else {
                        containsUnknowns = true;
                    }
                }
                if (!containsUnknowns) { // don't perform this test if there are
                    for (var suffix in propertiesBySuffix.data) {
                        var entry = propertiesBySuffix.data[suffix];
                        var actual = entry.names;
                        var needsStandard = this.cssDataManager.isStandardProperty(suffix) && (actual.indexOf(suffix) === -1);
                        if (!needsStandard && actual.length === 1) {
                            continue; // only the non-vendor specific rule is used, that's fine, no warning
                        }
                        var expected = [];
                        for (var i = 0, len = LintVisitor.prefixes.length; i < len; i++) {
                            var prefix = LintVisitor.prefixes[i];
                            if (this.cssDataManager.isStandardProperty(prefix + suffix)) {
                                expected.push(prefix + suffix);
                            }
                        }
                        var missingVendorSpecific = this.getMissingNames(expected, actual);
                        if (missingVendorSpecific || needsStandard) {
                            for (var _e = 0, _f = entry.nodes; _e < _f.length; _e++) {
                                var node_2 = _f[_e];
                                if (needsStandard) {
                                    var message = localize('property.standard.missing', "Also define the standard property '{0}' for compatibility", suffix);
                                    this.addEntry(node_2, lintRules_1.Rules.IncludeStandardPropertyWhenUsingVendorPrefix, message);
                                }
                                if (missingVendorSpecific) {
                                    var message = localize('property.vendorspecific.missing', "Always include all vendor specific properties: Missing: {0}", missingVendorSpecific);
                                    this.addEntry(node_2, lintRules_1.Rules.AllVendorPrefixes, message);
                                }
                            }
                        }
                    }
                }
            }
            return true;
        };
        LintVisitor.prototype.visitPrio = function (node) {
            /////////////////////////////////////////////////////////////
            //	Don't use !important
            /////////////////////////////////////////////////////////////
            this.addEntry(node, lintRules_1.Rules.AvoidImportant);
            return true;
        };
        LintVisitor.prototype.visitNumericValue = function (node) {
            /////////////////////////////////////////////////////////////
            //	0 has no following unit
            /////////////////////////////////////////////////////////////
            var funcDecl = node.findParent(nodes.NodeType.Function);
            if (funcDecl && funcDecl.getName() === 'calc') {
                return true;
            }
            var decl = node.findParent(nodes.NodeType.Declaration);
            if (decl) {
                var declValue = decl.getValue();
                if (declValue) {
                    var value = node.getValue();
                    if (!value.unit || languageFacts.units.length.indexOf(value.unit.toLowerCase()) === -1) {
                        return true;
                    }
                    if (parseFloat(value.value) === 0.0 && !!value.unit && !this.validProperties[decl.getFullPropertyName()]) {
                        this.addEntry(node, lintRules_1.Rules.ZeroWithUnit);
                    }
                }
            }
            return true;
        };
        LintVisitor.prototype.visitFontFace = function (node) {
            var declarations = node.getDeclarations();
            if (!declarations) {
                // syntax error
                return false;
            }
            var definesSrc = false, definesFontFamily = false;
            var containsUnknowns = false;
            for (var _i = 0, _a = declarations.getChildren(); _i < _a.length; _i++) {
                var node_3 = _a[_i];
                if (this.isCSSDeclaration(node_3)) {
                    var name = node_3.getProperty().getName().toLowerCase();
                    if (name === 'src') {
                        definesSrc = true;
                    }
                    if (name === 'font-family') {
                        definesFontFamily = true;
                    }
                }
                else {
                    containsUnknowns = true;
                }
            }
            if (!containsUnknowns && (!definesSrc || !definesFontFamily)) {
                this.addEntry(node, lintRules_1.Rules.RequiredPropertiesForFontFace);
            }
            return true;
        };
        LintVisitor.prototype.isCSSDeclaration = function (node) {
            if (node instanceof nodes.Declaration) {
                if (!node.getValue()) {
                    return false;
                }
                var property = node.getProperty();
                if (!property) {
                    return false;
                }
                var identifier = property.getIdentifier();
                if (!identifier || identifier.containsInterpolation()) {
                    return false;
                }
                return true;
            }
            return false;
        };
        LintVisitor.prototype.visitHexColorValue = function (node) {
            // Rule: #eeff0011 or #eeff00 or #ef01 or #ef0
            var length = node.length;
            if (length !== 9 && length !== 7 && length !== 5 && length !== 4) {
                this.addEntry(node, lintRules_1.Rules.HexColorLength);
            }
            return false;
        };
        LintVisitor.prototype.visitFunction = function (node) {
            var fnName = node.getName().toLowerCase();
            var expectedAttrCount = -1;
            var actualAttrCount = 0;
            switch (fnName) {
                case 'rgb(':
                case 'hsl(':
                    expectedAttrCount = 3;
                    break;
                case 'rgba(':
                case 'hsla(':
                    expectedAttrCount = 4;
                    break;
            }
            if (expectedAttrCount !== -1) {
                node.getArguments().accept(function (n) {
                    if (n instanceof nodes.BinaryExpression) {
                        actualAttrCount += 1;
                        return false;
                    }
                    return true;
                });
                if (actualAttrCount !== expectedAttrCount) {
                    this.addEntry(node, lintRules_1.Rules.ArgsInColorFunction);
                }
            }
            return true;
        };
        LintVisitor.prefixes = [
            '-ms-', '-moz-', '-o-', '-webkit-',
        ];
        return LintVisitor;
    }());
    exports.LintVisitor = LintVisitor;
});

(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/services/cssValidation',["require", "exports", "../parser/cssNodes", "./lintRules", "./lint", "../cssLanguageTypes"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.CSSValidation = void 0;
    var nodes = require("../parser/cssNodes");
    var lintRules_1 = require("./lintRules");
    var lint_1 = require("./lint");
    var cssLanguageTypes_1 = require("../cssLanguageTypes");
    var CSSValidation = /** @class */ (function () {
        function CSSValidation(cssDataManager) {
            this.cssDataManager = cssDataManager;
        }
        CSSValidation.prototype.configure = function (settings) {
            this.settings = settings;
        };
        CSSValidation.prototype.doValidation = function (document, stylesheet, settings) {
            if (settings === void 0) { settings = this.settings; }
            if (settings && settings.validate === false) {
                return [];
            }
            var entries = [];
            entries.push.apply(entries, nodes.ParseErrorCollector.entries(stylesheet));
            entries.push.apply(entries, lint_1.LintVisitor.entries(stylesheet, document, new lintRules_1.LintConfigurationSettings(settings && settings.lint), this.cssDataManager));
            var ruleIds = [];
            for (var r in lintRules_1.Rules) {
                ruleIds.push(lintRules_1.Rules[r].id);
            }
            function toDiagnostic(marker) {
                var range = cssLanguageTypes_1.Range.create(document.positionAt(marker.getOffset()), document.positionAt(marker.getOffset() + marker.getLength()));
                var source = document.languageId;
                return {
                    code: marker.getRule().id,
                    source: source,
                    message: marker.getMessage(),
                    severity: marker.getLevel() === nodes.Level.Warning ? cssLanguageTypes_1.DiagnosticSeverity.Warning : cssLanguageTypes_1.DiagnosticSeverity.Error,
                    range: range
                };
            }
            return entries.filter(function (entry) { return entry.getLevel() !== nodes.Level.Ignore; }).map(toDiagnostic);
        };
        return CSSValidation;
    }());
    exports.CSSValidation = CSSValidation;
});

var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/parser/scssScanner',["require", "exports", "./cssScanner"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.SCSSScanner = exports.Module = exports.Ellipsis = exports.SmallerEqualsOperator = exports.GreaterEqualsOperator = exports.NotEqualsOperator = exports.EqualsOperator = exports.Default = exports.InterpolationFunction = exports.VariableName = void 0;
    var cssScanner_1 = require("./cssScanner");
    var _FSL = '/'.charCodeAt(0);
    var _NWL = '\n'.charCodeAt(0);
    var _CAR = '\r'.charCodeAt(0);
    var _LFD = '\f'.charCodeAt(0);
    var _DLR = '$'.charCodeAt(0);
    var _HSH = '#'.charCodeAt(0);
    var _CUL = '{'.charCodeAt(0);
    var _EQS = '='.charCodeAt(0);
    var _BNG = '!'.charCodeAt(0);
    var _LAN = '<'.charCodeAt(0);
    var _RAN = '>'.charCodeAt(0);
    var _DOT = '.'.charCodeAt(0);
    var _ATS = '@'.charCodeAt(0);
    var customTokenValue = cssScanner_1.TokenType.CustomToken;
    exports.VariableName = customTokenValue++;
    exports.InterpolationFunction = customTokenValue++;
    exports.Default = customTokenValue++;
    exports.EqualsOperator = customTokenValue++;
    exports.NotEqualsOperator = customTokenValue++;
    exports.GreaterEqualsOperator = customTokenValue++;
    exports.SmallerEqualsOperator = customTokenValue++;
    exports.Ellipsis = customTokenValue++;
    exports.Module = customTokenValue++;
    var SCSSScanner = /** @class */ (function (_super) {
        __extends(SCSSScanner, _super);
        function SCSSScanner() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        SCSSScanner.prototype.scanNext = function (offset) {
            // scss variable
            if (this.stream.advanceIfChar(_DLR)) {
                var content = ['$'];
                if (this.ident(content)) {
                    return this.finishToken(offset, exports.VariableName, content.join(''));
                }
                else {
                    this.stream.goBackTo(offset);
                }
            }
            // scss: interpolation function #{..})
            if (this.stream.advanceIfChars([_HSH, _CUL])) {
                return this.finishToken(offset, exports.InterpolationFunction);
            }
            // operator ==
            if (this.stream.advanceIfChars([_EQS, _EQS])) {
                return this.finishToken(offset, exports.EqualsOperator);
            }
            // operator !=
            if (this.stream.advanceIfChars([_BNG, _EQS])) {
                return this.finishToken(offset, exports.NotEqualsOperator);
            }
            // operators <, <=
            if (this.stream.advanceIfChar(_LAN)) {
                if (this.stream.advanceIfChar(_EQS)) {
                    return this.finishToken(offset, exports.SmallerEqualsOperator);
                }
                return this.finishToken(offset, cssScanner_1.TokenType.Delim);
            }
            // ooperators >, >=
            if (this.stream.advanceIfChar(_RAN)) {
                if (this.stream.advanceIfChar(_EQS)) {
                    return this.finishToken(offset, exports.GreaterEqualsOperator);
                }
                return this.finishToken(offset, cssScanner_1.TokenType.Delim);
            }
            // ellipis
            if (this.stream.advanceIfChars([_DOT, _DOT, _DOT])) {
                return this.finishToken(offset, exports.Ellipsis);
            }
            return _super.prototype.scanNext.call(this, offset);
        };
        SCSSScanner.prototype.comment = function () {
            if (_super.prototype.comment.call(this)) {
                return true;
            }
            if (!this.inURL && this.stream.advanceIfChars([_FSL, _FSL])) {
                this.stream.advanceWhileChar(function (ch) {
                    switch (ch) {
                        case _NWL:
                        case _CAR:
                        case _LFD:
                            return false;
                        default:
                            return true;
                    }
                });
                return true;
            }
            else {
                return false;
            }
        };
        return SCSSScanner;
    }(cssScanner_1.Scanner));
    exports.SCSSScanner = SCSSScanner;
});

(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/parser/scssErrors',["require", "exports", "vscode-nls"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.SCSSParseError = exports.SCSSIssueType = void 0;
    var nls = require("vscode-nls");
    var localize = nls.loadMessageBundle();
    var SCSSIssueType = /** @class */ (function () {
        function SCSSIssueType(id, message) {
            this.id = id;
            this.message = message;
        }
        return SCSSIssueType;
    }());
    exports.SCSSIssueType = SCSSIssueType;
    exports.SCSSParseError = {
        FromExpected: new SCSSIssueType('scss-fromexpected', localize('expected.from', "'from' expected")),
        ThroughOrToExpected: new SCSSIssueType('scss-throughexpected', localize('expected.through', "'through' or 'to' expected")),
        InExpected: new SCSSIssueType('scss-fromexpected', localize('expected.in', "'in' expected")),
    };
});

var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/parser/scssParser',["require", "exports", "./scssScanner", "./cssScanner", "./cssParser", "./cssNodes", "./scssErrors", "./cssErrors"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.SCSSParser = void 0;
    var scssScanner = require("./scssScanner");
    var cssScanner_1 = require("./cssScanner");
    var cssParser = require("./cssParser");
    var nodes = require("./cssNodes");
    var scssErrors_1 = require("./scssErrors");
    var cssErrors_1 = require("./cssErrors");
    /// <summary>
    /// A parser for scss
    /// http://sass-lang.com/documentation/file.SASS_REFERENCE.html
    /// </summary>
    var SCSSParser = /** @class */ (function (_super) {
        __extends(SCSSParser, _super);
        function SCSSParser() {
            return _super.call(this, new scssScanner.SCSSScanner()) || this;
        }
        SCSSParser.prototype._parseStylesheetStatement = function (isNested) {
            if (isNested === void 0) { isNested = false; }
            if (this.peek(cssScanner_1.TokenType.AtKeyword)) {
                return this._parseWarnAndDebug() // @warn, @debug and @error statements
                    || this._parseControlStatement() // @if, @while, @for, @each
                    || this._parseMixinDeclaration() // @mixin
                    || this._parseMixinContent() // @content
                    || this._parseMixinReference() // @include
                    || this._parseFunctionDeclaration() // @function
                    || this._parseForward() // @forward
                    || this._parseUse() // @use
                    || this._parseRuleset(isNested) // @at-rule
                    || _super.prototype._parseStylesheetAtStatement.call(this, isNested);
            }
            return this._parseRuleset(true) || this._parseVariableDeclaration();
        };
        SCSSParser.prototype._parseImport = function () {
            if (!this.peekKeyword('@import')) {
                return null;
            }
            var node = this.create(nodes.Import);
            this.consumeToken();
            if (!node.addChild(this._parseURILiteral()) && !node.addChild(this._parseStringLiteral())) {
                return this.finish(node, cssErrors_1.ParseError.URIOrStringExpected);
            }
            while (this.accept(cssScanner_1.TokenType.Comma)) {
                if (!node.addChild(this._parseURILiteral()) && !node.addChild(this._parseStringLiteral())) {
                    return this.finish(node, cssErrors_1.ParseError.URIOrStringExpected);
                }
            }
            if (!this.peek(cssScanner_1.TokenType.SemiColon) && !this.peek(cssScanner_1.TokenType.EOF)) {
                node.setMedialist(this._parseMediaQueryList());
            }
            return this.finish(node);
        };
        // scss variables: $font-size: 12px;
        SCSSParser.prototype._parseVariableDeclaration = function (panic) {
            if (panic === void 0) { panic = []; }
            if (!this.peek(scssScanner.VariableName)) {
                return null;
            }
            var node = this.create(nodes.VariableDeclaration);
            if (!node.setVariable(this._parseVariable())) {
                return null;
            }
            if (!this.accept(cssScanner_1.TokenType.Colon)) {
                return this.finish(node, cssErrors_1.ParseError.ColonExpected);
            }
            if (this.prevToken) {
                node.colonPosition = this.prevToken.offset;
            }
            if (!node.setValue(this._parseExpr())) {
                return this.finish(node, cssErrors_1.ParseError.VariableValueExpected, [], panic);
            }
            while (this.peek(cssScanner_1.TokenType.Exclamation)) {
                if (node.addChild(this._tryParsePrio())) {
                    // !important
                }
                else {
                    this.consumeToken();
                    if (!this.peekRegExp(cssScanner_1.TokenType.Ident, /^(default|global)$/)) {
                        return this.finish(node, cssErrors_1.ParseError.UnknownKeyword);
                    }
                    this.consumeToken();
                }
            }
            if (this.peek(cssScanner_1.TokenType.SemiColon)) {
                node.semicolonPosition = this.token.offset; // not part of the declaration, but useful information for code assist
            }
            return this.finish(node);
        };
        SCSSParser.prototype._parseMediaContentStart = function () {
            return this._parseInterpolation();
        };
        SCSSParser.prototype._parseMediaFeatureName = function () {
            return this._parseModuleMember()
                || this._parseFunction() // function before ident
                || this._parseIdent()
                || this._parseVariable();
        };
        SCSSParser.prototype._parseKeyframeSelector = function () {
            return this._tryParseKeyframeSelector()
                || this._parseControlStatement(this._parseKeyframeSelector.bind(this))
                || this._parseVariableDeclaration()
                || this._parseMixinContent();
        };
        SCSSParser.prototype._parseVariable = function () {
            if (!this.peek(scssScanner.VariableName)) {
                return null;
            }
            var node = this.create(nodes.Variable);
            this.consumeToken();
            return node;
        };
        SCSSParser.prototype._parseModuleMember = function () {
            var pos = this.mark();
            var node = this.create(nodes.Module);
            if (!node.setIdentifier(this._parseIdent([nodes.ReferenceType.Module]))) {
                return null;
            }
            if (this.hasWhitespace()
                || !this.acceptDelim('.')
                || this.hasWhitespace()) {
                this.restoreAtMark(pos);
                return null;
            }
            if (!node.addChild(this._parseVariable() || this._parseFunction())) {
                return this.finish(node, cssErrors_1.ParseError.IdentifierOrVariableExpected);
            }
            return node;
        };
        SCSSParser.prototype._parseIdent = function (referenceTypes) {
            var _this = this;
            if (!this.peek(cssScanner_1.TokenType.Ident) && !this.peek(scssScanner.InterpolationFunction) && !this.peekDelim('-')) {
                return null;
            }
            var node = this.create(nodes.Identifier);
            node.referenceTypes = referenceTypes;
            node.isCustomProperty = this.peekRegExp(cssScanner_1.TokenType.Ident, /^--/);
            var hasContent = false;
            var indentInterpolation = function () {
                var pos = _this.mark();
                if (_this.acceptDelim('-')) {
                    if (!_this.hasWhitespace()) {
                        _this.acceptDelim('-');
                    }
                    if (_this.hasWhitespace()) {
                        _this.restoreAtMark(pos);
                        return null;
                    }
                }
                return _this._parseInterpolation();
            };
            while (this.accept(cssScanner_1.TokenType.Ident) || node.addChild(indentInterpolation()) || (hasContent && this.acceptRegexp(/^[\w-]/))) {
                hasContent = true;
                if (this.hasWhitespace()) {
                    break;
                }
            }
            return hasContent ? this.finish(node) : null;
        };
        SCSSParser.prototype._parseTermExpression = function () {
            return this._parseModuleMember() ||
                this._parseVariable() ||
                this._parseSelectorCombinator() ||
                //this._tryParsePrio() ||
                _super.prototype._parseTermExpression.call(this);
        };
        SCSSParser.prototype._parseInterpolation = function () {
            if (this.peek(scssScanner.InterpolationFunction)) {
                var node = this.create(nodes.Interpolation);
                this.consumeToken();
                if (!node.addChild(this._parseExpr()) && !this._parseSelectorCombinator()) {
                    if (this.accept(cssScanner_1.TokenType.CurlyR)) {
                        return this.finish(node);
                    }
                    return this.finish(node, cssErrors_1.ParseError.ExpressionExpected);
                }
                if (!this.accept(cssScanner_1.TokenType.CurlyR)) {
                    return this.finish(node, cssErrors_1.ParseError.RightCurlyExpected);
                }
                return this.finish(node);
            }
            return null;
        };
        SCSSParser.prototype._parseOperator = function () {
            if (this.peek(scssScanner.EqualsOperator) || this.peek(scssScanner.NotEqualsOperator)
                || this.peek(scssScanner.GreaterEqualsOperator) || this.peek(scssScanner.SmallerEqualsOperator)
                || this.peekDelim('>') || this.peekDelim('<')
                || this.peekIdent('and') || this.peekIdent('or')
                || this.peekDelim('%')) {
                var node = this.createNode(nodes.NodeType.Operator);
                this.consumeToken();
                return this.finish(node);
            }
            return _super.prototype._parseOperator.call(this);
        };
        SCSSParser.prototype._parseUnaryOperator = function () {
            if (this.peekIdent('not')) {
                var node = this.create(nodes.Node);
                this.consumeToken();
                return this.finish(node);
            }
            return _super.prototype._parseUnaryOperator.call(this);
        };
        SCSSParser.prototype._parseRuleSetDeclaration = function () {
            if (this.peek(cssScanner_1.TokenType.AtKeyword)) {
                return this._parseKeyframe() // nested @keyframe
                    || this._parseImport() // nested @import
                    || this._parseMedia(true) // nested @media
                    || this._parseFontFace() // nested @font-face
                    || this._parseWarnAndDebug() // @warn, @debug and @error statements
                    || this._parseControlStatement() // @if, @while, @for, @each
                    || this._parseFunctionDeclaration() // @function
                    || this._parseExtends() // @extends
                    || this._parseMixinReference() // @include
                    || this._parseMixinContent() // @content
                    || this._parseMixinDeclaration() // nested @mixin
                    || this._parseRuleset(true) // @at-rule
                    || this._parseSupports(true) // @supports
                    || _super.prototype._parseRuleSetDeclarationAtStatement.call(this);
            }
            return this._parseVariableDeclaration() // variable declaration
                || this._tryParseRuleset(true) // nested ruleset
                || _super.prototype._parseRuleSetDeclaration.call(this); // try css ruleset declaration as last so in the error case, the ast will contain a declaration
        };
        SCSSParser.prototype._parseDeclaration = function (stopTokens) {
            var custonProperty = this._tryParseCustomPropertyDeclaration(stopTokens);
            if (custonProperty) {
                return custonProperty;
            }
            var node = this.create(nodes.Declaration);
            if (!node.setProperty(this._parseProperty())) {
                return null;
            }
            if (!this.accept(cssScanner_1.TokenType.Colon)) {
                return this.finish(node, cssErrors_1.ParseError.ColonExpected, [cssScanner_1.TokenType.Colon], stopTokens || [cssScanner_1.TokenType.SemiColon]);
            }
            if (this.prevToken) {
                node.colonPosition = this.prevToken.offset;
            }
            var hasContent = false;
            if (node.setValue(this._parseExpr())) {
                hasContent = true;
                node.addChild(this._parsePrio());
            }
            if (this.peek(cssScanner_1.TokenType.CurlyL)) {
                node.setNestedProperties(this._parseNestedProperties());
            }
            else {
                if (!hasContent) {
                    return this.finish(node, cssErrors_1.ParseError.PropertyValueExpected);
                }
            }
            if (this.peek(cssScanner_1.TokenType.SemiColon)) {
                node.semicolonPosition = this.token.offset; // not part of the declaration, but useful information for code assist
            }
            return this.finish(node);
        };
        SCSSParser.prototype._parseNestedProperties = function () {
            var node = this.create(nodes.NestedProperties);
            return this._parseBody(node, this._parseDeclaration.bind(this));
        };
        SCSSParser.prototype._parseExtends = function () {
            if (this.peekKeyword('@extend')) {
                var node = this.create(nodes.ExtendsReference);
                this.consumeToken();
                if (!node.getSelectors().addChild(this._parseSimpleSelector())) {
                    return this.finish(node, cssErrors_1.ParseError.SelectorExpected);
                }
                while (this.accept(cssScanner_1.TokenType.Comma)) {
                    node.getSelectors().addChild(this._parseSimpleSelector());
                }
                if (this.accept(cssScanner_1.TokenType.Exclamation)) {
                    if (!this.acceptIdent('optional')) {
                        return this.finish(node, cssErrors_1.ParseError.UnknownKeyword);
                    }
                }
                return this.finish(node);
            }
            return null;
        };
        SCSSParser.prototype._parseSimpleSelectorBody = function () {
            return this._parseSelectorCombinator() || this._parseSelectorPlaceholder() || _super.prototype._parseSimpleSelectorBody.call(this);
        };
        SCSSParser.prototype._parseSelectorCombinator = function () {
            if (this.peekDelim('&')) {
                var node = this.createNode(nodes.NodeType.SelectorCombinator);
                this.consumeToken();
                while (!this.hasWhitespace() && (this.acceptDelim('-') || this.accept(cssScanner_1.TokenType.Num) || this.accept(cssScanner_1.TokenType.Dimension) || node.addChild(this._parseIdent()) || this.acceptDelim('&'))) {
                    //  support &-foo-1
                }
                return this.finish(node);
            }
            return null;
        };
        SCSSParser.prototype._parseSelectorPlaceholder = function () {
            if (this.peekDelim('%')) {
                var node = this.createNode(nodes.NodeType.SelectorPlaceholder);
                this.consumeToken();
                this._parseIdent();
                return this.finish(node);
            }
            else if (this.peekKeyword('@at-root')) {
                var node = this.createNode(nodes.NodeType.SelectorPlaceholder);
                this.consumeToken();
                return this.finish(node);
            }
            return null;
        };
        SCSSParser.prototype._parseElementName = function () {
            var pos = this.mark();
            var node = _super.prototype._parseElementName.call(this);
            if (node && !this.hasWhitespace() && this.peek(cssScanner_1.TokenType.ParenthesisL)) { // for #49589
                this.restoreAtMark(pos);
                return null;
            }
            return node;
        };
        SCSSParser.prototype._tryParsePseudoIdentifier = function () {
            return this._parseInterpolation() || _super.prototype._tryParsePseudoIdentifier.call(this); // for #49589
        };
        SCSSParser.prototype._parseWarnAndDebug = function () {
            if (!this.peekKeyword('@debug')
                && !this.peekKeyword('@warn')
                && !this.peekKeyword('@error')) {
                return null;
            }
            var node = this.createNode(nodes.NodeType.Debug);
            this.consumeToken(); // @debug, @warn or @error
            node.addChild(this._parseExpr()); // optional
            return this.finish(node);
        };
        SCSSParser.prototype._parseControlStatement = function (parseStatement) {
            if (parseStatement === void 0) { parseStatement = this._parseRuleSetDeclaration.bind(this); }
            if (!this.peek(cssScanner_1.TokenType.AtKeyword)) {
                return null;
            }
            return this._parseIfStatement(parseStatement) || this._parseForStatement(parseStatement)
                || this._parseEachStatement(parseStatement) || this._parseWhileStatement(parseStatement);
        };
        SCSSParser.prototype._parseIfStatement = function (parseStatement) {
            if (!this.peekKeyword('@if')) {
                return null;
            }
            return this._internalParseIfStatement(parseStatement);
        };
        SCSSParser.prototype._internalParseIfStatement = function (parseStatement) {
            var node = this.create(nodes.IfStatement);
            this.consumeToken(); // @if or if
            if (!node.setExpression(this._parseExpr(true))) {
                return this.finish(node, cssErrors_1.ParseError.ExpressionExpected);
            }
            this._parseBody(node, parseStatement);
            if (this.acceptKeyword('@else')) {
                if (this.peekIdent('if')) {
                    node.setElseClause(this._internalParseIfStatement(parseStatement));
                }
                else if (this.peek(cssScanner_1.TokenType.CurlyL)) {
                    var elseNode = this.create(nodes.ElseStatement);
                    this._parseBody(elseNode, parseStatement);
                    node.setElseClause(elseNode);
                }
            }
            return this.finish(node);
        };
        SCSSParser.prototype._parseForStatement = function (parseStatement) {
            if (!this.peekKeyword('@for')) {
                return null;
            }
            var node = this.create(nodes.ForStatement);
            this.consumeToken(); // @for
            if (!node.setVariable(this._parseVariable())) {
                return this.finish(node, cssErrors_1.ParseError.VariableNameExpected, [cssScanner_1.TokenType.CurlyR]);
            }
            if (!this.acceptIdent('from')) {
                return this.finish(node, scssErrors_1.SCSSParseError.FromExpected, [cssScanner_1.TokenType.CurlyR]);
            }
            if (!node.addChild(this._parseBinaryExpr())) {
                return this.finish(node, cssErrors_1.ParseError.ExpressionExpected, [cssScanner_1.TokenType.CurlyR]);
            }
            if (!this.acceptIdent('to') && !this.acceptIdent('through')) {
                return this.finish(node, scssErrors_1.SCSSParseError.ThroughOrToExpected, [cssScanner_1.TokenType.CurlyR]);
            }
            if (!node.addChild(this._parseBinaryExpr())) {
                return this.finish(node, cssErrors_1.ParseError.ExpressionExpected, [cssScanner_1.TokenType.CurlyR]);
            }
            return this._parseBody(node, parseStatement);
        };
        SCSSParser.prototype._parseEachStatement = function (parseStatement) {
            if (!this.peekKeyword('@each')) {
                return null;
            }
            var node = this.create(nodes.EachStatement);
            this.consumeToken(); // @each
            var variables = node.getVariables();
            if (!variables.addChild(this._parseVariable())) {
                return this.finish(node, cssErrors_1.ParseError.VariableNameExpected, [cssScanner_1.TokenType.CurlyR]);
            }
            while (this.accept(cssScanner_1.TokenType.Comma)) {
                if (!variables.addChild(this._parseVariable())) {
                    return this.finish(node, cssErrors_1.ParseError.VariableNameExpected, [cssScanner_1.TokenType.CurlyR]);
                }
            }
            this.finish(variables);
            if (!this.acceptIdent('in')) {
                return this.finish(node, scssErrors_1.SCSSParseError.InExpected, [cssScanner_1.TokenType.CurlyR]);
            }
            if (!node.addChild(this._parseExpr())) {
                return this.finish(node, cssErrors_1.ParseError.ExpressionExpected, [cssScanner_1.TokenType.CurlyR]);
            }
            return this._parseBody(node, parseStatement);
        };
        SCSSParser.prototype._parseWhileStatement = function (parseStatement) {
            if (!this.peekKeyword('@while')) {
                return null;
            }
            var node = this.create(nodes.WhileStatement);
            this.consumeToken(); // @while
            if (!node.addChild(this._parseBinaryExpr())) {
                return this.finish(node, cssErrors_1.ParseError.ExpressionExpected, [cssScanner_1.TokenType.CurlyR]);
            }
            return this._parseBody(node, parseStatement);
        };
        SCSSParser.prototype._parseFunctionBodyDeclaration = function () {
            return this._parseVariableDeclaration() || this._parseReturnStatement() || this._parseWarnAndDebug()
                || this._parseControlStatement(this._parseFunctionBodyDeclaration.bind(this));
        };
        SCSSParser.prototype._parseFunctionDeclaration = function () {
            if (!this.peekKeyword('@function')) {
                return null;
            }
            var node = this.create(nodes.FunctionDeclaration);
            this.consumeToken(); // @function
            if (!node.setIdentifier(this._parseIdent([nodes.ReferenceType.Function]))) {
                return this.finish(node, cssErrors_1.ParseError.IdentifierExpected, [cssScanner_1.TokenType.CurlyR]);
            }
            if (!this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                return this.finish(node, cssErrors_1.ParseError.LeftParenthesisExpected, [cssScanner_1.TokenType.CurlyR]);
            }
            if (node.getParameters().addChild(this._parseParameterDeclaration())) {
                while (this.accept(cssScanner_1.TokenType.Comma)) {
                    if (this.peek(cssScanner_1.TokenType.ParenthesisR)) {
                        break;
                    }
                    if (!node.getParameters().addChild(this._parseParameterDeclaration())) {
                        return this.finish(node, cssErrors_1.ParseError.VariableNameExpected);
                    }
                }
            }
            if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected, [cssScanner_1.TokenType.CurlyR]);
            }
            return this._parseBody(node, this._parseFunctionBodyDeclaration.bind(this));
        };
        SCSSParser.prototype._parseReturnStatement = function () {
            if (!this.peekKeyword('@return')) {
                return null;
            }
            var node = this.createNode(nodes.NodeType.ReturnStatement);
            this.consumeToken(); // @function
            if (!node.addChild(this._parseExpr())) {
                return this.finish(node, cssErrors_1.ParseError.ExpressionExpected);
            }
            return this.finish(node);
        };
        SCSSParser.prototype._parseMixinDeclaration = function () {
            if (!this.peekKeyword('@mixin')) {
                return null;
            }
            var node = this.create(nodes.MixinDeclaration);
            this.consumeToken();
            if (!node.setIdentifier(this._parseIdent([nodes.ReferenceType.Mixin]))) {
                return this.finish(node, cssErrors_1.ParseError.IdentifierExpected, [cssScanner_1.TokenType.CurlyR]);
            }
            if (this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                if (node.getParameters().addChild(this._parseParameterDeclaration())) {
                    while (this.accept(cssScanner_1.TokenType.Comma)) {
                        if (this.peek(cssScanner_1.TokenType.ParenthesisR)) {
                            break;
                        }
                        if (!node.getParameters().addChild(this._parseParameterDeclaration())) {
                            return this.finish(node, cssErrors_1.ParseError.VariableNameExpected);
                        }
                    }
                }
                if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                    return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected, [cssScanner_1.TokenType.CurlyR]);
                }
            }
            return this._parseBody(node, this._parseRuleSetDeclaration.bind(this));
        };
        SCSSParser.prototype._parseParameterDeclaration = function () {
            var node = this.create(nodes.FunctionParameter);
            if (!node.setIdentifier(this._parseVariable())) {
                return null;
            }
            if (this.accept(scssScanner.Ellipsis)) {
                // ok
            }
            if (this.accept(cssScanner_1.TokenType.Colon)) {
                if (!node.setDefaultValue(this._parseExpr(true))) {
                    return this.finish(node, cssErrors_1.ParseError.VariableValueExpected, [], [cssScanner_1.TokenType.Comma, cssScanner_1.TokenType.ParenthesisR]);
                }
            }
            return this.finish(node);
        };
        SCSSParser.prototype._parseMixinContent = function () {
            if (!this.peekKeyword('@content')) {
                return null;
            }
            var node = this.create(nodes.MixinContentReference);
            this.consumeToken();
            if (this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                if (node.getArguments().addChild(this._parseFunctionArgument())) {
                    while (this.accept(cssScanner_1.TokenType.Comma)) {
                        if (this.peek(cssScanner_1.TokenType.ParenthesisR)) {
                            break;
                        }
                        if (!node.getArguments().addChild(this._parseFunctionArgument())) {
                            return this.finish(node, cssErrors_1.ParseError.ExpressionExpected);
                        }
                    }
                }
                if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                    return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected);
                }
            }
            return this.finish(node);
        };
        SCSSParser.prototype._parseMixinReference = function () {
            if (!this.peekKeyword('@include')) {
                return null;
            }
            var node = this.create(nodes.MixinReference);
            this.consumeToken();
            // Could be module or mixin identifier, set as mixin as default.
            var firstIdent = this._parseIdent([nodes.ReferenceType.Mixin]);
            if (!node.setIdentifier(firstIdent)) {
                return this.finish(node, cssErrors_1.ParseError.IdentifierExpected, [cssScanner_1.TokenType.CurlyR]);
            }
            // Is a module accessor.
            if (!this.hasWhitespace() && this.acceptDelim('.') && !this.hasWhitespace()) {
                var secondIdent = this._parseIdent([nodes.ReferenceType.Mixin]);
                if (!secondIdent) {
                    return this.finish(node, cssErrors_1.ParseError.IdentifierExpected, [cssScanner_1.TokenType.CurlyR]);
                }
                var moduleToken = this.create(nodes.Module);
                // Re-purpose first matched ident as identifier for module token.
                firstIdent.referenceTypes = [nodes.ReferenceType.Module];
                moduleToken.setIdentifier(firstIdent);
                // Override identifier with second ident.
                node.setIdentifier(secondIdent);
                node.addChild(moduleToken);
            }
            if (this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                if (node.getArguments().addChild(this._parseFunctionArgument())) {
                    while (this.accept(cssScanner_1.TokenType.Comma)) {
                        if (this.peek(cssScanner_1.TokenType.ParenthesisR)) {
                            break;
                        }
                        if (!node.getArguments().addChild(this._parseFunctionArgument())) {
                            return this.finish(node, cssErrors_1.ParseError.ExpressionExpected);
                        }
                    }
                }
                if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                    return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected);
                }
            }
            if (this.peekIdent('using') || this.peek(cssScanner_1.TokenType.CurlyL)) {
                node.setContent(this._parseMixinContentDeclaration());
            }
            return this.finish(node);
        };
        SCSSParser.prototype._parseMixinContentDeclaration = function () {
            var node = this.create(nodes.MixinContentDeclaration);
            if (this.acceptIdent('using')) {
                if (!this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                    return this.finish(node, cssErrors_1.ParseError.LeftParenthesisExpected, [cssScanner_1.TokenType.CurlyL]);
                }
                if (node.getParameters().addChild(this._parseParameterDeclaration())) {
                    while (this.accept(cssScanner_1.TokenType.Comma)) {
                        if (this.peek(cssScanner_1.TokenType.ParenthesisR)) {
                            break;
                        }
                        if (!node.getParameters().addChild(this._parseParameterDeclaration())) {
                            return this.finish(node, cssErrors_1.ParseError.VariableNameExpected);
                        }
                    }
                }
                if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                    return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected, [cssScanner_1.TokenType.CurlyL]);
                }
            }
            if (this.peek(cssScanner_1.TokenType.CurlyL)) {
                this._parseBody(node, this._parseMixinReferenceBodyStatement.bind(this));
            }
            return this.finish(node);
        };
        SCSSParser.prototype._parseMixinReferenceBodyStatement = function () {
            return this._tryParseKeyframeSelector() || this._parseRuleSetDeclaration();
        };
        SCSSParser.prototype._parseFunctionArgument = function () {
            // [variableName ':'] expression | variableName '...'
            var node = this.create(nodes.FunctionArgument);
            var pos = this.mark();
            var argument = this._parseVariable();
            if (argument) {
                if (!this.accept(cssScanner_1.TokenType.Colon)) {
                    if (this.accept(scssScanner.Ellipsis)) { // optional
                        node.setValue(argument);
                        return this.finish(node);
                    }
                    else {
                        this.restoreAtMark(pos);
                    }
                }
                else {
                    node.setIdentifier(argument);
                }
            }
            if (node.setValue(this._parseExpr(true))) {
                this.accept(scssScanner.Ellipsis); // #43746
                node.addChild(this._parsePrio()); // #9859
                return this.finish(node);
            }
            else if (node.setValue(this._tryParsePrio())) {
                return this.finish(node);
            }
            return null;
        };
        SCSSParser.prototype._parseURLArgument = function () {
            var pos = this.mark();
            var node = _super.prototype._parseURLArgument.call(this);
            if (!node || !this.peek(cssScanner_1.TokenType.ParenthesisR)) {
                this.restoreAtMark(pos);
                var node_1 = this.create(nodes.Node);
                node_1.addChild(this._parseBinaryExpr());
                return this.finish(node_1);
            }
            return node;
        };
        SCSSParser.prototype._parseOperation = function () {
            if (!this.peek(cssScanner_1.TokenType.ParenthesisL)) {
                return null;
            }
            var node = this.create(nodes.Node);
            this.consumeToken();
            while (node.addChild(this._parseListElement())) {
                this.accept(cssScanner_1.TokenType.Comma); // optional
            }
            if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected);
            }
            return this.finish(node);
        };
        SCSSParser.prototype._parseListElement = function () {
            var node = this.create(nodes.ListEntry);
            var child = this._parseBinaryExpr();
            if (!child) {
                return null;
            }
            if (this.accept(cssScanner_1.TokenType.Colon)) {
                node.setKey(child);
                if (!node.setValue(this._parseBinaryExpr())) {
                    return this.finish(node, cssErrors_1.ParseError.ExpressionExpected);
                }
            }
            else {
                node.setValue(child);
            }
            return this.finish(node);
        };
        SCSSParser.prototype._parseUse = function () {
            if (!this.peekKeyword('@use')) {
                return null;
            }
            var node = this.create(nodes.Use);
            this.consumeToken(); // @use
            if (!node.addChild(this._parseStringLiteral())) {
                return this.finish(node, cssErrors_1.ParseError.StringLiteralExpected);
            }
            if (!this.peek(cssScanner_1.TokenType.SemiColon) && !this.peek(cssScanner_1.TokenType.EOF)) {
                if (!this.peekRegExp(cssScanner_1.TokenType.Ident, /as|with/)) {
                    return this.finish(node, cssErrors_1.ParseError.UnknownKeyword);
                }
                if (this.acceptIdent('as') &&
                    (!node.setIdentifier(this._parseIdent([nodes.ReferenceType.Module])) && !this.acceptDelim('*'))) {
                    return this.finish(node, cssErrors_1.ParseError.IdentifierOrWildcardExpected);
                }
                if (this.acceptIdent('with')) {
                    if (!this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                        return this.finish(node, cssErrors_1.ParseError.LeftParenthesisExpected, [cssScanner_1.TokenType.ParenthesisR]);
                    }
                    // First variable statement, no comma.
                    if (!node.getParameters().addChild(this._parseModuleConfigDeclaration())) {
                        return this.finish(node, cssErrors_1.ParseError.VariableNameExpected);
                    }
                    while (this.accept(cssScanner_1.TokenType.Comma)) {
                        if (this.peek(cssScanner_1.TokenType.ParenthesisR)) {
                            break;
                        }
                        if (!node.getParameters().addChild(this._parseModuleConfigDeclaration())) {
                            return this.finish(node, cssErrors_1.ParseError.VariableNameExpected);
                        }
                    }
                    if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                        return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected);
                    }
                }
            }
            if (!this.accept(cssScanner_1.TokenType.SemiColon) && !this.accept(cssScanner_1.TokenType.EOF)) {
                return this.finish(node, cssErrors_1.ParseError.SemiColonExpected);
            }
            return this.finish(node);
        };
        SCSSParser.prototype._parseModuleConfigDeclaration = function () {
            var node = this.create(nodes.ModuleConfiguration);
            if (!node.setIdentifier(this._parseVariable())) {
                return null;
            }
            if (!this.accept(cssScanner_1.TokenType.Colon) || !node.setValue(this._parseExpr(true))) {
                return this.finish(node, cssErrors_1.ParseError.VariableValueExpected, [], [cssScanner_1.TokenType.Comma, cssScanner_1.TokenType.ParenthesisR]);
            }
            if (this.accept(cssScanner_1.TokenType.Exclamation)) {
                if (this.hasWhitespace() || !this.acceptIdent('default')) {
                    return this.finish(node, cssErrors_1.ParseError.UnknownKeyword);
                }
            }
            return this.finish(node);
        };
        SCSSParser.prototype._parseForward = function () {
            if (!this.peekKeyword('@forward')) {
                return null;
            }
            var node = this.create(nodes.Forward);
            this.consumeToken();
            if (!node.addChild(this._parseStringLiteral())) {
                return this.finish(node, cssErrors_1.ParseError.StringLiteralExpected);
            }
            if (this.acceptIdent('with')) {
                if (!this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                    return this.finish(node, cssErrors_1.ParseError.LeftParenthesisExpected, [cssScanner_1.TokenType.ParenthesisR]);
                }
                // First variable statement, no comma.
                if (!node.getParameters().addChild(this._parseModuleConfigDeclaration())) {
                    return this.finish(node, cssErrors_1.ParseError.VariableNameExpected);
                }
                while (this.accept(cssScanner_1.TokenType.Comma)) {
                    if (this.peek(cssScanner_1.TokenType.ParenthesisR)) {
                        break;
                    }
                    if (!node.getParameters().addChild(this._parseModuleConfigDeclaration())) {
                        return this.finish(node, cssErrors_1.ParseError.VariableNameExpected);
                    }
                }
                if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                    return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected);
                }
            }
            if (!this.peek(cssScanner_1.TokenType.SemiColon) && !this.peek(cssScanner_1.TokenType.EOF)) {
                if (!this.peekRegExp(cssScanner_1.TokenType.Ident, /as|hide|show/)) {
                    return this.finish(node, cssErrors_1.ParseError.UnknownKeyword);
                }
                if (this.acceptIdent('as')) {
                    var identifier = this._parseIdent([nodes.ReferenceType.Forward]);
                    if (!node.setIdentifier(identifier)) {
                        return this.finish(node, cssErrors_1.ParseError.IdentifierExpected);
                    }
                    // Wildcard must be the next character after the identifier string.
                    if (this.hasWhitespace() || !this.acceptDelim('*')) {
                        return this.finish(node, cssErrors_1.ParseError.WildcardExpected);
                    }
                }
                if (this.peekIdent('hide') || this.peekIdent('show')) {
                    if (!node.addChild(this._parseForwardVisibility())) {
                        return this.finish(node, cssErrors_1.ParseError.IdentifierOrVariableExpected);
                    }
                }
            }
            if (!this.accept(cssScanner_1.TokenType.SemiColon) && !this.accept(cssScanner_1.TokenType.EOF)) {
                return this.finish(node, cssErrors_1.ParseError.SemiColonExpected);
            }
            return this.finish(node);
        };
        SCSSParser.prototype._parseForwardVisibility = function () {
            var node = this.create(nodes.ForwardVisibility);
            // Assume to be "hide" or "show".
            node.setIdentifier(this._parseIdent());
            while (node.addChild(this._parseVariable() || this._parseIdent())) {
                // Consume all variables and idents ahead.
                this.accept(cssScanner_1.TokenType.Comma);
            }
            // More than just identifier 
            return node.getChildren().length > 1 ? node : null;
        };
        SCSSParser.prototype._parseSupportsCondition = function () {
            return this._parseInterpolation() || _super.prototype._parseSupportsCondition.call(this);
        };
        return SCSSParser;
    }(cssParser.Parser));
    exports.SCSSParser = SCSSParser;
});

var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/services/scssCompletion',["require", "exports", "./cssCompletion", "../parser/cssNodes", "../cssLanguageTypes", "vscode-nls"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.SCSSCompletion = void 0;
    var cssCompletion_1 = require("./cssCompletion");
    var nodes = require("../parser/cssNodes");
    var cssLanguageTypes_1 = require("../cssLanguageTypes");
    var nls = require("vscode-nls");
    var localize = nls.loadMessageBundle();
    var SCSSCompletion = /** @class */ (function (_super) {
        __extends(SCSSCompletion, _super);
        function SCSSCompletion(lsServiceOptions, cssDataManager) {
            var _this = _super.call(this, '$', lsServiceOptions, cssDataManager) || this;
            addReferencesToDocumentation(SCSSCompletion.scssModuleLoaders);
            addReferencesToDocumentation(SCSSCompletion.scssModuleBuiltIns);
            return _this;
        }
        SCSSCompletion.prototype.isImportPathParent = function (type) {
            return type === nodes.NodeType.Forward
                || type === nodes.NodeType.Use
                || _super.prototype.isImportPathParent.call(this, type);
        };
        SCSSCompletion.prototype.getCompletionForImportPath = function (importPathNode, result) {
            var parentType = importPathNode.getParent().type;
            if (parentType === nodes.NodeType.Forward || parentType === nodes.NodeType.Use) {
                for (var _i = 0, _a = SCSSCompletion.scssModuleBuiltIns; _i < _a.length; _i++) {
                    var p = _a[_i];
                    var item = {
                        label: p.label,
                        documentation: p.documentation,
                        textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(importPathNode), "'" + p.label + "'"),
                        kind: cssLanguageTypes_1.CompletionItemKind.Module
                    };
                    result.items.push(item);
                }
            }
            return _super.prototype.getCompletionForImportPath.call(this, importPathNode, result);
        };
        SCSSCompletion.prototype.createReplaceFunction = function () {
            var tabStopCounter = 1;
            return function (_match, p1) {
                return '\\' + p1 + ': ${' + tabStopCounter++ + ':' + (SCSSCompletion.variableDefaults[p1] || '') + '}';
            };
        };
        SCSSCompletion.prototype.createFunctionProposals = function (proposals, existingNode, sortToEnd, result) {
            for (var _i = 0, proposals_1 = proposals; _i < proposals_1.length; _i++) {
                var p = proposals_1[_i];
                var insertText = p.func.replace(/\[?(\$\w+)\]?/g, this.createReplaceFunction());
                var label = p.func.substr(0, p.func.indexOf('('));
                var item = {
                    label: label,
                    detail: p.func,
                    documentation: p.desc,
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), insertText),
                    insertTextFormat: cssLanguageTypes_1.InsertTextFormat.Snippet,
                    kind: cssLanguageTypes_1.CompletionItemKind.Function
                };
                if (sortToEnd) {
                    item.sortText = 'z';
                }
                result.items.push(item);
            }
            return result;
        };
        SCSSCompletion.prototype.getCompletionsForSelector = function (ruleSet, isNested, result) {
            this.createFunctionProposals(SCSSCompletion.selectorFuncs, null, true, result);
            return _super.prototype.getCompletionsForSelector.call(this, ruleSet, isNested, result);
        };
        SCSSCompletion.prototype.getTermProposals = function (entry, existingNode, result) {
            var functions = SCSSCompletion.builtInFuncs;
            if (entry) {
                functions = functions.filter(function (f) { return !f.type || !entry.restrictions || entry.restrictions.indexOf(f.type) !== -1; });
            }
            this.createFunctionProposals(functions, existingNode, true, result);
            return _super.prototype.getTermProposals.call(this, entry, existingNode, result);
        };
        SCSSCompletion.prototype.getColorProposals = function (entry, existingNode, result) {
            this.createFunctionProposals(SCSSCompletion.colorProposals, existingNode, false, result);
            return _super.prototype.getColorProposals.call(this, entry, existingNode, result);
        };
        SCSSCompletion.prototype.getCompletionsForDeclarationProperty = function (declaration, result) {
            this.getCompletionForAtDirectives(result);
            this.getCompletionsForSelector(null, true, result);
            return _super.prototype.getCompletionsForDeclarationProperty.call(this, declaration, result);
        };
        SCSSCompletion.prototype.getCompletionsForExtendsReference = function (_extendsRef, existingNode, result) {
            var symbols = this.getSymbolContext().findSymbolsAtOffset(this.offset, nodes.ReferenceType.Rule);
            for (var _i = 0, symbols_1 = symbols; _i < symbols_1.length; _i++) {
                var symbol = symbols_1[_i];
                var suggest = {
                    label: symbol.name,
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), symbol.name),
                    kind: cssLanguageTypes_1.CompletionItemKind.Function,
                };
                result.items.push(suggest);
            }
            return result;
        };
        SCSSCompletion.prototype.getCompletionForAtDirectives = function (result) {
            var _a;
            (_a = result.items).push.apply(_a, SCSSCompletion.scssAtDirectives);
            return result;
        };
        SCSSCompletion.prototype.getCompletionForTopLevel = function (result) {
            this.getCompletionForAtDirectives(result);
            this.getCompletionForModuleLoaders(result);
            _super.prototype.getCompletionForTopLevel.call(this, result);
            return result;
        };
        SCSSCompletion.prototype.getCompletionForModuleLoaders = function (result) {
            var _a;
            (_a = result.items).push.apply(_a, SCSSCompletion.scssModuleLoaders);
            return result;
        };
        SCSSCompletion.variableDefaults = {
            '$red': '1',
            '$green': '2',
            '$blue': '3',
            '$alpha': '1.0',
            '$color': '#000000',
            '$weight': '0.5',
            '$hue': '0',
            '$saturation': '0%',
            '$lightness': '0%',
            '$degrees': '0',
            '$amount': '0',
            '$string': '""',
            '$substring': '"s"',
            '$number': '0',
            '$limit': '1'
        };
        SCSSCompletion.colorProposals = [
            { func: 'red($color)', desc: localize('scss.builtin.red', 'Gets the red component of a color.') },
            { func: 'green($color)', desc: localize('scss.builtin.green', 'Gets the green component of a color.') },
            { func: 'blue($color)', desc: localize('scss.builtin.blue', 'Gets the blue component of a color.') },
            { func: 'mix($color, $color, [$weight])', desc: localize('scss.builtin.mix', 'Mixes two colors together.') },
            { func: 'hue($color)', desc: localize('scss.builtin.hue', 'Gets the hue component of a color.') },
            { func: 'saturation($color)', desc: localize('scss.builtin.saturation', 'Gets the saturation component of a color.') },
            { func: 'lightness($color)', desc: localize('scss.builtin.lightness', 'Gets the lightness component of a color.') },
            { func: 'adjust-hue($color, $degrees)', desc: localize('scss.builtin.adjust-hue', 'Changes the hue of a color.') },
            { func: 'lighten($color, $amount)', desc: localize('scss.builtin.lighten', 'Makes a color lighter.') },
            { func: 'darken($color, $amount)', desc: localize('scss.builtin.darken', 'Makes a color darker.') },
            { func: 'saturate($color, $amount)', desc: localize('scss.builtin.saturate', 'Makes a color more saturated.') },
            { func: 'desaturate($color, $amount)', desc: localize('scss.builtin.desaturate', 'Makes a color less saturated.') },
            { func: 'grayscale($color)', desc: localize('scss.builtin.grayscale', 'Converts a color to grayscale.') },
            { func: 'complement($color)', desc: localize('scss.builtin.complement', 'Returns the complement of a color.') },
            { func: 'invert($color)', desc: localize('scss.builtin.invert', 'Returns the inverse of a color.') },
            { func: 'alpha($color)', desc: localize('scss.builtin.alpha', 'Gets the opacity component of a color.') },
            { func: 'opacity($color)', desc: 'Gets the alpha component (opacity) of a color.' },
            { func: 'rgba($color, $alpha)', desc: localize('scss.builtin.rgba', 'Changes the alpha component for a color.') },
            { func: 'opacify($color, $amount)', desc: localize('scss.builtin.opacify', 'Makes a color more opaque.') },
            { func: 'fade-in($color, $amount)', desc: localize('scss.builtin.fade-in', 'Makes a color more opaque.') },
            { func: 'transparentize($color, $amount)', desc: localize('scss.builtin.transparentize', 'Makes a color more transparent.') },
            { func: 'fade-out($color, $amount)', desc: localize('scss.builtin.fade-out', 'Makes a color more transparent.') },
            { func: 'adjust-color($color, [$red], [$green], [$blue], [$hue], [$saturation], [$lightness], [$alpha])', desc: localize('scss.builtin.adjust-color', 'Increases or decreases one or more components of a color.') },
            { func: 'scale-color($color, [$red], [$green], [$blue], [$saturation], [$lightness], [$alpha])', desc: localize('scss.builtin.scale-color', 'Fluidly scales one or more properties of a color.') },
            { func: 'change-color($color, [$red], [$green], [$blue], [$hue], [$saturation], [$lightness], [$alpha])', desc: localize('scss.builtin.change-color', 'Changes one or more properties of a color.') },
            { func: 'ie-hex-str($color)', desc: localize('scss.builtin.ie-hex-str', 'Converts a color into the format understood by IE filters.') }
        ];
        SCSSCompletion.selectorFuncs = [
            { func: 'selector-nest($selectors…)', desc: localize('scss.builtin.selector-nest', 'Nests selector beneath one another like they would be nested in the stylesheet.') },
            { func: 'selector-append($selectors…)', desc: localize('scss.builtin.selector-append', 'Appends selectors to one another without spaces in between.') },
            { func: 'selector-extend($selector, $extendee, $extender)', desc: localize('scss.builtin.selector-extend', 'Extends $extendee with $extender within $selector.') },
            { func: 'selector-replace($selector, $original, $replacement)', desc: localize('scss.builtin.selector-replace', 'Replaces $original with $replacement within $selector.') },
            { func: 'selector-unify($selector1, $selector2)', desc: localize('scss.builtin.selector-unify', 'Unifies two selectors to produce a selector that matches elements matched by both.') },
            { func: 'is-superselector($super, $sub)', desc: localize('scss.builtin.is-superselector', 'Returns whether $super matches all the elements $sub does, and possibly more.') },
            { func: 'simple-selectors($selector)', desc: localize('scss.builtin.simple-selectors', 'Returns the simple selectors that comprise a compound selector.') },
            { func: 'selector-parse($selector)', desc: localize('scss.builtin.selector-parse', 'Parses a selector into the format returned by &.') }
        ];
        SCSSCompletion.builtInFuncs = [
            { func: 'unquote($string)', desc: localize('scss.builtin.unquote', 'Removes quotes from a string.') },
            { func: 'quote($string)', desc: localize('scss.builtin.quote', 'Adds quotes to a string.') },
            { func: 'str-length($string)', desc: localize('scss.builtin.str-length', 'Returns the number of characters in a string.') },
            { func: 'str-insert($string, $insert, $index)', desc: localize('scss.builtin.str-insert', 'Inserts $insert into $string at $index.') },
            { func: 'str-index($string, $substring)', desc: localize('scss.builtin.str-index', 'Returns the index of the first occurance of $substring in $string.') },
            { func: 'str-slice($string, $start-at, [$end-at])', desc: localize('scss.builtin.str-slice', 'Extracts a substring from $string.') },
            { func: 'to-upper-case($string)', desc: localize('scss.builtin.to-upper-case', 'Converts a string to upper case.') },
            { func: 'to-lower-case($string)', desc: localize('scss.builtin.to-lower-case', 'Converts a string to lower case.') },
            { func: 'percentage($number)', desc: localize('scss.builtin.percentage', 'Converts a unitless number to a percentage.'), type: 'percentage' },
            { func: 'round($number)', desc: localize('scss.builtin.round', 'Rounds a number to the nearest whole number.') },
            { func: 'ceil($number)', desc: localize('scss.builtin.ceil', 'Rounds a number up to the next whole number.') },
            { func: 'floor($number)', desc: localize('scss.builtin.floor', 'Rounds a number down to the previous whole number.') },
            { func: 'abs($number)', desc: localize('scss.builtin.abs', 'Returns the absolute value of a number.') },
            { func: 'min($numbers)', desc: localize('scss.builtin.min', 'Finds the minimum of several numbers.') },
            { func: 'max($numbers)', desc: localize('scss.builtin.max', 'Finds the maximum of several numbers.') },
            { func: 'random([$limit])', desc: localize('scss.builtin.random', 'Returns a random number.') },
            { func: 'length($list)', desc: localize('scss.builtin.length', 'Returns the length of a list.') },
            { func: 'nth($list, $n)', desc: localize('scss.builtin.nth', 'Returns a specific item in a list.') },
            { func: 'set-nth($list, $n, $value)', desc: localize('scss.builtin.set-nth', 'Replaces the nth item in a list.') },
            { func: 'join($list1, $list2, [$separator])', desc: localize('scss.builtin.join', 'Joins together two lists into one.') },
            { func: 'append($list1, $val, [$separator])', desc: localize('scss.builtin.append', 'Appends a single value onto the end of a list.') },
            { func: 'zip($lists)', desc: localize('scss.builtin.zip', 'Combines several lists into a single multidimensional list.') },
            { func: 'index($list, $value)', desc: localize('scss.builtin.index', 'Returns the position of a value within a list.') },
            { func: 'list-separator(#list)', desc: localize('scss.builtin.list-separator', 'Returns the separator of a list.') },
            { func: 'map-get($map, $key)', desc: localize('scss.builtin.map-get', 'Returns the value in a map associated with a given key.') },
            { func: 'map-merge($map1, $map2)', desc: localize('scss.builtin.map-merge', 'Merges two maps together into a new map.') },
            { func: 'map-remove($map, $keys)', desc: localize('scss.builtin.map-remove', 'Returns a new map with keys removed.') },
            { func: 'map-keys($map)', desc: localize('scss.builtin.map-keys', 'Returns a list of all keys in a map.') },
            { func: 'map-values($map)', desc: localize('scss.builtin.map-values', 'Returns a list of all values in a map.') },
            { func: 'map-has-key($map, $key)', desc: localize('scss.builtin.map-has-key', 'Returns whether a map has a value associated with a given key.') },
            { func: 'keywords($args)', desc: localize('scss.builtin.keywords', 'Returns the keywords passed to a function that takes variable arguments.') },
            { func: 'feature-exists($feature)', desc: localize('scss.builtin.feature-exists', 'Returns whether a feature exists in the current Sass runtime.') },
            { func: 'variable-exists($name)', desc: localize('scss.builtin.variable-exists', 'Returns whether a variable with the given name exists in the current scope.') },
            { func: 'global-variable-exists($name)', desc: localize('scss.builtin.global-variable-exists', 'Returns whether a variable with the given name exists in the global scope.') },
            { func: 'function-exists($name)', desc: localize('scss.builtin.function-exists', 'Returns whether a function with the given name exists.') },
            { func: 'mixin-exists($name)', desc: localize('scss.builtin.mixin-exists', 'Returns whether a mixin with the given name exists.') },
            { func: 'inspect($value)', desc: localize('scss.builtin.inspect', 'Returns the string representation of a value as it would be represented in Sass.') },
            { func: 'type-of($value)', desc: localize('scss.builtin.type-of', 'Returns the type of a value.') },
            { func: 'unit($number)', desc: localize('scss.builtin.unit', 'Returns the unit(s) associated with a number.') },
            { func: 'unitless($number)', desc: localize('scss.builtin.unitless', 'Returns whether a number has units.') },
            { func: 'comparable($number1, $number2)', desc: localize('scss.builtin.comparable', 'Returns whether two numbers can be added, subtracted, or compared.') },
            { func: 'call($name, $args…)', desc: localize('scss.builtin.call', 'Dynamically calls a Sass function.') }
        ];
        SCSSCompletion.scssAtDirectives = [
            {
                label: "@extend",
                documentation: localize("scss.builtin.@extend", "Inherits the styles of another selector."),
                kind: cssLanguageTypes_1.CompletionItemKind.Keyword
            },
            {
                label: "@at-root",
                documentation: localize("scss.builtin.@at-root", "Causes one or more rules to be emitted at the root of the document."),
                kind: cssLanguageTypes_1.CompletionItemKind.Keyword
            },
            {
                label: "@debug",
                documentation: localize("scss.builtin.@debug", "Prints the value of an expression to the standard error output stream. Useful for debugging complicated Sass files."),
                kind: cssLanguageTypes_1.CompletionItemKind.Keyword
            },
            {
                label: "@warn",
                documentation: localize("scss.builtin.@warn", "Prints the value of an expression to the standard error output stream. Useful for libraries that need to warn users of deprecations or recovering from minor mixin usage mistakes. Warnings can be turned off with the `--quiet` command-line option or the `:quiet` Sass option."),
                kind: cssLanguageTypes_1.CompletionItemKind.Keyword
            },
            {
                label: "@error",
                documentation: localize("scss.builtin.@error", "Throws the value of an expression as a fatal error with stack trace. Useful for validating arguments to mixins and functions."),
                kind: cssLanguageTypes_1.CompletionItemKind.Keyword
            },
            {
                label: "@if",
                documentation: localize("scss.builtin.@if", "Includes the body if the expression does not evaluate to `false` or `null`."),
                insertText: "@if ${1:expr} {\n\t$0\n}",
                insertTextFormat: cssLanguageTypes_1.InsertTextFormat.Snippet,
                kind: cssLanguageTypes_1.CompletionItemKind.Keyword
            },
            {
                label: "@for",
                documentation: localize("scss.builtin.@for", "For loop that repeatedly outputs a set of styles for each `$var` in the `from/through` or `from/to` clause."),
                insertText: "@for \\$${1:var} from ${2:start} ${3|to,through|} ${4:end} {\n\t$0\n}",
                insertTextFormat: cssLanguageTypes_1.InsertTextFormat.Snippet,
                kind: cssLanguageTypes_1.CompletionItemKind.Keyword
            },
            {
                label: "@each",
                documentation: localize("scss.builtin.@each", "Each loop that sets `$var` to each item in the list or map, then outputs the styles it contains using that value of `$var`."),
                insertText: "@each \\$${1:var} in ${2:list} {\n\t$0\n}",
                insertTextFormat: cssLanguageTypes_1.InsertTextFormat.Snippet,
                kind: cssLanguageTypes_1.CompletionItemKind.Keyword
            },
            {
                label: "@while",
                documentation: localize("scss.builtin.@while", "While loop that takes an expression and repeatedly outputs the nested styles until the statement evaluates to `false`."),
                insertText: "@while ${1:condition} {\n\t$0\n}",
                insertTextFormat: cssLanguageTypes_1.InsertTextFormat.Snippet,
                kind: cssLanguageTypes_1.CompletionItemKind.Keyword
            },
            {
                label: "@mixin",
                documentation: localize("scss.builtin.@mixin", "Defines styles that can be re-used throughout the stylesheet with `@include`."),
                insertText: "@mixin ${1:name} {\n\t$0\n}",
                insertTextFormat: cssLanguageTypes_1.InsertTextFormat.Snippet,
                kind: cssLanguageTypes_1.CompletionItemKind.Keyword
            },
            {
                label: "@include",
                documentation: localize("scss.builtin.@include", "Includes the styles defined by another mixin into the current rule."),
                kind: cssLanguageTypes_1.CompletionItemKind.Keyword
            },
            {
                label: "@function",
                documentation: localize("scss.builtin.@function", "Defines complex operations that can be re-used throughout stylesheets."),
                kind: cssLanguageTypes_1.CompletionItemKind.Keyword
            }
        ];
        SCSSCompletion.scssModuleLoaders = [
            {
                label: "@use",
                documentation: localize("scss.builtin.@use", "Loads mixins, functions, and variables from other Sass stylesheets as 'modules', and combines CSS from multiple stylesheets together."),
                references: [{ name: 'Sass documentation', url: 'https://sass-lang.com/documentation/at-rules/use' }],
                insertText: "@use $0;",
                insertTextFormat: cssLanguageTypes_1.InsertTextFormat.Snippet,
                kind: cssLanguageTypes_1.CompletionItemKind.Keyword
            },
            {
                label: "@forward",
                documentation: localize("scss.builtin.@forward", "Loads a Sass stylesheet and makes its mixins, functions, and variables available when this stylesheet is loaded with the @use rule."),
                references: [{ name: 'Sass documentation', url: 'https://sass-lang.com/documentation/at-rules/forward' }],
                insertText: "@forward $0;",
                insertTextFormat: cssLanguageTypes_1.InsertTextFormat.Snippet,
                kind: cssLanguageTypes_1.CompletionItemKind.Keyword
            },
        ];
        SCSSCompletion.scssModuleBuiltIns = [
            {
                label: 'sass:math',
                documentation: localize('scss.builtin.sass:math', 'Provides functions that operate on numbers.'),
                references: [{ name: 'Sass documentation', url: 'https://sass-lang.com/documentation/modules/math' }]
            },
            {
                label: 'sass:string',
                documentation: localize('scss.builtin.sass:string', 'Makes it easy to combine, search, or split apart strings.'),
                references: [{ name: 'Sass documentation', url: 'https://sass-lang.com/documentation/modules/string' }]
            },
            {
                label: 'sass:color',
                documentation: localize('scss.builtin.sass:color', 'Generates new colors based on existing ones, making it easy to build color themes.'),
                references: [{ name: 'Sass documentation', url: 'https://sass-lang.com/documentation/modules/color' }]
            },
            {
                label: 'sass:list',
                documentation: localize('scss.builtin.sass:list', 'Lets you access and modify values in lists.'),
                references: [{ name: 'Sass documentation', url: 'https://sass-lang.com/documentation/modules/list' }]
            },
            {
                label: 'sass:map',
                documentation: localize('scss.builtin.sass:map', 'Makes it possible to look up the value associated with a key in a map, and much more.'),
                references: [{ name: 'Sass documentation', url: 'https://sass-lang.com/documentation/modules/map' }]
            },
            {
                label: 'sass:selector',
                documentation: localize('scss.builtin.sass:selector', 'Provides access to Sass’s powerful selector engine.'),
                references: [{ name: 'Sass documentation', url: 'https://sass-lang.com/documentation/modules/selector' }]
            },
            {
                label: 'sass:meta',
                documentation: localize('scss.builtin.sass:meta', 'Exposes the details of Sass’s inner workings.'),
                references: [{ name: 'Sass documentation', url: 'https://sass-lang.com/documentation/modules/meta' }]
            },
        ];
        return SCSSCompletion;
    }(cssCompletion_1.CSSCompletion));
    exports.SCSSCompletion = SCSSCompletion;
    /**
     * Todo @Pine: Remove this and do it through custom data
     */
    function addReferencesToDocumentation(items) {
        items.forEach(function (i) {
            if (i.documentation && i.references && i.references.length > 0) {
                var markdownDoc = typeof i.documentation === 'string'
                    ? { kind: 'markdown', value: i.documentation }
                    : { kind: 'markdown', value: i.documentation.value };
                markdownDoc.value += '\n\n';
                markdownDoc.value += i.references
                    .map(function (r) {
                    return "[" + r.name + "](" + r.url + ")";
                })
                    .join(' | ');
                i.documentation = markdownDoc;
            }
        });
    }
});

var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/parser/lessScanner',["require", "exports", "./cssScanner"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.LESSScanner = exports.Ellipsis = void 0;
    var scanner = require("./cssScanner");
    var _FSL = '/'.charCodeAt(0);
    var _NWL = '\n'.charCodeAt(0);
    var _CAR = '\r'.charCodeAt(0);
    var _LFD = '\f'.charCodeAt(0);
    var _TIC = '`'.charCodeAt(0);
    var _DOT = '.'.charCodeAt(0);
    var customTokenValue = scanner.TokenType.CustomToken;
    exports.Ellipsis = customTokenValue++;
    var LESSScanner = /** @class */ (function (_super) {
        __extends(LESSScanner, _super);
        function LESSScanner() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        LESSScanner.prototype.scanNext = function (offset) {
            // LESS: escaped JavaScript code `const a = "dddd"`
            var tokenType = this.escapedJavaScript();
            if (tokenType !== null) {
                return this.finishToken(offset, tokenType);
            }
            if (this.stream.advanceIfChars([_DOT, _DOT, _DOT])) {
                return this.finishToken(offset, exports.Ellipsis);
            }
            return _super.prototype.scanNext.call(this, offset);
        };
        LESSScanner.prototype.comment = function () {
            if (_super.prototype.comment.call(this)) {
                return true;
            }
            if (!this.inURL && this.stream.advanceIfChars([_FSL, _FSL])) {
                this.stream.advanceWhileChar(function (ch) {
                    switch (ch) {
                        case _NWL:
                        case _CAR:
                        case _LFD:
                            return false;
                        default:
                            return true;
                    }
                });
                return true;
            }
            else {
                return false;
            }
        };
        LESSScanner.prototype.escapedJavaScript = function () {
            var ch = this.stream.peekChar();
            if (ch === _TIC) {
                this.stream.advance(1);
                this.stream.advanceWhileChar(function (ch) { return ch !== _TIC; });
                return this.stream.advanceIfChar(_TIC) ? scanner.TokenType.EscapedJavaScript : scanner.TokenType.BadEscapedJavaScript;
            }
            return null;
        };
        return LESSScanner;
    }(scanner.Scanner));
    exports.LESSScanner = LESSScanner;
});

var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/parser/lessParser',["require", "exports", "./lessScanner", "./cssScanner", "./cssParser", "./cssNodes", "./cssErrors"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.LESSParser = void 0;
    var lessScanner = require("./lessScanner");
    var cssScanner_1 = require("./cssScanner");
    var cssParser = require("./cssParser");
    var nodes = require("./cssNodes");
    var cssErrors_1 = require("./cssErrors");
    /// <summary>
    /// A parser for LESS
    /// http://lesscss.org/
    /// </summary>
    var LESSParser = /** @class */ (function (_super) {
        __extends(LESSParser, _super);
        function LESSParser() {
            return _super.call(this, new lessScanner.LESSScanner()) || this;
        }
        LESSParser.prototype._parseStylesheetStatement = function (isNested) {
            if (isNested === void 0) { isNested = false; }
            if (this.peek(cssScanner_1.TokenType.AtKeyword)) {
                return this._parseVariableDeclaration()
                    || this._parsePlugin()
                    || _super.prototype._parseStylesheetAtStatement.call(this, isNested);
            }
            return this._tryParseMixinDeclaration()
                || this._tryParseMixinReference()
                || this._parseFunction()
                || this._parseRuleset(true);
        };
        LESSParser.prototype._parseImport = function () {
            if (!this.peekKeyword('@import') && !this.peekKeyword('@import-once') /* deprecated in less 1.4.1 */) {
                return null;
            }
            var node = this.create(nodes.Import);
            this.consumeToken();
            // less 1.4.1: @import (css) "lib"
            if (this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                if (!this.accept(cssScanner_1.TokenType.Ident)) {
                    return this.finish(node, cssErrors_1.ParseError.IdentifierExpected, [cssScanner_1.TokenType.SemiColon]);
                }
                do {
                    if (!this.accept(cssScanner_1.TokenType.Comma)) {
                        break;
                    }
                } while (this.accept(cssScanner_1.TokenType.Ident));
                if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                    return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected, [cssScanner_1.TokenType.SemiColon]);
                }
            }
            if (!node.addChild(this._parseURILiteral()) && !node.addChild(this._parseStringLiteral())) {
                return this.finish(node, cssErrors_1.ParseError.URIOrStringExpected, [cssScanner_1.TokenType.SemiColon]);
            }
            if (!this.peek(cssScanner_1.TokenType.SemiColon) && !this.peek(cssScanner_1.TokenType.EOF)) {
                node.setMedialist(this._parseMediaQueryList());
            }
            return this.finish(node);
        };
        LESSParser.prototype._parsePlugin = function () {
            if (!this.peekKeyword('@plugin')) {
                return null;
            }
            var node = this.createNode(nodes.NodeType.Plugin);
            this.consumeToken(); // @import
            if (!node.addChild(this._parseStringLiteral())) {
                return this.finish(node, cssErrors_1.ParseError.StringLiteralExpected);
            }
            if (!this.accept(cssScanner_1.TokenType.SemiColon)) {
                return this.finish(node, cssErrors_1.ParseError.SemiColonExpected);
            }
            return this.finish(node);
        };
        LESSParser.prototype._parseMediaQuery = function (resyncStopToken) {
            var node = _super.prototype._parseMediaQuery.call(this, resyncStopToken);
            if (!node) {
                var node_1 = this.create(nodes.MediaQuery);
                if (node_1.addChild(this._parseVariable())) {
                    return this.finish(node_1);
                }
                return null;
            }
            return node;
        };
        LESSParser.prototype._parseMediaDeclaration = function (isNested) {
            if (isNested === void 0) { isNested = false; }
            return this._tryParseRuleset(isNested)
                || this._tryToParseDeclaration()
                || this._tryParseMixinDeclaration()
                || this._tryParseMixinReference()
                || this._parseDetachedRuleSetMixin()
                || this._parseStylesheetStatement(isNested);
        };
        LESSParser.prototype._parseMediaFeatureName = function () {
            return this._parseIdent() || this._parseVariable();
        };
        LESSParser.prototype._parseVariableDeclaration = function (panic) {
            if (panic === void 0) { panic = []; }
            var node = this.create(nodes.VariableDeclaration);
            var mark = this.mark();
            if (!node.setVariable(this._parseVariable(true))) {
                return null;
            }
            if (this.accept(cssScanner_1.TokenType.Colon)) {
                if (this.prevToken) {
                    node.colonPosition = this.prevToken.offset;
                }
                if (node.setValue(this._parseDetachedRuleSet())) {
                    node.needsSemicolon = false;
                }
                else if (!node.setValue(this._parseExpr())) {
                    return this.finish(node, cssErrors_1.ParseError.VariableValueExpected, [], panic);
                }
                node.addChild(this._parsePrio());
            }
            else {
                this.restoreAtMark(mark);
                return null; // at keyword, but no ':', not a variable declaration but some at keyword
            }
            if (this.peek(cssScanner_1.TokenType.SemiColon)) {
                node.semicolonPosition = this.token.offset; // not part of the declaration, but useful information for code assist
            }
            return this.finish(node);
        };
        LESSParser.prototype._parseDetachedRuleSet = function () {
            var mark = this.mark();
            // "Anonymous mixin" used in each() and possibly a generic type in the future
            if (this.peekDelim('#') || this.peekDelim('.')) {
                this.consumeToken();
                if (!this.hasWhitespace() && this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                    var node = this.create(nodes.MixinDeclaration);
                    if (node.getParameters().addChild(this._parseMixinParameter())) {
                        while (this.accept(cssScanner_1.TokenType.Comma) || this.accept(cssScanner_1.TokenType.SemiColon)) {
                            if (this.peek(cssScanner_1.TokenType.ParenthesisR)) {
                                break;
                            }
                            if (!node.getParameters().addChild(this._parseMixinParameter())) {
                                this.markError(node, cssErrors_1.ParseError.IdentifierExpected, [], [cssScanner_1.TokenType.ParenthesisR]);
                            }
                        }
                    }
                    if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                        this.restoreAtMark(mark);
                        return null;
                    }
                }
                else {
                    this.restoreAtMark(mark);
                    return null;
                }
            }
            if (!this.peek(cssScanner_1.TokenType.CurlyL)) {
                return null;
            }
            var content = this.create(nodes.BodyDeclaration);
            this._parseBody(content, this._parseDetachedRuleSetBody.bind(this));
            return this.finish(content);
        };
        LESSParser.prototype._parseDetachedRuleSetBody = function () {
            return this._tryParseKeyframeSelector() || this._parseRuleSetDeclaration();
        };
        LESSParser.prototype._addLookupChildren = function (node) {
            if (!node.addChild(this._parseLookupValue())) {
                return false;
            }
            var expectsValue = false;
            while (true) {
                if (this.peek(cssScanner_1.TokenType.BracketL)) {
                    expectsValue = true;
                }
                if (!node.addChild(this._parseLookupValue())) {
                    break;
                }
                expectsValue = false;
            }
            return !expectsValue;
        };
        LESSParser.prototype._parseLookupValue = function () {
            var node = this.create(nodes.Node);
            var mark = this.mark();
            if (!this.accept(cssScanner_1.TokenType.BracketL)) {
                this.restoreAtMark(mark);
                return null;
            }
            if (((node.addChild(this._parseVariable(false, true)) ||
                node.addChild(this._parsePropertyIdentifier())) &&
                this.accept(cssScanner_1.TokenType.BracketR)) || this.accept(cssScanner_1.TokenType.BracketR)) {
                return node;
            }
            this.restoreAtMark(mark);
            return null;
        };
        LESSParser.prototype._parseVariable = function (declaration, insideLookup) {
            if (declaration === void 0) { declaration = false; }
            if (insideLookup === void 0) { insideLookup = false; }
            var isPropertyReference = !declaration && this.peekDelim('$');
            if (!this.peekDelim('@') && !isPropertyReference && !this.peek(cssScanner_1.TokenType.AtKeyword)) {
                return null;
            }
            var node = this.create(nodes.Variable);
            var mark = this.mark();
            while (this.acceptDelim('@') || (!declaration && this.acceptDelim('$'))) {
                if (this.hasWhitespace()) {
                    this.restoreAtMark(mark);
                    return null;
                }
            }
            if (!this.accept(cssScanner_1.TokenType.AtKeyword) && !this.accept(cssScanner_1.TokenType.Ident)) {
                this.restoreAtMark(mark);
                return null;
            }
            if (!insideLookup && this.peek(cssScanner_1.TokenType.BracketL)) {
                if (!this._addLookupChildren(node)) {
                    this.restoreAtMark(mark);
                    return null;
                }
            }
            return node;
        };
        LESSParser.prototype._parseTermExpression = function () {
            return this._parseVariable() ||
                this._parseEscaped() ||
                _super.prototype._parseTermExpression.call(this) || // preference for colors before mixin references
                this._tryParseMixinReference(false);
        };
        LESSParser.prototype._parseEscaped = function () {
            if (this.peek(cssScanner_1.TokenType.EscapedJavaScript) ||
                this.peek(cssScanner_1.TokenType.BadEscapedJavaScript)) {
                var node = this.createNode(nodes.NodeType.EscapedValue);
                this.consumeToken();
                return this.finish(node);
            }
            if (this.peekDelim('~')) {
                var node = this.createNode(nodes.NodeType.EscapedValue);
                this.consumeToken();
                if (this.accept(cssScanner_1.TokenType.String) || this.accept(cssScanner_1.TokenType.EscapedJavaScript)) {
                    return this.finish(node);
                }
                else {
                    return this.finish(node, cssErrors_1.ParseError.TermExpected);
                }
            }
            return null;
        };
        LESSParser.prototype._parseOperator = function () {
            var node = this._parseGuardOperator();
            if (node) {
                return node;
            }
            else {
                return _super.prototype._parseOperator.call(this);
            }
        };
        LESSParser.prototype._parseGuardOperator = function () {
            if (this.peekDelim('>')) {
                var node = this.createNode(nodes.NodeType.Operator);
                this.consumeToken();
                this.acceptDelim('=');
                return node;
            }
            else if (this.peekDelim('=')) {
                var node = this.createNode(nodes.NodeType.Operator);
                this.consumeToken();
                this.acceptDelim('<');
                return node;
            }
            else if (this.peekDelim('<')) {
                var node = this.createNode(nodes.NodeType.Operator);
                this.consumeToken();
                this.acceptDelim('=');
                return node;
            }
            return null;
        };
        LESSParser.prototype._parseRuleSetDeclaration = function () {
            if (this.peek(cssScanner_1.TokenType.AtKeyword)) {
                return this._parseKeyframe()
                    || this._parseMedia(true)
                    || this._parseImport()
                    || this._parseSupports(true) // @supports
                    || this._parseDetachedRuleSetMixin() // less detached ruleset mixin
                    || this._parseVariableDeclaration() // Variable declarations
                    || _super.prototype._parseRuleSetDeclarationAtStatement.call(this);
            }
            return this._tryParseMixinDeclaration()
                || this._tryParseRuleset(true) // nested ruleset
                || this._tryParseMixinReference() // less mixin reference
                || this._parseFunction()
                || this._parseExtend() // less extend declaration
                || _super.prototype._parseRuleSetDeclaration.call(this); // try css ruleset declaration as the last option
        };
        LESSParser.prototype._parseKeyframeIdent = function () {
            return this._parseIdent([nodes.ReferenceType.Keyframe]) || this._parseVariable();
        };
        LESSParser.prototype._parseKeyframeSelector = function () {
            return this._parseDetachedRuleSetMixin() // less detached ruleset mixin
                || _super.prototype._parseKeyframeSelector.call(this);
        };
        LESSParser.prototype._parseSimpleSelectorBody = function () {
            return this._parseSelectorCombinator() || _super.prototype._parseSimpleSelectorBody.call(this);
        };
        LESSParser.prototype._parseSelector = function (isNested) {
            // CSS Guards
            var node = this.create(nodes.Selector);
            var hasContent = false;
            if (isNested) {
                // nested selectors can start with a combinator
                hasContent = node.addChild(this._parseCombinator());
            }
            while (node.addChild(this._parseSimpleSelector())) {
                hasContent = true;
                var mark = this.mark();
                if (node.addChild(this._parseGuard()) && this.peek(cssScanner_1.TokenType.CurlyL)) {
                    break;
                }
                this.restoreAtMark(mark);
                node.addChild(this._parseCombinator()); // optional
            }
            return hasContent ? this.finish(node) : null;
        };
        LESSParser.prototype._parseSelectorCombinator = function () {
            if (this.peekDelim('&')) {
                var node = this.createNode(nodes.NodeType.SelectorCombinator);
                this.consumeToken();
                while (!this.hasWhitespace() && (this.acceptDelim('-') || this.accept(cssScanner_1.TokenType.Num) || this.accept(cssScanner_1.TokenType.Dimension) || node.addChild(this._parseIdent()) || this.acceptDelim('&'))) {
                    //  support &-foo
                }
                return this.finish(node);
            }
            return null;
        };
        LESSParser.prototype._parseSelectorIdent = function () {
            if (!this.peekInterpolatedIdent()) {
                return null;
            }
            var node = this.createNode(nodes.NodeType.SelectorInterpolation);
            var hasContent = this._acceptInterpolatedIdent(node);
            return hasContent ? this.finish(node) : null;
        };
        LESSParser.prototype._parsePropertyIdentifier = function (inLookup) {
            if (inLookup === void 0) { inLookup = false; }
            var propertyRegex = /^[\w-]+/;
            if (!this.peekInterpolatedIdent() && !this.peekRegExp(this.token.type, propertyRegex)) {
                return null;
            }
            var mark = this.mark();
            var node = this.create(nodes.Identifier);
            node.isCustomProperty = this.acceptDelim('-') && this.acceptDelim('-');
            var childAdded = false;
            if (!inLookup) {
                if (node.isCustomProperty) {
                    childAdded = this._acceptInterpolatedIdent(node);
                }
                else {
                    childAdded = this._acceptInterpolatedIdent(node, propertyRegex);
                }
            }
            else {
                if (node.isCustomProperty) {
                    childAdded = node.addChild(this._parseIdent());
                }
                else {
                    childAdded = node.addChild(this._parseRegexp(propertyRegex));
                }
            }
            if (!childAdded) {
                this.restoreAtMark(mark);
                return null;
            }
            if (!inLookup && !this.hasWhitespace()) {
                this.acceptDelim('+');
                if (!this.hasWhitespace()) {
                    this.acceptIdent('_');
                }
            }
            return this.finish(node);
        };
        LESSParser.prototype.peekInterpolatedIdent = function () {
            return this.peek(cssScanner_1.TokenType.Ident) ||
                this.peekDelim('@') ||
                this.peekDelim('$') ||
                this.peekDelim('-');
        };
        LESSParser.prototype._acceptInterpolatedIdent = function (node, identRegex) {
            var _this = this;
            var hasContent = false;
            var indentInterpolation = function () {
                var pos = _this.mark();
                if (_this.acceptDelim('-')) {
                    if (!_this.hasWhitespace()) {
                        _this.acceptDelim('-');
                    }
                    if (_this.hasWhitespace()) {
                        _this.restoreAtMark(pos);
                        return null;
                    }
                }
                return _this._parseInterpolation();
            };
            var accept = identRegex ?
                function () { return _this.acceptRegexp(identRegex); } :
                function () { return _this.accept(cssScanner_1.TokenType.Ident); };
            while (accept() ||
                node.addChild(this._parseInterpolation() ||
                    this.try(indentInterpolation))) {
                hasContent = true;
                if (this.hasWhitespace()) {
                    break;
                }
            }
            return hasContent;
        };
        LESSParser.prototype._parseInterpolation = function () {
            // @{name} Variable or
            // ${name} Property
            var mark = this.mark();
            if (this.peekDelim('@') || this.peekDelim('$')) {
                var node = this.createNode(nodes.NodeType.Interpolation);
                this.consumeToken();
                if (this.hasWhitespace() || !this.accept(cssScanner_1.TokenType.CurlyL)) {
                    this.restoreAtMark(mark);
                    return null;
                }
                if (!node.addChild(this._parseIdent())) {
                    return this.finish(node, cssErrors_1.ParseError.IdentifierExpected);
                }
                if (!this.accept(cssScanner_1.TokenType.CurlyR)) {
                    return this.finish(node, cssErrors_1.ParseError.RightCurlyExpected);
                }
                return this.finish(node);
            }
            return null;
        };
        LESSParser.prototype._tryParseMixinDeclaration = function () {
            var mark = this.mark();
            var node = this.create(nodes.MixinDeclaration);
            if (!node.setIdentifier(this._parseMixinDeclarationIdentifier()) || !this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                this.restoreAtMark(mark);
                return null;
            }
            if (node.getParameters().addChild(this._parseMixinParameter())) {
                while (this.accept(cssScanner_1.TokenType.Comma) || this.accept(cssScanner_1.TokenType.SemiColon)) {
                    if (this.peek(cssScanner_1.TokenType.ParenthesisR)) {
                        break;
                    }
                    if (!node.getParameters().addChild(this._parseMixinParameter())) {
                        this.markError(node, cssErrors_1.ParseError.IdentifierExpected, [], [cssScanner_1.TokenType.ParenthesisR]);
                    }
                }
            }
            if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                this.restoreAtMark(mark);
                return null;
            }
            node.setGuard(this._parseGuard());
            if (!this.peek(cssScanner_1.TokenType.CurlyL)) {
                this.restoreAtMark(mark);
                return null;
            }
            return this._parseBody(node, this._parseMixInBodyDeclaration.bind(this));
        };
        LESSParser.prototype._parseMixInBodyDeclaration = function () {
            return this._parseFontFace() || this._parseRuleSetDeclaration();
        };
        LESSParser.prototype._parseMixinDeclarationIdentifier = function () {
            var identifier;
            if (this.peekDelim('#') || this.peekDelim('.')) {
                identifier = this.create(nodes.Identifier);
                this.consumeToken(); // # or .
                if (this.hasWhitespace() || !identifier.addChild(this._parseIdent())) {
                    return null;
                }
            }
            else if (this.peek(cssScanner_1.TokenType.Hash)) {
                identifier = this.create(nodes.Identifier);
                this.consumeToken(); // TokenType.Hash
            }
            else {
                return null;
            }
            identifier.referenceTypes = [nodes.ReferenceType.Mixin];
            return this.finish(identifier);
        };
        LESSParser.prototype._parsePseudo = function () {
            if (!this.peek(cssScanner_1.TokenType.Colon)) {
                return null;
            }
            var mark = this.mark();
            var node = this.create(nodes.ExtendsReference);
            this.consumeToken(); // :
            if (this.acceptIdent('extend')) {
                return this._completeExtends(node);
            }
            this.restoreAtMark(mark);
            return _super.prototype._parsePseudo.call(this);
        };
        LESSParser.prototype._parseExtend = function () {
            if (!this.peekDelim('&')) {
                return null;
            }
            var mark = this.mark();
            var node = this.create(nodes.ExtendsReference);
            this.consumeToken(); // &
            if (this.hasWhitespace() || !this.accept(cssScanner_1.TokenType.Colon) || !this.acceptIdent('extend')) {
                this.restoreAtMark(mark);
                return null;
            }
            return this._completeExtends(node);
        };
        LESSParser.prototype._completeExtends = function (node) {
            if (!this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                return this.finish(node, cssErrors_1.ParseError.LeftParenthesisExpected);
            }
            var selectors = node.getSelectors();
            if (!selectors.addChild(this._parseSelector(true))) {
                return this.finish(node, cssErrors_1.ParseError.SelectorExpected);
            }
            while (this.accept(cssScanner_1.TokenType.Comma)) {
                if (!selectors.addChild(this._parseSelector(true))) {
                    return this.finish(node, cssErrors_1.ParseError.SelectorExpected);
                }
            }
            if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected);
            }
            return this.finish(node);
        };
        LESSParser.prototype._parseDetachedRuleSetMixin = function () {
            if (!this.peek(cssScanner_1.TokenType.AtKeyword)) {
                return null;
            }
            var mark = this.mark();
            var node = this.create(nodes.MixinReference);
            if (node.addChild(this._parseVariable(true)) && (this.hasWhitespace() || !this.accept(cssScanner_1.TokenType.ParenthesisL))) {
                this.restoreAtMark(mark);
                return null;
            }
            if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected);
            }
            return this.finish(node);
        };
        LESSParser.prototype._tryParseMixinReference = function (atRoot) {
            if (atRoot === void 0) { atRoot = true; }
            var mark = this.mark();
            var node = this.create(nodes.MixinReference);
            var identifier = this._parseMixinDeclarationIdentifier();
            while (identifier) {
                this.acceptDelim('>');
                var nextId = this._parseMixinDeclarationIdentifier();
                if (nextId) {
                    node.getNamespaces().addChild(identifier);
                    identifier = nextId;
                }
                else {
                    break;
                }
            }
            if (!node.setIdentifier(identifier)) {
                this.restoreAtMark(mark);
                return null;
            }
            var hasArguments = false;
            if (this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                hasArguments = true;
                if (node.getArguments().addChild(this._parseMixinArgument())) {
                    while (this.accept(cssScanner_1.TokenType.Comma) || this.accept(cssScanner_1.TokenType.SemiColon)) {
                        if (this.peek(cssScanner_1.TokenType.ParenthesisR)) {
                            break;
                        }
                        if (!node.getArguments().addChild(this._parseMixinArgument())) {
                            return this.finish(node, cssErrors_1.ParseError.ExpressionExpected);
                        }
                    }
                }
                if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                    return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected);
                }
                identifier.referenceTypes = [nodes.ReferenceType.Mixin];
            }
            else {
                identifier.referenceTypes = [nodes.ReferenceType.Mixin, nodes.ReferenceType.Rule];
            }
            if (this.peek(cssScanner_1.TokenType.BracketL)) {
                if (!atRoot) {
                    this._addLookupChildren(node);
                }
            }
            else {
                node.addChild(this._parsePrio());
            }
            if (!hasArguments && !this.peek(cssScanner_1.TokenType.SemiColon) && !this.peek(cssScanner_1.TokenType.CurlyR) && !this.peek(cssScanner_1.TokenType.EOF)) {
                this.restoreAtMark(mark);
                return null;
            }
            return this.finish(node);
        };
        LESSParser.prototype._parseMixinArgument = function () {
            // [variableName ':'] expression | variableName '...'
            var node = this.create(nodes.FunctionArgument);
            var pos = this.mark();
            var argument = this._parseVariable();
            if (argument) {
                if (!this.accept(cssScanner_1.TokenType.Colon)) {
                    this.restoreAtMark(pos);
                }
                else {
                    node.setIdentifier(argument);
                }
            }
            if (node.setValue(this._parseDetachedRuleSet() || this._parseExpr(true))) {
                return this.finish(node);
            }
            this.restoreAtMark(pos);
            return null;
        };
        LESSParser.prototype._parseMixinParameter = function () {
            var node = this.create(nodes.FunctionParameter);
            // special rest variable: @rest...
            if (this.peekKeyword('@rest')) {
                var restNode = this.create(nodes.Node);
                this.consumeToken();
                if (!this.accept(lessScanner.Ellipsis)) {
                    return this.finish(node, cssErrors_1.ParseError.DotExpected, [], [cssScanner_1.TokenType.Comma, cssScanner_1.TokenType.ParenthesisR]);
                }
                node.setIdentifier(this.finish(restNode));
                return this.finish(node);
            }
            // special const args: ...
            if (this.peek(lessScanner.Ellipsis)) {
                var varargsNode = this.create(nodes.Node);
                this.consumeToken();
                node.setIdentifier(this.finish(varargsNode));
                return this.finish(node);
            }
            var hasContent = false;
            // default variable declaration: @param: 12 or @name
            if (node.setIdentifier(this._parseVariable())) {
                this.accept(cssScanner_1.TokenType.Colon);
                hasContent = true;
            }
            if (!node.setDefaultValue(this._parseDetachedRuleSet() || this._parseExpr(true)) && !hasContent) {
                return null;
            }
            return this.finish(node);
        };
        LESSParser.prototype._parseGuard = function () {
            if (!this.peekIdent('when')) {
                return null;
            }
            var node = this.create(nodes.LessGuard);
            this.consumeToken(); // when
            node.isNegated = this.acceptIdent('not');
            if (!node.getConditions().addChild(this._parseGuardCondition())) {
                return this.finish(node, cssErrors_1.ParseError.ConditionExpected);
            }
            while (this.acceptIdent('and') || this.accept(cssScanner_1.TokenType.Comma)) {
                if (!node.getConditions().addChild(this._parseGuardCondition())) {
                    return this.finish(node, cssErrors_1.ParseError.ConditionExpected);
                }
            }
            return this.finish(node);
        };
        LESSParser.prototype._parseGuardCondition = function () {
            if (!this.peek(cssScanner_1.TokenType.ParenthesisL)) {
                return null;
            }
            var node = this.create(nodes.GuardCondition);
            this.consumeToken(); // ParenthesisL
            if (!node.addChild(this._parseExpr())) {
                // empty (?)
            }
            if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected);
            }
            return this.finish(node);
        };
        LESSParser.prototype._parseFunction = function () {
            var pos = this.mark();
            var node = this.create(nodes.Function);
            if (!node.setIdentifier(this._parseFunctionIdentifier())) {
                return null;
            }
            if (this.hasWhitespace() || !this.accept(cssScanner_1.TokenType.ParenthesisL)) {
                this.restoreAtMark(pos);
                return null;
            }
            if (node.getArguments().addChild(this._parseMixinArgument())) {
                while (this.accept(cssScanner_1.TokenType.Comma) || this.accept(cssScanner_1.TokenType.SemiColon)) {
                    if (this.peek(cssScanner_1.TokenType.ParenthesisR)) {
                        break;
                    }
                    if (!node.getArguments().addChild(this._parseMixinArgument())) {
                        return this.finish(node, cssErrors_1.ParseError.ExpressionExpected);
                    }
                }
            }
            if (!this.accept(cssScanner_1.TokenType.ParenthesisR)) {
                return this.finish(node, cssErrors_1.ParseError.RightParenthesisExpected);
            }
            return this.finish(node);
        };
        LESSParser.prototype._parseFunctionIdentifier = function () {
            if (this.peekDelim('%')) {
                var node = this.create(nodes.Identifier);
                node.referenceTypes = [nodes.ReferenceType.Function];
                this.consumeToken();
                return this.finish(node);
            }
            return _super.prototype._parseFunctionIdentifier.call(this);
        };
        LESSParser.prototype._parseURLArgument = function () {
            var pos = this.mark();
            var node = _super.prototype._parseURLArgument.call(this);
            if (!node || !this.peek(cssScanner_1.TokenType.ParenthesisR)) {
                this.restoreAtMark(pos);
                var node_2 = this.create(nodes.Node);
                node_2.addChild(this._parseBinaryExpr());
                return this.finish(node_2);
            }
            return node;
        };
        return LESSParser;
    }(cssParser.Parser));
    exports.LESSParser = LESSParser;
});

var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/services/lessCompletion',["require", "exports", "./cssCompletion", "../cssLanguageTypes", "vscode-nls"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.LESSCompletion = void 0;
    var cssCompletion_1 = require("./cssCompletion");
    var cssLanguageTypes_1 = require("../cssLanguageTypes");
    var nls = require("vscode-nls");
    var localize = nls.loadMessageBundle();
    var LESSCompletion = /** @class */ (function (_super) {
        __extends(LESSCompletion, _super);
        function LESSCompletion(lsOptions, cssDataManager) {
            return _super.call(this, '@', lsOptions, cssDataManager) || this;
        }
        LESSCompletion.prototype.createFunctionProposals = function (proposals, existingNode, sortToEnd, result) {
            for (var _i = 0, proposals_1 = proposals; _i < proposals_1.length; _i++) {
                var p = proposals_1[_i];
                var item = {
                    label: p.name,
                    detail: p.example,
                    documentation: p.description,
                    textEdit: cssLanguageTypes_1.TextEdit.replace(this.getCompletionRange(existingNode), p.name + '($0)'),
                    insertTextFormat: cssLanguageTypes_1.InsertTextFormat.Snippet,
                    kind: cssLanguageTypes_1.CompletionItemKind.Function
                };
                if (sortToEnd) {
                    item.sortText = 'z';
                }
                result.items.push(item);
            }
            return result;
        };
        LESSCompletion.prototype.getTermProposals = function (entry, existingNode, result) {
            var functions = LESSCompletion.builtInProposals;
            if (entry) {
                functions = functions.filter(function (f) { return !f.type || !entry.restrictions || entry.restrictions.indexOf(f.type) !== -1; });
            }
            this.createFunctionProposals(functions, existingNode, true, result);
            return _super.prototype.getTermProposals.call(this, entry, existingNode, result);
        };
        LESSCompletion.prototype.getColorProposals = function (entry, existingNode, result) {
            this.createFunctionProposals(LESSCompletion.colorProposals, existingNode, false, result);
            return _super.prototype.getColorProposals.call(this, entry, existingNode, result);
        };
        LESSCompletion.prototype.getCompletionsForDeclarationProperty = function (declaration, result) {
            this.getCompletionsForSelector(null, true, result);
            return _super.prototype.getCompletionsForDeclarationProperty.call(this, declaration, result);
        };
        LESSCompletion.builtInProposals = [
            // Boolean functions
            {
                'name': 'if',
                'example': 'if(condition, trueValue [, falseValue]);',
                'description': localize('less.builtin.if', 'returns one of two values depending on a condition.')
            },
            {
                'name': 'boolean',
                'example': 'boolean(condition);',
                'description': localize('less.builtin.boolean', '"store" a boolean test for later evaluation in a guard or if().')
            },
            // List functions
            {
                'name': 'length',
                'example': 'length(@list);',
                'description': localize('less.builtin.length', 'returns the number of elements in a value list')
            },
            {
                'name': 'extract',
                'example': 'extract(@list, index);',
                'description': localize('less.builtin.extract', 'returns a value at the specified position in the list')
            },
            {
                'name': 'range',
                'example': 'range([start, ] end [, step]);',
                'description': localize('less.builtin.range', 'generate a list spanning a range of values')
            },
            {
                'name': 'each',
                'example': 'each(@list, ruleset);',
                'description': localize('less.builtin.each', 'bind the evaluation of a ruleset to each member of a list.')
            },
            // Other built-ins
            {
                'name': 'escape',
                'example': 'escape(@string);',
                'description': localize('less.builtin.escape', 'URL encodes a string')
            },
            {
                'name': 'e',
                'example': 'e(@string);',
                'description': localize('less.builtin.e', 'escape string content')
            },
            {
                'name': 'replace',
                'example': 'replace(@string, @pattern, @replacement[, @flags]);',
                'description': localize('less.builtin.replace', 'string replace')
            },
            {
                'name': 'unit',
                'example': 'unit(@dimension, [@unit: \'\']);',
                'description': localize('less.builtin.unit', 'remove or change the unit of a dimension')
            },
            {
                'name': 'color',
                'example': 'color(@string);',
                'description': localize('less.builtin.color', 'parses a string to a color'),
                'type': 'color'
            },
            {
                'name': 'convert',
                'example': 'convert(@value, unit);',
                'description': localize('less.builtin.convert', 'converts numbers from one type into another')
            },
            {
                'name': 'data-uri',
                'example': 'data-uri([mimetype,] url);',
                'description': localize('less.builtin.data-uri', 'inlines a resource and falls back to `url()`'),
                'type': 'url'
            },
            {
                'name': 'abs',
                'description': localize('less.builtin.abs', 'absolute value of a number'),
                'example': 'abs(number);'
            },
            {
                'name': 'acos',
                'description': localize('less.builtin.acos', 'arccosine - inverse of cosine function'),
                'example': 'acos(number);'
            },
            {
                'name': 'asin',
                'description': localize('less.builtin.asin', 'arcsine - inverse of sine function'),
                'example': 'asin(number);'
            },
            {
                'name': 'ceil',
                'example': 'ceil(@number);',
                'description': localize('less.builtin.ceil', 'rounds up to an integer')
            },
            {
                'name': 'cos',
                'description': localize('less.builtin.cos', 'cosine function'),
                'example': 'cos(number);'
            },
            {
                'name': 'floor',
                'description': localize('less.builtin.floor', 'rounds down to an integer'),
                'example': 'floor(@number);'
            },
            {
                'name': 'percentage',
                'description': localize('less.builtin.percentage', 'converts to a %, e.g. 0.5 > 50%'),
                'example': 'percentage(@number);',
                'type': 'percentage'
            },
            {
                'name': 'round',
                'description': localize('less.builtin.round', 'rounds a number to a number of places'),
                'example': 'round(number, [places: 0]);'
            },
            {
                'name': 'sqrt',
                'description': localize('less.builtin.sqrt', 'calculates square root of a number'),
                'example': 'sqrt(number);'
            },
            {
                'name': 'sin',
                'description': localize('less.builtin.sin', 'sine function'),
                'example': 'sin(number);'
            },
            {
                'name': 'tan',
                'description': localize('less.builtin.tan', 'tangent function'),
                'example': 'tan(number);'
            },
            {
                'name': 'atan',
                'description': localize('less.builtin.atan', 'arctangent - inverse of tangent function'),
                'example': 'atan(number);'
            },
            {
                'name': 'pi',
                'description': localize('less.builtin.pi', 'returns pi'),
                'example': 'pi();'
            },
            {
                'name': 'pow',
                'description': localize('less.builtin.pow', 'first argument raised to the power of the second argument'),
                'example': 'pow(@base, @exponent);'
            },
            {
                'name': 'mod',
                'description': localize('less.builtin.mod', 'first argument modulus second argument'),
                'example': 'mod(number, number);'
            },
            {
                'name': 'min',
                'description': localize('less.builtin.min', 'returns the lowest of one or more values'),
                'example': 'min(@x, @y);'
            },
            {
                'name': 'max',
                'description': localize('less.builtin.max', 'returns the lowest of one or more values'),
                'example': 'max(@x, @y);'
            }
        ];
        LESSCompletion.colorProposals = [
            {
                'name': 'argb',
                'example': 'argb(@color);',
                'description': localize('less.builtin.argb', 'creates a #AARRGGBB')
            },
            {
                'name': 'hsl',
                'example': 'hsl(@hue, @saturation, @lightness);',
                'description': localize('less.builtin.hsl', 'creates a color')
            },
            {
                'name': 'hsla',
                'example': 'hsla(@hue, @saturation, @lightness, @alpha);',
                'description': localize('less.builtin.hsla', 'creates a color')
            },
            {
                'name': 'hsv',
                'example': 'hsv(@hue, @saturation, @value);',
                'description': localize('less.builtin.hsv', 'creates a color')
            },
            {
                'name': 'hsva',
                'example': 'hsva(@hue, @saturation, @value, @alpha);',
                'description': localize('less.builtin.hsva', 'creates a color')
            },
            {
                'name': 'hue',
                'example': 'hue(@color);',
                'description': localize('less.builtin.hue', 'returns the `hue` channel of `@color` in the HSL space')
            },
            {
                'name': 'saturation',
                'example': 'saturation(@color);',
                'description': localize('less.builtin.saturation', 'returns the `saturation` channel of `@color` in the HSL space')
            },
            {
                'name': 'lightness',
                'example': 'lightness(@color);',
                'description': localize('less.builtin.lightness', 'returns the `lightness` channel of `@color` in the HSL space')
            },
            {
                'name': 'hsvhue',
                'example': 'hsvhue(@color);',
                'description': localize('less.builtin.hsvhue', 'returns the `hue` channel of `@color` in the HSV space')
            },
            {
                'name': 'hsvsaturation',
                'example': 'hsvsaturation(@color);',
                'description': localize('less.builtin.hsvsaturation', 'returns the `saturation` channel of `@color` in the HSV space')
            },
            {
                'name': 'hsvvalue',
                'example': 'hsvvalue(@color);',
                'description': localize('less.builtin.hsvvalue', 'returns the `value` channel of `@color` in the HSV space')
            },
            {
                'name': 'red',
                'example': 'red(@color);',
                'description': localize('less.builtin.red', 'returns the `red` channel of `@color`')
            },
            {
                'name': 'green',
                'example': 'green(@color);',
                'description': localize('less.builtin.green', 'returns the `green` channel of `@color`')
            },
            {
                'name': 'blue',
                'example': 'blue(@color);',
                'description': localize('less.builtin.blue', 'returns the `blue` channel of `@color`')
            },
            {
                'name': 'alpha',
                'example': 'alpha(@color);',
                'description': localize('less.builtin.alpha', 'returns the `alpha` channel of `@color`')
            },
            {
                'name': 'luma',
                'example': 'luma(@color);',
                'description': localize('less.builtin.luma', 'returns the `luma` value (perceptual brightness) of `@color`')
            },
            {
                'name': 'saturate',
                'example': 'saturate(@color, 10%);',
                'description': localize('less.builtin.saturate', 'return `@color` 10% points more saturated')
            },
            {
                'name': 'desaturate',
                'example': 'desaturate(@color, 10%);',
                'description': localize('less.builtin.desaturate', 'return `@color` 10% points less saturated')
            },
            {
                'name': 'lighten',
                'example': 'lighten(@color, 10%);',
                'description': localize('less.builtin.lighten', 'return `@color` 10% points lighter')
            },
            {
                'name': 'darken',
                'example': 'darken(@color, 10%);',
                'description': localize('less.builtin.darken', 'return `@color` 10% points darker')
            },
            {
                'name': 'fadein',
                'example': 'fadein(@color, 10%);',
                'description': localize('less.builtin.fadein', 'return `@color` 10% points less transparent')
            },
            {
                'name': 'fadeout',
                'example': 'fadeout(@color, 10%);',
                'description': localize('less.builtin.fadeout', 'return `@color` 10% points more transparent')
            },
            {
                'name': 'fade',
                'example': 'fade(@color, 50%);',
                'description': localize('less.builtin.fade', 'return `@color` with 50% transparency')
            },
            {
                'name': 'spin',
                'example': 'spin(@color, 10);',
                'description': localize('less.builtin.spin', 'return `@color` with a 10 degree larger in hue')
            },
            {
                'name': 'mix',
                'example': 'mix(@color1, @color2, [@weight: 50%]);',
                'description': localize('less.builtin.mix', 'return a mix of `@color1` and `@color2`')
            },
            {
                'name': 'greyscale',
                'example': 'greyscale(@color);',
                'description': localize('less.builtin.greyscale', 'returns a grey, 100% desaturated color'),
            },
            {
                'name': 'contrast',
                'example': 'contrast(@color1, [@darkcolor: black], [@lightcolor: white], [@threshold: 43%]);',
                'description': localize('less.builtin.contrast', 'return `@darkcolor` if `@color1 is> 43% luma` otherwise return `@lightcolor`, see notes')
            },
            {
                'name': 'multiply',
                'example': 'multiply(@color1, @color2);'
            },
            {
                'name': 'screen',
                'example': 'screen(@color1, @color2);'
            },
            {
                'name': 'overlay',
                'example': 'overlay(@color1, @color2);'
            },
            {
                'name': 'softlight',
                'example': 'softlight(@color1, @color2);'
            },
            {
                'name': 'hardlight',
                'example': 'hardlight(@color1, @color2);'
            },
            {
                'name': 'difference',
                'example': 'difference(@color1, @color2);'
            },
            {
                'name': 'exclusion',
                'example': 'exclusion(@color1, @color2);'
            },
            {
                'name': 'average',
                'example': 'average(@color1, @color2);'
            },
            {
                'name': 'negation',
                'example': 'negation(@color1, @color2);'
            }
        ];
        return LESSCompletion;
    }(cssCompletion_1.CSSCompletion));
    exports.LESSCompletion = LESSCompletion;
});

(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/services/cssFolding',["require", "exports", "../parser/cssScanner", "../parser/scssScanner", "../parser/lessScanner"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.getFoldingRanges = void 0;
    var cssScanner_1 = require("../parser/cssScanner");
    var scssScanner_1 = require("../parser/scssScanner");
    var lessScanner_1 = require("../parser/lessScanner");
    function getFoldingRanges(document, context) {
        var ranges = computeFoldingRanges(document);
        return limitFoldingRanges(ranges, context);
    }
    exports.getFoldingRanges = getFoldingRanges;
    function computeFoldingRanges(document) {
        function getStartLine(t) {
            return document.positionAt(t.offset).line;
        }
        function getEndLine(t) {
            return document.positionAt(t.offset + t.len).line;
        }
        function getScanner() {
            switch (document.languageId) {
                case 'scss':
                    return new scssScanner_1.SCSSScanner();
                case 'less':
                    return new lessScanner_1.LESSScanner();
                default:
                    return new cssScanner_1.Scanner();
            }
        }
        function tokenToRange(t, kind) {
            var startLine = getStartLine(t);
            var endLine = getEndLine(t);
            if (startLine !== endLine) {
                return {
                    startLine: startLine,
                    endLine: endLine,
                    kind: kind
                };
            }
            else {
                return null;
            }
        }
        var ranges = [];
        var delimiterStack = [];
        var scanner = getScanner();
        scanner.ignoreComment = false;
        scanner.setSource(document.getText());
        var token = scanner.scan();
        var prevToken = null;
        var _loop_1 = function () {
            switch (token.type) {
                case cssScanner_1.TokenType.CurlyL:
                case scssScanner_1.InterpolationFunction:
                    {
                        delimiterStack.push({ line: getStartLine(token), type: 'brace', isStart: true });
                        break;
                    }
                case cssScanner_1.TokenType.CurlyR: {
                    if (delimiterStack.length !== 0) {
                        var prevDelimiter = popPrevStartDelimiterOfType(delimiterStack, 'brace');
                        if (!prevDelimiter) {
                            break;
                        }
                        var endLine = getEndLine(token);
                        if (prevDelimiter.type === 'brace') {
                            /**
                             * Other than the case when curly brace is not on a new line by itself, for example
                             * .foo {
                             *   color: red; }
                             * Use endLine minus one to show ending curly brace
                             */
                            if (prevToken && getEndLine(prevToken) !== endLine) {
                                endLine--;
                            }
                            if (prevDelimiter.line !== endLine) {
                                ranges.push({
                                    startLine: prevDelimiter.line,
                                    endLine: endLine,
                                    kind: undefined
                                });
                            }
                        }
                    }
                    break;
                }
                /**
                 * In CSS, there is no single line comment prefixed with //
                 * All comments are marked as `Comment`
                 */
                case cssScanner_1.TokenType.Comment: {
                    var commentRegionMarkerToDelimiter_1 = function (marker) {
                        if (marker === '#region') {
                            return { line: getStartLine(token), type: 'comment', isStart: true };
                        }
                        else {
                            return { line: getEndLine(token), type: 'comment', isStart: false };
                        }
                    };
                    var getCurrDelimiter = function (token) {
                        var matches = token.text.match(/^\s*\/\*\s*(#region|#endregion)\b\s*(.*?)\s*\*\//);
                        if (matches) {
                            return commentRegionMarkerToDelimiter_1(matches[1]);
                        }
                        else if (document.languageId === 'scss' || document.languageId === 'less') {
                            var matches_1 = token.text.match(/^\s*\/\/\s*(#region|#endregion)\b\s*(.*?)\s*/);
                            if (matches_1) {
                                return commentRegionMarkerToDelimiter_1(matches_1[1]);
                            }
                        }
                        return null;
                    };
                    var currDelimiter = getCurrDelimiter(token);
                    // /* */ comment region folding
                    // All #region and #endregion cases
                    if (currDelimiter) {
                        if (currDelimiter.isStart) {
                            delimiterStack.push(currDelimiter);
                        }
                        else {
                            var prevDelimiter = popPrevStartDelimiterOfType(delimiterStack, 'comment');
                            if (!prevDelimiter) {
                                break;
                            }
                            if (prevDelimiter.type === 'comment') {
                                if (prevDelimiter.line !== currDelimiter.line) {
                                    ranges.push({
                                        startLine: prevDelimiter.line,
                                        endLine: currDelimiter.line,
                                        kind: 'region'
                                    });
                                }
                            }
                        }
                    }
                    // Multiline comment case
                    else {
                        var range = tokenToRange(token, 'comment');
                        if (range) {
                            ranges.push(range);
                        }
                    }
                    break;
                }
            }
            prevToken = token;
            token = scanner.scan();
        };
        while (token.type !== cssScanner_1.TokenType.EOF) {
            _loop_1();
        }
        return ranges;
    }
    function popPrevStartDelimiterOfType(stack, type) {
        if (stack.length === 0) {
            return null;
        }
        for (var i = stack.length - 1; i >= 0; i--) {
            if (stack[i].type === type && stack[i].isStart) {
                return stack.splice(i, 1)[0];
            }
        }
        return null;
    }
    /**
     * - Sort regions
     * - Remove invalid regions (intersections)
     * - If limit exceeds, only return `rangeLimit` amount of ranges
     */
    function limitFoldingRanges(ranges, context) {
        var maxRanges = context && context.rangeLimit || Number.MAX_VALUE;
        var sortedRanges = ranges.sort(function (r1, r2) {
            var diff = r1.startLine - r2.startLine;
            if (diff === 0) {
                diff = r1.endLine - r2.endLine;
            }
            return diff;
        });
        var validRanges = [];
        var prevEndLine = -1;
        sortedRanges.forEach(function (r) {
            if (!(r.startLine < prevEndLine && prevEndLine < r.endLine)) {
                validRanges.push(r);
                prevEndLine = r.endLine;
            }
        });
        if (validRanges.length < maxRanges) {
            return validRanges;
        }
        else {
            return validRanges.slice(0, maxRanges);
        }
    }
});

/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
// file generated from vscode-web-custom-data NPM package
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define('vscode-css-languageservice/data/webCustomData',["require", "exports"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.cssData = void 0;
    exports.cssData = {
        "version": 1.1,
        "properties": [
            {
                "name": "additive-symbols",
                "browsers": [
                    "FF33"
                ],
                "syntax": "[ <integer> && <symbol> ]#",
                "relevance": 50,
                "description": "@counter-style descriptor. Specifies the symbols used by the marker-construction algorithm specified by the system descriptor. Needs to be specified if the counter system is 'additive'.",
                "restrictions": [
                    "integer",
                    "string",
                    "image",
                    "identifier"
                ]
            },
            {
                "name": "align-content",
                "values": [
                    {
                        "name": "center",
                        "description": "Lines are packed toward the center of the flex container."
                    },
                    {
                        "name": "flex-end",
                        "description": "Lines are packed toward the end of the flex container."
                    },
                    {
                        "name": "flex-start",
                        "description": "Lines are packed toward the start of the flex container."
                    },
                    {
                        "name": "space-around",
                        "description": "Lines are evenly distributed in the flex container, with half-size spaces on either end."
                    },
                    {
                        "name": "space-between",
                        "description": "Lines are evenly distributed in the flex container."
                    },
                    {
                        "name": "stretch",
                        "description": "Lines stretch to take up the remaining space."
                    }
                ],
                "syntax": "normal | <baseline-position> | <content-distribution> | <overflow-position>? <content-position>",
                "relevance": 60,
                "description": "Aligns a flex container’s lines within the flex container when there is extra space in the cross-axis, similar to how 'justify-content' aligns individual items within the main-axis.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "align-items",
                "values": [
                    {
                        "name": "baseline",
                        "description": "If the flex item’s inline axis is the same as the cross axis, this value is identical to 'flex-start'. Otherwise, it participates in baseline alignment."
                    },
                    {
                        "name": "center",
                        "description": "The flex item’s margin box is centered in the cross axis within the line."
                    },
                    {
                        "name": "flex-end",
                        "description": "The cross-end margin edge of the flex item is placed flush with the cross-end edge of the line."
                    },
                    {
                        "name": "flex-start",
                        "description": "The cross-start margin edge of the flex item is placed flush with the cross-start edge of the line."
                    },
                    {
                        "name": "stretch",
                        "description": "If the cross size property of the flex item computes to auto, and neither of the cross-axis margins are auto, the flex item is stretched."
                    }
                ],
                "syntax": "normal | stretch | <baseline-position> | [ <overflow-position>? <self-position> ]",
                "relevance": 83,
                "description": "Aligns flex items along the cross axis of the current line of the flex container.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "justify-items",
                "values": [
                    {
                        "name": "auto"
                    },
                    {
                        "name": "normal"
                    },
                    {
                        "name": "end"
                    },
                    {
                        "name": "start"
                    },
                    {
                        "name": "flex-end",
                        "description": "\"Flex items are packed toward the end of the line.\""
                    },
                    {
                        "name": "flex-start",
                        "description": "\"Flex items are packed toward the start of the line.\""
                    },
                    {
                        "name": "self-end",
                        "description": "The item is packed flush to the edge of the alignment container of the end side of the item, in the appropriate axis."
                    },
                    {
                        "name": "self-start",
                        "description": "The item is packed flush to the edge of the alignment container of the start side of the item, in the appropriate axis.."
                    },
                    {
                        "name": "center",
                        "description": "The items are packed flush to each other toward the center of the of the alignment container."
                    },
                    {
                        "name": "left"
                    },
                    {
                        "name": "right"
                    },
                    {
                        "name": "baseline"
                    },
                    {
                        "name": "first baseline"
                    },
                    {
                        "name": "last baseline"
                    },
                    {
                        "name": "stretch",
                        "description": "If the cross size property of the flex item computes to auto, and neither of the cross-axis margins are auto, the flex item is stretched."
                    },
                    {
                        "name": "save"
                    },
                    {
                        "name": "unsave"
                    },
                    {
                        "name": "legacy"
                    }
                ],
                "syntax": "normal | stretch | <baseline-position> | <overflow-position>? [ <self-position> | left | right ] | legacy | legacy && [ left | right | center ]",
                "relevance": 51,
                "description": "Defines the default justify-self for all items of the box, giving them the default way of justifying each box along the appropriate axis",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "justify-self",
                "values": [
                    {
                        "name": "auto"
                    },
                    {
                        "name": "normal"
                    },
                    {
                        "name": "end"
                    },
                    {
                        "name": "start"
                    },
                    {
                        "name": "flex-end",
                        "description": "\"Flex items are packed toward the end of the line.\""
                    },
                    {
                        "name": "flex-start",
                        "description": "\"Flex items are packed toward the start of the line.\""
                    },
                    {
                        "name": "self-end",
                        "description": "The item is packed flush to the edge of the alignment container of the end side of the item, in the appropriate axis."
                    },
                    {
                        "name": "self-start",
                        "description": "The item is packed flush to the edge of the alignment container of the start side of the item, in the appropriate axis.."
                    },
                    {
                        "name": "center",
                        "description": "The items are packed flush to each other toward the center of the of the alignment container."
                    },
                    {
                        "name": "left"
                    },
                    {
                        "name": "right"
                    },
                    {
                        "name": "baseline"
                    },
                    {
                        "name": "first baseline"
                    },
                    {
                        "name": "last baseline"
                    },
                    {
                        "name": "stretch",
                        "description": "If the cross size property of the flex item computes to auto, and neither of the cross-axis margins are auto, the flex item is stretched."
                    },
                    {
                        "name": "save"
                    },
                    {
                        "name": "unsave"
                    }
                ],
                "syntax": "auto | normal | stretch | <baseline-position> | <overflow-position>? [ <self-position> | left | right ]",
                "relevance": 52,
                "description": "Defines the way of justifying a box inside its container along the appropriate axis.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "align-self",
                "values": [
                    {
                        "name": "auto",
                        "description": "Computes to the value of 'align-items' on the element’s parent, or 'stretch' if the element has no parent. On absolutely positioned elements, it computes to itself."
                    },
                    {
                        "name": "baseline",
                        "description": "If the flex item’s inline axis is the same as the cross axis, this value is identical to 'flex-start'. Otherwise, it participates in baseline alignment."
                    },
                    {
                        "name": "center",
                        "description": "The flex item’s margin box is centered in the cross axis within the line."
                    },
                    {
                        "name": "flex-end",
                        "description": "The cross-end margin edge of the flex item is placed flush with the cross-end edge of the line."
                    },
                    {
                        "name": "flex-start",
                        "description": "The cross-start margin edge of the flex item is placed flush with the cross-start edge of the line."
                    },
                    {
                        "name": "stretch",
                        "description": "If the cross size property of the flex item computes to auto, and neither of the cross-axis margins are auto, the flex item is stretched."
                    }
                ],
                "syntax": "auto | normal | stretch | <baseline-position> | <overflow-position>? <self-position>",
                "relevance": 70,
                "description": "Allows the default alignment along the cross axis to be overridden for individual flex items.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "all",
                "browsers": [
                    "E79",
                    "FF27",
                    "S9.1",
                    "C37",
                    "O24"
                ],
                "values": [],
                "syntax": "initial | inherit | unset | revert",
                "relevance": 52,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/all"
                    }
                ],
                "description": "Shorthand that resets all properties except 'direction' and 'unicode-bidi'.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "alt",
                "browsers": [
                    "S9"
                ],
                "values": [],
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/alt"
                    }
                ],
                "description": "Provides alternative text for assistive technology to replace the generated content of a ::before or ::after element.",
                "restrictions": [
                    "string",
                    "enum"
                ]
            },
            {
                "name": "animation",
                "values": [
                    {
                        "name": "alternate",
                        "description": "The animation cycle iterations that are odd counts are played in the normal direction, and the animation cycle iterations that are even counts are played in a reverse direction."
                    },
                    {
                        "name": "alternate-reverse",
                        "description": "The animation cycle iterations that are odd counts are played in the reverse direction, and the animation cycle iterations that are even counts are played in a normal direction."
                    },
                    {
                        "name": "backwards",
                        "description": "The beginning property value (as defined in the first @keyframes at-rule) is applied before the animation is displayed, during the period defined by 'animation-delay'."
                    },
                    {
                        "name": "both",
                        "description": "Both forwards and backwards fill modes are applied."
                    },
                    {
                        "name": "forwards",
                        "description": "The final property value (as defined in the last @keyframes at-rule) is maintained after the animation completes."
                    },
                    {
                        "name": "infinite",
                        "description": "Causes the animation to repeat forever."
                    },
                    {
                        "name": "none",
                        "description": "No animation is performed"
                    },
                    {
                        "name": "normal",
                        "description": "Normal playback."
                    },
                    {
                        "name": "reverse",
                        "description": "All iterations of the animation are played in the reverse direction from the way they were specified."
                    }
                ],
                "syntax": "<single-animation>#",
                "relevance": 80,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/animation"
                    }
                ],
                "description": "Shorthand property combines six of the animation properties into a single property.",
                "restrictions": [
                    "time",
                    "timing-function",
                    "enum",
                    "identifier",
                    "number"
                ]
            },
            {
                "name": "animation-delay",
                "syntax": "<time>#",
                "relevance": 62,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/animation-delay"
                    }
                ],
                "description": "Defines when the animation will start.",
                "restrictions": [
                    "time"
                ]
            },
            {
                "name": "animation-direction",
                "values": [
                    {
                        "name": "alternate",
                        "description": "The animation cycle iterations that are odd counts are played in the normal direction, and the animation cycle iterations that are even counts are played in a reverse direction."
                    },
                    {
                        "name": "alternate-reverse",
                        "description": "The animation cycle iterations that are odd counts are played in the reverse direction, and the animation cycle iterations that are even counts are played in a normal direction."
                    },
                    {
                        "name": "normal",
                        "description": "Normal playback."
                    },
                    {
                        "name": "reverse",
                        "description": "All iterations of the animation are played in the reverse direction from the way they were specified."
                    }
                ],
                "syntax": "<single-animation-direction>#",
                "relevance": 56,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/animation-direction"
                    }
                ],
                "description": "Defines whether or not the animation should play in reverse on alternate cycles.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "animation-duration",
                "syntax": "<time>#",
                "relevance": 65,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/animation-duration"
                    }
                ],
                "description": "Defines the length of time that an animation takes to complete one cycle.",
                "restrictions": [
                    "time"
                ]
            },
            {
                "name": "animation-fill-mode",
                "values": [
                    {
                        "name": "backwards",
                        "description": "The beginning property value (as defined in the first @keyframes at-rule) is applied before the animation is displayed, during the period defined by 'animation-delay'."
                    },
                    {
                        "name": "both",
                        "description": "Both forwards and backwards fill modes are applied."
                    },
                    {
                        "name": "forwards",
                        "description": "The final property value (as defined in the last @keyframes at-rule) is maintained after the animation completes."
                    },
                    {
                        "name": "none",
                        "description": "There is no change to the property value between the time the animation is applied and the time the animation begins playing or after the animation completes."
                    }
                ],
                "syntax": "<single-animation-fill-mode>#",
                "relevance": 62,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/animation-fill-mode"
                    }
                ],
                "description": "Defines what values are applied by the animation outside the time it is executing.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "animation-iteration-count",
                "values": [
                    {
                        "name": "infinite",
                        "description": "Causes the animation to repeat forever."
                    }
                ],
                "syntax": "<single-animation-iteration-count>#",
                "relevance": 59,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/animation-iteration-count"
                    }
                ],
                "description": "Defines the number of times an animation cycle is played. The default value is one, meaning the animation will play from beginning to end once.",
                "restrictions": [
                    "number",
                    "enum"
                ]
            },
            {
                "name": "animation-name",
                "values": [
                    {
                        "name": "none",
                        "description": "No animation is performed"
                    }
                ],
                "syntax": "[ none | <keyframes-name> ]#",
                "relevance": 65,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/animation-name"
                    }
                ],
                "description": "Defines a list of animations that apply. Each name is used to select the keyframe at-rule that provides the property values for the animation.",
                "restrictions": [
                    "identifier",
                    "enum"
                ]
            },
            {
                "name": "animation-play-state",
                "values": [
                    {
                        "name": "paused",
                        "description": "A running animation will be paused."
                    },
                    {
                        "name": "running",
                        "description": "Resume playback of a paused animation."
                    }
                ],
                "syntax": "<single-animation-play-state>#",
                "relevance": 53,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/animation-play-state"
                    }
                ],
                "description": "Defines whether the animation is running or paused.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "animation-timing-function",
                "syntax": "<easing-function>#",
                "relevance": 68,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/animation-timing-function"
                    }
                ],
                "description": "Describes how the animation will progress over one cycle of its duration.",
                "restrictions": [
                    "timing-function"
                ]
            },
            {
                "name": "backface-visibility",
                "values": [
                    {
                        "name": "hidden",
                        "description": "Back side is hidden."
                    },
                    {
                        "name": "visible",
                        "description": "Back side is visible."
                    }
                ],
                "syntax": "visible | hidden",
                "relevance": 59,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/backface-visibility"
                    }
                ],
                "description": "Determines whether or not the 'back' side of a transformed element is visible when facing the viewer. With an identity transform, the front side of an element faces the viewer.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "background",
                "values": [
                    {
                        "name": "fixed",
                        "description": "The background is fixed with regard to the viewport. In paged media where there is no viewport, a 'fixed' background is fixed with respect to the page box and therefore replicated on every page."
                    },
                    {
                        "name": "local",
                        "description": "The background is fixed with regard to the element's contents: if the element has a scrolling mechanism, the background scrolls with the element's contents."
                    },
                    {
                        "name": "none",
                        "description": "A value of 'none' counts as an image layer but draws nothing."
                    },
                    {
                        "name": "scroll",
                        "description": "The background is fixed with regard to the element itself and does not scroll with its contents. (It is effectively attached to the element's border.)"
                    }
                ],
                "syntax": "[ <bg-layer> , ]* <final-bg-layer>",
                "relevance": 93,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/background"
                    }
                ],
                "description": "Shorthand property for setting most background properties at the same place in the style sheet.",
                "restrictions": [
                    "enum",
                    "image",
                    "color",
                    "position",
                    "length",
                    "repeat",
                    "percentage",
                    "box"
                ]
            },
            {
                "name": "background-attachment",
                "values": [
                    {
                        "name": "fixed",
                        "description": "The background is fixed with regard to the viewport. In paged media where there is no viewport, a 'fixed' background is fixed with respect to the page box and therefore replicated on every page."
                    },
                    {
                        "name": "local",
                        "description": "The background is fixed with regard to the element’s contents: if the element has a scrolling mechanism, the background scrolls with the element’s contents."
                    },
                    {
                        "name": "scroll",
                        "description": "The background is fixed with regard to the element itself and does not scroll with its contents. (It is effectively attached to the element’s border.)"
                    }
                ],
                "syntax": "<attachment>#",
                "relevance": 54,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/background-attachment"
                    }
                ],
                "description": "Specifies whether the background images are fixed with regard to the viewport ('fixed') or scroll along with the element ('scroll') or its contents ('local').",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "background-blend-mode",
                "browsers": [
                    "E79",
                    "FF30",
                    "S8",
                    "C35",
                    "O22"
                ],
                "values": [
                    {
                        "name": "normal",
                        "description": "Default attribute which specifies no blending"
                    },
                    {
                        "name": "multiply",
                        "description": "The source color is multiplied by the destination color and replaces the destination."
                    },
                    {
                        "name": "screen",
                        "description": "Multiplies the complements of the backdrop and source color values, then complements the result."
                    },
                    {
                        "name": "overlay",
                        "description": "Multiplies or screens the colors, depending on the backdrop color value."
                    },
                    {
                        "name": "darken",
                        "description": "Selects the darker of the backdrop and source colors."
                    },
                    {
                        "name": "lighten",
                        "description": "Selects the lighter of the backdrop and source colors."
                    },
                    {
                        "name": "color-dodge",
                        "description": "Brightens the backdrop color to reflect the source color."
                    },
                    {
                        "name": "color-burn",
                        "description": "Darkens the backdrop color to reflect the source color."
                    },
                    {
                        "name": "hard-light",
                        "description": "Multiplies or screens the colors, depending on the source color value."
                    },
                    {
                        "name": "soft-light",
                        "description": "Darkens or lightens the colors, depending on the source color value."
                    },
                    {
                        "name": "difference",
                        "description": "Subtracts the darker of the two constituent colors from the lighter color.."
                    },
                    {
                        "name": "exclusion",
                        "description": "Produces an effect similar to that of the Difference mode but lower in contrast."
                    },
                    {
                        "name": "hue",
                        "browsers": [
                            "E79",
                            "FF30",
                            "S8",
                            "C35",
                            "O22"
                        ],
                        "description": "Creates a color with the hue of the source color and the saturation and luminosity of the backdrop color."
                    },
                    {
                        "name": "saturation",
                        "browsers": [
                            "E79",
                            "FF30",
                            "S8",
                            "C35",
                            "O22"
                        ],
                        "description": "Creates a color with the saturation of the source color and the hue and luminosity of the backdrop color."
                    },
                    {
                        "name": "color",
                        "browsers": [
                            "E79",
                            "FF30",
                            "S8",
                            "C35",
                            "O22"
                        ],
                        "description": "Creates a color with the hue and saturation of the source color and the luminosity of the backdrop color."
                    },
                    {
                        "name": "luminosity",
                        "browsers": [
                            "E79",
                            "FF30",
                            "S8",
                            "C35",
                            "O22"
                        ],
                        "description": "Creates a color with the luminosity of the source color and the hue and saturation of the backdrop color."
                    }
                ],
                "syntax": "<blend-mode>#",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/background-blend-mode"
                    }
                ],
                "description": "Defines the blending mode of each background layer.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "background-clip",
                "syntax": "<box>#",
                "relevance": 67,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/background-clip"
                    }
                ],
                "description": "Determines the background painting area.",
                "restrictions": [
                    "box"
                ]
            },
            {
                "name": "background-color",
                "syntax": "<color>",
                "relevance": 94,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/background-color"
                    }
                ],
                "description": "Sets the background color of an element.",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "background-image",
                "values": [
                    {
                        "name": "none",
                        "description": "Counts as an image layer but draws nothing."
                    }
                ],
                "syntax": "<bg-image>#",
                "relevance": 89,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/background-image"
                    }
                ],
                "description": "Sets the background image(s) of an element.",
                "restrictions": [
                    "image",
                    "enum"
                ]
            },
            {
                "name": "background-origin",
                "syntax": "<box>#",
                "relevance": 53,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/background-origin"
                    }
                ],
                "description": "For elements rendered as a single box, specifies the background positioning area. For elements rendered as multiple boxes (e.g., inline boxes on several lines, boxes on several pages) specifies which boxes 'box-decoration-break' operates on to determine the background positioning area(s).",
                "restrictions": [
                    "box"
                ]
            },
            {
                "name": "background-position",
                "syntax": "<bg-position>#",
                "relevance": 88,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/background-position"
                    }
                ],
                "description": "Specifies the initial position of the background image(s) (after any resizing) within their corresponding background positioning area.",
                "restrictions": [
                    "position",
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "background-position-x",
                "values": [
                    {
                        "name": "center",
                        "description": "Equivalent to '50%' ('left 50%') for the horizontal position if the horizontal position is not otherwise specified, or '50%' ('top 50%') for the vertical position if it is."
                    },
                    {
                        "name": "left",
                        "description": "Equivalent to '0%' for the horizontal position if one or two values are given, otherwise specifies the left edge as the origin for the next offset."
                    },
                    {
                        "name": "right",
                        "description": "Equivalent to '100%' for the horizontal position if one or two values are given, otherwise specifies the right edge as the origin for the next offset."
                    }
                ],
                "status": "experimental",
                "syntax": "[ center | [ [ left | right | x-start | x-end ]? <length-percentage>? ]! ]#",
                "relevance": 53,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/background-position-x"
                    }
                ],
                "description": "If background images have been specified, this property specifies their initial position (after any resizing) within their corresponding background positioning area.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "background-position-y",
                "values": [
                    {
                        "name": "bottom",
                        "description": "Equivalent to '100%' for the vertical position if one or two values are given, otherwise specifies the bottom edge as the origin for the next offset."
                    },
                    {
                        "name": "center",
                        "description": "Equivalent to '50%' ('left 50%') for the horizontal position if the horizontal position is not otherwise specified, or '50%' ('top 50%') for the vertical position if it is."
                    },
                    {
                        "name": "top",
                        "description": "Equivalent to '0%' for the vertical position if one or two values are given, otherwise specifies the top edge as the origin for the next offset."
                    }
                ],
                "status": "experimental",
                "syntax": "[ center | [ [ top | bottom | y-start | y-end ]? <length-percentage>? ]! ]#",
                "relevance": 53,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/background-position-y"
                    }
                ],
                "description": "If background images have been specified, this property specifies their initial position (after any resizing) within their corresponding background positioning area.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "background-repeat",
                "values": [],
                "syntax": "<repeat-style>#",
                "relevance": 86,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/background-repeat"
                    }
                ],
                "description": "Specifies how background images are tiled after they have been sized and positioned.",
                "restrictions": [
                    "repeat"
                ]
            },
            {
                "name": "background-size",
                "values": [
                    {
                        "name": "auto",
                        "description": "Resolved by using the image’s intrinsic ratio and the size of the other dimension, or failing that, using the image’s intrinsic size, or failing that, treating it as 100%."
                    },
                    {
                        "name": "contain",
                        "description": "Scale the image, while preserving its intrinsic aspect ratio (if any), to the largest size such that both its width and its height can fit inside the background positioning area."
                    },
                    {
                        "name": "cover",
                        "description": "Scale the image, while preserving its intrinsic aspect ratio (if any), to the smallest size such that both its width and its height can completely cover the background positioning area."
                    }
                ],
                "syntax": "<bg-size>#",
                "relevance": 86,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/background-size"
                    }
                ],
                "description": "Specifies the size of the background images.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "behavior",
                "browsers": [
                    "IE6"
                ],
                "relevance": 50,
                "description": "IE only. Used to extend behaviors of the browser.",
                "restrictions": [
                    "url"
                ]
            },
            {
                "name": "block-size",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C57",
                    "O44"
                ],
                "values": [
                    {
                        "name": "auto",
                        "description": "Depends on the values of other properties."
                    }
                ],
                "syntax": "<'width'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/block-size"
                    }
                ],
                "description": "Logical 'width'. Mapping depends on the element’s 'writing-mode'.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "border",
                "syntax": "<line-width> || <line-style> || <color>",
                "relevance": 96,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border"
                    }
                ],
                "description": "Shorthand property for setting border width, style, and color.",
                "restrictions": [
                    "length",
                    "line-width",
                    "line-style",
                    "color"
                ]
            },
            {
                "name": "border-block-end",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "syntax": "<'border-top-width'> || <'border-top-style'> || <color>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-block-end"
                    }
                ],
                "description": "Logical 'border-bottom'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "length",
                    "line-width",
                    "line-style",
                    "color"
                ]
            },
            {
                "name": "border-block-start",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "syntax": "<'border-top-width'> || <'border-top-style'> || <color>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-block-start"
                    }
                ],
                "description": "Logical 'border-top'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "length",
                    "line-width",
                    "line-style",
                    "color"
                ]
            },
            {
                "name": "border-block-end-color",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "syntax": "<'border-top-color'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-block-end-color"
                    }
                ],
                "description": "Logical 'border-bottom-color'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "border-block-start-color",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "syntax": "<'border-top-color'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-block-start-color"
                    }
                ],
                "description": "Logical 'border-top-color'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "border-block-end-style",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "syntax": "<'border-top-style'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-block-end-style"
                    }
                ],
                "description": "Logical 'border-bottom-style'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "line-style"
                ]
            },
            {
                "name": "border-block-start-style",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "syntax": "<'border-top-style'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-block-start-style"
                    }
                ],
                "description": "Logical 'border-top-style'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "line-style"
                ]
            },
            {
                "name": "border-block-end-width",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "syntax": "<'border-top-width'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-block-end-width"
                    }
                ],
                "description": "Logical 'border-bottom-width'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "length",
                    "line-width"
                ]
            },
            {
                "name": "border-block-start-width",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "syntax": "<'border-top-width'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-block-start-width"
                    }
                ],
                "description": "Logical 'border-top-width'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "length",
                    "line-width"
                ]
            },
            {
                "name": "border-bottom",
                "syntax": "<line-width> || <line-style> || <color>",
                "relevance": 89,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-bottom"
                    }
                ],
                "description": "Shorthand property for setting border width, style and color.",
                "restrictions": [
                    "length",
                    "line-width",
                    "line-style",
                    "color"
                ]
            },
            {
                "name": "border-bottom-color",
                "syntax": "<'border-top-color'>",
                "relevance": 71,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-bottom-color"
                    }
                ],
                "description": "Sets the color of the bottom border.",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "border-bottom-left-radius",
                "syntax": "<length-percentage>{1,2}",
                "relevance": 75,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-bottom-left-radius"
                    }
                ],
                "description": "Defines the radii of the bottom left outer border edge.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "border-bottom-right-radius",
                "syntax": "<length-percentage>{1,2}",
                "relevance": 74,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-bottom-right-radius"
                    }
                ],
                "description": "Defines the radii of the bottom right outer border edge.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "border-bottom-style",
                "syntax": "<line-style>",
                "relevance": 57,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-bottom-style"
                    }
                ],
                "description": "Sets the style of the bottom border.",
                "restrictions": [
                    "line-style"
                ]
            },
            {
                "name": "border-bottom-width",
                "syntax": "<line-width>",
                "relevance": 62,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-bottom-width"
                    }
                ],
                "description": "Sets the thickness of the bottom border.",
                "restrictions": [
                    "length",
                    "line-width"
                ]
            },
            {
                "name": "border-collapse",
                "values": [
                    {
                        "name": "collapse",
                        "description": "Selects the collapsing borders model."
                    },
                    {
                        "name": "separate",
                        "description": "Selects the separated borders border model."
                    }
                ],
                "syntax": "collapse | separate",
                "relevance": 75,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-collapse"
                    }
                ],
                "description": "Selects a table's border model.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "border-color",
                "values": [],
                "syntax": "<color>{1,4}",
                "relevance": 87,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-color"
                    }
                ],
                "description": "The color of the border around all four edges of an element.",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "border-image",
                "values": [
                    {
                        "name": "auto",
                        "description": "If 'auto' is specified then the border image width is the intrinsic width or height (whichever is applicable) of the corresponding image slice. If the image does not have the required intrinsic dimension then the corresponding border-width is used instead."
                    },
                    {
                        "name": "fill",
                        "description": "Causes the middle part of the border-image to be preserved."
                    },
                    {
                        "name": "none",
                        "description": "Use the border styles."
                    },
                    {
                        "name": "repeat",
                        "description": "The image is tiled (repeated) to fill the area."
                    },
                    {
                        "name": "round",
                        "description": "The image is tiled (repeated) to fill the area. If it does not fill the area with a whole number of tiles, the image is rescaled so that it does."
                    },
                    {
                        "name": "space",
                        "description": "The image is tiled (repeated) to fill the area. If it does not fill the area with a whole number of tiles, the extra space is distributed around the tiles."
                    },
                    {
                        "name": "stretch",
                        "description": "The image is stretched to fill the area."
                    },
                    {
                        "name": "url()"
                    }
                ],
                "syntax": "<'border-image-source'> || <'border-image-slice'> [ / <'border-image-width'> | / <'border-image-width'>? / <'border-image-outset'> ]? || <'border-image-repeat'>",
                "relevance": 52,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-image"
                    }
                ],
                "description": "Shorthand property for setting 'border-image-source', 'border-image-slice', 'border-image-width', 'border-image-outset' and 'border-image-repeat'. Omitted values are set to their initial values.",
                "restrictions": [
                    "length",
                    "percentage",
                    "number",
                    "url",
                    "enum"
                ]
            },
            {
                "name": "border-image-outset",
                "syntax": "[ <length> | <number> ]{1,4}",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-image-outset"
                    }
                ],
                "description": "The values specify the amount by which the border image area extends beyond the border box on the top, right, bottom, and left sides respectively. If the fourth value is absent, it is the same as the second. If the third one is also absent, it is the same as the first. If the second one is also absent, it is the same as the first. Numbers represent multiples of the corresponding border-width.",
                "restrictions": [
                    "length",
                    "number"
                ]
            },
            {
                "name": "border-image-repeat",
                "values": [
                    {
                        "name": "repeat",
                        "description": "The image is tiled (repeated) to fill the area."
                    },
                    {
                        "name": "round",
                        "description": "The image is tiled (repeated) to fill the area. If it does not fill the area with a whole number of tiles, the image is rescaled so that it does."
                    },
                    {
                        "name": "space",
                        "description": "The image is tiled (repeated) to fill the area. If it does not fill the area with a whole number of tiles, the extra space is distributed around the tiles."
                    },
                    {
                        "name": "stretch",
                        "description": "The image is stretched to fill the area."
                    }
                ],
                "syntax": "[ stretch | repeat | round | space ]{1,2}",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-image-repeat"
                    }
                ],
                "description": "Specifies how the images for the sides and the middle part of the border image are scaled and tiled. If the second keyword is absent, it is assumed to be the same as the first.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "border-image-slice",
                "values": [
                    {
                        "name": "fill",
                        "description": "Causes the middle part of the border-image to be preserved."
                    }
                ],
                "syntax": "<number-percentage>{1,4} && fill?",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-image-slice"
                    }
                ],
                "description": "Specifies inward offsets from the top, right, bottom, and left edges of the image, dividing it into nine regions: four corners, four edges and a middle.",
                "restrictions": [
                    "number",
                    "percentage"
                ]
            },
            {
                "name": "border-image-source",
                "values": [
                    {
                        "name": "none",
                        "description": "Use the border styles."
                    }
                ],
                "syntax": "none | <image>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-image-source"
                    }
                ],
                "description": "Specifies an image to use instead of the border styles given by the 'border-style' properties and as an additional background layer for the element. If the value is 'none' or if the image cannot be displayed, the border styles will be used.",
                "restrictions": [
                    "image"
                ]
            },
            {
                "name": "border-image-width",
                "values": [
                    {
                        "name": "auto",
                        "description": "The border image width is the intrinsic width or height (whichever is applicable) of the corresponding image slice. If the image does not have the required intrinsic dimension then the corresponding border-width is used instead."
                    }
                ],
                "syntax": "[ <length-percentage> | <number> | auto ]{1,4}",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-image-width"
                    }
                ],
                "description": "The four values of 'border-image-width' specify offsets that are used to divide the border image area into nine parts. They represent inward distances from the top, right, bottom, and left sides of the area, respectively.",
                "restrictions": [
                    "length",
                    "percentage",
                    "number"
                ]
            },
            {
                "name": "border-inline-end",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "syntax": "<'border-top-width'> || <'border-top-style'> || <color>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-inline-end"
                    }
                ],
                "description": "Logical 'border-right'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "length",
                    "line-width",
                    "line-style",
                    "color"
                ]
            },
            {
                "name": "border-inline-start",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "syntax": "<'border-top-width'> || <'border-top-style'> || <color>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-inline-start"
                    }
                ],
                "description": "Logical 'border-left'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "length",
                    "line-width",
                    "line-style",
                    "color"
                ]
            },
            {
                "name": "border-inline-end-color",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "syntax": "<'border-top-color'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-inline-end-color"
                    }
                ],
                "description": "Logical 'border-right-color'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "border-inline-start-color",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "syntax": "<'border-top-color'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-inline-start-color"
                    }
                ],
                "description": "Logical 'border-left-color'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "border-inline-end-style",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "syntax": "<'border-top-style'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-inline-end-style"
                    }
                ],
                "description": "Logical 'border-right-style'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "line-style"
                ]
            },
            {
                "name": "border-inline-start-style",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "syntax": "<'border-top-style'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-inline-start-style"
                    }
                ],
                "description": "Logical 'border-left-style'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "line-style"
                ]
            },
            {
                "name": "border-inline-end-width",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "syntax": "<'border-top-width'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-inline-end-width"
                    }
                ],
                "description": "Logical 'border-right-width'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "length",
                    "line-width"
                ]
            },
            {
                "name": "border-inline-start-width",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "syntax": "<'border-top-width'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-inline-start-width"
                    }
                ],
                "description": "Logical 'border-left-width'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "length",
                    "line-width"
                ]
            },
            {
                "name": "border-left",
                "syntax": "<line-width> || <line-style> || <color>",
                "relevance": 83,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-left"
                    }
                ],
                "description": "Shorthand property for setting border width, style and color",
                "restrictions": [
                    "length",
                    "line-width",
                    "line-style",
                    "color"
                ]
            },
            {
                "name": "border-left-color",
                "syntax": "<color>",
                "relevance": 65,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-left-color"
                    }
                ],
                "description": "Sets the color of the left border.",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "border-left-style",
                "syntax": "<line-style>",
                "relevance": 54,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-left-style"
                    }
                ],
                "description": "Sets the style of the left border.",
                "restrictions": [
                    "line-style"
                ]
            },
            {
                "name": "border-left-width",
                "syntax": "<line-width>",
                "relevance": 58,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-left-width"
                    }
                ],
                "description": "Sets the thickness of the left border.",
                "restrictions": [
                    "length",
                    "line-width"
                ]
            },
            {
                "name": "border-radius",
                "syntax": "<length-percentage>{1,4} [ / <length-percentage>{1,4} ]?",
                "relevance": 92,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-radius"
                    }
                ],
                "description": "Defines the radii of the outer border edge.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "border-right",
                "syntax": "<line-width> || <line-style> || <color>",
                "relevance": 81,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-right"
                    }
                ],
                "description": "Shorthand property for setting border width, style and color",
                "restrictions": [
                    "length",
                    "line-width",
                    "line-style",
                    "color"
                ]
            },
            {
                "name": "border-right-color",
                "syntax": "<color>",
                "relevance": 64,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-right-color"
                    }
                ],
                "description": "Sets the color of the right border.",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "border-right-style",
                "syntax": "<line-style>",
                "relevance": 54,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-right-style"
                    }
                ],
                "description": "Sets the style of the right border.",
                "restrictions": [
                    "line-style"
                ]
            },
            {
                "name": "border-right-width",
                "syntax": "<line-width>",
                "relevance": 60,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-right-width"
                    }
                ],
                "description": "Sets the thickness of the right border.",
                "restrictions": [
                    "length",
                    "line-width"
                ]
            },
            {
                "name": "border-spacing",
                "syntax": "<length> <length>?",
                "relevance": 68,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-spacing"
                    }
                ],
                "description": "The lengths specify the distance that separates adjoining cell borders. If one length is specified, it gives both the horizontal and vertical spacing. If two are specified, the first gives the horizontal spacing and the second the vertical spacing. Lengths may not be negative.",
                "restrictions": [
                    "length"
                ]
            },
            {
                "name": "border-style",
                "values": [],
                "syntax": "<line-style>{1,4}",
                "relevance": 80,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-style"
                    }
                ],
                "description": "The style of the border around edges of an element.",
                "restrictions": [
                    "line-style"
                ]
            },
            {
                "name": "border-top",
                "syntax": "<line-width> || <line-style> || <color>",
                "relevance": 88,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-top"
                    }
                ],
                "description": "Shorthand property for setting border width, style and color",
                "restrictions": [
                    "length",
                    "line-width",
                    "line-style",
                    "color"
                ]
            },
            {
                "name": "border-top-color",
                "syntax": "<color>",
                "relevance": 72,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-top-color"
                    }
                ],
                "description": "Sets the color of the top border.",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "border-top-left-radius",
                "syntax": "<length-percentage>{1,2}",
                "relevance": 75,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-top-left-radius"
                    }
                ],
                "description": "Defines the radii of the top left outer border edge.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "border-top-right-radius",
                "syntax": "<length-percentage>{1,2}",
                "relevance": 73,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-top-right-radius"
                    }
                ],
                "description": "Defines the radii of the top right outer border edge.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "border-top-style",
                "syntax": "<line-style>",
                "relevance": 57,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-top-style"
                    }
                ],
                "description": "Sets the style of the top border.",
                "restrictions": [
                    "line-style"
                ]
            },
            {
                "name": "border-top-width",
                "syntax": "<line-width>",
                "relevance": 61,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-top-width"
                    }
                ],
                "description": "Sets the thickness of the top border.",
                "restrictions": [
                    "length",
                    "line-width"
                ]
            },
            {
                "name": "border-width",
                "values": [],
                "syntax": "<line-width>{1,4}",
                "relevance": 82,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/border-width"
                    }
                ],
                "description": "Shorthand that sets the four 'border-*-width' properties. If it has four values, they set top, right, bottom and left in that order. If left is missing, it is the same as right; if bottom is missing, it is the same as top; if right is missing, it is the same as top.",
                "restrictions": [
                    "length",
                    "line-width"
                ]
            },
            {
                "name": "bottom",
                "values": [
                    {
                        "name": "auto",
                        "description": "For non-replaced elements, the effect of this value depends on which of related properties have the value 'auto' as well"
                    }
                ],
                "syntax": "<length> | <percentage> | auto",
                "relevance": 90,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/bottom"
                    }
                ],
                "description": "Specifies how far an absolutely positioned box's bottom margin edge is offset above the bottom edge of the box's 'containing block'.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "box-decoration-break",
                "browsers": [
                    "E79",
                    "FF32",
                    "S6.1",
                    "C22",
                    "O15"
                ],
                "values": [
                    {
                        "name": "clone",
                        "description": "Each box is independently wrapped with the border and padding."
                    },
                    {
                        "name": "slice",
                        "description": "The effect is as though the element were rendered with no breaks present, and then sliced by the breaks afterward."
                    }
                ],
                "syntax": "slice | clone",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/box-decoration-break"
                    }
                ],
                "description": "Specifies whether individual boxes are treated as broken pieces of one continuous box, or whether each box is individually wrapped with the border and padding.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "box-shadow",
                "values": [
                    {
                        "name": "inset",
                        "description": "Changes the drop shadow from an outer shadow (one that shadows the box onto the canvas, as if it were lifted above the canvas) to an inner shadow (one that shadows the canvas onto the box, as if the box were cut out of the canvas and shifted behind it)."
                    },
                    {
                        "name": "none",
                        "description": "No shadow."
                    }
                ],
                "syntax": "none | <shadow>#",
                "relevance": 90,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/box-shadow"
                    }
                ],
                "description": "Attaches one or more drop-shadows to the box. The property is a comma-separated list of shadows, each specified by 2-4 length values, an optional color, and an optional 'inset' keyword. Omitted lengths are 0; omitted colors are a user agent chosen color.",
                "restrictions": [
                    "length",
                    "color",
                    "enum"
                ]
            },
            {
                "name": "box-sizing",
                "values": [
                    {
                        "name": "border-box",
                        "description": "The specified width and height (and respective min/max properties) on this element determine the border box of the element."
                    },
                    {
                        "name": "content-box",
                        "description": "Behavior of width and height as specified by CSS2.1. The specified width and height (and respective min/max properties) apply to the width and height respectively of the content box of the element."
                    }
                ],
                "syntax": "content-box | border-box",
                "relevance": 92,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/box-sizing"
                    }
                ],
                "description": "Specifies the behavior of the 'width' and 'height' properties.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "break-after",
                "values": [
                    {
                        "name": "always",
                        "description": "Always force a page break before/after the generated box."
                    },
                    {
                        "name": "auto",
                        "description": "Neither force nor forbid a page/column break before/after the principal box."
                    },
                    {
                        "name": "avoid",
                        "description": "Avoid a break before/after the principal box."
                    },
                    {
                        "name": "avoid-column",
                        "description": "Avoid a column break before/after the principal box."
                    },
                    {
                        "name": "avoid-page",
                        "description": "Avoid a page break before/after the principal box."
                    },
                    {
                        "name": "column",
                        "description": "Always force a column break before/after the principal box."
                    },
                    {
                        "name": "left",
                        "description": "Force one or two page breaks before/after the generated box so that the next page is formatted as a left page."
                    },
                    {
                        "name": "page",
                        "description": "Always force a page break before/after the principal box."
                    },
                    {
                        "name": "right",
                        "description": "Force one or two page breaks before/after the generated box so that the next page is formatted as a right page."
                    }
                ],
                "syntax": "auto | avoid | always | all | avoid-page | page | left | right | recto | verso | avoid-column | column | avoid-region | region",
                "relevance": 50,
                "description": "Describes the page/column/region break behavior after the generated box.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "break-before",
                "values": [
                    {
                        "name": "always",
                        "description": "Always force a page break before/after the generated box."
                    },
                    {
                        "name": "auto",
                        "description": "Neither force nor forbid a page/column break before/after the principal box."
                    },
                    {
                        "name": "avoid",
                        "description": "Avoid a break before/after the principal box."
                    },
                    {
                        "name": "avoid-column",
                        "description": "Avoid a column break before/after the principal box."
                    },
                    {
                        "name": "avoid-page",
                        "description": "Avoid a page break before/after the principal box."
                    },
                    {
                        "name": "column",
                        "description": "Always force a column break before/after the principal box."
                    },
                    {
                        "name": "left",
                        "description": "Force one or two page breaks before/after the generated box so that the next page is formatted as a left page."
                    },
                    {
                        "name": "page",
                        "description": "Always force a page break before/after the principal box."
                    },
                    {
                        "name": "right",
                        "description": "Force one or two page breaks before/after the generated box so that the next page is formatted as a right page."
                    }
                ],
                "syntax": "auto | avoid | always | all | avoid-page | page | left | right | recto | verso | avoid-column | column | avoid-region | region",
                "relevance": 50,
                "description": "Describes the page/column/region break behavior before the generated box.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "break-inside",
                "values": [
                    {
                        "name": "auto",
                        "description": "Impose no additional breaking constraints within the box."
                    },
                    {
                        "name": "avoid",
                        "description": "Avoid breaks within the box."
                    },
                    {
                        "name": "avoid-column",
                        "description": "Avoid a column break within the box."
                    },
                    {
                        "name": "avoid-page",
                        "description": "Avoid a page break within the box."
                    }
                ],
                "syntax": "auto | avoid | avoid-page | avoid-column | avoid-region",
                "relevance": 50,
                "description": "Describes the page/column/region break behavior inside the principal box.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "caption-side",
                "values": [
                    {
                        "name": "bottom",
                        "description": "Positions the caption box below the table box."
                    },
                    {
                        "name": "top",
                        "description": "Positions the caption box above the table box."
                    }
                ],
                "syntax": "top | bottom | block-start | block-end | inline-start | inline-end",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/caption-side"
                    }
                ],
                "description": "Specifies the position of the caption box with respect to the table box.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "caret-color",
                "browsers": [
                    "E79",
                    "FF53",
                    "S11.1",
                    "C57",
                    "O44"
                ],
                "values": [
                    {
                        "name": "auto",
                        "description": "The user agent selects an appropriate color for the caret. This is generally currentcolor, but the user agent may choose a different color to ensure good visibility and contrast with the surrounding content, taking into account the value of currentcolor, the background, shadows, and other factors."
                    }
                ],
                "syntax": "auto | <color>",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/caret-color"
                    }
                ],
                "description": "Controls the color of the text insertion indicator.",
                "restrictions": [
                    "color",
                    "enum"
                ]
            },
            {
                "name": "clear",
                "values": [
                    {
                        "name": "both",
                        "description": "The clearance of the generated box is set to the amount necessary to place the top border edge below the bottom outer edge of any right-floating and left-floating boxes that resulted from elements earlier in the source document."
                    },
                    {
                        "name": "left",
                        "description": "The clearance of the generated box is set to the amount necessary to place the top border edge below the bottom outer edge of any left-floating boxes that resulted from elements earlier in the source document."
                    },
                    {
                        "name": "none",
                        "description": "No constraint on the box's position with respect to floats."
                    },
                    {
                        "name": "right",
                        "description": "The clearance of the generated box is set to the amount necessary to place the top border edge below the bottom outer edge of any right-floating boxes that resulted from elements earlier in the source document."
                    }
                ],
                "syntax": "none | left | right | both | inline-start | inline-end",
                "relevance": 85,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/clear"
                    }
                ],
                "description": "Indicates which sides of an element's box(es) may not be adjacent to an earlier floating box. The 'clear' property does not consider floats inside the element itself or in other block formatting contexts.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "clip",
                "values": [
                    {
                        "name": "auto",
                        "description": "The element does not clip."
                    },
                    {
                        "name": "rect()",
                        "description": "Specifies offsets from the edges of the border box."
                    }
                ],
                "syntax": "<shape> | auto",
                "relevance": 73,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/clip"
                    }
                ],
                "description": "Deprecated. Use the 'clip-path' property when support allows. Defines the visible portion of an element’s box.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "clip-path",
                "values": [
                    {
                        "name": "none",
                        "description": "No clipping path gets created."
                    },
                    {
                        "name": "url()",
                        "description": "References a <clipPath> element to create a clipping path."
                    }
                ],
                "syntax": "<clip-source> | [ <basic-shape> || <geometry-box> ] | none",
                "relevance": 55,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/clip-path"
                    }
                ],
                "description": "Specifies a clipping path where everything inside the path is visible and everything outside is clipped out.",
                "restrictions": [
                    "url",
                    "shape",
                    "geometry-box",
                    "enum"
                ]
            },
            {
                "name": "clip-rule",
                "browsers": [
                    "E",
                    "C5",
                    "FF3",
                    "IE10",
                    "O9",
                    "S6"
                ],
                "values": [
                    {
                        "name": "evenodd",
                        "description": "Determines the ‘insideness’ of a point on the canvas by drawing a ray from that point to infinity in any direction and counting the number of path segments from the given shape that the ray crosses."
                    },
                    {
                        "name": "nonzero",
                        "description": "Determines the ‘insideness’ of a point on the canvas by drawing a ray from that point to infinity in any direction and then examining the places where a segment of the shape crosses the ray."
                    }
                ],
                "relevance": 50,
                "description": "Indicates the algorithm which is to be used to determine what parts of the canvas are included inside the shape.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "color",
                "syntax": "<color>",
                "relevance": 95,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/color"
                    }
                ],
                "description": "Sets the color of an element's text",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "color-interpolation-filters",
                "browsers": [
                    "E",
                    "C5",
                    "FF3",
                    "IE10",
                    "O9",
                    "S6"
                ],
                "values": [
                    {
                        "name": "auto",
                        "description": "Color operations are not required to occur in a particular color space."
                    },
                    {
                        "name": "linearRGB",
                        "description": "Color operations should occur in the linearized RGB color space."
                    },
                    {
                        "name": "sRGB",
                        "description": "Color operations should occur in the sRGB color space."
                    }
                ],
                "relevance": 50,
                "description": "Specifies the color space for imaging operations performed via filter effects.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "column-count",
                "values": [
                    {
                        "name": "auto",
                        "description": "Determines the number of columns by the 'column-width' property and the element width."
                    }
                ],
                "syntax": "<integer> | auto",
                "relevance": 52,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/column-count"
                    }
                ],
                "description": "Describes the optimal number of columns into which the content of the element will be flowed.",
                "restrictions": [
                    "integer",
                    "enum"
                ]
            },
            {
                "name": "column-fill",
                "values": [
                    {
                        "name": "auto",
                        "description": "Fills columns sequentially."
                    },
                    {
                        "name": "balance",
                        "description": "Balance content equally between columns, if possible."
                    }
                ],
                "syntax": "auto | balance | balance-all",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/column-fill"
                    }
                ],
                "description": "In continuous media, this property will only be consulted if the length of columns has been constrained. Otherwise, columns will automatically be balanced.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "column-gap",
                "values": [
                    {
                        "name": "normal",
                        "description": "User agent specific and typically equivalent to 1em."
                    }
                ],
                "syntax": "normal | <length-percentage>",
                "relevance": 52,
                "description": "Sets the gap between columns. If there is a column rule between columns, it will appear in the middle of the gap.",
                "restrictions": [
                    "length",
                    "enum"
                ]
            },
            {
                "name": "column-rule",
                "syntax": "<'column-rule-width'> || <'column-rule-style'> || <'column-rule-color'>",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/column-rule"
                    }
                ],
                "description": "Shorthand for setting 'column-rule-width', 'column-rule-style', and 'column-rule-color' at the same place in the style sheet. Omitted values are set to their initial values.",
                "restrictions": [
                    "length",
                    "line-width",
                    "line-style",
                    "color"
                ]
            },
            {
                "name": "column-rule-color",
                "syntax": "<color>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/column-rule-color"
                    }
                ],
                "description": "Sets the color of the column rule",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "column-rule-style",
                "syntax": "<'border-style'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/column-rule-style"
                    }
                ],
                "description": "Sets the style of the rule between columns of an element.",
                "restrictions": [
                    "line-style"
                ]
            },
            {
                "name": "column-rule-width",
                "syntax": "<'border-width'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/column-rule-width"
                    }
                ],
                "description": "Sets the width of the rule between columns. Negative values are not allowed.",
                "restrictions": [
                    "length",
                    "line-width"
                ]
            },
            {
                "name": "columns",
                "values": [
                    {
                        "name": "auto",
                        "description": "The width depends on the values of other properties."
                    }
                ],
                "syntax": "<'column-width'> || <'column-count'>",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/columns"
                    }
                ],
                "description": "A shorthand property which sets both 'column-width' and 'column-count'.",
                "restrictions": [
                    "length",
                    "integer",
                    "enum"
                ]
            },
            {
                "name": "column-span",
                "values": [
                    {
                        "name": "all",
                        "description": "The element spans across all columns. Content in the normal flow that appears before the element is automatically balanced across all columns before the element appear."
                    },
                    {
                        "name": "none",
                        "description": "The element does not span multiple columns."
                    }
                ],
                "syntax": "none | all",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/column-span"
                    }
                ],
                "description": "Describes the page/column break behavior after the generated box.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "column-width",
                "values": [
                    {
                        "name": "auto",
                        "description": "The width depends on the values of other properties."
                    }
                ],
                "syntax": "<length> | auto",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/column-width"
                    }
                ],
                "description": "Describes the width of columns in multicol elements.",
                "restrictions": [
                    "length",
                    "enum"
                ]
            },
            {
                "name": "contain",
                "browsers": [
                    "E79",
                    "FF69",
                    "C52",
                    "O40"
                ],
                "values": [
                    {
                        "name": "none",
                        "description": "Indicates that the property has no effect."
                    },
                    {
                        "name": "strict",
                        "description": "Turns on all forms of containment for the element."
                    },
                    {
                        "name": "content",
                        "description": "All containment rules except size are applied to the element."
                    },
                    {
                        "name": "size",
                        "description": "For properties that can have effects on more than just an element and its descendants, those effects don't escape the containing element."
                    },
                    {
                        "name": "layout",
                        "description": "Turns on layout containment for the element."
                    },
                    {
                        "name": "style",
                        "description": "Turns on style containment for the element."
                    },
                    {
                        "name": "paint",
                        "description": "Turns on paint containment for the element."
                    }
                ],
                "syntax": "none | strict | content | [ size || layout || style || paint ]",
                "relevance": 55,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/contain"
                    }
                ],
                "description": "Indicates that an element and its contents are, as much as possible, independent of the rest of the document tree.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "content",
                "values": [
                    {
                        "name": "attr()",
                        "description": "The attr(n) function returns as a string the value of attribute n for the subject of the selector."
                    },
                    {
                        "name": "counter(name)",
                        "description": "Counters are denoted by identifiers (see the 'counter-increment' and 'counter-reset' properties)."
                    },
                    {
                        "name": "icon",
                        "description": "The (pseudo-)element is replaced in its entirety by the resource referenced by its 'icon' property, and treated as a replaced element."
                    },
                    {
                        "name": "none",
                        "description": "On elements, this inhibits the children of the element from being rendered as children of this element, as if the element was empty. On pseudo-elements it causes the pseudo-element to have no content."
                    },
                    {
                        "name": "normal",
                        "description": "See http://www.w3.org/TR/css3-content/#content for computation rules."
                    },
                    {
                        "name": "url()"
                    }
                ],
                "syntax": "normal | none | [ <content-replacement> | <content-list> ] [/ <string> ]?",
                "relevance": 89,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/content"
                    }
                ],
                "description": "Determines which page-based occurrence of a given element is applied to a counter or string value.",
                "restrictions": [
                    "string",
                    "url"
                ]
            },
            {
                "name": "counter-increment",
                "values": [
                    {
                        "name": "none",
                        "description": "This element does not alter the value of any counters."
                    }
                ],
                "syntax": "[ <custom-ident> <integer>? ]+ | none",
                "relevance": 52,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/counter-increment"
                    }
                ],
                "description": "Manipulate the value of existing counters.",
                "restrictions": [
                    "identifier",
                    "integer"
                ]
            },
            {
                "name": "counter-reset",
                "values": [
                    {
                        "name": "none",
                        "description": "The counter is not modified."
                    }
                ],
                "syntax": "[ <custom-ident> <integer>? ]+ | none",
                "relevance": 52,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/counter-reset"
                    }
                ],
                "description": "Property accepts one or more names of counters (identifiers), each one optionally followed by an integer. The integer gives the value that the counter is set to on each occurrence of the element.",
                "restrictions": [
                    "identifier",
                    "integer"
                ]
            },
            {
                "name": "cursor",
                "values": [
                    {
                        "name": "alias",
                        "description": "Indicates an alias of/shortcut to something is to be created. Often rendered as an arrow with a small curved arrow next to it."
                    },
                    {
                        "name": "all-scroll",
                        "description": "Indicates that the something can be scrolled in any direction. Often rendered as arrows pointing up, down, left, and right with a dot in the middle."
                    },
                    {
                        "name": "auto",
                        "description": "The UA determines the cursor to display based on the current context."
                    },
                    {
                        "name": "cell",
                        "description": "Indicates that a cell or set of cells may be selected. Often rendered as a thick plus-sign with a dot in the middle."
                    },
                    {
                        "name": "col-resize",
                        "description": "Indicates that the item/column can be resized horizontally. Often rendered as arrows pointing left and right with a vertical bar separating them."
                    },
                    {
                        "name": "context-menu",
                        "description": "A context menu is available for the object under the cursor. Often rendered as an arrow with a small menu-like graphic next to it."
                    },
                    {
                        "name": "copy",
                        "description": "Indicates something is to be copied. Often rendered as an arrow with a small plus sign next to it."
                    },
                    {
                        "name": "crosshair",
                        "description": "A simple crosshair (e.g., short line segments resembling a '+' sign). Often used to indicate a two dimensional bitmap selection mode."
                    },
                    {
                        "name": "default",
                        "description": "The platform-dependent default cursor. Often rendered as an arrow."
                    },
                    {
                        "name": "e-resize",
                        "description": "Indicates that east edge is to be moved."
                    },
                    {
                        "name": "ew-resize",
                        "description": "Indicates a bidirectional east-west resize cursor."
                    },
                    {
                        "name": "grab",
                        "description": "Indicates that something can be grabbed."
                    },
                    {
                        "name": "grabbing",
                        "description": "Indicates that something is being grabbed."
                    },
                    {
                        "name": "help",
                        "description": "Help is available for the object under the cursor. Often rendered as a question mark or a balloon."
                    },
                    {
                        "name": "move",
                        "description": "Indicates something is to be moved."
                    },
                    {
                        "name": "-moz-grab",
                        "description": "Indicates that something can be grabbed."
                    },
                    {
                        "name": "-moz-grabbing",
                        "description": "Indicates that something is being grabbed."
                    },
                    {
                        "name": "-moz-zoom-in",
                        "description": "Indicates that something can be zoomed (magnified) in."
                    },
                    {
                        "name": "-moz-zoom-out",
                        "description": "Indicates that something can be zoomed (magnified) out."
                    },
                    {
                        "name": "ne-resize",
                        "description": "Indicates that movement starts from north-east corner."
                    },
                    {
                        "name": "nesw-resize",
                        "description": "Indicates a bidirectional north-east/south-west cursor."
                    },
                    {
                        "name": "no-drop",
                        "description": "Indicates that the dragged item cannot be dropped at the current cursor location. Often rendered as a hand or pointer with a small circle with a line through it."
                    },
                    {
                        "name": "none",
                        "description": "No cursor is rendered for the element."
                    },
                    {
                        "name": "not-allowed",
                        "description": "Indicates that the requested action will not be carried out. Often rendered as a circle with a line through it."
                    },
                    {
                        "name": "n-resize",
                        "description": "Indicates that north edge is to be moved."
                    },
                    {
                        "name": "ns-resize",
                        "description": "Indicates a bidirectional north-south cursor."
                    },
                    {
                        "name": "nw-resize",
                        "description": "Indicates that movement starts from north-west corner."
                    },
                    {
                        "name": "nwse-resize",
                        "description": "Indicates a bidirectional north-west/south-east cursor."
                    },
                    {
                        "name": "pointer",
                        "description": "The cursor is a pointer that indicates a link."
                    },
                    {
                        "name": "progress",
                        "description": "A progress indicator. The program is performing some processing, but is different from 'wait' in that the user may still interact with the program. Often rendered as a spinning beach ball, or an arrow with a watch or hourglass."
                    },
                    {
                        "name": "row-resize",
                        "description": "Indicates that the item/row can be resized vertically. Often rendered as arrows pointing up and down with a horizontal bar separating them."
                    },
                    {
                        "name": "se-resize",
                        "description": "Indicates that movement starts from south-east corner."
                    },
                    {
                        "name": "s-resize",
                        "description": "Indicates that south edge is to be moved."
                    },
                    {
                        "name": "sw-resize",
                        "description": "Indicates that movement starts from south-west corner."
                    },
                    {
                        "name": "text",
                        "description": "Indicates text that may be selected. Often rendered as a vertical I-beam."
                    },
                    {
                        "name": "vertical-text",
                        "description": "Indicates vertical-text that may be selected. Often rendered as a horizontal I-beam."
                    },
                    {
                        "name": "wait",
                        "description": "Indicates that the program is busy and the user should wait. Often rendered as a watch or hourglass."
                    },
                    {
                        "name": "-webkit-grab",
                        "description": "Indicates that something can be grabbed."
                    },
                    {
                        "name": "-webkit-grabbing",
                        "description": "Indicates that something is being grabbed."
                    },
                    {
                        "name": "-webkit-zoom-in",
                        "description": "Indicates that something can be zoomed (magnified) in."
                    },
                    {
                        "name": "-webkit-zoom-out",
                        "description": "Indicates that something can be zoomed (magnified) out."
                    },
                    {
                        "name": "w-resize",
                        "description": "Indicates that west edge is to be moved."
                    },
                    {
                        "name": "zoom-in",
                        "description": "Indicates that something can be zoomed (magnified) in."
                    },
                    {
                        "name": "zoom-out",
                        "description": "Indicates that something can be zoomed (magnified) out."
                    }
                ],
                "syntax": "[ [ <url> [ <x> <y> ]? , ]* [ auto | default | none | context-menu | help | pointer | progress | wait | cell | crosshair | text | vertical-text | alias | copy | move | no-drop | not-allowed | e-resize | n-resize | ne-resize | nw-resize | s-resize | se-resize | sw-resize | w-resize | ew-resize | ns-resize | nesw-resize | nwse-resize | col-resize | row-resize | all-scroll | zoom-in | zoom-out | grab | grabbing ] ]",
                "relevance": 92,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/cursor"
                    }
                ],
                "description": "Allows control over cursor appearance in an element",
                "restrictions": [
                    "url",
                    "number",
                    "enum"
                ]
            },
            {
                "name": "direction",
                "values": [
                    {
                        "name": "ltr",
                        "description": "Left-to-right direction."
                    },
                    {
                        "name": "rtl",
                        "description": "Right-to-left direction."
                    }
                ],
                "syntax": "ltr | rtl",
                "relevance": 69,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/direction"
                    }
                ],
                "description": "Specifies the inline base direction or directionality of any bidi paragraph, embedding, isolate, or override established by the box. Note: for HTML content use the 'dir' attribute and 'bdo' element rather than this property.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "display",
                "values": [
                    {
                        "name": "block",
                        "description": "The element generates a block-level box"
                    },
                    {
                        "name": "contents",
                        "description": "The element itself does not generate any boxes, but its children and pseudo-elements still generate boxes as normal."
                    },
                    {
                        "name": "flex",
                        "description": "The element generates a principal flex container box and establishes a flex formatting context."
                    },
                    {
                        "name": "flexbox",
                        "description": "The element lays out its contents using flow layout (block-and-inline layout). Standardized as 'flex'."
                    },
                    {
                        "name": "flow-root",
                        "description": "The element generates a block container box, and lays out its contents using flow layout."
                    },
                    {
                        "name": "grid",
                        "description": "The element generates a principal grid container box, and establishes a grid formatting context."
                    },
                    {
                        "name": "inline",
                        "description": "The element generates an inline-level box."
                    },
                    {
                        "name": "inline-block",
                        "description": "A block box, which itself is flowed as a single inline box, similar to a replaced element. The inside of an inline-block is formatted as a block box, and the box itself is formatted as an inline box."
                    },
                    {
                        "name": "inline-flex",
                        "description": "Inline-level flex container."
                    },
                    {
                        "name": "inline-flexbox",
                        "description": "Inline-level flex container. Standardized as 'inline-flex'"
                    },
                    {
                        "name": "inline-table",
                        "description": "Inline-level table wrapper box containing table box."
                    },
                    {
                        "name": "list-item",
                        "description": "One or more block boxes and one marker box."
                    },
                    {
                        "name": "-moz-box",
                        "description": "The element lays out its contents using flow layout (block-and-inline layout). Standardized as 'flex'."
                    },
                    {
                        "name": "-moz-deck"
                    },
                    {
                        "name": "-moz-grid"
                    },
                    {
                        "name": "-moz-grid-group"
                    },
                    {
                        "name": "-moz-grid-line"
                    },
                    {
                        "name": "-moz-groupbox"
                    },
                    {
                        "name": "-moz-inline-box",
                        "description": "Inline-level flex container. Standardized as 'inline-flex'"
                    },
                    {
                        "name": "-moz-inline-grid"
                    },
                    {
                        "name": "-moz-inline-stack"
                    },
                    {
                        "name": "-moz-marker"
                    },
                    {
                        "name": "-moz-popup"
                    },
                    {
                        "name": "-moz-stack"
                    },
                    {
                        "name": "-ms-flexbox",
                        "description": "The element lays out its contents using flow layout (block-and-inline layout). Standardized as 'flex'."
                    },
                    {
                        "name": "-ms-grid",
                        "description": "The element generates a principal grid container box, and establishes a grid formatting context."
                    },
                    {
                        "name": "-ms-inline-flexbox",
                        "description": "Inline-level flex container. Standardized as 'inline-flex'"
                    },
                    {
                        "name": "-ms-inline-grid",
                        "description": "Inline-level grid container."
                    },
                    {
                        "name": "none",
                        "description": "The element and its descendants generates no boxes."
                    },
                    {
                        "name": "ruby",
                        "description": "The element generates a principal ruby container box, and establishes a ruby formatting context."
                    },
                    {
                        "name": "ruby-base"
                    },
                    {
                        "name": "ruby-base-container"
                    },
                    {
                        "name": "ruby-text"
                    },
                    {
                        "name": "ruby-text-container"
                    },
                    {
                        "name": "run-in",
                        "description": "The element generates a run-in box. Run-in elements act like inlines or blocks, depending on the surrounding elements."
                    },
                    {
                        "name": "table",
                        "description": "The element generates a principal table wrapper box containing an additionally-generated table box, and establishes a table formatting context."
                    },
                    {
                        "name": "table-caption"
                    },
                    {
                        "name": "table-cell"
                    },
                    {
                        "name": "table-column"
                    },
                    {
                        "name": "table-column-group"
                    },
                    {
                        "name": "table-footer-group"
                    },
                    {
                        "name": "table-header-group"
                    },
                    {
                        "name": "table-row"
                    },
                    {
                        "name": "table-row-group"
                    },
                    {
                        "name": "-webkit-box",
                        "description": "The element lays out its contents using flow layout (block-and-inline layout). Standardized as 'flex'."
                    },
                    {
                        "name": "-webkit-flex",
                        "description": "The element lays out its contents using flow layout (block-and-inline layout)."
                    },
                    {
                        "name": "-webkit-inline-box",
                        "description": "Inline-level flex container. Standardized as 'inline-flex'"
                    },
                    {
                        "name": "-webkit-inline-flex",
                        "description": "Inline-level flex container."
                    }
                ],
                "syntax": "[ <display-outside> || <display-inside> ] | <display-listitem> | <display-internal> | <display-box> | <display-legacy>",
                "relevance": 96,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/display"
                    }
                ],
                "description": "In combination with 'float' and 'position', determines the type of box or boxes that are generated for an element.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "empty-cells",
                "values": [
                    {
                        "name": "hide",
                        "description": "No borders or backgrounds are drawn around/behind empty cells."
                    },
                    {
                        "name": "-moz-show-background"
                    },
                    {
                        "name": "show",
                        "description": "Borders and backgrounds are drawn around/behind empty cells (like normal cells)."
                    }
                ],
                "syntax": "show | hide",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/empty-cells"
                    }
                ],
                "description": "In the separated borders model, this property controls the rendering of borders and backgrounds around cells that have no visible content.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "enable-background",
                "values": [
                    {
                        "name": "accumulate",
                        "description": "If the ancestor container element has a property of new, then all graphics elements within the current container are rendered both on the parent's background image and onto the target."
                    },
                    {
                        "name": "new",
                        "description": "Create a new background image canvas. All children of the current container element can access the background, and they will be rendered onto both the parent's background image canvas in addition to the target device."
                    }
                ],
                "relevance": 50,
                "description": "Deprecated. Use 'isolation' property instead when support allows. Specifies how the accumulation of the background image is managed.",
                "restrictions": [
                    "integer",
                    "length",
                    "percentage",
                    "enum"
                ]
            },
            {
                "name": "fallback",
                "browsers": [
                    "FF33"
                ],
                "syntax": "<counter-style-name>",
                "relevance": 50,
                "description": "@counter-style descriptor. Specifies a fallback counter style to be used when the current counter style can’t create a representation for a given counter value.",
                "restrictions": [
                    "identifier"
                ]
            },
            {
                "name": "fill",
                "values": [
                    {
                        "name": "url()",
                        "description": "A URL reference to a paint server element, which is an element that defines a paint server: ‘hatch’, ‘linearGradient’, ‘mesh’, ‘pattern’, ‘radialGradient’ and ‘solidcolor’."
                    },
                    {
                        "name": "none",
                        "description": "No paint is applied in this layer."
                    }
                ],
                "relevance": 75,
                "description": "Paints the interior of the given graphical element.",
                "restrictions": [
                    "color",
                    "enum",
                    "url"
                ]
            },
            {
                "name": "fill-opacity",
                "relevance": 52,
                "description": "Specifies the opacity of the painting operation used to paint the interior the current object.",
                "restrictions": [
                    "number(0-1)"
                ]
            },
            {
                "name": "fill-rule",
                "values": [
                    {
                        "name": "evenodd",
                        "description": "Determines the ‘insideness’ of a point on the canvas by drawing a ray from that point to infinity in any direction and counting the number of path segments from the given shape that the ray crosses."
                    },
                    {
                        "name": "nonzero",
                        "description": "Determines the ‘insideness’ of a point on the canvas by drawing a ray from that point to infinity in any direction and then examining the places where a segment of the shape crosses the ray."
                    }
                ],
                "relevance": 50,
                "description": "Indicates the algorithm (or winding rule) which is to be used to determine what parts of the canvas are included inside the shape.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "filter",
                "browsers": [
                    "E12",
                    "FF35",
                    "S9.1",
                    "C53",
                    "O40"
                ],
                "values": [
                    {
                        "name": "none",
                        "description": "No filter effects are applied."
                    },
                    {
                        "name": "blur()",
                        "description": "Applies a Gaussian blur to the input image."
                    },
                    {
                        "name": "brightness()",
                        "description": "Applies a linear multiplier to input image, making it appear more or less bright."
                    },
                    {
                        "name": "contrast()",
                        "description": "Adjusts the contrast of the input."
                    },
                    {
                        "name": "drop-shadow()",
                        "description": "Applies a drop shadow effect to the input image."
                    },
                    {
                        "name": "grayscale()",
                        "description": "Converts the input image to grayscale."
                    },
                    {
                        "name": "hue-rotate()",
                        "description": "Applies a hue rotation on the input image. "
                    },
                    {
                        "name": "invert()",
                        "description": "Inverts the samples in the input image."
                    },
                    {
                        "name": "opacity()",
                        "description": "Applies transparency to the samples in the input image."
                    },
                    {
                        "name": "saturate()",
                        "description": "Saturates the input image."
                    },
                    {
                        "name": "sepia()",
                        "description": "Converts the input image to sepia."
                    },
                    {
                        "name": "url()",
                        "browsers": [
                            "E12",
                            "FF35",
                            "S9.1",
                            "C53",
                            "O40"
                        ],
                        "description": "A filter reference to a <filter> element."
                    }
                ],
                "syntax": "none | <filter-function-list>",
                "relevance": 65,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/filter"
                    }
                ],
                "description": "Processes an element’s rendering before it is displayed in the document, by applying one or more filter effects.",
                "restrictions": [
                    "enum",
                    "url"
                ]
            },
            {
                "name": "flex",
                "values": [
                    {
                        "name": "auto",
                        "description": "Retrieves the value of the main size property as the used 'flex-basis'."
                    },
                    {
                        "name": "content",
                        "description": "Indicates automatic sizing, based on the flex item’s content."
                    },
                    {
                        "name": "none",
                        "description": "Expands to '0 0 auto'."
                    }
                ],
                "syntax": "none | [ <'flex-grow'> <'flex-shrink'>? || <'flex-basis'> ]",
                "relevance": 78,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/flex"
                    }
                ],
                "description": "Specifies the components of a flexible length: the flex grow factor and flex shrink factor, and the flex basis.",
                "restrictions": [
                    "length",
                    "number",
                    "percentage"
                ]
            },
            {
                "name": "flex-basis",
                "values": [
                    {
                        "name": "auto",
                        "description": "Retrieves the value of the main size property as the used 'flex-basis'."
                    },
                    {
                        "name": "content",
                        "description": "Indicates automatic sizing, based on the flex item’s content."
                    }
                ],
                "syntax": "content | <'width'>",
                "relevance": 63,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/flex-basis"
                    }
                ],
                "description": "Sets the flex basis.",
                "restrictions": [
                    "length",
                    "number",
                    "percentage"
                ]
            },
            {
                "name": "flex-direction",
                "values": [
                    {
                        "name": "column",
                        "description": "The flex container’s main axis has the same orientation as the block axis of the current writing mode."
                    },
                    {
                        "name": "column-reverse",
                        "description": "Same as 'column', except the main-start and main-end directions are swapped."
                    },
                    {
                        "name": "row",
                        "description": "The flex container’s main axis has the same orientation as the inline axis of the current writing mode."
                    },
                    {
                        "name": "row-reverse",
                        "description": "Same as 'row', except the main-start and main-end directions are swapped."
                    }
                ],
                "syntax": "row | row-reverse | column | column-reverse",
                "relevance": 80,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/flex-direction"
                    }
                ],
                "description": "Specifies how flex items are placed in the flex container, by setting the direction of the flex container’s main axis.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "flex-flow",
                "values": [
                    {
                        "name": "column",
                        "description": "The flex container’s main axis has the same orientation as the block axis of the current writing mode."
                    },
                    {
                        "name": "column-reverse",
                        "description": "Same as 'column', except the main-start and main-end directions are swapped."
                    },
                    {
                        "name": "nowrap",
                        "description": "The flex container is single-line."
                    },
                    {
                        "name": "row",
                        "description": "The flex container’s main axis has the same orientation as the inline axis of the current writing mode."
                    },
                    {
                        "name": "row-reverse",
                        "description": "Same as 'row', except the main-start and main-end directions are swapped."
                    },
                    {
                        "name": "wrap",
                        "description": "The flexbox is multi-line."
                    },
                    {
                        "name": "wrap-reverse",
                        "description": "Same as 'wrap', except the cross-start and cross-end directions are swapped."
                    }
                ],
                "syntax": "<'flex-direction'> || <'flex-wrap'>",
                "relevance": 59,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/flex-flow"
                    }
                ],
                "description": "Specifies how flexbox items are placed in the flexbox.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "flex-grow",
                "syntax": "<number>",
                "relevance": 73,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/flex-grow"
                    }
                ],
                "description": "Sets the flex grow factor. Negative numbers are invalid.",
                "restrictions": [
                    "number"
                ]
            },
            {
                "name": "flex-shrink",
                "syntax": "<number>",
                "relevance": 71,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/flex-shrink"
                    }
                ],
                "description": "Sets the flex shrink factor. Negative numbers are invalid.",
                "restrictions": [
                    "number"
                ]
            },
            {
                "name": "flex-wrap",
                "values": [
                    {
                        "name": "nowrap",
                        "description": "The flex container is single-line."
                    },
                    {
                        "name": "wrap",
                        "description": "The flexbox is multi-line."
                    },
                    {
                        "name": "wrap-reverse",
                        "description": "Same as 'wrap', except the cross-start and cross-end directions are swapped."
                    }
                ],
                "syntax": "nowrap | wrap | wrap-reverse",
                "relevance": 76,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/flex-wrap"
                    }
                ],
                "description": "Controls whether the flex container is single-line or multi-line, and the direction of the cross-axis, which determines the direction new lines are stacked in.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "float",
                "values": [
                    {
                        "name": "inline-end",
                        "description": "A keyword indicating that the element must float on the end side of its containing block. That is the right side with ltr scripts, and the left side with rtl scripts."
                    },
                    {
                        "name": "inline-start",
                        "description": "A keyword indicating that the element must float on the start side of its containing block. That is the left side with ltr scripts, and the right side with rtl scripts."
                    },
                    {
                        "name": "left",
                        "description": "The element generates a block box that is floated to the left. Content flows on the right side of the box, starting at the top (subject to the 'clear' property)."
                    },
                    {
                        "name": "none",
                        "description": "The box is not floated."
                    },
                    {
                        "name": "right",
                        "description": "Similar to 'left', except the box is floated to the right, and content flows on the left side of the box, starting at the top."
                    }
                ],
                "syntax": "left | right | none | inline-start | inline-end",
                "relevance": 92,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/float"
                    }
                ],
                "description": "Specifies how a box should be floated. It may be set for any element, but only applies to elements that generate boxes that are not absolutely positioned.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "flood-color",
                "browsers": [
                    "E",
                    "C5",
                    "FF3",
                    "IE10",
                    "O9",
                    "S6"
                ],
                "relevance": 50,
                "description": "Indicates what color to use to flood the current filter primitive subregion.",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "flood-opacity",
                "browsers": [
                    "E",
                    "C5",
                    "FF3",
                    "IE10",
                    "O9",
                    "S6"
                ],
                "relevance": 50,
                "description": "Indicates what opacity to use to flood the current filter primitive subregion.",
                "restrictions": [
                    "number(0-1)",
                    "percentage"
                ]
            },
            {
                "name": "font",
                "values": [
                    {
                        "name": "100",
                        "description": "Thin"
                    },
                    {
                        "name": "200",
                        "description": "Extra Light (Ultra Light)"
                    },
                    {
                        "name": "300",
                        "description": "Light"
                    },
                    {
                        "name": "400",
                        "description": "Normal"
                    },
                    {
                        "name": "500",
                        "description": "Medium"
                    },
                    {
                        "name": "600",
                        "description": "Semi Bold (Demi Bold)"
                    },
                    {
                        "name": "700",
                        "description": "Bold"
                    },
                    {
                        "name": "800",
                        "description": "Extra Bold (Ultra Bold)"
                    },
                    {
                        "name": "900",
                        "description": "Black (Heavy)"
                    },
                    {
                        "name": "bold",
                        "description": "Same as 700"
                    },
                    {
                        "name": "bolder",
                        "description": "Specifies the weight of the face bolder than the inherited value."
                    },
                    {
                        "name": "caption",
                        "description": "The font used for captioned controls (e.g., buttons, drop-downs, etc.)."
                    },
                    {
                        "name": "icon",
                        "description": "The font used to label icons."
                    },
                    {
                        "name": "italic",
                        "description": "Selects a font that is labeled 'italic', or, if that is not available, one labeled 'oblique'."
                    },
                    {
                        "name": "large"
                    },
                    {
                        "name": "larger"
                    },
                    {
                        "name": "lighter",
                        "description": "Specifies the weight of the face lighter than the inherited value."
                    },
                    {
                        "name": "medium"
                    },
                    {
                        "name": "menu",
                        "description": "The font used in menus (e.g., dropdown menus and menu lists)."
                    },
                    {
                        "name": "message-box",
                        "description": "The font used in dialog boxes."
                    },
                    {
                        "name": "normal",
                        "description": "Specifies a face that is not labeled as a small-caps font."
                    },
                    {
                        "name": "oblique",
                        "description": "Selects a font that is labeled 'oblique'."
                    },
                    {
                        "name": "small"
                    },
                    {
                        "name": "small-caps",
                        "description": "Specifies a font that is labeled as a small-caps font. If a genuine small-caps font is not available, user agents should simulate a small-caps font."
                    },
                    {
                        "name": "small-caption",
                        "description": "The font used for labeling small controls."
                    },
                    {
                        "name": "smaller"
                    },
                    {
                        "name": "status-bar",
                        "description": "The font used in window status bars."
                    },
                    {
                        "name": "x-large"
                    },
                    {
                        "name": "x-small"
                    },
                    {
                        "name": "xx-large"
                    },
                    {
                        "name": "xx-small"
                    }
                ],
                "syntax": "[ [ <'font-style'> || <font-variant-css21> || <'font-weight'> || <'font-stretch'> ]? <'font-size'> [ / <'line-height'> ]? <'font-family'> ] | caption | icon | menu | message-box | small-caption | status-bar",
                "relevance": 82,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font"
                    }
                ],
                "description": "Shorthand property for setting 'font-style', 'font-variant', 'font-weight', 'font-size', 'line-height', and 'font-family', at the same place in the style sheet. The syntax of this property is based on a traditional typographical shorthand notation to set multiple properties related to fonts.",
                "restrictions": [
                    "font"
                ]
            },
            {
                "name": "font-family",
                "values": [
                    {
                        "name": "-apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif"
                    },
                    {
                        "name": "Arial, Helvetica, sans-serif"
                    },
                    {
                        "name": "Cambria, Cochin, Georgia, Times, 'Times New Roman', serif"
                    },
                    {
                        "name": "'Courier New', Courier, monospace"
                    },
                    {
                        "name": "cursive"
                    },
                    {
                        "name": "fantasy"
                    },
                    {
                        "name": "'Franklin Gothic Medium', 'Arial Narrow', Arial, sans-serif"
                    },
                    {
                        "name": "Georgia, 'Times New Roman', Times, serif"
                    },
                    {
                        "name": "'Gill Sans', 'Gill Sans MT', Calibri, 'Trebuchet MS', sans-serif"
                    },
                    {
                        "name": "Impact, Haettenschweiler, 'Arial Narrow Bold', sans-serif"
                    },
                    {
                        "name": "'Lucida Sans', 'Lucida Sans Regular', 'Lucida Grande', 'Lucida Sans Unicode', Geneva, Verdana, sans-serif"
                    },
                    {
                        "name": "monospace"
                    },
                    {
                        "name": "sans-serif"
                    },
                    {
                        "name": "'Segoe UI', Tahoma, Geneva, Verdana, sans-serif"
                    },
                    {
                        "name": "serif"
                    },
                    {
                        "name": "'Times New Roman', Times, serif"
                    },
                    {
                        "name": "'Trebuchet MS', 'Lucida Sans Unicode', 'Lucida Grande', 'Lucida Sans', Arial, sans-serif"
                    },
                    {
                        "name": "Verdana, Geneva, Tahoma, sans-serif"
                    }
                ],
                "syntax": "<family-name>",
                "relevance": 93,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font-family"
                    }
                ],
                "description": "Specifies a prioritized list of font family names or generic family names. A user agent iterates through the list of family names until it matches an available font that contains a glyph for the character to be rendered.",
                "restrictions": [
                    "font"
                ]
            },
            {
                "name": "font-feature-settings",
                "values": [
                    {
                        "name": "\"aalt\"",
                        "description": "Access All Alternates."
                    },
                    {
                        "name": "\"abvf\"",
                        "description": "Above-base Forms. Required in Khmer script."
                    },
                    {
                        "name": "\"abvm\"",
                        "description": "Above-base Mark Positioning. Required in Indic scripts."
                    },
                    {
                        "name": "\"abvs\"",
                        "description": "Above-base Substitutions. Required in Indic scripts."
                    },
                    {
                        "name": "\"afrc\"",
                        "description": "Alternative Fractions."
                    },
                    {
                        "name": "\"akhn\"",
                        "description": "Akhand. Required in most Indic scripts."
                    },
                    {
                        "name": "\"blwf\"",
                        "description": "Below-base Form. Required in a number of Indic scripts."
                    },
                    {
                        "name": "\"blwm\"",
                        "description": "Below-base Mark Positioning. Required in Indic scripts."
                    },
                    {
                        "name": "\"blws\"",
                        "description": "Below-base Substitutions. Required in Indic scripts."
                    },
                    {
                        "name": "\"calt\"",
                        "description": "Contextual Alternates."
                    },
                    {
                        "name": "\"case\"",
                        "description": "Case-Sensitive Forms. Applies only to European scripts; particularly prominent in Spanish-language setting."
                    },
                    {
                        "name": "\"ccmp\"",
                        "description": "Glyph Composition/Decomposition."
                    },
                    {
                        "name": "\"cfar\"",
                        "description": "Conjunct Form After Ro. Required in Khmer scripts."
                    },
                    {
                        "name": "\"cjct\"",
                        "description": "Conjunct Forms. Required in Indic scripts that show similarity to Devanagari."
                    },
                    {
                        "name": "\"clig\"",
                        "description": "Contextual Ligatures."
                    },
                    {
                        "name": "\"cpct\"",
                        "description": "Centered CJK Punctuation. Used primarily in Chinese fonts."
                    },
                    {
                        "name": "\"cpsp\"",
                        "description": "Capital Spacing. Should not be used in connecting scripts (e.g. most Arabic)."
                    },
                    {
                        "name": "\"cswh\"",
                        "description": "Contextual Swash."
                    },
                    {
                        "name": "\"curs\"",
                        "description": "Cursive Positioning. Can be used in any cursive script."
                    },
                    {
                        "name": "\"c2pc\"",
                        "description": "Petite Capitals From Capitals. Applies only to bicameral scripts."
                    },
                    {
                        "name": "\"c2sc\"",
                        "description": "Small Capitals From Capitals. Applies only to bicameral scripts."
                    },
                    {
                        "name": "\"dist\"",
                        "description": "Distances. Required in Indic scripts."
                    },
                    {
                        "name": "\"dlig\"",
                        "description": "Discretionary ligatures."
                    },
                    {
                        "name": "\"dnom\"",
                        "description": "Denominators."
                    },
                    {
                        "name": "\"dtls\"",
                        "description": "Dotless Forms. Applied to math formula layout."
                    },
                    {
                        "name": "\"expt\"",
                        "description": "Expert Forms. Applies only to Japanese."
                    },
                    {
                        "name": "\"falt\"",
                        "description": "Final Glyph on Line Alternates. Can be used in any cursive script."
                    },
                    {
                        "name": "\"fin2\"",
                        "description": "Terminal Form #2. Used only with the Syriac script."
                    },
                    {
                        "name": "\"fin3\"",
                        "description": "Terminal Form #3. Used only with the Syriac script."
                    },
                    {
                        "name": "\"fina\"",
                        "description": "Terminal Forms. Can be used in any alphabetic script."
                    },
                    {
                        "name": "\"flac\"",
                        "description": "Flattened ascent forms. Applied to math formula layout."
                    },
                    {
                        "name": "\"frac\"",
                        "description": "Fractions."
                    },
                    {
                        "name": "\"fwid\"",
                        "description": "Full Widths. Applies to any script which can use monospaced forms."
                    },
                    {
                        "name": "\"half\"",
                        "description": "Half Forms. Required in Indic scripts that show similarity to Devanagari."
                    },
                    {
                        "name": "\"haln\"",
                        "description": "Halant Forms. Required in Indic scripts."
                    },
                    {
                        "name": "\"halt\"",
                        "description": "Alternate Half Widths. Used only in CJKV fonts."
                    },
                    {
                        "name": "\"hist\"",
                        "description": "Historical Forms."
                    },
                    {
                        "name": "\"hkna\"",
                        "description": "Horizontal Kana Alternates. Applies only to fonts that support kana (hiragana and katakana)."
                    },
                    {
                        "name": "\"hlig\"",
                        "description": "Historical Ligatures."
                    },
                    {
                        "name": "\"hngl\"",
                        "description": "Hangul. Korean only."
                    },
                    {
                        "name": "\"hojo\"",
                        "description": "Hojo Kanji Forms (JIS X 0212-1990 Kanji Forms). Used only with Kanji script."
                    },
                    {
                        "name": "\"hwid\"",
                        "description": "Half Widths. Generally used only in CJKV fonts."
                    },
                    {
                        "name": "\"init\"",
                        "description": "Initial Forms. Can be used in any alphabetic script."
                    },
                    {
                        "name": "\"isol\"",
                        "description": "Isolated Forms. Can be used in any cursive script."
                    },
                    {
                        "name": "\"ital\"",
                        "description": "Italics. Applies mostly to Latin; note that many non-Latin fonts contain Latin as well."
                    },
                    {
                        "name": "\"jalt\"",
                        "description": "Justification Alternates. Can be used in any cursive script."
                    },
                    {
                        "name": "\"jp78\"",
                        "description": "JIS78 Forms. Applies only to Japanese."
                    },
                    {
                        "name": "\"jp83\"",
                        "description": "JIS83 Forms. Applies only to Japanese."
                    },
                    {
                        "name": "\"jp90\"",
                        "description": "JIS90 Forms. Applies only to Japanese."
                    },
                    {
                        "name": "\"jp04\"",
                        "description": "JIS2004 Forms. Applies only to Japanese."
                    },
                    {
                        "name": "\"kern\"",
                        "description": "Kerning."
                    },
                    {
                        "name": "\"lfbd\"",
                        "description": "Left Bounds."
                    },
                    {
                        "name": "\"liga\"",
                        "description": "Standard Ligatures."
                    },
                    {
                        "name": "\"ljmo\"",
                        "description": "Leading Jamo Forms. Required for Hangul script when Ancient Hangul writing system is supported."
                    },
                    {
                        "name": "\"lnum\"",
                        "description": "Lining Figures."
                    },
                    {
                        "name": "\"locl\"",
                        "description": "Localized Forms."
                    },
                    {
                        "name": "\"ltra\"",
                        "description": "Left-to-right glyph alternates."
                    },
                    {
                        "name": "\"ltrm\"",
                        "description": "Left-to-right mirrored forms."
                    },
                    {
                        "name": "\"mark\"",
                        "description": "Mark Positioning."
                    },
                    {
                        "name": "\"med2\"",
                        "description": "Medial Form #2. Used only with the Syriac script."
                    },
                    {
                        "name": "\"medi\"",
                        "description": "Medial Forms."
                    },
                    {
                        "name": "\"mgrk\"",
                        "description": "Mathematical Greek."
                    },
                    {
                        "name": "\"mkmk\"",
                        "description": "Mark to Mark Positioning."
                    },
                    {
                        "name": "\"nalt\"",
                        "description": "Alternate Annotation Forms."
                    },
                    {
                        "name": "\"nlck\"",
                        "description": "NLC Kanji Forms. Used only with Kanji script."
                    },
                    {
                        "name": "\"nukt\"",
                        "description": "Nukta Forms. Required in Indic scripts.."
                    },
                    {
                        "name": "\"numr\"",
                        "description": "Numerators."
                    },
                    {
                        "name": "\"onum\"",
                        "description": "Oldstyle Figures."
                    },
                    {
                        "name": "\"opbd\"",
                        "description": "Optical Bounds."
                    },
                    {
                        "name": "\"ordn\"",
                        "description": "Ordinals. Applies mostly to Latin script."
                    },
                    {
                        "name": "\"ornm\"",
                        "description": "Ornaments."
                    },
                    {
                        "name": "\"palt\"",
                        "description": "Proportional Alternate Widths. Used mostly in CJKV fonts."
                    },
                    {
                        "name": "\"pcap\"",
                        "description": "Petite Capitals."
                    },
                    {
                        "name": "\"pkna\"",
                        "description": "Proportional Kana. Generally used only in Japanese fonts."
                    },
                    {
                        "name": "\"pnum\"",
                        "description": "Proportional Figures."
                    },
                    {
                        "name": "\"pref\"",
                        "description": "Pre-base Forms. Required in Khmer and Myanmar (Burmese) scripts and southern Indic scripts that may display a pre-base form of Ra."
                    },
                    {
                        "name": "\"pres\"",
                        "description": "Pre-base Substitutions. Required in Indic scripts."
                    },
                    {
                        "name": "\"pstf\"",
                        "description": "Post-base Forms. Required in scripts of south and southeast Asia that have post-base forms for consonants eg: Gurmukhi, Malayalam, Khmer."
                    },
                    {
                        "name": "\"psts\"",
                        "description": "Post-base Substitutions."
                    },
                    {
                        "name": "\"pwid\"",
                        "description": "Proportional Widths."
                    },
                    {
                        "name": "\"qwid\"",
                        "description": "Quarter Widths. Generally used only in CJKV fonts."
                    },
                    {
                        "name": "\"rand\"",
                        "description": "Randomize."
                    },
                    {
                        "name": "\"rclt\"",
                        "description": "Required Contextual Alternates. May apply to any script, but is especially important for many styles of Arabic."
                    },
                    {
                        "name": "\"rlig\"",
                        "description": "Required Ligatures. Applies to Arabic and Syriac. May apply to some other scripts."
                    },
                    {
                        "name": "\"rkrf\"",
                        "description": "Rakar Forms. Required in Devanagari and Gujarati scripts."
                    },
                    {
                        "name": "\"rphf\"",
                        "description": "Reph Form. Required in Indic scripts. E.g. Devanagari, Kannada."
                    },
                    {
                        "name": "\"rtbd\"",
                        "description": "Right Bounds."
                    },
                    {
                        "name": "\"rtla\"",
                        "description": "Right-to-left alternates."
                    },
                    {
                        "name": "\"rtlm\"",
                        "description": "Right-to-left mirrored forms."
                    },
                    {
                        "name": "\"ruby\"",
                        "description": "Ruby Notation Forms. Applies only to Japanese."
                    },
                    {
                        "name": "\"salt\"",
                        "description": "Stylistic Alternates."
                    },
                    {
                        "name": "\"sinf\"",
                        "description": "Scientific Inferiors."
                    },
                    {
                        "name": "\"size\"",
                        "description": "Optical size."
                    },
                    {
                        "name": "\"smcp\"",
                        "description": "Small Capitals. Applies only to bicameral scripts."
                    },
                    {
                        "name": "\"smpl\"",
                        "description": "Simplified Forms. Applies only to Chinese and Japanese."
                    },
                    {
                        "name": "\"ssty\"",
                        "description": "Math script style alternates."
                    },
                    {
                        "name": "\"stch\"",
                        "description": "Stretching Glyph Decomposition."
                    },
                    {
                        "name": "\"subs\"",
                        "description": "Subscript."
                    },
                    {
                        "name": "\"sups\"",
                        "description": "Superscript."
                    },
                    {
                        "name": "\"swsh\"",
                        "description": "Swash. Does not apply to ideographic scripts."
                    },
                    {
                        "name": "\"titl\"",
                        "description": "Titling."
                    },
                    {
                        "name": "\"tjmo\"",
                        "description": "Trailing Jamo Forms. Required for Hangul script when Ancient Hangul writing system is supported."
                    },
                    {
                        "name": "\"tnam\"",
                        "description": "Traditional Name Forms. Applies only to Japanese."
                    },
                    {
                        "name": "\"tnum\"",
                        "description": "Tabular Figures."
                    },
                    {
                        "name": "\"trad\"",
                        "description": "Traditional Forms. Applies only to Chinese and Japanese."
                    },
                    {
                        "name": "\"twid\"",
                        "description": "Third Widths. Generally used only in CJKV fonts."
                    },
                    {
                        "name": "\"unic\"",
                        "description": "Unicase."
                    },
                    {
                        "name": "\"valt\"",
                        "description": "Alternate Vertical Metrics. Applies only to scripts with vertical writing modes."
                    },
                    {
                        "name": "\"vatu\"",
                        "description": "Vattu Variants. Used for Indic scripts. E.g. Devanagari."
                    },
                    {
                        "name": "\"vert\"",
                        "description": "Vertical Alternates. Applies only to scripts with vertical writing modes."
                    },
                    {
                        "name": "\"vhal\"",
                        "description": "Alternate Vertical Half Metrics. Used only in CJKV fonts."
                    },
                    {
                        "name": "\"vjmo\"",
                        "description": "Vowel Jamo Forms. Required for Hangul script when Ancient Hangul writing system is supported."
                    },
                    {
                        "name": "\"vkna\"",
                        "description": "Vertical Kana Alternates. Applies only to fonts that support kana (hiragana and katakana)."
                    },
                    {
                        "name": "\"vkrn\"",
                        "description": "Vertical Kerning."
                    },
                    {
                        "name": "\"vpal\"",
                        "description": "Proportional Alternate Vertical Metrics. Used mostly in CJKV fonts."
                    },
                    {
                        "name": "\"vrt2\"",
                        "description": "Vertical Alternates and Rotation. Applies only to scripts with vertical writing modes."
                    },
                    {
                        "name": "\"zero\"",
                        "description": "Slashed Zero."
                    },
                    {
                        "name": "normal",
                        "description": "No change in glyph substitution or positioning occurs."
                    },
                    {
                        "name": "off",
                        "description": "Disable feature."
                    },
                    {
                        "name": "on",
                        "description": "Enable feature."
                    }
                ],
                "syntax": "normal | <feature-tag-value>#",
                "relevance": 54,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font-feature-settings"
                    }
                ],
                "description": "Provides low-level control over OpenType font features. It is intended as a way of providing access to font features that are not widely used but are needed for a particular use case.",
                "restrictions": [
                    "string",
                    "integer"
                ]
            },
            {
                "name": "font-kerning",
                "browsers": [
                    "E79",
                    "FF32",
                    "S9",
                    "C33",
                    "O20"
                ],
                "values": [
                    {
                        "name": "auto",
                        "description": "Specifies that kerning is applied at the discretion of the user agent."
                    },
                    {
                        "name": "none",
                        "description": "Specifies that kerning is not applied."
                    },
                    {
                        "name": "normal",
                        "description": "Specifies that kerning is applied."
                    }
                ],
                "syntax": "auto | normal | none",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font-kerning"
                    }
                ],
                "description": "Kerning is the contextual adjustment of inter-glyph spacing. This property controls metric kerning, kerning that utilizes adjustment data contained in the font.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "font-language-override",
                "browsers": [
                    "FF34"
                ],
                "values": [
                    {
                        "name": "normal",
                        "description": "Implies that when rendering with OpenType fonts the language of the document is used to infer the OpenType language system, used to select language specific features when rendering."
                    }
                ],
                "syntax": "normal | <string>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font-language-override"
                    }
                ],
                "description": "The value of 'normal' implies that when rendering with OpenType fonts the language of the document is used to infer the OpenType language system, used to select language specific features when rendering.",
                "restrictions": [
                    "string"
                ]
            },
            {
                "name": "font-size",
                "values": [
                    {
                        "name": "large"
                    },
                    {
                        "name": "larger"
                    },
                    {
                        "name": "medium"
                    },
                    {
                        "name": "small"
                    },
                    {
                        "name": "smaller"
                    },
                    {
                        "name": "x-large"
                    },
                    {
                        "name": "x-small"
                    },
                    {
                        "name": "xx-large"
                    },
                    {
                        "name": "xx-small"
                    }
                ],
                "syntax": "<absolute-size> | <relative-size> | <length-percentage>",
                "relevance": 94,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font-size"
                    }
                ],
                "description": "Indicates the desired height of glyphs from the font. For scalable fonts, the font-size is a scale factor applied to the EM unit of the font. (Note that certain glyphs may bleed outside their EM box.) For non-scalable fonts, the font-size is converted into absolute units and matched against the declared font-size of the font, using the same absolute coordinate space for both of the matched values.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "font-size-adjust",
                "browsers": [
                    "E79",
                    "FF40",
                    "C43",
                    "O30"
                ],
                "values": [
                    {
                        "name": "none",
                        "description": "Do not preserve the font’s x-height."
                    }
                ],
                "syntax": "none | <number>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font-size-adjust"
                    }
                ],
                "description": "Preserves the readability of text when font fallback occurs by adjusting the font-size so that the x-height is the same regardless of the font used.",
                "restrictions": [
                    "number"
                ]
            },
            {
                "name": "font-stretch",
                "values": [
                    {
                        "name": "condensed"
                    },
                    {
                        "name": "expanded"
                    },
                    {
                        "name": "extra-condensed"
                    },
                    {
                        "name": "extra-expanded"
                    },
                    {
                        "name": "narrower",
                        "description": "Indicates a narrower value relative to the width of the parent element."
                    },
                    {
                        "name": "normal"
                    },
                    {
                        "name": "semi-condensed"
                    },
                    {
                        "name": "semi-expanded"
                    },
                    {
                        "name": "ultra-condensed"
                    },
                    {
                        "name": "ultra-expanded"
                    },
                    {
                        "name": "wider",
                        "description": "Indicates a wider value relative to the width of the parent element."
                    }
                ],
                "syntax": "<font-stretch-absolute>{1,2}",
                "relevance": 53,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font-stretch"
                    }
                ],
                "description": "Selects a normal, condensed, or expanded face from a font family.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "font-style",
                "values": [
                    {
                        "name": "italic",
                        "description": "Selects a font that is labeled as an 'italic' face, or an 'oblique' face if one is not"
                    },
                    {
                        "name": "normal",
                        "description": "Selects a face that is classified as 'normal'."
                    },
                    {
                        "name": "oblique",
                        "description": "Selects a font that is labeled as an 'oblique' face, or an 'italic' face if one is not."
                    }
                ],
                "syntax": "normal | italic | oblique <angle>{0,2}",
                "relevance": 83,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font-style"
                    }
                ],
                "description": "Allows italic or oblique faces to be selected. Italic forms are generally cursive in nature while oblique faces are typically sloped versions of the regular face.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "font-synthesis",
                "browsers": [
                    "FF34",
                    "S9"
                ],
                "values": [
                    {
                        "name": "none",
                        "description": "Disallow all synthetic faces."
                    },
                    {
                        "name": "style",
                        "description": "Allow synthetic italic faces."
                    },
                    {
                        "name": "weight",
                        "description": "Allow synthetic bold faces."
                    }
                ],
                "syntax": "none | [ weight || style ]",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font-synthesis"
                    }
                ],
                "description": "Controls whether user agents are allowed to synthesize bold or oblique font faces when a font family lacks bold or italic faces.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "font-variant",
                "values": [
                    {
                        "name": "normal",
                        "description": "Specifies a face that is not labeled as a small-caps font."
                    },
                    {
                        "name": "small-caps",
                        "description": "Specifies a font that is labeled as a small-caps font. If a genuine small-caps font is not available, user agents should simulate a small-caps font."
                    }
                ],
                "syntax": "normal | none | [ <common-lig-values> || <discretionary-lig-values> || <historical-lig-values> || <contextual-alt-values> || stylistic(<feature-value-name>) || historical-forms || styleset(<feature-value-name>#) || character-variant(<feature-value-name>#) || swash(<feature-value-name>) || ornaments(<feature-value-name>) || annotation(<feature-value-name>) || [ small-caps | all-small-caps | petite-caps | all-petite-caps | unicase | titling-caps ] || <numeric-figure-values> || <numeric-spacing-values> || <numeric-fraction-values> || ordinal || slashed-zero || <east-asian-variant-values> || <east-asian-width-values> || ruby ]",
                "relevance": 64,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font-variant"
                    }
                ],
                "description": "Specifies variant representations of the font",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "font-variant-alternates",
                "browsers": [
                    "FF34"
                ],
                "values": [
                    {
                        "name": "annotation()",
                        "description": "Enables display of alternate annotation forms."
                    },
                    {
                        "name": "character-variant()",
                        "description": "Enables display of specific character variants."
                    },
                    {
                        "name": "historical-forms",
                        "description": "Enables display of historical forms."
                    },
                    {
                        "name": "normal",
                        "description": "None of the features are enabled."
                    },
                    {
                        "name": "ornaments()",
                        "description": "Enables replacement of default glyphs with ornaments, if provided in the font."
                    },
                    {
                        "name": "styleset()",
                        "description": "Enables display with stylistic sets."
                    },
                    {
                        "name": "stylistic()",
                        "description": "Enables display of stylistic alternates."
                    },
                    {
                        "name": "swash()",
                        "description": "Enables display of swash glyphs."
                    }
                ],
                "syntax": "normal | [ stylistic( <feature-value-name> ) || historical-forms || styleset( <feature-value-name># ) || character-variant( <feature-value-name># ) || swash( <feature-value-name> ) || ornaments( <feature-value-name> ) || annotation( <feature-value-name> ) ]",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font-variant-alternates"
                    }
                ],
                "description": "For any given character, fonts can provide a variety of alternate glyphs in addition to the default glyph for that character. This property provides control over the selection of these alternate glyphs.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "font-variant-caps",
                "browsers": [
                    "E79",
                    "FF34",
                    "C52",
                    "O39"
                ],
                "values": [
                    {
                        "name": "all-petite-caps",
                        "description": "Enables display of petite capitals for both upper and lowercase letters."
                    },
                    {
                        "name": "all-small-caps",
                        "description": "Enables display of small capitals for both upper and lowercase letters."
                    },
                    {
                        "name": "normal",
                        "description": "None of the features are enabled."
                    },
                    {
                        "name": "petite-caps",
                        "description": "Enables display of petite capitals."
                    },
                    {
                        "name": "small-caps",
                        "description": "Enables display of small capitals. Small-caps glyphs typically use the form of uppercase letters but are reduced to the size of lowercase letters."
                    },
                    {
                        "name": "titling-caps",
                        "description": "Enables display of titling capitals."
                    },
                    {
                        "name": "unicase",
                        "description": "Enables display of mixture of small capitals for uppercase letters with normal lowercase letters."
                    }
                ],
                "syntax": "normal | small-caps | all-small-caps | petite-caps | all-petite-caps | unicase | titling-caps",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font-variant-caps"
                    }
                ],
                "description": "Specifies control over capitalized forms.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "font-variant-east-asian",
                "browsers": [
                    "E79",
                    "FF34",
                    "C63",
                    "O50"
                ],
                "values": [
                    {
                        "name": "full-width",
                        "description": "Enables rendering of full-width variants."
                    },
                    {
                        "name": "jis04",
                        "description": "Enables rendering of JIS04 forms."
                    },
                    {
                        "name": "jis78",
                        "description": "Enables rendering of JIS78 forms."
                    },
                    {
                        "name": "jis83",
                        "description": "Enables rendering of JIS83 forms."
                    },
                    {
                        "name": "jis90",
                        "description": "Enables rendering of JIS90 forms."
                    },
                    {
                        "name": "normal",
                        "description": "None of the features are enabled."
                    },
                    {
                        "name": "proportional-width",
                        "description": "Enables rendering of proportionally-spaced variants."
                    },
                    {
                        "name": "ruby",
                        "description": "Enables display of ruby variant glyphs."
                    },
                    {
                        "name": "simplified",
                        "description": "Enables rendering of simplified forms."
                    },
                    {
                        "name": "traditional",
                        "description": "Enables rendering of traditional forms."
                    }
                ],
                "syntax": "normal | [ <east-asian-variant-values> || <east-asian-width-values> || ruby ]",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font-variant-east-asian"
                    }
                ],
                "description": "Allows control of glyph substitute and positioning in East Asian text.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "font-variant-ligatures",
                "browsers": [
                    "E79",
                    "FF34",
                    "S9.1",
                    "C34",
                    "O21"
                ],
                "values": [
                    {
                        "name": "additional-ligatures",
                        "description": "Enables display of additional ligatures."
                    },
                    {
                        "name": "common-ligatures",
                        "description": "Enables display of common ligatures."
                    },
                    {
                        "name": "contextual",
                        "browsers": [
                            "E79",
                            "FF34",
                            "S9.1",
                            "C34",
                            "O21"
                        ],
                        "description": "Enables display of contextual alternates."
                    },
                    {
                        "name": "discretionary-ligatures",
                        "description": "Enables display of discretionary ligatures."
                    },
                    {
                        "name": "historical-ligatures",
                        "description": "Enables display of historical ligatures."
                    },
                    {
                        "name": "no-additional-ligatures",
                        "description": "Disables display of additional ligatures."
                    },
                    {
                        "name": "no-common-ligatures",
                        "description": "Disables display of common ligatures."
                    },
                    {
                        "name": "no-contextual",
                        "browsers": [
                            "E79",
                            "FF34",
                            "S9.1",
                            "C34",
                            "O21"
                        ],
                        "description": "Disables display of contextual alternates."
                    },
                    {
                        "name": "no-discretionary-ligatures",
                        "description": "Disables display of discretionary ligatures."
                    },
                    {
                        "name": "no-historical-ligatures",
                        "description": "Disables display of historical ligatures."
                    },
                    {
                        "name": "none",
                        "browsers": [
                            "E79",
                            "FF34",
                            "S9.1",
                            "C34",
                            "O21"
                        ],
                        "description": "Disables all ligatures."
                    },
                    {
                        "name": "normal",
                        "description": "Implies that the defaults set by the font are used."
                    }
                ],
                "syntax": "normal | none | [ <common-lig-values> || <discretionary-lig-values> || <historical-lig-values> || <contextual-alt-values> ]",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font-variant-ligatures"
                    }
                ],
                "description": "Specifies control over which ligatures are enabled or disabled. A value of ‘normal’ implies that the defaults set by the font are used.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "font-variant-numeric",
                "browsers": [
                    "E79",
                    "FF34",
                    "S9.1",
                    "C52",
                    "O39"
                ],
                "values": [
                    {
                        "name": "diagonal-fractions",
                        "description": "Enables display of lining diagonal fractions."
                    },
                    {
                        "name": "lining-nums",
                        "description": "Enables display of lining numerals."
                    },
                    {
                        "name": "normal",
                        "description": "None of the features are enabled."
                    },
                    {
                        "name": "oldstyle-nums",
                        "description": "Enables display of old-style numerals."
                    },
                    {
                        "name": "ordinal",
                        "description": "Enables display of letter forms used with ordinal numbers."
                    },
                    {
                        "name": "proportional-nums",
                        "description": "Enables display of proportional numerals."
                    },
                    {
                        "name": "slashed-zero",
                        "description": "Enables display of slashed zeros."
                    },
                    {
                        "name": "stacked-fractions",
                        "description": "Enables display of lining stacked fractions."
                    },
                    {
                        "name": "tabular-nums",
                        "description": "Enables display of tabular numerals."
                    }
                ],
                "syntax": "normal | [ <numeric-figure-values> || <numeric-spacing-values> || <numeric-fraction-values> || ordinal || slashed-zero ]",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font-variant-numeric"
                    }
                ],
                "description": "Specifies control over numerical forms.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "font-variant-position",
                "browsers": [
                    "FF34"
                ],
                "values": [
                    {
                        "name": "normal",
                        "description": "None of the features are enabled."
                    },
                    {
                        "name": "sub",
                        "description": "Enables display of subscript variants (OpenType feature: subs)."
                    },
                    {
                        "name": "super",
                        "description": "Enables display of superscript variants (OpenType feature: sups)."
                    }
                ],
                "syntax": "normal | sub | super",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font-variant-position"
                    }
                ],
                "description": "Specifies the vertical position",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "font-weight",
                "values": [
                    {
                        "name": "100",
                        "description": "Thin"
                    },
                    {
                        "name": "200",
                        "description": "Extra Light (Ultra Light)"
                    },
                    {
                        "name": "300",
                        "description": "Light"
                    },
                    {
                        "name": "400",
                        "description": "Normal"
                    },
                    {
                        "name": "500",
                        "description": "Medium"
                    },
                    {
                        "name": "600",
                        "description": "Semi Bold (Demi Bold)"
                    },
                    {
                        "name": "700",
                        "description": "Bold"
                    },
                    {
                        "name": "800",
                        "description": "Extra Bold (Ultra Bold)"
                    },
                    {
                        "name": "900",
                        "description": "Black (Heavy)"
                    },
                    {
                        "name": "bold",
                        "description": "Same as 700"
                    },
                    {
                        "name": "bolder",
                        "description": "Specifies the weight of the face bolder than the inherited value."
                    },
                    {
                        "name": "lighter",
                        "description": "Specifies the weight of the face lighter than the inherited value."
                    },
                    {
                        "name": "normal",
                        "description": "Same as 400"
                    }
                ],
                "syntax": "<font-weight-absolute>{1,2}",
                "relevance": 93,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/font-weight"
                    }
                ],
                "description": "Specifies weight of glyphs in the font, their degree of blackness or stroke thickness.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "glyph-orientation-horizontal",
                "relevance": 50,
                "description": "Controls glyph orientation when the inline-progression-direction is horizontal.",
                "restrictions": [
                    "angle",
                    "number"
                ]
            },
            {
                "name": "glyph-orientation-vertical",
                "values": [
                    {
                        "name": "auto",
                        "description": "Sets the orientation based on the fullwidth or non-fullwidth characters and the most common orientation."
                    }
                ],
                "relevance": 50,
                "description": "Controls glyph orientation when the inline-progression-direction is vertical.",
                "restrictions": [
                    "angle",
                    "number",
                    "enum"
                ]
            },
            {
                "name": "grid-area",
                "browsers": [
                    "E16",
                    "FF52",
                    "S10.1",
                    "C57",
                    "O44"
                ],
                "values": [
                    {
                        "name": "auto",
                        "description": "The property contributes nothing to the grid item’s placement, indicating auto-placement, an automatic span, or a default span of one."
                    },
                    {
                        "name": "span",
                        "description": "Contributes a grid span to the grid item’s placement such that the corresponding edge of the grid item’s grid area is N lines from its opposite edge."
                    }
                ],
                "syntax": "<grid-line> [ / <grid-line> ]{0,3}",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/grid-area"
                    }
                ],
                "description": "Determine a grid item’s size and location within the grid by contributing a line, a span, or nothing (automatic) to its grid placement. Shorthand for 'grid-row-start', 'grid-column-start', 'grid-row-end', and 'grid-column-end'.",
                "restrictions": [
                    "identifier",
                    "integer"
                ]
            },
            {
                "name": "grid",
                "browsers": [
                    "E16",
                    "FF52",
                    "S10.1",
                    "C57",
                    "O44"
                ],
                "syntax": "<'grid-template'> | <'grid-template-rows'> / [ auto-flow && dense? ] <'grid-auto-columns'>? | [ auto-flow && dense? ] <'grid-auto-rows'>? / <'grid-template-columns'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/grid"
                    }
                ],
                "description": "The grid CSS property is a shorthand property that sets all of the explicit grid properties ('grid-template-rows', 'grid-template-columns', and 'grid-template-areas'), and all the implicit grid properties ('grid-auto-rows', 'grid-auto-columns', and 'grid-auto-flow'), in a single declaration.",
                "restrictions": [
                    "identifier",
                    "length",
                    "percentage",
                    "string",
                    "enum"
                ]
            },
            {
                "name": "grid-auto-columns",
                "values": [
                    {
                        "name": "min-content",
                        "description": "Represents the largest min-content contribution of the grid items occupying the grid track."
                    },
                    {
                        "name": "max-content",
                        "description": "Represents the largest max-content contribution of the grid items occupying the grid track."
                    },
                    {
                        "name": "auto",
                        "description": "As a maximum, identical to 'max-content'. As a minimum, represents the largest minimum size (as specified by min-width/min-height) of the grid items occupying the grid track."
                    },
                    {
                        "name": "minmax()",
                        "description": "Defines a size range greater than or equal to min and less than or equal to max."
                    }
                ],
                "syntax": "<track-size>+",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/grid-auto-columns"
                    }
                ],
                "description": "Specifies the size of implicitly created columns.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "grid-auto-flow",
                "browsers": [
                    "E16",
                    "FF52",
                    "S10.1",
                    "C57",
                    "O44"
                ],
                "values": [
                    {
                        "name": "row",
                        "description": "The auto-placement algorithm places items by filling each row in turn, adding new rows as necessary."
                    },
                    {
                        "name": "column",
                        "description": "The auto-placement algorithm places items by filling each column in turn, adding new columns as necessary."
                    },
                    {
                        "name": "dense",
                        "description": "If specified, the auto-placement algorithm uses a “dense” packing algorithm, which attempts to fill in holes earlier in the grid if smaller items come up later."
                    }
                ],
                "syntax": "[ row | column ] || dense",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/grid-auto-flow"
                    }
                ],
                "description": "Controls how the auto-placement algorithm works, specifying exactly how auto-placed items get flowed into the grid.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "grid-auto-rows",
                "values": [
                    {
                        "name": "min-content",
                        "description": "Represents the largest min-content contribution of the grid items occupying the grid track."
                    },
                    {
                        "name": "max-content",
                        "description": "Represents the largest max-content contribution of the grid items occupying the grid track."
                    },
                    {
                        "name": "auto",
                        "description": "As a maximum, identical to 'max-content'. As a minimum, represents the largest minimum size (as specified by min-width/min-height) of the grid items occupying the grid track."
                    },
                    {
                        "name": "minmax()",
                        "description": "Defines a size range greater than or equal to min and less than or equal to max."
                    }
                ],
                "syntax": "<track-size>+",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/grid-auto-rows"
                    }
                ],
                "description": "Specifies the size of implicitly created rows.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "grid-column",
                "browsers": [
                    "E16",
                    "FF52",
                    "S10.1",
                    "C57",
                    "O44"
                ],
                "values": [
                    {
                        "name": "auto",
                        "description": "The property contributes nothing to the grid item’s placement, indicating auto-placement, an automatic span, or a default span of one."
                    },
                    {
                        "name": "span",
                        "description": "Contributes a grid span to the grid item’s placement such that the corresponding edge of the grid item’s grid area is N lines from its opposite edge."
                    }
                ],
                "syntax": "<grid-line> [ / <grid-line> ]?",
                "relevance": 52,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/grid-column"
                    }
                ],
                "description": "Shorthand for 'grid-column-start' and 'grid-column-end'.",
                "restrictions": [
                    "identifier",
                    "integer",
                    "enum"
                ]
            },
            {
                "name": "grid-column-end",
                "browsers": [
                    "E16",
                    "FF52",
                    "S10.1",
                    "C57",
                    "O44"
                ],
                "values": [
                    {
                        "name": "auto",
                        "description": "The property contributes nothing to the grid item’s placement, indicating auto-placement, an automatic span, or a default span of one."
                    },
                    {
                        "name": "span",
                        "description": "Contributes a grid span to the grid item’s placement such that the corresponding edge of the grid item’s grid area is N lines from its opposite edge."
                    }
                ],
                "syntax": "<grid-line>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/grid-column-end"
                    }
                ],
                "description": "Determine a grid item’s size and location within the grid by contributing a line, a span, or nothing (automatic) to its grid placement.",
                "restrictions": [
                    "identifier",
                    "integer",
                    "enum"
                ]
            },
            {
                "name": "grid-column-gap",
                "browsers": [
                    "FF52",
                    "C57",
                    "S10.1",
                    "O44"
                ],
                "status": "obsolete",
                "syntax": "<length-percentage>",
                "relevance": 1,
                "description": "Specifies the gutters between grid columns. Replaced by 'column-gap' property.",
                "restrictions": [
                    "length"
                ]
            },
            {
                "name": "grid-column-start",
                "browsers": [
                    "E16",
                    "FF52",
                    "S10.1",
                    "C57",
                    "O44"
                ],
                "values": [
                    {
                        "name": "auto",
                        "description": "The property contributes nothing to the grid item’s placement, indicating auto-placement, an automatic span, or a default span of one."
                    },
                    {
                        "name": "span",
                        "description": "Contributes a grid span to the grid item’s placement such that the corresponding edge of the grid item’s grid area is N lines from its opposite edge."
                    }
                ],
                "syntax": "<grid-line>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/grid-column-start"
                    }
                ],
                "description": "Determine a grid item’s size and location within the grid by contributing a line, a span, or nothing (automatic) to its grid placement.",
                "restrictions": [
                    "identifier",
                    "integer",
                    "enum"
                ]
            },
            {
                "name": "grid-gap",
                "browsers": [
                    "FF52",
                    "C57",
                    "S10.1",
                    "O44"
                ],
                "status": "obsolete",
                "syntax": "<'grid-row-gap'> <'grid-column-gap'>?",
                "relevance": 2,
                "description": "Shorthand that specifies the gutters between grid columns and grid rows in one declaration. Replaced by 'gap' property.",
                "restrictions": [
                    "length"
                ]
            },
            {
                "name": "grid-row",
                "browsers": [
                    "E16",
                    "FF52",
                    "S10.1",
                    "C57",
                    "O44"
                ],
                "values": [
                    {
                        "name": "auto",
                        "description": "The property contributes nothing to the grid item’s placement, indicating auto-placement, an automatic span, or a default span of one."
                    },
                    {
                        "name": "span",
                        "description": "Contributes a grid span to the grid item’s placement such that the corresponding edge of the grid item’s grid area is N lines from its opposite edge."
                    }
                ],
                "syntax": "<grid-line> [ / <grid-line> ]?",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/grid-row"
                    }
                ],
                "description": "Shorthand for 'grid-row-start' and 'grid-row-end'.",
                "restrictions": [
                    "identifier",
                    "integer",
                    "enum"
                ]
            },
            {
                "name": "grid-row-end",
                "browsers": [
                    "E16",
                    "FF52",
                    "S10.1",
                    "C57",
                    "O44"
                ],
                "values": [
                    {
                        "name": "auto",
                        "description": "The property contributes nothing to the grid item’s placement, indicating auto-placement, an automatic span, or a default span of one."
                    },
                    {
                        "name": "span",
                        "description": "Contributes a grid span to the grid item’s placement such that the corresponding edge of the grid item’s grid area is N lines from its opposite edge."
                    }
                ],
                "syntax": "<grid-line>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/grid-row-end"
                    }
                ],
                "description": "Determine a grid item’s size and location within the grid by contributing a line, a span, or nothing (automatic) to its grid placement.",
                "restrictions": [
                    "identifier",
                    "integer",
                    "enum"
                ]
            },
            {
                "name": "grid-row-gap",
                "browsers": [
                    "FF52",
                    "C57",
                    "S10.1",
                    "O44"
                ],
                "status": "obsolete",
                "syntax": "<length-percentage>",
                "relevance": 1,
                "description": "Specifies the gutters between grid rows. Replaced by 'row-gap' property.",
                "restrictions": [
                    "length"
                ]
            },
            {
                "name": "grid-row-start",
                "browsers": [
                    "E16",
                    "FF52",
                    "S10.1",
                    "C57",
                    "O44"
                ],
                "values": [
                    {
                        "name": "auto",
                        "description": "The property contributes nothing to the grid item’s placement, indicating auto-placement, an automatic span, or a default span of one."
                    },
                    {
                        "name": "span",
                        "description": "Contributes a grid span to the grid item’s placement such that the corresponding edge of the grid item’s grid area is N lines from its opposite edge."
                    }
                ],
                "syntax": "<grid-line>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/grid-row-start"
                    }
                ],
                "description": "Determine a grid item’s size and location within the grid by contributing a line, a span, or nothing (automatic) to its grid placement.",
                "restrictions": [
                    "identifier",
                    "integer",
                    "enum"
                ]
            },
            {
                "name": "grid-template",
                "browsers": [
                    "E16",
                    "FF52",
                    "S10.1",
                    "C57",
                    "O44"
                ],
                "values": [
                    {
                        "name": "none",
                        "description": "Sets all three properties to their initial values."
                    },
                    {
                        "name": "min-content",
                        "description": "Represents the largest min-content contribution of the grid items occupying the grid track."
                    },
                    {
                        "name": "max-content",
                        "description": "Represents the largest max-content contribution of the grid items occupying the grid track."
                    },
                    {
                        "name": "auto",
                        "description": "As a maximum, identical to 'max-content'. As a minimum, represents the largest minimum size (as specified by min-width/min-height) of the grid items occupying the grid track."
                    },
                    {
                        "name": "subgrid",
                        "description": "Sets 'grid-template-rows' and 'grid-template-columns' to 'subgrid', and 'grid-template-areas' to its initial value."
                    },
                    {
                        "name": "minmax()",
                        "description": "Defines a size range greater than or equal to min and less than or equal to max."
                    },
                    {
                        "name": "repeat()",
                        "description": "Represents a repeated fragment of the track list, allowing a large number of columns or rows that exhibit a recurring pattern to be written in a more compact form."
                    }
                ],
                "syntax": "none | [ <'grid-template-rows'> / <'grid-template-columns'> ] | [ <line-names>? <string> <track-size>? <line-names>? ]+ [ / <explicit-track-list> ]?",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/grid-template"
                    }
                ],
                "description": "Shorthand for setting grid-template-columns, grid-template-rows, and grid-template-areas in a single declaration.",
                "restrictions": [
                    "identifier",
                    "length",
                    "percentage",
                    "string",
                    "enum"
                ]
            },
            {
                "name": "grid-template-areas",
                "browsers": [
                    "E16",
                    "FF52",
                    "S10.1",
                    "C57",
                    "O44"
                ],
                "values": [
                    {
                        "name": "none",
                        "description": "The grid container doesn’t define any named grid areas."
                    }
                ],
                "syntax": "none | <string>+",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/grid-template-areas"
                    }
                ],
                "description": "Specifies named grid areas, which are not associated with any particular grid item, but can be referenced from the grid-placement properties.",
                "restrictions": [
                    "string"
                ]
            },
            {
                "name": "grid-template-columns",
                "values": [
                    {
                        "name": "none",
                        "description": "There is no explicit grid; any rows/columns will be implicitly generated."
                    },
                    {
                        "name": "min-content",
                        "description": "Represents the largest min-content contribution of the grid items occupying the grid track."
                    },
                    {
                        "name": "max-content",
                        "description": "Represents the largest max-content contribution of the grid items occupying the grid track."
                    },
                    {
                        "name": "auto",
                        "description": "As a maximum, identical to 'max-content'. As a minimum, represents the largest minimum size (as specified by min-width/min-height) of the grid items occupying the grid track."
                    },
                    {
                        "name": "subgrid",
                        "description": "Indicates that the grid will align to its parent grid in that axis."
                    },
                    {
                        "name": "minmax()",
                        "description": "Defines a size range greater than or equal to min and less than or equal to max."
                    },
                    {
                        "name": "repeat()",
                        "description": "Represents a repeated fragment of the track list, allowing a large number of columns or rows that exhibit a recurring pattern to be written in a more compact form."
                    }
                ],
                "syntax": "none | <track-list> | <auto-track-list> | subgrid <line-name-list>?",
                "relevance": 56,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/grid-template-columns"
                    }
                ],
                "description": "specifies, as a space-separated track list, the line names and track sizing functions of the grid.",
                "restrictions": [
                    "identifier",
                    "length",
                    "percentage",
                    "enum"
                ]
            },
            {
                "name": "grid-template-rows",
                "values": [
                    {
                        "name": "none",
                        "description": "There is no explicit grid; any rows/columns will be implicitly generated."
                    },
                    {
                        "name": "min-content",
                        "description": "Represents the largest min-content contribution of the grid items occupying the grid track."
                    },
                    {
                        "name": "max-content",
                        "description": "Represents the largest max-content contribution of the grid items occupying the grid track."
                    },
                    {
                        "name": "auto",
                        "description": "As a maximum, identical to 'max-content'. As a minimum, represents the largest minimum size (as specified by min-width/min-height) of the grid items occupying the grid track."
                    },
                    {
                        "name": "subgrid",
                        "description": "Indicates that the grid will align to its parent grid in that axis."
                    },
                    {
                        "name": "minmax()",
                        "description": "Defines a size range greater than or equal to min and less than or equal to max."
                    },
                    {
                        "name": "repeat()",
                        "description": "Represents a repeated fragment of the track list, allowing a large number of columns or rows that exhibit a recurring pattern to be written in a more compact form."
                    }
                ],
                "syntax": "none | <track-list> | <auto-track-list> | subgrid <line-name-list>?",
                "relevance": 52,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/grid-template-rows"
                    }
                ],
                "description": "specifies, as a space-separated track list, the line names and track sizing functions of the grid.",
                "restrictions": [
                    "identifier",
                    "length",
                    "percentage",
                    "string",
                    "enum"
                ]
            },
            {
                "name": "height",
                "values": [
                    {
                        "name": "auto",
                        "description": "The height depends on the values of other properties."
                    },
                    {
                        "name": "fit-content",
                        "description": "Use the fit-content inline size or fit-content block size, as appropriate to the writing mode."
                    },
                    {
                        "name": "max-content",
                        "description": "Use the max-content inline size or max-content block size, as appropriate to the writing mode."
                    },
                    {
                        "name": "min-content",
                        "description": "Use the min-content inline size or min-content block size, as appropriate to the writing mode."
                    }
                ],
                "syntax": "<viewport-length>{1,2}",
                "relevance": 96,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/height"
                    }
                ],
                "description": "Specifies the height of the content area, padding area or border area (depending on 'box-sizing') of certain boxes.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "hyphens",
                "values": [
                    {
                        "name": "auto",
                        "description": "Conditional hyphenation characters inside a word, if present, take priority over automatic resources when determining hyphenation points within the word."
                    },
                    {
                        "name": "manual",
                        "description": "Words are only broken at line breaks where there are characters inside the word that suggest line break opportunities"
                    },
                    {
                        "name": "none",
                        "description": "Words are not broken at line breaks, even if characters inside the word suggest line break points."
                    }
                ],
                "syntax": "none | manual | auto",
                "relevance": 53,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/hyphens"
                    }
                ],
                "description": "Controls whether hyphenation is allowed to create more break opportunities within a line of text.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "image-orientation",
                "browsers": [
                    "E81",
                    "FF26",
                    "S13.1",
                    "C81",
                    "O67"
                ],
                "values": [
                    {
                        "name": "flip",
                        "description": "After rotating by the precededing angle, the image is flipped horizontally. Defaults to 0deg if the angle is ommitted."
                    },
                    {
                        "name": "from-image",
                        "description": "If the image has an orientation specified in its metadata, such as EXIF, this value computes to the angle that the metadata specifies is necessary to correctly orient the image."
                    }
                ],
                "syntax": "from-image | <angle> | [ <angle>? flip ]",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/image-orientation"
                    }
                ],
                "description": "Specifies an orthogonal rotation to be applied to an image before it is laid out.",
                "restrictions": [
                    "angle"
                ]
            },
            {
                "name": "image-rendering",
                "browsers": [
                    "E79",
                    "FF3.6",
                    "S6",
                    "C13",
                    "O15"
                ],
                "values": [
                    {
                        "name": "auto",
                        "description": "The image should be scaled with an algorithm that maximizes the appearance of the image."
                    },
                    {
                        "name": "crisp-edges",
                        "description": "The image must be scaled with an algorithm that preserves contrast and edges in the image, and which does not smooth colors or introduce blur to the image in the process."
                    },
                    {
                        "name": "-moz-crisp-edges",
                        "browsers": [
                            "E79",
                            "FF3.6",
                            "S6",
                            "C13",
                            "O15"
                        ]
                    },
                    {
                        "name": "optimizeQuality",
                        "description": "Deprecated."
                    },
                    {
                        "name": "optimizeSpeed",
                        "description": "Deprecated."
                    },
                    {
                        "name": "pixelated",
                        "description": "When scaling the image up, the 'nearest neighbor' or similar algorithm must be used, so that the image appears to be simply composed of very large pixels."
                    }
                ],
                "syntax": "auto | crisp-edges | pixelated",
                "relevance": 55,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/image-rendering"
                    }
                ],
                "description": "Provides a hint to the user-agent about what aspects of an image are most important to preserve when the image is scaled, to aid the user-agent in the choice of an appropriate scaling algorithm.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "ime-mode",
                "browsers": [
                    "E12",
                    "FF3",
                    "IE5"
                ],
                "values": [
                    {
                        "name": "active",
                        "description": "The input method editor is initially active; text entry is performed using it unless the user specifically dismisses it."
                    },
                    {
                        "name": "auto",
                        "description": "No change is made to the current input method editor state. This is the default."
                    },
                    {
                        "name": "disabled",
                        "description": "The input method editor is disabled and may not be activated by the user."
                    },
                    {
                        "name": "inactive",
                        "description": "The input method editor is initially inactive, but the user may activate it if they wish."
                    },
                    {
                        "name": "normal",
                        "description": "The IME state should be normal; this value can be used in a user style sheet to override the page setting."
                    }
                ],
                "status": "obsolete",
                "syntax": "auto | normal | active | inactive | disabled",
                "relevance": 0,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/ime-mode"
                    }
                ],
                "description": "Controls the state of the input method editor for text fields.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "inline-size",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C57",
                    "O44"
                ],
                "values": [
                    {
                        "name": "auto",
                        "description": "Depends on the values of other properties."
                    }
                ],
                "syntax": "<'width'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/inline-size"
                    }
                ],
                "description": "Logical 'height'. Mapping depends on the element’s 'writing-mode'.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "isolation",
                "browsers": [
                    "E79",
                    "FF36",
                    "S8",
                    "C41",
                    "O30"
                ],
                "values": [
                    {
                        "name": "auto",
                        "description": "Elements are not isolated unless an operation is applied that causes the creation of a stacking context."
                    },
                    {
                        "name": "isolate",
                        "description": "In CSS will turn the element into a stacking context."
                    }
                ],
                "syntax": "auto | isolate",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/isolation"
                    }
                ],
                "description": "In CSS setting to 'isolate' will turn the element into a stacking context. In SVG, it defines whether an element is isolated or not.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "justify-content",
                "values": [
                    {
                        "name": "center",
                        "description": "Flex items are packed toward the center of the line."
                    },
                    {
                        "name": "start",
                        "description": "The items are packed flush to each other toward the start edge of the alignment container in the main axis."
                    },
                    {
                        "name": "end",
                        "description": "The items are packed flush to each other toward the end edge of the alignment container in the main axis."
                    },
                    {
                        "name": "left",
                        "description": "The items are packed flush to each other toward the left edge of the alignment container in the main axis."
                    },
                    {
                        "name": "right",
                        "description": "The items are packed flush to each other toward the right edge of the alignment container in the main axis."
                    },
                    {
                        "name": "safe",
                        "description": "If the size of the item overflows the alignment container, the item is instead aligned as if the alignment mode were start."
                    },
                    {
                        "name": "unsafe",
                        "description": "Regardless of the relative sizes of the item and alignment container, the given alignment value is honored."
                    },
                    {
                        "name": "stretch",
                        "description": "If the combined size of the alignment subjects is less than the size of the alignment container, any auto-sized alignment subjects have their size increased equally (not proportionally), while still respecting the constraints imposed by max-height/max-width (or equivalent functionality), so that the combined size exactly fills the alignment container."
                    },
                    {
                        "name": "space-evenly",
                        "description": "The items are evenly distributed within the alignment container along the main axis."
                    },
                    {
                        "name": "flex-end",
                        "description": "Flex items are packed toward the end of the line."
                    },
                    {
                        "name": "flex-start",
                        "description": "Flex items are packed toward the start of the line."
                    },
                    {
                        "name": "space-around",
                        "description": "Flex items are evenly distributed in the line, with half-size spaces on either end."
                    },
                    {
                        "name": "space-between",
                        "description": "Flex items are evenly distributed in the line."
                    },
                    {
                        "name": "baseline",
                        "description": "Specifies participation in first-baseline alignment."
                    },
                    {
                        "name": "first baseline",
                        "description": "Specifies participation in first-baseline alignment."
                    },
                    {
                        "name": "last baseline",
                        "description": "Specifies participation in last-baseline alignment."
                    }
                ],
                "syntax": "normal | <content-distribution> | <overflow-position>? [ <content-position> | left | right ]",
                "relevance": 84,
                "description": "Aligns flex items along the main axis of the current line of the flex container.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "kerning",
                "values": [
                    {
                        "name": "auto",
                        "description": "Indicates that the user agent should adjust inter-glyph spacing based on kerning tables that are included in the font that will be used."
                    }
                ],
                "relevance": 50,
                "description": "Indicates whether the user agent should adjust inter-glyph spacing based on kerning tables that are included in the relevant font or instead disable auto-kerning and set inter-character spacing to a specific length.",
                "restrictions": [
                    "length",
                    "enum"
                ]
            },
            {
                "name": "left",
                "values": [
                    {
                        "name": "auto",
                        "description": "For non-replaced elements, the effect of this value depends on which of related properties have the value 'auto' as well"
                    }
                ],
                "syntax": "<length> | <percentage> | auto",
                "relevance": 95,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/left"
                    }
                ],
                "description": "Specifies how far an absolutely positioned box's left margin edge is offset to the right of the left edge of the box's 'containing block'.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "letter-spacing",
                "values": [
                    {
                        "name": "normal",
                        "description": "The spacing is the normal spacing for the current font. It is typically zero-length."
                    }
                ],
                "syntax": "normal | <length>",
                "relevance": 80,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/letter-spacing"
                    }
                ],
                "description": "Specifies the minimum, maximum, and optimal spacing between grapheme clusters.",
                "restrictions": [
                    "length"
                ]
            },
            {
                "name": "lighting-color",
                "browsers": [
                    "E",
                    "C5",
                    "FF3",
                    "IE10",
                    "O9",
                    "S6"
                ],
                "relevance": 50,
                "description": "Defines the color of the light source for filter primitives 'feDiffuseLighting' and 'feSpecularLighting'.",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "line-break",
                "values": [
                    {
                        "name": "auto",
                        "description": "The UA determines the set of line-breaking restrictions to use for CJK scripts, and it may vary the restrictions based on the length of the line; e.g., use a less restrictive set of line-break rules for short lines."
                    },
                    {
                        "name": "loose",
                        "description": "Breaks text using the least restrictive set of line-breaking rules. Typically used for short lines, such as in newspapers."
                    },
                    {
                        "name": "normal",
                        "description": "Breaks text using the most common set of line-breaking rules."
                    },
                    {
                        "name": "strict",
                        "description": "Breaks CJK scripts using a more restrictive set of line-breaking rules than 'normal'."
                    }
                ],
                "syntax": "auto | loose | normal | strict | anywhere",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/line-break"
                    }
                ],
                "description": "Specifies what set of line breaking restrictions are in effect within the element.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "line-height",
                "values": [
                    {
                        "name": "normal",
                        "description": "Tells user agents to set the computed value to a 'reasonable' value based on the font size of the element."
                    }
                ],
                "syntax": "normal | <number> | <length> | <percentage>",
                "relevance": 93,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/line-height"
                    }
                ],
                "description": "Determines the block-progression dimension of the text content area of an inline box.",
                "restrictions": [
                    "number",
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "list-style",
                "values": [
                    {
                        "name": "armenian"
                    },
                    {
                        "name": "circle",
                        "description": "A hollow circle."
                    },
                    {
                        "name": "decimal"
                    },
                    {
                        "name": "decimal-leading-zero"
                    },
                    {
                        "name": "disc",
                        "description": "A filled circle."
                    },
                    {
                        "name": "georgian"
                    },
                    {
                        "name": "inside",
                        "description": "The marker box is outside the principal block box, as described in the section on the ::marker pseudo-element below."
                    },
                    {
                        "name": "lower-alpha"
                    },
                    {
                        "name": "lower-greek"
                    },
                    {
                        "name": "lower-latin"
                    },
                    {
                        "name": "lower-roman"
                    },
                    {
                        "name": "none"
                    },
                    {
                        "name": "outside",
                        "description": "The ::marker pseudo-element is an inline element placed immediately before all ::before pseudo-elements in the principal block box, after which the element's content flows."
                    },
                    {
                        "name": "square",
                        "description": "A filled square."
                    },
                    {
                        "name": "symbols()",
                        "description": "Allows a counter style to be defined inline."
                    },
                    {
                        "name": "upper-alpha"
                    },
                    {
                        "name": "upper-latin"
                    },
                    {
                        "name": "upper-roman"
                    },
                    {
                        "name": "url()"
                    }
                ],
                "syntax": "<'list-style-type'> || <'list-style-position'> || <'list-style-image'>",
                "relevance": 85,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/list-style"
                    }
                ],
                "description": "Shorthand for setting 'list-style-type', 'list-style-position' and 'list-style-image'",
                "restrictions": [
                    "image",
                    "enum",
                    "url"
                ]
            },
            {
                "name": "list-style-image",
                "values": [
                    {
                        "name": "none",
                        "description": "The default contents of the of the list item’s marker are given by 'list-style-type' instead."
                    }
                ],
                "syntax": "<url> | none",
                "relevance": 52,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/list-style-image"
                    }
                ],
                "description": "Sets the image that will be used as the list item marker. When the image is available, it will replace the marker set with the 'list-style-type' marker.",
                "restrictions": [
                    "image"
                ]
            },
            {
                "name": "list-style-position",
                "values": [
                    {
                        "name": "inside",
                        "description": "The marker box is outside the principal block box, as described in the section on the ::marker pseudo-element below."
                    },
                    {
                        "name": "outside",
                        "description": "The ::marker pseudo-element is an inline element placed immediately before all ::before pseudo-elements in the principal block box, after which the element's content flows."
                    }
                ],
                "syntax": "inside | outside",
                "relevance": 55,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/list-style-position"
                    }
                ],
                "description": "Specifies the position of the '::marker' pseudo-element's box in the list item.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "list-style-type",
                "values": [
                    {
                        "name": "armenian",
                        "description": "Traditional uppercase Armenian numbering."
                    },
                    {
                        "name": "circle",
                        "description": "A hollow circle."
                    },
                    {
                        "name": "decimal",
                        "description": "Western decimal numbers."
                    },
                    {
                        "name": "decimal-leading-zero",
                        "description": "Decimal numbers padded by initial zeros."
                    },
                    {
                        "name": "disc",
                        "description": "A filled circle."
                    },
                    {
                        "name": "georgian",
                        "description": "Traditional Georgian numbering."
                    },
                    {
                        "name": "lower-alpha",
                        "description": "Lowercase ASCII letters."
                    },
                    {
                        "name": "lower-greek",
                        "description": "Lowercase classical Greek."
                    },
                    {
                        "name": "lower-latin",
                        "description": "Lowercase ASCII letters."
                    },
                    {
                        "name": "lower-roman",
                        "description": "Lowercase ASCII Roman numerals."
                    },
                    {
                        "name": "none",
                        "description": "No marker"
                    },
                    {
                        "name": "square",
                        "description": "A filled square."
                    },
                    {
                        "name": "symbols()",
                        "description": "Allows a counter style to be defined inline."
                    },
                    {
                        "name": "upper-alpha",
                        "description": "Uppercase ASCII letters."
                    },
                    {
                        "name": "upper-latin",
                        "description": "Uppercase ASCII letters."
                    },
                    {
                        "name": "upper-roman",
                        "description": "Uppercase ASCII Roman numerals."
                    }
                ],
                "syntax": "<counter-style> | <string> | none",
                "relevance": 75,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/list-style-type"
                    }
                ],
                "description": "Used to construct the default contents of a list item’s marker",
                "restrictions": [
                    "enum",
                    "string"
                ]
            },
            {
                "name": "margin",
                "values": [
                    {
                        "name": "auto"
                    }
                ],
                "syntax": "[ <length> | <percentage> | auto ]{1,4}",
                "relevance": 95,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/margin"
                    }
                ],
                "description": "Shorthand property to set values the thickness of the margin area. If left is omitted, it is the same as right. If bottom is omitted it is the same as top, if right is omitted it is the same as top. Negative values for margin properties are allowed, but there may be implementation-specific limits.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "margin-block-end",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "values": [
                    {
                        "name": "auto"
                    }
                ],
                "syntax": "<'margin-left'>",
                "relevance": 53,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/margin-block-end"
                    }
                ],
                "description": "Logical 'margin-bottom'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "margin-block-start",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "values": [
                    {
                        "name": "auto"
                    }
                ],
                "syntax": "<'margin-left'>",
                "relevance": 52,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/margin-block-start"
                    }
                ],
                "description": "Logical 'margin-top'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "margin-bottom",
                "values": [
                    {
                        "name": "auto"
                    }
                ],
                "syntax": "<length> | <percentage> | auto",
                "relevance": 91,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/margin-bottom"
                    }
                ],
                "description": "Shorthand property to set values the thickness of the margin area. If left is omitted, it is the same as right. If bottom is omitted it is the same as top, if right is omitted it is the same as top. Negative values for margin properties are allowed, but there may be implementation-specific limits..",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "margin-inline-end",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "values": [
                    {
                        "name": "auto"
                    }
                ],
                "syntax": "<'margin-left'>",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/margin-inline-end"
                    }
                ],
                "description": "Logical 'margin-right'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "margin-inline-start",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C69",
                    "O56"
                ],
                "values": [
                    {
                        "name": "auto"
                    }
                ],
                "syntax": "<'margin-left'>",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/margin-inline-start"
                    }
                ],
                "description": "Logical 'margin-left'. Mapping depends on the parent element’s 'writing-mode', 'direction', and 'text-orientation'.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "margin-left",
                "values": [
                    {
                        "name": "auto"
                    }
                ],
                "syntax": "<length> | <percentage> | auto",
                "relevance": 91,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/margin-left"
                    }
                ],
                "description": "Shorthand property to set values the thickness of the margin area. If left is omitted, it is the same as right. If bottom is omitted it is the same as top, if right is omitted it is the same as top. Negative values for margin properties are allowed, but there may be implementation-specific limits..",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "margin-right",
                "values": [
                    {
                        "name": "auto"
                    }
                ],
                "syntax": "<length> | <percentage> | auto",
                "relevance": 91,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/margin-right"
                    }
                ],
                "description": "Shorthand property to set values the thickness of the margin area. If left is omitted, it is the same as right. If bottom is omitted it is the same as top, if right is omitted it is the same as top. Negative values for margin properties are allowed, but there may be implementation-specific limits..",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "margin-top",
                "values": [
                    {
                        "name": "auto"
                    }
                ],
                "syntax": "<length> | <percentage> | auto",
                "relevance": 95,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/margin-top"
                    }
                ],
                "description": "Shorthand property to set values the thickness of the margin area. If left is omitted, it is the same as right. If bottom is omitted it is the same as top, if right is omitted it is the same as top. Negative values for margin properties are allowed, but there may be implementation-specific limits..",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "marker",
                "values": [
                    {
                        "name": "none",
                        "description": "Indicates that no marker symbol will be drawn at the given vertex or vertices."
                    },
                    {
                        "name": "url()",
                        "description": "Indicates that the <marker> element referenced will be used."
                    }
                ],
                "relevance": 50,
                "description": "Specifies the marker symbol that shall be used for all points on the sets the value for all vertices on the given ‘path’ element or basic shape.",
                "restrictions": [
                    "url"
                ]
            },
            {
                "name": "marker-end",
                "values": [
                    {
                        "name": "none",
                        "description": "Indicates that no marker symbol will be drawn at the given vertex or vertices."
                    },
                    {
                        "name": "url()",
                        "description": "Indicates that the <marker> element referenced will be used."
                    }
                ],
                "relevance": 50,
                "description": "Specifies the marker that will be drawn at the last vertices of the given markable element.",
                "restrictions": [
                    "url"
                ]
            },
            {
                "name": "marker-mid",
                "values": [
                    {
                        "name": "none",
                        "description": "Indicates that no marker symbol will be drawn at the given vertex or vertices."
                    },
                    {
                        "name": "url()",
                        "description": "Indicates that the <marker> element referenced will be used."
                    }
                ],
                "relevance": 50,
                "description": "Specifies the marker that will be drawn at all vertices except the first and last.",
                "restrictions": [
                    "url"
                ]
            },
            {
                "name": "marker-start",
                "values": [
                    {
                        "name": "none",
                        "description": "Indicates that no marker symbol will be drawn at the given vertex or vertices."
                    },
                    {
                        "name": "url()",
                        "description": "Indicates that the <marker> element referenced will be used."
                    }
                ],
                "relevance": 50,
                "description": "Specifies the marker that will be drawn at the first vertices of the given markable element.",
                "restrictions": [
                    "url"
                ]
            },
            {
                "name": "mask-image",
                "browsers": [
                    "E16",
                    "FF53",
                    "S4",
                    "C1",
                    "O15"
                ],
                "values": [
                    {
                        "name": "none",
                        "description": "Counts as a transparent black image layer."
                    },
                    {
                        "name": "url()",
                        "description": "Reference to a <mask element or to a CSS image."
                    }
                ],
                "syntax": "<mask-reference>#",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/mask-image"
                    }
                ],
                "description": "Sets the mask layer image of an element.",
                "restrictions": [
                    "url",
                    "image",
                    "enum"
                ]
            },
            {
                "name": "mask-mode",
                "browsers": [
                    "FF53"
                ],
                "values": [
                    {
                        "name": "alpha",
                        "description": "Alpha values of the mask layer image should be used as the mask values."
                    },
                    {
                        "name": "auto",
                        "description": "Use alpha values if 'mask-image' is an image, luminance if a <mask> element or a CSS image."
                    },
                    {
                        "name": "luminance",
                        "description": "Luminance values of the mask layer image should be used as the mask values."
                    }
                ],
                "syntax": "<masking-mode>#",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/mask-mode"
                    }
                ],
                "description": "Indicates whether the mask layer image is treated as luminance mask or alpha mask.",
                "restrictions": [
                    "url",
                    "image",
                    "enum"
                ]
            },
            {
                "name": "mask-origin",
                "browsers": [
                    "E79",
                    "FF53",
                    "S4",
                    "C1",
                    "O15"
                ],
                "syntax": "<geometry-box>#",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/mask-origin"
                    }
                ],
                "description": "Specifies the mask positioning area.",
                "restrictions": [
                    "geometry-box",
                    "enum"
                ]
            },
            {
                "name": "mask-position",
                "browsers": [
                    "E18",
                    "FF53",
                    "S3.2",
                    "C1",
                    "O15"
                ],
                "syntax": "<position>#",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/mask-position"
                    }
                ],
                "description": "Specifies how mask layer images are positioned.",
                "restrictions": [
                    "position",
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "mask-repeat",
                "browsers": [
                    "E18",
                    "FF53",
                    "S3.2",
                    "C1",
                    "O15"
                ],
                "syntax": "<repeat-style>#",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/mask-repeat"
                    }
                ],
                "description": "Specifies how mask layer images are tiled after they have been sized and positioned.",
                "restrictions": [
                    "repeat"
                ]
            },
            {
                "name": "mask-size",
                "browsers": [
                    "E18",
                    "FF53",
                    "S4",
                    "C4",
                    "O15"
                ],
                "values": [
                    {
                        "name": "auto",
                        "description": "Resolved by using the image’s intrinsic ratio and the size of the other dimension, or failing that, using the image’s intrinsic size, or failing that, treating it as 100%."
                    },
                    {
                        "name": "contain",
                        "description": "Scale the image, while preserving its intrinsic aspect ratio (if any), to the largest size such that both its width and its height can fit inside the background positioning area."
                    },
                    {
                        "name": "cover",
                        "description": "Scale the image, while preserving its intrinsic aspect ratio (if any), to the smallest size such that both its width and its height can completely cover the background positioning area."
                    }
                ],
                "syntax": "<bg-size>#",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/mask-size"
                    }
                ],
                "description": "Specifies the size of the mask layer images.",
                "restrictions": [
                    "length",
                    "percentage",
                    "enum"
                ]
            },
            {
                "name": "mask-type",
                "browsers": [
                    "E79",
                    "FF35",
                    "S6.1",
                    "C24",
                    "O15"
                ],
                "values": [
                    {
                        "name": "alpha",
                        "description": "Indicates that the alpha values of the mask should be used."
                    },
                    {
                        "name": "luminance",
                        "description": "Indicates that the luminance values of the mask should be used."
                    }
                ],
                "syntax": "luminance | alpha",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/mask-type"
                    }
                ],
                "description": "Defines whether the content of the <mask> element is treated as as luminance mask or alpha mask.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "max-block-size",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C57",
                    "O44"
                ],
                "values": [
                    {
                        "name": "none",
                        "description": "No limit on the width of the box."
                    }
                ],
                "syntax": "<'max-width'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/max-block-size"
                    }
                ],
                "description": "Logical 'max-width'. Mapping depends on the element’s 'writing-mode'.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "max-height",
                "values": [
                    {
                        "name": "none",
                        "description": "No limit on the height of the box."
                    },
                    {
                        "name": "fit-content",
                        "description": "Use the fit-content inline size or fit-content block size, as appropriate to the writing mode."
                    },
                    {
                        "name": "max-content",
                        "description": "Use the max-content inline size or max-content block size, as appropriate to the writing mode."
                    },
                    {
                        "name": "min-content",
                        "description": "Use the min-content inline size or min-content block size, as appropriate to the writing mode."
                    }
                ],
                "syntax": "<viewport-length>",
                "relevance": 85,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/max-height"
                    }
                ],
                "description": "Allows authors to constrain content height to a certain range.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "max-inline-size",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C57",
                    "O44"
                ],
                "values": [
                    {
                        "name": "none",
                        "description": "No limit on the height of the box."
                    }
                ],
                "syntax": "<'max-width'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/max-inline-size"
                    }
                ],
                "description": "Logical 'max-height'. Mapping depends on the element’s 'writing-mode'.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "max-width",
                "values": [
                    {
                        "name": "none",
                        "description": "No limit on the width of the box."
                    },
                    {
                        "name": "fit-content",
                        "description": "Use the fit-content inline size or fit-content block size, as appropriate to the writing mode."
                    },
                    {
                        "name": "max-content",
                        "description": "Use the max-content inline size or max-content block size, as appropriate to the writing mode."
                    },
                    {
                        "name": "min-content",
                        "description": "Use the min-content inline size or min-content block size, as appropriate to the writing mode."
                    }
                ],
                "syntax": "<viewport-length>",
                "relevance": 90,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/max-width"
                    }
                ],
                "description": "Allows authors to constrain content width to a certain range.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "min-block-size",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C57",
                    "O44"
                ],
                "syntax": "<'min-width'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/min-block-size"
                    }
                ],
                "description": "Logical 'min-width'. Mapping depends on the element’s 'writing-mode'.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "min-height",
                "values": [
                    {
                        "name": "auto"
                    },
                    {
                        "name": "fit-content",
                        "description": "Use the fit-content inline size or fit-content block size, as appropriate to the writing mode."
                    },
                    {
                        "name": "max-content",
                        "description": "Use the max-content inline size or max-content block size, as appropriate to the writing mode."
                    },
                    {
                        "name": "min-content",
                        "description": "Use the min-content inline size or min-content block size, as appropriate to the writing mode."
                    }
                ],
                "syntax": "<viewport-length>",
                "relevance": 89,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/min-height"
                    }
                ],
                "description": "Allows authors to constrain content height to a certain range.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "min-inline-size",
                "browsers": [
                    "E79",
                    "FF41",
                    "S12.1",
                    "C57",
                    "O44"
                ],
                "syntax": "<'min-width'>",
                "relevance": 50,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/min-inline-size"
                    }
                ],
                "description": "Logical 'min-height'. Mapping depends on the element’s 'writing-mode'.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "min-width",
                "values": [
                    {
                        "name": "auto"
                    },
                    {
                        "name": "fit-content",
                        "description": "Use the fit-content inline size or fit-content block size, as appropriate to the writing mode."
                    },
                    {
                        "name": "max-content",
                        "description": "Use the max-content inline size or max-content block size, as appropriate to the writing mode."
                    },
                    {
                        "name": "min-content",
                        "description": "Use the min-content inline size or min-content block size, as appropriate to the writing mode."
                    }
                ],
                "syntax": "<viewport-length>",
                "relevance": 88,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/min-width"
                    }
                ],
                "description": "Allows authors to constrain content width to a certain range.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "mix-blend-mode",
                "browsers": [
                    "E79",
                    "FF32",
                    "S8",
                    "C41",
                    "O28"
                ],
                "values": [
                    {
                        "name": "normal",
                        "description": "Default attribute which specifies no blending"
                    },
                    {
                        "name": "multiply",
                        "description": "The source color is multiplied by the destination color and replaces the destination."
                    },
                    {
                        "name": "screen",
                        "description": "Multiplies the complements of the backdrop and source color values, then complements the result."
                    },
                    {
                        "name": "overlay",
                        "description": "Multiplies or screens the colors, depending on the backdrop color value."
                    },
                    {
                        "name": "darken",
                        "description": "Selects the darker of the backdrop and source colors."
                    },
                    {
                        "name": "lighten",
                        "description": "Selects the lighter of the backdrop and source colors."
                    },
                    {
                        "name": "color-dodge",
                        "description": "Brightens the backdrop color to reflect the source color."
                    },
                    {
                        "name": "color-burn",
                        "description": "Darkens the backdrop color to reflect the source color."
                    },
                    {
                        "name": "hard-light",
                        "description": "Multiplies or screens the colors, depending on the source color value."
                    },
                    {
                        "name": "soft-light",
                        "description": "Darkens or lightens the colors, depending on the source color value."
                    },
                    {
                        "name": "difference",
                        "description": "Subtracts the darker of the two constituent colors from the lighter color.."
                    },
                    {
                        "name": "exclusion",
                        "description": "Produces an effect similar to that of the Difference mode but lower in contrast."
                    },
                    {
                        "name": "hue",
                        "browsers": [
                            "E79",
                            "FF32",
                            "S8",
                            "C41",
                            "O28"
                        ],
                        "description": "Creates a color with the hue of the source color and the saturation and luminosity of the backdrop color."
                    },
                    {
                        "name": "saturation",
                        "browsers": [
                            "E79",
                            "FF32",
                            "S8",
                            "C41",
                            "O28"
                        ],
                        "description": "Creates a color with the saturation of the source color and the hue and luminosity of the backdrop color."
                    },
                    {
                        "name": "color",
                        "browsers": [
                            "E79",
                            "FF32",
                            "S8",
                            "C41",
                            "O28"
                        ],
                        "description": "Creates a color with the hue and saturation of the source color and the luminosity of the backdrop color."
                    },
                    {
                        "name": "luminosity",
                        "browsers": [
                            "E79",
                            "FF32",
                            "S8",
                            "C41",
                            "O28"
                        ],
                        "description": "Creates a color with the luminosity of the source color and the hue and saturation of the backdrop color."
                    }
                ],
                "syntax": "<blend-mode>",
                "relevance": 51,
                "references": [
                    {
                        "name": "MDN Reference",
                        "url": "https://developer.mozilla.org/docs/Web/CSS/mix-blend-mode"
                    }
                ],
                "description": "Defines the formula that must be used to mix the colors with the backdrop.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "motion",
                "browsers": [
                    "C46",
                    "O33"
                ],
                "values": [
                    {
                        "name": "none",
                        "description": "No motion path gets created."
                    },
                    {
                        "name": "path()",
                        "description": "Defines an SVG path as a string, with optional 'fill-rule' as the first argument."
                    },
                    {
                        "name": "auto",
                        "description": "Indicates that the object is rotated by the angle of the direction of the motion path."
                    },
                    {
                        "name": "reverse",
                        "description": "Indicates that the object is rotated by the angle of the direction of the motion path plus 180 degrees."
                    }
                ],
                "relevance": 50,
                "description": "Shorthand property for setting 'motion-path', 'motion-offset' and 'motion-rotation'.",
                "restrictions": [
                    "url",
                    "length",
                    "percentage",
                    "angle",
                    "shape",
                    "geometry-box",
                    "enum"
                ]
            },
            {
                "name": "motion-offset",
                "browsers": [
                    "C46",
                    "O33"
                ],
                "relevance": 50,
                "description": "A distance that describes the position along the specified motion path.",
                "restrictions": [
                    "length",
                    "percentage"
                ]
            },
            {
                "name": "motion-path",
                "browsers": [
                    "C46",
                    "O33"
                ],
                "values": [
                    {
                        "name": "none",
                        "description": "No motion path gets created."
                    },
                    {
                        "name": "path()",
                        "description": "Defines an SVG path as a string, with optional 'fill-rule' as the first argument."
                    }
                ],
                "relevance": 50,
                "description": "Specifies the motion path the element gets positioned at.",
                "restrictions": [
                    "url",
                    "shape",
                    "geometry-box",
                    "enum"
                ]
            },
            {
                "name": "motion-rotation",
                "browsers": [
                    "C46",
                    "O33"
                ],
                "values": [
                    {
                        "name": "auto",
                        "description": "Indicates that the object is rotated by the angle of the direction of the motion path."
                    },
                    {
                        "name": "reverse",
                        "description": "Indicates that the object is rotated by the angle of the direction of the motion path plus 180 degrees."
                    }
                ],
                "relevance": 50,
                "description": "Defines the direction of the element while positioning along the motion path.",
                "restrictions": [
                    "angle"
                ]
            },
            {
                "name": "-moz-animation",
                "browsers": [
                    "FF9"
                ],
                "values": [
                    {
                        "name": "alternate",
                        "description": "The animation cycle iterations that are odd counts are played in the normal direction, and the animation cycle iterations that are even counts are played in a reverse direction."
                    },
                    {
                        "name": "alternate-reverse",
                        "description": "The animation cycle iterations that are odd counts are played in the reverse direction, and the animation cycle iterations that are even counts are played in a normal direction."
                    },
                    {
                        "name": "backwards",
                        "description": "The beginning property value (as defined in the first @keyframes at-rule) is applied before the animation is displayed, during the period defined by 'animation-delay'."
                    },
                    {
                        "name": "both",
                        "description": "Both forwards and backwards fill modes are applied."
                    },
                    {
                        "name": "forwards",
                        "description": "The final property value (as defined in the last @keyframes at-rule) is maintained after the animation completes."
                    },
                    {
                        "name": "infinite",
                        "description": "Causes the animation to repeat forever."
                    },
                    {
                        "name": "none",
                        "description": "No animation is performed"
                    },
                    {
                        "name": "normal",
                        "description": "Normal playback."
                    },
                    {
                        "name": "reverse",
                        "description": "All iterations of the animation are played in the reverse direction from the way they were specified."
                    }
                ],
                "relevance": 50,
                "description": "Shorthand property combines six of the animation properties into a single property.",
                "restrictions": [
                    "time",
                    "enum",
                    "timing-function",
                    "identifier",
                    "number"
                ]
            },
            {
                "name": "-moz-animation-delay",
                "browsers": [
                    "FF9"
                ],
                "relevance": 50,
                "description": "Defines when the animation will start.",
                "restrictions": [
                    "time"
                ]
            },
            {
                "name": "-moz-animation-direction",
                "browsers": [
                    "FF9"
                ],
                "values": [
                    {
                        "name": "alternate",
                        "description": "The animation cycle iterations that are odd counts are played in the normal direction, and the animation cycle iterations that are even counts are played in a reverse direction."
                    },
                    {
                        "name": "alternate-reverse",
                        "description": "The animation cycle iterations that are odd counts are played in the reverse direction, and the animation cycle iterations that are even counts are played in a normal direction."
                    },
                    {
                        "name": "normal",
                        "description": "Normal playback."
                    },
                    {
                        "name": "reverse",
                        "description": "All iterations of the animation are played in the reverse direction from the way they were specified."
                    }
                ],
                "relevance": 50,
                "description": "Defines whether or not the animation should play in reverse on alternate cycles.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "-moz-animation-duration",
                "browsers": [
                    "FF9"
                ],
                "relevance": 50,
                "description": "Defines the length of time that an animation takes to complete one cycle.",
                "restrictions": [
                    "time"
                ]
            },
            {
                "name": "-moz-animation-iteration-count",
                "browsers": [
                    "FF9"
                ],
                "values": [
                    {
                        "name": "infinite",
                        "description": "Causes the animation to repeat forever."
                    }
                ],
                "relevance": 50,
                "description": "Defines the number of times an animation cycle is played. The default value is one, meaning the animation will play from beginning to end once.",
                "restrictions": [
                    "number",
                    "enum"
                ]
            },
            {
                "name": "-moz-animation-name",
                "browsers": [
                    "FF9"
                ],
                "values": [
                    {
                        "name": "none",
                        "description": "No animation is performed"
                    }
                ],
                "relevance": 50,
                "description": "Defines a list of animations that apply. Each name is used to select the keyframe at-rule that provides the property values for the animation.",
                "restrictions": [
                    "identifier",
                    "enum"
                ]
            },
            {
                "name": "-moz-animation-play-state",
                "browsers": [
                    "FF9"
                ],
                "values": [
                    {
                        "name": "paused",
                        "description": "A running animation will be paused."
                    },
                    {
                        "name": "running",
                        "description": "Resume playback of a paused animation."
                    }
                ],
                "relevance": 50,
                "description": "Defines whether the animation is running or paused.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "-moz-animation-timing-function",
                "browsers": [
                    "FF9"
                ],
                "relevance": 50,
                "description": "Describes how the animation will progress over one cycle of its duration. See the 'transition-timing-function'.",
                "restrictions": [
                    "timing-function"
                ]
            },
            {
                "name": "-moz-appearance",
                "browsers": [
                    "FF1"
                ],
                "values": [
                    {
                        "name": "button"
                    },
                    {
                        "name": "button-arrow-down"
                    },
                    {
                        "name": "button-arrow-next"
                    },
                    {
                        "name": "button-arrow-previous"
                    },
                    {
                        "name": "button-arrow-up"
                    },
                    {
                        "name": "button-bevel"
                    },
                    {
                        "name": "checkbox"
                    },
                    {
                        "name": "checkbox-container"
                    },
                    {
                        "name": "checkbox-label"
                    },
                    {
                        "name": "dialog"
                    },
                    {
                        "name": "groupbox"
                    },
                    {
                        "name": "listbox"
                    },
                    {
                        "name": "menuarrow"
                    },
                    {
                        "name": "menuimage"
                    },
                    {
                        "name": "menuitem"
                    },
                    {
                        "name": "menuitemtext"
                    },
                    {
                        "name": "menulist"
                    },
                    {
                        "name": "menulist-button"
                    },
                    {
                        "name": "menulist-text"
                    },
                    {
                        "name": "menulist-textfield"
                    },
                    {
                        "name": "menupopup"
                    },
                    {
                        "name": "menuradio"
                    },
                    {
                        "name": "menuseparator"
                    },
                    {
                        "name": "-moz-mac-unified-toolbar"
                    },
                    {
                        "name": "-moz-win-borderless-glass"
                    },
                    {
                        "name": "-moz-win-browsertabbar-toolbox"
                    },
                    {
                        "name": "-moz-win-communications-toolbox"
                    },
                    {
                        "name": "-moz-win-glass"
                    },
                    {
                        "name": "-moz-win-media-toolbox"
                    },
                    {
                        "name": "none"
                    },
                    {
                        "name": "progressbar"
                    },
                    {
                        "name": "progresschunk"
                    },
                    {
                        "name": "radio"
                    },
                    {
                        "name": "radio-container"
                    },
                    {
                        "name": "radio-label"
                    },
                    {
                        "name": "radiomenuitem"
                    },
                    {
                        "name": "resizer"
                    },
                    {
                        "name": "resizerpanel"
                    },
                    {
                        "name": "scrollbarbutton-down"
                    },
                    {
                        "name": "scrollbarbutton-left"
                    },
                    {
                        "name": "scrollbarbutton-right"
                    },
                    {
                        "name": "scrollbarbutton-up"
                    },
                    {
                        "name": "scrollbar-small"
                    },
                    {
                        "name": "scrollbartrack-horizontal"
                    },
                    {
                        "name": "scrollbartrack-vertical"
                    },
                    {
                        "name": "separator"
                    },
                    {
                        "name": "spinner"
                    },
                    {
                        "name": "spinner-downbutton"
                    },
                    {
                        "name": "spinner-textfield"
                    },
                    {
                        "name": "spinner-upbutton"
                    },
                    {
                        "name": "statusbar"
                    },
                    {
                        "name": "statusbarpanel"
                    },
                    {
                        "name": "tab"
                    },
                    {
                        "name": "tabpanels"
                    },
                    {
                        "name": "tab-scroll-arrow-back"
                    },
                    {
                        "name": "tab-scroll-arrow-forward"
                    },
                    {
                        "name": "textfield"
                    },
                    {
                        "name": "textfield-multiline"
                    },
                    {
                        "name": "toolbar"
                    },
                    {
                        "name": "toolbox"
                    },
                    {
                        "name": "tooltip"
                    },
                    {
                        "name": "treeheadercell"
                    },
                    {
                        "name": "treeheadersortarrow"
                    },
                    {
                        "name": "treeitem"
                    },
                    {
                        "name": "treetwistyopen"
                    },
                    {
                        "name": "treeview"
                    },
                    {
                        "name": "treewisty"
                    },
                    {
                        "name": "window"
                    }
                ],
                "status": "nonstandard",
                "syntax": "none | button | button-arrow-down | button-arrow-next | button-arrow-previous | button-arrow-up | button-bevel | button-focus | caret | checkbox | checkbox-container | checkbox-label | checkmenuitem | dualbutton | groupbox | listbox | listitem | menuarrow | menubar | menucheckbox | menuimage | menuitem | menuitemtext | menulist | menulist-button | menulist-text | menulist-textfield | menupopup | menuradio | menuseparator | meterbar | meterchunk | progressbar | progressbar-vertical | progresschunk | progresschunk-vertical | radio | radio-container | radio-label | radiomenuitem | range | range-thumb | resizer | resizerpanel | scale-horizontal | scalethumbend | scalethumb-horizontal | scalethumbstart | scalethumbtick | scalethumb-vertical | scale-vertical | scrollbarbutton-down | scrollbarbutton-left | scrollbarbutton-right | scrollbarbutton-up | scrollbarthumb-horizontal | scrollbarthumb-vertical | scrollbartrack-horizontal | scrollbartrack-vertical | searchfield | separator | sheet | spinner | spinner-downbutton | spinner-textfield | spinner-upbutton | splitter | statusbar | statusbarpanel | tab | tabpanel | tabpanels | tab-scroll-arrow-back | tab-scroll-arrow-forward | textfield | textfield-multiline | toolbar | toolbarbutton | toolbarbutton-dropdown | toolbargripper | toolbox | tooltip | treeheader | treeheadercell | treeheadersortarrow | treeitem | treeline | treetwisty | treetwistyopen | treeview | -moz-mac-unified-toolbar | -moz-win-borderless-glass | -moz-win-browsertabbar-toolbox | -moz-win-communicationstext | -moz-win-communications-toolbox | -moz-win-exclude-glass | -moz-win-glass | -moz-win-mediatext | -moz-win-media-toolbox | -moz-window-button-box | -moz-window-button-box-maximized | -moz-window-button-close | -moz-window-button-maximize | -moz-window-button-minimize | -moz-window-button-restore | -moz-window-frame-bottom | -moz-window-frame-left | -moz-window-frame-right | -moz-window-titlebar | -moz-window-titlebar-maximized",
                "relevance": 0,
                "description": "Used in Gecko (Firefox) to display an element using a platform-native styling based on the operating system's theme.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "-moz-backface-visibility",
                "browsers": [
                    "FF10"
                ],
                "values": [
                    {
                        "name": "hidden"
                    },
                    {
                        "name": "visible"
                    }
                ],
                "relevance": 50,
                "description": "Determines whether or not the 'back' side of a transformed element is visible when facing the viewer. With an identity transform, the front side of an element faces the viewer.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "-moz-background-clip",
                "browsers": [
                    "FF1-3.6"
                ],
                "values": [
                    {
                        "name": "padding"
                    }
                ],
                "relevance": 50,
                "description": "Determines the background painting area.",
                "restrictions": [
                    "box",
                    "enum"
                ]
            },
            {
                "name": "-moz-background-inline-policy",
                "browsers": [
                    "FF1"
                ],
                "values": [
                    {
                        "name": "bounding-box"
                    },
                    {
                        "name": "continuous"
                    },
                    {
                        "name": "each-box"
                    }
                ],
                "relevance": 50,
                "description": "In Gecko-based applications like Firefox, the -moz-background-inline-policy CSS property specifies how the background image of an inline element is determined when the content of the inline element wraps onto multiple lines. The choice of position has significant effects on repetition.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "-moz-background-origin",
                "browsers": [
                    "FF1"
                ],
                "relevance": 50,
                "description": "For elements rendered as a single box, specifies the background positioning area. For elements rendered as multiple boxes (e.g., inline boxes on several lines, boxes on several pages) specifies which boxes 'box-decoration-break' operates on to determine the background positioning area(s).",
                "restrictions": [
                    "box"
                ]
            },
            {
                "name": "-moz-border-bottom-colors",
                "browsers": [
                    "FF1"
                ],
                "status": "nonstandard",
                "syntax": "<color>+ | none",
                "relevance": 0,
                "description": "Sets a list of colors for the bottom border.",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "-moz-border-image",
                "browsers": [
                    "FF3.6"
                ],
                "values": [
                    {
                        "name": "auto",
                        "description": "If 'auto' is specified then the border image width is the intrinsic width or height (whichever is applicable) of the corresponding image slice. If the image does not have the required intrinsic dimension then the corresponding border-width is used instead."
                    },
                    {
                        "name": "fill",
                        "description": "Causes the middle part of the border-image to be preserved."
                    },
                    {
                        "name": "none"
                    },
                    {
                        "name": "repeat",
                        "description": "The image is tiled (repeated) to fill the area."
                    },
                    {
                        "name": "round",
                        "description": "The image is tiled (repeated) to fill the area. If it does not fill the area with a whole number of tiles, the image is rescaled so that it does."
                    },
                    {
                        "name": "space",
                        "description": "The image is tiled (repeated) to fill the area. If it does not fill the area with a whole number of tiles, the extra space is distributed around the tiles."
                    },
                    {
                        "name": "stretch",
                        "description": "The image is stretched to fill the area."
                    },
                    {
                        "name": "url()"
                    }
                ],
                "relevance": 50,
                "description": "Shorthand property for setting 'border-image-source', 'border-image-slice', 'border-image-width', 'border-image-outset' and 'border-image-repeat'. Omitted values are set to their initial values.",
                "restrictions": [
                    "length",
                    "percentage",
                    "number",
                    "url",
                    "enum"
                ]
            },
            {
                "name": "-moz-border-left-colors",
                "browsers": [
                    "FF1"
                ],
                "status": "nonstandard",
                "syntax": "<color>+ | none",
                "relevance": 0,
                "description": "Sets a list of colors for the bottom border.",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "-moz-border-right-colors",
                "browsers": [
                    "FF1"
                ],
                "status": "nonstandard",
                "syntax": "<color>+ | none",
                "relevance": 0,
                "description": "Sets a list of colors for the bottom border.",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "-moz-border-top-colors",
                "browsers": [
                    "FF1"
                ],
                "status": "nonstandard",
                "syntax": "<color>+ | none",
                "relevance": 0,
                "description": "Ske Firefox, -moz-border-bottom-colors sets a list of colors for the bottom border.",
                "restrictions": [
                    "color"
                ]
            },
            {
                "name": "-moz-box-align",
                "browsers": [
                    "FF1"
                ],
                "values": [
                    {
                        "name": "baseline",
                        "description": "If this box orientation is inline-axis or horizontal, all children are placed with their baselines aligned, and extra space placed before or after as necessary. For block flows, the baseline of the first non-empty line box located within the element is used. For tables, the baseline of the first cell is used."
                    },
                    {
                        "name": "center",
                        "description": "Any extra space is divided evenly, with half placed above the child and the other half placed after the child."
                    },
                    {
                        "name": "end",
                        "description": "For normal direction boxes, the bottom edge of each child is placed along the bottom of the box. Extra space is placed above the element. For reverse direction boxes, the top edge of each child is placed along the top of the box. Extra space is placed below the element."
                    },
                    {
                        "name": "start",
                        "description": "For normal direction boxes, the top edge of each child is placed along the top of the box. Extra space is placed below the element. For reverse direction boxes, the bottom edge of each child is placed along the bottom of the box. Extra space is placed above the element."
                    },
                    {
                        "name": "stretch",
                        "description": "The height of each child is adjusted to that of the containing block."
                    }
                ],
                "relevance": 50,
                "description": "Specifies how a XUL box aligns its contents across (perpendicular to) the direction of its layout. The effect of this is only visible if there is extra space in the box.",
                "restrictions": [
                    "enum"
                ]
            },
            {
                "name": "-moz-box-direction",
                "browsers": [
                    "FF1"
                ],
                "values": [
                    {
                        "name": "normal",
                        "description": "A box with a computed value of horizontal for box-orient displays its children from left to right. A box with a computed value of vertical displays its children from top to bottom."
                    },
                    {
                        "name": "reverse",
                        "description": "A box with a computed value of horizontal for box-orient displays its children from right to l