"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _classCallCheck2 = _interopRequireDefault(require("@babel/runtime/helpers/classCallCheck"));

var _createClass2 = _interopRequireDefault(require("@babel/runtime/helpers/createClass"));

var _history = require("history");

var _queryString = _interopRequireDefault(require("./queryString"));

function isNumericString(num) {
  return !isNaN(num);
}

function toSingleValue(val) {
  return Array.isArray(val) ? val[val.length - 1] : val;
}

function toSingleValueInteger(num) {
  return toInteger(toSingleValue(num));
}

function toInteger(num) {
  if (!isNumericString(num)) return;
  return parseInt(num, 10);
}

function parseFiltersFromQueryParams(queryParams) {
  return queryParams.filters;
}

function parseCurrentFromQueryParams(queryParams) {
  return toSingleValueInteger(queryParams.current);
}

function parseSearchTermFromQueryParams(queryParams) {
  return toSingleValue(queryParams.q);
}

function parseSortFromQueryParams(queryParams) {
  var sortField = toSingleValue(queryParams["sort-field"]);
  var sortDirection = toSingleValue(queryParams["sort-direction"]);
  if (sortField) return [sortField, sortDirection];
  return [];
}

function parseSizeFromQueryParams(queryParams) {
  return toSingleValueInteger(queryParams.size);
}

function paramsToState(queryParams) {
  var state = {
    current: parseCurrentFromQueryParams(queryParams),
    filters: parseFiltersFromQueryParams(queryParams),
    searchTerm: parseSearchTermFromQueryParams(queryParams),
    resultsPerPage: parseSizeFromQueryParams(queryParams),
    sortField: parseSortFromQueryParams(queryParams)[0],
    sortDirection: parseSortFromQueryParams(queryParams)[1]
  };
  return Object.keys(state).reduce(function (acc, key) {
    var value = state[key];
    if (value) acc[key] = value;
    return acc;
  }, {});
}

function stateToParams(_ref) {
  var searchTerm = _ref.searchTerm,
      current = _ref.current,
      filters = _ref.filters,
      resultsPerPage = _ref.resultsPerPage,
      sortDirection = _ref.sortDirection,
      sortField = _ref.sortField;
  var params = {};
  if (current > 1) params.current = current;
  if (searchTerm) params.q = searchTerm;
  if (resultsPerPage) params.size = resultsPerPage;

  if (filters && filters.length > 0) {
    params["filters"] = filters;
  }

  if (sortField) {
    params["sort-field"] = sortField;
    params["sort-direction"] = sortDirection;
  }

  return params;
}

function stateToQueryString(state) {
  return _queryString.default.stringify(stateToParams(state));
}
/**
 * The URL Manager is responsible for synchronizing state between
 * SearchDriver and the URL. There are 3 main cases we handle when
 * synchronizing:
 *
 * 1. When the app loads, SearchDriver will need to
 * read the current state from the URL, in order to perform the search
 * expressed by the query string. `getStateFromURL` is used for this case.
 *
 * 2. When the URL changes as a result of `pushState` or `replaceState`,
 * SearchDriver will need to be notified and given the updated state, so that
 * it can re-run the current search. `onURLStateChange` is used for this case.
 *
 * 3. When state changes internally in the SearchDriver, as a result of an
 * Action, it will need to notify the URLManager of the change. `pushStateToURL`
 * is used for this case.
 */


var URLManager = /*#__PURE__*/function () {
  function URLManager() {
    (0, _classCallCheck2.default)(this, URLManager);
    this.history = (0, _history.createBrowserHistory)();
    this.lastPushSearchString = "";
  }
  /**
   * Parse the current URL into application state
   *
   * @return {Object} - The parsed state object
   */


  (0, _createClass2.default)(URLManager, [{
    key: "getStateFromURL",
    value: function getStateFromURL() {
      return paramsToState(_queryString.default.parse(this.history.location.search));
    }
    /**
     * Push the current state of the application to the URL
     *
     * @param {Object} state - The entire current state from the SearchDriver
     * @param {boolean} options
     * @param {boolean} options.replaceUrl - When pushing state to the URL, use history 'replace'
     * rather than 'push' to avoid adding a new history entry
     */

  }, {
    key: "pushStateToURL",
    value: function pushStateToURL(state) {
      var _ref2 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
          _ref2$replaceUrl = _ref2.replaceUrl,
          replaceUrl = _ref2$replaceUrl === void 0 ? false : _ref2$replaceUrl;

      var searchString = stateToQueryString(state);
      this.lastPushSearchString = searchString;
      var navigationFunction = replaceUrl ? this.history.replace : this.history.push;
      navigationFunction({
        search: "?".concat(searchString)
      });
    }
    /**
     * Add an event handler to be executed whenever state is pushed to the URL
     *
     * @callback requestCallback
     * @param {Object} state - Updated application state parsed from the new URL
     *
     * @param {requestCallback} callback
     */

  }, {
    key: "onURLStateChange",
    value: function onURLStateChange(callback) {
      var _this = this;

      this.unlisten = this.history.listen(function (location) {
        // If this URL is updated as a result of a pushState request, we don't
        // want to notify that the URL changed.
        if ("?".concat(_this.lastPushSearchString) === location.search) return; // Once we've decided to return based on lastPushSearchString, reset
        // it so that we don't break back / forward button.

        _this.lastPushSearchString = "";
        callback(paramsToState(_queryString.default.parse(location.search)));
      });
    }
  }, {
    key: "tearDown",
    value: function tearDown() {
      this.unlisten();
    }
  }]);
  return URLManager;
}();

exports.default = URLManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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