import _slicedToArray from "@babel/runtime/helpers/slicedToArray";
import _classCallCheck from "@babel/runtime/helpers/classCallCheck";
import _createClass from "@babel/runtime/helpers/createClass";
import _defineProperty from "@babel/runtime/helpers/defineProperty";

/**
 * minimal debounce function
 *
 * mostly for not spamming the server with requests when
 * searching with type ahead
 */
function debounce(func, wait) {
  var timeout;

  var debouncedFn = function debouncedFn() {
    var args = arguments;

    var later = function later() {
      func.apply(null, args);
    };

    clearTimeout(timeout);
    timeout = setTimeout(later, wait);
  };

  debouncedFn.cancel = function () {
    if (timeout) {
      clearTimeout(timeout);
      timeout = null;
    }
  };

  return debouncedFn;
}

var DebounceManager = /*#__PURE__*/function () {
  function DebounceManager() {
    _classCallCheck(this, DebounceManager);

    _defineProperty(this, "debounceCache", {});
  }

  _createClass(DebounceManager, [{
    key: "runWithDebounce",
    value:
    /**
     * Dynamically debounce and cache a debounced version of a function at the time of calling that function. This avoids
     * managing debounced version of functions locally.
     *
     * In other words, debounce usually works by debouncing based on
     * referential identity of a function. This works by comparing provided function names.
     *
     * This also has the ability to short-circuit a debounce all-together, if no wait
     * time is provided.
     *
     * Assumption:
     * Functions are debounced on a combination of unique function name and wait times. So debouncing won't work on
     * subsequent calls with different wait times or different functions. That also means that the debounce manager
     * can be used for different functions in parallel, and keep the two functions debounced separately.
     *
     * @param {number} wait Milliseconds to debounce. Executes immediately if falsey.
     * @param {function} fn Function to debounce
     * @param {function} functionName Name of function to debounce, used to create a unique key
     * @param {...any} parameters Parameters to pass to function
     */
    function runWithDebounce(wait, functionName, fn) {
      for (var _len = arguments.length, parameters = new Array(_len > 3 ? _len - 3 : 0), _key = 3; _key < _len; _key++) {
        parameters[_key - 3] = arguments[_key];
      }

      if (!wait) {
        return fn.apply(void 0, parameters);
      }

      var key = "".concat(functionName, "|").concat(wait.toString());
      var debounced = this.debounceCache[key];

      if (!debounced) {
        this.debounceCache[key] = debounce(fn, wait);
        debounced = this.debounceCache[key];
      }

      debounced.apply(void 0, parameters);
    }
    /**
     * Cancels existing debounced function calls.
     *
     * This will cancel any debounced function call, regardless of the debounce length that was provided.
     *
     * For example, making the following series of calls will create multiple debounced functions, because
     * they are cached by a combination of unique name and debounce length.
     *
     * runWithDebounce(1000, "_updateSearchResults", this._updateSearchResults)
     * runWithDebounce(500, "_updateSearchResults", this._updateSearchResults)
     * runWithDebounce(1000, "_updateSearchResults", this._updateSearchResults)
     *
     * Calling the following will cancel all of those, if they have not yet executed:
     *
     * cancelByName("_updateSearchResults")
     *
     * @param {string} functionName The name of the function that was debounced. This needs to match exactly what was provided
     * when runWithDebounce was called originally.
     */

  }, {
    key: "cancelByName",
    value: function cancelByName(functionName) {
      Object.entries(this.debounceCache).filter(function (_ref) {
        var _ref2 = _slicedToArray(_ref, 1),
            cachedKey = _ref2[0];

        return cachedKey.startsWith("".concat(functionName, "|"));
      }) // eslint-disable-next-line no-unused-vars
      .forEach(function (_ref3) {
        var _ref4 = _slicedToArray(_ref3, 2),
            _ = _ref4[0],
            cachedValue = _ref4[1];

        return cachedValue.cancel();
      });
    }
  }]);

  return DebounceManager;
}();
/**
 * Perform a standard debounce
 *
 * @param {number} wait Milliseconds to debounce. Executes immediately if falsey.
 * @param {function} fn Function to debounce
 */


DebounceManager.debounce = function (wait, fn) {
  return debounce(fn, wait);
};

export default DebounceManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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