package testvectors

const ChainTests = `
{
  "tests": {
    "test_ralph_sig_chain": {
      "_comment": "Ralph is a test user I created by hand on my local server. I fetched his sigs and keys from the API, and then massaged them into our input format. This test is mainly to make sure that the generated chains we're using in other tests bear some relationship to reality.  - Jack",
      "input": "ralph_chain.json",
      "len": 5,
      "sibkeys": 3,
      "subkeys": 2,
      "eldest": "second_eldest"
    },

    "test_simple_chain": {
      "_comment": "Test a simple chain, just one link.",
      "input": "simple_chain.json",
      "len": 1,
      "sibkeys": 1,
      "subkeys": 0
    },

    "test_error_unknown_key": {
      "_comment": "Check the case where a signing kid is simply missing from the list of available keys (as opposed to invalid for some other reason, like having been revoked).",
      "input": "missing_kid_chain.json",
      "eldest": "e",
      "err_type": "NONEXISTENT_KID"
    },

    "test_error_unknown_reverse_sig_key": {
      "_comment": "As above, but for a reverse sig.",
      "input": "missing_reverse_kid_chain.json",
      "eldest": "e",
      "err_type": "NONEXISTENT_KID"
    },

    "test_error_bad_signature": {
      "_comment": "Change some bytes from the valid signature, and confirm it gets rejected.",
      "input": "bad_signature_chain.json",
      "err_type": "VERIFY_FAILED"
    },

    "test_error_bad_reverse_signature": {
      "_comment": "Change some bytes from the valid reverse signature, and confirm it gets rejected.",
      "input": "bad_reverse_signature_chain.json",
      "err_type": "REVERSE_SIG_VERIFY_FAILED"
    },

    "test_error_mismatched_kid": {
      "_comment": "We need to use the server-provided KID to unbox a signature. We always need to check back after unboxing to make sure the internal KID matches the one we actually used. This test exercises that check. NOTE: I generated this chain by hacking some code into kbpgp to modify the payload right before it was signed.",
      "input": "mismatched_kid_chain.json",
      "err_type": "KID_MISMATCH"
    },

    "test_error_mismatched_fingerprint": {
      "_comment": "We don't use fingerprints in unboxing, but nonetheless we want to make sure that if a chain link claims to have been signed by a given fingerprint, that does in fact correspond to the KID of the PGP key that signed it. NOTE: I generated this chain by hacking some code into kbpgp to modify the payload right before it was signed.",
      "input": "mismatched_fingerprint_chain.json",
      "err_type": "FINGERPRINT_MISMATCH"
    },

    "test_revokes": {
      "_comment": "The chain is length 10, but after 2 sig revokes it should be length 8. Likewise, 6 keys are delegated, but after 2 sig revokes and 2 key revokes it should be down to 2 keys.",
      "input": "example_revokes_chain.json",
      "len": 13,
      "sibkeys": 2,
      "subkeys": 1
    },

    "test_error_revoked_key": {
      "_comment": "Try signing a link with a key that was previously revoked.",
      "input": "signed_with_revoked_key_chain.json",
      "err_type": "INVALID_SIBKEY"
    },

    "test_error_expired_key": {
      "_comment": "Try signing a link with a key that has expired.",
      "input": "expired_key_chain.json",
      "err_type": "EXPIRED_SIBKEY"
    },

    "test_error_bad_uid": {
      "input": "bad_uid_chain.json",
      "err_type": "WRONG_UID"
    },

    "test_error_bad_username": {
      "input": "bad_username_chain.json",
      "err_type": "WRONG_USERNAME"
    },

    "test_error_bad_prev": {
      "input": "bad_prev_chain.json",
      "err_type": "WRONG_PREV"
    },

    "test_error_bad_seqno": {
      "input": "bad_seqno_chain.json",
      "err_type": "WRONG_SEQNO"
    },

    "test_empty_sigchain": {
      "input": "empty_chain.json",
      "len": 0,
      "sibkeys": 1,
      "subkeys": 0,
      "eldest": "ralph"
    },

    "test_error_empty_sigchain_bad_key_userid": {
      "input": "empty_chain.json",
      "eldest": "steve",
      "err_type": "KEY_OWNERSHIP"
    },

    "test_error_empty_sigchain_nonexistent_pgp": {
      "input": "empty_chain.json",
      "eldest": "nonexistent",
      "err_type": "NONEXISTENT_KID"
    },

    "test_error_empty_sigchain_eldest_key_is_nacl": {
      "_comment": "NaCl keys don't have any internal userid info, so they can't stand alone as an eldest key without having signed some links.",
      "input": "empty_chain.json",
      "eldest": "naclkey",
      "err_type": "KEY_OWNERSHIP"
    },

    "test_jack_chain": {
      "_comment": "Just a sanity check with my own sigchain",
      "input": "jack_chain.json",
      "len": 29,
      "sibkeys": 1,
      "subkeys": 0
    },

    "test_error_missing_first_link": {
      "_comment": "We need to make sure the server can't drop links off the front. This would be a risk if we assumed that, for example, they belonged to a previous eldest key.",
      "input": "jack_chain_missing_first_link.json",
      "err_type": "WRONG_SEQNO"
    },

    "test_error_substitute_first_link": {
      "_comment": "Here the first link has been substituted for one signed by a different eldest key. We should notice the payload hash mismatch.",
      "input": "jack_chain_substitute_first_link.json",
      "eldest": "real_eldest",
      "err_type": "WRONG_PREV"
    },

    "test_ralph_with_earlier_eldest_key": {
      "_comment": "Eldest keys can be reused, which means there can be subchains other than the latest one that correspond to the current eldest key. Those must be ignored. Previously, before we supported repeating eldest keys at all, we used to require that this was an error. Note that if the latest links don't correspond to the current eldest key, as we're testing here, that's an implicit reset, and we return no links.",
      "input": "ralph_chain.json",
      "eldest": "first_eldest",
      "len": 0,
      "sibkeys": 1,
      "subkeys": 0
    },

    "test_ralph_with_new_eldest_key": {
      "input": "ralph_chain.json",
      "eldest": "new_eldest",
      "len": 0,
      "sibkeys": 1,
      "subkeys": 0
    },

    "test_melvin_with_expiring_pgp_key": {
      "_comment": "Right now this just exercises some of the code that recognizes the expiration time. Eventually we should add tests that make sure the etime is really enforced.",
      "input": "melvin_chain.json",
      "len": 1,
      "sibkeys": 1,
      "subkeys": 0
    },

    "test_error_ralph_with_missing_subkey": {
      "input": "nonexistent_subkey_chain.json",
      "err_type": "NONEXISTENT_KID"
    },

    "test_helen_chain": {
      "_comment": "A copy of t_helen's sigchain (also available as a permanent test user). Tests PGP keys with changing subkeys. She signed up with a PGP key, added a bitcoin address, and then changed that address, for a total of 3 sigchain links (but the first bitcoin link is revoked, so the final len is 2). In between each one she deleted the signing subkey that was used before and added a new one. So the only way to validate this sigchain is to do proper PGP key merging on all the bundles that the server returns. Note that the order of the key bundles given is intentionally rearranged, to make it less likely that the Go client will spuriously pass this test by only checking the last link against the last key.",
      "input": "t_helen_chain.json",
      "len": 2,
      "sibkeys": 1,
      "subkeys": 0
    },
    "test_pgp_subkey_after_revoke": {
      "_comment": "The eldest link is signed with a PGP key with an extra signing subkey, then a pgp_update link is used to update the key to a version where that subkey has been revoked. Then, another pgp_update link appears which tries to revert to the original version of the key, but is signed by the revoked subkey.",
      "input": "old_pgp_key.json",
      "err_type": "VERIFY_FAILED"
    },
    "test_pgp_bad_full_hash": {
      "_comment": "The eldest link is signed with a PGP key and includes a full_hash which doesn't match the key",
      "input": "bad_full_hash.json",
      "err_type": "NO_KEY_WITH_THIS_HASH"
    },
    "test_pnelsen_sigchain": {
      "_comment": "pnelsen was one of several users affected by the 15 Sep 2015 white-space stripping bug on the server",
      "input": "pnelsen_chain.json",
      "len": 21,
      "sibkeys": 1,
      "subkeys": 0
    },
    "test_akalin_sigchain": {
      "_comment": "Fred has some unverifiable links in his chain from old bugs of ours. These need to be special cased.",
      "input": "akalin_chain.json",
      "len": 31,
      "sibkeys": 1,
      "subkeys": 0
    },
    "test_repeat_eldest": {
      "_comment": "We now define 'eldest' links to imply a sigchain reset. This is a basic test case with two links, both of type eldest, delegating the same eldest key. Old clients would've parsed this as a 2-link chain (where the second link was a no-op), but conforming clients should respect the reset adn return a 1-link chain.",
      "input": "repeat_eldest.json",
      "len": 1,
      "sibkeys": 1,
      "subkeys": 0
    },
    "test_hardcoded_reset": {
      "_comment": "Several users (6) managed to do a sigchain reset and reuse their eldest key without an 'eldest' link, before we prohibited that on the server. This test uses the chain of one of those users, as a canary to make sure they're covered.",
      "input": "hardcoded_reset.json",
      "len": 1,
      "sibkeys": 1,
      "subkeys": 0
    }
  }
}
`

var ChainTestInputs = map[string]string{
	"bad_prev_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgtAfp8c2vIsq6o/a4fzGays16KagtWSwxbObC7mU2UngKp3BheWxvYWTFASp7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMGI0MDdlOWYxY2RhZjIyY2FiYWEzZjZiODdmMzE5YWNhY2Q3YTI5YTgyZDU5MmMzMTZjZTZjMmVlNjUzNjUyNzgwYSIsInVpZCI6Ijc0YzM4Y2Y3Y2ViOTQ3ZjU2MzIwNDVkOGNhNWQ0ODE5IiwidXNlcm5hbWUiOiJtYXgzMiJ9LCJ0eXBlIjoiZWxkZXN0IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMjI1OTA0LCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjpudWxsLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjEsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RALK1iab8sGnnb4waKvKzJAO2kpRvTyaCy7u6W8ZI2+OALN0SmeZZ4ZZk9A+qDWNWFaw7LRDg7/wafrn4sVjwlDKhzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
            "payload_hash": "df5d9f31e681f921d15e1dc98dd875f3f299dcebdf5f4267827416a2117cb22d",
            "sig_id": "3dc0ed1000573ccde355fd4004274c2aa45aed986df5fcbc187ce1ceffdfcb560f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"0120b407e9f1cdaf22cabaa3f6b87f319acacd7a29a82d592c316ce6c2ee653652780a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432225904,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "0120b407e9f1cdaf22cabaa3f6b87f319acacd7a29a82d592c316ce6c2ee653652780a",
            "ctime": 1432225904
        },
        {
            "seqno": 2,
            "prev": "deadbeef00000000000000000000000000000000000000000000000000000000",
            "sig": "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",
            "payload_hash": "bca91e71714a849c255c531c60bc6e4a69c33782edeba8afcab9fc22b286171e",
            "sig_id": "10a9ec422404d3b0b062e56c809db16e3301bd01deb4d54ee77e5895c29a79660f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120b407e9f1cdaf22cabaa3f6b87f319acacd7a29a82d592c316ce6c2ee653652780a\",\"host\":\"keybase.io\",\"kid\":\"0120b407e9f1cdaf22cabaa3f6b87f319acacd7a29a82d592c316ce6c2ee653652780a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0120126e4cb40127102247b460ac6e4872c35b53075af0598ccb04034b961cf740cd0a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgEm5MtAEnECJHtGCsbkhyw1tTB1rwWYzLBANLlhz3QM0Kp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYjQwN2U5ZjFjZGFmMjJjYWJhYTNmNmI4N2YzMTlhY2FjZDdhMjlhODJkNTkyYzMxNmNlNmMyZWU2NTM2NTI3ODBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYjQwN2U5ZjFjZGFmMjJjYWJhYTNmNmI4N2YzMTlhY2FjZDdhMjlhODJkNTkyYzMxNmNlNmMyZWU2NTM2NTI3ODBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwMTI2ZTRjYjQwMTI3MTAyMjQ3YjQ2MGFjNmU0ODcyYzM1YjUzMDc1YWYwNTk4Y2NiMDQwMzRiOTYxY2Y3NDBjZDBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMjI2MDA0LCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiZGVhZGJlZWYwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMCIsInNlcV90eXBlIjoxLCJzZXFubyI6MiwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEB7MkCVRfz1rQw9QyfguqbncdwBve1YGTcopsxfibwjRj98Io1n4kAo8i5pBP4Wk7HiTSsr7DJNF6GRtUiJLMkCqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432226004,\"expire_in\":10000000,\"prev\":\"deadbeef00000000000000000000000000000000000000000000000000000000\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "0120b407e9f1cdaf22cabaa3f6b87f319acacd7a29a82d592c316ce6c2ee653652780a",
            "ctime": 1432226004
        }
    ],
    "keys": [
        "0120b407e9f1cdaf22cabaa3f6b87f319acacd7a29a82d592c316ce6c2ee653652780a",
        "0120126e4cb40127102247b460ac6e4872c35b53075af0598ccb04034b961cf740cd0a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"bad_reverse_signature_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgd1JXIa21+kSY41/Nx2jLADJcFxGLZ0Rgxug6xaFH67EKp3BheWxvYWTFASp7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMDc3NTI1NzIxYWRiNWZhNDQ5OGUzNWZjZGM3NjhjYjAwMzI1YzE3MTE4YjY3NDQ2MGM2ZTgzYWM1YTE0N2ViYjEwYSIsInVpZCI6Ijc0YzM4Y2Y3Y2ViOTQ3ZjU2MzIwNDVkOGNhNWQ0ODE5IiwidXNlcm5hbWUiOiJtYXgzMiJ9LCJ0eXBlIjoiZWxkZXN0IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0MjQ1LCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjpudWxsLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjEsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAz1C4EVTGxElMcIHOJqRFTmqA0XeFS8oAweX4Bih+6yfzi6sqiOefKBg5AvG3gXeWnHdsSCzJIun8Anwuhf+zAqhzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
            "payload_hash": "2eddd3fa83f7f34808526b28bb887a95c09dc81f833c88b234a4be2311dd89f4",
            "sig_id": "7c7ef1f35cf56e63b62f42d1afee7682022bed0c58981f9ca4f8601929cb923c0f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"012077525721adb5fa4498e35fcdc768cb00325c17118b674460c6e83ac5a147ebb10a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432144245,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "012077525721adb5fa4498e35fcdc768cb00325c17118b674460c6e83ac5a147ebb10a",
            "ctime": 1432144245
        },
        {
            "seqno": 2,
            "prev": "2eddd3fa83f7f34808526b28bb887a95c09dc81f833c88b234a4be2311dd89f4",
            "sig": "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",
            "payload_hash": "4ccc4185a501dd16f2e2226182d3fadf8558aa59b593b5e4a239aea35581102d",
            "sig_id": "2d8dcd8fe4e7ceb9c67a76c18bbee71524d0ba0aa7c2e10298ce5688cf3cd2c80f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"012077525721adb5fa4498e35fcdc768cb00325c17118b674460c6e83ac5a147ebb10a\",\"host\":\"keybase.io\",\"kid\":\"012077525721adb5fa4498e35fcdc768cb00325c17118b674460c6e83ac5a147ebb10a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"012057257eeb8feed5d331d63ba382b0d5cffba17c95d32a83c7a58b5527bad873250a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgVyV+64/u1dMx1jujgrDVz/uhfJXTKoPHpYtVJ7rYcyUKp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwNzc1MjU3MjFhZGI1ZmE0NDk4ZTM1ZmNkYzc2OGNiMDAzMjVjMTcxMThiNjc0NDYwYzZlODNhYzVhMTQ3ZWJiMTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwNzc1MjU3MjFhZGI1ZmE0NDk4ZTM1ZmNkYzc2OGNiMDAzMjVjMTcxMThiNjc0NDYwYzZlODNhYzVhMTQ3ZWJiMTBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwNTcyNTdlZWI4ZmVlZDVkMzMxZDYzYmEzODJiMGQ1Y2ZmYmExN2M5NWQzMmE4M2M3YTU4YjU1MjdiYWQ4NzMyNTBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0MzQ1LCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiMmVkZGQzZmE4M2Y3ZjM0ODA4NTI2YjI4YmI4ODdhOTVjMDlkYzgxZjgzM2M4OGIyMzRhNGJlMjMxMWRkODlmNCIsInNlcV90eXBlIjoxLCJzZXFubyI6MiwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEAAAAAAQcgOiJ9NVPw9R/PVasiuow1AKdCTncJcmqJTYcG+0U2CSVUsKbT6iop7qEuJn7kD5NNoz3kRE31SJqsJqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432144345,\"expire_in\":10000000,\"prev\":\"2eddd3fa83f7f34808526b28bb887a95c09dc81f833c88b234a4be2311dd89f4\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "012077525721adb5fa4498e35fcdc768cb00325c17118b674460c6e83ac5a147ebb10a",
            "ctime": 1432144345
        }
    ],
    "keys": [
        "012077525721adb5fa4498e35fcdc768cb00325c17118b674460c6e83ac5a147ebb10a",
        "012057257eeb8feed5d331d63ba382b0d5cffba17c95d32a83c7a58b5527bad873250a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"bad_seqno_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgsDiGCNcoqXQu5wYLlcIpCJv1Vlj7OJ3mmHN1pC8If7cKp3BheWxvYWTFASp7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMGIwMzg4NjA4ZDcyOGE5NzQyZWU3MDYwYjk1YzIyOTA4OWJmNTU2NThmYjM4OWRlNjk4NzM3NWE0MmYwODdmYjcwYSIsInVpZCI6Ijc0YzM4Y2Y3Y2ViOTQ3ZjU2MzIwNDVkOGNhNWQ0ODE5IiwidXNlcm5hbWUiOiJtYXgzMiJ9LCJ0eXBlIjoiZWxkZXN0IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0MzQzLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjpudWxsLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjEsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAzscTjEcAa9aF8Ze5MOUpDaR2BRSFObI7CtWlFLSAySxG0kKv99hqh53WjTyUL9G4CZcg14o1WSJcS9P05XI2DahzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
            "payload_hash": "fafd9bbd0b7e905b96cb4b6c8022f7ee712098ef78ae048acb2fa296220d760f",
            "sig_id": "8dcc6e4d4de09e9bf0a1bf543e9bafbef589a2dce7797bd45296414a7b7cbff50f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"0120b0388608d728a9742ee7060b95c229089bf55658fb389de6987375a42f087fb70a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432144343,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "0120b0388608d728a9742ee7060b95c229089bf55658fb389de6987375a42f087fb70a",
            "ctime": 1432144343
        },
        {
            "seqno": 3,
            "prev": "fafd9bbd0b7e905b96cb4b6c8022f7ee712098ef78ae048acb2fa296220d760f",
            "sig": "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",
            "payload_hash": "db4a190291265dceb1d047089ca40c21cea4258e2eb7c353fd7aa19729e7b0c1",
            "sig_id": "9a0eb84b985fa57a38a850189206cd0cd257db3d9c6c1f6169373ba1b5b1e4be0f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120b0388608d728a9742ee7060b95c229089bf55658fb389de6987375a42f087fb70a\",\"host\":\"keybase.io\",\"kid\":\"0120b0388608d728a9742ee7060b95c229089bf55658fb389de6987375a42f087fb70a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"012015d09e9f3d13242c0b531228825b2762163fc4a72eca3439f91867734da305e20a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgFdCenz0TJCwLUxIoglsnYhY/xKcuyjQ5+Rhnc02jBeIKp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYjAzODg2MDhkNzI4YTk3NDJlZTcwNjBiOTVjMjI5MDg5YmY1NTY1OGZiMzg5ZGU2OTg3Mzc1YTQyZjA4N2ZiNzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYjAzODg2MDhkNzI4YTk3NDJlZTcwNjBiOTVjMjI5MDg5YmY1NTY1OGZiMzg5ZGU2OTg3Mzc1YTQyZjA4N2ZiNzBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwMTVkMDllOWYzZDEzMjQyYzBiNTMxMjI4ODI1YjI3NjIxNjNmYzRhNzJlY2EzNDM5ZjkxODY3NzM0ZGEzMDVlMjBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0NDQzLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiZmFmZDliYmQwYjdlOTA1Yjk2Y2I0YjZjODAyMmY3ZWU3MTIwOThlZjc4YWUwNDhhY2IyZmEyOTYyMjBkNzYwZiIsInNlcV90eXBlIjoxLCJzZXFubyI6MywidGFnIjoic2lnbmF0dXJlIn2jc2lnxEBZPnISDDH3rPasRuEiblt2HHP587d05UMxqNZwloGuVmIR2+SZMGfRGWCuNz5ZXoa47+GSErv6QD7sVwB4h9wPqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432144443,\"expire_in\":10000000,\"prev\":\"fafd9bbd0b7e905b96cb4b6c8022f7ee712098ef78ae048acb2fa296220d760f\",\"seq_type\":1,\"seqno\":3,\"tag\":\"signature\"}",
            "kid": "0120b0388608d728a9742ee7060b95c229089bf55658fb389de6987375a42f087fb70a",
            "ctime": 1432144443
        }
    ],
    "keys": [
        "0120b0388608d728a9742ee7060b95c229089bf55658fb389de6987375a42f087fb70a",
        "012015d09e9f3d13242c0b531228825b2762163fc4a72eca3439f91867734da305e20a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"bad_signature_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nyMDyAnicbZG9axVBFMU3fkEeCMGIYBWcLriE+d6ZBwYFRaysLCQxj9mZO+uSl93N\n7r4kL/EVgtj4B2ir2FinEMRCRCvFYGchigSxsLKxC7q7wc5pZs65v3suw31z8mjQ\nm03O725/+bTJpt69ejsKbgwuzeygOHdj1N9Bq9BdPs0SKIsyzWrURwriSHBLY08l\njihgLZTwAFQLACV47LGVlhMUott51XY0MbGpYCHNG68Rg9Q17n/41a6ACWHaSSo4\ncAXESkUki7yWxmPNsQTBBaWWRwBaSUUxBmdjCk4bxb3jkYwdNk3cqIuLuGXK+shC\nrHnkhWQUc+GUNcJxRXQLVlBmZg0aes1sMYomIarHRath6KD5RIg2oKzSPEN90hRt\nnbY04YwyLDQTIYKtIi1hkLYEPjwhKkrYQP1sNByGqIL1wWEo6USWd6/aJM2YKk0y\nU49KQJPXK8eC2V5w4viRdhlBb3rm34o2f08Fd1/+ef7h5t6vp/OTa+P3V65eeHbv\n89yD/cUnZ/ZXitPLt4KDU0s/zj76ee5r/uLO/MWH673h3lJSP853P377foCu37/8\nF4rxoso=\n=Zv9D\n-----END PGP MESSAGE-----\n",
            "payload_hash": "eea2476670616f48f64531bcad5ed592e053671dae9aa8ce35550af03151e77a",
            "sig_id": "dfc1b98e0a9094f3f68b74f698b7a084006bb88cf26ca592c84eac49663e7fd40f",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"8eb754c2bf26072e09585fee295ee854bf0c6c41\",\"host\":\"keybase.io\",\"key_id\":\"295ee854bf0c6c41\",\"kid\":\"01139d6254e48e1c681637f96af09406e54522c47ee9868200edcb2ed9a84fd476bd0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432305935,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "01139d6254e48e1c681637f96af09406e54522c47ee9868200edcb2ed9a84fd476bd0a",
            "ctime": 1432305935
        },
        {
            "seqno": 2,
            "prev": "eea2476670616f48f64531bcad5ed592e053671dae9aa8ce35550af03151e77a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nyMQhAnicrVU7jONEAM0dHBIrIR06CShXaRCXZeOxPf5EOgknsZPdxInzcbIJ0S1j\ne5xMsh5/87GPEy2cREWBkKBBSNAjIdEhPhUFHaK6ElEfBS1O9tA1VzLNaGbePL15\n1nv++ZUXCkd35qVvs8d/bLkbv/7CrQvmpXL7QdHynbRYeVBc4cOErxwcJ5cr4hQr\nRQYATnYEFvKYlzCwBQkInOjKAnIZmWcEDHnIsjYvYixLgsQyDHZsi8WOjCTedXhR\nsBwGFU+KLqFzHAURoUlOK2FLhLzNWi4rMCKLGRlK0MWYlSHGEuQtl7EFmwf5xYUf\n72/k4iwU41Pi53v54vIg7zn4/1n3+kAn8jYn2a5oY0vmRRcKHMvw0JFsBB1eAvIe\nGOOIIg/naA/tOLb48KQYE+upqc9UuVBE2OWB6LqAs5BrQwe7ruxYEmsBKOXM2AEy\ny7D5ieQyyIKOCCwRW5bA5Tj2oCrCGxzF+DIm85z27f2oqo2zzrHRMI51dTBQGuph\nd0ZHOZD4tHLcunbwuBtguodt2FPmFDAzWvM9D9OkcrxIkiCulMvPvC7bURok/ozO\naKprE4USO5qeiQj0q5rubFxVG899zZ+Mm4tJbdzOjPUuqDbKczVQY0MRlpkRdaZr\n0eM2m5zBR71d3G7UYb0xWqnxTrXUga/76nxORkG7kW76rLaO7KE9bDCe19BLTrm8\ndpue22FFWh/P6NghUuxiUxRFRpeTDeezGSpTrXRxsVp4aid3sa+vtpuGMvA9LoW1\nFmnqtaZeVVdaVRCqM4pb3vm05iM00NmmGvcutHq47p1tx17Y0nvDTUvpLzuTUrVV\nWtWSkLQnV03W7wejvrNajbegm7uFhYCAPun3VMs3TdVXJwrSNWenQWF41V6I46Yz\nIj1+yq55U8mG422Uthy+LBulKc5SMKOmvM1EZKS8srswzUVD7ztVnMY1IA2Y7SIM\nmRAqZlie1FDH9/lNuF3kjx3II00PalAdajM6OCcZM4kVfRrNO82RKnR7+rxPWsiO\nfNJihr1+g27r9Hzlw244CPkaMDt4xDEmkuz6FgW5k5bOoiB25W5KRleqPpB7yxb2\nLH9lLqnWaJyPESGskrZIdEYAqsqMdLGcnmf6vLHt6ECZ0RZJz5farpn2zGWWxPMz\nhQyN1XYbmmQ7aWWjZjDvKaHb18nZWatjEhDVkJh2+2ikdM2J5OWvSNZubzNts/w2\n6lUTq8PLqxrkYraj1cXh1KMOhUa2EUStzYYydUpiHfBJVo+wZCQrWZrR9jBKXSqk\n67KRCAHVciHZRbldHRqTJeY4bLgl2DE2YakbBoFtRrQEE7oEcsk1/FLIcTO6TLFB\nkwzW+5NEWDP3ZvSeCna5QYf0qJ36cxK1T3aSBvukPw34SXFznbJiBeSHdkL2PQB4\njuUYgeHgSRHvAhLhS7JHMNfjpBjkMc5JMEZsXjiCyAhAcHnJFXjIActGDsQOlPN2\nhJwgAgdhGeUfEHMQQibvMQ5AgEVx3wcxDi+vJYHDgvrFCpurRPODyDlFyTrCxYc/\n3n+xcOeo8NKtm/vqLxy9fPu/H8JPb94ofCxnDz599/PLQj6+Pvrqw2/eWP3wW124\n+/j9t+7yf43jG4Xvgn/okz/fe/L9l+88un/zM+x9cvX6R68+qvx967XTD34nX/wL\nbL0JeQ==\n=965F\n-----END PGP MESSAGE-----\n",
            "payload_hash": "3eafef1d80fc9e3c2d559bbc4a7b57522caaefc48d3a399cf9c2d7213284b0a7",
            "sig_id": "7b0b5f4fb66663c2290d78cddb1898236d6e065eb783e9aa85c8c67ac941e1a40f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01139d6254e48e1c681637f96af09406e54522c47ee9868200edcb2ed9a84fd476bd0a\",\"fingerprint\":\"8eb754c2bf26072e09585fee295ee854bf0c6c41\",\"host\":\"keybase.io\",\"key_id\":\"295ee854bf0c6c41\",\"kid\":\"01139d6254e48e1c681637f96af09406e54522c47ee9868200edcb2ed9a84fd476bd0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0113f57aef417ff13bafc5deff9db82b158632ed19202fc58f0ab5d71b7ebb63ff920a\",\"reverse_sig\":\"-----BEGIN PGP MESSAGE-----\\nVersion: Keybase OpenPGP v2.0.10\\nComment: https://keybase.io/crypto\\n\\nyMFYAnicrZI7a1RBFMdvfEFWgoFoYWHhYCWLzPuxpBG/gEpEsPA6jzPrNZu7m3vv\\nyoaQxsLGD5DGVkEsxEbESoMoEggiVpLGyvR2FurcTcTG0mmGM+d//ufHmfN27nDW\\nWdi8sfeU7770M9tv3o2za/nF+XXkhmEN9dbRMkwvGASom3y5CKiHMCHMBEkFB66B\\neKmJZCoaaSM2HEsQXFDquQIwWmqKMQTvKARjNY+BK+kCtqiLYlH2oRpVRdkkWw1O\\nCe6pi1RiRQEboUUEoEYAaMFdxF56TlLh7WHdViQ4Z2u4UAzTWwryKd4/9P+Zezy1\\nU9wz7aPy4AxXUUhGMRdBeysC18S0whqq0q5AUq/YCaNoo4vqwh0M9S9VFMpC5ETF\\nSJiz0YsAMZrgNHVE6OQMgRiKacroiK0TQRGnwDnJko5OqSq4C1UNeV30Ua8cDwap\\nWbM2apsf9OyiVlEMS9QjKembokUjnFGGJWaii2AyKirIi1aB908XjZJzMgGwNM1A\\nKiyJjFxHyQUjztsgIAiTPkwwqUiwYKzVHpgQAqfRMiIIKNUi1rCa7yORaVAOUY8m\\nStufQvZL24wrQBtbN49kC53s2NFD7TZmndn5Pzv67FL2q9nd+7D14tzDre8Ptk98\\nLTryfn6yu/Pt6pnFjztzX/LBTPYje33ePf+5NPvq+OqppcUrn+5tnj19+fPo+q33\\njyePntz5DRYt6u0=\\n=E1xp\\n-----END PGP MESSAGE-----\\n\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432306035,\"expire_in\":10000000,\"prev\":\"eea2476670616f48f64531bcad5ed592e053671dae9aa8ce35550af03151e77a\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "01139d6254e48e1c681637f96af09406e54522c47ee9868200edcb2ed9a84fd476bd0a",
            "ctime": 1432306035
        }
    ],
    "keys": [
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nxm8EVV9BDxMFK4EEACIDAwRSzcsRfEMpwQ7RvwWHPxXf97lwjf8mqqCeTZXkntaK\nLAYBBi4ZH2dPpVL2Nk6Bh0K7Zc9II0ksc0BL+z0fJ/3hIDhq9NgfusjqjiX8NYZ7\ndbeT+gyDPp5gzXaxPqrF0vXNBW1heDMywpIEExMKABoFAlVfQQ8CGwEDCwkHAxUK\nCAIeAQIXgAIZAQAKCRApXuhUvwxsQUGxAX9Y1nv+tbCxZGE4P22kbVQbi4BTyIkL\nYgNKqeoOlNxuWeDdh0xaDsimsTpEKWtSagMBgK179Xs4gsE9jPBtHGuzYpbHbROS\nhVCq9ssZjvId45D2UHFSjyAm8spEeFJzLARqEs5WBFVfQQ8SCCqGSM49AwEHAgME\nntY21MrNuVBiA28QJRFx/+nw8O6URDuF7P+a1Ou+c/mzeH8bH9NB+fozm+wt0+kU\nsAQ1rrmAdK9oXfbxFHw+VgMBCgnChwQYEwoADwUCVV9BDwUJDwmcAAIbDAAKCRAp\nXuhUvwxsQYFyAYCuu59mWoB68DUaSFY1YoIjXt10oKkJyqeaF/MDKCs4RZvnLkcM\nVfm4ANHRx8P4jTwBgKF4bBRJYKbGCh9Sdve3ivaihIjKueXbIkIwzlHKnDhH0ryF\nZy7AuYErrAMqZiEyG85SBFVfQQ8TCCqGSM49AwEHAgME2hMvKEfCqKxsX7+B70Lq\nfOOQg3mAP5vNEE9fP/O4CHS3nG7DRv4Di1vZlHE7u2OHAfHSAFq3ir935z4C0d7B\ncsLAJwQYEwoADwUCVV9BDwUJDwmcAAIbIgBqCRApXuhUvwxsQV8gBBkTCgAGBQJV\nX0EPAAoJEGcrs3re2HczDekBAJOT5B0avXmVYnPGkTwVZ3oeoXNwQEpYOm/kdPEx\nBFVhAQCKGjWboi7vVftw1cU/IFp1uh7lYWEzDlB9yvUCyd7/INoFAYC9wV1R++yE\nroIfq9hkzttJnvmuobSYaJHLMIQXnIADOBLHbHxrKInDtYqIToUoYsoBf2CMDj0r\nVwx1bGn7KprTMnu89hv/rIjnAcDYorsmqfECoXXZNBrVHS/DBpDROPPpag==\n=wdBi\n-----END PGP PUBLIC KEY BLOCK-----\n",
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nxm8EVV9BEBMFK4EEACIDAwS9o76Gzhjx/aRI+8EdNO7F011t/m6clC6VHpKASpc/\nmnygwK6qvxou750c6WgrBlFtnskjjpJcY2zqkgSKi+MKtpt6PJwWfQ1CR6aJBqNe\nSJyJVWS0yLuvf94v8u+e2c/NBW1heDMywpIEExMKABoFAlVfQRACGwEDCwkHAxUK\nCAIeAQIXgAIZAQAKCRCRTHSn/50zJ11MAYDI+TPMhrTb73FWiUxTkRvehyfWrLzk\nSFQS0RZpujJpzaaI2tE8Q/g5KU2oLEZpvdkBgPubVOmKL0LSWaaQzGaPrv7pR5+D\naGjOMf1VlUYSprgDiHrPnft1EiAlUsWsfX8yKs5WBFVfQRASCCqGSM49AwEHAgME\n77H3BHjofaqD5KYjvEkq9SAxc9i8VBuSrWplGelgeG/Ctr9UIvindI7uoRhHqAVa\nbjTFylO+CZy88cKdsRcj3wMBCgnChwQYEwoADwUCVV9BEAUJDwmcAAIbDAAKCRCR\nTHSn/50zJ+KDAX9nCQ75y87n0onZqDR3N+CC8QNvj6z+oaxvEh9w8AoF5auU9AQn\ny+/L5h5TNuQ/UkwBgLhredcA4cJ1+iBUOIQKJituuVYwvZ5wRFzRw6O7bs9XrbQZ\nZhki4WY7K64PuLvSQs5SBFVfQRATCCqGSM49AwEHAgMErT7N/XUzhm0YrnBF5uCk\nl9UIbLoiN4Zv8Vfj+KRJ8cWxQZVGKUd1lbTSQpE7js9adGEGqloqCEaxPaAYWtor\nQcLAJwQYEwoADwUCVV9BEAUJDwmcAAIbIgBqCRCRTHSn/50zJ18gBBkTCgAGBQJV\nX0EQAAoJEJda6Ks03LhjKjcBAOc8qE3v+yh8buW/vgeHbxTZRQLKaJoGSHC+TvgZ\nCe9/APwPIo89pm4CLE5vukuv0rSV3kQUF/LLkAtG2kwcvIUGO/2kAX9+aHMa/KlU\nLR4NJzhprqbXG9hWqR/R0DJLCm3HrP/J7v2glQ/yRWrAqdaYvxhASZoBfiTH1x2U\nL1C2MzFN8F/M8MQiobo5Kp4O/jLvj7leBQk/mXgaDAZrnAulV0cYn9xgfA==\n=PeLw\n-----END PGP PUBLIC KEY BLOCK-----\n"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"bad_uid_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgj8Et28Wy41nCNQ2UVwC2dpyeKou5x1Z51AtO17SyjGsKp3BheWxvYWTFASp7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMDhmYzEyZGRiYzViMmUzNTljMjM1MGQ5NDU3MDBiNjc2OWM5ZTJhOGJiOWM3NTY3OWQ0MGI0ZWQ3YjRiMjhjNmIwYSIsInVpZCI6ImRlYWRiZWVmMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwIiwidXNlcm5hbWUiOiJtYXgzMiJ9LCJ0eXBlIjoiZWxkZXN0IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0NTk1LCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjpudWxsLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjEsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAc4WxvRU1roBEB1HStIiE2d48q4cOLgHlYSu9sGawOYOc4nvDvrzPKq72jrTgjSJFOuwa9t0bQDpXMTD1/s/XA6hzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
            "payload_hash": "4ffdc8466a0ffa380da8c4ef6477cc80e3a80734eda97d148f659bc0410c7344",
            "sig_id": "84083329f6ebe6fc7157cb7be43a6ea7526db482416949d1f54eb5159eb8c4890f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"01208fc12ddbc5b2e359c2350d945700b6769c9e2a8bb9c75679d40b4ed7b4b28c6b0a\",\"uid\":\"deadbeef000000000000000000000000\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432144595,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "01208fc12ddbc5b2e359c2350d945700b6769c9e2a8bb9c75679d40b4ed7b4b28c6b0a",
            "ctime": 1432144595
        }
    ],
    "keys": [
        "01208fc12ddbc5b2e359c2350d945700b6769c9e2a8bb9c75679d40b4ed7b4b28c6b0a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"bad_username_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEguI0gTYv5jJi0OvFbTgAsh7xXonZn2UXGOIZlClWJN/oKp3BheWxvYWTFATF7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMGI4OGQyMDRkOGJmOThjOThiNDNhZjE1YjRlMDAyYzg3YmM1N2EyNzY2N2Q5NDVjNjM4ODY2NTBhNTU4OTM3ZmEwYSIsInVpZCI6Ijc0YzM4Y2Y3Y2ViOTQ3ZjU2MzIwNDVkOGNhNWQ0ODE5IiwidXNlcm5hbWUiOiJnYXJiYWdlX3VzZXIifSwidHlwZSI6ImVsZGVzdCIsInZlcnNpb24iOjF9LCJjdGltZSI6MTQzMjE0NDY0MSwiZXhwaXJlX2luIjoxMDAwMDAwMCwicHJldiI6bnVsbCwic2VxX3R5cGUiOjEsInNlcW5vIjoxLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQG0IAj1rTzCBW2H64Hh66oer8vzOAmiL70khO1/49o8hL/QjXRUm8/8aabGrHmESu7FC+TU6oyvO1sH++bK2LgSoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "a867b0e83da675b6076e3ba7e5f34e2020ea96836fba50e6e1a4092c9b653aac",
            "sig_id": "a9d87f33ed10c1295bedb5b7369b9af11bbb69fc38872b8a7a85cb660049dc480f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"0120b88d204d8bf98c98b43af15b4e002c87bc57a27667d945c63886650a558937fa0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"garbage_user\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432144641,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "0120b88d204d8bf98c98b43af15b4e002c87bc57a27667d945c63886650a558937fa0a",
            "ctime": 1432144641
        }
    ],
    "keys": [
        "0120b88d204d8bf98c98b43af15b4e002c87bc57a27667d945c63886650a558937fa0a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"example_revokes_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgKvldOpi5fFXv0twJsMnGBv0LXqp+NbWaV14zbexQTUkKp3BheWxvYWTFASp7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMDJhZjk1ZDNhOThiOTdjNTVlZmQyZGMwOWIwYzljNjA2ZmQwYjVlYWE3ZTM1YjU5YTU3NWUzMzZkZWM1MDRkNDkwYSIsInVpZCI6Ijc0YzM4Y2Y3Y2ViOTQ3ZjU2MzIwNDVkOGNhNWQ0ODE5IiwidXNlcm5hbWUiOiJtYXgzMiJ9LCJ0eXBlIjoiZWxkZXN0IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0NzAwLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjpudWxsLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjEsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAelPb6xpVpK5Kdk57aOoiA438uOVw8Ha6eHBGydFnWYLw12EWT02pKXQYnIvCVSQNOJIMeXXn99hPYLSS3cv2DKhzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
            "payload_hash": "69a9b17aca289091b869bddb00385857147515fb4c335d157fd0be86f0bb5e9c",
            "sig_id": "71e44e4c690d0d7ac6bcb244d9f72cf173bfae15e3fa7d4cb0ee2f24c3d6de9f0f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432144700,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a",
            "ctime": 1432144700
        },
        {
            "seqno": 2,
            "prev": "69a9b17aca289091b869bddb00385857147515fb4c335d157fd0be86f0bb5e9c",
            "sig": "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",
            "payload_hash": "29ef238a6283a937300a13b3e5542a92e3b9014380c0798979a9b27133d6c8ee",
            "sig_id": "77a2a70d5c8af0f1201027076504065e80ddd6d5ff2a622eb57e149dc8fa00470f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0120d20fe982e4e7ba1a08dd60dd527a027bbe0225d4fe89eef979020c5b9efedbfb0a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg0g/pguTnuhoI3WDdUnoCe74CJdT+ie75eQIMW57+2/sKp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwZDIwZmU5ODJlNGU3YmExYTA4ZGQ2MGRkNTI3YTAyN2JiZTAyMjVkNGZlODllZWY5NzkwMjBjNWI5ZWZlZGJmYjBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0ODAwLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiNjlhOWIxN2FjYTI4OTA5MWI4NjliZGRiMDAzODU4NTcxNDc1MTVmYjRjMzM1ZDE1N2ZkMGJlODZmMGJiNWU5YyIsInNlcV90eXBlIjoxLCJzZXFubyI6MiwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEBjqhTAjgisDTZInxciSQyXFdMBVRal98GzY1ZuVLpUOU5gyr7Hk8xrpRH7fPtXpWGRp+uZsigkVVmU/1mrLcAOqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432144800,\"expire_in\":10000000,\"prev\":\"69a9b17aca289091b869bddb00385857147515fb4c335d157fd0be86f0bb5e9c\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a",
            "ctime": 1432144800
        },
        {
            "seqno": 3,
            "prev": "29ef238a6283a937300a13b3e5542a92e3b9014380c0798979a9b27133d6c8ee",
            "sig": "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",
            "payload_hash": "c3deb587280bf27e7b7554c0304c5b58c4183d888306c65386d7a57287ba0a8d",
            "sig_id": "e65e67f0e1748ccfb4c1b05c84c87eb8311771a5d6bb908093edfb68a4e691180f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d20fe982e4e7ba1a08dd60dd527a027bbe0225d4fe89eef979020c5b9efedbfb0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"012031fe75cae16e364949f99b6fc4fe79fab0b07d1234dd50b9eafb0d731571bf6c0a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgMf51yuFuNklJ+ZtvxP55+rCwfRI03VC56vsNcxVxv2wKp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDIwZmU5ODJlNGU3YmExYTA4ZGQ2MGRkNTI3YTAyN2JiZTAyMjVkNGZlODllZWY5NzkwMjBjNWI5ZWZlZGJmYjBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwMzFmZTc1Y2FlMTZlMzY0OTQ5Zjk5YjZmYzRmZTc5ZmFiMGIwN2QxMjM0ZGQ1MGI5ZWFmYjBkNzMxNTcxYmY2YzBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0OTAwLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiMjllZjIzOGE2MjgzYTkzNzMwMGExM2IzZTU1NDJhOTJlM2I5MDE0MzgwYzA3OTg5NzlhOWIyNzEzM2Q2YzhlZSIsInNlcV90eXBlIjoxLCJzZXFubyI6MywidGFnIjoic2lnbmF0dXJlIn2jc2lnxEAcpFUiEVRD2yvqIa0t93Vynjdf8lSrpVXevhsHt06LeKsR+SDXa98EJo6Ym/fjpWxJj8ejsnKBLagqEtQfEjAMqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432144900,\"expire_in\":10000000,\"prev\":\"29ef238a6283a937300a13b3e5542a92e3b9014380c0798979a9b27133d6c8ee\",\"seq_type\":1,\"seqno\":3,\"tag\":\"signature\"}",
            "kid": "0120d20fe982e4e7ba1a08dd60dd527a027bbe0225d4fe89eef979020c5b9efedbfb0a",
            "ctime": 1432144900
        },
        {
            "seqno": 4,
            "prev": "c3deb587280bf27e7b7554c0304c5b58c4183d888306c65386d7a57287ba0a8d",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgMf51yuFuNklJ+ZtvxP55+rCwfRI03VC56vsNcxVxv2wKp3BheWxvYWTFBfl7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMzFmZTc1Y2FlMTZlMzY0OTQ5Zjk5YjZmYzRmZTc5ZmFiMGIwN2QxMjM0ZGQ1MGI5ZWFmYjBkNzMxNTcxYmY2YzBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwZmY1YWZmM2I1NjAyNmQ1MGY2MjI1ZmY1YzUzZmVkOWJiOWMyN2U1MjMwYWJlNjM2YzVlMDNiZTFjODQxZGQ3NjBhIiwicmV2ZXJzZV9zaWciOiJnNlJpYjJSNWhxaGtaWFJoWTJobFpNT3BhR0Z6YUY5MGVYQmxDcU5yWlhuRUl3RWcvMXIvTzFZQ2JWRDJJbC8xeFQvdG03bkNmbEl3cStZMnhlQTc0Y2hCM1hZS3AzQmhlV3h2WVdURkFpdDdJbUp2WkhraU9uc2lhMlY1SWpwN0ltVnNaR1Z6ZEY5cmFXUWlPaUl3TVRJd01tRm1PVFZrTTJFNU9HSTVOMk0xTldWbVpESmtZekE1WWpCak9XTTJNRFptWkRCaU5XVmhZVGRsTXpWaU5UbGhOVGMxWlRNek5tUmxZelV3TkdRME9UQmhJaXdpYUc5emRDSTZJbXRsZVdKaGMyVXVhVzhpTENKcmFXUWlPaUl3TVRJd016Rm1aVGMxWTJGbE1UWmxNelkwT1RRNVpqazVZalptWXpSbVpUYzVabUZpTUdJd04yUXhNak0wWkdRMU1HSTVaV0ZtWWpCa056TXhOVGN4WW1ZMll6QmhJaXdpZFdsa0lqb2lOelJqTXpoalpqZGpaV0k1TkRkbU5UWXpNakEwTldRNFkyRTFaRFE0TVRraUxDSjFjMlZ5Ym1GdFpTSTZJbTFoZURNeUluMHNJbk5wWW10bGVTSTZleUpyYVdRaU9pSXdNVEl3Wm1ZMVlXWm1NMkkxTmpBeU5tUTFNR1kyTWpJMVptWTFZelV6Wm1Wa09XSmlPV015TjJVMU1qTXdZV0psTmpNMll6VmxNRE5pWlRGak9EUXhaR1EzTmpCaElpd2ljbVYyWlhKelpWOXphV2NpT201MWJHeDlMQ0owZVhCbElqb2ljMmxpYTJWNUlpd2lkbVZ5YzJsdmJpSTZNWDBzSW1OMGFXMWxJam94TkRNeU1UUTFNREF3TENKbGVIQnBjbVZmYVc0aU9qRXdNREF3TURBd0xDSndjbVYySWpvaVl6TmtaV0kxT0RjeU9EQmlaakkzWlRkaU56VTFOR013TXpBMFl6VmlOVGhqTkRFNE0yUTRPRGd6TURaak5qVXpPRFprTjJFMU56STROMkpoTUdFNFpDSXNJbk5sY1Y5MGVYQmxJam94TENKelpYRnVieUk2TkN3aWRHRm5Jam9pYzJsbmJtRjBkWEpsSW4yamMybG54RUNzSzBoU3V6UVRKVS9STjNNNzYyRzJ0N2NrSk1DRGtVSzdqQTZyZzNwRnJveTBYRVp3TGhQQWlNODdtelVWVEJRSGU5VFVIbXJFb2Y5YWVoVTg2NElEcUhOcFoxOTBlWEJsSUtOMFlXZk5BZ0tuZG1WeWMybHZiZ0U9In0sInR5cGUiOiJzaWJrZXkiLCJ2ZXJzaW9uIjoxfSwiY3RpbWUiOjE0MzIxNDUwMDAsImV4cGlyZV9pbiI6MTAwMDAwMDAsInByZXYiOiJjM2RlYjU4NzI4MGJmMjdlN2I3NTU0YzAzMDRjNWI1OGM0MTgzZDg4ODMwNmM2NTM4NmQ3YTU3Mjg3YmEwYThkIiwic2VxX3R5cGUiOjEsInNlcW5vIjo0LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQHKgrodQ97nLF0vdV+Tvde8zto29t6XWwGbOeNvAri3JGVCCH5U0Pab/4RkZt5aSvv+rP2/AWlJqiA+ea/pikA2oc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "e89c2dc5a08333fe90d9af7dcb5ec251feb72f87daf44b281ffc9645bb49f025",
            "sig_id": "506bf48c368a84c9e9ab6fc22438359241ee12784c72f2e68c2afa1d3dd71aae0f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"012031fe75cae16e364949f99b6fc4fe79fab0b07d1234dd50b9eafb0d731571bf6c0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0120ff5aff3b56026d50f6225ff5c53fed9bb9c27e5230abe636c5e03be1c841dd760a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg/1r/O1YCbVD2Il/1xT/tm7nCflIwq+Y2xeA74chB3XYKp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMzFmZTc1Y2FlMTZlMzY0OTQ5Zjk5YjZmYzRmZTc5ZmFiMGIwN2QxMjM0ZGQ1MGI5ZWFmYjBkNzMxNTcxYmY2YzBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwZmY1YWZmM2I1NjAyNmQ1MGY2MjI1ZmY1YzUzZmVkOWJiOWMyN2U1MjMwYWJlNjM2YzVlMDNiZTFjODQxZGQ3NjBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ1MDAwLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiYzNkZWI1ODcyODBiZjI3ZTdiNzU1NGMwMzA0YzViNThjNDE4M2Q4ODgzMDZjNjUzODZkN2E1NzI4N2JhMGE4ZCIsInNlcV90eXBlIjoxLCJzZXFubyI6NCwidGFnIjoic2lnbmF0dXJlIn2jc2lnxECsK0hSuzQTJU/RN3M762G2t7ckJMCDkUK7jA6rg3pFroy0XEZwLhPAiM87mzUVTBQHe9TUHmrEof9aehU864IDqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432145000,\"expire_in\":10000000,\"prev\":\"c3deb587280bf27e7b7554c0304c5b58c4183d888306c65386d7a57287ba0a8d\",\"seq_type\":1,\"seqno\":4,\"tag\":\"signature\"}",
            "kid": "012031fe75cae16e364949f99b6fc4fe79fab0b07d1234dd50b9eafb0d731571bf6c0a",
            "ctime": 1432145000
        },
        {
            "seqno": 5,
            "prev": "e89c2dc5a08333fe90d9af7dcb5ec251feb72f87daf44b281ffc9645bb49f025",
            "sig": "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",
            "payload_hash": "33b022b996f1faaff00ad6eafdb9f5745ec0c0baa08588cab74017edeaf0f233",
            "sig_id": "b0d3a36fc061b317a260514eff8f5ac1b8ab1cfc60c54731e79e979f9e638b320f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120ff5aff3b56026d50f6225ff5c53fed9bb9c27e5230abe636c5e03be1c841dd760a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0120818b638074d0b15e766af7a61c08fb5fd2314cf0e2d0d5a104d8347d21e6f5ed0a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEggYtjgHTQsV52avemHAj7X9IxTPDi0NWhBNg0fSHm9e0Kp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZmY1YWZmM2I1NjAyNmQ1MGY2MjI1ZmY1YzUzZmVkOWJiOWMyN2U1MjMwYWJlNjM2YzVlMDNiZTFjODQxZGQ3NjBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwODE4YjYzODA3NGQwYjE1ZTc2NmFmN2E2MWMwOGZiNWZkMjMxNGNmMGUyZDBkNWExMDRkODM0N2QyMWU2ZjVlZDBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ1MTAwLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiZTg5YzJkYzVhMDgzMzNmZTkwZDlhZjdkY2I1ZWMyNTFmZWI3MmY4N2RhZjQ0YjI4MWZmYzk2NDViYjQ5ZjAyNSIsInNlcV90eXBlIjoxLCJzZXFubyI6NSwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEBN4dDSklpCQVTgSSbfFMPEGFO+9ZF0smlN0nHRbEMP1HsOgnuTgHaqtBtAI0fO3OhZIz7mKVAeAJDI24tDrR0MqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432145100,\"expire_in\":10000000,\"prev\":\"e89c2dc5a08333fe90d9af7dcb5ec251feb72f87daf44b281ffc9645bb49f025\",\"seq_type\":1,\"seqno\":5,\"tag\":\"signature\"}",
            "kid": "0120ff5aff3b56026d50f6225ff5c53fed9bb9c27e5230abe636c5e03be1c841dd760a",
            "ctime": 1432145100
        },
        {
            "seqno": 6,
            "prev": "33b022b996f1faaff00ad6eafdb9f5745ec0c0baa08588cab74017edeaf0f233",
            "sig": "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",
            "payload_hash": "81356413d54fb759fdb6f3bdbcf020c7208fba6f3635a81469f9581be2be2340",
            "sig_id": "f52a55c59023f5e188e85cf0933e0301b1143b462c7f361abcdc7989492b1bf00f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120818b638074d0b15e766af7a61c08fb5fd2314cf0e2d0d5a104d8347d21e6f5ed0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwODE4YjYzODA3NGQwYjE1ZTc2NmFmN2E2MWMwOGZiNWZkMjMxNGNmMGUyZDBkNWExMDRkODM0N2QyMWU2ZjVlZDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ1MjAwLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiMzNiMDIyYjk5NmYxZmFhZmYwMGFkNmVhZmRiOWY1NzQ1ZWMwYzBiYWEwODU4OGNhYjc0MDE3ZWRlYWYwZjIzMyIsInNlcV90eXBlIjoxLCJzZXFubyI6NiwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEC79ZsL2owjcjaW+hKyjeTX01rEbAeCJV1NhjPJI6PQBOgKLpQFbWsrxle82B+zfCy6BFhTlfmrELidnVWD+G4AqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432145200,\"expire_in\":10000000,\"prev\":\"33b022b996f1faaff00ad6eafdb9f5745ec0c0baa08588cab74017edeaf0f233\",\"seq_type\":1,\"seqno\":6,\"tag\":\"signature\"}",
            "kid": "0120818b638074d0b15e766af7a61c08fb5fd2314cf0e2d0d5a104d8347d21e6f5ed0a",
            "ctime": 1432145200
        },
        {
            "seqno": 7,
            "prev": "81356413d54fb759fdb6f3bdbcf020c7208fba6f3635a81469f9581be2be2340",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAoF7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInN1YmtleSI6eyJraWQiOiIwMTIxMzBiZjU3Y2MwMTY4ZjJhZGQzYTYxZGI1ZmNiOTNkMTlmN2MwZTQyM2EyNjZmZDVkNWE5OWQ1NmQ4MWMzNWM1NTBhIiwicGFyZW50X2tpZCI6IjAxMjBkNTY4NmNkMDM2YmIzMjUzY2MwMjY3NzU5MDM2NDg4MGFiYjk5NDU3NjU3OThjYWM1MGEyODViYWNkYTM2MjYwMGEiLCJyZXZlcnNlX3NpZyI6bnVsbH0sInR5cGUiOiJzdWJrZXkiLCJ2ZXJzaW9uIjoxfSwiY3RpbWUiOjE0MzIxNDUzMDAsImV4cGlyZV9pbiI6MTAwMDAwMDAsInByZXYiOiI4MTM1NjQxM2Q1NGZiNzU5ZmRiNmYzYmRiY2YwMjBjNzIwOGZiYTZmMzYzNWE4MTQ2OWY5NTgxYmUyYmUyMzQwIiwic2VxX3R5cGUiOjEsInNlcW5vIjo3LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQOhVHMGds7KBnZCG0IkbF7E8nEpHYhoFI2s54a69Cl8LikH6JdcCkAkFrHM7Oeu2gOFIHk5/w3XU7z51n+DKBw6oc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "ba371fdc2fc138718b7183204c92a2ae0eb1f7b3717f457a68ac7add0ff34bcb",
            "sig_id": "81824a207a2a3ecbc958074167346990f0f5e61fd4b0ebd22b7c7b1b0987b7920f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"subkey\":{\"kid\":\"012130bf57cc0168f2add3a61db5fcb93d19f7c0e423a266fd5d5a99d56d81c35c550a\",\"parent_kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"reverse_sig\":null},\"type\":\"subkey\",\"version\":1},\"ctime\":1432145300,\"expire_in\":10000000,\"prev\":\"81356413d54fb759fdb6f3bdbcf020c7208fba6f3635a81469f9581be2be2340\",\"seq_type\":1,\"seqno\":7,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432145300
        },
        {
            "seqno": 8,
            "prev": "ba371fdc2fc138718b7183204c92a2ae0eb1f7b3717f457a68ac7add0ff34bcb",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAoF7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInN1YmtleSI6eyJraWQiOiIwMTIxNTYwMWM0YzRiNDZlMTlmMzU2NTRhMWRlYmM4ZGMwZTNmNjgwMTNjMzUxN2ViODA5ZTVmMzY1NDE3MjZmZDMzNjBhIiwicGFyZW50X2tpZCI6IjAxMjBkNTY4NmNkMDM2YmIzMjUzY2MwMjY3NzU5MDM2NDg4MGFiYjk5NDU3NjU3OThjYWM1MGEyODViYWNkYTM2MjYwMGEiLCJyZXZlcnNlX3NpZyI6bnVsbH0sInR5cGUiOiJzdWJrZXkiLCJ2ZXJzaW9uIjoxfSwiY3RpbWUiOjE0MzIxNDU0MDAsImV4cGlyZV9pbiI6MTAwMDAwMDAsInByZXYiOiJiYTM3MWZkYzJmYzEzODcxOGI3MTgzMjA0YzkyYTJhZTBlYjFmN2IzNzE3ZjQ1N2E2OGFjN2FkZDBmZjM0YmNiIiwic2VxX3R5cGUiOjEsInNlcW5vIjo4LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQMUM7CpuUEq39bXVQE8Idxp7Ezolki56TrLsua40gmrzdj7bTFmj0YeEqRegn50dxqgAzoi/eqECouYHAdxb7Amoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "f28bad7570323729cbfbb3cd945e03b4ab9957450006fc339e297f3788add3c0",
            "sig_id": "a535a3972951c2077647d567949c73209672afd5c586fe7a0f4407d5894099000f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"subkey\":{\"kid\":\"01215601c4c4b46e19f35654a1debc8dc0e3f68013c3517eb809e5f36541726fd3360a\",\"parent_kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"reverse_sig\":null},\"type\":\"subkey\",\"version\":1},\"ctime\":1432145400,\"expire_in\":10000000,\"prev\":\"ba371fdc2fc138718b7183204c92a2ae0eb1f7b3717f457a68ac7add0ff34bcb\",\"seq_type\":1,\"seqno\":8,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432145400
        },
        {
            "seqno": 9,
            "prev": "f28bad7570323729cbfbb3cd945e03b4ab9957450006fc339e297f3788add3c0",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAoF7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInN1YmtleSI6eyJraWQiOiIwMTIxNWNiOTAyZTRiNmEwNzgzZjQ1NThlNGI0ODNmMDkxODUyZWI1ODVhZWNkZWMwMTkzNTZlYjgxM2FjNTA0MGQwMjBhIiwicGFyZW50X2tpZCI6IjAxMjBkNTY4NmNkMDM2YmIzMjUzY2MwMjY3NzU5MDM2NDg4MGFiYjk5NDU3NjU3OThjYWM1MGEyODViYWNkYTM2MjYwMGEiLCJyZXZlcnNlX3NpZyI6bnVsbH0sInR5cGUiOiJzdWJrZXkiLCJ2ZXJzaW9uIjoxfSwiY3RpbWUiOjE0MzIxNDU1MDAsImV4cGlyZV9pbiI6MTAwMDAwMDAsInByZXYiOiJmMjhiYWQ3NTcwMzIzNzI5Y2JmYmIzY2Q5NDVlMDNiNGFiOTk1NzQ1MDAwNmZjMzM5ZTI5N2YzNzg4YWRkM2MwIiwic2VxX3R5cGUiOjEsInNlcW5vIjo5LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQJEPySgBxHJeapOmXGKSEDP7H2EL4pB78+oM6j22Qa5CaEIzC1NPE/nGlpb3a1/iVxORoMWLohakYoHEgJVYOAaoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "d56391bc51ce01b9c6851edd75d7b5e8d8b3c7622c5d7600b71437075adaa232",
            "sig_id": "d5f5e75c809a6c1207d35791dd8361c68b5a7a0de9853d9ee647ba3f1620b9e00f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"subkey\":{\"kid\":\"01215cb902e4b6a0783f4558e4b483f091852eb585aecdec019356eb813ac5040d020a\",\"parent_kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"reverse_sig\":null},\"type\":\"subkey\",\"version\":1},\"ctime\":1432145500,\"expire_in\":10000000,\"prev\":\"f28bad7570323729cbfbb3cd945e03b4ab9957450006fc339e297f3788add3c0\",\"seq_type\":1,\"seqno\":9,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432145500
        },
        {
            "seqno": 10,
            "prev": "d56391bc51ce01b9c6851edd75d7b5e8d8b3c7622c5d7600b71437075adaa232",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAhl7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInJldm9rZSI6eyJraWQiOiIwMTIwZDIwZmU5ODJlNGU3YmExYTA4ZGQ2MGRkNTI3YTAyN2JiZTAyMjVkNGZlODllZWY5NzkwMjBjNWI5ZWZlZGJmYjBhIn0sInR5cGUiOiJyZXZva2UiLCJ2ZXJzaW9uIjoxfSwiY3RpbWUiOjE0MzIxNDU2MDAsImV4cGlyZV9pbiI6MTAwMDAwMDAsInByZXYiOiJkNTYzOTFiYzUxY2UwMWI5YzY4NTFlZGQ3NWQ3YjVlOGQ4YjNjNzYyMmM1ZDc2MDBiNzE0MzcwNzVhZGFhMjMyIiwic2VxX3R5cGUiOjEsInNlcW5vIjoxMCwidGFnIjoic2lnbmF0dXJlIn2jc2lnxECvZgjAO0Re95cU0cwiw2IMaomncIU6ir6lbQup6QM789tiDoed42Xf9+vKXB+lyHmNQwPOt6Qd/cYHafREYR8HqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=",
            "payload_hash": "601b3462742a14955c12017a7bbf5615cb3cad8bea04734d00dd61d492ad1952",
            "sig_id": "4d4a63b6914fc2750ff9e03f7f5f54ea62cfaf91c17d4311b0991fbedefbff760f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"kid\":\"0120d20fe982e4e7ba1a08dd60dd527a027bbe0225d4fe89eef979020c5b9efedbfb0a\"},\"type\":\"revoke\",\"version\":1},\"ctime\":1432145600,\"expire_in\":10000000,\"prev\":\"d56391bc51ce01b9c6851edd75d7b5e8d8b3c7622c5d7600b71437075adaa232\",\"seq_type\":1,\"seqno\":10,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432145600
        },
        {
            "seqno": 11,
            "prev": "601b3462742a14955c12017a7bbf5615cb3cad8bea04734d00dd61d492ad1952",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAhl7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInJldm9rZSI6eyJraWQiOiIwMTIwZDIwZmU5ODJlNGU3YmExYTA4ZGQ2MGRkNTI3YTAyN2JiZTAyMjVkNGZlODllZWY5NzkwMjBjNWI5ZWZlZGJmYjBhIn0sInR5cGUiOiJyZXZva2UiLCJ2ZXJzaW9uIjoxfSwiY3RpbWUiOjE0MzIxNDU3MDAsImV4cGlyZV9pbiI6MTAwMDAwMDAsInByZXYiOiI2MDFiMzQ2Mjc0MmExNDk1NWMxMjAxN2E3YmJmNTYxNWNiM2NhZDhiZWEwNDczNGQwMGRkNjFkNDkyYWQxOTUyIiwic2VxX3R5cGUiOjEsInNlcW5vIjoxMSwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEALFHOFpo85gpD2rMddo8E1U6ZxP3wRFOyj9eoxDGFxdtwwNbWCNb8bO0U1MsaiMFJCd/WOyg2NEHEcSR+5stsKqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=",
            "payload_hash": "dce3f0a140b4549038a21b43aec620d5362c1ee99b462cc2c8d7c8efbf4cf233",
            "sig_id": "be6da4ea06918c7f5f6255834c9e41da9b94c75745edbe64659fe885c3c432a10f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"kid\":\"0120d20fe982e4e7ba1a08dd60dd527a027bbe0225d4fe89eef979020c5b9efedbfb0a\"},\"type\":\"revoke\",\"version\":1},\"ctime\":1432145700,\"expire_in\":10000000,\"prev\":\"601b3462742a14955c12017a7bbf5615cb3cad8bea04734d00dd61d492ad1952\",\"seq_type\":1,\"seqno\":11,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432145700
        },
        {
            "seqno": 12,
            "prev": "dce3f0a140b4549038a21b43aec620d5362c1ee99b462cc2c8d7c8efbf4cf233",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAhx7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInJldm9rZSI6eyJraWRzIjpbIjAxMjAzMWZlNzVjYWUxNmUzNjQ5NDlmOTliNmZjNGZlNzlmYWIwYjA3ZDEyMzRkZDUwYjllYWZiMGQ3MzE1NzFiZjZjMGEiXX0sInR5cGUiOiJyZXZva2UiLCJ2ZXJzaW9uIjoxfSwiY3RpbWUiOjE0MzIxNDU4MDAsImV4cGlyZV9pbiI6MTAwMDAwMDAsInByZXYiOiJkY2UzZjBhMTQwYjQ1NDkwMzhhMjFiNDNhZWM2MjBkNTM2MmMxZWU5OWI0NjJjYzJjOGQ3YzhlZmJmNGNmMjMzIiwic2VxX3R5cGUiOjEsInNlcW5vIjoxMiwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEBw0vZNoLLGZ+77z68uKIPVtTY9L+LB/Le7Qy+8ViETnpz1SMN8TFzNVtoTZvtCp8lUVzYqhCCGCczNkT6BfJkJqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=",
            "payload_hash": "8689d6aa08e1a0818f955de82520fa736f61ced7eb640070c45660e127d63bda",
            "sig_id": "cb6682dcddf0e407635260320b21957c2df306f81813a5a7c8507936b296c5150f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"kids\":[\"012031fe75cae16e364949f99b6fc4fe79fab0b07d1234dd50b9eafb0d731571bf6c0a\"]},\"type\":\"revoke\",\"version\":1},\"ctime\":1432145800,\"expire_in\":10000000,\"prev\":\"dce3f0a140b4549038a21b43aec620d5362c1ee99b462cc2c8d7c8efbf4cf233\",\"seq_type\":1,\"seqno\":12,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432145800
        },
        {
            "seqno": 13,
            "prev": "8689d6aa08e1a0818f955de82520fa736f61ced7eb640070c45660e127d63bda",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAhh7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInJldm9rZSI6eyJzaWdfaWQiOiI1MDZiZjQ4YzM2OGE4NGM5ZTlhYjZmYzIyNDM4MzU5MjQxZWUxMjc4NGM3MmYyZTY4YzJhZmExZDNkZDcxYWFlMGYifSwidHlwZSI6InJldm9rZSIsInZlcnNpb24iOjF9LCJjdGltZSI6MTQzMjE0NTkwMCwiZXhwaXJlX2luIjoxMDAwMDAwMCwicHJldiI6Ijg2ODlkNmFhMDhlMWEwODE4Zjk1NWRlODI1MjBmYTczNmY2MWNlZDdlYjY0MDA3MGM0NTY2MGUxMjdkNjNiZGEiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjEzLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQAVTx2OScgQI1tbC07Cg1VfRW4Zow+p7PFX3g4HCck5ZBjf/2wsj1JYihQZgmDnm8IFVdmUEXNui5F2MVxvYSweoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "2a8200e9f3f59f2db54f404a51a727784571586282b7786755a17196ec2f722e",
            "sig_id": "33214539afe47c134f16b520728da821a8f0f0259e954805d11a0ea285309f700f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"sig_id\":\"506bf48c368a84c9e9ab6fc22438359241ee12784c72f2e68c2afa1d3dd71aae0f\"},\"type\":\"revoke\",\"version\":1},\"ctime\":1432145900,\"expire_in\":10000000,\"prev\":\"8689d6aa08e1a0818f955de82520fa736f61ced7eb640070c45660e127d63bda\",\"seq_type\":1,\"seqno\":13,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432145900
        },
        {
            "seqno": 14,
            "prev": "2a8200e9f3f59f2db54f404a51a727784571586282b7786755a17196ec2f722e",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAhh7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInJldm9rZSI6eyJzaWdfaWQiOiI1MDZiZjQ4YzM2OGE4NGM5ZTlhYjZmYzIyNDM4MzU5MjQxZWUxMjc4NGM3MmYyZTY4YzJhZmExZDNkZDcxYWFlMGYifSwidHlwZSI6InJldm9rZSIsInZlcnNpb24iOjF9LCJjdGltZSI6MTQzMjE0NjAwMCwiZXhwaXJlX2luIjoxMDAwMDAwMCwicHJldiI6IjJhODIwMGU5ZjNmNTlmMmRiNTRmNDA0YTUxYTcyNzc4NDU3MTU4NjI4MmI3Nzg2NzU1YTE3MTk2ZWMyZjcyMmUiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjE0LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQK2ipVbmehRpRKz8yoBDxNlxipjLFEfSFZpbz3HBB7pOCSQKcEQP9jDn4Das48z08CS6Bb59ZUD+hlYjI6joWQ+oc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "d0ff71f5026ff1a92b8baec3b06336df8d46f961077f39442fb019553b2b0a18",
            "sig_id": "67d4abca611b70d909cbfa3fcf3311320791a57309f54799a6e2ae49db858a9e0f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"sig_id\":\"506bf48c368a84c9e9ab6fc22438359241ee12784c72f2e68c2afa1d3dd71aae0f\"},\"type\":\"revoke\",\"version\":1},\"ctime\":1432146000,\"expire_in\":10000000,\"prev\":\"2a8200e9f3f59f2db54f404a51a727784571586282b7786755a17196ec2f722e\",\"seq_type\":1,\"seqno\":14,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432146000
        },
        {
            "seqno": 15,
            "prev": "d0ff71f5026ff1a92b8baec3b06336df8d46f961077f39442fb019553b2b0a18",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAmB7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInJldm9rZSI6eyJzaWdfaWRzIjpbIjc3YTJhNzBkNWM4YWYwZjEyMDEwMjcwNzY1MDQwNjVlODBkZGQ2ZDVmZjJhNjIyZWI1N2UxNDlkYzhmYTAwNDcwZiIsImIwZDNhMzZmYzA2MWIzMTdhMjYwNTE0ZWZmOGY1YWMxYjhhYjFjZmM2MGM1NDczMWU3OWU5NzlmOWU2MzhiMzIwZiJdfSwidHlwZSI6InJldm9rZSIsInZlcnNpb24iOjF9LCJjdGltZSI6MTQzMjE0NjEwMCwiZXhwaXJlX2luIjoxMDAwMDAwMCwicHJldiI6ImQwZmY3MWY1MDI2ZmYxYTkyYjhiYWVjM2IwNjMzNmRmOGQ0NmY5NjEwNzdmMzk0NDJmYjAxOTU1M2IyYjBhMTgiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjE1LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQH9HYHx8/DWAwcC5rRKcNQ+Bwk//XSc196Nr89E6dS1XrKYGurIL0+wqWvfRn1NxLT/zSvcwAXT/OoEu5PtHDguoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "159fd5164720001518c6a6d194c755ed6bca0eeedce5ac3f409f67fe629e655d",
            "sig_id": "577d4cc5654ced3ed4740b35eac9312701bb46177a68249bd44fec4a274669680f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"sig_ids\":[\"77a2a70d5c8af0f1201027076504065e80ddd6d5ff2a622eb57e149dc8fa00470f\",\"b0d3a36fc061b317a260514eff8f5ac1b8ab1cfc60c54731e79e979f9e638b320f\"]},\"type\":\"revoke\",\"version\":1},\"ctime\":1432146100,\"expire_in\":10000000,\"prev\":\"d0ff71f5026ff1a92b8baec3b06336df8d46f961077f39442fb019553b2b0a18\",\"seq_type\":1,\"seqno\":15,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432146100
        },
        {
            "seqno": 16,
            "prev": "159fd5164720001518c6a6d194c755ed6bca0eeedce5ac3f409f67fe629e655d",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgKvldOpi5fFXv0twJsMnGBv0LXqp+NbWaV14zbexQTUkKp3BheWxvYWTFAhl7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInJldm9rZSI6eyJraWQiOiIwMTIxMzBiZjU3Y2MwMTY4ZjJhZGQzYTYxZGI1ZmNiOTNkMTlmN2MwZTQyM2EyNjZmZDVkNWE5OWQ1NmQ4MWMzNWM1NTBhIn0sInR5cGUiOiJyZXZva2UiLCJ2ZXJzaW9uIjoxfSwiY3RpbWUiOjE0MzIxNDYyMDAsImV4cGlyZV9pbiI6MTAwMDAwMDAsInByZXYiOiIxNTlmZDUxNjQ3MjAwMDE1MThjNmE2ZDE5NGM3NTVlZDZiY2EwZWVlZGNlNWFjM2Y0MDlmNjdmZTYyOWU2NTVkIiwic2VxX3R5cGUiOjEsInNlcW5vIjoxNiwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEBRTqKeefnwZwlcVwjun1IsWKFhkP3xNdSliTIq/YVuuBOS4Iw66xtvShzdfwrqUR569G4mcU0AkLV320hWVVQGqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=",
            "payload_hash": "ba47575973c4e7f3faf6c6c3ee346a8f5d7a1c37f5c03bc0df7e15c5603774fd",
            "sig_id": "00ce703cbc1ec4576392064e378aaf5fd78983a934e4cfae97af3d6d8231743e0f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"kid\":\"012130bf57cc0168f2add3a61db5fcb93d19f7c0e423a266fd5d5a99d56d81c35c550a\"},\"type\":\"revoke\",\"version\":1},\"ctime\":1432146200,\"expire_in\":10000000,\"prev\":\"159fd5164720001518c6a6d194c755ed6bca0eeedce5ac3f409f67fe629e655d\",\"seq_type\":1,\"seqno\":16,\"tag\":\"signature\"}",
            "kid": "01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a",
            "ctime": 1432146200
        },
        {
            "seqno": 17,
            "prev": "ba47575973c4e7f3faf6c6c3ee346a8f5d7a1c37f5c03bc0df7e15c5603774fd",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgKvldOpi5fFXv0twJsMnGBv0LXqp+NbWaV14zbexQTUkKp3BheWxvYWTFAhh7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInJldm9rZSI6eyJzaWdfaWQiOiJhNTM1YTM5NzI5NTFjMjA3NzY0N2Q1Njc5NDljNzMyMDk2NzJhZmQ1YzU4NmZlN2EwZjQ0MDdkNTg5NDA5OTAwMGYifSwidHlwZSI6InJldm9rZSIsInZlcnNpb24iOjF9LCJjdGltZSI6MTQzMjE0NjMwMCwiZXhwaXJlX2luIjoxMDAwMDAwMCwicHJldiI6ImJhNDc1NzU5NzNjNGU3ZjNmYWY2YzZjM2VlMzQ2YThmNWQ3YTFjMzdmNWMwM2JjMGRmN2UxNWM1NjAzNzc0ZmQiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjE3LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQKXqUkdDSab896qNZKjMQ9sF3n3SbDS66JvGc9vFZ7/sBXNjTlt2SWRy40ukK3VIGE10D6QT8DBiEeSz5ffAmwaoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "ae4ed6b838a8395e9f44f7e1e5fbad442134de72691dcfc5dc04eb9b47da00f2",
            "sig_id": "ee77d80410f106b29f3b6294ae5de84308166188ba01e64d7ce3e1628044ce380f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"sig_id\":\"a535a3972951c2077647d567949c73209672afd5c586fe7a0f4407d5894099000f\"},\"type\":\"revoke\",\"version\":1},\"ctime\":1432146300,\"expire_in\":10000000,\"prev\":\"ba47575973c4e7f3faf6c6c3ee346a8f5d7a1c37f5c03bc0df7e15c5603774fd\",\"seq_type\":1,\"seqno\":17,\"tag\":\"signature\"}",
            "kid": "01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a",
            "ctime": 1432146300
        }
    ],
    "keys": [
        "01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a",
        "0120d20fe982e4e7ba1a08dd60dd527a027bbe0225d4fe89eef979020c5b9efedbfb0a",
        "012031fe75cae16e364949f99b6fc4fe79fab0b07d1234dd50b9eafb0d731571bf6c0a",
        "0120ff5aff3b56026d50f6225ff5c53fed9bb9c27e5230abe636c5e03be1c841dd760a",
        "0120818b638074d0b15e766af7a61c08fb5fd2314cf0e2d0d5a104d8347d21e6f5ed0a",
        "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
        "012130bf57cc0168f2add3a61db5fcb93d19f7c0e423a266fd5d5a99d56d81c35c550a",
        "01215601c4c4b46e19f35654a1debc8dc0e3f68013c3517eb809e5f36541726fd3360a",
        "01215cb902e4b6a0783f4558e4b483f091852eb585aecdec019356eb813ac5040d020a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"expired_key_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgDTTfqXtH50CdFdtB6s5oGZiTHBiSB8+PU6MnAafmgr4Kp3BheWxvYWTFASN7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMDBkMzRkZmE5N2I0N2U3NDA5ZDE1ZGI0MWVhY2U2ODE5OTg5MzFjMTg5MjA3Y2Y4ZjUzYTMyNzAxYTdlNjgyYmUwYSIsInVpZCI6Ijc0YzM4Y2Y3Y2ViOTQ3ZjU2MzIwNDVkOGNhNWQ0ODE5IiwidXNlcm5hbWUiOiJtYXgzMiJ9LCJ0eXBlIjoiZWxkZXN0IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0NzUwLCJleHBpcmVfaW4iOjEsInByZXYiOm51bGwsInNlcV90eXBlIjoxLCJzZXFubyI6MSwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEAi+mx5hSdvAHor5WhocvcOzK0M/JjQcB6/6BNzIm7DHXopkvVGK6qXrg6QvydIoZ0OW8lD23eWrKdIHaEITJUOqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=",
            "payload_hash": "cc2f840119fc2734c852c41c3274b69e53fd6771f15a31b2c06bb57c6a9f1752",
            "sig_id": "bdc3c02698b224ba0e27ed3c9782106ecc83ffa5c5a436e438b3de7dbc42554e0f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"01200d34dfa97b47e7409d15db41eace681998931c189207cf8f53a32701a7e682be0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432144750,\"expire_in\":1,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "01200d34dfa97b47e7409d15db41eace681998931c189207cf8f53a32701a7e682be0a",
            "ctime": 1432144750
        },
        {
            "seqno": 2,
            "prev": "cc2f840119fc2734c852c41c3274b69e53fd6771f15a31b2c06bb57c6a9f1752",
            "sig": "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",
            "payload_hash": "d945735df7741a001ceeee778c79fc4d3dfd55d2fa5d6b672b7bd198395ab746",
            "sig_id": "79579a28083e801d565a330e3c8a7d0cd76dda85f07603976ccb016f259cfadf0f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01200d34dfa97b47e7409d15db41eace681998931c189207cf8f53a32701a7e682be0a\",\"host\":\"keybase.io\",\"kid\":\"01200d34dfa97b47e7409d15db41eace681998931c189207cf8f53a32701a7e682be0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"012063557949acce3245ce74063632e9dd2a61ccc0350e9be1000107a8cee0d213180a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgY1V5SazOMkXOdAY2MundKmHMwDUOm+EAAQeozuDSExgKp3BheWxvYWTFAiR7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMGQzNGRmYTk3YjQ3ZTc0MDlkMTVkYjQxZWFjZTY4MTk5ODkzMWMxODkyMDdjZjhmNTNhMzI3MDFhN2U2ODJiZTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMGQzNGRmYTk3YjQ3ZTc0MDlkMTVkYjQxZWFjZTY4MTk5ODkzMWMxODkyMDdjZjhmNTNhMzI3MDFhN2U2ODJiZTBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwNjM1NTc5NDlhY2NlMzI0NWNlNzQwNjM2MzJlOWRkMmE2MWNjYzAzNTBlOWJlMTAwMDEwN2E4Y2VlMGQyMTMxODBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0ODUwLCJleHBpcmVfaW4iOjEsInByZXYiOiJjYzJmODQwMTE5ZmMyNzM0Yzg1MmM0MWMzMjc0YjY5ZTUzZmQ2NzcxZjE1YTMxYjJjMDZiYjU3YzZhOWYxNzUyIiwic2VxX3R5cGUiOjEsInNlcW5vIjoyLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQH7iM4vtfx9PQgmxX3eXfBeP4BhtA+CGudKoGMY0fEdj4LFPn98DONwV/Me+QNzXYfkMqZt8Rp/ES2Hzxr7qew2oc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432144850,\"expire_in\":1,\"prev\":\"cc2f840119fc2734c852c41c3274b69e53fd6771f15a31b2c06bb57c6a9f1752\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "01200d34dfa97b47e7409d15db41eace681998931c189207cf8f53a32701a7e682be0a",
            "ctime": 1432144850
        }
    ],
    "keys": [
        "01200d34dfa97b47e7409d15db41eace681998931c189207cf8f53a32701a7e682be0a",
        "012063557949acce3245ce74063632e9dd2a61ccc0350e9be1000107a8cee0d213180a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"mismatched_fingerprint_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nyMDvAnicbZG7axVBFMY3RoVcEQIRCSIkTL3I7Dzu7lzbFEljJ4KELPM4cx1ys7vZ\n3bskXPInWJgqpBC1SGFlbyPGB6KIiL2CGFDsRAULdXaDnVPMnO+c33wzh/Pk7HTQ\nm7u/vHDm/blP+1MvHz0dB1fTmzMTpHKzjQYTtA7dYV02hLIoXVajAbJYqEhThmWM\nrd+UtIrQPnBBCafCJoJwIjVBIbqRV+0Nb6NkBZdc7nNepM747H/49a6Ao4gaE2uq\niSSaMaIg0sJwAoqbGCjTShCB+8YyoRRlJAL/DR9LIfrWYgwmxtLbjTu7mGmaaBtr\nUILFlvcpwYybREtuWBKJFqygzOQGeHpDblGCdkJUbxethpEB30SIGigrl2doEPmi\nrl1LR4yStomEhQi2CldC6loCH68QFSU0aJCNR6MQVbCZHptGncjyLqrl0D9TuWEm\n63EJaOfx2slgrhecPnWiHUbQm5n9N6Lf94I/zR3Y273VvLq9+uvD5Ojaef79+dc3\n+dri4rfNg8vvmo9TweefR4cLP97OPpwvXj9gq8/SlbsrF64sHV4vvlyky/DiL4nZ\npE0=\n=S2XN\n-----END PGP MESSAGE-----\n",
            "payload_hash": "2fe005da2e66271bc5904ce234a0960c018e6a408fd4ee12aa1c03b31dc8cd81",
            "sig_id": "a486871a4705c9d1690125cd6ba2f6bd4ec74b129a466ad0dc0b78b0dd0bf3630f",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"f09b1c340a70f0a7bafb236e5932539f89252ac2\",\"host\":\"keybase.io\",\"key_id\":\"5932539f89252ac2\",\"kid\":\"0113dd7c3c2a2c442be1c9d52eb5d7e34cb92906df49bb3421e0a749ba996ff00ed70a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432325384,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "0113dd7c3c2a2c442be1c9d52eb5d7e34cb92906df49bb3421e0a749ba996ff00ed70a",
            "ctime": 1432325384
        },
        {
            "seqno": 2,
            "prev": "2fe005da2e66271bc5904ce234a0960c018e6a408fd4ee12aa1c03b31dc8cd81",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nyMQeAnicrZU7jNxEGMcvvEQORUIJRSo43O7ldjx+n3TFetdr7+7t6/Zpa8XJ9oz3\nZh9jr1+73iiK0lGQghZBD2noKRGpUlGlTMGjQUJQIKVB4N0EpUFUTOOZ+b75+6fP\n/n/z3Y3XDw5vPTI+eOfZez99du3JYy45GFw+PLzLOD7KmNO7zBzvH3iBcBRfzgli\nThnAshxCksu50IYuz0MHs66CBIgdAUmY411HgQoQkccrjsPxkMXAlvK5rSii5wGA\nkQRs5pjxCJ3iMAgJjXNZhG3kYJzH/3vkB6/8aHcih3PsCJ8QP9/LF5d7PEHhoMAp\nnqxAAdou3MX+X+5kLyfxLie7nuRiR+ElTxA5CHgBya4tIF5mlV1ihENqL3GevbQ3\nHGTuHTMRcV4W9RWVk2NxtsxhmUcOBwRWcnjPyWE9qHDAFt2cESIJCjwQPcS6HOR5\nWeJYx8OIhza7pwpxisMIX0Zkmsve2Q1V02uto47eOWpqvV5J1/a7EzrME4lPT48a\nLyp41A4w3aWl8AScsGBCy/5yiWl8enQVx0F0Wiy+qnXRDbMg9id0QrNmdVyixA2t\nRtHeXKjVpsvVUwN0a2TeI16/bDS0ZR9vk21rbAzNUaOXEVPXRwXPWEbJRTKhkpIp\n5fXCiEqrWqGUbKJhKYrPS8ONJtZ66kqqqiYczoOu5skV2gcztLDazWJnmXqe4lEi\nT2gsoqVfcMjaLAw3CepYBamTtlLHAHgcxWmr3NcqMeBsTWqoYB1LlVV8EQeaBIg9\naGLBnVA7nFcrQ7PWbJLInelTMoSNcZXMui1Dw7be5Ls1cxBAPWbNlWwJPVSmw0Ab\nymPa2xh1S5jQZneVVcLBlpSNorRp2uwaDbkLYzprCOv2RUFLN+p8IG1M1ySCvKir\nlVZQpl5S5NrtYFz2owldKDat+qC3MuV+ZdHX+gsftTPdWk2j8LyUog1bQSwbuZkm\nng/P54KeBF7jqr1O6qabS+gTCgSyuKDTvlkBfp8OSSlr81W9rmqFgin0ylq/Mt8a\nAShtbWtQHwlkNKqOhCHXO49sZTwrORPqdJVQzjqd6moeyvOqCsl6LvVICqzZVCub\ny0CfXnBquKyPLGE81oB0ZVpqMpiWrvhgZoGcoa/Wa0tH3/hRjbgNPeityKKvN4hq\njXF50NVHs9Z5go3maLREVlMvrJep5WRdWDF91R60pxPKtkBgnXtdn9PrVttrwUUG\nYC1rbzeFMN5WV3Sx5SOxSWm00YxMKYwsEo0NThQTRyW8l/+T6CrtKIIrYVGW0Jrv\nCMWkE86K5mws5pM0c7zWgi/2LWXrjQzOM4VuYatIYlJwlpLA4tmEBt3Uc/KPprqp\n312fnU3omR5H8YTu3aO1Kv/iqJ2z4yzYOf2lwY+Z9IXLmFM2D7ox2fUBludg3p54\nmT9m8CYgIb4ku4yXve2YCXIb5yLQwwAIyIZYFKHEOq6gAN7FkONtoIjABayMRZsH\nsod4jFlo26wLOIdjkSu7SGbz10d4dfkCid0vqM+cwpzSnu4hp9SOkxAz97796I2D\nW4cHb7352q71Hxxef/efC+H79w/+un/2FH86+OLt7f0r/5OvnuBHH5YePBVu/Hnz\n9h3m698W22sHzx88rv0SPr+5aT7Mvrx+6/ZJ138G//jm589Lv/7w9PcfP/4bjE0M\nZg==\n=4X5j\n-----END PGP MESSAGE-----\n",
            "payload_hash": "dffbf787f044211a12ac4f601ff477c351675634e8a6d6fc8aba468b7ced26f0",
            "sig_id": "deed996f790f6b4441a4dcd06f610e281a7c782111368c5da66806ade383347d0f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0113dd7c3c2a2c442be1c9d52eb5d7e34cb92906df49bb3421e0a749ba996ff00ed70a\",\"fingerprint\":\"deadbeef00000000000000000000000000000000\",\"host\":\"keybase.io\",\"key_id\":\"5932539f89252ac2\",\"kid\":\"0113dd7c3c2a2c442be1c9d52eb5d7e34cb92906df49bb3421e0a749ba996ff00ed70a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0113bdd73a83e84db30517b4fb3ddf2930a6c52e2d725406fd1c32448731bfed42a10a\",\"reverse_sig\":\"-----BEGIN PGP MESSAGE-----\\nVersion: Keybase OpenPGP v2.0.10\\nComment: https://keybase.io/crypto\\n\\nyMFXAnicrZK/axRBFMc3JvH0QIikSifTCHKEmTezuzNXHVYWKSyiYGGW+fHmsuRu\\n79y9CwlHsAqI+AuxsVAstLAVxE6ISBq7FBY2VkpQEf8DnT0jdlZOM/Pmvff9fni8\\nt6dmo+biwY+VxudPZ+7PvNvbH0eXstvNCTEDt03aE7KB0wt7DqtRtpE70iaUMe5c\\narkFDVYIMMiscjGgiV2KXFijQNHEeaGM4QIYUp2Gt1Yq8Z5SdCnVpEV8XnSxHJZ5\\nMQqyDrUziCH/7xMa1wdV3RHgjK5wOR+EvxBkU7xYcYi58lJBDNpCnfu/3OOpXCos\\nl9anFo0SqY8TDlTETlodOyGZqgsrLAvdx1Dd11scyE6LVLk5GupfKhOwuJYcpXCG\\n05ilRngTYD0oTnViAyO4FGJBE++Y5SCETDkzHp0AzaZUJW5iWWFW5V3SLsa9XjAb\\nbQ9r8yPPFqkr8kFB2iwk7Siv0ZjgECYmpGgR3BrmJWZ5XXE07hYZBuUgAh4pjZ0G\\nTBJImbGxosIicKGpSqilTGKiBZXeCUQGWjNLueHMWWmdZMG+wmvZbyQ2DYoBaUOg\\n1N0pZLfQo3GJZOfN2ly02IyOzx+rtzFqnlz4s6MnnsxEHy9+WZisXH366ubBi4fn\\ndhvP95c7e687dw4P5/uPrj/YjX6uPrvybfNl4/TZ9zfWH3fY5Q+z976u+bm751ej\\npQvfb+EvBcvoQw==\\n=Gtst\\n-----END PGP MESSAGE-----\\n\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432325484,\"expire_in\":10000000,\"prev\":\"2fe005da2e66271bc5904ce234a0960c018e6a408fd4ee12aa1c03b31dc8cd81\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "0113dd7c3c2a2c442be1c9d52eb5d7e34cb92906df49bb3421e0a749ba996ff00ed70a",
            "ctime": 1432325484
        }
    ],
    "keys": [
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nxm8EVV+NCBMFK4EEACIDAwSxdoYt7Gw2ZhzqSD7gxAANbFbNJFDslJ0hqd6blexK\nu9a2iSGdnWI7zz8o0IJj+mv+Vbf9IYJ4QWnDMIK8JHPP60BfqIkrjcSZn45LY8q9\nkkQlXP88yFjIrAqxca7GWJrNBW1heDMywpIEExMKABoFAlVfjQgCGwEDCwkHAxUK\nCAIeAQIXgAIZAQAKCRBZMlOfiSUqwmTwAYCzAH6p3Y1AhHLupXtmfoLn2LpbIDCf\nbgTRUwKDYTrM9/Si2w2mDwDUde893GQ8TXgBfROubbwMm7mko9kVG08/+Dgsq297\n+681BmBRwCWt7pJLTAfNG0cCrvSs12DePfHXYs5WBFVfjQgSCCqGSM49AwEHAgME\n+3JxQRgAllBCy62bYqxbi3ABKdvFtay8GhNDasLD9qZLdAebWJmHbuM1FebN4V8+\nXuBqNhDePGCE0vT1F8Sr5gMBCgnChwQYEwoADwUCVV+NCAUJDwmcAAIbDAAKCRBZ\nMlOfiSUqwjyeAYD6kut9MSuZozqQOfOS80WCqJe+mrf8ifCp1FvndALhnfo9F9qZ\nVja22hB3wWvxkOoBgLXVrYq6cmbRrmbqRSZ+4XpyoCNFQer6wsa6fvyTdMhcyBUi\nlTYjwzKKOxFopfemv85SBFVfjQgTCCqGSM49AwEHAgMECIKXe20+/fgR1Sv3iCGo\nGGH1tCUkdBsTy9ooXhtw52HA1ua0D94Uij2fOinADT5nRfSA601HUVLtIbl9kie2\nbMLAJwQYEwoADwUCVV+NCAUJDwmcAAIbIgBqCRBZMlOfiSUqwl8gBBkTCgAGBQJV\nX40IAAoJEKpIHwvfFOWaqoUBANYAO2WFA+1ykf52R2zDZWjktVd+bs8VlVVGU6ek\n8EV/AQDmKu+utOxSyXMRIe6sHCnMS7TcKHF1tpCuBv0dahwWOYqcAYD2Q2ntJ/W0\nI+rjw2T8uHbDIpMVERfg48E/YV2c2nJMAl6EPSbCZoCWs/PuXZfOQp0Bf0HqVZ3L\nBm/NhmTS3jvcpiFzJ/oM/RJf0mxVIV3ECTTvCF6nKaXCCTnKe3seNADTVA==\n=gaPN\n-----END PGP PUBLIC KEY BLOCK-----\n",
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nxm8EVV+NCRMFK4EEACIDAwRX2ASuVhbQWsemulbD6U1PRTZV/tAUb7P4IeRzklSs\ntWk9coRRpepboFjTWiAZjOz/rpVLd+DgUfEcUt8yH7jb3K8meEbiVGZLgeBL6hPY\nXpm73qI1saVTCA31AXSdJZnNBW1heDMywpIEExMKABoFAlVfjQkCGwEDCwkHAxUK\nCAIeAQIXgAIZAQAKCRCd/xzIsqmWA2Q6AYC5ayGbUfWSOhd9OiS83Dtf+91Ege7z\nECVUym9+RHIUS7mQaOV8lQu0zhugvsqlmTwBf2eQMX+xvMtM5YdDI33lZLqibTJS\nx36yabMk+l1C3jdNQOKYL2iJU3dJMjlaTiWE8M5WBFVfjQkSCCqGSM49AwEHAgME\nyz7k0POd3bNOpuz4Jc2djhmtgIlyCHaeUgfAnrP7QqlC8Q8WzzmBFz+od1EM5tEQ\nRB07+YGPFL1Kbq/DO8OJAgMBCgnChwQYEwoADwUCVV+NCQUJDwmcAAIbDAAKCRCd\n/xzIsqmWAzVQAYC33nryFGGQnMSRQjqm3K9xe/l8/Zz+UjWBBviEh96ccsngJGLw\nKadsWyYuOuos2y4BgO1ChHwRNiSGXrpe1bBD1HJCvPxOhqQvNaqyjK7BYdPW6HhR\nnzGp0HlAnNLb7lj2/c5SBFVfjQkTCCqGSM49AwEHAgMEMVlURn6Qe9pHS2I5pFII\nLru49zxqsoOws8+oz6ojqWxJUZeb1lEHrcb1kvNpTL0gs/TAQMJY+LFwU7E/PeEc\nO8LAJwQYEwoADwUCVV+NCQUJDwmcAAIbIgBqCRCd/xzIsqmWA18gBBkTCgAGBQJV\nX40JAAoJENLwTAfm5SCSIh8BAM7UnJtmN0gMpEfx2Z7BS0CjKm+dGkuMz4mCW9U2\nsrDHAP4ukoSjPtSs//isCS+474UOfRu6MGubTWB4qFTSzOl9Aj8SAX9q07KDj3Vn\neSOQEaqzB8WFDK/Ye2+Vz7WUgzXpGt5ZE+D8buy92f2pN1bgfnL9s2UBf3IrS3pC\nrqgRPXzqJSbJk8Pg3PRtEF19LLII2QMsWaIypTOtGtNyR/mNmQTgqnzikw==\n=euv7\n-----END PGP PUBLIC KEY BLOCK-----\n"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"mismatched_kid_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nyMDzAnicbZG/axRBFMdXTYQcCJIUYulAmrDovJ3ZmdkDBcEqlSBWCR7z8xxy2T13\n946LRxrBJhaCKHZ2goKFnWghIRoQhKCdWPoHSBAr8cTZDXZOM/P9vs/7Pob3/tSJ\nqLN0dmdl58PB5d1jH3f3R9F1M340RaowW6g7RRu2vZzP+7Yclj6vUReRTEgADkwb\nJYAZcFoYh9PEcDBOOCZERiEFFKObRdV0hBglK3veF8ELoudNcP/Db7QFDEAowVkm\nWaozxbFzWhPlpEsxYGlkwhkmMhFGc0gsKCO1YKAoU6CNTZmkDMsQN2rjONVEaMe1\nVRnlLmUkwTQ1QsvUUAFZA1a2zOWmDfSmnJAEbceo3ho22g6MDZ+I0diWlS9y1IVQ\n1LVvaKAkYZQT4DGyk6Evbc83BD46MRqWdoy6+WgwiFFlb/WOQqEVedG+atkPYyrf\nz2U9Ki3a3rsxFy11opPzx5tlRJ2F0/9WdFdHs8X+2qs3c7+ffn229unOjxfn5s88\nXP7y9sq9Wf5y9fG7B4fRnwuX9hbu82+/ru5f/Hl7Oll/ffj9M3++fO3J4mx15YCs\n/wUBuaVy\n=jTqg\n-----END PGP MESSAGE-----\n",
            "payload_hash": "fb7a58e58f6acde118349495d44438bbd8577e88795ed81c9149fda3b4628d76",
            "sig_id": "d92e49fb62ac5009f53989807cc82113fc5a806a8b717097f0ca8be8822553840f",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"398a11716cdb816d1fc8df052d71df8f68894151\",\"host\":\"keybase.io\",\"key_id\":\"2d71df8f68894151\",\"kid\":\"011343099a65c9b70ffcc3bfaf5010ada27603a28dc712e1bdac861b46b1cde56a460a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432647317,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "011343099a65c9b70ffcc3bfaf5010ada27603a28dc712e1bdac861b46b1cde56a460a",
            "ctime": 1432647317
        },
        {
            "seqno": 2,
            "prev": "fb7a58e58f6acde118349495d44438bbd8577e88795ed81c9149fda3b4628d76",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nyMQ2AnicpVVLiOREAJ31iyuCuHpYT2Mf3XGTSiqpZGTB/mZ60p/pTv9tHJJUJZ3u\ndOXb+fSy6E1YFBXBFU+CiwcvK3jZi7CIiih49ORlj4JHr6LpmZW97M26FFX16uXx\nkvfy/XOP7128dPnmqzd/+rV878IvP/CbvSFOPrleMjyclw6vl1bkbCIuJlF8unJw\n6bDEAsBDnpVlXRRM2UCsZZkmb1i6JbCA1bHOIZHldU7CJgIcAQbWTUkEBhQNYGIi\niDoUWb10ULIcapPQDx0aF7S8LOkAICCa2JCAiIFlSthiBQ4jgC3JEiVJhkAAxcWF\nF+1uFOIMPSJXHa/YKxanZ/IegT/XjYmODUIs9n+Ngm5zRoegyUumhUxiyBBZgshz\nLBSwZOoChhKQd8CIhFRfkwK91jOeK904KEWO8cDUh25ignSe42S2MKBwjLU4CbAm\nhJgQicMcx/OmgCAwBGLwQBQ41oDAQsjgrMIlxJ+7GZKEhBE5jRy7oH1tNyp1pdnZ\nP1FO9tt1TSsr9bPdOR0VQMejh/vquYP7XZ/QHSzhrrJXATunVW+9JjQ+3F/EsR8d\nMsxDrxkzzP3Ym9M5zduNaZk6ZjhTkQ76FaU9SuquPGtMB7DB5nJ+tN3SaOg0PBW7\nxGXXBIs5MTZCjPBKGXXmtBDoq2O2rqbrJtXKHtey+71RUx2vbco1nOZivBqaRNhk\nbUPTp/qsOznpGinaipusCtylPqekGjGtyRWIyLQ7CSGZxZaUhWwWLVmmF7DZ2OYH\n6qQvB2UjBjhfNqJm2W0oUdw/MnUtzuZ02Rb6CsuvlFYtW6ycSqAdqd10tbJPJora\nU8bVXtKt5LBdlx372PMFtue6iZS3eobqjhercuGWeSwEbAU6dSQ4NZrVaTYcao6r\nTCrNBsdwQeIrdthdVxdWPon9lsNMhyhjREZKZdrxj+dUCBb50PDtzqzbr7JeCgEU\n0i0YeEq8aJhhNWyOV5Nl13DVanwc+DkBg2CSe5avKnX9OFXnVGuvYN2eTNtkVhu2\ne4NeQJcr78hNR80eVWqtZt33tWEaa4NgqUYdRVWTXpg0u/piUjOWpDGntaEhd5PU\ni6Ab9DEMh+VE8/rJqtP3Us3ut+RQPU7XU6bFtlGYjWDHDzwq5APNGGhmZQTnVI/1\nVdcuD2ez0JcbSyVotcZqLrjDuh8chWnVh0e+WAYdRh+p/nCZjUaLZCtU7I2zEIP6\nUJtTn7jt5mDTUHrblt2arjtQcDXIR9nJkXBliddLc7S1GORtvSsCOlmpXcb1CIMk\nO2eMFqsP5nSRYprEJ6G7RWS8TE2eAcIVk5/I8jpxmKOUE7qN1FJxhjYDeWIhylhD\nJh7LEkGyi7vFV03pjKMuL9SmvIZw89qcXgvrNWVOz9JT79QekahdsuPc3yX9QcAP\nSsl5ykqHoDg0Y2fXAwDynAiLLKODEsl8JySnzg7xoFwOSn4R44LEMpAuSEQoakwv\nOhMAiYcylItmgZCXDANLAkJEkpAsECwBUwZQtrDOFyVb9C4Si8dHJDg9lwTOFtQr\nHXKFSt0+E2kXb2sTktKN7956Yu/Sxb2nnnxsV/17F595/r8fwv1be//El9MPnz3w\nPrj957feK9Jjb37duf2p+nH88vv3P7rzXv70nQt7P/99943fX/greOnLFy8tfvzi\nt9z9fPL6Z2/fu7X/zTt3v3r3j38BlbAMxw==\n=U60b\n-----END PGP MESSAGE-----\n",
            "payload_hash": "b0c1a6a4f8af46483e1c249ef49a727c991d601537cee9e799d414357f1edddf",
            "sig_id": "3edc748f9de94269229d930ca1090c78caf3ade6148bfd1571290eb6c05284860f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"011343099a65c9b70ffcc3bfaf5010ada27603a28dc712e1bdac861b46b1cde56a460a\",\"fingerprint\":\"398a11716cdb816d1fc8df052d71df8f68894151\",\"host\":\"keybase.io\",\"key_id\":\"2d71df8f68894151\",\"kid\":\"deadbeef00000000000000000000000000000000000000000000000000000000000000\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0113de7a32290a1112e0f2810c44dee82d2233c5741b5eb316520b41f77b2fd1f7360a\",\"reverse_sig\":\"-----BEGIN PGP MESSAGE-----\\nVersion: Keybase OpenPGP v2.0.10\\nComment: https://keybase.io/crypto\\n\\nyMFYAnicrZK7a1RBGMVvEl9ZFYT4F0y9yHzznsUiFoKdlel0med6yebu5t7dkGVN\\nEGIpKW0EKwmInSAo2LgRQVIKWmgn2FiIhWkUce5uxMbSaYaZOXPObw7z6uxC1lja\\neCs/LX+47eYOXr4eZtf8xr0xsj0/Qq0xWg3TKXR9qAbt1dyjFsIAlFGstRHcaStx\\njM5RG03kGLDxhkiBqSHKOwkkgPXGKQGWCQvOBy4ME9igJop50Qllv8yLQbKlWhkA\\nCcJ5q0B4iE75iDnxEnxUUSilGXBIF2/2qvpGgrOmChfyXtpLi/YU7x/6/8w9nNpJ\\n5qhyUbpgNZORC0ow4145wz1ToGthFcrCrIWkXjOblKCtJqpye1TqXyofpKGEaJwK\\nSMk4EgXYMeZDUMQTQqnjkoHlwVIQnGDLIEppSUwtSTqjKsNGKKvQrvIOahXDbjeF\\nDUb9Ovwos4lqRd4rUAvSoRvkNRowSgRL9rKJwmY/L0M7rxV4Npqon5yTSbTScBV4\\natakOgAUZZrp9FjGqLLWKy5lUEpqHrwCp4Hp6A1N/aVKpUjxVVhvz5Bguih6qEUS\\npelMITuFGQzLgLYmN45lS43sxPH5+jdmjcVzf/7ozo+57PkKO/loe/78gy/bL0aT\\nhwdnvtPrlz7eWjwc3/15+c3X99mvi/Hw25OFwfKdx7uT9Xf7n/fU/tW98e79ldOn\\nnnZ2nl35DY3S7dI=\\n=rEDG\\n-----END PGP MESSAGE-----\\n\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432647417,\"expire_in\":10000000,\"prev\":\"fb7a58e58f6acde118349495d44438bbd8577e88795ed81c9149fda3b4628d76\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "011343099a65c9b70ffcc3bfaf5010ada27603a28dc712e1bdac861b46b1cde56a460a",
            "ctime": 1432647417
        }
    ],
    "keys": [
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nxm8EVWR2lRMFK4EEACIDAwTHLrzQ9a9ejgFgBkMeCo3GZxk+st5RQT7ogWdd/ReZ\nx+a0RSzVQoJ1YWxMOH1gXe2g3Ez89shdUwdRhAIqiHu+3Dpju9gLULaggupPHTcH\nDubfmoXLs9XZlWdfwnebobDNBW1heDMywpIEExMKABoFAlVkdpUCGwEDCwkHAxUK\nCAIeAQIXgAIZAQAKCRAtcd+PaIlBUYbrAYDbujynJ196wGcCxBatpJkEvWfl33HT\n7h7LflVFH7gRpVBWy2qJxhwpMJXySlhKFfcBf06AZbOg3B5UaNh7e+xp6J/8Ey6j\nN1zYxdnrDFqjIyPQO1nLLkQx9vDVT6iMwuki+s5WBFVkdpUSCCqGSM49AwEHAgME\nuJXDKpNHbnnKFnjHZYBo7/j945n6YC/w0uEmEspqY1pAGhDBr83onBBiTJrDMM88\nrzA2voGwkdsFTlFDtps2BQMBCgnChwQYEwoADwUCVWR2lQUJDwmcAAIbDAAKCRAt\ncd+PaIlBUYC5AXsFr9Q8NeRhUqhu5QQ1q2ZRHc6noVTCYx5rljmHACiqFfM6MjFy\nGjfXfnm1HSe3cfUBgOSj+1sDKahuZg/F0NC4fPSNFdlDKbP2awmCX8fxBOePN+zm\nZgfA+9Fn3PINLiUkqM5SBFVkdpUTCCqGSM49AwEHAgMEkVmgczonctdljbaeqhVK\n24Y0PWoqn5inr6/kIG2jez1UW1Io8F4Nyc7NZl+UPHE1+hKht2Y0438Jg89cE9aY\noMLAJwQYEwoADwUCVWR2lQUJDwmcAAIbIgBqCRAtcd+PaIlBUV8gBBkTCgAGBQJV\nZHaVAAoJEBmKKorIzkHAcm8A/3rb4Kbt4BHk93U4AaNukEZWS+kPqKGJGEJBHsas\ni1IeAP4qFQIvPcUFgCBwM0aOvQoIKY3u0XqlfPAmY6YJo/mf/w9BAYD0afRCV6U3\nCPUZshdS5jO5/3Y23QoSfI2ibq6UWgAmKXo7rgLwsSJpFIriCIN9vrwBfjneWxQ9\nOb/HHvXW0jBP3qMZhfLQdQPM+FYBYIpoES8UNblYyNkPXVXyojuNn0pKHg==\n=4ZY+\n-----END PGP PUBLIC KEY BLOCK-----\n",
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nxm8EVWR2lRMFK4EEACIDAwQNQ19PK1/SLtX48z0hz3dPFw7m8wgGQHKZtr9+cSSh\n93/8rC/WzmAwwA42iswhS/n0QCIe00JsKvku5MoU1wtJa0klQfKOpf2JMgx56Ei8\nWoEfWkVQBYjbZo2TwHzwIUjNBW1heDMywpIEExMKABoFAlVkdpUCGwEDCwkHAxUK\nCAIeAQIXgAIZAQAKCRD6HZfQ/y2R/RY+AX0VG+yecb/FPERfFpID8PgDwisbDYYV\nu9n7rxWXweO4r/XLo8KQR9FmTwlNgua5oX4Bf3do5Lro6gTWsqDvxcYo/6BDCC6s\nU0GlFSYX9lo+mJNY5Fhj5BM/YEFznEMVrmEFAc5WBFVkdpUSCCqGSM49AwEHAgME\nICRjm0O9+5c5ylqcWEhpK42kJOvN2WzMsuAXPve8YXwXZWoI4mBKWlhpnxMD2i2U\nNDj9FgbsOWxAYTwBNl8/wQMBCgnChwQYEwoADwUCVWR2lQUJDwmcAAIbDAAKCRD6\nHZfQ/y2R/QaIAYCEzFPDc3EWSjl3cudokf4K4/dd9nODpNqB/PaGex2tQv4jaetr\nRf/HcTCqICyuBUQBfAvDoRiShMCUqIg+/Uj279VbdDw5ncGpMF8qeM8CxyTartVL\nwTPEsN3DC9puTb59/s5SBFVkdpUTCCqGSM49AwEHAgMEzpgc1Zzf4tsT3K8Fp6ld\n6drFm6y/vnUDbRW1OiHgn3xHPYjOHtb5SKPTg7Hcaa9JinnJcGVwPpkRYm7ixRVE\nFMLAJwQYEwoADwUCVWR2lQUJDwmcAAIbIgBqCRD6HZfQ/y2R/V8gBBkTCgAGBQJV\nZHaVAAoJEHbTN+NA24BjqYkA+gNxwnl8x2yCNjHtKT7vhLBZYvL+qXyUYpOVWjaH\nbe6yAP0QZNQTWZxt5ArLWBpjE7PrwjG7Nu2ljvGA7Z0K7CBpj1w3AYDP9Cuq7xnQ\nG4PL8NfiRvgyvaHiUFYuRQA2z97u3zdBsZzN53dj85Q8s7i5niUfSdMBgL5Fu0H8\nJGBMySeMF03CLsEOS9saWXVqTqY/kcc09A91bhQ0shxPBbpXZgHisQEIZA==\n=h1vu\n-----END PGP PUBLIC KEY BLOCK-----\n"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"missing_kid_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nyMD3AnicbVG/SxxBGF01CjmIBI4UtgMhCJsws7O7M7uNiIUQCSGQNCF4zM58ew6e\nu+fu3qmIlWCVJuA/IEKaQIrYpMkPJAkRhSvENNpYaGGjpSBiZlfS5WPgm/e9N+8x\nfD/uDVi1er33aPfB2f7vvt3vPzvWK7U2sYyiVC2hcBnNQtVinTQha2c6KVCIVOzG\nipHYA0yxkswNvJgTRxDsE8kw9SXnhsXIRjNpXr4wNpHI4YlOzcyAhlZm+h/9bEVg\nQmiEAYQUWAQKRIyF9CNg4GHGmOIBMB74Jp3ziDhY+R6JCI+oIlx5rlTmYGHsOpUd\ncyXlMmYSosBlsedTB7ue4lJ4yuUkKIU5ZImYA6OeE4vUQSs2KpbaJYaWAvMJG3Uh\ny3WaoJAYUha6VBOXOiacUmwjWGzrDBq6VODbslE7gy4Kk06rZaMc5hu3pqQCSVrd\nCtE0MbluJqLoZIBWtqfvWPWaNTTYXy7Dqt29/29FWwt9Vu/l2/drf76NfhnfOnm9\n0Qpf1I9XNz+PPR7onY5fjUx+sm4mrw/ere5tXv768PThxZF4NrEz1X84vHO+3fv6\ncf3g+Zu/KrimzA==\n=9oxM\n-----END PGP MESSAGE-----\n",
            "payload_hash": "a9e948e5353741bd48065c251c7b8224ff52d8de16f4f824b7548e074dd99d67",
            "sig_id": "3d80007ad5e64c9367fe566b03c1918bff04d522537787d7385fdc4bed274a740f",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"df4fd71f5e030dc7495f812a1061c7036c881f50\",\"host\":\"keybase.io\",\"key_id\":\"1061c7036c881f50\",\"kid\":\"0113b0eeaca0a9deaf0ac6be7e50777d89e789603088b120d651b18b3d18d54cd4cd0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432651330,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "0113b0eeaca0a9deaf0ac6be7e50777d89e789603088b120d651b18b3d18d54cd4cd0a",
            "ctime": 1432651330
        },
        {
            "seqno": 2,
            "prev": "a9e948e5353741bd48065c251c7b8224ff52d8de16f4f824b7548e074dd99d67",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nyMQmAnicrVXPq+NEHH+76oIrorAsiAd5xIOHPreZ/JxURZo27Ut/JOmP9LUl7CPJ\nzLRpm0mapH3teyze9iYK6lFBlD0J4tWjrigsugcP3vwDPHjXBTHtW9nLHh1Cwsx8\nvp/55Aufz9x/8ZmD6zduPHzjwc0/fv3pyoMfiqsDG92tXDBehLZM6YKZ4/0HLxBO\ns9N5gJgSwwLAeyzGru+yroKwS1jXlzwsY5GVZRlBBctQkViehdADHIskEXgAejwC\nEImCj/KHdZkjhgR0gpM4CWiW0yIiECQDIuK8EvmyoIgEAs4FrAR8meUlH8J8l80L\np1G6q8jFeW6KbwVRvpZPTvfynoL/n3Wv9nSy4PPQJ7KPPUWQiSjxHCuICPquiAQI\nlB0wxQl1Q5yjQ3fDc8ydIyYNvMdNfaKKk0SMZdHDPIKQ5xUZChwBvIgwkmRPFjgB\nQUJ4EctE4DzPZbEn5ScCUYKeh1x3ryrBa5yk+DQNJjntm7uhanXdOLTq1mFb6/XK\ndW2/6tBBDgwiWjpsXnbw0Iwx3cHW3C32FmAdWonCENOsdDjNsjgtFYtPel30k22c\nRQ516LZdOynTwE/GDcPddDb1NqJgMEa23gmkqs2PvJrY58aqRSZqECXDFl2I7YmM\n6ADQ8TxrOVRC9tquHWvetozVUSEYTOsw6BFVXwWxvpiBQQa3rBANZ4LbmZv9omUp\nZB6Sc65dn5s8mTv0DLPxat3nfJMshALpi81l0R9qBufVSJewICv3M1VrW4M+GG1n\n/FJnz/R0MyubumFIqaY51Gts2UhTJ/G4P2z2AtRLt2cgHnfkpj1ZhLOWZdSwb6LZ\n5swDdjtoVdtxI+wi0DfTahVIXYfqFbWOl0hoRnoI1HIaDMxRz0XAatsbaCwFN9AD\nWrUHtM0f+z24UgMeW8V+MRyu3Xhh+Q5tY67ih2o/iZqdWV1frY/HddwUG9xkkjTL\n6flkOcFhd1Bnh5E18CgiSq84qHWa5YqdtpsQOHQk0R4te5Vp2V8vBie+v1Sqo3LD\nFoyuO4HHTUkLNGCMpRFYLrXYnSZ1tSN26bKmWNN2Ixk4dCladWIUzqebuGYFYofC\nbnVQ8I36qm6QzWZTzVkaFWU6r/rNtYEkV9XPN9tOOB13E007O3PoZhQaCHZ7ZiUJ\nRiAzmh1FtZtQ0dR50V4Yq8Vouk1aM97tJmsNs8mqKo8621YWTYOZFXG2QxthOk1h\ny2Y3rU4htmoduPANMUQmUu2Roit41bBmvtiex1DY9mRiFq1I0SSl0IgLRUue5hrG\nWOZdrm8WfHlYgGuvJom+tOyThqz4J+uthIayIaf8mmOVoneiwH5ZNs6VIrG6qxEh\nFYcqq+35DBVWyGSrDn3HjwLRoXvnaEb1KW7auTrbxjuXPzb3EbO+dBhTAvmmnwW7\nDADCzuj5mz1i8CYOEnwa7BDs5Thi4tzCOYmrYEWAWORFXhaAl4cJm/8CJ+ap5kGO\nEwgROQQRBlKemJATPFnM4awsIKQoeWDkx6d4eXopCewnNGJKXK7SnexFTqibrRLM\n3Pnu9rMHN64fXHvu6i72D64///J/l8E988rBn5//mIHvP7390kc3H/7y4Tfv/nzR\ne+WD1y/mtX/+voY+1l+7cvD+l5y5uKff/eToEf7ts6+4L+4/0vDbb7337QtXfz/6\n6+tX/wU94wjD\n=mdjZ\n-----END PGP MESSAGE-----\n",
            "payload_hash": "a6eb21956740c5d06a82248710b0243ed7f4a9d83a0ab0ad345971cb0532d3f6",
            "sig_id": "03a631ece380553b7a422c0d53ff9492a0a036594580304fbbea14af928761c10f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0113b0eeaca0a9deaf0ac6be7e50777d89e789603088b120d651b18b3d18d54cd4cd0a\",\"fingerprint\":\"df4fd71f5e030dc7495f812a1061c7036c881f50\",\"host\":\"keybase.io\",\"key_id\":\"1061c7036c881f50\",\"kid\":\"0113b0eeaca0a9deaf0ac6be7e50777d89e789603088b120d651b18b3d18d54cd4cd0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0113265ee75be3d883397842f135ded67b7424d8ff35e7f42bba0eb67f51568bbdaa0a\",\"reverse_sig\":\"-----BEGIN PGP MESSAGE-----\\nVersion: Keybase OpenPGP v2.0.10\\nComment: https://keybase.io/crypto\\n\\nyMFWAnicrZJNaxQxGMdn1VZdUIQi6DU3YbF5T2ZBPfgBiorXLnl5Mg7dnV1nZktL\\n6dUvUFHEbyAeBY+iVhG8iSfBIuipIlj1Vt8y04oXj4aQkOT/PP9fkmfz2MGkO3fk\\nwe0puvT2cOfl4+fT5Kq/cXEN2bFfRf01tATtBEMPVT1Yyj3qI0wIsxjAOINN6sEE\\nbJy0oEBgpZTXKSidSsyw1pZQ7KUglmjLPNFecOdjxwb1UMiLDMpJmRd1TOsDD16R\\nICBGeqd4KoIm1BAsiVOYSad1PMUx8Nq4aiIinDUVnM3HcS8uBi3eP/T/mXvaplPc\\nMe2CcmBTroKQjGIuvHZGeK5J2ggrKAszgqgemRVG0XoPVbndf9S/VFQKACUsMK81\\nY6nSnAbChAcvlVWccq9DYAJU4NRag8HK6EiE1NZ6Y1qqEpahrGBQ5RnqF9PhMJrV\\nq5PGfN+zhxpFPi5Qn8RDV+cNGuGNfxxxD8HKJC9hkDcKvNd6aBIzxyQmhZRrEEww\\nxYmNd8RSOCriY1tNKQ9BUK89EBk/UlNulYhyrLj3aRrvEe0ruD7YQyLtohijPo2U\\nJmshs8LU0xLQ+pPFQ8lcN5mdOdBUY9I9euJPjc5Mkp84yS7fO/Po9E69+Jp+/P7h\\nxZe73a2TO+c7X+8vbF65c6qTfJ79cWvy6dX7N7s3v209/bW98TA7Nz9/fPRuYffC\\n9uyzjd+udO0D\\n=coi5\\n-----END PGP MESSAGE-----\\n\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432651430,\"expire_in\":10000000,\"prev\":\"a9e948e5353741bd48065c251c7b8224ff52d8de16f4f824b7548e074dd99d67\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "0113b0eeaca0a9deaf0ac6be7e50777d89e789603088b120d651b18b3d18d54cd4cd0a",
            "ctime": 1432651430
        }
    ],
    "keys": [
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nxm8EVWSGQxMFK4EEACIDAwTUgD4K0SSmhOHwgdWN2M8LT0NVQWSwgIYd0jeH1fM5\nJsE/Dsuem9GQwHLGwUILIINMzOPHbPscjOC7Rf3+8uAHiTDj00ejRgDoJpCKIXjH\nEM9MMxofqG0eDvEldTWKZibNBW1heDMywpIEExMKABoFAlVkhkMCGwEDCwkHAxUK\nCAIeAQIXgAIZAQAKCRB0waG+wsm+NVgWAYC5Q86W0VFs6IGqOgD2HIBAhShCo0Bp\nOivlkcK4iO7xFfYfbXusj74byp13ct8Qzk8Bf2VY8vJF6eha8dfcngiCZsEOAGVE\n3abIgE/YOERD/xm1sfzYs50QHNSj+LBU7pBqLs5WBFVkhkMSCCqGSM49AwEHAgME\naomMzHfI91CQZmMq3mYYh1szK9lBxoK8jYMHa0iVxCKd3mcTtvRxMBEak6XSo2U2\ni8i6iv25vyTO1yfJPcmBOwMBCgnChwQYEwoADwUCVWSGQwUJDwmcAAIbDAAKCRB0\nwaG+wsm+NXxRAYDlZo+z7gpcXFyILtxBbhBKz0qqueRVBfkCP8y4ANJJSjH6MnC2\nEcLwj6BNX8yq3TEBewf4GinIZHMXWj8hOH8KKtyidPLhR5WPCgPe3SYP0lZzFYgO\nb1YYrIVicghxSB6UTM5SBFVkhkMTCCqGSM49AwEHAgMEZ5qBf/Nw3XSK0OSz23Kz\nslf6ixA7xs+TNE72MwuzoVDGypTHTF5tz9Oz1/oDH8eCzZHLDHtF3gtgyyTv+864\nvMLAJwQYEwoADwUCVWSGQwUJDwmcAAIbIgBqCRB0waG+wsm+NV8gBBkTCgAGBQJV\nZIZDAAoJEAi0l3UiUd0H8R4BALO/bP6Ir0dBdzHMjsRitZ5cf1gNlTOsALhfTaRm\nCigeAQD2/vmEl6C7TA1hlSuvGgp7Z7UPgUgJizoYJGu2NpAwgqEQAYCtFBHE8FmQ\nKpbovcWv783nqAiwsVRQPgefcE7tZsneUgBa5pT3tFmsPBH0229y8IoBgPcPRToY\nLELCkzhNSfsxY6UI7Qlp57lFIjnxrDIrz5kYrVRmNeLIrbZK+56xv6x3rQ==\n=vv7t\n-----END PGP PUBLIC KEY BLOCK-----\n"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32",
    "label_kids": {
        "e": "0113b0eeaca0a9deaf0ac6be7e50777d89e789603088b120d651b18b3d18d54cd4cd0a",
        "sib1": "0113265ee75be3d883397842f135ded67b7424d8ff35e7f42bba0eb67f51568bbdaa0a"
    },
    "label_sigs": {
        "e": "3d80007ad5e64c9367fe566b03c1918bff04d522537787d7385fdc4bed274a740f",
        "sib1": "03a631ece380553b7a422c0d53ff9492a0a036594580304fbbea14af928761c10f"
    }
}
`,
	"missing_reverse_kid_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nyMDuAnicZZE9b9RAEIYdEpA4gUBKl3LLw0T7Ya/XJ9EGpUIKBAoUjvXO2GxyZzu2\nL+RDJ2jCD6BAAgoafgM1IEJzQkhItEgUdHRIIAoU1o6omGbn3XnmnV3Nu/PzXm/x\n0+cLo9/9w/m52eujibcOj1YOSFLAHhkckC3sjtTmGVZlZfOGDEgSc4PSgBJcSaV4\nClzHFGRCNY3RQKQlTQE48cm9om47nE2ia1y2hbtzYmih9fmf3+oKlDHBqJIQIYRR\niCmgjFDEjINOGRcyUCHFgIuAGa4ppG44UJkI7R4VBzyEQFHt7CadXRQYoUwaGUzi\nIEpDKTgNQlBGtyCLW7DGKtdjdPRY7wpOpj5p9spW4wjQfcInO1jVtsjJgLmiaWxL\ns0BwGTIhhE9wt7QVDm1L0JPwSVnhDhnkk9HIJzVuD09MWSfyossanbkxtc1y3Uwq\nJNO3dxa8xZ535vSpdhle7+zFfys63JjzPq78Wt3cKM9dW7t16eXS7M+Hy9P9pQfk\nuP/02c2r+vmad3x7/cqXH1k1vv7maHN2/9X7hezFk/705/ev/Yc3Ht9d/vYXEwyi\nWA==\n=H6Xl\n-----END PGP MESSAGE-----\n",
            "payload_hash": "00169139cc7d3d241bcd0537fa7dcf1f1d6c13f2ef581bf402424fbfd169bed1",
            "sig_id": "ea8a3358e12cdaa444d211fa71ffc4b421040244fd5b72ea2b387d722c11d8170f",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"b92ce6cd83286882fd2a90d6b0a09ecd7a60fdd2\",\"host\":\"keybase.io\",\"key_id\":\"b0a09ecd7a60fdd2\",\"kid\":\"01131086d7ed575efde67e3912daf12364850e42341c2a0df0d6d06b3acd89425d480a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432651333,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "01131086d7ed575efde67e3912daf12364850e42341c2a0df0d6d06b3acd89425d480a",
            "ctime": 1432651333
        },
        {
            "seqno": 2,
            "prev": "00169139cc7d3d241bcd0537fa7dcf1f1d6c13f2ef581bf402424fbfd169bed1",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nyMQyAnicrZU7jONEGMf3OEDiJB7SFVBQrIKugOxtPDN+Bh1S4jhONnFeTjaJL2Jl\nz8NxnDiJ7ThOTidEARIlFRUFFSUdBSUCIUGBhEQHNRJPUVBwDTi7h665kilszfj/\nff7Np/l/88Wz149u3Pzu++fnf7/2zvVr33z5+uZoQN6t3ss5S7LLFe/lfHr5onNC\no/jC90iumOMAQICTRSJRIkgCZYSKEkUKgMRmACKRlwWO8hDxAEObI4wjIuFEB9mY\nyAoPBcLLnJ07yTEvcGm4Cr0gztI6CsRUzCQIyqIsQ0agrWShDmdzCsVEskWOEQKz\nwOkyOkRkcI4d0VNvma1lk4tLvMfo/2fuzWU6icdIxkzC1FF4iQkighwvEBnbByFQ\nDsKIhoG9oJl6YacI5u6f5CLPeVjUR1RUoUBCPHUY4QUAOYlIDCkytHmKAWYCoNQW\nOIHjFYlS3gGiQ3kO2BBKLCuRJFxShTShYUQvIs/N0t4+jLKm11vHHb1zbGimWdK1\ny9VJcJ4JvWVQPG5cVfC4vaLBQZbAU+4UcJNAXS4WNIiLx9M4XkXFQuFRrQs43K3i\n5SSYBDujSkqBh0OrLtmgV64aJGGGtskbqjvWtf1svHaMdl4gaTTtDdRKXXNtVaBW\ne8iGptCQJoEoQcmflvv5tTsoq4grAXfj+jUtneFyh9fc5Xo4H6penY/TVmqMdwM5\n3VqJVMBI2LcbBTQJJH9n+xAbTHKSPpvLfAybsdzya2upE06bIzcdmSNv0GxXhK3m\nlGxStnZ1Hah6r22spktjErizeqMklmCj0tS3XUefWvZq6XmCb6R7c1ftcpoKNMWq\nLPzFpttwBv50fEZJrRrNC4vUVBeTgPQCtG60NLlqBo1uoeT07Hg2X9vLLpm1ulyp\nU14mih3aGtVxOCqJkjyU9Fpfp51Cz9l3sl3gs33bVBWzxqe1Oef352Ubm+5wFUvD\nruUMt9uwsTXT2bmJLAs3m8P6fLSJV6E42y73lmbVM4aS7o9XW9vQqn7DwWaHtcrG\nwAzXjfrO4GemQUF3sBgLrZ0Odw03PR/PR1Fjtx3kzwnnifok8PhNCnV1PkoZW8e+\nF5vNpGr3W/zaYEJlYPv8WVgdsV49nOE9l0RCc6g6a34oYma0zOl8ErQXhicrYqmk\nipbMqedeydWZrpZdZhnlulBe4W3VdALfiLpJz4or42ZVnlpYWnDTqFbuKpOA29g9\nLnBqfgupleV47ofDURXhuS8Df5/nidLvLvbYLid7Q6Ro7FGn3xtWZFnOj0Yz1upM\nggQBmO026ESgVWNKKiciwfsWSgpWrSI02UoU8x0Ew1Z/pMhkCPJWgpypjGddQdnU\nKlF2Js/arJIWEiRMnUKBr81WzbXVvXNnEtw5H/HZwb90kdaqPMZZB4fHu9XB8Q+N\nfpJLrtyWK4LsI469Qz8APIKikD3RSY6mKy+kF95BwV2Nk9wqs/OhO3BAVABSMJYI\nIpAHDiacgDLfSwQzwAARMUAMUibIwGE8B3nIs6yPZGEOJSD7fUTXF1dI4HISLHNF\nmFHa7iWkG9jxJqS5+5+/+eTRzRtHTz/1xOEKOLrxzAv/XQz3xkf/0LsPHrxU/eRH\n9+PbP9984/db5T86qvnq6U+/fPbrtPJX8aNrRx/231t89Vz8weju2688/dYPL37a\nvf0+2nx98tufL4vfPtje+heUNA3B\n=okGo\n-----END PGP MESSAGE-----\n",
            "payload_hash": "525de3a305ed703e4419add54649cedd083464f3b76f296c0bd9c7ca2bff3bb9",
            "sig_id": "eca9bd4e3be4d3be48501d5be8bc01efdbad86b51c9887a11279c44a9eb7de4b0f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01131086d7ed575efde67e3912daf12364850e42341c2a0df0d6d06b3acd89425d480a\",\"fingerprint\":\"b92ce6cd83286882fd2a90d6b0a09ecd7a60fdd2\",\"host\":\"keybase.io\",\"key_id\":\"b0a09ecd7a60fdd2\",\"kid\":\"01131086d7ed575efde67e3912daf12364850e42341c2a0df0d6d06b3acd89425d480a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0113e9e1734ebfd451207d7f3982a4ec1cf51eea5050497ee4b16be401a227fa90750a\",\"reverse_sig\":\"-----BEGIN PGP MESSAGE-----\\nVersion: Keybase OpenPGP v2.0.10\\nComment: https://keybase.io/crypto\\n\\nyMFdAnicrZI7a1RBFMdvfMEu+MCgYGEzjYqbMO+5dxshRUCDIEgaC5eZOWfWS5K7\\n6727khBT+qgUBC30A1gugkHExjcBP4EgoqWlWCiI4txNxMYyU8xwZv7/c35zOK/3\\n7kyak2cMf7bvTfl84t2Lt8NkHq7PrhLXgxXSXiULOD5wEbAadBZyIG1CGROMphoM\\ngjIKA6A2KDLGwQbGhZapoii5kMxzSyFQ0EC1E9ZDmkmuQKbUkhYJedHFsl/mxSCm\\ndRn3qKNE8FSnKQ/AbRatjlqaoQdjNQ0APBov9araEeGcrXA678W7GHTGeP/RbzP3\\ncJzOSC9SH4xHl0kTlBacSgWpt7WQZbWwwrKwSxjVS3ZZcLLWIlXutpr6jwozZEZI\\ndAGkYpwaMEFkKbcSPfNBMUSrqKIyM4jSMe1QUmY5NyG2yKgxVYlXsKywU+Vd0i6G\\ni4ux2GClXxffqtkitSLvFaTN4qMf5DUak4JrFXfRIrjcz0vs5LWCbq4W6cfMNShl\\nOmMi896AAC6Z80CViAgGfGCBgfZMBI5BpcwFSbnkMsQvRZtDYLF8hZc7m0hsHBQ9\\n0uaR0nbHkN3CDoYlkrWXF3clk81kz+4d9TQmzcaBvzM6e3YiebTRWD888+XXjfNP\\nv312jSMnPs1NHf9x8v6dczN3v/ZHD5Lfp66+P32rNTX98dW1+Zv3bh8cjQ59uHDs\\n6JOfDx/v35hb//4HjpLqZQ==\\n=VX4o\\n-----END PGP MESSAGE-----\\n\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432651433,\"expire_in\":10000000,\"prev\":\"00169139cc7d3d241bcd0537fa7dcf1f1d6c13f2ef581bf402424fbfd169bed1\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "01131086d7ed575efde67e3912daf12364850e42341c2a0df0d6d06b3acd89425d480a",
            "ctime": 1432651433
        }
    ],
    "keys": [
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nxm8EVWSGRRMFK4EEACIDAwTGPp5WCEi30jKYsqP+DUV1NWMxi53YT9axzEbrEyU/\n7UdKsWZPZJnOsYpjCr8+VXqIHY56kO1oiR0HqpnoJhsRZ1imNqHq4L4d/QMhyNGs\nICqaFQJsSShlwQwvUffq0urNBW1heDMywpIEExMKABoFAlVkhkUCGwEDCwkHAxUK\nCAIeAQIXgAIZAQAKCRCwoJ7NemD90gI7AX9zOItdXE2tVZuNus0+HvqcVI4cvlXD\nqt/0HTIly/mDP8tgNcPSpgOuzPVCQy+dnYMBfiGFbDQnwge9Cea0tHHIbY8qAOax\niX6yRDuMPaqhbleIKaKU1hJBrtDMLz6pnHLtn85WBFVkhkUSCCqGSM49AwEHAgME\nHZOEO11yQS3sZtB43vEHD9YbSHpvhOtgArJl4/dQB5Jr4DFrmBAA5WPlUbF8EBgc\nJ7oD6qVGETtJDX1R78qo0QMBCgnChwQYEwoADwUCVWSGRQUJDwmcAAIbDAAKCRCw\noJ7NemD90tS0AX9gy15Vv10RalrRSHhjcT/BW6r6MGHy/Z+w33tUAND7EUDBwXvc\n4u4i/+bTOJY6TcYBf2fN+q2Zcajt91IJiqd/4su/SwcpJb9mcHJpc4M1wXen6rFn\nmHVVB47WTJpk+ORFcc5SBFVkhkUTCCqGSM49AwEHAgMEBHRJTnd1BZp6x9KrE1p1\nPJ4McaYM8lu1JysuL+BJdV/yw98XnKBIv8nhnBikJgntcj1jAcWyixuQJvbreZn+\nD8LAJwQYEwoADwUCVWSGRQUJDwmcAAIbIgBqCRCwoJ7NemD90l8gBBkTCgAGBQJV\nZIZFAAoJENLVD2z5KoUDfcsBAKpn+wqezR1uPhU2aG1YRgwTlSr2Qkf/n1Ob8gss\nkCilAP4+zmdLJgv017cEYrMAxozL/zTFhCRns+BBMLnJCaRNDov4AX93/JS4CYca\naw7nB4SNhxHaMgyV6jhsXmO1psuC8uAVqtkQPuo3hf6ttcftfxbnv9gBgNJwoYyk\nkEStMA8gnT4841eHX2JMzuZ9kfqYcKuf+rqkSgX+wyMnuU4rKtKWHc+mZg==\n=pMMp\n-----END PGP PUBLIC KEY BLOCK-----\n"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32",
    "label_kids": {
        "e": "01131086d7ed575efde67e3912daf12364850e42341c2a0df0d6d06b3acd89425d480a",
        "sib1": "0113e9e1734ebfd451207d7f3982a4ec1cf51eea5050497ee4b16be401a227fa90750a"
    },
    "label_sigs": {
        "e": "ea8a3358e12cdaa444d211fa71ffc4b421040244fd5b72ea2b387d722c11d8170f",
        "sib1": "eca9bd4e3be4d3be48501d5be8bc01efdbad86b51c9887a11279c44a9eb7de4b0f"
    }
}
`,
	"signed_with_revoked_key_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgNkE13AJPYlXXSNyO4DTlRVMLDNln4Gji/wPNwVgvxwYKp3BheWxvYWTFASp7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMDM2NDEzNWRjMDI0ZjYyNTVkNzQ4ZGM4ZWUwMzRlNTQ1NTMwYjBjZDk2N2UwNjhlMmZmMDNjZGMxNTgyZmM3MDYwYSIsInVpZCI6Ijc0YzM4Y2Y3Y2ViOTQ3ZjU2MzIwNDVkOGNhNWQ0ODE5IiwidXNlcm5hbWUiOiJtYXgzMiJ9LCJ0eXBlIjoiZWxkZXN0IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ1NzYzLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjpudWxsLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjEsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAaH1Gcdn5h66O/2llNDAaZKGzZAEQBuXgTN9JuxAa8hV2up2yn144/aRucaRx3lE/fYVDJsgy6lg/fJI4c+i6CahzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
            "payload_hash": "9aa395f90c7c4cc42a9e3934d7b88d85318c7a8f4be72def31b3ed9e1f1e8088",
            "sig_id": "53e8e98d74d1bb3ab2710a2ce955e6dfc111c400478d7f4dd18a359cdd79350f0f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"0120364135dc024f6255d748dc8ee034e545530b0cd967e068e2ff03cdc1582fc7060a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432145763,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "0120364135dc024f6255d748dc8ee034e545530b0cd967e068e2ff03cdc1582fc7060a",
            "ctime": 1432145763
        },
        {
            "seqno": 2,
            "prev": "9aa395f90c7c4cc42a9e3934d7b88d85318c7a8f4be72def31b3ed9e1f1e8088",
            "sig": "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",
            "payload_hash": "d845211b2ba21e87141d51553c43ecc3701fbf37203d63e5b82ede4c0fa023bc",
            "sig_id": "ecdbdc174a75a79fb175b9c4729765eaf0d03d1b9d910f244f90293b6f96e8b40f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120364135dc024f6255d748dc8ee034e545530b0cd967e068e2ff03cdc1582fc7060a\",\"host\":\"keybase.io\",\"kid\":\"0120364135dc024f6255d748dc8ee034e545530b0cd967e068e2ff03cdc1582fc7060a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0120787908851d57913b40e4b79e21cfce3ca731e310052509efee75b8969ea75f4d0a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgeHkIhR1XkTtA5LeeIc/OPKcx4xAFJQnv7nW4lp6nX00Kp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMzY0MTM1ZGMwMjRmNjI1NWQ3NDhkYzhlZTAzNGU1NDU1MzBiMGNkOTY3ZTA2OGUyZmYwM2NkYzE1ODJmYzcwNjBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMzY0MTM1ZGMwMjRmNjI1NWQ3NDhkYzhlZTAzNGU1NDU1MzBiMGNkOTY3ZTA2OGUyZmYwM2NkYzE1ODJmYzcwNjBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwNzg3OTA4ODUxZDU3OTEzYjQwZTRiNzllMjFjZmNlM2NhNzMxZTMxMDA1MjUwOWVmZWU3NWI4OTY5ZWE3NWY0ZDBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ1ODYzLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiOWFhMzk1ZjkwYzdjNGNjNDJhOWUzOTM0ZDdiODhkODUzMThjN2E4ZjRiZTcyZGVmMzFiM2VkOWUxZjFlODA4OCIsInNlcV90eXBlIjoxLCJzZXFubyI6MiwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEDiskJVcUsN6nC8c7BngKGNZaMW/Gn1iae0a0WLHRvnyPbLJqJ1aA3Jqzd5KcsjTaEK3/JKjBFw4538tC+t0OEPqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432145863,\"expire_in\":10000000,\"prev\":\"9aa395f90c7c4cc42a9e3934d7b88d85318c7a8f4be72def31b3ed9e1f1e8088\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "0120364135dc024f6255d748dc8ee034e545530b0cd967e068e2ff03cdc1582fc7060a",
            "ctime": 1432145863
        },
        {
            "seqno": 3,
            "prev": "d845211b2ba21e87141d51553c43ecc3701fbf37203d63e5b82ede4c0fa023bc",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgNkE13AJPYlXXSNyO4DTlRVMLDNln4Gji/wPNwVgvxwYKp3BheWxvYWTFAhh7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMzY0MTM1ZGMwMjRmNjI1NWQ3NDhkYzhlZTAzNGU1NDU1MzBiMGNkOTY3ZTA2OGUyZmYwM2NkYzE1ODJmYzcwNjBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMzY0MTM1ZGMwMjRmNjI1NWQ3NDhkYzhlZTAzNGU1NDU1MzBiMGNkOTY3ZTA2OGUyZmYwM2NkYzE1ODJmYzcwNjBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInJldm9rZSI6eyJraWQiOiIwMTIwNzg3OTA4ODUxZDU3OTEzYjQwZTRiNzllMjFjZmNlM2NhNzMxZTMxMDA1MjUwOWVmZWU3NWI4OTY5ZWE3NWY0ZDBhIn0sInR5cGUiOiJyZXZva2UiLCJ2ZXJzaW9uIjoxfSwiY3RpbWUiOjE0MzIxNDU5NjMsImV4cGlyZV9pbiI6MTAwMDAwMDAsInByZXYiOiJkODQ1MjExYjJiYTIxZTg3MTQxZDUxNTUzYzQzZWNjMzcwMWZiZjM3MjAzZDYzZTViODJlZGU0YzBmYTAyM2JjIiwic2VxX3R5cGUiOjEsInNlcW5vIjozLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQIN1Yd5VY6JwixVs2xEJhQ0cz7PqHm00OCbuNFamehp6RfMzDkFK9L8YfUwkKoRQ89czbCeDw4my5vT/n4eutgyoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "3fd60b4b2317261b678b92b23dfc46e3d369b8d94a79b5b3dbfc2581de6a6741",
            "sig_id": "384d5b0d6ce5497364589b4fc3c907c0dc26b264aeb523b2212367cb96b170920f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120364135dc024f6255d748dc8ee034e545530b0cd967e068e2ff03cdc1582fc7060a\",\"host\":\"keybase.io\",\"kid\":\"0120364135dc024f6255d748dc8ee034e545530b0cd967e068e2ff03cdc1582fc7060a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"kid\":\"0120787908851d57913b40e4b79e21cfce3ca731e310052509efee75b8969ea75f4d0a\"},\"type\":\"revoke\",\"version\":1},\"ctime\":1432145963,\"expire_in\":10000000,\"prev\":\"d845211b2ba21e87141d51553c43ecc3701fbf37203d63e5b82ede4c0fa023bc\",\"seq_type\":1,\"seqno\":3,\"tag\":\"signature\"}",
            "kid": "0120364135dc024f6255d748dc8ee034e545530b0cd967e068e2ff03cdc1582fc7060a",
            "ctime": 1432145963
        },
        {
            "seqno": 4,
            "prev": "3fd60b4b2317261b678b92b23dfc46e3d369b8d94a79b5b3dbfc2581de6a6741",
            "sig": "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",
            "payload_hash": "ff34e6ffa732ada20e3e2413f82fddb835f4c981c9acf43033da7ff8275333f5",
            "sig_id": "7eb99ec28f5cea7163e1e3dc05fae2690c7d040fd6b99a70ebe2266786e637500f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120364135dc024f6255d748dc8ee034e545530b0cd967e068e2ff03cdc1582fc7060a\",\"host\":\"keybase.io\",\"kid\":\"0120787908851d57913b40e4b79e21cfce3ca731e310052509efee75b8969ea75f4d0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"01207a93ab35003f48262dd5593bd01be53d3acda0c09db5dac6187c6c6e1d60b9e00a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgepOrNQA/SCYt1Vk70BvlPTrNoMCdtdrGGHxsbh1gueAKp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMzY0MTM1ZGMwMjRmNjI1NWQ3NDhkYzhlZTAzNGU1NDU1MzBiMGNkOTY3ZTA2OGUyZmYwM2NkYzE1ODJmYzcwNjBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwNzg3OTA4ODUxZDU3OTEzYjQwZTRiNzllMjFjZmNlM2NhNzMxZTMxMDA1MjUwOWVmZWU3NWI4OTY5ZWE3NWY0ZDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwN2E5M2FiMzUwMDNmNDgyNjJkZDU1OTNiZDAxYmU1M2QzYWNkYTBjMDlkYjVkYWM2MTg3YzZjNmUxZDYwYjllMDBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ2MDYzLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiM2ZkNjBiNGIyMzE3MjYxYjY3OGI5MmIyM2RmYzQ2ZTNkMzY5YjhkOTRhNzliNWIzZGJmYzI1ODFkZTZhNjc0MSIsInNlcV90eXBlIjoxLCJzZXFubyI6NCwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEDJMN6kYBA9owVIKWRe+9s2Oti0NKtTzjrsdIefvfb3t92PGFzm0A5D4t973MsGTqcguIzoajNvysOUw5oAOTQPqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432146063,\"expire_in\":10000000,\"prev\":\"3fd60b4b2317261b678b92b23dfc46e3d369b8d94a79b5b3dbfc2581de6a6741\",\"seq_type\":1,\"seqno\":4,\"tag\":\"signature\"}",
            "kid": "0120787908851d57913b40e4b79e21cfce3ca731e310052509efee75b8969ea75f4d0a",
            "ctime": 1432146063
        }
    ],
    "keys": [
        "0120364135dc024f6255d748dc8ee034e545530b0cd967e068e2ff03cdc1582fc7060a",
        "0120787908851d57913b40e4b79e21cfce3ca731e310052509efee75b8969ea75f4d0a",
        "01207a93ab35003f48262dd5593bd01be53d3acda0c09db5dac6187c6c6e1d60b9e00a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"simple_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEggg05gbQjjIO9nT6AnnoCfDEbP8W+/Qc9K/tIlLm+hrgKp3BheWxvYWTFASp7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMDgyMGQzOTgxYjQyMzhjODNiZDlkM2U4MDllN2EwMjdjMzExYjNmYzViZWZkMDczZDJiZmI0ODk0YjliZTg2YjgwYSIsInVpZCI6Ijc0YzM4Y2Y3Y2ViOTQ3ZjU2MzIwNDVkOGNhNWQ0ODE5IiwidXNlcm5hbWUiOiJtYXgzMiJ9LCJ0eXBlIjoiZWxkZXN0IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMxOTc5MjE0LCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjpudWxsLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjEsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RABjZFgXiUCNQR16g+RRL4xUtp+O2DI72+1gbMwoUJVlMxHGcXItlBFSFjfWGtOxU7//fYS0aVEH02q17IUCOXBqhzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
            "payload_hash": "223e25fd456ea18cc39b608bac8b5f35302d240d5f95e6e42e5a85b1136ac5e7",
            "sig_id": "f0a633afc45db7d611a0e484e2db45800e87033f7c4bc1f62577bc1eba156f060f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"0120820d3981b4238c83bd9d3e809e7a027c311b3fc5befd073d2bfb4894b9be86b80a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1431979214,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "0120820d3981b4238c83bd9d3e809e7a027c311b3fc5befd073d2bfb4894b9be86b80a",
            "ctime": 1431979214
        }
    ],
    "keys": [
        "0120820d3981b4238c83bd9d3e809e7a027c311b3fc5befd073d2bfb4894b9be86b80a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"empty_chain.json": `{
    "_comment": "The 'ralph' key has ralph@keybase.io signed into it. The 'steve' key does not. This is for testing key ownership when the sigchain in empty.",
    "username": "ralph",
    "uid": "bf65266d0d8df3ad5d1b367f578e6819",
    "chain": [],
    "keys": [
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v2\n\nmI0EVXnWkQEEALmSERRK4+fbTg5C28GOXxpiI/IfQZDFmJfWzNKNW6CrX1uj4o8G\nu3292bU5meCRCJstd09LFaELOASbknUOiXFQcnuzpY7J9Ou8uqxKcOYxi4xWjXih\nK8KQfL4khBIpB+7B0tp16qalqbd1+V2wIAJzmNdkY+cFGeXZE45gr+i3ABEBAAG0\nGHJhbHBoIDxyYWxwaEBrZXliYXNlLmlvPoi5BBMBCAAjBQJVedaRAhsDBwsJCAcD\nAgEGFQgCCQoLBBYCAwECHgECF4AACgkQUGs4lh8OKGN+TgQAhTSvOZV+8me6UuIc\ntBJTsZzAkzWsP3QpdSZ6zUyZ40yA5WMZMpnQb+6kd3xzo/tkm9imTK2xLtoCj52d\n3QZRN3BOqweetaiWng6sgIkMZc8EaQp68bPeVhtcAYU0Y/I9Jcim6d2nSXQcwuph\na3RWghsf37A9CfXlbZl36PvQ2ko=\n=qGhY\n-----END PGP PUBLIC KEY BLOCK-----\n",
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v2\n\nmI0EVXnWyQEEAMUSBo5Q/X3et/ExeX6UHvsH9Wv5IoaEmMqdT7zSxaphcFS+vo5E\n63QQrokfIUKZeMDmWjI65SNVNhpWng5mhrkz/bRql0pdikhwXnALLVnwT+0J/Spl\nyWiQKVfZ6pqz2anto56/rlhLui7oTLnG7q2CttIaAp1aaMzaHwMH4iW1ABEBAAG0\nBXN0ZXZliLkEEwEIACMFAlV51skCGwMHCwkIBwMCAQYVCAIJCgsEFgIDAQIeAQIX\ngAAKCRA0feCGaTASHtOGA/4u7MAY2z3UzVKzH4XxYknep4hD+75HH6TY1fCUHBcS\nAfVptSna59alV37JJtsBQhmNowyIqvNeUSl3VcN2oOBvJwxGyPhoh6Roxj+4phhO\nPGH9dzzaZVQVSYMBMU8Wuf9OsEM/2k7Wo2oes9XLiNVBqYNwvk6Q2BupT77FyzDI\ntg==\n=zpuJ\n-----END PGP PUBLIC KEY BLOCK-----\n",
        "0120a4dabfb39e2bc635eadc861b72d9f9512dfe4261562822433d261718235d5b6f0a"
    ],
    "label_kids": {
        "ralph": "0101de638c5bee00e1b473b5f3fa7065e55cde33e88d230a2cd253da5d78a1c5a8910a",
        "steve": "0101d17f13d98d9a21e3fcc2ba30ef38fe1ca0e805950839f7ecfb8b77ad5d128e4f0a",
        "nonexistent": "0101deadbeef0000000000000000000000000000000000000000000000000000000000",
        "naclkey": "0120a4dabfb39e2bc635eadc861b72d9f9512dfe4261562822433d261718235d5b6f0a"
    }
}
`,
	"jack_chain.json": `{
    "username": "oconnor663",
    "uid": "41b1f75fb55046d370608425a3208100",
    "chain": [
        {
            "seqno": 1,
            "payload_hash": "b421b35b23fc5da8a69fab3978ff34689fcaaf0048fc8af822aa32b5121c8193",
            "sig_id": "e43b8971ff4bdc1beb57fecc15c1b3455997d1c6ea5596e14f25bdf4d5ad47580f",
            "sig_id_short": "5DuJcf9L3BvrV_7MFcGzRVmX0cbqVZbhTyW9",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowGbwMvMwMFY2J6dsdZD14jx9IGPSQwh0yOXVisl5adUKllVKyXnZKbmlYBYeYm5\nqUpWStmplUmJxal6mfkKefkpqXpZxQpQNTpKZalFxZn5eUBVBnrmesZKtTog5SDN\naZl56alFBUWZILOUTC0SjUxMU4xTLJKMTJMNTdPMklON09LSzA0tzM2SzCwSU0ws\njFKMjYBGZuQXl6DYqgQ2Mz4zBSgKVu9kZuHoAlTvAlZfCpYwMUwyTDM3TUsyNTUw\nMUsxNjcwM7AwMTJNNDYysDA0MAApLE4tgnopPzk/Ly+/yMwM7GKgeFlmciqSl0vK\nM0tKUovwaSqpLAAJlqcmxUP1xydl5qUAfY0cLIZAlcklmSD9hiaGlkYGpgZGljpK\nqRUFmUWp8ZkgFabmZhYGQKCjVFCUWqZklVeakwNyU2FePlAWaFFiOtCe4sz0vMSS\n0qJUpdpORhkWBkYOBjZWJlDMMXBxCsDic9kVDobOWXPTXnulBP/5+WXyw94r/dXi\nayb80JuTNUfjzJXohNsN04zSOku9Xv77VyD46tHWzPqi+xe25ESJmS+d6JHZvOh+\n/KHTETHKb/1FKlborXgV9cO6bL3k3mXef9a93/Djg76cYFmxF2P5gcmXxK+IvjjF\nu7IhjvdNvMTbMP5V+qqKF9KdFjo093V/6VGeUKP8TvnFpv6sgIoXVv3HcvzermQ2\n+d/0dhrDj86Y3o8FB2ae5uu5dNv0zifxzjdX0yMzjy51Of36cnF8pMwj3zQrQyPP\nqPmlvcFzJrZlXeiZbda+sEV4Z6pFD9dKg8IPFT18nSE7TjPyde1rfbHydFjQPL3p\n736G/OYocJgxqw8A\n=bgPh\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"service\":{\"name\":\"twitter\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205029,\"expire_in\":157680000,\"prev\":null,\"seqno\":1,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1419205029,
            "etime": 1576885029,
            "rtime": null,
            "sig_status": 0,
            "prev": null,
            "proof_id": "756a07b444d2f1044a162210",
            "proof_type": 2,
            "proof_text_check": "Verifying myself: I am oconnor663 on Keybase.io. 5DuJcf9L3BvrV_7MFcGzRVmX0cbqVZbhTyW9 /",
            "proof_text_full": "Verifying myself: I am oconnor663 on Keybase.io. 5DuJcf9L3BvrV_7MFcGzRVmX0cbqVZbhTyW9 / https://keybase.io/oconnor663/sigs/5DuJcf9L3BvrV_7MFcGzRVmX0cbqVZbhTyW9\n",
            "check_data_json": "{\"name\":\"twitter\",\"username\":\"oconnor663\"}",
            "remote_id": "546811878073061377",
            "api_url": "https://twitter.com/oconnor663/status/546811878073061377",
            "human_url": "https://twitter.com/oconnor663/status/546811878073061377",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 337,
            "hard_fail_count": 0,
            "last_check": 1434439958,
            "last_success": 1434439958,
            "version": 2,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 2,
            "payload_hash": "382e1d68b4b81c6643483a74715ac0964e291fb2b52c4e9fa6c73d304ab9108f",
            "sig_id": "9d5b60c0d486f34089a922bcc5e77fb643b9781cbcc894af7e123fad50645d6a0f",
            "sig_id_short": "nVtgwNSG80CJqSK8xed_tkO5eBy8yJSvfhI_",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kltoFFcYx3et1VWrWBQJsaCdFq3tusy5zhn1QYOUVh9MglQq1vWcM+dkJ5eZ\ndWaTmNqIWmlTrDRtpaLSghdaIqYS6614wbsWry/6oEVoi6AvkgRFVHA9s1iqL52n\nme/8vv/3/893uke/ksqkl3/ZVOj9YDpMX7grUgs3LR610hKh12HNWGnJZl8FpeQt\n4C3KmmE1qQ7BY5Xzw8lB6KlcYzz5OZO12lQU+2FgKDvn5JDVmU3wpFn7QYOKipGf\naFmEcYiJhzwmIJGAaCoV0lo7gDlUUMY9zKCHoJEshHHppalWRTPve6Za4WsomzPX\n8HMrfGvlAAMBtEO0IMTG1EOOTW2GIeEI2gzYdgLGKnoeKZRhEIQRpRXHLSpqalb5\nKAwrsWXJTyCAgQttYiPHeOJxwbR51HGl4EBShRHChAosJQEMONz2kKnbLuFcJY1A\nEQKAhK459Lh0lJBcciopcIiHkbax4Awg1xFMCcQEEwQJ6EmICMXCVo4rHMk5ocYC\n0tyMMTNNiFgtD0JjzrjDbmfyHbX5Ur2wrga/VGgV/5e31FFMiu1K5A2yK+nPCz/w\nzMZeXCkw6Mv/AiCQtdSKoh+pvJ8QxKHMNk/WKkaqzWgKDIFARECkJfE449TVXJiY\nTGuEKXO1CaVtGzMtGdcMQm42JAiAQDLgov8SQuOTNxjJ2G8IeKk1UlbnV+k3hqbS\nmdSwV4ckdzY1csTYfy9yqjeTOji8+KB672c9PasmiqPhVFS+N38n+e2cqvqkdikt\nrC4e7e//R7w57/j8R69lahr3/Tx4/mI0ru/bqku98Cy41f/37acXa8+sv5a7scb9\n/tjA/m/WvfPk5M3amikb3qvecvvWYOav4VtP73n9u3dnLdzx9YGldX0f7uiCi7qf\n/vrWD79kV/fdf7v2fPHwPae+ceKPB2bWb/+U7qsfv/Exm3R/wF3c/WTtwZ5j8wZ3\nnZnQdm7KF3WNh6ovjKH4Sv+GzPvTRi+b+cd6sK5c9/Dy6SHtH338aFthSbpqQupU\nze/l9s/317XUXyuvmj30z40DW6ubx95Zc6hq77jC7gXl+PD18XPOnuw6Me2nI11X\nN496Bg==\n=diQc\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419205037,\"hash\":\"d679cba1c6e433456b4cc51817a0d3a1c095aae41921e5511c29817dac7ebcaca6c6175d43f04ba81397b8eb38b8b53b2dc23564b0e79b7caa562053facc56e4\",\"seqno\":114149},\"service\":{\"name\":\"github\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205131,\"expire_in\":157680000,\"prev\":\"b421b35b23fc5da8a69fab3978ff34689fcaaf0048fc8af822aa32b5121c8193\",\"seqno\":2,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1419205131,
            "etime": 1576885131,
            "rtime": null,
            "sig_status": 0,
            "prev": "b421b35b23fc5da8a69fab3978ff34689fcaaf0048fc8af822aa32b5121c8193",
            "proof_id": "034487589bf1339ef37e6210",
            "proof_type": 3,
            "proof_text_check": "\n\nowF9kltoFFcYx3et1VWrWBQJsaCdFq3tusy5zhn1QYOUVh9MglQq1vWcM+dkJ5eZ\ndWaTmNqIWmlTrDRtpaLSghdaIqYS6614wbsWry/6oEVoi6AvkgRFVHA9s1iqL52n\nme/8vv/3/893uke/ksqkl3/ZVOj9YDpMX7grUgs3LR610hKh12HNWGnJZl8FpeQt\n4C3KmmE1qQ7BY5Xzw8lB6KlcYzz5OZO12lQU+2FgKDvn5JDVmU3wpFn7QYOKipGf\naFmEcYiJhzwmIJGAaCoV0lo7gDlUUMY9zKCHoJEshHHppalWRTPve6Za4WsomzPX\n8HMrfGvlAAMBtEO0IMTG1EOOTW2GIeEI2gzYdgLGKnoeKZRhEIQRpRXHLSpqalb5\nKAwrsWXJTyCAgQttYiPHeOJxwbR51HGl4EBShRHChAosJQEMONz2kKnbLuFcJY1A\nEQKAhK459Lh0lJBcciopcIiHkbax4Awg1xFMCcQEEwQJ6EmICMXCVo4rHMk5ocYC\n0tyMMTNNiFgtD0JjzrjDbmfyHbX5Ur2wrga/VGgV/5e31FFMiu1K5A2yK+nPCz/w\nzMZeXCkw6Mv/AiCQtdSKoh+pvJ8QxKHMNk/WKkaqzWgKDIFARECkJfE449TVXJiY\nTGuEKXO1CaVtGzMtGdcMQm42JAiAQDLgov8SQuOTNxjJ2G8IeKk1UlbnV+k3hqbS\nmdSwV4ckdzY1csTYfy9yqjeTOji8+KB672c9PasmiqPhVFS+N38n+e2cqvqkdikt\nrC4e7e//R7w57/j8R69lahr3/Tx4/mI0ru/bqku98Cy41f/37acXa8+sv5a7scb9\n/tjA/m/WvfPk5M3amikb3qvecvvWYOav4VtP73n9u3dnLdzx9YGldX0f7uiCi7qf\n/vrWD79kV/fdf7v2fPHwPae+ceKPB2bWb/+U7qsfv/Exm3R/wF3c/WTtwZ5j8wZ3\nnZnQdm7KF3WNh6ovjKH4Sv+GzPvTRi+b+cd6sK5c9/Dy6SHtH338aFthSbpqQupU\nze/l9s/317XUXyuvmj30z40DW6ubx95Zc6hq77jC7gXl+PD18XPOnuw6Me2nI11X\nN496Bg==\n",
            "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am oconnor663 on github.\n  * I am oconnor663 (https://keybase.io/oconnor663) on keybase.\n  * I have a public key whose fingerprint is 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32\n\nTo claim this, I am signing this object:\n\n` + "`" + `` + "`" + `` + "`" + `json\n{\n    \"body\": {\n        \"client\": {\n            \"name\": \"keybase.io node.js client\",\n            \"version\": \"0.7.3\"\n        },\n        \"key\": {\n            \"fingerprint\": \"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"71876B68AD482D32\",\n            \"uid\": \"41b1f75fb55046d370608425a3208100\",\n            \"username\": \"oconnor663\"\n        },\n        \"merkle_root\": {\n            \"ctime\": 1419205037,\n            \"hash\": \"d679cba1c6e433456b4cc51817a0d3a1c095aae41921e5511c29817dac7ebcaca6c6175d43f04ba81397b8eb38b8b53b2dc23564b0e79b7caa562053facc56e4\",\n            \"seqno\": 114149\n        },\n        \"service\": {\n            \"name\": \"github\",\n            \"username\": \"oconnor663\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1419205131,\n    \"expire_in\": 157680000,\n    \"prev\": \"b421b35b23fc5da8a69fab3978ff34689fcaaf0048fc8af822aa32b5121c8193\",\n    \"seqno\": 2,\n    \"tag\": \"signature\"\n}\n` + "`" + `` + "`" + `` + "`" + `\n\nwith the PGP key whose fingerprint is\n[58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32](https://keybase.io/oconnor663)\n(captured above as ` + "`" + `body.key.fingerprint` + "`" + `), yielding the PGP signature:\n\n` + "`" + `` + "`" + `` + "`" + `\n-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kltoFFcYx3et1VWrWBQJsaCdFq3tusy5zhn1QYOUVh9MglQq1vWcM+dkJ5eZ\ndWaTmNqIWmlTrDRtpaLSghdaIqYS6614wbsWry/6oEVoi6AvkgRFVHA9s1iqL52n\nme/8vv/3/893uke/ksqkl3/ZVOj9YDpMX7grUgs3LR610hKh12HNWGnJZl8FpeQt\n4C3KmmE1qQ7BY5Xzw8lB6KlcYzz5OZO12lQU+2FgKDvn5JDVmU3wpFn7QYOKipGf\naFmEcYiJhzwmIJGAaCoV0lo7gDlUUMY9zKCHoJEshHHppalWRTPve6Za4WsomzPX\n8HMrfGvlAAMBtEO0IMTG1EOOTW2GIeEI2gzYdgLGKnoeKZRhEIQRpRXHLSpqalb5\nKAwrsWXJTyCAgQttYiPHeOJxwbR51HGl4EBShRHChAosJQEMONz2kKnbLuFcJY1A\nEQKAhK459Lh0lJBcciopcIiHkbax4Awg1xFMCcQEEwQJ6EmICMXCVo4rHMk5ocYC\n0tyMMTNNiFgtD0JjzrjDbmfyHbX5Ur2wrga/VGgV/5e31FFMiu1K5A2yK+nPCz/w\nzMZeXCkw6Mv/AiCQtdSKoh+pvJ8QxKHMNk/WKkaqzWgKDIFARECkJfE449TVXJiY\nTGuEKXO1CaVtGzMtGdcMQm42JAiAQDLgov8SQuOTNxjJ2G8IeKk1UlbnV+k3hqbS\nmdSwV4ckdzY1csTYfy9yqjeTOji8+KB672c9PasmiqPhVFS+N38n+e2cqvqkdikt\nrC4e7e//R7w57/j8R69lahr3/Tx4/mI0ru/bqku98Cy41f/37acXa8+sv5a7scb9\n/tjA/m/WvfPk5M3amikb3qvecvvWYOav4VtP73n9u3dnLdzx9YGldX0f7uiCi7qf\n/vrWD79kV/fdf7v2fPHwPae+ceKPB2bWb/+U7qsfv/Exm3R/wF3c/WTtwZ5j8wZ3\nnZnQdm7KF3WNh6ovjKH4Sv+GzPvTRi+b+cd6sK5c9/Dy6SHtH338aFthSbpqQupU\nze/l9s/317XUXyuvmj30z40DW6ubx95Zc6hq77jC7gXl+PD18XPOnuw6Me2nI11X\nN496Bg==\n=diQc\n-----END PGP MESSAGE-----\n\n` + "`" + `` + "`" + `` + "`" + `\n\nAnd finally, I am proving ownership of the github account by posting this as a gist.\n\n### My publicly-auditable identity:\n\nhttps://keybase.io/oconnor663\n\n### From the command line:\n\nConsider the [keybase command line program](https://keybase.io/docs/command_line).\n\n` + "`" + `` + "`" + `` + "`" + `bash\n# look me up\nkeybase id oconnor663\n\n# encrypt a message to me\nkeybase encrypt oconnor663 -m 'a secret message...'\n\n# ...and more...\n` + "`" + `` + "`" + `` + "`" + `\n",
            "check_data_json": "{\"name\":\"github\",\"username\":\"oconnor663\"}",
            "remote_id": "df527ad8438a424efcec",
            "api_url": "https://gist.githubusercontent.com/oconnor663/df527ad8438a424efcec/raw/0c24241710275124f07db83ab5b6f97fc6952165/keybase.md",
            "human_url": "https://gist.github.com/df527ad8438a424efcec",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 142,
            "hard_fail_count": 0,
            "last_check": 1434467728,
            "last_success": 1434467728,
            "version": 3,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 3,
            "payload_hash": "44a567581a31b596f31745c4633feced0806f427bac69e720aecfda02279b9bb",
            "sig_id": "a5039b256eb381b395e2fe9018e931c7dbe052dc1f1cf213b9e8e930ef0feabf0f",
            "sig_id_short": "pQObJW6zgbOV4v6QGOkxx9vgUtwfHPITuejp",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kltoHUUcxk9sm9ZqSIsXfIgStwUpTU7mPrMnDdUawWiIgkIxUk/mtidrkt2T\n3ZOYGKMFi02EqiCaFvom7dFqHqrRqOhDQdBYifqgra3QSERaxQuitUorzh4qti/O\n01x+/2++b/7zXMOK3Jq64d0D/bN3tqK6o2dU7v6Z3hcnPBWbca8w4enB0EaVbBbJ\nIesVvAE7rmRq82HcHMXG5h9Omy8yLd6oTdIwjhwF8jyPvcmWDM+KgzAq2aSchJmW\nR4VEhBpshEJUQxowbXEQBBwKzhQT0hCBDEZOsj9OK5fd6tU0i6FxuzV+GxO3dTq+\ns8aP1A4IVDDgNFCUAsIM5oABQRCVGAEBAcjA1CYXI8U6jqI4YazmeMgmA4O2mMRx\nLbauhBkECfQRoJAR50mm/a5MCetTbQgEkkAmsG8UVr6EPsYKSYWg70NEEUOMBZZw\nHViojUvGCIIKcQmUIpJBjQx3qSH2XQjOKIQ+0BhA3z2L5BAjoxhWGLoCLgjnwAgI\nEcbEhUjtcBQ7c84dRZPZOhkNtb2kXYk1Jqz8X97KeDnbfMSqokMOZfVFFUbGdezS\nlkKHXv4WiIMWz46Vw8QWw4ygnAngRotXTuyo08QCWWiYUEQJqBkjmAgsOeGQSg18\nRizyYaCQokgT6weSaY4NBkQqHwIR/JcQO5+y5CTTsBTJykhivcnpuqaVubo1ufpV\nV2R/Nrf2ynX/fuSNJ1b/vaF1+be0Mdf1Tld0w9lVi3ubCu/KRXnhlpOf98z1jd13\nCOyZOn3q+lj90Lc4N/P2L4WP83eLXYdv36zr35w+cu1rbe9tuKP86plkHiwffKvh\ncPT6tvOn/Zd3/rV759LWA6a1c/aJ6vvb0dahPY+dO9n42UfV/cvtv3+3aWFkxYmk\ntGVmqffPVx7atfR0Kej+dKHvp59fWMnEV6ee2etddWx9YcfqWx+/50hPd1fV7mi/\naazj7Lpvp+YXrut444NfG/ddaKebJo43w3T25gce7H6pybtmcbrx+z+eXf/j5vPD\nc0+Ob+FTg4/e9c3XN66tHudH91ePbU8+vHqk3FD/VFvH/L2fHKx/vld/ea4t/uIf\n=i2ny\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419205164,\"hash\":\"b8e95cd410a416839db3b9a1933b2ab219912526266fe47cfe1cdd486421b27a0bb4a61c2d776b1394827651190c30195f6a7132db63b317a0784770d8112334\",\"seqno\":114152},\"service\":{\"name\":\"reddit\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205270,\"expire_in\":157680000,\"prev\":\"382e1d68b4b81c6643483a74715ac0964e291fb2b52c4e9fa6c73d304ab9108f\",\"seqno\":3,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1419205270,
            "etime": 1576885270,
            "rtime": null,
            "sig_status": 0,
            "prev": "382e1d68b4b81c6643483a74715ac0964e291fb2b52c4e9fa6c73d304ab9108f",
            "proof_id": "7374183938ae0741e08d3c10",
            "proof_type": 4,
            "proof_text_check": "\n\nowF9kltoHUUcxk9sm9ZqSIsXfIgStwUpTU7mPrMnDdUawWiIgkIxUk/mtidrkt2T\n3ZOYGKMFi02EqiCaFvom7dFqHqrRqOhDQdBYifqgra3QSERaxQuitUorzh4qti/O\n01x+/2++b/7zXMOK3Jq64d0D/bN3tqK6o2dU7v6Z3hcnPBWbca8w4enB0EaVbBbJ\nIesVvAE7rmRq82HcHMXG5h9Omy8yLd6oTdIwjhwF8jyPvcmWDM+KgzAq2aSchJmW\nR4VEhBpshEJUQxowbXEQBBwKzhQT0hCBDEZOsj9OK5fd6tU0i6FxuzV+GxO3dTq+\ns8aP1A4IVDDgNFCUAsIM5oABQRCVGAEBAcjA1CYXI8U6jqI4YazmeMgmA4O2mMRx\nLbauhBkECfQRoJAR50mm/a5MCetTbQgEkkAmsG8UVr6EPsYKSYWg70NEEUOMBZZw\nHViojUvGCIIKcQmUIpJBjQx3qSH2XQjOKIQ+0BhA3z2L5BAjoxhWGLoCLgjnwAgI\nEcbEhUjtcBQ7c84dRZPZOhkNtb2kXYk1Jqz8X97KeDnbfMSqokMOZfVFFUbGdezS\nlkKHXv4WiIMWz46Vw8QWw4ygnAngRotXTuyo08QCWWiYUEQJqBkjmAgsOeGQSg18\nRizyYaCQokgT6weSaY4NBkQqHwIR/JcQO5+y5CTTsBTJykhivcnpuqaVubo1ufpV\nV2R/Nrf2ynX/fuSNJ1b/vaF1+be0Mdf1Tld0w9lVi3ubCu/KRXnhlpOf98z1jd13\nCOyZOn3q+lj90Lc4N/P2L4WP83eLXYdv36zr35w+cu1rbe9tuKP86plkHiwffKvh\ncPT6tvOn/Zd3/rV759LWA6a1c/aJ6vvb0dahPY+dO9n42UfV/cvtv3+3aWFkxYmk\ntGVmqffPVx7atfR0Kej+dKHvp59fWMnEV6ee2etddWx9YcfqWx+/50hPd1fV7mi/\naazj7Lpvp+YXrut444NfG/ddaKebJo43w3T25gce7H6pybtmcbrx+z+eXf/j5vPD\nc0+Ob+FTg4/e9c3XN66tHudH91ePbU8+vHqk3FD/VFvH/L2fHKx/vld/ea4t/uIf\n",
            "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am [oconnor663](https://www.reddit.com/user/oconnor663) on reddit.\n  * I am [oconnor663](https://keybase.io/oconnor663) on keybase.\n  * I have a public key whose fingerprint is 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32\n\nTo claim this, I am signing this object:\n\n\n    {\n        \"body\": {\n            \"client\": {\n                \"name\": \"keybase.io node.js client\",\n                \"version\": \"0.7.3\"\n            },\n            \"key\": {\n                \"fingerprint\": \"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\n                \"host\": \"keybase.io\",\n                \"key_id\": \"71876B68AD482D32\",\n                \"uid\": \"41b1f75fb55046d370608425a3208100\",\n                \"username\": \"oconnor663\"\n            },\n            \"merkle_root\": {\n                \"ctime\": 1419205164,\n                \"hash\": \"b8e95cd410a416839db3b9a1933b2ab219912526266fe47cfe1cdd486421b27a0bb4a61c2d776b1394827651190c30195f6a7132db63b317a0784770d8112334\",\n                \"seqno\": 114152\n            },\n            \"service\": {\n                \"name\": \"reddit\",\n                \"username\": \"oconnor663\"\n            },\n            \"type\": \"web_service_binding\",\n            \"version\": 1\n        },\n        \"ctime\": 1419205270,\n        \"expire_in\": 157680000,\n        \"prev\": \"382e1d68b4b81c6643483a74715ac0964e291fb2b52c4e9fa6c73d304ab9108f\",\n        \"seqno\": 3,\n        \"tag\": \"signature\"\n    }\n\nwith the PGP key referenced above, yielding the PGP signature:\n\n\n    -----BEGIN PGP MESSAGE-----\n    Version: GnuPG v2\n    \n    owF9kltoHUUcxk9sm9ZqSIsXfIgStwUpTU7mPrMnDdUawWiIgkIxUk/mtidrkt2T\n    3ZOYGKMFi02EqiCaFvom7dFqHqrRqOhDQdBYifqgra3QSERaxQuitUorzh4qti/O\n    01x+/2++b/7zXMOK3Jq64d0D/bN3tqK6o2dU7v6Z3hcnPBWbca8w4enB0EaVbBbJ\n    IesVvAE7rmRq82HcHMXG5h9Omy8yLd6oTdIwjhwF8jyPvcmWDM+KgzAq2aSchJmW\n    R4VEhBpshEJUQxowbXEQBBwKzhQT0hCBDEZOsj9OK5fd6tU0i6FxuzV+GxO3dTq+\n    s8aP1A4IVDDgNFCUAsIM5oABQRCVGAEBAcjA1CYXI8U6jqI4YazmeMgmA4O2mMRx\n    LbauhBkECfQRoJAR50mm/a5MCetTbQgEkkAmsG8UVr6EPsYKSYWg70NEEUOMBZZw\n    HViojUvGCIIKcQmUIpJBjQx3qSH2XQjOKIQ+0BhA3z2L5BAjoxhWGLoCLgjnwAgI\n    EcbEhUjtcBQ7c84dRZPZOhkNtb2kXYk1Jqz8X97KeDnbfMSqokMOZfVFFUbGdezS\n    lkKHXv4WiIMWz46Vw8QWw4ygnAngRotXTuyo08QCWWiYUEQJqBkjmAgsOeGQSg18\n    RizyYaCQokgT6weSaY4NBkQqHwIR/JcQO5+y5CTTsBTJykhivcnpuqaVubo1ufpV\n    V2R/Nrf2ynX/fuSNJ1b/vaF1+be0Mdf1Tld0w9lVi3ubCu/KRXnhlpOf98z1jd13\n    COyZOn3q+lj90Lc4N/P2L4WP83eLXYdv36zr35w+cu1rbe9tuKP86plkHiwffKvh\n    cPT6tvOn/Zd3/rV759LWA6a1c/aJ6vvb0dahPY+dO9n42UfV/cvtv3+3aWFkxYmk\n    tGVmqffPVx7atfR0Kej+dKHvp59fWMnEV6ee2etddWx9YcfqWx+/50hPd1fV7mi/\n    aazj7Lpvp+YXrut444NfG/ddaKebJo43w3T25gce7H6pybtmcbrx+z+eXf/j5vPD\n    c0+Ob+FTg4/e9c3XN66tHudH91ePbU8+vHqk3FD/VFvH/L2fHKx/vld/ea4t/uIf\n    =i2ny\n    -----END PGP MESSAGE-----\n\nFinally, I am proving my reddit account by posting it in [KeybaseProofs](https://www.reddit.com/r/KeybaseProofs).\n",
            "check_data_json": "{\"name\":\"reddit\",\"username\":\"oconnor663\"}",
            "remote_id": "t3_2q0kk1",
            "api_url": "https://www.reddit.com/r/KeybaseProofs/comments/2q0kk1/my_keybase_proof_redditoconnor663/.json",
            "human_url": "https://www.reddit.com/r/KeybaseProofs/comments/2q0kk1/my_keybase_proof_redditoconnor663/",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 4,
            "hard_fail_count": 0,
            "last_check": 1434485079,
            "last_success": 1434485079,
            "version": 9,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 4,
            "payload_hash": "00279b0704b01ac6171beef1a5718a8de52d5f0fd0ab63dc947dd6f111bb673f",
            "sig_id": "89e70d05df9c659d1ad8b0e2ead5790cc500d29345979af2e5807edefab5666f0f",
            "sig_id_short": "iecNBd-cZZ0a2LDi6tV5DMUA0pNFl5ry5YB-",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kk1sFVUYhu+9SlsKJARRiGJoJkIa017P/zmDC6J2URMEF9UoKNfzN+3YMnOZ\nub20aa/SVetCRRaUSMPKGCJlgQqYaGpK1DRFY4yp1aj8JBgEqQJBgwsTz1wxwsbJ\nLM6ceb73vO/3nT3L7sg15XeO9vYc6WxH+VMXVa5rfOuVIU/FZtDbMOTpvtBGlWwV\nyR3W2+D12kElU1sM45YoNrb4Ytpyk2nzqjZJwzhyFCjyIvZqbRmeFQdh1G2TchJm\nWh4VEhFqsBEKUQ1pwLTFQRBwKDhTTEhDBDIYOcmeOK3cdqpX1yyFxu3W+UeZeKTD\n8R11vr/+g0AFA04DRSkgzGAOGBAEUYkREBCADExtcjNSrOMoihPG6o532KS3z5aS\nOK7H1pUwgyCBPgIUceg8ybTHlVkCiK+sltkJDHJiqDBCIy10AJUvlQCWc0JsYAJo\nKEQGYUZ96rtXQGYA0pJqQYEwhHLgK2PdkktnllJOhAOkJpZQoaRUriAwWAPMfCNc\nBfZdiNTujGJnzrmjuJZ9J9VQ21vGpeMwyjr3f4krg+Vsc5dVpXTyH4WSCiPjZnbr\nUKFDb+8GZqDNswPlMLGlMCMoZwK4p80rJ7aaDYJIyjgVUGKoqM8C7PpENWEYB1Zb\nAwRgAUFcSc18yxGQVgdGAoS4r3yl/stInE/Z7STTsDuSlf7EerVX8mvuzOWbcg2L\nCtmtzTUvXv7vVd5Wbsq9e7JhazT+wIGZqOdc4zNdy/5sPDDxzfaBLYdHPi8Uvrs4\nvef15hfkj61zuSvrTo7WVnVVv7w8NNCyce10n/71s2eX/3D/iYYnL7//06FhOLl/\nYWV/x0O/zc9utBfODy+91vpS68d7fz/2xFV4+vRrl8pjbMnKa4eO8qOjm1bPXT0y\nvmK2c8Ru3l+899hSVug4c8+3xS/u27Tm/NjM3ZfOPpVfe2JbemPFxKmpwvW3v1qY\n2vfgDTC1+K/Hdg3Mr9+3W8y9sb755erh7Wc/fPiD4teLDrafWfLR7PePH9zCnnt1\n83vz7+xddVfjW+uuT/5SrM0Nf7L7XLWl1E7109Njx9+80PfpBH7+55k/VneOHPcW\n/gY=\n=2nAo\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419205271,\"hash\":\"e4049beca46d36174d58d8c2c8cf1b9ab80e7744efdf1d512d2365959959816d02ca5c8508d45709bde08d7a70655748d02ac4e458baab16dfd3c0369d88d439\",\"seqno\":114153},\"service\":{\"name\":\"coinbase\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205360,\"expire_in\":157680000,\"prev\":\"44a567581a31b596f31745c4633feced0806f427bac69e720aecfda02279b9bb\",\"seqno\":4,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1419205360,
            "etime": 1576885360,
            "rtime": null,
            "sig_status": 0,
            "prev": "44a567581a31b596f31745c4633feced0806f427bac69e720aecfda02279b9bb",
            "proof_id": "de92300787097c3f7ae36910",
            "proof_type": 5,
            "proof_text_check": "\n\nowF9kk1sFVUYhu+9SlsKJARRiGJoJkIa017P/zmDC6J2URMEF9UoKNfzN+3YMnOZ\nub20aa/SVetCRRaUSMPKGCJlgQqYaGpK1DRFY4yp1aj8JBgEqQJBgwsTz1wxwsbJ\nLM6ceb73vO/3nT3L7sg15XeO9vYc6WxH+VMXVa5rfOuVIU/FZtDbMOTpvtBGlWwV\nyR3W2+D12kElU1sM45YoNrb4Ytpyk2nzqjZJwzhyFCjyIvZqbRmeFQdh1G2TchJm\nWh4VEhFqsBEKUQ1pwLTFQRBwKDhTTEhDBDIYOcmeOK3cdqpX1yyFxu3W+UeZeKTD\n8R11vr/+g0AFA04DRSkgzGAOGBAEUYkREBCADExtcjNSrOMoihPG6o532KS3z5aS\nOK7H1pUwgyCBPgIUceg8ybTHlVkCiK+sltkJDHJiqDBCIy10AJUvlQCWc0JsYAJo\nKEQGYUZ96rtXQGYA0pJqQYEwhHLgK2PdkktnllJOhAOkJpZQoaRUriAwWAPMfCNc\nBfZdiNTujGJnzrmjuJZ9J9VQ21vGpeMwyjr3f4krg+Vsc5dVpXTyH4WSCiPjZnbr\nUKFDb+8GZqDNswPlMLGlMCMoZwK4p80rJ7aaDYJIyjgVUGKoqM8C7PpENWEYB1Zb\nAwRgAUFcSc18yxGQVgdGAoS4r3yl/stInE/Z7STTsDuSlf7EerVX8mvuzOWbcg2L\nCtmtzTUvXv7vVd5Wbsq9e7JhazT+wIGZqOdc4zNdy/5sPDDxzfaBLYdHPi8Uvrs4\nvef15hfkj61zuSvrTo7WVnVVv7w8NNCyce10n/71s2eX/3D/iYYnL7//06FhOLl/\nYWV/x0O/zc9utBfODy+91vpS68d7fz/2xFV4+vRrl8pjbMnKa4eO8qOjm1bPXT0y\nvmK2c8Ru3l+899hSVug4c8+3xS/u27Tm/NjM3ZfOPpVfe2JbemPFxKmpwvW3v1qY\n2vfgDTC1+K/Hdg3Mr9+3W8y9sb755erh7Wc/fPiD4teLDrafWfLR7PePH9zCnnt1\n83vz7+xddVfjW+uuT/5SrM0Nf7L7XLWl1E7109Njx9+80PfpBH7+55k/VneOHPcW\n/gY=\n",
            "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am oconnor663 on coinbase.\n  * I am oconnor663 on keybase.\n  * I have a public key whose fingerprint is 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32\n\nTo claim this, I am signing this object:\n\n` + "`" + `` + "`" + `` + "`" + `json\n{\n    \"body\": {\n        \"client\": {\n            \"name\": \"keybase.io node.js client\",\n            \"version\": \"0.7.3\"\n        },\n        \"key\": {\n            \"fingerprint\": \"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"71876B68AD482D32\",\n            \"uid\": \"41b1f75fb55046d370608425a3208100\",\n            \"username\": \"oconnor663\"\n        },\n        \"merkle_root\": {\n            \"ctime\": 1419205271,\n            \"hash\": \"e4049beca46d36174d58d8c2c8cf1b9ab80e7744efdf1d512d2365959959816d02ca5c8508d45709bde08d7a70655748d02ac4e458baab16dfd3c0369d88d439\",\n            \"seqno\": 114153\n        },\n        \"service\": {\n            \"name\": \"coinbase\",\n            \"username\": \"oconnor663\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1419205360,\n    \"expire_in\": 157680000,\n    \"prev\": \"44a567581a31b596f31745c4633feced0806f427bac69e720aecfda02279b9bb\",\n    \"seqno\": 4,\n    \"tag\": \"signature\"\n}\n` + "`" + `` + "`" + `` + "`" + `\n\nwith the PGP key whose fingerprint is 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32 (captured above as ` + "`" + `body.key.fingerprint` + "`" + `), yielding the PGP signature:\n\n` + "`" + `` + "`" + `` + "`" + `\n-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kk1sFVUYhu+9SlsKJARRiGJoJkIa017P/zmDC6J2URMEF9UoKNfzN+3YMnOZ\nub20aa/SVetCRRaUSMPKGCJlgQqYaGpK1DRFY4yp1aj8JBgEqQJBgwsTz1wxwsbJ\nLM6ceb73vO/3nT3L7sg15XeO9vYc6WxH+VMXVa5rfOuVIU/FZtDbMOTpvtBGlWwV\nyR3W2+D12kElU1sM45YoNrb4Ytpyk2nzqjZJwzhyFCjyIvZqbRmeFQdh1G2TchJm\nWh4VEhFqsBEKUQ1pwLTFQRBwKDhTTEhDBDIYOcmeOK3cdqpX1yyFxu3W+UeZeKTD\n8R11vr/+g0AFA04DRSkgzGAOGBAEUYkREBCADExtcjNSrOMoihPG6o532KS3z5aS\nOK7H1pUwgyCBPgIUceg8ybTHlVkCiK+sltkJDHJiqDBCIy10AJUvlQCWc0JsYAJo\nKEQGYUZ96rtXQGYA0pJqQYEwhHLgK2PdkktnllJOhAOkJpZQoaRUriAwWAPMfCNc\nBfZdiNTujGJnzrmjuJZ9J9VQ21vGpeMwyjr3f4krg+Vsc5dVpXTyH4WSCiPjZnbr\nUKFDb+8GZqDNswPlMLGlMCMoZwK4p80rJ7aaDYJIyjgVUGKoqM8C7PpENWEYB1Zb\nAwRgAUFcSc18yxGQVgdGAoS4r3yl/stInE/Z7STTsDuSlf7EerVX8mvuzOWbcg2L\nCtmtzTUvXv7vVd5Wbsq9e7JhazT+wIGZqOdc4zNdy/5sPDDxzfaBLYdHPi8Uvrs4\nvef15hfkj61zuSvrTo7WVnVVv7w8NNCyce10n/71s2eX/3D/iYYnL7//06FhOLl/\nYWV/x0O/zc9utBfODy+91vpS68d7fz/2xFV4+vRrl8pjbMnKa4eO8qOjm1bPXT0y\nvmK2c8Ru3l+899hSVug4c8+3xS/u27Tm/NjM3ZfOPpVfe2JbemPFxKmpwvW3v1qY\n2vfgDTC1+K/Hdg3Mr9+3W8y9sb755erh7Wc/fPiD4teLDrafWfLR7PePH9zCnnt1\n83vz7+xddVfjW+uuT/5SrM0Nf7L7XLWl1E7109Njx9+80PfpBH7+55k/VneOHPcW\n/gY=\n=2nAo\n-----END PGP MESSAGE-----\n\n` + "`" + `` + "`" + `` + "`" + `\n\nAnd finally, I am proving ownership of the coinbase account by posting this on coinbase.com\n\n### My publicly-auditable identity:\n\nhttps://keybase.io/oconnor663\n\n` + "`" + `` + "`" + `` + "`" + `\n",
            "check_data_json": "{\"name\":\"coinbase\",\"username\":\"oconnor663\"}",
            "remote_id": "oconnor663",
            "api_url": "https://coinbase.com/oconnor663/public-key",
            "human_url": "https://coinbase.com/oconnor663/public-key",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 192,
            "hard_fail_count": 0,
            "last_check": 1434481477,
            "last_success": 1434481477,
            "version": 3,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 5,
            "payload_hash": "d64c8d89811ef70c83ab58cf6350d4e847f1750cff3c468d7e8556c65073e09a",
            "sig_id": "8b4bc6a736a2a651763300384d54e72daa0ee3f9464ae98fd2df4f06f93d59f60f",
            "sig_id_short": "i0vGpzaiplF2MwA4TVTnLaoO4_lGSumP0t9P",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9km2IVFUYx2e23ZwEJRctBYPda2DlOJxz7nm5o/lBd3uxkMi2/ZDYdF5nb7ve\nO957d7dRBmwNRC3ayFIye1nKEsoUMTIUjcoWAwk0tk39YJCUFpZIBqtu5w4b6ZfO\np/Pye/7n+T/PMzjllkwuu3pDd9fHD89H2W9/FZmOrStqax0RqqqzYK0je3wdJOku\n4Ku0s8Dp1lXBY13ww5YgVLrwbNwyweSdPh3FfhhYChRYwXVq+RRPg40flHVUifxU\nyyEeR5goV3kCEQmJoVK7xhgGPUYF9bjCHlIuspJdYZzc9KtT1yz5yt7W+SXUW9xu\n+fY631t/wFBAw4gRhABMlcsABR5GhLsIeBCAFIx1NGEplGEQhBGl9YxX6ai7R5ei\nMKzblomfQhDDIgIEQ2Jz4nGXDdPY0xoY7hFBsGcwABxZI0QVFRESY44MAspwyj1F\nDVSEYcklUYBpVwpPmSIVwGhoFJWQy6Isaq05QBwSz37kcokos89WESlsMFZS2HjK\nCGMQedyaiPXqILTJ2ewIqaXnqM+X+oZ2dXHZbY3q/vj/PCfVSnrZr8Xu0oRGSfiB\nsl27sa3QojfXgwCYd/RzFT/SJT8lCKMesCvvVCLdl04CQKwoAANYAOuSQgaF1gZy\nYrtnK6MJUsQAowAX1FWyiJlKywWhEJS55j+XtvIJL1vJ2C8HPOmNtFPbmJ3dmMnm\nMrc2NaRzm5l82+3/DnPSlstsPvyGOXpg194r5dnN2zrfGg7K58Ym/Z1cCzq+mffj\nH51fPrX7euflphkHK18cGbq/9Obcx7KtT4+NrF/y1V1t4wfId4s+jB4d/awpM6V5\nXUNruP3M4uL++Pqex38aOPLzlgv4oca2X4ZWztq7+fjyma8fL7RefHfr9E/m9b49\nkKBTi1ZmT48NLl1zbOelXYeH57y4b+bUu0/uG3llzed//p7cM+3c90eP+cuevPf9\n4Q9O/fbIlcYTGShySX9nw9yhQs+lOwfveH75xa9HWs4/8d4zHevzB18YLu+c9FHu\ntWrrnJ6XPt30wH3b/2qunR24Ot5iTrw6Om1F+zuj29pPL1s469CDS3dUR9tGftgS\nnnx5HHf8Aw==\n=B0oP\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419205415,\"hash\":\"e48ee0fa85b548f400a26ce5d9d5bc44a2f20dfa6a8d6f1d574cac5d07e3cb8df96b0fe1fd6c1ac9c9eeea02a1585413ac267e1fe5d2d4f44dcbac567577128a\",\"seqno\":114155},\"service\":{\"name\":\"hackernews\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205501,\"expire_in\":157680000,\"prev\":\"00279b0704b01ac6171beef1a5718a8de52d5f0fd0ab63dc947dd6f111bb673f\",\"seqno\":5,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1419205501,
            "etime": 1576885501,
            "rtime": null,
            "sig_status": 0,
            "prev": "00279b0704b01ac6171beef1a5718a8de52d5f0fd0ab63dc947dd6f111bb673f",
            "proof_id": "ff25378bd64cc2b0d2942f10",
            "proof_type": 6,
            "proof_text_check": "i0vGpzaiplF2MwA4TVTnLaoO4_lGSumP0t9PBvk9WfY",
            "proof_text_full": "[ my public key: https://keybase.io/oconnor663; my proof: https://keybase.io/oconnor663/sigs/i0vGpzaiplF2MwA4TVTnLaoO4_lGSumP0t9PBvk9WfY ]\n",
            "check_data_json": "{\"name\":\"hackernews\",\"username\":\"oconnor663\"}",
            "remote_id": "oconnor663",
            "api_url": "https://hacker-news.firebaseio.com/v0/user/oconnor663/about.json",
            "human_url": "https://news.ycombinator.com/user?id=oconnor663",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 210,
            "hard_fail_count": 0,
            "last_check": 1434471730,
            "last_success": 1434471730,
            "version": 8,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 6,
            "payload_hash": "37ad41cf064989dfd849a0af178aa066dd396b3452433a9883a69e88e35e75b8",
            "sig_id": "5941324c9d6c5c830c832f44494cbc73af0cfaff6e3c9e69ae88bbda45e72c3e0f",
            "sig_id_short": "WUEyTJ1sXIMMgy9ESUy8c68M-v9uPJ5proi7",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v1.1.6\nComment: https://keybase.io/crypto\n\nyMHMAnicO8LLzMDFWNienbHWQ9eI8fSBXaUMIdNj86qVkvJTKpWsqpWyU8FUWmZe\nempRQVFmXomSlZKpRaKRiWmKcYpFkpFpsqFpmllyqnFaWpq5oYW5WZKZRWKKiYVR\nirGRko5SRn4xSAfQmKTE4lS9zHygGJATn5kCFMWivhQsYWKYZJhmbpqWZGpqYGKW\nYmxuYGZgYWJkmmhsZGBhaGAAUlicWpSXmJsKVJ2fnJ+Xl19kZmasVKujVFJZABIs\nT02KByopy0xOjU/KzEsBegCoqyy1qDgzP0/JyhCoMrkkE6Tf0MTQ0sjA1NLSQkcp\ntaIgsyg1PhOkwtTczMIACHSUCopSy4BGppiZJFukWFhaGBqmppkbJFsYJyaZWiSn\nmRmbGqSYpFqYmKcZmpsaJKelGSebmFmkmKdamJqaJZuZGpgbpxpYJgLtL04tzMtX\nsjIDOjMxHWhkcWZ6XmJJaVGqUu2hAzEsDIxcDGysTKAoYODiFIBFjLMKB8OJ2qe/\nZr3gLy6SvvjN507UXm6uyy+2Jr2S2Jmconh4rcmchIcyYbsTGpuXb+XbvvHYnyK+\n6Qr1z9Vku4vzZgazFKw7pivUk3NdcXWEterdmca6ynKPhS78UWX5l2veJ7FycfPu\nSJfvTz3uTQzgft/3eM+CH29FHyfOy57Z1hf5/c1qt8XH93Umxs0/tca7+0n5oj/t\ni953tFf9d/bvu3HXTeGx5vIL6zdL+3qzfSu4nx0p9tZG2MhAte2tfMaFD6sZOP+r\nRzwPDotbUiO+3PQGSzGf99aK4Iebbk4XfF4WK3TycN/VWaX+W2LSCgzXbXlwiOu0\nFM+WnNRVXZHdM2cHnpxq+eXelcO9MplbLTUA7nMASQ==\n=dQNg\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205998,\"expire_in\":157680000,\"prev\":\"d64c8d89811ef70c83ab58cf6350d4e847f1750cff3c468d7e8556c65073e09a\",\"seqno\":6,\"tag\":\"signature\"}",
            "sig_type": 1,
            "ctime": 1419205998,
            "etime": 1576885998,
            "rtime": null,
            "sig_status": 0,
            "prev": "d64c8d89811ef70c83ab58cf6350d4e847f1750cff3c468d7e8556c65073e09a",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 7,
            "payload_hash": "2d8dfe977e6d359a03430c5cf22993d16438b1e10546b1b247055abc9cad3edf",
            "sig_id": "bd1445bcc75e2d6ed5031c245306863fbaddf8673e674903fe5ac7c1414136a90f",
            "sig_id_short": "vRRFvMdeLW7VAxwkUwaGP7rd-Gc-Z0kD_lrH",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowFdkl1oHFUUxzdNTFI1WKEIQcU4FTF1E++dO/djAlYiGwzBwLakpU0iy/1MJh8z\n25lNNMZVMMYiRoiGFlpt6IfUNqUP2qJQqAj1QVofLKgPpU0fpAShopJIo1XqnSWK\ndV5mOPPjd//3nDPbUJ2pr9q9Z3jwVGeLW3XxR5Hp2b94e9IRkZpw2iYdORLosJR+\nhXxUO23OsJ4QPNGtQdQURkq3DiVNa0zWGddxEkShpUArbUVOOevIeKJYiuRYHOtQ\nVoxcqVgniYVgV0deIt0ebMW93dtCd+jFwZ5Sx3bk+uM+kjkPdVhnaaKYHiuCkoyC\nMFXaBKnHBOGAjotxkMZzMOOuhxVSTLhYQmyI1MgYQyGjRBDGlcdchVxrHIyS0h0X\ncSrOQqBstcI/S1h7zvK5Cj9W+eFBAQ3FRmAMPKIQBQQwz8UcuYBBAFIw0fFalyIZ\nhWEUE1JpwqiOh0d0IY6iSidlKUgh6EHfauzLZuLJYNo2ZrRA1FMeMEYyV1BglJBA\nu8rFyBYo1QBITJXxPGIMI4hSSY3CwIeSA0WQwD4hRkHGMZGA23zWRSQiHqXGYE8Y\nbL22IJFluCKaad9XkGBNtZP9osZJ9O4wsvEgxswv/zuB/03yP9OG6ZzvuJOL/Kyj\nXyoGsS4EKYEpYcA+WacY63FrQ9ROBEoDiOczXxnFPJ8DbiC1oQAhSiGfCORh10OI\n+4whTnzNmEY2JxbMnr+Wk9qEfMAqk2Ag5KWxWDvlt6oeqslU1Wdq71qXrnPm7vUb\n/tnxh6vrbtWWy6XtP8MDr7xTx/uvwtX7Z3a0Ty18dG3nwqGVB/sy1+tOrLuc25J9\n/M39BdEc7+t9ufYFlv9w58hfPcstS/7Sei7nZ16DF9798/zlngfsNu+6ZzrbfOP0\n9PkF58xXb8//seWDba9ePF68WX6jb/rCvb8ttXS/3/jknrlre59o/P7o4aN9lz7d\nHG98rHDwyqaVTb9c/Xbfro+f75KP/PB07uu5hmcefS65cXOme1asXL/vm8bFs8dX\nDx+Yqu89NtwwuVw32bn597nmM+VCvvG96qm9Q8nW0defaspnv5v7ZMftYLHfPXf6\ny18b8h1G9M/PfvbTQVF/5MqxkVvLhc7PazZeOrnQdgisgq6/AQ==\n=5wxk\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"cryptocurrency\":{\"address\":\"1JEPc3eAiQ5ZMRn2jwhTtEU329v93cD43E\",\"type\":\"bitcoin\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419370141,\"hash\":\"08feb374d40ffc82b70fdbc0e2d253c8277e00c57df446ff86377c7fd5091ca0d63b5966fd18a56c0a5a340f6c36477ff54bf582b6c3c318aad6e8e99d165e7e\",\"seqno\":115589},\"type\":\"cryptocurrency\",\"version\":1},\"ctime\":1419370239,\"expire_in\":157680000,\"prev\":\"37ad41cf064989dfd849a0af178aa066dd396b3452433a9883a69e88e35e75b8\",\"seqno\":7,\"tag\":\"signature\"}",
            "sig_type": 6,
            "ctime": 1419370239,
            "etime": 1577050239,
            "rtime": null,
            "sig_status": 0,
            "prev": "37ad41cf064989dfd849a0af178aa066dd396b3452433a9883a69e88e35e75b8",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 8,
            "payload_hash": "57c95524e2fdb546201af092292ae9d27a055a8934d81a8f8488eec777441d1e",
            "sig_id": "f6689d2fa3ff3b2fc130fcadb73e7de28ca55b3c59b493de45f9c61d980b5d220f",
            "sig_id_short": "9midL6P_Oy_BMPyttz594oylWzxZtJPeRfnG",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMPVAnicdVRriFVVGJ2Z7OGIYpJQoBKHULAx9/uhRFmBIUExzWiadvn2a+5xZu4Z\n7znXZpCh6EcpRknmg8wyEhmFgdQ/EQSTjJhFREHSixBMS4gIS1Aiap87kzpQ59e+\nZ5+71vrW961vfOZNLe2tm17qrY4+uoS0fjre32jp2n9wfEtiMjeULNuS2L7U14ry\nVIN+nyxLev2Qgdzfl2Z3P+tN0pFs9vU8zWrxylmNLGfeSaqUos5ggICYFByhIL30\nhlPALiTDHSVMCRrSWo+vD9TTkiPhCgjjjjplCLeYB2E9DSFIrKQwQoFjijhKIms1\ny4spapImZiV18e1/fN/bvEAYYeoRp4IIcIQKJbySJIDkRjDgRiGKCdcSGW5BgjTS\naWWpjj8UidcYCIII12jCMWxwkDwYzhETjkokkGKEAyVIYYTKD3Nfn3Qus1mtltWF\noKUBRR1sb2lB1J/avDylrnLNTNqR9EFexHoqtgrRpGQZZlgzoSPXFNiNvVme9w3V\nStCmquiiwEgSpCEo7oTVlhBttGIaO9ZUNel+adjUDoCLD3UoFi6Mk0R6BSGaxYAA\nJyYIDt5H46Y4qoXQ2DOJqXDeaBY4ozggIYLmIBQTgRJpo4GeMBdw7CoHA9hC/A/X\nzkdHo/a6788KXxmoZ1mIdjwdZ69I+6/XzTjvSGyjXo+sWHNPhMLEK6SARS7GvA4o\ncjMfOEGSUmcDcCuIZAp5Rpx3XALCJZaP8v0kOJcSswnwSQWlK00VzfmvettbcVBA\nZWNeNuZaEHrSotowyf/2oglRKYYGfLObeQFFyRdvcr+plsVjPKU9ExOLqaRWO2kt\nt8wBozErgLShAjtHLA9Ee8kRRVh5jJ3FKGCMGFIxUIYYikIygTbBR4Y3NGkqBaR9\npeYBGOrLwFWqkFcjnRAids0aqaWmJlCOZcyblwikhtJuMPFMtI4Z4LGHcSyAQqBg\nsHbO8ORaEeSGIhh45wXyQkpJNMVGeqGIDlRT6wL3DAclwAYvnQXr4pCCxiw48CTm\nisaakuEyGc0SJgNyw34prbthJASWcbXERg4OpHVfSWvNZgqF4lO67zdHEOKUC17L\nqMRRrgFRRuOOsiEmQlOHBaPKYI8RZ8JEJ5lEsV4TIwOOeheuF6qiMOiJkLHcGhSN\nuk+GT4ytn9bS2t5yy81t5dZsaZ8++99dutvd+vfqHSu2LX5415d3kjNzVp6ffezq\nocZrzyx4e+v+eR+8PPPcqSOd/o87BjeN/3Xpu8PvVc3H73e+sKHt3PjJ4qnuGc9n\ne3etubTnwY/2zOrad1z2v37xnauPLThxYIa7evL+Kztnbb/3q4Xzz46ceujyyZ2X\nD7W17lytty1ad3r7+cE3vnl8JF/Z/cSZhXNH0wvTalnl21+6nrxr2x74fu46Li7f\nvqrozBf/9EX3xcaVH7d2fP57/4YDoQuvfW70zZGtvz1ycN7RkWM7zp7+dfT4J0fC\n0uVn1/9wbN6fe98dGT+84nx7embtqle+vqAPHl236J4H5nfMectNH7tt6e4FY2PJ\n4L41n21esvziq9K92KY6Pyx+/ges9xbZ\n=2qFk\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"dc90c54ed738883db1aaf0476500f7e7eb53a1df\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":3,\"last_id_change\":1419469550,\"username\":\"jkosslyn\"},\"id\":\"58a6107209af85d6c9c229b98491d400\",\"key\":{\"key_fingerprint\":\"adddd3d07b76bd727e8af6264a2a52bf65aeed23\",\"kid\":\"010196691e47136deb94f5431f066f95a6846f327c782e24df1f6c5aba1cae4759de0a\"},\"remote_proofs\":[{\"ctime\":1419469455,\"curr\":\"195e26812e808a4f0644e9f094f4ef520733dcfa5c627480e42ded57a011946e\",\"etime\":1577149455,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"jkosslyn\"},\"proof_type\":3,\"state\":1},\"seqno\":1,\"sig_id\":\"1373c9d7cc5c4da43eb5a09b361dd2c5f29e7503018e11dc10f1104083a1b2b30f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"666196cb79793bf35175c1e70a79a4194ab70a299a7a55a6adda3af3ab19ddb5\",\"seqno\":2,\"sig_id\":\"4aede60e67772931b7e6829f393cdf5e41f86acfe7dcacd610a914fdae21a2310f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1419617476,\"expire_in\":157680000,\"prev\":\"2d8dfe977e6d359a03430c5cf22993d16438b1e10546b1b247055abc9cad3edf\",\"seqno\":8,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1419617476,
            "etime": 1577297476,
            "rtime": null,
            "sig_status": 0,
            "prev": "2d8dfe977e6d359a03430c5cf22993d16438b1e10546b1b247055abc9cad3edf",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 9,
            "payload_hash": "d70f647a155416bd94ad669254e5513aa667c03ed55f047f81cd6fc9bc3ed89c",
            "sig_id": "7ec6ea814573f2d7db7947262a624e4b0990ae5cc48feff1c8572b69f60b769e0f",
            "sig_id_short": "fsbqgUVz8tfbeUcmKmJOSwmQrlzEj-_xyFcr",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMSiAnicrVVbiFVVGD6T3RyNmUqSLqZtUYxGW/fLeRgDMSytqCQIJ07rOrOdyznu\ns0edxklfNMtECroR3R5KIiQJKy2K0hgljXpworCUsIcghHkoRLLs38dxmiEfeuht\nrb3W/v7//77v/9cXV00pNTeteaK7a9eyhaTp8PDS/tLK3Sv2DSa26geS8mDietLQ\nlxerPtMbknLSHQasqYdFaXXOumCTtmRtyOpptQ+OCCYOReWFwcypwG1kHgtBuVOe\nWaGc5I4F5ZKhtgKmAI1pX2fIallaxEi4MoRxT72yhDvMo3CBxhglVlIAgPFMEU8J\nRO2q1vNJ2SQNzErq4etF7nc3DhBGmAbEqSDCeEKFEkFJEo3kVjDDrUIUE64lstwZ\naaSVXitHNWwUgWNsCDIA19+AY9jiKHm0nCMmPJVIIMUIN5QghREqLtZDNsZc1VX7\n+qoZEFIQkGfGdRcUQP6pqxer1FfGycSyLekx9RwKqrguAyzBN4Y101hRPAm3nmep\nLSAbOYVIAtNa4OC9MVIizSzjwRcyRNfIaYz7gq7J/MsYHFMMu4C9IMxJLEBEawwV\nmgmBuLMkeGcm8cmVlVJoHaWlhmjuGQtWaWohoCc8aiCYwf+KBsGIDJgbuAFkI0KU\nlhr4hNyz0FvNQ6WWVasRyFgFzsvT3kbRSHHBIZW2xPVnGUTVInoZoieE+Yg8kpTJ\naDhxyjjLDbJBw4ZFLpiGpRXUe4WZgoysB1dB+mEMnAuo8zx4LQtrAdxiHyiVETGG\nPC7K9TGGwIKWQAgEYIobISkKihZyO8miYlQSbzTHTCbjtRT8Nupp9FFXcN0Vb3JT\nWV0vBB5sWPiCNXq8q2Zhkav2JkUm1RzM0gMHXXleKxf8NIAq+UCtyBoh1Aaym7zY\nwGE9rOmrJmUOq7TzQgsIo0B7QaGnQHzDDZHYYWMDEgoJr0kAj2ojsNZgcWMl904G\nY72PoB8qWCrQzockQ22TFBFMCo7HFVFG6+A5U0Y55r2nnHAFvSuUpNQJb6F7qCch\nYm09olw6473TRmkvYmx01ARFKDkPPqbI/yD3f1RkTI3ONO/qt8nFm2yiEPRiKogJ\nKuDIHMwT6zTCzikwkpKehUAU9z5gQqR2gkQsEMxL0EVZ6D4rJOcSBgoT/1LhkUaY\nSm7SniLhmhnoqRpf6TL1rkIGbkA7G3GA/mPRIGUdjgp7bS3DVGHFsVYIBYcxtLfg\nIRiNYGRHwqTyxZwcKwKjCVUI6GwfWSGiRRRkhSFKvBIgeQzSUOMcgYkZJXS0DdYE\nypSMMDItPAXWFV4aKiZeo4axwTfh3Si4G7cWDE6GMAH1w/pamoVKWtzgEjzbcP2Y\nJ8BBmnPCAoneciYIwiYiTYgmJmhPpEGcm4IFsIJR0KFKheCklIxhj8M/lWpIzHQW\nCqedfSbvz0IytP+zjktLTc2lyy+7pHgNS81TWy+8kdPeveLcHV89fe9o86nHsk8O\nr/781NeHdz3XvX354jmbPt6zYNuts7a+cfWh0y2lqvtwSm3JsH372S1Tl3fcvm7D\nxv3zjy75+eDJcph/6pmZ59666Zf81yU/tR1o//73v+4cXf5N6+md82Zsapr96Ob3\nbmg5eeTmE9c//uLw8dbnj7llO6d/ef+ylU/evbXlxtKxo/v63xxomdYxPb3unhcO\nnX3/24fnvnZ88we37B356OVzZ+c9sG36gbL59GD3suGlZ3pqw90L4ow/XmnvGN14\n4rYNO3e80zMyJz515Mot5dWjuzpeXbG79aXy3M6WWav2nhzh4s8HL9+x59Dxa9fe\n1b69fYDNfH3qD/sPDK2/r7L4msEzv/3YMfuh0ncLR/4GukqcnQ==\n=TY+X\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":17,\"last_id_change\":1419491831,\"username\":\"strib\"},\"id\":\"ef2e49961eddaa77094b45ed635cfc00\",\"key\":{\"key_fingerprint\":\"7fec4841ce1d624c7165bfbaa36946605cb2edca\",\"kid\":\"010158b77699f7b3a295d44eb893b35cd25f9872469483e6427e15a44e4a502289790a\"},\"remote_proofs\":[{\"ctime\":1408565716,\"curr\":\"96fd7efd224df0d07347fa52c8acb5a0be952c4f5649e95b63dd8148b35bd15f\",\"etime\":1566245716,\"prev\":\"b1de337f0440d12edcdffee4e974c7acb485a6730e830608c74f84372da95147\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"oldcore.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":5,\"sig_id\":\"716a8094638a2d63a5a271c1abe06806d92e0819a619998cab75dc7eabddfeb80f\",\"sig_type\":2},{\"ctime\":1408647651,\"curr\":\"8a99ed548a8c4ddd35258ce368733c6db7823d2ef19bd0357caddc9a89d6ff0a\",\"etime\":1566327651,\"prev\":\"96fd7efd224df0d07347fa52c8acb5a0be952c4f5649e95b63dd8148b35bd15f\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"strib\"},\"proof_type\":3,\"state\":1},\"seqno\":6,\"sig_id\":\"1f4c125bc901cc8ffe87d4ee285dde12279c62f160c8e0688b841b67557046460f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"85afebbf1e8934fa08bc1f81d9bb413818519800ec1124c65eea900f8f2478d2\",\"seqno\":10,\"sig_id\":\"67b3df4b782b03ce30132d869edfe7a3acc270bf7022bebae3487fca7bd6abc80f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1420840121,\"expire_in\":157680000,\"prev\":\"57c95524e2fdb546201af092292ae9d27a055a8934d81a8f8488eec777441d1e\",\"seqno\":9,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1420840121,
            "etime": 1578520121,
            "rtime": null,
            "sig_status": 0,
            "prev": "57c95524e2fdb546201af092292ae9d27a055a8934d81a8f8488eec777441d1e",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 10,
            "payload_hash": "251d1c8b830c3c4b131a182d27188712bf85a7026755bd0ad16ec2f58c880a07",
            "sig_id": "2c449b4270eeb6094bd48557866343e89dde1983a1694a5625b8076ed793fbc30f",
            "sig_id_short": "LESbQnDutglL1IVXhmND6J3eGYOhaUpWJbgH",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n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\n/fOhx5fP2ffg1w6ed+015rc3fOvBp++87PDF/wN3bEWj\n=S3O8\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":103,\"last_id_change\":1419928822,\"username\":\"max\"},\"id\":\"dbb165b7879fe7b1174df73bed0b9500\",\"key\":{\"key_fingerprint\":\"8efbe2e4dd56b35273634e8f6052b2ad31a6631c\",\"kid\":\"01013ef90b4c4e62121d12a51d18569b57996002c8bdccc9b2740935c9e4a07d20b40a\"},\"remote_proofs\":[{\"ctime\":1391720984,\"curr\":\"35782a3561e40a5e7217500bd67b5aadac5d7f6f9594dcd5d15cd06879da6e19\",\"etime\":1549400984,\"prev\":\"0146be7c3669a5f5da89e9908e8bb5f1fdb9fcfd6d2d66a51664f1075a7637b1\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"maxtaco\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"9f731fa7c75b64e1c9d70300b4383196a8fb432294a4308f8d7a379376a0b1900f\",\"sig_type\":2},{\"ctime\":1392252388,\"curr\":\"c610bab1245d05ac6fa95eda4d008d67c3732bd441dcc5c74f48fc8f8ba44c31\",\"etime\":1549932388,\"prev\":\"35782a3561e40a5e7217500bd67b5aadac5d7f6f9594dcd5d15cd06879da6e19\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"maxtaco\"},\"proof_type\":2,\"state\":1},\"seqno\":4,\"sig_id\":\"667062cc7300f11292079f7c4da0ed8e53e52ca484bb55ae693e7d0aa614322a0f\",\"sig_type\":2},{\"ctime\":1395867108,\"curr\":\"7d72a9f98ca4739e3939c89339d3f67966a7160da9df6215acb6a27c4a1ea292\",\"etime\":1553547108,\"prev\":\"2df5de847a546d662d3dda502aa283328f477430bfc24ce0c681ca68c4d4a200\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"keybase.io\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":24,\"sig_id\":\"3517ddd1a5c9d708f21634d126814647129a64086a967daa5017e7966b89c9820f\",\"sig_type\":2},{\"ctime\":1395868002,\"curr\":\"ea61172238d18212d53a71316dc09e8e48f2e654490262872018d09854ab26e0\",\"etime\":1553548002,\"prev\":\"7d72a9f98ca4739e3939c89339d3f67966a7160da9df6215acb6a27c4a1ea292\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"oneshallpass.com\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":25,\"sig_id\":\"93734bd81d3b50d25401d7451a136e171757990d35110b585b729dbca2019d050f\",\"sig_type\":2},{\"ctime\":1397071049,\"curr\":\"eec8f9d3a754e793301b108ddb548dc46bbefaaaec189306b169a3ef71f22abd\",\"etime\":1554751049,\"prev\":\"b6168f3a49898b4aba5c5fb162c2b5b6df87407bca7649feba5364751e693c13\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"oneshallpass.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":34,\"sig_id\":\"0e577a1475085a07ad10663400de1cd7c321d2349cf2446de112e2f2f51a928b0f\",\"sig_type\":2},{\"ctime\":1397071368,\"curr\":\"ad3efd2c8ff323f96841b482bc3cfe92195bc72fe318ce0b95cf66ee3160aa55\",\"etime\":1554751368,\"prev\":\"eec8f9d3a754e793301b108ddb548dc46bbefaaaec189306b169a3ef71f22abd\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"maxk.org\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":35,\"sig_id\":\"baacd979d2ebe325482d2921ea72df72dbf9a31aeaa69f2b14b1eccfed31c72e0f\",\"sig_type\":2},{\"ctime\":1401977613,\"curr\":\"467a473af5b7d548f1e1a1d199398f7fe3a9ea40e4c615d261e6b44a56b96642\",\"etime\":1559657613,\"prev\":\"4111d9457743da3ad15826242bac5db8bb5ebf77d6de71a41a23ecfc4ed4bcf9\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"nutflex.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":49,\"sig_id\":\"4b4c67c83b4995aae5d4a5dde916fb35762551e67e35d4d4f1be99e2309238600f\",\"sig_type\":2},{\"ctime\":1407166100,\"curr\":\"acc5ffcf17cd722b5844b24564e2f644d265e010047bfdd378919be8f0603ba3\",\"etime\":1564846100,\"prev\":\"b328cdb4904ebcc196a49c67355f3ee9239bd735d7246e90cf20c7e7307e706b\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"maxtaco\"},\"proof_type\":4,\"state\":1},\"seqno\":55,\"sig_id\":\"095d87d153e6590c2d6c032752d3c43485a1f64e61b99d58f989f6da28b2a8320f\",\"sig_type\":2},{\"ctime\":1407167610,\"curr\":\"8d906bd4d0bb2d667f5c8191d6e7616f1de1e69e950f1f2e6ea181316d8d6dbd\",\"etime\":1564847610,\"prev\":\"acc5ffcf17cd722b5844b24564e2f644d265e010047bfdd378919be8f0603ba3\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"maxtaco\"},\"proof_type\":5,\"state\":1},\"seqno\":56,\"sig_id\":\"d0f9aaff73dc67f40fb6ebb6fdf473e11de43f11ae64a5fbffb00218914e19af0f\",\"sig_type\":2},{\"ctime\":1407186953,\"curr\":\"c9d4ca19adac4680957b2ce37d38280a1a32d2a7dfe606ae7236a9c982e2c75e\",\"etime\":1564866953,\"prev\":\"8d906bd4d0bb2d667f5c8191d6e7616f1de1e69e950f1f2e6ea181316d8d6dbd\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"maxtaco\"},\"proof_type\":6,\"state\":1},\"seqno\":57,\"sig_id\":\"a9fb80f43bed684a40e48de1e365e7ace765a3309819f9f79cdb27e3dd9802f50f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"040933ff11e2a44476ef0395989349a917a0aad723e41585d1087330c4365c3e\",\"seqno\":86,\"sig_id\":\"46ec4cb327088e808d068ab8236f39dd38a6cd840e229d5f8a5d65b4c7d6a58d0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1420840192,\"expire_in\":157680000,\"prev\":\"d70f647a155416bd94ad669254e5513aa667c03ed55f047f81cd6fc9bc3ed89c\",\"seqno\":10,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1420840192,
            "etime": 1578520192,
            "rtime": null,
            "sig_status": 0,
            "prev": "d70f647a155416bd94ad669254e5513aa667c03ed55f047f81cd6fc9bc3ed89c",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 11,
            "payload_hash": "a6808c59df1a6406a961e81b7038eb132c9ab04eb827ee844663ef5a7e651e90",
            "sig_id": "cf90e4629d273b677662b511269fff5756714d2889278c4297aebe9d723ac8240f",
            "sig_id_short": "z5DkYp0nO2d2YrURJp__V1ZxTSiJJ4xCl66-",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n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\nxl07Gran/Y2t75XvfPM3px744zLn55bn/gsCjHb2\n=ofws\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":167,\"last_id_change\":1418935207,\"username\":\"chris\"},\"id\":\"23260c2ce19420f97b58d7d95b68ca00\",\"key\":{\"key_fingerprint\":\"94aa3a5bdbd40ea549cabaf9fbc07d6a97016cb3\",\"kid\":\"0101d49265e97c636dbffa62070f29e07bdee44aca90b0eaa1645cc29fe5320f8a6f0a\"},\"remote_proofs\":[{\"ctime\":1391738324,\"curr\":\"461262ff766cf5da5eb40e5db29128a453ffb939840a23c0ffcb95073c888302\",\"etime\":1549418324,\"prev\":\"34152df8ec04aa33ad1bdbfd4927ed4d64d1be2ed3c4060043f313041a3722cf\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"malgorithms\"},\"proof_type\":3,\"state\":1},\"seqno\":4,\"sig_id\":\"3e1e20a1a11f4f0ee4ea4470b694fc115f928ce0b1f9aa2951a65081da8340820f\",\"sig_type\":2},{\"ctime\":1392240892,\"curr\":\"4396ec2c10ba4c4e1d931af42ed4d65ecea9f57a858fd44e86f2950071ef871f\",\"etime\":1549920892,\"prev\":\"d2284c9ab38c28650ccf4d1feb7af9f4ee46ee760ca45711a5e3afe008e518c3\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"malgorithms\"},\"proof_type\":2,\"state\":1},\"seqno\":7,\"sig_id\":\"0d9f6b71c043f2ef80b6dea44338471edc3ef76e2cee236dd397412ad9631aa90f\",\"sig_type\":2},{\"ctime\":1395868048,\"curr\":\"45c6893a87e18f27502266c08f0d0118f5ecd3fd0bd52432b18cddbd3f6520a8\",\"etime\":1553548048,\"prev\":\"26662070e4cd5345a302af7d513bc8d08a9a21cae783c6ad18f0113b039211f5\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"keybase.io\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":82,\"sig_id\":\"58ff6251889edccbb54495ea3f7d33f57c2d18e114dda887aeab6e06552aed780f\",\"sig_type\":2},{\"ctime\":1407170371,\"curr\":\"9fe1cf15276a8c34ebe03eae24e90f4ea44e6c6afd69a96bd6a85bf1ec025780\",\"etime\":1564850371,\"prev\":\"fe90114a30fcf13d0b9af77cb4291c1d550462fa567b0ecffdf436840f0fd8da\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"bitcoyne\"},\"proof_type\":5,\"state\":1},\"seqno\":202,\"sig_id\":\"c6d2c827fd7292ae36a359fef3ad04736ff373901b3db037af773ce13db7fab90f\",\"sig_type\":2},{\"ctime\":1407767238,\"curr\":\"510fb8ebf6ea77aa220f0a4cd7d0be212f9b294161f0cb2b43b6be024aa9ec67\",\"etime\":1565447238,\"prev\":\"044e1a2cc9d956c4780d4eb7fcaae9dfdf560eb9d5095ad653d8b37f63172c7f\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"malgorithms\"},\"proof_type\":6,\"state\":1},\"seqno\":209,\"sig_id\":\"c60cae4ab2b165fd15960433b0af48c66d37f3940e56a8d3b4955991947cb3f00f\",\"sig_type\":2},{\"ctime\":1408375965,\"curr\":\"57a78d3f45c749eab6a6b2796db103c6887ccbd7827a27e3f33cf3f0950cad8a\",\"etime\":1566055965,\"prev\":\"94e9a25ab3e551d43ff700e94be3cc446785ed4f44d83d93d00fb3ddc601070e\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"chriscoyne.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":221,\"sig_id\":\"dbf530c689d2f3af60c5b11341d5eb2ad229995d6ef39dfd1661962d088a75ec0f\",\"sig_type\":2},{\"ctime\":1408376008,\"curr\":\"ba697b9daa2617a80f53d94a8204a6db5aaf2215faf1feb4a43d2a09a95c6a7c\",\"etime\":1566056008,\"prev\":\"57a78d3f45c749eab6a6b2796db103c6887ccbd7827a27e3f33cf3f0950cad8a\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"chriscoyne.com\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":222,\"sig_id\":\"7dbc723fc7c0abfe761af6729fb27603ef9be58718153329ed2d4ddd8d6ddb910f\",\"sig_type\":2},{\"ctime\":1414608934,\"curr\":\"2f7e338d5d1915975611c4d99f225986f7978c2b0a89f7efed973cabc92b6e0b\",\"etime\":1572288934,\"prev\":\"b7036840be4000a1f5a006bc8b164c535300507c103651f389290e1ac42885e9\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"malgorithms\"},\"proof_type\":4,\"state\":1},\"seqno\":255,\"sig_id\":\"1a8fa75aa365dd96c95fd229722668dfb0026973d89e489b256d87ee06a4bd670f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"d24c186ca79e522e96c4d10f407646b9c1b5a2beb35fbaa03f2d070638af65b2\",\"seqno\":299,\"sig_id\":\"677463e4cd27e7cc4382f20aa7f21882f694feebdd4ddee623786b1cd21083940f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1421363548,\"expire_in\":157680000,\"prev\":\"251d1c8b830c3c4b131a182d27188712bf85a7026755bd0ad16ec2f58c880a07\",\"seqno\":11,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1421363548,
            "etime": 1579043548,
            "rtime": null,
            "sig_status": 0,
            "prev": "251d1c8b830c3c4b131a182d27188712bf85a7026755bd0ad16ec2f58c880a07",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 12,
            "payload_hash": "ed4c85c1ebc1765450203ad9f9c092dff4f1580367add60cf5db8eb894088521",
            "sig_id": "18c9a91230de00b8ce9f42845ffb24d2b690a2b2f9ee18efd66d0f14d000fb070f",
            "sig_id_short": "GMmpEjDeALjOn0KEX_sk0raQorL57hjv1m0P",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMQNAnicbVRriFVVGJ3xlQ5IjSVaIsYho3LM/X6MlKEgUf0wMxNKrvs59zjjPddz\n76ijTRGWk1GBhSU+wqSQyNESggzULCGsFJMIC0F60FCBgT8sMrPvXKfJoTh/9tnf\nZu211re+fXT8yKaW5pV9neX+e2eS5s+OHu5uWnTgwgPrEpv5nqR9XeK60lCpF6uK\nWRGS9qQz9FhTC3em2c2rg03aklUhr6VZBUoEE4ei8sJg5lTgNjKPhaDcKc+sUE5y\nx4JySW9bAVOAxrTSEfJqnhZ3JFwZwrinXlnCHeZRuEBjjBIrKQDAeKaIpwRuLWe1\n+jA2SQOzlHrY/Z/znY0CwgjTgDgVRBhPqFAiKEmikdwKZrhViGLCtUSWOyONtNJr\n5aiGH0WgjA1BBuC6G3AMWxwlj5ZzxISnEgmkGOGGEqQwQsXBWsgHnctcVqlkORhS\nGFDPjessLAD+qasVq9SXhszEqC3pMrU6CCq5sgGXYI8RzLggFA3H7apllVowlc4C\nt0FMIS85UdxEJjUKxunIvfaSYKUlbxAbbEDh2fAmeBXBO8KRjdpbKaJHDBTiaKP0\ngtiIGdXYxWGmCqSpsxhBo6kmRjAbHAFnDDOUIbA7WIqFoyoGb63knFEhFDFGMasU\nVWAqcM/DiqweStU8yyI48ijEr56uGFLONeNtievO80IilZZSEayWQXBZtFkor4kF\nVEcExdIaLb2jAQvLtSdQw4ITUhCwTgL9MAjOpcb0Cng1D6sAXAuFEGGRUaQ1Z9Bk\nHbSAD8QLo4ziBKxByBENumW0IFkyxRwcNkAgGdJS+NvQ0ximcnCdJW/qprS8VnR5\naKo60nq52yb/7evMorEzG51twJTqPVUo0rakVjf1gj1UamFlJUvaGazSjisjYDnw\n4AgyCzyhhzo4DgRjIB5YSqqI4IGSECNG3BARVVBBSMwhvTwYjIr+FmhX7iO9SxvX\nlOom7Sp4V01PV2Z8qWxq5WL0iXNKBQiIFox6KbT2RjtGYNq0ZhBHH7CVVEMKuVba\n2kgFtwyuciGawrBBEfwqEQDHkDUM8RiopjIYqYkLoEgbYrzmgB5AHtcGIg6zjrj1\nHsPL4axTBh6jpLcYtYaEwYm76sEqrLs6YJJSLiAWa6ppHkpppRGNIgkIDSXDwL9y\nkKeIIeVIGC1wUCANURUspsRpYxELEEMZgmIMxj1EboqQ4qDRv0IxAWamAzBBb8XU\nu/OQ9H704WOjmppbmsaMHlG8w00t467753WeMHDNX7Om7WS/Xryp0rf43EPnp/75\nzdlfJsya/vE7m+tPrxs5ctOR9u3L75l3d/WFeX29Z15a+/OaU9f/3rpp0qQtO5bM\n/WT2xFf7zISBC/nst4/dsD9/ZePlhYcWXRpx7PT63Y/v2Xdp9Jtr9xx85qmpn645\n9psvTb3cv3dz85NfSXx2y6hbph9dMHb+qcq+Px6ptE75YODc/p0UzT9x0ew6+f72\nbG/rhq65L792/r3WGUd6x97+Xf8dPy7b83zPxN39Gt217IuB0wvXL3i29fUHmw7v\n37b4vpMHv2Rzfrp/ylvd/mz7mJYnug/Yvq27ftg658xz045/Xl7y/dfjb0tvXTru\n+IY3Di3aMXnX6rXbN1778Il3F9747eQZq178G3IcKcE=\n=dEMP\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":10,\"last_id_change\":1421456230,\"username\":\"olsonseank\"},\"id\":\"80d75285af4790eac9f5d9d721897500\",\"key\":{\"key_fingerprint\":\"d8f6b6250bf9db76fd04fb51fbf7d62bf14391cf\",\"kid\":\"01016093cb108d4392a64bec275fa4a340362eb316c38fedbb755436682aa84b88380a\"},\"remote_proofs\":[{\"ctime\":1421455945,\"curr\":\"837b336eb97e6572d3268d92b82ac26317ba97dc3e16b59d2d9216522edbbbc7\",\"etime\":1579135945,\"prev\":\"9680024f43099541f79e96969f146a8a852db700c2993c7fb2a67484c954a82a\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"olson-sean-k\"},\"proof_type\":3,\"state\":1},\"seqno\":4,\"sig_id\":\"b5a825025bb706fd9ec5484fe2d4c9738265e32eff105a26f8e8e671525a5ea10f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"22cc88e7d69643d7699da9c423e09940d7de1b739d9d5989bbf365b45eacefaa\",\"seqno\":5,\"sig_id\":\"7d640ba405fe3937ea792ceb709a2ad953e0e70659aac948205bdd1c15cbc8ac0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1421473356,\"expire_in\":157680000,\"prev\":\"a6808c59df1a6406a961e81b7038eb132c9ab04eb827ee844663ef5a7e651e90\",\"seqno\":12,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1421473356,
            "etime": 1579153356,
            "rtime": null,
            "sig_status": 0,
            "prev": "a6808c59df1a6406a961e81b7038eb132c9ab04eb827ee844663ef5a7e651e90",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 13,
            "payload_hash": "1421995af7068ad3bf3982028330e56b89e187a1c94aa1f5426cf61696e1478c",
            "sig_id": "19da9eba50589b026df038f1ca162d5b008a101fa102abc728d3b4a44a4747e00f",
            "sig_id_short": "GdqeulBYmwJt8DjxyhYtWwCKEB-hAqvHKNO0",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMSlAnicrVVbiFVVGJ7RdNLMMhKxyHRDBDaN634Zgx4ydKywSEspOazrnO2ZOWfm\nnD2j0zBBRkkvGhRZWqMMpd00sx5EfPASUkghJRaFJAQ9lFY+KNZQ9O8z0+iYDz3E\nedlnr7W/9X+X/1+fXD+xYWpj94ZScdeSu0njsaPLehqWH/50sD+xFd+XtPYnriMN\n5Sx/KpvOkLQmpdBnTS20pJW5a4NNmpPeUK2llTIsEUwcisoLg5lTgdvIPBaCcqc8\ns0I5yR0LyiUDzTlMDhrTcnuodlXT/IyEK0MY99QrS7jDPAoXaIxRYiUFABjPFPGU\nwKnFSi0bV01SxyykHt5eZX+pvoAwwjQgTgURxhMqlAhKkmgkt4IZbhWimHAtkeXO\nSCOt9Fo5quGPIrCMDUEG4HrqcAxbHCWPlnPEhKcSCaQY4YYSpDBC+cZaqI4qV3GV\ncrlSBUFyAbKqcaVcAqg/dbX8KfWFMTExbk46TC0DQgVXNKASvGMEa6UIkeNw16Tl\nPtiQg9ar0gzR4IGlRsJqRqKXVODIkBFS83pVo+rngo13wDoRHDaaeSYUd4jSiLkj\nxFtjKWFEemmQ0mycolZijJTJNypMpGUaS6UoDoFIYxR8qkBfbSRzNkqvlHWQCwiI\nVhojI0FRqL0aOitZKHRVK5UIcjwB2cvSzhHaSDKFFWpOXE+1Cqc6xOGtRY5HilBk\nmigThHSCgQMsUEENlKoIM4xaDXVT5mjkHMy3GlsH5YdRcC7BsRHwrmrozV0VlDjt\nIQKEY084E5FFUMAH4iOREXRhNhDtBRce5LXSQEQD0xqEM1olY1xyfet86p1UDK5U\n8CYzhTW13OKxlmpPs2KPTa5makeaS1PHKGR9XbBCm5NaZrK8dFiphe5yJWkl8JS2\nj4SfQPEgAQHDpHDeQddZQi10FaJaMw7U8mjbqBgOSkShCKZYcQVfaOskiskI2sh5\nZKB5vBNcEMnGnLBSYOuxYvCjAnHBlICIkBg80cbrgDWjxggjsbXQaRFJa4XhUojg\n4dsrnKAj4KNO/A82/0cn8nEyqnw5rK21uFpLua+nJfiepC5/Bs3bAYvFLOtqvdIS\n6HR0NVfoZa5QzHV0hjpGPIJu0pw6w2w0MGMY8lxK5DS0HHKEek4shRFiggsGW2Sc\nsP9yZXX9mEJm0o6cQJfp66gYXyiaWhGO81Z4ZDDMAOMB0UXtGJPEMhO1IpRAW+d+\nGaOFh6MxBw6CEYE8i4i7XLlREvIyEhyqZjxoC32NIPAseoYFV9EFL+AkyJkJ0gjM\nuVKcAV3LYVzkbhAkBFiYDOSjr05hdAJedoHk0l1KGsnNVwLysa4rrYZCWq5nRChU\nl3s0IsHDVQN3RbAOS8EZRwRRyB0QRhr6FVoXc5jqEErvBXKRe6uChRGGoESCLxHF\nYFdm2gET+JZN1lMNycDhg09e09A4tWHypAn5vdgwdcqN/9yW33zd9OfsZSc+Wnhu\n4dJXhtKh6+YcX/Lgw3tX7Hx/i+ufuXtz/7Y9Nw33tr45f92JhyY2vds29yVz1+M7\nu+2SzXu2bl/wQvx5/X3PT/9t5oIflv7YO2vlycFFpWNixSPL27ZtL71z9tfu2wcX\nvPz0yROHLhydsvoD7p57asZrG5uWvd0y+7sPC4e+Onhmy+vTVr03Ye/p9Yv3DQ+u\nmnPg91vWzMOLvmi+Z96FO7cfmLNpx5ns0dPz93dM3r3p4ucTe88uXnnvbZNmvHHt\n/lk7OndNb/t2cHjGefP9x037yS/PnNpzvrWzePzmixMee5XuaywsPzVh2sY/yl3k\nyE/tf20YttNunf/ZA0ND8v5nV5Mv5dbSDefaVh2Z9NYdL/4NNTaZww==\n=nhwa\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":11,\"last_id_change\":1421988227,\"username\":\"jinyang\"},\"id\":\"9403ede05906b942fd7361f40a679500\",\"key\":{\"key_fingerprint\":\"bc6ec1a94d4685c033f15c22dbab32427d7a0894\",\"kid\":\"0101b71108a15c28127b49178831ee27aa8bab875b9a74cbf7d88bcd168e598910a70a\"},\"remote_proofs\":[{\"ctime\":1420748180,\"curr\":\"c05142b0c5f300f4928ae67c640604e363a27d824a43b924234c3f5513eb91bc\",\"etime\":1578428180,\"prev\":\"4632c9d0b5251d2546f4f2dbde2df27f85c4be29d656d42fb7aff7e4991a9a98\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"jinyangli\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"22c94e32ab376cdcc75b23bd3d039945c9dca7abf841e86f6821318582ab9bc70f\",\"sig_type\":2},{\"ctime\":1420756274,\"curr\":\"b761bd184848360564867a02fed29ad9e1943aa6a71bb72ff07bb6a5766ed761\",\"etime\":1578436274,\"prev\":\"c05142b0c5f300f4928ae67c640604e363a27d824a43b924234c3f5513eb91bc\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"news.cs.nyu.edu\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":3,\"sig_id\":\"3159fca3c42d0dba953ca4bfaa7b40d5770c96850c23d52b3320aecea1b0ac6b0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"db6d0a1059adc23cf9c4472b4af982323f14836aa96d7701510064260d4f05cc\",\"seqno\":7,\"sig_id\":\"595345e9b27a0e494fd41658fced69adb23ae7a61558854ca3b5a083eb9206600f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422051486,\"expire_in\":157680000,\"prev\":\"ed4c85c1ebc1765450203ad9f9c092dff4f1580367add60cf5db8eb894088521\",\"seqno\":13,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1422051486,
            "etime": 1579731486,
            "rtime": null,
            "sig_status": 0,
            "prev": "ed4c85c1ebc1765450203ad9f9c092dff4f1580367add60cf5db8eb894088521",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 14,
            "payload_hash": "1d21eed55eeb085102bf28673e316156187760fc5836ebc0af9aa579ba5e9b7f",
            "sig_id": "c7b288e398279c6d0de3d9accfcbd93dc601a0b51460b92bd965523a7fc246390f",
            "sig_id_short": "x7KI45gnnG0N49msz8vZPcYBoLUUYLkr2WVS",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n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\nS3d98rr6+daBpx5901rrkf8B5q5j+A==\n=997G\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":54,\"last_id_change\":1421371756,\"username\":\"patrick\"},\"id\":\"1563ec26dc20fd162a4f783551141200\",\"key\":{\"key_fingerprint\":\"246201479f89c599c08f29b3fa706aef3d045f20\",\"kid\":\"01017c13682b0965e718feaead9f04e468086c3864ba8294072c97594a5957a99f6f0a\"},\"remote_proofs\":[{\"ctime\":1392412788,\"curr\":\"c20d2386bb1714908f0de84f76d9aabc7c2911a1bc6ee451f98bb8833a528fac\",\"etime\":1550092788,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"patrickxb\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"fff3da1be82b15fa7a5633f85d2ae19e4f128986e6e80e64a4859dedb56f47080f\",\"sig_type\":2},{\"ctime\":1392412864,\"curr\":\"e4e726145a1e148ed4bb2c8481d6a6f399d0e65d04af5e590be98c2ca2f23705\",\"etime\":1550092864,\"prev\":\"c20d2386bb1714908f0de84f76d9aabc7c2911a1bc6ee451f98bb8833a528fac\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"patrickxb\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"81a9003a098ec52cd98a9f4983ad7edea81e9beae23064ceb97518138e3cd0f50f\",\"sig_type\":2},{\"ctime\":1395889480,\"curr\":\"7a1d453f123e38a125f0e872c6ec4ef0ba80119008996742abfad2675a16110e\",\"etime\":1553569480,\"prev\":\"bff52a25614ccd76fafabd54adbede2a2832899a0c0fe324a433de6100845331\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"patrickcrosby.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":11,\"sig_id\":\"8e0043c22d467a8b966c528c17882118c23bec1b7803fd4b5430c500e04352e20f\",\"sig_type\":2},{\"ctime\":1420864565,\"curr\":\"171a9d6e5b305d8eaf72f78e75a4bc814bbc2c5e5a429724640f0727212c0cca\",\"etime\":1578544565,\"prev\":\"eae03042fbc56ad02ffd97e0d07b43a5d9c4f4959a03bdc3a5d1659d0f28797d\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"patrickxb\"},\"proof_type\":6,\"state\":1},\"seqno\":23,\"sig_id\":\"9127676da00562183585508aaf5b5fff4e936f208fc238e388d37a40eb99b3fa0f\",\"sig_type\":2},{\"ctime\":1420864719,\"curr\":\"db4612f774b4e4689b5b8d9ab9c99447d30bce0a969a131d65fa2810b352cc44\",\"etime\":1578544719,\"prev\":\"171a9d6e5b305d8eaf72f78e75a4bc814bbc2c5e5a429724640f0727212c0cca\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"stathat.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":24,\"sig_id\":\"250861cbd50c104723977c779c4a7ab7919d664dabf0228a61fc65d24dd4001d0f\",\"sig_type\":2},{\"ctime\":1420864917,\"curr\":\"6c1866e5442084ff8b6892ca7a8cf2c5555e5c6281553331593e3d8efb911ff7\",\"etime\":1578544917,\"prev\":\"db4612f774b4e4689b5b8d9ab9c99447d30bce0a969a131d65fa2810b352cc44\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"numerotron.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":25,\"sig_id\":\"3c666d4cf57d5f1b11a556def8d0f78e89ec92b994867a4ce0245f5bd187355b0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"bfdae4f98d9910608c2760cf87fb6e3e1ea3527fa57ef2221e4d6e5cafd25cb3\",\"seqno\":31,\"sig_id\":\"baeb664889dc7f1e203bffe3cb364249a55661f45f9c84e3a470bbde36e34f360f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422051540,\"expire_in\":157680000,\"prev\":\"1421995af7068ad3bf3982028330e56b89e187a1c94aa1f5426cf61696e1478c\",\"seqno\":14,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1422051540,
            "etime": 1579731540,
            "rtime": null,
            "sig_status": 0,
            "prev": "1421995af7068ad3bf3982028330e56b89e187a1c94aa1f5426cf61696e1478c",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 15,
            "payload_hash": "185e2496a0dc012deea1c883759a74e0f7c33e38e11a40530aac76cd1916c7f5",
            "sig_id": "de2c80dde617e34b0715fd082b593a2b7eff7d4915c2095e4f11e92871d92d9f0f",
            "sig_id_short": "3iyA3eYX40sHFf0IK1k6K37_fUkVwgleTxHp",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n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\nv9Ii8F4=\n=Az9U\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":45,\"last_id_change\":1421805379,\"username\":\"gabrielh\"},\"id\":\"b7c2eaddcced7727bcb229751d91e800\",\"key\":{\"key_fingerprint\":\"3f6e504945ae44e6a53d49845431890fe1eebcde\",\"kid\":\"0101230aa5a191471449cc05ee95ce3637ebb16e409bb3b92ac0a8cc28eef92c0b500a\"},\"remote_proofs\":[{\"ctime\":1413405722,\"curr\":\"55d1e2013c91f4ef410538cb03cadfcd0b40e86dfa8a0594a42036eedfd61fff\",\"etime\":1571085722,\"prev\":\"461f0051ad08190465b5c8f4cab2fb8c5f04290c3f3cbd53794e7dbe5b89d448\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"gabrlh\"},\"proof_type\":2,\"state\":1},\"seqno\":19,\"sig_id\":\"dbbb07105a43c18ab1a9d01b564c591a6d4d57e35dd4ba3f1a0ed8e33f8517eb0f\",\"sig_type\":2},{\"ctime\":1413405744,\"curr\":\"5381de2b714095bcf65f41c1b235c11ab6fb87cea8b2c9a3bcb95095ab26e191\",\"etime\":1571085744,\"prev\":\"55d1e2013c91f4ef410538cb03cadfcd0b40e86dfa8a0594a42036eedfd61fff\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"gabriel\"},\"proof_type\":3,\"state\":1},\"seqno\":20,\"sig_id\":\"916bccc5e1fd7e784340cca2795d1ae352922bd498ec196bbff6c00b7a304d2d0f\",\"sig_type\":2},{\"ctime\":1413406032,\"curr\":\"eba1b91ed0048983d7e32efeee7fecce16547998ff3e05b67045a751ef99be1c\",\"etime\":1571086032,\"prev\":\"5381de2b714095bcf65f41c1b235c11ab6fb87cea8b2c9a3bcb95095ab26e191\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"gabrielh\"},\"proof_type\":5,\"state\":1},\"seqno\":21,\"sig_id\":\"2c9e345b5acedbcfe3d5d9b2d6115b59e9d6ad31ad262bb07266ccc6db0b95940f\",\"sig_type\":2},{\"ctime\":1421558613,\"curr\":\"fdd474158530c9816ca9707c3cda809aaa24a0f69b5bcc06f5776a54cea54fff\",\"etime\":1579238613,\"prev\":\"3f0c4b94e2c2d6435aa583012793a77222c3ec7b2ab48ff95359df23fa5f422c\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"rel.me\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":25,\"sig_id\":\"fdee6f84bdd854d055fe51be696b175e5f2d2d6b143bc09a52f7bad57dfe05160f\",\"sig_type\":2},{\"ctime\":1421558741,\"curr\":\"158668859ea086ccd02ce16724f2b35a6a71834f98867429cd1210a075346443\",\"etime\":1579238741,\"prev\":\"fdd474158530c9816ca9707c3cda809aaa24a0f69b5bcc06f5776a54cea54fff\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"rel.me\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":26,\"sig_id\":\"9c8edb92c68188a43266c7037e1148c9320a969b248b97059aeca65ddfc39e100f\",\"sig_type\":2},{\"ctime\":1421805307,\"curr\":\"a0a74b05db02a6c66c9cf2ffd3754449ff680cd4fb61f80cba2038a23e7c0bfb\",\"etime\":1579485307,\"prev\":\"a71816d134d54dae20073688eea5753fd756903173b0461213aa3956a3c66de5\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"gabrielhn\"},\"proof_type\":6,\"state\":1},\"seqno\":28,\"sig_id\":\"2787406a0b57a2eb7de822c5ecea6bdcd28a3ed96191a27d46459d747a6b292c0f\",\"sig_type\":2},{\"ctime\":1421805358,\"curr\":\"b8e2fb00585d67ee5144b37483a94e251444144c9dcd5949ca30bf275bafe59f\",\"etime\":1579485358,\"prev\":\"a0a74b05db02a6c66c9cf2ffd3754449ff680cd4fb61f80cba2038a23e7c0bfb\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"gabrlh\"},\"proof_type\":4,\"state\":1},\"seqno\":29,\"sig_id\":\"a3a73a75b6941ff0aeabdededc9640d27a45cff49f2a9b57538b4608a45ba6590f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"b8e2fb00585d67ee5144b37483a94e251444144c9dcd5949ca30bf275bafe59f\",\"seqno\":29,\"sig_id\":\"a3a73a75b6941ff0aeabdededc9640d27a45cff49f2a9b57538b4608a45ba6590f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422051554,\"expire_in\":157680000,\"prev\":\"1d21eed55eeb085102bf28673e316156187760fc5836ebc0af9aa579ba5e9b7f\",\"seqno\":15,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1422051554,
            "etime": 1579731554,
            "rtime": null,
            "sig_status": 0,
            "prev": "1d21eed55eeb085102bf28673e316156187760fc5836ebc0af9aa579ba5e9b7f",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 16,
            "payload_hash": "e11940eb12b36e89eadfd73af91558819262952825faca0af1ccc744fb896a6a",
            "sig_id": "08fa6ab7f3d57852666fc4cc2ccf946557970be7bccd64cf6894d2940d90e53d0f",
            "sig_id_short": "CPpqt_PVeFJmb8TMLM-UZVeXC-e8zWTPaJTS",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowGtVVuIVWUYPaNpGgk+GNgN7ASRzaX/fjHL64NERmWQ3Tz91zl7LmcfzzkzOdpA\nRhEVSpRPhmZoD0USYoYXBi1ItJsGFURaFEl5gwxJQm369p5pmEEfEjpPm/N/e/3r\nW2t93351yvjCpKblL3aWty5qJU2fnbCFhz/+Vq8q2tT3FWetKrquJFQa2VPFdIfi\nrGJn6LOmHtqSdEYl9aGtoz5juKal2Btq9SStQBVqk2202N+SlWcvx6TSHmrVWpJh\nFbkyhHFPvbKEO8yjcIHGGCVWUlihjGeKeEoAspzWG2NuLeaYpcTDv3n9fKHmLYT6\nhXl9T37AsMVR8mg5R0x4KpFAihFuKEEKI5QV1kNtuKXUpZVKWhMiZ9wdap1doVRL\n07xt10iyIswIQZwpqoCTqZfhNaKkshS56IVhRivNGUPBWapJDJFQzyUmkQaCcAhO\nMsGww5ZF5yNVxJIorceYRUkwtoRGbKK22vgIgDoI77VDkjpjNGXKGCWR9FEzAhiC\nY8mhiXpYXkmBHNWSU+DeqBnXmbEGsRJXz54SXxqxBdOWYpepN0C9kisbsCRvDGst\nqFBjNemqp5V6MJXOTJOxImihW4q5zgp5yYniJjKpUTBOR+61h44UMMp1Hg5A5tnY\nEHgVwWvCkY3aWymiRwwMw9FG6QWxETOqsYsZRn4ZwggLpKmzGCkPh8QIZoMjYDQY\nQBmiggRLsXBUxeCtlZwzKoQioB6zCsxDJmunFrrTRihVweMIIj0+2mXMONeMQ889\ntVrWIpWWUhGslkFwmcVSKK+JBVRHBMXSGi29owELy7UncIYFJyQjYJ0E+mEYnEuN\n6RD4MINMlZxFnrRycJ0lbxqm1FHP7BqZufakUe6xxUsNas0cas0tymFKjb4qHILN\n9YZpZHf2j4SEwVPSPjQ4lhsFyhMOGiFQXgcH2WYxkOOeOS0hn4IHSkKMGHFDRFRB\nBSExhxniwWCU2ZLBDV1I+lvGaAiBwpyNaBjBHIYIgeDAjFNhA/c+S4xGMJEOvJTK\nw1vKWi24l1oI4pFmAgMZo8RYDYUcAr8yDV2aVLIVchkVR2I+WkN+OQ3lKA2ZBnuR\ncdwKaTkxNID9lsJCo9FBLK3SzED4FCw3LYITmFJtYNBBQMihuETCJ/NrSg2TdGW8\nq6avKzW+NLxt/gcN/1MT+2ZccRv92ebJmxheQKO+BPjy2yOsqCa1UEoquaVCIfhl\n+odeAMGKBwJ1BnmHMPEBAudgeiXXRrKAonSUBqoCxoYhTpExTgrnsYbRl3H0YhTA\nzLQDJnRcMY2eWij2v9R001WFpkmFiRPGZd+6wjWTp/77Afz0+asHO9ae+vHm/o5f\n1qzdqH9vffDNu2bPWaR2pJM3L9ozezF+qof2Du7a/uEN+sC2R1f/dvrkpu+++WPl\n4Y3b1DN41Xn7xvHTT6y/eOvEo+u+Otf79bKlZ+bsf2ja0bffXT3ur7OquG/dETew\nTC4pP3fgpw33Frae+fWRO5dunDD/gw175u2+3x2qbe/e23awOvDYwMy/F7i3Zurm\nu6dM9ys+Gb9307Xd6nxHYUeTOtn85cs/XEi/WP50877Wc5/fNnDdO/rG9Rc2d0zf\neeiBbfsX306X7D92ik21r63ZufT6uR+tPjhhy9xnX5g6f8HhOy7esnLa+68vXtb8\n8xl5T/m9P1/ZW921+/sjxw6fRbPvG2g61zI4GE78Aw==\n=mZ4m\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1422054838,\"hash\":\"2878b30cfd6a4a9895440ecb392fef23d5712f3e201eec74641c1b4fcdf382b2f7bd114f7211b23f1af9b9adffd69e6dd9c073caa9348aa8707df9421b465175\",\"seqno\":139753},\"track\":{\"basics\":{\"id_version\":13,\"last_id_change\":1421996368,\"username\":\"olsonseank\"},\"ctime\":1422054969,\"id\":\"80d75285af4790eac9f5d9d721897500\",\"key\":{\"key_fingerprint\":\"d8f6b6250bf9db76fd04fb51fbf7d62bf14391cf\",\"kid\":\"01016093cb108d4392a64bec275fa4a340362eb316c38fedbb755436682aa84b88380a\"},\"remote_proofs\":[{\"ctime\":1421455945,\"curr\":\"837b336eb97e6572d3268d92b82ac26317ba97dc3e16b59d2d9216522edbbbc7\",\"etime\":1579135945,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"olson-sean-k\"},\"proof_type\":3,\"state\":1},\"seqno\":4,\"sig_id\":\"b5a825025bb706fd9ec5484fe2d4c9738265e32eff105a26f8e8e671525a5ea10f\",\"sig_type\":2},{\"ctime\":1421996154,\"curr\":\"fa344022969f7136be5dd479090a32cd4378d2198bb965d79662d094614c9a86\",\"etime\":1579676154,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"olsonseank\"},\"proof_type\":5,\"state\":1},\"seqno\":7,\"sig_id\":\"49bbc0ac5b67b52a3eb59b325c3fcbecb894a66886ce96ec61339ab23a5e80a60f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"fa344022969f7136be5dd479090a32cd4378d2198bb965d79662d094614c9a86\",\"seqno\":7,\"sig_id\":\"49bbc0ac5b67b52a3eb59b325c3fcbecb894a66886ce96ec61339ab23a5e80a60f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422054969,\"expire_in\":157680000,\"prev\":\"185e2496a0dc012deea1c883759a74e0f7c33e38e11a40530aac76cd1916c7f5\",\"seqno\":16,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1422054969,
            "etime": 1579734969,
            "rtime": null,
            "sig_status": 0,
            "prev": "185e2496a0dc012deea1c883759a74e0f7c33e38e11a40530aac76cd1916c7f5",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 17,
            "payload_hash": "c49c8c4bef4862d13da267f19416f3edc650159b18f25abfe8a0f3c5417159e2",
            "sig_id": "8afe41e09f1bb4b99765f8fc97ad11aad3d2de26c110d5b858628db443decf600f",
            "sig_id_short": "iv5B4J8btLmXZfj8l60RqtPS3ibBENW4WGKN",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMIXAnicbVJbSBVBGD5aiimRRVeDiK1AymJmdueyp16OZUYiBdaDkMns7qxtJ3eP\nZ4+WliFU9FJhHeqlwiAzMizRStColKLIIqXLQy8ZEVqYdBGrhy6zVm/Ny/B///d9\n/8zHf3f6lFBmStXh6I62DStRSn/v6urQlnvv/b2K4Vm1SnivEhWTl+24FSIeiztu\nQgkrmHGkYUu1mIGwCbFNTKHatk0ho8QgjFsaQ5aKlDxlh+cHCmljcF+scjyJyaLc\nsST6H350sgEggKoAWCWIcAuphBHBKLI5xQbRODYYUCHCOgUGNjnl1KCWzkxVlwVD\nsg05AlzaVU/aadCANsW2gTHQiKVSQADTEOYqAgwCEBB9EXd5pZBsz/Rc14sToir1\neYrEaxxTBCFYXiV3XMnYyc2o9+cvsbiXkIJdErVcPxAkamOBy25hlP/VlhuOa8kA\nJb1GxH3Hkx5QMs2EEwyEGkIIE4pxniL2xJy4KA+mQEwJA/IEQ0SNtBQQ6hoQBkSG\nKvPQBbdsi6rc1iHGjEEdEaRjJAOwuckBt6FpmlTTbIPphJMgD19UuZ70pvKdvEJ6\n+k6FyxPVcaHU993ZNjWUkhlKT0sNdiCUOS3732b0nMsIXfuU3ZpfNrBWtQqalTWf\nmlakHr31bLyl43gSJAc/tnWlNXe2v1pwe+BUyZcZpb/yu78tSj/biJu/Nik/IpXn\nu3+ODNWNL9566XWscHvR5YWbI8cKk7MbNl4fpjlzJ2Yv35/VfyK1r7X+frL/wb2b\ng1UTnaONXQfG2ExcMPYhq+GKvQsDOON138uHkRv2nPCbh/6+OYUt4UfhK+klRWrv\nuo+Rk6Uds0JyDzpOd3w4WDwUKsjyvnubiusOPW1PvDiybMqlRPLz456RocS03uGM\ncO5YfmRe1ysr7dvohXd33hqbcrau/1wy+vZ52Zkl+5bWLTPrL169OL+2ZSw3M+sJ\nPbOxu/E3EGAmLQ==\n=QKaN\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"service\":{\"domain\":\"jacko.io\",\"protocol\":\"dns\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1422256755,\"expire_in\":157680000,\"prev\":\"e11940eb12b36e89eadfd73af91558819262952825faca0af1ccc744fb896a6a\",\"seqno\":17,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1422256755,
            "etime": 1579936755,
            "rtime": null,
            "sig_status": 0,
            "prev": "e11940eb12b36e89eadfd73af91558819262952825faca0af1ccc744fb896a6a",
            "proof_id": "c8f453c788deabc4839f5210",
            "proof_type": 1001,
            "proof_text_check": "keybase-site-verification=iv5B4J8btLmXZfj8l60RqtPS3ibBENW4WGKNtEPez2A",
            "proof_text_full": "keybase-site-verification=iv5B4J8btLmXZfj8l60RqtPS3ibBENW4WGKNtEPez2A",
            "check_data_json": "{\"domain\":\"jacko.io\",\"protocol\":\"dns\"}",
            "remote_id": "dns://jacko.io",
            "api_url": "dns://jacko.io",
            "human_url": "dns://jacko.io",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 737,
            "hard_fail_count": 0,
            "last_check": 1434480198,
            "last_success": 1434480198,
            "version": 2,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 18,
            "payload_hash": "88d873d0340dfc5aa391dcfbe4a07cce979d8805c910b535af043c8e17579119",
            "sig_id": "e8ea962435235ba18daa87ad739bb59415706dbf82be6663e18e02529b9c38320f",
            "sig_id_short": "6OqWJDUjW6GNqoetc5u1lBVwbb-CvmZj4Y4C",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMSGAnicrVVbiFVVGB5TsyZHjHroJZQtGugU634ZTYquWJpCxNRkp3Wdsztzzhn3\n2aNO42BB2oASWtSTD1oYlWVgvQRJWd5ActCiUEyIkB4iqSDNkuzfZ6ZpxiQIett7\nrbW/9X+X/9/72ya3tE5a9Xyl/M79t5JJRw4u72t5+FjbxYHE1n1/0jGQuJ401PLi\nqWaqIelIKqHfmka4La3PXhNs0p6sDlkjrddgi2DiUFReGMycCtxG5rEQlDvlmRXK\nSe5YUC4ZbC9gCtCY1rpD1pulxR0JV4Yw7qlXlnCHeRQu0BijxEoKADCeKeIpgVvL\n9UY+oZqkiVlKPaxe4XyluYEwwjQgTgURxhMqlAhKkmgkt4IZbhWimHAtkeXOSCOt\n9Fo5quFFEdjGhiADcH1NOIYtjpJHyzliwlOJBFKMcEMJUhih4mAjZKPK1V29Vqtn\nIEghQJ4ZVykkgPpT1yieUl8aE1O3Jz2mkQOfkisbECnpwIwQyRTjZAJsqKY9/VVX\nNWX4DoCblRGjsNTNQiKRhIaoIgIfLDwh16xs1IFCtIku4KCRidSzGD0HLSxjyggQ\ngoK9HvgxFzS1cYKqBgfnqbYCa+YwHLZBgnpOI4kQskxJrpgNkXjEhYeYYEOtoFpT\nLKIArUxRexaq9TyUerN6PYIkXZC/PK2OcZeCAHfXl2UFb6YQCp5RjgJjjEvhMHXS\nMGKRM1wyjh2zPiIiJOImaIJRDMEwZbXwQkD5YRScK8TICHhvFlYXCdKcaM9YIBRZ\n4RGxkGAonGhjpFZaS6kk5FsIpjEOAVFOLLM0CsEpNUXiRrkU+jb5NLupHFyl5E1u\nSk81CpvH2srV01qR5ORfvW0ClfL+Xtjk7UkjN3lRP+w0wqpaPekABo20e6QLgCi1\n0HyQTM4sFi56Y7jXUVqMLHRp4I5CzwrGI2IRXI1eQHBi0JFYA2olI2gj95HB9svs\ngCioMTskioyHKAkzAWSKnCsP7nvMYE1bFZxWEsQzxMAIEBQ+D1YRjQiWRih+uR0j\n4KN2/A9e/zc78jVpnodsohujRoQsbVQvd4NcyQ06zg1jHfcMhEBgBg1cR+vAA2IQ\nRQraQWBho7EghcEKpAlEcGBDLcTewOo/3FjZvKaUm7SnKLzX9PfUjS+VTaMM18UI\nXYkCIoFia4JBXEpOLcQZpJMwlwkSIB6m0mEEiS5OUB2cgGxjrVkyRoKNIyECZUVP\ne3AQZqsTmgbKsQ0UWlkSL7jH0cqoZGSE+oCDD5GzAPdYa5gBEoPF7GtSGB2B4/4g\nhXTjEqaVgJRALtb2plkopcUJsB6igNBYNBzTTrlitDAliMfUG5AtwhiCwUKDd4Ij\nzLXFKsJgtjEoqII6zrCE5UD+Jorhqtx0AybwrZm8LwvJ4CcfPz6lZVJry9VTryp+\njC2t187863f5bcu0S+b1Z5P193zwyE+L2ru/XndJLX5r09YHbpcPzjr24sUNF15Y\nXjlZP/9Etvrsh8cXrliwaO8Bu7N1/V07liy+6dG2qUuOz1k594ste7c/tHMb2br0\n3uV37pr/9onrrvlteM8Nv+7b3bXrUH3h/i/XHejccfjJb2auXzbn3elH/tgx15+v\nVLoGZs/Smzsv/Hz9d3vmLxv6bJve1XnLPLdh3UuCts0+03Hu6CubLvxee/q+T6cf\n+vFUfvqjjafDmf43Zmz39R+G1v7y2InDN++7cXjDgnlL+cCJrs5NSW1O9c1Tw++n\nKw4ffa805diWu4fPPfcav3jH5OTzkxvZtDPlZw7OODhkNie9L3916tWS3P79wNmz\nuxcP/Qm5OaOL\n=HJ4a\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":9,\"last_id_change\":1422748452,\"username\":\"emilymcmahon\"},\"id\":\"2a81795a320f2723ef8f05c8b3ef0c00\",\"key\":{\"key_fingerprint\":\"1e90af3d4ffd5597b448a67d932c0d8424ce93bf\",\"kid\":\"0101a1ecd39b6194c17d9be77b7c907000b487584bef2d056dc0f1a3b6399316f6100a\"},\"remote_proofs\":[{\"ctime\":1422747622,\"curr\":\"e4800ed4350e444576c13c7a42b0ca57451c4bdf026705ae9210feea48b96d66\",\"etime\":1580427622,\"prev\":\"79529d44e230b6d02bd4b56d29aa798997787bf4664911ee0352b4b3f66533a2\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"emilymcmahon\"},\"proof_type\":5,\"state\":1},\"seqno\":2,\"sig_id\":\"48b3b35c46d54b16cfdaa5d9f7b10bd6ae5c3212645f04f4cefd62a8fe9f2ba10f\",\"sig_type\":2},{\"ctime\":1422748428,\"curr\":\"70f45ef724aed29f558dc90d14f729b8ec987aa7a2ae8c63748eb8290217a685\",\"etime\":1580428428,\"prev\":\"e4800ed4350e444576c13c7a42b0ca57451c4bdf026705ae9210feea48b96d66\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"mcmahonerism\"},\"proof_type\":2,\"state\":1},\"seqno\":3,\"sig_id\":\"abc5d49b8054b3e59fbcd6a2a03080f1616bfaba68a188ebe2652673b747aa680f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"ff0100e02e31baea057753b2306c174d12062b0137c10798ea0539ec68991994\",\"seqno\":4,\"sig_id\":\"6e347c90d8c6d48c693e351be363972d65d1fb7f87f423de1edef54e013bba4a0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422986748,\"expire_in\":157680000,\"prev\":\"c49c8c4bef4862d13da267f19416f3edc650159b18f25abfe8a0f3c5417159e2\",\"seqno\":18,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1422986748,
            "etime": 1580666748,
            "rtime": null,
            "sig_status": 0,
            "prev": "c49c8c4bef4862d13da267f19416f3edc650159b18f25abfe8a0f3c5417159e2",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 19,
            "payload_hash": "abf4894d0e34671e497ab1f2982dcb24c8e8974d10a00576eeb81ef236d75da2",
            "sig_id": "e7ee174db64a5f37d2ce5765367b8bd073aa81627356b9aec84af7b0136ff6380f",
            "sig_id_short": "5-4XTbZKXzfSzldlNnuL0HOqgWJzVrmuyEr3",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.4\nComment: https://keybase.io/crypto\n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\nk5E=\n=vfBK\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":19,\"last_id_change\":1419475392,\"username\":\"willwilson\"},\"id\":\"dc1944ea9df3dcddecbf994031a89e00\",\"key\":{\"key_fingerprint\":\"dd613e6f6261ad1ea6a72c3d32c8cec143040cb1\",\"kid\":\"010180b3d73a72827897132d4cc422ec07e5786db6212b6a9ddacad48b767a1c88610a\"},\"remote_proofs\":[{\"ctime\":1419468116,\"curr\":\"c4b068dc645df3ac1352e1bb5647f37633fecd0024593e716f98cb2d4832fcdf\",\"etime\":1577148116,\"prev\":\"b8da9a936fa9489b58943c64c170e095209f58a750da769bfa821b24e42a8422\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"williamaugustwilson.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":5,\"sig_id\":\"5f576a0731d37e3e025f4170d84ef0cb8f54709d2d3882bb247cbfc8db446cf90f\",\"sig_type\":2},{\"ctime\":1419468293,\"curr\":\"b7c1e5f80170b827f93cdea9831bcd6612b9b4c1997310d7b7cbe0ab3f6b154c\",\"etime\":1577148293,\"prev\":\"c4b068dc645df3ac1352e1bb5647f37633fecd0024593e716f98cb2d4832fcdf\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"wwilson\"},\"proof_type\":3,\"state\":1},\"seqno\":6,\"sig_id\":\"41a916d2bc13a407b22d71271e4b85d624d0ca61110198eae7f9afd8694731e10f\",\"sig_type\":2},{\"ctime\":1419468468,\"curr\":\"00ed23721676c981446cbfe1d8fecb5e08af42c562c27b6502e7dc0686a9f738\",\"etime\":1577148468,\"prev\":\"b7c1e5f80170b827f93cdea9831bcd6612b9b4c1997310d7b7cbe0ab3f6b154c\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"willwilsonFDB\"},\"proof_type\":2,\"state\":1},\"seqno\":7,\"sig_id\":\"bafbd853dd34d2398538d9015f3ea2d596dfbd191e7618cb1a5ad35535200d450f\",\"sig_type\":2},{\"ctime\":1419469445,\"curr\":\"8a1a8927a662c0f0ecd438b7c4c5de1ef91ad48c93382bda5467c8c30af328fe\",\"etime\":1577149445,\"prev\":\"6304bb04a2777bd09fbdd8556e0434d6b4c77c0d1247c5b017278547fd886f12\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"willwilson\"},\"proof_type\":5,\"state\":1},\"seqno\":10,\"sig_id\":\"ded5deb64539634e097a17d666d1a7e637e26470b7861c482a3273c63728bfb10f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"8a1a8927a662c0f0ecd438b7c4c5de1ef91ad48c93382bda5467c8c30af328fe\",\"seqno\":10,\"sig_id\":\"ded5deb64539634e097a17d666d1a7e637e26470b7861c482a3273c63728bfb10f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424279037,\"expire_in\":157680000,\"prev\":\"88d873d0340dfc5aa391dcfbe4a07cce979d8805c910b535af043c8e17579119\",\"seqno\":19,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1424279037,
            "etime": 1581959037,
            "rtime": null,
            "sig_status": 0,
            "prev": "88d873d0340dfc5aa391dcfbe4a07cce979d8805c910b535af043c8e17579119",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 20,
            "payload_hash": "64868e2b2a3dac78b9040722ab3217be565b887be8282c34a52842de6ae55035",
            "sig_id": "3c756a399e14d984ca1c36e1414a12c1d3d1b6cb46451d36ca8e21068d8ecc5d0f",
            "sig_id_short": "PHVqOZ4U2YTKHDbhQUoSwdPRtstGRR02yo4h",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.4\nComment: https://keybase.io/crypto\n\nyMPYAnicfVRbiFVVGD6jpTURZmWFFdkuinLSdb/4YOGLUl7KTMms07rO2Z7x7HGf\nPdogk6ZUD6JBBaVBhFAo1ERgWApOFy+R0kUMSsoUCisssCwsRfv3cTIHov2wWWev\ndb7/u/z/2nHp8Ep72+Kn67U3pt1B2vbsKHoqc46ePrUssZnvTSYtS1xXGhpFuWqY\nRSGZlNRDrzXNMCHNxi0NNulIloS8mWYN2NJUC8GNDU5qJ7lWQvNAmVFca424JTp6\nZZxI+jpKmBI0po3OkHfnaVkj4coQxj31yhLuMI/CBRpjlFhJYYUyniniKYGqtaxZ\nDGGTtDCrqYev/3G+3tpAGGEaEKeCCOMJFUoEJUk0klvBDLcKUUy4lshyZ6SRVnqt\nHNXwQxHYxoYgA3A9LTiGLY6SR8s5YsJTiQRSjHBDCVIYofJgM+SDzmUuazSyXAha\nGlDkxtVLC4B/6prlKvXVc2ayjqTLNAvQU3U1AyYlkzAjTOGSxxDYAMes6V1o6qHE\nbRGjjAevTTAIOYV1qQxsDTQQBC+sk3MBlJ4NDQFLEoIxzAZig7deRyogxWCsBOuo\nddiAXZQPMZUDNR+FpMLAOyIGQZLADefaBeNxjNYIaQkCdyT3BkyNhEUCDBkUiWAq\ncM/DoqwI1e48yyI48hC0X5Eu+lc6Z6ojcT15DlWtDsECMnNBOgoqlZTBWsUgVkod\nQip6SSimzkRpovcUK4S11UghyEkC/TAIzhVh+iz4IIPSlRaL1gjUgqtXvSlMdWGz\nzObcLHSmRa3HJv8XRwulWvR2wx7tSJqFKcqSsNMMixsZLGGVdlYH+0l5LIzxPjpu\nbYxCEeU50QJTHyz1qJRnEXYOeU+g6wzy0KgOGhZ77lFMzqKdrUf6Hm6VqRYm7Spp\nd5versz4as00a2VwHLMI0UmLvNUMCUc0ch7hAAWFx5SCu4oiiwgiOjDKmEGUuOCi\nd8aWHTAogpwngiPKtGPgO5aUoWiEhqHTESRgLLAVkpNgudWWB4YjxgEzHokKVjJN\nJAcRfeV8tCQMjsl5t0xp3dCu0FhAlo91p3mopuUJLoVC8JTuhyUAYmxkSjMPzc+E\nxAHSNjC5RMPt4CwBkUFpyTwGPxH8GzpL4RDhfvBls5LzhAJoYToBE/Q2TNGTQ8of\nvLfggkpbe2XEhcPKy7PSfvFl/1ypa64deRr9emSDmzIiue/IzlummNVvrdjSv2nN\ndds6ybyjs59dvnP4H5X9Z/r3x3Tb3AOH3j+ih42/c+e8dwZ+eeDVU3jq7mvC8faD\n4xZ89f2KUZP7B2Z9uGv+/OQiUfl05uP51Ok/nfym47kv/lq1+/rZcz4/kP1uZ25b\nvXXJK23f3rVszOhdj/y2ecPafQPLT0x/5vB3u57fsnflx9WXl8b1e57cVJ0x5p5P\nNq5+6m7846jKhNtW3bz36hfHfaZXrV1579f9yUeHTp04fdXEg4evXPHm/cdHT2yb\nsvuSfTeysY+u26j0TSuPXXF4w/i+gZ/r2384ue7yOdtf2zr35Pw/b7h10+Yzr09b\nfGzGS2Nvf/vdF2adeKLLf9n34OSRfwNe3h4Z\n=8sIp\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":4,\"last_id_change\":1424817825,\"username\":\"eastbayjake\"},\"id\":\"345ed9aea00c8194a5b5d3e3e20e3e19\",\"key\":{\"key_fingerprint\":\"172eeaa4be2bedbd9f36e34eab73e03bc1aa7535\",\"kid\":\"01015248df6736af67f04a242e5a559cead1ffba67b2042575da98cf24f2ea04edbf0a\"},\"remote_proofs\":[{\"ctime\":1424817548,\"curr\":\"b9eebad14ce7c3c81877ebb84d2333c008fd72313caf7afdd318019b90807067\",\"etime\":1582497548,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"eastbayjake\"},\"proof_type\":3,\"state\":1},\"seqno\":1,\"sig_id\":\"418d16aaddfc5bbff6828d529613deb3d084d2b01cc0dd2550a0d5cac9701d5d0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"0514f1527b0db9406c290cd01e9616d13314c830b02029e4344a032cecfdcab5\",\"seqno\":2,\"sig_id\":\"50349c43ca17340fa690139f13d1161b6752eb5b9b5e41f11e145f28eb7492750f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424817916,\"expire_in\":157680000,\"prev\":\"abf4894d0e34671e497ab1f2982dcb24c8e8974d10a00576eeb81ef236d75da2\",\"seqno\":20,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1424817916,
            "etime": 1582497916,
            "rtime": null,
            "sig_status": 0,
            "prev": "abf4894d0e34671e497ab1f2982dcb24c8e8974d10a00576eeb81ef236d75da2",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 21,
            "payload_hash": "435c23da23b8f0cdc5d01386c4fb45cabc0c8cc18dec204da64e92e4183ea522",
            "sig_id": "de8c6c5e3a795efa0134ece15854a3b27f66285e1aa54a889c056b344b992bde0f",
            "sig_id_short": "3oxsXjp5XvoBNOzhWFSjsn9mKF4apUqInAVr",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n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\nOPH+6enZmZn/Ag==\n=Xn8y\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.7\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1424822283,\"hash\":\"1e6cff7e202e00d95da40c270129ac4a2b2da7fd20b63742a51192d9e3c8b37181272499a8c8ccd3359aaacd22493d961de6f0e9427b63f934b61ddd4e1f7a7c\",\"seqno\":161166},\"track\":{\"basics\":{\"id_version\":10,\"last_id_change\":1424818329,\"username\":\"eastbayjake\"},\"ctime\":1424822326,\"id\":\"345ed9aea00c8194a5b5d3e3e20e3e19\",\"key\":{\"key_fingerprint\":\"172eeaa4be2bedbd9f36e34eab73e03bc1aa7535\",\"kid\":\"01015248df6736af67f04a242e5a559cead1ffba67b2042575da98cf24f2ea04edbf0a\"},\"remote_proofs\":[{\"ctime\":1424817993,\"curr\":\"d0732de68050ee3c1d182e10c17e5833eee5d800e587f668baa5607382a4b420\",\"etime\":1582497993,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"eastbayjake\"},\"proof_type\":2,\"state\":1},\"seqno\":3,\"sig_id\":\"ebd456fa9d628a45430314a99e31f3f09bbd42b0f0b7f99d939210a2081af2d60f\",\"sig_type\":2},{\"ctime\":1424817548,\"curr\":\"b9eebad14ce7c3c81877ebb84d2333c008fd72313caf7afdd318019b90807067\",\"etime\":1582497548,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"eastbayjake\"},\"proof_type\":3,\"state\":1},\"seqno\":1,\"sig_id\":\"418d16aaddfc5bbff6828d529613deb3d084d2b01cc0dd2550a0d5cac9701d5d0f\",\"sig_type\":2},{\"ctime\":1424817824,\"curr\":\"0514f1527b0db9406c290cd01e9616d13314c830b02029e4344a032cecfdcab5\",\"etime\":1582497824,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"eastbayjake\"},\"proof_type\":6,\"state\":1},\"seqno\":2,\"sig_id\":\"50349c43ca17340fa690139f13d1161b6752eb5b9b5e41f11e145f28eb7492750f\",\"sig_type\":2},{\"ctime\":1424818115,\"curr\":\"4a8a01b5f1fa0e85e6f1ff5f303b3fde520996845715e2c0256408b349cc8214\",\"etime\":1582498115,\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"eastbayjake.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":4,\"sig_id\":\"4391f1c718270ec52aa912be0ba8be237203676c3b84ee4e2baf8c703162dd940f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"b4930677467667ea292ba7c7320e1c336dc29bb575863a52806cd02fc3d6b633\",\"seqno\":6,\"sig_id\":\"82b1f91595ca62c66d4ac2166cc08ca777ee1a20dfdb8e69ae8989531afdcac50f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424822326,\"expire_in\":157680000,\"prev\":\"64868e2b2a3dac78b9040722ab3217be565b887be8282c34a52842de6ae55035\",\"seqno\":21,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1424822326,
            "etime": 1582502326,
            "rtime": null,
            "sig_status": 0,
            "prev": "64868e2b2a3dac78b9040722ab3217be565b887be8282c34a52842de6ae55035",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 22,
            "payload_hash": "3061ceab5f82a87df7cc4065c58baccd3bde0513326e40a4e71547278f93eca9",
            "sig_id": "f8aedd7699783c95ed1ca93f23dce9fc7a0ff6e394d080ce1c8a09687845311c0f",
            "sig_id_short": "-K7ddpl4PJXtHKk_I9zp_HoP9uOU0IDOHIoJ",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.4\nComment: https://keybase.io/crypto\n\nyMWyAnictZZ/aJ5XFcfTZd2w0FIRZh2Ozdc/1tqs3N8/KsOqhbZoZYPqH6vycu+5\n5ybPkrxv9rxP2iVZEBVaSq2VOkRp7boOxvyjUKGCdozpNkHSoihsf+xHdbNzE90P\nRdYOrfU8b9MsSVPaYiUQnjzPzbnnfj/nfO95bmlvz5JFD+wcHDiy8S6x6MTUsdGe\nLW/d+spEI7bTWGPtRAOGCmxV9VMrDGNjbWMQx2Lo4Jqifcd2jI2+xjYsO0W7RZ+8\n9MboEBGsB6u9M16jVMFp7z3TUficXADTmOyrw9RBc9Hqx3KkLOo9GtoFoXSSyUWh\ngetsAGXO2XJnTTQuJOVEkoJ2HWh3qjnZNLoxm0WitwusH+x+YJxxiUxLI0xIQhpn\n0FmRg9XRqKCjY5IL7S2LGoINNtrkHUhPfzhBn3kQLFC40W44xSPPVueoNVMmScsM\nc0roIAVznLF6YQfLaeXa0G612qUxshagKgMM1hJQ/gV06qciNWfE5KKvMRQ6FR2o\nCQOBVKJ3StB5lHF+TtxOhf3DocCyDtvNyydKJ7IgrGAWQhYy2qwBhFX0wH1jRv9a\nsnkMUISUeHI2A9NGBmEM55KphNkpqx13LDPMczW1ERBD0FL67EPOxnPp0DJuk0mC\ndEMrueVI/8xINCmUDJBiAhE5BEWaUu4lDrcrbI6U7XYmQbZS9VXF8MzBpRV0cBgt\nyzrNEIEBJsEzN14hM5YLmxJakUA6oPNmCTqiRZ6UMVmFxK02RhFL9DVDnA6uiSy7\nEHw6g1qVbhbdDhhAGGymUIXm/Z0azUwrVNuLqiLZ59AIrVTi9jlMuqGa1dgIfSes\nnSpU9b70pYMPtNr0SE9F/4XaTdpZn7Pjhs4XJNNKcpeC5sEHICGCsSi5p1o0SFUs\nrEavFXWLpHUaFavJ1NGm95vsmyej4kbNyCi9BfpJglQRKiurfKL6VxkAYkyGkpAG\nA3WB4iJzzayjvpQ8Wk7VoOR8GS8EHylxGwXPQlDPO2eT1JSm9kI4SwvoF0OVEmQj\njRFaWJN0SAa0FtQ9OluQXCFvXCsQaBet2g4al++P2Sz0QizkLBY5oPXMKCmkJBNg\nntxJimRBSA0KFUaGzBMaCwk495A8GEWlKR3TVOhXZkEkZ1jYnLxgggySLCdQ+UYK\nAElgTgwDWGV5AE1EKCcOkUzJSOrElKkzSUZ3CYtu8GkW1wH0tbEoMaWimtcbl0Wh\nFkKhZqGIDASSWZPuTEGmKhZRgQcXNCK3SjpDlpcgJqUQQGWes9RCY9aevEtcGQXF\nnUFBquRgIJN7xpCNY6ADaqdr3RU5m6fX4DS5mmO+/iulaJkidIl2R7gERTf4NIrr\nwPnaUPQX1cBovFoUciEUehYKRvl6pYUw0WcbgKVM3gqCJ6aR9NCRLkEJ3BrPEpfk\nzcoZppPnTJJJXYVDaWU+MHpt6suYMclEVoZmgySlypFKwTK6YBQK0BkRwUuOwols\nkmUuS8PpwovqEhTd4Bcd6n/nfJUo6nll4RtiDbSHG10EFU0IQ7RgoKpG1s7HQuME\nW4iMmUWGU5GQs3oD0lBLUE1HusyjJMmZMwkcgUohBu+B01uDGaio6jFNMxNJz/lk\nvtbdplmFYqg+xEgYG2qH1BwInYHrg+b/c4jJer7qHmF6zJo1pdbSzak2TeMg1ciD\nI0WJzaJeoS1VQlfu6TJRZPdCpkBzlMuMvF4nmnmcIZuJdPV2RxEHQPc0giC3CUah\nF6i4kxioUT44qKD7vwr99a1U9LdCNVpiY/KZX3z1xp5FS3puWnxDPXz3LPnQ8osj\nee/vbj7f3rTmHbb6oc0Pb3G7vn/2k08+fW5q5bd/v+KJ2x788rs7Nj43tW/L+O5N\n5174+LPvvv36dz+1rXf0yKj/2/Dj+27tP/q5vd9a1XrvliM/+8z4e2e/uHLrb5a5\nP3x61eSBTfcfOrX2vo2j+6fe/9iJczdvHj/6+R889tZtW5c+0fzLquNT/7j77IGP\nTL68eM/iN2/c8+cfnql2Ln9qYsc9av1fz+987cVXv77yC3/f8MJPTx78xlfWbXjq\nmz/vY2N/3H188U969j09cvLAm4/wX42deaMHD7WOPdp/3+vPH1p3/vbTW8Z//Kf1\ny94/+KN7d5VnJwb//dvPnvnl6uHeU79efvr2/6yQ+75zy8reN/710h0nO3L/lw6f\n/t6HP3p47z/vVEs/cerl/wLKKfqA\n=4aXe\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":12,\"last_id_change\":1424824689,\"username\":\"stegmaier\"},\"id\":\"9d046b0a27207caf23b7f5cc274b7f19\",\"key\":{\"key_fingerprint\":\"5e2add1d87fc0563a26611304def84758180f0ef\",\"kid\":\"010137bceeaa5339f9aff69138e7017d6d2b81e73171e18001b13243acdbdc2b1ca40a\"},\"remote_proofs\":[{\"ctime\":1424823729,\"curr\":\"5abc0ced21f1694e067127dde72dc38c7f5f3c5be7e1d466f4ad1756649cae9a\",\"etime\":1582503729,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"andrewstegmaier\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"d5879ff816c0ca3054318da51a9ac142a67e3195b86e626275e9545c133185e40f\",\"sig_type\":2},{\"ctime\":1424824164,\"curr\":\"397c7c7d29ca24f4749d2594fcccbbd66c036ea7d9412f15078c1531b71f8443\",\"etime\":1582504164,\"prev\":\"f22b29887d3575e5922871648710e4ddcf636625276d5ad6c5524255f7c314e1\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"stegmaier\"},\"proof_type\":5,\"state\":1},\"seqno\":3,\"sig_id\":\"fae790643233c390924532d7c235c4e4eb0e0951a7cdc119cd9c645ab38050010f\",\"sig_type\":2},{\"ctime\":1424824319,\"curr\":\"7fd9202ac698ca5f3b10fcd2efd0eac7471ac56c07901cb39c63f0edfdef6628\",\"etime\":1582504319,\"prev\":\"397c7c7d29ca24f4749d2594fcccbbd66c036ea7d9412f15078c1531b71f8443\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"astegmaier\"},\"proof_type\":4,\"state\":1},\"seqno\":4,\"sig_id\":\"b0c2e6e85c404cf2412b4c9c8a5ee17438646bdcbd44ecc4f1ff3525ef5904d20f\",\"sig_type\":2},{\"ctime\":1424824404,\"curr\":\"fccfa6cf07cbaf680c5ae58501cb41389cbac851e780989cbddb704ac6d4ecec\",\"etime\":1582504404,\"prev\":\"7fd9202ac698ca5f3b10fcd2efd0eac7471ac56c07901cb39c63f0edfdef6628\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"astegmaier\"},\"proof_type\":3,\"state\":1},\"seqno\":5,\"sig_id\":\"0b10945226b9f7ac0dfad1c21d05e8095b0843c17690d134e048605d910333140f\",\"sig_type\":2},{\"ctime\":1424824546,\"curr\":\"5560b5c00302f463d8d334fbe85707014e2c5feeec931e282f6d708f361cc2b4\",\"etime\":1582504546,\"prev\":\"fccfa6cf07cbaf680c5ae58501cb41389cbac851e780989cbddb704ac6d4ecec\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"andrewstegmaier.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":6,\"sig_id\":\"1edf0e496c36743244b2adb340f086dc8faddaba99c1b346efcfde8695506b570f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"5560b5c00302f463d8d334fbe85707014e2c5feeec931e282f6d708f361cc2b4\",\"seqno\":6,\"sig_id\":\"1edf0e496c36743244b2adb340f086dc8faddaba99c1b346efcfde8695506b570f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424825053,\"expire_in\":157680000,\"prev\":\"435c23da23b8f0cdc5d01386c4fb45cabc0c8cc18dec204da64e92e4183ea522\",\"seqno\":22,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1424825053,
            "etime": 1582505053,
            "rtime": null,
            "sig_status": 0,
            "prev": "435c23da23b8f0cdc5d01386c4fb45cabc0c8cc18dec204da64e92e4183ea522",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 23,
            "payload_hash": "7b38bc16fc9a65e8f7fe77cadd751a561378f59975d24ed7274ffaad09a04392",
            "sig_id": "49eb0fc70bc3f269b2c82b7ec8ebae60fa4da0263e0e532eb9c31e603d6e90fd0f",
            "sig_id_short": "SesPxwvD8mmyyCt-yOuuYPpNoCY-DlMuucMe",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMT+AnicrVZZjF5jGJ6xRYsSkVSRqEOQGM23vd8ySUMisVdcKEr5837b/MfM/P/0\n/KdTo9oJRcVasRUXlpC0ohe9sVyIJZao2Jcb0rhBJIJELDfEe/6OMaNNkPivzn++\nc573eZ/nXc5rh+0/sHBwzebR9o5zTxeDb7+5Ze3AygO++mp94btxqhheX4SxMnXq\n5qqD46kYLkbTlMdeWlZ2l65LvhgqJlPVK7sdOhJRmywsWC6dF4bbaCD6YLnBIJVF\nZpUSAKLYMNTANKC57IykaqIqmxgFWBQKoozWCwgcsg5J5pwJymivLUZlRZSCora7\nvXoem6KP2Soj3d3H86P9A8YZl4mB1EJjFFJbnawRGQ14rRC8ZZILcIZ5CGjQeBOd\nDdLRHyvomKNgSHBr+3CKe54NZA/AlI7SME05CkApmOWMNQ/2UjWjXDd0O51upbVs\nBKgrDKONBMS/DL3mqoytWTHtUDGGvZryaYU2kkjFMFdCS2OdsvNgJ7DXRt5A9jlZ\nTEIpcAmMsMyJhFJqw7jg0iBwV8xq38g1X38vXfKYlLYefFacA7cqAEvWgccGOegM\nEObp6Z0PwiEkoywLKSG9YYJnwYhoQQsGUXDOEw8ych1j1krapLVwkShxQ3oS9yqN\nd+vUmqi63UxiXEmVV5fjf2UtuRoqwtqqamgmE6WS3DKiJ2zgIeholGKuMYr0dZhF\npngI0iitDU+YkAeWjLOIluinGXCwisEe8BkGjSp9Fv3qb6cw2opYY+uaXmPLbBtU\nKcaynm8wjpRjKeBkGsdOk1MfplVPTdAhBejVWDcx6aSX1nS6dElX5ciemnVEGWK0\npGXwGLTiwICS8BytSSobyyIlyJ1NoCCLlACQx4woZKLqZbnYg7Ynntgw9HcJtYVZ\nCXPkgvQACRGVVWBUMFZpQEsVrZKjPg6W2kKBd9EboZnjJlpNbBKHEPaSsA8+UaVJ\nAg8h5uCoBjz1A3nsJfWJUC77bHx2qJAmgqHADFlQVJ0cmXQ65mhYNK7pr/9mRr2u\nrOtUzXfj4qYvLsDxqV6bWP3NDrEvO9QcO6hmibsTylukCRSYdWQE3WPUUFHkjNT2\nghJTqEgn9FG5YGVC5rJk6P7ZDmvlrB2GZks2QjCXEL2hmRKDi+A0l1aR4Yk01lEl\najfmQGmRhKV5k1ElaqqEe9nRB5+x43/w+r/Z0abJRjakdb1/3x96X4bAHEMUDR/Q\nINGSzl5oFSJAv54M6cBAS+ET5qCl5i5G0kgnnZNlCriibbCXIVf1w7RqLMca7hM4\nNdbF2GpT2TSrSCVyNZumej2QozKFhGCbGcNo4BAmaNQ0TXy2hhlDE47GkKLINAWx\nWYwzSZi5SWSg+uBZc+EkveytR6+k8lkwxtHRIhHeewXN7pExMpVTsjogtUaWwVIS\nG5rd0U9hZoXM2cCNdHOLjFxuJlu6dqKsUqtsngCjLaPfbHVIpjll5iFbQZMmZhMC\nFRsEIG7UydJHWphcStKTGlYlw0EZYWx2JAm6vxIVVHI1jhAm5dvBem2Vig2vvrz6\ngIHBhQMHHbhf82ExsHDBEX9+bjz//sEDm277We9frHh5+olNn73x0YGv7LdmkO1+\n4dyRlWLJiT/uPPb33uJdl8jjJp/bvPqRZffuvviF39677qBtx04X/pSzv1m16c6R\nN7bvUk+3T/jxtw8vXPzD6q9/Obn8YMFLR565fmv6btGVi8UV6tbigXeP2rFix7fT\n52358JSzLvvk1kNvfNEuv2Xi58eOXnn0Rb88cMvGjQ8v2TZ40xe35eHPLzM3PDnw\n6Y3v/HrNXcNBbvvygzOqe65/6LhjRu5csWrp8unT3srPnNR9d8lj9y168Pvr7r/j\n48lzln1ywbbPFh353M7DD8Wdl9++vdJ3b772keNPvWv3mi0Ltj41fdj7p9//+uDN\nY48uXbr80h8O2fjazl2vf/rs1p/Of6n6PDx+9R9hSf1+\n=m57g\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"2d67f2858139b2718d75dbc817ac348a08442552\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":8,\"last_id_change\":1426378948,\"username\":\"pasha1\"},\"id\":\"8ae24459e5728092ea3367012137a519\",\"key\":{\"key_fingerprint\":\"b39ebae468b5bf4115184c50e895ba8ae2c6f55c\",\"kid\":\"0101b9bc29a5e7480ceea4c57cb0c72d856205d2111e1c3d16ddf6438e6629d137170a\"},\"remote_proofs\":[{\"ctime\":1426378314,\"curr\":\"be7d34318018428c1cc6d7440939ca6639af2f1c3a53746671eaea1c0e798aa8\",\"etime\":1584058314,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"agilecaveman\"},\"proof_type\":4,\"state\":1},\"seqno\":1,\"sig_id\":\"9f2f5dd8c29cbac641505f1cb1a87e4f780d093198e545f2ee55a1dfaa23e5b60f\",\"sig_type\":2},{\"ctime\":1426378685,\"curr\":\"fd12c0e535da484574c78465a8b1f4e97f2c8fa745b9db7260917d86b1ae15cc\",\"etime\":1584058685,\"prev\":\"ccdfc97cbb060137b31b1249fbf7bf9a4a17a7da40a0c43671a0396dfd70d79a\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"PashaKamyshev\"},\"proof_type\":2,\"state\":1},\"seqno\":4,\"sig_id\":\"d21601924b8afffc0891a81600ea3d2ffa5fb231b4a4fa7abd49c83ea09f30a90f\",\"sig_type\":2},{\"ctime\":1426378883,\"curr\":\"725bf72209eaab7f75dc9d59613848e5e8406d4e101095462e286d3fa4e3d1ea\",\"etime\":1584058883,\"prev\":\"fd12c0e535da484574c78465a8b1f4e97f2c8fa745b9db7260917d86b1ae15cc\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"agilecaveman\"},\"proof_type\":6,\"state\":1},\"seqno\":5,\"sig_id\":\"4e2c5653a89c8b264cd55f7bf973fa05632beafc63619dd4626e6fe80451476b0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"54ea7af737b3b530a3ecea58093901cc14756a6aa8bf8707711ecc64dd480cab\",\"seqno\":7,\"sig_id\":\"4f509f1f612936aab8bab434bf2001a9b552bbb45a7a73dd04fee86cafd7f3c80f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1426384614,\"expire_in\":157680000,\"prev\":\"3061ceab5f82a87df7cc4065c58baccd3bde0513326e40a4e71547278f93eca9\",\"seqno\":23,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1426384614,
            "etime": 1584064614,
            "rtime": null,
            "sig_status": 0,
            "prev": "3061ceab5f82a87df7cc4065c58baccd3bde0513326e40a4e71547278f93eca9",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 24,
            "payload_hash": "5cae5e8dc5d0cd6b233f12f6aeb03f45ca4c12b3fc2c337ed31112f0ebaba2a7",
            "sig_id": "c930a19b6e372177c3ba9e840544cb27fad487f3b72cbafd5da863b97eed66cc0f",
            "sig_id_short": "yTChm243IXfDup6EBUTLJ_rUh_O3LLr9Xahj",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n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\n05sPP/YVP/+Hi1/684vq4MTZb/3x/SOH33n9jq//+GMf3fBO\n=ftZy\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"2d67f2858139b2718d75dbc817ac348a08442552\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":38,\"last_id_change\":1427140928,\"username\":\"akalin\"},\"id\":\"ebbe1d99410ab70123262cf8dfc87900\",\"key\":{\"key_fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\"},\"remote_proofs\":[{\"ctime\":1409475047,\"curr\":\"d8cd14f0d6da7dad0b0227b024ecb99c920e6110ef9486262be08df3b6431e00\",\"etime\":1567155047,\"prev\":\"738bebce2b4ed88c9319075bfed58779734efd03f3ee5386b15b38b61d2024f5\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"fakalin\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"1a67183170ef64b00e805c74d39c5286a6a9b50939b3604b15f0722aae27500b0f\",\"sig_type\":2},{\"ctime\":1409475097,\"curr\":\"99456cb27d7c6c8e6978be4c0928c19a07428e236fa9f7ed4d823d926f054701\",\"etime\":1567155097,\"prev\":\"d8cd14f0d6da7dad0b0227b024ecb99c920e6110ef9486262be08df3b6431e00\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"akalin\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"68a547eed57a11b62bfe2511fe128cc10cfb70f199453aa9cb964b97999f5d800f\",\"sig_type\":2},{\"ctime\":1409475156,\"curr\":\"a2a877a4cda04a32aa02de40137d22b2a0a04f35f7f677bdcae15fbf91f4ec04\",\"etime\":1567155156,\"prev\":\"99456cb27d7c6c8e6978be4c0928c19a07428e236fa9f7ed4d823d926f054701\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"akalin\"},\"proof_type\":4,\"state\":1},\"seqno\":4,\"sig_id\":\"dece17e125e06d6b44438218d29ab55a7785beaeb7bf372d8f98eb228009635c0f\",\"sig_type\":2},{\"ctime\":1409475252,\"curr\":\"297ac58d197e52956fae5d628b4db81de68e8cebf1bfd17fbb4404cee7edba6f\",\"etime\":1567155252,\"prev\":\"a2a877a4cda04a32aa02de40137d22b2a0a04f35f7f677bdcae15fbf91f4ec04\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"akalin\"},\"proof_type\":6,\"state\":1},\"seqno\":5,\"sig_id\":\"fc16a961ab69ae81910f3a76ca000fca19699dec8461fc7ec0e8fcb78764069b0f\",\"sig_type\":2},{\"ctime\":1409475374,\"curr\":\"04e5fba11ba32227cacb8b8345cec9bee563bfd4c25b77aa716b3763c3857735\",\"etime\":1567155374,\"prev\":\"297ac58d197e52956fae5d628b4db81de68e8cebf1bfd17fbb4404cee7edba6f\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"akalin.cx\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":6,\"sig_id\":\"ac377cb7d4df7bc9db7c50b3bc06146e05e29fd0eb715b478ea461074a8770550f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"ce162011e380c954de15f30db28f8b7b358866d2721143d9d0d4424166ce5ed8\",\"seqno\":25,\"sig_id\":\"48267f0e3484b2f97859829503e20c2f598529b42c1d840a8fc1eceda71458400f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1427999616,\"expire_in\":157680000,\"prev\":\"7b38bc16fc9a65e8f7fe77cadd751a561378f59975d24ed7274ffaad09a04392\",\"seqno\":24,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1427999616,
            "etime": 1585679616,
            "rtime": null,
            "sig_status": 0,
            "prev": "7b38bc16fc9a65e8f7fe77cadd751a561378f59975d24ed7274ffaad09a04392",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 25,
            "payload_hash": "91e23015983d77e62a467470aedbe563db62ffdcebcb5bacb22b1e81a81e0575",
            "sig_id": "d46203a6ef3dec608d5a61520b4310f3f72073ab461289d1937a3d50db4ab03c0f",
            "sig_id_short": "1GIDpu897GCNWmFSC0MQ8_cgc6tGEonRk3o9",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n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\n5dbu/q/dvOvr737o6RP/6bz/vxjQa/Q=\n=MXEx\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"a83b5e55a841454ff3f9eb75c03f5e269ecd30ba\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":17,\"last_id_change\":1430212043,\"username\":\"marks\"},\"id\":\"e3d11da4dd6ea5b4e328df9a80ad2119\",\"key\":{\"key_fingerprint\":\"5766b7358932d52ac7edd391d279eb4d0dd8be48\",\"kid\":\"01019c72c962ba85496f17dfa5cd39f5b1a7a98b8fc22dc26e4920e7d6bf5311081f0a\"},\"remote_proofs\":[{\"ctime\":1429044821,\"curr\":\"299f4e2461ae29d61153d712a4d5dd42b19b84ebbe3c6b6ae0cd31a27ea9587c\",\"etime\":1586724821,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"laebshade\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"666bab030581509664ad2649ff34804fff664adb96ef6a765cff47e86321d0a80f\",\"sig_type\":2},{\"ctime\":1429044876,\"curr\":\"643678824683fc60026c2ac03344fdeacff1f6bf95dc02189caefd825cf07b3f\",\"etime\":1586724876,\"prev\":\"299f4e2461ae29d61153d712a4d5dd42b19b84ebbe3c6b6ae0cd31a27ea9587c\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"laebshade\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"835dc147ade856af2320f4d40fda968bf19d45dd29934c84d5f47fc7462f9e440f\",\"sig_type\":2},{\"ctime\":1429044990,\"curr\":\"e6e5c443846efe1c1a2c8e453155cab11b0d758ec16ff78d80f10c2581bd8174\",\"etime\":1586724990,\"prev\":\"643678824683fc60026c2ac03344fdeacff1f6bf95dc02189caefd825cf07b3f\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"laebshade\"},\"proof_type\":4,\"state\":1},\"seqno\":3,\"sig_id\":\"072a030922dd777f4a640e0e61fb341a0995656adc500115ec2bb88c37d600640f\",\"sig_type\":2},{\"ctime\":1429045103,\"curr\":\"8dbfa743fd9ddf216390572927f980c34fc393dadcf6f72e6b25d49fb184d79a\",\"etime\":1586725103,\"prev\":\"e6e5c443846efe1c1a2c8e453155cab11b0d758ec16ff78d80f10c2581bd8174\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"markashields.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":4,\"sig_id\":\"1e176953f3cea0043172ae39900a5921145fdb745acf1e21835c4964c581da390f\",\"sig_type\":2},{\"ctime\":1429122182,\"curr\":\"cf5fceaa9c459258d4ffdbba3840047d38f166421049b88ddfdf11f7f18e7de3\",\"etime\":1586802182,\"prev\":\"2c32724fa9517a33521692796b830b5572dc6fbc51aa0a8ff8e6e6c9b32bf1d3\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"zzrbikes.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":15,\"sig_id\":\"a299f5a3b4a9af5c249c7c8bad66071b1d998eb466a5664e8af800d3426796b30f\",\"sig_type\":2},{\"ctime\":1429122394,\"curr\":\"c20dbded98b9cfb1585329c5889091312da3776d3bf33d2179e60cf88e9c6d18\",\"etime\":1586802394,\"prev\":\"cf5fceaa9c459258d4ffdbba3840047d38f166421049b88ddfdf11f7f18e7de3\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"gentoogeek.net\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":16,\"sig_id\":\"d7900625801bcd7c7c896a095e28f0e34473bae75e0c479019a5e75d4a7edb8c0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"6e3d94f68da91ddb895f1196d258fc552dfabfdcbb7141c1b267e82d6dd31cfc\",\"seqno\":21,\"sig_id\":\"5a778fe9f838ca35d90548bf9048305ba08cab2c3aa6ad92cda41439a27daf920f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1430419594,\"expire_in\":157680000,\"prev\":\"5cae5e8dc5d0cd6b233f12f6aeb03f45ca4c12b3fc2c337ed31112f0ebaba2a7\",\"seqno\":25,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1430419594,
            "etime": 1588099594,
            "rtime": null,
            "sig_status": 0,
            "prev": "5cae5e8dc5d0cd6b233f12f6aeb03f45ca4c12b3fc2c337ed31112f0ebaba2a7",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 26,
            "payload_hash": "b11dae23b92ba9d7b6100fb81bf201dfdcd406ccbf637876daf31f139d477af7",
            "sig_id": "b152e28cde4399080d248318b4dc10114bbc9388dcd38823bc227cb4365dcbba0f",
            "sig_id_short": "sVLijN5DmQgNJIMYtNwQEUu8k4jc04gjvCJ8",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMIYAnicbVJbSBVBGD6nLMlIuglSVLQRiJjNzuzM7p6KehHNwgsZFUk6szt7XE/u\nHs/FC2IvRVkhiRZkavbQ5Uk4kfgQSWaGZVoZIWRRFERB9ytdHmrW7K15Gf7v/75v\n5v/4h+bN9KX4qw+FKnry1kD/7Wsb477tmzetbpCYa9ZLgQYpxKcuy3aCPBKO2E5M\nCkhYo1DBJjI1BrEhY4sYHFmWpcqaShjRqKlo0ERQypIq3KinEDaMRnm27QpMFGW2\nKdD/8ENTDSADGXGAEYGEmhARjXBNhRZVMSMKxUwDSIZYVwHDBlWpylRT1wyki0KD\noi1TCKiwi0/ZKTKTLRVbDGOgEBOpgABNgZgiCDQZAI8Y5RGHVnHBdg3XcdwIIUhq\nzJIEXmMb3AvBm2WaU0mNkPt3mnDEjQnJXoFWxGLhgCeK1Yc9Vi1nZdP6MmY7pghR\nCGp4JGq7jhSQBdOI2Z6hrCCgEaQDPUvidWE7wstsj4FVogFxvGd4jbDUZQ4RkLGu\nIVNVOYFUIaqiAspNxjFBJiPQskyDM4NhRg0GIZO5yEOTRZ4qlryJqh1XCkAi/kmD\nwjNqBx0ai0e41Dg4UJrk86f4Zs+a4e2BL2XO/H/bcWZt8q/k0aazvcVXHzw/lVbP\nk6z1heXjG1/3XAq23xtZwBe/Dlf1o8lMZf9IZU5RNzh3rqBx8nzdxcnxHcu2skRT\na/rzxDZ+Q4//mDt8GOUMuvWjvuG7F1prC9xd5SeSattKF+aPL2n61Zm5YnIotahr\nS8fY3H3vd+c9q+56Uvq+s235omO9n/wZ2f5LX6qrKlPTmzsaLJyUeVbZkEj83vO4\nb+/xzx+D3/ISY63mztqhpQf6T968MvGzeVXBi4PSPN+75pKg8XD4dFrLy1t3r7Pi\ni28+Pr3/tq05oy233Yre6eteN1yXW5jW8ehV8EGB/P1y6srZR/q+ogk9P6174ENL\nyVF/zcv4H5rwJkk=\n=Wdqc\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"service\":{\"hostname\":\"jacko.io\",\"protocol\":\"http:\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1430863909,\"expire_in\":157680000,\"prev\":\"91e23015983d77e62a467470aedbe563db62ffdcebcb5bacb22b1e81a81e0575\",\"seqno\":26,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1430863909,
            "etime": 1588543909,
            "rtime": null,
            "sig_status": 0,
            "prev": "91e23015983d77e62a467470aedbe563db62ffdcebcb5bacb22b1e81a81e0575",
            "proof_id": "3195d2432b6853b6f3ceae10",
            "proof_type": 1000,
            "proof_text_check": "\n\nyMIYAnicbVJbSBVBGD6nLMlIuglSVLQRiJjNzuzM7p6KehHNwgsZFUk6szt7XE/u\nHs/FC2IvRVkhiRZkavbQ5Uk4kfgQSWaGZVoZIWRRFERB9ytdHmrW7K15Gf7v/75v\n5v/4h+bN9KX4qw+FKnry1kD/7Wsb477tmzetbpCYa9ZLgQYpxKcuy3aCPBKO2E5M\nCkhYo1DBJjI1BrEhY4sYHFmWpcqaShjRqKlo0ERQypIq3KinEDaMRnm27QpMFGW2\nKdD/8ENTDSADGXGAEYGEmhARjXBNhRZVMSMKxUwDSIZYVwHDBlWpylRT1wyki0KD\noi1TCKiwi0/ZKTKTLRVbDGOgEBOpgABNgZgiCDQZAI8Y5RGHVnHBdg3XcdwIIUhq\nzJIEXmMb3AvBm2WaU0mNkPt3mnDEjQnJXoFWxGLhgCeK1Yc9Vi1nZdP6MmY7pghR\nCGp4JGq7jhSQBdOI2Z6hrCCgEaQDPUvidWE7wstsj4FVogFxvGd4jbDUZQ4RkLGu\nIVNVOYFUIaqiAspNxjFBJiPQskyDM4NhRg0GIZO5yEOTRZ4qlryJqh1XCkAi/kmD\nwjNqBx0ai0e41Dg4UJrk86f4Zs+a4e2BL2XO/H/bcWZt8q/k0aazvcVXHzw/lVbP\nk6z1heXjG1/3XAq23xtZwBe/Dlf1o8lMZf9IZU5RNzh3rqBx8nzdxcnxHcu2skRT\na/rzxDZ+Q4//mDt8GOUMuvWjvuG7F1prC9xd5SeSattKF+aPL2n61Zm5YnIotahr\nS8fY3H3vd+c9q+56Uvq+s235omO9n/wZ2f5LX6qrKlPTmzsaLJyUeVbZkEj83vO4\nb+/xzx+D3/ISY63mztqhpQf6T968MvGzeVXBi4PSPN+75pKg8XD4dFrLy1t3r7Pi\ni28+Pr3/tq05oy233Yre6eteN1yXW5jW8ehV8EGB/P1y6srZR/q+ogk9P6174ENL\nyVF/zcv4H5rwJkk=\n",
            "proof_text_full": "==================================================================\nhttps://keybase.io/oconnor663\n--------------------------------------------------------------------\n\nI hereby claim:\n\n  * I am an admin of http://jacko.io\n  * I am oconnor663 (https://keybase.io/oconnor663) on keybase.\n  * I have a public key with fingerprint 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32\n\nTo do so, I am signing this object:\n\n{\n    \"body\": {\n        \"key\": {\n            \"fingerprint\": \"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"71876b68ad482d32\",\n            \"kid\": \"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\n            \"uid\": \"41b1f75fb55046d370608425a3208100\",\n            \"username\": \"oconnor663\"\n        },\n        \"service\": {\n            \"hostname\": \"jacko.io\",\n            \"protocol\": \"http:\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1430863909,\n    \"expire_in\": 157680000,\n    \"prev\": \"91e23015983d77e62a467470aedbe563db62ffdcebcb5bacb22b1e81a81e0575\",\n    \"seqno\": 26,\n    \"tag\": \"signature\"\n}\n\nwhich yields the signature:\n\n-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMIYAnicbVJbSBVBGD6nLMlIuglSVLQRiJjNzuzM7p6KehHNwgsZFUk6szt7XE/u\nHs/FC2IvRVkhiRZkavbQ5Uk4kfgQSWaGZVoZIWRRFERB9ytdHmrW7K15Gf7v/75v\n5v/4h+bN9KX4qw+FKnry1kD/7Wsb477tmzetbpCYa9ZLgQYpxKcuy3aCPBKO2E5M\nCkhYo1DBJjI1BrEhY4sYHFmWpcqaShjRqKlo0ERQypIq3KinEDaMRnm27QpMFGW2\nKdD/8ENTDSADGXGAEYGEmhARjXBNhRZVMSMKxUwDSIZYVwHDBlWpylRT1wyki0KD\noi1TCKiwi0/ZKTKTLRVbDGOgEBOpgABNgZgiCDQZAI8Y5RGHVnHBdg3XcdwIIUhq\nzJIEXmMb3AvBm2WaU0mNkPt3mnDEjQnJXoFWxGLhgCeK1Yc9Vi1nZdP6MmY7pghR\nCGp4JGq7jhSQBdOI2Z6hrCCgEaQDPUvidWE7wstsj4FVogFxvGd4jbDUZQ4RkLGu\nIVNVOYFUIaqiAspNxjFBJiPQskyDM4NhRg0GIZO5yEOTRZ4qlryJqh1XCkAi/kmD\nwjNqBx0ai0e41Dg4UJrk86f4Zs+a4e2BL2XO/H/bcWZt8q/k0aazvcVXHzw/lVbP\nk6z1heXjG1/3XAq23xtZwBe/Dlf1o8lMZf9IZU5RNzh3rqBx8nzdxcnxHcu2skRT\na/rzxDZ+Q4//mDt8GOUMuvWjvuG7F1prC9xd5SeSattKF+aPL2n61Zm5YnIotahr\nS8fY3H3vd+c9q+56Uvq+s235omO9n/wZ2f5LX6qrKlPTmzsaLJyUeVbZkEj83vO4\nb+/xzx+D3/ISY63mztqhpQf6T968MvGzeVXBi4PSPN+75pKg8XD4dFrLy1t3r7Pi\ni28+Pr3/tq05oy233Yre6eteN1yXW5jW8ehV8EGB/P1y6srZR/q+ogk9P6174ENL\nyVF/zcv4H5rwJkk=\n=Wdqc\n-----END PGP MESSAGE-----\n\nAnd finally, I am proving ownership of this host by posting or\nappending to this document.\n\nView my publicly-auditable identity here: https://keybase.io/oconnor663\n\n==================================================================\n",
            "check_data_json": "{\"hostname\":\"jacko.io\",\"protocol\":\"http:\"}",
            "remote_id": "http://jacko.io/keybase.txt",
            "api_url": "http://jacko.io/keybase.txt",
            "human_url": "http://jacko.io/keybase.txt",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 54,
            "hard_fail_count": 0,
            "last_check": 1434483168,
            "last_success": 1434483168,
            "version": 3,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 27,
            "payload_hash": "e6ed42cb01be8914b2fba11faffd431729fecdd21c2e1d1470c58167c7122aa7",
            "sig_id": "4e965585ac3389cc9c86937c5f632b43d8487d6e723767ad2b5c325ee54c1c9d0f",
            "sig_id_short": "TpZVhawzicychpN8X2MrQ9hIfW5yN2etK1wy",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMSBAnicpVVbiFVVGJ6ZrEwRS+khMJA9KhGjrvtlKlQs9MEefJgMHDmu65zdzDln\n3GePOshkQWmWhlBBOKFSGinYBaEkLe3CIBheMinIF/HBLhDB4IhE0L/PjJOTBUXn\n4bD3Xotv/d/l/9eX025rmtK8bmt3+dDy+aT51NDivqaOFb/HTYmt+f6kfVPietJQ\nzYunqqmEpD3pDv3W1MOCtDZ7Q7BJW7I+ZPW0VoUlaZHBTmuvPdJRBsIYCyYyh4nx\nXjiBiCJck2SgrYApQGNa7QpZb5YWZyRcGcK4p15Zwh3mUbhAY4wSKymsUMYzRTwl\ncGq5Vs8nVJM0MEupLwq5dX93YwFhhGlAnAoijCdUKBGUJNFIbgUz3CpEMZQokeXO\nSCOt9Fo5quEFSrcKG4IMwPU14Bi2OEoeLeeICU8lEkgxwg0lSGGEio31kI0pV3O1\narWWCUELAfLMuO5CAqg/dfXiKfWlcTFVW9Jj6jnwKbmyAZGSdsyIZFoLzSbAunpu\nKqZeYDaKotFRrAJn3PFIZQieORsZ08QTi7FOxsUv9JpoALEBU484CdoYLqQXHASQ\nGkfPkcScRautoXqCoN4zLanUoDdx2ESqrSeCcmIsZU5G54TFzhKpkPLOWWmNN5Z4\nGxSl1GIQFGrPQqWWh1JvVqtFUGM1RC9PK6O0mWZYYt2WuL4sg1MD9Q4zZkwwwjJv\nJTHM0qiRUZFaKTR1SoiChUA8KqM0/EUUvAKDCPJQfhgD54oIMgrem4X1Bbj0MTov\nIjFKGucM40KoQK3GgkqwGnsFoWDKhSLTwkXBhFbWaeYZ5jIZ51Lo2+DTaKRycN0l\nb3JTeqpeODzeUfmGNM9DlvyTqw2IUt7fC99JWwILeVE5rNTDumpt9GPaNRp9IqyT\nRdmQAkqc804xB3JzRznVRSy8CNCjKkYMvUkCKMWBLVdeRWs1isko2th5A21/NUIx\nOW6EpAgyCbGg1EeQTFPEFHJgPfc8BMExtQ4ZpzCI6hx3UkfoMeICfIP+u8WIBvgN\nI/6/y//NiK40L/fZf+UD/Tsf6E0+KEU9IZg6hwVynntHjYswHSWjlsnApbWYRKRl\nNIJ6Q3xwBjJkbJCEs3CLD2sax5Ryk/YUJfea/p6a8aWyqZeLMjnmhJBicKFAFPQ+\nwTCHifDOGEbAbCwt0YFrgbU2iBIpir5EHiEYKI3DRklgdhML75EN3sEmaYUxWHFP\nOJUQfqXBGh4dl4gq4EgRhx7ggkUYUF4gKwOsAIuBYtY1OIyNvJtujEK78XBRjJHS\nQkAkNvamWSilxQ4uhULwG08FjDBvAoF2JNZoD3XBoIVQYRsJzKII1TIE3GyEboVr\nwMNAwhFT7Rn0RZR/MiUQtdx0ASbwrZq8LwvJwOcnOic1NU9puuP2luIibJpy1903\nrsddlyc3Hc2PvHZo0pO7Hh68790l83df/PDq4CfTl+1bvuD64EPtC9+Y9cVnG195\nYX3p9JbD01fs/PHjhXPC8DvXWprQsbeHD1y89s3rW66++N3c7Zc2Pznv5NyfRp5/\necbF62d3Ht043Xz/wDyX7DjwcysZ7pzzyNBQfuGDziu8ZWttaceB90/MvLr5rXrH\nqpHz5d2Ht9/5no1+z7ODi1/t2rF8w7Gnh2trW/esWXZ571dfzzi9YNuhXedas3Pm\n1Jv79u5dOXjP/tVXZs6iU1s/HT737TMvtaw4v+S3k4ua+2dPe+Lgc7pemfzRmUsj\nq6ZWlp699/5lj47MGl7ZcfrBypGDj+/Xj/3w66m127Yt2nJ850jX8V8uDJ2JnX8A\nhtqkhQ==\n=aIyH\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"7b0a1c99d9d09f7e2444eaf4c12add6c60282592\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":8,\"last_id_change\":1427499694,\"username\":\"cstamas\"},\"id\":\"3fc318e545c5f37eed4cbf4492d2b119\",\"key\":{\"key_fingerprint\":\"2be13d052e9aa567d657d9791fd507154fb9ba39\",\"kid\":\"0101dd49737968a2c1af39bd26352ab34c7fcc6b1cb27808dccb7badab2dbe8333b10a\"},\"remote_proofs\":[{\"ctime\":1424941719,\"curr\":\"e3dc144aaea6b4db72a4b3f90a8f3b7693c86652e9605f8a89f8af0ed806020d\",\"etime\":1582621719,\"prev\":\"e7dffcd6f2a87acca45668e3b91637b551d8d9848ced6c66cf64698bc94d4157\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"cstamas\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"26bc77accfc332ccdc84cdab5c35398e54d6ec998ff1eaf2ec865cd658d8fbb90f\",\"sig_type\":2},{\"ctime\":1424941847,\"curr\":\"73027454f33dfd6f930480c52a5d5ee6513bc0ac817dfcc5c79fb802ceac8d23\",\"etime\":1582621847,\"prev\":\"e3dc144aaea6b4db72a4b3f90a8f3b7693c86652e9605f8a89f8af0ed806020d\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"cstamas\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"883d2213cc160cd5dc3acf1c9743b47e57bb12f097fa63da2decabc9abe7254e0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"c515222970b0e2845c219d926dcaa42ff117b29e596199a03276cc6b0d00694f\",\"seqno\":14,\"sig_id\":\"dd0bedcd007b6aa185d2537848891445fc57038c1630598b564f969d60b7e7030f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1431108966,\"expire_in\":157680000,\"prev\":\"b11dae23b92ba9d7b6100fb81bf201dfdcd406ccbf637876daf31f139d477af7\",\"seqno\":27,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1431108966,
            "etime": 1588788966,
            "rtime": null,
            "sig_status": 0,
            "prev": "b11dae23b92ba9d7b6100fb81bf201dfdcd406ccbf637876daf31f139d477af7",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 28,
            "payload_hash": "937fff13c30312edfe384a04f09b8714fc72f12e22ead9c30811064528a22afe",
            "sig_id": "a912d4392c93a36cd89d9136839aa339a6df9f37156a18f2dc8ed46213b055520f",
            "sig_id_short": "qRLUOSyTo2zYnZE2g5qjOabfnzcVahjy3I7U",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n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\nL3ztrpN3PrC04yN/Ov/MfwCug/Aw\n=AtX0\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"f41a7d6fbb57b25bbfc781a4496dce572f77e86e\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":15,\"last_id_change\":1430852617,\"username\":\"mitsuhiko\"},\"id\":\"67aaabb56ac60890936d56f7347ccb00\",\"key\":{\"key_fingerprint\":\"d0c6a9995649e5b407483e541d8fbe76fdd3e6f2\",\"kid\":\"0101587b8827659e835fe5a9a53f70866a0dcc8634c574452b799cec2dcd68e788e90a\"},\"remote_proofs\":[{\"ctime\":1404156355,\"curr\":\"2e611ebd672c8d2a2192ace994e0c68b8c897c815e187dd7a01a148ae870f628\",\"etime\":1561836355,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"mitsuhiko\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"86d652c363de52c28013fab6fb8f69a1880f60cda87963921cc9061aa333b20b0f\",\"sig_type\":2},{\"ctime\":1404156413,\"curr\":\"c4b72d2ddd4bc9b9d61bf5b85b867670ab0c09541b2638dd02d8abdae0b683d7\",\"etime\":1561836413,\"prev\":\"2e611ebd672c8d2a2192ace994e0c68b8c897c815e187dd7a01a148ae870f628\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"lucumr.pocoo.org\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":2,\"sig_id\":\"b5eed595d035c7df78c4c41b64258f6e4dce60a8f5ce65ec5b34d3ca95d7053e0f\",\"sig_type\":2},{\"ctime\":1404156497,\"curr\":\"27f5ab0f489054c81fe81f107151a551ea11407751b3200cc73b00cfc53694ab\",\"etime\":1561836497,\"prev\":\"c4b72d2ddd4bc9b9d61bf5b85b867670ab0c09541b2638dd02d8abdae0b683d7\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"mitsuhiko\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"ced4fccf0fb74fc4b5146e4839af435231a4f3460de7b220d6df6cd39e1053eb0f\",\"sig_type\":2},{\"ctime\":1430852524,\"curr\":\"31e4fa83b6ffb964173e79b5f5d3d0e6ec746c8d51504d4ad89e2fa8e9131633\",\"etime\":1588532524,\"prev\":\"f25be273916f6e96f5c25682a26ec80a7909be0690655d6b3edccf16bf222a21\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"mitsuhiko\"},\"proof_type\":4,\"state\":1},\"seqno\":5,\"sig_id\":\"8c5c690c601e42ec0b3e185f228af08178a9591fea3b1a385df8730ff0b1a9110f\",\"sig_type\":2},{\"ctime\":1430852571,\"curr\":\"8b7528d662cb7d89883146a0fa13af9e630b81732516d7e163e75ad3d5c295e3\",\"etime\":1588532571,\"prev\":\"31e4fa83b6ffb964173e79b5f5d3d0e6ec746c8d51504d4ad89e2fa8e9131633\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"the_mitsuhiko\"},\"proof_type\":6,\"state\":1},\"seqno\":6,\"sig_id\":\"76ea910f1781cbded6ae954cbcb250a15417c7845a23a8d7d3f83cffcf53b56c0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"8b7528d662cb7d89883146a0fa13af9e630b81732516d7e163e75ad3d5c295e3\",\"seqno\":6,\"sig_id\":\"76ea910f1781cbded6ae954cbcb250a15417c7845a23a8d7d3f83cffcf53b56c0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1433274004,\"expire_in\":157680000,\"prev\":\"e6ed42cb01be8914b2fba11faffd431729fecdd21c2e1d1470c58167c7122aa7\",\"seqno\":28,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1433274004,
            "etime": 1590954004,
            "rtime": null,
            "sig_status": 0,
            "prev": "e6ed42cb01be8914b2fba11faffd431729fecdd21c2e1d1470c58167c7122aa7",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 29,
            "payload_hash": "9efc3ff3f6515f5c720b660bb8f705cbc242c4859e666d965243ed99ad7aedb1",
            "sig_id": "234f77b28a058b7625cbd73ebe28e2c09a4f5780f08657cec0e6a71ccabad1520f",
            "sig_id_short": "I093sooFi3Yly9c-vijiwJpPV4DwhlfOwOan",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMORAnictVRdiFVVFL6m5jhlTkQZMqKdMkNG279n7zNTyBRDgRGCpmnFdf/OPd6Z\nc2bOOTPjKBNEUUiSBkGU+NCv9TAyPRSiEjqZKA1kTyERGYj2o/MQgZIErXOdbAZ6\nrPNw2Peudb71fd9ae51YMLvSPKv/lXpt9PFVZNZXJ4qBylPbn35wZ6BTOxy07wxM\nT+ySojwlqtcF7UHdDWuVu9VxumzI6aAtGHRZHqcJhDzDStjQa82FJlxrb4TEirEo\ntMZxQbwQToYuGGkrYUpQHyfdLuvL4rJGwKUijFtqJXxuMPehcdR7L7AUoQ6lskwS\nSwlUraV5MYNN0MCsxhb+/Zf8eiOAMMLUIU5DEipLaAhsJPBSguuQKa4lopjwSCDN\njRJKaGEjaWgEPyQoAjUEKYAbaMAxrLEX3GvOEQstFShEkhGuKEESI1Qm5i6bci41\naZKkWRjS0oAiU6ZeWgD8Y5OXp9hWb5hJ24IelRegp2pqCkwK2jGjNGKERmQGbHec\n5LU4KTEbpELlOBLMIQRCkKPYUOGItT7iTkkcBTfML/2a2QDCpPVEIu18pIlVhkUO\nK2NDRDAlBlqhCObIzjDUERcKB1k4dFoxbOCEQ2IElcJFmFsLL6oZEVIjxzXDTtsQ\nHHdYS2YFGArcM9ebFq7al6WpBzeegdEr4t5pslnYFpiBLIOqEkmpHGhhWlEUychS\nwLfQMG0k48TKslU8ohHHJb6JvBZcCIJdGCopgb6bAucRUL0OPsWgdKXBojH+NWfq\nVasKVd2Wl325cQ+KobgoXDazxd0qrrskiYfTTO0YKGU1kKrFcB+EoW95oYqyLERy\n15+kcIRT3H19biVSAkYtEswbGEgqBFcMVCAvpbPGG+QsxUwoyrDHBlzxFFPLhLRY\nIIuQD66jTdUbea5RplqouKek3qeGe1JlqzWV1/4bG/8fESPl/WhImLom07ZMad30\nyeAw5hz6ub0vzlw1LjO4CCWCp3TfDQJIRAWMLqaGltfbWe+oZAoxjyItBQaisAMg\nQIhTNoIsiTEKwQHYR0R5949QEgEz1Q2YoDdRxUAG22z82LNzKrOaKzfPvalcnpXm\n+S1/r9QncFPl5OqJg6eOPd/54S0X1nYt/rL96KKW8fM/n9nwOVn85Lyuw3Gw7sij\nh9X5Vzv2T34698+fks6Jesuauw+8u3LN7vtHO3cfWHR21+/37l12rfXIqjP9nc0L\nJ17Yt/SNP35sb58vjnUtObt29Nal9JHDJ+v6vb2tmy9/e6WjFR+qNB0fX7fhwtqL\ndxRXP3n/t/1fn/6m5dTGhRfH4i9ejza9s/7jK6uHBt7a80B+9fgP62f7cxuTdHLH\noYe2nFfff9ax5ZeuXx+bvWvf8tvnH73z4MrxPcGC7+5qbRn56J5zm5dc3pSNdaxJ\nx15bsWLxbQdeWt7U2zK2dXLe2y+a0W3L77tUXHsz33r65YfHF3WNTFwcvDTnA/oX\nDIMM3w==\n=uNE8\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"f41a7d6fbb57b25bbfc781a4496dce572f77e86e\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":3,\"last_id_change\":1433942392,\"username\":\"ginshin\"},\"id\":\"6ae5074e002590e31c37e2ddf95ea819\",\"key\":{\"key_fingerprint\":\"248df280bef9b2dac49e1acd602132cfffa2150d\",\"kid\":\"0101e2e67eacd16eba41ccd1162c7387e915dd9153b4278b0e5b41ebd675be1b84d70a\"},\"remote_proofs\":[{\"ctime\":1433942346,\"curr\":\"8088ae8194ba30989d33b4da7abc8452d85b81593951d70ac9fb757721e66a88\",\"etime\":1591622346,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"gaikenniyorazu\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"80a79ca974fc5b63775a49390f88edcfc0ed3147a341f1c942f313d478d170d00f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"8088ae8194ba30989d33b4da7abc8452d85b81593951d70ac9fb757721e66a88\",\"seqno\":1,\"sig_id\":\"80a79ca974fc5b63775a49390f88edcfc0ed3147a341f1c942f313d478d170d00f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1433950255,\"expire_in\":157680000,\"prev\":\"937fff13c30312edfe384a04f09b8714fc72f12e22ead9c30811064528a22afe\",\"seqno\":29,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1433950255,
            "etime": 1591630255,
            "rtime": null,
            "sig_status": 0,
            "prev": "937fff13c30312edfe384a04f09b8714fc72f12e22ead9c30811064528a22afe",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        }
    ],
    "keys": [
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v2\n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\n8PZq\n=Bh81\n-----END PGP PUBLIC KEY BLOCK-----"
    ]
}
`,
	"jack_chain_missing_first_link.json": `{
    "username": "oconnor663",
    "uid": "41b1f75fb55046d370608425a3208100",
    "chain": [
        {
            "seqno": 2,
            "payload_hash": "382e1d68b4b81c6643483a74715ac0964e291fb2b52c4e9fa6c73d304ab9108f",
            "sig_id": "9d5b60c0d486f34089a922bcc5e77fb643b9781cbcc894af7e123fad50645d6a0f",
            "sig_id_short": "nVtgwNSG80CJqSK8xed_tkO5eBy8yJSvfhI_",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kltoFFcYx3et1VWrWBQJsaCdFq3tusy5zhn1QYOUVh9MglQq1vWcM+dkJ5eZ\ndWaTmNqIWmlTrDRtpaLSghdaIqYS6614wbsWry/6oEVoi6AvkgRFVHA9s1iqL52n\nme/8vv/3/893uke/ksqkl3/ZVOj9YDpMX7grUgs3LR610hKh12HNWGnJZl8FpeQt\n4C3KmmE1qQ7BY5Xzw8lB6KlcYzz5OZO12lQU+2FgKDvn5JDVmU3wpFn7QYOKipGf\naFmEcYiJhzwmIJGAaCoV0lo7gDlUUMY9zKCHoJEshHHppalWRTPve6Za4WsomzPX\n8HMrfGvlAAMBtEO0IMTG1EOOTW2GIeEI2gzYdgLGKnoeKZRhEIQRpRXHLSpqalb5\nKAwrsWXJTyCAgQttYiPHeOJxwbR51HGl4EBShRHChAosJQEMONz2kKnbLuFcJY1A\nEQKAhK459Lh0lJBcciopcIiHkbax4Awg1xFMCcQEEwQJ6EmICMXCVo4rHMk5ocYC\n0tyMMTNNiFgtD0JjzrjDbmfyHbX5Ur2wrga/VGgV/5e31FFMiu1K5A2yK+nPCz/w\nzMZeXCkw6Mv/AiCQtdSKoh+pvJ8QxKHMNk/WKkaqzWgKDIFARECkJfE449TVXJiY\nTGuEKXO1CaVtGzMtGdcMQm42JAiAQDLgov8SQuOTNxjJ2G8IeKk1UlbnV+k3hqbS\nmdSwV4ckdzY1csTYfy9yqjeTOji8+KB672c9PasmiqPhVFS+N38n+e2cqvqkdikt\nrC4e7e//R7w57/j8R69lahr3/Tx4/mI0ru/bqku98Cy41f/37acXa8+sv5a7scb9\n/tjA/m/WvfPk5M3amikb3qvecvvWYOav4VtP73n9u3dnLdzx9YGldX0f7uiCi7qf\n/vrWD79kV/fdf7v2fPHwPae+ceKPB2bWb/+U7qsfv/Exm3R/wF3c/WTtwZ5j8wZ3\nnZnQdm7KF3WNh6ovjKH4Sv+GzPvTRi+b+cd6sK5c9/Dy6SHtH338aFthSbpqQupU\nze/l9s/317XUXyuvmj30z40DW6ubx95Zc6hq77jC7gXl+PD18XPOnuw6Me2nI11X\nN496Bg==\n=diQc\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419205037,\"hash\":\"d679cba1c6e433456b4cc51817a0d3a1c095aae41921e5511c29817dac7ebcaca6c6175d43f04ba81397b8eb38b8b53b2dc23564b0e79b7caa562053facc56e4\",\"seqno\":114149},\"service\":{\"name\":\"github\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205131,\"expire_in\":157680000,\"prev\":\"b421b35b23fc5da8a69fab3978ff34689fcaaf0048fc8af822aa32b5121c8193\",\"seqno\":2,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1419205131,
            "etime": 1576885131,
            "rtime": null,
            "sig_status": 0,
            "prev": "b421b35b23fc5da8a69fab3978ff34689fcaaf0048fc8af822aa32b5121c8193",
            "proof_id": "034487589bf1339ef37e6210",
            "proof_type": 3,
            "proof_text_check": "\n\nowF9kltoFFcYx3et1VWrWBQJsaCdFq3tusy5zhn1QYOUVh9MglQq1vWcM+dkJ5eZ\ndWaTmNqIWmlTrDRtpaLSghdaIqYS6614wbsWry/6oEVoi6AvkgRFVHA9s1iqL52n\nme/8vv/3/893uke/ksqkl3/ZVOj9YDpMX7grUgs3LR610hKh12HNWGnJZl8FpeQt\n4C3KmmE1qQ7BY5Xzw8lB6KlcYzz5OZO12lQU+2FgKDvn5JDVmU3wpFn7QYOKipGf\naFmEcYiJhzwmIJGAaCoV0lo7gDlUUMY9zKCHoJEshHHppalWRTPve6Za4WsomzPX\n8HMrfGvlAAMBtEO0IMTG1EOOTW2GIeEI2gzYdgLGKnoeKZRhEIQRpRXHLSpqalb5\nKAwrsWXJTyCAgQttYiPHeOJxwbR51HGl4EBShRHChAosJQEMONz2kKnbLuFcJY1A\nEQKAhK459Lh0lJBcciopcIiHkbax4Awg1xFMCcQEEwQJ6EmICMXCVo4rHMk5ocYC\n0tyMMTNNiFgtD0JjzrjDbmfyHbX5Ur2wrga/VGgV/5e31FFMiu1K5A2yK+nPCz/w\nzMZeXCkw6Mv/AiCQtdSKoh+pvJ8QxKHMNk/WKkaqzWgKDIFARECkJfE449TVXJiY\nTGuEKXO1CaVtGzMtGdcMQm42JAiAQDLgov8SQuOTNxjJ2G8IeKk1UlbnV+k3hqbS\nmdSwV4ckdzY1csTYfy9yqjeTOji8+KB672c9PasmiqPhVFS+N38n+e2cqvqkdikt\nrC4e7e//R7w57/j8R69lahr3/Tx4/mI0ru/bqku98Cy41f/37acXa8+sv5a7scb9\n/tjA/m/WvfPk5M3amikb3qvecvvWYOav4VtP73n9u3dnLdzx9YGldX0f7uiCi7qf\n/vrWD79kV/fdf7v2fPHwPae+ceKPB2bWb/+U7qsfv/Exm3R/wF3c/WTtwZ5j8wZ3\nnZnQdm7KF3WNh6ovjKH4Sv+GzPvTRi+b+cd6sK5c9/Dy6SHtH338aFthSbpqQupU\nze/l9s/317XUXyuvmj30z40DW6ubx95Zc6hq77jC7gXl+PD18XPOnuw6Me2nI11X\nN496Bg==\n",
            "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am oconnor663 on github.\n  * I am oconnor663 (https://keybase.io/oconnor663) on keybase.\n  * I have a public key whose fingerprint is 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32\n\nTo claim this, I am signing this object:\n\n` + "`" + `` + "`" + `` + "`" + `json\n{\n    \"body\": {\n        \"client\": {\n            \"name\": \"keybase.io node.js client\",\n            \"version\": \"0.7.3\"\n        },\n        \"key\": {\n            \"fingerprint\": \"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"71876B68AD482D32\",\n            \"uid\": \"41b1f75fb55046d370608425a3208100\",\n            \"username\": \"oconnor663\"\n        },\n        \"merkle_root\": {\n            \"ctime\": 1419205037,\n            \"hash\": \"d679cba1c6e433456b4cc51817a0d3a1c095aae41921e5511c29817dac7ebcaca6c6175d43f04ba81397b8eb38b8b53b2dc23564b0e79b7caa562053facc56e4\",\n            \"seqno\": 114149\n        },\n        \"service\": {\n            \"name\": \"github\",\n            \"username\": \"oconnor663\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1419205131,\n    \"expire_in\": 157680000,\n    \"prev\": \"b421b35b23fc5da8a69fab3978ff34689fcaaf0048fc8af822aa32b5121c8193\",\n    \"seqno\": 2,\n    \"tag\": \"signature\"\n}\n` + "`" + `` + "`" + `` + "`" + `\n\nwith the PGP key whose fingerprint is\n[58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32](https://keybase.io/oconnor663)\n(captured above as ` + "`" + `body.key.fingerprint` + "`" + `), yielding the PGP signature:\n\n` + "`" + `` + "`" + `` + "`" + `\n-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kltoFFcYx3et1VWrWBQJsaCdFq3tusy5zhn1QYOUVh9MglQq1vWcM+dkJ5eZ\ndWaTmNqIWmlTrDRtpaLSghdaIqYS6614wbsWry/6oEVoi6AvkgRFVHA9s1iqL52n\nme/8vv/3/893uke/ksqkl3/ZVOj9YDpMX7grUgs3LR610hKh12HNWGnJZl8FpeQt\n4C3KmmE1qQ7BY5Xzw8lB6KlcYzz5OZO12lQU+2FgKDvn5JDVmU3wpFn7QYOKipGf\naFmEcYiJhzwmIJGAaCoV0lo7gDlUUMY9zKCHoJEshHHppalWRTPve6Za4WsomzPX\n8HMrfGvlAAMBtEO0IMTG1EOOTW2GIeEI2gzYdgLGKnoeKZRhEIQRpRXHLSpqalb5\nKAwrsWXJTyCAgQttYiPHeOJxwbR51HGl4EBShRHChAosJQEMONz2kKnbLuFcJY1A\nEQKAhK459Lh0lJBcciopcIiHkbax4Awg1xFMCcQEEwQJ6EmICMXCVo4rHMk5ocYC\n0tyMMTNNiFgtD0JjzrjDbmfyHbX5Ur2wrga/VGgV/5e31FFMiu1K5A2yK+nPCz/w\nzMZeXCkw6Mv/AiCQtdSKoh+pvJ8QxKHMNk/WKkaqzWgKDIFARECkJfE449TVXJiY\nTGuEKXO1CaVtGzMtGdcMQm42JAiAQDLgov8SQuOTNxjJ2G8IeKk1UlbnV+k3hqbS\nmdSwV4ckdzY1csTYfy9yqjeTOji8+KB672c9PasmiqPhVFS+N38n+e2cqvqkdikt\nrC4e7e//R7w57/j8R69lahr3/Tx4/mI0ru/bqku98Cy41f/37acXa8+sv5a7scb9\n/tjA/m/WvfPk5M3amikb3qvecvvWYOav4VtP73n9u3dnLdzx9YGldX0f7uiCi7qf\n/vrWD79kV/fdf7v2fPHwPae+ceKPB2bWb/+U7qsfv/Exm3R/wF3c/WTtwZ5j8wZ3\nnZnQdm7KF3WNh6ovjKH4Sv+GzPvTRi+b+cd6sK5c9/Dy6SHtH338aFthSbpqQupU\nze/l9s/317XUXyuvmj30z40DW6ubx95Zc6hq77jC7gXl+PD18XPOnuw6Me2nI11X\nN496Bg==\n=diQc\n-----END PGP MESSAGE-----\n\n` + "`" + `` + "`" + `` + "`" + `\n\nAnd finally, I am proving ownership of the github account by posting this as a gist.\n\n### My publicly-auditable identity:\n\nhttps://keybase.io/oconnor663\n\n### From the command line:\n\nConsider the [keybase command line program](https://keybase.io/docs/command_line).\n\n` + "`" + `` + "`" + `` + "`" + `bash\n# look me up\nkeybase id oconnor663\n\n# encrypt a message to me\nkeybase encrypt oconnor663 -m 'a secret message...'\n\n# ...and more...\n` + "`" + `` + "`" + `` + "`" + `\n",
            "check_data_json": "{\"name\":\"github\",\"username\":\"oconnor663\"}",
            "remote_id": "df527ad8438a424efcec",
            "api_url": "https://gist.githubusercontent.com/oconnor663/df527ad8438a424efcec/raw/0c24241710275124f07db83ab5b6f97fc6952165/keybase.md",
            "human_url": "https://gist.github.com/df527ad8438a424efcec",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 142,
            "hard_fail_count": 0,
            "last_check": 1434467728,
            "last_success": 1434467728,
            "version": 3,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 3,
            "payload_hash": "44a567581a31b596f31745c4633feced0806f427bac69e720aecfda02279b9bb",
            "sig_id": "a5039b256eb381b395e2fe9018e931c7dbe052dc1f1cf213b9e8e930ef0feabf0f",
            "sig_id_short": "pQObJW6zgbOV4v6QGOkxx9vgUtwfHPITuejp",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kltoHUUcxk9sm9ZqSIsXfIgStwUpTU7mPrMnDdUawWiIgkIxUk/mtidrkt2T\n3ZOYGKMFi02EqiCaFvom7dFqHqrRqOhDQdBYifqgra3QSERaxQuitUorzh4qti/O\n01x+/2++b/7zXMOK3Jq64d0D/bN3tqK6o2dU7v6Z3hcnPBWbca8w4enB0EaVbBbJ\nIesVvAE7rmRq82HcHMXG5h9Omy8yLd6oTdIwjhwF8jyPvcmWDM+KgzAq2aSchJmW\nR4VEhBpshEJUQxowbXEQBBwKzhQT0hCBDEZOsj9OK5fd6tU0i6FxuzV+GxO3dTq+\ns8aP1A4IVDDgNFCUAsIM5oABQRCVGAEBAcjA1CYXI8U6jqI4YazmeMgmA4O2mMRx\nLbauhBkECfQRoJAR50mm/a5MCetTbQgEkkAmsG8UVr6EPsYKSYWg70NEEUOMBZZw\nHViojUvGCIIKcQmUIpJBjQx3qSH2XQjOKIQ+0BhA3z2L5BAjoxhWGLoCLgjnwAgI\nEcbEhUjtcBQ7c84dRZPZOhkNtb2kXYk1Jqz8X97KeDnbfMSqokMOZfVFFUbGdezS\nlkKHXv4WiIMWz46Vw8QWw4ygnAngRotXTuyo08QCWWiYUEQJqBkjmAgsOeGQSg18\nRizyYaCQokgT6weSaY4NBkQqHwIR/JcQO5+y5CTTsBTJykhivcnpuqaVubo1ufpV\nV2R/Nrf2ynX/fuSNJ1b/vaF1+be0Mdf1Tld0w9lVi3ubCu/KRXnhlpOf98z1jd13\nCOyZOn3q+lj90Lc4N/P2L4WP83eLXYdv36zr35w+cu1rbe9tuKP86plkHiwffKvh\ncPT6tvOn/Zd3/rV759LWA6a1c/aJ6vvb0dahPY+dO9n42UfV/cvtv3+3aWFkxYmk\ntGVmqffPVx7atfR0Kej+dKHvp59fWMnEV6ee2etddWx9YcfqWx+/50hPd1fV7mi/\naazj7Lpvp+YXrut444NfG/ddaKebJo43w3T25gce7H6pybtmcbrx+z+eXf/j5vPD\nc0+Ob+FTg4/e9c3XN66tHudH91ePbU8+vHqk3FD/VFvH/L2fHKx/vld/ea4t/uIf\n=i2ny\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419205164,\"hash\":\"b8e95cd410a416839db3b9a1933b2ab219912526266fe47cfe1cdd486421b27a0bb4a61c2d776b1394827651190c30195f6a7132db63b317a0784770d8112334\",\"seqno\":114152},\"service\":{\"name\":\"reddit\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205270,\"expire_in\":157680000,\"prev\":\"382e1d68b4b81c6643483a74715ac0964e291fb2b52c4e9fa6c73d304ab9108f\",\"seqno\":3,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1419205270,
            "etime": 1576885270,
            "rtime": null,
            "sig_status": 0,
            "prev": "382e1d68b4b81c6643483a74715ac0964e291fb2b52c4e9fa6c73d304ab9108f",
            "proof_id": "7374183938ae0741e08d3c10",
            "proof_type": 4,
            "proof_text_check": "\n\nowF9kltoHUUcxk9sm9ZqSIsXfIgStwUpTU7mPrMnDdUawWiIgkIxUk/mtidrkt2T\n3ZOYGKMFi02EqiCaFvom7dFqHqrRqOhDQdBYifqgra3QSERaxQuitUorzh4qti/O\n01x+/2++b/7zXMOK3Jq64d0D/bN3tqK6o2dU7v6Z3hcnPBWbca8w4enB0EaVbBbJ\nIesVvAE7rmRq82HcHMXG5h9Omy8yLd6oTdIwjhwF8jyPvcmWDM+KgzAq2aSchJmW\nR4VEhBpshEJUQxowbXEQBBwKzhQT0hCBDEZOsj9OK5fd6tU0i6FxuzV+GxO3dTq+\ns8aP1A4IVDDgNFCUAsIM5oABQRCVGAEBAcjA1CYXI8U6jqI4YazmeMgmA4O2mMRx\nLbauhBkECfQRoJAR50mm/a5MCetTbQgEkkAmsG8UVr6EPsYKSYWg70NEEUOMBZZw\nHViojUvGCIIKcQmUIpJBjQx3qSH2XQjOKIQ+0BhA3z2L5BAjoxhWGLoCLgjnwAgI\nEcbEhUjtcBQ7c84dRZPZOhkNtb2kXYk1Jqz8X97KeDnbfMSqokMOZfVFFUbGdezS\nlkKHXv4WiIMWz46Vw8QWw4ygnAngRotXTuyo08QCWWiYUEQJqBkjmAgsOeGQSg18\nRizyYaCQokgT6weSaY4NBkQqHwIR/JcQO5+y5CTTsBTJykhivcnpuqaVubo1ufpV\nV2R/Nrf2ynX/fuSNJ1b/vaF1+be0Mdf1Tld0w9lVi3ubCu/KRXnhlpOf98z1jd13\nCOyZOn3q+lj90Lc4N/P2L4WP83eLXYdv36zr35w+cu1rbe9tuKP86plkHiwffKvh\ncPT6tvOn/Zd3/rV759LWA6a1c/aJ6vvb0dahPY+dO9n42UfV/cvtv3+3aWFkxYmk\ntGVmqffPVx7atfR0Kej+dKHvp59fWMnEV6ee2etddWx9YcfqWx+/50hPd1fV7mi/\naazj7Lpvp+YXrut444NfG/ddaKebJo43w3T25gce7H6pybtmcbrx+z+eXf/j5vPD\nc0+Ob+FTg4/e9c3XN66tHudH91ePbU8+vHqk3FD/VFvH/L2fHKx/vld/ea4t/uIf\n",
            "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am [oconnor663](https://www.reddit.com/user/oconnor663) on reddit.\n  * I am [oconnor663](https://keybase.io/oconnor663) on keybase.\n  * I have a public key whose fingerprint is 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32\n\nTo claim this, I am signing this object:\n\n\n    {\n        \"body\": {\n            \"client\": {\n                \"name\": \"keybase.io node.js client\",\n                \"version\": \"0.7.3\"\n            },\n            \"key\": {\n                \"fingerprint\": \"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\n                \"host\": \"keybase.io\",\n                \"key_id\": \"71876B68AD482D32\",\n                \"uid\": \"41b1f75fb55046d370608425a3208100\",\n                \"username\": \"oconnor663\"\n            },\n            \"merkle_root\": {\n                \"ctime\": 1419205164,\n                \"hash\": \"b8e95cd410a416839db3b9a1933b2ab219912526266fe47cfe1cdd486421b27a0bb4a61c2d776b1394827651190c30195f6a7132db63b317a0784770d8112334\",\n                \"seqno\": 114152\n            },\n            \"service\": {\n                \"name\": \"reddit\",\n                \"username\": \"oconnor663\"\n            },\n            \"type\": \"web_service_binding\",\n            \"version\": 1\n        },\n        \"ctime\": 1419205270,\n        \"expire_in\": 157680000,\n        \"prev\": \"382e1d68b4b81c6643483a74715ac0964e291fb2b52c4e9fa6c73d304ab9108f\",\n        \"seqno\": 3,\n        \"tag\": \"signature\"\n    }\n\nwith the PGP key referenced above, yielding the PGP signature:\n\n\n    -----BEGIN PGP MESSAGE-----\n    Version: GnuPG v2\n    \n    owF9kltoHUUcxk9sm9ZqSIsXfIgStwUpTU7mPrMnDdUawWiIgkIxUk/mtidrkt2T\n    3ZOYGKMFi02EqiCaFvom7dFqHqrRqOhDQdBYifqgra3QSERaxQuitUorzh4qti/O\n    01x+/2++b/7zXMOK3Jq64d0D/bN3tqK6o2dU7v6Z3hcnPBWbca8w4enB0EaVbBbJ\n    IesVvAE7rmRq82HcHMXG5h9Omy8yLd6oTdIwjhwF8jyPvcmWDM+KgzAq2aSchJmW\n    R4VEhBpshEJUQxowbXEQBBwKzhQT0hCBDEZOsj9OK5fd6tU0i6FxuzV+GxO3dTq+\n    s8aP1A4IVDDgNFCUAsIM5oABQRCVGAEBAcjA1CYXI8U6jqI4YazmeMgmA4O2mMRx\n    LbauhBkECfQRoJAR50mm/a5MCetTbQgEkkAmsG8UVr6EPsYKSYWg70NEEUOMBZZw\n    HViojUvGCIIKcQmUIpJBjQx3qSH2XQjOKIQ+0BhA3z2L5BAjoxhWGLoCLgjnwAgI\n    EcbEhUjtcBQ7c84dRZPZOhkNtb2kXYk1Jqz8X97KeDnbfMSqokMOZfVFFUbGdezS\n    lkKHXv4WiIMWz46Vw8QWw4ygnAngRotXTuyo08QCWWiYUEQJqBkjmAgsOeGQSg18\n    RizyYaCQokgT6weSaY4NBkQqHwIR/JcQO5+y5CTTsBTJykhivcnpuqaVubo1ufpV\n    V2R/Nrf2ynX/fuSNJ1b/vaF1+be0Mdf1Tld0w9lVi3ubCu/KRXnhlpOf98z1jd13\n    COyZOn3q+lj90Lc4N/P2L4WP83eLXYdv36zr35w+cu1rbe9tuKP86plkHiwffKvh\n    cPT6tvOn/Zd3/rV759LWA6a1c/aJ6vvb0dahPY+dO9n42UfV/cvtv3+3aWFkxYmk\n    tGVmqffPVx7atfR0Kej+dKHvp59fWMnEV6ee2etddWx9YcfqWx+/50hPd1fV7mi/\n    aazj7Lpvp+YXrut444NfG/ddaKebJo43w3T25gce7H6pybtmcbrx+z+eXf/j5vPD\n    c0+Ob+FTg4/e9c3XN66tHudH91ePbU8+vHqk3FD/VFvH/L2fHKx/vld/ea4t/uIf\n    =i2ny\n    -----END PGP MESSAGE-----\n\nFinally, I am proving my reddit account by posting it in [KeybaseProofs](https://www.reddit.com/r/KeybaseProofs).\n",
            "check_data_json": "{\"name\":\"reddit\",\"username\":\"oconnor663\"}",
            "remote_id": "t3_2q0kk1",
            "api_url": "https://www.reddit.com/r/KeybaseProofs/comments/2q0kk1/my_keybase_proof_redditoconnor663/.json",
            "human_url": "https://www.reddit.com/r/KeybaseProofs/comments/2q0kk1/my_keybase_proof_redditoconnor663/",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 4,
            "hard_fail_count": 0,
            "last_check": 1434485079,
            "last_success": 1434485079,
            "version": 9,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 4,
            "payload_hash": "00279b0704b01ac6171beef1a5718a8de52d5f0fd0ab63dc947dd6f111bb673f",
            "sig_id": "89e70d05df9c659d1ad8b0e2ead5790cc500d29345979af2e5807edefab5666f0f",
            "sig_id_short": "iecNBd-cZZ0a2LDi6tV5DMUA0pNFl5ry5YB-",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kk1sFVUYhu+9SlsKJARRiGJoJkIa017P/zmDC6J2URMEF9UoKNfzN+3YMnOZ\nub20aa/SVetCRRaUSMPKGCJlgQqYaGpK1DRFY4yp1aj8JBgEqQJBgwsTz1wxwsbJ\nLM6ceb73vO/3nT3L7sg15XeO9vYc6WxH+VMXVa5rfOuVIU/FZtDbMOTpvtBGlWwV\nyR3W2+D12kElU1sM45YoNrb4Ytpyk2nzqjZJwzhyFCjyIvZqbRmeFQdh1G2TchJm\nWh4VEhFqsBEKUQ1pwLTFQRBwKDhTTEhDBDIYOcmeOK3cdqpX1yyFxu3W+UeZeKTD\n8R11vr/+g0AFA04DRSkgzGAOGBAEUYkREBCADExtcjNSrOMoihPG6o532KS3z5aS\nOK7H1pUwgyCBPgIUceg8ybTHlVkCiK+sltkJDHJiqDBCIy10AJUvlQCWc0JsYAJo\nKEQGYUZ96rtXQGYA0pJqQYEwhHLgK2PdkktnllJOhAOkJpZQoaRUriAwWAPMfCNc\nBfZdiNTujGJnzrmjuJZ9J9VQ21vGpeMwyjr3f4krg+Vsc5dVpXTyH4WSCiPjZnbr\nUKFDb+8GZqDNswPlMLGlMCMoZwK4p80rJ7aaDYJIyjgVUGKoqM8C7PpENWEYB1Zb\nAwRgAUFcSc18yxGQVgdGAoS4r3yl/stInE/Z7STTsDuSlf7EerVX8mvuzOWbcg2L\nCtmtzTUvXv7vVd5Wbsq9e7JhazT+wIGZqOdc4zNdy/5sPDDxzfaBLYdHPi8Uvrs4\nvef15hfkj61zuSvrTo7WVnVVv7w8NNCyce10n/71s2eX/3D/iYYnL7//06FhOLl/\nYWV/x0O/zc9utBfODy+91vpS68d7fz/2xFV4+vRrl8pjbMnKa4eO8qOjm1bPXT0y\nvmK2c8Ru3l+899hSVug4c8+3xS/u27Tm/NjM3ZfOPpVfe2JbemPFxKmpwvW3v1qY\n2vfgDTC1+K/Hdg3Mr9+3W8y9sb755erh7Wc/fPiD4teLDrafWfLR7PePH9zCnnt1\n83vz7+xddVfjW+uuT/5SrM0Nf7L7XLWl1E7109Njx9+80PfpBH7+55k/VneOHPcW\n/gY=\n=2nAo\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419205271,\"hash\":\"e4049beca46d36174d58d8c2c8cf1b9ab80e7744efdf1d512d2365959959816d02ca5c8508d45709bde08d7a70655748d02ac4e458baab16dfd3c0369d88d439\",\"seqno\":114153},\"service\":{\"name\":\"coinbase\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205360,\"expire_in\":157680000,\"prev\":\"44a567581a31b596f31745c4633feced0806f427bac69e720aecfda02279b9bb\",\"seqno\":4,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1419205360,
            "etime": 1576885360,
            "rtime": null,
            "sig_status": 0,
            "prev": "44a567581a31b596f31745c4633feced0806f427bac69e720aecfda02279b9bb",
            "proof_id": "de92300787097c3f7ae36910",
            "proof_type": 5,
            "proof_text_check": "\n\nowF9kk1sFVUYhu+9SlsKJARRiGJoJkIa017P/zmDC6J2URMEF9UoKNfzN+3YMnOZ\nub20aa/SVetCRRaUSMPKGCJlgQqYaGpK1DRFY4yp1aj8JBgEqQJBgwsTz1wxwsbJ\nLM6ceb73vO/3nT3L7sg15XeO9vYc6WxH+VMXVa5rfOuVIU/FZtDbMOTpvtBGlWwV\nyR3W2+D12kElU1sM45YoNrb4Ytpyk2nzqjZJwzhyFCjyIvZqbRmeFQdh1G2TchJm\nWh4VEhFqsBEKUQ1pwLTFQRBwKDhTTEhDBDIYOcmeOK3cdqpX1yyFxu3W+UeZeKTD\n8R11vr/+g0AFA04DRSkgzGAOGBAEUYkREBCADExtcjNSrOMoihPG6o532KS3z5aS\nOK7H1pUwgyCBPgIUceg8ybTHlVkCiK+sltkJDHJiqDBCIy10AJUvlQCWc0JsYAJo\nKEQGYUZ96rtXQGYA0pJqQYEwhHLgK2PdkktnllJOhAOkJpZQoaRUriAwWAPMfCNc\nBfZdiNTujGJnzrmjuJZ9J9VQ21vGpeMwyjr3f4krg+Vsc5dVpXTyH4WSCiPjZnbr\nUKFDb+8GZqDNswPlMLGlMCMoZwK4p80rJ7aaDYJIyjgVUGKoqM8C7PpENWEYB1Zb\nAwRgAUFcSc18yxGQVgdGAoS4r3yl/stInE/Z7STTsDuSlf7EerVX8mvuzOWbcg2L\nCtmtzTUvXv7vVd5Wbsq9e7JhazT+wIGZqOdc4zNdy/5sPDDxzfaBLYdHPi8Uvrs4\nvef15hfkj61zuSvrTo7WVnVVv7w8NNCyce10n/71s2eX/3D/iYYnL7//06FhOLl/\nYWV/x0O/zc9utBfODy+91vpS68d7fz/2xFV4+vRrl8pjbMnKa4eO8qOjm1bPXT0y\nvmK2c8Ru3l+899hSVug4c8+3xS/u27Tm/NjM3ZfOPpVfe2JbemPFxKmpwvW3v1qY\n2vfgDTC1+K/Hdg3Mr9+3W8y9sb755erh7Wc/fPiD4teLDrafWfLR7PePH9zCnnt1\n83vz7+xddVfjW+uuT/5SrM0Nf7L7XLWl1E7109Njx9+80PfpBH7+55k/VneOHPcW\n/gY=\n",
            "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am oconnor663 on coinbase.\n  * I am oconnor663 on keybase.\n  * I have a public key whose fingerprint is 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32\n\nTo claim this, I am signing this object:\n\n` + "`" + `` + "`" + `` + "`" + `json\n{\n    \"body\": {\n        \"client\": {\n            \"name\": \"keybase.io node.js client\",\n            \"version\": \"0.7.3\"\n        },\n        \"key\": {\n            \"fingerprint\": \"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"71876B68AD482D32\",\n            \"uid\": \"41b1f75fb55046d370608425a3208100\",\n            \"username\": \"oconnor663\"\n        },\n        \"merkle_root\": {\n            \"ctime\": 1419205271,\n            \"hash\": \"e4049beca46d36174d58d8c2c8cf1b9ab80e7744efdf1d512d2365959959816d02ca5c8508d45709bde08d7a70655748d02ac4e458baab16dfd3c0369d88d439\",\n            \"seqno\": 114153\n        },\n        \"service\": {\n            \"name\": \"coinbase\",\n            \"username\": \"oconnor663\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1419205360,\n    \"expire_in\": 157680000,\n    \"prev\": \"44a567581a31b596f31745c4633feced0806f427bac69e720aecfda02279b9bb\",\n    \"seqno\": 4,\n    \"tag\": \"signature\"\n}\n` + "`" + `` + "`" + `` + "`" + `\n\nwith the PGP key whose fingerprint is 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32 (captured above as ` + "`" + `body.key.fingerprint` + "`" + `), yielding the PGP signature:\n\n` + "`" + `` + "`" + `` + "`" + `\n-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kk1sFVUYhu+9SlsKJARRiGJoJkIa017P/zmDC6J2URMEF9UoKNfzN+3YMnOZ\nub20aa/SVetCRRaUSMPKGCJlgQqYaGpK1DRFY4yp1aj8JBgEqQJBgwsTz1wxwsbJ\nLM6ceb73vO/3nT3L7sg15XeO9vYc6WxH+VMXVa5rfOuVIU/FZtDbMOTpvtBGlWwV\nyR3W2+D12kElU1sM45YoNrb4Ytpyk2nzqjZJwzhyFCjyIvZqbRmeFQdh1G2TchJm\nWh4VEhFqsBEKUQ1pwLTFQRBwKDhTTEhDBDIYOcmeOK3cdqpX1yyFxu3W+UeZeKTD\n8R11vr/+g0AFA04DRSkgzGAOGBAEUYkREBCADExtcjNSrOMoihPG6o532KS3z5aS\nOK7H1pUwgyCBPgIUceg8ybTHlVkCiK+sltkJDHJiqDBCIy10AJUvlQCWc0JsYAJo\nKEQGYUZ96rtXQGYA0pJqQYEwhHLgK2PdkktnllJOhAOkJpZQoaRUriAwWAPMfCNc\nBfZdiNTujGJnzrmjuJZ9J9VQ21vGpeMwyjr3f4krg+Vsc5dVpXTyH4WSCiPjZnbr\nUKFDb+8GZqDNswPlMLGlMCMoZwK4p80rJ7aaDYJIyjgVUGKoqM8C7PpENWEYB1Zb\nAwRgAUFcSc18yxGQVgdGAoS4r3yl/stInE/Z7STTsDuSlf7EerVX8mvuzOWbcg2L\nCtmtzTUvXv7vVd5Wbsq9e7JhazT+wIGZqOdc4zNdy/5sPDDxzfaBLYdHPi8Uvrs4\nvef15hfkj61zuSvrTo7WVnVVv7w8NNCyce10n/71s2eX/3D/iYYnL7//06FhOLl/\nYWV/x0O/zc9utBfODy+91vpS68d7fz/2xFV4+vRrl8pjbMnKa4eO8qOjm1bPXT0y\nvmK2c8Ru3l+899hSVug4c8+3xS/u27Tm/NjM3ZfOPpVfe2JbemPFxKmpwvW3v1qY\n2vfgDTC1+K/Hdg3Mr9+3W8y9sb755erh7Wc/fPiD4teLDrafWfLR7PePH9zCnnt1\n83vz7+xddVfjW+uuT/5SrM0Nf7L7XLWl1E7109Njx9+80PfpBH7+55k/VneOHPcW\n/gY=\n=2nAo\n-----END PGP MESSAGE-----\n\n` + "`" + `` + "`" + `` + "`" + `\n\nAnd finally, I am proving ownership of the coinbase account by posting this on coinbase.com\n\n### My publicly-auditable identity:\n\nhttps://keybase.io/oconnor663\n\n` + "`" + `` + "`" + `` + "`" + `\n",
            "check_data_json": "{\"name\":\"coinbase\",\"username\":\"oconnor663\"}",
            "remote_id": "oconnor663",
            "api_url": "https://coinbase.com/oconnor663/public-key",
            "human_url": "https://coinbase.com/oconnor663/public-key",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 192,
            "hard_fail_count": 0,
            "last_check": 1434481477,
            "last_success": 1434481477,
            "version": 3,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 5,
            "payload_hash": "d64c8d89811ef70c83ab58cf6350d4e847f1750cff3c468d7e8556c65073e09a",
            "sig_id": "8b4bc6a736a2a651763300384d54e72daa0ee3f9464ae98fd2df4f06f93d59f60f",
            "sig_id_short": "i0vGpzaiplF2MwA4TVTnLaoO4_lGSumP0t9P",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9km2IVFUYx2e23ZwEJRctBYPda2DlOJxz7nm5o/lBd3uxkMi2/ZDYdF5nb7ve\nO957d7dRBmwNRC3ayFIye1nKEsoUMTIUjcoWAwk0tk39YJCUFpZIBqtu5w4b6ZfO\np/Pye/7n+T/PMzjllkwuu3pDd9fHD89H2W9/FZmOrStqax0RqqqzYK0je3wdJOku\n4Ku0s8Dp1lXBY13ww5YgVLrwbNwyweSdPh3FfhhYChRYwXVq+RRPg40flHVUifxU\nyyEeR5goV3kCEQmJoVK7xhgGPUYF9bjCHlIuspJdYZzc9KtT1yz5yt7W+SXUW9xu\n+fY631t/wFBAw4gRhABMlcsABR5GhLsIeBCAFIx1NGEplGEQhBGl9YxX6ai7R5ei\nMKzblomfQhDDIgIEQ2Jz4nGXDdPY0xoY7hFBsGcwABxZI0QVFRESY44MAspwyj1F\nDVSEYcklUYBpVwpPmSIVwGhoFJWQy6Isaq05QBwSz37kcokos89WESlsMFZS2HjK\nCGMQedyaiPXqILTJ2ewIqaXnqM+X+oZ2dXHZbY3q/vj/PCfVSnrZr8Xu0oRGSfiB\nsl27sa3QojfXgwCYd/RzFT/SJT8lCKMesCvvVCLdl04CQKwoAANYAOuSQgaF1gZy\nYrtnK6MJUsQAowAX1FWyiJlKywWhEJS55j+XtvIJL1vJ2C8HPOmNtFPbmJ3dmMnm\nMrc2NaRzm5l82+3/DnPSlstsPvyGOXpg194r5dnN2zrfGg7K58Ym/Z1cCzq+mffj\nH51fPrX7euflphkHK18cGbq/9Obcx7KtT4+NrF/y1V1t4wfId4s+jB4d/awpM6V5\nXUNruP3M4uL++Pqex38aOPLzlgv4oca2X4ZWztq7+fjyma8fL7RefHfr9E/m9b49\nkKBTi1ZmT48NLl1zbOelXYeH57y4b+bUu0/uG3llzed//p7cM+3c90eP+cuevPf9\n4Q9O/fbIlcYTGShySX9nw9yhQs+lOwfveH75xa9HWs4/8d4zHevzB18YLu+c9FHu\ntWrrnJ6XPt30wH3b/2qunR24Ot5iTrw6Om1F+zuj29pPL1s469CDS3dUR9tGftgS\nnnx5HHf8Aw==\n=B0oP\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419205415,\"hash\":\"e48ee0fa85b548f400a26ce5d9d5bc44a2f20dfa6a8d6f1d574cac5d07e3cb8df96b0fe1fd6c1ac9c9eeea02a1585413ac267e1fe5d2d4f44dcbac567577128a\",\"seqno\":114155},\"service\":{\"name\":\"hackernews\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205501,\"expire_in\":157680000,\"prev\":\"00279b0704b01ac6171beef1a5718a8de52d5f0fd0ab63dc947dd6f111bb673f\",\"seqno\":5,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1419205501,
            "etime": 1576885501,
            "rtime": null,
            "sig_status": 0,
            "prev": "00279b0704b01ac6171beef1a5718a8de52d5f0fd0ab63dc947dd6f111bb673f",
            "proof_id": "ff25378bd64cc2b0d2942f10",
            "proof_type": 6,
            "proof_text_check": "i0vGpzaiplF2MwA4TVTnLaoO4_lGSumP0t9PBvk9WfY",
            "proof_text_full": "[ my public key: https://keybase.io/oconnor663; my proof: https://keybase.io/oconnor663/sigs/i0vGpzaiplF2MwA4TVTnLaoO4_lGSumP0t9PBvk9WfY ]\n",
            "check_data_json": "{\"name\":\"hackernews\",\"username\":\"oconnor663\"}",
            "remote_id": "oconnor663",
            "api_url": "https://hacker-news.firebaseio.com/v0/user/oconnor663/about.json",
            "human_url": "https://news.ycombinator.com/user?id=oconnor663",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 210,
            "hard_fail_count": 0,
            "last_check": 1434471730,
            "last_success": 1434471730,
            "version": 8,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 6,
            "payload_hash": "37ad41cf064989dfd849a0af178aa066dd396b3452433a9883a69e88e35e75b8",
            "sig_id": "5941324c9d6c5c830c832f44494cbc73af0cfaff6e3c9e69ae88bbda45e72c3e0f",
            "sig_id_short": "WUEyTJ1sXIMMgy9ESUy8c68M-v9uPJ5proi7",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v1.1.6\nComment: https://keybase.io/crypto\n\nyMHMAnicO8LLzMDFWNienbHWQ9eI8fSBXaUMIdNj86qVkvJTKpWsqpWyU8FUWmZe\nempRQVFmXomSlZKpRaKRiWmKcYpFkpFpsqFpmllyqnFaWpq5oYW5WZKZRWKKiYVR\nirGRko5SRn4xSAfQmKTE4lS9zHygGJATn5kCFMWivhQsYWKYZJhmbpqWZGpqYGKW\nYmxuYGZgYWJkmmhsZGBhaGAAUlicWpSXmJsKVJ2fnJ+Xl19kZmasVKujVFJZABIs\nT02KByopy0xOjU/KzEsBegCoqyy1qDgzP0/JyhCoMrkkE6Tf0MTQ0sjA1NLSQkcp\ntaIgsyg1PhOkwtTczMIACHSUCopSy4BGppiZJFukWFhaGBqmppkbJFsYJyaZWiSn\nmRmbGqSYpFqYmKcZmpsaJKelGSebmFmkmKdamJqaJZuZGpgbpxpYJgLtL04tzMtX\nsjIDOjMxHWhkcWZ6XmJJaVGqUu2hAzEsDIxcDGysTKAoYODiFIBFjLMKB8OJ2qe/\nZr3gLy6SvvjN507UXm6uyy+2Jr2S2Jmconh4rcmchIcyYbsTGpuXb+XbvvHYnyK+\n6Qr1z9Vku4vzZgazFKw7pivUk3NdcXWEterdmca6ynKPhS78UWX5l2veJ7FycfPu\nSJfvTz3uTQzgft/3eM+CH29FHyfOy57Z1hf5/c1qt8XH93Umxs0/tca7+0n5oj/t\ni953tFf9d/bvu3HXTeGx5vIL6zdL+3qzfSu4nx0p9tZG2MhAte2tfMaFD6sZOP+r\nRzwPDotbUiO+3PQGSzGf99aK4Iebbk4XfF4WK3TycN/VWaX+W2LSCgzXbXlwiOu0\nFM+WnNRVXZHdM2cHnpxq+eXelcO9MplbLTUA7nMASQ==\n=dQNg\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205998,\"expire_in\":157680000,\"prev\":\"d64c8d89811ef70c83ab58cf6350d4e847f1750cff3c468d7e8556c65073e09a\",\"seqno\":6,\"tag\":\"signature\"}",
            "sig_type": 1,
            "ctime": 1419205998,
            "etime": 1576885998,
            "rtime": null,
            "sig_status": 0,
            "prev": "d64c8d89811ef70c83ab58cf6350d4e847f1750cff3c468d7e8556c65073e09a",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 7,
            "payload_hash": "2d8dfe977e6d359a03430c5cf22993d16438b1e10546b1b247055abc9cad3edf",
            "sig_id": "bd1445bcc75e2d6ed5031c245306863fbaddf8673e674903fe5ac7c1414136a90f",
            "sig_id_short": "vRRFvMdeLW7VAxwkUwaGP7rd-Gc-Z0kD_lrH",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowFdkl1oHFUUxzdNTFI1WKEIQcU4FTF1E++dO/djAlYiGwzBwLakpU0iy/1MJh8z\n25lNNMZVMMYiRoiGFlpt6IfUNqUP2qJQqAj1QVofLKgPpU0fpAShopJIo1XqnSWK\ndV5mOPPjd//3nDPbUJ2pr9q9Z3jwVGeLW3XxR5Hp2b94e9IRkZpw2iYdORLosJR+\nhXxUO23OsJ4QPNGtQdQURkq3DiVNa0zWGddxEkShpUArbUVOOevIeKJYiuRYHOtQ\nVoxcqVgniYVgV0deIt0ebMW93dtCd+jFwZ5Sx3bk+uM+kjkPdVhnaaKYHiuCkoyC\nMFXaBKnHBOGAjotxkMZzMOOuhxVSTLhYQmyI1MgYQyGjRBDGlcdchVxrHIyS0h0X\ncSrOQqBstcI/S1h7zvK5Cj9W+eFBAQ3FRmAMPKIQBQQwz8UcuYBBAFIw0fFalyIZ\nhWEUE1JpwqiOh0d0IY6iSidlKUgh6EHfauzLZuLJYNo2ZrRA1FMeMEYyV1BglJBA\nu8rFyBYo1QBITJXxPGIMI4hSSY3CwIeSA0WQwD4hRkHGMZGA23zWRSQiHqXGYE8Y\nbL22IJFluCKaad9XkGBNtZP9osZJ9O4wsvEgxswv/zuB/03yP9OG6ZzvuJOL/Kyj\nXyoGsS4EKYEpYcA+WacY63FrQ9ROBEoDiOczXxnFPJ8DbiC1oQAhSiGfCORh10OI\n+4whTnzNmEY2JxbMnr+Wk9qEfMAqk2Ag5KWxWDvlt6oeqslU1Wdq71qXrnPm7vUb\n/tnxh6vrbtWWy6XtP8MDr7xTx/uvwtX7Z3a0Ty18dG3nwqGVB/sy1+tOrLuc25J9\n/M39BdEc7+t9ufYFlv9w58hfPcstS/7Sei7nZ16DF9798/zlngfsNu+6ZzrbfOP0\n9PkF58xXb8//seWDba9ePF68WX6jb/rCvb8ttXS/3/jknrlre59o/P7o4aN9lz7d\nHG98rHDwyqaVTb9c/Xbfro+f75KP/PB07uu5hmcefS65cXOme1asXL/vm8bFs8dX\nDx+Yqu89NtwwuVw32bn597nmM+VCvvG96qm9Q8nW0defaspnv5v7ZMftYLHfPXf6\ny18b8h1G9M/PfvbTQVF/5MqxkVvLhc7PazZeOrnQdgisgq6/AQ==\n=5wxk\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"cryptocurrency\":{\"address\":\"1JEPc3eAiQ5ZMRn2jwhTtEU329v93cD43E\",\"type\":\"bitcoin\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419370141,\"hash\":\"08feb374d40ffc82b70fdbc0e2d253c8277e00c57df446ff86377c7fd5091ca0d63b5966fd18a56c0a5a340f6c36477ff54bf582b6c3c318aad6e8e99d165e7e\",\"seqno\":115589},\"type\":\"cryptocurrency\",\"version\":1},\"ctime\":1419370239,\"expire_in\":157680000,\"prev\":\"37ad41cf064989dfd849a0af178aa066dd396b3452433a9883a69e88e35e75b8\",\"seqno\":7,\"tag\":\"signature\"}",
            "sig_type": 6,
            "ctime": 1419370239,
            "etime": 1577050239,
            "rtime": null,
            "sig_status": 0,
            "prev": "37ad41cf064989dfd849a0af178aa066dd396b3452433a9883a69e88e35e75b8",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 8,
            "payload_hash": "57c95524e2fdb546201af092292ae9d27a055a8934d81a8f8488eec777441d1e",
            "sig_id": "f6689d2fa3ff3b2fc130fcadb73e7de28ca55b3c59b493de45f9c61d980b5d220f",
            "sig_id_short": "9midL6P_Oy_BMPyttz594oylWzxZtJPeRfnG",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMPVAnicdVRriFVVGJ2Z7OGIYpJQoBKHULAx9/uhRFmBIUExzWiadvn2a+5xZu4Z\n7znXZpCh6EcpRknmg8wyEhmFgdQ/EQSTjJhFREHSixBMS4gIS1Aiap87kzpQ59e+\nZ5+71vrW961vfOZNLe2tm17qrY4+uoS0fjre32jp2n9wfEtiMjeULNuS2L7U14ry\nVIN+nyxLev2Qgdzfl2Z3P+tN0pFs9vU8zWrxylmNLGfeSaqUos5ggICYFByhIL30\nhlPALiTDHSVMCRrSWo+vD9TTkiPhCgjjjjplCLeYB2E9DSFIrKQwQoFjijhKIms1\ny4spapImZiV18e1/fN/bvEAYYeoRp4IIcIQKJbySJIDkRjDgRiGKCdcSGW5BgjTS\naWWpjj8UidcYCIII12jCMWxwkDwYzhETjkokkGKEAyVIYYTKD3Nfn3Qus1mtltWF\noKUBRR1sb2lB1J/avDylrnLNTNqR9EFexHoqtgrRpGQZZlgzoSPXFNiNvVme9w3V\nStCmquiiwEgSpCEo7oTVlhBttGIaO9ZUNel+adjUDoCLD3UoFi6Mk0R6BSGaxYAA\nJyYIDt5H46Y4qoXQ2DOJqXDeaBY4ozggIYLmIBQTgRJpo4GeMBdw7CoHA9hC/A/X\nzkdHo/a6788KXxmoZ1mIdjwdZ69I+6/XzTjvSGyjXo+sWHNPhMLEK6SARS7GvA4o\ncjMfOEGSUmcDcCuIZAp5Rpx3XALCJZaP8v0kOJcSswnwSQWlK00VzfmvettbcVBA\nZWNeNuZaEHrSotowyf/2oglRKYYGfLObeQFFyRdvcr+plsVjPKU9ExOLqaRWO2kt\nt8wBozErgLShAjtHLA9Ee8kRRVh5jJ3FKGCMGFIxUIYYikIygTbBR4Y3NGkqBaR9\npeYBGOrLwFWqkFcjnRAids0aqaWmJlCOZcyblwikhtJuMPFMtI4Z4LGHcSyAQqBg\nsHbO8ORaEeSGIhh45wXyQkpJNMVGeqGIDlRT6wL3DAclwAYvnQXr4pCCxiw48CTm\nisaakuEyGc0SJgNyw34prbthJASWcbXERg4OpHVfSWvNZgqF4lO67zdHEOKUC17L\nqMRRrgFRRuOOsiEmQlOHBaPKYI8RZ8JEJ5lEsV4TIwOOeheuF6qiMOiJkLHcGhSN\nuk+GT4ytn9bS2t5yy81t5dZsaZ8++99dutvd+vfqHSu2LX5415d3kjNzVp6ffezq\nocZrzyx4e+v+eR+8PPPcqSOd/o87BjeN/3Xpu8PvVc3H73e+sKHt3PjJ4qnuGc9n\ne3etubTnwY/2zOrad1z2v37xnauPLThxYIa7evL+Kztnbb/3q4Xzz46ceujyyZ2X\nD7W17lytty1ad3r7+cE3vnl8JF/Z/cSZhXNH0wvTalnl21+6nrxr2x74fu46Li7f\nvqrozBf/9EX3xcaVH7d2fP57/4YDoQuvfW70zZGtvz1ycN7RkWM7zp7+dfT4J0fC\n0uVn1/9wbN6fe98dGT+84nx7embtqle+vqAPHl236J4H5nfMectNH7tt6e4FY2PJ\n4L41n21esvziq9K92KY6Pyx+/ges9xbZ\n=2qFk\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"dc90c54ed738883db1aaf0476500f7e7eb53a1df\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":3,\"last_id_change\":1419469550,\"username\":\"jkosslyn\"},\"id\":\"58a6107209af85d6c9c229b98491d400\",\"key\":{\"key_fingerprint\":\"adddd3d07b76bd727e8af6264a2a52bf65aeed23\",\"kid\":\"010196691e47136deb94f5431f066f95a6846f327c782e24df1f6c5aba1cae4759de0a\"},\"remote_proofs\":[{\"ctime\":1419469455,\"curr\":\"195e26812e808a4f0644e9f094f4ef520733dcfa5c627480e42ded57a011946e\",\"etime\":1577149455,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"jkosslyn\"},\"proof_type\":3,\"state\":1},\"seqno\":1,\"sig_id\":\"1373c9d7cc5c4da43eb5a09b361dd2c5f29e7503018e11dc10f1104083a1b2b30f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"666196cb79793bf35175c1e70a79a4194ab70a299a7a55a6adda3af3ab19ddb5\",\"seqno\":2,\"sig_id\":\"4aede60e67772931b7e6829f393cdf5e41f86acfe7dcacd610a914fdae21a2310f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1419617476,\"expire_in\":157680000,\"prev\":\"2d8dfe977e6d359a03430c5cf22993d16438b1e10546b1b247055abc9cad3edf\",\"seqno\":8,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1419617476,
            "etime": 1577297476,
            "rtime": null,
            "sig_status": 0,
            "prev": "2d8dfe977e6d359a03430c5cf22993d16438b1e10546b1b247055abc9cad3edf",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 9,
            "payload_hash": "d70f647a155416bd94ad669254e5513aa667c03ed55f047f81cd6fc9bc3ed89c",
            "sig_id": "7ec6ea814573f2d7db7947262a624e4b0990ae5cc48feff1c8572b69f60b769e0f",
            "sig_id_short": "fsbqgUVz8tfbeUcmKmJOSwmQrlzEj-_xyFcr",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMSiAnicrVVbiFVVGD6T3RyNmUqSLqZtUYxGW/fLeRgDMSytqCQIJ07rOrOdyznu\ns0edxklfNMtECroR3R5KIiQJKy2K0hgljXpworCUsIcghHkoRLLs38dxmiEfeuht\nrb3W/v7//77v/9cXV00pNTeteaK7a9eyhaTp8PDS/tLK3Sv2DSa26geS8mDietLQ\nlxerPtMbknLSHQasqYdFaXXOumCTtmRtyOpptQ+OCCYOReWFwcypwG1kHgtBuVOe\nWaGc5I4F5ZKhtgKmAI1pX2fIallaxEi4MoRxT72yhDvMo3CBxhglVlIAgPFMEU8J\nRO2q1vNJ2SQNzErq4etF7nc3DhBGmAbEqSDCeEKFEkFJEo3kVjDDrUIUE64lstwZ\naaSVXitHNWwUgWNsCDIA19+AY9jiKHm0nCMmPJVIIMUIN5QghREqLtZDNsZc1VX7\n+qoZEFIQkGfGdRcUQP6pqxer1FfGycSyLekx9RwKqrguAyzBN4Y101hRPAm3nmep\nLSAbOYVIAtNa4OC9MVIizSzjwRcyRNfIaYz7gq7J/MsYHFMMu4C9IMxJLEBEawwV\nmgmBuLMkeGcm8cmVlVJoHaWlhmjuGQtWaWohoCc8aiCYwf+KBsGIDJgbuAFkI0KU\nlhr4hNyz0FvNQ6WWVasRyFgFzsvT3kbRSHHBIZW2xPVnGUTVInoZoieE+Yg8kpTJ\naDhxyjjLDbJBw4ZFLpiGpRXUe4WZgoysB1dB+mEMnAuo8zx4LQtrAdxiHyiVETGG\nPC7K9TGGwIKWQAgEYIobISkKihZyO8miYlQSbzTHTCbjtRT8Nupp9FFXcN0Vb3JT\nWV0vBB5sWPiCNXq8q2Zhkav2JkUm1RzM0gMHXXleKxf8NIAq+UCtyBoh1Aaym7zY\nwGE9rOmrJmUOq7TzQgsIo0B7QaGnQHzDDZHYYWMDEgoJr0kAj2ojsNZgcWMl904G\nY72PoB8qWCrQzockQ22TFBFMCo7HFVFG6+A5U0Y55r2nnHAFvSuUpNQJb6F7qCch\nYm09olw6473TRmkvYmx01ARFKDkPPqbI/yD3f1RkTI3ONO/qt8nFm2yiEPRiKogJ\nKuDIHMwT6zTCzikwkpKehUAU9z5gQqR2gkQsEMxL0EVZ6D4rJOcSBgoT/1LhkUaY\nSm7SniLhmhnoqRpf6TL1rkIGbkA7G3GA/mPRIGUdjgp7bS3DVGHFsVYIBYcxtLfg\nIRiNYGRHwqTyxZwcKwKjCVUI6GwfWSGiRRRkhSFKvBIgeQzSUOMcgYkZJXS0DdYE\nypSMMDItPAXWFV4aKiZeo4axwTfh3Si4G7cWDE6GMAH1w/pamoVKWtzgEjzbcP2Y\nJ8BBmnPCAoneciYIwiYiTYgmJmhPpEGcm4IFsIJR0KFKheCklIxhj8M/lWpIzHQW\nCqedfSbvz0IytP+zjktLTc2lyy+7pHgNS81TWy+8kdPeveLcHV89fe9o86nHsk8O\nr/781NeHdz3XvX354jmbPt6zYNuts7a+cfWh0y2lqvtwSm3JsH372S1Tl3fcvm7D\nxv3zjy75+eDJcph/6pmZ59666Zf81yU/tR1o//73v+4cXf5N6+md82Zsapr96Ob3\nbmg5eeTmE9c//uLw8dbnj7llO6d/ef+ylU/evbXlxtKxo/v63xxomdYxPb3unhcO\nnX3/24fnvnZ88we37B356OVzZ+c9sG36gbL59GD3suGlZ3pqw90L4ow/XmnvGN14\n4rYNO3e80zMyJz515Mot5dWjuzpeXbG79aXy3M6WWav2nhzh4s8HL9+x59Dxa9fe\n1b69fYDNfH3qD/sPDK2/r7L4msEzv/3YMfuh0ncLR/4GukqcnQ==\n=TY+X\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":17,\"last_id_change\":1419491831,\"username\":\"strib\"},\"id\":\"ef2e49961eddaa77094b45ed635cfc00\",\"key\":{\"key_fingerprint\":\"7fec4841ce1d624c7165bfbaa36946605cb2edca\",\"kid\":\"010158b77699f7b3a295d44eb893b35cd25f9872469483e6427e15a44e4a502289790a\"},\"remote_proofs\":[{\"ctime\":1408565716,\"curr\":\"96fd7efd224df0d07347fa52c8acb5a0be952c4f5649e95b63dd8148b35bd15f\",\"etime\":1566245716,\"prev\":\"b1de337f0440d12edcdffee4e974c7acb485a6730e830608c74f84372da95147\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"oldcore.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":5,\"sig_id\":\"716a8094638a2d63a5a271c1abe06806d92e0819a619998cab75dc7eabddfeb80f\",\"sig_type\":2},{\"ctime\":1408647651,\"curr\":\"8a99ed548a8c4ddd35258ce368733c6db7823d2ef19bd0357caddc9a89d6ff0a\",\"etime\":1566327651,\"prev\":\"96fd7efd224df0d07347fa52c8acb5a0be952c4f5649e95b63dd8148b35bd15f\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"strib\"},\"proof_type\":3,\"state\":1},\"seqno\":6,\"sig_id\":\"1f4c125bc901cc8ffe87d4ee285dde12279c62f160c8e0688b841b67557046460f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"85afebbf1e8934fa08bc1f81d9bb413818519800ec1124c65eea900f8f2478d2\",\"seqno\":10,\"sig_id\":\"67b3df4b782b03ce30132d869edfe7a3acc270bf7022bebae3487fca7bd6abc80f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1420840121,\"expire_in\":157680000,\"prev\":\"57c95524e2fdb546201af092292ae9d27a055a8934d81a8f8488eec777441d1e\",\"seqno\":9,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1420840121,
            "etime": 1578520121,
            "rtime": null,
            "sig_status": 0,
            "prev": "57c95524e2fdb546201af092292ae9d27a055a8934d81a8f8488eec777441d1e",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 10,
            "payload_hash": "251d1c8b830c3c4b131a182d27188712bf85a7026755bd0ad16ec2f58c880a07",
            "sig_id": "2c449b4270eeb6094bd48557866343e89dde1983a1694a5625b8076ed793fbc30f",
            "sig_id_short": "LESbQnDutglL1IVXhmND6J3eGYOhaUpWJbgH",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMjuAnicrVhtiKVlGR4t3XZDCn/0ww2Sk5LSuj3fH1KkYrJbWqsUVJrT/Tz3/cyc\nZnbOeM4ZdV2WEH9IlKFIkPujKIgoirZf2QckVrqRkdkPMe1DhKIIKbRoK7LrPTPO\nnlmH3Z3cHzOc877vuZ/nua/7vq7rfn983mtmdpx1y90L89/cc5k562dPfG9l5oNH\nrps52CsDPtC7/GCvLvZladx9WqL90ru8tyAHCo1kd39w4W1Sert6t8pw1B8s4ZbR\npqqWOJB2NYkvzbEOwfqa2JWQavTVSaq9Q7u6MF3Q1l+ak+HysN+t0fOJjPNsORXj\nq/YtVLGttahTDAhA7JJha7Dq/GA03rCb3iTmbJ9xdZPnFyY3lFbaivI2mEBsbEhB\nUjSNoi/BkS9JWW18jqr4SpFiiZxTtRlfksFtTUYRwq1MwjlddIu+Fe+VC2yjCio5\n48kalbRS3YMjGa5lblAHS0uDIRLSJWA8pLrQpQD779dR96nPs+vJ1Mru6i3SaIwT\nzdZ5Qppw0emcTUrGbAi8n27vIk62xKXo4EtMMTeJRevouEVbhFXJfrKltdR32dqY\n/iStiBHH7EOx3kQbLPBqQXlTDLHVhM3rekI6W1bFAdgA/DVrQx7/kw+5+JhzUMrU\nVLjWmouJTmXUQxZHKrLBD5FO7H0o+wdjmV0eDgYNubgRhTfud4fTNutoVE5uV6+u\nDIdY1XpgQdYHLfi1l2h0xMkKh1g8EVP1HFto2WfHlT1rX1kFZIQpiM7YvqwF9y47\ntRp8eSi3To7kQpFYbQiZfPNMKUvOKkkqxTfduORWGwc2HAKOGoJrWkVPMVjku7d+\nli6/k/NM2mhe6sIs05hmPzHq8F3vp7n+eH6l9E4EdEx10CVmEmF2fGAZ11ERozGN\nu43jzkhuWRqsXuzPrdZ9BtIaxVwn5Sy6Zo7KIjXOJqtzoNTw0ZjsyFmV0KqRbMw2\nBlJFZ6VabzXa6nrm0K5pHIzxxqa0jkMNWhUqumtZ5amGRtkLk2OlQALIYbSmsHMa\n4PsaXXOpVaxayLlq9UYcsl0NvobDGQB5aziMb+uPxzI8LSDMZkC4KSBCABWYWiOS\n37Q22Sj0Y6yOSQmDGq14U8klh6LyJCFbiayIgu7woZMD4VOIWh0HInI0lBuIily0\nWWwGYaVsbWbbQsyo06iDYsrc0KSAqgQy2A1pIextGghvvVsNvgaEYTSBJBfJg+JC\nAJ8yk1eGyCRrTWouRlRTadW4KqqGpCuB7B07kKU6fSA6Sn+FyPQmmR+DOxdxfX48\nXh5dfiIcYFq1GSJmGhLrdWRmTX7SFakZDXYDXWG/LuDMJlNwKgXKITLhhDpKl7uS\ncs3JnBKSBKJbh0SAJHgLJQ0qBC+yt8DA6sBVZUmCTjASvHNZmWAgQUonBhF5R8UE\nUSdCshp8DZIzgPfWIRksyWieFheXaTTaXQf7XxUwfpq0QBOucNJsi1dsvFOao/Oa\ntEUnR/Q9ZEQxAATf+ARxM5lLRXHpDOI5OTBRoZpdPg6MgIGQJyi+A7zWKmg46IoL\nkswV5F+kEZFUjYyqADXNBI2LuqEvC28AxkW/GnwNmBJ0SM2SyymnAihRbHAHGlRg\nCiiZW4IARuw9Bgd9xn0buiAdAVRtTx8YHuyn/tIpYeGl0SaY6E11ZLpZlPgY4eJA\nuclDqYm1gvRDKxmqwqB2KL2xLtdmHGhBwHFimmlADQalnBoTWK91TOAspDFcQmsQ\ngZZDgrGCcyvV1ibZ6OxLhUsTqxMYBj6mthAEXwMo0/tXYDIJvobJGQB8q5hANRZ2\nD4Zz/z8W0/1RiCpn6JqRIhbtAUuLJkYrR3Az/krDhjUJ1CM3U7QrWioSB7+GrMnJ\nsECr5RiDtutYuBA7UqGGNmOs1bSgERm+0+bUIjCgLOSUOMi/ZwN9DsXBOocC3nEb\ntSQHvxp8DQuntebsfCcYTBZV5RPoz5nSyXnpHJaUFiPclURNDm7bSm3wlpgfatuC\nqK9jsbQyboty+6tqja7D1+Fw8LrwNskWeBY4EfHQOc8sWYcG1xyD8V1DR7G4wzCH\nBf5RjFUZahBO6rHADfCTuiPNl1sDzqnBb+qIrjMgkeRcgeGCuzMtOAcAvMCGKxdL\nY0wgKetc4NkxidhCdhqOALuxGvxluoJ6V8Y5lJNSa2cQ0dEhWu+bFTSezYXxDSu7\nIFmh2VWFKFqFf+iULXusoTD3x6dlsdxmQPjpvlDZc4ooIQshxebgx6uyJno4lOqs\nA3FppEiCLjmzBwmk3ALDtWCWSfakgj4BAqV7HIjEGScGoKqUzvnHhqlWZ81B8Fxo\nGhwIFhfMWBgSoO1COk0EH1aYN+pGB8Rq8DUgzgDKWwOiDvpLncE6LSj8plCEKShY\ngYIIc7pllE9zqpUgpYTGMIcW4sDiLHwwjC56pZXWCvwM+FZjSMnUTgVFCtkfpyj4\nN1cJv8MQ4GC7so/FVLGRbTJJkcb4zYYiNwkqEEYHC1PXeTgxmIzkBCjCavA1KM4A\nzluDYp7qAhCQ20anBUbYFIw4BQacYUmquW7mh5JOuDp1u7YookgV5/AEGcw4VsNQ\nkkEBBmzFnJOCgr8CjI9N1pkdU3+x2/cyHVgcEM/O02i+a8NuorcN6IrBXIfClqZg\niNFt8AeE6Z0g0ihrKw50j0FNpYjl0aLBV9uhsXaKNF1SIJzqKhgKVh2zNwQbwx2V\nBCwbfC6gplA54XAYZ9m3BA4OHtwM8SCPp1vvUPeWZXKGtZctU++quuStV5hRqRNC\nGGy5fbk/lNlOOTSYHK57ii4xNaAZYYtgMjRqBFM0CiRDkQWMbyG/0AVlhUGf6NSG\nOYhDqxk2BjNfrsdP2rX+mOYQE+ddovHKUHqHHn7optfOnLVj5txzzu5ewc3s2P7G\nl1/M3XDn62buPvKiabe8bfmlL/7iq9v2Pfv1P11/4KJ/X7Iib7p253uOPb/nD4v3\nfPf5Z9zTb/3ywZ2/++gVv1Tb7rv5kweeeu6SL13xgzK396/lgWeefuD6s991/hM/\nevHzb7n/p4NLf/P25aN37dy5/dEr//zYP57Ycc7vL/n1kQ/v/vY1/Qs+8ORVe/4z\n+MsjN1936+K9L1xw8codl/4kvu/9j/38bx9595OHn92294cXvf5X/3rDkb/D/V61\n99j57zy27Su0/XPfOXbujfte2POhb8xtO/q1e66+/953PLzvs/c+uve92w/vuvkz\nd7X40sOLjx++8ppHzr/vC1ffed0fd765f8fR73/8LqNuunrblempxQs/ZT999L/P\n/fOhx5fP2ffg1w6ed+015rc3fOvBp++87PDF/wN3bEWj\n=S3O8\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":103,\"last_id_change\":1419928822,\"username\":\"max\"},\"id\":\"dbb165b7879fe7b1174df73bed0b9500\",\"key\":{\"key_fingerprint\":\"8efbe2e4dd56b35273634e8f6052b2ad31a6631c\",\"kid\":\"01013ef90b4c4e62121d12a51d18569b57996002c8bdccc9b2740935c9e4a07d20b40a\"},\"remote_proofs\":[{\"ctime\":1391720984,\"curr\":\"35782a3561e40a5e7217500bd67b5aadac5d7f6f9594dcd5d15cd06879da6e19\",\"etime\":1549400984,\"prev\":\"0146be7c3669a5f5da89e9908e8bb5f1fdb9fcfd6d2d66a51664f1075a7637b1\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"maxtaco\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"9f731fa7c75b64e1c9d70300b4383196a8fb432294a4308f8d7a379376a0b1900f\",\"sig_type\":2},{\"ctime\":1392252388,\"curr\":\"c610bab1245d05ac6fa95eda4d008d67c3732bd441dcc5c74f48fc8f8ba44c31\",\"etime\":1549932388,\"prev\":\"35782a3561e40a5e7217500bd67b5aadac5d7f6f9594dcd5d15cd06879da6e19\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"maxtaco\"},\"proof_type\":2,\"state\":1},\"seqno\":4,\"sig_id\":\"667062cc7300f11292079f7c4da0ed8e53e52ca484bb55ae693e7d0aa614322a0f\",\"sig_type\":2},{\"ctime\":1395867108,\"curr\":\"7d72a9f98ca4739e3939c89339d3f67966a7160da9df6215acb6a27c4a1ea292\",\"etime\":1553547108,\"prev\":\"2df5de847a546d662d3dda502aa283328f477430bfc24ce0c681ca68c4d4a200\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"keybase.io\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":24,\"sig_id\":\"3517ddd1a5c9d708f21634d126814647129a64086a967daa5017e7966b89c9820f\",\"sig_type\":2},{\"ctime\":1395868002,\"curr\":\"ea61172238d18212d53a71316dc09e8e48f2e654490262872018d09854ab26e0\",\"etime\":1553548002,\"prev\":\"7d72a9f98ca4739e3939c89339d3f67966a7160da9df6215acb6a27c4a1ea292\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"oneshallpass.com\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":25,\"sig_id\":\"93734bd81d3b50d25401d7451a136e171757990d35110b585b729dbca2019d050f\",\"sig_type\":2},{\"ctime\":1397071049,\"curr\":\"eec8f9d3a754e793301b108ddb548dc46bbefaaaec189306b169a3ef71f22abd\",\"etime\":1554751049,\"prev\":\"b6168f3a49898b4aba5c5fb162c2b5b6df87407bca7649feba5364751e693c13\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"oneshallpass.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":34,\"sig_id\":\"0e577a1475085a07ad10663400de1cd7c321d2349cf2446de112e2f2f51a928b0f\",\"sig_type\":2},{\"ctime\":1397071368,\"curr\":\"ad3efd2c8ff323f96841b482bc3cfe92195bc72fe318ce0b95cf66ee3160aa55\",\"etime\":1554751368,\"prev\":\"eec8f9d3a754e793301b108ddb548dc46bbefaaaec189306b169a3ef71f22abd\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"maxk.org\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":35,\"sig_id\":\"baacd979d2ebe325482d2921ea72df72dbf9a31aeaa69f2b14b1eccfed31c72e0f\",\"sig_type\":2},{\"ctime\":1401977613,\"curr\":\"467a473af5b7d548f1e1a1d199398f7fe3a9ea40e4c615d261e6b44a56b96642\",\"etime\":1559657613,\"prev\":\"4111d9457743da3ad15826242bac5db8bb5ebf77d6de71a41a23ecfc4ed4bcf9\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"nutflex.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":49,\"sig_id\":\"4b4c67c83b4995aae5d4a5dde916fb35762551e67e35d4d4f1be99e2309238600f\",\"sig_type\":2},{\"ctime\":1407166100,\"curr\":\"acc5ffcf17cd722b5844b24564e2f644d265e010047bfdd378919be8f0603ba3\",\"etime\":1564846100,\"prev\":\"b328cdb4904ebcc196a49c67355f3ee9239bd735d7246e90cf20c7e7307e706b\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"maxtaco\"},\"proof_type\":4,\"state\":1},\"seqno\":55,\"sig_id\":\"095d87d153e6590c2d6c032752d3c43485a1f64e61b99d58f989f6da28b2a8320f\",\"sig_type\":2},{\"ctime\":1407167610,\"curr\":\"8d906bd4d0bb2d667f5c8191d6e7616f1de1e69e950f1f2e6ea181316d8d6dbd\",\"etime\":1564847610,\"prev\":\"acc5ffcf17cd722b5844b24564e2f644d265e010047bfdd378919be8f0603ba3\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"maxtaco\"},\"proof_type\":5,\"state\":1},\"seqno\":56,\"sig_id\":\"d0f9aaff73dc67f40fb6ebb6fdf473e11de43f11ae64a5fbffb00218914e19af0f\",\"sig_type\":2},{\"ctime\":1407186953,\"curr\":\"c9d4ca19adac4680957b2ce37d38280a1a32d2a7dfe606ae7236a9c982e2c75e\",\"etime\":1564866953,\"prev\":\"8d906bd4d0bb2d667f5c8191d6e7616f1de1e69e950f1f2e6ea181316d8d6dbd\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"maxtaco\"},\"proof_type\":6,\"state\":1},\"seqno\":57,\"sig_id\":\"a9fb80f43bed684a40e48de1e365e7ace765a3309819f9f79cdb27e3dd9802f50f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"040933ff11e2a44476ef0395989349a917a0aad723e41585d1087330c4365c3e\",\"seqno\":86,\"sig_id\":\"46ec4cb327088e808d068ab8236f39dd38a6cd840e229d5f8a5d65b4c7d6a58d0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1420840192,\"expire_in\":157680000,\"prev\":\"d70f647a155416bd94ad669254e5513aa667c03ed55f047f81cd6fc9bc3ed89c\",\"seqno\":10,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1420840192,
            "etime": 1578520192,
            "rtime": null,
            "sig_status": 0,
            "prev": "d70f647a155416bd94ad669254e5513aa667c03ed55f047f81cd6fc9bc3ed89c",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 11,
            "payload_hash": "a6808c59df1a6406a961e81b7038eb132c9ab04eb827ee844663ef5a7e651e90",
            "sig_id": "cf90e4629d273b677662b511269fff5756714d2889278c4297aebe9d723ac8240f",
            "sig_id_short": "z5DkYp0nO2d2YrURJp__V1ZxTSiJJ4xCl66-",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n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\nxl07Gran/Y2t75XvfPM3px744zLn55bn/gsCjHb2\n=ofws\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":167,\"last_id_change\":1418935207,\"username\":\"chris\"},\"id\":\"23260c2ce19420f97b58d7d95b68ca00\",\"key\":{\"key_fingerprint\":\"94aa3a5bdbd40ea549cabaf9fbc07d6a97016cb3\",\"kid\":\"0101d49265e97c636dbffa62070f29e07bdee44aca90b0eaa1645cc29fe5320f8a6f0a\"},\"remote_proofs\":[{\"ctime\":1391738324,\"curr\":\"461262ff766cf5da5eb40e5db29128a453ffb939840a23c0ffcb95073c888302\",\"etime\":1549418324,\"prev\":\"34152df8ec04aa33ad1bdbfd4927ed4d64d1be2ed3c4060043f313041a3722cf\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"malgorithms\"},\"proof_type\":3,\"state\":1},\"seqno\":4,\"sig_id\":\"3e1e20a1a11f4f0ee4ea4470b694fc115f928ce0b1f9aa2951a65081da8340820f\",\"sig_type\":2},{\"ctime\":1392240892,\"curr\":\"4396ec2c10ba4c4e1d931af42ed4d65ecea9f57a858fd44e86f2950071ef871f\",\"etime\":1549920892,\"prev\":\"d2284c9ab38c28650ccf4d1feb7af9f4ee46ee760ca45711a5e3afe008e518c3\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"malgorithms\"},\"proof_type\":2,\"state\":1},\"seqno\":7,\"sig_id\":\"0d9f6b71c043f2ef80b6dea44338471edc3ef76e2cee236dd397412ad9631aa90f\",\"sig_type\":2},{\"ctime\":1395868048,\"curr\":\"45c6893a87e18f27502266c08f0d0118f5ecd3fd0bd52432b18cddbd3f6520a8\",\"etime\":1553548048,\"prev\":\"26662070e4cd5345a302af7d513bc8d08a9a21cae783c6ad18f0113b039211f5\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"keybase.io\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":82,\"sig_id\":\"58ff6251889edccbb54495ea3f7d33f57c2d18e114dda887aeab6e06552aed780f\",\"sig_type\":2},{\"ctime\":1407170371,\"curr\":\"9fe1cf15276a8c34ebe03eae24e90f4ea44e6c6afd69a96bd6a85bf1ec025780\",\"etime\":1564850371,\"prev\":\"fe90114a30fcf13d0b9af77cb4291c1d550462fa567b0ecffdf436840f0fd8da\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"bitcoyne\"},\"proof_type\":5,\"state\":1},\"seqno\":202,\"sig_id\":\"c6d2c827fd7292ae36a359fef3ad04736ff373901b3db037af773ce13db7fab90f\",\"sig_type\":2},{\"ctime\":1407767238,\"curr\":\"510fb8ebf6ea77aa220f0a4cd7d0be212f9b294161f0cb2b43b6be024aa9ec67\",\"etime\":1565447238,\"prev\":\"044e1a2cc9d956c4780d4eb7fcaae9dfdf560eb9d5095ad653d8b37f63172c7f\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"malgorithms\"},\"proof_type\":6,\"state\":1},\"seqno\":209,\"sig_id\":\"c60cae4ab2b165fd15960433b0af48c66d37f3940e56a8d3b4955991947cb3f00f\",\"sig_type\":2},{\"ctime\":1408375965,\"curr\":\"57a78d3f45c749eab6a6b2796db103c6887ccbd7827a27e3f33cf3f0950cad8a\",\"etime\":1566055965,\"prev\":\"94e9a25ab3e551d43ff700e94be3cc446785ed4f44d83d93d00fb3ddc601070e\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"chriscoyne.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":221,\"sig_id\":\"dbf530c689d2f3af60c5b11341d5eb2ad229995d6ef39dfd1661962d088a75ec0f\",\"sig_type\":2},{\"ctime\":1408376008,\"curr\":\"ba697b9daa2617a80f53d94a8204a6db5aaf2215faf1feb4a43d2a09a95c6a7c\",\"etime\":1566056008,\"prev\":\"57a78d3f45c749eab6a6b2796db103c6887ccbd7827a27e3f33cf3f0950cad8a\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"chriscoyne.com\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":222,\"sig_id\":\"7dbc723fc7c0abfe761af6729fb27603ef9be58718153329ed2d4ddd8d6ddb910f\",\"sig_type\":2},{\"ctime\":1414608934,\"curr\":\"2f7e338d5d1915975611c4d99f225986f7978c2b0a89f7efed973cabc92b6e0b\",\"etime\":1572288934,\"prev\":\"b7036840be4000a1f5a006bc8b164c535300507c103651f389290e1ac42885e9\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"malgorithms\"},\"proof_type\":4,\"state\":1},\"seqno\":255,\"sig_id\":\"1a8fa75aa365dd96c95fd229722668dfb0026973d89e489b256d87ee06a4bd670f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"d24c186ca79e522e96c4d10f407646b9c1b5a2beb35fbaa03f2d070638af65b2\",\"seqno\":299,\"sig_id\":\"677463e4cd27e7cc4382f20aa7f21882f694feebdd4ddee623786b1cd21083940f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1421363548,\"expire_in\":157680000,\"prev\":\"251d1c8b830c3c4b131a182d27188712bf85a7026755bd0ad16ec2f58c880a07\",\"seqno\":11,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1421363548,
            "etime": 1579043548,
            "rtime": null,
            "sig_status": 0,
            "prev": "251d1c8b830c3c4b131a182d27188712bf85a7026755bd0ad16ec2f58c880a07",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 12,
            "payload_hash": "ed4c85c1ebc1765450203ad9f9c092dff4f1580367add60cf5db8eb894088521",
            "sig_id": "18c9a91230de00b8ce9f42845ffb24d2b690a2b2f9ee18efd66d0f14d000fb070f",
            "sig_id_short": "GMmpEjDeALjOn0KEX_sk0raQorL57hjv1m0P",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMQNAnicbVRriFVVGJ3xlQ5IjSVaIsYho3LM/X6MlKEgUf0wMxNKrvs59zjjPddz\n76ijTRGWk1GBhSU+wqSQyNESggzULCGsFJMIC0F60FCBgT8sMrPvXKfJoTh/9tnf\nZu211re+fXT8yKaW5pV9neX+e2eS5s+OHu5uWnTgwgPrEpv5nqR9XeK60lCpF6uK\nWRGS9qQz9FhTC3em2c2rg03aklUhr6VZBUoEE4ei8sJg5lTgNjKPhaDcKc+sUE5y\nx4JySW9bAVOAxrTSEfJqnhZ3JFwZwrinXlnCHeZRuEBjjBIrKQDAeKaIpwRuLWe1\n+jA2SQOzlHrY/Z/znY0CwgjTgDgVRBhPqFAiKEmikdwKZrhViGLCtUSWOyONtNJr\n5aiGH0WgjA1BBuC6G3AMWxwlj5ZzxISnEgmkGOGGEqQwQsXBWsgHnctcVqlkORhS\nGFDPjessLAD+qasVq9SXhszEqC3pMrU6CCq5sgGXYI8RzLggFA3H7apllVowlc4C\nt0FMIS85UdxEJjUKxunIvfaSYKUlbxAbbEDh2fAmeBXBO8KRjdpbKaJHDBTiaKP0\ngtiIGdXYxWGmCqSpsxhBo6kmRjAbHAFnDDOUIbA7WIqFoyoGb63knFEhFDFGMasU\nVWAqcM/DiqweStU8yyI48ijEr56uGFLONeNtievO80IilZZSEayWQXBZtFkor4kF\nVEcExdIaLb2jAQvLtSdQw4ITUhCwTgL9MAjOpcb0Cng1D6sAXAuFEGGRUaQ1Z9Bk\nHbSAD8QLo4ziBKxByBENumW0IFkyxRwcNkAgGdJS+NvQ0ximcnCdJW/qprS8VnR5\naKo60nq52yb/7evMorEzG51twJTqPVUo0rakVjf1gj1UamFlJUvaGazSjisjYDnw\n4AgyCzyhhzo4DgRjIB5YSqqI4IGSECNG3BARVVBBSMwhvTwYjIr+FmhX7iO9SxvX\nlOom7Sp4V01PV2Z8qWxq5WL0iXNKBQiIFox6KbT2RjtGYNq0ZhBHH7CVVEMKuVba\n2kgFtwyuciGawrBBEfwqEQDHkDUM8RiopjIYqYkLoEgbYrzmgB5AHtcGIg6zjrj1\nHsPL4axTBh6jpLcYtYaEwYm76sEqrLs6YJJSLiAWa6ppHkpppRGNIgkIDSXDwL9y\nkKeIIeVIGC1wUCANURUspsRpYxELEEMZgmIMxj1EboqQ4qDRv0IxAWamAzBBb8XU\nu/OQ9H704WOjmppbmsaMHlG8w00t467753WeMHDNX7Om7WS/Xryp0rf43EPnp/75\nzdlfJsya/vE7m+tPrxs5ctOR9u3L75l3d/WFeX29Z15a+/OaU9f/3rpp0qQtO5bM\n/WT2xFf7zISBC/nst4/dsD9/ZePlhYcWXRpx7PT63Y/v2Xdp9Jtr9xx85qmpn645\n9psvTb3cv3dz85NfSXx2y6hbph9dMHb+qcq+Px6ptE75YODc/p0UzT9x0ew6+f72\nbG/rhq65L792/r3WGUd6x97+Xf8dPy7b83zPxN39Gt217IuB0wvXL3i29fUHmw7v\n37b4vpMHv2Rzfrp/ylvd/mz7mJYnug/Yvq27ftg658xz045/Xl7y/dfjb0tvXTru\n+IY3Di3aMXnX6rXbN1778Il3F9747eQZq178G3IcKcE=\n=dEMP\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":10,\"last_id_change\":1421456230,\"username\":\"olsonseank\"},\"id\":\"80d75285af4790eac9f5d9d721897500\",\"key\":{\"key_fingerprint\":\"d8f6b6250bf9db76fd04fb51fbf7d62bf14391cf\",\"kid\":\"01016093cb108d4392a64bec275fa4a340362eb316c38fedbb755436682aa84b88380a\"},\"remote_proofs\":[{\"ctime\":1421455945,\"curr\":\"837b336eb97e6572d3268d92b82ac26317ba97dc3e16b59d2d9216522edbbbc7\",\"etime\":1579135945,\"prev\":\"9680024f43099541f79e96969f146a8a852db700c2993c7fb2a67484c954a82a\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"olson-sean-k\"},\"proof_type\":3,\"state\":1},\"seqno\":4,\"sig_id\":\"b5a825025bb706fd9ec5484fe2d4c9738265e32eff105a26f8e8e671525a5ea10f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"22cc88e7d69643d7699da9c423e09940d7de1b739d9d5989bbf365b45eacefaa\",\"seqno\":5,\"sig_id\":\"7d640ba405fe3937ea792ceb709a2ad953e0e70659aac948205bdd1c15cbc8ac0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1421473356,\"expire_in\":157680000,\"prev\":\"a6808c59df1a6406a961e81b7038eb132c9ab04eb827ee844663ef5a7e651e90\",\"seqno\":12,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1421473356,
            "etime": 1579153356,
            "rtime": null,
            "sig_status": 0,
            "prev": "a6808c59df1a6406a961e81b7038eb132c9ab04eb827ee844663ef5a7e651e90",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 13,
            "payload_hash": "1421995af7068ad3bf3982028330e56b89e187a1c94aa1f5426cf61696e1478c",
            "sig_id": "19da9eba50589b026df038f1ca162d5b008a101fa102abc728d3b4a44a4747e00f",
            "sig_id_short": "GdqeulBYmwJt8DjxyhYtWwCKEB-hAqvHKNO0",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMSlAnicrVVbiFVVGJ7RdNLMMhKxyHRDBDaN634Zgx4ydKywSEspOazrnO2ZOWfm\nnD2j0zBBRkkvGhRZWqMMpd00sx5EfPASUkghJRaFJAQ9lFY+KNZQ9O8z0+iYDz3E\nedlnr7W/9X+X/1+fXD+xYWpj94ZScdeSu0njsaPLehqWH/50sD+xFd+XtPYnriMN\n5Sx/KpvOkLQmpdBnTS20pJW5a4NNmpPeUK2llTIsEUwcisoLg5lTgdvIPBaCcqc8\ns0I5yR0LyiUDzTlMDhrTcnuodlXT/IyEK0MY99QrS7jDPAoXaIxRYiUFABjPFPGU\nwKnFSi0bV01SxyykHt5eZX+pvoAwwjQgTgURxhMqlAhKkmgkt4IZbhWimHAtkeXO\nSCOt9Fo5quGPIrCMDUEG4HrqcAxbHCWPlnPEhKcSCaQY4YYSpDBC+cZaqI4qV3GV\ncrlSBUFyAbKqcaVcAqg/dbX8KfWFMTExbk46TC0DQgVXNKASvGMEa6UIkeNw16Tl\nPtiQg9ar0gzR4IGlRsJqRqKXVODIkBFS83pVo+rngo13wDoRHDaaeSYUd4jSiLkj\nxFtjKWFEemmQ0mycolZijJTJNypMpGUaS6UoDoFIYxR8qkBfbSRzNkqvlHWQCwiI\nVhojI0FRqL0aOitZKHRVK5UIcjwB2cvSzhHaSDKFFWpOXE+1Cqc6xOGtRY5HilBk\nmigThHSCgQMsUEENlKoIM4xaDXVT5mjkHMy3GlsH5YdRcC7BsRHwrmrozV0VlDjt\nIQKEY084E5FFUMAH4iOREXRhNhDtBRce5LXSQEQD0xqEM1olY1xyfet86p1UDK5U\n8CYzhTW13OKxlmpPs2KPTa5makeaS1PHKGR9XbBCm5NaZrK8dFiphe5yJWkl8JS2\nj4SfQPEgAQHDpHDeQddZQi10FaJaMw7U8mjbqBgOSkShCKZYcQVfaOskiskI2sh5\nZKB5vBNcEMnGnLBSYOuxYvCjAnHBlICIkBg80cbrgDWjxggjsbXQaRFJa4XhUojg\n4dsrnKAj4KNO/A82/0cn8nEyqnw5rK21uFpLua+nJfiepC5/Bs3bAYvFLOtqvdIS\n6HR0NVfoZa5QzHV0hjpGPIJu0pw6w2w0MGMY8lxK5DS0HHKEek4shRFiggsGW2Sc\nsP9yZXX9mEJm0o6cQJfp66gYXyiaWhGO81Z4ZDDMAOMB0UXtGJPEMhO1IpRAW+d+\nGaOFh6MxBw6CEYE8i4i7XLlREvIyEhyqZjxoC32NIPAseoYFV9EFL+AkyJkJ0gjM\nuVKcAV3LYVzkbhAkBFiYDOSjr05hdAJedoHk0l1KGsnNVwLysa4rrYZCWq5nRChU\nl3s0IsHDVQN3RbAOS8EZRwRRyB0QRhr6FVoXc5jqEErvBXKRe6uChRGGoESCLxHF\nYFdm2gET+JZN1lMNycDhg09e09A4tWHypAn5vdgwdcqN/9yW33zd9OfsZSc+Wnhu\n4dJXhtKh6+YcX/Lgw3tX7Hx/i+ufuXtz/7Y9Nw33tr45f92JhyY2vds29yVz1+M7\nu+2SzXu2bl/wQvx5/X3PT/9t5oIflv7YO2vlycFFpWNixSPL27ZtL71z9tfu2wcX\nvPz0yROHLhydsvoD7p57asZrG5uWvd0y+7sPC4e+Onhmy+vTVr03Ye/p9Yv3DQ+u\nmnPg91vWzMOLvmi+Z96FO7cfmLNpx5ns0dPz93dM3r3p4ucTe88uXnnvbZNmvHHt\n/lk7OndNb/t2cHjGefP9x037yS/PnNpzvrWzePzmixMee5XuaywsPzVh2sY/yl3k\nyE/tf20YttNunf/ZA0ND8v5nV5Mv5dbSDefaVh2Z9NYdL/4NNTaZww==\n=nhwa\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":11,\"last_id_change\":1421988227,\"username\":\"jinyang\"},\"id\":\"9403ede05906b942fd7361f40a679500\",\"key\":{\"key_fingerprint\":\"bc6ec1a94d4685c033f15c22dbab32427d7a0894\",\"kid\":\"0101b71108a15c28127b49178831ee27aa8bab875b9a74cbf7d88bcd168e598910a70a\"},\"remote_proofs\":[{\"ctime\":1420748180,\"curr\":\"c05142b0c5f300f4928ae67c640604e363a27d824a43b924234c3f5513eb91bc\",\"etime\":1578428180,\"prev\":\"4632c9d0b5251d2546f4f2dbde2df27f85c4be29d656d42fb7aff7e4991a9a98\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"jinyangli\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"22c94e32ab376cdcc75b23bd3d039945c9dca7abf841e86f6821318582ab9bc70f\",\"sig_type\":2},{\"ctime\":1420756274,\"curr\":\"b761bd184848360564867a02fed29ad9e1943aa6a71bb72ff07bb6a5766ed761\",\"etime\":1578436274,\"prev\":\"c05142b0c5f300f4928ae67c640604e363a27d824a43b924234c3f5513eb91bc\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"news.cs.nyu.edu\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":3,\"sig_id\":\"3159fca3c42d0dba953ca4bfaa7b40d5770c96850c23d52b3320aecea1b0ac6b0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"db6d0a1059adc23cf9c4472b4af982323f14836aa96d7701510064260d4f05cc\",\"seqno\":7,\"sig_id\":\"595345e9b27a0e494fd41658fced69adb23ae7a61558854ca3b5a083eb9206600f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422051486,\"expire_in\":157680000,\"prev\":\"ed4c85c1ebc1765450203ad9f9c092dff4f1580367add60cf5db8eb894088521\",\"seqno\":13,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1422051486,
            "etime": 1579731486,
            "rtime": null,
            "sig_status": 0,
            "prev": "ed4c85c1ebc1765450203ad9f9c092dff4f1580367add60cf5db8eb894088521",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 14,
            "payload_hash": "1d21eed55eeb085102bf28673e316156187760fc5836ebc0af9aa579ba5e9b7f",
            "sig_id": "c7b288e398279c6d0de3d9accfcbd93dc601a0b51460b92bd965523a7fc246390f",
            "sig_id_short": "x7KI45gnnG0N49msz8vZPcYBoLUUYLkr2WVS",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n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\nS3d98rr6+daBpx5901rrkf8B5q5j+A==\n=997G\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":54,\"last_id_change\":1421371756,\"username\":\"patrick\"},\"id\":\"1563ec26dc20fd162a4f783551141200\",\"key\":{\"key_fingerprint\":\"246201479f89c599c08f29b3fa706aef3d045f20\",\"kid\":\"01017c13682b0965e718feaead9f04e468086c3864ba8294072c97594a5957a99f6f0a\"},\"remote_proofs\":[{\"ctime\":1392412788,\"curr\":\"c20d2386bb1714908f0de84f76d9aabc7c2911a1bc6ee451f98bb8833a528fac\",\"etime\":1550092788,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"patrickxb\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"fff3da1be82b15fa7a5633f85d2ae19e4f128986e6e80e64a4859dedb56f47080f\",\"sig_type\":2},{\"ctime\":1392412864,\"curr\":\"e4e726145a1e148ed4bb2c8481d6a6f399d0e65d04af5e590be98c2ca2f23705\",\"etime\":1550092864,\"prev\":\"c20d2386bb1714908f0de84f76d9aabc7c2911a1bc6ee451f98bb8833a528fac\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"patrickxb\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"81a9003a098ec52cd98a9f4983ad7edea81e9beae23064ceb97518138e3cd0f50f\",\"sig_type\":2},{\"ctime\":1395889480,\"curr\":\"7a1d453f123e38a125f0e872c6ec4ef0ba80119008996742abfad2675a16110e\",\"etime\":1553569480,\"prev\":\"bff52a25614ccd76fafabd54adbede2a2832899a0c0fe324a433de6100845331\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"patrickcrosby.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":11,\"sig_id\":\"8e0043c22d467a8b966c528c17882118c23bec1b7803fd4b5430c500e04352e20f\",\"sig_type\":2},{\"ctime\":1420864565,\"curr\":\"171a9d6e5b305d8eaf72f78e75a4bc814bbc2c5e5a429724640f0727212c0cca\",\"etime\":1578544565,\"prev\":\"eae03042fbc56ad02ffd97e0d07b43a5d9c4f4959a03bdc3a5d1659d0f28797d\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"patrickxb\"},\"proof_type\":6,\"state\":1},\"seqno\":23,\"sig_id\":\"9127676da00562183585508aaf5b5fff4e936f208fc238e388d37a40eb99b3fa0f\",\"sig_type\":2},{\"ctime\":1420864719,\"curr\":\"db4612f774b4e4689b5b8d9ab9c99447d30bce0a969a131d65fa2810b352cc44\",\"etime\":1578544719,\"prev\":\"171a9d6e5b305d8eaf72f78e75a4bc814bbc2c5e5a429724640f0727212c0cca\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"stathat.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":24,\"sig_id\":\"250861cbd50c104723977c779c4a7ab7919d664dabf0228a61fc65d24dd4001d0f\",\"sig_type\":2},{\"ctime\":1420864917,\"curr\":\"6c1866e5442084ff8b6892ca7a8cf2c5555e5c6281553331593e3d8efb911ff7\",\"etime\":1578544917,\"prev\":\"db4612f774b4e4689b5b8d9ab9c99447d30bce0a969a131d65fa2810b352cc44\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"numerotron.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":25,\"sig_id\":\"3c666d4cf57d5f1b11a556def8d0f78e89ec92b994867a4ce0245f5bd187355b0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"bfdae4f98d9910608c2760cf87fb6e3e1ea3527fa57ef2221e4d6e5cafd25cb3\",\"seqno\":31,\"sig_id\":\"baeb664889dc7f1e203bffe3cb364249a55661f45f9c84e3a470bbde36e34f360f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422051540,\"expire_in\":157680000,\"prev\":\"1421995af7068ad3bf3982028330e56b89e187a1c94aa1f5426cf61696e1478c\",\"seqno\":14,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1422051540,
            "etime": 1579731540,
            "rtime": null,
            "sig_status": 0,
            "prev": "1421995af7068ad3bf3982028330e56b89e187a1c94aa1f5426cf61696e1478c",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 15,
            "payload_hash": "185e2496a0dc012deea1c883759a74e0f7c33e38e11a40530aac76cd1916c7f5",
            "sig_id": "de2c80dde617e34b0715fd082b593a2b7eff7d4915c2095e4f11e92871d92d9f0f",
            "sig_id_short": "3iyA3eYX40sHFf0IK1k6K37_fUkVwgleTxHp",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n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\nv9Ii8F4=\n=Az9U\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":45,\"last_id_change\":1421805379,\"username\":\"gabrielh\"},\"id\":\"b7c2eaddcced7727bcb229751d91e800\",\"key\":{\"key_fingerprint\":\"3f6e504945ae44e6a53d49845431890fe1eebcde\",\"kid\":\"0101230aa5a191471449cc05ee95ce3637ebb16e409bb3b92ac0a8cc28eef92c0b500a\"},\"remote_proofs\":[{\"ctime\":1413405722,\"curr\":\"55d1e2013c91f4ef410538cb03cadfcd0b40e86dfa8a0594a42036eedfd61fff\",\"etime\":1571085722,\"prev\":\"461f0051ad08190465b5c8f4cab2fb8c5f04290c3f3cbd53794e7dbe5b89d448\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"gabrlh\"},\"proof_type\":2,\"state\":1},\"seqno\":19,\"sig_id\":\"dbbb07105a43c18ab1a9d01b564c591a6d4d57e35dd4ba3f1a0ed8e33f8517eb0f\",\"sig_type\":2},{\"ctime\":1413405744,\"curr\":\"5381de2b714095bcf65f41c1b235c11ab6fb87cea8b2c9a3bcb95095ab26e191\",\"etime\":1571085744,\"prev\":\"55d1e2013c91f4ef410538cb03cadfcd0b40e86dfa8a0594a42036eedfd61fff\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"gabriel\"},\"proof_type\":3,\"state\":1},\"seqno\":20,\"sig_id\":\"916bccc5e1fd7e784340cca2795d1ae352922bd498ec196bbff6c00b7a304d2d0f\",\"sig_type\":2},{\"ctime\":1413406032,\"curr\":\"eba1b91ed0048983d7e32efeee7fecce16547998ff3e05b67045a751ef99be1c\",\"etime\":1571086032,\"prev\":\"5381de2b714095bcf65f41c1b235c11ab6fb87cea8b2c9a3bcb95095ab26e191\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"gabrielh\"},\"proof_type\":5,\"state\":1},\"seqno\":21,\"sig_id\":\"2c9e345b5acedbcfe3d5d9b2d6115b59e9d6ad31ad262bb07266ccc6db0b95940f\",\"sig_type\":2},{\"ctime\":1421558613,\"curr\":\"fdd474158530c9816ca9707c3cda809aaa24a0f69b5bcc06f5776a54cea54fff\",\"etime\":1579238613,\"prev\":\"3f0c4b94e2c2d6435aa583012793a77222c3ec7b2ab48ff95359df23fa5f422c\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"rel.me\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":25,\"sig_id\":\"fdee6f84bdd854d055fe51be696b175e5f2d2d6b143bc09a52f7bad57dfe05160f\",\"sig_type\":2},{\"ctime\":1421558741,\"curr\":\"158668859ea086ccd02ce16724f2b35a6a71834f98867429cd1210a075346443\",\"etime\":1579238741,\"prev\":\"fdd474158530c9816ca9707c3cda809aaa24a0f69b5bcc06f5776a54cea54fff\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"rel.me\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":26,\"sig_id\":\"9c8edb92c68188a43266c7037e1148c9320a969b248b97059aeca65ddfc39e100f\",\"sig_type\":2},{\"ctime\":1421805307,\"curr\":\"a0a74b05db02a6c66c9cf2ffd3754449ff680cd4fb61f80cba2038a23e7c0bfb\",\"etime\":1579485307,\"prev\":\"a71816d134d54dae20073688eea5753fd756903173b0461213aa3956a3c66de5\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"gabrielhn\"},\"proof_type\":6,\"state\":1},\"seqno\":28,\"sig_id\":\"2787406a0b57a2eb7de822c5ecea6bdcd28a3ed96191a27d46459d747a6b292c0f\",\"sig_type\":2},{\"ctime\":1421805358,\"curr\":\"b8e2fb00585d67ee5144b37483a94e251444144c9dcd5949ca30bf275bafe59f\",\"etime\":1579485358,\"prev\":\"a0a74b05db02a6c66c9cf2ffd3754449ff680cd4fb61f80cba2038a23e7c0bfb\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"gabrlh\"},\"proof_type\":4,\"state\":1},\"seqno\":29,\"sig_id\":\"a3a73a75b6941ff0aeabdededc9640d27a45cff49f2a9b57538b4608a45ba6590f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"b8e2fb00585d67ee5144b37483a94e251444144c9dcd5949ca30bf275bafe59f\",\"seqno\":29,\"sig_id\":\"a3a73a75b6941ff0aeabdededc9640d27a45cff49f2a9b57538b4608a45ba6590f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422051554,\"expire_in\":157680000,\"prev\":\"1d21eed55eeb085102bf28673e316156187760fc5836ebc0af9aa579ba5e9b7f\",\"seqno\":15,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1422051554,
            "etime": 1579731554,
            "rtime": null,
            "sig_status": 0,
            "prev": "1d21eed55eeb085102bf28673e316156187760fc5836ebc0af9aa579ba5e9b7f",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 16,
            "payload_hash": "e11940eb12b36e89eadfd73af91558819262952825faca0af1ccc744fb896a6a",
            "sig_id": "08fa6ab7f3d57852666fc4cc2ccf946557970be7bccd64cf6894d2940d90e53d0f",
            "sig_id_short": "CPpqt_PVeFJmb8TMLM-UZVeXC-e8zWTPaJTS",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowGtVVuIVWUYPaNpGgk+GNgN7ASRzaX/fjHL64NERmWQ3Tz91zl7LmcfzzkzOdpA\nRhEVSpRPhmZoD0USYoYXBi1ItJsGFURaFEl5gwxJQm369p5pmEEfEjpPm/N/e/3r\nW2t93351yvjCpKblL3aWty5qJU2fnbCFhz/+Vq8q2tT3FWetKrquJFQa2VPFdIfi\nrGJn6LOmHtqSdEYl9aGtoz5juKal2Btq9SStQBVqk2202N+SlWcvx6TSHmrVWpJh\nFbkyhHFPvbKEO8yjcIHGGCVWUlihjGeKeEoAspzWG2NuLeaYpcTDv3n9fKHmLYT6\nhXl9T37AsMVR8mg5R0x4KpFAihFuKEEKI5QV1kNtuKXUpZVKWhMiZ9wdap1doVRL\n07xt10iyIswIQZwpqoCTqZfhNaKkshS56IVhRivNGUPBWapJDJFQzyUmkQaCcAhO\nMsGww5ZF5yNVxJIorceYRUkwtoRGbKK22vgIgDoI77VDkjpjNGXKGCWR9FEzAhiC\nY8mhiXpYXkmBHNWSU+DeqBnXmbEGsRJXz54SXxqxBdOWYpepN0C9kisbsCRvDGst\nqFBjNemqp5V6MJXOTJOxImihW4q5zgp5yYniJjKpUTBOR+61h44UMMp1Hg5A5tnY\nEHgVwWvCkY3aWymiRwwMw9FG6QWxETOqsYsZRn4ZwggLpKmzGCkPh8QIZoMjYDQY\nQBmiggRLsXBUxeCtlZwzKoQioB6zCsxDJmunFrrTRihVweMIIj0+2mXMONeMQ889\ntVrWIpWWUhGslkFwmcVSKK+JBVRHBMXSGi29owELy7UncIYFJyQjYJ0E+mEYnEuN\n6RD4MINMlZxFnrRycJ0lbxqm1FHP7BqZufakUe6xxUsNas0cas0tymFKjb4qHILN\n9YZpZHf2j4SEwVPSPjQ4lhsFyhMOGiFQXgcH2WYxkOOeOS0hn4IHSkKMGHFDRFRB\nBSExhxniwWCU2ZLBDV1I+lvGaAiBwpyNaBjBHIYIgeDAjFNhA/c+S4xGMJEOvJTK\nw1vKWi24l1oI4pFmAgMZo8RYDYUcAr8yDV2aVLIVchkVR2I+WkN+OQ3lKA2ZBnuR\ncdwKaTkxNID9lsJCo9FBLK3SzED4FCw3LYITmFJtYNBBQMihuETCJ/NrSg2TdGW8\nq6avKzW+NLxt/gcN/1MT+2ZccRv92ebJmxheQKO+BPjy2yOsqCa1UEoquaVCIfhl\n+odeAMGKBwJ1BnmHMPEBAudgeiXXRrKAonSUBqoCxoYhTpExTgrnsYbRl3H0YhTA\nzLQDJnRcMY2eWij2v9R001WFpkmFiRPGZd+6wjWTp/77Afz0+asHO9ae+vHm/o5f\n1qzdqH9vffDNu2bPWaR2pJM3L9ozezF+qof2Du7a/uEN+sC2R1f/dvrkpu+++WPl\n4Y3b1DN41Xn7xvHTT6y/eOvEo+u+Otf79bKlZ+bsf2ja0bffXT3ur7OquG/dETew\nTC4pP3fgpw33Frae+fWRO5dunDD/gw175u2+3x2qbe/e23awOvDYwMy/F7i3Zurm\nu6dM9ys+Gb9307Xd6nxHYUeTOtn85cs/XEi/WP50877Wc5/fNnDdO/rG9Rc2d0zf\neeiBbfsX306X7D92ik21r63ZufT6uR+tPjhhy9xnX5g6f8HhOy7esnLa+68vXtb8\n8xl5T/m9P1/ZW921+/sjxw6fRbPvG2g61zI4GE78Aw==\n=mZ4m\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1422054838,\"hash\":\"2878b30cfd6a4a9895440ecb392fef23d5712f3e201eec74641c1b4fcdf382b2f7bd114f7211b23f1af9b9adffd69e6dd9c073caa9348aa8707df9421b465175\",\"seqno\":139753},\"track\":{\"basics\":{\"id_version\":13,\"last_id_change\":1421996368,\"username\":\"olsonseank\"},\"ctime\":1422054969,\"id\":\"80d75285af4790eac9f5d9d721897500\",\"key\":{\"key_fingerprint\":\"d8f6b6250bf9db76fd04fb51fbf7d62bf14391cf\",\"kid\":\"01016093cb108d4392a64bec275fa4a340362eb316c38fedbb755436682aa84b88380a\"},\"remote_proofs\":[{\"ctime\":1421455945,\"curr\":\"837b336eb97e6572d3268d92b82ac26317ba97dc3e16b59d2d9216522edbbbc7\",\"etime\":1579135945,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"olson-sean-k\"},\"proof_type\":3,\"state\":1},\"seqno\":4,\"sig_id\":\"b5a825025bb706fd9ec5484fe2d4c9738265e32eff105a26f8e8e671525a5ea10f\",\"sig_type\":2},{\"ctime\":1421996154,\"curr\":\"fa344022969f7136be5dd479090a32cd4378d2198bb965d79662d094614c9a86\",\"etime\":1579676154,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"olsonseank\"},\"proof_type\":5,\"state\":1},\"seqno\":7,\"sig_id\":\"49bbc0ac5b67b52a3eb59b325c3fcbecb894a66886ce96ec61339ab23a5e80a60f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"fa344022969f7136be5dd479090a32cd4378d2198bb965d79662d094614c9a86\",\"seqno\":7,\"sig_id\":\"49bbc0ac5b67b52a3eb59b325c3fcbecb894a66886ce96ec61339ab23a5e80a60f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422054969,\"expire_in\":157680000,\"prev\":\"185e2496a0dc012deea1c883759a74e0f7c33e38e11a40530aac76cd1916c7f5\",\"seqno\":16,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1422054969,
            "etime": 1579734969,
            "rtime": null,
            "sig_status": 0,
            "prev": "185e2496a0dc012deea1c883759a74e0f7c33e38e11a40530aac76cd1916c7f5",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 17,
            "payload_hash": "c49c8c4bef4862d13da267f19416f3edc650159b18f25abfe8a0f3c5417159e2",
            "sig_id": "8afe41e09f1bb4b99765f8fc97ad11aad3d2de26c110d5b858628db443decf600f",
            "sig_id_short": "iv5B4J8btLmXZfj8l60RqtPS3ibBENW4WGKN",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMIXAnicbVJbSBVBGD5aiimRRVeDiK1AymJmdueyp16OZUYiBdaDkMns7qxtJ3eP\nZ4+WliFU9FJhHeqlwiAzMizRStColKLIIqXLQy8ZEVqYdBGrhy6zVm/Ny/B///d9\n/8zHf3f6lFBmStXh6I62DStRSn/v6urQlnvv/b2K4Vm1SnivEhWTl+24FSIeiztu\nQgkrmHGkYUu1mIGwCbFNTKHatk0ho8QgjFsaQ5aKlDxlh+cHCmljcF+scjyJyaLc\nsST6H350sgEggKoAWCWIcAuphBHBKLI5xQbRODYYUCHCOgUGNjnl1KCWzkxVlwVD\nsg05AlzaVU/aadCANsW2gTHQiKVSQADTEOYqAgwCEBB9EXd5pZBsz/Rc14sToir1\neYrEaxxTBCFYXiV3XMnYyc2o9+cvsbiXkIJdErVcPxAkamOBy25hlP/VlhuOa8kA\nJb1GxH3Hkx5QMs2EEwyEGkIIE4pxniL2xJy4KA+mQEwJA/IEQ0SNtBQQ6hoQBkSG\nKvPQBbdsi6rc1iHGjEEdEaRjJAOwuckBt6FpmlTTbIPphJMgD19UuZ70pvKdvEJ6\n+k6FyxPVcaHU993ZNjWUkhlKT0sNdiCUOS3732b0nMsIXfuU3ZpfNrBWtQqalTWf\nmlakHr31bLyl43gSJAc/tnWlNXe2v1pwe+BUyZcZpb/yu78tSj/biJu/Nik/IpXn\nu3+ODNWNL9566XWscHvR5YWbI8cKk7MbNl4fpjlzJ2Yv35/VfyK1r7X+frL/wb2b\ng1UTnaONXQfG2ExcMPYhq+GKvQsDOON138uHkRv2nPCbh/6+OYUt4UfhK+klRWrv\nuo+Rk6Uds0JyDzpOd3w4WDwUKsjyvnubiusOPW1PvDiybMqlRPLz456RocS03uGM\ncO5YfmRe1ysr7dvohXd33hqbcrau/1wy+vZ52Zkl+5bWLTPrL169OL+2ZSw3M+sJ\nPbOxu/E3EGAmLQ==\n=QKaN\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"service\":{\"domain\":\"jacko.io\",\"protocol\":\"dns\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1422256755,\"expire_in\":157680000,\"prev\":\"e11940eb12b36e89eadfd73af91558819262952825faca0af1ccc744fb896a6a\",\"seqno\":17,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1422256755,
            "etime": 1579936755,
            "rtime": null,
            "sig_status": 0,
            "prev": "e11940eb12b36e89eadfd73af91558819262952825faca0af1ccc744fb896a6a",
            "proof_id": "c8f453c788deabc4839f5210",
            "proof_type": 1001,
            "proof_text_check": "keybase-site-verification=iv5B4J8btLmXZfj8l60RqtPS3ibBENW4WGKNtEPez2A",
            "proof_text_full": "keybase-site-verification=iv5B4J8btLmXZfj8l60RqtPS3ibBENW4WGKNtEPez2A",
            "check_data_json": "{\"domain\":\"jacko.io\",\"protocol\":\"dns\"}",
            "remote_id": "dns://jacko.io",
            "api_url": "dns://jacko.io",
            "human_url": "dns://jacko.io",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 737,
            "hard_fail_count": 0,
            "last_check": 1434480198,
            "last_success": 1434480198,
            "version": 2,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 18,
            "payload_hash": "88d873d0340dfc5aa391dcfbe4a07cce979d8805c910b535af043c8e17579119",
            "sig_id": "e8ea962435235ba18daa87ad739bb59415706dbf82be6663e18e02529b9c38320f",
            "sig_id_short": "6OqWJDUjW6GNqoetc5u1lBVwbb-CvmZj4Y4C",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMSGAnicrVVbiFVVGB5TsyZHjHroJZQtGugU634ZTYquWJpCxNRkp3Wdsztzzhn3\n2aNO42BB2oASWtSTD1oYlWVgvQRJWd5ActCiUEyIkB4iqSDNkuzfZ6ZpxiQIett7\nrbW/9X+X/9/72ya3tE5a9Xyl/M79t5JJRw4u72t5+FjbxYHE1n1/0jGQuJ401PLi\nqWaqIelIKqHfmka4La3PXhNs0p6sDlkjrddgi2DiUFReGMycCtxG5rEQlDvlmRXK\nSe5YUC4ZbC9gCtCY1rpD1pulxR0JV4Yw7qlXlnCHeRQu0BijxEoKADCeKeIpgVvL\n9UY+oZqkiVlKPaxe4XyluYEwwjQgTgURxhMqlAhKkmgkt4IZbhWimHAtkeXOSCOt\n9Fo5quFFEdjGhiADcH1NOIYtjpJHyzliwlOJBFKMcEMJUhih4mAjZKPK1V29Vqtn\nIEghQJ4ZVykkgPpT1yieUl8aE1O3Jz2mkQOfkisbECnpwIwQyRTjZAJsqKY9/VVX\nNWX4DoCblRGjsNTNQiKRhIaoIgIfLDwh16xs1IFCtIku4KCRidSzGD0HLSxjyggQ\ngoK9HvgxFzS1cYKqBgfnqbYCa+YwHLZBgnpOI4kQskxJrpgNkXjEhYeYYEOtoFpT\nLKIArUxRexaq9TyUerN6PYIkXZC/PK2OcZeCAHfXl2UFb6YQCp5RjgJjjEvhMHXS\nMGKRM1wyjh2zPiIiJOImaIJRDMEwZbXwQkD5YRScK8TICHhvFlYXCdKcaM9YIBRZ\n4RGxkGAonGhjpFZaS6kk5FsIpjEOAVFOLLM0CsEpNUXiRrkU+jb5NLupHFyl5E1u\nSk81CpvH2srV01qR5ORfvW0ClfL+Xtjk7UkjN3lRP+w0wqpaPekABo20e6QLgCi1\n0HyQTM4sFi56Y7jXUVqMLHRp4I5CzwrGI2IRXI1eQHBi0JFYA2olI2gj95HB9svs\ngCioMTskioyHKAkzAWSKnCsP7nvMYE1bFZxWEsQzxMAIEBQ+D1YRjQiWRih+uR0j\n4KN2/A9e/zc78jVpnodsohujRoQsbVQvd4NcyQ06zg1jHfcMhEBgBg1cR+vAA2IQ\nRQraQWBho7EghcEKpAlEcGBDLcTewOo/3FjZvKaUm7SnKLzX9PfUjS+VTaMM18UI\nXYkCIoFia4JBXEpOLcQZpJMwlwkSIB6m0mEEiS5OUB2cgGxjrVkyRoKNIyECZUVP\ne3AQZqsTmgbKsQ0UWlkSL7jH0cqoZGSE+oCDD5GzAPdYa5gBEoPF7GtSGB2B4/4g\nhXTjEqaVgJRALtb2plkopcUJsB6igNBYNBzTTrlitDAliMfUG5AtwhiCwUKDd4Ij\nzLXFKsJgtjEoqII6zrCE5UD+Jorhqtx0AybwrZm8LwvJ4CcfPz6lZVJry9VTryp+\njC2t187863f5bcu0S+b1Z5P193zwyE+L2ru/XndJLX5r09YHbpcPzjr24sUNF15Y\nXjlZP/9Etvrsh8cXrliwaO8Bu7N1/V07liy+6dG2qUuOz1k594ste7c/tHMb2br0\n3uV37pr/9onrrvlteM8Nv+7b3bXrUH3h/i/XHejccfjJb2auXzbn3elH/tgx15+v\nVLoGZs/Smzsv/Hz9d3vmLxv6bJve1XnLPLdh3UuCts0+03Hu6CubLvxee/q+T6cf\n+vFUfvqjjafDmf43Zmz39R+G1v7y2InDN++7cXjDgnlL+cCJrs5NSW1O9c1Tw++n\nKw4ffa805diWu4fPPfcav3jH5OTzkxvZtDPlZw7OODhkNie9L3916tWS3P79wNmz\nuxcP/Qm5OaOL\n=HJ4a\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":9,\"last_id_change\":1422748452,\"username\":\"emilymcmahon\"},\"id\":\"2a81795a320f2723ef8f05c8b3ef0c00\",\"key\":{\"key_fingerprint\":\"1e90af3d4ffd5597b448a67d932c0d8424ce93bf\",\"kid\":\"0101a1ecd39b6194c17d9be77b7c907000b487584bef2d056dc0f1a3b6399316f6100a\"},\"remote_proofs\":[{\"ctime\":1422747622,\"curr\":\"e4800ed4350e444576c13c7a42b0ca57451c4bdf026705ae9210feea48b96d66\",\"etime\":1580427622,\"prev\":\"79529d44e230b6d02bd4b56d29aa798997787bf4664911ee0352b4b3f66533a2\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"emilymcmahon\"},\"proof_type\":5,\"state\":1},\"seqno\":2,\"sig_id\":\"48b3b35c46d54b16cfdaa5d9f7b10bd6ae5c3212645f04f4cefd62a8fe9f2ba10f\",\"sig_type\":2},{\"ctime\":1422748428,\"curr\":\"70f45ef724aed29f558dc90d14f729b8ec987aa7a2ae8c63748eb8290217a685\",\"etime\":1580428428,\"prev\":\"e4800ed4350e444576c13c7a42b0ca57451c4bdf026705ae9210feea48b96d66\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"mcmahonerism\"},\"proof_type\":2,\"state\":1},\"seqno\":3,\"sig_id\":\"abc5d49b8054b3e59fbcd6a2a03080f1616bfaba68a188ebe2652673b747aa680f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"ff0100e02e31baea057753b2306c174d12062b0137c10798ea0539ec68991994\",\"seqno\":4,\"sig_id\":\"6e347c90d8c6d48c693e351be363972d65d1fb7f87f423de1edef54e013bba4a0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422986748,\"expire_in\":157680000,\"prev\":\"c49c8c4bef4862d13da267f19416f3edc650159b18f25abfe8a0f3c5417159e2\",\"seqno\":18,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1422986748,
            "etime": 1580666748,
            "rtime": null,
            "sig_status": 0,
            "prev": "c49c8c4bef4862d13da267f19416f3edc650159b18f25abfe8a0f3c5417159e2",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 19,
            "payload_hash": "abf4894d0e34671e497ab1f2982dcb24c8e8974d10a00576eeb81ef236d75da2",
            "sig_id": "e7ee174db64a5f37d2ce5765367b8bd073aa81627356b9aec84af7b0136ff6380f",
            "sig_id_short": "5-4XTbZKXzfSzldlNnuL0HOqgWJzVrmuyEr3",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.4\nComment: https://keybase.io/crypto\n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\nk5E=\n=vfBK\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":19,\"last_id_change\":1419475392,\"username\":\"willwilson\"},\"id\":\"dc1944ea9df3dcddecbf994031a89e00\",\"key\":{\"key_fingerprint\":\"dd613e6f6261ad1ea6a72c3d32c8cec143040cb1\",\"kid\":\"010180b3d73a72827897132d4cc422ec07e5786db6212b6a9ddacad48b767a1c88610a\"},\"remote_proofs\":[{\"ctime\":1419468116,\"curr\":\"c4b068dc645df3ac1352e1bb5647f37633fecd0024593e716f98cb2d4832fcdf\",\"etime\":1577148116,\"prev\":\"b8da9a936fa9489b58943c64c170e095209f58a750da769bfa821b24e42a8422\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"williamaugustwilson.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":5,\"sig_id\":\"5f576a0731d37e3e025f4170d84ef0cb8f54709d2d3882bb247cbfc8db446cf90f\",\"sig_type\":2},{\"ctime\":1419468293,\"curr\":\"b7c1e5f80170b827f93cdea9831bcd6612b9b4c1997310d7b7cbe0ab3f6b154c\",\"etime\":1577148293,\"prev\":\"c4b068dc645df3ac1352e1bb5647f37633fecd0024593e716f98cb2d4832fcdf\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"wwilson\"},\"proof_type\":3,\"state\":1},\"seqno\":6,\"sig_id\":\"41a916d2bc13a407b22d71271e4b85d624d0ca61110198eae7f9afd8694731e10f\",\"sig_type\":2},{\"ctime\":1419468468,\"curr\":\"00ed23721676c981446cbfe1d8fecb5e08af42c562c27b6502e7dc0686a9f738\",\"etime\":1577148468,\"prev\":\"b7c1e5f80170b827f93cdea9831bcd6612b9b4c1997310d7b7cbe0ab3f6b154c\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"willwilsonFDB\"},\"proof_type\":2,\"state\":1},\"seqno\":7,\"sig_id\":\"bafbd853dd34d2398538d9015f3ea2d596dfbd191e7618cb1a5ad35535200d450f\",\"sig_type\":2},{\"ctime\":1419469445,\"curr\":\"8a1a8927a662c0f0ecd438b7c4c5de1ef91ad48c93382bda5467c8c30af328fe\",\"etime\":1577149445,\"prev\":\"6304bb04a2777bd09fbdd8556e0434d6b4c77c0d1247c5b017278547fd886f12\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"willwilson\"},\"proof_type\":5,\"state\":1},\"seqno\":10,\"sig_id\":\"ded5deb64539634e097a17d666d1a7e637e26470b7861c482a3273c63728bfb10f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"8a1a8927a662c0f0ecd438b7c4c5de1ef91ad48c93382bda5467c8c30af328fe\",\"seqno\":10,\"sig_id\":\"ded5deb64539634e097a17d666d1a7e637e26470b7861c482a3273c63728bfb10f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424279037,\"expire_in\":157680000,\"prev\":\"88d873d0340dfc5aa391dcfbe4a07cce979d8805c910b535af043c8e17579119\",\"seqno\":19,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1424279037,
            "etime": 1581959037,
            "rtime": null,
            "sig_status": 0,
            "prev": "88d873d0340dfc5aa391dcfbe4a07cce979d8805c910b535af043c8e17579119",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 20,
            "payload_hash": "64868e2b2a3dac78b9040722ab3217be565b887be8282c34a52842de6ae55035",
            "sig_id": "3c756a399e14d984ca1c36e1414a12c1d3d1b6cb46451d36ca8e21068d8ecc5d0f",
            "sig_id_short": "PHVqOZ4U2YTKHDbhQUoSwdPRtstGRR02yo4h",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.4\nComment: https://keybase.io/crypto\n\nyMPYAnicfVRbiFVVGD6jpTURZmWFFdkuinLSdb/4YOGLUl7KTMms07rO2Z7x7HGf\nPdogk6ZUD6JBBaVBhFAo1ERgWApOFy+R0kUMSsoUCisssCwsRfv3cTIHov2wWWev\ndb7/u/z/2nHp8Ep72+Kn67U3pt1B2vbsKHoqc46ePrUssZnvTSYtS1xXGhpFuWqY\nRSGZlNRDrzXNMCHNxi0NNulIloS8mWYN2NJUC8GNDU5qJ7lWQvNAmVFca424JTp6\nZZxI+jpKmBI0po3OkHfnaVkj4coQxj31yhLuMI/CBRpjlFhJYYUyniniKYGqtaxZ\nDGGTtDCrqYev/3G+3tpAGGEaEKeCCOMJFUoEJUk0klvBDLcKUUy4lshyZ6SRVnqt\nHNXwQxHYxoYgA3A9LTiGLY6SR8s5YsJTiQRSjHBDCVIYofJgM+SDzmUuazSyXAha\nGlDkxtVLC4B/6prlKvXVc2ayjqTLNAvQU3U1AyYlkzAjTOGSxxDYAMes6V1o6qHE\nbRGjjAevTTAIOYV1qQxsDTQQBC+sk3MBlJ4NDQFLEoIxzAZig7deRyogxWCsBOuo\nddiAXZQPMZUDNR+FpMLAOyIGQZLADefaBeNxjNYIaQkCdyT3BkyNhEUCDBkUiWAq\ncM/DoqwI1e48yyI48hC0X5Eu+lc6Z6ojcT15DlWtDsECMnNBOgoqlZTBWsUgVkod\nQip6SSimzkRpovcUK4S11UghyEkC/TAIzhVh+iz4IIPSlRaL1gjUgqtXvSlMdWGz\nzObcLHSmRa3HJv8XRwulWvR2wx7tSJqFKcqSsNMMixsZLGGVdlYH+0l5LIzxPjpu\nbYxCEeU50QJTHyz1qJRnEXYOeU+g6wzy0KgOGhZ77lFMzqKdrUf6Hm6VqRYm7Spp\nd5versz4as00a2VwHLMI0UmLvNUMCUc0ch7hAAWFx5SCu4oiiwgiOjDKmEGUuOCi\nd8aWHTAogpwngiPKtGPgO5aUoWiEhqHTESRgLLAVkpNgudWWB4YjxgEzHokKVjJN\nJAcRfeV8tCQMjsl5t0xp3dCu0FhAlo91p3mopuUJLoVC8JTuhyUAYmxkSjMPzc+E\nxAHSNjC5RMPt4CwBkUFpyTwGPxH8GzpL4RDhfvBls5LzhAJoYToBE/Q2TNGTQ8of\nvLfggkpbe2XEhcPKy7PSfvFl/1ypa64deRr9emSDmzIiue/IzlummNVvrdjSv2nN\ndds6ybyjs59dvnP4H5X9Z/r3x3Tb3AOH3j+ih42/c+e8dwZ+eeDVU3jq7mvC8faD\n4xZ89f2KUZP7B2Z9uGv+/OQiUfl05uP51Ok/nfym47kv/lq1+/rZcz4/kP1uZ25b\nvXXJK23f3rVszOhdj/y2ecPafQPLT0x/5vB3u57fsnflx9WXl8b1e57cVJ0x5p5P\nNq5+6m7846jKhNtW3bz36hfHfaZXrV1579f9yUeHTp04fdXEg4evXPHm/cdHT2yb\nsvuSfTeysY+u26j0TSuPXXF4w/i+gZ/r2384ue7yOdtf2zr35Pw/b7h10+Yzr09b\nfGzGS2Nvf/vdF2adeKLLf9n34OSRfwNe3h4Z\n=8sIp\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":4,\"last_id_change\":1424817825,\"username\":\"eastbayjake\"},\"id\":\"345ed9aea00c8194a5b5d3e3e20e3e19\",\"key\":{\"key_fingerprint\":\"172eeaa4be2bedbd9f36e34eab73e03bc1aa7535\",\"kid\":\"01015248df6736af67f04a242e5a559cead1ffba67b2042575da98cf24f2ea04edbf0a\"},\"remote_proofs\":[{\"ctime\":1424817548,\"curr\":\"b9eebad14ce7c3c81877ebb84d2333c008fd72313caf7afdd318019b90807067\",\"etime\":1582497548,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"eastbayjake\"},\"proof_type\":3,\"state\":1},\"seqno\":1,\"sig_id\":\"418d16aaddfc5bbff6828d529613deb3d084d2b01cc0dd2550a0d5cac9701d5d0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"0514f1527b0db9406c290cd01e9616d13314c830b02029e4344a032cecfdcab5\",\"seqno\":2,\"sig_id\":\"50349c43ca17340fa690139f13d1161b6752eb5b9b5e41f11e145f28eb7492750f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424817916,\"expire_in\":157680000,\"prev\":\"abf4894d0e34671e497ab1f2982dcb24c8e8974d10a00576eeb81ef236d75da2\",\"seqno\":20,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1424817916,
            "etime": 1582497916,
            "rtime": null,
            "sig_status": 0,
            "prev": "abf4894d0e34671e497ab1f2982dcb24c8e8974d10a00576eeb81ef236d75da2",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 21,
            "payload_hash": "435c23da23b8f0cdc5d01386c4fb45cabc0c8cc18dec204da64e92e4183ea522",
            "sig_id": "de8c6c5e3a795efa0134ece15854a3b27f66285e1aa54a889c056b344b992bde0f",
            "sig_id_short": "3oxsXjp5XvoBNOzhWFSjsn9mKF4apUqInAVr",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n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\nOPH+6enZmZn/Ag==\n=Xn8y\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.7\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1424822283,\"hash\":\"1e6cff7e202e00d95da40c270129ac4a2b2da7fd20b63742a51192d9e3c8b37181272499a8c8ccd3359aaacd22493d961de6f0e9427b63f934b61ddd4e1f7a7c\",\"seqno\":161166},\"track\":{\"basics\":{\"id_version\":10,\"last_id_change\":1424818329,\"username\":\"eastbayjake\"},\"ctime\":1424822326,\"id\":\"345ed9aea00c8194a5b5d3e3e20e3e19\",\"key\":{\"key_fingerprint\":\"172eeaa4be2bedbd9f36e34eab73e03bc1aa7535\",\"kid\":\"01015248df6736af67f04a242e5a559cead1ffba67b2042575da98cf24f2ea04edbf0a\"},\"remote_proofs\":[{\"ctime\":1424817993,\"curr\":\"d0732de68050ee3c1d182e10c17e5833eee5d800e587f668baa5607382a4b420\",\"etime\":1582497993,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"eastbayjake\"},\"proof_type\":2,\"state\":1},\"seqno\":3,\"sig_id\":\"ebd456fa9d628a45430314a99e31f3f09bbd42b0f0b7f99d939210a2081af2d60f\",\"sig_type\":2},{\"ctime\":1424817548,\"curr\":\"b9eebad14ce7c3c81877ebb84d2333c008fd72313caf7afdd318019b90807067\",\"etime\":1582497548,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"eastbayjake\"},\"proof_type\":3,\"state\":1},\"seqno\":1,\"sig_id\":\"418d16aaddfc5bbff6828d529613deb3d084d2b01cc0dd2550a0d5cac9701d5d0f\",\"sig_type\":2},{\"ctime\":1424817824,\"curr\":\"0514f1527b0db9406c290cd01e9616d13314c830b02029e4344a032cecfdcab5\",\"etime\":1582497824,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"eastbayjake\"},\"proof_type\":6,\"state\":1},\"seqno\":2,\"sig_id\":\"50349c43ca17340fa690139f13d1161b6752eb5b9b5e41f11e145f28eb7492750f\",\"sig_type\":2},{\"ctime\":1424818115,\"curr\":\"4a8a01b5f1fa0e85e6f1ff5f303b3fde520996845715e2c0256408b349cc8214\",\"etime\":1582498115,\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"eastbayjake.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":4,\"sig_id\":\"4391f1c718270ec52aa912be0ba8be237203676c3b84ee4e2baf8c703162dd940f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"b4930677467667ea292ba7c7320e1c336dc29bb575863a52806cd02fc3d6b633\",\"seqno\":6,\"sig_id\":\"82b1f91595ca62c66d4ac2166cc08ca777ee1a20dfdb8e69ae8989531afdcac50f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424822326,\"expire_in\":157680000,\"prev\":\"64868e2b2a3dac78b9040722ab3217be565b887be8282c34a52842de6ae55035\",\"seqno\":21,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1424822326,
            "etime": 1582502326,
            "rtime": null,
            "sig_status": 0,
            "prev": "64868e2b2a3dac78b9040722ab3217be565b887be8282c34a52842de6ae55035",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 22,
            "payload_hash": "3061ceab5f82a87df7cc4065c58baccd3bde0513326e40a4e71547278f93eca9",
            "sig_id": "f8aedd7699783c95ed1ca93f23dce9fc7a0ff6e394d080ce1c8a09687845311c0f",
            "sig_id_short": "-K7ddpl4PJXtHKk_I9zp_HoP9uOU0IDOHIoJ",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.4\nComment: https://keybase.io/crypto\n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\n/t6HP3p47z/vVEs/cerl/wLKKfqA\n=4aXe\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":12,\"last_id_change\":1424824689,\"username\":\"stegmaier\"},\"id\":\"9d046b0a27207caf23b7f5cc274b7f19\",\"key\":{\"key_fingerprint\":\"5e2add1d87fc0563a26611304def84758180f0ef\",\"kid\":\"010137bceeaa5339f9aff69138e7017d6d2b81e73171e18001b13243acdbdc2b1ca40a\"},\"remote_proofs\":[{\"ctime\":1424823729,\"curr\":\"5abc0ced21f1694e067127dde72dc38c7f5f3c5be7e1d466f4ad1756649cae9a\",\"etime\":1582503729,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"andrewstegmaier\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"d5879ff816c0ca3054318da51a9ac142a67e3195b86e626275e9545c133185e40f\",\"sig_type\":2},{\"ctime\":1424824164,\"curr\":\"397c7c7d29ca24f4749d2594fcccbbd66c036ea7d9412f15078c1531b71f8443\",\"etime\":1582504164,\"prev\":\"f22b29887d3575e5922871648710e4ddcf636625276d5ad6c5524255f7c314e1\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"stegmaier\"},\"proof_type\":5,\"state\":1},\"seqno\":3,\"sig_id\":\"fae790643233c390924532d7c235c4e4eb0e0951a7cdc119cd9c645ab38050010f\",\"sig_type\":2},{\"ctime\":1424824319,\"curr\":\"7fd9202ac698ca5f3b10fcd2efd0eac7471ac56c07901cb39c63f0edfdef6628\",\"etime\":1582504319,\"prev\":\"397c7c7d29ca24f4749d2594fcccbbd66c036ea7d9412f15078c1531b71f8443\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"astegmaier\"},\"proof_type\":4,\"state\":1},\"seqno\":4,\"sig_id\":\"b0c2e6e85c404cf2412b4c9c8a5ee17438646bdcbd44ecc4f1ff3525ef5904d20f\",\"sig_type\":2},{\"ctime\":1424824404,\"curr\":\"fccfa6cf07cbaf680c5ae58501cb41389cbac851e780989cbddb704ac6d4ecec\",\"etime\":1582504404,\"prev\":\"7fd9202ac698ca5f3b10fcd2efd0eac7471ac56c07901cb39c63f0edfdef6628\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"astegmaier\"},\"proof_type\":3,\"state\":1},\"seqno\":5,\"sig_id\":\"0b10945226b9f7ac0dfad1c21d05e8095b0843c17690d134e048605d910333140f\",\"sig_type\":2},{\"ctime\":1424824546,\"curr\":\"5560b5c00302f463d8d334fbe85707014e2c5feeec931e282f6d708f361cc2b4\",\"etime\":1582504546,\"prev\":\"fccfa6cf07cbaf680c5ae58501cb41389cbac851e780989cbddb704ac6d4ecec\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"andrewstegmaier.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":6,\"sig_id\":\"1edf0e496c36743244b2adb340f086dc8faddaba99c1b346efcfde8695506b570f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"5560b5c00302f463d8d334fbe85707014e2c5feeec931e282f6d708f361cc2b4\",\"seqno\":6,\"sig_id\":\"1edf0e496c36743244b2adb340f086dc8faddaba99c1b346efcfde8695506b570f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424825053,\"expire_in\":157680000,\"prev\":\"435c23da23b8f0cdc5d01386c4fb45cabc0c8cc18dec204da64e92e4183ea522\",\"seqno\":22,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1424825053,
            "etime": 1582505053,
            "rtime": null,
            "sig_status": 0,
            "prev": "435c23da23b8f0cdc5d01386c4fb45cabc0c8cc18dec204da64e92e4183ea522",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 23,
            "payload_hash": "7b38bc16fc9a65e8f7fe77cadd751a561378f59975d24ed7274ffaad09a04392",
            "sig_id": "49eb0fc70bc3f269b2c82b7ec8ebae60fa4da0263e0e532eb9c31e603d6e90fd0f",
            "sig_id_short": "SesPxwvD8mmyyCt-yOuuYPpNoCY-DlMuucMe",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMT+AnicrVZZjF5jGJ6xRYsSkVSRqEOQGM23vd8ySUMisVdcKEr5837b/MfM/P/0\n/KdTo9oJRcVasRUXlpC0ohe9sVyIJZao2Jcb0rhBJIJELDfEe/6OMaNNkPivzn++\nc573eZ/nXc5rh+0/sHBwzebR9o5zTxeDb7+5Ze3AygO++mp94btxqhheX4SxMnXq\n5qqD46kYLkbTlMdeWlZ2l65LvhgqJlPVK7sdOhJRmywsWC6dF4bbaCD6YLnBIJVF\nZpUSAKLYMNTANKC57IykaqIqmxgFWBQKoozWCwgcsg5J5pwJymivLUZlRZSCora7\nvXoem6KP2Soj3d3H86P9A8YZl4mB1EJjFFJbnawRGQ14rRC8ZZILcIZ5CGjQeBOd\nDdLRHyvomKNgSHBr+3CKe54NZA/AlI7SME05CkApmOWMNQ/2UjWjXDd0O51upbVs\nBKgrDKONBMS/DL3mqoytWTHtUDGGvZryaYU2kkjFMFdCS2OdsvNgJ7DXRt5A9jlZ\nTEIpcAmMsMyJhFJqw7jg0iBwV8xq38g1X38vXfKYlLYefFacA7cqAEvWgccGOegM\nEObp6Z0PwiEkoywLKSG9YYJnwYhoQQsGUXDOEw8ych1j1krapLVwkShxQ3oS9yqN\nd+vUmqi63UxiXEmVV5fjf2UtuRoqwtqqamgmE6WS3DKiJ2zgIeholGKuMYr0dZhF\npngI0iitDU+YkAeWjLOIluinGXCwisEe8BkGjSp9Fv3qb6cw2opYY+uaXmPLbBtU\nKcaynm8wjpRjKeBkGsdOk1MfplVPTdAhBejVWDcx6aSX1nS6dElX5ciemnVEGWK0\npGXwGLTiwICS8BytSSobyyIlyJ1NoCCLlACQx4woZKLqZbnYg7Ynntgw9HcJtYVZ\nCXPkgvQACRGVVWBUMFZpQEsVrZKjPg6W2kKBd9EboZnjJlpNbBKHEPaSsA8+UaVJ\nAg8h5uCoBjz1A3nsJfWJUC77bHx2qJAmgqHADFlQVJ0cmXQ65mhYNK7pr/9mRr2u\nrOtUzXfj4qYvLsDxqV6bWP3NDrEvO9QcO6hmibsTylukCRSYdWQE3WPUUFHkjNT2\nghJTqEgn9FG5YGVC5rJk6P7ZDmvlrB2GZks2QjCXEL2hmRKDi+A0l1aR4Yk01lEl\najfmQGmRhKV5k1ElaqqEe9nRB5+x43/w+r/Z0abJRjakdb1/3x96X4bAHEMUDR/Q\nINGSzl5oFSJAv54M6cBAS+ET5qCl5i5G0kgnnZNlCriibbCXIVf1w7RqLMca7hM4\nNdbF2GpT2TSrSCVyNZumej2QozKFhGCbGcNo4BAmaNQ0TXy2hhlDE47GkKLINAWx\nWYwzSZi5SWSg+uBZc+EkveytR6+k8lkwxtHRIhHeewXN7pExMpVTsjogtUaWwVIS\nG5rd0U9hZoXM2cCNdHOLjFxuJlu6dqKsUqtsngCjLaPfbHVIpjll5iFbQZMmZhMC\nFRsEIG7UydJHWphcStKTGlYlw0EZYWx2JAm6vxIVVHI1jhAm5dvBem2Vig2vvrz6\ngIHBhQMHHbhf82ExsHDBEX9+bjz//sEDm277We9frHh5+olNn73x0YGv7LdmkO1+\n4dyRlWLJiT/uPPb33uJdl8jjJp/bvPqRZffuvviF39677qBtx04X/pSzv1m16c6R\nN7bvUk+3T/jxtw8vXPzD6q9/Obn8YMFLR565fmv6btGVi8UV6tbigXeP2rFix7fT\n52358JSzLvvk1kNvfNEuv2Xi58eOXnn0Rb88cMvGjQ8v2TZ40xe35eHPLzM3PDnw\n6Y3v/HrNXcNBbvvygzOqe65/6LhjRu5csWrp8unT3srPnNR9d8lj9y168Pvr7r/j\n48lzln1ywbbPFh353M7DD8Wdl9++vdJ3b772keNPvWv3mi0Ltj41fdj7p9//+uDN\nY48uXbr80h8O2fjazl2vf/rs1p/Of6n6PDx+9R9hSf1+\n=m57g\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"2d67f2858139b2718d75dbc817ac348a08442552\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":8,\"last_id_change\":1426378948,\"username\":\"pasha1\"},\"id\":\"8ae24459e5728092ea3367012137a519\",\"key\":{\"key_fingerprint\":\"b39ebae468b5bf4115184c50e895ba8ae2c6f55c\",\"kid\":\"0101b9bc29a5e7480ceea4c57cb0c72d856205d2111e1c3d16ddf6438e6629d137170a\"},\"remote_proofs\":[{\"ctime\":1426378314,\"curr\":\"be7d34318018428c1cc6d7440939ca6639af2f1c3a53746671eaea1c0e798aa8\",\"etime\":1584058314,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"agilecaveman\"},\"proof_type\":4,\"state\":1},\"seqno\":1,\"sig_id\":\"9f2f5dd8c29cbac641505f1cb1a87e4f780d093198e545f2ee55a1dfaa23e5b60f\",\"sig_type\":2},{\"ctime\":1426378685,\"curr\":\"fd12c0e535da484574c78465a8b1f4e97f2c8fa745b9db7260917d86b1ae15cc\",\"etime\":1584058685,\"prev\":\"ccdfc97cbb060137b31b1249fbf7bf9a4a17a7da40a0c43671a0396dfd70d79a\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"PashaKamyshev\"},\"proof_type\":2,\"state\":1},\"seqno\":4,\"sig_id\":\"d21601924b8afffc0891a81600ea3d2ffa5fb231b4a4fa7abd49c83ea09f30a90f\",\"sig_type\":2},{\"ctime\":1426378883,\"curr\":\"725bf72209eaab7f75dc9d59613848e5e8406d4e101095462e286d3fa4e3d1ea\",\"etime\":1584058883,\"prev\":\"fd12c0e535da484574c78465a8b1f4e97f2c8fa745b9db7260917d86b1ae15cc\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"agilecaveman\"},\"proof_type\":6,\"state\":1},\"seqno\":5,\"sig_id\":\"4e2c5653a89c8b264cd55f7bf973fa05632beafc63619dd4626e6fe80451476b0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"54ea7af737b3b530a3ecea58093901cc14756a6aa8bf8707711ecc64dd480cab\",\"seqno\":7,\"sig_id\":\"4f509f1f612936aab8bab434bf2001a9b552bbb45a7a73dd04fee86cafd7f3c80f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1426384614,\"expire_in\":157680000,\"prev\":\"3061ceab5f82a87df7cc4065c58baccd3bde0513326e40a4e71547278f93eca9\",\"seqno\":23,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1426384614,
            "etime": 1584064614,
            "rtime": null,
            "sig_status": 0,
            "prev": "3061ceab5f82a87df7cc4065c58baccd3bde0513326e40a4e71547278f93eca9",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 24,
            "payload_hash": "5cae5e8dc5d0cd6b233f12f6aeb03f45ca4c12b3fc2c337ed31112f0ebaba2a7",
            "sig_id": "c930a19b6e372177c3ba9e840544cb27fad487f3b72cbafd5da863b97eed66cc0f",
            "sig_id_short": "yTChm243IXfDup6EBUTLJ_rUh_O3LLr9Xahj",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n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\n05sPP/YVP/+Hi1/684vq4MTZb/3x/SOH33n9jq//+GMf3fBO\n=ftZy\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"2d67f2858139b2718d75dbc817ac348a08442552\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":38,\"last_id_change\":1427140928,\"username\":\"akalin\"},\"id\":\"ebbe1d99410ab70123262cf8dfc87900\",\"key\":{\"key_fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\"},\"remote_proofs\":[{\"ctime\":1409475047,\"curr\":\"d8cd14f0d6da7dad0b0227b024ecb99c920e6110ef9486262be08df3b6431e00\",\"etime\":1567155047,\"prev\":\"738bebce2b4ed88c9319075bfed58779734efd03f3ee5386b15b38b61d2024f5\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"fakalin\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"1a67183170ef64b00e805c74d39c5286a6a9b50939b3604b15f0722aae27500b0f\",\"sig_type\":2},{\"ctime\":1409475097,\"curr\":\"99456cb27d7c6c8e6978be4c0928c19a07428e236fa9f7ed4d823d926f054701\",\"etime\":1567155097,\"prev\":\"d8cd14f0d6da7dad0b0227b024ecb99c920e6110ef9486262be08df3b6431e00\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"akalin\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"68a547eed57a11b62bfe2511fe128cc10cfb70f199453aa9cb964b97999f5d800f\",\"sig_type\":2},{\"ctime\":1409475156,\"curr\":\"a2a877a4cda04a32aa02de40137d22b2a0a04f35f7f677bdcae15fbf91f4ec04\",\"etime\":1567155156,\"prev\":\"99456cb27d7c6c8e6978be4c0928c19a07428e236fa9f7ed4d823d926f054701\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"akalin\"},\"proof_type\":4,\"state\":1},\"seqno\":4,\"sig_id\":\"dece17e125e06d6b44438218d29ab55a7785beaeb7bf372d8f98eb228009635c0f\",\"sig_type\":2},{\"ctime\":1409475252,\"curr\":\"297ac58d197e52956fae5d628b4db81de68e8cebf1bfd17fbb4404cee7edba6f\",\"etime\":1567155252,\"prev\":\"a2a877a4cda04a32aa02de40137d22b2a0a04f35f7f677bdcae15fbf91f4ec04\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"akalin\"},\"proof_type\":6,\"state\":1},\"seqno\":5,\"sig_id\":\"fc16a961ab69ae81910f3a76ca000fca19699dec8461fc7ec0e8fcb78764069b0f\",\"sig_type\":2},{\"ctime\":1409475374,\"curr\":\"04e5fba11ba32227cacb8b8345cec9bee563bfd4c25b77aa716b3763c3857735\",\"etime\":1567155374,\"prev\":\"297ac58d197e52956fae5d628b4db81de68e8cebf1bfd17fbb4404cee7edba6f\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"akalin.cx\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":6,\"sig_id\":\"ac377cb7d4df7bc9db7c50b3bc06146e05e29fd0eb715b478ea461074a8770550f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"ce162011e380c954de15f30db28f8b7b358866d2721143d9d0d4424166ce5ed8\",\"seqno\":25,\"sig_id\":\"48267f0e3484b2f97859829503e20c2f598529b42c1d840a8fc1eceda71458400f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1427999616,\"expire_in\":157680000,\"prev\":\"7b38bc16fc9a65e8f7fe77cadd751a561378f59975d24ed7274ffaad09a04392\",\"seqno\":24,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1427999616,
            "etime": 1585679616,
            "rtime": null,
            "sig_status": 0,
            "prev": "7b38bc16fc9a65e8f7fe77cadd751a561378f59975d24ed7274ffaad09a04392",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 25,
            "payload_hash": "91e23015983d77e62a467470aedbe563db62ffdcebcb5bacb22b1e81a81e0575",
            "sig_id": "d46203a6ef3dec608d5a61520b4310f3f72073ab461289d1937a3d50db4ab03c0f",
            "sig_id_short": "1GIDpu897GCNWmFSC0MQ8_cgc6tGEonRk3o9",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n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\n5dbu/q/dvOvr737o6RP/6bz/vxjQa/Q=\n=MXEx\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"a83b5e55a841454ff3f9eb75c03f5e269ecd30ba\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":17,\"last_id_change\":1430212043,\"username\":\"marks\"},\"id\":\"e3d11da4dd6ea5b4e328df9a80ad2119\",\"key\":{\"key_fingerprint\":\"5766b7358932d52ac7edd391d279eb4d0dd8be48\",\"kid\":\"01019c72c962ba85496f17dfa5cd39f5b1a7a98b8fc22dc26e4920e7d6bf5311081f0a\"},\"remote_proofs\":[{\"ctime\":1429044821,\"curr\":\"299f4e2461ae29d61153d712a4d5dd42b19b84ebbe3c6b6ae0cd31a27ea9587c\",\"etime\":1586724821,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"laebshade\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"666bab030581509664ad2649ff34804fff664adb96ef6a765cff47e86321d0a80f\",\"sig_type\":2},{\"ctime\":1429044876,\"curr\":\"643678824683fc60026c2ac03344fdeacff1f6bf95dc02189caefd825cf07b3f\",\"etime\":1586724876,\"prev\":\"299f4e2461ae29d61153d712a4d5dd42b19b84ebbe3c6b6ae0cd31a27ea9587c\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"laebshade\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"835dc147ade856af2320f4d40fda968bf19d45dd29934c84d5f47fc7462f9e440f\",\"sig_type\":2},{\"ctime\":1429044990,\"curr\":\"e6e5c443846efe1c1a2c8e453155cab11b0d758ec16ff78d80f10c2581bd8174\",\"etime\":1586724990,\"prev\":\"643678824683fc60026c2ac03344fdeacff1f6bf95dc02189caefd825cf07b3f\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"laebshade\"},\"proof_type\":4,\"state\":1},\"seqno\":3,\"sig_id\":\"072a030922dd777f4a640e0e61fb341a0995656adc500115ec2bb88c37d600640f\",\"sig_type\":2},{\"ctime\":1429045103,\"curr\":\"8dbfa743fd9ddf216390572927f980c34fc393dadcf6f72e6b25d49fb184d79a\",\"etime\":1586725103,\"prev\":\"e6e5c443846efe1c1a2c8e453155cab11b0d758ec16ff78d80f10c2581bd8174\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"markashields.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":4,\"sig_id\":\"1e176953f3cea0043172ae39900a5921145fdb745acf1e21835c4964c581da390f\",\"sig_type\":2},{\"ctime\":1429122182,\"curr\":\"cf5fceaa9c459258d4ffdbba3840047d38f166421049b88ddfdf11f7f18e7de3\",\"etime\":1586802182,\"prev\":\"2c32724fa9517a33521692796b830b5572dc6fbc51aa0a8ff8e6e6c9b32bf1d3\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"zzrbikes.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":15,\"sig_id\":\"a299f5a3b4a9af5c249c7c8bad66071b1d998eb466a5664e8af800d3426796b30f\",\"sig_type\":2},{\"ctime\":1429122394,\"curr\":\"c20dbded98b9cfb1585329c5889091312da3776d3bf33d2179e60cf88e9c6d18\",\"etime\":1586802394,\"prev\":\"cf5fceaa9c459258d4ffdbba3840047d38f166421049b88ddfdf11f7f18e7de3\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"gentoogeek.net\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":16,\"sig_id\":\"d7900625801bcd7c7c896a095e28f0e34473bae75e0c479019a5e75d4a7edb8c0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"6e3d94f68da91ddb895f1196d258fc552dfabfdcbb7141c1b267e82d6dd31cfc\",\"seqno\":21,\"sig_id\":\"5a778fe9f838ca35d90548bf9048305ba08cab2c3aa6ad92cda41439a27daf920f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1430419594,\"expire_in\":157680000,\"prev\":\"5cae5e8dc5d0cd6b233f12f6aeb03f45ca4c12b3fc2c337ed31112f0ebaba2a7\",\"seqno\":25,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1430419594,
            "etime": 1588099594,
            "rtime": null,
            "sig_status": 0,
            "prev": "5cae5e8dc5d0cd6b233f12f6aeb03f45ca4c12b3fc2c337ed31112f0ebaba2a7",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 26,
            "payload_hash": "b11dae23b92ba9d7b6100fb81bf201dfdcd406ccbf637876daf31f139d477af7",
            "sig_id": "b152e28cde4399080d248318b4dc10114bbc9388dcd38823bc227cb4365dcbba0f",
            "sig_id_short": "sVLijN5DmQgNJIMYtNwQEUu8k4jc04gjvCJ8",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMIYAnicbVJbSBVBGD6nLMlIuglSVLQRiJjNzuzM7p6KehHNwgsZFUk6szt7XE/u\nHs/FC2IvRVkhiRZkavbQ5Uk4kfgQSWaGZVoZIWRRFERB9ytdHmrW7K15Gf7v/75v\n5v/4h+bN9KX4qw+FKnry1kD/7Wsb477tmzetbpCYa9ZLgQYpxKcuy3aCPBKO2E5M\nCkhYo1DBJjI1BrEhY4sYHFmWpcqaShjRqKlo0ERQypIq3KinEDaMRnm27QpMFGW2\nKdD/8ENTDSADGXGAEYGEmhARjXBNhRZVMSMKxUwDSIZYVwHDBlWpylRT1wyki0KD\noi1TCKiwi0/ZKTKTLRVbDGOgEBOpgABNgZgiCDQZAI8Y5RGHVnHBdg3XcdwIIUhq\nzJIEXmMb3AvBm2WaU0mNkPt3mnDEjQnJXoFWxGLhgCeK1Yc9Vi1nZdP6MmY7pghR\nCGp4JGq7jhSQBdOI2Z6hrCCgEaQDPUvidWE7wstsj4FVogFxvGd4jbDUZQ4RkLGu\nIVNVOYFUIaqiAspNxjFBJiPQskyDM4NhRg0GIZO5yEOTRZ4qlryJqh1XCkAi/kmD\nwjNqBx0ai0e41Dg4UJrk86f4Zs+a4e2BL2XO/H/bcWZt8q/k0aazvcVXHzw/lVbP\nk6z1heXjG1/3XAq23xtZwBe/Dlf1o8lMZf9IZU5RNzh3rqBx8nzdxcnxHcu2skRT\na/rzxDZ+Q4//mDt8GOUMuvWjvuG7F1prC9xd5SeSattKF+aPL2n61Zm5YnIotahr\nS8fY3H3vd+c9q+56Uvq+s235omO9n/wZ2f5LX6qrKlPTmzsaLJyUeVbZkEj83vO4\nb+/xzx+D3/ISY63mztqhpQf6T968MvGzeVXBi4PSPN+75pKg8XD4dFrLy1t3r7Pi\ni28+Pr3/tq05oy233Yre6eteN1yXW5jW8ehV8EGB/P1y6srZR/q+ogk9P6174ENL\nyVF/zcv4H5rwJkk=\n=Wdqc\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"service\":{\"hostname\":\"jacko.io\",\"protocol\":\"http:\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1430863909,\"expire_in\":157680000,\"prev\":\"91e23015983d77e62a467470aedbe563db62ffdcebcb5bacb22b1e81a81e0575\",\"seqno\":26,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1430863909,
            "etime": 1588543909,
            "rtime": null,
            "sig_status": 0,
            "prev": "91e23015983d77e62a467470aedbe563db62ffdcebcb5bacb22b1e81a81e0575",
            "proof_id": "3195d2432b6853b6f3ceae10",
            "proof_type": 1000,
            "proof_text_check": "\n\nyMIYAnicbVJbSBVBGD6nLMlIuglSVLQRiJjNzuzM7p6KehHNwgsZFUk6szt7XE/u\nHs/FC2IvRVkhiRZkavbQ5Uk4kfgQSWaGZVoZIWRRFERB9ytdHmrW7K15Gf7v/75v\n5v/4h+bN9KX4qw+FKnry1kD/7Wsb477tmzetbpCYa9ZLgQYpxKcuy3aCPBKO2E5M\nCkhYo1DBJjI1BrEhY4sYHFmWpcqaShjRqKlo0ERQypIq3KinEDaMRnm27QpMFGW2\nKdD/8ENTDSADGXGAEYGEmhARjXBNhRZVMSMKxUwDSIZYVwHDBlWpylRT1wyki0KD\noi1TCKiwi0/ZKTKTLRVbDGOgEBOpgABNgZgiCDQZAI8Y5RGHVnHBdg3XcdwIIUhq\nzJIEXmMb3AvBm2WaU0mNkPt3mnDEjQnJXoFWxGLhgCeK1Yc9Vi1nZdP6MmY7pghR\nCGp4JGq7jhSQBdOI2Z6hrCCgEaQDPUvidWE7wstsj4FVogFxvGd4jbDUZQ4RkLGu\nIVNVOYFUIaqiAspNxjFBJiPQskyDM4NhRg0GIZO5yEOTRZ4qlryJqh1XCkAi/kmD\nwjNqBx0ai0e41Dg4UJrk86f4Zs+a4e2BL2XO/H/bcWZt8q/k0aazvcVXHzw/lVbP\nk6z1heXjG1/3XAq23xtZwBe/Dlf1o8lMZf9IZU5RNzh3rqBx8nzdxcnxHcu2skRT\na/rzxDZ+Q4//mDt8GOUMuvWjvuG7F1prC9xd5SeSattKF+aPL2n61Zm5YnIotahr\nS8fY3H3vd+c9q+56Uvq+s235omO9n/wZ2f5LX6qrKlPTmzsaLJyUeVbZkEj83vO4\nb+/xzx+D3/ISY63mztqhpQf6T968MvGzeVXBi4PSPN+75pKg8XD4dFrLy1t3r7Pi\ni28+Pr3/tq05oy233Yre6eteN1yXW5jW8ehV8EGB/P1y6srZR/q+ogk9P6174ENL\nyVF/zcv4H5rwJkk=\n",
            "proof_text_full": "==================================================================\nhttps://keybase.io/oconnor663\n--------------------------------------------------------------------\n\nI hereby claim:\n\n  * I am an admin of http://jacko.io\n  * I am oconnor663 (https://keybase.io/oconnor663) on keybase.\n  * I have a public key with fingerprint 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32\n\nTo do so, I am signing this object:\n\n{\n    \"body\": {\n        \"key\": {\n            \"fingerprint\": \"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"71876b68ad482d32\",\n            \"kid\": \"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\n            \"uid\": \"41b1f75fb55046d370608425a3208100\",\n            \"username\": \"oconnor663\"\n        },\n        \"service\": {\n            \"hostname\": \"jacko.io\",\n            \"protocol\": \"http:\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1430863909,\n    \"expire_in\": 157680000,\n    \"prev\": \"91e23015983d77e62a467470aedbe563db62ffdcebcb5bacb22b1e81a81e0575\",\n    \"seqno\": 26,\n    \"tag\": \"signature\"\n}\n\nwhich yields the signature:\n\n-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMIYAnicbVJbSBVBGD6nLMlIuglSVLQRiJjNzuzM7p6KehHNwgsZFUk6szt7XE/u\nHs/FC2IvRVkhiRZkavbQ5Uk4kfgQSWaGZVoZIWRRFERB9ytdHmrW7K15Gf7v/75v\n5v/4h+bN9KX4qw+FKnry1kD/7Wsb477tmzetbpCYa9ZLgQYpxKcuy3aCPBKO2E5M\nCkhYo1DBJjI1BrEhY4sYHFmWpcqaShjRqKlo0ERQypIq3KinEDaMRnm27QpMFGW2\nKdD/8ENTDSADGXGAEYGEmhARjXBNhRZVMSMKxUwDSIZYVwHDBlWpylRT1wyki0KD\noi1TCKiwi0/ZKTKTLRVbDGOgEBOpgABNgZgiCDQZAI8Y5RGHVnHBdg3XcdwIIUhq\nzJIEXmMb3AvBm2WaU0mNkPt3mnDEjQnJXoFWxGLhgCeK1Yc9Vi1nZdP6MmY7pghR\nCGp4JGq7jhSQBdOI2Z6hrCCgEaQDPUvidWE7wstsj4FVogFxvGd4jbDUZQ4RkLGu\nIVNVOYFUIaqiAspNxjFBJiPQskyDM4NhRg0GIZO5yEOTRZ4qlryJqh1XCkAi/kmD\nwjNqBx0ai0e41Dg4UJrk86f4Zs+a4e2BL2XO/H/bcWZt8q/k0aazvcVXHzw/lVbP\nk6z1heXjG1/3XAq23xtZwBe/Dlf1o8lMZf9IZU5RNzh3rqBx8nzdxcnxHcu2skRT\na/rzxDZ+Q4//mDt8GOUMuvWjvuG7F1prC9xd5SeSattKF+aPL2n61Zm5YnIotahr\nS8fY3H3vd+c9q+56Uvq+s235omO9n/wZ2f5LX6qrKlPTmzsaLJyUeVbZkEj83vO4\nb+/xzx+D3/ISY63mztqhpQf6T968MvGzeVXBi4PSPN+75pKg8XD4dFrLy1t3r7Pi\ni28+Pr3/tq05oy233Yre6eteN1yXW5jW8ehV8EGB/P1y6srZR/q+ogk9P6174ENL\nyVF/zcv4H5rwJkk=\n=Wdqc\n-----END PGP MESSAGE-----\n\nAnd finally, I am proving ownership of this host by posting or\nappending to this document.\n\nView my publicly-auditable identity here: https://keybase.io/oconnor663\n\n==================================================================\n",
            "check_data_json": "{\"hostname\":\"jacko.io\",\"protocol\":\"http:\"}",
            "remote_id": "http://jacko.io/keybase.txt",
            "api_url": "http://jacko.io/keybase.txt",
            "human_url": "http://jacko.io/keybase.txt",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 54,
            "hard_fail_count": 0,
            "last_check": 1434483168,
            "last_success": 1434483168,
            "version": 3,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 27,
            "payload_hash": "e6ed42cb01be8914b2fba11faffd431729fecdd21c2e1d1470c58167c7122aa7",
            "sig_id": "4e965585ac3389cc9c86937c5f632b43d8487d6e723767ad2b5c325ee54c1c9d0f",
            "sig_id_short": "TpZVhawzicychpN8X2MrQ9hIfW5yN2etK1wy",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMSBAnicpVVbiFVVGJ6ZrEwRS+khMJA9KhGjrvtlKlQs9MEefJgMHDmu65zdzDln\n3GePOshkQWmWhlBBOKFSGinYBaEkLe3CIBheMinIF/HBLhDB4IhE0L/PjJOTBUXn\n4bD3Xotv/d/l/9eX025rmtK8bmt3+dDy+aT51NDivqaOFb/HTYmt+f6kfVPietJQ\nzYunqqmEpD3pDv3W1MOCtDZ7Q7BJW7I+ZPW0VoUlaZHBTmuvPdJRBsIYCyYyh4nx\nXjiBiCJck2SgrYApQGNa7QpZb5YWZyRcGcK4p15Zwh3mUbhAY4wSKymsUMYzRTwl\ncGq5Vs8nVJM0MEupLwq5dX93YwFhhGlAnAoijCdUKBGUJNFIbgUz3CpEMZQokeXO\nSCOt9Fo5quEFSrcKG4IMwPU14Bi2OEoeLeeICU8lEkgxwg0lSGGEio31kI0pV3O1\narWWCUELAfLMuO5CAqg/dfXiKfWlcTFVW9Jj6jnwKbmyAZGSdsyIZFoLzSbAunpu\nKqZeYDaKotFRrAJn3PFIZQieORsZ08QTi7FOxsUv9JpoALEBU484CdoYLqQXHASQ\nGkfPkcScRautoXqCoN4zLanUoDdx2ESqrSeCcmIsZU5G54TFzhKpkPLOWWmNN5Z4\nGxSl1GIQFGrPQqWWh1JvVqtFUGM1RC9PK6O0mWZYYt2WuL4sg1MD9Q4zZkwwwjJv\nJTHM0qiRUZFaKTR1SoiChUA8KqM0/EUUvAKDCPJQfhgD54oIMgrem4X1Bbj0MTov\nIjFKGucM40KoQK3GgkqwGnsFoWDKhSLTwkXBhFbWaeYZ5jIZ51Lo2+DTaKRycN0l\nb3JTeqpeODzeUfmGNM9DlvyTqw2IUt7fC99JWwILeVE5rNTDumpt9GPaNRp9IqyT\nRdmQAkqc804xB3JzRznVRSy8CNCjKkYMvUkCKMWBLVdeRWs1isko2th5A21/NUIx\nOW6EpAgyCbGg1EeQTFPEFHJgPfc8BMExtQ4ZpzCI6hx3UkfoMeICfIP+u8WIBvgN\nI/6/y//NiK40L/fZf+UD/Tsf6E0+KEU9IZg6hwVynntHjYswHSWjlsnApbWYRKRl\nNIJ6Q3xwBjJkbJCEs3CLD2sax5Ryk/YUJfea/p6a8aWyqZeLMjnmhJBicKFAFPQ+\nwTCHifDOGEbAbCwt0YFrgbU2iBIpir5EHiEYKI3DRklgdhML75EN3sEmaYUxWHFP\nOJUQfqXBGh4dl4gq4EgRhx7ggkUYUF4gKwOsAIuBYtY1OIyNvJtujEK78XBRjJHS\nQkAkNvamWSilxQ4uhULwG08FjDBvAoF2JNZoD3XBoIVQYRsJzKII1TIE3GyEboVr\nwMNAwhFT7Rn0RZR/MiUQtdx0ASbwrZq8LwvJwOcnOic1NU9puuP2luIibJpy1903\nrsddlyc3Hc2PvHZo0pO7Hh68790l83df/PDq4CfTl+1bvuD64EPtC9+Y9cVnG195\nYX3p9JbD01fs/PHjhXPC8DvXWprQsbeHD1y89s3rW66++N3c7Zc2Pznv5NyfRp5/\necbF62d3Ht043Xz/wDyX7DjwcysZ7pzzyNBQfuGDziu8ZWttaceB90/MvLr5rXrH\nqpHz5d2Ht9/5no1+z7ODi1/t2rF8w7Gnh2trW/esWXZ571dfzzi9YNuhXedas3Pm\n1Jv79u5dOXjP/tVXZs6iU1s/HT737TMvtaw4v+S3k4ua+2dPe+Lgc7pemfzRmUsj\nq6ZWlp699/5lj47MGl7ZcfrBypGDj+/Xj/3w66m127Yt2nJ850jX8V8uDJ2JnX8A\nhtqkhQ==\n=aIyH\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"7b0a1c99d9d09f7e2444eaf4c12add6c60282592\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":8,\"last_id_change\":1427499694,\"username\":\"cstamas\"},\"id\":\"3fc318e545c5f37eed4cbf4492d2b119\",\"key\":{\"key_fingerprint\":\"2be13d052e9aa567d657d9791fd507154fb9ba39\",\"kid\":\"0101dd49737968a2c1af39bd26352ab34c7fcc6b1cb27808dccb7badab2dbe8333b10a\"},\"remote_proofs\":[{\"ctime\":1424941719,\"curr\":\"e3dc144aaea6b4db72a4b3f90a8f3b7693c86652e9605f8a89f8af0ed806020d\",\"etime\":1582621719,\"prev\":\"e7dffcd6f2a87acca45668e3b91637b551d8d9848ced6c66cf64698bc94d4157\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"cstamas\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"26bc77accfc332ccdc84cdab5c35398e54d6ec998ff1eaf2ec865cd658d8fbb90f\",\"sig_type\":2},{\"ctime\":1424941847,\"curr\":\"73027454f33dfd6f930480c52a5d5ee6513bc0ac817dfcc5c79fb802ceac8d23\",\"etime\":1582621847,\"prev\":\"e3dc144aaea6b4db72a4b3f90a8f3b7693c86652e9605f8a89f8af0ed806020d\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"cstamas\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"883d2213cc160cd5dc3acf1c9743b47e57bb12f097fa63da2decabc9abe7254e0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"c515222970b0e2845c219d926dcaa42ff117b29e596199a03276cc6b0d00694f\",\"seqno\":14,\"sig_id\":\"dd0bedcd007b6aa185d2537848891445fc57038c1630598b564f969d60b7e7030f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1431108966,\"expire_in\":157680000,\"prev\":\"b11dae23b92ba9d7b6100fb81bf201dfdcd406ccbf637876daf31f139d477af7\",\"seqno\":27,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1431108966,
            "etime": 1588788966,
            "rtime": null,
            "sig_status": 0,
            "prev": "b11dae23b92ba9d7b6100fb81bf201dfdcd406ccbf637876daf31f139d477af7",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 28,
            "payload_hash": "937fff13c30312edfe384a04f09b8714fc72f12e22ead9c30811064528a22afe",
            "sig_id": "a912d4392c93a36cd89d9136839aa339a6df9f37156a18f2dc8ed46213b055520f",
            "sig_id_short": "qRLUOSyTo2zYnZE2g5qjOabfnzcVahjy3I7U",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n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\nL3ztrpN3PrC04yN/Ov/MfwCug/Aw\n=AtX0\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"f41a7d6fbb57b25bbfc781a4496dce572f77e86e\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":15,\"last_id_change\":1430852617,\"username\":\"mitsuhiko\"},\"id\":\"67aaabb56ac60890936d56f7347ccb00\",\"key\":{\"key_fingerprint\":\"d0c6a9995649e5b407483e541d8fbe76fdd3e6f2\",\"kid\":\"0101587b8827659e835fe5a9a53f70866a0dcc8634c574452b799cec2dcd68e788e90a\"},\"remote_proofs\":[{\"ctime\":1404156355,\"curr\":\"2e611ebd672c8d2a2192ace994e0c68b8c897c815e187dd7a01a148ae870f628\",\"etime\":1561836355,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"mitsuhiko\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"86d652c363de52c28013fab6fb8f69a1880f60cda87963921cc9061aa333b20b0f\",\"sig_type\":2},{\"ctime\":1404156413,\"curr\":\"c4b72d2ddd4bc9b9d61bf5b85b867670ab0c09541b2638dd02d8abdae0b683d7\",\"etime\":1561836413,\"prev\":\"2e611ebd672c8d2a2192ace994e0c68b8c897c815e187dd7a01a148ae870f628\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"lucumr.pocoo.org\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":2,\"sig_id\":\"b5eed595d035c7df78c4c41b64258f6e4dce60a8f5ce65ec5b34d3ca95d7053e0f\",\"sig_type\":2},{\"ctime\":1404156497,\"curr\":\"27f5ab0f489054c81fe81f107151a551ea11407751b3200cc73b00cfc53694ab\",\"etime\":1561836497,\"prev\":\"c4b72d2ddd4bc9b9d61bf5b85b867670ab0c09541b2638dd02d8abdae0b683d7\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"mitsuhiko\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"ced4fccf0fb74fc4b5146e4839af435231a4f3460de7b220d6df6cd39e1053eb0f\",\"sig_type\":2},{\"ctime\":1430852524,\"curr\":\"31e4fa83b6ffb964173e79b5f5d3d0e6ec746c8d51504d4ad89e2fa8e9131633\",\"etime\":1588532524,\"prev\":\"f25be273916f6e96f5c25682a26ec80a7909be0690655d6b3edccf16bf222a21\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"mitsuhiko\"},\"proof_type\":4,\"state\":1},\"seqno\":5,\"sig_id\":\"8c5c690c601e42ec0b3e185f228af08178a9591fea3b1a385df8730ff0b1a9110f\",\"sig_type\":2},{\"ctime\":1430852571,\"curr\":\"8b7528d662cb7d89883146a0fa13af9e630b81732516d7e163e75ad3d5c295e3\",\"etime\":1588532571,\"prev\":\"31e4fa83b6ffb964173e79b5f5d3d0e6ec746c8d51504d4ad89e2fa8e9131633\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"the_mitsuhiko\"},\"proof_type\":6,\"state\":1},\"seqno\":6,\"sig_id\":\"76ea910f1781cbded6ae954cbcb250a15417c7845a23a8d7d3f83cffcf53b56c0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"8b7528d662cb7d89883146a0fa13af9e630b81732516d7e163e75ad3d5c295e3\",\"seqno\":6,\"sig_id\":\"76ea910f1781cbded6ae954cbcb250a15417c7845a23a8d7d3f83cffcf53b56c0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1433274004,\"expire_in\":157680000,\"prev\":\"e6ed42cb01be8914b2fba11faffd431729fecdd21c2e1d1470c58167c7122aa7\",\"seqno\":28,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1433274004,
            "etime": 1590954004,
            "rtime": null,
            "sig_status": 0,
            "prev": "e6ed42cb01be8914b2fba11faffd431729fecdd21c2e1d1470c58167c7122aa7",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 29,
            "payload_hash": "9efc3ff3f6515f5c720b660bb8f705cbc242c4859e666d965243ed99ad7aedb1",
            "sig_id": "234f77b28a058b7625cbd73ebe28e2c09a4f5780f08657cec0e6a71ccabad1520f",
            "sig_id_short": "I093sooFi3Yly9c-vijiwJpPV4DwhlfOwOan",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMORAnictVRdiFVVFL6m5jhlTkQZMqKdMkNG279n7zNTyBRDgRGCpmnFdf/OPd6Z\nc2bOOTPjKBNEUUiSBkGU+NCv9TAyPRSiEjqZKA1kTyERGYj2o/MQgZIErXOdbAZ6\nrPNw2Peudb71fd9ae51YMLvSPKv/lXpt9PFVZNZXJ4qBylPbn35wZ6BTOxy07wxM\nT+ySojwlqtcF7UHdDWuVu9VxumzI6aAtGHRZHqcJhDzDStjQa82FJlxrb4TEirEo\ntMZxQbwQToYuGGkrYUpQHyfdLuvL4rJGwKUijFtqJXxuMPehcdR7L7AUoQ6lskwS\nSwlUraV5MYNN0MCsxhb+/Zf8eiOAMMLUIU5DEipLaAhsJPBSguuQKa4lopjwSCDN\njRJKaGEjaWgEPyQoAjUEKYAbaMAxrLEX3GvOEQstFShEkhGuKEESI1Qm5i6bci41\naZKkWRjS0oAiU6ZeWgD8Y5OXp9hWb5hJ24IelRegp2pqCkwK2jGjNGKERmQGbHec\n5LU4KTEbpELlOBLMIQRCkKPYUOGItT7iTkkcBTfML/2a2QDCpPVEIu18pIlVhkUO\nK2NDRDAlBlqhCObIzjDUERcKB1k4dFoxbOCEQ2IElcJFmFsLL6oZEVIjxzXDTtsQ\nHHdYS2YFGArcM9ebFq7al6WpBzeegdEr4t5pslnYFpiBLIOqEkmpHGhhWlEUychS\nwLfQMG0k48TKslU8ohHHJb6JvBZcCIJdGCopgb6bAucRUL0OPsWgdKXBojH+NWfq\nVasKVd2Wl325cQ+KobgoXDazxd0qrrskiYfTTO0YKGU1kKrFcB+EoW95oYqyLERy\n15+kcIRT3H19biVSAkYtEswbGEgqBFcMVCAvpbPGG+QsxUwoyrDHBlzxFFPLhLRY\nIIuQD66jTdUbea5RplqouKek3qeGe1JlqzWV1/4bG/8fESPl/WhImLom07ZMad30\nyeAw5hz6ub0vzlw1LjO4CCWCp3TfDQJIRAWMLqaGltfbWe+oZAoxjyItBQaisAMg\nQIhTNoIsiTEKwQHYR0R5949QEgEz1Q2YoDdRxUAG22z82LNzKrOaKzfPvalcnpXm\n+S1/r9QncFPl5OqJg6eOPd/54S0X1nYt/rL96KKW8fM/n9nwOVn85Lyuw3Gw7sij\nh9X5Vzv2T34698+fks6Jesuauw+8u3LN7vtHO3cfWHR21+/37l12rfXIqjP9nc0L\nJ17Yt/SNP35sb58vjnUtObt29Nal9JHDJ+v6vb2tmy9/e6WjFR+qNB0fX7fhwtqL\ndxRXP3n/t/1fn/6m5dTGhRfH4i9ejza9s/7jK6uHBt7a80B+9fgP62f7cxuTdHLH\noYe2nFfff9ax5ZeuXx+bvWvf8tvnH73z4MrxPcGC7+5qbRn56J5zm5dc3pSNdaxJ\nx15bsWLxbQdeWt7U2zK2dXLe2y+a0W3L77tUXHsz33r65YfHF3WNTFwcvDTnA/oX\nDIMM3w==\n=uNE8\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"f41a7d6fbb57b25bbfc781a4496dce572f77e86e\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":3,\"last_id_change\":1433942392,\"username\":\"ginshin\"},\"id\":\"6ae5074e002590e31c37e2ddf95ea819\",\"key\":{\"key_fingerprint\":\"248df280bef9b2dac49e1acd602132cfffa2150d\",\"kid\":\"0101e2e67eacd16eba41ccd1162c7387e915dd9153b4278b0e5b41ebd675be1b84d70a\"},\"remote_proofs\":[{\"ctime\":1433942346,\"curr\":\"8088ae8194ba30989d33b4da7abc8452d85b81593951d70ac9fb757721e66a88\",\"etime\":1591622346,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"gaikenniyorazu\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"80a79ca974fc5b63775a49390f88edcfc0ed3147a341f1c942f313d478d170d00f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"8088ae8194ba30989d33b4da7abc8452d85b81593951d70ac9fb757721e66a88\",\"seqno\":1,\"sig_id\":\"80a79ca974fc5b63775a49390f88edcfc0ed3147a341f1c942f313d478d170d00f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1433950255,\"expire_in\":157680000,\"prev\":\"937fff13c30312edfe384a04f09b8714fc72f12e22ead9c30811064528a22afe\",\"seqno\":29,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1433950255,
            "etime": 1591630255,
            "rtime": null,
            "sig_status": 0,
            "prev": "937fff13c30312edfe384a04f09b8714fc72f12e22ead9c30811064528a22afe",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        }
    ],
    "keys": [
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v2\n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\n8PZq\n=Bh81\n-----END PGP PUBLIC KEY BLOCK-----"
    ]
}
`,
	"jack_chain_substitute_first_link.json": `{
  "username": "oconnor663",
  "uid": "41b1f75fb55046d370608425a3208100",
  "chain": [
    {
      "seqno": 1,
      "prev": null,
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.12\nComment: https://keybase.io/crypto\n\nyMDuAnicbZE9aBRBGIb3/MUT4eCwsJ16Ob7Z+du9wtJGMJWNhcfs7sxlyGVm3Z9w\n4YgKgdQGSyuLdGniD4iCEREiiYUIKtZCSsHaIs5usHOame/93veZGb4PV84G/eGD\nF73nx0dbJ72jd4dNcPvhk+0FSl2+jsYLtKK6TRs7VWVRGlujMeIJUEgE1SIjSnMK\nWY5B4iwjguGEyDjKOeeZRCFadlWb8JhUVmpknNd8MTG5V//jX+kagDEBSTRWkRSK\nUppEWoqMx4RQYJIBj9OEYB/P0wiAiUSAhwiCVZxqRrTINbS4psNRnGItmE4ZA8pz\nIoBDTCMmSQQxBmiNlSqtXFXe7TJnrSs5J2gjRPV60Ypqliv/kxCtqbIyzqIx9s2s\nNm0EU+KfiEUShUjNC1OqiWkdcLpCVJRqDY1tM5uFqFL3JqdQ3BXWdadaTv01lZla\nWTelQhvv754Lhv3gwvkz7USC/qXBvzm9OegF33fm8ku8c/9ls7/6+BHSP+bR8e7h\nZ/ZscOvP22HxtRd8PBmEvzefflvc1L9u3GGj0dWly+5g//XPveWL1699evUXZpSl\nkw==\n=H94g\n-----END PGP MESSAGE-----\n",
      "payload_hash": "490aa72c0676474061d7519d83e5e57535d656431aeb8010f81b1b6ef8152038",
      "sig_id": "e4948fe86eb6cee5b986aba51270e584960eb45229a48207e7ad6220e3af366b0f",
      "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"69040974f7c3ef640cd10a1cc375193a82d666ca\",\"host\":\"keybase.io\",\"key_id\":\"c375193a82d666ca\",\"kid\":\"01130a3f1e2a7e44492fa7c6833405a5068b931375db2005797082d731e8bf53f7df0a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1434491792,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
      "kid": "01130a3f1e2a7e44492fa7c6833405a5068b931375db2005797082d731e8bf53f7df0a",
      "ctime": 1434491792
    },
    {
      "seqno": 2,
      "payload_hash": "382e1d68b4b81c6643483a74715ac0964e291fb2b52c4e9fa6c73d304ab9108f",
      "sig_id": "9d5b60c0d486f34089a922bcc5e77fb643b9781cbcc894af7e123fad50645d6a0f",
      "sig_id_short": "nVtgwNSG80CJqSK8xed_tkO5eBy8yJSvfhI_",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kltoFFcYx3et1VWrWBQJsaCdFq3tusy5zhn1QYOUVh9MglQq1vWcM+dkJ5eZ\ndWaTmNqIWmlTrDRtpaLSghdaIqYS6614wbsWry/6oEVoi6AvkgRFVHA9s1iqL52n\nme/8vv/3/893uke/ksqkl3/ZVOj9YDpMX7grUgs3LR610hKh12HNWGnJZl8FpeQt\n4C3KmmE1qQ7BY5Xzw8lB6KlcYzz5OZO12lQU+2FgKDvn5JDVmU3wpFn7QYOKipGf\naFmEcYiJhzwmIJGAaCoV0lo7gDlUUMY9zKCHoJEshHHppalWRTPve6Za4WsomzPX\n8HMrfGvlAAMBtEO0IMTG1EOOTW2GIeEI2gzYdgLGKnoeKZRhEIQRpRXHLSpqalb5\nKAwrsWXJTyCAgQttYiPHeOJxwbR51HGl4EBShRHChAosJQEMONz2kKnbLuFcJY1A\nEQKAhK459Lh0lJBcciopcIiHkbax4Awg1xFMCcQEEwQJ6EmICMXCVo4rHMk5ocYC\n0tyMMTNNiFgtD0JjzrjDbmfyHbX5Ur2wrga/VGgV/5e31FFMiu1K5A2yK+nPCz/w\nzMZeXCkw6Mv/AiCQtdSKoh+pvJ8QxKHMNk/WKkaqzWgKDIFARECkJfE449TVXJiY\nTGuEKXO1CaVtGzMtGdcMQm42JAiAQDLgov8SQuOTNxjJ2G8IeKk1UlbnV+k3hqbS\nmdSwV4ckdzY1csTYfy9yqjeTOji8+KB672c9PasmiqPhVFS+N38n+e2cqvqkdikt\nrC4e7e//R7w57/j8R69lahr3/Tx4/mI0ru/bqku98Cy41f/37acXa8+sv5a7scb9\n/tjA/m/WvfPk5M3amikb3qvecvvWYOav4VtP73n9u3dnLdzx9YGldX0f7uiCi7qf\n/vrWD79kV/fdf7v2fPHwPae+ceKPB2bWb/+U7qsfv/Exm3R/wF3c/WTtwZ5j8wZ3\nnZnQdm7KF3WNh6ovjKH4Sv+GzPvTRi+b+cd6sK5c9/Dy6SHtH338aFthSbpqQupU\nze/l9s/317XUXyuvmj30z40DW6ubx95Zc6hq77jC7gXl+PD18XPOnuw6Me2nI11X\nN496Bg==\n=diQc\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419205037,\"hash\":\"d679cba1c6e433456b4cc51817a0d3a1c095aae41921e5511c29817dac7ebcaca6c6175d43f04ba81397b8eb38b8b53b2dc23564b0e79b7caa562053facc56e4\",\"seqno\":114149},\"service\":{\"name\":\"github\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205131,\"expire_in\":157680000,\"prev\":\"b421b35b23fc5da8a69fab3978ff34689fcaaf0048fc8af822aa32b5121c8193\",\"seqno\":2,\"tag\":\"signature\"}",
      "sig_type": 2,
      "ctime": 1419205131,
      "etime": 1576885131,
      "rtime": null,
      "sig_status": 0,
      "prev": "b421b35b23fc5da8a69fab3978ff34689fcaaf0048fc8af822aa32b5121c8193",
      "proof_id": "034487589bf1339ef37e6210",
      "proof_type": 3,
      "proof_text_check": "\n\nowF9kltoFFcYx3et1VWrWBQJsaCdFq3tusy5zhn1QYOUVh9MglQq1vWcM+dkJ5eZ\ndWaTmNqIWmlTrDRtpaLSghdaIqYS6614wbsWry/6oEVoi6AvkgRFVHA9s1iqL52n\nme/8vv/3/893uke/ksqkl3/ZVOj9YDpMX7grUgs3LR610hKh12HNWGnJZl8FpeQt\n4C3KmmE1qQ7BY5Xzw8lB6KlcYzz5OZO12lQU+2FgKDvn5JDVmU3wpFn7QYOKipGf\naFmEcYiJhzwmIJGAaCoV0lo7gDlUUMY9zKCHoJEshHHppalWRTPve6Za4WsomzPX\n8HMrfGvlAAMBtEO0IMTG1EOOTW2GIeEI2gzYdgLGKnoeKZRhEIQRpRXHLSpqalb5\nKAwrsWXJTyCAgQttYiPHeOJxwbR51HGl4EBShRHChAosJQEMONz2kKnbLuFcJY1A\nEQKAhK459Lh0lJBcciopcIiHkbax4Awg1xFMCcQEEwQJ6EmICMXCVo4rHMk5ocYC\n0tyMMTNNiFgtD0JjzrjDbmfyHbX5Ur2wrga/VGgV/5e31FFMiu1K5A2yK+nPCz/w\nzMZeXCkw6Mv/AiCQtdSKoh+pvJ8QxKHMNk/WKkaqzWgKDIFARECkJfE449TVXJiY\nTGuEKXO1CaVtGzMtGdcMQm42JAiAQDLgov8SQuOTNxjJ2G8IeKk1UlbnV+k3hqbS\nmdSwV4ckdzY1csTYfy9yqjeTOji8+KB672c9PasmiqPhVFS+N38n+e2cqvqkdikt\nrC4e7e//R7w57/j8R69lahr3/Tx4/mI0ru/bqku98Cy41f/37acXa8+sv5a7scb9\n/tjA/m/WvfPk5M3amikb3qvecvvWYOav4VtP73n9u3dnLdzx9YGldX0f7uiCi7qf\n/vrWD79kV/fdf7v2fPHwPae+ceKPB2bWb/+U7qsfv/Exm3R/wF3c/WTtwZ5j8wZ3\nnZnQdm7KF3WNh6ovjKH4Sv+GzPvTRi+b+cd6sK5c9/Dy6SHtH338aFthSbpqQupU\nze/l9s/317XUXyuvmj30z40DW6ubx95Zc6hq77jC7gXl+PD18XPOnuw6Me2nI11X\nN496Bg==\n",
      "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am oconnor663 on github.\n  * I am oconnor663 (https://keybase.io/oconnor663) on keybase.\n  * I have a public key whose fingerprint is 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32\n\nTo claim this, I am signing this object:\n\n` + "`" + `` + "`" + `` + "`" + `json\n{\n    \"body\": {\n        \"client\": {\n            \"name\": \"keybase.io node.js client\",\n            \"version\": \"0.7.3\"\n        },\n        \"key\": {\n            \"fingerprint\": \"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"71876B68AD482D32\",\n            \"uid\": \"41b1f75fb55046d370608425a3208100\",\n            \"username\": \"oconnor663\"\n        },\n        \"merkle_root\": {\n            \"ctime\": 1419205037,\n            \"hash\": \"d679cba1c6e433456b4cc51817a0d3a1c095aae41921e5511c29817dac7ebcaca6c6175d43f04ba81397b8eb38b8b53b2dc23564b0e79b7caa562053facc56e4\",\n            \"seqno\": 114149\n        },\n        \"service\": {\n            \"name\": \"github\",\n            \"username\": \"oconnor663\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1419205131,\n    \"expire_in\": 157680000,\n    \"prev\": \"b421b35b23fc5da8a69fab3978ff34689fcaaf0048fc8af822aa32b5121c8193\",\n    \"seqno\": 2,\n    \"tag\": \"signature\"\n}\n` + "`" + `` + "`" + `` + "`" + `\n\nwith the PGP key whose fingerprint is\n[58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32](https://keybase.io/oconnor663)\n(captured above as ` + "`" + `body.key.fingerprint` + "`" + `), yielding the PGP signature:\n\n` + "`" + `` + "`" + `` + "`" + `\n-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kltoFFcYx3et1VWrWBQJsaCdFq3tusy5zhn1QYOUVh9MglQq1vWcM+dkJ5eZ\ndWaTmNqIWmlTrDRtpaLSghdaIqYS6614wbsWry/6oEVoi6AvkgRFVHA9s1iqL52n\nme/8vv/3/893uke/ksqkl3/ZVOj9YDpMX7grUgs3LR610hKh12HNWGnJZl8FpeQt\n4C3KmmE1qQ7BY5Xzw8lB6KlcYzz5OZO12lQU+2FgKDvn5JDVmU3wpFn7QYOKipGf\naFmEcYiJhzwmIJGAaCoV0lo7gDlUUMY9zKCHoJEshHHppalWRTPve6Za4WsomzPX\n8HMrfGvlAAMBtEO0IMTG1EOOTW2GIeEI2gzYdgLGKnoeKZRhEIQRpRXHLSpqalb5\nKAwrsWXJTyCAgQttYiPHeOJxwbR51HGl4EBShRHChAosJQEMONz2kKnbLuFcJY1A\nEQKAhK459Lh0lJBcciopcIiHkbax4Awg1xFMCcQEEwQJ6EmICMXCVo4rHMk5ocYC\n0tyMMTNNiFgtD0JjzrjDbmfyHbX5Ur2wrga/VGgV/5e31FFMiu1K5A2yK+nPCz/w\nzMZeXCkw6Mv/AiCQtdSKoh+pvJ8QxKHMNk/WKkaqzWgKDIFARECkJfE449TVXJiY\nTGuEKXO1CaVtGzMtGdcMQm42JAiAQDLgov8SQuOTNxjJ2G8IeKk1UlbnV+k3hqbS\nmdSwV4ckdzY1csTYfy9yqjeTOji8+KB672c9PasmiqPhVFS+N38n+e2cqvqkdikt\nrC4e7e//R7w57/j8R69lahr3/Tx4/mI0ru/bqku98Cy41f/37acXa8+sv5a7scb9\n/tjA/m/WvfPk5M3amikb3qvecvvWYOav4VtP73n9u3dnLdzx9YGldX0f7uiCi7qf\n/vrWD79kV/fdf7v2fPHwPae+ceKPB2bWb/+U7qsfv/Exm3R/wF3c/WTtwZ5j8wZ3\nnZnQdm7KF3WNh6ovjKH4Sv+GzPvTRi+b+cd6sK5c9/Dy6SHtH338aFthSbpqQupU\nze/l9s/317XUXyuvmj30z40DW6ubx95Zc6hq77jC7gXl+PD18XPOnuw6Me2nI11X\nN496Bg==\n=diQc\n-----END PGP MESSAGE-----\n\n` + "`" + `` + "`" + `` + "`" + `\n\nAnd finally, I am proving ownership of the github account by posting this as a gist.\n\n### My publicly-auditable identity:\n\nhttps://keybase.io/oconnor663\n\n### From the command line:\n\nConsider the [keybase command line program](https://keybase.io/docs/command_line).\n\n` + "`" + `` + "`" + `` + "`" + `bash\n# look me up\nkeybase id oconnor663\n\n# encrypt a message to me\nkeybase encrypt oconnor663 -m 'a secret message...'\n\n# ...and more...\n` + "`" + `` + "`" + `` + "`" + `\n",
      "check_data_json": "{\"name\":\"github\",\"username\":\"oconnor663\"}",
      "remote_id": "df527ad8438a424efcec",
      "api_url": "https://gist.githubusercontent.com/oconnor663/df527ad8438a424efcec/raw/0c24241710275124f07db83ab5b6f97fc6952165/keybase.md",
      "human_url": "https://gist.github.com/df527ad8438a424efcec",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 142,
      "hard_fail_count": 0,
      "last_check": 1434467728,
      "last_success": 1434467728,
      "version": 3,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 3,
      "payload_hash": "44a567581a31b596f31745c4633feced0806f427bac69e720aecfda02279b9bb",
      "sig_id": "a5039b256eb381b395e2fe9018e931c7dbe052dc1f1cf213b9e8e930ef0feabf0f",
      "sig_id_short": "pQObJW6zgbOV4v6QGOkxx9vgUtwfHPITuejp",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kltoHUUcxk9sm9ZqSIsXfIgStwUpTU7mPrMnDdUawWiIgkIxUk/mtidrkt2T\n3ZOYGKMFi02EqiCaFvom7dFqHqrRqOhDQdBYifqgra3QSERaxQuitUorzh4qti/O\n01x+/2++b/7zXMOK3Jq64d0D/bN3tqK6o2dU7v6Z3hcnPBWbca8w4enB0EaVbBbJ\nIesVvAE7rmRq82HcHMXG5h9Omy8yLd6oTdIwjhwF8jyPvcmWDM+KgzAq2aSchJmW\nR4VEhBpshEJUQxowbXEQBBwKzhQT0hCBDEZOsj9OK5fd6tU0i6FxuzV+GxO3dTq+\ns8aP1A4IVDDgNFCUAsIM5oABQRCVGAEBAcjA1CYXI8U6jqI4YazmeMgmA4O2mMRx\nLbauhBkECfQRoJAR50mm/a5MCetTbQgEkkAmsG8UVr6EPsYKSYWg70NEEUOMBZZw\nHViojUvGCIIKcQmUIpJBjQx3qSH2XQjOKIQ+0BhA3z2L5BAjoxhWGLoCLgjnwAgI\nEcbEhUjtcBQ7c84dRZPZOhkNtb2kXYk1Jqz8X97KeDnbfMSqokMOZfVFFUbGdezS\nlkKHXv4WiIMWz46Vw8QWw4ygnAngRotXTuyo08QCWWiYUEQJqBkjmAgsOeGQSg18\nRizyYaCQokgT6weSaY4NBkQqHwIR/JcQO5+y5CTTsBTJykhivcnpuqaVubo1ufpV\nV2R/Nrf2ynX/fuSNJ1b/vaF1+be0Mdf1Tld0w9lVi3ubCu/KRXnhlpOf98z1jd13\nCOyZOn3q+lj90Lc4N/P2L4WP83eLXYdv36zr35w+cu1rbe9tuKP86plkHiwffKvh\ncPT6tvOn/Zd3/rV759LWA6a1c/aJ6vvb0dahPY+dO9n42UfV/cvtv3+3aWFkxYmk\ntGVmqffPVx7atfR0Kej+dKHvp59fWMnEV6ee2etddWx9YcfqWx+/50hPd1fV7mi/\naazj7Lpvp+YXrut444NfG/ddaKebJo43w3T25gce7H6pybtmcbrx+z+eXf/j5vPD\nc0+Ob+FTg4/e9c3XN66tHudH91ePbU8+vHqk3FD/VFvH/L2fHKx/vld/ea4t/uIf\n=i2ny\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419205164,\"hash\":\"b8e95cd410a416839db3b9a1933b2ab219912526266fe47cfe1cdd486421b27a0bb4a61c2d776b1394827651190c30195f6a7132db63b317a0784770d8112334\",\"seqno\":114152},\"service\":{\"name\":\"reddit\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205270,\"expire_in\":157680000,\"prev\":\"382e1d68b4b81c6643483a74715ac0964e291fb2b52c4e9fa6c73d304ab9108f\",\"seqno\":3,\"tag\":\"signature\"}",
      "sig_type": 2,
      "ctime": 1419205270,
      "etime": 1576885270,
      "rtime": null,
      "sig_status": 0,
      "prev": "382e1d68b4b81c6643483a74715ac0964e291fb2b52c4e9fa6c73d304ab9108f",
      "proof_id": "7374183938ae0741e08d3c10",
      "proof_type": 4,
      "proof_text_check": "\n\nowF9kltoHUUcxk9sm9ZqSIsXfIgStwUpTU7mPrMnDdUawWiIgkIxUk/mtidrkt2T\n3ZOYGKMFi02EqiCaFvom7dFqHqrRqOhDQdBYifqgra3QSERaxQuitUorzh4qti/O\n01x+/2++b/7zXMOK3Jq64d0D/bN3tqK6o2dU7v6Z3hcnPBWbca8w4enB0EaVbBbJ\nIesVvAE7rmRq82HcHMXG5h9Omy8yLd6oTdIwjhwF8jyPvcmWDM+KgzAq2aSchJmW\nR4VEhBpshEJUQxowbXEQBBwKzhQT0hCBDEZOsj9OK5fd6tU0i6FxuzV+GxO3dTq+\ns8aP1A4IVDDgNFCUAsIM5oABQRCVGAEBAcjA1CYXI8U6jqI4YazmeMgmA4O2mMRx\nLbauhBkECfQRoJAR50mm/a5MCetTbQgEkkAmsG8UVr6EPsYKSYWg70NEEUOMBZZw\nHViojUvGCIIKcQmUIpJBjQx3qSH2XQjOKIQ+0BhA3z2L5BAjoxhWGLoCLgjnwAgI\nEcbEhUjtcBQ7c84dRZPZOhkNtb2kXYk1Jqz8X97KeDnbfMSqokMOZfVFFUbGdezS\nlkKHXv4WiIMWz46Vw8QWw4ygnAngRotXTuyo08QCWWiYUEQJqBkjmAgsOeGQSg18\nRizyYaCQokgT6weSaY4NBkQqHwIR/JcQO5+y5CTTsBTJykhivcnpuqaVubo1ufpV\nV2R/Nrf2ynX/fuSNJ1b/vaF1+be0Mdf1Tld0w9lVi3ubCu/KRXnhlpOf98z1jd13\nCOyZOn3q+lj90Lc4N/P2L4WP83eLXYdv36zr35w+cu1rbe9tuKP86plkHiwffKvh\ncPT6tvOn/Zd3/rV759LWA6a1c/aJ6vvb0dahPY+dO9n42UfV/cvtv3+3aWFkxYmk\ntGVmqffPVx7atfR0Kej+dKHvp59fWMnEV6ee2etddWx9YcfqWx+/50hPd1fV7mi/\naazj7Lpvp+YXrut444NfG/ddaKebJo43w3T25gce7H6pybtmcbrx+z+eXf/j5vPD\nc0+Ob+FTg4/e9c3XN66tHudH91ePbU8+vHqk3FD/VFvH/L2fHKx/vld/ea4t/uIf\n",
      "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am [oconnor663](https://www.reddit.com/user/oconnor663) on reddit.\n  * I am [oconnor663](https://keybase.io/oconnor663) on keybase.\n  * I have a public key whose fingerprint is 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32\n\nTo claim this, I am signing this object:\n\n\n    {\n        \"body\": {\n            \"client\": {\n                \"name\": \"keybase.io node.js client\",\n                \"version\": \"0.7.3\"\n            },\n            \"key\": {\n                \"fingerprint\": \"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\n                \"host\": \"keybase.io\",\n                \"key_id\": \"71876B68AD482D32\",\n                \"uid\": \"41b1f75fb55046d370608425a3208100\",\n                \"username\": \"oconnor663\"\n            },\n            \"merkle_root\": {\n                \"ctime\": 1419205164,\n                \"hash\": \"b8e95cd410a416839db3b9a1933b2ab219912526266fe47cfe1cdd486421b27a0bb4a61c2d776b1394827651190c30195f6a7132db63b317a0784770d8112334\",\n                \"seqno\": 114152\n            },\n            \"service\": {\n                \"name\": \"reddit\",\n                \"username\": \"oconnor663\"\n            },\n            \"type\": \"web_service_binding\",\n            \"version\": 1\n        },\n        \"ctime\": 1419205270,\n        \"expire_in\": 157680000,\n        \"prev\": \"382e1d68b4b81c6643483a74715ac0964e291fb2b52c4e9fa6c73d304ab9108f\",\n        \"seqno\": 3,\n        \"tag\": \"signature\"\n    }\n\nwith the PGP key referenced above, yielding the PGP signature:\n\n\n    -----BEGIN PGP MESSAGE-----\n    Version: GnuPG v2\n    \n    owF9kltoHUUcxk9sm9ZqSIsXfIgStwUpTU7mPrMnDdUawWiIgkIxUk/mtidrkt2T\n    3ZOYGKMFi02EqiCaFvom7dFqHqrRqOhDQdBYifqgra3QSERaxQuitUorzh4qti/O\n    01x+/2++b/7zXMOK3Jq64d0D/bN3tqK6o2dU7v6Z3hcnPBWbca8w4enB0EaVbBbJ\n    IesVvAE7rmRq82HcHMXG5h9Omy8yLd6oTdIwjhwF8jyPvcmWDM+KgzAq2aSchJmW\n    R4VEhBpshEJUQxowbXEQBBwKzhQT0hCBDEZOsj9OK5fd6tU0i6FxuzV+GxO3dTq+\n    s8aP1A4IVDDgNFCUAsIM5oABQRCVGAEBAcjA1CYXI8U6jqI4YazmeMgmA4O2mMRx\n    LbauhBkECfQRoJAR50mm/a5MCetTbQgEkkAmsG8UVr6EPsYKSYWg70NEEUOMBZZw\n    HViojUvGCIIKcQmUIpJBjQx3qSH2XQjOKIQ+0BhA3z2L5BAjoxhWGLoCLgjnwAgI\n    EcbEhUjtcBQ7c84dRZPZOhkNtb2kXYk1Jqz8X97KeDnbfMSqokMOZfVFFUbGdezS\n    lkKHXv4WiIMWz46Vw8QWw4ygnAngRotXTuyo08QCWWiYUEQJqBkjmAgsOeGQSg18\n    RizyYaCQokgT6weSaY4NBkQqHwIR/JcQO5+y5CTTsBTJykhivcnpuqaVubo1ufpV\n    V2R/Nrf2ynX/fuSNJ1b/vaF1+be0Mdf1Tld0w9lVi3ubCu/KRXnhlpOf98z1jd13\n    COyZOn3q+lj90Lc4N/P2L4WP83eLXYdv36zr35w+cu1rbe9tuKP86plkHiwffKvh\n    cPT6tvOn/Zd3/rV759LWA6a1c/aJ6vvb0dahPY+dO9n42UfV/cvtv3+3aWFkxYmk\n    tGVmqffPVx7atfR0Kej+dKHvp59fWMnEV6ee2etddWx9YcfqWx+/50hPd1fV7mi/\n    aazj7Lpvp+YXrut444NfG/ddaKebJo43w3T25gce7H6pybtmcbrx+z+eXf/j5vPD\n    c0+Ob+FTg4/e9c3XN66tHudH91ePbU8+vHqk3FD/VFvH/L2fHKx/vld/ea4t/uIf\n    =i2ny\n    -----END PGP MESSAGE-----\n\nFinally, I am proving my reddit account by posting it in [KeybaseProofs](https://www.reddit.com/r/KeybaseProofs).\n",
      "check_data_json": "{\"name\":\"reddit\",\"username\":\"oconnor663\"}",
      "remote_id": "t3_2q0kk1",
      "api_url": "https://www.reddit.com/r/KeybaseProofs/comments/2q0kk1/my_keybase_proof_redditoconnor663/.json",
      "human_url": "https://www.reddit.com/r/KeybaseProofs/comments/2q0kk1/my_keybase_proof_redditoconnor663/",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 4,
      "hard_fail_count": 0,
      "last_check": 1434485079,
      "last_success": 1434485079,
      "version": 9,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 4,
      "payload_hash": "00279b0704b01ac6171beef1a5718a8de52d5f0fd0ab63dc947dd6f111bb673f",
      "sig_id": "89e70d05df9c659d1ad8b0e2ead5790cc500d29345979af2e5807edefab5666f0f",
      "sig_id_short": "iecNBd-cZZ0a2LDi6tV5DMUA0pNFl5ry5YB-",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kk1sFVUYhu+9SlsKJARRiGJoJkIa017P/zmDC6J2URMEF9UoKNfzN+3YMnOZ\nub20aa/SVetCRRaUSMPKGCJlgQqYaGpK1DRFY4yp1aj8JBgEqQJBgwsTz1wxwsbJ\nLM6ceb73vO/3nT3L7sg15XeO9vYc6WxH+VMXVa5rfOuVIU/FZtDbMOTpvtBGlWwV\nyR3W2+D12kElU1sM45YoNrb4Ytpyk2nzqjZJwzhyFCjyIvZqbRmeFQdh1G2TchJm\nWh4VEhFqsBEKUQ1pwLTFQRBwKDhTTEhDBDIYOcmeOK3cdqpX1yyFxu3W+UeZeKTD\n8R11vr/+g0AFA04DRSkgzGAOGBAEUYkREBCADExtcjNSrOMoihPG6o532KS3z5aS\nOK7H1pUwgyCBPgIUceg8ybTHlVkCiK+sltkJDHJiqDBCIy10AJUvlQCWc0JsYAJo\nKEQGYUZ96rtXQGYA0pJqQYEwhHLgK2PdkktnllJOhAOkJpZQoaRUriAwWAPMfCNc\nBfZdiNTujGJnzrmjuJZ9J9VQ21vGpeMwyjr3f4krg+Vsc5dVpXTyH4WSCiPjZnbr\nUKFDb+8GZqDNswPlMLGlMCMoZwK4p80rJ7aaDYJIyjgVUGKoqM8C7PpENWEYB1Zb\nAwRgAUFcSc18yxGQVgdGAoS4r3yl/stInE/Z7STTsDuSlf7EerVX8mvuzOWbcg2L\nCtmtzTUvXv7vVd5Wbsq9e7JhazT+wIGZqOdc4zNdy/5sPDDxzfaBLYdHPi8Uvrs4\nvef15hfkj61zuSvrTo7WVnVVv7w8NNCyce10n/71s2eX/3D/iYYnL7//06FhOLl/\nYWV/x0O/zc9utBfODy+91vpS68d7fz/2xFV4+vRrl8pjbMnKa4eO8qOjm1bPXT0y\nvmK2c8Ru3l+899hSVug4c8+3xS/u27Tm/NjM3ZfOPpVfe2JbemPFxKmpwvW3v1qY\n2vfgDTC1+K/Hdg3Mr9+3W8y9sb755erh7Wc/fPiD4teLDrafWfLR7PePH9zCnnt1\n83vz7+xddVfjW+uuT/5SrM0Nf7L7XLWl1E7109Njx9+80PfpBH7+55k/VneOHPcW\n/gY=\n=2nAo\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419205271,\"hash\":\"e4049beca46d36174d58d8c2c8cf1b9ab80e7744efdf1d512d2365959959816d02ca5c8508d45709bde08d7a70655748d02ac4e458baab16dfd3c0369d88d439\",\"seqno\":114153},\"service\":{\"name\":\"coinbase\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205360,\"expire_in\":157680000,\"prev\":\"44a567581a31b596f31745c4633feced0806f427bac69e720aecfda02279b9bb\",\"seqno\":4,\"tag\":\"signature\"}",
      "sig_type": 2,
      "ctime": 1419205360,
      "etime": 1576885360,
      "rtime": null,
      "sig_status": 0,
      "prev": "44a567581a31b596f31745c4633feced0806f427bac69e720aecfda02279b9bb",
      "proof_id": "de92300787097c3f7ae36910",
      "proof_type": 5,
      "proof_text_check": "\n\nowF9kk1sFVUYhu+9SlsKJARRiGJoJkIa017P/zmDC6J2URMEF9UoKNfzN+3YMnOZ\nub20aa/SVetCRRaUSMPKGCJlgQqYaGpK1DRFY4yp1aj8JBgEqQJBgwsTz1wxwsbJ\nLM6ceb73vO/3nT3L7sg15XeO9vYc6WxH+VMXVa5rfOuVIU/FZtDbMOTpvtBGlWwV\nyR3W2+D12kElU1sM45YoNrb4Ytpyk2nzqjZJwzhyFCjyIvZqbRmeFQdh1G2TchJm\nWh4VEhFqsBEKUQ1pwLTFQRBwKDhTTEhDBDIYOcmeOK3cdqpX1yyFxu3W+UeZeKTD\n8R11vr/+g0AFA04DRSkgzGAOGBAEUYkREBCADExtcjNSrOMoihPG6o532KS3z5aS\nOK7H1pUwgyCBPgIUceg8ybTHlVkCiK+sltkJDHJiqDBCIy10AJUvlQCWc0JsYAJo\nKEQGYUZ96rtXQGYA0pJqQYEwhHLgK2PdkktnllJOhAOkJpZQoaRUriAwWAPMfCNc\nBfZdiNTujGJnzrmjuJZ9J9VQ21vGpeMwyjr3f4krg+Vsc5dVpXTyH4WSCiPjZnbr\nUKFDb+8GZqDNswPlMLGlMCMoZwK4p80rJ7aaDYJIyjgVUGKoqM8C7PpENWEYB1Zb\nAwRgAUFcSc18yxGQVgdGAoS4r3yl/stInE/Z7STTsDuSlf7EerVX8mvuzOWbcg2L\nCtmtzTUvXv7vVd5Wbsq9e7JhazT+wIGZqOdc4zNdy/5sPDDxzfaBLYdHPi8Uvrs4\nvef15hfkj61zuSvrTo7WVnVVv7w8NNCyce10n/71s2eX/3D/iYYnL7//06FhOLl/\nYWV/x0O/zc9utBfODy+91vpS68d7fz/2xFV4+vRrl8pjbMnKa4eO8qOjm1bPXT0y\nvmK2c8Ru3l+899hSVug4c8+3xS/u27Tm/NjM3ZfOPpVfe2JbemPFxKmpwvW3v1qY\n2vfgDTC1+K/Hdg3Mr9+3W8y9sb755erh7Wc/fPiD4teLDrafWfLR7PePH9zCnnt1\n83vz7+xddVfjW+uuT/5SrM0Nf7L7XLWl1E7109Njx9+80PfpBH7+55k/VneOHPcW\n/gY=\n",
      "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am oconnor663 on coinbase.\n  * I am oconnor663 on keybase.\n  * I have a public key whose fingerprint is 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32\n\nTo claim this, I am signing this object:\n\n` + "`" + `` + "`" + `` + "`" + `json\n{\n    \"body\": {\n        \"client\": {\n            \"name\": \"keybase.io node.js client\",\n            \"version\": \"0.7.3\"\n        },\n        \"key\": {\n            \"fingerprint\": \"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"71876B68AD482D32\",\n            \"uid\": \"41b1f75fb55046d370608425a3208100\",\n            \"username\": \"oconnor663\"\n        },\n        \"merkle_root\": {\n            \"ctime\": 1419205271,\n            \"hash\": \"e4049beca46d36174d58d8c2c8cf1b9ab80e7744efdf1d512d2365959959816d02ca5c8508d45709bde08d7a70655748d02ac4e458baab16dfd3c0369d88d439\",\n            \"seqno\": 114153\n        },\n        \"service\": {\n            \"name\": \"coinbase\",\n            \"username\": \"oconnor663\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1419205360,\n    \"expire_in\": 157680000,\n    \"prev\": \"44a567581a31b596f31745c4633feced0806f427bac69e720aecfda02279b9bb\",\n    \"seqno\": 4,\n    \"tag\": \"signature\"\n}\n` + "`" + `` + "`" + `` + "`" + `\n\nwith the PGP key whose fingerprint is 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32 (captured above as ` + "`" + `body.key.fingerprint` + "`" + `), yielding the PGP signature:\n\n` + "`" + `` + "`" + `` + "`" + `\n-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kk1sFVUYhu+9SlsKJARRiGJoJkIa017P/zmDC6J2URMEF9UoKNfzN+3YMnOZ\nub20aa/SVetCRRaUSMPKGCJlgQqYaGpK1DRFY4yp1aj8JBgEqQJBgwsTz1wxwsbJ\nLM6ceb73vO/3nT3L7sg15XeO9vYc6WxH+VMXVa5rfOuVIU/FZtDbMOTpvtBGlWwV\nyR3W2+D12kElU1sM45YoNrb4Ytpyk2nzqjZJwzhyFCjyIvZqbRmeFQdh1G2TchJm\nWh4VEhFqsBEKUQ1pwLTFQRBwKDhTTEhDBDIYOcmeOK3cdqpX1yyFxu3W+UeZeKTD\n8R11vr/+g0AFA04DRSkgzGAOGBAEUYkREBCADExtcjNSrOMoihPG6o532KS3z5aS\nOK7H1pUwgyCBPgIUceg8ybTHlVkCiK+sltkJDHJiqDBCIy10AJUvlQCWc0JsYAJo\nKEQGYUZ96rtXQGYA0pJqQYEwhHLgK2PdkktnllJOhAOkJpZQoaRUriAwWAPMfCNc\nBfZdiNTujGJnzrmjuJZ9J9VQ21vGpeMwyjr3f4krg+Vsc5dVpXTyH4WSCiPjZnbr\nUKFDb+8GZqDNswPlMLGlMCMoZwK4p80rJ7aaDYJIyjgVUGKoqM8C7PpENWEYB1Zb\nAwRgAUFcSc18yxGQVgdGAoS4r3yl/stInE/Z7STTsDuSlf7EerVX8mvuzOWbcg2L\nCtmtzTUvXv7vVd5Wbsq9e7JhazT+wIGZqOdc4zNdy/5sPDDxzfaBLYdHPi8Uvrs4\nvef15hfkj61zuSvrTo7WVnVVv7w8NNCyce10n/71s2eX/3D/iYYnL7//06FhOLl/\nYWV/x0O/zc9utBfODy+91vpS68d7fz/2xFV4+vRrl8pjbMnKa4eO8qOjm1bPXT0y\nvmK2c8Ru3l+899hSVug4c8+3xS/u27Tm/NjM3ZfOPpVfe2JbemPFxKmpwvW3v1qY\n2vfgDTC1+K/Hdg3Mr9+3W8y9sb755erh7Wc/fPiD4teLDrafWfLR7PePH9zCnnt1\n83vz7+xddVfjW+uuT/5SrM0Nf7L7XLWl1E7109Njx9+80PfpBH7+55k/VneOHPcW\n/gY=\n=2nAo\n-----END PGP MESSAGE-----\n\n` + "`" + `` + "`" + `` + "`" + `\n\nAnd finally, I am proving ownership of the coinbase account by posting this on coinbase.com\n\n### My publicly-auditable identity:\n\nhttps://keybase.io/oconnor663\n\n` + "`" + `` + "`" + `` + "`" + `\n",
      "check_data_json": "{\"name\":\"coinbase\",\"username\":\"oconnor663\"}",
      "remote_id": "oconnor663",
      "api_url": "https://coinbase.com/oconnor663/public-key",
      "human_url": "https://coinbase.com/oconnor663/public-key",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 192,
      "hard_fail_count": 0,
      "last_check": 1434481477,
      "last_success": 1434481477,
      "version": 3,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 5,
      "payload_hash": "d64c8d89811ef70c83ab58cf6350d4e847f1750cff3c468d7e8556c65073e09a",
      "sig_id": "8b4bc6a736a2a651763300384d54e72daa0ee3f9464ae98fd2df4f06f93d59f60f",
      "sig_id_short": "i0vGpzaiplF2MwA4TVTnLaoO4_lGSumP0t9P",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9km2IVFUYx2e23ZwEJRctBYPda2DlOJxz7nm5o/lBd3uxkMi2/ZDYdF5nb7ve\nO957d7dRBmwNRC3ayFIye1nKEsoUMTIUjcoWAwk0tk39YJCUFpZIBqtu5w4b6ZfO\np/Pye/7n+T/PMzjllkwuu3pDd9fHD89H2W9/FZmOrStqax0RqqqzYK0je3wdJOku\n4Ku0s8Dp1lXBY13ww5YgVLrwbNwyweSdPh3FfhhYChRYwXVq+RRPg40flHVUifxU\nyyEeR5goV3kCEQmJoVK7xhgGPUYF9bjCHlIuspJdYZzc9KtT1yz5yt7W+SXUW9xu\n+fY631t/wFBAw4gRhABMlcsABR5GhLsIeBCAFIx1NGEplGEQhBGl9YxX6ai7R5ei\nMKzblomfQhDDIgIEQ2Jz4nGXDdPY0xoY7hFBsGcwABxZI0QVFRESY44MAspwyj1F\nDVSEYcklUYBpVwpPmSIVwGhoFJWQy6Isaq05QBwSz37kcokos89WESlsMFZS2HjK\nCGMQedyaiPXqILTJ2ewIqaXnqM+X+oZ2dXHZbY3q/vj/PCfVSnrZr8Xu0oRGSfiB\nsl27sa3QojfXgwCYd/RzFT/SJT8lCKMesCvvVCLdl04CQKwoAANYAOuSQgaF1gZy\nYrtnK6MJUsQAowAX1FWyiJlKywWhEJS55j+XtvIJL1vJ2C8HPOmNtFPbmJ3dmMnm\nMrc2NaRzm5l82+3/DnPSlstsPvyGOXpg194r5dnN2zrfGg7K58Ym/Z1cCzq+mffj\nH51fPrX7euflphkHK18cGbq/9Obcx7KtT4+NrF/y1V1t4wfId4s+jB4d/awpM6V5\nXUNruP3M4uL++Pqex38aOPLzlgv4oca2X4ZWztq7+fjyma8fL7RefHfr9E/m9b49\nkKBTi1ZmT48NLl1zbOelXYeH57y4b+bUu0/uG3llzed//p7cM+3c90eP+cuevPf9\n4Q9O/fbIlcYTGShySX9nw9yhQs+lOwfveH75xa9HWs4/8d4zHevzB18YLu+c9FHu\ntWrrnJ6XPt30wH3b/2qunR24Ot5iTrw6Om1F+zuj29pPL1s469CDS3dUR9tGftgS\nnnx5HHf8Aw==\n=B0oP\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419205415,\"hash\":\"e48ee0fa85b548f400a26ce5d9d5bc44a2f20dfa6a8d6f1d574cac5d07e3cb8df96b0fe1fd6c1ac9c9eeea02a1585413ac267e1fe5d2d4f44dcbac567577128a\",\"seqno\":114155},\"service\":{\"name\":\"hackernews\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205501,\"expire_in\":157680000,\"prev\":\"00279b0704b01ac6171beef1a5718a8de52d5f0fd0ab63dc947dd6f111bb673f\",\"seqno\":5,\"tag\":\"signature\"}",
      "sig_type": 2,
      "ctime": 1419205501,
      "etime": 1576885501,
      "rtime": null,
      "sig_status": 0,
      "prev": "00279b0704b01ac6171beef1a5718a8de52d5f0fd0ab63dc947dd6f111bb673f",
      "proof_id": "ff25378bd64cc2b0d2942f10",
      "proof_type": 6,
      "proof_text_check": "i0vGpzaiplF2MwA4TVTnLaoO4_lGSumP0t9PBvk9WfY",
      "proof_text_full": "[ my public key: https://keybase.io/oconnor663; my proof: https://keybase.io/oconnor663/sigs/i0vGpzaiplF2MwA4TVTnLaoO4_lGSumP0t9PBvk9WfY ]\n",
      "check_data_json": "{\"name\":\"hackernews\",\"username\":\"oconnor663\"}",
      "remote_id": "oconnor663",
      "api_url": "https://hacker-news.firebaseio.com/v0/user/oconnor663/about.json",
      "human_url": "https://news.ycombinator.com/user?id=oconnor663",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 210,
      "hard_fail_count": 0,
      "last_check": 1434471730,
      "last_success": 1434471730,
      "version": 8,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 6,
      "payload_hash": "37ad41cf064989dfd849a0af178aa066dd396b3452433a9883a69e88e35e75b8",
      "sig_id": "5941324c9d6c5c830c832f44494cbc73af0cfaff6e3c9e69ae88bbda45e72c3e0f",
      "sig_id_short": "WUEyTJ1sXIMMgy9ESUy8c68M-v9uPJ5proi7",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v1.1.6\nComment: https://keybase.io/crypto\n\nyMHMAnicO8LLzMDFWNienbHWQ9eI8fSBXaUMIdNj86qVkvJTKpWsqpWyU8FUWmZe\nempRQVFmXomSlZKpRaKRiWmKcYpFkpFpsqFpmllyqnFaWpq5oYW5WZKZRWKKiYVR\nirGRko5SRn4xSAfQmKTE4lS9zHygGJATn5kCFMWivhQsYWKYZJhmbpqWZGpqYGKW\nYmxuYGZgYWJkmmhsZGBhaGAAUlicWpSXmJsKVJ2fnJ+Xl19kZmasVKujVFJZABIs\nT02KByopy0xOjU/KzEsBegCoqyy1qDgzP0/JyhCoMrkkE6Tf0MTQ0sjA1NLSQkcp\ntaIgsyg1PhOkwtTczMIACHSUCopSy4BGppiZJFukWFhaGBqmppkbJFsYJyaZWiSn\nmRmbGqSYpFqYmKcZmpsaJKelGSebmFmkmKdamJqaJZuZGpgbpxpYJgLtL04tzMtX\nsjIDOjMxHWhkcWZ6XmJJaVGqUu2hAzEsDIxcDGysTKAoYODiFIBFjLMKB8OJ2qe/\nZr3gLy6SvvjN507UXm6uyy+2Jr2S2Jmconh4rcmchIcyYbsTGpuXb+XbvvHYnyK+\n6Qr1z9Vku4vzZgazFKw7pivUk3NdcXWEterdmca6ynKPhS78UWX5l2veJ7FycfPu\nSJfvTz3uTQzgft/3eM+CH29FHyfOy57Z1hf5/c1qt8XH93Umxs0/tca7+0n5oj/t\ni953tFf9d/bvu3HXTeGx5vIL6zdL+3qzfSu4nx0p9tZG2MhAte2tfMaFD6sZOP+r\nRzwPDotbUiO+3PQGSzGf99aK4Iebbk4XfF4WK3TycN/VWaX+W2LSCgzXbXlwiOu0\nFM+WnNRVXZHdM2cHnpxq+eXelcO9MplbLTUA7nMASQ==\n=dQNg\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205998,\"expire_in\":157680000,\"prev\":\"d64c8d89811ef70c83ab58cf6350d4e847f1750cff3c468d7e8556c65073e09a\",\"seqno\":6,\"tag\":\"signature\"}",
      "sig_type": 1,
      "ctime": 1419205998,
      "etime": 1576885998,
      "rtime": null,
      "sig_status": 0,
      "prev": "d64c8d89811ef70c83ab58cf6350d4e847f1750cff3c468d7e8556c65073e09a",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 7,
      "payload_hash": "2d8dfe977e6d359a03430c5cf22993d16438b1e10546b1b247055abc9cad3edf",
      "sig_id": "bd1445bcc75e2d6ed5031c245306863fbaddf8673e674903fe5ac7c1414136a90f",
      "sig_id_short": "vRRFvMdeLW7VAxwkUwaGP7rd-Gc-Z0kD_lrH",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowFdkl1oHFUUxzdNTFI1WKEIQcU4FTF1E++dO/djAlYiGwzBwLakpU0iy/1MJh8z\n25lNNMZVMMYiRoiGFlpt6IfUNqUP2qJQqAj1QVofLKgPpU0fpAShopJIo1XqnSWK\ndV5mOPPjd//3nDPbUJ2pr9q9Z3jwVGeLW3XxR5Hp2b94e9IRkZpw2iYdORLosJR+\nhXxUO23OsJ4QPNGtQdQURkq3DiVNa0zWGddxEkShpUArbUVOOevIeKJYiuRYHOtQ\nVoxcqVgniYVgV0deIt0ebMW93dtCd+jFwZ5Sx3bk+uM+kjkPdVhnaaKYHiuCkoyC\nMFXaBKnHBOGAjotxkMZzMOOuhxVSTLhYQmyI1MgYQyGjRBDGlcdchVxrHIyS0h0X\ncSrOQqBstcI/S1h7zvK5Cj9W+eFBAQ3FRmAMPKIQBQQwz8UcuYBBAFIw0fFalyIZ\nhWEUE1JpwqiOh0d0IY6iSidlKUgh6EHfauzLZuLJYNo2ZrRA1FMeMEYyV1BglJBA\nu8rFyBYo1QBITJXxPGIMI4hSSY3CwIeSA0WQwD4hRkHGMZGA23zWRSQiHqXGYE8Y\nbL22IJFluCKaad9XkGBNtZP9osZJ9O4wsvEgxswv/zuB/03yP9OG6ZzvuJOL/Kyj\nXyoGsS4EKYEpYcA+WacY63FrQ9ROBEoDiOczXxnFPJ8DbiC1oQAhSiGfCORh10OI\n+4whTnzNmEY2JxbMnr+Wk9qEfMAqk2Ag5KWxWDvlt6oeqslU1Wdq71qXrnPm7vUb\n/tnxh6vrbtWWy6XtP8MDr7xTx/uvwtX7Z3a0Ty18dG3nwqGVB/sy1+tOrLuc25J9\n/M39BdEc7+t9ufYFlv9w58hfPcstS/7Sei7nZ16DF9798/zlngfsNu+6ZzrbfOP0\n9PkF58xXb8//seWDba9ePF68WX6jb/rCvb8ttXS/3/jknrlre59o/P7o4aN9lz7d\nHG98rHDwyqaVTb9c/Xbfro+f75KP/PB07uu5hmcefS65cXOme1asXL/vm8bFs8dX\nDx+Yqu89NtwwuVw32bn597nmM+VCvvG96qm9Q8nW0defaspnv5v7ZMftYLHfPXf6\ny18b8h1G9M/PfvbTQVF/5MqxkVvLhc7PazZeOrnQdgisgq6/AQ==\n=5wxk\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"cryptocurrency\":{\"address\":\"1JEPc3eAiQ5ZMRn2jwhTtEU329v93cD43E\",\"type\":\"bitcoin\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419370141,\"hash\":\"08feb374d40ffc82b70fdbc0e2d253c8277e00c57df446ff86377c7fd5091ca0d63b5966fd18a56c0a5a340f6c36477ff54bf582b6c3c318aad6e8e99d165e7e\",\"seqno\":115589},\"type\":\"cryptocurrency\",\"version\":1},\"ctime\":1419370239,\"expire_in\":157680000,\"prev\":\"37ad41cf064989dfd849a0af178aa066dd396b3452433a9883a69e88e35e75b8\",\"seqno\":7,\"tag\":\"signature\"}",
      "sig_type": 6,
      "ctime": 1419370239,
      "etime": 1577050239,
      "rtime": null,
      "sig_status": 0,
      "prev": "37ad41cf064989dfd849a0af178aa066dd396b3452433a9883a69e88e35e75b8",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 8,
      "payload_hash": "57c95524e2fdb546201af092292ae9d27a055a8934d81a8f8488eec777441d1e",
      "sig_id": "f6689d2fa3ff3b2fc130fcadb73e7de28ca55b3c59b493de45f9c61d980b5d220f",
      "sig_id_short": "9midL6P_Oy_BMPyttz594oylWzxZtJPeRfnG",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMPVAnicdVRriFVVGJ2Z7OGIYpJQoBKHULAx9/uhRFmBIUExzWiadvn2a+5xZu4Z\n7znXZpCh6EcpRknmg8wyEhmFgdQ/EQSTjJhFREHSixBMS4gIS1Aiap87kzpQ59e+\nZ5+71vrW961vfOZNLe2tm17qrY4+uoS0fjre32jp2n9wfEtiMjeULNuS2L7U14ry\nVIN+nyxLev2Qgdzfl2Z3P+tN0pFs9vU8zWrxylmNLGfeSaqUos5ggICYFByhIL30\nhlPALiTDHSVMCRrSWo+vD9TTkiPhCgjjjjplCLeYB2E9DSFIrKQwQoFjijhKIms1\ny4spapImZiV18e1/fN/bvEAYYeoRp4IIcIQKJbySJIDkRjDgRiGKCdcSGW5BgjTS\naWWpjj8UidcYCIII12jCMWxwkDwYzhETjkokkGKEAyVIYYTKD3Nfn3Qus1mtltWF\noKUBRR1sb2lB1J/avDylrnLNTNqR9EFexHoqtgrRpGQZZlgzoSPXFNiNvVme9w3V\nStCmquiiwEgSpCEo7oTVlhBttGIaO9ZUNel+adjUDoCLD3UoFi6Mk0R6BSGaxYAA\nJyYIDt5H46Y4qoXQ2DOJqXDeaBY4ozggIYLmIBQTgRJpo4GeMBdw7CoHA9hC/A/X\nzkdHo/a6788KXxmoZ1mIdjwdZ69I+6/XzTjvSGyjXo+sWHNPhMLEK6SARS7GvA4o\ncjMfOEGSUmcDcCuIZAp5Rpx3XALCJZaP8v0kOJcSswnwSQWlK00VzfmvettbcVBA\nZWNeNuZaEHrSotowyf/2oglRKYYGfLObeQFFyRdvcr+plsVjPKU9ExOLqaRWO2kt\nt8wBozErgLShAjtHLA9Ee8kRRVh5jJ3FKGCMGFIxUIYYikIygTbBR4Y3NGkqBaR9\npeYBGOrLwFWqkFcjnRAids0aqaWmJlCOZcyblwikhtJuMPFMtI4Z4LGHcSyAQqBg\nsHbO8ORaEeSGIhh45wXyQkpJNMVGeqGIDlRT6wL3DAclwAYvnQXr4pCCxiw48CTm\nisaakuEyGc0SJgNyw34prbthJASWcbXERg4OpHVfSWvNZgqF4lO67zdHEOKUC17L\nqMRRrgFRRuOOsiEmQlOHBaPKYI8RZ8JEJ5lEsV4TIwOOeheuF6qiMOiJkLHcGhSN\nuk+GT4ytn9bS2t5yy81t5dZsaZ8++99dutvd+vfqHSu2LX5415d3kjNzVp6ffezq\nocZrzyx4e+v+eR+8PPPcqSOd/o87BjeN/3Xpu8PvVc3H73e+sKHt3PjJ4qnuGc9n\ne3etubTnwY/2zOrad1z2v37xnauPLThxYIa7evL+Kztnbb/3q4Xzz46ceujyyZ2X\nD7W17lytty1ad3r7+cE3vnl8JF/Z/cSZhXNH0wvTalnl21+6nrxr2x74fu46Li7f\nvqrozBf/9EX3xcaVH7d2fP57/4YDoQuvfW70zZGtvz1ycN7RkWM7zp7+dfT4J0fC\n0uVn1/9wbN6fe98dGT+84nx7embtqle+vqAPHl236J4H5nfMectNH7tt6e4FY2PJ\n4L41n21esvziq9K92KY6Pyx+/ges9xbZ\n=2qFk\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"dc90c54ed738883db1aaf0476500f7e7eb53a1df\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":3,\"last_id_change\":1419469550,\"username\":\"jkosslyn\"},\"id\":\"58a6107209af85d6c9c229b98491d400\",\"key\":{\"key_fingerprint\":\"adddd3d07b76bd727e8af6264a2a52bf65aeed23\",\"kid\":\"010196691e47136deb94f5431f066f95a6846f327c782e24df1f6c5aba1cae4759de0a\"},\"remote_proofs\":[{\"ctime\":1419469455,\"curr\":\"195e26812e808a4f0644e9f094f4ef520733dcfa5c627480e42ded57a011946e\",\"etime\":1577149455,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"jkosslyn\"},\"proof_type\":3,\"state\":1},\"seqno\":1,\"sig_id\":\"1373c9d7cc5c4da43eb5a09b361dd2c5f29e7503018e11dc10f1104083a1b2b30f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"666196cb79793bf35175c1e70a79a4194ab70a299a7a55a6adda3af3ab19ddb5\",\"seqno\":2,\"sig_id\":\"4aede60e67772931b7e6829f393cdf5e41f86acfe7dcacd610a914fdae21a2310f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1419617476,\"expire_in\":157680000,\"prev\":\"2d8dfe977e6d359a03430c5cf22993d16438b1e10546b1b247055abc9cad3edf\",\"seqno\":8,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1419617476,
      "etime": 1577297476,
      "rtime": null,
      "sig_status": 0,
      "prev": "2d8dfe977e6d359a03430c5cf22993d16438b1e10546b1b247055abc9cad3edf",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 9,
      "payload_hash": "d70f647a155416bd94ad669254e5513aa667c03ed55f047f81cd6fc9bc3ed89c",
      "sig_id": "7ec6ea814573f2d7db7947262a624e4b0990ae5cc48feff1c8572b69f60b769e0f",
      "sig_id_short": "fsbqgUVz8tfbeUcmKmJOSwmQrlzEj-_xyFcr",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMSiAnicrVVbiFVVGD6T3RyNmUqSLqZtUYxGW/fLeRgDMSytqCQIJ07rOrOdyznu\ns0edxklfNMtECroR3R5KIiQJKy2K0hgljXpworCUsIcghHkoRLLs38dxmiEfeuht\nrb3W/v7//77v/9cXV00pNTeteaK7a9eyhaTp8PDS/tLK3Sv2DSa26geS8mDietLQ\nlxerPtMbknLSHQasqYdFaXXOumCTtmRtyOpptQ+OCCYOReWFwcypwG1kHgtBuVOe\nWaGc5I4F5ZKhtgKmAI1pX2fIallaxEi4MoRxT72yhDvMo3CBxhglVlIAgPFMEU8J\nRO2q1vNJ2SQNzErq4etF7nc3DhBGmAbEqSDCeEKFEkFJEo3kVjDDrUIUE64lstwZ\naaSVXitHNWwUgWNsCDIA19+AY9jiKHm0nCMmPJVIIMUIN5QghREqLtZDNsZc1VX7\n+qoZEFIQkGfGdRcUQP6pqxer1FfGycSyLekx9RwKqrguAyzBN4Y101hRPAm3nmep\nLSAbOYVIAtNa4OC9MVIizSzjwRcyRNfIaYz7gq7J/MsYHFMMu4C9IMxJLEBEawwV\nmgmBuLMkeGcm8cmVlVJoHaWlhmjuGQtWaWohoCc8aiCYwf+KBsGIDJgbuAFkI0KU\nlhr4hNyz0FvNQ6WWVasRyFgFzsvT3kbRSHHBIZW2xPVnGUTVInoZoieE+Yg8kpTJ\naDhxyjjLDbJBw4ZFLpiGpRXUe4WZgoysB1dB+mEMnAuo8zx4LQtrAdxiHyiVETGG\nPC7K9TGGwIKWQAgEYIobISkKihZyO8miYlQSbzTHTCbjtRT8Nupp9FFXcN0Vb3JT\nWV0vBB5sWPiCNXq8q2Zhkav2JkUm1RzM0gMHXXleKxf8NIAq+UCtyBoh1Aaym7zY\nwGE9rOmrJmUOq7TzQgsIo0B7QaGnQHzDDZHYYWMDEgoJr0kAj2ojsNZgcWMl904G\nY72PoB8qWCrQzockQ22TFBFMCo7HFVFG6+A5U0Y55r2nnHAFvSuUpNQJb6F7qCch\nYm09olw6473TRmkvYmx01ARFKDkPPqbI/yD3f1RkTI3ONO/qt8nFm2yiEPRiKogJ\nKuDIHMwT6zTCzikwkpKehUAU9z5gQqR2gkQsEMxL0EVZ6D4rJOcSBgoT/1LhkUaY\nSm7SniLhmhnoqRpf6TL1rkIGbkA7G3GA/mPRIGUdjgp7bS3DVGHFsVYIBYcxtLfg\nIRiNYGRHwqTyxZwcKwKjCVUI6GwfWSGiRRRkhSFKvBIgeQzSUOMcgYkZJXS0DdYE\nypSMMDItPAXWFV4aKiZeo4axwTfh3Si4G7cWDE6GMAH1w/pamoVKWtzgEjzbcP2Y\nJ8BBmnPCAoneciYIwiYiTYgmJmhPpEGcm4IFsIJR0KFKheCklIxhj8M/lWpIzHQW\nCqedfSbvz0IytP+zjktLTc2lyy+7pHgNS81TWy+8kdPeveLcHV89fe9o86nHsk8O\nr/781NeHdz3XvX354jmbPt6zYNuts7a+cfWh0y2lqvtwSm3JsH372S1Tl3fcvm7D\nxv3zjy75+eDJcph/6pmZ59666Zf81yU/tR1o//73v+4cXf5N6+md82Zsapr96Ob3\nbmg5eeTmE9c//uLw8dbnj7llO6d/ef+ylU/evbXlxtKxo/v63xxomdYxPb3unhcO\nnX3/24fnvnZ88we37B356OVzZ+c9sG36gbL59GD3suGlZ3pqw90L4ow/XmnvGN14\n4rYNO3e80zMyJz515Mot5dWjuzpeXbG79aXy3M6WWav2nhzh4s8HL9+x59Dxa9fe\n1b69fYDNfH3qD/sPDK2/r7L4msEzv/3YMfuh0ncLR/4GukqcnQ==\n=TY+X\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":17,\"last_id_change\":1419491831,\"username\":\"strib\"},\"id\":\"ef2e49961eddaa77094b45ed635cfc00\",\"key\":{\"key_fingerprint\":\"7fec4841ce1d624c7165bfbaa36946605cb2edca\",\"kid\":\"010158b77699f7b3a295d44eb893b35cd25f9872469483e6427e15a44e4a502289790a\"},\"remote_proofs\":[{\"ctime\":1408565716,\"curr\":\"96fd7efd224df0d07347fa52c8acb5a0be952c4f5649e95b63dd8148b35bd15f\",\"etime\":1566245716,\"prev\":\"b1de337f0440d12edcdffee4e974c7acb485a6730e830608c74f84372da95147\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"oldcore.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":5,\"sig_id\":\"716a8094638a2d63a5a271c1abe06806d92e0819a619998cab75dc7eabddfeb80f\",\"sig_type\":2},{\"ctime\":1408647651,\"curr\":\"8a99ed548a8c4ddd35258ce368733c6db7823d2ef19bd0357caddc9a89d6ff0a\",\"etime\":1566327651,\"prev\":\"96fd7efd224df0d07347fa52c8acb5a0be952c4f5649e95b63dd8148b35bd15f\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"strib\"},\"proof_type\":3,\"state\":1},\"seqno\":6,\"sig_id\":\"1f4c125bc901cc8ffe87d4ee285dde12279c62f160c8e0688b841b67557046460f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"85afebbf1e8934fa08bc1f81d9bb413818519800ec1124c65eea900f8f2478d2\",\"seqno\":10,\"sig_id\":\"67b3df4b782b03ce30132d869edfe7a3acc270bf7022bebae3487fca7bd6abc80f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1420840121,\"expire_in\":157680000,\"prev\":\"57c95524e2fdb546201af092292ae9d27a055a8934d81a8f8488eec777441d1e\",\"seqno\":9,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1420840121,
      "etime": 1578520121,
      "rtime": null,
      "sig_status": 0,
      "prev": "57c95524e2fdb546201af092292ae9d27a055a8934d81a8f8488eec777441d1e",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 10,
      "payload_hash": "251d1c8b830c3c4b131a182d27188712bf85a7026755bd0ad16ec2f58c880a07",
      "sig_id": "2c449b4270eeb6094bd48557866343e89dde1983a1694a5625b8076ed793fbc30f",
      "sig_id_short": "LESbQnDutglL1IVXhmND6J3eGYOhaUpWJbgH",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n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\n/fOhx5fP2ffg1w6ed+015rc3fOvBp++87PDF/wN3bEWj\n=S3O8\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":103,\"last_id_change\":1419928822,\"username\":\"max\"},\"id\":\"dbb165b7879fe7b1174df73bed0b9500\",\"key\":{\"key_fingerprint\":\"8efbe2e4dd56b35273634e8f6052b2ad31a6631c\",\"kid\":\"01013ef90b4c4e62121d12a51d18569b57996002c8bdccc9b2740935c9e4a07d20b40a\"},\"remote_proofs\":[{\"ctime\":1391720984,\"curr\":\"35782a3561e40a5e7217500bd67b5aadac5d7f6f9594dcd5d15cd06879da6e19\",\"etime\":1549400984,\"prev\":\"0146be7c3669a5f5da89e9908e8bb5f1fdb9fcfd6d2d66a51664f1075a7637b1\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"maxtaco\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"9f731fa7c75b64e1c9d70300b4383196a8fb432294a4308f8d7a379376a0b1900f\",\"sig_type\":2},{\"ctime\":1392252388,\"curr\":\"c610bab1245d05ac6fa95eda4d008d67c3732bd441dcc5c74f48fc8f8ba44c31\",\"etime\":1549932388,\"prev\":\"35782a3561e40a5e7217500bd67b5aadac5d7f6f9594dcd5d15cd06879da6e19\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"maxtaco\"},\"proof_type\":2,\"state\":1},\"seqno\":4,\"sig_id\":\"667062cc7300f11292079f7c4da0ed8e53e52ca484bb55ae693e7d0aa614322a0f\",\"sig_type\":2},{\"ctime\":1395867108,\"curr\":\"7d72a9f98ca4739e3939c89339d3f67966a7160da9df6215acb6a27c4a1ea292\",\"etime\":1553547108,\"prev\":\"2df5de847a546d662d3dda502aa283328f477430bfc24ce0c681ca68c4d4a200\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"keybase.io\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":24,\"sig_id\":\"3517ddd1a5c9d708f21634d126814647129a64086a967daa5017e7966b89c9820f\",\"sig_type\":2},{\"ctime\":1395868002,\"curr\":\"ea61172238d18212d53a71316dc09e8e48f2e654490262872018d09854ab26e0\",\"etime\":1553548002,\"prev\":\"7d72a9f98ca4739e3939c89339d3f67966a7160da9df6215acb6a27c4a1ea292\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"oneshallpass.com\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":25,\"sig_id\":\"93734bd81d3b50d25401d7451a136e171757990d35110b585b729dbca2019d050f\",\"sig_type\":2},{\"ctime\":1397071049,\"curr\":\"eec8f9d3a754e793301b108ddb548dc46bbefaaaec189306b169a3ef71f22abd\",\"etime\":1554751049,\"prev\":\"b6168f3a49898b4aba5c5fb162c2b5b6df87407bca7649feba5364751e693c13\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"oneshallpass.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":34,\"sig_id\":\"0e577a1475085a07ad10663400de1cd7c321d2349cf2446de112e2f2f51a928b0f\",\"sig_type\":2},{\"ctime\":1397071368,\"curr\":\"ad3efd2c8ff323f96841b482bc3cfe92195bc72fe318ce0b95cf66ee3160aa55\",\"etime\":1554751368,\"prev\":\"eec8f9d3a754e793301b108ddb548dc46bbefaaaec189306b169a3ef71f22abd\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"maxk.org\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":35,\"sig_id\":\"baacd979d2ebe325482d2921ea72df72dbf9a31aeaa69f2b14b1eccfed31c72e0f\",\"sig_type\":2},{\"ctime\":1401977613,\"curr\":\"467a473af5b7d548f1e1a1d199398f7fe3a9ea40e4c615d261e6b44a56b96642\",\"etime\":1559657613,\"prev\":\"4111d9457743da3ad15826242bac5db8bb5ebf77d6de71a41a23ecfc4ed4bcf9\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"nutflex.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":49,\"sig_id\":\"4b4c67c83b4995aae5d4a5dde916fb35762551e67e35d4d4f1be99e2309238600f\",\"sig_type\":2},{\"ctime\":1407166100,\"curr\":\"acc5ffcf17cd722b5844b24564e2f644d265e010047bfdd378919be8f0603ba3\",\"etime\":1564846100,\"prev\":\"b328cdb4904ebcc196a49c67355f3ee9239bd735d7246e90cf20c7e7307e706b\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"maxtaco\"},\"proof_type\":4,\"state\":1},\"seqno\":55,\"sig_id\":\"095d87d153e6590c2d6c032752d3c43485a1f64e61b99d58f989f6da28b2a8320f\",\"sig_type\":2},{\"ctime\":1407167610,\"curr\":\"8d906bd4d0bb2d667f5c8191d6e7616f1de1e69e950f1f2e6ea181316d8d6dbd\",\"etime\":1564847610,\"prev\":\"acc5ffcf17cd722b5844b24564e2f644d265e010047bfdd378919be8f0603ba3\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"maxtaco\"},\"proof_type\":5,\"state\":1},\"seqno\":56,\"sig_id\":\"d0f9aaff73dc67f40fb6ebb6fdf473e11de43f11ae64a5fbffb00218914e19af0f\",\"sig_type\":2},{\"ctime\":1407186953,\"curr\":\"c9d4ca19adac4680957b2ce37d38280a1a32d2a7dfe606ae7236a9c982e2c75e\",\"etime\":1564866953,\"prev\":\"8d906bd4d0bb2d667f5c8191d6e7616f1de1e69e950f1f2e6ea181316d8d6dbd\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"maxtaco\"},\"proof_type\":6,\"state\":1},\"seqno\":57,\"sig_id\":\"a9fb80f43bed684a40e48de1e365e7ace765a3309819f9f79cdb27e3dd9802f50f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"040933ff11e2a44476ef0395989349a917a0aad723e41585d1087330c4365c3e\",\"seqno\":86,\"sig_id\":\"46ec4cb327088e808d068ab8236f39dd38a6cd840e229d5f8a5d65b4c7d6a58d0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1420840192,\"expire_in\":157680000,\"prev\":\"d70f647a155416bd94ad669254e5513aa667c03ed55f047f81cd6fc9bc3ed89c\",\"seqno\":10,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1420840192,
      "etime": 1578520192,
      "rtime": null,
      "sig_status": 0,
      "prev": "d70f647a155416bd94ad669254e5513aa667c03ed55f047f81cd6fc9bc3ed89c",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 11,
      "payload_hash": "a6808c59df1a6406a961e81b7038eb132c9ab04eb827ee844663ef5a7e651e90",
      "sig_id": "cf90e4629d273b677662b511269fff5756714d2889278c4297aebe9d723ac8240f",
      "sig_id_short": "z5DkYp0nO2d2YrURJp__V1ZxTSiJJ4xCl66-",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n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\nxl07Gran/Y2t75XvfPM3px744zLn55bn/gsCjHb2\n=ofws\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":167,\"last_id_change\":1418935207,\"username\":\"chris\"},\"id\":\"23260c2ce19420f97b58d7d95b68ca00\",\"key\":{\"key_fingerprint\":\"94aa3a5bdbd40ea549cabaf9fbc07d6a97016cb3\",\"kid\":\"0101d49265e97c636dbffa62070f29e07bdee44aca90b0eaa1645cc29fe5320f8a6f0a\"},\"remote_proofs\":[{\"ctime\":1391738324,\"curr\":\"461262ff766cf5da5eb40e5db29128a453ffb939840a23c0ffcb95073c888302\",\"etime\":1549418324,\"prev\":\"34152df8ec04aa33ad1bdbfd4927ed4d64d1be2ed3c4060043f313041a3722cf\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"malgorithms\"},\"proof_type\":3,\"state\":1},\"seqno\":4,\"sig_id\":\"3e1e20a1a11f4f0ee4ea4470b694fc115f928ce0b1f9aa2951a65081da8340820f\",\"sig_type\":2},{\"ctime\":1392240892,\"curr\":\"4396ec2c10ba4c4e1d931af42ed4d65ecea9f57a858fd44e86f2950071ef871f\",\"etime\":1549920892,\"prev\":\"d2284c9ab38c28650ccf4d1feb7af9f4ee46ee760ca45711a5e3afe008e518c3\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"malgorithms\"},\"proof_type\":2,\"state\":1},\"seqno\":7,\"sig_id\":\"0d9f6b71c043f2ef80b6dea44338471edc3ef76e2cee236dd397412ad9631aa90f\",\"sig_type\":2},{\"ctime\":1395868048,\"curr\":\"45c6893a87e18f27502266c08f0d0118f5ecd3fd0bd52432b18cddbd3f6520a8\",\"etime\":1553548048,\"prev\":\"26662070e4cd5345a302af7d513bc8d08a9a21cae783c6ad18f0113b039211f5\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"keybase.io\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":82,\"sig_id\":\"58ff6251889edccbb54495ea3f7d33f57c2d18e114dda887aeab6e06552aed780f\",\"sig_type\":2},{\"ctime\":1407170371,\"curr\":\"9fe1cf15276a8c34ebe03eae24e90f4ea44e6c6afd69a96bd6a85bf1ec025780\",\"etime\":1564850371,\"prev\":\"fe90114a30fcf13d0b9af77cb4291c1d550462fa567b0ecffdf436840f0fd8da\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"bitcoyne\"},\"proof_type\":5,\"state\":1},\"seqno\":202,\"sig_id\":\"c6d2c827fd7292ae36a359fef3ad04736ff373901b3db037af773ce13db7fab90f\",\"sig_type\":2},{\"ctime\":1407767238,\"curr\":\"510fb8ebf6ea77aa220f0a4cd7d0be212f9b294161f0cb2b43b6be024aa9ec67\",\"etime\":1565447238,\"prev\":\"044e1a2cc9d956c4780d4eb7fcaae9dfdf560eb9d5095ad653d8b37f63172c7f\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"malgorithms\"},\"proof_type\":6,\"state\":1},\"seqno\":209,\"sig_id\":\"c60cae4ab2b165fd15960433b0af48c66d37f3940e56a8d3b4955991947cb3f00f\",\"sig_type\":2},{\"ctime\":1408375965,\"curr\":\"57a78d3f45c749eab6a6b2796db103c6887ccbd7827a27e3f33cf3f0950cad8a\",\"etime\":1566055965,\"prev\":\"94e9a25ab3e551d43ff700e94be3cc446785ed4f44d83d93d00fb3ddc601070e\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"chriscoyne.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":221,\"sig_id\":\"dbf530c689d2f3af60c5b11341d5eb2ad229995d6ef39dfd1661962d088a75ec0f\",\"sig_type\":2},{\"ctime\":1408376008,\"curr\":\"ba697b9daa2617a80f53d94a8204a6db5aaf2215faf1feb4a43d2a09a95c6a7c\",\"etime\":1566056008,\"prev\":\"57a78d3f45c749eab6a6b2796db103c6887ccbd7827a27e3f33cf3f0950cad8a\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"chriscoyne.com\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":222,\"sig_id\":\"7dbc723fc7c0abfe761af6729fb27603ef9be58718153329ed2d4ddd8d6ddb910f\",\"sig_type\":2},{\"ctime\":1414608934,\"curr\":\"2f7e338d5d1915975611c4d99f225986f7978c2b0a89f7efed973cabc92b6e0b\",\"etime\":1572288934,\"prev\":\"b7036840be4000a1f5a006bc8b164c535300507c103651f389290e1ac42885e9\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"malgorithms\"},\"proof_type\":4,\"state\":1},\"seqno\":255,\"sig_id\":\"1a8fa75aa365dd96c95fd229722668dfb0026973d89e489b256d87ee06a4bd670f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"d24c186ca79e522e96c4d10f407646b9c1b5a2beb35fbaa03f2d070638af65b2\",\"seqno\":299,\"sig_id\":\"677463e4cd27e7cc4382f20aa7f21882f694feebdd4ddee623786b1cd21083940f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1421363548,\"expire_in\":157680000,\"prev\":\"251d1c8b830c3c4b131a182d27188712bf85a7026755bd0ad16ec2f58c880a07\",\"seqno\":11,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1421363548,
      "etime": 1579043548,
      "rtime": null,
      "sig_status": 0,
      "prev": "251d1c8b830c3c4b131a182d27188712bf85a7026755bd0ad16ec2f58c880a07",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 12,
      "payload_hash": "ed4c85c1ebc1765450203ad9f9c092dff4f1580367add60cf5db8eb894088521",
      "sig_id": "18c9a91230de00b8ce9f42845ffb24d2b690a2b2f9ee18efd66d0f14d000fb070f",
      "sig_id_short": "GMmpEjDeALjOn0KEX_sk0raQorL57hjv1m0P",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMQNAnicbVRriFVVGJ3xlQ5IjSVaIsYho3LM/X6MlKEgUf0wMxNKrvs59zjjPddz\n76ijTRGWk1GBhSU+wqSQyNESggzULCGsFJMIC0F60FCBgT8sMrPvXKfJoTh/9tnf\nZu211re+fXT8yKaW5pV9neX+e2eS5s+OHu5uWnTgwgPrEpv5nqR9XeK60lCpF6uK\nWRGS9qQz9FhTC3em2c2rg03aklUhr6VZBUoEE4ei8sJg5lTgNjKPhaDcKc+sUE5y\nx4JySW9bAVOAxrTSEfJqnhZ3JFwZwrinXlnCHeZRuEBjjBIrKQDAeKaIpwRuLWe1\n+jA2SQOzlHrY/Z/znY0CwgjTgDgVRBhPqFAiKEmikdwKZrhViGLCtUSWOyONtNJr\n5aiGH0WgjA1BBuC6G3AMWxwlj5ZzxISnEgmkGOGGEqQwQsXBWsgHnctcVqlkORhS\nGFDPjessLAD+qasVq9SXhszEqC3pMrU6CCq5sgGXYI8RzLggFA3H7apllVowlc4C\nt0FMIS85UdxEJjUKxunIvfaSYKUlbxAbbEDh2fAmeBXBO8KRjdpbKaJHDBTiaKP0\ngtiIGdXYxWGmCqSpsxhBo6kmRjAbHAFnDDOUIbA7WIqFoyoGb63knFEhFDFGMasU\nVWAqcM/DiqweStU8yyI48ijEr56uGFLONeNtievO80IilZZSEayWQXBZtFkor4kF\nVEcExdIaLb2jAQvLtSdQw4ITUhCwTgL9MAjOpcb0Cng1D6sAXAuFEGGRUaQ1Z9Bk\nHbSAD8QLo4ziBKxByBENumW0IFkyxRwcNkAgGdJS+NvQ0ximcnCdJW/qprS8VnR5\naKo60nq52yb/7evMorEzG51twJTqPVUo0rakVjf1gj1UamFlJUvaGazSjisjYDnw\n4AgyCzyhhzo4DgRjIB5YSqqI4IGSECNG3BARVVBBSMwhvTwYjIr+FmhX7iO9SxvX\nlOom7Sp4V01PV2Z8qWxq5WL0iXNKBQiIFox6KbT2RjtGYNq0ZhBHH7CVVEMKuVba\n2kgFtwyuciGawrBBEfwqEQDHkDUM8RiopjIYqYkLoEgbYrzmgB5AHtcGIg6zjrj1\nHsPL4axTBh6jpLcYtYaEwYm76sEqrLs6YJJSLiAWa6ppHkpppRGNIgkIDSXDwL9y\nkKeIIeVIGC1wUCANURUspsRpYxELEEMZgmIMxj1EboqQ4qDRv0IxAWamAzBBb8XU\nu/OQ9H704WOjmppbmsaMHlG8w00t467753WeMHDNX7Om7WS/Xryp0rf43EPnp/75\nzdlfJsya/vE7m+tPrxs5ctOR9u3L75l3d/WFeX29Z15a+/OaU9f/3rpp0qQtO5bM\n/WT2xFf7zISBC/nst4/dsD9/ZePlhYcWXRpx7PT63Y/v2Xdp9Jtr9xx85qmpn645\n9psvTb3cv3dz85NfSXx2y6hbph9dMHb+qcq+Px6ptE75YODc/p0UzT9x0ew6+f72\nbG/rhq65L792/r3WGUd6x97+Xf8dPy7b83zPxN39Gt217IuB0wvXL3i29fUHmw7v\n37b4vpMHv2Rzfrp/ylvd/mz7mJYnug/Yvq27ftg658xz045/Xl7y/dfjb0tvXTru\n+IY3Di3aMXnX6rXbN1778Il3F9747eQZq178G3IcKcE=\n=dEMP\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":10,\"last_id_change\":1421456230,\"username\":\"olsonseank\"},\"id\":\"80d75285af4790eac9f5d9d721897500\",\"key\":{\"key_fingerprint\":\"d8f6b6250bf9db76fd04fb51fbf7d62bf14391cf\",\"kid\":\"01016093cb108d4392a64bec275fa4a340362eb316c38fedbb755436682aa84b88380a\"},\"remote_proofs\":[{\"ctime\":1421455945,\"curr\":\"837b336eb97e6572d3268d92b82ac26317ba97dc3e16b59d2d9216522edbbbc7\",\"etime\":1579135945,\"prev\":\"9680024f43099541f79e96969f146a8a852db700c2993c7fb2a67484c954a82a\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"olson-sean-k\"},\"proof_type\":3,\"state\":1},\"seqno\":4,\"sig_id\":\"b5a825025bb706fd9ec5484fe2d4c9738265e32eff105a26f8e8e671525a5ea10f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"22cc88e7d69643d7699da9c423e09940d7de1b739d9d5989bbf365b45eacefaa\",\"seqno\":5,\"sig_id\":\"7d640ba405fe3937ea792ceb709a2ad953e0e70659aac948205bdd1c15cbc8ac0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1421473356,\"expire_in\":157680000,\"prev\":\"a6808c59df1a6406a961e81b7038eb132c9ab04eb827ee844663ef5a7e651e90\",\"seqno\":12,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1421473356,
      "etime": 1579153356,
      "rtime": null,
      "sig_status": 0,
      "prev": "a6808c59df1a6406a961e81b7038eb132c9ab04eb827ee844663ef5a7e651e90",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 13,
      "payload_hash": "1421995af7068ad3bf3982028330e56b89e187a1c94aa1f5426cf61696e1478c",
      "sig_id": "19da9eba50589b026df038f1ca162d5b008a101fa102abc728d3b4a44a4747e00f",
      "sig_id_short": "GdqeulBYmwJt8DjxyhYtWwCKEB-hAqvHKNO0",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMSlAnicrVVbiFVVGJ7RdNLMMhKxyHRDBDaN634Zgx4ydKywSEspOazrnO2ZOWfm\nnD2j0zBBRkkvGhRZWqMMpd00sx5EfPASUkghJRaFJAQ9lFY+KNZQ9O8z0+iYDz3E\nedlnr7W/9X+X/1+fXD+xYWpj94ZScdeSu0njsaPLehqWH/50sD+xFd+XtPYnriMN\n5Sx/KpvOkLQmpdBnTS20pJW5a4NNmpPeUK2llTIsEUwcisoLg5lTgdvIPBaCcqc8\ns0I5yR0LyiUDzTlMDhrTcnuodlXT/IyEK0MY99QrS7jDPAoXaIxRYiUFABjPFPGU\nwKnFSi0bV01SxyykHt5eZX+pvoAwwjQgTgURxhMqlAhKkmgkt4IZbhWimHAtkeXO\nSCOt9Fo5quGPIrCMDUEG4HrqcAxbHCWPlnPEhKcSCaQY4YYSpDBC+cZaqI4qV3GV\ncrlSBUFyAbKqcaVcAqg/dbX8KfWFMTExbk46TC0DQgVXNKASvGMEa6UIkeNw16Tl\nPtiQg9ar0gzR4IGlRsJqRqKXVODIkBFS83pVo+rngo13wDoRHDaaeSYUd4jSiLkj\nxFtjKWFEemmQ0mycolZijJTJNypMpGUaS6UoDoFIYxR8qkBfbSRzNkqvlHWQCwiI\nVhojI0FRqL0aOitZKHRVK5UIcjwB2cvSzhHaSDKFFWpOXE+1Cqc6xOGtRY5HilBk\nmigThHSCgQMsUEENlKoIM4xaDXVT5mjkHMy3GlsH5YdRcC7BsRHwrmrozV0VlDjt\nIQKEY084E5FFUMAH4iOREXRhNhDtBRce5LXSQEQD0xqEM1olY1xyfet86p1UDK5U\n8CYzhTW13OKxlmpPs2KPTa5makeaS1PHKGR9XbBCm5NaZrK8dFiphe5yJWkl8JS2\nj4SfQPEgAQHDpHDeQddZQi10FaJaMw7U8mjbqBgOSkShCKZYcQVfaOskiskI2sh5\nZKB5vBNcEMnGnLBSYOuxYvCjAnHBlICIkBg80cbrgDWjxggjsbXQaRFJa4XhUojg\n4dsrnKAj4KNO/A82/0cn8nEyqnw5rK21uFpLua+nJfiepC5/Bs3bAYvFLOtqvdIS\n6HR0NVfoZa5QzHV0hjpGPIJu0pw6w2w0MGMY8lxK5DS0HHKEek4shRFiggsGW2Sc\nsP9yZXX9mEJm0o6cQJfp66gYXyiaWhGO81Z4ZDDMAOMB0UXtGJPEMhO1IpRAW+d+\nGaOFh6MxBw6CEYE8i4i7XLlREvIyEhyqZjxoC32NIPAseoYFV9EFL+AkyJkJ0gjM\nuVKcAV3LYVzkbhAkBFiYDOSjr05hdAJedoHk0l1KGsnNVwLysa4rrYZCWq5nRChU\nl3s0IsHDVQN3RbAOS8EZRwRRyB0QRhr6FVoXc5jqEErvBXKRe6uChRGGoESCLxHF\nYFdm2gET+JZN1lMNycDhg09e09A4tWHypAn5vdgwdcqN/9yW33zd9OfsZSc+Wnhu\n4dJXhtKh6+YcX/Lgw3tX7Hx/i+ufuXtz/7Y9Nw33tr45f92JhyY2vds29yVz1+M7\nu+2SzXu2bl/wQvx5/X3PT/9t5oIflv7YO2vlycFFpWNixSPL27ZtL71z9tfu2wcX\nvPz0yROHLhydsvoD7p57asZrG5uWvd0y+7sPC4e+Onhmy+vTVr03Ye/p9Yv3DQ+u\nmnPg91vWzMOLvmi+Z96FO7cfmLNpx5ns0dPz93dM3r3p4ucTe88uXnnvbZNmvHHt\n/lk7OndNb/t2cHjGefP9x037yS/PnNpzvrWzePzmixMee5XuaywsPzVh2sY/yl3k\nyE/tf20YttNunf/ZA0ND8v5nV5Mv5dbSDefaVh2Z9NYdL/4NNTaZww==\n=nhwa\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":11,\"last_id_change\":1421988227,\"username\":\"jinyang\"},\"id\":\"9403ede05906b942fd7361f40a679500\",\"key\":{\"key_fingerprint\":\"bc6ec1a94d4685c033f15c22dbab32427d7a0894\",\"kid\":\"0101b71108a15c28127b49178831ee27aa8bab875b9a74cbf7d88bcd168e598910a70a\"},\"remote_proofs\":[{\"ctime\":1420748180,\"curr\":\"c05142b0c5f300f4928ae67c640604e363a27d824a43b924234c3f5513eb91bc\",\"etime\":1578428180,\"prev\":\"4632c9d0b5251d2546f4f2dbde2df27f85c4be29d656d42fb7aff7e4991a9a98\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"jinyangli\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"22c94e32ab376cdcc75b23bd3d039945c9dca7abf841e86f6821318582ab9bc70f\",\"sig_type\":2},{\"ctime\":1420756274,\"curr\":\"b761bd184848360564867a02fed29ad9e1943aa6a71bb72ff07bb6a5766ed761\",\"etime\":1578436274,\"prev\":\"c05142b0c5f300f4928ae67c640604e363a27d824a43b924234c3f5513eb91bc\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"news.cs.nyu.edu\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":3,\"sig_id\":\"3159fca3c42d0dba953ca4bfaa7b40d5770c96850c23d52b3320aecea1b0ac6b0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"db6d0a1059adc23cf9c4472b4af982323f14836aa96d7701510064260d4f05cc\",\"seqno\":7,\"sig_id\":\"595345e9b27a0e494fd41658fced69adb23ae7a61558854ca3b5a083eb9206600f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422051486,\"expire_in\":157680000,\"prev\":\"ed4c85c1ebc1765450203ad9f9c092dff4f1580367add60cf5db8eb894088521\",\"seqno\":13,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1422051486,
      "etime": 1579731486,
      "rtime": null,
      "sig_status": 0,
      "prev": "ed4c85c1ebc1765450203ad9f9c092dff4f1580367add60cf5db8eb894088521",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 14,
      "payload_hash": "1d21eed55eeb085102bf28673e316156187760fc5836ebc0af9aa579ba5e9b7f",
      "sig_id": "c7b288e398279c6d0de3d9accfcbd93dc601a0b51460b92bd965523a7fc246390f",
      "sig_id_short": "x7KI45gnnG0N49msz8vZPcYBoLUUYLkr2WVS",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n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\nS3d98rr6+daBpx5901rrkf8B5q5j+A==\n=997G\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":54,\"last_id_change\":1421371756,\"username\":\"patrick\"},\"id\":\"1563ec26dc20fd162a4f783551141200\",\"key\":{\"key_fingerprint\":\"246201479f89c599c08f29b3fa706aef3d045f20\",\"kid\":\"01017c13682b0965e718feaead9f04e468086c3864ba8294072c97594a5957a99f6f0a\"},\"remote_proofs\":[{\"ctime\":1392412788,\"curr\":\"c20d2386bb1714908f0de84f76d9aabc7c2911a1bc6ee451f98bb8833a528fac\",\"etime\":1550092788,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"patrickxb\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"fff3da1be82b15fa7a5633f85d2ae19e4f128986e6e80e64a4859dedb56f47080f\",\"sig_type\":2},{\"ctime\":1392412864,\"curr\":\"e4e726145a1e148ed4bb2c8481d6a6f399d0e65d04af5e590be98c2ca2f23705\",\"etime\":1550092864,\"prev\":\"c20d2386bb1714908f0de84f76d9aabc7c2911a1bc6ee451f98bb8833a528fac\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"patrickxb\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"81a9003a098ec52cd98a9f4983ad7edea81e9beae23064ceb97518138e3cd0f50f\",\"sig_type\":2},{\"ctime\":1395889480,\"curr\":\"7a1d453f123e38a125f0e872c6ec4ef0ba80119008996742abfad2675a16110e\",\"etime\":1553569480,\"prev\":\"bff52a25614ccd76fafabd54adbede2a2832899a0c0fe324a433de6100845331\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"patrickcrosby.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":11,\"sig_id\":\"8e0043c22d467a8b966c528c17882118c23bec1b7803fd4b5430c500e04352e20f\",\"sig_type\":2},{\"ctime\":1420864565,\"curr\":\"171a9d6e5b305d8eaf72f78e75a4bc814bbc2c5e5a429724640f0727212c0cca\",\"etime\":1578544565,\"prev\":\"eae03042fbc56ad02ffd97e0d07b43a5d9c4f4959a03bdc3a5d1659d0f28797d\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"patrickxb\"},\"proof_type\":6,\"state\":1},\"seqno\":23,\"sig_id\":\"9127676da00562183585508aaf5b5fff4e936f208fc238e388d37a40eb99b3fa0f\",\"sig_type\":2},{\"ctime\":1420864719,\"curr\":\"db4612f774b4e4689b5b8d9ab9c99447d30bce0a969a131d65fa2810b352cc44\",\"etime\":1578544719,\"prev\":\"171a9d6e5b305d8eaf72f78e75a4bc814bbc2c5e5a429724640f0727212c0cca\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"stathat.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":24,\"sig_id\":\"250861cbd50c104723977c779c4a7ab7919d664dabf0228a61fc65d24dd4001d0f\",\"sig_type\":2},{\"ctime\":1420864917,\"curr\":\"6c1866e5442084ff8b6892ca7a8cf2c5555e5c6281553331593e3d8efb911ff7\",\"etime\":1578544917,\"prev\":\"db4612f774b4e4689b5b8d9ab9c99447d30bce0a969a131d65fa2810b352cc44\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"numerotron.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":25,\"sig_id\":\"3c666d4cf57d5f1b11a556def8d0f78e89ec92b994867a4ce0245f5bd187355b0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"bfdae4f98d9910608c2760cf87fb6e3e1ea3527fa57ef2221e4d6e5cafd25cb3\",\"seqno\":31,\"sig_id\":\"baeb664889dc7f1e203bffe3cb364249a55661f45f9c84e3a470bbde36e34f360f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422051540,\"expire_in\":157680000,\"prev\":\"1421995af7068ad3bf3982028330e56b89e187a1c94aa1f5426cf61696e1478c\",\"seqno\":14,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1422051540,
      "etime": 1579731540,
      "rtime": null,
      "sig_status": 0,
      "prev": "1421995af7068ad3bf3982028330e56b89e187a1c94aa1f5426cf61696e1478c",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 15,
      "payload_hash": "185e2496a0dc012deea1c883759a74e0f7c33e38e11a40530aac76cd1916c7f5",
      "sig_id": "de2c80dde617e34b0715fd082b593a2b7eff7d4915c2095e4f11e92871d92d9f0f",
      "sig_id_short": "3iyA3eYX40sHFf0IK1k6K37_fUkVwgleTxHp",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n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\nv9Ii8F4=\n=Az9U\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":45,\"last_id_change\":1421805379,\"username\":\"gabrielh\"},\"id\":\"b7c2eaddcced7727bcb229751d91e800\",\"key\":{\"key_fingerprint\":\"3f6e504945ae44e6a53d49845431890fe1eebcde\",\"kid\":\"0101230aa5a191471449cc05ee95ce3637ebb16e409bb3b92ac0a8cc28eef92c0b500a\"},\"remote_proofs\":[{\"ctime\":1413405722,\"curr\":\"55d1e2013c91f4ef410538cb03cadfcd0b40e86dfa8a0594a42036eedfd61fff\",\"etime\":1571085722,\"prev\":\"461f0051ad08190465b5c8f4cab2fb8c5f04290c3f3cbd53794e7dbe5b89d448\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"gabrlh\"},\"proof_type\":2,\"state\":1},\"seqno\":19,\"sig_id\":\"dbbb07105a43c18ab1a9d01b564c591a6d4d57e35dd4ba3f1a0ed8e33f8517eb0f\",\"sig_type\":2},{\"ctime\":1413405744,\"curr\":\"5381de2b714095bcf65f41c1b235c11ab6fb87cea8b2c9a3bcb95095ab26e191\",\"etime\":1571085744,\"prev\":\"55d1e2013c91f4ef410538cb03cadfcd0b40e86dfa8a0594a42036eedfd61fff\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"gabriel\"},\"proof_type\":3,\"state\":1},\"seqno\":20,\"sig_id\":\"916bccc5e1fd7e784340cca2795d1ae352922bd498ec196bbff6c00b7a304d2d0f\",\"sig_type\":2},{\"ctime\":1413406032,\"curr\":\"eba1b91ed0048983d7e32efeee7fecce16547998ff3e05b67045a751ef99be1c\",\"etime\":1571086032,\"prev\":\"5381de2b714095bcf65f41c1b235c11ab6fb87cea8b2c9a3bcb95095ab26e191\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"gabrielh\"},\"proof_type\":5,\"state\":1},\"seqno\":21,\"sig_id\":\"2c9e345b5acedbcfe3d5d9b2d6115b59e9d6ad31ad262bb07266ccc6db0b95940f\",\"sig_type\":2},{\"ctime\":1421558613,\"curr\":\"fdd474158530c9816ca9707c3cda809aaa24a0f69b5bcc06f5776a54cea54fff\",\"etime\":1579238613,\"prev\":\"3f0c4b94e2c2d6435aa583012793a77222c3ec7b2ab48ff95359df23fa5f422c\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"rel.me\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":25,\"sig_id\":\"fdee6f84bdd854d055fe51be696b175e5f2d2d6b143bc09a52f7bad57dfe05160f\",\"sig_type\":2},{\"ctime\":1421558741,\"curr\":\"158668859ea086ccd02ce16724f2b35a6a71834f98867429cd1210a075346443\",\"etime\":1579238741,\"prev\":\"fdd474158530c9816ca9707c3cda809aaa24a0f69b5bcc06f5776a54cea54fff\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"rel.me\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":26,\"sig_id\":\"9c8edb92c68188a43266c7037e1148c9320a969b248b97059aeca65ddfc39e100f\",\"sig_type\":2},{\"ctime\":1421805307,\"curr\":\"a0a74b05db02a6c66c9cf2ffd3754449ff680cd4fb61f80cba2038a23e7c0bfb\",\"etime\":1579485307,\"prev\":\"a71816d134d54dae20073688eea5753fd756903173b0461213aa3956a3c66de5\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"gabrielhn\"},\"proof_type\":6,\"state\":1},\"seqno\":28,\"sig_id\":\"2787406a0b57a2eb7de822c5ecea6bdcd28a3ed96191a27d46459d747a6b292c0f\",\"sig_type\":2},{\"ctime\":1421805358,\"curr\":\"b8e2fb00585d67ee5144b37483a94e251444144c9dcd5949ca30bf275bafe59f\",\"etime\":1579485358,\"prev\":\"a0a74b05db02a6c66c9cf2ffd3754449ff680cd4fb61f80cba2038a23e7c0bfb\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"gabrlh\"},\"proof_type\":4,\"state\":1},\"seqno\":29,\"sig_id\":\"a3a73a75b6941ff0aeabdededc9640d27a45cff49f2a9b57538b4608a45ba6590f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"b8e2fb00585d67ee5144b37483a94e251444144c9dcd5949ca30bf275bafe59f\",\"seqno\":29,\"sig_id\":\"a3a73a75b6941ff0aeabdededc9640d27a45cff49f2a9b57538b4608a45ba6590f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422051554,\"expire_in\":157680000,\"prev\":\"1d21eed55eeb085102bf28673e316156187760fc5836ebc0af9aa579ba5e9b7f\",\"seqno\":15,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1422051554,
      "etime": 1579731554,
      "rtime": null,
      "sig_status": 0,
      "prev": "1d21eed55eeb085102bf28673e316156187760fc5836ebc0af9aa579ba5e9b7f",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 16,
      "payload_hash": "e11940eb12b36e89eadfd73af91558819262952825faca0af1ccc744fb896a6a",
      "sig_id": "08fa6ab7f3d57852666fc4cc2ccf946557970be7bccd64cf6894d2940d90e53d0f",
      "sig_id_short": "CPpqt_PVeFJmb8TMLM-UZVeXC-e8zWTPaJTS",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowGtVVuIVWUYPaNpGgk+GNgN7ASRzaX/fjHL64NERmWQ3Tz91zl7LmcfzzkzOdpA\nRhEVSpRPhmZoD0USYoYXBi1ItJsGFURaFEl5gwxJQm369p5pmEEfEjpPm/N/e/3r\nW2t93351yvjCpKblL3aWty5qJU2fnbCFhz/+Vq8q2tT3FWetKrquJFQa2VPFdIfi\nrGJn6LOmHtqSdEYl9aGtoz5juKal2Btq9SStQBVqk2202N+SlWcvx6TSHmrVWpJh\nFbkyhHFPvbKEO8yjcIHGGCVWUlihjGeKeEoAspzWG2NuLeaYpcTDv3n9fKHmLYT6\nhXl9T37AsMVR8mg5R0x4KpFAihFuKEEKI5QV1kNtuKXUpZVKWhMiZ9wdap1doVRL\n07xt10iyIswIQZwpqoCTqZfhNaKkshS56IVhRivNGUPBWapJDJFQzyUmkQaCcAhO\nMsGww5ZF5yNVxJIorceYRUkwtoRGbKK22vgIgDoI77VDkjpjNGXKGCWR9FEzAhiC\nY8mhiXpYXkmBHNWSU+DeqBnXmbEGsRJXz54SXxqxBdOWYpepN0C9kisbsCRvDGst\nqFBjNemqp5V6MJXOTJOxImihW4q5zgp5yYniJjKpUTBOR+61h44UMMp1Hg5A5tnY\nEHgVwWvCkY3aWymiRwwMw9FG6QWxETOqsYsZRn4ZwggLpKmzGCkPh8QIZoMjYDQY\nQBmiggRLsXBUxeCtlZwzKoQioB6zCsxDJmunFrrTRihVweMIIj0+2mXMONeMQ889\ntVrWIpWWUhGslkFwmcVSKK+JBVRHBMXSGi29owELy7UncIYFJyQjYJ0E+mEYnEuN\n6RD4MINMlZxFnrRycJ0lbxqm1FHP7BqZufakUe6xxUsNas0cas0tymFKjb4qHILN\n9YZpZHf2j4SEwVPSPjQ4lhsFyhMOGiFQXgcH2WYxkOOeOS0hn4IHSkKMGHFDRFRB\nBSExhxniwWCU2ZLBDV1I+lvGaAiBwpyNaBjBHIYIgeDAjFNhA/c+S4xGMJEOvJTK\nw1vKWi24l1oI4pFmAgMZo8RYDYUcAr8yDV2aVLIVchkVR2I+WkN+OQ3lKA2ZBnuR\ncdwKaTkxNID9lsJCo9FBLK3SzED4FCw3LYITmFJtYNBBQMihuETCJ/NrSg2TdGW8\nq6avKzW+NLxt/gcN/1MT+2ZccRv92ebJmxheQKO+BPjy2yOsqCa1UEoquaVCIfhl\n+odeAMGKBwJ1BnmHMPEBAudgeiXXRrKAonSUBqoCxoYhTpExTgrnsYbRl3H0YhTA\nzLQDJnRcMY2eWij2v9R001WFpkmFiRPGZd+6wjWTp/77Afz0+asHO9ae+vHm/o5f\n1qzdqH9vffDNu2bPWaR2pJM3L9ozezF+qof2Du7a/uEN+sC2R1f/dvrkpu+++WPl\n4Y3b1DN41Xn7xvHTT6y/eOvEo+u+Otf79bKlZ+bsf2ja0bffXT3ur7OquG/dETew\nTC4pP3fgpw33Frae+fWRO5dunDD/gw175u2+3x2qbe/e23awOvDYwMy/F7i3Zurm\nu6dM9ys+Gb9307Xd6nxHYUeTOtn85cs/XEi/WP50877Wc5/fNnDdO/rG9Rc2d0zf\neeiBbfsX306X7D92ik21r63ZufT6uR+tPjhhy9xnX5g6f8HhOy7esnLa+68vXtb8\n8xl5T/m9P1/ZW921+/sjxw6fRbPvG2g61zI4GE78Aw==\n=mZ4m\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1422054838,\"hash\":\"2878b30cfd6a4a9895440ecb392fef23d5712f3e201eec74641c1b4fcdf382b2f7bd114f7211b23f1af9b9adffd69e6dd9c073caa9348aa8707df9421b465175\",\"seqno\":139753},\"track\":{\"basics\":{\"id_version\":13,\"last_id_change\":1421996368,\"username\":\"olsonseank\"},\"ctime\":1422054969,\"id\":\"80d75285af4790eac9f5d9d721897500\",\"key\":{\"key_fingerprint\":\"d8f6b6250bf9db76fd04fb51fbf7d62bf14391cf\",\"kid\":\"01016093cb108d4392a64bec275fa4a340362eb316c38fedbb755436682aa84b88380a\"},\"remote_proofs\":[{\"ctime\":1421455945,\"curr\":\"837b336eb97e6572d3268d92b82ac26317ba97dc3e16b59d2d9216522edbbbc7\",\"etime\":1579135945,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"olson-sean-k\"},\"proof_type\":3,\"state\":1},\"seqno\":4,\"sig_id\":\"b5a825025bb706fd9ec5484fe2d4c9738265e32eff105a26f8e8e671525a5ea10f\",\"sig_type\":2},{\"ctime\":1421996154,\"curr\":\"fa344022969f7136be5dd479090a32cd4378d2198bb965d79662d094614c9a86\",\"etime\":1579676154,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"olsonseank\"},\"proof_type\":5,\"state\":1},\"seqno\":7,\"sig_id\":\"49bbc0ac5b67b52a3eb59b325c3fcbecb894a66886ce96ec61339ab23a5e80a60f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"fa344022969f7136be5dd479090a32cd4378d2198bb965d79662d094614c9a86\",\"seqno\":7,\"sig_id\":\"49bbc0ac5b67b52a3eb59b325c3fcbecb894a66886ce96ec61339ab23a5e80a60f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422054969,\"expire_in\":157680000,\"prev\":\"185e2496a0dc012deea1c883759a74e0f7c33e38e11a40530aac76cd1916c7f5\",\"seqno\":16,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1422054969,
      "etime": 1579734969,
      "rtime": null,
      "sig_status": 0,
      "prev": "185e2496a0dc012deea1c883759a74e0f7c33e38e11a40530aac76cd1916c7f5",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 17,
      "payload_hash": "c49c8c4bef4862d13da267f19416f3edc650159b18f25abfe8a0f3c5417159e2",
      "sig_id": "8afe41e09f1bb4b99765f8fc97ad11aad3d2de26c110d5b858628db443decf600f",
      "sig_id_short": "iv5B4J8btLmXZfj8l60RqtPS3ibBENW4WGKN",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMIXAnicbVJbSBVBGD5aiimRRVeDiK1AymJmdueyp16OZUYiBdaDkMns7qxtJ3eP\nZ4+WliFU9FJhHeqlwiAzMizRStColKLIIqXLQy8ZEVqYdBGrhy6zVm/Ny/B///d9\n/8zHf3f6lFBmStXh6I62DStRSn/v6urQlnvv/b2K4Vm1SnivEhWTl+24FSIeiztu\nQgkrmHGkYUu1mIGwCbFNTKHatk0ho8QgjFsaQ5aKlDxlh+cHCmljcF+scjyJyaLc\nsST6H350sgEggKoAWCWIcAuphBHBKLI5xQbRODYYUCHCOgUGNjnl1KCWzkxVlwVD\nsg05AlzaVU/aadCANsW2gTHQiKVSQADTEOYqAgwCEBB9EXd5pZBsz/Rc14sToir1\neYrEaxxTBCFYXiV3XMnYyc2o9+cvsbiXkIJdErVcPxAkamOBy25hlP/VlhuOa8kA\nJb1GxH3Hkx5QMs2EEwyEGkIIE4pxniL2xJy4KA+mQEwJA/IEQ0SNtBQQ6hoQBkSG\nKvPQBbdsi6rc1iHGjEEdEaRjJAOwuckBt6FpmlTTbIPphJMgD19UuZ70pvKdvEJ6\n+k6FyxPVcaHU993ZNjWUkhlKT0sNdiCUOS3732b0nMsIXfuU3ZpfNrBWtQqalTWf\nmlakHr31bLyl43gSJAc/tnWlNXe2v1pwe+BUyZcZpb/yu78tSj/biJu/Nik/IpXn\nu3+ODNWNL9566XWscHvR5YWbI8cKk7MbNl4fpjlzJ2Yv35/VfyK1r7X+frL/wb2b\ng1UTnaONXQfG2ExcMPYhq+GKvQsDOON138uHkRv2nPCbh/6+OYUt4UfhK+klRWrv\nuo+Rk6Uds0JyDzpOd3w4WDwUKsjyvnubiusOPW1PvDiybMqlRPLz456RocS03uGM\ncO5YfmRe1ysr7dvohXd33hqbcrau/1wy+vZ52Zkl+5bWLTPrL169OL+2ZSw3M+sJ\nPbOxu/E3EGAmLQ==\n=QKaN\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"service\":{\"domain\":\"jacko.io\",\"protocol\":\"dns\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1422256755,\"expire_in\":157680000,\"prev\":\"e11940eb12b36e89eadfd73af91558819262952825faca0af1ccc744fb896a6a\",\"seqno\":17,\"tag\":\"signature\"}",
      "sig_type": 2,
      "ctime": 1422256755,
      "etime": 1579936755,
      "rtime": null,
      "sig_status": 0,
      "prev": "e11940eb12b36e89eadfd73af91558819262952825faca0af1ccc744fb896a6a",
      "proof_id": "c8f453c788deabc4839f5210",
      "proof_type": 1001,
      "proof_text_check": "keybase-site-verification=iv5B4J8btLmXZfj8l60RqtPS3ibBENW4WGKNtEPez2A",
      "proof_text_full": "keybase-site-verification=iv5B4J8btLmXZfj8l60RqtPS3ibBENW4WGKNtEPez2A",
      "check_data_json": "{\"domain\":\"jacko.io\",\"protocol\":\"dns\"}",
      "remote_id": "dns://jacko.io",
      "api_url": "dns://jacko.io",
      "human_url": "dns://jacko.io",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 737,
      "hard_fail_count": 0,
      "last_check": 1434480198,
      "last_success": 1434480198,
      "version": 2,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 18,
      "payload_hash": "88d873d0340dfc5aa391dcfbe4a07cce979d8805c910b535af043c8e17579119",
      "sig_id": "e8ea962435235ba18daa87ad739bb59415706dbf82be6663e18e02529b9c38320f",
      "sig_id_short": "6OqWJDUjW6GNqoetc5u1lBVwbb-CvmZj4Y4C",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMSGAnicrVVbiFVVGB5TsyZHjHroJZQtGugU634ZTYquWJpCxNRkp3Wdsztzzhn3\n2aNO42BB2oASWtSTD1oYlWVgvQRJWd5ActCiUEyIkB4iqSDNkuzfZ6ZpxiQIett7\nrbW/9X+X/9/72ya3tE5a9Xyl/M79t5JJRw4u72t5+FjbxYHE1n1/0jGQuJ401PLi\nqWaqIelIKqHfmka4La3PXhNs0p6sDlkjrddgi2DiUFReGMycCtxG5rEQlDvlmRXK\nSe5YUC4ZbC9gCtCY1rpD1pulxR0JV4Yw7qlXlnCHeRQu0BijxEoKADCeKeIpgVvL\n9UY+oZqkiVlKPaxe4XyluYEwwjQgTgURxhMqlAhKkmgkt4IZbhWimHAtkeXOSCOt\n9Fo5quFFEdjGhiADcH1NOIYtjpJHyzliwlOJBFKMcEMJUhih4mAjZKPK1V29Vqtn\nIEghQJ4ZVykkgPpT1yieUl8aE1O3Jz2mkQOfkisbECnpwIwQyRTjZAJsqKY9/VVX\nNWX4DoCblRGjsNTNQiKRhIaoIgIfLDwh16xs1IFCtIku4KCRidSzGD0HLSxjyggQ\ngoK9HvgxFzS1cYKqBgfnqbYCa+YwHLZBgnpOI4kQskxJrpgNkXjEhYeYYEOtoFpT\nLKIArUxRexaq9TyUerN6PYIkXZC/PK2OcZeCAHfXl2UFb6YQCp5RjgJjjEvhMHXS\nMGKRM1wyjh2zPiIiJOImaIJRDMEwZbXwQkD5YRScK8TICHhvFlYXCdKcaM9YIBRZ\n4RGxkGAonGhjpFZaS6kk5FsIpjEOAVFOLLM0CsEpNUXiRrkU+jb5NLupHFyl5E1u\nSk81CpvH2srV01qR5ORfvW0ClfL+Xtjk7UkjN3lRP+w0wqpaPekABo20e6QLgCi1\n0HyQTM4sFi56Y7jXUVqMLHRp4I5CzwrGI2IRXI1eQHBi0JFYA2olI2gj95HB9svs\ngCioMTskioyHKAkzAWSKnCsP7nvMYE1bFZxWEsQzxMAIEBQ+D1YRjQiWRih+uR0j\n4KN2/A9e/zc78jVpnodsohujRoQsbVQvd4NcyQ06zg1jHfcMhEBgBg1cR+vAA2IQ\nRQraQWBho7EghcEKpAlEcGBDLcTewOo/3FjZvKaUm7SnKLzX9PfUjS+VTaMM18UI\nXYkCIoFia4JBXEpOLcQZpJMwlwkSIB6m0mEEiS5OUB2cgGxjrVkyRoKNIyECZUVP\ne3AQZqsTmgbKsQ0UWlkSL7jH0cqoZGSE+oCDD5GzAPdYa5gBEoPF7GtSGB2B4/4g\nhXTjEqaVgJRALtb2plkopcUJsB6igNBYNBzTTrlitDAliMfUG5AtwhiCwUKDd4Ij\nzLXFKsJgtjEoqII6zrCE5UD+Jorhqtx0AybwrZm8LwvJ4CcfPz6lZVJry9VTryp+\njC2t187863f5bcu0S+b1Z5P193zwyE+L2ru/XndJLX5r09YHbpcPzjr24sUNF15Y\nXjlZP/9Etvrsh8cXrliwaO8Bu7N1/V07liy+6dG2qUuOz1k594ste7c/tHMb2br0\n3uV37pr/9onrrvlteM8Nv+7b3bXrUH3h/i/XHejccfjJb2auXzbn3elH/tgx15+v\nVLoGZs/Smzsv/Hz9d3vmLxv6bJve1XnLPLdh3UuCts0+03Hu6CubLvxee/q+T6cf\n+vFUfvqjjafDmf43Zmz39R+G1v7y2InDN++7cXjDgnlL+cCJrs5NSW1O9c1Tw++n\nKw4ffa805diWu4fPPfcav3jH5OTzkxvZtDPlZw7OODhkNie9L3916tWS3P79wNmz\nuxcP/Qm5OaOL\n=HJ4a\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":9,\"last_id_change\":1422748452,\"username\":\"emilymcmahon\"},\"id\":\"2a81795a320f2723ef8f05c8b3ef0c00\",\"key\":{\"key_fingerprint\":\"1e90af3d4ffd5597b448a67d932c0d8424ce93bf\",\"kid\":\"0101a1ecd39b6194c17d9be77b7c907000b487584bef2d056dc0f1a3b6399316f6100a\"},\"remote_proofs\":[{\"ctime\":1422747622,\"curr\":\"e4800ed4350e444576c13c7a42b0ca57451c4bdf026705ae9210feea48b96d66\",\"etime\":1580427622,\"prev\":\"79529d44e230b6d02bd4b56d29aa798997787bf4664911ee0352b4b3f66533a2\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"emilymcmahon\"},\"proof_type\":5,\"state\":1},\"seqno\":2,\"sig_id\":\"48b3b35c46d54b16cfdaa5d9f7b10bd6ae5c3212645f04f4cefd62a8fe9f2ba10f\",\"sig_type\":2},{\"ctime\":1422748428,\"curr\":\"70f45ef724aed29f558dc90d14f729b8ec987aa7a2ae8c63748eb8290217a685\",\"etime\":1580428428,\"prev\":\"e4800ed4350e444576c13c7a42b0ca57451c4bdf026705ae9210feea48b96d66\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"mcmahonerism\"},\"proof_type\":2,\"state\":1},\"seqno\":3,\"sig_id\":\"abc5d49b8054b3e59fbcd6a2a03080f1616bfaba68a188ebe2652673b747aa680f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"ff0100e02e31baea057753b2306c174d12062b0137c10798ea0539ec68991994\",\"seqno\":4,\"sig_id\":\"6e347c90d8c6d48c693e351be363972d65d1fb7f87f423de1edef54e013bba4a0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422986748,\"expire_in\":157680000,\"prev\":\"c49c8c4bef4862d13da267f19416f3edc650159b18f25abfe8a0f3c5417159e2\",\"seqno\":18,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1422986748,
      "etime": 1580666748,
      "rtime": null,
      "sig_status": 0,
      "prev": "c49c8c4bef4862d13da267f19416f3edc650159b18f25abfe8a0f3c5417159e2",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 19,
      "payload_hash": "abf4894d0e34671e497ab1f2982dcb24c8e8974d10a00576eeb81ef236d75da2",
      "sig_id": "e7ee174db64a5f37d2ce5765367b8bd073aa81627356b9aec84af7b0136ff6380f",
      "sig_id_short": "5-4XTbZKXzfSzldlNnuL0HOqgWJzVrmuyEr3",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.4\nComment: https://keybase.io/crypto\n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\nk5E=\n=vfBK\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":19,\"last_id_change\":1419475392,\"username\":\"willwilson\"},\"id\":\"dc1944ea9df3dcddecbf994031a89e00\",\"key\":{\"key_fingerprint\":\"dd613e6f6261ad1ea6a72c3d32c8cec143040cb1\",\"kid\":\"010180b3d73a72827897132d4cc422ec07e5786db6212b6a9ddacad48b767a1c88610a\"},\"remote_proofs\":[{\"ctime\":1419468116,\"curr\":\"c4b068dc645df3ac1352e1bb5647f37633fecd0024593e716f98cb2d4832fcdf\",\"etime\":1577148116,\"prev\":\"b8da9a936fa9489b58943c64c170e095209f58a750da769bfa821b24e42a8422\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"williamaugustwilson.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":5,\"sig_id\":\"5f576a0731d37e3e025f4170d84ef0cb8f54709d2d3882bb247cbfc8db446cf90f\",\"sig_type\":2},{\"ctime\":1419468293,\"curr\":\"b7c1e5f80170b827f93cdea9831bcd6612b9b4c1997310d7b7cbe0ab3f6b154c\",\"etime\":1577148293,\"prev\":\"c4b068dc645df3ac1352e1bb5647f37633fecd0024593e716f98cb2d4832fcdf\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"wwilson\"},\"proof_type\":3,\"state\":1},\"seqno\":6,\"sig_id\":\"41a916d2bc13a407b22d71271e4b85d624d0ca61110198eae7f9afd8694731e10f\",\"sig_type\":2},{\"ctime\":1419468468,\"curr\":\"00ed23721676c981446cbfe1d8fecb5e08af42c562c27b6502e7dc0686a9f738\",\"etime\":1577148468,\"prev\":\"b7c1e5f80170b827f93cdea9831bcd6612b9b4c1997310d7b7cbe0ab3f6b154c\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"willwilsonFDB\"},\"proof_type\":2,\"state\":1},\"seqno\":7,\"sig_id\":\"bafbd853dd34d2398538d9015f3ea2d596dfbd191e7618cb1a5ad35535200d450f\",\"sig_type\":2},{\"ctime\":1419469445,\"curr\":\"8a1a8927a662c0f0ecd438b7c4c5de1ef91ad48c93382bda5467c8c30af328fe\",\"etime\":1577149445,\"prev\":\"6304bb04a2777bd09fbdd8556e0434d6b4c77c0d1247c5b017278547fd886f12\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"willwilson\"},\"proof_type\":5,\"state\":1},\"seqno\":10,\"sig_id\":\"ded5deb64539634e097a17d666d1a7e637e26470b7861c482a3273c63728bfb10f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"8a1a8927a662c0f0ecd438b7c4c5de1ef91ad48c93382bda5467c8c30af328fe\",\"seqno\":10,\"sig_id\":\"ded5deb64539634e097a17d666d1a7e637e26470b7861c482a3273c63728bfb10f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424279037,\"expire_in\":157680000,\"prev\":\"88d873d0340dfc5aa391dcfbe4a07cce979d8805c910b535af043c8e17579119\",\"seqno\":19,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1424279037,
      "etime": 1581959037,
      "rtime": null,
      "sig_status": 0,
      "prev": "88d873d0340dfc5aa391dcfbe4a07cce979d8805c910b535af043c8e17579119",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 20,
      "payload_hash": "64868e2b2a3dac78b9040722ab3217be565b887be8282c34a52842de6ae55035",
      "sig_id": "3c756a399e14d984ca1c36e1414a12c1d3d1b6cb46451d36ca8e21068d8ecc5d0f",
      "sig_id_short": "PHVqOZ4U2YTKHDbhQUoSwdPRtstGRR02yo4h",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.4\nComment: https://keybase.io/crypto\n\nyMPYAnicfVRbiFVVGD6jpTURZmWFFdkuinLSdb/4YOGLUl7KTMms07rO2Z7x7HGf\nPdogk6ZUD6JBBaVBhFAo1ERgWApOFy+R0kUMSsoUCisssCwsRfv3cTIHov2wWWev\ndb7/u/z/2nHp8Ep72+Kn67U3pt1B2vbsKHoqc46ePrUssZnvTSYtS1xXGhpFuWqY\nRSGZlNRDrzXNMCHNxi0NNulIloS8mWYN2NJUC8GNDU5qJ7lWQvNAmVFca424JTp6\nZZxI+jpKmBI0po3OkHfnaVkj4coQxj31yhLuMI/CBRpjlFhJYYUyniniKYGqtaxZ\nDGGTtDCrqYev/3G+3tpAGGEaEKeCCOMJFUoEJUk0klvBDLcKUUy4lshyZ6SRVnqt\nHNXwQxHYxoYgA3A9LTiGLY6SR8s5YsJTiQRSjHBDCVIYofJgM+SDzmUuazSyXAha\nGlDkxtVLC4B/6prlKvXVc2ayjqTLNAvQU3U1AyYlkzAjTOGSxxDYAMes6V1o6qHE\nbRGjjAevTTAIOYV1qQxsDTQQBC+sk3MBlJ4NDQFLEoIxzAZig7deRyogxWCsBOuo\nddiAXZQPMZUDNR+FpMLAOyIGQZLADefaBeNxjNYIaQkCdyT3BkyNhEUCDBkUiWAq\ncM/DoqwI1e48yyI48hC0X5Eu+lc6Z6ojcT15DlWtDsECMnNBOgoqlZTBWsUgVkod\nQip6SSimzkRpovcUK4S11UghyEkC/TAIzhVh+iz4IIPSlRaL1gjUgqtXvSlMdWGz\nzObcLHSmRa3HJv8XRwulWvR2wx7tSJqFKcqSsNMMixsZLGGVdlYH+0l5LIzxPjpu\nbYxCEeU50QJTHyz1qJRnEXYOeU+g6wzy0KgOGhZ77lFMzqKdrUf6Hm6VqRYm7Spp\nd5versz4as00a2VwHLMI0UmLvNUMCUc0ch7hAAWFx5SCu4oiiwgiOjDKmEGUuOCi\nd8aWHTAogpwngiPKtGPgO5aUoWiEhqHTESRgLLAVkpNgudWWB4YjxgEzHokKVjJN\nJAcRfeV8tCQMjsl5t0xp3dCu0FhAlo91p3mopuUJLoVC8JTuhyUAYmxkSjMPzc+E\nxAHSNjC5RMPt4CwBkUFpyTwGPxH8GzpL4RDhfvBls5LzhAJoYToBE/Q2TNGTQ8of\nvLfggkpbe2XEhcPKy7PSfvFl/1ypa64deRr9emSDmzIiue/IzlummNVvrdjSv2nN\ndds6ybyjs59dvnP4H5X9Z/r3x3Tb3AOH3j+ih42/c+e8dwZ+eeDVU3jq7mvC8faD\n4xZ89f2KUZP7B2Z9uGv+/OQiUfl05uP51Ok/nfym47kv/lq1+/rZcz4/kP1uZ25b\nvXXJK23f3rVszOhdj/y2ecPafQPLT0x/5vB3u57fsnflx9WXl8b1e57cVJ0x5p5P\nNq5+6m7846jKhNtW3bz36hfHfaZXrV1579f9yUeHTp04fdXEg4evXPHm/cdHT2yb\nsvuSfTeysY+u26j0TSuPXXF4w/i+gZ/r2384ue7yOdtf2zr35Pw/b7h10+Yzr09b\nfGzGS2Nvf/vdF2adeKLLf9n34OSRfwNe3h4Z\n=8sIp\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":4,\"last_id_change\":1424817825,\"username\":\"eastbayjake\"},\"id\":\"345ed9aea00c8194a5b5d3e3e20e3e19\",\"key\":{\"key_fingerprint\":\"172eeaa4be2bedbd9f36e34eab73e03bc1aa7535\",\"kid\":\"01015248df6736af67f04a242e5a559cead1ffba67b2042575da98cf24f2ea04edbf0a\"},\"remote_proofs\":[{\"ctime\":1424817548,\"curr\":\"b9eebad14ce7c3c81877ebb84d2333c008fd72313caf7afdd318019b90807067\",\"etime\":1582497548,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"eastbayjake\"},\"proof_type\":3,\"state\":1},\"seqno\":1,\"sig_id\":\"418d16aaddfc5bbff6828d529613deb3d084d2b01cc0dd2550a0d5cac9701d5d0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"0514f1527b0db9406c290cd01e9616d13314c830b02029e4344a032cecfdcab5\",\"seqno\":2,\"sig_id\":\"50349c43ca17340fa690139f13d1161b6752eb5b9b5e41f11e145f28eb7492750f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424817916,\"expire_in\":157680000,\"prev\":\"abf4894d0e34671e497ab1f2982dcb24c8e8974d10a00576eeb81ef236d75da2\",\"seqno\":20,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1424817916,
      "etime": 1582497916,
      "rtime": null,
      "sig_status": 0,
      "prev": "abf4894d0e34671e497ab1f2982dcb24c8e8974d10a00576eeb81ef236d75da2",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 21,
      "payload_hash": "435c23da23b8f0cdc5d01386c4fb45cabc0c8cc18dec204da64e92e4183ea522",
      "sig_id": "de8c6c5e3a795efa0134ece15854a3b27f66285e1aa54a889c056b344b992bde0f",
      "sig_id_short": "3oxsXjp5XvoBNOzhWFSjsn9mKF4apUqInAVr",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n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\nOPH+6enZmZn/Ag==\n=Xn8y\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.7\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1424822283,\"hash\":\"1e6cff7e202e00d95da40c270129ac4a2b2da7fd20b63742a51192d9e3c8b37181272499a8c8ccd3359aaacd22493d961de6f0e9427b63f934b61ddd4e1f7a7c\",\"seqno\":161166},\"track\":{\"basics\":{\"id_version\":10,\"last_id_change\":1424818329,\"username\":\"eastbayjake\"},\"ctime\":1424822326,\"id\":\"345ed9aea00c8194a5b5d3e3e20e3e19\",\"key\":{\"key_fingerprint\":\"172eeaa4be2bedbd9f36e34eab73e03bc1aa7535\",\"kid\":\"01015248df6736af67f04a242e5a559cead1ffba67b2042575da98cf24f2ea04edbf0a\"},\"remote_proofs\":[{\"ctime\":1424817993,\"curr\":\"d0732de68050ee3c1d182e10c17e5833eee5d800e587f668baa5607382a4b420\",\"etime\":1582497993,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"eastbayjake\"},\"proof_type\":2,\"state\":1},\"seqno\":3,\"sig_id\":\"ebd456fa9d628a45430314a99e31f3f09bbd42b0f0b7f99d939210a2081af2d60f\",\"sig_type\":2},{\"ctime\":1424817548,\"curr\":\"b9eebad14ce7c3c81877ebb84d2333c008fd72313caf7afdd318019b90807067\",\"etime\":1582497548,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"eastbayjake\"},\"proof_type\":3,\"state\":1},\"seqno\":1,\"sig_id\":\"418d16aaddfc5bbff6828d529613deb3d084d2b01cc0dd2550a0d5cac9701d5d0f\",\"sig_type\":2},{\"ctime\":1424817824,\"curr\":\"0514f1527b0db9406c290cd01e9616d13314c830b02029e4344a032cecfdcab5\",\"etime\":1582497824,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"eastbayjake\"},\"proof_type\":6,\"state\":1},\"seqno\":2,\"sig_id\":\"50349c43ca17340fa690139f13d1161b6752eb5b9b5e41f11e145f28eb7492750f\",\"sig_type\":2},{\"ctime\":1424818115,\"curr\":\"4a8a01b5f1fa0e85e6f1ff5f303b3fde520996845715e2c0256408b349cc8214\",\"etime\":1582498115,\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"eastbayjake.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":4,\"sig_id\":\"4391f1c718270ec52aa912be0ba8be237203676c3b84ee4e2baf8c703162dd940f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"b4930677467667ea292ba7c7320e1c336dc29bb575863a52806cd02fc3d6b633\",\"seqno\":6,\"sig_id\":\"82b1f91595ca62c66d4ac2166cc08ca777ee1a20dfdb8e69ae8989531afdcac50f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424822326,\"expire_in\":157680000,\"prev\":\"64868e2b2a3dac78b9040722ab3217be565b887be8282c34a52842de6ae55035\",\"seqno\":21,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1424822326,
      "etime": 1582502326,
      "rtime": null,
      "sig_status": 0,
      "prev": "64868e2b2a3dac78b9040722ab3217be565b887be8282c34a52842de6ae55035",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 22,
      "payload_hash": "3061ceab5f82a87df7cc4065c58baccd3bde0513326e40a4e71547278f93eca9",
      "sig_id": "f8aedd7699783c95ed1ca93f23dce9fc7a0ff6e394d080ce1c8a09687845311c0f",
      "sig_id_short": "-K7ddpl4PJXtHKk_I9zp_HoP9uOU0IDOHIoJ",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.4\nComment: https://keybase.io/crypto\n\nyMWyAnictZZ/aJ5XFcfTZd2w0FIRZh2Ozdc/1tqs3N8/KsOqhbZoZYPqH6vycu+5\n5ybPkrxv9rxP2iVZEBVaSq2VOkRp7boOxvyjUKGCdozpNkHSoihsf+xHdbNzE90P\nRdYOrfU8b9MsSVPaYiUQnjzPzbnnfj/nfO95bmlvz5JFD+wcHDiy8S6x6MTUsdGe\nLW/d+spEI7bTWGPtRAOGCmxV9VMrDGNjbWMQx2Lo4Jqifcd2jI2+xjYsO0W7RZ+8\n9MboEBGsB6u9M16jVMFp7z3TUficXADTmOyrw9RBc9Hqx3KkLOo9GtoFoXSSyUWh\ngetsAGXO2XJnTTQuJOVEkoJ2HWh3qjnZNLoxm0WitwusH+x+YJxxiUxLI0xIQhpn\n0FmRg9XRqKCjY5IL7S2LGoINNtrkHUhPfzhBn3kQLFC40W44xSPPVueoNVMmScsM\nc0roIAVznLF6YQfLaeXa0G612qUxshagKgMM1hJQ/gV06qciNWfE5KKvMRQ6FR2o\nCQOBVKJ3StB5lHF+TtxOhf3DocCyDtvNyydKJ7IgrGAWQhYy2qwBhFX0wH1jRv9a\nsnkMUISUeHI2A9NGBmEM55KphNkpqx13LDPMczW1ERBD0FL67EPOxnPp0DJuk0mC\ndEMrueVI/8xINCmUDJBiAhE5BEWaUu4lDrcrbI6U7XYmQbZS9VXF8MzBpRV0cBgt\nyzrNEIEBJsEzN14hM5YLmxJakUA6oPNmCTqiRZ6UMVmFxK02RhFL9DVDnA6uiSy7\nEHw6g1qVbhbdDhhAGGymUIXm/Z0azUwrVNuLqiLZ59AIrVTi9jlMuqGa1dgIfSes\nnSpU9b70pYMPtNr0SE9F/4XaTdpZn7Pjhs4XJNNKcpeC5sEHICGCsSi5p1o0SFUs\nrEavFXWLpHUaFavJ1NGm95vsmyej4kbNyCi9BfpJglQRKiurfKL6VxkAYkyGkpAG\nA3WB4iJzzayjvpQ8Wk7VoOR8GS8EHylxGwXPQlDPO2eT1JSm9kI4SwvoF0OVEmQj\njRFaWJN0SAa0FtQ9OluQXCFvXCsQaBet2g4al++P2Sz0QizkLBY5oPXMKCmkJBNg\nntxJimRBSA0KFUaGzBMaCwk495A8GEWlKR3TVOhXZkEkZ1jYnLxgggySLCdQ+UYK\nAElgTgwDWGV5AE1EKCcOkUzJSOrElKkzSUZ3CYtu8GkW1wH0tbEoMaWimtcbl0Wh\nFkKhZqGIDASSWZPuTEGmKhZRgQcXNCK3SjpDlpcgJqUQQGWes9RCY9aevEtcGQXF\nnUFBquRgIJN7xpCNY6ADaqdr3RU5m6fX4DS5mmO+/iulaJkidIl2R7gERTf4NIrr\nwPnaUPQX1cBovFoUciEUehYKRvl6pYUw0WcbgKVM3gqCJ6aR9NCRLkEJ3BrPEpfk\nzcoZppPnTJJJXYVDaWU+MHpt6suYMclEVoZmgySlypFKwTK6YBQK0BkRwUuOwols\nkmUuS8PpwovqEhTd4Bcd6n/nfJUo6nll4RtiDbSHG10EFU0IQ7RgoKpG1s7HQuME\nW4iMmUWGU5GQs3oD0lBLUE1HusyjJMmZMwkcgUohBu+B01uDGaio6jFNMxNJz/lk\nvtbdplmFYqg+xEgYG2qH1BwInYHrg+b/c4jJer7qHmF6zJo1pdbSzak2TeMg1ciD\nI0WJzaJeoS1VQlfu6TJRZPdCpkBzlMuMvF4nmnmcIZuJdPV2RxEHQPc0giC3CUah\nF6i4kxioUT44qKD7vwr99a1U9LdCNVpiY/KZX3z1xp5FS3puWnxDPXz3LPnQ8osj\nee/vbj7f3rTmHbb6oc0Pb3G7vn/2k08+fW5q5bd/v+KJ2x788rs7Nj43tW/L+O5N\n5174+LPvvv36dz+1rXf0yKj/2/Dj+27tP/q5vd9a1XrvliM/+8z4e2e/uHLrb5a5\nP3x61eSBTfcfOrX2vo2j+6fe/9iJczdvHj/6+R889tZtW5c+0fzLquNT/7j77IGP\nTL68eM/iN2/c8+cfnql2Ln9qYsc9av1fz+987cVXv77yC3/f8MJPTx78xlfWbXjq\nmz/vY2N/3H188U969j09cvLAm4/wX42deaMHD7WOPdp/3+vPH1p3/vbTW8Z//Kf1\ny94/+KN7d5VnJwb//dvPnvnl6uHeU79efvr2/6yQ+75zy8reN/710h0nO3L/lw6f\n/t6HP3p47z/vVEs/cerl/wLKKfqA\n=4aXe\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":12,\"last_id_change\":1424824689,\"username\":\"stegmaier\"},\"id\":\"9d046b0a27207caf23b7f5cc274b7f19\",\"key\":{\"key_fingerprint\":\"5e2add1d87fc0563a26611304def84758180f0ef\",\"kid\":\"010137bceeaa5339f9aff69138e7017d6d2b81e73171e18001b13243acdbdc2b1ca40a\"},\"remote_proofs\":[{\"ctime\":1424823729,\"curr\":\"5abc0ced21f1694e067127dde72dc38c7f5f3c5be7e1d466f4ad1756649cae9a\",\"etime\":1582503729,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"andrewstegmaier\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"d5879ff816c0ca3054318da51a9ac142a67e3195b86e626275e9545c133185e40f\",\"sig_type\":2},{\"ctime\":1424824164,\"curr\":\"397c7c7d29ca24f4749d2594fcccbbd66c036ea7d9412f15078c1531b71f8443\",\"etime\":1582504164,\"prev\":\"f22b29887d3575e5922871648710e4ddcf636625276d5ad6c5524255f7c314e1\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"stegmaier\"},\"proof_type\":5,\"state\":1},\"seqno\":3,\"sig_id\":\"fae790643233c390924532d7c235c4e4eb0e0951a7cdc119cd9c645ab38050010f\",\"sig_type\":2},{\"ctime\":1424824319,\"curr\":\"7fd9202ac698ca5f3b10fcd2efd0eac7471ac56c07901cb39c63f0edfdef6628\",\"etime\":1582504319,\"prev\":\"397c7c7d29ca24f4749d2594fcccbbd66c036ea7d9412f15078c1531b71f8443\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"astegmaier\"},\"proof_type\":4,\"state\":1},\"seqno\":4,\"sig_id\":\"b0c2e6e85c404cf2412b4c9c8a5ee17438646bdcbd44ecc4f1ff3525ef5904d20f\",\"sig_type\":2},{\"ctime\":1424824404,\"curr\":\"fccfa6cf07cbaf680c5ae58501cb41389cbac851e780989cbddb704ac6d4ecec\",\"etime\":1582504404,\"prev\":\"7fd9202ac698ca5f3b10fcd2efd0eac7471ac56c07901cb39c63f0edfdef6628\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"astegmaier\"},\"proof_type\":3,\"state\":1},\"seqno\":5,\"sig_id\":\"0b10945226b9f7ac0dfad1c21d05e8095b0843c17690d134e048605d910333140f\",\"sig_type\":2},{\"ctime\":1424824546,\"curr\":\"5560b5c00302f463d8d334fbe85707014e2c5feeec931e282f6d708f361cc2b4\",\"etime\":1582504546,\"prev\":\"fccfa6cf07cbaf680c5ae58501cb41389cbac851e780989cbddb704ac6d4ecec\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"andrewstegmaier.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":6,\"sig_id\":\"1edf0e496c36743244b2adb340f086dc8faddaba99c1b346efcfde8695506b570f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"5560b5c00302f463d8d334fbe85707014e2c5feeec931e282f6d708f361cc2b4\",\"seqno\":6,\"sig_id\":\"1edf0e496c36743244b2adb340f086dc8faddaba99c1b346efcfde8695506b570f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424825053,\"expire_in\":157680000,\"prev\":\"435c23da23b8f0cdc5d01386c4fb45cabc0c8cc18dec204da64e92e4183ea522\",\"seqno\":22,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1424825053,
      "etime": 1582505053,
      "rtime": null,
      "sig_status": 0,
      "prev": "435c23da23b8f0cdc5d01386c4fb45cabc0c8cc18dec204da64e92e4183ea522",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 23,
      "payload_hash": "7b38bc16fc9a65e8f7fe77cadd751a561378f59975d24ed7274ffaad09a04392",
      "sig_id": "49eb0fc70bc3f269b2c82b7ec8ebae60fa4da0263e0e532eb9c31e603d6e90fd0f",
      "sig_id_short": "SesPxwvD8mmyyCt-yOuuYPpNoCY-DlMuucMe",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMT+AnicrVZZjF5jGJ6xRYsSkVSRqEOQGM23vd8ySUMisVdcKEr5837b/MfM/P/0\n/KdTo9oJRcVasRUXlpC0ohe9sVyIJZao2Jcb0rhBJIJELDfEe/6OMaNNkPivzn++\nc573eZ/nXc5rh+0/sHBwzebR9o5zTxeDb7+5Ze3AygO++mp94btxqhheX4SxMnXq\n5qqD46kYLkbTlMdeWlZ2l65LvhgqJlPVK7sdOhJRmywsWC6dF4bbaCD6YLnBIJVF\nZpUSAKLYMNTANKC57IykaqIqmxgFWBQKoozWCwgcsg5J5pwJymivLUZlRZSCora7\nvXoem6KP2Soj3d3H86P9A8YZl4mB1EJjFFJbnawRGQ14rRC8ZZILcIZ5CGjQeBOd\nDdLRHyvomKNgSHBr+3CKe54NZA/AlI7SME05CkApmOWMNQ/2UjWjXDd0O51upbVs\nBKgrDKONBMS/DL3mqoytWTHtUDGGvZryaYU2kkjFMFdCS2OdsvNgJ7DXRt5A9jlZ\nTEIpcAmMsMyJhFJqw7jg0iBwV8xq38g1X38vXfKYlLYefFacA7cqAEvWgccGOegM\nEObp6Z0PwiEkoywLKSG9YYJnwYhoQQsGUXDOEw8ych1j1krapLVwkShxQ3oS9yqN\nd+vUmqi63UxiXEmVV5fjf2UtuRoqwtqqamgmE6WS3DKiJ2zgIeholGKuMYr0dZhF\npngI0iitDU+YkAeWjLOIluinGXCwisEe8BkGjSp9Fv3qb6cw2opYY+uaXmPLbBtU\nKcaynm8wjpRjKeBkGsdOk1MfplVPTdAhBejVWDcx6aSX1nS6dElX5ciemnVEGWK0\npGXwGLTiwICS8BytSSobyyIlyJ1NoCCLlACQx4woZKLqZbnYg7Ynntgw9HcJtYVZ\nCXPkgvQACRGVVWBUMFZpQEsVrZKjPg6W2kKBd9EboZnjJlpNbBKHEPaSsA8+UaVJ\nAg8h5uCoBjz1A3nsJfWJUC77bHx2qJAmgqHADFlQVJ0cmXQ65mhYNK7pr/9mRr2u\nrOtUzXfj4qYvLsDxqV6bWP3NDrEvO9QcO6hmibsTylukCRSYdWQE3WPUUFHkjNT2\nghJTqEgn9FG5YGVC5rJk6P7ZDmvlrB2GZks2QjCXEL2hmRKDi+A0l1aR4Yk01lEl\najfmQGmRhKV5k1ElaqqEe9nRB5+x43/w+r/Z0abJRjakdb1/3x96X4bAHEMUDR/Q\nINGSzl5oFSJAv54M6cBAS+ET5qCl5i5G0kgnnZNlCriibbCXIVf1w7RqLMca7hM4\nNdbF2GpT2TSrSCVyNZumej2QozKFhGCbGcNo4BAmaNQ0TXy2hhlDE47GkKLINAWx\nWYwzSZi5SWSg+uBZc+EkveytR6+k8lkwxtHRIhHeewXN7pExMpVTsjogtUaWwVIS\nG5rd0U9hZoXM2cCNdHOLjFxuJlu6dqKsUqtsngCjLaPfbHVIpjll5iFbQZMmZhMC\nFRsEIG7UydJHWphcStKTGlYlw0EZYWx2JAm6vxIVVHI1jhAm5dvBem2Vig2vvrz6\ngIHBhQMHHbhf82ExsHDBEX9+bjz//sEDm277We9frHh5+olNn73x0YGv7LdmkO1+\n4dyRlWLJiT/uPPb33uJdl8jjJp/bvPqRZffuvviF39677qBtx04X/pSzv1m16c6R\nN7bvUk+3T/jxtw8vXPzD6q9/Obn8YMFLR565fmv6btGVi8UV6tbigXeP2rFix7fT\n52358JSzLvvk1kNvfNEuv2Xi58eOXnn0Rb88cMvGjQ8v2TZ40xe35eHPLzM3PDnw\n6Y3v/HrNXcNBbvvygzOqe65/6LhjRu5csWrp8unT3srPnNR9d8lj9y168Pvr7r/j\n48lzln1ywbbPFh353M7DD8Wdl9++vdJ3b772keNPvWv3mi0Ltj41fdj7p9//+uDN\nY48uXbr80h8O2fjazl2vf/rs1p/Of6n6PDx+9R9hSf1+\n=m57g\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"2d67f2858139b2718d75dbc817ac348a08442552\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":8,\"last_id_change\":1426378948,\"username\":\"pasha1\"},\"id\":\"8ae24459e5728092ea3367012137a519\",\"key\":{\"key_fingerprint\":\"b39ebae468b5bf4115184c50e895ba8ae2c6f55c\",\"kid\":\"0101b9bc29a5e7480ceea4c57cb0c72d856205d2111e1c3d16ddf6438e6629d137170a\"},\"remote_proofs\":[{\"ctime\":1426378314,\"curr\":\"be7d34318018428c1cc6d7440939ca6639af2f1c3a53746671eaea1c0e798aa8\",\"etime\":1584058314,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"agilecaveman\"},\"proof_type\":4,\"state\":1},\"seqno\":1,\"sig_id\":\"9f2f5dd8c29cbac641505f1cb1a87e4f780d093198e545f2ee55a1dfaa23e5b60f\",\"sig_type\":2},{\"ctime\":1426378685,\"curr\":\"fd12c0e535da484574c78465a8b1f4e97f2c8fa745b9db7260917d86b1ae15cc\",\"etime\":1584058685,\"prev\":\"ccdfc97cbb060137b31b1249fbf7bf9a4a17a7da40a0c43671a0396dfd70d79a\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"PashaKamyshev\"},\"proof_type\":2,\"state\":1},\"seqno\":4,\"sig_id\":\"d21601924b8afffc0891a81600ea3d2ffa5fb231b4a4fa7abd49c83ea09f30a90f\",\"sig_type\":2},{\"ctime\":1426378883,\"curr\":\"725bf72209eaab7f75dc9d59613848e5e8406d4e101095462e286d3fa4e3d1ea\",\"etime\":1584058883,\"prev\":\"fd12c0e535da484574c78465a8b1f4e97f2c8fa745b9db7260917d86b1ae15cc\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"agilecaveman\"},\"proof_type\":6,\"state\":1},\"seqno\":5,\"sig_id\":\"4e2c5653a89c8b264cd55f7bf973fa05632beafc63619dd4626e6fe80451476b0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"54ea7af737b3b530a3ecea58093901cc14756a6aa8bf8707711ecc64dd480cab\",\"seqno\":7,\"sig_id\":\"4f509f1f612936aab8bab434bf2001a9b552bbb45a7a73dd04fee86cafd7f3c80f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1426384614,\"expire_in\":157680000,\"prev\":\"3061ceab5f82a87df7cc4065c58baccd3bde0513326e40a4e71547278f93eca9\",\"seqno\":23,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1426384614,
      "etime": 1584064614,
      "rtime": null,
      "sig_status": 0,
      "prev": "3061ceab5f82a87df7cc4065c58baccd3bde0513326e40a4e71547278f93eca9",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 24,
      "payload_hash": "5cae5e8dc5d0cd6b233f12f6aeb03f45ca4c12b3fc2c337ed31112f0ebaba2a7",
      "sig_id": "c930a19b6e372177c3ba9e840544cb27fad487f3b72cbafd5da863b97eed66cc0f",
      "sig_id_short": "yTChm243IXfDup6EBUTLJ_rUh_O3LLr9Xahj",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n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\n05sPP/YVP/+Hi1/684vq4MTZb/3x/SOH33n9jq//+GMf3fBO\n=ftZy\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"2d67f2858139b2718d75dbc817ac348a08442552\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":38,\"last_id_change\":1427140928,\"username\":\"akalin\"},\"id\":\"ebbe1d99410ab70123262cf8dfc87900\",\"key\":{\"key_fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\"},\"remote_proofs\":[{\"ctime\":1409475047,\"curr\":\"d8cd14f0d6da7dad0b0227b024ecb99c920e6110ef9486262be08df3b6431e00\",\"etime\":1567155047,\"prev\":\"738bebce2b4ed88c9319075bfed58779734efd03f3ee5386b15b38b61d2024f5\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"fakalin\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"1a67183170ef64b00e805c74d39c5286a6a9b50939b3604b15f0722aae27500b0f\",\"sig_type\":2},{\"ctime\":1409475097,\"curr\":\"99456cb27d7c6c8e6978be4c0928c19a07428e236fa9f7ed4d823d926f054701\",\"etime\":1567155097,\"prev\":\"d8cd14f0d6da7dad0b0227b024ecb99c920e6110ef9486262be08df3b6431e00\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"akalin\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"68a547eed57a11b62bfe2511fe128cc10cfb70f199453aa9cb964b97999f5d800f\",\"sig_type\":2},{\"ctime\":1409475156,\"curr\":\"a2a877a4cda04a32aa02de40137d22b2a0a04f35f7f677bdcae15fbf91f4ec04\",\"etime\":1567155156,\"prev\":\"99456cb27d7c6c8e6978be4c0928c19a07428e236fa9f7ed4d823d926f054701\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"akalin\"},\"proof_type\":4,\"state\":1},\"seqno\":4,\"sig_id\":\"dece17e125e06d6b44438218d29ab55a7785beaeb7bf372d8f98eb228009635c0f\",\"sig_type\":2},{\"ctime\":1409475252,\"curr\":\"297ac58d197e52956fae5d628b4db81de68e8cebf1bfd17fbb4404cee7edba6f\",\"etime\":1567155252,\"prev\":\"a2a877a4cda04a32aa02de40137d22b2a0a04f35f7f677bdcae15fbf91f4ec04\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"akalin\"},\"proof_type\":6,\"state\":1},\"seqno\":5,\"sig_id\":\"fc16a961ab69ae81910f3a76ca000fca19699dec8461fc7ec0e8fcb78764069b0f\",\"sig_type\":2},{\"ctime\":1409475374,\"curr\":\"04e5fba11ba32227cacb8b8345cec9bee563bfd4c25b77aa716b3763c3857735\",\"etime\":1567155374,\"prev\":\"297ac58d197e52956fae5d628b4db81de68e8cebf1bfd17fbb4404cee7edba6f\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"akalin.cx\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":6,\"sig_id\":\"ac377cb7d4df7bc9db7c50b3bc06146e05e29fd0eb715b478ea461074a8770550f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"ce162011e380c954de15f30db28f8b7b358866d2721143d9d0d4424166ce5ed8\",\"seqno\":25,\"sig_id\":\"48267f0e3484b2f97859829503e20c2f598529b42c1d840a8fc1eceda71458400f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1427999616,\"expire_in\":157680000,\"prev\":\"7b38bc16fc9a65e8f7fe77cadd751a561378f59975d24ed7274ffaad09a04392\",\"seqno\":24,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1427999616,
      "etime": 1585679616,
      "rtime": null,
      "sig_status": 0,
      "prev": "7b38bc16fc9a65e8f7fe77cadd751a561378f59975d24ed7274ffaad09a04392",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 25,
      "payload_hash": "91e23015983d77e62a467470aedbe563db62ffdcebcb5bacb22b1e81a81e0575",
      "sig_id": "d46203a6ef3dec608d5a61520b4310f3f72073ab461289d1937a3d50db4ab03c0f",
      "sig_id_short": "1GIDpu897GCNWmFSC0MQ8_cgc6tGEonRk3o9",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n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\n5dbu/q/dvOvr737o6RP/6bz/vxjQa/Q=\n=MXEx\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"a83b5e55a841454ff3f9eb75c03f5e269ecd30ba\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":17,\"last_id_change\":1430212043,\"username\":\"marks\"},\"id\":\"e3d11da4dd6ea5b4e328df9a80ad2119\",\"key\":{\"key_fingerprint\":\"5766b7358932d52ac7edd391d279eb4d0dd8be48\",\"kid\":\"01019c72c962ba85496f17dfa5cd39f5b1a7a98b8fc22dc26e4920e7d6bf5311081f0a\"},\"remote_proofs\":[{\"ctime\":1429044821,\"curr\":\"299f4e2461ae29d61153d712a4d5dd42b19b84ebbe3c6b6ae0cd31a27ea9587c\",\"etime\":1586724821,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"laebshade\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"666bab030581509664ad2649ff34804fff664adb96ef6a765cff47e86321d0a80f\",\"sig_type\":2},{\"ctime\":1429044876,\"curr\":\"643678824683fc60026c2ac03344fdeacff1f6bf95dc02189caefd825cf07b3f\",\"etime\":1586724876,\"prev\":\"299f4e2461ae29d61153d712a4d5dd42b19b84ebbe3c6b6ae0cd31a27ea9587c\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"laebshade\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"835dc147ade856af2320f4d40fda968bf19d45dd29934c84d5f47fc7462f9e440f\",\"sig_type\":2},{\"ctime\":1429044990,\"curr\":\"e6e5c443846efe1c1a2c8e453155cab11b0d758ec16ff78d80f10c2581bd8174\",\"etime\":1586724990,\"prev\":\"643678824683fc60026c2ac03344fdeacff1f6bf95dc02189caefd825cf07b3f\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"laebshade\"},\"proof_type\":4,\"state\":1},\"seqno\":3,\"sig_id\":\"072a030922dd777f4a640e0e61fb341a0995656adc500115ec2bb88c37d600640f\",\"sig_type\":2},{\"ctime\":1429045103,\"curr\":\"8dbfa743fd9ddf216390572927f980c34fc393dadcf6f72e6b25d49fb184d79a\",\"etime\":1586725103,\"prev\":\"e6e5c443846efe1c1a2c8e453155cab11b0d758ec16ff78d80f10c2581bd8174\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"markashields.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":4,\"sig_id\":\"1e176953f3cea0043172ae39900a5921145fdb745acf1e21835c4964c581da390f\",\"sig_type\":2},{\"ctime\":1429122182,\"curr\":\"cf5fceaa9c459258d4ffdbba3840047d38f166421049b88ddfdf11f7f18e7de3\",\"etime\":1586802182,\"prev\":\"2c32724fa9517a33521692796b830b5572dc6fbc51aa0a8ff8e6e6c9b32bf1d3\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"zzrbikes.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":15,\"sig_id\":\"a299f5a3b4a9af5c249c7c8bad66071b1d998eb466a5664e8af800d3426796b30f\",\"sig_type\":2},{\"ctime\":1429122394,\"curr\":\"c20dbded98b9cfb1585329c5889091312da3776d3bf33d2179e60cf88e9c6d18\",\"etime\":1586802394,\"prev\":\"cf5fceaa9c459258d4ffdbba3840047d38f166421049b88ddfdf11f7f18e7de3\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"gentoogeek.net\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":16,\"sig_id\":\"d7900625801bcd7c7c896a095e28f0e34473bae75e0c479019a5e75d4a7edb8c0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"6e3d94f68da91ddb895f1196d258fc552dfabfdcbb7141c1b267e82d6dd31cfc\",\"seqno\":21,\"sig_id\":\"5a778fe9f838ca35d90548bf9048305ba08cab2c3aa6ad92cda41439a27daf920f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1430419594,\"expire_in\":157680000,\"prev\":\"5cae5e8dc5d0cd6b233f12f6aeb03f45ca4c12b3fc2c337ed31112f0ebaba2a7\",\"seqno\":25,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1430419594,
      "etime": 1588099594,
      "rtime": null,
      "sig_status": 0,
      "prev": "5cae5e8dc5d0cd6b233f12f6aeb03f45ca4c12b3fc2c337ed31112f0ebaba2a7",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 26,
      "payload_hash": "b11dae23b92ba9d7b6100fb81bf201dfdcd406ccbf637876daf31f139d477af7",
      "sig_id": "b152e28cde4399080d248318b4dc10114bbc9388dcd38823bc227cb4365dcbba0f",
      "sig_id_short": "sVLijN5DmQgNJIMYtNwQEUu8k4jc04gjvCJ8",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMIYAnicbVJbSBVBGD6nLMlIuglSVLQRiJjNzuzM7p6KehHNwgsZFUk6szt7XE/u\nHs/FC2IvRVkhiRZkavbQ5Uk4kfgQSWaGZVoZIWRRFERB9ytdHmrW7K15Gf7v/75v\n5v/4h+bN9KX4qw+FKnry1kD/7Wsb477tmzetbpCYa9ZLgQYpxKcuy3aCPBKO2E5M\nCkhYo1DBJjI1BrEhY4sYHFmWpcqaShjRqKlo0ERQypIq3KinEDaMRnm27QpMFGW2\nKdD/8ENTDSADGXGAEYGEmhARjXBNhRZVMSMKxUwDSIZYVwHDBlWpylRT1wyki0KD\noi1TCKiwi0/ZKTKTLRVbDGOgEBOpgABNgZgiCDQZAI8Y5RGHVnHBdg3XcdwIIUhq\nzJIEXmMb3AvBm2WaU0mNkPt3mnDEjQnJXoFWxGLhgCeK1Yc9Vi1nZdP6MmY7pghR\nCGp4JGq7jhSQBdOI2Z6hrCCgEaQDPUvidWE7wstsj4FVogFxvGd4jbDUZQ4RkLGu\nIVNVOYFUIaqiAspNxjFBJiPQskyDM4NhRg0GIZO5yEOTRZ4qlryJqh1XCkAi/kmD\nwjNqBx0ai0e41Dg4UJrk86f4Zs+a4e2BL2XO/H/bcWZt8q/k0aazvcVXHzw/lVbP\nk6z1heXjG1/3XAq23xtZwBe/Dlf1o8lMZf9IZU5RNzh3rqBx8nzdxcnxHcu2skRT\na/rzxDZ+Q4//mDt8GOUMuvWjvuG7F1prC9xd5SeSattKF+aPL2n61Zm5YnIotahr\nS8fY3H3vd+c9q+56Uvq+s235omO9n/wZ2f5LX6qrKlPTmzsaLJyUeVbZkEj83vO4\nb+/xzx+D3/ISY63mztqhpQf6T968MvGzeVXBi4PSPN+75pKg8XD4dFrLy1t3r7Pi\ni28+Pr3/tq05oy233Yre6eteN1yXW5jW8ehV8EGB/P1y6srZR/q+ogk9P6174ENL\nyVF/zcv4H5rwJkk=\n=Wdqc\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"service\":{\"hostname\":\"jacko.io\",\"protocol\":\"http:\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1430863909,\"expire_in\":157680000,\"prev\":\"91e23015983d77e62a467470aedbe563db62ffdcebcb5bacb22b1e81a81e0575\",\"seqno\":26,\"tag\":\"signature\"}",
      "sig_type": 2,
      "ctime": 1430863909,
      "etime": 1588543909,
      "rtime": null,
      "sig_status": 0,
      "prev": "91e23015983d77e62a467470aedbe563db62ffdcebcb5bacb22b1e81a81e0575",
      "proof_id": "3195d2432b6853b6f3ceae10",
      "proof_type": 1000,
      "proof_text_check": "\n\nyMIYAnicbVJbSBVBGD6nLMlIuglSVLQRiJjNzuzM7p6KehHNwgsZFUk6szt7XE/u\nHs/FC2IvRVkhiRZkavbQ5Uk4kfgQSWaGZVoZIWRRFERB9ytdHmrW7K15Gf7v/75v\n5v/4h+bN9KX4qw+FKnry1kD/7Wsb477tmzetbpCYa9ZLgQYpxKcuy3aCPBKO2E5M\nCkhYo1DBJjI1BrEhY4sYHFmWpcqaShjRqKlo0ERQypIq3KinEDaMRnm27QpMFGW2\nKdD/8ENTDSADGXGAEYGEmhARjXBNhRZVMSMKxUwDSIZYVwHDBlWpylRT1wyki0KD\noi1TCKiwi0/ZKTKTLRVbDGOgEBOpgABNgZgiCDQZAI8Y5RGHVnHBdg3XcdwIIUhq\nzJIEXmMb3AvBm2WaU0mNkPt3mnDEjQnJXoFWxGLhgCeK1Yc9Vi1nZdP6MmY7pghR\nCGp4JGq7jhSQBdOI2Z6hrCCgEaQDPUvidWE7wstsj4FVogFxvGd4jbDUZQ4RkLGu\nIVNVOYFUIaqiAspNxjFBJiPQskyDM4NhRg0GIZO5yEOTRZ4qlryJqh1XCkAi/kmD\nwjNqBx0ai0e41Dg4UJrk86f4Zs+a4e2BL2XO/H/bcWZt8q/k0aazvcVXHzw/lVbP\nk6z1heXjG1/3XAq23xtZwBe/Dlf1o8lMZf9IZU5RNzh3rqBx8nzdxcnxHcu2skRT\na/rzxDZ+Q4//mDt8GOUMuvWjvuG7F1prC9xd5SeSattKF+aPL2n61Zm5YnIotahr\nS8fY3H3vd+c9q+56Uvq+s235omO9n/wZ2f5LX6qrKlPTmzsaLJyUeVbZkEj83vO4\nb+/xzx+D3/ISY63mztqhpQf6T968MvGzeVXBi4PSPN+75pKg8XD4dFrLy1t3r7Pi\ni28+Pr3/tq05oy233Yre6eteN1yXW5jW8ehV8EGB/P1y6srZR/q+ogk9P6174ENL\nyVF/zcv4H5rwJkk=\n",
      "proof_text_full": "==================================================================\nhttps://keybase.io/oconnor663\n--------------------------------------------------------------------\n\nI hereby claim:\n\n  * I am an admin of http://jacko.io\n  * I am oconnor663 (https://keybase.io/oconnor663) on keybase.\n  * I have a public key with fingerprint 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32\n\nTo do so, I am signing this object:\n\n{\n    \"body\": {\n        \"key\": {\n            \"fingerprint\": \"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"71876b68ad482d32\",\n            \"kid\": \"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\n            \"uid\": \"41b1f75fb55046d370608425a3208100\",\n            \"username\": \"oconnor663\"\n        },\n        \"service\": {\n            \"hostname\": \"jacko.io\",\n            \"protocol\": \"http:\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1430863909,\n    \"expire_in\": 157680000,\n    \"prev\": \"91e23015983d77e62a467470aedbe563db62ffdcebcb5bacb22b1e81a81e0575\",\n    \"seqno\": 26,\n    \"tag\": \"signature\"\n}\n\nwhich yields the signature:\n\n-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMIYAnicbVJbSBVBGD6nLMlIuglSVLQRiJjNzuzM7p6KehHNwgsZFUk6szt7XE/u\nHs/FC2IvRVkhiRZkavbQ5Uk4kfgQSWaGZVoZIWRRFERB9ytdHmrW7K15Gf7v/75v\n5v/4h+bN9KX4qw+FKnry1kD/7Wsb477tmzetbpCYa9ZLgQYpxKcuy3aCPBKO2E5M\nCkhYo1DBJjI1BrEhY4sYHFmWpcqaShjRqKlo0ERQypIq3KinEDaMRnm27QpMFGW2\nKdD/8ENTDSADGXGAEYGEmhARjXBNhRZVMSMKxUwDSIZYVwHDBlWpylRT1wyki0KD\noi1TCKiwi0/ZKTKTLRVbDGOgEBOpgABNgZgiCDQZAI8Y5RGHVnHBdg3XcdwIIUhq\nzJIEXmMb3AvBm2WaU0mNkPt3mnDEjQnJXoFWxGLhgCeK1Yc9Vi1nZdP6MmY7pghR\nCGp4JGq7jhSQBdOI2Z6hrCCgEaQDPUvidWE7wstsj4FVogFxvGd4jbDUZQ4RkLGu\nIVNVOYFUIaqiAspNxjFBJiPQskyDM4NhRg0GIZO5yEOTRZ4qlryJqh1XCkAi/kmD\nwjNqBx0ai0e41Dg4UJrk86f4Zs+a4e2BL2XO/H/bcWZt8q/k0aazvcVXHzw/lVbP\nk6z1heXjG1/3XAq23xtZwBe/Dlf1o8lMZf9IZU5RNzh3rqBx8nzdxcnxHcu2skRT\na/rzxDZ+Q4//mDt8GOUMuvWjvuG7F1prC9xd5SeSattKF+aPL2n61Zm5YnIotahr\nS8fY3H3vd+c9q+56Uvq+s235omO9n/wZ2f5LX6qrKlPTmzsaLJyUeVbZkEj83vO4\nb+/xzx+D3/ISY63mztqhpQf6T968MvGzeVXBi4PSPN+75pKg8XD4dFrLy1t3r7Pi\ni28+Pr3/tq05oy233Yre6eteN1yXW5jW8ehV8EGB/P1y6srZR/q+ogk9P6174ENL\nyVF/zcv4H5rwJkk=\n=Wdqc\n-----END PGP MESSAGE-----\n\nAnd finally, I am proving ownership of this host by posting or\nappending to this document.\n\nView my publicly-auditable identity here: https://keybase.io/oconnor663\n\n==================================================================\n",
      "check_data_json": "{\"hostname\":\"jacko.io\",\"protocol\":\"http:\"}",
      "remote_id": "http://jacko.io/keybase.txt",
      "api_url": "http://jacko.io/keybase.txt",
      "human_url": "http://jacko.io/keybase.txt",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 54,
      "hard_fail_count": 0,
      "last_check": 1434483168,
      "last_success": 1434483168,
      "version": 3,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 27,
      "payload_hash": "e6ed42cb01be8914b2fba11faffd431729fecdd21c2e1d1470c58167c7122aa7",
      "sig_id": "4e965585ac3389cc9c86937c5f632b43d8487d6e723767ad2b5c325ee54c1c9d0f",
      "sig_id_short": "TpZVhawzicychpN8X2MrQ9hIfW5yN2etK1wy",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMSBAnicpVVbiFVVGJ6ZrEwRS+khMJA9KhGjrvtlKlQs9MEefJgMHDmu65zdzDln\n3GePOshkQWmWhlBBOKFSGinYBaEkLe3CIBheMinIF/HBLhDB4IhE0L/PjJOTBUXn\n4bD3Xotv/d/l/9eX025rmtK8bmt3+dDy+aT51NDivqaOFb/HTYmt+f6kfVPietJQ\nzYunqqmEpD3pDv3W1MOCtDZ7Q7BJW7I+ZPW0VoUlaZHBTmuvPdJRBsIYCyYyh4nx\nXjiBiCJck2SgrYApQGNa7QpZb5YWZyRcGcK4p15Zwh3mUbhAY4wSKymsUMYzRTwl\ncGq5Vs8nVJM0MEupLwq5dX93YwFhhGlAnAoijCdUKBGUJNFIbgUz3CpEMZQokeXO\nSCOt9Fo5quEFSrcKG4IMwPU14Bi2OEoeLeeICU8lEkgxwg0lSGGEio31kI0pV3O1\narWWCUELAfLMuO5CAqg/dfXiKfWlcTFVW9Jj6jnwKbmyAZGSdsyIZFoLzSbAunpu\nKqZeYDaKotFRrAJn3PFIZQieORsZ08QTi7FOxsUv9JpoALEBU484CdoYLqQXHASQ\nGkfPkcScRautoXqCoN4zLanUoDdx2ESqrSeCcmIsZU5G54TFzhKpkPLOWWmNN5Z4\nGxSl1GIQFGrPQqWWh1JvVqtFUGM1RC9PK6O0mWZYYt2WuL4sg1MD9Q4zZkwwwjJv\nJTHM0qiRUZFaKTR1SoiChUA8KqM0/EUUvAKDCPJQfhgD54oIMgrem4X1Bbj0MTov\nIjFKGucM40KoQK3GgkqwGnsFoWDKhSLTwkXBhFbWaeYZ5jIZ51Lo2+DTaKRycN0l\nb3JTeqpeODzeUfmGNM9DlvyTqw2IUt7fC99JWwILeVE5rNTDumpt9GPaNRp9IqyT\nRdmQAkqc804xB3JzRznVRSy8CNCjKkYMvUkCKMWBLVdeRWs1isko2th5A21/NUIx\nOW6EpAgyCbGg1EeQTFPEFHJgPfc8BMExtQ4ZpzCI6hx3UkfoMeICfIP+u8WIBvgN\nI/6/y//NiK40L/fZf+UD/Tsf6E0+KEU9IZg6hwVynntHjYswHSWjlsnApbWYRKRl\nNIJ6Q3xwBjJkbJCEs3CLD2sax5Ryk/YUJfea/p6a8aWyqZeLMjnmhJBicKFAFPQ+\nwTCHifDOGEbAbCwt0YFrgbU2iBIpir5EHiEYKI3DRklgdhML75EN3sEmaYUxWHFP\nOJUQfqXBGh4dl4gq4EgRhx7ggkUYUF4gKwOsAIuBYtY1OIyNvJtujEK78XBRjJHS\nQkAkNvamWSilxQ4uhULwG08FjDBvAoF2JNZoD3XBoIVQYRsJzKII1TIE3GyEboVr\nwMNAwhFT7Rn0RZR/MiUQtdx0ASbwrZq8LwvJwOcnOic1NU9puuP2luIibJpy1903\nrsddlyc3Hc2PvHZo0pO7Hh68790l83df/PDq4CfTl+1bvuD64EPtC9+Y9cVnG195\nYX3p9JbD01fs/PHjhXPC8DvXWprQsbeHD1y89s3rW66++N3c7Zc2Pznv5NyfRp5/\necbF62d3Ht043Xz/wDyX7DjwcysZ7pzzyNBQfuGDziu8ZWttaceB90/MvLr5rXrH\nqpHz5d2Ht9/5no1+z7ODi1/t2rF8w7Gnh2trW/esWXZ571dfzzi9YNuhXedas3Pm\n1Jv79u5dOXjP/tVXZs6iU1s/HT737TMvtaw4v+S3k4ua+2dPe+Lgc7pemfzRmUsj\nq6ZWlp699/5lj47MGl7ZcfrBypGDj+/Xj/3w66m127Yt2nJ850jX8V8uDJ2JnX8A\nhtqkhQ==\n=aIyH\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"7b0a1c99d9d09f7e2444eaf4c12add6c60282592\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":8,\"last_id_change\":1427499694,\"username\":\"cstamas\"},\"id\":\"3fc318e545c5f37eed4cbf4492d2b119\",\"key\":{\"key_fingerprint\":\"2be13d052e9aa567d657d9791fd507154fb9ba39\",\"kid\":\"0101dd49737968a2c1af39bd26352ab34c7fcc6b1cb27808dccb7badab2dbe8333b10a\"},\"remote_proofs\":[{\"ctime\":1424941719,\"curr\":\"e3dc144aaea6b4db72a4b3f90a8f3b7693c86652e9605f8a89f8af0ed806020d\",\"etime\":1582621719,\"prev\":\"e7dffcd6f2a87acca45668e3b91637b551d8d9848ced6c66cf64698bc94d4157\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"cstamas\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"26bc77accfc332ccdc84cdab5c35398e54d6ec998ff1eaf2ec865cd658d8fbb90f\",\"sig_type\":2},{\"ctime\":1424941847,\"curr\":\"73027454f33dfd6f930480c52a5d5ee6513bc0ac817dfcc5c79fb802ceac8d23\",\"etime\":1582621847,\"prev\":\"e3dc144aaea6b4db72a4b3f90a8f3b7693c86652e9605f8a89f8af0ed806020d\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"cstamas\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"883d2213cc160cd5dc3acf1c9743b47e57bb12f097fa63da2decabc9abe7254e0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"c515222970b0e2845c219d926dcaa42ff117b29e596199a03276cc6b0d00694f\",\"seqno\":14,\"sig_id\":\"dd0bedcd007b6aa185d2537848891445fc57038c1630598b564f969d60b7e7030f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1431108966,\"expire_in\":157680000,\"prev\":\"b11dae23b92ba9d7b6100fb81bf201dfdcd406ccbf637876daf31f139d477af7\",\"seqno\":27,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1431108966,
      "etime": 1588788966,
      "rtime": null,
      "sig_status": 0,
      "prev": "b11dae23b92ba9d7b6100fb81bf201dfdcd406ccbf637876daf31f139d477af7",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 28,
      "payload_hash": "937fff13c30312edfe384a04f09b8714fc72f12e22ead9c30811064528a22afe",
      "sig_id": "a912d4392c93a36cd89d9136839aa339a6df9f37156a18f2dc8ed46213b055520f",
      "sig_id_short": "qRLUOSyTo2zYnZE2g5qjOabfnzcVahjy3I7U",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMWyAnictZZ/iFxXFcc3bYxmRRFtwAUrZTQquK739z03pKUYLKXBGpBSrNHh3HvP\n3Xnd3ZndmTdJ1rCKBWn+EAIhQUSJ/yhpGyuIsaiN0kobpME/qlQhtdj6C6EWEVux\ntqjnTcbNbrOVDcRhhrlv3rxzz/l+zo/72FuunZjcsnTvXOfBWz+ktpx/4vRw4o7u\ntuOHW7GXl1u7DrfSfEXdull1cYFau1pztBxxQDNV74aDFFvTrQPUH1S9Lt8qRqLP\nrsRofVQ2xpI8SDQmuJzIelW8J3DUWpluzDRGS9Wdpf5iv2r2aFlAZWzWGfjxJG1x\niXQpxUvwLjrAbEBlrXjXTm9Qr/OmNbLZrjL/usH/50Y3hBRSk7DaKYdZacfeAPuF\n3kZn0EYQWiobvIg2oUcffQ6QdOAL4Ig4GiWQzQ1H5oyMsnhborXCuKy9cAKMsqiV\nAClE88cB9cfK9VKv2+31ndONAHUf01wjAftfpUGzqnJ7VUxpp1vzOKg5oHbqIKvE\nvxktwCon/Tq7C1U9GHaquV5jduSX84jIFBwmdiiIoF22rnhtfEpx5NdY/0ay9Qyy\nSA5DCNaZQDYa4Q1oskZmKJG8KzlrcmW9phZ8BFDe2UCgbSGLAa0uXoBzKHJK4LRJ\n1htjVfQhJEoqp+yAPAAF1pR979NCr6b2Yr/XKyzIpzj76mphFLgw0jptWZQ07Pd5\nV0VOSorZeZUgK1QyKEwUgiEOACIkCD6BtMS5kLNHIVEaQMYtilPA7tPYuHUS9EXj\nYw8aVUZejCqgQ2munbHG9t2DBs1qKdQHq7qmfuv1aYyMtOvlRb6jpluDGutmR74z\noKVuj5e8qmYvZi247KxK7Eom/lbAuVowcj1BcQElAHsuUkbwwemgZEpBOImotY5K\nRFFaF62N91uZfq2ARupVAZOJXmWVczYxhRiyk7FwhvPbeecFRpFEYO5ROQ05C5UB\nY0YSXFg6+8sEHBlf7NOBq0NnkyhybwGrpvnMD9NwoT+zyFXWm+n1Z1sj9Wu+nG+y\nujt4LQ+uT7kRErUGSbRE2QabhbbJ5+IhmcSSOC5yhkKG+5oTCMXyt6VkozZZJ+Qn\nPHcZ2gSS4C/ltC+WZS+GS9YalqcQf6Tw0kq0VhJKfsp7KyM3GJGS11zMqSTuZ8Fg\nvBxJY3yM5CrwvrLqmK3qzjBusjj0RiT0GhKJsikpFVGi54WJVhrWH3TAYrRVmudM\n0caJTDx7lMgu8/TIOpBsQPzv4hi1VavMKgktyRQEzcVXYuDU9pp8iLY0w0mQo+SN\n48y2kht/NpghEE8R7mRSS6f1WhIAVl80PiZReJCQ8jpIxykUHCePsg64StguCPTc\nsCMJx8VtbXZRE/fPIl0sSjWldMUk+pRzVW+ShNmIhF3bppJN7BtPFhZJURLsoATL\nzgEWHnseOP0D5y7qKFGDzQW8FqUIvgxSboKEl6skIHqrIDunEs9iCACawaMoKDWW\nQE4LnsqeFZYue2LxyVtkSixqsLQBicb4mMRVwHxlJDo88xkBHRysp1F3qP26RNxG\nRNwaIt4RCysKSy9TzJQdEldyiomPUQIlF7Xnk5ixqDRC9lkX0KmUVKzmQ0K6jMin\nR9u0a6zmG+cXcXm+h7ndwUHn6iD5/wSx0hyrRiGMT1drDqeNdJeyTCtvhOBUp0OL\nVZ/a1ejE5R0Ifq2mBydANhykkJEgSBNVidyCC5aSjZZehUIpZx7EimSWxotkQTp2\nU3Khor8UqAL2DGfZJsfbxXrY50PwTx7Zv3Viy+TEtjdc05y5Jya3v+2/J/Gvijf+\n+64vvPuZW478bnFm120/e+7m4/t//MCzO1ZWnvj6Cfnml35+/dldtx95/8zHH/vX\ncOfSU+d3/3LyyEPbj9+z/7pX//Dib+D5/dcdm/rr3VN/vPCBfQ9PnvnkTZ95/JGz\nb33616fetE1/9+hNW7fc99Nnf7T18Vun8p/FQ+9557fP/H727/tOTt2709+/+8an\nXjn0jhf3PvmJHVMnXrjt84eBnvtwed83/vLRvf84cePpPVOf/dbUl89N3H/h9lOt\n/qsfVN+59p7fHv3cNQ/vuf7M8y/tu+OVtx/+4oP1fU+/fOHkqfee3fvyDcu7v3fg\nhzfv/dLpPX+be/KbraUfHLrl2M5flePvmjt37M5zR7f/4uCj32/vO/WVj518lPCf\nL3ztrpN3PrC04yN/Ov/MfwCug/Aw\n=AtX0\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"f41a7d6fbb57b25bbfc781a4496dce572f77e86e\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":15,\"last_id_change\":1430852617,\"username\":\"mitsuhiko\"},\"id\":\"67aaabb56ac60890936d56f7347ccb00\",\"key\":{\"key_fingerprint\":\"d0c6a9995649e5b407483e541d8fbe76fdd3e6f2\",\"kid\":\"0101587b8827659e835fe5a9a53f70866a0dcc8634c574452b799cec2dcd68e788e90a\"},\"remote_proofs\":[{\"ctime\":1404156355,\"curr\":\"2e611ebd672c8d2a2192ace994e0c68b8c897c815e187dd7a01a148ae870f628\",\"etime\":1561836355,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"mitsuhiko\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"86d652c363de52c28013fab6fb8f69a1880f60cda87963921cc9061aa333b20b0f\",\"sig_type\":2},{\"ctime\":1404156413,\"curr\":\"c4b72d2ddd4bc9b9d61bf5b85b867670ab0c09541b2638dd02d8abdae0b683d7\",\"etime\":1561836413,\"prev\":\"2e611ebd672c8d2a2192ace994e0c68b8c897c815e187dd7a01a148ae870f628\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"lucumr.pocoo.org\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":2,\"sig_id\":\"b5eed595d035c7df78c4c41b64258f6e4dce60a8f5ce65ec5b34d3ca95d7053e0f\",\"sig_type\":2},{\"ctime\":1404156497,\"curr\":\"27f5ab0f489054c81fe81f107151a551ea11407751b3200cc73b00cfc53694ab\",\"etime\":1561836497,\"prev\":\"c4b72d2ddd4bc9b9d61bf5b85b867670ab0c09541b2638dd02d8abdae0b683d7\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"mitsuhiko\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"ced4fccf0fb74fc4b5146e4839af435231a4f3460de7b220d6df6cd39e1053eb0f\",\"sig_type\":2},{\"ctime\":1430852524,\"curr\":\"31e4fa83b6ffb964173e79b5f5d3d0e6ec746c8d51504d4ad89e2fa8e9131633\",\"etime\":1588532524,\"prev\":\"f25be273916f6e96f5c25682a26ec80a7909be0690655d6b3edccf16bf222a21\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"mitsuhiko\"},\"proof_type\":4,\"state\":1},\"seqno\":5,\"sig_id\":\"8c5c690c601e42ec0b3e185f228af08178a9591fea3b1a385df8730ff0b1a9110f\",\"sig_type\":2},{\"ctime\":1430852571,\"curr\":\"8b7528d662cb7d89883146a0fa13af9e630b81732516d7e163e75ad3d5c295e3\",\"etime\":1588532571,\"prev\":\"31e4fa83b6ffb964173e79b5f5d3d0e6ec746c8d51504d4ad89e2fa8e9131633\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"the_mitsuhiko\"},\"proof_type\":6,\"state\":1},\"seqno\":6,\"sig_id\":\"76ea910f1781cbded6ae954cbcb250a15417c7845a23a8d7d3f83cffcf53b56c0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"8b7528d662cb7d89883146a0fa13af9e630b81732516d7e163e75ad3d5c295e3\",\"seqno\":6,\"sig_id\":\"76ea910f1781cbded6ae954cbcb250a15417c7845a23a8d7d3f83cffcf53b56c0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1433274004,\"expire_in\":157680000,\"prev\":\"e6ed42cb01be8914b2fba11faffd431729fecdd21c2e1d1470c58167c7122aa7\",\"seqno\":28,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1433274004,
      "etime": 1590954004,
      "rtime": null,
      "sig_status": 0,
      "prev": "e6ed42cb01be8914b2fba11faffd431729fecdd21c2e1d1470c58167c7122aa7",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    },
    {
      "seqno": 29,
      "payload_hash": "9efc3ff3f6515f5c720b660bb8f705cbc242c4859e666d965243ed99ad7aedb1",
      "sig_id": "234f77b28a058b7625cbd73ebe28e2c09a4f5780f08657cec0e6a71ccabad1520f",
      "sig_id_short": "I093sooFi3Yly9c-vijiwJpPV4DwhlfOwOan",
      "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMORAnictVRdiFVVFL6m5jhlTkQZMqKdMkNG279n7zNTyBRDgRGCpmnFdf/OPd6Z\nc2bOOTPjKBNEUUiSBkGU+NCv9TAyPRSiEjqZKA1kTyERGYj2o/MQgZIErXOdbAZ6\nrPNw2Peudb71fd9ae51YMLvSPKv/lXpt9PFVZNZXJ4qBylPbn35wZ6BTOxy07wxM\nT+ySojwlqtcF7UHdDWuVu9VxumzI6aAtGHRZHqcJhDzDStjQa82FJlxrb4TEirEo\ntMZxQbwQToYuGGkrYUpQHyfdLuvL4rJGwKUijFtqJXxuMPehcdR7L7AUoQ6lskwS\nSwlUraV5MYNN0MCsxhb+/Zf8eiOAMMLUIU5DEipLaAhsJPBSguuQKa4lopjwSCDN\njRJKaGEjaWgEPyQoAjUEKYAbaMAxrLEX3GvOEQstFShEkhGuKEESI1Qm5i6bci41\naZKkWRjS0oAiU6ZeWgD8Y5OXp9hWb5hJ24IelRegp2pqCkwK2jGjNGKERmQGbHec\n5LU4KTEbpELlOBLMIQRCkKPYUOGItT7iTkkcBTfML/2a2QDCpPVEIu18pIlVhkUO\nK2NDRDAlBlqhCObIzjDUERcKB1k4dFoxbOCEQ2IElcJFmFsLL6oZEVIjxzXDTtsQ\nHHdYS2YFGArcM9ebFq7al6WpBzeegdEr4t5pslnYFpiBLIOqEkmpHGhhWlEUychS\nwLfQMG0k48TKslU8ohHHJb6JvBZcCIJdGCopgb6bAucRUL0OPsWgdKXBojH+NWfq\nVasKVd2Wl325cQ+KobgoXDazxd0qrrskiYfTTO0YKGU1kKrFcB+EoW95oYqyLERy\n15+kcIRT3H19biVSAkYtEswbGEgqBFcMVCAvpbPGG+QsxUwoyrDHBlzxFFPLhLRY\nIIuQD66jTdUbea5RplqouKek3qeGe1JlqzWV1/4bG/8fESPl/WhImLom07ZMad30\nyeAw5hz6ub0vzlw1LjO4CCWCp3TfDQJIRAWMLqaGltfbWe+oZAoxjyItBQaisAMg\nQIhTNoIsiTEKwQHYR0R5949QEgEz1Q2YoDdRxUAG22z82LNzKrOaKzfPvalcnpXm\n+S1/r9QncFPl5OqJg6eOPd/54S0X1nYt/rL96KKW8fM/n9nwOVn85Lyuw3Gw7sij\nh9X5Vzv2T34698+fks6Jesuauw+8u3LN7vtHO3cfWHR21+/37l12rfXIqjP9nc0L\nJ17Yt/SNP35sb58vjnUtObt29Nal9JHDJ+v6vb2tmy9/e6WjFR+qNB0fX7fhwtqL\ndxRXP3n/t/1fn/6m5dTGhRfH4i9ejza9s/7jK6uHBt7a80B+9fgP62f7cxuTdHLH\noYe2nFfff9ax5ZeuXx+bvWvf8tvnH73z4MrxPcGC7+5qbRn56J5zm5dc3pSNdaxJ\nx15bsWLxbQdeWt7U2zK2dXLe2y+a0W3L77tUXHsz33r65YfHF3WNTFwcvDTnA/oX\nDIMM3w==\n=uNE8\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"f41a7d6fbb57b25bbfc781a4496dce572f77e86e\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":3,\"last_id_change\":1433942392,\"username\":\"ginshin\"},\"id\":\"6ae5074e002590e31c37e2ddf95ea819\",\"key\":{\"key_fingerprint\":\"248df280bef9b2dac49e1acd602132cfffa2150d\",\"kid\":\"0101e2e67eacd16eba41ccd1162c7387e915dd9153b4278b0e5b41ebd675be1b84d70a\"},\"remote_proofs\":[{\"ctime\":1433942346,\"curr\":\"8088ae8194ba30989d33b4da7abc8452d85b81593951d70ac9fb757721e66a88\",\"etime\":1591622346,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"gaikenniyorazu\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"80a79ca974fc5b63775a49390f88edcfc0ed3147a341f1c942f313d478d170d00f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"8088ae8194ba30989d33b4da7abc8452d85b81593951d70ac9fb757721e66a88\",\"seqno\":1,\"sig_id\":\"80a79ca974fc5b63775a49390f88edcfc0ed3147a341f1c942f313d478d170d00f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1433950255,\"expire_in\":157680000,\"prev\":\"937fff13c30312edfe384a04f09b8714fc72f12e22ead9c30811064528a22afe\",\"seqno\":29,\"tag\":\"signature\"}",
      "sig_type": 3,
      "ctime": 1433950255,
      "etime": 1591630255,
      "rtime": null,
      "sig_status": 0,
      "prev": "937fff13c30312edfe384a04f09b8714fc72f12e22ead9c30811064528a22afe",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
    }
  ],
  "keys": [
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v2\n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\n8PZq\n=Bh81\n-----END PGP PUBLIC KEY BLOCK-----",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: Keybase OpenPGP v2.0.12\nComment: https://keybase.io/crypto\n\nxm8EVYCbkBMFK4EEACIDAwS+Lq/XrKqk2rU7v9E5SgT7tS0NVacPwjGNfpn7vxER\n/BnCn9O0rWpy/kWsMXLIBHAnNllHFRQCByHiIy24OHft113uLqcbm4qOPmzFYHWA\ndeZNo4tsfMJOvzQmPgQRgG/NCm9jb25ub3I2NjPCkgQTEwoAGgUCVYCbkAIbAQML\nCQcDFQoIAh4BAheAAhkBAAoJEMN1GTqC1mbK3UoBf1SEPEo0gaKlmTVtbUZWW7dL\npnFWrfgCjHMXpL3OPR0vkMs+hRdIwA1+OKhmNFMWmwGAgRx/HYFBpEJ6M9pqjEeO\noowx9uXKcmgbYwfLSEkdmbTKmPrx8KvEOPprDFuusRg5zlYEVYCbkBIIKoZIzj0D\nAQcCAwSNEJ98q1RjOpZtT/4TPnPW/mCuvMdWQCyGUOCXs+a7HPksKMTVa1qK0eR+\n7klw5Bbmtqk7l+Jy1srfaRSXywz2AwEKCcKHBBgTCgAPBQJVgJuQBQkPCZwAAhsM\nAAoJEMN1GTqC1mbK0LEBfjdrB4hfYjCqVOjHkKHnv6F51nHrnZcydu7c0f6yvYzD\nZRuHNzt6zWQdxKhtr0ZTugF7Bs3uWUn4u78Y2WwlUGpqHrRHz/G5yQWrDeNPS9R/\nE6HcCbA30/MlAjw8NoGAhy/PzlIEVYCbkBMIKoZIzj0DAQcCAwQL2zLTtDUkDuBC\nF4a3SiMloVaUrEScvX0FCiunSzJjCWV6LlpKduUD8I3+maul5TIFqgtDppn1gGPu\nG7DOikDowsAnBBgTCgAPBQJVgJuQBQkPCZwAAhsiAGoJEMN1GTqC1mbKXyAEGRMK\nAAYFAlWAm5AACgkQf7QBs+bLhv9uWgD9HMqGIEZJxnv3meya1h5w0rhBJETBy3rX\n0UtC35y3ZGMBAJ6D5P+9Bs9r9gt5tMRxNHQRsDY5iS4QCLh0ENLNdWwZYZoBf3T7\n6y5wJrJHN2DBMd1faT9l59AHKDeluEiapwbBhBdJ2EyBoUf//4jBYHCt8u9AaAGA\nqLQKTLOcPAj/75K6wtm6pXvFdKF7qyVtXUN0j23rCU9c73KZFwg0CyDgkRF/uUG9\n=gL89\n-----END PGP PUBLIC KEY BLOCK-----\n"
  ],
  "label_kids": {
    "real_eldest": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
    "fake_eldest": "01130a3f1e2a7e44492fa7c6833405a5068b931375db2005797082d731e8bf53f7df0a"
  }
}
`,
	"melvin_chain.json": `{
  "username": "melvin",
  "uid": "08e486faf56b304cf08915984f92b619",
  "chain": [
    {
      "seqno": 1,
      "payload_hash": "a00d01fc45ce203b2c4349b36b65a0d76339e6a7839f606a047ae2445714865b",
      "sig_id": "98edf54d8bbbf69e79174ae25e92efbba3ed6c22d0f193ad5f29a195d81c63980f",
      "sig_id_short": "mO31TYu79p55F0riXpLvu6PtbCLQ8ZOtXymh",
      "kid": "0101ab90843b164381844663a438e52de43288982be6828f37c105ed74aca0d0be160a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowFtUU1oE0EUTlJbsFKUHpTqoTgeDLrqzM7s7GwQhVKoHrReBAuWdWZ3Nh2TbuJu\nklpCGvEHqvVQ8aIgpejNg4dCEDEU6alQUAShxYNQaLwLgnhocbbizbm8ed/fg/fm\n+7pSmfTTu+vH7PZMIb32YVWkrj5smXUgSv40yNVBQe6WQIV5GZUjFVZADgTCYZwi\nnzqOL6CNqU8Cwi3mYGb6tomxLTxbUgEMMFGKE4eOETyWp1VJY7pxla/R/+gLuwRE\nEHHhQEawQJRghhghlGKuv9IyfUmwyZjDTCEpM1mAbQ9BS/o24R6HPhQSUch1XPVv\nHJOE0YAHFhUYEi+AzEGWw0jgmIIiJxHGMgr5pNTqSVmsqRA0DFCZLifAlBSupmvK\nk65Qoa93oR01GcWqFIIc0kqvohIvItiCjCGbGEDeLqtIuipRWDZlUD8DlCNZA7mw\nWiwaIJa3wpJm9SCe13NilQ95pRpJ0OidfbEnlc6kerozyT1SvXsP/LvS/ZWuHetg\n89HJlz1ns8vNVffx0uCa/JH91CrfFO9fb/rLv44PDRndp5pz9vyXe1NLW8/bAxe/\nGqN9pDV4/bMLDh/5mT2x0DiXnru0b/zO6MjsxscR91DwZnzrVf+ZzoW3l9e/Fxdv\n7O//faXdmbn27tliffjJ8M63zbHthbHzD1asowOdPw==\n=/ACy\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"fb98a61d699db0736d4f4a589382d72337bc7e6b\",\"host\":\"keybase.io\",\"key_id\":\"9382d72337bc7e6b\",\"kid\":\"0101ab90843b164381844663a438e52de43288982be6828f37c105ed74aca0d0be160a\",\"uid\":\"08e486faf56b304cf08915984f92b619\",\"username\":\"melvin\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1435088174,\"expire_in\":157680000,\"prev\":null,\"seqno\":1,\"tag\":\"signature\"}\n",
      "sig_type": 1,
      "ctime": 1435088174,
      "etime": 1592768174,
      "rtime": null,
      "sig_status": 0,
      "prev": null,
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "fb98a61d699db0736d4f4a589382d72337bc7e6b"
    }
  ],
  "keys": [
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v2\n\nmI0EVYm1EwEEAPynyMcyNok+alsFd9GYM6femohjRckCeFv5IU4nJ/th3S7Iwp0+\nTVi5jl5TGhJHTqo6S09J/tovDUXzPFTNlSfR7iY0ijNvr+5Yunwg/Gr962pJsbHH\nYDfAuSLyOgX7+YfM4bVJbma4knxEcxvWlQDUuZv5ZqMj+n2eqdJ+JTwdABEBAAG0\nBk1lbHZpboi/BBMBAgApBQJVibUTAhsDBQkHhM4ABwsJCAcDAgEGFQgCCQoLBBYC\nAwECHgECF4AACgkQk4LXIze8fms4zgP+NvVnQvYpXHlVk3Igmg3uMW+c7EMsvhSW\nnl45Hpd2ExV5QZuJPzB2qPS8T+GsQs7q2uiQpcK6JxU/79QmHuer0o63syHr6Uvh\nqmNiH02A59iAXlM+29gGpuSC0Wxck4E/qxcQcje+8w++vguqF6DboOdOanRU7VMA\n91DwSuUFz8q4jQRVibUTAQQA557twd/jg2cuF3N7MbpEf6qA0j4qDrNz/N2FJtOg\nB/Save4ovTP7ISySXjJu94aSsh4ePQt9io7SwZ2rbxHrqhKa8S80pm4Rp7Nyn8be\n6Exn9hllEAEmBmxeK85mVKt2XDpx8mo7pBes2Ia4BsjtVOzlBg3zWXDvsxNGzLkZ\nV00AEQEAAYilBBgBAgAPBQJVibUTAhsMBQkHhM4AAAoJEJOC1yM3vH5rk6UEALwv\nfcT9BQp9k7wX8gtFXMcaGpKJkKHDC71pe4Q9BmXQ4ZjGqPyIwaTEeE2uWCLeayfX\niU5W60iZJiCRxTj2C7hJdUTfrljDvLQvzfHC7Ryz/jOg+O0BCy2kbk+5ZAee7sfB\nYBF+ZLMK932igqwRVVJHOZvw0jczmp2dJzQ4F+pR\n=7zUs\n-----END PGP PUBLIC KEY BLOCK-----"
  ]
}
`,
	"nonexistent_subkey_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgyakGGaFdDQpMpW0GuYDIxls3tgVRHwtOgq71fwfPBCUKp3BheWxvYWTFASt7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMGM5YTkwNjE5YTE1ZDBkMGE0Y2E1NmQwNmI5ODBjOGM2NWIzN2I2MDU1MTFmMGI0ZTgyYWVmNTdmMDdjZjA0MjUwYSIsInVpZCI6Ijc0YzM4Y2Y3Y2ViOTQ3ZjU2MzIwNDVkOGNhNWQ0ODE5IiwidXNlcm5hbWUiOiJtYXgzMiJ9LCJ0eXBlIjoiZWxkZXN0IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDM1MDk1NzQxLCJleHBpcmVfaW4iOjMxNDQ5NjAwMCwicHJldiI6bnVsbCwic2VxX3R5cGUiOjEsInNlcW5vIjoxLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQCj9Ka+PuO9jiy9SAB5inJjKHkB4oq00QExNDIG1u1TW42CPC1VUk+v/A6zAnUVon+Vp+EYSmwC+Izr8+xg49QCoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "57e3d3757262e4b0692c6c1ccc0a57443aa314af8ec7f7d21760a024148951cf",
            "sig_id": "29cba60382bc5c81244acf8cc40cbe2ff14cf35f7b471fb16a5ccf42d15694ef0f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"0120c9a90619a15d0d0a4ca56d06b980c8c65b37b605511f0b4e82aef57f07cf04250a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1435095741,\"expire_in\":314496000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "0120c9a90619a15d0d0a4ca56d06b980c8c65b37b605511f0b4e82aef57f07cf04250a",
            "ctime": 1435095741
        },
        {
            "seqno": 2,
            "prev": "57e3d3757262e4b0692c6c1ccc0a57443aa314af8ec7f7d21760a024148951cf",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgyakGGaFdDQpMpW0GuYDIxls3tgVRHwtOgq71fwfPBCUKp3BheWxvYWTFAoJ7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYzlhOTA2MTlhMTVkMGQwYTRjYTU2ZDA2Yjk4MGM4YzY1YjM3YjYwNTUxMWYwYjRlODJhZWY1N2YwN2NmMDQyNTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYzlhOTA2MTlhMTVkMGQwYTRjYTU2ZDA2Yjk4MGM4YzY1YjM3YjYwNTUxMWYwYjRlODJhZWY1N2YwN2NmMDQyNTBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInN1YmtleSI6eyJraWQiOiIwMTIxMWIyM2ZlZmM3Yzg5NTFlMzQzNTgwMTJlZWNkYTNmNzg5YjkyNTAwYTc0OWMzMTg0NzI1YWEwYTU2MmU0OWY2NjBhIiwicGFyZW50X2tpZCI6IjAxMjBjOWE5MDYxOWExNWQwZDBhNGNhNTZkMDZiOTgwYzhjNjViMzdiNjA1NTExZjBiNGU4MmFlZjU3ZjA3Y2YwNDI1MGEiLCJyZXZlcnNlX3NpZyI6bnVsbH0sInR5cGUiOiJzdWJrZXkiLCJ2ZXJzaW9uIjoxfSwiY3RpbWUiOjE0MzUwOTU3NDEsImV4cGlyZV9pbiI6MzE0NDk2MDAwLCJwcmV2IjoiNTdlM2QzNzU3MjYyZTRiMDY5MmM2YzFjY2MwYTU3NDQzYWEzMTRhZjhlYzdmN2QyMTc2MGEwMjQxNDg5NTFjZiIsInNlcV90eXBlIjoxLCJzZXFubyI6MiwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEDopHjQ1+pUxyQ65K9Z4c77uhVWhF21+UmHqJqV+1E/nMX822yTus5QqAPgfZPC4lc7vBsmdcV7rWsrXrN6qC8JqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=",
            "payload_hash": "df9bc2d737831dd3c5ea6e8f0e66a33ced4371e76afbe6dec3dba2c28e65f3a1",
            "sig_id": "fc147293e3f476c1e37038bf90dc49233d3b51ff31479abbab3c286fdea954900f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120c9a90619a15d0d0a4ca56d06b980c8c65b37b605511f0b4e82aef57f07cf04250a\",\"host\":\"keybase.io\",\"kid\":\"0120c9a90619a15d0d0a4ca56d06b980c8c65b37b605511f0b4e82aef57f07cf04250a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"subkey\":{\"kid\":\"01211b23fefc7c8951e34358012eecda3f789b92500a749c3184725aa0a562e49f660a\",\"parent_kid\":\"0120c9a90619a15d0d0a4ca56d06b980c8c65b37b605511f0b4e82aef57f07cf04250a\",\"reverse_sig\":null},\"type\":\"subkey\",\"version\":1},\"ctime\":1435095741,\"expire_in\":314496000,\"prev\":\"57e3d3757262e4b0692c6c1ccc0a57443aa314af8ec7f7d21760a024148951cf\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "0120c9a90619a15d0d0a4ca56d06b980c8c65b37b605511f0b4e82aef57f07cf04250a",
            "ctime": 1435095741
        }
    ],
    "_comment": "DELETED A SUBKEY KID HERE",
    "keys": [
        "0120c9a90619a15d0d0a4ca56d06b980c8c65b37b605511f0b4e82aef57f07cf04250a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32",
    "label_kids": {
        "e": "0120c9a90619a15d0d0a4ca56d06b980c8c65b37b605511f0b4e82aef57f07cf04250a",
        "s": "01211b23fefc7c8951e34358012eecda3f789b92500a749c3184725aa0a562e49f660a"
    },
    "label_sigs": {
        "e": "29cba60382bc5c81244acf8cc40cbe2ff14cf35f7b471fb16a5ccf42d15694ef0f",
        "s": "fc147293e3f476c1e37038bf90dc49233d3b51ff31479abbab3c286fdea954900f"
    }
}
`,
	"ralph_chain.json": `{
  "username": "ralph",
  "uid": "bf65266d0d8df3ad5d1b367f578e6819",
  "chain": [
    {
      "seqno": 1,
      "payload_hash": "f4f8779457bc3053ee4f04637934ace1705344aeee8a4b38dcffbbf9c9268d4e",
      "sig_id": "55b70fe0e7d5d8605e6607de1daf77c3e862050ad6ae491b5b7829e05130bf580f",
      "sig_id_short": "VbcP4OfV2GBeZgfeHa93w-hiBQrWrkkbW3gp",
      "kid": "0101dbd7f3aaf965df1f60839b970f92b5f368046d693d5ea4e20aa564cab672820c0a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMHhAnicbVFbSFRBGF4rIzdbLOuhoqAxuu22ncuey6xmWFFBIVb0YCnLnJ0568H1\n7Hb2oqIrWwkSUkGZ+GAG3ZAeukiCpLkiIYEJ2kXzRSg0SGQDRci0yxzJIGgeZpjv\n/77vn2/+l2uWW6wpaVOt+Rva68dS+nvmI5Yzx97UVwElgCuBuwqUksVD1XQfMYKG\npoeBG6iizLugC0kcIwsQ0p1DhPci6PIiHmJJIRBipBDgACWBkKmgNgoKEacWoBi9\neDRM0f/wSxcLDMuwWMGSyiOkQlHAKquKjMxDBUqMCjlFUHlRZlwiFiGPBYJchGMQ\nEkRqqIgSJ3OMl0HULrJop6iiwIkiZrCMqSUWMKvwoqQKkkxEmYUmMUQMHZURyjaQ\nP1gCYg5AoajmJWb+pRLBWAv/Sw+TUJgYf0XhyqCJlhPF80fvUTQd0/+jsigxQlpA\nB26WMr1hzTRgXTxNJsgC6wCkIqgZxKOZDEGiCelygKBBosCtR/x+803n9QCt0kbI\nR/uENJ+OwhGDgFhvT9EKS4rVsjJ1mTlDizUtY2myte2rLDeSZ5ubh/Zdi7UTpu/j\ns11FTi7JJ9TMo7aiK0O+TeLzvtGqd6fzMivu5naNOpn01WV3Be5O/ER8z60PLV6Y\n+YJEh34Wz/ZeLs0erBl7W9y48wBM+BYOFcRc3U79/ffD+V9q5wau385Jn9/deaHh\n0qyjI9u1/oGRtOcWfDp5b+NU9XjQOpLX5YjO5D+9HxtOTrRlPII2jRnP+Tzds3m7\nezCrcWDHj7zH9tYyUBiPObNuavpkUwTMCfHpjrriznOJq/vnXk96LiaD/pkt5b+E\n/t7uU3sbvz15tXX0SLa9sMmaVtNQXWdLtKxdaMtY17Hta+7BhyMTzQt2/bhtOPU3\nnTkY3w==\n=DD0n\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"f683494a72085992082ae3ca94ca39d7be99dabe\",\"host\":\"keybase.io\",\"key_id\":\"94ca39d7be99dabe\",\"kid\":\"0101dbd7f3aaf965df1f60839b970f92b5f368046d693d5ea4e20aa564cab672820c0a\",\"uid\":\"bf65266d0d8df3ad5d1b367f578e6819\",\"username\":\"ralph\"},\"service\":{\"name\":\"reddit\",\"username\":\"testerralph\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1430835851,\"expire_in\":157680000,\"prev\":null,\"seqno\":1,\"tag\":\"signature\"}",
      "sig_type": 2,
      "ctime": 1430835851,
      "etime": 1588515851,
      "rtime": null,
      "sig_status": 1,
      "prev": null,
      "proof_id": "af85baace455686a5e809a10",
      "proof_type": 4,
      "proof_text_check": "\n\nyMHhAnicbVFbSFRBGF4rIzdbLOuhoqAxuu22ncuey6xmWFFBIVb0YCnLnJ0568H1\n7Hb2oqIrWwkSUkGZ+GAG3ZAeukiCpLkiIYEJ2kXzRSg0SGQDRci0yxzJIGgeZpjv\n/77vn2/+l2uWW6wpaVOt+Rva68dS+nvmI5Yzx97UVwElgCuBuwqUksVD1XQfMYKG\npoeBG6iizLugC0kcIwsQ0p1DhPci6PIiHmJJIRBipBDgACWBkKmgNgoKEacWoBi9\neDRM0f/wSxcLDMuwWMGSyiOkQlHAKquKjMxDBUqMCjlFUHlRZlwiFiGPBYJchGMQ\nEkRqqIgSJ3OMl0HULrJop6iiwIkiZrCMqSUWMKvwoqQKkkxEmYUmMUQMHZURyjaQ\nP1gCYg5AoajmJWb+pRLBWAv/Sw+TUJgYf0XhyqCJlhPF80fvUTQd0/+jsigxQlpA\nB26WMr1hzTRgXTxNJsgC6wCkIqgZxKOZDEGiCelygKBBosCtR/x+803n9QCt0kbI\nR/uENJ+OwhGDgFhvT9EKS4rVsjJ1mTlDizUtY2myte2rLDeSZ5ubh/Zdi7UTpu/j\ns11FTi7JJ9TMo7aiK0O+TeLzvtGqd6fzMivu5naNOpn01WV3Be5O/ER8z60PLV6Y\n+YJEh34Wz/ZeLs0erBl7W9y48wBM+BYOFcRc3U79/ffD+V9q5wau385Jn9/deaHh\n0qyjI9u1/oGRtOcWfDp5b+NU9XjQOpLX5YjO5D+9HxtOTrRlPII2jRnP+Tzds3m7\nezCrcWDHj7zH9tYyUBiPObNuavpkUwTMCfHpjrriznOJq/vnXk96LiaD/pkt5b+E\n/t7uU3sbvz15tXX0SLa9sMmaVtNQXWdLtKxdaMtY17Hta+7BhyMTzQt2/bhtOPU3\nnTkY3w==\n",
      "check_data_json": "{\"name\":\"reddit\",\"username\":\"testerralph\"}",
      "remote_id": "",
      "api_url": null,
      "human_url": null,
      "proof_state": 7,
      "proof_status": 0,
      "retry_count": 0,
      "hard_fail_count": 0,
      "last_check": 1430835852,
      "last_success": null,
      "version": 0,
      "fingerprint": "f683494a72085992082ae3ca94ca39d7be99dabe"
    },
    {
      "seqno": 2,
      "payload_hash": "bdc9430e56f7d59eb9e6709cb719b4a8479ad3272c046e3c915a113e0892f8a7",
      "sig_id": "32c195e0fca5c310237f6180b29315afd2cfc4ddce9cb307b50cc4a9172bdbd30f",
      "sig_id_short": "MsGV4PylwxAjf2GAspMVr9LPxN3OnLMHtQzE",
      "kid": "0101dbd7f3aaf965df1f60839b970f92b5f368046d693d5ea4e20aa564cab672820c0a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMIQAnicbZJbSFRBGMd33a21JbsQFV0wPBVFSJ37nNnSpKDbQ0UlQhe2mTMzuwfb\nS2eP642NSIrKh3wQMgojMs0IMcyH7KZZLy09dH3oYg+RdqMLEZFGNEcsCJqHGeb/\n/b8f8/2Z23k+T9A74cP5jdO76we82b5Qhad03YOWWgEnSLUQqhXK6ejBrHiE2knb\nijtCSGC6oahQRUAWDQ1CvsuIKiaCqokUSACmEBKEqVAoRBMpt4NjMErRpVaCa/wS\ntghX/+MvHy2IkigRTABTEGJQ1wiTmC4aCsQQiAzKWGOKboiqTnSoEI0ilcoiQprO\ngVgHsiGLpog4rmIUh5muybpORGIQjiQakbCiA6YBg+qGBF1jitpxFKPcbaM9yaiQ\nKRS4lLZM6s4/VnIqLceh9r9+h6a49rfLqU66aiXF4TFAGFtxwgPkbWlqp6xEXAhJ\n3Gk6lguQVIWPphlALxRoVdKyadhyHRrgI/JVKCRtmnZjV5kBAFQ1gE1F1BRKVcYz\nUABUVGRSCXBNVRGl1EAqVgxiMoYxgyaUdYOobrwpujeeEEIyfyaKcGTKisSRU2FT\nIXOrd6ff4w16xo/Lcb+AJzhhyp+PMSTmejrfFQy1X27+OLO/Z/LZWHBFpvMKe2K/\nbiw7PjUdrjy6c97Kc9saJW/s1Podfq1uX+3TlsH6o5ULh0/kPPu1o6Z4c6ZkV11u\nnOYfPpM9MuIPRtnzNqXg0TIjFvha9rmv5cdsWvzK77m4YNqnu48/VfW31g7EFiXq\nNkxquD/Um9d+GkVnPPB9+/kQvOwpuT67FXdFam4u7TqQCfjWZPOzDYOXqjsXG59b\nr395I83qPpg350ejtGTgwsSO0q2HcvZfqw88n78q2QS2tNH84ZIb7xt9y1d9752S\n2zH57srm7vKiL0xc33Rv99zlq9du6D148e1gUeDF9o6T8Nkd/9Vj6ZFIzaaPvwGx\nDCYq\n=4hWA\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"f683494a72085992082ae3ca94ca39d7be99dabe\",\"host\":\"keybase.io\",\"key_id\":\"94ca39d7be99dabe\",\"kid\":\"0101dbd7f3aaf965df1f60839b970f92b5f368046d693d5ea4e20aa564cab672820c0a\",\"uid\":\"bf65266d0d8df3ad5d1b367f578e6819\",\"username\":\"ralph\"},\"service\":{\"name\":\"twitter\",\"username\":\"testerralph\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1430835876,\"expire_in\":157680000,\"prev\":\"f4f8779457bc3053ee4f04637934ace1705344aeee8a4b38dcffbbf9c9268d4e\",\"seqno\":2,\"tag\":\"signature\"}",
      "sig_type": 2,
      "ctime": 1430835876,
      "etime": 1588515876,
      "rtime": null,
      "sig_status": 1,
      "prev": "f4f8779457bc3053ee4f04637934ace1705344aeee8a4b38dcffbbf9c9268d4e",
      "proof_id": "9d8b0ec56bbda063e5241b10",
      "proof_type": 2,
      "proof_text_check": "Verifying myself: I am ralph on Keybase.io. MsGV4PylwxAjf2GAspMVr9LPxN3OnLMHtQzE /",
      "check_data_json": "{\"name\":\"twitter\",\"username\":\"testerralph\"}",
      "remote_id": "595594965541064704",
      "api_url": "https://twitter.com/testerralph/status/595594965541064704",
      "human_url": "https://twitter.com/testerralph/status/595594965541064704",
      "proof_state": 5,
      "proof_status": 1,
      "retry_count": 1,
      "hard_fail_count": 0,
      "last_check": 1430835881,
      "last_success": 1430835881,
      "version": 1,
      "fingerprint": "f683494a72085992082ae3ca94ca39d7be99dabe"
    },
    {
      "seqno": 3,
      "payload_hash": "4cc9ac722ffcd53b5de91a37b3a7ede9e80cc3893eb64bdc5988b18eb26a97d6",
      "sig_id": "4200f01dc416139995cd534d577ef5c14b39fde69a6e57ccd3f1b6d9212b53fc0f",
      "sig_id_short": "QgDwHcQWE5mVzVNNV371wUs5_eaablfM0_G2",
      "kid": "0101c304e8c86c8f4b6773478eed4d05e9ffdddc81c7068c50db1b5bad9a904f5f890a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n\nyMIRAnicbZJ/aI1RGMfvHZdcxpLtD7RysoQ73ve+73ve817dpikzkpKVP65u57zn\nnO1t3Hu973vvzELajJjLsCxDITVKSZuhxQwh/7BsiSn/XG1+FVqK/DjvQinn/HF6\nvs/3+fQ8p+du/gRf0H/ie0/94iWnyvyPbkfSvqpVT4saAEnSehBpALVs/OFWoprZ\nKdtKuCACiBmGHCschxkRV9UUGVFEDQ0TLCPOsQplrjAKQqAm6XgVAkOww5ZYSaGJ\nIG5Rof7HXzuekGRJNhVJZchE0ERcJVDXFVVHjFGVShozOKeUmkg2dQkiU5MokYlG\nMDWwIalc48iQsMClx3GEQy0MIZVEk1zBVKMyUaDONQGESDY8o8PsBN7ChNvGm1M1\nYEcICCljmcyb/3fKrbNcl9n/+l3mCO1vlVuf8tQ6RuK/AXFiJaj4QFGWYbZjJRMg\nIgun6VoeQFYVCSkwrMIQYNtSls3ilufQdIgkcUIgZbOMNwY1DeFlGuQ61QxGDAZ1\nyTCJLhtExUjVDUyVsB42JRUyxTRkDcuywiRkhDnCOvBG2ppIgogi2sTVAulY1Qns\npm0GdvT3xSb6/EHfpECetwK+4JSCP4vRu2fyj7LCn5vO95VP2+jMel46ve/b3OyG\njoHDu0pocVOnYa0YGrx29VhbO/h8q7JluHGdFRut7CjomXO2u2NLNIQr5795dZpc\n+XBr2tjxHBs7dGaA3wvsHGwcKh/JP5DX+6T8kO7ezsaO7PeN5Hbff7fww9Zs07Kl\nu5yW3M2VRWvCB8HjpRezjdGmVrrv3Eh38mT/TlrjwPbQpzs/jGAq+WxW4dvKsRkL\nLn166RbldS1fFr8Drs/uHdo+83Xz9tU3Wi42ZFpjP9FoSdtDe2rL0ZNzvrad3hu4\nolVcfvF476J5pSNv1w32fEk3+7r82su1Dypao8NnAlWnhqO59Rfq8z/a74s7X/wC\nT2QsJA==\n=eyGB\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"bc26fa3fa2ebebe45318d8d95aba18ffa461f3ed\",\"host\":\"keybase.io\",\"key_id\":\"5aba18ffa461f3ed\",\"kid\":\"0101c304e8c86c8f4b6773478eed4d05e9ffdddc81c7068c50db1b5bad9a904f5f890a\",\"uid\":\"bf65266d0d8df3ad5d1b367f578e6819\",\"username\":\"ralph\"},\"service\":{\"name\":\"twitter\",\"username\":\"testerralph\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1430836246,\"expire_in\":157680000,\"prev\":\"bdc9430e56f7d59eb9e6709cb719b4a8479ad3272c046e3c915a113e0892f8a7\",\"seqno\":3,\"tag\":\"signature\"}",
      "sig_type": 2,
      "ctime": 1430836246,
      "etime": 1588516246,
      "rtime": null,
      "sig_status": 0,
      "prev": "bdc9430e56f7d59eb9e6709cb719b4a8479ad3272c046e3c915a113e0892f8a7",
      "proof_id": "35dc42fe28fdab9085277e10",
      "proof_type": 2,
      "proof_text_check": "Verifying myself: I am ralph on Keybase.io. QgDwHcQWE5mVzVNNV371wUs5_eaablfM0_G2 /",
      "check_data_json": "{\"name\":\"twitter\",\"username\":\"testerralph\"}",
      "remote_id": "595596523460796418",
      "api_url": "https://twitter.com/testerralph/status/595596523460796418",
      "human_url": "https://twitter.com/testerralph/status/595596523460796418",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 5,
      "hard_fail_count": 0,
      "last_check": 1431006765,
      "last_success": 1431006765,
      "version": 1,
      "fingerprint": "bc26fa3fa2ebebe45318d8d95aba18ffa461f3ed"
    },
    {
      "seqno": 4,
      "payload_hash": "6069efa710426907e15a734f5f15e32730aa02b133feb8cf718b22cf44faa640",
      "sig_id": "5c4da82af7e168cc4c6708be87d28c8558c7e8801b826d7af8b7049d246c32600f",
      "sig_id_short": "XE2oKvfhaMxMZwi-h9KMhVjH6IAbgm16-LcE",
      "kid": "0101c304e8c86c8f4b6773478eed4d05e9ffdddc81c7068c50db1b5bad9a904f5f890a",
      "sig": "-----BEGIN PGP MESSAGE-----\nComment: https://keybase.io/download\nVersion: Keybase Go 0.1.12 (linux)\n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\nbGdcbgZ/DD5krt8lsCABbPZeuiuGQlHHvhd6pQeG\n=E0qo\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"f9f52a2350bf0804804347dba83ca918\",\"type\":\"desktop\",\"kid\":\"0120afac9383132b317336866b4ce48836906cad88de18a92bddf29ffb949db8bc420a\",\"description\":\"stuff\",\"status\":1},\"key\":{\"eldest_kid\":\"0101c304e8c86c8f4b6773478eed4d05e9ffdddc81c7068c50db1b5bad9a904f5f890a\",\"fingerprint\":\"bc26fa3fa2ebebe45318d8d95aba18ffa461f3ed\",\"host\":\"keybase.io\",\"key_id\":\"5ABA18FFA461F3ED\",\"kid\":\"0101c304e8c86c8f4b6773478eed4d05e9ffdddc81c7068c50db1b5bad9a904f5f890a\",\"uid\":\"bf65266d0d8df3ad5d1b367f578e6819\",\"username\":\"ralph\"},\"sibkey\":{\"kid\":\"0120afac9383132b317336866b4ce48836906cad88de18a92bddf29ffb949db8bc420a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgr6yTgxMrMXM2hmtM5Ig2kGytiN4YqSvd8p/7lJ24vEIKp3BheWxvYWTFBBx7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiZjlmNTJhMjM1MGJmMDgwNDgwNDM0N2RiYTgzY2E5MTgiLCJ0eXBlIjoiZGVza3RvcCIsImtpZCI6IjAxMjBhZmFjOTM4MzEzMmIzMTczMzY4NjZiNGNlNDg4MzY5MDZjYWQ4OGRlMThhOTJiZGRmMjlmZmI5NDlkYjhiYzQyMGEiLCJkZXNjcmlwdGlvbiI6InN0dWZmIiwic3RhdHVzIjoxfSwia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTAxYzMwNGU4Yzg2YzhmNGI2NzczNDc4ZWVkNGQwNWU5ZmZkZGRjODFjNzA2OGM1MGRiMWI1YmFkOWE5MDRmNWY4OTBhIiwiZmluZ2VycHJpbnQiOiJiYzI2ZmEzZmEyZWJlYmU0NTMxOGQ4ZDk1YWJhMThmZmE0NjFmM2VkIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJrZXlfaWQiOiI1QUJBMThGRkE0NjFGM0VEIiwia2lkIjoiMDEwMWMzMDRlOGM4NmM4ZjRiNjc3MzQ3OGVlZDRkMDVlOWZmZGRkYzgxYzcwNjhjNTBkYjFiNWJhZDlhOTA0ZjVmODkwYSIsInVpZCI6ImJmNjUyNjZkMGQ4ZGYzYWQ1ZDFiMzY3ZjU3OGU2ODE5IiwidXNlcm5hbWUiOiJyYWxwaCJ9LCJzaWJrZXkiOnsia2lkIjoiMDEyMGFmYWM5MzgzMTMyYjMxNzMzNjg2NmI0Y2U0ODgzNjkwNmNhZDg4ZGUxOGE5MmJkZGYyOWZmYjk0OWRiOGJjNDIwYSIsInJldmVyc2Vfc2lnIjpudWxsfSwidHlwZSI6InNpYmtleSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjAuMS4xMiJ9LCJjdGltZSI6MTQzMDgzNjM3OSwiZXhwaXJlX2luIjo5NDYwODAwMCwibWVya2xlX3Jvb3QiOnsiY3RpbWUiOjE0MzA4MzYyNDcsImhhc2giOiI4Y2EzOWU2NTc0N2M2M2VmNDcwNTk5YTc1OGU4Njc0MzUzODNhNDhiYzk0ZWUwODczYjhiNzU0ZTIxODI5MzZiZmJkYTNmZDU4ZTBiZTBhOGMyNDMxYjUyYjQ2NWFlZmQ3ZGY3YWVmYjBmYmM1OTY1NWQzYWViOTU4Njc0YWU4YSIsInNlcW5vIjo3MDR9LCJwcmV2IjoiNGNjOWFjNzIyZmZjZDUzYjVkZTkxYTM3YjNhN2VkZTllODBjYzM4OTNlYjY0YmRjNTk4OGIxOGViMjZhOTdkNiIsInNlcW5vIjo0LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQPCZWkGA2C7XR11/OBO8Iu5twquWsXw3/vXnxyJNt5l2Fhttenyle9Z7N+9KOVey/1SoVNRoK1SdUOu7KyLbrQCoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==\"},\"type\":\"sibkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"0.1.12\"},\"ctime\":1430836379,\"expire_in\":94608000,\"merkle_root\":{\"ctime\":1430836247,\"hash\":\"8ca39e65747c63ef470599a758e867435383a48bc94ee0873b8b754e2182936bfbda3fd58e0be0a8c2431b52b465aefd7df7aefb0fbc59655d3aeb958674ae8a\",\"seqno\":704},\"prev\":\"4cc9ac722ffcd53b5de91a37b3a7ede9e80cc3893eb64bdc5988b18eb26a97d6\",\"seqno\":4,\"tag\":\"signature\"}",
      "sig_type": 1,
      "ctime": 1430836379,
      "etime": 1525444379,
      "rtime": null,
      "sig_status": 0,
      "prev": "4cc9ac722ffcd53b5de91a37b3a7ede9e80cc3893eb64bdc5988b18eb26a97d6",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "bc26fa3fa2ebebe45318d8d95aba18ffa461f3ed"
    },
    {
      "seqno": 5,
      "payload_hash": "ba29de60fd420e579cfb241c3419bd781763a28f0ee9be18c5f58a1ae1e1a017",
      "sig_id": "b29a6b21766b48215e3ce7b75d6a9bd61b36e6212bf5b050b8b318cd972f742f0f",
      "sig_id_short": "spprIXZrSCFePOe3XWqb1hs25iEr9bBQuLMY",
      "kid": "0120afac9383132b317336866b4ce48836906cad88de18a92bddf29ffb949db8bc420a",
      "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgr6yTgxMrMXM2hmtM5Ig2kGytiN4YqSvd8p/7lJ24vEIKp3BheWxvYWTFBAp7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiZjlmNTJhMjM1MGJmMDgwNDgwNDM0N2RiYTgzY2E5MTgiLCJ0eXBlIjoiZGVza3RvcCIsImtpZCI6IjAxMjE2N2YxYjAyOTFhMDdmZGIxNGI3Mjk0NzUxNDdhYmYzN2QzMWM5NDc3ZDhkNjlkY2JmYTg3NDhjYjc2MDEwZDYxMGEiLCJkZXNjcmlwdGlvbiI6InN0dWZmIiwic3RhdHVzIjoxfSwia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTAxYzMwNGU4Yzg2YzhmNGI2NzczNDc4ZWVkNGQwNWU5ZmZkZGRjODFjNzA2OGM1MGRiMWI1YmFkOWE5MDRmNWY4OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYWZhYzkzODMxMzJiMzE3MzM2ODY2YjRjZTQ4ODM2OTA2Y2FkODhkZTE4YTkyYmRkZjI5ZmZiOTQ5ZGI4YmM0MjBhIiwidWlkIjoiYmY2NTI2NmQwZDhkZjNhZDVkMWIzNjdmNTc4ZTY4MTkiLCJ1c2VybmFtZSI6InJhbHBoIn0sInN1YmtleSI6eyJraWQiOiIwMTIxNjdmMWIwMjkxYTA3ZmRiMTRiNzI5NDc1MTQ3YWJmMzdkMzFjOTQ3N2Q4ZDY5ZGNiZmE4NzQ4Y2I3NjAxMGQ2MTBhIiwicGFyZW50X2tpZCI6IjAxMjBhZmFjOTM4MzEzMmIzMTczMzY4NjZiNGNlNDg4MzY5MDZjYWQ4OGRlMThhOTJiZGRmMjlmZmI5NDlkYjhiYzQyMGEifSwidHlwZSI6InN1YmtleSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjAuMS4xMiJ9LCJjdGltZSI6MTQzMDgzNjM3OSwiZXhwaXJlX2luIjo5NDYwODAwMCwibWVya2xlX3Jvb3QiOnsiY3RpbWUiOjE0MzA4MzYyNDcsImhhc2giOiI4Y2EzOWU2NTc0N2M2M2VmNDcwNTk5YTc1OGU4Njc0MzUzODNhNDhiYzk0ZWUwODczYjhiNzU0ZTIxODI5MzZiZmJkYTNmZDU4ZTBiZTBhOGMyNDMxYjUyYjQ2NWFlZmQ3ZGY3YWVmYjBmYmM1OTY1NWQzYWViOTU4Njc0YWU4YSIsInNlcW5vIjo3MDR9LCJwcmV2IjoiNjA2OWVmYTcxMDQyNjkwN2UxNWE3MzRmNWYxNWUzMjczMGFhMDJiMTMzZmViOGNmNzE4YjIyY2Y0NGZhYTY0MCIsInNlcW5vIjo1LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQBeo141bOEMnr0m/IHl72SGtJ/8wf9sVkw8uZRU0O3nZFpF0jrydXbQOMpZOv8c7eAEa7QdecLbdsmlZUtlaUAWoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"f9f52a2350bf0804804347dba83ca918\",\"type\":\"desktop\",\"kid\":\"012167f1b0291a07fdb14b729475147abf37d31c9477d8d69dcbfa8748cb76010d610a\",\"description\":\"stuff\",\"status\":1},\"key\":{\"eldest_kid\":\"0101c304e8c86c8f4b6773478eed4d05e9ffdddc81c7068c50db1b5bad9a904f5f890a\",\"host\":\"keybase.io\",\"kid\":\"0120afac9383132b317336866b4ce48836906cad88de18a92bddf29ffb949db8bc420a\",\"uid\":\"bf65266d0d8df3ad5d1b367f578e6819\",\"username\":\"ralph\"},\"subkey\":{\"kid\":\"012167f1b0291a07fdb14b729475147abf37d31c9477d8d69dcbfa8748cb76010d610a\",\"parent_kid\":\"0120afac9383132b317336866b4ce48836906cad88de18a92bddf29ffb949db8bc420a\"},\"type\":\"subkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"0.1.12\"},\"ctime\":1430836379,\"expire_in\":94608000,\"merkle_root\":{\"ctime\":1430836247,\"hash\":\"8ca39e65747c63ef470599a758e867435383a48bc94ee0873b8b754e2182936bfbda3fd58e0be0a8c2431b52b465aefd7df7aefb0fbc59655d3aeb958674ae8a\",\"seqno\":704},\"prev\":\"6069efa710426907e15a734f5f15e32730aa02b133feb8cf718b22cf44faa640\",\"seqno\":5,\"tag\":\"signature\"}",
      "sig_type": 1,
      "ctime": 1430836379,
      "etime": 1525444379,
      "rtime": null,
      "sig_status": 0,
      "prev": "6069efa710426907e15a734f5f15e32730aa02b133feb8cf718b22cf44faa640",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 6,
      "payload_hash": "cdd5ae013a6a879c605ba9611c3efff8dbfc473b9b14b2b99e8a6c00f62bbbba",
      "sig_id": "bce8a4dd488f24d42ce89ee667d549ab723d80e8b0258915af3fa8f3495dde620f",
      "sig_id_short": "vOik3UiPJNQs6J7mZ9VJq3I9gOiwJYkVrz-o",
      "kid": "0120afac9383132b317336866b4ce48836906cad88de18a92bddf29ffb949db8bc420a",
      "sig": "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",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"fb15d9d8583198e62486a623af1b4d18\",\"type\":\"web\",\"kid\":\"0120fdaf819278cb690178d38f89ff1e4464c6a39320f75632b1b5734820bd1037c70a\",\"status\":1},\"key\":{\"eldest_kid\":\"0101c304e8c86c8f4b6773478eed4d05e9ffdddc81c7068c50db1b5bad9a904f5f890a\",\"host\":\"keybase.io\",\"kid\":\"0120afac9383132b317336866b4ce48836906cad88de18a92bddf29ffb949db8bc420a\",\"uid\":\"bf65266d0d8df3ad5d1b367f578e6819\",\"username\":\"ralph\"},\"sibkey\":{\"kid\":\"0120fdaf819278cb690178d38f89ff1e4464c6a39320f75632b1b5734820bd1037c70a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg/a+BknjLaQF404+J/x5EZMajkyD3VjKxtXNIIL0QN8cKp3BheWxvYWTFA617ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiZmIxNWQ5ZDg1ODMxOThlNjI0ODZhNjIzYWYxYjRkMTgiLCJ0eXBlIjoid2ViIiwia2lkIjoiMDEyMGZkYWY4MTkyNzhjYjY5MDE3OGQzOGY4OWZmMWU0NDY0YzZhMzkzMjBmNzU2MzJiMWI1NzM0ODIwYmQxMDM3YzcwYSIsInN0YXR1cyI6MX0sImtleSI6eyJlbGRlc3Rfa2lkIjoiMDEwMWMzMDRlOGM4NmM4ZjRiNjc3MzQ3OGVlZDRkMDVlOWZmZGRkYzgxYzcwNjhjNTBkYjFiNWJhZDlhOTA0ZjVmODkwYSIsImhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMGFmYWM5MzgzMTMyYjMxNzMzNjg2NmI0Y2U0ODgzNjkwNmNhZDg4ZGUxOGE5MmJkZGYyOWZmYjk0OWRiOGJjNDIwYSIsInVpZCI6ImJmNjUyNjZkMGQ4ZGYzYWQ1ZDFiMzY3ZjU3OGU2ODE5IiwidXNlcm5hbWUiOiJyYWxwaCJ9LCJzaWJrZXkiOnsia2lkIjoiMDEyMGZkYWY4MTkyNzhjYjY5MDE3OGQzOGY4OWZmMWU0NDY0YzZhMzkzMjBmNzU2MzJiMWI1NzM0ODIwYmQxMDM3YzcwYSIsInJldmVyc2Vfc2lnIjpudWxsfSwidHlwZSI6InNpYmtleSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjAuMS4xMiJ9LCJjdGltZSI6MTQzMDgzNjM3OSwiZXhwaXJlX2luIjo5NDYwODAwMCwibWVya2xlX3Jvb3QiOnsiY3RpbWUiOjE0MzA4MzYyNDcsImhhc2giOiI4Y2EzOWU2NTc0N2M2M2VmNDcwNTk5YTc1OGU4Njc0MzUzODNhNDhiYzk0ZWUwODczYjhiNzU0ZTIxODI5MzZiZmJkYTNmZDU4ZTBiZTBhOGMyNDMxYjUyYjQ2NWFlZmQ3ZGY3YWVmYjBmYmM1OTY1NWQzYWViOTU4Njc0YWU4YSIsInNlcW5vIjo3MDR9LCJwcmV2IjoiYmEyOWRlNjBmZDQyMGU1NzljZmIyNDFjMzQxOWJkNzgxNzYzYTI4ZjBlZTliZTE4YzVmNThhMWFlMWUxYTAxNyIsInNlcW5vIjo2LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQCzPlWOmzXM2yqlf9xvzI/st9NoZl0N1C5dwaIzPgE+S37xPlb5/Q14U/SGgn4qIvuTsDF0nq/1xOtABiaJ4yQqoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==\"},\"type\":\"sibkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"0.1.12\"},\"ctime\":1430836379,\"expire_in\":94608000,\"merkle_root\":{\"ctime\":1430836247,\"hash\":\"8ca39e65747c63ef470599a758e867435383a48bc94ee0873b8b754e2182936bfbda3fd58e0be0a8c2431b52b465aefd7df7aefb0fbc59655d3aeb958674ae8a\",\"seqno\":704},\"prev\":\"ba29de60fd420e579cfb241c3419bd781763a28f0ee9be18c5f58a1ae1e1a017\",\"seqno\":6,\"tag\":\"signature\"}",
      "sig_type": 1,
      "ctime": 1430836379,
      "etime": 1525444379,
      "rtime": null,
      "sig_status": 0,
      "prev": "ba29de60fd420e579cfb241c3419bd781763a28f0ee9be18c5f58a1ae1e1a017",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 7,
      "payload_hash": "ab5399ff8a1bc85c80e621d19462ced75095544c84a854f9b9f1dbde90fd1a0e",
      "sig_id": "c55da1d425a0f9ee917b5cd3b35b31135ccb0241e51ff93cb926dbb3bba1f98b0f",
      "sig_id_short": "xV2h1CWg-e6Re1zTs1sxE1zLAkHlH_k8uSbb",
      "kid": "0120fdaf819278cb690178d38f89ff1e4464c6a39320f75632b1b5734820bd1037c70a",
      "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg/a+BknjLaQF404+J/x5EZMajkyD3VjKxtXNIIL0QN8cKp3BheWxvYWTFA/B7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiZmIxNWQ5ZDg1ODMxOThlNjI0ODZhNjIzYWYxYjRkMTgiLCJ0eXBlIjoid2ViIiwia2lkIjoiMDEyMWI4NWIyNWFkZmRjNDgzYWUwMDQzNTU3NGJkODI3MmM1Y2M5MzZjNDYzZDE3OWQ4ODkwMGM0Yjk2YTM3NGNkNmMwYSIsInN0YXR1cyI6MX0sImtleSI6eyJlbGRlc3Rfa2lkIjoiMDEwMWMzMDRlOGM4NmM4ZjRiNjc3MzQ3OGVlZDRkMDVlOWZmZGRkYzgxYzcwNjhjNTBkYjFiNWJhZDlhOTA0ZjVmODkwYSIsImhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMGZkYWY4MTkyNzhjYjY5MDE3OGQzOGY4OWZmMWU0NDY0YzZhMzkzMjBmNzU2MzJiMWI1NzM0ODIwYmQxMDM3YzcwYSIsInVpZCI6ImJmNjUyNjZkMGQ4ZGYzYWQ1ZDFiMzY3ZjU3OGU2ODE5IiwidXNlcm5hbWUiOiJyYWxwaCJ9LCJzdWJrZXkiOnsia2lkIjoiMDEyMWI4NWIyNWFkZmRjNDgzYWUwMDQzNTU3NGJkODI3MmM1Y2M5MzZjNDYzZDE3OWQ4ODkwMGM0Yjk2YTM3NGNkNmMwYSIsInBhcmVudF9raWQiOiIwMTIwZmRhZjgxOTI3OGNiNjkwMTc4ZDM4Zjg5ZmYxZTQ0NjRjNmEzOTMyMGY3NTYzMmIxYjU3MzQ4MjBiZDEwMzdjNzBhIn0sInR5cGUiOiJzdWJrZXkiLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIwLjEuMTIifSwiY3RpbWUiOjE0MzA4MzYzNzksImV4cGlyZV9pbiI6OTQ2MDgwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDMwODM2MjQ3LCJoYXNoIjoiOGNhMzllNjU3NDdjNjNlZjQ3MDU5OWE3NThlODY3NDM1MzgzYTQ4YmM5NGVlMDg3M2I4Yjc1NGUyMTgyOTM2YmZiZGEzZmQ1OGUwYmUwYThjMjQzMWI1MmI0NjVhZWZkN2RmN2FlZmIwZmJjNTk2NTVkM2FlYjk1ODY3NGFlOGEiLCJzZXFubyI6NzA0fSwicHJldiI6ImNkZDVhZTAxM2E2YTg3OWM2MDViYTk2MTFjM2VmZmY4ZGJmYzQ3M2I5YjE0YjJiOTllOGE2YzAwZjYyYmJiYmEiLCJzZXFubyI6NywidGFnIjoic2lnbmF0dXJlIn2jc2lnxEBHqe4TZEOiNGSd8Vanr53/W1aMX1wD7PSk8ag8lnvN2jmzrfzIz6PqRyiFFVRFnDuk3buCEDvQZ2EsA/aZfLUEqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"fb15d9d8583198e62486a623af1b4d18\",\"type\":\"web\",\"kid\":\"0121b85b25adfdc483ae00435574bd8272c5cc936c463d179d88900c4b96a374cd6c0a\",\"status\":1},\"key\":{\"eldest_kid\":\"0101c304e8c86c8f4b6773478eed4d05e9ffdddc81c7068c50db1b5bad9a904f5f890a\",\"host\":\"keybase.io\",\"kid\":\"0120fdaf819278cb690178d38f89ff1e4464c6a39320f75632b1b5734820bd1037c70a\",\"uid\":\"bf65266d0d8df3ad5d1b367f578e6819\",\"username\":\"ralph\"},\"subkey\":{\"kid\":\"0121b85b25adfdc483ae00435574bd8272c5cc936c463d179d88900c4b96a374cd6c0a\",\"parent_kid\":\"0120fdaf819278cb690178d38f89ff1e4464c6a39320f75632b1b5734820bd1037c70a\"},\"type\":\"subkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"0.1.12\"},\"ctime\":1430836379,\"expire_in\":94608000,\"merkle_root\":{\"ctime\":1430836247,\"hash\":\"8ca39e65747c63ef470599a758e867435383a48bc94ee0873b8b754e2182936bfbda3fd58e0be0a8c2431b52b465aefd7df7aefb0fbc59655d3aeb958674ae8a\",\"seqno\":704},\"prev\":\"cdd5ae013a6a879c605ba9611c3efff8dbfc473b9b14b2b99e8a6c00f62bbbba\",\"seqno\":7,\"tag\":\"signature\"}",
      "sig_type": 1,
      "ctime": 1430836379,
      "etime": 1525444379,
      "rtime": null,
      "sig_status": 0,
      "prev": "cdd5ae013a6a879c605ba9611c3efff8dbfc473b9b14b2b99e8a6c00f62bbbba",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    }
  ],
  "keys": [
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n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\nxZRM1+LWTRcng3OuRBlbA7nevuYBR42XdmPls5vpLAljA6jsMpQ=\n=BVOL\n-----END PGP PUBLIC KEY BLOCK-----",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: Keybase OpenPGP v2.0.8\nComment: https://keybase.io/crypto\n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\n48t0rjfPLiTMvWIEyIWZXyrvsZViyRBsaGAoU/uT9AAMu+ANPg==\n=sF0S\n-----END PGP PUBLIC KEY BLOCK-----",
    "0120afac9383132b317336866b4ce48836906cad88de18a92bddf29ffb949db8bc420a",
    "012167f1b0291a07fdb14b729475147abf37d31c9477d8d69dcbfa8748cb76010d610a",
    "0121b85b25adfdc483ae00435574bd8272c5cc936c463d179d88900c4b96a374cd6c0a",
    "0120fdaf819278cb690178d38f89ff1e4464c6a39320f75632b1b5734820bd1037c70a",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v2\n\nmI0EVYGTSQEEAOHnED1wHyg3+Jm2x6K1jG1t48v6/xUt/mui7C63VKqvwHTDk3TM\nUzV7oN7Qvek2YoXClQIUMPBvJs+nie5mohCTpHEZJr7tCwTlvPIf+4qJqQAtMr5z\n8ubnxHNsIPKg8w0R5HanM2qy2Tz+kxR4lmP/nBFNJTwRvIPDglHAh6a3ABEBAAG0\nGHJhbHBoIDxyYWxwaEBrZXliYXNlLmlvPoi5BBMBCAAjBQJVgZNJAhsDBwsJCAcD\nAgEGFQgCCQoLBBYCAwECHgECF4AACgkQ1XTwnazAlYOQQQP/aWmXflNWXMqiTwHW\nQj5J3YC3zFr312yt6sgR3IrYBLdp1aSHmk0DbfI/U2JaRzrr5Epv3efSCkK9R7vQ\nRvXxE9InsRK28PZvx+bmjuZAGD0abfPj/Ex9MiVq52veNqYzMN20o2EACdoTE9/5\ndkTKiRUGbPIolbpqjCGW6LPBJ6i4jQRVgZNJAQQAvaCnEIQ9LfPt0qECCU+9Nrz8\nNLC1XreQQ7Mm31vcSOkbypEhil1eK5+TkP84UTBkdMzQvX77r+BqRrqT4k2oXZD2\nw+KjDV48Lzb2qmRd+kezTK7jLKNwqcEvMyJMhi8wyU8HD95KM9is1TdFr045JlIY\nzThgskx1jXHfnteWq7sAEQEAAYifBBgBCAAJBQJVgZNJAhsMAAoJENV08J2swJWD\nf0oEAML7lg1p/m6auEY/WH13hRUndb2JLvnqMxpd4dwh5xQzyFLSMBKG2xnUO4P8\n9zliTnelXHQVzJVzv0k0JHzc1027aJO+YAPjT5RTd+QwuoA90DBhDzNC9GQBn3kl\n0q45q+rHj3/1XWXKjJNUKasAlKK7F+fJoLVHBTmQUoNt9ZT+\n=jMe5\n-----END PGP PUBLIC KEY BLOCK-----"
  ],
  "label_kids": {
    "first_eldest": "0101dbd7f3aaf965df1f60839b970f92b5f368046d693d5ea4e20aa564cab672820c0a",
    "second_eldest": "0101c304e8c86c8f4b6773478eed4d05e9ffdddc81c7068c50db1b5bad9a904f5f890a",
    "new_eldest": "01014d569e80644c482092be218bc9faee07f068219c317233d2f8823f742a519dca0a"
  }
}
`,
	"t_helen_chain.json": `{
    "username": "t_helen",
    "uid": "9cbca30c38afba6ab02d76b206515919",
    "chain": [
        {
            "seqno": 1,
            "payload_hash": "ed2d411f26584e8f9aaac8497715840f7701f10df114b097a554a8b6a8d20e23",
            "sig_id": "d020f47d51011e8fe3ca042ca6639ca56acb6b53f53c8f8db25da308fd10c2b50f",
            "sig_id_short": "0CD0fVEBHo_jygQspmOcpWrLa1P1PI-Nsl2j",
            "kid": "01016690eb0ddc554cb4ac827497b398abf96d1d3d2ddfff9a118a05b70d91be59090a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowFtkT2IE0EUxzc5FbzzMIhgo80UV63H7OfsBGxECwtFg9HOOLPzNjck2Y27+TCG\nq0QuKnjiB4Kx8QMFEcNdYyNyeuKhuVII2AhqdYXXWHiF4GzAzmmG+b/f//8e825O\nT2jZzPvFXVu/HotmZrjykWvFp1fPdhGPRAflu6gC4yuQYRnieizDBsojUwjBPSEC\nE7DtkcAOuAU+8WyPWS4w3xOWzYASpKO5KEkdKoazBGZlpDT1KEmh1P/wlXEBG9hw\nXYqBYyF8x7F9bivMJDYl3KIe4wF1hSEsoUYJgoAyw/AYdjjBghocHIopZiquOY6j\nPveZhX3LYwFnLuPYFMTlJnYdw6EGTcEE4pDVQNGN0hxUIUTzOmp06qnSBl5S9Zb0\nocRlKNRnKEsL4kRGIcobivQbMjUbtuVSQmzP1hFcrMsYSjIlHOJ6WB0d1WNooXzY\nrFZ1lMCFMFJV1YiVVZ9ElkPWaMaA5id7D7Zpmay2Y3s2XYg2uTP3b023fk/8yV0b\nnDq+fyEKlk7cr15fSFZXDyc1rXDo9sbe9ubDIvl2uX/n3KV3vBPM3rtxxfzxoV7p\ntzePjAp3148uvdz4ZMaj0Zl9W93M1HftTX9wfuX5yQNrvc8ztWPF4W59Gs+sL69N\nLdK3X15TeLJn8OInOZhbPv2sV7S+PnpVGKK/\n=ZlOG\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"2dddb8ddf2e0487f4fb3ec7848a36eac8d34ae97\",\"host\":\"keybase.io\",\"key_id\":\"48a36eac8d34ae97\",\"kid\":\"01016690eb0ddc554cb4ac827497b398abf96d1d3d2ddfff9a118a05b70d91be59090a\",\"uid\":\"9cbca30c38afba6ab02d76b206515919\",\"username\":\"t_helen\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1436977484,\"expire_in\":157680000,\"prev\":null,\"seqno\":1,\"tag\":\"signature\"}\n",
            "sig_type": 1,
            "ctime": 1436977484,
            "etime": 1594657484,
            "rtime": null,
            "sig_status": 0,
            "prev": null,
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "2dddb8ddf2e0487f4fb3ec7848a36eac8d34ae97"
        },
        {
            "seqno": 2,
            "payload_hash": "f075c6f9b90d03787307c57842682574a99d267b80e189420b0fc5971016f03c",
            "sig_id": "4e6251c541a5d77912a02370c707d39ad38193330e43f4507d45c3e1917d41850f",
            "sig_id_short": "TmJRxUGl13kSoCNwxwfTmtOBkzMOQ_RQfUXD",
            "kid": "01016690eb0ddc554cb4ac827497b398abf96d1d3d2ddfff9a118a05b70d91be59090a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowFtkt9LFUEUx/derUgNCsuiHqQJiuhmMzu7OzM36AcR9MOMlLgkwWVmZ1Yndfc6\nu4oXE3pLMikIIsEgAnsvK3roJfAtkpCIyP4Box5LjIjmXuolmsdzvucz3+/h3N7U\n4ORzLcXeK89udh/OvV4RzsVHN/aPA5HIKiiOg9BUK1kSjhij4rBe4VIalaagCPAZ\nRjNUqpzsJfhE19CFUa1CE1d1ycTD3f2lrs4qKICsWlFWK3QWJjoGEwUwoOqcSMd9\nylSMjjPbd6WUgkoZuQp6lEReJLAKCfUox4HiIZXY44oRS+xP0tqExQieqg6dgDqz\nrKWt/kc/UG9ABFEQMKgElDL0fS8UnpW5xGNEYEa5iFggkcTSWomiiHGEKIe+IFAy\nJJTPIIPc4kbqOBaKkGMYYsojwQMuoCtJIFwY+MhniNWEqTIxH6qlz8r9alDV0//Z\nxz97LYBRZVKdxKCIrCjMdG0OeThghAQ+LgA1VtFGlXVN4ZOAQvsKoGLUqKUpa9pD\nKHIDn3qKWvPcZrPJCLIFGBECUYSgjBDyBGSE2/ycioBT6ULlYvt/qobjBBS3udYi\n77PMVPfFPBsxCkw0Tc42Orm8s35dvnYdTtPGzX9P5uithl8d2cvy8heWbd3Z2dL2\nEOtPTcuLL2aPrB64PDg1eX9foXktm5nDrbsnro59e9Las+fc246FpeufS/zS1OLP\n+Wmz8vHrMdAMnjf+SN7fyz9d2/F4w+ntpx58bzt78G7Se779+Exudm/JaVm4tvpq\nbiicbzz0IY7fLL27syvnbGlun/4N\n=CoBU\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"cryptocurrency\":{\"address\":\"3J98t1WpEZ73CNmQviecrnyiWrnqRhWNLy\",\"type\":\"bitcoin\"},\"key\":{\"fingerprint\":\"2dddb8ddf2e0487f4fb3ec7848a36eac8d34ae97\",\"host\":\"keybase.io\",\"key_id\":\"48a36eac8d34ae97\",\"kid\":\"01016690eb0ddc554cb4ac827497b398abf96d1d3d2ddfff9a118a05b70d91be59090a\",\"uid\":\"9cbca30c38afba6ab02d76b206515919\",\"username\":\"t_helen\"},\"type\":\"cryptocurrency\",\"version\":1},\"ctime\":1436977653,\"expire_in\":157680000,\"prev\":\"ed2d411f26584e8f9aaac8497715840f7701f10df114b097a554a8b6a8d20e23\",\"seqno\":2,\"tag\":\"signature\"}\n",
            "sig_type": 6,
            "ctime": 1436977653,
            "etime": 1594657653,
            "rtime": 1436977851,
            "sig_status": 1,
            "prev": "ed2d411f26584e8f9aaac8497715840f7701f10df114b097a554a8b6a8d20e23",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "2dddb8ddf2e0487f4fb3ec7848a36eac8d34ae97"
        },
        {
            "seqno": 3,
            "payload_hash": "123134c84cd9e592cb7f9ead0b24a29252ce22789a718007ea865612eba88e9e",
            "sig_id": "bef80a641d1c020b2cb4dab82074fbd4ef7f7c9c0182163aac807ada77cb3a3b0f",
            "sig_id_short": "vvgKZB0cAgsstNq4IHT71O9_fJwBghY6rIB6",
            "kid": "01016690eb0ddc554cb4ac827497b398abf96d1d3d2ddfff9a118a05b70d91be59090a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowFtkk1IVUEYhq+ahmI/0M+uoCkI6pIzZ86cOXMJRAOjqxglIRUh86uHW+fee87x\n0kUMWgSlJRn0RxG2MEQQzEUU1sIWUWRRm2gTgRDkKoqgjVRzL9QimtXwzfs9vN87\n3+U1danamifzN+93mK/NNS+XRerIvZH5ISDyqgwyQ0BG5UKSl4NRpENZrXClIh3H\nIAMQzZaP7iuFmLZbdZc4pjoPRb24t1v00qyMS21ZkAZJuaCtVgSJzAchGE6DnK5y\nTBD266gQBWFi3x2llPCVMo6Grk+NawTWkvquz7GnufQVdrlm1BIH8nGlw2IEj/We\nIA+qzL5A2ep/9LnqA0QQeR6DWkClJCGuFK6VOdRlVGDmc2GYp5DCyloxxjCOkM8h\nERQqhoQmDDLILW6wimNSSI6hxD43gntcQEdRTzjQI4gwxCrCWEchP1WZPukb0Cd1\ndfpIl/I5XQkgDvqtZxvkceBqzyFIEhdxoihlyOHQwRRKCqnCjCvsI4YxhtrFxiW2\n6BKJNWLI3pBPoAEnhv9m/c+fpUFJR3GQD0EGWZFMgoon5GKPURswSgN9uhBEus9+\nT6aICPV8aE8aFKxXizOQEukZJhhUEFOfYkglsZ2O5zuEupwx5XhU+FAjn7kOFNBI\nwmglcAOxtAZiXQzzIIOtRd5vkXb0kCeDkQbDTedvr0rV1KYa6msrm5dqalz/Zx0n\nZutWmlu354p7z4z5yYFLP+bPjuCDGzeh6WT36xfr3jY8y35eurBCJ161j7asbpk7\nfG3pzY0di+ON7+62BV2Ppkd7aA/ZumXzzJeR7xuK29Y23gqXL6786p5pvfL4Kl+o\n79j1YfbBzinvTvBz/+K3yYXU0tOHn+auf2xfHi++P9c5OfV8zPwG\n=bVAb\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"cryptocurrency\":{\"address\":\"17JyYCvn37BodyLbZdKQrW3WNbW7JcsvAJ\",\"type\":\"bitcoin\"},\"key\":{\"fingerprint\":\"2dddb8ddf2e0487f4fb3ec7848a36eac8d34ae97\",\"host\":\"keybase.io\",\"key_id\":\"48a36eac8d34ae97\",\"kid\":\"01016690eb0ddc554cb4ac827497b398abf96d1d3d2ddfff9a118a05b70d91be59090a\",\"uid\":\"9cbca30c38afba6ab02d76b206515919\",\"username\":\"t_helen\"},\"revoke\":{\"sig_ids\":[\"4e6251c541a5d77912a02370c707d39ad38193330e43f4507d45c3e1917d41850f\"]},\"type\":\"cryptocurrency\",\"version\":1},\"ctime\":1436977841,\"expire_in\":157680000,\"prev\":\"f075c6f9b90d03787307c57842682574a99d267b80e189420b0fc5971016f03c\",\"seqno\":3,\"tag\":\"signature\"}\n",
            "sig_type": 6,
            "ctime": 1436977841,
            "etime": 1594657841,
            "rtime": null,
            "sig_status": 0,
            "prev": "f075c6f9b90d03787307c57842682574a99d267b80e189420b0fc5971016f03c",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "2dddb8ddf2e0487f4fb3ec7848a36eac8d34ae97"
        }
    ],
    "_comment": "NOTE: I've reordered these keys so that the last one here is the middle one used to sign. This is intentionally to trip up the Go client, which likes to only verify the last signature (and therefore by default used the last key and spuriously passed this test).",
    "keys": [
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v2\n\nmI0EVaaJBQEEAK4gOAQlgExzLwhMqqvLXE1Lld5K6K1aKn/G9o8CWWph8G17gOx3\nwnPfWEZ7Sm553GclYK3OiCzrbrxlatsStv7bQjIofRVIo2xjbEboh1/6HM7udCFO\nB7tdRvqrw+Gw/57s1jApKbKvfL/tAeIRdEN0GE5XyJd107OPNcS8M9vrABEBAAG0\nVUhlbGVuICh0ZXN0aW5nIGtleXMgd2l0aCBzaWduaW5nIHN1YmtleXMgdGhhdCBj\naGFuZ2Ugb3ZlciB0aW1lKSA8dF9oZWxlbkBleGFtcGxlLmNvbT6IuQQTAQIAIwUC\nVaaJBQIbAwcLCQgHAwIBBhUIAgkKCwQWAgMBAh4BAheAAAoJEEijbqyNNK6X1uwD\n/jOjk4NrQmdUiYbTHfKOWaltZu2vVFm5rBRSxwUIItK5iguGxJ1w+nfSIRpb9XcG\nLqXrin8AyPkuw4wgox7oHahWvF4R4qF9bfVf5W/n14xJiqQ1EFaIvO0eoQdmpI68\nx7FNl7sTOLwK23V+NgeJuyK2KuB88uhrmRqD6M29qg2AuI0EVaaJBQEEAMIX2acX\nimcfGRgQfXV88vThixQNj3RDxna+i2/ji4it2C77xI2fOrD7G9V/1fyI/xV5VkDc\nvebKpg13vcV2tBts5XeDiAzu1UJ7MXVPiKxZ7h7KzsBQexQVYlG/Eo9uJAuvkUKD\nCjM1TF1gc24gzdWSn95+aU+KpSXYJcnV21inABEBAAGInwQYAQIACQUCVaaJBQIb\nDAAKCRBIo26sjTSulxWFBACgAsPYYrqIS1z+bAbGcUGdrCZ6Qcj2gnTrNz2R+OCg\nWWjwOVCOGwH//CmXldetuQuTxWv+uE4jwL1rvNMZGFXDqSfP5QPYTDjUOzDlHmiK\na0v7BePO4TWV9Rymg8F4SdWxQDzhZgudsXl/TCDYGkVbByyOET8t1a3DYUlcqZwA\nBriNBFWmiaUBBAC4rsIlUx2wFOpLUX2RA4M57NFtRs5IadU0k3JdMq2sUqcVqBoY\nM3Fb+aI+kidTc2ViwO3EJ/dHJ04GH99gL0JdPLq4gIHZ6cbLDJn6IcJWjHXp3M5O\nIlkJ4bUwJ+GQPEqwqSzbMU2B5bNmZIJed+L1Lcg4RCyyRT9QKDPNNRR2MwARAQAB\niQE9BBgBAgAJBQJVpomlAhsCAKgJEEijbqyNNK6XnSAEGQECAAYFAlWmiaUACgkQ\nDDpaarWMUjvhCAP+KrnFTQqnSJOcyMY2GvNjeJxWIKfQLuBOUP9JNTxQW92gOgZy\nUbf4pPZShYduMW3vNQKOsuYI27UbwF9+SWPadhU1/1KDOcAc2I6gcJKKNUKjXIBs\ny2FViVgjfTQeAdWhXv+giSrGOB+spiLZKMFK/e8Ws5trUSmk9sruR94l6rl7WQP/\ndcDx57sJPXaMDV9WcL2lkZ8qGnNv0sn2Hkn3LBlOLcNldLZ46xtQhPWC/2J+ZLrd\nqOxdH77rboQRkCRIGU74xtFcIrFWE71lFNh2rGp3LTOUO8Dxl4lOna3DICjyQZtF\n/7wtrrTFtHqvJKFoUrBeZ3VYPok9J4Nkwie44o3ySic=\n=Z+fb\n-----END PGP PUBLIC KEY BLOCK-----",
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v2\n\nmI0EVaaJBQEEAK4gOAQlgExzLwhMqqvLXE1Lld5K6K1aKn/G9o8CWWph8G17gOx3\nwnPfWEZ7Sm553GclYK3OiCzrbrxlatsStv7bQjIofRVIo2xjbEboh1/6HM7udCFO\nB7tdRvqrw+Gw/57s1jApKbKvfL/tAeIRdEN0GE5XyJd107OPNcS8M9vrABEBAAG0\nVUhlbGVuICh0ZXN0aW5nIGtleXMgd2l0aCBzaWduaW5nIHN1YmtleXMgdGhhdCBj\naGFuZ2Ugb3ZlciB0aW1lKSA8dF9oZWxlbkBleGFtcGxlLmNvbT6IuQQTAQIAIwUC\nVaaJBQIbAwcLCQgHAwIBBhUIAgkKCwQWAgMBAh4BAheAAAoJEEijbqyNNK6X1uwD\n/jOjk4NrQmdUiYbTHfKOWaltZu2vVFm5rBRSxwUIItK5iguGxJ1w+nfSIRpb9XcG\nLqXrin8AyPkuw4wgox7oHahWvF4R4qF9bfVf5W/n14xJiqQ1EFaIvO0eoQdmpI68\nx7FNl7sTOLwK23V+NgeJuyK2KuB88uhrmRqD6M29qg2AuI0EVaaJBQEEAMIX2acX\nimcfGRgQfXV88vThixQNj3RDxna+i2/ji4it2C77xI2fOrD7G9V/1fyI/xV5VkDc\nvebKpg13vcV2tBts5XeDiAzu1UJ7MXVPiKxZ7h7KzsBQexQVYlG/Eo9uJAuvkUKD\nCjM1TF1gc24gzdWSn95+aU+KpSXYJcnV21inABEBAAGInwQYAQIACQUCVaaJBQIb\nDAAKCRBIo26sjTSulxWFBACgAsPYYrqIS1z+bAbGcUGdrCZ6Qcj2gnTrNz2R+OCg\nWWjwOVCOGwH//CmXldetuQuTxWv+uE4jwL1rvNMZGFXDqSfP5QPYTDjUOzDlHmiK\na0v7BePO4TWV9Rymg8F4SdWxQDzhZgudsXl/TCDYGkVbByyOET8t1a3DYUlcqZwA\nBriNBFWmioMBBAC4ll4aBxfFepR/vfXs1TVaeDBi5wWKJNYrmfHAwXOZIFgrIPNx\n9ikGXZaiUBsekRDqcxmlsIMtZJWdpmIk180Tvm0evTQ7jCi8glltcS8OSCpFv9uW\nx8phrv6eY+U0AkTTpzPZqeO/IIywpbu3JTPsZxODETbo0l9/TpwV2+w6OwARAQAB\niQE9BBgBAgAJBQJVpoqDAhsCAKgJEEijbqyNNK6XnSAEGQECAAYFAlWmioMACgkQ\nvbuZsUZm8QwQCwP/ehPkqLCLJtMMLXEzIk0vFyEA8HJqaHmTTpmoWH2Z71x7usgN\n5/11FBqAcSL8lbk+5rwtTGs9mogkt6Gdm5TG+kypkOgTxq1cTqWn0MoM4rr2czv7\nfuyYFkCP6tiqhIPBBp8g4/jWfhbPFzY4q9IqhZENVM3Ljq0GLtqcZGTH8YFe1QP7\nBEz3aMPXH+TnGYKSDJ+PinhunEN5WEa2/sPqPmuBEjsB4NC3fQXt7UC5JOMSNMAz\nN53BOmdExycT2CjEDj+pyhNViBEWTHuCf/MkKcmItDdcUDSvAQ9H2wlFEbUJwmbU\nDJqF9Foqo3P7k7YVxGf6g2vVNAjyqs3xsdh7ch88GNo=\n=JxSo\n-----END PGP PUBLIC KEY BLOCK-----",
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v2\n\nmI0EVaaJBQEEAK4gOAQlgExzLwhMqqvLXE1Lld5K6K1aKn/G9o8CWWph8G17gOx3\nwnPfWEZ7Sm553GclYK3OiCzrbrxlatsStv7bQjIofRVIo2xjbEboh1/6HM7udCFO\nB7tdRvqrw+Gw/57s1jApKbKvfL/tAeIRdEN0GE5XyJd107OPNcS8M9vrABEBAAG0\nVUhlbGVuICh0ZXN0aW5nIGtleXMgd2l0aCBzaWduaW5nIHN1YmtleXMgdGhhdCBj\naGFuZ2Ugb3ZlciB0aW1lKSA8dF9oZWxlbkBleGFtcGxlLmNvbT6IuQQTAQIAIwUC\nVaaJBQIbAwcLCQgHAwIBBhUIAgkKCwQWAgMBAh4BAheAAAoJEEijbqyNNK6X1uwD\n/jOjk4NrQmdUiYbTHfKOWaltZu2vVFm5rBRSxwUIItK5iguGxJ1w+nfSIRpb9XcG\nLqXrin8AyPkuw4wgox7oHahWvF4R4qF9bfVf5W/n14xJiqQ1EFaIvO0eoQdmpI68\nx7FNl7sTOLwK23V+NgeJuyK2KuB88uhrmRqD6M29qg2AuI0EVaaJBQEEAMIX2acX\nimcfGRgQfXV88vThixQNj3RDxna+i2/ji4it2C77xI2fOrD7G9V/1fyI/xV5VkDc\nvebKpg13vcV2tBts5XeDiAzu1UJ7MXVPiKxZ7h7KzsBQexQVYlG/Eo9uJAuvkUKD\nCjM1TF1gc24gzdWSn95+aU+KpSXYJcnV21inABEBAAGInwQYAQIACQUCVaaJBQIb\nDAAKCRBIo26sjTSulxWFBACgAsPYYrqIS1z+bAbGcUGdrCZ6Qcj2gnTrNz2R+OCg\nWWjwOVCOGwH//CmXldetuQuTxWv+uE4jwL1rvNMZGFXDqSfP5QPYTDjUOzDlHmiK\na0v7BePO4TWV9Rymg8F4SdWxQDzhZgudsXl/TCDYGkVbByyOET8t1a3DYUlcqZwA\nBriNBFWmiQ0BBADGbDlbbQsqG0rR9CiOYTXnQH7gXbPW8JQGc+n27N05NO7k8b8U\nMhYXg7aDR89AvVHyL8UIL/+1c7aAU3xcsu/yBfwsAuDOjlZnB5K+IWEGxuFAS1I2\nPxhehRkct9aLh7lz26MPcaPV8Jn1GgsC9ZVc4IoUE5S+Rx28IxbSWb9XrQARAQAB\niQE9BBgBAgAJBQJVpokNAhsCAKgJEEijbqyNNK6XnSAEGQECAAYFAlWmiQ0ACgkQ\nxI8M+fSjZHW2SwQAk35hPHEoeJdoTEWN0fiJjcGocASsOtcuYl7SRIvqQ8xiFxUv\nlIFAiAua3Tu85Bf7vvUkfjbMQHUru1jfrGdVHWRS445hDpGo2JIOOG53MPbNMVTW\nhk6T/GdYH7tyEoSU3EpUPn3I6+9ThI9RF2UHNWk245PE5IbP2QOSRqI4OGey5gP/\nVBkq/7vweVO05MwaCeWGUJtutSi/BEarWjXJGvRHsF5ynxPw8I74NMN3eNWF3GJ9\nREKidCZaJiMUFuhnM0q0hs0JQLutyPJdPdk0Q+Lwp74GH8XAd3+9cHGjQsoKKv9I\nIgxzqpptFCn9YShewNGexOtRJBuiiNQNmGUFapkHPpM=\n=YNYZ\n-----END PGP PUBLIC KEY BLOCK-----"
    ]
}
`,
	"bad_full_hash.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.44\nComment: https://keybase.io/crypto\n\nyMHlAnicbZFdSBRRGIbXskxjo4Qs88L2hES0yPzPzupNISQRCJJppa5nZs6ugzo7\nOzMriq1EWUFSaWq5ZBn4lyWSEf0QiWteuRj+hBYUohC2ZYFYUFHS2Y3uOjfnfO/7\nfs8H5xuzrrckxeVdsw7w7JGmuPDIkt9SEFnLqgOiV64FzjpQgWKXW1E9SNd0RTWB\nE7CQp0URsYQgEpJAiTIlQRHSLMeLDp6kHZCnIEtzDLADt7+y0lUOjXLc5vZ6Rahj\nsdxrRDGYLUIDZSperOHCpchY/Q+kImYQJEHyNJQhhY9EQcSSPMFJjOAWSZJ384KI\nCJpHLJQExs2LlOTgBMbBUxSDOAIKIhZJAmKcP4ZzCDJDy6KbFSBHS4JIQ5JzUCyL\nEOTdLClEgwbSVViFcFrzaC6/JkMTuUxkmEgHATswa7WohyplLOF8NdINxasCJ4lN\nyVSinSTDkLTgoGnSDlCNpujIpeAEjXWBIwjCDjQdVQOnij/KDgzkc/2lkrFC9cZe\nJvTgOYbiUaHp1xEIjI4Ux1vikiwbN6yL7suSlLj13xZThhN+J+96/HxL5O6lve/1\nuUehkX5PxteHt2pmX84Fe3yR8e3XCbCzs2BifPBHSfHPhtbQvsBkbtnhC6GB1QOd\nyeXhHO3K4HzHu9fWuvb46ZnktuVIWyl579WNm5k9yyVpQdvVvqppX0bg7AqVveR7\nuvdzo407OvRsreVB6OSMlLrkD9frXXRaWkJTd2//wcSpFwuLjR+maj/2NXe0n4N7\nPpWhyRZC7aU6LxLbSrNP5x/P/1VovT25eKer7HtIuG/aihoKx77szlvLH229XJHj\nSjnTdWxT0LZgbi4+cehtpv4muDp8Kj1smaW7pcT6UO4Embpj/8qQFkhvLvp2fng+\n60nKH6ETE6k=\n=8ZaC\n-----END PGP MESSAGE-----\n",
            "payload_hash": "9d6cea32a5cc0a0d9cfb9a58d60f0c1a9720e748dcd83901afb32e9de9c8b03c",
            "sig_id": "3530b961e15781b6e833f8e9bae1e857a9b52b3aeb1a3c34e83a0e4e91b5d29f0f",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"5a73bbe509b0c92bd2caba3567b87138a72a5364\",\"full_hash\":\"foobar\",\"host\":\"keybase.io\",\"key_id\":\"67b87138a72a5364\",\"kid\":\"010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a\",\"uid\":\"89d43dbf59a63c9b3a168255eea7f519\",\"username\":\"pgp_update_tester\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1441398331,\"expire_in\":314496000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a",
            "ctime": 1441398331
        }
    ],
    "keys": [
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v1\n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\nnQ==\n=olf1\n-----END PGP PUBLIC KEY BLOCK-----"
    ],
    "uid": "89d43dbf59a63c9b3a168255eea7f519",
    "username": "pgp_update_tester",
    "label_kids": {
        "start": "010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a"
    },
    "label_sigs": {
        "undefined": "3530b961e15781b6e833f8e9bae1e857a9b52b3aeb1a3c34e83a0e4e91b5d29f0f"
    }
}
`,
	"old_pgp_key.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.44\nComment: https://keybase.io/crypto\n\nyMINAnicbVFdSFRBFL6aGS4VgRb0R3Ufyspy5v7NnaUiyRBKyBd/kpV15t65u5fV\nvdve3VQWqSclkqgljEiIjH4oeyjBiAz7WRIqUoIyilDDl4qoNCvJstmN3pqXM+f7\nvvOdM3NSi+YJnqx9nYt6kFp+POvxvXVxofLdbGdCpI7ZInoTYohlgmWHAywaidrh\nmOgVVYJkSpkKMAUGlqgpGYQSWdUQ1RGUdYIkosqaIhaJVryhwR8kbpCXGYYpWbIM\nLV2B0NSJYWGKgKTIGkEWI6qhAMiYiSWky0Chmk5kSQcmIoAxwwSM2wUdNz0An4oS\nl221HY7xxG+bHP1P+1CGABBAJBOTSPwYEmEqREAzFGxRCJGFMGVARkwlBlYsRCVD\n17CiI0lSmAYIphyEgHC7eMZOx6Yim9RSMdFkA1OZQE2XVJUx/g4V4rTQZdEwaWRc\nHQlE/PGISWLMH2NujEXF1iIx1hJJc6zB5BDXH2RR13bCohdy0ojZ6UqoKFDGCENQ\nJLLmiB1lfpsrZI5jDQCORqLsoOgN8y8uEl12wP/XFWaSsJO5xUiA93HtQJjE4lEm\ntt6/68sRsjxC7vzs9KYFT96Sf/tfUbpgbte8yt5f5rm5ZRXZbaN9yQczyQFv0+0P\nouCL1xsnl/V01ZQ9Kvwx3Z2cG2iZeFY3PPnhcl9Wzv2hN8/bn166Obp+6c8rtYUd\n1dUzK3YWKsJEbX+dnmq9OPvNU9DeUbW6a2ifebYqeOhzaGNqU2nlLmLuJYN1ZSPC\nnmF94H3b2xfTFa+Mgtmx7Uf73eSt7z9OBH4vbRqZmio5VVhb3VFcnL83lfO6eXne\naHMvWDuhZUsfxUT5sYdnSnyfjicGfU554sYX347r67bgO2+OvNztdK168mzx/smV\nG0L5dE3Fte7Go03nld3l23uczRuvHq6vO70wuG38ws658Zpc39epkrGhqT8huR8G\n=wXBT\n-----END PGP MESSAGE-----\n",
            "payload_hash": "c324627b05df3adde4b179f0a432fedc0c4068ad04d5abef314450fa20cada57",
            "sig_id": "9f659ab568d9a98a82623c1669c9f6b297a76a0dace66dece0808e99c0a572cd0f",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"5a73bbe509b0c92bd2caba3567b87138a72a5364\",\"full_hash\":\"ccd2f331f8411d8acf9b702436a7fea5c401eed9278304b68a3280d7a0eecd0e\",\"host\":\"keybase.io\",\"key_id\":\"67b87138a72a5364\",\"kid\":\"010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a\",\"uid\":\"89d43dbf59a63c9b3a168255eea7f519\",\"username\":\"pgp_update_tester\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1441397910,\"expire_in\":314496000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a",
            "ctime": 1441397910
        },
        {
            "seqno": 2,
            "prev": "c324627b05df3adde4b179f0a432fedc0c4068ad04d5abef314450fa20cada57",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.44\nComment: https://keybase.io/crypto\n\nyMJRAnicrVNbaFRHGN7V3JqQaFvUquDleMHaGGbmzOVMJAoRhdJY68MmMWiWmTNz\nNsfE3fWc3aBIQNBgHlbwwUYDNlKhwVu1rT5Y6IOkEdQERNJ6YUEFJaJBi4oKrZbO\nphQq+CTOy/B/fP/3f/8H/1Dl5Eh5dF1v5UlGGvZFhwe/ykZiD14d2GHJlNpu1e6w\n2vXEpzuUDjPxdl9ZtRaAADJbKIHMc5HQBDJAXcw9CSHzGJca2EwT4XLsMYlch3Ls\nMISwpkBwaUAIhFVteX4yoYN04CczRpYIZkupCeASuBxJhVwhhU0okw6DtiMYEsSm\n2DS2pcJChzEnRahr/JTBTBGfsPcW/nv2nZ2Qc7jCtpIe4YLaLpe2gNRBhGgtmEcg\nLxBDHSTFFm3Y6UQ6nk0rkdHxjIlSB1ZX9f/AQsjvHIeX7eiIt4mwzbRJAajwECKM\nCkgc6mFIFXAxklIBxhWCCNkSUQmFZpAz2xWulBh7RBGiKH/faZktM9vTbyZgZnTq\nIPRTSasWGoKb8QsZQYyhzRmHoNrS29J+oOO+YdgG5xQAg6YD3WmUXBthipgERHnG\nodJYQsY9ILCNPK1csy6gjlAAKyKk9goKBHgCAdfsQ5iZH+qt8X99wYkimbJqkbEq\nEkY/9BNJkckG2uoaPL+xKBItj5QUTyocRqT8g6n/nUvxvNJXVeNHb/DTfSR2NQq6\nfw1aer8Rzc9iuRcL8v1z6HewZu3av7un3+4ZLpmX21uXP3Zl8f4LK07sHmu5/PvW\nnluXRhtTH224c3LX3eZVs196fzV2xVrx8/XPi6xDFzc1HZ/b177665WZ5Y0D6394\nWlc53vLn9w8qz//YNK2o4lnzWP3N1oN1YcPjD39bOf5Hw9MjS8mXjxJTzuW/jZ66\n1r9w5pmy7pqq2OBYe+/Zij0k1/fZw1zpkos9N4YmHxwZKV1U3//6SUvFml8eH/54\nRvHiZZ14tCtfVfb5GTDXOdC6OrdoVfSTeAn6oinfP2tz6ueBn56sSNXfu38kOrTz\nUO+t+XtOX/90yQWqji/7B92ma9g=\n=lrM0\n-----END PGP MESSAGE-----\n",
            "payload_hash": "c21f7c7854deffe705eb01c476e526e74a8c05e7b74ef4921613d2b530e9f1df",
            "sig_id": "dff433e6fa28c2167a283a89d4982cb909410e86ff4ff5dd3f8431464877fb0a0f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a\",\"fingerprint\":\"5a73bbe509b0c92bd2caba3567b87138a72a5364\",\"host\":\"keybase.io\",\"key_id\":\"67b87138a72a5364\",\"kid\":\"010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a\",\"uid\":\"89d43dbf59a63c9b3a168255eea7f519\",\"username\":\"pgp_update_tester\"},\"pgp_update\":{\"fingerprint\":\"5a73bbe509b0c92bd2caba3567b87138a72a5364\",\"full_hash\":\"ba06af22576a1586f416d0c42bbd079d21223b26b1ae71973cacbb44f5d55d69\",\"kid\":\"010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a\"},\"type\":\"pgp_update\",\"version\":1},\"ctime\":1441397910,\"expire_in\":314496000,\"prev\":\"c324627b05df3adde4b179f0a432fedc0c4068ad04d5abef314450fa20cada57\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a",
            "ctime": 1441397910
        },
        {
            "seqno": 3,
            "prev": "c21f7c7854deffe705eb01c476e526e74a8c05e7b74ef4921613d2b530e9f1df",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.44\nComment: https://keybase.io/crypto\n\nyMJLAnicrVNdaFRHFN61UZIFiRVttSaoF4qppmF+79yJFYxVELQ1IC2i0mXmzpnd\nq3F33btrEtQiYkTRRs3DStqiVAVTMG2hBnyQKLGpoSmlkIeUVgoVLAh9SIsNFIN0\nNiK00KfivAzn4zvn+84H56v5LyRSyW2V+dcE33o2OT7SXk6883Dm/EFP502313rQ\n2wuzH3QYiEvpvZHxWj2EERZUGUXcC4kCjgXyQyatxlhYITUgKoCrUDIrNAkDX7JA\nEMLAR0pqB2KkvGbPRrkMFAvFKFdyY7kSVGvgSGoUSqINCZVWlPtCBwLTQAmiOPWZ\na8zm42qHM6dVDC1R3mGuSM/a+w/+c/Zdnh0XSMOo0ZZL5dNQaqqwHxDOAZSwHMsq\nMYZiTu0Dxy5kCulywagSpEsuSih6h5v/AVZD/t9x2HJHRzqr4qxrC0NDLKXYBgxj\nE6jQSi0QYdR3rkDxkCEMYCQRAUVM+4GiJEBGKAQQGgTPOy23Zam78O8EnMYBKMZR\nPue1YkcIS1E1I8wYplJIjJo96CpERUhHjkEdLn2EHFoowoHqjgRbEYqAMwPWgkAc\nNMIhEz5w4oNgKggdJrRgYJkk2MfUEM0pAmmxsU4/hv3pp77wbJHLOyFnVWXc/DjK\n5FSpXATv8Mjt3TWJZCoxb+6c6mEkUnULnp3L1hW1iRN80713m34YSi9pXNx0vLJz\n+yf1607cS51Jnro1/Pgn+8UHl9Zf5KvMub4Nk+Lyp4dGpxa+1TAwenfg529aOsT5\nhS+9rU8mzi2uU8Nji7q+/3VpAx/syd6Z+LHtym9DaOBBNvHqG9currmw+fW6XVu6\nH26fWdsz6U/VZBrGCjUX2pb1fzdcP/TX3WC8lq34tn/OLfXozRv3+dTKaZgY3h/3\n7Glpa1+9473xj0cbK029Lyb/qH1tpG9fpuvk/b6pMy/f/KjXljuP7WkYPPr1kbG+\nzVe/3LKeTn/2y9Hlfy7pn77zeQWnHuwcnCDQcuPJh4tOz1xvfOX932t6ezb6nZP9\n45X6Tr6BHln7NwUNazw=\n=WqS2\n-----END PGP MESSAGE-----\n",
            "payload_hash": "a0de15403085ab1ec0bfc6672b335284189694eb935bebb7592abe48698796d9",
            "sig_id": "90c8903c7c40d32d6d85a93abe3dd00db0c368da38b645393a7338a9cc55c8cf0f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a\",\"fingerprint\":\"5a73bbe509b0c92bd2caba3567b87138a72a5364\",\"host\":\"keybase.io\",\"key_id\":\"67b87138a72a5364\",\"kid\":\"010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a\",\"uid\":\"89d43dbf59a63c9b3a168255eea7f519\",\"username\":\"pgp_update_tester\"},\"pgp_update\":{\"fingerprint\":\"5a73bbe509b0c92bd2caba3567b87138a72a5364\",\"full_hash\":\"ccd2f331f8411d8acf9b702436a7fea5c401eed9278304b68a3280d7a0eecd0e\",\"kid\":\"010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a\"},\"type\":\"pgp_update\",\"version\":1},\"ctime\":1441397910,\"expire_in\":314496000,\"prev\":\"c21f7c7854deffe705eb01c476e526e74a8c05e7b74ef4921613d2b530e9f1df\",\"seq_type\":1,\"seqno\":3,\"tag\":\"signature\"}",
            "kid": "010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a",
            "ctime": 1441397910
        }
    ],
    "keys": [
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v1\n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\nnQ==\n=olf1\n-----END PGP PUBLIC KEY BLOCK-----",
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v1\n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\nrNyd\n=NgHr\n-----END PGP PUBLIC KEY BLOCK-----"
    ],
    "uid": "89d43dbf59a63c9b3a168255eea7f519",
    "username": "pgp_update_tester",
    "label_kids": {
        "start": "010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a",
        "sub_revoked": "010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a"
    }
}
`,
	"pnelsen_chain.json": `{
   "username" : "pnelsen",
   "uid" : "619e6eddd3df78745ecc4b3755295b19",
   "chain" : [
      {
         "version" : 2,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "sig_type" : 2,
         "last_check" : 1424366438,
         "proof_text_check" : "\n\nowGbwMvMwMSYyKDYzFR3ZSPj6QOvkhhCnklbVisl5adUKllVKyXnZKbmlYBYeYm5\nqUpWStmplUmJxal6mfkKefkpqXpZxQpQNTpKZalFxZn5eUBVBnrmeqZKtTog5SDN\naZl56alFBUWZILOUkhNTTJPSki2SjIwtjZIMDI1MjQzNE40MzQwNDIwMLYwNjMxT\nU0ySDIFGZuQXl6DYqgQ2Mz4zBSiKpN7VxcQJpL4UKmGZapaakpJinJJmbmFuYpqa\nnGySZGxuampkaZpkaAlSWJxaBPVSQV5qTnFqHsi5QMGyzORUJP+mZ5ZklCbh1FBS\nWQASKU9NiofqjU/KzEsBehc5PAyBKpNLMkGaDU2MTIzNzIwtjXWUUisKMotS4zNB\nKkzNzSwMgEBHqaAotUzJKq80JwfknsK8fKAs0KLEdKA9xZnpeYklpUWpSrWdTDIs\nDIxMDGysTKAoY+DiFIBFZONM/v+h1l+OXBTj51dlV7AP/sYzN5SvZMfGp7+rmfYu\nP6GfmS+vVWJwYa2LKmv4g2kP5kg0f/1+MfqrzdWAJ9XyKpXNV025C0q+HZbmfasv\nVrpl9ZG/mgfWPvh5fk7THMne5XuS6xVmXVKc/cts/06uv2Uh5WEX1D8nNeg+5GHN\n8RLWWa0vpc14Vll7YkqpxpdA3dr5i9cGN+6492/1M4acrK6zzfN2NLD9k6v8OTs6\npqXqZ6D85bdrL707uebz9KWLxOqVi059fODipztxqWRbbsDqrjZxkeXv+74I2Yio\nrI0J75jKwemQoNJdWRR0zTFGQ37vdy7jmxGB/bUanIVskdkVr27EaMm/zk/yLZx9\n6MERiWc2/P9eG2boXOmXq/KcpTrnnvwN45wLT0vWbTsmst7tj/SR9HXsNdrtJ1Om\n6ssapn9g4bKf07gwxbSwZMvuEC6H/7OzXyz9IbLshurc4Ivf4+cd3W7lL21v1L81\na77ci1/P90Y72gaW1a18uGaSWMpzxUcJmS8szwSwq674sVXm5kmBvd+LRVaKuEjH\nRmWFVn1gZu1ZU/ngyt/Jl8wKT/jv/uqVfcng2mnPmpXuQr+ibYyErRguGOaq3Na9\nbGD12vRu4aNf6xvCWQMzFb+JLzk0WeDjdo60xrv/C9PVMkNnt3qulnubr/zLxfpb\n9UY2rYVrnuTL7L4dEO74YUaPtv6GDc7tO9dsMnt0FAA=\n",
         "etime" : 1582046393,
         "proof_status" : 201,
         "rtime" : null,
         "sig_id_short" : "C-NwsMxT7aLb_VXkYpBnI2G1FplnK-6XfW3n",
         "remote_id" : "",
         "seqno" : 1,
         "last_success" : null,
         "proof_id" : "f68028db075d576cd6639910",
         "ctime" : 1424366393,
         "retry_count" : 1,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n\nowGbwMvMwMSYyKDYzFR3ZSPj6QOvkhhCnklbVisl5adUKllVKyXnZKbmlYBYeYm5\nqUpWStmplUmJxal6mfkKefkpqXpZxQpQNTpKZalFxZn5eUBVBnrmeqZKtTog5SDN\naZl56alFBUWZILOUkhNTTJPSki2SjIwtjZIMDI1MjQzNE40MzQwNDIwMLYwNjMxT\nU0ySDIFGZuQXl6DYqgQ2Mz4zBSiKpN7VxcQJpL4UKmGZapaakpJinJJmbmFuYpqa\nnGySZGxuampkaZpkaAlSWJxaBPVSQV5qTnFqHsi5QMGyzORUJP+mZ5ZklCbh1FBS\nWQASKU9NiofqjU/KzEsBehc5PAyBKpNLMkGaDU2MTIzNzIwtjXWUUisKMotS4zNB\nKkzNzSwMgEBHqaAotUzJKq80JwfknsK8fKAs0KLEdKA9xZnpeYklpUWpSrWdTDIs\nDIxMDGysTKAoY+DiFIBFZONM/v+h1l+OXBTj51dlV7AP/sYzN5SvZMfGp7+rmfYu\nP6GfmS+vVWJwYa2LKmv4g2kP5kg0f/1+MfqrzdWAJ9XyKpXNV025C0q+HZbmfasv\nVrpl9ZG/mgfWPvh5fk7THMne5XuS6xVmXVKc/cts/06uv2Uh5WEX1D8nNeg+5GHN\n8RLWWa0vpc14Vll7YkqpxpdA3dr5i9cGN+6492/1M4acrK6zzfN2NLD9k6v8OTs6\npqXqZ6D85bdrL707uebz9KWLxOqVi059fODipztxqWRbbsDqrjZxkeXv+74I2Yio\nrI0J75jKwemQoNJdWRR0zTFGQ37vdy7jmxGB/bUanIVskdkVr27EaMm/zk/yLZx9\n6MERiWc2/P9eG2boXOmXq/KcpTrnnvwN45wLT0vWbTsmst7tj/SR9HXsNdrtJ1Om\n6ssapn9g4bKf07gwxbSwZMvuEC6H/7OzXyz9IbLshurc4Ivf4+cd3W7lL21v1L81\na77ci1/P90Y72gaW1a18uGaSWMpzxUcJmS8szwSwq674sVXm5kmBvd+LRVaKuEjH\nRmWFVn1gZu1ZU/ngyt/Jl8wKT/jv/uqVfcng2mnPmpXuQr+ibYyErRguGOaq3Na9\nbGD12vRu4aNf6xvCWQMzFb+JLzk0WeDjdo60xrv/C9PVMkNnt3qulnubr/zLxfpb\n9UY2rYVrnuTL7L4dEO74YUaPtv6GDc7tO9dsMnt0FAA=\n=y8Q8\n-----END PGP MESSAGE-----",
         "api_url" : null,
         "human_url" : null,
         "hard_fail_count" : 1,
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.5\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ED4B1\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"service\":{\"name\":\"github\",\"username\":\"pnelsen\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1424366393,\"expire_in\":157680000,\"prev\":null,\"seqno\":1,\"tag\":\"signature\"}",
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "payload_hash" : "ff7e15625797458ef40006bbfe68ead03ef8b87b3bb35a0a02d22570c15d6de0",
         "proof_state" : 8,
         "proof_type" : 3,
         "proof_text_full" : "### Keybase proof\n\nI hereby claim:\n\n  * I am pnelsen on github.\n  * I am pnelsen (https://keybase.io/pnelsen) on keybase.\n  * I have a public key whose fingerprint is CAD5 BFC8 B239 2B01 2521  7A21 6100 2183 027E D4B1\n\nTo claim this, I am signing this object:\n\n` + "`" + `` + "`" + `` + "`" + `json\n{\n    \"body\": {\n        \"client\": {\n            \"name\": \"keybase.io node.js client\",\n            \"version\": \"0.7.5\"\n        },\n        \"key\": {\n            \"fingerprint\": \"cad5bfc8b2392b0125217a2161002183027ed4b1\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"61002183027ED4B1\",\n            \"uid\": \"619e6eddd3df78745ecc4b3755295b19\",\n            \"username\": \"pnelsen\"\n        },\n        \"service\": {\n            \"name\": \"github\",\n            \"username\": \"pnelsen\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1424366393,\n    \"expire_in\": 157680000,\n    \"prev\": null,\n    \"seqno\": 1,\n    \"tag\": \"signature\"\n}\n` + "`" + `` + "`" + `` + "`" + `\n\nwith the key [CAD5 BFC8 B239 2B01 2521  7A21 6100 2183 027E D4B1](https://keybase.io/pnelsen), yielding the signature:\n\n` + "`" + `` + "`" + `` + "`" + `\n-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n\nowGbwMvMwMSYyKDYzFR3ZSPj6QOvkhhCnklbVisl5adUKllVKyXnZKbmlYBYeYm5\nqUpWStmplUmJxal6mfkKefkpqXpZxQpQNTpKZalFxZn5eUBVBnrmeqZKtTog5SDN\naZl56alFBUWZILOUkhNTTJPSki2SjIwtjZIMDI1MjQzNE40MzQwNDIwMLYwNjMxT\nU0ySDIFGZuQXl6DYqgQ2Mz4zBSiKpN7VxcQJpL4UKmGZapaakpJinJJmbmFuYpqa\nnGySZGxuampkaZpkaAlSWJxaBPVSQV5qTnFqHsi5QMGyzORUJP+mZ5ZklCbh1FBS\nWQASKU9NiofqjU/KzEsBehc5PAyBKpNLMkGaDU2MTIzNzIwtjXWUUisKMotS4zNB\nKkzNzSwMgEBHqaAotUzJKq80JwfknsK8fKAs0KLEdKA9xZnpeYklpUWpSrWdTDIs\nDIxMDGysTKAoY+DiFIBFZONM/v+h1l+OXBTj51dlV7AP/sYzN5SvZMfGp7+rmfYu\nP6GfmS+vVWJwYa2LKmv4g2kP5kg0f/1+MfqrzdWAJ9XyKpXNV025C0q+HZbmfasv\nVrpl9ZG/mgfWPvh5fk7THMne5XuS6xVmXVKc/cts/06uv2Uh5WEX1D8nNeg+5GHN\n8RLWWa0vpc14Vll7YkqpxpdA3dr5i9cGN+6492/1M4acrK6zzfN2NLD9k6v8OTs6\npqXqZ6D85bdrL707uebz9KWLxOqVi059fODipztxqWRbbsDqrjZxkeXv+74I2Yio\nrI0J75jKwemQoNJdWRR0zTFGQ37vdy7jmxGB/bUanIVskdkVr27EaMm/zk/yLZx9\n6MERiWc2/P9eG2boXOmXq/KcpTrnnvwN45wLT0vWbTsmst7tj/SR9HXsNdrtJ1Om\n6ssapn9g4bKf07gwxbSwZMvuEC6H/7OzXyz9IbLshurc4Ivf4+cd3W7lL21v1L81\na77ci1/P90Y72gaW1a18uGaSWMpzxUcJmS8szwSwq674sVXm5kmBvd+LRVaKuEjH\nRmWFVn1gZu1ZU/ngyt/Jl8wKT/jv/uqVfcng2mnPmpXuQr+ibYyErRguGOaq3Na9\nbGD12vRu4aNf6xvCWQMzFb+JLzk0WeDjdo60xrv/C9PVMkNnt3qulnubr/zLxfpb\n9UY2rYVrnuTL7L4dEO74YUaPtv6GDc7tO9dsMnt0FAA=\n=y8Q8\n-----END PGP MESSAGE-----\n\n` + "`" + `` + "`" + `` + "`" + `\n\nAnd finally, I am proving ownership of the github account by posting this as a gist.\n\n### My publicly-auditable identity:\n\nhttps://keybase.io/pnelsen\n\n### From the command line:\n\nConsider the [keybase command line program](https://keybase.io/docs/command_line).\n\n` + "`" + `` + "`" + `` + "`" + `bash\n# look me up\nkeybase id pnelsen\n\n# encrypt a message to me\nkeybase encrypt pnelsen -m 'a secret message...'\n\n# ...and more...\n` + "`" + `` + "`" + `` + "`" + `\n",
         "prev" : null,
         "sig_status" : 2,
         "check_data_json" : "{\"name\":\"github\",\"username\":\"pnelsen\"}",
         "sig_id" : "0be370b0cc53eda2dbfd55e46290672361b51699672bee977d6de7181d589d090f"
      },
      {
         "last_success" : 1442332249,
         "proof_id" : "be6330f3e749b34ccbed1010",
         "remote_id" : "9dd53655aeb13984b738",
         "sig_id_short" : "p5qjiwznQmMZnhti9g00K_QDzeyhnprHsSDf",
         "seqno" : 2,
         "rtime" : null,
         "proof_status" : 1,
         "etime" : 1582046461,
         "proof_text_check" : "\n\nowF1Un1MVWUYv/cqpCIFGEtRCE6GIYTn+4OPpqIoSyabH4O4eT3vOe+Bw8e5l3sv\nGMEFFL2CoFymkWwQoLkb1iBCb/xhGwkVgRpMmDmjBAYYLQdWm2OJneNoUVvvX+/7\nPL/n9zy/3/O6fJfpDHpeF1ZuKBnu0A/8DHT7p9Y7ihBgFguRmCJEyJGhYtduCp8L\nkRgkGxYC3gajZXOoYhZhdJYtdBEThRRAq002KyoKjWaiKcQRpcG1YklWMqDVYpU1\nLkTgRQpIAgtwgsMBiuEUjjE8jtEYiuIYS6A4A0USYCplptlm/1dX5BmnSRbV6BL8\nzh3kdg2fv5jgIA1FUSREiWEZkoKCQAKCoSicowDGaUAbtC5KsigwxwYVbdxcaM3O\ngSar2fxMs2CXNQRG4iRB0yTGqQPxtkxNII5TDA0kHCUhTnJAhAwvoWojUqA4DsMx\nyAkijUqqUoygJDUg8Cih1kgCSuMCDilGoClOZCWS5wmaA0A1gicATUER4BwnoRAH\nFCBQnhZREeIoEFkBJyQJUiwrcaSqwAbzFLM6HMXiFOrQ3tYCWYBLdpUh2zPzwf+K\ntRdatMgRCEyLtSbwoayI6q6WLhNTof8xgsaiEPiORbZCk6whVCtYVD1RiMUKC1RO\nSWIgRtGqXk51n4USqWZpACRIs5AXUQJKLGAZQABAUDzKo7ioGYoKGCXSIkT/kYer\nc/IZKqVNzlB4e74VIo5Kw4blOr1B5+1l0H6rbtVKv7+/cGzxC0/W/tn1o+1q08r3\ny9pfi13+4q8v9W5+9PVBV+Hq4bcs8WVt0pm8mD9CqlvuMR1r8sfHZmivTEWvS55+\n3XCZqD5lDPS9E7C72TV4fN3N9DSxvdrxpnnaUL/R229hTDcf3F+j2+E++GhXgvXL\nYxciwz76YG+58rutGG572lehCClHnubG+uSEZzd1ZJN7DrseXPUa+6Tsp5Ynng2f\ns0ET/Vt8GlrLW5zGxAHyh3UR5hBP4e7Ojgl3z/z2vof62TSPu+tG96u7+OZS/62T\n6XcPd42NdxxyGPcdiwwwWBrlNl3C3frmioZXIjpTNrXTD/Q3uPrf9vo9vNb6mfvi\n8zWjXz1mQ+Xrz5lNTFCUrmhVVd3o0XT/4D0Hwkf2HT/d41kx/V5dUNb6A8Xx523O\nxvMjtaVJDclh94d8Ao5O9MU9/i5VXu07TTFbT+28/a4QWTPEEt+4Ku3LbibNbOFC\nplynx+M+nYpfc+1sQ2tud0/f+But/AAJjCfi8kYvjPmlezu/MIVHOD1Bk9ua7oy4\nhxd0h769+PGVtEnvtP0Lrktx3VWx9+vmrldcjjc6Pb0Tm38J/L406+XU2tm5+TNz\n7kRn/9nUxraZvoRO/2rvUVcKcvLS27UDYOOsPncwkXXejr4Se+84s6LkXP9gCVER\nfiJpqLFubXbcJuutZOVcIB6cUDkZZOxJq+Krem/9BQ==\n",
         "sig_type" : 2,
         "last_check" : 1442332249,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "version" : 5,
         "sig_id" : "a79aa38b0ce74263199e1b62f60d342bf403cdeca19e9ac7b120df52737945d10f",
         "check_data_json" : "{\"name\":\"github\",\"username\":\"pnelsen\"}",
         "sig_status" : 2,
         "proof_type" : 3,
         "proof_state" : 8,
         "proof_text_full" : "### Keybase proof\n\nI hereby claim:\n\n  * I am pnelsen on github.\n  * I am pnelsen (https://keybase.io/pnelsen) on keybase.\n  * I have a public key whose fingerprint is CAD5 BFC8 B239 2B01 2521  7A21 6100 2183 027E D4B1\n\nTo claim this, I am signing this object:\n\n` + "`" + `` + "`" + `` + "`" + `json\n{\n    \"body\": {\n        \"client\": {\n            \"name\": \"keybase.io node.js client\",\n            \"version\": \"0.7.5\"\n        },\n        \"key\": {\n            \"fingerprint\": \"cad5bfc8b2392b0125217a2161002183027ed4b1\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"61002183027ED4B1\",\n            \"uid\": \"619e6eddd3df78745ecc4b3755295b19\",\n            \"username\": \"pnelsen\"\n        },\n        \"merkle_root\": {\n            \"ctime\": 1424366419,\n            \"hash\": \"022576bf204e249bde7af045e4c599121e9cd60fad5135f121ca03257fc062c2e57c659d8f4aa369bb2b0a3b65edb299f0e2b5b30a6d0de20bd8c23ffe588f94\",\n            \"seqno\": 158250\n        },\n        \"service\": {\n            \"name\": \"github\",\n            \"username\": \"pnelsen\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1424366461,\n    \"expire_in\": 157680000,\n    \"prev\": \"ff7e15625797458ef40006bbfe68ead03ef8b87b3bb35a0a02d22570c15d6de0\",\n    \"seqno\": 2,\n    \"tag\": \"signature\"\n}\n` + "`" + `` + "`" + `` + "`" + `\n\nwith the key [CAD5 BFC8 B239 2B01 2521  7A21 6100 2183 027E D4B1](https://keybase.io/pnelsen), yielding the signature:\n\n` + "`" + `` + "`" + `` + "`" + `\n-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n\nowF1Un1MVWUYv/cqpCIFGEtRCE6GIYTn+4OPpqIoSyabH4O4eT3vOe+Bw8e5l3sv\nGMEFFL2CoFymkWwQoLkb1iBCb/xhGwkVgRpMmDmjBAYYLQdWm2OJneNoUVvvX+/7\nPL/n9zy/3/O6fJfpDHpeF1ZuKBnu0A/8DHT7p9Y7ihBgFguRmCJEyJGhYtduCp8L\nkRgkGxYC3gajZXOoYhZhdJYtdBEThRRAq002KyoKjWaiKcQRpcG1YklWMqDVYpU1\nLkTgRQpIAgtwgsMBiuEUjjE8jtEYiuIYS6A4A0USYCplptlm/1dX5BmnSRbV6BL8\nzh3kdg2fv5jgIA1FUSREiWEZkoKCQAKCoSicowDGaUAbtC5KsigwxwYVbdxcaM3O\ngSar2fxMs2CXNQRG4iRB0yTGqQPxtkxNII5TDA0kHCUhTnJAhAwvoWojUqA4DsMx\nyAkijUqqUoygJDUg8Cih1kgCSuMCDilGoClOZCWS5wmaA0A1gicATUER4BwnoRAH\nFCBQnhZREeIoEFkBJyQJUiwrcaSqwAbzFLM6HMXiFOrQ3tYCWYBLdpUh2zPzwf+K\ntRdatMgRCEyLtSbwoayI6q6WLhNTof8xgsaiEPiORbZCk6whVCtYVD1RiMUKC1RO\nSWIgRtGqXk51n4USqWZpACRIs5AXUQJKLGAZQABAUDzKo7ioGYoKGCXSIkT/kYer\nc/IZKqVNzlB4e74VIo5Kw4blOr1B5+1l0H6rbtVKv7+/cGzxC0/W/tn1o+1q08r3\ny9pfi13+4q8v9W5+9PVBV+Hq4bcs8WVt0pm8mD9CqlvuMR1r8sfHZmivTEWvS55+\n3XCZqD5lDPS9E7C72TV4fN3N9DSxvdrxpnnaUL/R229hTDcf3F+j2+E++GhXgvXL\nYxciwz76YG+58rutGG572lehCClHnubG+uSEZzd1ZJN7DrseXPUa+6Tsp5Ynng2f\ns0ET/Vt8GlrLW5zGxAHyh3UR5hBP4e7Ojgl3z/z2vof62TSPu+tG96u7+OZS/62T\n6XcPd42NdxxyGPcdiwwwWBrlNl3C3frmioZXIjpTNrXTD/Q3uPrf9vo9vNb6mfvi\n8zWjXz1mQ+Xrz5lNTFCUrmhVVd3o0XT/4D0Hwkf2HT/d41kx/V5dUNb6A8Xx523O\nxvMjtaVJDclh94d8Ao5O9MU9/i5VXu07TTFbT+28/a4QWTPEEt+4Ku3LbibNbOFC\nplynx+M+nYpfc+1sQ2tud0/f+But/AAJjCfi8kYvjPmlezu/MIVHOD1Bk9ua7oy4\nhxd0h769+PGVtEnvtP0Lrktx3VWx9+vmrldcjjc6Pb0Tm38J/L406+XU2tm5+TNz\n7kRn/9nUxraZvoRO/2rvUVcKcvLS27UDYOOsPncwkXXejr4Se+84s6LkXP9gCVER\nfiJpqLFubXbcJuutZOVcIB6cUDkZZOxJq+Krem/9BQ==\n=IFF0\n-----END PGP MESSAGE-----\n\n` + "`" + `` + "`" + `` + "`" + `\n\nAnd finally, I am proving ownership of the github account by posting this as a gist.\n\n### My publicly-auditable identity:\n\nhttps://keybase.io/pnelsen\n\n### From the command line:\n\nConsider the [keybase command line program](https://keybase.io/docs/command_line).\n\n` + "`" + `` + "`" + `` + "`" + `bash\n# look me up\nkeybase id pnelsen\n\n# encrypt a message to me\nkeybase encrypt pnelsen -m 'a secret message...'\n\n# ...and more...\n` + "`" + `` + "`" + `` + "`" + `\n",
         "prev" : "ff7e15625797458ef40006bbfe68ead03ef8b87b3bb35a0a02d22570c15d6de0",
         "payload_hash" : "7d9be7a8fdf0682780edcde42b0be17d35f000d37903462346541f51c725e115",
         "hard_fail_count" : 0,
         "human_url" : "https://gist.github.com/9dd53655aeb13984b738",
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.5\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ED4B1\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"merkle_root\":{\"ctime\":1424366419,\"hash\":\"022576bf204e249bde7af045e4c599121e9cd60fad5135f121ca03257fc062c2e57c659d8f4aa369bb2b0a3b65edb299f0e2b5b30a6d0de20bd8c23ffe588f94\",\"seqno\":158250},\"service\":{\"name\":\"github\",\"username\":\"pnelsen\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1424366461,\"expire_in\":157680000,\"prev\":\"ff7e15625797458ef40006bbfe68ead03ef8b87b3bb35a0a02d22570c15d6de0\",\"seqno\":2,\"tag\":\"signature\"}",
         "api_url" : "https://gist.githubusercontent.com/pnelsen/9dd53655aeb13984b738/raw/2043284f9960885e73e16f45cbc160e4d9e2c421/keybase.md",
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n\nowF1Un1MVWUYv/cqpCIFGEtRCE6GIYTn+4OPpqIoSyabH4O4eT3vOe+Bw8e5l3sv\nGMEFFL2CoFymkWwQoLkb1iBCb/xhGwkVgRpMmDmjBAYYLQdWm2OJneNoUVvvX+/7\nPL/n9zy/3/O6fJfpDHpeF1ZuKBnu0A/8DHT7p9Y7ihBgFguRmCJEyJGhYtduCp8L\nkRgkGxYC3gajZXOoYhZhdJYtdBEThRRAq002KyoKjWaiKcQRpcG1YklWMqDVYpU1\nLkTgRQpIAgtwgsMBiuEUjjE8jtEYiuIYS6A4A0USYCplptlm/1dX5BmnSRbV6BL8\nzh3kdg2fv5jgIA1FUSREiWEZkoKCQAKCoSicowDGaUAbtC5KsigwxwYVbdxcaM3O\ngSar2fxMs2CXNQRG4iRB0yTGqQPxtkxNII5TDA0kHCUhTnJAhAwvoWojUqA4DsMx\nyAkijUqqUoygJDUg8Cih1kgCSuMCDilGoClOZCWS5wmaA0A1gicATUER4BwnoRAH\nFCBQnhZREeIoEFkBJyQJUiwrcaSqwAbzFLM6HMXiFOrQ3tYCWYBLdpUh2zPzwf+K\ntRdatMgRCEyLtSbwoayI6q6WLhNTof8xgsaiEPiORbZCk6whVCtYVD1RiMUKC1RO\nSWIgRtGqXk51n4USqWZpACRIs5AXUQJKLGAZQABAUDzKo7ioGYoKGCXSIkT/kYer\nc/IZKqVNzlB4e74VIo5Kw4blOr1B5+1l0H6rbtVKv7+/cGzxC0/W/tn1o+1q08r3\ny9pfi13+4q8v9W5+9PVBV+Hq4bcs8WVt0pm8mD9CqlvuMR1r8sfHZmivTEWvS55+\n3XCZqD5lDPS9E7C72TV4fN3N9DSxvdrxpnnaUL/R229hTDcf3F+j2+E++GhXgvXL\nYxciwz76YG+58rutGG572lehCClHnubG+uSEZzd1ZJN7DrseXPUa+6Tsp5Ynng2f\ns0ET/Vt8GlrLW5zGxAHyh3UR5hBP4e7Ojgl3z/z2vof62TSPu+tG96u7+OZS/62T\n6XcPd42NdxxyGPcdiwwwWBrlNl3C3frmioZXIjpTNrXTD/Q3uPrf9vo9vNb6mfvi\n8zWjXz1mQ+Xrz5lNTFCUrmhVVd3o0XT/4D0Hwkf2HT/d41kx/V5dUNb6A8Xx523O\nxvMjtaVJDclh94d8Ao5O9MU9/i5VXu07TTFbT+28/a4QWTPEEt+4Ku3LbibNbOFC\nplynx+M+nYpfc+1sQ2tud0/f+But/AAJjCfi8kYvjPmlezu/MIVHOD1Bk9ua7oy4\nhxd0h769+PGVtEnvtP0Lrktx3VWx9+vmrldcjjc6Pb0Tm38J/L406+XU2tm5+TNz\n7kRn/9nUxraZvoRO/2rvUVcKcvLS27UDYOOsPncwkXXejr4Se+84s6LkXP9gCVER\nfiJpqLFubXbcJuutZOVcIB6cUDkZZOxJq+Krem/9BQ==\n=IFF0\n-----END PGP MESSAGE-----",
         "ctime" : 1424366461,
         "retry_count" : 213
      },
      {
         "api_url" : null,
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ed4b1\",\"kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"track\":{\"basics\":{\"id_version\":4,\"last_id_change\":1424366227,\"username\":\"dburman\"},\"id\":\"a7bb4137ada5faef5bbf32685a779800\",\"key\":{\"key_fingerprint\":\"5e7dda3ca4646326d1cf9921c4ce31d1a1f8d6f7\",\"kid\":\"0101658ad5d530214caba491eed0a2690339c84ecc4754f1b34017fe59902eb89ded0a\"},\"remote_proofs\":[{\"ctime\":1424366077,\"curr\":\"d199b52afef93d36d13a88dd79bf51cfabbd31ca990557fdf26cd15e0fc911de\",\"etime\":1582046077,\"prev\":\"4dac75dc62094d680a3d11d486cdc46e913e899b237009debf82860490670f4e\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"dburman\"},\"proof_type\":3,\"state\":1},\"seqno\":8,\"sig_id\":\"eecbf4175daef756ccff5a00004c197ae3d58474c306a86f6f5d4e71e87926f50f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"d199b52afef93d36d13a88dd79bf51cfabbd31ca990557fdf26cd15e0fc911de\",\"seqno\":8,\"sig_id\":\"eecbf4175daef756ccff5a00004c197ae3d58474c306a86f6f5d4e71e87926f50f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424366751,\"expire_in\":157680000,\"prev\":\"7d9be7a8fdf0682780edcde42b0be17d35f000d37903462346541f51c725e115\",\"seqno\":3,\"tag\":\"signature\"}\n",
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "human_url" : null,
         "hard_fail_count" : null,
         "retry_count" : null,
         "ctime" : 1424366751,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n\nowG1VGlsFVUUbkuLAkFQUEMAhSGyWfDembvMLSSo0QooEdAQVl/vNn3Ptu+V96Yt\ntRYkJQoxreyVRDbDFhYJsraAQCASwIgimyBSpOwQQYkgCnjnUdEm+k9nMsmduWe+\nc77vfOdObdkkJS2Vp3SuSBt/cE3qvosi5fWzHT4ts0RMlVpZZZbMj+ioH6yivEBb\nWVaeLhU8oXtHYp1KtLAyrWIdT0RiUbPFHEYI5kJLyiTFzCUMawdxFzPGABY285TL\nJbHKMwOYANSLRHN1vDAeCXJYkissPOkK22G2ANDGNqTchgQCYEPXATbVCglosoZj\nCb9RNVYSMxRR5us/xOclNwA0t5BUMyohIBQDV0BPO5RpiqmjAWMux1ASjJVSFEMC\nPAQo87hLbSAJ4BoBbuCKGvIwTbSJdJRHXYqwlhIJh2JsMywgCwITOt6gXGFU5yd0\nNGDvx7nMC/ib4iMyEawiKnRfSZRp5fOEb8iEZJgbhawsiGzkEGLbtBGmEkXxAp7E\nTFbEqRAIOpQrjj2uPSyE59jExZxS5gJg3Vc+EKux+lhTpbgjOSKImJ8UlB5jNpRI\nagcqyKHnKuLRRmoS7JqmKWy0hkhywRGDWivAbcKA4zDpokAUipEHhYMApJ42dgC2\nFi5TQWBQe1wXxHwdKozHYp5RY5TxnR8p+Is2oIa2LIrHA8qQMYFt7mmPOcoxZTrc\ndU23mPBM6zwuhHKg5IHlMPWUZxOpINbAkwxCpU35ugEcuzZA98AL47rYgCPFjXOV\nJDZgSBEXcEeZn5BrMCQimkFHuya/7VAATP3Cc22XAMSMmYxVAvAGLoG+ST7JKQpr\nmRdS3OehNxNBh++PU27EDxcJ69+amkQI+aWF5ruTaSV87geFm52EHhuNWVmuWUVy\n79leayk8BE39pvMUEyk9D3NgLiQho1w7CruIIukAwl3iEQ8rpCnULmW2eQGelbl9\nbRLvXka7fEwyUcjnkfyg6EJemh/jKhTmifB/04r/h0Z5MGNJCg2j9rdjKhCvsbvM\nnBtPjCuMxHUoEkRgajpvrvu2oIoJTblrKADi2tQFWkmlkTmkhIZUOdgz4cqcI8BB\nxDYPRjCQgNpYQ4j/Imqa6PNcA2noRrlfFNdWefMpaR3SU1LTUppmpAWnb0rzZq3/\nPJLPz2t1p2vHd99fqTYtyt7wxpralfszU2v319bdLdj31G1SIXaxtWWwZX1Nt5ra\nwQMPlLTunfnrltYnrvycaNNm0uWC/su+D+e8V111/cjeEUMnHZp9a3LTj65dvLt5\nxJWJ2e+t7/TZOHX+obUvXe8+qc+RJb/H+2w+9+qQ7K0zx26a9WFPeSzl5IzJn9T0\nT6x4tFf0icpjVZHosl0zRlSXdRi6qeOp4xMXPX2zrmbCtvqShU+V2VO+e2jv6uZb\n0iqvdV9wKGNI9UtT4MfbHxw2avTOL8o2z/xm5BeHFvetdv0eLVcUr5uvFl4dNj99\neLNB4K2Rx65/O7Uu/VZ9yvTa31Z3OXrp/Ohhc746Maddhn2n756JE1v1mDvoufrj\nm0ZmVLQft39E1bSqqqGzu/bMqcjN33Gyru7smRYHnRbb12ct1Uuzb29M7P168Ic5\nfV6IfX74+X1dSfHtH9f0+4l5A7td8q/XVr4T37ig1wxSVto+nPrAl3VH9S/uuicv\n73vnh5zl6frLhctJdmpJxz1dbuZs/WDSmIr5nbdtyDp4YXGr4e3rH37sJjmz7NoH\n/ZzZTfP83au6LL6xLOf5BTem7RLRNecu9o6sH1LZv+Tyyxtewej9U8VNDsxccvXK\nI1MPvzhgwqX0rMWi5tmuu9NanF5Fdl4eO5nK1a8NbHurvFPF6cTFthd2TD9Q3nHR\nrGce79Uf22kt5s1tVpC9YHx1RlbnAW+3u/1suN3yPwA=\n=OVlb\n-----END PGP MESSAGE-----",
         "sig_status" : 2,
         "sig_id" : "ae98364138aebf280b57047f57d7c2866530628042515b72957cd163b761ac200f",
         "check_data_json" : null,
         "payload_hash" : "7ad8a4ff1674d62b1dec55ecb5ba0fde2cfdfb060a0f1bd6562eef3fa6f1533f",
         "proof_state" : null,
         "proof_text_full" : null,
         "prev" : "7d9be7a8fdf0682780edcde42b0be17d35f000d37903462346541f51c725e115",
         "proof_type" : null,
         "last_check" : null,
         "sig_type" : 3,
         "etime" : 1582046751,
         "proof_text_check" : null,
         "version" : null,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "sig_id_short" : "rpg2QTiuvygLVwR_V9fChmUwYoBCUVtylXzR",
         "remote_id" : null,
         "seqno" : 3,
         "proof_id" : null,
         "last_success" : null,
         "proof_status" : null,
         "rtime" : null
      },
      {
         "payload_hash" : "18f6757dca80f14a0a215585305277341ffce00ebe835d5701640cdd6a07fc55",
         "proof_type" : null,
         "proof_state" : null,
         "proof_text_full" : null,
         "prev" : "7ad8a4ff1674d62b1dec55ecb5ba0fde2cfdfb060a0f1bd6562eef3fa6f1533f",
         "sig_status" : 2,
         "sig_id" : "d2dfff954d31942085b0fdab8595a7da061e11e893d25ee9ec06cace1e16fe080f",
         "check_data_json" : null,
         "ctime" : 1424366870,
         "retry_count" : null,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n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\ncd3ph0Z3/WjDg7N3PnXtwSefyn1w5VW9q6P7bxn81d8s/y8=\n=8pKV\n-----END PGP MESSAGE-----",
         "api_url" : null,
         "hard_fail_count" : null,
         "human_url" : null,
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ed4b1\",\"kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"track\":{\"basics\":{\"id_version\":9,\"last_id_change\":1404367871,\"username\":\"lanshark\"},\"id\":\"a349349f14da86fb286acc02baacce00\",\"key\":{\"key_fingerprint\":\"250140f039a301954fe00dd05580d2964a0e8b6d\",\"kid\":\"010194032ab846e5f4bb75cdf9afdebd952426284e5ef6c35e1b824bf8df237ea7ee0a\"},\"remote_proofs\":[{\"ctime\":1395158671,\"curr\":\"dd87570f4d83beb41ffe82a2829f901abc027958bbf2edef3586fa667e801f1b\",\"etime\":1552838671,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"darksun\"},\"proof_type\":3,\"state\":1},\"seqno\":1,\"sig_id\":\"09062531a6b02bddaf268bef04cdc53977483c80cc820a6b2d1f68862a75242c0f\",\"sig_type\":2},{\"ctime\":1395158973,\"curr\":\"0d9b64cfe79e03d9eb3d2adbb2078c64b50c81ff1dbf7bbc36811e235f8d0fb9\",\"etime\":1552838973,\"prev\":\"dd87570f4d83beb41ffe82a2829f901abc027958bbf2edef3586fa667e801f1b\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"adamlanners\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"d3f66399e44854fb856c5d512c750b77ac9e8ec48bf0075b42005994fbb094550f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"59d1f816e2d7cfbd8cb89052867a32a2550abb514adc1abd7123f88978bb61f8\",\"seqno\":10,\"sig_id\":\"f437eb195028814529177047838b26eaa98e835b36da0453e62ade783bc653ca0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424366870,\"expire_in\":157680000,\"prev\":\"7ad8a4ff1674d62b1dec55ecb5ba0fde2cfdfb060a0f1bd6562eef3fa6f1533f\",\"seqno\":4,\"tag\":\"signature\"}\n",
         "proof_status" : null,
         "rtime" : null,
         "remote_id" : null,
         "sig_id_short" : "0t__lU0xlCCFsP2rhZWn2gYeEeiT0l7p7AbK",
         "seqno" : 4,
         "last_success" : null,
         "proof_id" : null,
         "version" : null,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "sig_type" : 3,
         "last_check" : null,
         "proof_text_check" : null,
         "etime" : 1582046870
      },
      {
         "check_data_json" : null,
         "sig_id" : "06b6c340ea102f5945337ea759289de50603a9e2ad543237343eb90050de40380f",
         "sig_status" : 2,
         "proof_type" : null,
         "proof_state" : null,
         "proof_text_full" : null,
         "prev" : "18f6757dca80f14a0a215585305277341ffce00ebe835d5701640cdd6a07fc55",
         "payload_hash" : "788bef4ab571d16accad4ce4c0e615dc627096a8464273559d074de2e77151dd",
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ed4b1\",\"kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"track\":{\"basics\":{\"id_version\":10,\"last_id_change\":1405487500,\"username\":\"byers\"},\"id\":\"a8f186f933d2da59c9c744e8ddd0bc00\",\"key\":{\"key_fingerprint\":\"1c0a0d19c1a8ccddb68b3d797f7a48d96f8113ea\",\"kid\":\"01014c28af0b56b2d8e1507d266a0f4a3eb195cc50c5004acdd2cb26cb7aff34165b0a\"},\"remote_proofs\":[{\"ctime\":1397514685,\"curr\":\"6d1837ce8ae0cc136035f56f92078d384e80811db42f1176d3845291b5b98590\",\"etime\":1555194685,\"prev\":\"9ee701e60a6cb8a36abe22e8bb7b3cde2114b5db4207343f911d06d91adfe72b\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"byers\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"7b01e397c965a6692f580fc892bba1491991b6322670ae8c9dd043eb6ea552830f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"6d1837ce8ae0cc136035f56f92078d384e80811db42f1176d3845291b5b98590\",\"seqno\":2,\"sig_id\":\"7b01e397c965a6692f580fc892bba1491991b6322670ae8c9dd043eb6ea552830f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424368292,\"expire_in\":157680000,\"prev\":\"18f6757dca80f14a0a215585305277341ffce00ebe835d5701640cdd6a07fc55\",\"seqno\":5,\"tag\":\"signature\"}\n",
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "hard_fail_count" : null,
         "human_url" : null,
         "api_url" : null,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n\nowG1VGtsVEUYbbe8K0J5KkWRi4ChS5m5986dGWhrJbamtPIILRirWebOndtd2u4u\nu9tCLVWBgNBqAiGAQKgJIhhBnqU+CBEqz1IIIhRRQZBSEiAgL5EawblLQZvwU7P7\nYzIz93zfOd85s7BrXIwrlsUMnON6+/iW2EOXzJjcZmVzuWIGrDJlZLnCi3zCH3FW\nflYslJFKoSgzWVgk+wLPTRem4lZKRSjsC/jlEdWoYSBmCo4px4gSgyKh6YwgSilA\npkptizBuKBVuB8YBtX3+AhEKhnxODYUzC5k2J6aqUdUEUEUqxEyFBgRAhUQDKhaW\nbkJZ1RsIR9p0o0QxPT5L7j7mfmH0AED5MzkWFHMIDIwAMaEtNEwFRlgTgFLCEOQG\nQpZlYQQNYOsAU5sRrAJuACZ0wCRcSWsdKgwhb2qWjQnWkeBcNzWMkEqRCalzMSxC\nrcoF/aIoLPwO+0iI8UKHv2zex8POymd5HikJgVspYuGIZOPhXiYlkns6QDqRHYM2\noGaZ/MiBjDbEiA2JYVNNs1SLIcrlHHRdENmipA2A8kh4R6u24kMOGLAg5ZARzi3L\nNIipWZhiGzOdWNSwCYSaYG3E1LlKmA1MZJiqRQREAFuqYTApG9OElABxjoD8A51J\nTJWbqsFNzGxb06GBTCmm7D0kigMR4QmGAgFbipEvbRfxOfSgRuUQdIMgt8JLQiFH\nc0sOFnNBmACcQ80AGrKR5KwCTCyNSLZANmqZumpDiA1nS44Dmsik0oiOBKIVHCEE\n6QPwYEiUOg4WAgMo5Jxll4RphjSzqgpimtjUuCVUCHUTOdgAa7pmU1kIGBaFzLIF\nVp04tHJx9I3yiYbIK3ihx2IR5pkadgb8KE0Fvoi3xFQeP9Lo955IWVDuam4lHGER\np215EhbT/AFlpCpXvoIHnscyL0LKxamMoGFQ1UYEyCzJIJkM6hRSqYGhqaqBpYsJ\np9ISuhyRIZi0q8yKrThouzYXtFZUK96MFvJEmK/IaTnIyooCzPJ4Wdj73wzi/6FR\n4QQsSqE1Z/96oxzxHnpLV3XNICqV5cWMoC8kPD7nBsIGAcCJWaspILHlS4EtzmQn\nUGdAvkgIEaQBpGJpA2jbXAAgTEE0ZCFpIEMH0uwyBdjmCP1DVBotwgokpKTrZ5GS\nkFAquixw9W8XE+uK6dDe5Ty9MV06d3/4Hp/r0e2vZ7yrWxLd4YnLVx7L7Tjum7r4\nU+rw/clDS3/65aUBnjXZk/1bq5cvTODn2by0W03t/xzUI2XNlCW8793DHZrGVFfu\n3LXx8mC+v+Phfc1J7+U925B3nZ+54Pu9qt7VWMQKfyzVPYs67fZ6xqVnXH1t7M6M\n/Mwbn/Sef+d26Cny5KlBrySeuXl+7ssZp6edP7qIXXBd27Ot6ut7JSn5b9R27zBs\nWOTTm7mu0Qfz+90vGjVMPbJp9zPpa3elep6YmpJZ0/m7a9evPX22GofrVm05MHvv\nockr4yb098aJjIH128sqSzd9/mLXxKaba0blXB2zYfbU/jEHilanf1+cXZ2ztmlU\nZdIHnx0Zsv+L2TU3ZpHgl/MOzO1TOnbVjt+WJZyYW508unrZ3tELSPzHg2c2ZF5K\nez4t0sN7q9yVVr6pYe2MidMnfbt4cWr2H7f7BZTe8+Lh0oa8yi2ZF6ua4uyjPVNn\nbsj59WJL44j2J+tPTImvff/VAQfrsy6+ULLy9eH+pLpZxadjh6C6rNraj3oKT0u3\nfVfGYm3o9o0nUgcpJ+9vD3Rflbg+s5M3AZ/74W7O+BUJxyq69rrgc6eBK6HY5uIR\nc7Iak1Ys9t9ZPX5rr7fm7zi8bt3SzD3d/FVn+9zKTgnunlK3zaj56p3lWY0FE8rq\nY1suHzyWeCiwoMuYRPe99cdr5rw7LfPDqvTTE24s6dtuUvLQvBGeWeuaf879Gw==\n=iNau\n-----END PGP MESSAGE-----",
         "retry_count" : null,
         "ctime" : 1424368292,
         "proof_id" : null,
         "last_success" : null,
         "remote_id" : null,
         "seqno" : 5,
         "sig_id_short" : "BrbDQOoQL1lFM36nWSid5QYDqeKtVDI3ND65",
         "rtime" : null,
         "proof_status" : null,
         "etime" : 1582048292,
         "proof_text_check" : null,
         "last_check" : null,
         "sig_type" : 3,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "version" : null
      },
      {
         "check_data_json" : null,
         "sig_id" : "a8c27c8bfb3c231b065d8193817da7f2359170055a48742e6e62d2e56e7049dc0f",
         "sig_status" : 2,
         "proof_state" : null,
         "proof_type" : null,
         "proof_text_full" : null,
         "prev" : "788bef4ab571d16accad4ce4c0e615dc627096a8464273559d074de2e77151dd",
         "payload_hash" : "d3920272a9064f50d0d6ec76c4f8f0a921d1bbd40c932cd0478204432a09dbd8",
         "human_url" : null,
         "hard_fail_count" : null,
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.5\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ED4B1\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"merkle_root\":{\"ctime\":1424899771,\"hash\":\"e93b7b99b4ceaf80ec326249d0db3a5cf10f51c167e8e1a0ca27e358bf4412a2584030c922fcb1d2b57314525db0602606c654a6110f3d1f4674e70a66d3fdc3\",\"seqno\":163321},\"track\":{\"basics\":{\"id_version\":15,\"last_id_change\":1413691456,\"username\":\"lbredeso\"},\"ctime\":1424899843,\"id\":\"68eb53e6f2763ca1bcf46bd0462cd900\",\"key\":{\"key_fingerprint\":\"95564c8994229ce4a83191f3378d97eca04603b5\",\"kid\":\"0101c828024ab190b29b6ba9b21b7b5dd974201e7f4b74da640ed1a7d7b5f4f202de0a\"},\"remote_proofs\":[{\"ctime\":1394659711,\"curr\":\"3e77356af455edcafbde1e95411998783810fed43fa6bd47caf302b240f00499\",\"etime\":1552339711,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"lbredeso\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"3415d1c8fc7c3b97af4a825bd7c9ed456053205d6060e4e92556432b14f62ab50f\",\"sig_type\":2},{\"ctime\":1394725533,\"curr\":\"4ee6cf5ebece3a3e1374e6960d9e85777be9bd20902c0bea8306f4f7b8989e2a\",\"etime\":1552405533,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"lbredeso\"},\"proof_type\":3,\"state\":1},\"seqno\":4,\"sig_id\":\"067f7d07d7790e2e565bf22eb6abad95f6260d0a3c2979224b99a5e534fc291c0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"5f749cdc6b7d2894c0c79b08601de02ae1188b0f5efb00b9e1b3250c9beeae6c\",\"seqno\":13,\"sig_id\":\"62e6b314af0330921df04717dc73e5c307bb3f38d91fd648579a145fd0b2a69c0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424899843,\"expire_in\":157680000,\"prev\":\"788bef4ab571d16accad4ce4c0e615dc627096a8464273559d074de2e77151dd\",\"seqno\":6,\"tag\":\"signature\"}",
         "api_url" : null,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n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\n6/ozbTWk92a3/d0foSdmzPv+4Btt/wI=\n=05XJ\n-----END PGP MESSAGE-----",
         "ctime" : 1424899843,
         "retry_count" : null,
         "last_success" : null,
         "proof_id" : null,
         "remote_id" : null,
         "seqno" : 6,
         "sig_id_short" : "qMJ8i_s8IxsGXYGTgX2n8jWRcAVaSHQubmLS",
         "rtime" : null,
         "proof_status" : null,
         "etime" : 1582579843,
         "proof_text_check" : null,
         "sig_type" : 3,
         "last_check" : null,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "version" : null
      },
      {
         "sig_id" : "a1d6f082e8cae7db1466ef8d0f00faec8d5a278fc87bb7374d399f3a73c363190f",
         "check_data_json" : null,
         "sig_status" : 2,
         "prev" : "d3920272a9064f50d0d6ec76c4f8f0a921d1bbd40c932cd0478204432a09dbd8",
         "proof_state" : null,
         "proof_text_full" : null,
         "proof_type" : null,
         "payload_hash" : "4be366f49c0dab0d8466c6139af9463fd005140130997dd3da140a995029c582",
         "human_url" : null,
         "hard_fail_count" : null,
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.5\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ED4B1\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"merkle_root\":{\"ctime\":1424907843,\"hash\":\"8304f090361d52ccc6288e7b2f153f4bce8234bc8f8a102434c38c29cac5f8284cad1a28cec71b6139442041463cd33f81744124784ae7bf6999c6b8fe311197\",\"seqno\":163565},\"track\":{\"basics\":{\"id_version\":34,\"last_id_change\":1421868407,\"username\":\"kevinvandenberge\"},\"ctime\":1424907860,\"id\":\"5e9faa7093e4e5d90402ad6faab1ef00\",\"key\":{\"key_fingerprint\":\"e177d740bb6b666399e7fce637c6f2ec8bf249f7\",\"kid\":\"0101087ed75f764c5451817d4de5710f922d09b113f283a9bde4d1ad060e36e1e8c80a\"},\"remote_proofs\":[{\"ctime\":1407544575,\"curr\":\"997fe56968d38af4251a2fdd1172a1fa179839e20b81e9e56091e197386adb33\",\"etime\":1565224575,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"kvberge\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"a7b080c03b7a3b90ea07ae92dc27b2d4340da132d3dd95e1b26ecf5ea45e750e0f\",\"sig_type\":2},{\"ctime\":1414697229,\"curr\":\"a839d569d5090bbab585b257d9f5ad882a01d39df2fb66580412d15aab359110\",\"etime\":1572377229,\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"www.kevinvandenberge.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":8,\"sig_id\":\"a68108ee18450fde61bd9444bc1c254a3fda3c68cd17ab4662ec6e1c967162640f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"a09db57e382b0131b57a93a249b0c7ec4c283a338d2a40f1d29ebf12f70ba57d\",\"seqno\":14,\"sig_id\":\"9ceaad0e7b363cee8c433870c5fc63a53973b52f05610622a7e9ee0fdb3eb4c90f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424907860,\"expire_in\":157680000,\"prev\":\"d3920272a9064f50d0d6ec76c4f8f0a921d1bbd40c932cd0478204432a09dbd8\",\"seqno\":7,\"tag\":\"signature\"}",
         "api_url" : null,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n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\n+f6Wx/b+/KPnr532bw==\n=dyjl\n-----END PGP MESSAGE-----",
         "ctime" : 1424907860,
         "retry_count" : null,
         "last_success" : null,
         "proof_id" : null,
         "sig_id_short" : "odbwgujK59sUZu-NDwD67I1aJ4_Ie7c3TTmf",
         "remote_id" : null,
         "seqno" : 7,
         "rtime" : null,
         "proof_status" : null,
         "etime" : 1582587860,
         "proof_text_check" : null,
         "sig_type" : 3,
         "last_check" : null,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "version" : null
      },
      {
         "retry_count" : null,
         "ctime" : 1426175653,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n\nowFtVHtQVUUc5iKSkIaZNQ6SY0dLy4vu7tk9DzKxEsdJ1NLERpI7u3v2wAm493rv\nRUS8RFomaGqKqSNWMD4anTEp056GWmDiC/MxozY4A2ZMZpnKZFPSHrAHM539Z5/f\n7/t9v+93VvXrFRProTEPLY4tO1XnaWpnMTM9e+pLFRawSpS0UoUXOMIfcWd+WiiU\nNCVflDAaFqOdwNBiwRSvMk+Ewk7AL4+Qpek2MogBVZMhHRqWTizGDahTrmKDAgNj\nRAhSol4XxgW1HX+uCAVDjhtD4dQizOYGQ6qJGICIIPkWQQ0CgKChAqQLCzMoo+YF\nwpEebJQuTJ9jyd3/uZ/fdQCgHIzrwtQ5BJpOgMGgLVTdFDrRVQFM06AEco0Qy5Ls\noQZsDHTTpoaOANcAFRhQCVd0J44pNCFvqpatGzomgnPMVF3maBIGTfdiWITuKBf0\ni4Kw8LvZR0KU57v5S/IOD7szx/L9oyT0KgU0HJHJ+HgelQrJLamcCnQ5emCyQG6u\nYMLF7GKEEVYN3VIJx5omCDSwzjhGKjaBwN2M7ijvitVTfYEFs/Z7EcKECUKBioVK\nVWrYGHMktQEqhUzV5KqHnhozkWZCCNxwgOpY0wGVwuiGtAMXGmYAqUiagGhMQxZl\nUEOmsDUTY8tm8h112YdEYSAifMFQIGBLPbLneJWwmOuLUKfAZRukJQUBavnyaDhP\nxqVQ1gRISrYsJucW0BGxAFc5JZY0oAqJTQEG2CaStUGRCTCyKbchwhxYtqF0ofsD\nXVKHndxu20jHUIxVQm1mcQQ0KEzJ1TIkCKYmxJDZGOmagW2LCtMgnGnMgpqqYhsA\nXePAVqJucUuCbmm6a/yf/oDyjEecwu5qaq4gRPUqYn7QCQmf494gEh3Iz6sEQ2Ke\nW08mVE2zsSlpUwYsQ5aVa7K/qG1iTbUtAAjEAKrSubrrQypX1DQJQCYnBvo3UUMS\no7kSUqbrp5GikDRNYkVsSlyMJzYmvnes2/YxiQn9//4XvFKWdJvs2uuFC4fnvn33\n5mNt03yXHm0aEt88fFDunD9bDjXWDJu5ct8j981qLQ9e3r665kTZkeNFqcE+DTWV\nm7aHa0+DSQMS1rDsCcebF1yal/nL0e9S+m8782Xo2S8+uZxw4+mmhszEP86vLxqz\nlo5uTZy2/PXBiy5cvbLkrmVbZjfET9k4/Ty/vjM+9al+70Rxs3MzY4OBmjIOHuAj\nqqrW4bG3JjyQF7gH12SgrKSlB473qcufUbS28HYbboi/He3NU+Zro3aNf2nV78kV\n6eCJ6LULaR2TVgcRevPhz5vUbdEtHe9Vpl8defrbramLFk1rGTvy2Lkzzbd21AXb\n4sbELhyX2df35OGXT5feTD83st5zv7/h1bf0s3uu1+TUj0hMemZh39kL4g5//dEx\nknbverK7deen1paYF7XKZUptpGPK0I12Smnfgt2TqzavXTDTd3JU588D2JXq4qys\naK3s++LOHcPU1462V94YtaNC937w1eqp69oeT/ZmzDo6vbG2eNtzWUsGJm+4Vl1+\nJO3BqZ0dZ6uuX8z7aUZ5/VwB13wTevWgZ/Ihz8XQu3HtUb4/Wx3IX4jLORdJ/XBl\nY215ekL1Y3PEis4+WgWfOLh28ZDkQZ5Z2ydmhX7MfP6N38rfv9ayal9Ov937fxjQ\nfKr11xPVdZ+taK3xZbcEGwtubt3bvOhk2bhLN3LazyRx0xn/cXBpr02Hvl++6y8=\n=OxOX\n-----END PGP MESSAGE-----",
         "api_url" : null,
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"2d67f2858139b2718d75dbc817ac348a08442552\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ed4b1\",\"kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"track\":{\"basics\":{\"id_version\":1,\"last_id_change\":1425307070,\"username\":\"boggebe\"},\"id\":\"424387d35c466e51847bc423490e4b19\",\"key\":{\"key_fingerprint\":\"e4ebd2245be5a034e3a3a8f44c23e003a1b3644c\",\"kid\":\"01016b9269110e5180a74670a19e78d67ce64b023281756b62dab1629ef6944dfb110a\"},\"remote_proofs\":[],\"seq_tail\":{\"payload_hash\":\"a179f0e00fbc7ccd0725d0c3ca5d285315fa0404f50038a29042facf124c0df8\",\"seqno\":1,\"sig_id\":\"183a4435afbdc2061e9162d85004a9141bf427684fdae985cb6bd16334f0076c0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1426175653,\"expire_in\":157680000,\"prev\":\"4be366f49c0dab0d8466c6139af9463fd005140130997dd3da140a995029c582\",\"seqno\":8,\"tag\":\"signature\"}\n",
         "human_url" : null,
         "hard_fail_count" : null,
         "payload_hash" : "65a62af1247817a88c841f0f45aa86876c0d681cd0d25aec326a86a2d5568412",
         "proof_state" : null,
         "proof_text_full" : null,
         "proof_type" : null,
         "prev" : "4be366f49c0dab0d8466c6139af9463fd005140130997dd3da140a995029c582",
         "sig_status" : 2,
         "sig_id" : "fa0034c804ff5179af9fed5742688939984c7c3fb34c21c193d4942ee99c9af20f",
         "check_data_json" : null,
         "version" : null,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "last_check" : null,
         "sig_type" : 3,
         "etime" : 1583855653,
         "proof_text_check" : null,
         "proof_status" : null,
         "rtime" : null,
         "remote_id" : null,
         "seqno" : 8,
         "sig_id_short" : "-gA0yAT_UXmvn-1XQmiJOZhMfD-zTCHBk9SU",
         "proof_id" : null,
         "last_success" : null
      },
      {
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "version" : null,
         "proof_text_check" : null,
         "etime" : 1587401499,
         "sig_type" : 3,
         "last_check" : null,
         "rtime" : null,
         "proof_status" : null,
         "last_success" : null,
         "proof_id" : null,
         "remote_id" : null,
         "seqno" : 9,
         "sig_id_short" : "Fg53UCl1THhCRsO4lfJVt_A_Ln_HldB3Viqi",
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n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\nc1vbfwA=\n=hBn5\n-----END PGP MESSAGE-----",
         "ctime" : 1429721499,
         "retry_count" : null,
         "human_url" : null,
         "hard_fail_count" : null,
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"2d67f2858139b2718d75dbc817ac348a08442552\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ed4b1\",\"kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"track\":{\"basics\":{\"id_version\":9,\"last_id_change\":1428087044,\"username\":\"joneilloine\"},\"id\":\"52b1f4132cb3625164ce231f62f76d19\",\"key\":{\"key_fingerprint\":\"dc14c51ca9b555bf2df756f64d8c5a528e73c001\",\"kid\":\"01014ad50518fc5e96fca02dc3e910670cfefa21be4540e564b7d3cd31d088223ea30a\"},\"remote_proofs\":[{\"ctime\":1428086623,\"curr\":\"00c53535a951a9affd158cd2eb203b175d582db62e4430bf4e78654217483aa5\",\"etime\":1585766623,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"joneilloine\"},\"proof_type\":3,\"state\":1},\"seqno\":1,\"sig_id\":\"211f34e646014b39b39aabf03c5b5ce2e52bb5bc0045a63a3e2fe1bde9d793460f\",\"sig_type\":2},{\"ctime\":1428086935,\"curr\":\"c68527dfe977386effd41fe9a6a771ef4089a1bcf572da2ad126d1c36ed4454b\",\"etime\":1585766935,\"prev\":\"00c53535a951a9affd158cd2eb203b175d582db62e4430bf4e78654217483aa5\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"joneilloine\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"317b075435e12091ab5ecf1a264a3c0fe36e0b229fdf38be30ca1c39b557c5720f\",\"sig_type\":2},{\"ctime\":1428086989,\"curr\":\"801a08f151875da0b455104a69e3fa24f1911454b0ff79fcc0d108a721598a85\",\"etime\":1585766989,\"prev\":\"c68527dfe977386effd41fe9a6a771ef4089a1bcf572da2ad126d1c36ed4454b\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"joneilloine\"},\"proof_type\":4,\"state\":1},\"seqno\":3,\"sig_id\":\"f0350a30fbc27301c16beeb23d5bd54d53b79082f1b69663e3587a588b595ce10f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"801a08f151875da0b455104a69e3fa24f1911454b0ff79fcc0d108a721598a85\",\"seqno\":3,\"sig_id\":\"f0350a30fbc27301c16beeb23d5bd54d53b79082f1b69663e3587a588b595ce10f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1429721499,\"expire_in\":157680000,\"prev\":\"65a62af1247817a88c841f0f45aa86876c0d681cd0d25aec326a86a2d5568412\",\"seqno\":9,\"tag\":\"signature\"}\n",
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "api_url" : null,
         "proof_text_full" : null,
         "proof_state" : null,
         "proof_type" : null,
         "prev" : "65a62af1247817a88c841f0f45aa86876c0d681cd0d25aec326a86a2d5568412",
         "payload_hash" : "37069f84adfad38043651e257053a83e5e22ae2b5cc06909879507bf74efbc53",
         "check_data_json" : null,
         "sig_id" : "160e775029754c784246c3b895f255b7f03f2e7fc795d077562aa221ed7703420f",
         "sig_status" : 2
      },
      {
         "proof_state" : null,
         "proof_type" : null,
         "proof_text_full" : null,
         "prev" : "37069f84adfad38043651e257053a83e5e22ae2b5cc06909879507bf74efbc53",
         "payload_hash" : "181bf1660e64f8d0399243b18d2b71209a6f20a093b06527f56417d9499709d1",
         "check_data_json" : null,
         "sig_id" : "821f8b8415f4af8035da234abaebe09d4da998602b568399b5594955d261c6e40f",
         "sig_status" : 2,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n\nowFtk3tsFEUcx6+HCEcBCZb6oNjrFoWGAjOzOzuzgLx8AHJYU6GxQFt2dmfp2uvd\n9e5aKEhJeYUSVMqlpQ0VCoRCEESwGIsBJfKQAGkRSIg8VB5GwFAkJQgodbZWYxN3\n/5md+c339/1+ZmdNn24ud5zuSlnqLju7J+7EDeaaob3wyUKJBc1SadRCyfDbPBB1\nRgG9kEujpAJeyvQIH2EHvfM4k9KlEh6O2MGAWEKmSixEMYWyxhCB1CTYZAaFRDdk\nheqAKgrCGEmL0h0ZR9SyA3N5OBS2nR6SoZuYWQZlSNYQAxBhJPYiqEIAEKQyQISb\nCoOia34wEu3iRurQzLNNMfs/9QUdCwCKlxmEa8SAQCUYUAYtLhONE0xkDjSN6hga\nKsamKdxDFVgKIJqlU4KAoQKdK0AXcsWdfTSuclEpmxahRMHcMBQmE5FRwwxqTmGE\nhzvJhQLcH+EBJ300rBsFTn5h3jYizsg28/4lCdMlvx6JijB5Rr4uCIkpBWmqDFVV\n7aJZaBfwwpA/4oh2WKIqUnSuMwiZyYRvwgwALZmoiCIqd1jqRO/Q6oqfYIt8PU9V\ndACobgIo6xrCGuKKbFAETcJUhClTxMl2AWpSaDDMEcSYGar44oomW0gjhOlQVSjV\nOeKUyJByGQNFEKRYIzowTQMoRHOACvdhXhiM8rxQOBi0BJBZOQ6m0pAT8m9a//nT\noFgzonZhJxcqThLgdInPD9lhnmc7FZioFIgnXQqFeYkQkQlQNYsqumnppkyBIqsY\ncoTFTlmnMhcBkDDKsGGIQqBRomFAmEUUbjEDy6J/hBcFgkJbiEb1uUIzYs8N6NHi\nMJcW9apwJz3hinO7nuzudm6Qq5en3z/Xao72VPusiacupky7MHLdUlfPnkP3xvr/\ncTX8WkN5Xeta9GDr7b718/s3eye/ZNYcX1szesKfM2/dO++dem1nRWaSD3/kw2p2\nQt3VV342EpO3bGz98OqdCd+PS+0TN3yOPzvxypZ74fLZGbeP9zDpvu05v6U17fUM\ndp9Lj01eE91y+OLltqScnwqfrQrtz7nR8nFVecO1Mb5Jp/lXA0fVH6gr9v2y6sCt\nYTPlzKzr6ro3a9sy3h3emH8ka+W2GQmHsnbQuLoLmUtGTojDPbLLzo8/1zK/bUP/\nqh+bDi2LT737flbJ7rKjT8dPuRR+fKW2+vqumpTeA778ZnjqnboXc1c/Uzbn+cGH\nE05s/ODkMNlzafDDY/KS5P1wlqdWye9e2pCYMX73vppHWYu7nXlZyc0oyRiQMLXf\n9E1Xvmg9vX7xiJDdfLRFu1wZ+/X+KlZD9l37NuZ/+7t5OzyfJ72zeUzi78nHcryt\ndPum6uqtKwfumL0rI7ZhwblXS6XU+LEP7zYfblrhTfY17HxU8WliWuKknhCl7WmH\n/XIy7w+dWTvsweP46X3PLM8vOjj05LL2sa5T47r1Pl7uScvzNuauqO81+qzZ9ENq\nn+W59a1F/hU3QvIbBdkblsa8lerjm6tTNj9HWlZPaUzxrW9P9TS0DvEVeG8eLNq2\nfNCgKfejR90LmisaPyu+YI3XWl5/q7p7cmnlkbZp1987dmB0bMhf\n=9cMh\n-----END PGP MESSAGE-----",
         "retry_count" : null,
         "ctime" : 1429806705,
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"2d67f2858139b2718d75dbc817ac348a08442552\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ed4b1\",\"kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"track\":{\"basics\":{\"id_version\":1,\"last_id_change\":1429631666,\"username\":\"mikempls\"},\"id\":\"8624aeab11bdb9fa7bc01f3762828319\",\"key\":{\"key_fingerprint\":\"75f764a008ad013a92592e43c821d7b6258b4271\",\"kid\":\"0101d81cb5e2155bc6d81e493f2977ba16488ae2e87318e3504e408597a0ddc047940a\"},\"remote_proofs\":[]},\"type\":\"track\",\"version\":1},\"ctime\":1429806705,\"expire_in\":157680000,\"prev\":\"37069f84adfad38043651e257053a83e5e22ae2b5cc06909879507bf74efbc53\",\"seqno\":10,\"tag\":\"signature\"}\n",
         "human_url" : null,
         "hard_fail_count" : null,
         "api_url" : null,
         "rtime" : null,
         "proof_status" : null,
         "proof_id" : null,
         "last_success" : null,
         "sig_id_short" : "gh-LhBX0r4A12iNKuuvgnU2pmGArVoOZtVlJ",
         "remote_id" : null,
         "seqno" : 10,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "version" : null,
         "etime" : 1587486705,
         "proof_text_check" : null,
         "last_check" : null,
         "sig_type" : 3
      },
      {
         "version" : null,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "sig_type" : 3,
         "last_check" : null,
         "proof_text_check" : null,
         "etime" : 1588094983,
         "proof_status" : null,
         "rtime" : null,
         "remote_id" : null,
         "seqno" : 11,
         "sig_id_short" : "Ng8Bg67BNf7DeMEhWmKuUWC4UgVuZb3Bmn0E",
         "last_success" : null,
         "proof_id" : null,
         "ctime" : 1430414983,
         "retry_count" : null,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n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\nafumzkEHxny/6/fPm5b8BQ==\n=Iux8\n-----END PGP MESSAGE-----",
         "api_url" : null,
         "human_url" : null,
         "hard_fail_count" : null,
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"360c42463e20df0ff132e91e17bdbc94f2435156\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ed4b1\",\"kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"track\":{\"basics\":{\"id_version\":10,\"last_id_change\":1429906357,\"username\":\"muxneo\"},\"id\":\"840aa7e4bdcf11710b111e9a01b88d00\",\"key\":{\"key_fingerprint\":\"87f85e2d02ca0c0405820b896e29f38a21640b8c\",\"kid\":\"01011a9eb010b256125ac2a6026f7df2a4f19bf784b574d4af539e3b4066ff06950a0a\"},\"remote_proofs\":[{\"ctime\":1429906221,\"curr\":\"c89f2af02b59dc7a31f102a18edb413dd73de7351dfde9c12a46f7d582dfdbfd\",\"etime\":1587586221,\"prev\":\"a2da2652763b65550b47e760c5ad7bf502a3728b6a3ae5fc9fac7d3f5582f1d5\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"muxneo\"},\"proof_type\":3,\"state\":1},\"seqno\":7,\"sig_id\":\"7eba87a53ab0fcb4bf91a23e600294531820f08e2bcca31f0c37f8e21cccae260f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"5b11ae1044b17ac96d3953d2c0ee32fee9677baf14cd0f1bee10101d3a1da009\",\"seqno\":11,\"sig_id\":\"daaea871cd483d33683d9cfefb50de4116c997b7d8d488f77a7d0e03599e9af60f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1430414983,\"expire_in\":157680000,\"prev\":\"181bf1660e64f8d0399243b18d2b71209a6f20a093b06527f56417d9499709d1\",\"seqno\":11,\"tag\":\"signature\"}\n",
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "payload_hash" : "24876c04fcbee2543202274c5145354f0b6b34a38e37d27824ab0260f98d419b",
         "proof_state" : null,
         "proof_text_full" : null,
         "proof_type" : null,
         "prev" : "181bf1660e64f8d0399243b18d2b71209a6f20a093b06527f56417d9499709d1",
         "sig_status" : 2,
         "check_data_json" : null,
         "sig_id" : "360f0183aec135fec378c1215a62ae5160b852056e65bdc19a7d04ec2d2ac72d0f"
      },
      {
         "proof_status" : null,
         "rtime" : null,
         "remote_id" : null,
         "sig_id_short" : "FjfqxBESjZxyqPKq1UIk5ptMznevnhWxWwOf",
         "seqno" : 12,
         "last_success" : null,
         "proof_id" : null,
         "version" : null,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "sig_type" : 3,
         "last_check" : null,
         "proof_text_check" : null,
         "etime" : 1588095020,
         "payload_hash" : "8b9f2b83ae4c160c03c5ea39af51b257d5a561f9211cb99bd3aaf0be9638dad9",
         "proof_type" : null,
         "proof_state" : null,
         "proof_text_full" : null,
         "prev" : "24876c04fcbee2543202274c5145354f0b6b34a38e37d27824ab0260f98d419b",
         "sig_status" : 2,
         "sig_id" : "1637eac411128d9c72a8f2aad54224e69b4cce77af9e15b15b039f2b65a4266f0f",
         "check_data_json" : null,
         "ctime" : 1430415020,
         "retry_count" : null,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n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\nv377BzW9H5fW8I1HGipqrszUjDjyJQ==\n=vtaW\n-----END PGP MESSAGE-----",
         "api_url" : null,
         "human_url" : null,
         "hard_fail_count" : null,
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"360c42463e20df0ff132e91e17bdbc94f2435156\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ed4b1\",\"kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"track\":{\"basics\":{\"id_version\":13,\"last_id_change\":1406184490,\"username\":\"scottvlaminck\"},\"id\":\"cddcde66a89b5d47044a11cdcf961b00\",\"key\":{\"key_fingerprint\":\"d64936887c0e09f329f00f78de2c49dd4b40bd50\",\"kid\":\"0101856e1f9379dac8efa6255bdfc462a9c12f60f60648fe27c50c177c52e16439fd0a\"},\"remote_proofs\":[{\"ctime\":1394818406,\"curr\":\"16f911ed734f1c8695638423b5a0523ffce92a4f9e7d0834c07e355b121af23a\",\"etime\":1552498406,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"scottvlaminck\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"7a1740607b28ba6a23e3085c8aef1242f34c7a600a0e52f4199fc5c30c6bc0680f\",\"sig_type\":2},{\"ctime\":1394819042,\"curr\":\"cadcc05b4839bb4bb49d605b25ac3ac8ca5b763aa299ba9d640d0202f6f3ced6\",\"etime\":1552499042,\"prev\":\"16f911ed734f1c8695638423b5a0523ffce92a4f9e7d0834c07e355b121af23a\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"scottvlaminck\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"6e93dc268566c599c1c8954848d045e516a6c2200d0de50c158171fed0603e280f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"8c4ea2b1bd18623b0d675fb4df890003498a7b840fb9c6e6fcfc1a18f139e3c8\",\"seqno\":53,\"sig_id\":\"53a7b82766bd90d6d0b8dac5f1ff6c320959b1193f692698f858b317e6b477850f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1430415020,\"expire_in\":157680000,\"prev\":\"24876c04fcbee2543202274c5145354f0b6b34a38e37d27824ab0260f98d419b\",\"seqno\":12,\"tag\":\"signature\"}\n",
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1"
      },
      {
         "version" : null,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "last_check" : null,
         "sig_type" : 3,
         "etime" : 1588095245,
         "proof_text_check" : null,
         "proof_status" : null,
         "rtime" : null,
         "remote_id" : null,
         "sig_id_short" : "P75D8aps5ZgJwSVLxA2DUaLBCTfyYaGDbrZr",
         "seqno" : 13,
         "proof_id" : null,
         "last_success" : null,
         "retry_count" : null,
         "ctime" : 1430415245,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n\nowFtk3tQFVUcxy9XcXwODkw1pImuKFZ0PY89Z3dBfJBo2jSj+BYLztk9i8vj3su9\nF5MQNcuKAkfKwhrTsqaMQMOSMcowfDU6jo9x0AbHV2ZqqNMomcFEZ4lMZ9r958zu\nd7/n9/18z64d1MvjjWKeES95l5+ojzp0hXvmpOdWlio8YJUoKaWKWeAIf8Rd+Vmh\nUFKUfFHCWVj4nMBwf8ASvrzw8B5NsrJEhMJOwC9VwKf5NKUs2ZW7H9uOP1eEgiHH\n9VJMZhFumzpH2EAcQEQQ1BiCFAKAoI4B0oSlcigtFwfCkft2Vbo9sx1LPr1HnzFZ\nTXf1xT0vDEGFZVnYsjVdU4kwTZVjjRBkEA4NVxgWoZ5IQb8oCAu/O26hCOUXiOxQ\nINCd2Yw4rgKqGKiQQELlQCy82A2ICYSIqrZqm5QwqOvunoJbmo5URgHiiFs6oTqA\nJrJNjRu6ZkNgqoZMZhOs2sK2CKW2wAwKYskbyMS2RAKJTM9tyFXV0BBFgFLMVQxt\n1bSEtOUUa7pMEBZF/oCSgiDGEMjZIyFm5rtTS1KOGXZXjpV9txOcrBSwcESSyzYX\nM1mHmwsZOtAMAu7y2H3GJVLo5IvCYEHYRXI/A6SSZKWbsU5lUME4hNzihs00bgJo\nY40iHem4m3FP+W5f9x8AAnXN0GQzWMNAN3UVECg0gnVhU02zVWBhFXGduR7dmwEo\nvQWjpqoCCqlQGWYUEg4QsCiDBqZCsgNcHitGpC2QRqqOkWoDwU2D6pDrgLlxQqIw\nEBHZQVmxLRllPeuSKwm6qf8BeM8xhv8fXywNOiGR7bgKosmK5ZWsBENiiYtFspCj\nYyZUE1JgAmwSwbDBbAI5IppFGKHQNhCEJjcMbmHGbMCFQbFuMcv4r1goG4uwXOkZ\ndnL9LFIcEkpZuXdob0+U19Mn2uv+qJ7+/Qb/+/f22hPTtejLJ2o+ivJP2jguK/aq\nuvW1qqaNqfMd/FDqsHk7Wu3O3eWj4rahH2ey6oXfdCTFTZh7dueSfjBtyruT+Pas\nQxM2vHK7pHhVfBqbOibzyj60bNDM8oHK73HvtzX93Jz+9s3DHZMuDv5wU3lXvb59\nf15nfPvp5avGXRo+UmMtDWdjBuTGBa9UNgaOi9Klc4e1tJX+ci2YU/d9Y+z4XaNI\nYkNN4eA5Ca2zWxpOndp5dEsqf2rCw6v50Eier3dddXpR1+EDUUn5F1aOTTwwYsje\n2be+jqpqWjMTdYzsWPT0p18s877x7frU6621CTld8fX0VrFde+tOl+fOAysSlXYP\nvXpx5+4fYmBj7KwhyiiRdGnLZ5+kVvuje01d5VT5MscWrVyZ8OZJ6h2WxSuchINH\nFrYfqepTP/mvZwZenvFy0aLW/n+mnlo3MfHo+NLn97K+eZXNFZvXTUsy17zweuaG\n0T95fY+u63zw+PVtW3ecn1LXpk4zE9PiG/pVB98pgzVNl3uvj7GiFdH34LHN3+25\nUenfF3Mu57llSuyQ/vNuvrp1VuOMx8o+3zRi7iN1KU8u9U4npef2ZAWunTm0377Y\n3LV2xe1j89NapobOZBy5NmDamNO/pTk3PCfK3ruQtS+06a3avPbRHc0Laqas9gYq\nvpr4h+fF89E5bcW3I6vHZahjPvi4oqqz9tKujeVXF8w7Of3xyK+Zl+Pb/gY=\n=nCVg\n-----END PGP MESSAGE-----",
         "api_url" : null,
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.7\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ED4B1\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"merkle_root\":{\"ctime\":1430415156,\"hash\":\"03511264f4fc65a188619eebd7824a602b2bd856801c2fc7b987f10c49ed4f534fefd566fe3a1e5d5d504b1fb23157edbf1b44972620663b431f4cdeb2bb6378\",\"seqno\":213310},\"track\":{\"basics\":{\"id_version\":3,\"last_id_change\":1429807950,\"username\":\"mikempls\"},\"ctime\":1430415245,\"id\":\"8624aeab11bdb9fa7bc01f3762828319\",\"key\":{\"key_fingerprint\":\"51879778737308c84051e7538ef677f40d342b8a\",\"kid\":\"0101fea6c440616e4a3a615b020d6a1936e1fb0b5bfa58730ef648324f0ebc9681b80a\"},\"remote_proofs\":[]},\"type\":\"track\",\"version\":1},\"ctime\":1430415245,\"expire_in\":157680000,\"prev\":\"8b9f2b83ae4c160c03c5ea39af51b257d5a561f9211cb99bd3aaf0be9638dad9\",\"seqno\":13,\"tag\":\"signature\"}",
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "hard_fail_count" : null,
         "human_url" : null,
         "payload_hash" : "479ed752fa2eb513cf59f835b69098e9ed845593a8e96b400baeab20f88e5a76",
         "proof_text_full" : null,
         "proof_state" : null,
         "proof_type" : null,
         "prev" : "8b9f2b83ae4c160c03c5ea39af51b257d5a561f9211cb99bd3aaf0be9638dad9",
         "sig_status" : 2,
         "check_data_json" : null,
         "sig_id" : "3fbe43f1aa6ce59809c1254bc40d8351a2c10937f261a1836eb66bd5186c2b250f"
      },
      {
         "last_check" : null,
         "sig_type" : 3,
         "etime" : 1589328187,
         "proof_text_check" : null,
         "version" : null,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "seqno" : 14,
         "remote_id" : null,
         "sig_id_short" : "jMOmIJ1s5vaPFp5mLBUKfud2h72BNG_mfiSP",
         "proof_id" : null,
         "last_success" : null,
         "proof_status" : null,
         "rtime" : null,
         "api_url" : null,
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"f41a7d6fbb57b25bbfc781a4496dce572f77e86e\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ed4b1\",\"kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"track\":{\"basics\":{\"id_version\":11,\"last_id_change\":1429364269,\"username\":\"andrewsterian\"},\"id\":\"f2896753a9fa6bebe59986a1ac153300\",\"key\":{\"key_fingerprint\":\"da6dcc39c64659b634e66c30f4653eab346f2c58\",\"kid\":\"0101925437405dbd8fc3865841bb61a88ada8b2904359db3000d7b4806d440fe38050a\"},\"remote_proofs\":[{\"ctime\":1418773663,\"curr\":\"737f34cbdc2f98be791e3b8705929ffe77ef66ed123914902bb9e9bff6f05272\",\"etime\":1576453663,\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"andrewsterian.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":1,\"sig_id\":\"0d8d26612a0b6d5f266198b466a8df1da9bee43fc739a5699495928c036a9b680f\",\"sig_type\":2},{\"ctime\":1418835791,\"curr\":\"59adb62cf6b5ce119901b2828f5c839ebc89fde67852398199bd33336b58a183\",\"etime\":1576515791,\"prev\":\"2d2a2b66fb74f40bb9751e8e9377938b63f4f3239ea9a14a123c6572969f02f8\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"steriana\"},\"proof_type\":2,\"state\":1},\"seqno\":4,\"sig_id\":\"adb072500be02202ebb9ecc82d7f65f83414b0b8bbdec3e7a892235599dd969b0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"80881a036552fa43a8c9dc93559aa44f03ccc0f33de4339d688e15d44a3f67ef\",\"seqno\":13,\"sig_id\":\"c215115ae3d975ff32752a707263725ec07df0596818e86560670bcc234078740f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1431648187,\"expire_in\":157680000,\"prev\":\"479ed752fa2eb513cf59f835b69098e9ed845593a8e96b400baeab20f88e5a76\",\"seqno\":14,\"tag\":\"signature\"}\n",
         "human_url" : null,
         "hard_fail_count" : null,
         "retry_count" : null,
         "ctime" : 1431648187,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n\nowGNVXtsleUZb8u46BCqG46JxPXMTImFvPcLJQwTYDJ1rmhNh8rZe+059PSc7pwP\nSi3otCsi04XOBFzkkjhj5eLQxYnJ2ASCMsBsa2SjZoPJ4hZ1GyJxk3mJe77TjtGN\nP9b+857zfuf3PL/L83zrLxlTU1drahp66+45+mztkbdtTUuL3N+TsSXfnZndk3GF\nfCgm6aloOkJmdqY9dFtTCbPypS90BZtpzKwI5Uq+VISryLCRXkRrubSEWxudVNgw\npoV3gUsSpQxKhMzqxhQmBY35Ylsod5bzaY2MM57Dj5QlVBOLMOEES0OwwAgRrCgi\nMnhmMVTNlSrJqG4yVcxs3sO3F3i+vXqBMPxbJ4OWDiMhOVIWx0ClDpJLGpDWynDs\nBOfee8mxQJEhqaNRkiAnkAkMGYBbPlJHBxHgSeqjVJLx4ByzVHJONLdYpw9WQnlE\nuc5iKFRCMWWflI1rT/lD83lXSU95nz2nJMaNmYKpJMAm63IGJILvGNFUMCL0KFBT\n9OXQVUlCOW+q0NXGIlEa2FEDnQsbbOBATBhsHOaUIpQ5Z0Cq2WgTvAG3HNVOMMG1\nFZQFIRwFIQSnwVjKRCSOq1GiasIZlQxxb72KjirBFcPWCmyUMt6ApRoxyrW3UB55\naZlCwjOGQH2FOIgKvZdDRykJ2c5yqRRBlDsgfkm+o0oeKympELQx45aXy1BVUhkp\nc9Y7ErWyQWocqFUScU10jAGiFgWYgyFLmGlErNVB2xhFRJxIAu2HEXAuBePD4CMd\npKpUu6iOQC649qw3ickuq6T29FTTdyEDZrlSBwDDT5OSKxXgOpcknbNTblW4bNLd\nmVYECRozlcQk6Qe4rIRvFUtwhFO+bTjDyCtPhMDEICs8j+kZeDIhjPIRe6NtCIzC\niFFtuNCaaWCuHKICroRCMTOMNlySrG4cpaaiHBQ7pybXxltBXHxLWO4CxlojbIki\nKnKnqA7WKR19EFJxEFTBvfUU/uBxZWDURsvJ8TB6ZzmsAHTiiSFWwGaQDOYJnIDR\nCipoKqWmCjIWWaSAG4w2mBnwDEZQEi10RCSmUfs/jRkxJenKJ+DI6Pkb8cj8txvk\nQlaw86wAaZAkHCEbECGIhDRKziniZRQ8Ksows8gqa31wNEijNCGUw8x5DxTs/1hx\nV7VMNjH5Qtp0p+kulIzP5kwlB+UUUrA1wUdYI9EwapTT3ukU0MAyjYg651Ck1IP/\nVHuhVMAcZsnQKCD1mf/kiZ7HwhHMMeYmUA/yR9BbcmIkMBMU2AWHpIfR2DtRC4UV\n7GkuYEPCtnSEwg6E9QY8VqfLq8piZIedt/9T9c4ljGLBAEZCLFZ25sshmy8OR0Oh\navhHksFg88KeBZ6gKsfURa5BUG6FRhoCErxiwBs0CFpYBhYYWEEEReDMjRTncQXH\nEtOW+pxvK5pkeRneMxc/WHflp2pq62rGja1L32s1F19U/++X3ZWNkz+e4rtXfn/V\nSfTNQ4ue3tV79p3MzsJzt3c091896Y3TZ1f9YevPr39l4x93uE+6D5489t6xZ6a9\ng577ygPtR16952BxAttX+7vTl9+350/k7DpceWxf7jNbMi9f+qsZhVsGbvjB1fff\nn1m6e3orefFH/Zff+Vu+oK+na8fWm55+/MAAeu1rzaTn8NIdAw3Xvbb9Kf29/qv+\nebDkxi/+zZNrF8xtrds1+I1Fkyd997N+YE7Tznl+6tzPXfPoA9sGeuf/+sU1F9U3\nzezavaQv19Dixxx/JDn86p4ThYc33zDhl4Nb9hdntp4aqqNr/norn7d17T/ufvPM\nwfdbl/QdevLS9vefv6y2fzyfufTDVc2PrZjX8Pkfr1s/8Y27N7LDu165bv7CTeqJ\np858sm16ofnO54sHercP7R07edGtD78bbj86tFjEU/aDae8t+85mbqZe9XJT88rK\n7r1zf1g3ZUp7/rJvv/73w+v6l+6Z9ahEmxs2HH9h7bK4e86B3o03X5PjOT9u0tAH\nV+T2fXVg2vyPeOOyJX+b9OVs/X0v/GVJbsYja4o/G7prRtPYD7ddv+uJlzwav3DT\n9ltOPvNQ04PH3z1RPHnsJxPaV7Ys/Prgzdmm9XfQP79+4219t62Zs7z1Sxsvmdd2\non3w9LgNv1+w7oqjv2j56ZEx10794qHNe7ZsyIb6txYfaP34zL6Pam3PxJ3XbvOb\npn96a9e4Z88WT90bzEuDbw702cVd9fdOf/vx/Q/9Cw==\n=QG2X\n-----END PGP MESSAGE-----",
         "sig_status" : 2,
         "sig_id" : "8cc3a6209d6ce6f68f169e662c150a7ee77687bd81346fe67e248f2f40b469780f",
         "check_data_json" : null,
         "payload_hash" : "20639787fa9a049dd51059eb36c58b88181d2c97d856c84e610a98e225609903",
         "proof_state" : null,
         "proof_text_full" : null,
         "prev" : "479ed752fa2eb513cf59f835b69098e9ed845593a8e96b400baeab20f88e5a76",
         "proof_type" : null
      },
      {
         "proof_state" : null,
         "proof_text_full" : null,
         "proof_type" : null,
         "prev" : "20639787fa9a049dd51059eb36c58b88181d2c97d856c84e610a98e225609903",
         "payload_hash" : "ca08223a56cd5127ad5242912954c0b2c9b9345dd2475f09a4c3f9a7fa459e21",
         "check_data_json" : null,
         "sig_id" : "fe784a227da3a8b19b6b01a6dbbe257ed59ff2e73fd3050275479a1eee4d563d0f",
         "sig_status" : 2,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n\nowFtk3tsFUUUh++9WIOVRxtIACHULqKtXO3M7s7OTlUijyKEVikCEghcZmdn2+3j\n7u3e29qmtIaHhndRQMRAQRSkanmUEKmoQVKiFCOvlvqgCVABCUIK1JQoXpytaGzi\nzh87mXPmd875zpm1/fv4An7qS10SqG7Z5z9+1fDNLB/XWCkZjlkhZVZKrMjm4Zi3\nC9NiLmVKhbzCoFH+tO08+io3pKBUxt2o7YSFyVIhxaZmGQbChowMw2JYh1RViWYy\njrBsYcx1jUtVQU/GE7XscB53I67txZAYNZG4pBuyQmQDQBnJEFMZahAAGeoKkDE3\nVQOKqPlONNYrG6lHM2Sb4vR//At7DACKZTDMCWYQaBgB3YAWVzDhGGGFA0J0iiDT\nEDJNEyOoAUsFmFhUxzJgGqBcBVTIld6PQ7jGhadiWljHKuKMqYaCEZIJMiDxHKPc\nvU8uEuZFUR72qo+5lBV69YvkbRb1drYZ+pckDEpFNBoTxYRYPhWExJEqYwVACJRe\nmoUVRTziKfbkwwAihHNN0xWD61BRqMUAlRVV8Ed6Tz73uXuoerOXiayLi4djBFiI\naJAR4HVL/AlQoMKAxhVIVJ2SXjRVg+lMFVcMRpBmYQ5UwCg1DUAthAhAmDIDcoub\nBHpttahmUl3RVaQCBTFT0BTZu7zYifFQxHUcS9CYO89jVBHxKvwb1X/GDAobi9nF\nPVAUhahivnBQ4uUR2+Uh2/NAWNOB+IJSxOVlXnFAU4jokEUJBSoxTQQFKm4omuBi\n6DrUoSkzgk0daUxXuRggSnQuy0gDXv0ifpSXhB1PW2RG84Rm1M4L01ipK8Y5cXlg\nxAM+f8D3YELAez6+xIeS/nlT43YNjI893LXmycUFGdWrLmZlB/1nLnb3x2tLwhmL\n4g15K2u3z24Zf6JzzvOnf819M+lAPH6grF8pOtcPaOaNFS982/TT9Z2zT9W0uoNC\nh9yRB/k9+dTFSSkD5m6p+mVdv6xt1cuP+G8eGtB3wW+F868snfndz50/ZH+4avv6\nxQuzyVPt5+zavgWxUV3dg3J3nr8+f1rr8Wvv++vSmus3TfyUVQTr6naFJzSlPtFc\n/0HAGHwsYdrWpFDiwAA5sXlLcfbdnLQhm68uSjiX0ZLc4RQMAtOl2oV94g2fy5Nz\nLryL2l7fXZJeptSfqfvDmTu8eP8nwwu/WTE0OoUkZd3a0P797aPKSSf92W0rm4f2\nj15pKmu9tHHswFF9brx9+b1jz+3xH7nj25H945e3Jm+Z/dGdO7sT5sSdov2JHSPL\nar4etV2RrjcOOyu333zk5NbkhsEtb6VtajXzu5bMnDAtv2Nv2xg5vXhWStvv5Xva\n3MZN6UfnzbjQXDmm85mPo68NcapZHMZvUXXKAbqXpozImPRnW+XukmW5D69+Y1hV\nqb739hcLcOOaF7uO3p264/ZCWDm8oH7e5szH52jdo8dnplZZG7tBw8vXUpKc4+NS\na6Zfyqf7Oler6VY06eDdS41fPXYva2Ly1PVmy5nFtbM+S+s8XZ6z7OxLtTdHdLyT\nqVbnHsxqguvaazacX3N5KdOk0Yc3+BtnSOte+Qs=\n=Zdv2\n-----END PGP MESSAGE-----",
         "retry_count" : null,
         "ctime" : 1433944497,
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"f41a7d6fbb57b25bbfc781a4496dce572f77e86e\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ed4b1\",\"kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"track\":{\"basics\":{\"id_version\":1,\"last_id_change\":1427301103,\"username\":\"kylep\"},\"id\":\"c0599ee6683be8133afc0a2346dc5819\",\"key\":{\"key_fingerprint\":\"292866890f5961c9077e81c990313c06e31948a9\",\"kid\":\"01014bc8c40f5bc956f7e040caadb0af559057acb1efed91b239fa6da838454035cd0a\"},\"remote_proofs\":[]},\"type\":\"track\",\"version\":1},\"ctime\":1433944497,\"expire_in\":157680000,\"prev\":\"20639787fa9a049dd51059eb36c58b88181d2c97d856c84e610a98e225609903\",\"seqno\":15,\"tag\":\"signature\"}\n",
         "hard_fail_count" : null,
         "human_url" : null,
         "api_url" : null,
         "rtime" : null,
         "proof_status" : null,
         "proof_id" : null,
         "last_success" : null,
         "remote_id" : null,
         "seqno" : 15,
         "sig_id_short" : "_nhKIn2jqLGbawGm274lftWf8uc_0wUCdUea",
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "version" : null,
         "etime" : 1591624497,
         "proof_text_check" : null,
         "last_check" : null,
         "sig_type" : 3
      },
      {
         "version" : null,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "sig_type" : 3,
         "last_check" : null,
         "etime" : 1594685114,
         "proof_text_check" : null,
         "proof_status" : null,
         "rtime" : null,
         "seqno" : 16,
         "remote_id" : null,
         "sig_id_short" : "NYRmj50eVzf7CV7c5YV969N_DnY8UHHyY36e",
         "last_success" : null,
         "proof_id" : null,
         "ctime" : 1437005114,
         "retry_count" : null,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n\nowGtVGtsFUUY7QOIFqFEAggahBu0iIXMzM5jBxWiBQoI0Sj4quRmZnbm3qXtvbf3\n3gotVEREwBYCKCoEgtFaJEhrlYRHSIGAJoCAoEgIpAER0Ko0+AJBwdlSwSb8ZPfP\n7s7s+c453/lmcbfsjKxMkTFwTtbL3zRm7m2RGZPr/iiYEZJxryI0fEZIlfg6lg6e\nYqJUh4aHinWFFCk9zI8PmKZlKD/0kk6m/HjMLhkMBfOokZIwiYiURjEXCow59ZQm\nDBnGtEt1qCo/gAlAjR+L6GQi6Qc1Qkp4xP7kSuRwJAFEBEEmEKQQAARdByCmPSyh\nrRqNp9Id2ITaMMO+Z7/eZH9x2wKA9paKac4UBJQR4EpotMO4ZoQ5GnDuCgIVJcTz\nPEYgBQYDxo1wGQKKAqExEBauvL0O11TbnY5nmMsw0Uph6TBCECcS8mBjSifbnUvE\ndElKxwL16aRQxYF+S95XqeDJ98LXnST5oRKRSlsxYRUV1qHQcIgd6gCHE9YBMyXK\nk0JGi8tLRRJCGGC3MSMO5wIpRCDQmFHHRcxKwFBhSBFuY9begcC0jl3QSAForAxr\njXIIkggS4SLHGI94GmmNqYSORzq4yg1C2Hi218y+SUUZNMSTmgIPWXMwJi4Fythk\nuAJAyLFNiVBAu4YJ7khiXbXck7o0ntbhRDIeN9aVIpu/tF96Q75DQX5IlSeTtiq2\nJmssiWRQu5IBghExtrnIQVwqFdRWUnBKXcEJcLRWEGkOCRNYeJoYS1+3gxPucPca\neCKpXwqizDAVjvKIYExhGybsGMe+EGONEFYhsmBQWWcR544nHOgqKA2FyDpmROBv\nu5bA3zY9bdMU1ao47Im0CE9NBZ2+PlYRPx0tl6GbNdcGvSogZjHC6YqEXXHyQ6m0\nSAfU7UpKl8Xi1z76kWsDYE2VUjMbAE2lp7hV4lEgXSqMNBw5QDHEkVVHmSULDHEg\ndQlnLt3eZFytjAMCfwK8axVR1ZS2QuG08EsC2glRURIXXjgqUtFb04xbKuO6iKpg\n2toktA/d/w6swLwb+WIAEAixTcX0hJ/UYT/YQRh1gb2uB0MJ4CIbAUJtOKCdKo8g\njDi0A48VkEhxyR1sTw+EbVgAF1g5hgvLDxOuEbwhFFLLTESCPvuRmEiXJ+3BmLMg\n655OGZlZGV06ZwUHcUbO7T3+O50vx3KvFrGiY/iDrvldijIP7xmacXiV/mxN3vQe\n79916L6lS9/bUrWMf7n5l0lNz50dPWjk0vrfuk1cPiLXnfxdtx3hv5t2NWdcWDZj\n5e7SYT0nFL7C/tS/P9RrSendz8ZP9RjasLhT3tnuy4fsO549djvZcluuWLOiT4/K\nwfDKws2PNN/7Zk3XcbUrH/hjWmRcr0Vry65MGl/w0Nj9l85/NOcd2i+6be2DU+qa\n9cmybk9tM++OKHqiH/p04ZjcracreeKJH3Y0xJ6+WD5mzTPVOefWL1h158RNc1bP\nc45sPfhaXfMlv3Dkwtfdr44Xjb2zdebuOxLNK1+thqc39GvE9TtPDkqdqa5eO6p5\n0t66vbMadtfW9GqtKtpc+8GjsdrHfq48tOvs89PJxrFg1+Heo4bU54hRi9be33jg\nRBecBPjg5D1LX9g4c8tPs0yqpNePqV+XfLJkY+XoFZ0KDrVc2Lw/u/PRlvEDjq+o\nGnL8+y8q777cBF6cnvfPkZFnsme/tTpeNrDrpqmXxnfe/2TD1MZZ9XMf3z7i7eJ9\nQ3NraLUqW/drvGdM1FftmMD6n74jkne19xv+Ow9X7PzLNM4fOW3q/s+/LmwZcmrT\nhyfWDVrvby2cueVc5NkNYXTxfLT16raK9WOSQ49G+O1i/vqCQvL5sbkF3Vdd7j97\neYqebAZTcgYfaGlNbOi78urlKzUV3+ZuejzPax2++Is+D2TtOv3xwXlNff8F\n=ep5c\n-----END PGP MESSAGE-----",
         "api_url" : null,
         "human_url" : null,
         "hard_fail_count" : null,
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"f41a7d6fbb57b25bbfc781a4496dce572f77e86e\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ed4b1\",\"kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"track\":{\"basics\":{\"id_version\":5,\"last_id_change\":1436303957,\"username\":\"saurabhkumar111\"},\"id\":\"5399a2c2510e4763827a8741c4162419\",\"key\":{\"key_fingerprint\":\"e2c01fb37b1fc352b215a823ffd5de2ee46b13d5\",\"kid\":\"01019f224fd4497101bc671f5dbe60d2df7445860cfb578a01194fbbac0e8f7a93b50a\"},\"remote_proofs\":[{\"ctime\":1436303360,\"curr\":\"4529e4b5b71e8b705425f0bc2329bccdbe6cba9668a9503eec12e9157a4ade5f\",\"etime\":1593983360,\"prev\":\"f746a3cd5a77c402143f35a75f823a101212e1c7632993da318c1bf612c35fa9\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"saurabhst\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"f7abbe72c2e6bdc93cdd60b86afbf9230c7292d5a67fa90f5316859786f8ecf30f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"4529e4b5b71e8b705425f0bc2329bccdbe6cba9668a9503eec12e9157a4ade5f\",\"seqno\":3,\"sig_id\":\"f7abbe72c2e6bdc93cdd60b86afbf9230c7292d5a67fa90f5316859786f8ecf30f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1437005114,\"expire_in\":157680000,\"prev\":\"ca08223a56cd5127ad5242912954c0b2c9b9345dd2475f09a4c3f9a7fa459e21\",\"seqno\":16,\"tag\":\"signature\"}\n",
         "payload_hash" : "c3e489b75fa8b188eabd5df420f035865c46fb1dbe0530dcc030852713e5d8e8",
         "proof_state" : null,
         "proof_text_full" : null,
         "proof_type" : null,
         "prev" : "ca08223a56cd5127ad5242912954c0b2c9b9345dd2475f09a4c3f9a7fa459e21",
         "sig_status" : 2,
         "sig_id" : "3584668f9d1e5737fb095edce5857debd37f0e763c5071f2637e9ea17346641d0f",
         "check_data_json" : null
      },
      {
         "version" : null,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "sig_type" : 3,
         "last_check" : null,
         "etime" : 1597789923,
         "proof_text_check" : null,
         "proof_status" : null,
         "rtime" : null,
         "remote_id" : null,
         "sig_id_short" : "mp6xMZJeb-81P6sM9yfza0_yNkfMuqCffDvt",
         "seqno" : 17,
         "last_success" : null,
         "proof_id" : null,
         "ctime" : 1440109923,
         "retry_count" : null,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n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\ntbfva/rhDJl8e/XSSet+96v7/wU=\n=jZ2O\n-----END PGP MESSAGE-----",
         "api_url" : null,
         "human_url" : null,
         "hard_fail_count" : null,
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"7e4268c6deb18d6be4c21dc754a4829f1d330d7d\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ed4b1\",\"kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"track\":{\"basics\":{\"id_version\":12,\"last_id_change\":1430138479,\"username\":\"mckeed\"},\"id\":\"163eb265e77553004e7338d3ff9bf500\",\"key\":{\"key_fingerprint\":\"c586829056edae143c0104ed41e1cc39aa4ed6b6\",\"kid\":\"010147bcd3247660b62c8c01e2631b752145529286c5c59c5b26b104354035c022f50a\"},\"remote_proofs\":[{\"ctime\":1394554558,\"curr\":\"bd41453399c1dd139c606af52b93430e71aab63ebc844d3e7cb5b41def58556f\",\"etime\":1552234558,\"prev\":\"57434e9ad09c87a93be1d7a172340ee3148bb3417eec0bdbb4e4543aae93d3b1\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"mckeed\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"923431cb5273319e8c041dd80f361ba279de8c41cdaa828c35c5490770b3dda30f\",\"sig_type\":2},{\"ctime\":1394554650,\"curr\":\"abf632f6a31979612ad890939b7a538f485a339c81f0f779cb1a9733b1154e95\",\"etime\":1552234650,\"prev\":\"bd41453399c1dd139c606af52b93430e71aab63ebc844d3e7cb5b41def58556f\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"mckeed\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"fc537ebbb42a3edf5cf6299c1e5e2b602737d16f6b30836984e3f8ebe4641e6b0f\",\"sig_type\":2},{\"ctime\":1417557930,\"curr\":\"a8aaff1f33fcc870e8e76a177fd64b3261cb615dc82d82277de244f94e1ba436\",\"etime\":1575237930,\"prev\":\"48eaa22988cd2ae4abb7a4fb5044765e1b99d76ea94358cec429bdbdb914001f\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"mckeed\"},\"proof_type\":5,\"state\":1},\"seqno\":9,\"sig_id\":\"96ad40d89f0ca790778f8f029897599d8cc50f4f944a1e5ad0b204044ce22fa30f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"c721af97eed78f66261150da370b8cd1de0d35b3fa13364abf436885c8d2576a\",\"seqno\":14,\"sig_id\":\"95146fe075b8d26eaebfe3a50d2667d4cb12dd94bd17208fd0f333e035165b6f0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1440109923,\"expire_in\":157680000,\"prev\":\"c3e489b75fa8b188eabd5df420f035865c46fb1dbe0530dcc030852713e5d8e8\",\"seqno\":17,\"tag\":\"signature\"}\n",
         "payload_hash" : "bb4b8cd1484f7ed9f45af965b4f27d16602a6768f1a99f60a5a744d0d5da4d6b",
         "proof_state" : null,
         "proof_text_full" : null,
         "prev" : "c3e489b75fa8b188eabd5df420f035865c46fb1dbe0530dcc030852713e5d8e8",
         "proof_type" : null,
         "sig_status" : 2,
         "check_data_json" : null,
         "sig_id" : "9a9eb131925e6fef353fab0cf727f36b4ff23647ccbaa09f7c3bed60970cd2be0f"
      },
      {
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"7e4268c6deb18d6be4c21dc754a4829f1d330d7d\"},\"key\":{\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ed4b1\",\"kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"track\":{\"basics\":{\"id_version\":5,\"last_id_change\":1428084461,\"username\":\"mortent\"},\"id\":\"653c37f0fe5638cb0ff5a1292dde2419\",\"key\":{\"key_fingerprint\":\"090f42d6f824bf1308249176f24e19442738197a\",\"kid\":\"0101f176403437825378fa2641d112d1f19fbb27fe2ddd67e96a87ba3857c6401d100a\"},\"remote_proofs\":[],\"seq_tail\":{\"payload_hash\":\"7e2a50e4f5307c7d3020a42c4c7637628e5ced840c85f557d82fff4abb196afe\",\"seqno\":10,\"sig_id\":\"82f5d43d24f14ddc36c3b9ef86393ec15137fc9372e29433e90968dfdbc9691e0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1440625437,\"expire_in\":157680000,\"prev\":\"bb4b8cd1484f7ed9f45af965b4f27d16602a6768f1a99f60a5a744d0d5da4d6b\",\"seqno\":18,\"tag\":\"signature\"}\n",
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "human_url" : null,
         "hard_fail_count" : null,
         "api_url" : null,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n\nowFtVAtsFEUYPgpVqEpJhPBGXKAoLXVmdmZnF4IxolIwCEiKFtBzdh7cpu3d9e7a\n2tQiFKm8AhIoIKYKBAgUqFQQAwjRAlFQCggFH1BCKAcUGyNPI4LOFTQ0cTfZnez/\nzzff/33/vx881t6X1I75npyVNP1EbbvvLru+7EZzSKnhhkSJMazU4HmeDMYSqyDL\nl8YwI1eWuCwqM71Q/2LpGhlGkYxEvVBQh6jEyLK5JaQLbWG5EnMEBacEM2wjR0Fh\nmkBQYZRlJGASoMoLTpORcMRLnGFwJoiruO0i00EugIggSBmCFgQAQdsEiEqBXahP\nDYSisTZsjFZMvyf01//Jz20NAKhvl1PpUA6BRQmwXaikSR1JCTUlcBybEcgtQoQQ\nlEALKAyoo5hNEeAWYBIDpuEK75/jSEvqTFMoalNMJOfYNSkhyCEudBKJURm5r1w4\nKPOiMpioPhZhPDdRvybv8Whi5Qn/f0qSDCOPRWO6GD8PMK2QMQxiZAMbYwu2wcwP\nRWIJfzTmPUbE5CZVQElimTZ3gVKEQeQgISTCrYzuK58Qq636wNHFfp2JhKVshF0F\nTaDfDqSWQlhCB2NETRs6lLXRU+kEDExsUhsR/VAMWRgKCJHQMUe5LqJKagLC0rpb\nWkmXmTahXO/SaUDrqdlHZH4oJv3hSCiktB5T3sgworLAH2NeXoJtmJXkhZjwB1g0\n0NppiBEgsSImoJwK7TRgGHHMqWVSC9mScClsDLhNFCFU2EgphZmrTbGYkkYrejCk\ndQV66U271zc6iwhsCoQVxEJw0+Km60hlW6ZjSg4J1Npyx6RIIgebpnSAY9lCCZc7\nlgMlUEZZwt2ScMKbeyY/MCBQx3jMy2+1EwMLES1ahiHfDnsR6fcSGYRaNtBXhhGO\nyCIN4rrYtbmA2MZKN7OjMGHKsYiLFaICWhZAzNKbFGSOo3SDEkYxFkAQwbCewQcq\ntTUzNk1j6nqDLFYYkUZZytykPh187ZJ8DyUnJQbfl9Kpy79/g7MrU/+edqjh9pro\nQO/gwtrmoZuzzl1Ln5i5cPcL6SNyThgXqkct72Nfip+rKsBZtzvf/bCgrPzOyi8X\n1hU1H/tp//n0opd/Gz+aNjYWjU6enl6xdw/L3DT0s8GvL1q/btmkqYftFxctrexQ\n6rVfYDVsrN/lpVfVH112MFQ7P/DJ+dOT7F/3PfES+4ivqP8Y9p6bOqaw5W5kY9eU\nMVs7FPrzqzZtL2ufVXN9S8rlI5VDkhdcve6zR2xvvmIceGTJO+MOV4yILB5bfOtk\neqrvzKzuS9maOXldBv34Xk2PV6/122DcOnpyPNxzZNbDndaOdWSPPTNOh+fP7rMn\nt5YOmtdE0pq3FeccNvmhqU0Lj3erbJjXtznrlRvG2ed3DpjzTcXWE4OTdgxIr4kv\nPl5+y1+yYmWsY/e06mf37a25Oz+1x5U7o+tv9q4b3n/n5LQzw6+9dj39/dVXrz5+\nbPWQyTlX+67MiTdsaVHd1g6MN44Bg0cvx0VDp+Rvu2gP/Lk8JdgxsL62pf/Ngqq0\nyl4/3CnPXu8fUX0qUF3zVuXS5/ZlXIwHCtZd2nxha97ZpGeSG0Ytmb3tzabbR/48\nOmHmyC09r9T1e2jOyM4tf+1c/O2NwOqnx/bpOanCK91f/MfnR0vmz24pLv3q1K6m\nruPje2f0yh7n5UQPbICFBCT/Puf7d29bqycu5/GnMi90z94Zv/nLzFW7d3x67Ys6\nY8LWRztOOLLqHw==\n=sIrB\n-----END PGP MESSAGE-----",
         "retry_count" : null,
         "ctime" : 1440625437,
         "sig_id" : "47efdd7ab54c45384c5802d7e2a2e46d6cd2faaaad5da5ac1d19502fda8e8ddb0f",
         "check_data_json" : null,
         "sig_status" : 2,
         "proof_text_full" : null,
         "proof_state" : null,
         "proof_type" : null,
         "prev" : "bb4b8cd1484f7ed9f45af965b4f27d16602a6768f1a99f60a5a744d0d5da4d6b",
         "payload_hash" : "439cb3f0b4774688deb5db2fe2e92b9f411f1753e131e4e1cfb79abb904dd4b8",
         "etime" : 1598305437,
         "proof_text_check" : null,
         "last_check" : null,
         "sig_type" : 3,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "version" : null,
         "proof_id" : null,
         "last_success" : null,
         "sig_id_short" : "R-_derVMRThMWALX4qLkbWzS-qqtXaWsHRlQ",
         "remote_id" : null,
         "seqno" : 18,
         "rtime" : null,
         "proof_status" : null
      },
      {
         "sig_id" : "595a73fc649c2c8ccc1aa79384e0b3e7ab3049d8df838f75ef0edbcb5bbc42990f",
         "check_data_json" : null,
         "sig_status" : 2,
         "proof_state" : null,
         "proof_type" : null,
         "proof_text_full" : null,
         "prev" : "439cb3f0b4774688deb5db2fe2e92b9f411f1753e131e4e1cfb79abb904dd4b8",
         "payload_hash" : "09527db7672bf23a9681ac86c70826cdc01ed1e467252a76ca4bf4ad0964efd7",
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"7e4268c6deb18d6be4c21dc754a4829f1d330d7d\"},\"key\":{\"eldest_kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ed4b1\",\"kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"track\":{\"basics\":{\"id_version\":26,\"last_id_change\":1436229436,\"username\":\"acm\"},\"id\":\"62380b77e7bbd9ac3cef5b652d9ded19\",\"key\":{\"key_fingerprint\":\"d56565ce206b54d4ee29fbf33510ec4c3990d009\",\"kid\":\"0101888fd2ab6139520a27690b19e18d7cf2a811e3e235e06d186d6131077c7a680d0a\"},\"pgp_keys\":[{\"key_fingerprint\":\"d56565ce206b54d4ee29fbf33510ec4c3990d009\",\"kid\":\"0101888fd2ab6139520a27690b19e18d7cf2a811e3e235e06d186d6131077c7a680d0a\"}],\"remote_proofs\":[{\"ctime\":1427989138,\"curr\":\"2c16dd1dfea1ea5bebd47e9427bd8eb786864cd8780cf378b1a1bd24e5efc18e\",\"etime\":1585669138,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"acm1\"},\"proof_type\":3,\"state\":1},\"seqno\":1,\"sig_id\":\"938abf61f81c3a785435937ee88af64a9d47ab5533eff143c9b8034aaaab996f0f\",\"sig_type\":2},{\"ctime\":1427989268,\"curr\":\"8a4ee1d5de0914bb193118f4d71efbf67822033779dafab9a629c333d575c6d9\",\"etime\":1585669268,\"prev\":\"2c16dd1dfea1ea5bebd47e9427bd8eb786864cd8780cf378b1a1bd24e5efc18e\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"adammielke\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"8f81cc23a88f06e940f61e440a6d92cf0bba3df7e5174e422db97b61b6d0e1940f\",\"sig_type\":2},{\"ctime\":1427996287,\"curr\":\"86a300cdf8f88223249abdfe4318080125333f1c2359eb4c86c48f251428b3b0\",\"etime\":1585676287,\"prev\":\"8a4ee1d5de0914bb193118f4d71efbf67822033779dafab9a629c333d575c6d9\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"mielke.me\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":3,\"sig_id\":\"790fcabb2b9e0707f0b2cabf1d14ef248a7ba23a736e5861528c9fd521cd52890f\",\"sig_type\":2},{\"ctime\":1428074741,\"curr\":\"c92ebcf9a09922d3afb0e60e9c37f5c461a7bd0f156de1ab96c2b4b209f7f6aa\",\"etime\":1585754741,\"prev\":\"804cb73d7ba8819fad30f52823e06a43aa62d20b6caee22b094fcf81664aaf58\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"acm\"},\"proof_type\":5,\"state\":1},\"seqno\":13,\"sig_id\":\"99e3ad4a26cd115b5276066575177350fcf2a66ccd9552882f7d501e348ff3590f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"4b226b25ddbb7049aa75c47430340685a0631389850ed8f56c22e0fef8d37262\",\"seqno\":28,\"sig_id\":\"32e0d924dad237ce3c61c5587f6250c2c9e24d0e97d3cd8c583dd80e68b0f5520f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1442334643,\"expire_in\":157680000,\"prev\":\"439cb3f0b4774688deb5db2fe2e92b9f411f1753e131e4e1cfb79abb904dd4b8\",\"seqno\":19,\"tag\":\"signature\"}",
         "human_url" : null,
         "hard_fail_count" : null,
         "api_url" : null,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n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\n+MaKCZ07DzRPHVWKxbNb/ws=\n=z6bl\n-----END PGP MESSAGE-----",
         "retry_count" : null,
         "ctime" : 1442334643,
         "proof_id" : null,
         "last_success" : null,
         "sig_id_short" : "WVpz_GScLIzMGqeThOCz56swSdjfg4917w7b",
         "remote_id" : null,
         "seqno" : 19,
         "rtime" : null,
         "proof_status" : null,
         "etime" : 1600014643,
         "proof_text_check" : null,
         "last_check" : null,
         "sig_type" : 3,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "version" : null
      },
      {
         "ctime" : 1442334689,
         "retry_count" : null,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n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\n2/b1ouGzTnrr3PYHnlx57YGOve9++uDSX7Z3NLRdU5y7+veJQn/V+4I/AQ==\n=7yZE\n-----END PGP MESSAGE-----",
         "api_url" : null,
         "human_url" : null,
         "hard_fail_count" : null,
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "payload_json" : "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"7e4268c6deb18d6be4c21dc754a4829f1d330d7d\"},\"key\":{\"eldest_kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ed4b1\",\"kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"track\":{\"basics\":{\"id_version\":3,\"last_id_change\":1427123965,\"username\":\"robzienert\"},\"id\":\"12c11fff68bbe4a003a7c555f085b500\",\"key\":{\"key_fingerprint\":\"fa97622f2ea25d8d769eb3669a8662285c63e633\",\"kid\":\"0101b8069d1308440059ad0956461cd06b02f06472603f1441b689e1764c8cf53e130a\"},\"pgp_keys\":[{\"key_fingerprint\":\"fa97622f2ea25d8d769eb3669a8662285c63e633\",\"kid\":\"0101b8069d1308440059ad0956461cd06b02f06472603f1441b689e1764c8cf53e130a\"}],\"remote_proofs\":[{\"ctime\":1427123926,\"curr\":\"bf8f8cfd53a2a1026bd4a135fc32ef526c0a7eecc99f22b0a3f787fbfcddaa20\",\"etime\":1584803926,\"prev\":\"22c36f2e8d84480985cb34112876bfabf861549e24189e098881ac5bc668c805\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"robzienert\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"883dcf2a674b02a3ad331b85929ae6c49ab6d00727268230558f692a701bf2490f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"b52578ea4b0294e6cca324e7b9cdbfe96b9a4025e73ba21eb878c28e1f0cb475\",\"seqno\":32,\"sig_id\":\"2ccf32d96e6896849066d67f9b699b8ef697809dd3f496f79c32ed3711f2e3d30f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1442334689,\"expire_in\":157680000,\"prev\":\"09527db7672bf23a9681ac86c70826cdc01ed1e467252a76ca4bf4ad0964efd7\",\"seqno\":20,\"tag\":\"signature\"}",
         "payload_hash" : "f1509495f4f1d46e43dcdd341156b975f7ad19aefeb250a80fd2b236c517a891",
         "proof_text_full" : null,
         "proof_state" : null,
         "prev" : "09527db7672bf23a9681ac86c70826cdc01ed1e467252a76ca4bf4ad0964efd7",
         "proof_type" : null,
         "sig_status" : 2,
         "check_data_json" : null,
         "sig_id" : "e256078702afd7a15a24681259935b48342a49840ab6a90291b300961669790f0f",
         "version" : null,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "sig_type" : 3,
         "last_check" : null,
         "proof_text_check" : null,
         "etime" : 1600014689,
         "proof_status" : null,
         "rtime" : null,
         "seqno" : 20,
         "remote_id" : null,
         "sig_id_short" : "4lYHhwKv16FaJGgSWZNbSDQqSYQKtqkCkbMA",
         "last_success" : null,
         "proof_id" : null
      },
      {
         "ctime" : 1442351697,
         "retry_count" : 1,
         "sig" : "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v1\n\nowGtUm9QFGUYPw64QYIyC/oQ0LAkcw0ou3u7+94CWQMBwagpBSFU57u77x4LuHfe\nLkcCEsQE6EwIYaWiYuhRkA46KYP/TqgkCEH+lDQNYTVh0fSHQcVzPLM9h740fex9\nP7zzPO/v93t/7/M8TaH+Or0f1EXX6qu+OuE3PMfpcjw7N1VgnE3YhiVWYMXo/oFK\nBKSolmJJwBIxnNA2xwPEAp7AGUDjZo4QkQmwCNDAhHCWNUOa4BmaFgQB0ASDixQO\nWBGaAYnzDA4RhUMsHhMl2Yocdockq5osDwWaE3kzR5pYksMJkiYJAEmCIXCcJMwm\nnARIoDhCIxbaFB9DM8dBBa2WbFpOCyz37f0H/n/2Xbr0DosYpCFNggjMgKIRz1Oc\nCdA0ydIcwfqACnLIcAvS0HYZlShIxrbHY1rSKfHIV9elS7VMUlXk+BcDFhaqPry6\nze6LyxBnWaJaOEkWtOJpBCdyKJJNxhIJDcmrko9KUBRp0uyzIB5Dr9slB7JIPgQN\nGDOurXjM7kBOTVIkaJylWFqkREKgGESZ1gu89iGKIGiGYwEtAigQLEQi4kgah2Zc\nFEitPwxPa60xs77aKmirbMMSSUIzCq2aqCJZZaiWOhC2PXiHPiJA56fXGQL1vrnS\nBS9b/s+w3Sx/6F5Kw6waWlPkWdxcb/zRZYzp9lratn4wlRbrfy78POmtvNi/K3yX\noTwuMm9xw+17QS2ZT6jphQ/GpUx3XgslH4uwnjx384uEgLDql1LLiyNGOvPgLed4\n1/UCd1bMb017nJeSpqKy0suu9gdGvDJ4fSj0cVeufvL4jbBBd3Xe8KWOvucP3jGs\nNMLA5E2jYxOpnZE3vj8231VjW3etN7/nZfepkNhu/wA7duutP2+/V9enUGMZXb8X\n71ba6AecHxYX7HHlD72pvzt0xEKavBtfOMosT8phKt4+HnS5Hf+kbcx9YU2um0zr\nabyyG8sIAmFJI135uY8eyPupJGX/weiSv+obYWHzgGGV6wf/lU+WvpqedIFce0wZ\nv5rTevrAQkvaobV7PezoUx8VZbfPDZTDF7Ga7+LkMwPlq6oqB8xNC4aWkdH3x2d7\nfm09bU9o0P1sbHxtXeu33rMZpxIPzbDJy54+2y9/nTpXO1FfeXLNc64K2Hj4nazm\n0c9nBeNsZtFwVOyINcXzcdPU5GpXeET1leYtlbihIGpexDzZ3uk3EoJnxmDHxNj8\nzKDzxIqqi+frn/3lkaP79mKXQ3ptIZn5oK+9vvaZuQ0LZf3vflMTmRA8Xde4oju/\nJ3njZmN2ZO/OZPf+lsO6jtH+9SPpk9LknY6U2roK8GlM+2fRX54ZWljcd8Re9fAf\nDd67YX8D\n=92PB\n-----END PGP MESSAGE-----",
         "api_url" : null,
         "hard_fail_count" : 40,
         "human_url" : null,
         "fingerprint" : "cad5bfc8b2392b0125217a2161002183027ed4b1",
         "payload_json" : "{\"body\":{\"key\":{\"eldest_kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"fingerprint\":\"cad5bfc8b2392b0125217a2161002183027ed4b1\",\"host\":\"keybase.io\",\"key_id\":\"61002183027ed4b1\",\"kid\":\"01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a\",\"uid\":\"619e6eddd3df78745ecc4b3755295b19\",\"username\":\"pnelsen\"},\"service\":{\"name\":\"twitter\",\"username\":\"pahht\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1442351697,\"expire_in\":157680000,\"prev\":\"f1509495f4f1d46e43dcdd341156b975f7ad19aefeb250a80fd2b236c517a891\",\"seqno\":21,\"tag\":\"signature\"}",
         "payload_hash" : "605525686fef18180be692df6106c13dae39abb2799dc9e8bed1e2bb64e9b886",
         "proof_state" : 8,
         "proof_text_full" : "Verifying myself: I am pnelsen on Keybase.io. nBidbWRLrZWW94UZ2HCmhWJPgTr8HQ5JFVBz / https://keybase.io/pnelsen/sigs/nBidbWRLrZWW94UZ2HCmhWJPgTr8HQ5JFVBz\n",
         "proof_type" : 2,
         "prev" : "f1509495f4f1d46e43dcdd341156b975f7ad19aefeb250a80fd2b236c517a891",
         "sig_status" : 2,
         "check_data_json" : "{\"name\":\"twitter\",\"username\":\"pahht\"}",
         "sig_id" : "9c189d6d644bad9596f78519d870a685624f813afc1d0e49155073d3b0521f970f",
         "version" : 2,
         "kid" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a",
         "sig_type" : 2,
         "last_check" : 1442354273,
         "proof_text_check" : "Verifying myself: I am pnelsen on Keybase.io. nBidbWRLrZWW94UZ2HCmhWJPgTr8HQ5JFVBz /",
         "etime" : 1600031697,
         "proof_status" : 101,
         "rtime" : null,
         "remote_id" : null,
         "seqno" : 21,
         "sig_id_short" : "nBidbWRLrZWW94UZ2HCmhWJPgTr8HQ5JFVBz",
         "last_success" : null,
         "proof_id" : "d545d23630baa7495d195b10"
      }
   ],
   "keys" : [
      "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v1\n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\ngh72ne/Z9JIkqdSx1ujahthd\n=mbnu\n-----END PGP PUBLIC KEY BLOCK-----",
      "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v1\n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\ngh72ne/Z9JIkqdSx1ujahthd\n=mbnu\n-----END PGP PUBLIC KEY BLOCK-----",
      "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v1\n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\ngh72ne/Z9JIkqdSx1ujahthd\n=mbnu\n-----END PGP PUBLIC KEY BLOCK-----",
      "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v1\n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\ngh72ne/Z9JIkqdSx1ujahthd\n=mbnu\n-----END PGP PUBLIC KEY BLOCK-----",
      "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v1\n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\ngh72ne/Z9JIkqdSx1ujahthd\n=mbnu\n-----END PGP PUBLIC KEY BLOCK-----",
      "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v1\n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\nGNwgYjHG20par4J7G6kyQg4=\n=tNk+\n-----END PGP PUBLIC KEY BLOCK-----",
      "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v1\n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\nNBblvCY6pxirPBFTEDOv494v\n=+7cU\n-----END PGP PUBLIC KEY BLOCK-----"
   ],
   "label_kids" : {
      "eldest" : "01010bc7e97c1067508b1fe379e7573e0998a51c655ddd75160f4079fa8720c60ae40a"
   }
}

`,
	"akalin_chain.json": `{
    "username": "akalin",
    "uid": "ebbe1d99410ab70123262cf8dfc87900",
    "chain": [
        {
            "seqno": 1,
            "payload_hash": "738bebce2b4ed88c9319075bfed58779734efd03f3ee5386b15b38b61d2024f5",
            "sig_id": "5520adf71d91197d52cdb69d45fb31250d633bf30c5c07c4c4644bac263953860f",
            "sig_id_short": "VSCt9x2RGX1SzbadRfsxJQ1jO_MMXAfExGRL",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGbwMvMwMW4vyE6ITA1uJDx9IHKJIYQpocXqpWS8lMqlayqlbJTwVRaZl56alFB\nUWZeiZKVkoF5UpqhuXGySWKikYGZSZKFhYWRRSKQkWZhYJpkZmBqaGZqamxuqKSj\nlJFfDNIBNCYpsThVLzMfKAbkxGemAEWxqC8FS6QmJaUaplhamhgaJCaZGxgaGRuZ\nGSWnWaSkJVuYWxoYgBQWpxblJeamAlUnZifmZOYp1eoolVQWgATKU5PigdJlmcmp\n8UmZeSlAxwN1lKUWFWfm5ylZGQJVJpdkgvQamhhYmpibGhgY6iilVhRkFqXGZ4JU\nmJqbWRgAgY5SQVFqmZJVXmlOjo5ScWphXj5QFmhRYjrQnuLM9LzEktKiVKVark5G\nGRYGRi4GNlYmUAgycHEKwMJ16XwOho0zP+3Jav7+8uPF2J0Hjui4929Ze+rz35fx\n6+5kG8S0ijiKFTJZ/nE9ZqZbMDthy+GkreJLb8+vfDWvZvujxXl50f9rSn7eEfm1\nK16L4/dxvxyz+gviXNd6H7Nv2nLz/zIdiQ4Z+TKOH2sqnG0ztxwvFZufyrXsq4ES\na/jah+1y6/0zLz1+q5LSPz+ttIstxS/GgbvWm+HjG5GCP7PtG/u3TS2/L/tIfkb7\ne5kv7R7VAlvqd+nsKLT/18b76eZ0v6+HsyZM/nRcbdPJxplvH067sP3hteOLjtTG\nM5To1/AxbuZPODZHWka4vVZn14JuQXd2s415K7etE61eUbbkziMtNpZudl+lsD6R\nm/pbRAE=\n=M4q7\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1409475001,\"expire_in\":157680000,\"prev\":null,\"seqno\":1,\"tag\":\"signature\"}\n",
            "sig_type": 1,
            "ctime": 1409475001,
            "etime": 1567155001,
            "rtime": null,
            "sig_status": 0,
            "prev": null,
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 2,
            "payload_hash": "d8cd14f0d6da7dad0b0227b024ecb99c920e6110ef9486262be08df3b6431e00",
            "sig_id": "1a67183170ef64b00e805c74d39c5286a6a9b50939b3604b15f0722aae27500b0f",
            "sig_id_short": "GmcYMXDvZLAOgFx005xShqaptQk5s2BLFfBy",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowFtkVtIFFEYx2c1L60amUqYlnisINzqzH1mkR6iQNAX00hU2ObsnNkmdXadnfWC\nWCsKVkYaET50AS3IlITyApbtmj4oKpYvkmCCRVKQgmHRQ1IzYkTQeTnn/L//78/3\n8bUnRBJ2WzhYer4AF1bZpkc/I6IoYnmtHiCvXAec9aAcb12Kqnmw7tNVzQBOAHmk\nkDztZiSJghyDBEGgBMl8KAJkEQdZkmNZmieBA1zw+i3CjEGSHx9TvaZmflyqbKr/\n8Qe2ChghTMqiyJBQQjwkKZriKLciyIpb4EUILaMf65pUiU23VC5VqBpocABTq1bd\n2Op4u2bUqIaB9X8B5S9h1PkspQYj1zbsQqomm+OaSDXW/apXA07SdLoN1YJJBooM\nz0KGdwBc61N17FItB8tzAjSPA/h0XG1G8rSAMHJjCjFYFgS3SJMi5FmkYJkVeF7k\naQYrMqQVGmOWFjhEsshEOFKmIMUoLLDGqdK8wEmZbUoeM9KvejTJCOgYNNiv2tJ3\nEDY7ER0VYW2MsO/c/WeP52ZiieCNRaGzvyzubk1ld9JqV2nYNSgFUxxOR8LZguTa\n2cBgRWk4/smePP5STurM2MjYtfnRlbGObykv+ovyN+WN4+OpE7nqUPSvzfX7R+bf\nvTfmZHyireXiQHE8jHkcUEf6Jk6eTu972zT6ZnL6yvL+jAetMaEfzd/XSzpXpm7d\nHkg6nLb2LPdmo5IFuyKDiwuPpk7t6m5JyQwtaaGfZU83szu4ucZXxa9XssJdzw/k\n9fa2z8U9LPzYFrqux36Pa/Keac4GU9EHiXHi5cbC0dbikp7ZRE/6XnuLlPgh6jJO\n/JrZU9/6KaPq3vA+fvKOkbw+9GWYS14yDhWm+bnVnPzZ3w==\n=qYLb\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"service\":{\"name\":\"twitter\",\"username\":\"fakalin\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1409475047,\"expire_in\":157680000,\"prev\":\"738bebce2b4ed88c9319075bfed58779734efd03f3ee5386b15b38b61d2024f5\",\"seqno\":2,\"tag\":\"signature\"}\n",
            "sig_type": 2,
            "ctime": 1409475047,
            "etime": 1567155047,
            "rtime": null,
            "sig_status": 0,
            "prev": "738bebce2b4ed88c9319075bfed58779734efd03f3ee5386b15b38b61d2024f5",
            "proof_id": "a511b030f44e30d1aa00c410",
            "proof_type": 2,
            "proof_text_check": "Verifying myself: I am akalin on Keybase.io. GmcYMXDvZLAOgFx005xShqaptQk5s2BLFfBy /",
            "proof_text_full": "Verifying myself: I am akalin on Keybase.io. GmcYMXDvZLAOgFx005xShqaptQk5s2BLFfBy / https://keybase.io/akalin/sigs/GmcYMXDvZLAOgFx005xShqaptQk5s2BLFfBy\n",
            "check_data_json": "{\"name\":\"twitter\",\"username\":\"fakalin\"}",
            "remote_id": "506001226417180673",
            "api_url": "https://twitter.com/fakalin/status/506001226417180673",
            "human_url": "https://twitter.com/fakalin/status/506001226417180673",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 246,
            "hard_fail_count": 0,
            "last_check": 1443796918,
            "last_success": 1443796918,
            "version": 2,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 3,
            "payload_hash": "99456cb27d7c6c8e6978be4c0928c19a07428e236fa9f7ed4d823d926f054701",
            "sig_id": "68a547eed57a11b62bfe2511fe128cc10cfb70f199453aa9cb964b97999f5d800f",
            "sig_id_short": "aKVH7tV6EbYr_iUR_hKMwQz7cPGZRTqpy5ZL",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGbwMvMwMW4vyE6ITA1uJDx9IHnSQwhTI9kq5WS8lMqlayqlbJTwVRaZl56alFB\nUWZeiZKVkoF5UpqhuXGySWKikYGZSZKFhYWRRSKQkWZhYJpkZmBqaGZqamxuqKSj\nlJFfDNIBNCYpsThVLzMfKAbkxGemAEWxqC8FS6QmJaUaplhamhgaJCaZGxgaGRuZ\nGSWnWaSkJVuYWxoYgBQWpxblJeamAlUnZifmZOYp1eooAcXKMpNTQS6GyqVnlmSU\nJuFSX1JZABIoT02Kh2qNT8rMSwF6FqijLLWoODM/T8nKEKgyuSQTpNfQxMDSxNzU\nwNJcRym1oiCzKDU+E6TC1NzMwgAIdJQKilLLgEamWCSnGJqkGaSYpSSapySmGCQZ\nGBmZAwmT1OQkS8tkSyODVDNDQ4PUNEsTCzOg75JSDYDeM04yMzE2TAX7sDi1MC9f\nycoY6MzEdKCRxZnpeYklpUWpSrVcnYwyLAyMXAxsrEyg+GLg4hSAxaLKHfb/qWGc\n7rI9Gy7yeK52VJgqyZpfkMVde5tJ4m/EkvInerq6bAfVnCtTPILfbl1zxMfgT4la\nWyTfG4feRzYpizTuPn1Sra3/pftvyveKpUoGMwv7jt869XbtaX7/lm0/BcJMD0Yb\nPl7/av3nrVcaq5jv8vMWOq5Rlrg9/8Xt31dT2gWr4w4yPc1uqJrEeHn926Df91Y9\nLDtrws3Mva7gzWk/lfkaX8t4yrVebD3swRa2+1GHRRY307UL2e3MK1ZYXGHtbYmK\nrXpt+TDGRTK4dbIT5+bj5w+lvokr+hJQ7Teta9esFjGe7kOSJyTVtt5a/NuJ70D6\nl2L2k8alve6HguZdLFmrpKPx1CB2yd/f6VMB\n=/lLk\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"service\":{\"name\":\"github\",\"username\":\"akalin\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1409475097,\"expire_in\":157680000,\"prev\":\"d8cd14f0d6da7dad0b0227b024ecb99c920e6110ef9486262be08df3b6431e00\",\"seqno\":3,\"tag\":\"signature\"}\n",
            "sig_type": 2,
            "ctime": 1409475097,
            "etime": 1567155097,
            "rtime": null,
            "sig_status": 0,
            "prev": "d8cd14f0d6da7dad0b0227b024ecb99c920e6110ef9486262be08df3b6431e00",
            "proof_id": "ef1f23491d18d10b404c9c10",
            "proof_type": 3,
            "proof_text_check": "\n\nowGbwMvMwMW4vyE6ITA1uJDx9IHnSQwhTI9kq5WS8lMqlayqlbJTwVRaZl56alFB\nUWZeiZKVkoF5UpqhuXGySWKikYGZSZKFhYWRRSKQkWZhYJpkZmBqaGZqamxuqKSj\nlJFfDNIBNCYpsThVLzMfKAbkxGemAEWxqC8FS6QmJaUaplhamhgaJCaZGxgaGRuZ\nGSWnWaSkJVuYWxoYgBQWpxblJeamAlUnZifmZOYp1eooAcXKMpNTQS6GyqVnlmSU\nJuFSX1JZABIoT02Kh2qNT8rMSwF6FqijLLWoODM/T8nKEKgyuSQTpNfQxMDSxNzU\nwNJcRym1oiCzKDU+E6TC1NzMwgAIdJQKilLLgEamWCSnGJqkGaSYpSSapySmGCQZ\nGBmZAwmT1OQkS8tkSyODVDNDQ4PUNEsTCzOg75JSDYDeM04yMzE2TAX7sDi1MC9f\nycoY6MzEdKCRxZnpeYklpUWpSrVcnYwyLAyMXAxsrEyg+GLg4hSAxaLKHfb/qWGc\n7rI9Gy7yeK52VJgqyZpfkMVde5tJ4m/EkvInerq6bAfVnCtTPILfbl1zxMfgT4la\nWyTfG4feRzYpizTuPn1Sra3/pftvyveKpUoGMwv7jt869XbtaX7/lm0/BcJMD0Yb\nPl7/av3nrVcaq5jv8vMWOq5Rlrg9/8Xt31dT2gWr4w4yPc1uqJrEeHn926Df91Y9\nLDtrws3Mva7gzWk/lfkaX8t4yrVebD3swRa2+1GHRRY307UL2e3MK1ZYXGHtbYmK\nrXpt+TDGRTK4dbIT5+bj5w+lvokr+hJQ7Teta9esFjGe7kOSJyTVtt5a/NuJ70D6\nl2L2k8alve6HguZdLFmrpKPx1CB2yd/f6VMB\n",
            "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am akalin on github.\n  * I am akalin (https://keybase.io/akalin) on keybase.\n  * I have a public key whose fingerprint is 07BF 173C 4AA2 064B 8882  8A64 BF80 5B60 5165 5371\n\nTo claim this, I am signing this object:\n\n` + "`" + `` + "`" + `` + "`" + `json\n{\n    \"body\": {\n        \"key\": {\n            \"fingerprint\": \"07bf173c4aa2064b88828a64bf805b6051655371\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"bf805b6051655371\",\n            \"uid\": \"ebbe1d99410ab70123262cf8dfc87900\",\n            \"username\": \"akalin\"\n        },\n        \"service\": {\n            \"name\": \"github\",\n            \"username\": \"akalin\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1409475097,\n    \"expire_in\": 157680000,\n    \"prev\": \"d8cd14f0d6da7dad0b0227b024ecb99c920e6110ef9486262be08df3b6431e00\",\n    \"seqno\": 3,\n    \"tag\": \"signature\"\n}\n` + "`" + `` + "`" + `` + "`" + `\n\nwith the PGP key whose fingerprint is\n[07BF 173C 4AA2 064B 8882  8A64 BF80 5B60 5165 5371](https://keybase.io/akalin)\n(captured above as ` + "`" + `body.key.fingerprint` + "`" + `), yielding the PGP signature:\n\n` + "`" + `` + "`" + `` + "`" + `\n-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGbwMvMwMW4vyE6ITA1uJDx9IHnSQwhTI9kq5WS8lMqlayqlbJTwVRaZl56alFB\nUWZeiZKVkoF5UpqhuXGySWKikYGZSZKFhYWRRSKQkWZhYJpkZmBqaGZqamxuqKSj\nlJFfDNIBNCYpsThVLzMfKAbkxGemAEWxqC8FS6QmJaUaplhamhgaJCaZGxgaGRuZ\nGSWnWaSkJVuYWxoYgBQWpxblJeamAlUnZifmZOYp1eooAcXKMpNTQS6GyqVnlmSU\nJuFSX1JZABIoT02Kh2qNT8rMSwF6FqijLLWoODM/T8nKEKgyuSQTpNfQxMDSxNzU\nwNJcRym1oiCzKDU+E6TC1NzMwgAIdJQKilLLgEamWCSnGJqkGaSYpSSapySmGCQZ\nGBmZAwmT1OQkS8tkSyODVDNDQ4PUNEsTCzOg75JSDYDeM04yMzE2TAX7sDi1MC9f\nycoY6MzEdKCRxZnpeYklpUWpSrVcnYwyLAyMXAxsrEyg+GLg4hSAxaLKHfb/qWGc\n7rI9Gy7yeK52VJgqyZpfkMVde5tJ4m/EkvInerq6bAfVnCtTPILfbl1zxMfgT4la\nWyTfG4feRzYpizTuPn1Sra3/pftvyveKpUoGMwv7jt869XbtaX7/lm0/BcJMD0Yb\nPl7/av3nrVcaq5jv8vMWOq5Rlrg9/8Xt31dT2gWr4w4yPc1uqJrEeHn926Df91Y9\nLDtrws3Mva7gzWk/lfkaX8t4yrVebD3swRa2+1GHRRY307UL2e3MK1ZYXGHtbYmK\nrXpt+TDGRTK4dbIT5+bj5w+lvokr+hJQ7Teta9esFjGe7kOSJyTVtt5a/NuJ70D6\nl2L2k8alve6HguZdLFmrpKPx1CB2yd/f6VMB\n=/lLk\n-----END PGP MESSAGE-----\n\n` + "`" + `` + "`" + `` + "`" + `\n\nAnd finally, I am proving ownership of the github account by posting this as a gist.\n\n### My publicly-auditable identity:\n\nhttps://keybase.io/akalin\n\n### From the command line:\n\nConsider the [keybase command line program](https://keybase.io/docs/command_line).\n\n` + "`" + `` + "`" + `` + "`" + `bash\n# look me up\nkeybase id akalin\n\n# encrypt a message to me\nkeybase encrypt akalin -m 'a secret message...'\n\n# ...and more...\n` + "`" + `` + "`" + `` + "`" + `\n",
            "check_data_json": "{\"name\":\"github\",\"username\":\"akalin\"}",
            "remote_id": "cd238f392d451ea22da7",
            "api_url": "https://gist.githubusercontent.com/akalin/cd238f392d451ea22da7/raw/5adddb8173ed66e035a67cd2bce8555a7c798034/keybase.md",
            "human_url": "https://gist.github.com/cd238f392d451ea22da7",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 807,
            "hard_fail_count": 0,
            "last_check": 1443780855,
            "last_success": 1443780855,
            "version": 1,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 4,
            "payload_hash": "a2a877a4cda04a32aa02de40137d22b2a0a04f35f7f677bdcae15fbf91f4ec04",
            "sig_id": "dece17e125e06d6b44438218d29ab55a7785beaeb7bf372d8f98eb228009635c0f",
            "sig_id_short": "3s4X4SXgbWtEQ4IY0pq1WneFvq63vzctj5jr",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowF1UUtIVFEYnrEUZ3qg9KIcWpzQHox57p37nDYFtbBFEQZhNU73ce54UO+Md0Zz\nskGtREQSF5KoEUEZVgpqZUXRTGCKiUEmUTFKCmItXIT2GCrtXLFdnc055/u/7+P7\n+JvWrbLYrdHqU2eOobxS60hsVrYcT5rOrwSyXw0DdyUoQsuXhnUfMgIG1kPADSAv\naxTvUhhJoiHHyIIg0IJEHpoAWZmDLMWxrIungBMU+oOmgtjIUhDtxX6CkY8XqwT9\nB79seYBkGVGqKDIUlGQeUrSL5mhFE1RNEXgRQpMYRIYulSDCloqkYqyDiBMQrBwr\nyEy8MjOQquLQ//ihcMAEziLZuyL1ylhXSVmiKEdGEPt14KYIUwlhU0sxUGR4lmI5\nJ0AVAWwgLzYZLM8JkBwnCBionFiS7CynyDSv8gqnCIgTeUFGjAJFWlAoUYI8QwuI\ndnGaJGo8UhlVoF2qSHMaZBnSGJhlSnU/cDMkpuQjlkHs06VQmYFAxF5vday2WO2W\nlOQkc18Wuy3t7xZzalItfbY22/XtT+Ij9S/66x5/Vye5o73r38c8P5Y8sXjH+fyU\n0eZzvxrXFnfVJbVPeXrG98/mBsYHWr5cobunN1Tfe7ax4KkWmZpy1kZ21Q4NuBur\nEu+GPTVrhtoy0gOvotH2QLTl5cLXsh46O9Y/aWMeZR7RC7f8bswani55k+isT0/2\n3Rnc3emQLt3IvtsafjBxoC/c13Cy+dDtbXhnZvqOmydej+7D3qK0xMG0hapE9/zP\neJ2jcCbj4Vg4q2ux/+PWOXu8puBDMZt3nz98sSO4OJb2HHKnl77lXm2dzXHMwNTN\n8723dHZizl/LDpZcnqu40LBp7Jrr01s29rlpT3XWHw==\n=GIZO\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"service\":{\"name\":\"reddit\",\"username\":\"akalin\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1409475156,\"expire_in\":157680000,\"prev\":\"99456cb27d7c6c8e6978be4c0928c19a07428e236fa9f7ed4d823d926f054701\",\"seqno\":4,\"tag\":\"signature\"}\n",
            "sig_type": 2,
            "ctime": 1409475156,
            "etime": 1567155156,
            "rtime": null,
            "sig_status": 0,
            "prev": "99456cb27d7c6c8e6978be4c0928c19a07428e236fa9f7ed4d823d926f054701",
            "proof_id": "deea58754d6d77d60e621810",
            "proof_type": 4,
            "proof_text_check": "\n\nowF1UUtIVFEYnrEUZ3qg9KIcWpzQHox57p37nDYFtbBFEQZhNU73ce54UO+Md0Zz\nskGtREQSF5KoEUEZVgpqZUXRTGCKiUEmUTFKCmItXIT2GCrtXLFdnc055/u/7+P7\n+JvWrbLYrdHqU2eOobxS60hsVrYcT5rOrwSyXw0DdyUoQsuXhnUfMgIG1kPADSAv\naxTvUhhJoiHHyIIg0IJEHpoAWZmDLMWxrIungBMU+oOmgtjIUhDtxX6CkY8XqwT9\nB79seYBkGVGqKDIUlGQeUrSL5mhFE1RNEXgRQpMYRIYulSDCloqkYqyDiBMQrBwr\nyEy8MjOQquLQ//ihcMAEziLZuyL1ylhXSVmiKEdGEPt14KYIUwlhU0sxUGR4lmI5\nJ0AVAWwgLzYZLM8JkBwnCBionFiS7CynyDSv8gqnCIgTeUFGjAJFWlAoUYI8QwuI\ndnGaJGo8UhlVoF2qSHMaZBnSGJhlSnU/cDMkpuQjlkHs06VQmYFAxF5vday2WO2W\nlOQkc18Wuy3t7xZzalItfbY22/XtT+Ij9S/66x5/Vye5o73r38c8P5Y8sXjH+fyU\n0eZzvxrXFnfVJbVPeXrG98/mBsYHWr5cobunN1Tfe7ax4KkWmZpy1kZ21Q4NuBur\nEu+GPTVrhtoy0gOvotH2QLTl5cLXsh46O9Y/aWMeZR7RC7f8bswani55k+isT0/2\n3Rnc3emQLt3IvtsafjBxoC/c13Cy+dDtbXhnZvqOmydej+7D3qK0xMG0hapE9/zP\neJ2jcCbj4Vg4q2ux/+PWOXu8puBDMZt3nz98sSO4OJb2HHKnl77lXm2dzXHMwNTN\n8723dHZizl/LDpZcnqu40LBp7Jrr01s29rlpT3XWHw==\n",
            "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am [akalin](https://www.reddit.com/user/akalin) on reddit.\n  * I am [akalin](https://keybase.io/akalin) on keybase.\n  * I have a public key whose fingerprint is 07BF 173C 4AA2 064B 8882  8A64 BF80 5B60 5165 5371\n\nTo claim this, I am signing this object:\n\n\n    {\n        \"body\": {\n            \"key\": {\n                \"fingerprint\": \"07bf173c4aa2064b88828a64bf805b6051655371\",\n                \"host\": \"keybase.io\",\n                \"key_id\": \"bf805b6051655371\",\n                \"uid\": \"ebbe1d99410ab70123262cf8dfc87900\",\n                \"username\": \"akalin\"\n            },\n            \"service\": {\n                \"name\": \"reddit\",\n                \"username\": \"akalin\"\n            },\n            \"type\": \"web_service_binding\",\n            \"version\": 1\n        },\n        \"ctime\": 1409475156,\n        \"expire_in\": 157680000,\n        \"prev\": \"99456cb27d7c6c8e6978be4c0928c19a07428e236fa9f7ed4d823d926f054701\",\n        \"seqno\": 4,\n        \"tag\": \"signature\"\n    }\n\nwith the PGP key whose fingerprint is\n[07BF 173C 4AA2 064B 8882  8A64 BF80 5B60 5165 5371](https://keybase.io/akalin)\n(captured above as ` + "`" + `body.key.fingerprint` + "`" + `), yielding the PGP signature:\n\n\n    -----BEGIN PGP MESSAGE-----\n    Version: GnuPG/MacGPG2 v2.0.22 (Darwin)\n    Comment: GPGTools - https://gpgtools.org\n    \n    owF1UUtIVFEYnrEUZ3qg9KIcWpzQHox57p37nDYFtbBFEQZhNU73ce54UO+Md0Zz\n    skGtREQSF5KoEUEZVgpqZUXRTGCKiUEmUTFKCmItXIT2GCrtXLFdnc055/u/7+P7\n    +JvWrbLYrdHqU2eOobxS60hsVrYcT5rOrwSyXw0DdyUoQsuXhnUfMgIG1kPADSAv\n    axTvUhhJoiHHyIIg0IJEHpoAWZmDLMWxrIungBMU+oOmgtjIUhDtxX6CkY8XqwT9\n    B79seYBkGVGqKDIUlGQeUrSL5mhFE1RNEXgRQpMYRIYulSDCloqkYqyDiBMQrBwr\n    yEy8MjOQquLQ//ihcMAEziLZuyL1ylhXSVmiKEdGEPt14KYIUwlhU0sxUGR4lmI5\n    J0AVAWwgLzYZLM8JkBwnCBionFiS7CynyDSv8gqnCIgTeUFGjAJFWlAoUYI8QwuI\n    dnGaJGo8UhlVoF2qSHMaZBnSGJhlSnU/cDMkpuQjlkHs06VQmYFAxF5vday2WO2W\n    lOQkc18Wuy3t7xZzalItfbY22/XtT+Ij9S/66x5/Vye5o73r38c8P5Y8sXjH+fyU\n    0eZzvxrXFnfVJbVPeXrG98/mBsYHWr5cobunN1Tfe7ax4KkWmZpy1kZ21Q4NuBur\n    Eu+GPTVrhtoy0gOvotH2QLTl5cLXsh46O9Y/aWMeZR7RC7f8bswani55k+isT0/2\n    3Rnc3emQLt3IvtsafjBxoC/c13Cy+dDtbXhnZvqOmydej+7D3qK0xMG0hapE9/zP\n    eJ2jcCbj4Vg4q2ux/+PWOXu8puBDMZt3nz98sSO4OJb2HHKnl77lXm2dzXHMwNTN\n    8723dHZizl/LDpZcnqu40LBp7Jrr01s29rlpT3XWHw==\n    =GIZO\n    -----END PGP MESSAGE-----\n\nAnd finally, I am proving ownership of the reddit account by posting this on the subreddit [KeybaseProofs](https://www.reddit.com/r/KeybaseProofs).\n\n### My publicly-auditable identity:\n\nhttps://keybase.io/akalin\n\n### To join me:\n\nAfter a day of posting this and completing the proof, I'll be granted invitations to Keybase. Let me know\nif you would like access to the alpha.\n",
            "check_data_json": "{\"name\":\"reddit\",\"username\":\"akalin\"}",
            "remote_id": "t3_2f2l3y",
            "api_url": "https://www.reddit.com/r/KeybaseProofs/comments/2f2l3y/my_keybase_proof_redditakalin_keybaseakalin/.json",
            "human_url": "https://www.reddit.com/r/KeybaseProofs/comments/2f2l3y/my_keybase_proof_redditakalin_keybaseakalin/",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 33,
            "hard_fail_count": 0,
            "last_check": 1443790133,
            "last_success": 1443790133,
            "version": 24,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 5,
            "payload_hash": "297ac58d197e52956fae5d628b4db81de68e8cebf1bfd17fbb4404cee7edba6f",
            "sig_id": "fc16a961ab69ae81910f3a76ca000fca19699dec8461fc7ec0e8fcb78764069b0f",
            "sig_id_short": "_BapYatproGRDzp2ygAPyhlpneyEYfx-wOj8",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGbwMvMwMW4vyE6ITA1uJDx9IHXSQwhTI92VCsl5adUKllVK2Wngqm0zLz01KKC\nosy8EiUrJQPzpDRDc+Nkk8REIwMzkyQLCwsji0QgI83CwDTJzMDU0MzU1NjcUElH\nKSO/GKQDaExSYnGqXmY+UAzIic9MAYpiUV8KlkhNSko1TLG0NDE0SEwyNzA0MjYy\nM0pOs0hJS7YwtzQwACksTi3KS8xNBapOzE7MycxTqtVRAoqVZSanglwMlctITM4G\nKkwtL8alp6SyACRQnpoUD9Uen5SZlwL0MFBHWWpRcWZ+npKVIVBlckkmSK+hiYGl\nibmpkamRjlJqRUFmUWp8JkiFqbmZhQEQ6CgVFKWWgewwSrQwN080SU5JNDBJNDZK\nTDQwSkk1MTA0Nk8xMkoySjQAiqcZm6aZp5mZmyelJCemGpqmJaVZGqaZpCYbmCiB\nPFSYl69kZQp0ZmI60MjizPS8xJLSolSlWq5ORhkWBkYuBjZWJlCcMXBxCsBiMlCB\n/X++0Ivzb27KOOkcX77oYEHUY8mvdza5T9IwZsmcKerHve7BMqfyafvXSx09HFXv\nbsWpMO2qtmtd2YLvX/xn7lfV0hZ3NW1ZcTDpdNYXERl+jpv/L/AYpdppV0y70fF8\nkspFbd7jLh8Tf2vcKmfkiT9wU+qB7eLkkz7KWxliNFfN6l5t/YR/y6QlMnoSHEWW\ne3TSJugLrlA+8GShEc9bk/zo1brtJ6Y7SD9QPDoprbliR12/JG+N2nv9lanPX/9L\nSvTi43Hin6gXv4tRuYrnsdnCabsPHhbbEvfx9z6HD1s3tlnxx5aVzj+qvrCy6dOj\nJK9Tus4anyrMcjSrM+L3ZvxQnn34i4z5gaX2kv0SlgA=\n=RqrD\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"service\":{\"name\":\"hackernews\",\"username\":\"akalin\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1409475252,\"expire_in\":157680000,\"prev\":\"a2a877a4cda04a32aa02de40137d22b2a0a04f35f7f677bdcae15fbf91f4ec04\",\"seqno\":5,\"tag\":\"signature\"}\n",
            "sig_type": 2,
            "ctime": 1409475252,
            "etime": 1567155252,
            "rtime": null,
            "sig_status": 0,
            "prev": "a2a877a4cda04a32aa02de40137d22b2a0a04f35f7f677bdcae15fbf91f4ec04",
            "proof_id": "1e78f7bc1e740b086b64ef10",
            "proof_type": 6,
            "proof_text_check": "_BapYatproGRDzp2ygAPyhlpneyEYfx-wOj8t4dkBps",
            "proof_text_full": "[ my public key: https://keybase.io/akalin; my proof: https://keybase.io/akalin/sigs/_BapYatproGRDzp2ygAPyhlpneyEYfx-wOj8t4dkBps ]\n",
            "check_data_json": "{\"name\":\"hackernews\",\"username\":\"akalin\"}",
            "remote_id": "akalin",
            "api_url": "https://hacker-news.firebaseio.com/v0/user/akalin/about.json",
            "human_url": "https://news.ycombinator.com/user?id=akalin",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 172,
            "hard_fail_count": 0,
            "last_check": 1443810056,
            "last_success": 1443810056,
            "version": 22,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 6,
            "payload_hash": "04e5fba11ba32227cacb8b8345cec9bee563bfd4c25b77aa716b3763c3857735",
            "sig_id": "ac377cb7d4df7bc9db7c50b3bc06146e05e29fd0eb715b478ea461074a8770550f",
            "sig_id_short": "rDd8t9Tfe8nbfFCzvAYUbgXin9DrcVtHjqRh",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowFtUVtIFFEYnjXXciPYJBDSHhrDlhKdmZ3rUtRDFBZY4o0usp4zc2adVmfW2XW9\nJWw3osCMraA0RHMV8iHSSskStZsPWr0EQi4E9qBbD1JiYgZ1RvKt83LO//3f9/3/\nf/4bWzYQDtto5HRFISqqsU2OJSBRnPSFayKhoTSQnibSj9YuVdN9yAyYmh4iPSQl\nQJUW3DILAEPxLBRFkREBfqgixUGe4mie49wCTeaQlUbQUmAbCIIoVzMwhgOvpmD0\nP/zatQSCENGKJLE0BaBA0Yyb4RlZFRVVFgWJoixiEJk6qEaYDfygStPJ5hwSY2FN\nRlbHilENMLiezZXrsShgGiFDNqowrOhBSxFqCFgWdQh6/4m9UNMVPC6mh5EZ1Axs\nQmOmHNKsajRLSayAu2VzSFQf0EzktcrQnMCLFD5WERTGlowkAJkTFVoSEMdIHK8C\nxCk8I0JWgSKtIF5EoozwT0JVoQUVQpalWBkhASkQ8CppjVOjG6SHx20CH7YMaj4d\nhGpNRDY7rtoykwmbg0ixJ1kbIxypzvU95hEb//jbylrmj0y1vD/oW9w+7Mh8WVBx\nXJ7ImGZ25HculT9alj8VVZ4/2p1uTzt3K1p6J7GvpXy+qytt+G5b/IO0/Pzai7TL\nsZsX98R+j77bvfp4PjGQAGS90w8bGzp/fH/Wk1cCxg8/eBM5k5L9tTQ8qM2uXhox\n29u3tvqnJ4gr/U+mFiNsxq6ToqNuhpvJTl0pdrjMlWjcTOr72Xc9NnIv43VvR1XF\nCfsCd6ij0xntjxT8upD8Od7o2nS7tNG3kCVlndLm+P0fdw6m2xfPHhg7NuZpNebS\nl0bvd0z29hDjJc4hV7R7c9lsrPjtq2Dkm3tb/tPl7DqXgMShgcL4w71/AQ==\n=FZj7\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"service\":{\"domain\":\"akalin.cx\",\"protocol\":\"dns\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1409475374,\"expire_in\":157680000,\"prev\":\"297ac58d197e52956fae5d628b4db81de68e8cebf1bfd17fbb4404cee7edba6f\",\"seqno\":6,\"tag\":\"signature\"}\n",
            "sig_type": 2,
            "ctime": 1409475374,
            "etime": 1567155374,
            "rtime": null,
            "sig_status": 0,
            "prev": "297ac58d197e52956fae5d628b4db81de68e8cebf1bfd17fbb4404cee7edba6f",
            "proof_id": "c8968b3d21de18ae33d30210",
            "proof_type": 1001,
            "proof_text_check": "keybase-site-verification=rDd8t9Tfe8nbfFCzvAYUbgXin9DrcVtHjqRhB0qHcFU",
            "proof_text_full": "keybase-site-verification=rDd8t9Tfe8nbfFCzvAYUbgXin9DrcVtHjqRhB0qHcFU",
            "check_data_json": "{\"domain\":\"akalin.cx\",\"protocol\":\"dns\"}",
            "remote_id": "dns://akalin.cx",
            "api_url": "dns://akalin.cx",
            "human_url": "dns://akalin.cx",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 2518,
            "hard_fail_count": 0,
            "last_check": 1443826575,
            "last_success": 1443826575,
            "version": 2,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 7,
            "payload_hash": "bc78b22a430b8c6fd1b7f13b4d518db0b59a6ffa9356a501d087078fde529f5b",
            "sig_id": "043dabef311e0061afad845c64f4a9bd2098353bff93d8267e70c9d90d353c1f0f",
            "sig_id_short": "BD2r7zEeAGGvrYRcZPSpvSCYNTv_k9gmfnDJ",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowG1VFuIVVUYnuPYjE4K81AqNUFuMQqmXPfLRD70UEKIDZqWlse11l5rzp4zc/Zx\nn33UaRobCIx5CCRQ6CGMSCJFcYiUUBm7IWSh0ZQP1kxPXWQetKykUFvrOE4JIfTQ\ny97/3mvxff/3/Zed85ub2gpjwxs2ddvVmwunzuumNbOmzg5GOo0Hoq7ByPQltpKH\nqKL6bdQVle2AVjX7UJLeu9XqqDPaYrNaklb8kUHIQeCcdUZj7iiKqcKIcy6Z5JoC\nACwxAJNoqDPABFCXVHpsVs2SwBEBrh3k2BClEGBECyGQUD5wAlDNAIWMUsyhZy2l\ntfymbKIGZjGJ/d9/uV9vHFitLYylJBAozQFEGDFknIidEVwCEC7WbDYtVZVVX1IJ\n2eaZMuWQrydLTC1ESVycUQ5pZ9SnarlnL5qS8pL8PywZEQxgfhNmnvb3qnJdp1tr\n5SRAX8/LcQa9TZpCq4xT/u0EwVRiBrVt5DVtWNB4s2lGCetkjAzXsdPEEaktYZzF\nGjsFAeGaECpFMKHcIAMQQAqkw1z5AjlidRxLyxylEiKDDKOMIySsYloQf8VHMQNE\nSUMoUIhSYyRQIffM9qe5LVazNHXelA2+XfKkf0Y8lVJ0RqaeZYEVKK6hMDERmnKN\noBXEYkhjYxBBiCLLFYuhhtR/K2OoY0xL4qADGEnnfPp2GpxS4gvbAK9mdkvwABDt\nqx17C5E12AjnKysERIJiwbyXxlF/wQMzxYVvAWGtktIoyAxzMHgzrSX429DTaP6S\nNeVirHJV7K2FQs9MQU+Sl+o6unVpGzjFfKDqT3FnVMtVHtL3JzW7uZJGXdJHSc/1\nnhWGW+JtMVYoY5UyjCnhtH8AHpqDC86h8F5owbRhvl1FjJkfDuSMZVyD4FBAu86H\nhp5r0BRzlfSFxKtqoC9VcbGkaqVblONE9T8U5P+RMRQGriFieu7+sWOCeTd6jABJ\nPB7yA2a3VZPMFpPGLHIm/KoBM80BiKVOKwi130YIcaOMFlpgQo01flgsZVi7mBhE\nNedKccj8+mLYYEE5x/RvoZ4pVz0e0sutqLye2WiobaTQMbup0NbUctussDqb2ua2\n39inP/3Qem1w14VV2w7OG5m89MiKjjte6u9a504/4AaOFc6eO08m9u7e2Xx6weGN\nR0ZZyexp/w2f6V7+x7uLlq99/GF2dGzFyovdFw5exW+fudhy7IUDYwvoPeOz7zbN\n8xdWWt6Y/OXOg082jV4+crl+6K6lVx4kmx79mJ3bY0D164tPfLB4Ysmn30+OyVeu\npGv3L31qJbj9aHe+6J2NO19vXfzs8NxLB+ZNrdnzzfAzBXioY86H63/eseS7184+\n/dZj5sev3n/v5ODYSXJo1viy3qXjI2tevnpi6rOJ5TtWdXwkeveN7r7v/muVF79Y\nn33L9x3evvfL39uj53e1Hj+1LNv7ycKF2/88/uvnq1816f6Tb87Ztynv/As=\n=BdWf\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"c22f10ffefcb37f52d5a327779697b5000e4c034\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":15,\"last_id_change\":1396486037,\"username\":\"tomjakubowski\"},\"id\":\"ef761796b51eacfab51f84359361be00\",\"key\":{\"key_fingerprint\":\"ca8ef9d2c7bdfb4f49be4676db3fa1047b445981\",\"kid\":\"0101509f37ad5af4ebdd9e6f55912c2c6567228ea6b84ad58ead604a9c450a255cc90a\"},\"remote_proofs\":[{\"ctime\":1396485998,\"curr\":\"00a7b18cd48b57b21e84e315dcc242252e7a6d1b15c24acc5f66b94f1f0329ff\",\"etime\":1554165998,\"prev\":\"c04b051d7962ec3c8f2328812853861f8cf5b0524a6a78e1d8eea99ca16c6f11\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"tomjakubowski\"},\"proof_type\":3,\"state\":1},\"seqno\":9,\"sig_id\":\"8c7e4b21ce8aceaac66a8fb6a8077617787718032b86bc68798d36a642fce67b0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"00a7b18cd48b57b21e84e315dcc242252e7a6d1b15c24acc5f66b94f1f0329ff\",\"seqno\":9,\"sig_id\":\"8c7e4b21ce8aceaac66a8fb6a8077617787718032b86bc68798d36a642fce67b0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1409477827,\"expire_in\":157680000,\"prev\":\"04e5fba11ba32227cacb8b8345cec9bee563bfd4c25b77aa716b3763c3857735\",\"seqno\":7,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1409477827,
            "etime": 1567157827,
            "rtime": null,
            "sig_status": 0,
            "prev": "04e5fba11ba32227cacb8b8345cec9bee563bfd4c25b77aa716b3763c3857735",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 8,
            "payload_hash": "509a15dbf19107aa39362160bc2dd5a02175eac38b15315af8d845dabc2836b3",
            "sig_id": "2d0f5396223f80fb142bc5fc577c332533da015b4b7c0820b2959c267e6782040f",
            "sig_id_short": "LQ9TliI_gPsUK8X8V3wzJTPaAVtLfAggspWc",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGtVG1ollUYfrf1oQNNghqUlD3+EGwb5/tjCElBSK7CDBJEX8/n3meb7/vueZ5p\nUwZiBE5i9CMHs0hBS/pADQtNpDRHIbmCCCLGKAkWSFAZZRFp57wu04iI6Pl185zD\ndV/XdV/3eXZeS6m16d1tazescqsHmj48r0uPt0wt2Jromh1KurYmpj911SJWVbXR\nJV1JnxvSKnedaW3RZqeT9mSTy/K0Vg1HBiEPgffOG425p8hShRHnXDLJNQUAOGIA\nJslwe4SJoD6t9risnqWxRwK49pBjQ5RCgBEthEBChcILQDUDFDJKMYeha6WWF9ex\nSRqY5dSGv39zf7Bx4LR20EpJIFCaA4gwYsh4Yb0RXAIQL+Yum5Wq+lR/Wo1si0yZ\nvsg3NEtNHqvUlq8qp+1Jv8qL0LxsKiooSrogAZRhgqG4DrJ3MC/Sam4qg5WI2yAF\nnYZIYiMsslJooAj1ghhEmLMGNEjNuhUF/sUxJD1o6BFeY2SED/5rQ6EPDiIikWAA\ncewjRqMZgAAqZiDW1hACDGKKSusZxAxBBQgXyEKlJATMAq0VMQ5w5JgNnwTYWoUM\nUJF75jbWCleuZ7WaD46sDVkp0o1XpGMuMWOyPTGDWRa6ShlMZ0wQqa313FBmEOQO\nWwm9p0G1MhQgBwmX3lGsCTOGQUQ0CW2VijFzs+CUhVtXwGcZRFcaLBp5rTjTV7aq\nUOXePM7manCLzWlRuCz5p3k0YMrFUD2cofYkL1QRe4aT3A1Ua6EMVdpzJWWGCymA\nJR4hGMy3WEIjoBDUaG1dVAWDx1zFKYQ0hlAaLiklDISRQGxAnEpEm+03vK7Rplyo\ntD/yrquh/pqy5YrKK/+Ph/9CxMnkP8gYjivSEDG7Kde8CtG8P4MhKRZcBALuyXqa\nuXIab1DORMg5iP67TXF/AyuNkCIYaGGYt1BzHyJLLIXCaqCpVMx7JTEN8QXQAsEB\nFz7QDftAr5Eatq9QPQEyCK6qYjBzyXDrSNPCG0pNraWbbmyOj12pde6CP17AI7tu\nvryq6dSSxV89M3X/B8UbC16aefFnUYyufN3f+sil3345P1Kfs3xm/05T7Ridev+t\n4qlP9xw9NL3l+cXq5BOfFROnDt03fXTdmn29B7/Nxvpfnhlc/APeOwAfvrRjYGC9\nHdveJtq+x8vGhiZP7B9H88+27K59ueKOvd3vXJhHx++dPDZ9J7pwYs3Biz+tGDl7\n2y6oDu9/oPuu20vfDcxdf/Ch5yYvP71nfu2jjuadby85snx8UXHxzNjUo8eXvdn2\n8TcLV54bWVo6PWEefO3A0qXNW7Pz43dveI/U89Wf2K7jZzofG57b/euBWsemr+U9\nO0Z73b4ft7xKPz+37/SxOc0vuFcubB/5YtvuWzrHZ+wE+h0=\n=LHXi\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"c22f10ffefcb37f52d5a327779697b5000e4c034\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":5,\"last_id_change\":1405634318,\"username\":\"justinschuh\"},\"id\":\"1eb1293c8d2d98b0a45f84c246edc000\",\"key\":{\"key_fingerprint\":\"029f0994108fb32c8f0ffbc51fa202492860273f\",\"kid\":\"0101a6c13bdc440c26a59df613621a04782d1aa9106d0bba4ce072e6dddd903dda2c0a\"},\"remote_proofs\":[{\"ctime\":1403793669,\"curr\":\"99d9966849bddf7c56c217e3d91ff5d98ac502e1479fe53b46cc6124b4dddaab\",\"etime\":1561473669,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"justinschuh\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"c78980d4f221fb3d391c81885cbbde5d981dc47a49285b6bf8c795546002713c0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"99d9966849bddf7c56c217e3d91ff5d98ac502e1479fe53b46cc6124b4dddaab\",\"seqno\":1,\"sig_id\":\"c78980d4f221fb3d391c81885cbbde5d981dc47a49285b6bf8c795546002713c0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1409538781,\"expire_in\":157680000,\"prev\":\"bc78b22a430b8c6fd1b7f13b4d518db0b59a6ffa9356a501d087078fde529f5b\",\"seqno\":8,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1409538781,
            "etime": 1567218781,
            "rtime": null,
            "sig_status": 0,
            "prev": "bc78b22a430b8c6fd1b7f13b4d518db0b59a6ffa9356a501d087078fde529f5b",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 9,
            "payload_hash": "9ca79719d489099193dfefd2f76f9780cf73762c19305d4e377c1b4903a75278",
            "sig_id": "5bf1e508a73c3f37aa737731c2474d64446786f4615bfd7e04e407aa2278549e0f",
            "sig_id_short": "W_HlCKc8Pzeqc3cxwkdNZERnhvRhW_1-BOQH",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n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\nfnT2VVu+W/gz3vnle+f9/umjI9//Aw==\n=F1Eg\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"c22f10ffefcb37f52d5a327779697b5000e4c034\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":10,\"last_id_change\":1411673314,\"username\":\"gobiner\"},\"id\":\"e77a4f4103429648f544ca88ae9ac400\",\"key\":{\"key_fingerprint\":\"c29b5d863ea7e71d94f521b3ffa425daa3785f5f\",\"kid\":\"0101941c6e733b0a7a039b1eebe6bf4ebf8ae494fb6b500bf36078d27cae184ea8500a\"},\"remote_proofs\":[{\"ctime\":1411671108,\"curr\":\"7fc0d1dcc8e9de68d1352d8e67312e798eed2d75e12f6490e135ab39057bd2fd\",\"etime\":1569351108,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"Gobiner\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"a138e56f51d3d857b25576e12599d029873f2a19f7189a5e7813fc9c28c6809c0f\",\"sig_type\":2},{\"ctime\":1411671207,\"curr\":\"95d2e9a018ad49b9ff12f9830b943862e75b7e00027cecde9fc86cf4af7a8ee9\",\"etime\":1569351207,\"prev\":\"7fc0d1dcc8e9de68d1352d8e67312e798eed2d75e12f6490e135ab39057bd2fd\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"Gobiner\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"68fd1b7ac6a4ce123e92292b68a4c2c627cc6cd741cb3cd69dac702aa19363ab0f\",\"sig_type\":2},{\"ctime\":1411672221,\"curr\":\"01f8ca1c36d94eda2bd5165dd77a7a361b3b218c6ce886714975c3e6aeef3d98\",\"etime\":1569352221,\"prev\":\"23c40df880d6b972d48de4469ebd5e7a7120ae0d506f70bf31d5108ef69bf3e6\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"Gobiner\"},\"proof_type\":6,\"state\":1},\"seqno\":5,\"sig_id\":\"004ddea191ea4ef4b04632a8aca00c686c3657e15661c027c82b8f7c68907bf00f\",\"sig_type\":2},{\"ctime\":1411673282,\"curr\":\"e38c8530776b8c9f2b87545a7406695ba1f122ac4a1d5a17fdf4b6da8413bcf3\",\"etime\":1569353282,\"prev\":\"01f8ca1c36d94eda2bd5165dd77a7a361b3b218c6ce886714975c3e6aeef3d98\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"Gobiner\"},\"proof_type\":4,\"state\":1},\"seqno\":6,\"sig_id\":\"852c8d8bc5438ada66ab7e533f8185eef244bcbcc1cf8b1d230c3ad9855315550f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"165f7421dfcaba511f5c47bcd82100b93f7a81ce1008078f3fd6183d0edc1ffd\",\"seqno\":13,\"sig_id\":\"1d298f57b048c2d5407fa1636754ff8d73128880261ff9f95885c4d9399685140f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1411694605,\"expire_in\":157680000,\"prev\":\"509a15dbf19107aa39362160bc2dd5a02175eac38b15315af8d845dabc2836b3\",\"seqno\":9,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1411694605,
            "etime": 1569374605,
            "rtime": null,
            "sig_status": 0,
            "prev": "509a15dbf19107aa39362160bc2dd5a02175eac38b15315af8d845dabc2836b3",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 10,
            "payload_hash": "4cb503967897079bcbd6f37ad4a0446a0b4cd4b08ab7b271a1bfe98694384972",
            "sig_id": "769dbf1042afea9698d97bc36aad578f968f6a32893a641fdfcbf19727e8a21c0f",
            "sig_id_short": "dp2_EEKv6paY2XvDaq1Xj5aPajKJOmQf38vx",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGtVmmIVWUYnlFLnIwisEgl5Kg/glG+fTEUcYFCBoqZH4HL7Vvnnlnunbn3zEyD\njmlFIYEYJC2mBGKk9KPFJWyZSgS1RbACIUrJJLKiIqSsMXvPdRpHR0Khfx/nnPu8\n77N873s33Ty+rqF+cN3yhx4Izd31H52xdS2z3u9Zndiy70/mrU5cRxpKWX4qmc6Q\nzEvaQ7811TA3Lc/oCzZpTHpDpZqWS/DKERIxijFEZ6mMnHhuKJFSaqGl5QihwByi\nLBlozGFy0JiWWkOlq5LmNRIkbcSSOmYMQYJZpRRRBg5RIW4F4lhwTiWGqsVyNbus\nm6SGWUg9PL3K9z21F8HagL3WDCNjJcKEEkFcVD46JTVC+YfVUBmmatpNR1rKu80q\nxrXn/UKx1FXzU+oLI8xlY9JhqhkUL7iiAUbJPMwwFlJiTC6D7OqLOV6tGaYV8OQk\nMNDGBRsQ4dhHa4mGZ7VmhlXKiV2ulKFcMIIZRY5ySqkg0bNAqSEaBy+YN/AAW5Zj\n1IohjLDmmDGjuSeaYvh9UFRbjaMAoSnj3AkkrGEKySiNkoZJFXnAxMtgXPQBmbz3\nSugsZ6HQVSmXIyixHDKSpZ0jlIXiqjFxPZVKbgTWjDElFYrMByqCwxAQF6jSSllK\nUbBGYgAWXCIRtfOagDUE5FAM2wjth2FwLjRwroEPd5CrUuuiltNicO0FbzJTaKvm\nnowENutLsyxUkit9iDUnagCFrL8LnoJV1cxkeTV4Uw3dpTIc4ZS2XsyVJloRa4CF\nZ4YTYZShCiOlNXJIemSMJp4oFQQ2XBMKCQNfMcUawpgLmxPK0YbrDTReKZ7GfEQ8\ngqIHUVSMMgbKILOBGGeZt54jz3AMkBzoRtLASRRBKMKog9B7boOjdIx4NfCuSuj9\nD2e+vw5rrs+H1jQr9thrsIFezQYyygZisFUwUQz1CBTADmwRBkvtMfKeMxEIkpII\n7pTyyEetKNeMMyK04qDxNdig0YgNEcaWijZ45BxDEVTGHC4TQxZQY65roDHqQJim\nyjO4K9RJJHkUNBhJ+VgbcvBhG/4Hj6/PhSIMMpA/9FXHOHHRhw9mXnJCXM0JOsoJ\nRRD3kXOYFsTCHGJSMOS5DM4QC+tAGRUksjCQFEwZxBQMt2CZIYIamDRkjBMra2UK\nmUk78qa7TH9H2fhC0VSLUM5rQBIQTyYwIXDDIkOQXOE4pFOIgDEVDKTyxgZJnAmW\nKKQgwiEfHYgkIyTEKBJRUQg/TM68YxRQ4BLUV0p4DTdZRoi8ExwCxhiQicBBCRG9\nwFw7giBzoFy+I2oUhlfFqLWYSzc6XRoYwMIID3ellVBI8y8glAq246VYaGekllh7\npjTSGmvqYat6EiXcRLizLkoqYXPBCzAABr+UDqa9RtRITqS6RBQDaGZaARP4lkzW\nUwnJQMOG+ukT6uob6m68YVy+7usaJt3673+A5z+deGHtqj1y6PHmjw9P/mXt20sf\nfP18eq77jbdWrZ29YOfeFX81tfWen/z3thPP3ffqyZVTjxbv2Dm4a8WCl7fM7zy0\nfc69O1sOHqifNRWtPrfjlSWHPlk8NLB5WemxU0faTrBF4/Zt+3B/+c3Zu9IJC9V4\nsuie5n3LBh+R+4/sn3ns9+bBw0OnX1q35r1pv3319FP3P7n90c6zP3/z4jGR7iTv\nLH5h68Znw207bunfg7ubzv7pv14zMDB3+9De2P9H76Q5myZu3dp0amFlyozJ64+3\nodsP3rmBbJy7tDDtps2/vjt995LXLjzxeemLZ/ae+XLLt+uzQ013zz989PjiCy1r\nJhwgyzsP3DXp9Mkfpu6eUjj602ctP6rv/gE=\n=Oz7R\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"c22f10ffefcb37f52d5a327779697b5000e4c034\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":7,\"last_id_change\":1411677112,\"username\":\"pwf\"},\"id\":\"498aa252e4b50cebe0251dfbb292e400\",\"key\":{\"key_fingerprint\":\"a356421430c3533362fd4e33a291ed64da2fd1b4\",\"kid\":\"010195144a95d2931564e839b91f6b883455c606ba4807f7a87a478f5e12d7eacfde0a\"},\"remote_proofs\":[{\"ctime\":1411676858,\"curr\":\"b194448780f4de36ec110fce38988b330eba71e0a65706f9cd9210a20ce841bf\",\"etime\":1569356858,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"pwfff\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"92982baec1d4a526a8a38108990c07d0aa92d288e61a5923701ceb1319b60a480f\",\"sig_type\":2},{\"ctime\":1411676915,\"curr\":\"20fd06f8ff7fe34410e2acb4dbd50d41fee4bbae73e52f6e68243c051d5bec33\",\"etime\":1569356915,\"prev\":\"b194448780f4de36ec110fce38988b330eba71e0a65706f9cd9210a20ce841bf\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"pwfff\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"2a1b897ba3d04bb1c2986a179d10dd546e2077265c88d0df98359454269857690f\",\"sig_type\":2},{\"ctime\":1411676990,\"curr\":\"f00e8fbed0cc40f6821510140b359f9356e3ff9e24938d47ea3c7075f63ea735\",\"etime\":1569356990,\"prev\":\"20fd06f8ff7fe34410e2acb4dbd50d41fee4bbae73e52f6e68243c051d5bec33\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"pwf\"},\"proof_type\":6,\"state\":1},\"seqno\":3,\"sig_id\":\"8205df55a872b53347640d57eca2bc228a8e70bd648f6b048bb2eb4a263aba420f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"d9d64688b46122592f406ec6c5cd966e1136473edabe72caeb28080a6e356802\",\"seqno\":6,\"sig_id\":\"f83c11a29bc220e0e57bd5886d98e67f20cc656a144e70f8bb866fd6159c20d10f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1411694687,\"expire_in\":157680000,\"prev\":\"9ca79719d489099193dfefd2f76f9780cf73762c19305d4e377c1b4903a75278\",\"seqno\":10,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1411694687,
            "etime": 1569374687,
            "rtime": null,
            "sig_status": 0,
            "prev": "9ca79719d489099193dfefd2f76f9780cf73762c19305d4e377c1b4903a75278",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 11,
            "payload_hash": "a8d6f791591901bd574cc1f9ec97a713f772a59ffceb1de6ef07296b0be7cc8d",
            "sig_id": "7d2d653312c4644daf5b6089cb1c3651d3bb8a8f57b7055810ef804fa5c871310f",
            "sig_id_short": "fS1lMxLEZE2vW2CJyxw2UdO7io9XtwVYEO-A",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowG1VFuIVVUYnmNm02RZEaONGbYRRuo4rfvaa5QuU0JKBVIv4cR2Xefs5szZx332\nTA7ToYLAlG4zBkUGZhciBCvSqHlJBdMYM6QkejG1KWuwm0iUJLX2cRoVQnrp6Sz2\nWuf7v+/7v/9/7vKLmloKHz22ctUKe9/qwtiEarp/wY51Q4FKzGDQORTocmwrWX6q\nyD4bdAa9dlDJmu2Ik/kPWxUUgwGb1uKk4q80Qg4C56zTCnNHkaESI865YIIrCgCw\nRANMgnoxh8lBXVzpsWk1jfMaAeDKQY41kRIBRlQYhiiU/uBCQBUDFDJKMYe+aimp\nZeexCRqYUWz8139539+4sEpZaIQgEEjFAUQYMaRdaJwOuQAgf1iz6aRU2SvLcSVn\nm6VS9+Z8fbFY1/JTbKIp5bgYlGUt88UjXZJeUdAJCYQsJBTS8yAHkjSpJHE1yWSW\n5MgNWtoi5ek6aJRhyCAKuckNgIYpRhq0Jv3KJZ7vmQLcCoqIQ0Jarw4hQTnGyihF\nvXMEcQQ1oTbHaBQDEECpsQydERJjY6kJESMMUyEQkowy3zGDAbGOQmcVZxpqw5GU\nTulQQ6ERBTLnntq+JLNRNU0S5z1Z6dOSxX1nxRMuioHuT9OcpoWQ+j9Laoi3X0qN\nOBNSh9RQGiopOCFGUI9PBKTC/xjEFAkJsABpJjx9OwlOmcDsDHg1tQMeHAHEMEQh\nQta3G0PMKVDQaWcwlyBUyHHHhPXOMAt9MDSxDAAuufEP/LNgSkvub0NPI/slq3sj\nIzMZPVTL+zw1BD1xVupXwYVb28CJssGqbSSk5r/n9P1Nza6uJEGnj0Yt7jkTWent\n8K0OhQKepMUOU0cw5aH32mDhx0YoK7S3CBsDsOWSYO6nSno3qDEc5CJytDP1UP3B\nRpkok3E5J16Vg+VEmqgka6ULtWNH9b835P+RUc/nrSFicuzOWTG5eedmTBCOkE/G\nmmqc2ijOX1DOQr9pwFQ4iParBwvGQ8EBF0r7GXM+F4ZIQAiTQBFtiAKh3wgKcSih\nclaEHhuHRHB0ViiEnpns8Zheb0Vm/akN6i3rCtdNbyq0NM24eFq+OptaLr3yn316\n02hz05bi3LnNn4ebdy7p2Ds2btGzv5xYvz0aveW3jrJ7D/341bvF16Yt2PXm8LaN\nY+v/WnXwtnTiwMJXn9rbt3T2no1bb3jj8LIlbZ+MTT88/vZQ26n2d37eeslVL7cO\n/3T3S918eF97+eDSjiWbj+7/4fjvX5Se2d8VzVh+xzVdb83bNf3PFx/47MvZ0etX\nvP/xp8XThW7X3jZavOuF1jldx++5vVq+9pWeRUdawe5bb1zb/MFss2HZsa+ZmXls\nfP7IvCfvfaK17frTp+5cM7d522Xdi9KZszoWl3c/PvHdHzfvW3vgm/ZfBx/9fr45\n0bPnyMyTJ+3iqzd1P3J0LAm2HKrX0aHnl6/ZWf3w6e0L53w7a6SwYWTF3w==\n=qunj\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"c22f10ffefcb37f52d5a327779697b5000e4c034\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":3,\"last_id_change\":1411684515,\"username\":\"voronoipotato\"},\"id\":\"ce2bb60f1dbd62d2517d20641d6b6400\",\"key\":{\"key_fingerprint\":\"b07e9524f29aebe12295733bdbb564b42721c45e\",\"kid\":\"0101ac3a8fd9a33de5d82646359922a656777d304ef51feb76c1cd72aafbc8c19c250a\"},\"remote_proofs\":[{\"ctime\":1411684479,\"curr\":\"be115c19a5d4941aac2769ac85d558ba9744d958c149159c14d26b4840e02c69\",\"etime\":1569364479,\"prev\":\"2026312822e537313750b1fcfd37a08b2f7f69e2956e15b6c4e6007a7dd37fcf\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"voronoipotato\"},\"proof_type\":3,\"state\":1},\"seqno\":5,\"sig_id\":\"a27664089b056ee3f35f4357850ad39c039be9cd953dd03e7a43797ba5695dd70f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"be115c19a5d4941aac2769ac85d558ba9744d958c149159c14d26b4840e02c69\",\"seqno\":5,\"sig_id\":\"a27664089b056ee3f35f4357850ad39c039be9cd953dd03e7a43797ba5695dd70f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1411694722,\"expire_in\":157680000,\"prev\":\"4cb503967897079bcbd6f37ad4a0446a0b4cd4b08ab7b271a1bfe98694384972\",\"seqno\":11,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1411694722,
            "etime": 1569374722,
            "rtime": null,
            "sig_status": 0,
            "prev": "4cb503967897079bcbd6f37ad4a0446a0b4cd4b08ab7b271a1bfe98694384972",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 12,
            "payload_hash": "f7fd588cbcc7708776343be0ea13152ff7fd2f49ec6f27c4c0334dd7feb2c192",
            "sig_id": "73f473aa10f68bb05b817ce34fcfc22a94a44d092100d92b03deea274c3054db0f",
            "sig_id_short": "c_RzqhD2i7BbgXzjT8_CKpSkTQkhANkrA97q",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGVVWuIVGUY3jU12zI2rKxMsqPUj6b87hdbitXSbkYXy9Bk+q67p5mdWWfO6m7r\nWpSBUaGg1LIZEUT+KVNyKYq0MsQy/xSEiCBFIEmBVrAKgb1ndt1UDGoYho/vHJ73\nfS7vOxunXtTU0rz7ueVPPRweXdm8/5htWjLn8y39ia36vmRef+LKaahk+aliukIy\nLymFPmvq4ba0Omt1sEkhWRVq9bRagUeOkIhRjCE6S2XkxHNDiZRSCy0tRwgF5hBl\nyUAhh8lBY1rpCLXuWprXSJC0EUvqmDEECWaVUkQZOESFuBWIY8E5lRiqdlbr2Tnd\nJA3MYurh9gLv9zQeBGsD9lozjIyVCBNKBHFR+eiU1AjlL9ZDbYyqKZlyWsm7zWrG\nlfJ+oVjq6vkp9cVx5pgWkrKpZ1C96DoNUMrvoIrQCqNzMF3V9HaFzJRz2EZPgtAY\nOVdMcWjDOG6MdwzTgCM1rtHTmFg5v3MFo9wajKgjhiGlKTGCIgWUggwMMK10Ab5K\n5BiNYggjzA2zwltEqKWKU2IpvEWkUc5LTRzRnjusoDLlEk5M0hhQNFyovJIMyOS9\n10JXNQvF7lq1GkGQ5RCVLO0aJ66ElIXE9dRqUFVbRpRTnHDGjfWBO6+k9DYSqoMm\nPEYqrAvgisBReB2gM4Yitgxp6WXefhgD5xyDUg3w7lpYBeDESck1Dkz4wIikyNHA\nNTEqIFAfEJQNwDJYai3lIJHGTDMjAgbVuAjJOJdc3wafRvA7gysVvclM8el6bvL4\nBGSr0ywLteTffW2AFLO+bnhCCkk9M1neOzyph5WV6uhl2jEaVpgRprE3MjJDInIi\nCKRAcXATWUZBTYaMkIpRhYPG0jPuWdAeOa6pVgTFZBRtrN5A4TwrtCBq3ApJmdCR\nEi4wR1QyUEcya5WJBqZN8V8c8wruAyVI43yclYWyXAeELZfGnO/FKPqYF5Bcj40m\niDkjQWKYZux0IMgZY+BHMSCltA+aOcqshqwZsMQZHSMKjv5vLzrSrLPH/kcr6IWs\nEGdZIVwMTGurHXMCCec8g0EyVEvoVsoQBaVeUmrhEkRUwEV47JAW0ROfi3WeFSsa\nZYqZSct5092mr1w1vthp6p15OSEBnDMpILcRa0a1p9TgGCX3CgQM0qroYaZ9CJ4E\nxZVFFBnwBDIRcyvGSOCzWThYebDRPFeUcqcMYFkIPQ0Cw3bQBuYM1rAG8WE5YIED\nZtZZTiKsEUSGaU5jIF96DRJju++sPZ+LdyZgDEO+mGQQ6NDbndZCMc3f4BKWBXz+\nCYbyIkqNYVA1BMlzyZzDUQenpZGYRimJ4ZACFyz2AZKDJNHCIgs9OuXPogqlMtMB\nmEC4YrKeWkgGWl5qvn5iU3NL0+RJE/L/r6aWS1rP/Knd8s2Upi8H35x1eKSv/YnT\nWx+65/iNP7d9cvCDQ9sPvLr1SXeycsfJ9p2/D/UeODZp88iat/9aP3Tf5pnTyjuP\nHmrbsuyH8isvf3XdrHWffT+9c2nv1aXXFp6Y/Mz9fHDSyIF1i+9+7OYXtv+5YeqC\n03HbxrlrLp+/ZNeHRsyePfHd9sX9p9cODjUNL5v+Or5y8J1Fpb179l31x+OHW4YO\nXXHDjAml9oGTh2fs/viNPVML29ZuQHun3fvswW27pv248KOvN61d8OCyKY8M72o7\neOmeFwdaT8y5eMWnk/s7NsWl7920763vRk7d/kVpR8uCy9agia2tp46sXrdo/613\nPRCvbS+0td/ZPcyePz7/yIlvr/npaFxf/rV37sz3f9vxNw==\n=G+4d\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"c22f10ffefcb37f52d5a327779697b5000e4c034\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":13,\"last_id_change\":1394169810,\"username\":\"coaxmetal\"},\"id\":\"623ff558485dfcac5aadc413e1f3ac00\",\"key\":{\"key_fingerprint\":\"35ba103c2a408932a6308262e7e4f55b7ce7ce86\",\"kid\":\"01015a4b6db023b38532b3ce727a8cd792c29d5c18c003575c1473fe0fa56803c27e0a\"},\"remote_proofs\":[{\"ctime\":1394168677,\"curr\":\"9b428c852545abde5cd877dbf239e925ff36bce41061f6d9e3b340f1b4097d76\",\"etime\":1551848677,\"prev\":\"2c77591e46de42730c3e592a8e09814098be532eb3bb352a691494a6e12a456e\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"coaxmetal\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"5a3491da7f4a2f0c6e608fe032a0b4339440a6784381e917d45d4e9d0c5939820f\",\"sig_type\":2},{\"ctime\":1394169628,\"curr\":\"73469f325615037498b74bb8afa88885c4d8374e32091f10f8b1e959e01b57aa\",\"etime\":1551849628,\"prev\":\"adcd1a9204ca7a6ef171c9e20caaa0ca842a089de94c34b929da52aca9ff0ec3\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"coaxmetal\"},\"proof_type\":3,\"state\":1},\"seqno\":6,\"sig_id\":\"6cfe499b9c4c606ccd47e4a397c3477ef633d733be4a9f38a086d1c096fd2df10f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"6674a35476427f19439d33a1ff75d8a7ae7b8fd630deed2e858b030a0f8440fa\",\"seqno\":16,\"sig_id\":\"c5378dfd58335c8a5d8b6913e613089ae414c09929e7c161e14bcb52fc2a02310f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1411694742,\"expire_in\":157680000,\"prev\":\"a8d6f791591901bd574cc1f9ec97a713f772a59ffceb1de6ef07296b0be7cc8d\",\"seqno\":12,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1411694742,
            "etime": 1569374742,
            "rtime": null,
            "sig_status": 0,
            "prev": "a8d6f791591901bd574cc1f9ec97a713f772a59ffceb1de6ef07296b0be7cc8d",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 13,
            "payload_hash": "0521cd6cb3239aee69eeecb70866c8d88f0079883b2d3db473e16362e182cdf2",
            "sig_id": "96505ee653081f7c2e742283fbf559e13b3e330d943e964d3fc487c3438d07520f",
            "sig_id_short": "llBe5lMIH3wudCKD-_VZ4Ts-Mw2UPpZNP8SH",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowG1VWmIVWUYnilbHLQJNPVHWp5CEEb79mXsR1BIWhAypoGN12+dOc1yZ+496ow2\nIIJLGygotCgpiD9CKotissgxC7QMFcM2aJHSNJsMygUles+daXRKgqDun3u437nP\n+7zP87zvt270tVU11btXLFg0JzR0Vn90ylbN7Vc7lie26LuT+uWJa01De5Y/tZu2\nkNQnLaHbmnKYnhZvXxpsUpcsCaVyWmyHI021ENzY4KR2kmslNA+UGcW11ohboqNX\nxomkpy6HyUFj2t4USh2lNK+RIGkjltQxYwgSzCqliDLwEBX8XSCOBedUYqjaXCxn\nw9gkFcxC6uHXq7zfUjlAGGGvPXXGEketN8zmHIM3gWAVaSSaK8ulkTYIqp1HlBgh\nsUDUyaA94EZkAG5xBS5YGwBPM4yMlQgTSgRxUfnolNQI5S+WQ2lQOdNiWtP2vPms\nZFxL3j5wT105f0p9YUhIwuqSVlPOoJmCazagUFKPGRYEManlcMzW0FXOTFuxnONW\nSEUcKOVWcsO5jpwHpXxAijALNEmF1KD4uV7DDVDcY2qI8T7wgLTg1BNDiYTiFNoD\nC2NkhPlhggovpYmOYapYQNgRSq2MynEqhAzUO4+x0CJoaySj3gMfgjRxoKa0RgsQ\nFLiXQlsxC4WOUrEYQZEFEL0sbat0jihnSiDo3C0ulaAqd4HwAMWk88ozg4MOnpNg\nLJNEIIWkcojGqLiTXkmKsQxCBCYMp1hV8hAGwbnAhAyAd5TCEgCnGjCNCBbwRAyQ\nCOEDsI7WMu8hDBwr+NKeeweaIIqtNQREYIREHWQy1Euub6WfyiA1B9dS8CYzhcfK\nuctDE5UtTbMslJJ/MLaCUsi6O+CI1CVwkOXk4aQcOtuLST0kppw2DaRfSSZilNg6\nxKgB1xh05JDTjgSYLIEYqGS4kC4YRoWiIKEgFDLskDiplEExGYAbLNhTd6UZCsNI\naT5kBrY0aMawD1J5CJ0jOmhNggrOGa9o8JQx65EBFoIHrSAwMHggIg/SUjXMDK7o\nAPigGTFGgzgMsYLxtBqChA0hCEY10GgZxx5xpJlHKFrIPWdYWq0gDcxZwZT712Y0\nw2SCCWFpebgf5eByS/5qhriaGRhf6YYjkVjYOA4SjxysQodguLR0MCiYwn6BDcki\nIZKDuDBynKioJXgGliFn/2ZGY6VOITNpa867w3S3Fo0vNJty83/hRt+s/62Pnnz1\nVboY3IBXXB65fEMZI0zlxBQko6sjLYVCmr/BpVAIPpfDIaPnSjnrnJRISSkoozag\nYDDFHFYVnJPIdHAiEukY7AQKIywjzLbDmiSXO6XAzDTlPqdN7SZbXApJT80T1beO\nqKquqbr+umvyS7GqZuTNf96UMtxwccTRcSvWfrN7zOpzuL9h0vbrzzaU1087NnLu\n+Pt6+5a9OyG7aUz//BMzSd+MlhfjuY5H1516bsfm+Z+c3zp23enZv6/t227HnZm4\n//xdM4oXto3fMOpS7bjPjs3ou3f/5hMbvtwzobFr9YKtu6esPfTh9GmPT73w1Ovz\nXlk5deKuZ4qryk+uaVwyf0p89cAHta9l6cU1k+JPl+74dNPz22rJWxtvnHzgyPdn\nR/+oFu18+oVNt3V9XtV5fO+WMw98sXL93Q8dn3xy0S+H73z26843D07r3/PrnHf2\n/PzywYaNe82W2Qsbe6cfefi7o72Hd95Se2Dm/WMObRv13rwH3++/Z9+sR6rHzv7t\nq5U/vLRs1fjTb3/8bc3CN/bt6v0D\n=/VzW\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":24,\"last_id_change\":1416204797,\"username\":\"alexstamos\"},\"id\":\"f1e335b75a559f55e88de0824b410200\",\"key\":{\"key_fingerprint\":\"85d13a2adde5e09653d2a32741630125b2ff424d\",\"kid\":\"01016d77afc41384e01c233b7f8c53667e3dcd11696e9ba743dd88d2092cdf87ba960a\"},\"remote_proofs\":[{\"ctime\":1403548607,\"curr\":\"5ce25e4e07cd8d4a1e9ed52eab472608078c03ff85c7d873117e66e46a531871\",\"etime\":1561228607,\"prev\":\"39d8da6eb2ea6fe2956dedd8fbb4dd671518d679d5dc012031bba26d7422f9e7\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"alexstamos\"},\"proof_type\":2,\"state\":1},\"seqno\":4,\"sig_id\":\"8746ff71bc043a12549d8c0c9c2e4aa604472a567cea436838d462310ac0688a0f\",\"sig_type\":2},{\"ctime\":1408153795,\"curr\":\"1b3e9441de78d5a5c29e992e8eccad83ed344bd0a9d865e9816dda418d5e7b38\",\"etime\":1565833795,\"prev\":\"fffa05553818fb91161a2208f3e3fb451d05094d00fb1e35417b989ed4cb648c\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"secalex\"},\"proof_type\":6,\"state\":1},\"seqno\":11,\"sig_id\":\"8c2f2b3cac2cd0c599c0e5e97cc231332a34a4f2275140965528f9749dc0c0cb0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"1b3e9441de78d5a5c29e992e8eccad83ed344bd0a9d865e9816dda418d5e7b38\",\"seqno\":11,\"sig_id\":\"8c2f2b3cac2cd0c599c0e5e97cc231332a34a4f2275140965528f9749dc0c0cb0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424898168,\"expire_in\":157680000,\"prev\":\"f7fd588cbcc7708776343be0ea13152ff7fd2f49ec6f27c4c0334dd7feb2c192\",\"seqno\":13,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424898168,
            "etime": 1582578168,
            "rtime": null,
            "sig_status": 0,
            "prev": "f7fd588cbcc7708776343be0ea13152ff7fd2f49ec6f27c4c0334dd7feb2c192",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 14,
            "payload_hash": "c9c48bfe71bb8255b3b58a0fb093b9feadffa5679793ccdd1b87ac519a234c98",
            "sig_id": "0f2649140b094cd58f5e7fc9a01e7a417fbe0b41ea73cd82a33f25447801bd600f",
            "sig_id_short": "DyZJFAsJTNWPXn_JoB56QX--C0Hqc82Coz8l",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGtVVuIVVUYnjEtnRKcKKiQlO1bTbnuF8uiRNOXLLPBmOy4rjPbM845nrNndJhG\nRUEJAufFKBGRLhCCUIQ6JGovXlBBuhAUEWk+mEpDpRBJ1L+O4+SUQYEHNmzWWudb\n3+X//z009bamlubDGzpWPB9eWN188oJtWvqjOjaQ2Yrvz+YMZK47Dz1Feusxq0I2\nJyuHfmvq4dG8MnNNsFlb1hdq9bzSA1uaaiG4scFJ7STXSmgeKDOKa60Rt0RHr4wT\n2WBbgkmgMe/pDLVqLU93ZEjaiCV1zBiCBLNKKaIMvEQFfxeIY8E5lRhu7arUi3Fs\nsgZmKfewepPz5cYGwgh77akzljhqvWE2cQzeBIJVpJForiyXRtogqHYeUWKExAJR\nJ4P2gBuRAbjeBlywNgCeZhgZKxEmlAjiovLRKakRSgfroTbqnCmb7rwniS9qxpWT\nfOCeu3p6y31pzEhM27JuUy9ATMl1GXAI1hgWBDFF9DhMXy96YywSaIMRUQKpEAXD\nhGlFCPI2cmyslpxG2WA06nwya7z70gkrHfc8ECsoF0YTERGLIlBnXdCUSuJdNOPc\nhMiEjtYJ4zQ3VPhoonCWEjAKspaGc81gH0M9MCYiEAskGK0SRwtlYRL3WlhVKUKp\nWqtUItjRAXVX5EkgpmAuI0SDJa63VkvhWuwhGx8kjw5Zp4nCnFmtIINEBQXtDPNO\nBTikNIHMrKUWGUiTeSaBfhgF5xyDRw3wai30JT8jppgxT6K2yiDAdSRahpRyWEjF\nJY7ewwVUSew0FRE7BoQEdpzFEGk2piX529DT6KKu4MolbwpTWllPEY+1U7EmL4pQ\ny/4t1QZEqeivwjppy+qFKRJz2KmH1T2Va4t557W691RjC9VMmIJqBbYeWQ5xBhWU\n5yZS7gMNXFHhCHAOTFmnOIFg4ajQDsXsh2twoxcOtv09Ccr1WBLQnIj6EMAIeIKS\nFiEhOcFRcYSg73xqMm4Yp5Q7LoJIjYQd8jQyKJJ/JNEAH03iFsT8/5LozIuuXvuf\ngqA3C4LeEASoD5hyE6jQMEh4xIhocEkZHGXg1BKYG9xhg6E3EZGeQEYYIcKs0TB0\nIIjxOSxvXFMqTN6dKFdNf3fF+FKXqXfdmiDGRHwy95bKGEzjriFidOrd8MFI5l2v\nLgY1CwOLMaiJtdW8Fkp5OsGlUAh+Y2WBQJXzjQlDtQkBiIFuGL5KCKe8UhEhqZUC\nbp56yyQoEFQAUUWcj+QvqRiuKkwnYILeHlP01kI22PJa8/SJTc0tTbdPmpA+hE0t\nU6Zd/zoePTe56dBdjDy5vNTsw+V7du15ecPDez98ZviBKyNDa1uOfVrefvTA8LG+\nb2duah+a/+XpnUemnKr8bPfvRN+3nj65a8ubZ8nHnTMufXXh7s9WLurcxi8d2HF2\n+6Zdcy9deWjhnTtP3Dtnwdahvt3P3v/FbzM33ldtP3L1l2XllnnZT3vWP7Vw2/ER\n/O78FQc3VJcv2X9odwe6+uCmBWc268dfKi1FZ9ZN3Xf27dl7j4989EHH1lfWbVk/\nccK0Na2VgZFtT1w8987+V636ZsrXG2d/Pnz5YGXwZO+J4WVLnn7k/PDrz3Us3jxp\n8ns7vjv/1uFFvH3W4cfa/5B7FkyY8cbeX6evGNAvnmq9+H7r7/O2zFpcvmPfkjNZ\n+58=\n=64ao\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":13,\"last_id_change\":1416204829,\"username\":\"dstufft\"},\"id\":\"28608ef6412498220dbf51ab9753f700\",\"key\":{\"key_fingerprint\":\"7c6b7c5d5e2b6356a926f04f6e3cbce93372dcfa\",\"kid\":\"010107b69fbc6ac95a36dfaf6cb320f08ac7a5594fbc1759446f412e2ea98f641b990a\"},\"remote_proofs\":[{\"ctime\":1394142293,\"curr\":\"bb1d639de75fc0bc928154b980ab69fb0e9ca4dc8e39d892f80bb3b0acd04d47\",\"etime\":1551822293,\"prev\":\"df13144d2f9b8a054bc2fb4088c1678571fdde9c3871c936f1c4bb161c54fef3\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"dstufft\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"d391beed2487e9857d0b55e2e8e8d5af35de3e5836c2161e48bc85233757d69c0f\",\"sig_type\":2},{\"ctime\":1394142359,\"curr\":\"82803deec93ec9e87b0067521f8500371d2c3b5a45335c56e616031c0d3f46cb\",\"etime\":1551822359,\"prev\":\"bb1d639de75fc0bc928154b980ab69fb0e9ca4dc8e39d892f80bb3b0acd04d47\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"dstufft\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"003e135ae369e635f102993e8a1f7e53b2e1d5c1a11ab027d22e810024ba9d030f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"82803deec93ec9e87b0067521f8500371d2c3b5a45335c56e616031c0d3f46cb\",\"seqno\":3,\"sig_id\":\"003e135ae369e635f102993e8a1f7e53b2e1d5c1a11ab027d22e810024ba9d030f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424898244,\"expire_in\":157680000,\"prev\":\"0521cd6cb3239aee69eeecb70866c8d88f0079883b2d3db473e16362e182cdf2\",\"seqno\":14,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424898244,
            "etime": 1582578244,
            "rtime": null,
            "sig_status": 0,
            "prev": "0521cd6cb3239aee69eeecb70866c8d88f0079883b2d3db473e16362e182cdf2",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 15,
            "payload_hash": "d17b636b8e47993aa99a850e3140bb3c80f11f80c6517d8db5e225c747489d62",
            "sig_id": "20447ba9569d3e9cfb833be429b18afbac5561cc3916b4d35c43df77b7b1126a0f",
            "sig_id_short": "IER7qVadPpz7gzvkKbGK-6xVYcw5FrTTXEPf",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowF9VFuIVVUYnvESOaiURldJ2YGITbbulykSmiJLLUIfBJMz6+psZ+ac8Zw92iBD\nYb4olUJREVGJBGZKl4cke9DKQTEfmiDTyWq0G5V4QUl0XvrXmXFwINoPey/22vv7\nv+/7v39tmzaxoalx//Mr254Oy9Y2fv2XbVh+Vh3fkNmK781aNmSuMw/lIq3Kpitk\nLVlH6LWmFhbklTnrg82as3WhWssrZdjSVAvBjQ1Oaie5VkLzQJlRXGuNuCU6emWc\nyPqaE0wCjXl5dah2V/NUI0PSRiypY8YQJJhVShFlYBEV/C4Qx4JzKjFUba/UinFs\nsjpmKffw9j++76hvIIyw1546Y4mj1htmE8fgTSBYRRqJ5spyaaQNgmrnESVGSCwQ\ndTJoD7gRGYDrqcMFawPgaYaRsRJhQokgLiofnZIaofRhLVRHnTMdpjMvJ/FF1biO\nJB+4566WVrkvjRmJcXPWaWoFiCm5dgMOwTuqFUMCYT4O0+drOmoJrqfTJOQRWpII\n66yKKDIO3BChmhlECMLOiTqtUfuTY+NboDRjPriAsXQaae85QYopwixG1joDHmAn\nmRxnKQ3M6qCER9wzI8Ai7R1FDoozhBgPkTokebockInCYmirEdRGsJ17VOdeDV2V\nIpS6q5VKBE9WQviKvGtEOyNYYAm+uJ5qNUkkAVhRj7nEWtvAsLYxKuipgwbg6DHm\n0VKNCbfK40SOR66Rgn6SaDjQD6PgnGOlR8C7q2Fd8iBg43VUQXCCLfzPISNGOuUp\nCfCAFPtILYbKCG6BC0wkjUxGFyNBCXxUS/K3rqc+Su3BdZS8KUxpTS31eWymivV5\nUYRq9r+treOUit5u2CTNWa0wRaIPO7WwtlzJWiis8tWl0WQy75gWSgN3KaLmKCpG\nCaMKHHOSGOMUVYFRGoPm1mCpCRHIHDigUMAOQ3ayEbzRin2r6oVKhck7E/Vu09tZ\nMb7UbmrtUJCBvSJ46zVx3hqkDRJaCkuIAkdCVDQqrOBJrBIGcgTtl9Jb6iEuHuls\nTEZK/5gOrwRTyDMhoBkMxpVAGQMwBmPovgPvg8Oew8Cl+BFOGadKYikYc0gnFX1p\n4OoaRufuuiMruXctZIwwpRURUD48251XQylPX4B7CsE1lg6nHVM2BpmiQTi31HJl\nULRIU6tjMD5CwITUUlPnvMdWSeM41oZQ5rS6TinMcmFWAyboLZuipxqyvqbNjbMm\nNTQ2NdwweUI6ihuaptx07XzuX3Bjw74tL6uPBnqmf3lLdefeO2ee2H5m0zsvFa/M\nOvLJ61/9cn7rradmDF+8dOHKxXO3HZ7/8NQfp33/4Xt237Ydrf70QPj50saDT36x\ns+u7E0OfHz/wwbsLt29+e+Vw23NzTx566I4jZ97sf6NtRe+itnL/g0/sPnV464Th\n4Y1X588cbPz7rt8Glz8+7+g3i3+aMVUe2nFwqOne/XNaj3169+UXHh3o/PbsDze3\nlzc/1v3xPasGiv5Th1oXnivP/fOPrsVLe47u3nTs18kbBrbNPnl6+arWt+bO27Jg\nyqsLf182OPDa1aGjK6Z8dmFvw65HLmxcOnfSLnJ54H5537Tpsncwdix5YKJds6gY\natly+/k9z7z4/p4ri5+aveSffwE=\n=SdW+\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":11,\"last_id_change\":1398406015,\"username\":\"dijkstracula\"},\"id\":\"e726bcb8f0f4541002394a02201cc600\",\"key\":{\"key_fingerprint\":\"8944dece117c909dd52084824b10bbca9df1c747\",\"kid\":\"01013e4b9e86d05d4a6f0a9dc30c54140045ef3c0755555c023f6b18a6a63bf2185d0a\"},\"remote_proofs\":[{\"ctime\":1394216171,\"curr\":\"e2e8483d157199be419bff8c3bc2cf1fd115fb39125b8d19e865f590832a2fa5\",\"etime\":1551896171,\"prev\":\"8e1ad9f8e6521bb8d595ea7c8d32e7c8fd8df3b11990119e561273f47fcff205\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"dijkstracula\"},\"proof_type\":2,\"state\":1},\"seqno\":3,\"sig_id\":\"eb4dc4968959576f950f8432438571c72aac838e433fe95ba1792260a80e1c1f0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"45f56edbd92cdba09a06976b228205ef83f818ef82b86aca96a677db3d6f0d09\",\"seqno\":11,\"sig_id\":\"d86480d466e654a4b2f56aef8a113d1ce56ec1d5fc886d02534538717644c09f0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424898261,\"expire_in\":157680000,\"prev\":\"c9c48bfe71bb8255b3b58a0fb093b9feadffa5679793ccdd1b87ac519a234c98\",\"seqno\":15,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424898261,
            "etime": 1582578261,
            "rtime": null,
            "sig_status": 0,
            "prev": "c9c48bfe71bb8255b3b58a0fb093b9feadffa5679793ccdd1b87ac519a234c98",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 16,
            "payload_hash": "89a241dc68e2a9a74aeb144c704585282d7ff2f1cadfbe4838d269f6b738df34",
            "sig_id": "5e647529aa812916a15a2bbfd26d9e99af30db81a9884ef44bb8099369a4309c0f",
            "sig_id_short": "XmR1KaqBKRahWiu_0m2ema8w24GpiE70S7gJ",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGtVVuIVVUYnslSnG7SBbUp0V0PpoOz7pcRDCwfskIjM8TstK4zu5k553jOPtpk\noyX1IBKK1VOhiC+RGSVeICmdCiaKgvHFrlSOmkrYg6bRxfr3mXFy0KCg87Q4a63v\n//7v+9a/N147pqGpcf8zSx9/MDy0vPGTk7Zh0Sk1uCqxJd+TtK1KXFcailm+Kpru\nkLQlnaHHmmqYlZamrgw2aUlWhEo1LRVhS1MtBDc2OKmd5FoJzQNlRnGtNeKW6OiV\ncSLpbclhctCYFttDpVxJ8xoJkjZiSR0zhiDBrFKKKAOLqOC6QBwLzqnEULWjVM1G\nsUnqmIXUw7+XOd9Z30AYYa89dcYSR603zOYcgzeBYBVpJJory6WRNgiqnUeUGCGx\nQNTJoD3gRmQArlaHC9YGwNMMI2MlwoQSQVxUPjolNUL5wWqoDCtnOk1XWsybzyrG\ndebtA/fUVfNV6gsjQvKWpMtUM+il4DoMCJS0Yao5KIq4GAVZKRV9Cvdy0DojDwZY\n5h0jXoYgAkPa26BtzsoNMRpWPhdrtPrUKxBEgSw6gJMER2+dZpogb6RWznFknaF8\nlJoxBmFBN4o1DVZqyRXFElkZLBYCe4RY5FxHzIQl3qKANGitmAZmPFgEagL3Sugu\nZaFQrpRKEeRYCrnL0u6RvpXGoImrVSpQVTAFVXzgVDNquDeI8mhMVCQ65gIKVjgS\nopXKBMytIQIzyq0WgIOdcUA/DINDOBgdAh9mkKtSZ1HPfkdwnQVvMlN4opobM/II\nspVploHs/2hGHaOQ9ZRhg7Qk1cxkeUHYqYblxRIsYZW2D8U1YMJo1FZT7CySwhsu\nrbGIaY+ZseApE+ChIkJKMNFRCKXhDsIQGQEZUEyG0Ibr9bZcop8kI/opHjiDi44h\nBUYpFzS24DdCQchwAgnrcdDwcKTHmAsUwDWEVYBnLLD2RF0qYI5eroQV/487/82K\n9jTrqNl/5wS9nBPkIieIdohoFaKB5x8jtQzirX1wggoVWPDRc0tpCAHLGBzoBtMO\n3o30CiaBkZc4saxeppCZtCvnXDY9XSXjCx2m2gHlmLKEI86IsdY5ywKLSEZMIhhu\nsXMmCqUsBMNKJwlTnMBsijCPlJWMydyJ4SboRU0YZVj0lMM9bCFW0junLafeURQU\nIUh561HEkVOCIXTeKo8Vg64wjWI3JtBFbz6l6j0MD6uL5nyu3YV4MWClFZFQPzxZ\nTiuhkOYnuBQKwW8kFh5LCxpaEFFqTY3RGvKEAsUMWUsdzFWMYbo6wTGI6S0PhHAn\nmQR8L8jfnWKYgJlpB0xouGiyWiUkvU3rGm+9sqGxqWHsVVfk36+GpvETLnzUBheM\n+9MmCz5Yv0Rs31t57ba1h358t3nDPbV5026eff+i9mObvn3qpVe+mZw++vyM6TuO\nzVy8rGnObLf56rvXbv/18MGJY97qHHh/9Z1q7FT18/ls44Q5Lw8+vW7f7ob5B/ua\n2e7bJ33xR+u+7Y2DC7/etebZZdcdvW/cuJ5JMzdM3hYWfzjYv5P/dO+88uEbbzjU\n9RV7AZ04M/2WuZvWvNmK7trzwMmz3285/duX036/ae+pcw8np2t95daxS/o/LU5p\n/ug98Wp/47nG1jeOnEUvfn70sx/6dh0fWHhs7s5Zq2z/tvVba82rx5z5+LmJc8+2\nHR/Ysmf8kZV3vLN38cCUGefb13w358COX5ofeX38jL7z86+/5rED+9/u3vwX\n=1fCS\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":5,\"last_id_change\":1395759056,\"username\":\"rondiver\"},\"id\":\"d665b4dc42d7ee6e409dbe9b2cf8c900\",\"key\":{\"key_fingerprint\":\"3d8da483bd9e39621fdbc94920da798cc50bca35\",\"kid\":\"0101ffe6b3f23193eb797583170b7eb1661d004f559f146b2db0e09ae28494095eb00a\"},\"remote_proofs\":[{\"ctime\":1395758915,\"curr\":\"648975de53943a5da035faaf82fc4ce0eb6c2efb78ae15ba261435b967581cac\",\"etime\":1553438915,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"rondiver\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"e1243f9b931cb076da57bab049d14abd6646e9b82677c90c332aa5c056f425390f\",\"sig_type\":2},{\"ctime\":1395758972,\"curr\":\"85e54f42c40870b8ce91b39600e67e06bd1e96557d11560edb0018ea85619d28\",\"etime\":1553438972,\"prev\":\"648975de53943a5da035faaf82fc4ce0eb6c2efb78ae15ba261435b967581cac\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"rondiver\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"29c0298efae63ff3b4fe69dec6368e4edfd5b33eee17fec1b3abe3d87d8be1a70f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"48b250542abbccb4e4f07f12f646b1ccaf688b9b9b7c724852b57f6718b74478\",\"seqno\":3,\"sig_id\":\"a8a4fd359b91bf9b7dcc9b53dc30e82208dbd0f1f5321b93db8d1843ee13f6120f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424898273,\"expire_in\":157680000,\"prev\":\"d17b636b8e47993aa99a850e3140bb3c80f11f80c6517d8db5e225c747489d62\",\"seqno\":16,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424898273,
            "etime": 1582578273,
            "rtime": null,
            "sig_status": 0,
            "prev": "d17b636b8e47993aa99a850e3140bb3c80f11f80c6517d8db5e225c747489d62",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 17,
            "payload_hash": "e463e786a6fe37c0ae5108279292ddb5327fe9e2a76e851a618e223e883d0d93",
            "sig_id": "99ef2973d3f58d8f99bcc761357f34c566296169404f41ff461699f3cd74f18d0f",
            "sig_id_short": "me8pc9P1jY-ZvMdhNX80xWYpYWlAT0H_RhaZ",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGtVV2IVVUYnbGUaczwxYfAhzzli0y5/38MhiLC9KGUpESR6/6dOc2de6/3nhkd\ndKqpCPtlCpQIShwT8SEIszQMDa2MCTQ0KxJDy4ykYmiotMzpO9dxmEmDgu7Tvvvs\ns771rW/tdfqnXdPQ3Ljv0eUrF4f7VzV+fNY2LPlJ/bg2sWXfk8xbm7hiGkpZviqZ\nzpDMSzpCjzW1cFtavml1sElL0h2qtbRcgkeaaiG4scFJ7STXSmgeKDOKa60Rt0RH\nr4wTSW9LDpODxrTUFqqVaprXSJC0EUvqmDEECWaVUkQZWEQFrwvEseCcSgxV28u1\nbAKbpI5ZSD3sXuV8R/0Bwgh77akzljhqvWE25xi8CQSrSCPRXFkujbRBUO08osQI\niQWiTgbtATciA3BddbhgbQA8zTAyViJMKBHEReWjU1IjlB+sheqocqbDFNNS3nxW\nNa4jbx+4p66Wr1JfGBMS45akaGoZNFNw7QYUgj2GNNKcCj0Bs7OrmKVtxRz0UudM\nOMqF1QYpx7AJllulsURSKyLrjEaVz8WaqD6TVpIoQtROE66EcBxzLaApT6wH4bCV\nMjA5QU3hIyExIPhFR1gINFfcexEEjJor7hUXSEpmcYgeY0Mxx1qTiBBFQXhQE7hX\nQ2c5C4VKtVyOIMdy8F2W5g1iqhlhQgrekriuahWqCuuggMBRGMY0CO4jchF6NYyb\nwAVhXjBNmeTYczBfMNRjA4cIQ9SGAPTDKDgHKuQS+CiDXJU6i7r324PrKHiTmcJD\ntXwwY5cgW51mWagm/zSLOkQh66nAPmlJapnJ8nrwpBZWlcqwhFXaNupWLSIYkHFE\nYFIKGxYwbFgDWglLKTg5KAtXQ0UkFOYErGg4oeA4A3NmKCaX0Ebr9bb8XT5F1Zh8\nmHLGOEWKaCKNklJQg62B28CJRv575DFS1mpGnRIWSeekAltwD38VJ9FeoV8dvVIN\n3f/PcP7bJNrSrL3L/qtB0KsNgowbhAZVLWaYSUxZEFwZiyHKokORMO0tjjqClRV1\niBrjMex7B/s0amYwVlcMYkW9TCEzaTGnXDE9xbLxhXZTa8/LYSSVUxjCEjOnIHgM\n83BfDFMC7p0jGFMTiUcSWQuDpDESwVCAqJLcy7zpy25C47oQEphyCdcrGsJijBZ7\nxaiCUKWBEIOhIQgxSaivI0KqwMAookJqHNxOuMhJbx5S9SZGs2pczOfiXbYXmEtB\nsChQMayppNVQSPMTXAqVJ8KYLZQGKtg7oQIBz0oGwYQZcxIxiAh430ugErEDK9jA\nwFGeCB2FlbCKlI1rVQIz0waY0HDJZF3VkPQ2P9U489qGxuaGKZMn5Z+vhubrpl/+\npm1+qalh/8L+DSuX3nXxkerA9mfndO7qPrDlxo2zPvzk+oH03CuHa9+d2vHR8PGm\n6cWB1hmzT/z61vC2NV3fvDry3paTxzf+smzVlAWd9wwsefKFkW23Dp7aOXRs/5sr\nTh5qfXvqujMz37lw+1D57l0Ne97t29TH5t7bGt2+rdNOZ3P/mLz57GDp88c3ncIv\nH9n6Yuu54ae33vHw8/3nZvcvvPmxSZUTi35+f9bM+U3d+0/3Dd83tGfHwjliyZdN\nR2754tPPlu34/bW2qeuXH+3tazmDD8rzT6i2B18fGFm3+NsLa4YGZ6WLigcu7j58\nrG/pjG1v/LB+Q9z+3N7f9OAHD3w1svvrXQsOng8Hnxk0R09U5sy/Ye+dfw4NHfoL\n=Co9J\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":11,\"last_id_change\":1409095369,\"username\":\"multigl\"},\"id\":\"b46c356b9a08c41aeb5b891707982700\",\"key\":{\"key_fingerprint\":\"47b72f6ef9c925866c51596232d2bd01d1b77e47\",\"kid\":\"01016df22fe0000fc24ee35537dd6e68ac585d8560774b1efd11a3151992f0030e6d0a\"},\"remote_proofs\":[{\"ctime\":1394246765,\"curr\":\"6bc37d61f6a449262df0cf41aa45ae5624d64934751d54a8ea3d1adf02403bee\",\"etime\":1551926765,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"multigl\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"b96fdae450291781a4e1fdabaa316b335b6e8b7bf8f068152c7ea523b70a9a040f\",\"sig_type\":2},{\"ctime\":1394246838,\"curr\":\"1354453082927a87763a1bac3b5290d0d108bb943c86b07cc7872f5dc86852fb\",\"etime\":1551926838,\"prev\":\"6bc37d61f6a449262df0cf41aa45ae5624d64934751d54a8ea3d1adf02403bee\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"multigl\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"9a52b14147134e658ab1c79fc0f249db1f9f07783c03aad1c0fdc9db3f94a1180f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"91078c8179c14c858ba4d5d8a486596c2113af2d070bb3943ff2640ee2175d7b\",\"seqno\":10,\"sig_id\":\"67db157030fa24fffb1d84384aa3e22a19fcf0a723d3ff26095d613036791ec00f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424898282,\"expire_in\":157680000,\"prev\":\"89a241dc68e2a9a74aeb144c704585282d7ff2f1cadfbe4838d269f6b738df34\",\"seqno\":17,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424898282,
            "etime": 1582578282,
            "rtime": null,
            "sig_status": 0,
            "prev": "89a241dc68e2a9a74aeb144c704585282d7ff2f1cadfbe4838d269f6b738df34",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 18,
            "payload_hash": "63a81ade57cb9d522845f6b4ecbdf0a70ded314cf4868eaca28b5d1f88397262",
            "sig_id": "c593d6ee7b05a1290d16fa2b4a8a581bf4f2ae286829eefda632707c63db4b7d0f",
            "sig_id_short": "xZPW7nsFoSkNFvorSopYG_TyrihoKe79pjJw",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGtVlloXFUYTmJTa8AqioJajVwwDxLTsy95UFGRui8VXEodz/KfzJjJzHTmJmmU\nQIWCLWgREaSIVmweVLT6pKiVPlhxBaWCuODSKFSJGoRalfrgfydjTBdQwYdhDufc\n+53/+75/uQ+dfEJXX/eeTevuugnWbuh+b8533fKT+eXezNfjVDZ8bxaqFajlxarm\nxiAbzkZhyrsWDFXq50+CzwazCWi2KvUaHllulZLOQ9A2aGmNshK4cEZaa4n0zKZo\nXFDZ9GABU4CmSm0Emo1mpbgjI9onqnkQzjGihDfGMONwkQy+roikSkquKd5arrfy\nI6LJ2pilSsTd4zw/2j4glNBoIw/Os8B9dMIXMUJ0wKhJPDErjZfaaQ+K2xAJZ05p\nqggPGmxE3EQcwo234cB7QDwrKHFeE8o4UywkE1Mw2hJSPNiCZkc5N+qqlVpBPm+6\nMFrQx9groVWsKrG0KCQlg1nVtXIkUwplhwrhnqCSc0aoPQJzrFbPC8R2OMrwGLww\nCVA17gse0VJGKY1Ei3Y4HdkLpY6SPnJjEgVURaAYFiQYRzXoCFIGQSRIJCWPlNLi\noxCIIsZzzRUwFI37IBPjzvEYvWROUC0VlSwwh3oG8BYRvbYR40MpMfYmjNVzKDWa\n9XpCLdZh0uWVsUXOVBPkHMabTbxVCgLMJalTAJeScsEGCSL6gDxFNNQngsFT72Oy\nLEmZZEpWkyCj8xgghg8dcKk5oQvgjSZMILhGETgS55ErjYQsF1wEAjFEBpQ6RZTC\nv0App8pQzRgiU5qYI1wlbbNFLoW+bT7tEipDGC1Fl7vS3a3C38VayicreQ7N7LiW\ntt8v5VMN3GSDWSt3eRE2nrRgQ62+sFkZWch4SSO1OnhUK4jIbXSEKJp0VCG6FG2C\nSLUF602y6I3Dn1BcM2UhgNaOpGwBrXPf9ODRLhjCFl1QoIB4ZWSQNkoU13MmeNI6\nOSEgRa9UMjolSRNarkliNLlEQlKGgDf8GBfa4B0X/geL/5sLI5W8PO7/2QR+PBP4\nEhN4EtjkmMdMoQ4Mw1IwkmlCsKepGEGhFAK7jmGUm+RIsN5xLXVkLnLm7T+bgJ1m\n0QSnuBKYrYivse5wwbnWihKDfddawwhIzSTFtaaMaGpsUZmEOSxEHt0xJrTBOyb8\nDw7/SxOKRt4RfXJycqgQfqgGedbWPq+HehVPynneaA0fbQglhBzPE7G0MIzXMWJg\nniQOhntrA3Ea7aGOsKK7R2zsIRirOOYXjgUSUVZGkAnOoXSMJ+vb15RyV6kW4Tfc\nVLXuYqnsWuXCFAuFEpRZxkHhaKBEahMsJ9hCSFDOy0CMZkKhc5ILHZig4KhzYB2O\no+zv6rZLWBhhkxUiBMaj1yYab3A6RtRd4XhNXnLkWaQZGmkEJh+J6FQwOEajZhqn\nVjZdzJ02h874WTK5C+06ibaVCiYMZo9FaWFjo9KEUqXWzhG0ti14J0UAGwhoo5xK\nwDWKChJzj2mL3IvWz5lOgHI4rQAHB9aEAYaqGJxSJFr+N1VqMDQ3gphIuOby8SZk\n031bu1ct6+ru61re21N8knT1nXTqX98pYvOKrh1nnD20+6zXdh8uz577vb/yy5vv\n+6HW89W+ZR9MzuzadPIN608Z+nTLrlWHvnvwp8uu3snXv/nI5HXzTy975qwHLn7p\nrQs///H1y4fv2b8xDvw+v/2UP27UGwfihm/ePu+Q71lzfz7bn9S1/ees3vbB3BMX\nX7Vl30z/Zf3XtK5f+fLcmjPnX9mzcvb5g5fMQONg6fS56sHe8+t3Dfz6xRUT+z57\nccUbVz/Xc+Cib/d/0v8aefzAVwPbNves2hN2jO+484T9vffdvn17vdq3bfaR5R/N\nPTr6xc4TV3/upyY25+tXvPrzrjsu6Pnxt5mn1Gk3HX6s94kDz5I/5vfyCwbeeeHj\nJ3/e23vru8uzF9ZeeN2l5ZkPv374tvd3bvoT\n=4ya7\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":10,\"last_id_change\":1415332019,\"username\":\"mnot\"},\"id\":\"683dcb48fe28a3bb57ad912111d07400\",\"key\":{\"key_fingerprint\":\"0d388f1ec3b4ae29e5e8a17e7de55c405e5fc851\",\"kid\":\"010194aeec0608b3736e2e633bc5f23aa3ddb52a41756152c2a9cdceb9de5b79dbb50a\"},\"remote_proofs\":[{\"ctime\":1415331709,\"curr\":\"540e2af57fceaff6ac9c5e4dbc1214d81bf029e1bbdf92f55f5ff970c5dab6e2\",\"etime\":1573011709,\"prev\":\"70d33e5e3d36737393434c0edcd2e11a606611ac113168172297011f2a036f79\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"mnot\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"51d197cb317c4d39da0061f7d6cdafd9fed179e9b8f93aaa3aa4637269ece77a0f\",\"sig_type\":2},{\"ctime\":1415331802,\"curr\":\"6e6e0b685c59d50c5b3243f77fa44efdb66f87ff51fcdc70f21faf0cf680eb83\",\"etime\":1573011802,\"prev\":\"540e2af57fceaff6ac9c5e4dbc1214d81bf029e1bbdf92f55f5ff970c5dab6e2\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"mnot\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"3f45b22b1a61ae823bc8527001736dde6c7043bd82138fa0c9ba3757d2ad32b90f\",\"sig_type\":2},{\"ctime\":1415331941,\"curr\":\"a6364e5e7007e635e73377610886999820e5725199971207189c5f202aa9c3da\",\"etime\":1573011941,\"prev\":\"6e6e0b685c59d50c5b3243f77fa44efdb66f87ff51fcdc70f21faf0cf680eb83\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"www.mnot.net\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":4,\"sig_id\":\"58b7dd0cfb0f3e83b99c0a7ae81a0232a6d7e9cc8963121b2c0d64e20f68c4af0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"a9e21fa12923e6d9910578c930d2e0c6ab5c087246e5e5347c241ea1aae9ada4\",\"seqno\":29,\"sig_id\":\"849f944cc23db78d8b84b8d70f64a8fb5358b0173730841ae0d7fac8a20d72780f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424898290,\"expire_in\":157680000,\"prev\":\"e463e786a6fe37c0ae5108279292ddb5327fe9e2a76e851a618e223e883d0d93\",\"seqno\":18,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424898290,
            "etime": 1582578290,
            "rtime": null,
            "sig_status": 0,
            "prev": "e463e786a6fe37c0ae5108279292ddb5327fe9e2a76e851a618e223e883d0d93",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 19,
            "payload_hash": "c17a8f34177819e167e8aa727d8e391dbaadb33d2c58d8bdfe3a244ba4340828",
            "sig_id": "da45313ac0c2eaa0bc3669e3829b2b24b6f446e415d5652a31cfb054be1030760f",
            "sig_id_short": "2kUxOsDC6qC8NmnjgpsrJLb0RuQV1WUqMc-w",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGVVWuIVVUUnjEzm1BMMRRM8tCk1jTt98MKC1FDjMiRTJy87ued04z3Xs89o11s\nYEhUykIjCCWISBQCpShMMUvqRy8pE4oMo18WZUyvHyVI0Tp3xmGm7EfcP4u99/3W\nWt/3rXX2TLmqpa315OC6DQ+Grk2tpy7YltU/qUtbE1v1jWTR1sT1paGSF1HFbAzJ\noqQ3NKyph860etOWYJOOZHPI6mm1AleaaiG4scFJ7STXSmgeKDOKa60Rt0RHr4wT\nyUBHAVOAxrRSDlktS4scCZI2YkkdM4YgwaxSiigDQVTwd4E4FpxTiSFrT7Wej6sm\naWKWUg+nV3jf27xAGGGvPXXGEketN8wWNQZvAsEq0kg0V5ZLI20QVDuPKDFCYoGo\nk0F7wI3IAFx/Ey5YGwBPM4yMlQgTSgRxUfnolNQIFQ/rIRthzvSavrRSNJ9nxvUW\n7UPtqasXUepLo0Ri0ZH0mXoOzZRcjwGG4IwhRiWVEo3DtC5r1PICs1mQ8pIIQzh2\nQaOgVeQuascZpU4g1SxohPiCq/HkWx+k50hRaykP3kkjhPVaUqVcIKCdcVK5KMeR\naSK8c9LrSATTOFrLonMU8wjHwTPNACFyEa0GahmEOsiItCSURm0LMqH2LGys5qFU\ny6rVCGysA9vladEfBkcRJhjjHYnrzzLISrhhLiLiqbQMyGbIkSgVcoqyaJCkIK+V\nnjIbEHPeUWGEDz4KQbzEhEH5YQQcvKHJMPhIBQUrzSqa1u8JrrfkTW5Kj9YLXUZn\nIN+S5nnIkv+QoolQyhs1OCYdST03eZEObuphU6UKIURpedirWipWDIq0nipJhYxM\nYEwsKzpxDkQBq3KPHNJCRUyjQxTr4scodgoMmQyjjeQb6BjHHqhHMB1lj4ZgotdG\nK4lkCAhZDjIq4AzI0ZzzHxgmILgx2issDQo+aCywpioQcFAYRx9DYhi9loXNBQXg\nfMWs58SawLTDmnigt5CLwmKAiYMbAtLAgDPvqfAYFbaSmsMACZX8XyHKad7Tb8fr\n4FOYLh/6clP/pxj0imKIMWpEHQwJjmgXjRZOwHRbo4FsDYtI08gtjho7jSjMVwxK\nUcQJtVg465U3+l9qPNLMU8pN2lfUXTONvqrxpR5T74F0jPqIQF+kiHHMccOD8Z4J\nCfPmpYoBecQdEB88ETCZII7DXojokdTRFYvtchdyTBfKcfAMaI/gD7BJORUYdIF6\njdMGdJbGROWUtTgEHSjIFRWPRXv8qIZFDW0MFHuq2cTIuhqz6Qv2LnuMEaa0IhpY\nDI/V0iyU0uIFlwL2DUKj3hDUKGx84NJZDQ4hivEoLAvAHKwBiWBIKYbRhmlQwThD\nYBF7HIFi2BaCjGlVQ2WmDJjQcMXk/VlIBtqebJ0zsaW1rWXS1ROKL1hL27XTLn/W\njh2e3LL/xuWrfr/tpdLTc7+btmb52a5N761c9tT69beuvG/b0PMLyFDb7NVfPGG+\nH1xbP/nxhHlLLk7qap9A3743+/OX9z8YWnjhzVfcA8eX9t/V9+K+x+nnP849fEP5\noRk/751ZPt+z7+tzz7RPPtYol05suLvzuamnPppTujh/+tEds89UJi+duHPviTvn\n/frpNr17w5kdK47lk7oPLpt9/1+btw/OH1qwfdYfB3YNdk999/SZtYOd7edvP3kN\nWdl/yXXes3bV7qnfnrvluheW1GYsO3xozZJWObDimw9PDzXeOb7mq7d2Hliw7csj\nv8k7zu46+EnXyw8vvnlOubv7tZ3XV80e9Nn+6QtfbW88e2nmlMX5G0cOzZr/+sS/\nAQ==\n=8nRc\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":16,\"last_id_change\":1404373770,\"username\":\"bcrypt\"},\"id\":\"8d726a251ce90e98f5cf9c5433c60800\",\"key\":{\"key_fingerprint\":\"bde7d5083bb35edc7a66bd97388ce229fac78cf7\",\"kid\":\"0101afc7ac7d9f26491fbb4fcc315f7aced494d97f56fb929547f59e7f097233f9bf0a\"},\"remote_proofs\":[{\"ctime\":1396246445,\"curr\":\"25a4cf02d37b42cf40c2f780c834fa0739d3b7d34be04cdc36a6dedf662d7124\",\"etime\":1553926445,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"bcrypt\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"978486957bd387367f46112b4739dccde7b2c5d0c0968f13fc03191919431c880f\",\"sig_type\":2},{\"ctime\":1396388213,\"curr\":\"3eeafd9a98707ee00b501089d3d71955541288caa9d817a0ede9161938e2080e\",\"etime\":1554068213,\"prev\":\"b8df84bd52bae49c192d445cf02365ac3b52b234b4aa4dd36d10fac77950ab68\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"diracdeltas\"},\"proof_type\":3,\"state\":1},\"seqno\":16,\"sig_id\":\"f9ea2ec29cfa96c61d9ba9919928a93f5b1f91c90390efe8830523b16cbd8da90f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"43df0b2c082ac4c5a5eadd467229d78fe0d05c208ed2683b412c1d66fd079fc1\",\"seqno\":17,\"sig_id\":\"8c52c56380412206536184bfe8ac9a1957aaf8c8bb1ee9e32baf85f3f5b595b20f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424898296,\"expire_in\":157680000,\"prev\":\"63a81ade57cb9d522845f6b4ecbdf0a70ded314cf4868eaca28b5d1f88397262\",\"seqno\":19,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424898296,
            "etime": 1582578296,
            "rtime": null,
            "sig_status": 0,
            "prev": "63a81ade57cb9d522845f6b4ecbdf0a70ded314cf4868eaca28b5d1f88397262",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 20,
            "payload_hash": "11df754f5792a2e1fca7358791f7f3ad2c5ef2adb2696af84f3743d174c27b34",
            "sig_id": "1f2c30069e6632bf697a4729aa2bb0538e475444d9f6eec15a1a395e131468630f",
            "sig_id_short": "HywwBp5mMr9pekcpqiuwU45HVETZ9u7BWho5",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGtVVuIVVUYnmk0aaKSSFHIrC10c9B1v2gPapFYEEhRkOTxX7c5x5k5+8w5e7TR\nkYqiK0RaEphFRIJdSAfDMIh6ySQxwgdNitCHKIUwkEilqH+fOdXYBXrobe+z9vnW\nd1v/eu6ynq7e7g8fXrVmZbx7uPvgKdd1z+l6vjFzeRjNFm3M/GAt1ovyqQ5DMVuU\nDcRRB624oJZfuz66rC9bF5utWl7HJcutUhJc9Np6La1RVkYuwEhrLZGO2RQMeJVt\n6ithStBUq/fHZqNZK/fIiHaJau4FACNKOGMMM4APyeDfFZFUSck1xV2reau4gE3W\nxqzUAv76D98PtBcIJTTYwD045rkLIFzJMQaIjJrEE7PSOKlBu6i49YFwBkpTRbjX\n0QbETQQQbqQNF52LiGcFJeA0oYwzxXwyIXmjLSHlh63Y7DgHAzBYq5fiiyb4gVI+\ncq/5VvlUC5U/jKS8LxuEVoFiKr4K6BD+JpjQUlBGL8Dsb/QXeT7YKlHblJAHt5Fp\nrUKIJOlkU1I0Bsq1iK5NqWN96daF9htMyyRliVBOCE+jt8lhJoilTSKSkECY8mKS\nnZTaCFKpZEySQhj8GDly6a2KQhJtg0jKKJOEY4rS4MCFaLlyRsugGCRAO5F7Mw7l\nRaw0mnme0I9VWLyiNjShm1AhCEHdfqTZLNNVXEYffFAE0QFJ0sQ1AaToqKBAdcSS\ncUut8FEzrFOgyN5FQCbWlJ2NHXCptWET4I1mXIfgwgQdubZOJKkVtUxbjjW02num\nRXKo0oZADSeMQsR+EWBGg2SOW9BQVq2jpfS3rad9jKrRD1QCFFBZ2yoz3thu8F9S\nXJA3+7OSSl7kPh/ElWpRNFqLSofaUJVitFHyJoT0Za0CivIFF1txuJ5nixg+1fon\nzkDCdkdjvAlCoDEsGQeAcYJmAbDM1DOaPLFKmgQYFcqjStGTREoB0XlHUjYBN7En\n29T311CENX+EwikPmgYRZCIaj7kH4ssm8WRNwMzRzmANni/FmaTce6osWOa0EcE5\n6/8WShu8E8r/kPh/DKUTSLG+VhSxmf3rWZucBvunKPikKMpTxJxLjoMTgIQZvhIu\nAWkKcM6LiO3CqeGk9wSCDkQrxQQEghFR+FsSD7S3qRRQGyxJN2B0MIdQqUKritvR\nANIJRoU3KQpnOYKBN+gOVZgcEwlth+B0YMClJQEL7jElyhxz6aM7oRwSHRlykgzs\nSOQpWEwvMOaBWU9dxEAwK4oDInnbHv2inNeS4fkOeMg5ScxLjamjjE3l8GuL6MzA\nSddHad6f/RJWa2HQxfhgo9aMlVq93QxlSLv7nWJ4qrEEXFCsDE4iqnQ0gA3XwUSs\nA44b1Ml5QAYmGBdS5IDVciC4IHi3/CmUIWgB/YiJeutQjDRjtqn3qe6rp3R193Zd\nPPWi8lrs6r1k+u935TyY9uvynvl0/OYzPSvzqcsunzPnlXnbzp768YMTb751pu/b\nHz/Y9cbYV9NWNF7YfeindcOfv/Ss1+nxmfN675g965NlXwxfx3fvOLJ34fbN5298\nZfXpl589vLAY7g93PTM2Zc+M1w79dP7k+K1vfXjT02MLFt+w49J7lzx2+6drNtP3\nZ10z6+PZM9/eNX3fxiXVnQde3LCUH3vv3LI5x969aUPr7srCzduvOnr6u/X7j9+y\ndH99xTdb5j5ktqz+8rP7eqeMnyK37Zk6Y96xrxu/nG8dH3l97pktO6G679ZH5m94\nvu/7J8fWHnnsiTd++Hl1z81dh/ceeO3Vs0cWb1v+6L3nrhi60r5z17bs6G0Pbds6\nuvb++viNc4vtB7eeOHr9bw==\n=JlfJ\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":13,\"last_id_change\":1424754121,\"username\":\"gpgtools\"},\"id\":\"ab739e2776dde0f7f9ff61ed1374eb00\",\"key\":{\"key_fingerprint\":\"85e38f69046b44c1ec9fb07b76d78f0500d026c4\",\"kid\":\"01119ea566f88f54487b741235c96e45079d4f6868f4b2611dbabde936b875d62afa0a\"},\"remote_proofs\":[{\"ctime\":1420144001,\"curr\":\"b635ecdcd60d4fac9f1f370a8f0b141a17eac639194ce723c4d1f05bea68698b\",\"etime\":1577824001,\"prev\":\"48d7e379b4f57619279388897cc274fbf549dd183021ae1650a287a52b39a7a1\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"gpgtools.org\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":2,\"sig_id\":\"fbdae88c8d44f0b2f8baafb0a72dac7e1c21fc09658fa450c2716610554aebcb0f\",\"sig_type\":2},{\"ctime\":1420144498,\"curr\":\"313d71d4d5f07fd8ca0c69043f98dbde4f5d98f29632513cc169a92b784dbb9c\",\"etime\":1577824498,\"prev\":\"b635ecdcd60d4fac9f1f370a8f0b141a17eac639194ce723c4d1f05bea68698b\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"gpgtools\"},\"proof_type\":2,\"state\":1},\"seqno\":3,\"sig_id\":\"9fb02bbfb3ab4a141202b035a4d14abbc4ecc28dfb5cc0ad7d076624ad01fc1a0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"1da5b4214c8fe4b93766ac89191614224f2b7adb7d2a3590d388c98d12b2bfa0\",\"seqno\":5,\"sig_id\":\"50ce3fd913cd22ca29c1bed4f449178ffc9bec7944bf8525d6d75d30f2c573250f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424977483,\"expire_in\":157680000,\"prev\":\"c17a8f34177819e167e8aa727d8e391dbaadb33d2c58d8bdfe3a244ba4340828\",\"seqno\":20,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424977483,
            "etime": 1582657483,
            "rtime": null,
            "sig_status": 0,
            "prev": "c17a8f34177819e167e8aa727d8e391dbaadb33d2c58d8bdfe3a244ba4340828",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 21,
            "payload_hash": "daad146752067612cd4a10091d6ffb1246c1029b92a3e71e7fc86468e150b48b",
            "sig_id": "4b5a10b7a32536fe976be26eb7b42641ec1eab36a06327f1dd6f836ab7d910480f",
            "sig_id_short": "S1oQt6MlNv6Xa-Jut7QmQeweqzagYyfx3W-D",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGtVXuIFVUYv2uWtaSISlQYxVhgeJXznnOE1JTKPwoTIxKN63neHXf3znpnVt1s\nK4qSpDCSrIjFFIS2oDIhDfGFaGRJRRmiIEp/iNJiBD0os74Zt8VViYq4XDhzzszv\n+z2+c85Lo6+qtLbsfmrRkvl+wbKWT8+YykNnGytXRSZ1PdH0VZHtSHwjL0YN3emj\n6VG77zE689OS9LYV3kTVaLlvZknagCVFlRBcG29jZWOupFDcU6YlV0ohbogKTmor\not5qAVOAhqRR982uZlLUiFBsAo6pZVoTJJiRUhKpYRAkfC4Qx4JzGmOo2pZm+TA2\nUYlZSxzMXuH99nIBYYSdctRqQyw1TjNTcPROe4JloIEoLg2PdWy8oMo6RIkWMRaI\n2tgrB7gBaYDrLuG8MR7wFMNImxhhQokgNkgXrIwVQsWLmW8OOqfbdUfSKMTnTW3b\nC/nAPbFZMUpcbchITKpRh85yEFOzbRocgjkGBCUVnAzDzHVPR9rMViQhL4BLViGw\nOIAEi4wFqYhR71EIVCHMAi5ZDbpfGDY8ASaZI4RyCtZJxRFTghFuYyaE9VgYZrRE\njvJhjgoI3DIhNTLaKysMJohrbxhVhBFLKHzivPZCCEk4w545SwN3oAiYSgGOAvem\n70xzX+tqpmkASxZB7+VJZykdY8akFKIa2e5ms+y02FEHASnvsXKW2WA5zBjCsVSW\nEh40KdrOEaQF/CEXr50nYIYSTgB9PwjOhcLkAvggg8KVkkXZ/23ettecznVtaVaE\nM7QR8hVJnvvm8IzLJNJG5oFokvcUukqoWt7TBesQXpbrvKgLK5lf1khhCKOkfqFz\nOXMuYI84gSBAt8PEMR+4xggmuQUPpUGQLdPYceg0Cs1KkfESC3DJoBBdQBus11u9\n1EZF1JCNIfaSBgO2OUYNdYTh0x4anEnhVbCYOhEklbCjoQo8GSXiICy0e2CMCqsu\n87FE72r65f9PSP8wEZd26qRxuf3TbNoZlQnkqU074AXXyC7NBHYEvlIs5KJYMDWS\nY/jRWCtcJAQnEzJUOS5hA0AUHMExYhihzgdCjLBacCoUnIESIXdZLI+WZWq5TjoK\nAV3FNtau1qaztivn8m9j+c8i9oz+Wxm9xfFVihg8xS66AArzhpqNMBXHnMC28iu7\nkqavFQFhHhdACA01CZQNMWeBx4po4nGwOqYcTk8c4kC1I5YDEw1WCCV0kCzQmFGH\nY2ZJbCi7SCqkmOt60QVJvaHz7qaPelufb5k4stLSWrnm6hHFxVZpvW7sX7dd97lR\nv9967Mv9jS37+vuP1irzduZL6zMOjeyfemDniflrJ0y4e/2OQ5t/7ct7Zn/zxA9j\nts86f6p6bM9MO7mv88O39n9WyUaMe3vx60feO37ftlvmLJ23OXnhYP3avmz1qPVz\nBvbufpFmJ3bs2sfGvnlXtmEM/nj9xtkj7v1pxuPv1+pfdH9w/mRXPHXLJLNxYMkr\nfzw7YfzW754+vWzNwoF6Y6/97ebXvv4qXfPyw71P5u8efnXdt5P2b8OTf94gtj84\nrj7lo3Tx/Vtn3XQ92fRG0nJu7mN3njrKdh09oA+Pz3cer/x4/pG+G8imKevWzjx4\n/JOwfO7BZ9g9az6f//3qM0duFL/sWthvp0+84+xz1ZNt7wzcHj3wJw==\n=bbK7\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":12,\"last_id_change\":1421883652,\"username\":\"taylorswift\"},\"id\":\"ff47f7a7c0bcd3c043ee0ff39014f100\",\"key\":{\"key_fingerprint\":\"484d223535168950496425c7466ce16b4ba80d35\",\"kid\":\"01016ec7c468a0bae9c6b1205aeb439242c2380ddeae66682541e4dc3f5d21847f860a\"},\"remote_proofs\":[{\"ctime\":1411448866,\"curr\":\"937d3d39c9ee19dc4cfc5d3db25189c325fa28599d20a620a262eade204396d6\",\"etime\":1569128866,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"swiftonsecurity\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"54ddf1e05284d411d12d4ef5a10e055ce668b0a7c4a1d5f8d357a30be816114b0f\",\"sig_type\":2},{\"ctime\":1411448929,\"curr\":\"f7e83fbdb2d43b3d241ef0a486e9fc13d6f838abe57a13db967f6c410f4436c9\",\"etime\":1569128929,\"prev\":\"937d3d39c9ee19dc4cfc5d3db25189c325fa28599d20a620a262eade204396d6\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"swiftonsecurity.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":2,\"sig_id\":\"13b85151537a91df1e8820b39d581011d5505eeb423def22b6ca65369986800d0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"f7e83fbdb2d43b3d241ef0a486e9fc13d6f838abe57a13db967f6c410f4436c9\",\"seqno\":2,\"sig_id\":\"13b85151537a91df1e8820b39d581011d5505eeb423def22b6ca65369986800d0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424977526,\"expire_in\":157680000,\"prev\":\"11df754f5792a2e1fca7358791f7f3ad2c5ef2adb2696af84f3743d174c27b34\",\"seqno\":21,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424977526,
            "etime": 1582657526,
            "rtime": null,
            "sig_status": 0,
            "prev": "11df754f5792a2e1fca7358791f7f3ad2c5ef2adb2696af84f3743d174c27b34",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 22,
            "payload_hash": "694ed7166cee72449964e97bcd4be58243877718425c4dc655d2d80832bd5cdf",
            "sig_id": "2a0da9730f049133ce728ba30de8c91b6658b7a375e82c4b3528d7ddb1a21f7a0f",
            "sig_id_short": "Kg2pcw8EkTPOcoujDejJG2ZYt6N16CxLNSjX",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase Go CLI 0.1.4 (darwin)\n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\nlV3oFKu987EdVQilS37pKNHJ61z+OfbOhoWDNUWR\n=GlPj\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"device\":{\"id\":\"ad6b7972e84ac16d52b5283b42dfb718\",\"type\":\"desktop\",\"kid\":\"0120bad63d4a9b7852b1c092dea088bef3000efa1e14c8d72bf077025f7ad0550b050a\",\"description\":\"rMBP\",\"status\":1},\"key\":{\"eldest_kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"BF805B6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"sibkey\":{\"kid\":\"0120bad63d4a9b7852b1c092dea088bef3000efa1e14c8d72bf077025f7ad0550b050a\",\"reverse_sig\":{\"type\":\"kb\",\"sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgutY9Spt4UrHAkt6giL7zAA76HhTI1yvwdwJfetBVCwUKp3BheWxvYWTEqXsiY3RpbWUiOjE0MjcxNDA5MjcsImRlbGVnYXRlZF9ieSI6IjAxMDFkOWQzY2FiMmMzYmRhNGI2OTVlZWRhZTIxOGYzZjI5NThiNTdhN2JlNjM5Y2QwMzJhNjcxNjAzYzdlOWRmODBmMGEiLCJ1aWQiOiJlYmJlMWQ5OTQxMGFiNzAxMjMyNjJjZjhkZmM4NzkwMCIsInVzZXJuYW1lIjoiYWthbGluIn2jc2lnxEAVbt8oyulukwj+BnB/2IY17Weh8WMYnSyzYE+Ue5ppA0/ZeYmjNXnAvLvcPptt5f/01yRV0wc2X2+7wZAdnFQEqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"}},\"type\":\"sibkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"0.1.4\"},\"ctime\":1427140927,\"expire_in\":94608000,\"merkle_root\":{\"ctime\":1427140887,\"hash\":\"99cc99b537a724e7aeb8367425e692bb0ca0f0c6b5634bd3b477535a531306ff0cb3d941e0909b666717ae403bf6db0ea50a281465670a4e045873ec99112eb8\",\"seqno\":182592},\"prev\":\"daad146752067612cd4a10091d6ffb1246c1029b92a3e71e7fc86468e150b48b\",\"seqno\":22,\"tag\":\"signature\"}",
            "sig_type": 1,
            "ctime": 1427140927,
            "etime": 1521748927,
            "rtime": null,
            "sig_status": 0,
            "prev": "daad146752067612cd4a10091d6ffb1246c1029b92a3e71e7fc86468e150b48b",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 23,
            "payload_hash": "27bc88059a768a82b1a21dcc1c46f7fc61c2d2b80c445eb2d18fed3a5bb42e49",
            "sig_id": "eb5c7e7d3cf8370bed8ab55c0d8833ce9d74fd2c614cf2cd2d4c30feca4518fa0f",
            "sig_id_short": "61x-fTz4NwvtirVcDYgzzp10_SxhTPLNLUww",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase Go CLI 0.1.4 (darwin)\n\nxA0DAAoBv4BbYFFlU3EBy+F0AOIAAAAA6nsiYm9keSI6eyJkZXZpY2UiOnsiaWQi\nOiJhZDZiNzk3MmU4NGFjMTZkNTJiNTI4M2I0MmRmYjcxOCIsInR5cGUiOiJkZXNr\ndG9wIiwia2lkIjoiMDEyMTE4NzgxZWRlYWFlOWM2MTNkNTMwMjZhNGVmOTIwYzZj\nODM4MmExYzRmZDQwYzM5OTJhOWM3OTcwYTcwMTU4MzEwYSIsImRlc2NyaXB0aW9u\nIjoick1CUCIsInN0YXR1cyI6MX0sImtleSI6eyJlbGRlc3Rfa2lkIjoiMDEwMWQ5\nZDNjYWIyYzNiZGE0YjY5NWVlZGFlMjE4ZjNmMjk1OGI1N2E3YmU2MzljZDAzMmE2\nNzE2MDNjN2U5ZGY4MGYwYSIsImZpbmdlcnByaW50IjoiMDdiZjE3M2M0YWEyMDY0\nYjg4ODI4YTY0YmY4MDViNjA1MTY1NTM3MSIsImhvc3QiOiJrZXliYXNlLmlvIiwi\na2V5X2lkIjoiQkY4MDVCNjA1MTY1NTM3MSIsImtpZCI6IjAxMDFkOWQzY2FiMmMz\nYmRhNGI2OTVlZWRhZTIxOGYzZjI5NThiNTdhN2JlNjM5Y2QwMzJhNjcxNjAzYzdl\nOWRmODBmMGEiLCJ1aWQiOiJlYmJlMWQ5OTQxMGFiNzAxMjMyNjJjZjhkZmM4Nzkw\nMCIsInVzZXJuYW1lIjoiYWthbGluIn0sInN1YmtleSI6eyJraWQiOiIwMTIxMTg3\nODFlZGVhYWU5YzYxM2Q1MzAyNmE0ZWY5MjBjNmM4MzgyYTFjNGZkNDBjMzk5MmE5\nYzc5NzBhNzAxNTgzMTBhIiwicGFyZW50X2tpZCI6IjAxMDFkOWQzY2FiMmMzYmRh\nNGI2OTVlZWRhZTIxOGYzZjI5NThiNTdhN2JlNjM5Y2QwMzJhNjcxNjAzYzdlOWRm\nODBmMGEifSwidHlwZSI6InN1YmtleSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsi\nbmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjAuMS40In0s\nImN0aW1lIjoxNDI3MTQwOTI4LCJleHBpcmVfaW4iOjk0NjA4MDAwLCJtZXJrbGVf\ncm9vdCI6eyJjdGltZSI6MTQyNzE0MDg4NywiaGFzaCI6Ijk5Y2M5OWI1MzdhNzI0\nZTdhZWI4MzY3NDI1ZTY5MmJiMGNhMGYwYzZiNTYzNGJkM2I0Nzc1MzVhNTMxMzA2\nZmYwY2IzZDk0MWUwOTA5YjY2NjcxN2FlNDAzYmY2ZGIwZWE1MGEyODE0NjU2NzBh\nNGUwNDU4NzNlYzk5MTEyZWI4Iiwic2Vxbm8iOjE4MjU5Mn0sInByZXbmIjoiNjk0\nZWQ3MTY2Y2VlNzI0NDk5NjRlOTdiY2Q0YmU1ODI0Mzg3NzcxODQyNWM0ZGM2NTVk\nMmQ4MDgzMmJkNeVjZGYiLCJzZXFubyI6MjMsInRhZyI6InNpZ25hdHVyZeEifQDC\nwFwEAAEKABAFAlUQcUAJEL+AW2BRZVNxAACIaggAdOXNBG5V3ZyH/jqezi0gbOhp\nd88r4Fugil9n5jvY2Rp/yGWL+/Axh7NaReyw9Em3j1f9HBMwJLBO38j05KNDM1ES\nRsHBQz8YDU7QrmK2ZCbXJijfHbd6KgAJOUDMATVQxHcZX1ty3cgACTzbpGYsWNds\nC6lunXiGWlG7U5PFBupFWnI2BXXnA7PwL0ZuWPMfxFWT2xeIWYOgoe0XaRfP2hHH\nGtY5Bfv9rG4NLj9OuK0tK8gE35ikYw0aJY5BV1nbyiSNQNBrv7En+0luBGFcD3Ks\nRgGlu/LnJcttuk8aHLHZEr320LkA2tOlSnZMcWHnNoPMHfjIoOhM1AJk1QZLVw==\n=tm6V\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"device\":{\"id\":\"ad6b7972e84ac16d52b5283b42dfb718\",\"type\":\"desktop\",\"kid\":\"012118781edeaae9c613d53026a4ef920c6c8382a1c4fd40c3992a9c7970a70158310a\",\"description\":\"rMBP\",\"status\":1},\"key\":{\"eldest_kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"BF805B6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"subkey\":{\"kid\":\"012118781edeaae9c613d53026a4ef920c6c8382a1c4fd40c3992a9c7970a70158310a\",\"parent_kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\"},\"type\":\"subkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"0.1.4\"},\"ctime\":1427140928,\"expire_in\":94608000,\"merkle_root\":{\"ctime\":1427140887,\"hash\":\"99cc99b537a724e7aeb8367425e692bb0ca0f0c6b5634bd3b477535a531306ff0cb3d941e0909b666717ae403bf6db0ea50a281465670a4e045873ec99112eb8\",\"seqno\":182592},\"prev\":\"694ed7166cee72449964e97bcd4be58243877718425c4dc655d2d80832bd5cdf\",\"seqno\":23,\"tag\":\"signature\"}",
            "sig_type": 1,
            "ctime": 1427140928,
            "etime": 1521748928,
            "rtime": null,
            "sig_status": 0,
            "prev": "694ed7166cee72449964e97bcd4be58243877718425c4dc655d2d80832bd5cdf",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 24,
            "payload_hash": "12b594e44d9289349283f8b14a6f83ad144a17a3025a758e17d4eca70fbdc923",
            "sig_id": "0f175ef0d3b57a9991db5deb30f2432a85bc05922bbe727016f3fb660863a1890f",
            "sig_id_short": "Dxde8NO1epmR213rMPJDKoW8BZIrvnJwFvP7",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase Go CLI 0.1.4 (darwin)\n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\nkqI=\n=YWnj\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"device\":{\"id\":\"4c37e8fb6cb1c6ffcd7a7b8d9f533e18\",\"type\":\"web\",\"kid\":\"0120aa39aa0458019ae603412421f7136806ce8f1efc13ba1d0ccec5121df647df490a\",\"status\":1},\"key\":{\"eldest_kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"BF805B6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"sibkey\":{\"kid\":\"0120aa39aa0458019ae603412421f7136806ce8f1efc13ba1d0ccec5121df647df490a\",\"reverse_sig\":{\"type\":\"kb\",\"sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgqjmqBFgBmuYDQSQh9xNoBs6PHvwTuh0MzsUSHfZH30kKp3BheWxvYWTEqXsiY3RpbWUiOjE0MjcxNDA5MjgsImRlbGVnYXRlZF9ieSI6IjAxMDFkOWQzY2FiMmMzYmRhNGI2OTVlZWRhZTIxOGYzZjI5NThiNTdhN2JlNjM5Y2QwMzJhNjcxNjAzYzdlOWRmODBmMGEiLCJ1aWQiOiJlYmJlMWQ5OTQxMGFiNzAxMjMyNjJjZjhkZmM4NzkwMCIsInVzZXJuYW1lIjoiYWthbGluIn2jc2lnxEAdq9XpWaaPSfEeWeZwgj5kETElNTEsVgxpwD/ou4cMuvMCGIYItoDLPdoMhcr47n5DJbi9qPEXVEKToC4drqYLqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"}},\"type\":\"sibkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"0.1.4\"},\"ctime\":1427140928,\"expire_in\":94608000,\"merkle_root\":{\"ctime\":1427140887,\"hash\":\"99cc99b537a724e7aeb8367425e692bb0ca0f0c6b5634bd3b477535a531306ff0cb3d941e0909b666717ae403bf6db0ea50a281465670a4e045873ec99112eb8\",\"seqno\":182592},\"prev\":\"27bc88059a768a82b1a21dcc1c46f7fc61c2d2b80c445eb2d18fed3a5bb42e49\",\"seqno\":24,\"tag\":\"signature\"}",
            "sig_type": 1,
            "ctime": 1427140928,
            "etime": 1521748928,
            "rtime": null,
            "sig_status": 0,
            "prev": "27bc88059a768a82b1a21dcc1c46f7fc61c2d2b80c445eb2d18fed3a5bb42e49",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 25,
            "payload_hash": "ce162011e380c954de15f30db28f8b7b358866d2721143d9d0d4424166ce5ed8",
            "sig_id": "48267f0e3484b2f97859829503e20c2f598529b42c1d840a8fc1eceda71458400f",
            "sig_id_short": "SCZ_DjSEsvl4WYKVA-IML1mFKbQsHYQKj8Hs",
            "kid": "0120aa39aa0458019ae603412421f7136806ce8f1efc13ba1d0ccec5121df647df490a",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgqjmqBFgBmuYDQSQh9xNoBs6PHvwTuh0MzsUSHfZH30kKp3BheWxvYWTFA/R7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiYjhjOGEzMGU0NTMwZjc2OGZhNjhjYzA5NWQ5ZGQ0MTgiLCJ0eXBlIjoid2ViIiwia2lkIjoiMDEyMTRkNzNiN2Q1YjRlMjAwNGRiMzk1ZmRmNjdiODgxOGYzZmY0MzI0ZGZmMWJjMzAxNjg5ZjgwNjE1N2I4ZWRhNWEwYSIsInN0YXR1cyI6MX0sImtleSI6eyJlbGRlc3Rfa2lkIjoiMDEwMWQ5ZDNjYWIyYzNiZGE0YjY5NWVlZGFlMjE4ZjNmMjk1OGI1N2E3YmU2MzljZDAzMmE2NzE2MDNjN2U5ZGY4MGYwYSIsImhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMGFhMzlhYTA0NTgwMTlhZTYwMzQxMjQyMWY3MTM2ODA2Y2U4ZjFlZmMxM2JhMWQwY2NlYzUxMjFkZjY0N2RmNDkwYSIsInVpZCI6ImViYmUxZDk5NDEwYWI3MDEyMzI2MmNmOGRmYzg3OTAwIiwidXNlcm5hbWUiOiJha2FsaW4ifSwic3Via2V5Ijp7ImtpZCI6IjAxMjE0ZDczYjdkNWI0ZTIwMDRkYjM5NWZkZjY3Yjg4MThmM2ZmNDMyNGRmZjFiYzMwMTY4OWY4MDYxNTdiOGVkYTVhMGEiLCJwYXJlbnRfa2lkIjoiMDEyMGFhMzlhYTA0NTgwMTlhZTYwMzQxMjQyMWY3MTM2ODA2Y2U4ZjFlZmMxM2JhMWQwY2NlYzUxMjFkZjY0N2RmNDkwYSJ9LCJ0eXBlIjoic3Via2V5IiwidmVyc2lvbiI6MX0sImNsaWVudCI6eyJuYW1lIjoia2V5YmFzZS5pbyBnbyBjbGllbnQiLCJ2ZXJzaW9uIjoiMC4xLjQifSwiY3RpbWUiOjE0MjcxNDA5MjgsImV4cGlyZV9pbiI6OTQ2MDgwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDI3MTQwODg3LCJoYXNoIjoiOTljYzk5YjUzN2E3MjRlN2FlYjgzNjc0MjVlNjkyYmIwY2EwZjBjNmI1NjM0YmQzYjQ3NzUzNWE1MzEzMDZmZjBjYjNkOTQxZTA5MDliNjY2NzE3YWU0MDNiZjZkYjBlYTUwYTI4MTQ2NTY3MGE0ZTA0NTg3M2VjOTkxMTJlYjgiLCJzZXFubyI6MTgyNTkyfSwicHJldiI6IjEyYjU5NGU0NGQ5Mjg5MzQ5MjgzZjhiMTRhNmY4M2FkMTQ0YTE3YTMwMjVhNzU4ZTE3ZDRlY2E3MGZiZGM5MjMiLCJzZXFubyI6MjUsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAlN+pEeGja9mb7MqufdZQrb5scAuaTn/ECcNviA29FBgyeYXEEm164XDDXogDBalKMxizqP14/0Zmw8WkhGi8DqhzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
            "payload_json": "{\"body\":{\"device\":{\"id\":\"b8c8a30e4530f768fa68cc095d9dd418\",\"type\":\"web\",\"kid\":\"01214d73b7d5b4e2004db395fdf67b8818f3ff4324dff1bc301689f806157b8eda5a0a\",\"status\":1},\"key\":{\"eldest_kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"host\":\"keybase.io\",\"kid\":\"0120aa39aa0458019ae603412421f7136806ce8f1efc13ba1d0ccec5121df647df490a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"subkey\":{\"kid\":\"01214d73b7d5b4e2004db395fdf67b8818f3ff4324dff1bc301689f806157b8eda5a0a\",\"parent_kid\":\"0120aa39aa0458019ae603412421f7136806ce8f1efc13ba1d0ccec5121df647df490a\"},\"type\":\"subkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"0.1.4\"},\"ctime\":1427140928,\"expire_in\":94608000,\"merkle_root\":{\"ctime\":1427140887,\"hash\":\"99cc99b537a724e7aeb8367425e692bb0ca0f0c6b5634bd3b477535a531306ff0cb3d941e0909b666717ae403bf6db0ea50a281465670a4e045873ec99112eb8\",\"seqno\":182592},\"prev\":\"12b594e44d9289349283f8b14a6f83ad144a17a3025a758e17d4eca70fbdc923\",\"seqno\":25,\"tag\":\"signature\"}",
            "sig_type": 1,
            "ctime": 1427140928,
            "etime": 1521748928,
            "rtime": null,
            "sig_status": 0,
            "prev": "12b594e44d9289349283f8b14a6f83ad144a17a3025a758e17d4eca70fbdc923",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": ""
        },
        {
            "seqno": 26,
            "payload_hash": "55b1d9f7b6ac608ab5a1b361bd941c83cbd600f62e687071669904ad3500e109",
            "sig_id": "543be0b42d22ba4ee8f2cea0b8d58b8c9d0294bacc9e9884a0a6315a9943279d0f",
            "sig_id_short": "VDvgtC0iuk7o8s6guNWLjJ0ClLrMnpiEoKYx",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGbwMvMwMW4vyE6ITA1uJDx9IHtSQyhsneKq5WS8lMqlayqlbJTwVRaZl56alFB\nUWZeiZKVkoF5UpqhuXGySWKikYGZSZKFhYWRRSKQkWZhYJpkZmBqaGZqamxuqKSj\nlJFfDNIBNCYpsThVLzMfKAbkxGemAEWd3IDqnVDUl4IlUpOSUg1TLC1NDA0Sk8wN\nDI2MjcyMktMsUtKSLcwtDQxACotTi/ISc1OBqhOzE3My85RqdZRKKgtAAuWpSfFA\n6bLM5NT4pMy8FKDjgTrKUouKM/PzlKwMgSqTSzJBeg1NjCwMjAyMDQ11lFIrCjKL\nUuMzQSpMzc0sDIBAR6mgKLUMaGRyqqGZkYGhYaqxhUGypalJSqqhaZqxQUqSkUWa\nRZJ5krGphYWZWYqRuZGhoYlximWKQYqJiZGJoZlZcqppaooF0P7i1MK8fCUrIzOg\nOxPTgWYWZ6bnJZaUFqUq1XYyyrAwMHIxsLEygcKfgYtTABYrqys4GE5IZOgYFhr2\nNkxO2X/zysmXYUqLy9s3mT+8lSFzUOrvXi+DdfZ35kvZnO+dufrF+0LHWp9mzU3z\nZc7/2vCa+5TyvsfO9esVT4UdenVhj7yewf6NWZMk7m47rLXonoRK0Mq6CcuZV+Wz\n9f+QLQub5xFu4zF5ypZttYKBT42+nnc4uk5Nladr21ve2x0VK3daN1c/tlgmU8g/\nf+mc5vehLyUPVyQ8OONjonhfoaih5N7itnnWFUaGVpxPhL2i59S2pz3kVgiIS13h\neTXEZe7Fj93hkoFd050WvFQKe5I3ncX9UnO7kfSD3YItSbbWLHceq7I8nr+b/8X5\neb1H/3f/+RGWEVr7Ibj8kCSPySWjZAA=\n=foi1\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"BF805B6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1428020311,\"expire_in\":157680000,\"prev\":\"ce162011e380c954de15f30db28f8b7b358866d2721143d9d0d4424166ce5ed8\",\"seqno\":26,\"tag\":\"signature\"}",
            "sig_type": 1,
            "ctime": 1428020311,
            "etime": 1585700311,
            "rtime": null,
            "sig_status": 0,
            "prev": "ce162011e380c954de15f30db28f8b7b358866d2721143d9d0d4424166ce5ed8",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 27,
            "payload_hash": "60cc8edaa70c70a32591d35bb1e2f5f18a862b6b178b1aec27730ceeb166d9ea",
            "sig_id": "34a370138e5a3efe49946d049b52fd63fd825e702f76a0f560f550e7de43b0250f",
            "sig_id_short": "NKNwE45aPv5JlG0Em1L9Y_2CXnAvdqD1YPVQ",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n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\n6pMnPvHs3Qf2qDB28n0X/ws=\n=0H6x\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.7\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"BF805B6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"merkle_root\":{\"ctime\":1428026414,\"hash\":\"396523208ff0f485144d71755b26b654f25b5487307efa845973863ffbee6fdc95262959d587f35e0a8457fc54a2b4bcaddf12500c8ed5c7b158ba153fa16204\",\"seqno\":190010},\"track\":{\"basics\":{\"id_version\":50,\"last_id_change\":1426792504,\"username\":\"gabrielh\"},\"ctime\":1428026581,\"id\":\"b7c2eaddcced7727bcb229751d91e800\",\"key\":{\"key_fingerprint\":\"3f6e504945ae44e6a53d49845431890fe1eebcde\",\"kid\":\"0101230aa5a191471449cc05ee95ce3637ebb16e409bb3b92ac0a8cc28eef92c0b500a\"},\"remote_proofs\":[{\"ctime\":1413405722,\"curr\":\"55d1e2013c91f4ef410538cb03cadfcd0b40e86dfa8a0594a42036eedfd61fff\",\"etime\":1571085722,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"gabrlh\"},\"proof_type\":2,\"state\":1},\"seqno\":19,\"sig_id\":\"dbbb07105a43c18ab1a9d01b564c591a6d4d57e35dd4ba3f1a0ed8e33f8517eb0f\",\"sig_type\":2},{\"ctime\":1413405744,\"curr\":\"5381de2b714095bcf65f41c1b235c11ab6fb87cea8b2c9a3bcb95095ab26e191\",\"etime\":1571085744,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"gabriel\"},\"proof_type\":3,\"state\":1},\"seqno\":20,\"sig_id\":\"916bccc5e1fd7e784340cca2795d1ae352922bd498ec196bbff6c00b7a304d2d0f\",\"sig_type\":2},{\"ctime\":1413406032,\"curr\":\"eba1b91ed0048983d7e32efeee7fecce16547998ff3e05b67045a751ef99be1c\",\"etime\":1571086032,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"gabrielh\"},\"proof_type\":5,\"state\":1},\"seqno\":21,\"sig_id\":\"2c9e345b5acedbcfe3d5d9b2d6115b59e9d6ad31ad262bb07266ccc6db0b95940f\",\"sig_type\":2},{\"ctime\":1421805307,\"curr\":\"a0a74b05db02a6c66c9cf2ffd3754449ff680cd4fb61f80cba2038a23e7c0bfb\",\"etime\":1579485307,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"gabrielhn\"},\"proof_type\":6,\"state\":1},\"seqno\":28,\"sig_id\":\"2787406a0b57a2eb7de822c5ecea6bdcd28a3ed96191a27d46459d747a6b292c0f\",\"sig_type\":2},{\"ctime\":1421558613,\"curr\":\"fdd474158530c9816ca9707c3cda809aaa24a0f69b5bcc06f5776a54cea54fff\",\"etime\":1579238613,\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"rel.me\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":25,\"sig_id\":\"fdee6f84bdd854d055fe51be696b175e5f2d2d6b143bc09a52f7bad57dfe05160f\",\"sig_type\":2},{\"ctime\":1421558741,\"curr\":\"158668859ea086ccd02ce16724f2b35a6a71834f98867429cd1210a075346443\",\"etime\":1579238741,\"