// Copyright 2015 Keybase, Inc. All rights reserved. Use of
// this source code is governed by the included BSD license.

package libkb

import (
	"bytes"
	"encoding/base64"
	"testing"

	"github.com/keybase/go-crypto/openpgp"
	triplesec "github.com/keybase/go-triplesec"
	"github.com/ugorji/go/codec"
)

type Foo struct {
	Bar int    `codec:"bar"`
	Baz string `codec:"baz"`
}

func TestDecode0(t *testing.T) {
	data := "gqNiYXIKo2JheqRzaGl0"
	bytes, err := base64.StdEncoding.DecodeString(data)
	var h codec.MsgpackHandle
	if err != nil {
		t.Errorf("error decoding: %s", err)
	}
	var foo Foo
	err = MsgpackDecodeAll(bytes, &h, &foo)
	if err != nil {
		t.Errorf("Failed to decode packet: %s", err)
	}
	if foo.Bar != 10 {
		t.Errorf("Bad decoding of our FOO")
	}

}

func TestDecode1(t *testing.T) {
	key := "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"
	bytes, err := base64.StdEncoding.DecodeString(key)
	if err != nil {
		t.Errorf("error decoding: %s", err)
	}
	_, err = DecodePacket(bytes)
	if err != nil {
		t.Errorf("Failed to decode packet: %s", err)
	}
}

func TestDecodeSKBSequence(t *testing.T) {
	seq := "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"
	buf := bytes.NewBufferString(seq)
	decoder := base64.NewDecoder(base64.StdEncoding, buf)
	packets, err := DecodePacketsUnchecked(decoder)
	if err != nil {
		t.Errorf("Failed to decode packets: %s", err)
	}
	p3skbs, err := packets.ToListOfSKBs()
	if err != nil {
		t.Errorf("Failed to make a list of SKBs: %s", err)
	}
	if !FastByteArrayEq(p3skbs[1].Pub, p3skbs[0].Pub) {
		t.Errorf("Expected a repeat of the same key")
	}
}

func makeTestLKSec(t *testing.T, gc *GlobalContext) *LKSec {
	_, pps, err := StretchPassphrase("makeTestLKSec", []byte("saltsaltsaltsalt"))
	if err != nil {
		t.Fatal(err)
	}
	lks := NewLKSec(pps, "1111111111111111111111111111111111111111111111111111111111111119", gc)
	if err := lks.GenerateServerHalf(); err != nil {
		t.Fatal(err)
	}

	return lks
}

func makeTestSKB(t *testing.T, lks *LKSec) *SKB {
	email := "test@keybase.io"
	entity, err := openpgp.NewEntity("test name", "test comment", email, nil)
	if err != nil {
		t.Fatal(err)
	}

	skb, err := (NewGeneratedPGPKeyBundle(entity)).ToLksSKB(lks)
	if err != nil {
		t.Fatal(err)
	}
	skb.uid = lks.uid

	skb.newLKSecForTest = func(_ []byte) *LKSec {
		return lks
	}

	g := G
	salt, err := RandBytes(triplesec.SaltLen)
	if err != nil {
		t.Fatal(err)
	}
	g.createLoginState()
	err = g.LoginState().Account(func(a *Account) {
		a.CreateLoginSessionWithSalt(email, salt)
	}, "makeTestSKB")
	if err != nil {
		t.Fatal(err)
	}

	return skb
}

func testPromptAndUnlock(t *testing.T, skb *SKB, secretStore SecretStore) {
	// XXX check nil, nil at end of this...
	key, err := skb.PromptAndUnlock(nil, "test reason", "test which", secretStore, &TestSecretUI{Passphrase: "test passphrase", StoreSecret: true}, nil, nil)
	if err != nil {
		t.Fatal(err)
	}

	if key == nil {
		t.Errorf("PromptAndUnlock returned a nil key")
	}
}

func TestBasicSecretStore(t *testing.T) {
	tc := SetupTest(t, "skb_basic_secret_store")
	defer tc.Cleanup()

	lks := makeTestLKSec(t, G)
	expectedSecret, err := lks.GetSecret(nil)
	if err != nil {
		t.Fatal(err)
	}

	skb := makeTestSKB(t, lks)
	testSecretStore := TestSecretStore{}
	testPromptAndUnlock(t, skb, &testSecretStore)

	if string(testSecretStore.Secret) != string(expectedSecret) {
		t.Errorf("secret doesn't match expected value")
	}

	// Doing the prompt again should retrieve the secret from our
	// store and not call skb.newLKSecForTest.

	skb = makeTestSKB(t, lks)
	skb.newLKSecForTest = func(_ []byte) *LKSec {
		t.Errorf("newLKSecForTest unexpectedly called")
		return lks
	}
	testPromptAndUnlock(t, skb, &testSecretStore)
}

func TestCorruptSecretStore(t *testing.T) {
	tc := SetupTest(t, "skb_corrupt_secret_store")
	defer tc.Cleanup()

	lks := makeTestLKSec(t, G)
	expectedSecret, err := lks.GetSecret(nil)
	if err != nil {
		t.Fatal(err)
	}

	skb := makeTestSKB(t, lks)
	testSecretStore := TestSecretStore{
		Secret: []byte("corrupt"),
	}
	testPromptAndUnlock(t, skb, &testSecretStore)

	// The corrupt secret value should be overwritten by the new
	// correct one.
	if string(testSecretStore.Secret) != string(expectedSecret) {
		t.Errorf("secret doesn't match expected value")
	}
}

func TestUnusedSecretStore(t *testing.T) {
	tc := SetupTest(t, "skb_unused_secret_store")
	defer tc.Cleanup()

	lks := makeTestLKSec(t, G)

	skb := makeTestSKB(t, lks)
	// It doesn't matter what passphraseStream contains, as long
	// as it's the right size.
	err := tc.G.LoginState().Account(func(a *Account) {
		a.CreateStreamCache(nil, NewPassphraseStream(make([]byte, extraLen)))
	}, "TestUnusedSecretStore")
	if err != nil {
		t.Fatal(err)
	}

	testSecretStore := TestSecretStore{}
	testPromptAndUnlock(t, skb, &testSecretStore)

	// Since there is a non-nil passphraseStream in the login
	// state, nothing should be stored in the secret store (since
	// no prompt was shown).
	if len(testSecretStore.Secret) > 0 {
		t.Errorf("secret unexpectedly non-empty")
	}
}
