var documenterSearchIndex = {"docs": [

{
    "location": "index.html#",
    "page": "Home",
    "title": "Home",
    "category": "page",
    "text": ""
},

{
    "location": "index.html#Julia-Documentation-1",
    "page": "Home",
    "title": "Julia Documentation",
    "category": "section",
    "text": ""
},

{
    "location": "index.html#Manual-1",
    "page": "Home",
    "title": "Manual",
    "category": "section",
    "text": "Introduction\nGetting Started\nVariables\nIntegers and Floating-Point Numbers\nMathematical Operations and Elementary Functions\nComplex and Rational Numbers\nStrings\nFunctions\nControl Flow\nScope of Variables\nTypes\nMethods\nConstructors\nConversion and Promotion\nInterfaces\nModules\nDocumentation\nMetaprogramming\nMulti-dimensional Arrays\nLinear Algebra\nNetworking and Streams\nParallel Computing\nDate and DateTime\nRunning External Programs\nCalling C and Fortran Code\nHandling Operating System Variation\nEnvironment Variables\nInteracting With Julia\nEmbedding Julia\nPackages\nProfiling\nStack Traces\nPerformance Tips\nWorkflow Tips\nStyle Guide\nFrequently Asked Questions\nNoteworthy Differences from other Languages\nUnicode Input"
},

{
    "location": "index.html#Standard-Library-1",
    "page": "Home",
    "title": "Standard Library",
    "category": "section",
    "text": "Essentials\nCollections and Data Structures\nMathematics\nNumbers\nStrings\nArrays\nTasks and Parallel Computing\nLinear Algebra\nConstants\nFilesystem\nI/O and Network\nPunctuation\nSorting and Related Functions\nPackage Manager Functions\nDates and Time\nIteration utilities\nUnit Testing\nC Interface\nC Standard Library\nDynamic Linker\nProfiling\nStackTraces\nSIMD Support"
},

{
    "location": "index.html#Developer-Documentation-1",
    "page": "Home",
    "title": "Developer Documentation",
    "category": "section",
    "text": "Reflection and introspection\nDocumentation of Julia's Internals\nInitialization of the Julia runtime\nJulia ASTs\nMore about types\nMemory layout of Julia Objects\nEval of Julia code\nCalling Conventions\nHigh-level Overview of the Native-Code Generation Process\nJulia Functions\nBase.Cartesian\nTalking to the compiler (the :meta mechanism)\nSubArrays\nSystem Image Building\nWorking with LLVM\nprintf() and stdio in the Julia runtime\nBounds checking\nProper maintenance and care of multi-threading locks\nArrays with custom indices\nBase.LibGit2\nModule loading\nDeveloping/debugging Julia's C code\nReporting and analyzing crashes (segfaults)\ngdb debugging tips\nUsing Valgrind with Julia\nSanitizer support"
},

{
    "location": "manual/introduction.html#",
    "page": "Introduction",
    "title": "Introduction",
    "category": "page",
    "text": ""
},

{
    "location": "manual/introduction.html#man-introduction-1",
    "page": "Introduction",
    "title": "Introduction",
    "category": "section",
    "text": "Scientific computing has traditionally required the highest performance, yet domain experts have largely moved to slower dynamic languages for daily work. We believe there are many good reasons to prefer dynamic languages for these applications, and we do not expect their use to diminish. Fortunately, modern language design and compiler techniques make it possible to mostly eliminate the performance trade-off and provide a single environment productive enough for prototyping and efficient enough for deploying performance-intensive applications. The Julia programming language fills this role: it is a flexible dynamic language, appropriate for scientific and numerical computing, with performance comparable to traditional statically-typed languages.Because Julia's compiler is different from the interpreters used for languages like Python or R, you may find that Julia's performance is unintuitive at first. If you find that something is slow, we highly recommend reading through the Performance Tips section before trying anything else. Once you understand how Julia works, it's easy to write code that's nearly as fast as C.Julia features optional typing, multiple dispatch, and good performance, achieved using type inference and just-in-time (JIT) compilation, implemented using LLVM. It is multi-paradigm, combining features of imperative, functional, and object-oriented programming. Julia provides ease and expressiveness for high-level numerical computing, in the same way as languages such as R, MATLAB, and Python, but also supports general programming. To achieve this, Julia builds upon the lineage of mathematical programming languages, but also borrows much from popular dynamic languages, including Lisp, Perl, Python, Lua, and Ruby.The most significant departures of Julia from typical dynamic languages are:The core language imposes very little; the standard library is written in Julia itself, including primitive operations like integer arithmetic\nA rich language of types for constructing and describing objects, that can also optionally be used to make type declarations\nThe ability to define function behavior across many combinations of argument types via multiple dispatch\nAutomatic generation of efficient, specialized code for different argument types\nGood performance, approaching that of statically-compiled languages like CAlthough one sometimes speaks of dynamic languages as being \"typeless\", they are definitely not: every object, whether primitive or user-defined, has a type. The lack of type declarations in most dynamic languages, however, means that one cannot instruct the compiler about the types of values, and often cannot explicitly talk about types at all. In static languages, on the other hand, while one can – and usually must – annotate types for the compiler, types exist only at compile time and cannot be manipulated or expressed at run time. In Julia, types are themselves run-time objects, and can also be used to convey information to the compiler.While the casual programmer need not explicitly use types or multiple dispatch, they are the core unifying features of Julia: functions are defined on different combinations of argument types, and applied by dispatching to the most specific matching definition. This model is a good fit for mathematical programming, where it is unnatural for the first argument to \"own\" an operation as in traditional object-oriented dispatch. Operators are just functions with special notation – to extend addition to new user-defined data types, you define new methods for the + function. Existing code then seamlessly applies to the new data types.Partly because of run-time type inference (augmented by optional type annotations), and partly because of a strong focus on performance from the inception of the project, Julia's computational efficiency exceeds that of other dynamic languages, and even rivals that of statically-compiled languages. For large scale numerical problems, speed always has been, continues to be, and probably always will be crucial: the amount of data being processed has easily kept pace with Moore's Law over the past decades.Julia aims to create an unprecedented combination of ease-of-use, power, and efficiency in a single language. In addition to the above, some advantages of Julia over comparable systems include:Free and open source (MIT licensed)\nUser-defined types are as fast and compact as built-ins\nNo need to vectorize code for performance; devectorized code is fast\nDesigned for parallelism and distributed computation\nLightweight \"green\" threading (coroutines)\nUnobtrusive yet powerful type system\nElegant and extensible conversions and promotions for numeric and other types\nEfficient support for Unicode, including but not limited to UTF-8\nCall C functions directly (no wrappers or special APIs needed)\nPowerful shell-like capabilities for managing other processes\nLisp-like macros and other metaprogramming facilities"
},

{
    "location": "manual/getting-started.html#",
    "page": "Getting Started",
    "title": "Getting Started",
    "category": "page",
    "text": ""
},

{
    "location": "manual/getting-started.html#Getting-Started-1",
    "page": "Getting Started",
    "title": "Getting Started",
    "category": "section",
    "text": "Julia installation is straightforward, whether using precompiled binaries or compiling from source. Download and install Julia by following the instructions at https://julialang.org/downloads/.The easiest way to learn and experiment with Julia is by starting an interactive session (also known as a read-eval-print loop or \"repl\") by double-clicking the Julia executable or running julia from the command line:$ julia\n               _\n   _       _ _(_)_     |  A fresh approach to technical computing\n  (_)     | (_) (_)    |  Documentation: https://docs.julialang.org\n   _ _   _| |_  __ _   |  Type \"?help\" for help.\n  | | | | | | |/ _` |  |\n  | | |_| | | | (_| |  |  Version 0.5.0-dev+2440 (2016-02-01 02:22 UTC)\n _/ |\\__'_|_|_|\\__'_|  |  Commit 2bb94d6 (11 days old master)\n|__/                   |  x86_64-apple-darwin13.1.0\n\njulia> 1 + 2\n3\n\njulia> ans\n3To exit the interactive session, type ^D – the control key together with the d key or type quit(). When run in interactive mode, julia displays a banner and prompts the user for input. Once the user has entered a complete expression, such as 1 + 2, and hits enter, the interactive session evaluates the expression and shows its value. If an expression is entered into an interactive session with a trailing semicolon, its value is not shown. The variable ans is bound to the value of the last evaluated expression whether it is shown or not. The ans variable is only bound in interactive sessions, not when Julia code is run in other ways.To evaluate expressions written in a source file file.jl, write include(\"file.jl\").To run code in a file non-interactively, you can give it as the first argument to the julia command:$ julia script.jl arg1 arg2...As the example implies, the following command-line arguments to julia are taken as command-line arguments to the program script.jl, passed in the global constant ARGS. The name of the script itself is passed in as the global PROGRAM_FILE. Note that ARGS is also set when script code is given using the -e option on the command line (see the julia help output below) but PROGRAM_FILE will be empty. For example, to just print the arguments given to a script, you could do this:$ julia -e 'println(PROGRAM_FILE); for x in ARGS; println(x); end' foo bar\n\nfoo\nbarOr you could put that code into a script and run it:$ echo 'println(PROGRAM_FILE); for x in ARGS; println(x); end' > script.jl\n$ julia script.jl foo bar\nscript.jl\nfoo\nbarThe -- delimiter can be used to separate command-line args to the scriptfile from args to Julia:$ julia --color=yes -O -- foo.jl arg1 arg2..Julia can be started in parallel mode with either the -p or the --machinefile options. -p n will launch an additional n worker processes, while --machinefile file will launch a worker for each line in file file. The machines defined in file must be accessible via a passwordless ssh login, with Julia installed at the same location as the current host. Each machine definition takes the form [count*][user@]host[:port] [bind_addr[:port]] . user defaults to current user, port to the standard ssh port. count is the number of workers to spawn on the node, and defaults to 1. The optional bind-to bind_addr[:port] specifies the ip-address and port that other workers should use to connect to this worker.If you have code that you want executed whenever Julia is run, you can put it in ~/.juliarc.jl:$ echo 'println(\"Greetings! 你好! 안녕하세요?\")' > ~/.juliarc.jl\n$ julia\nGreetings! 你好! 안녕하세요?\n\n...There are various ways to run Julia code and provide options, similar to those available for the perl and ruby programs:julia [switches] -- [programfile] [args...]\n -v, --version             Display version information\n -h, --help                Print this message\n\n -J, --sysimage <file>     Start up with the given system image file\n --precompiled={yes|no}    Use precompiled code from system image if available\n --compilecache={yes|no}   Enable/disable incremental precompilation of modules\n -H, --home <dir>          Set location of `julia` executable\n --startup-file={yes|no}   Load ~/.juliarc.jl\n --handle-signals={yes|no} Enable or disable Julia's default signal handlers\n\n -e, --eval <expr>         Evaluate <expr>\n -E, --print <expr>        Evaluate and show <expr>\n -L, --load <file>         Load <file> immediately on all processors\n\n -p, --procs {N|auto}      Integer value N launches N additional local worker processes\n                           \"auto\" launches as many workers as the number of local cores\n --machinefile <file>      Run processes on hosts listed in <file>\n\n -i                        Interactive mode; REPL runs and isinteractive() is true\n -q, --quiet               Quiet startup (no banner)\n --color={yes|no}          Enable or disable color text\n --history-file={yes|no}   Load or save history\n\n --compile={yes|no|all|min}Enable or disable JIT compiler, or request exhaustive compilation\n -C, --cpu-target <target> Limit usage of cpu features up to <target>\n -O, --optimize={0,1,2,3}  Set the optimization level (default is 2 if unspecified or 3 if specified as -O)\n -g, -g <level>            Enable / Set the level of debug info generation (default is 1 if unspecified or 2 if specified as -g)\n --inline={yes|no}         Control whether inlining is permitted (overrides functions declared as @inline)\n --check-bounds={yes|no}   Emit bounds checks always or never (ignoring declarations)\n --math-mode={ieee,fast}   Disallow or enable unsafe floating point optimizations (overrides @fastmath declaration)\n\n --depwarn={yes|no|error}  Enable or disable syntax and method deprecation warnings (\"error\" turns warnings into errors)\n\n --output-o name           Generate an object file (including system image data)\n --output-ji name          Generate a system image data file (.ji)\n --output-bc name          Generate LLVM bitcode (.bc)\n --output-incremental=no   Generate an incremental output file (rather than complete)\n\n --code-coverage={none|user|all}, --code-coverage\n                           Count executions of source lines (omitting setting is equivalent to \"user\")\n --track-allocation={none|user|all}, --track-allocation\n                           Count bytes allocated by each source line"
},

{
    "location": "manual/getting-started.html#Resources-1",
    "page": "Getting Started",
    "title": "Resources",
    "category": "section",
    "text": "In addition to this manual, there are various other resources that may help new users get started with Julia:Julia and IJulia cheatsheet\nLearn Julia in a few minutes\nLearn Julia the Hard Way\nJulia by Example\nHands-on Julia\nTutorial for Homer Reid's numerical analysis class\nAn introductory presentation\nVideos from the Julia tutorial at MIT\nYouTube videos from the JuliaCons"
},

{
    "location": "manual/variables.html#",
    "page": "Variables",
    "title": "Variables",
    "category": "page",
    "text": ""
},

{
    "location": "manual/variables.html#Variables-1",
    "page": "Variables",
    "title": "Variables",
    "category": "section",
    "text": "A variable, in Julia, is a name associated (or bound) to a value. It's useful when you want to store a value (that you obtained after some math, for example) for later use. For example:# Assign the value 10 to the variable x\njulia> x = 10\n10\n\n# Doing math with x's value\njulia> x + 1\n11\n\n# Reassign x's value\njulia> x = 1 + 1\n2\n\n# You can assign values of other types, like strings of text\njulia> x = \"Hello World!\"\n\"Hello World!\"Julia provides an extremely flexible system for naming variables. Variable names are case-sensitive, and have no semantic meaning (that is, the language will not treat variables differently based on their names).julia> x = 1.0\n1.0\n\njulia> y = -3\n-3\n\njulia> Z = \"My string\"\n\"My string\"\n\njulia> customary_phrase = \"Hello world!\"\n\"Hello world!\"\n\njulia> UniversalDeclarationOfHumanRightsStart = \"人人生而自由，在尊严和权利上一律平等。\"\n\"人人生而自由，在尊严和权利上一律平等。\"Unicode names (in UTF-8 encoding) are allowed:julia> δ = 0.00001\n1.0e-5\n\njulia> 안녕하세요 = \"Hello\"\n\"Hello\"In the Julia REPL and several other Julia editing environments, you can type many Unicode math symbols by typing the backslashed LaTeX symbol name followed by tab. For example, the variable name δ can be entered by typing \\delta-tab, or even α̂₂ by \\alpha-tab-\\hat- tab-\\_2-tab. (If you find a symbol somewhere, e.g. in someone else's code, that you don't know how to type, the REPL help will tell you: just type ? and then paste the symbol.)Julia will even let you redefine built-in constants and functions if needed:julia> pi\nπ = 3.1415926535897...\n\njulia> pi = 3\nWARNING: imported binding for pi overwritten in module Main\n3\n\njulia> pi\n3\n\njulia> sqrt(100)\n10.0\n\njulia> sqrt = 4\nWARNING: imported binding for sqrt overwritten in module Main\n4However, this is obviously not recommended to avoid potential confusion."
},

{
    "location": "manual/variables.html#Allowed-Variable-Names-1",
    "page": "Variables",
    "title": "Allowed Variable Names",
    "category": "section",
    "text": "Variable names must begin with a letter (A-Z or a-z), underscore, or a subset of Unicode code points greater than 00A0; in particular, Unicode character categories Lu/Ll/Lt/Lm/Lo/Nl (letters), Sc/So (currency and other symbols), and a few other letter-like characters (e.g. a subset of the Sm math symbols) are allowed. Subsequent characters may also include ! and digits (0-9 and other characters in categories Nd/No), as well as other Unicode code points: diacritics and other modifying marks (categories Mn/Mc/Me/Sk), some punctuation connectors (category Pc), primes, and a few other characters.Operators like + are also valid identifiers, but are parsed specially. In some contexts, operators can be used just like variables; for example (+) refers to the addition function, and (+) = f will reassign it. Most of the Unicode infix operators (in category Sm), such as ⊕, are parsed as infix operators and are available for user-defined methods (e.g. you can use const ⊗ = kron to define ⊗ as an infix Kronecker product).The only explicitly disallowed names for variables are the names of built-in statements:julia> else = false\nERROR: syntax: unexpected \"else\"\n\njulia> try = \"No\"\nERROR: syntax: unexpected \"=\"Some Unicode characters are considered to be equivalent in identifiers. Different ways of entering Unicode combining characters (e.g., accents) are treated as equivalent (specifically, Julia identifiers are NFC-normalized). The Unicode characters ɛ (U+025B: Latin small letter open e) and µ (U+00B5: micro sign) are treated as equivalent to the corresponding Greek letters, because the former are easily accessible via some input methods."
},

{
    "location": "manual/variables.html#Stylistic-Conventions-1",
    "page": "Variables",
    "title": "Stylistic Conventions",
    "category": "section",
    "text": "While Julia imposes few restrictions on valid names, it has become useful to adopt the following conventions:Names of variables are in lower case.\nWord separation can be indicated by underscores ('_'), but use of underscores is discouraged unless the name would be hard to read otherwise.\nNames of Types and Modules begin with a capital letter and word separation is shown with upper camel case instead of underscores.\nNames of functions and macros are in lower case, without underscores.\nFunctions that write to their arguments have names that end in !. These are sometimes called \"mutating\" or \"in-place\" functions because they are intended to produce changes in their arguments after the function is called, not just return a value.For more information about stylistic conventions, see the Style Guide."
},

{
    "location": "manual/integers-and-floating-point-numbers.html#",
    "page": "Integers and Floating-Point Numbers",
    "title": "Integers and Floating-Point Numbers",
    "category": "page",
    "text": ""
},

{
    "location": "manual/integers-and-floating-point-numbers.html#Integers-and-Floating-Point-Numbers-1",
    "page": "Integers and Floating-Point Numbers",
    "title": "Integers and Floating-Point Numbers",
    "category": "section",
    "text": "Integers and floating-point values are the basic building blocks of arithmetic and computation. Built-in representations of such values are called numeric primitives, while representations of integers and floating-point numbers as immediate values in code are known as numeric literals. For example, 1 is an integer literal, while 1.0 is a floating-point literal; their binary in-memory representations as objects are numeric primitives.Julia provides a broad range of primitive numeric types, and a full complement of arithmetic and bitwise operators as well as standard mathematical functions are defined over them. These map directly onto numeric types and operations that are natively supported on modern computers, thus allowing Julia to take full advantage of computational resources. Additionally, Julia provides software support for Arbitrary Precision Arithmetic, which can handle operations on numeric values that cannot be represented effectively in native hardware representations, but at the cost of relatively slower performance.The following are Julia's primitive numeric types:Integer types:Type Signed? Number of bits Smallest value Largest value\nInt8 ✓ 8 -2^7 2^7 - 1\nUInt8   8 0 2^8 - 1\nInt16 ✓ 16 -2^15 2^15 - 1\nUInt16   16 0 2^16 - 1\nInt32 ✓ 32 -2^31 2^31 - 1\nUInt32   32 0 2^32 - 1\nInt64 ✓ 64 -2^63 2^63 - 1\nUInt64   64 0 2^64 - 1\nInt128 ✓ 128 -2^127 2^127 - 1\nUInt128   128 0 2^128 - 1\nBool N/A 8 false (0) true (1)Floating-point types:Type Precision Number of bits\nFloat16 half 16\nFloat32 single 32\nFloat64 double 64Additionally, full support for Complex and Rational Numbers is built on top of these primitive numeric types. All numeric types interoperate naturally without explicit casting, thanks to a flexible, user-extensible type promotion system."
},

{
    "location": "manual/integers-and-floating-point-numbers.html#Integers-1",
    "page": "Integers and Floating-Point Numbers",
    "title": "Integers",
    "category": "section",
    "text": "Literal integers are represented in the standard manner:julia> 1\n1\n\njulia> 1234\n1234The default type for an integer literal depends on whether the target system has a 32-bit architecture or a 64-bit architecture:# 32-bit system:\njulia> typeof(1)\nInt32\n\n# 64-bit system:\njulia> typeof(1)\nInt64The Julia internal variable Sys.WORD_SIZE indicates whether the target system is 32-bit or 64-bit:# 32-bit system:\njulia> Sys.WORD_SIZE\n32\n\n# 64-bit system:\njulia> Sys.WORD_SIZE\n64Julia also defines the types Int and UInt, which are aliases for the system's signed and unsigned native integer types respectively:# 32-bit system:\njulia> Int\nInt32\njulia> UInt\nUInt32\n\n# 64-bit system:\njulia> Int\nInt64\njulia> UInt\nUInt64Larger integer literals that cannot be represented using only 32 bits but can be represented in 64 bits always create 64-bit integers, regardless of the system type:# 32-bit or 64-bit system:\njulia> typeof(3000000000)\nInt64Unsigned integers are input and output using the 0x prefix and hexadecimal (base 16) digits 0-9a-f (the capitalized digits A-F also work for input). The size of the unsigned value is determined by the number of hex digits used:julia> 0x1\n0x01\n\njulia> typeof(ans)\nUInt8\n\njulia> 0x123\n0x0123\n\njulia> typeof(ans)\nUInt16\n\njulia> 0x1234567\n0x01234567\n\njulia> typeof(ans)\nUInt32\n\njulia> 0x123456789abcdef\n0x0123456789abcdef\n\njulia> typeof(ans)\nUInt64This behavior is based on the observation that when one uses unsigned hex literals for integer values, one typically is using them to represent a fixed numeric byte sequence, rather than just an integer value.Recall that the variable ans is set to the value of the last expression evaluated in an interactive session. This does not occur when Julia code is run in other ways.Binary and octal literals are also supported:julia> 0b10\n0x02\n\njulia> typeof(ans)\nUInt8\n\njulia> 0o10\n0x08\n\njulia> typeof(ans)\nUInt8The minimum and maximum representable values of primitive numeric types such as integers are given by the typemin() and typemax() functions:julia> (typemin(Int32), typemax(Int32))\n(-2147483648, 2147483647)\n\njulia> for T in [Int8,Int16,Int32,Int64,Int128,UInt8,UInt16,UInt32,UInt64,UInt128]\n           println(\"$(lpad(T,7)): [$(typemin(T)),$(typemax(T))]\")\n       end\n   Int8: [-128,127]\n  Int16: [-32768,32767]\n  Int32: [-2147483648,2147483647]\n  Int64: [-9223372036854775808,9223372036854775807]\n Int128: [-170141183460469231731687303715884105728,170141183460469231731687303715884105727]\n  UInt8: [0,255]\n UInt16: [0,65535]\n UInt32: [0,4294967295]\n UInt64: [0,18446744073709551615]\nUInt128: [0,340282366920938463463374607431768211455]The values returned by typemin() and typemax() are always of the given argument type. (The above expression uses several features we have yet to introduce, including for loops, Strings, and Interpolation, but should be easy enough to understand for users with some existing programming experience.)"
},

{
    "location": "manual/integers-and-floating-point-numbers.html#Overflow-behavior-1",
    "page": "Integers and Floating-Point Numbers",
    "title": "Overflow behavior",
    "category": "section",
    "text": "In Julia, exceeding the maximum representable value of a given type results in a wraparound behavior:julia> x = typemax(Int64)\n9223372036854775807\n\njulia> x + 1\n-9223372036854775808\n\njulia> x + 1 == typemin(Int64)\ntrueThus, arithmetic with Julia integers is actually a form of modular arithmetic. This reflects the characteristics of the underlying arithmetic of integers as implemented on modern computers. In applications where overflow is possible, explicit checking for wraparound produced by overflow is essential; otherwise, the BigInt type in Arbitrary Precision Arithmetic is recommended instead."
},

{
    "location": "manual/integers-and-floating-point-numbers.html#Division-errors-1",
    "page": "Integers and Floating-Point Numbers",
    "title": "Division errors",
    "category": "section",
    "text": "Integer division (the div function) has two exceptional cases: dividing by zero, and dividing the lowest negative number (typemin()) by -1. Both of these cases throw a DivideError. The remainder and modulus functions (rem and mod) throw a DivideError when their second argument is zero."
},

{
    "location": "manual/integers-and-floating-point-numbers.html#Floating-Point-Numbers-1",
    "page": "Integers and Floating-Point Numbers",
    "title": "Floating-Point Numbers",
    "category": "section",
    "text": "Literal floating-point numbers are represented in the standard formats:julia> 1.0\n1.0\n\njulia> 1.\n1.0\n\njulia> 0.5\n0.5\n\njulia> .5\n0.5\n\njulia> -1.23\n-1.23\n\njulia> 1e10\n1.0e10\n\njulia> 2.5e-4\n0.00025The above results are all Float64 values. Literal Float32 values can be entered by writing an f in place of e:julia> 0.5f0\n0.5f0\n\njulia> typeof(ans)\nFloat32\n\njulia> 2.5f-4\n0.00025f0Values can be converted to Float32 easily:julia> Float32(-1.5)\n-1.5f0\n\njulia> typeof(ans)\nFloat32Hexadecimal floating-point literals are also valid, but only as Float64 values:julia> 0x1p0\n1.0\n\njulia> 0x1.8p3\n12.0\n\njulia> 0x.4p-1\n0.125\n\njulia> typeof(ans)\nFloat64Half-precision floating-point numbers are also supported (Float16), but they are implemented in software and use Float32 for calculations.julia> sizeof(Float16(4.))\n2\n\njulia> 2*Float16(4.)\nFloat16(8.0)The underscore _ can be used as digit separator:julia> 10_000, 0.000_000_005, 0xdead_beef, 0b1011_0010\n(10000, 5.0e-9, 0xdeadbeef, 0xb2)"
},

{
    "location": "manual/integers-and-floating-point-numbers.html#Floating-point-zero-1",
    "page": "Integers and Floating-Point Numbers",
    "title": "Floating-point zero",
    "category": "section",
    "text": "Floating-point numbers have two zeros, positive zero and negative zero. They are equal to each other but have different binary representations, as can be seen using the bits function: :julia> 0.0 == -0.0\ntrue\n\njulia> bits(0.0)\n\"0000000000000000000000000000000000000000000000000000000000000000\"\n\njulia> bits(-0.0)\n\"1000000000000000000000000000000000000000000000000000000000000000\""
},

{
    "location": "manual/integers-and-floating-point-numbers.html#Special-floating-point-values-1",
    "page": "Integers and Floating-Point Numbers",
    "title": "Special floating-point values",
    "category": "section",
    "text": "There are three specified standard floating-point values that do not correspond to any point on the real number line:Float16 Float32 Float64 Name Description\nInf16 Inf32 Inf positive infinity a value greater than all finite floating-point values\n-Inf16 -Inf32 -Inf negative infinity a value less than all finite floating-point values\nNaN16 NaN32 NaN not a number a value not == to any floating-point value (including itself)For further discussion of how these non-finite floating-point values are ordered with respect to each other and other floats, see Numeric Comparisons. By the IEEE 754 standard, these floating-point values are the results of certain arithmetic operations:julia> 1/Inf\n0.0\n\njulia> 1/0\nInf\n\njulia> -5/0\n-Inf\n\njulia> 0.000001/0\nInf\n\njulia> 0/0\nNaN\n\njulia> 500 + Inf\nInf\n\njulia> 500 - Inf\n-Inf\n\njulia> Inf + Inf\nInf\n\njulia> Inf - Inf\nNaN\n\njulia> Inf * Inf\nInf\n\njulia> Inf / Inf\nNaN\n\njulia> 0 * Inf\nNaNThe typemin() and typemax() functions also apply to floating-point types:julia> (typemin(Float16),typemax(Float16))\n(-Inf16, Inf16)\n\njulia> (typemin(Float32),typemax(Float32))\n(-Inf32, Inf32)\n\njulia> (typemin(Float64),typemax(Float64))\n(-Inf, Inf)"
},

{
    "location": "manual/integers-and-floating-point-numbers.html#Machine-epsilon-1",
    "page": "Integers and Floating-Point Numbers",
    "title": "Machine epsilon",
    "category": "section",
    "text": "Most real numbers cannot be represented exactly with floating-point numbers, and so for many purposes it is important to know the distance between two adjacent representable floating-point numbers, which is often known as machine epsilon.Julia provides eps(), which gives the distance between 1.0 and the next larger representable floating-point value:julia> eps(Float32)\n1.1920929f-7\n\njulia> eps(Float64)\n2.220446049250313e-16\n\njulia> eps() # same as eps(Float64)\n2.220446049250313e-16These values are 2.0^-23 and 2.0^-52 as Float32 and Float64 values, respectively. The eps() function can also take a floating-point value as an argument, and gives the absolute difference between that value and the next representable floating point value. That is, eps(x) yields a value of the same type as x such that x + eps(x) is the next representable floating-point value larger than x:julia> eps(1.0)\n2.220446049250313e-16\n\njulia> eps(1000.)\n1.1368683772161603e-13\n\njulia> eps(1e-27)\n1.793662034335766e-43\n\njulia> eps(0.0)\n5.0e-324The distance between two adjacent representable floating-point numbers is not constant, but is smaller for smaller values and larger for larger values. In other words, the representable floating-point numbers are densest in the real number line near zero, and grow sparser exponentially as one moves farther away from zero. By definition, eps(1.0) is the same as eps(Float64) since 1.0 is a 64-bit floating-point value.Julia also provides the nextfloat() and prevfloat() functions which return the next largest or smallest representable floating-point number to the argument respectively:julia> x = 1.25f0\n1.25f0\n\njulia> nextfloat(x)\n1.2500001f0\n\njulia> prevfloat(x)\n1.2499999f0\n\njulia> bits(prevfloat(x))\n\"00111111100111111111111111111111\"\n\njulia> bits(x)\n\"00111111101000000000000000000000\"\n\njulia> bits(nextfloat(x))\n\"00111111101000000000000000000001\"This example highlights the general principle that the adjacent representable floating-point numbers also have adjacent binary integer representations."
},

{
    "location": "manual/integers-and-floating-point-numbers.html#Rounding-modes-1",
    "page": "Integers and Floating-Point Numbers",
    "title": "Rounding modes",
    "category": "section",
    "text": "If a number doesn't have an exact floating-point representation, it must be rounded to an appropriate representable value, however, if wanted, the manner in which this rounding is done can be changed according to the rounding modes presented in the IEEE 754 standard.julia> x = 1.1; y = 0.1;\n\njulia> x + y\n1.2000000000000002\n\njulia> setrounding(Float64,RoundDown) do\n           x + y\n       end\n1.2The default mode used is always RoundNearest, which rounds to the nearest representable value, with ties rounded towards the nearest value with an even least significant bit.warning: Warning\nRounding is generally only correct for basic arithmetic functions (+(), -(), *(), /() and sqrt()) and type conversion operations. Many other functions assume the default RoundNearest mode is set, and can give erroneous results when operating under other rounding modes."
},

{
    "location": "manual/integers-and-floating-point-numbers.html#Background-and-References-1",
    "page": "Integers and Floating-Point Numbers",
    "title": "Background and References",
    "category": "section",
    "text": "Floating-point arithmetic entails many subtleties which can be surprising to users who are unfamiliar with the low-level implementation details. However, these subtleties are described in detail in most books on scientific computation, and also in the following references:The definitive guide to floating point arithmetic is the IEEE 754-2008 Standard; however, it is not available for free online.\nFor a brief but lucid presentation of how floating-point numbers are represented, see John D. Cook's article on the subject as well as his introduction to some of the issues arising from how this representation differs in behavior from the idealized abstraction of real numbers.\nAlso recommended is Bruce Dawson's series of blog posts on floating-point numbers.\nFor an excellent, in-depth discussion of floating-point numbers and issues of numerical accuracy encountered when computing with them, see David Goldberg's paper What Every Computer Scientist Should Know About Floating-Point Arithmetic.\nFor even more extensive documentation of the history of, rationale for, and issues with floating-point numbers, as well as discussion of many other topics in numerical computing, see the collected writings of William Kahan, commonly known as the \"Father of Floating-Point\". Of particular interest may be An Interview with the Old Man of Floating-Point."
},

{
    "location": "manual/integers-and-floating-point-numbers.html#Arbitrary-Precision-Arithmetic-1",
    "page": "Integers and Floating-Point Numbers",
    "title": "Arbitrary Precision Arithmetic",
    "category": "section",
    "text": "To allow computations with arbitrary-precision integers and floating point numbers, Julia wraps the GNU Multiple Precision Arithmetic Library (GMP) and the GNU MPFR Library, respectively. The BigInt and BigFloat types are available in Julia for arbitrary precision integer and floating point numbers respectively.Constructors exist to create these types from primitive numerical types, and parse() can be used to construct them from AbstractStrings.  Once created, they participate in arithmetic with all other numeric types thanks to Julia's type promotion and conversion mechanism:julia> BigInt(typemax(Int64)) + 1\n9223372036854775808\n\njulia> parse(BigInt, \"123456789012345678901234567890\") + 1\n123456789012345678901234567891\n\njulia> parse(BigFloat, \"1.23456789012345678901\")\n1.234567890123456789010000000000000000000000000000000000000000000000000000000004\n\njulia> BigFloat(2.0^66) / 3\n2.459565876494606882133333333333333333333333333333333333333333333333333333333344e+19\n\njulia> factorial(BigInt(40))\n815915283247897734345611269596115894272000000000However, type promotion between the primitive types above and BigInt/BigFloat is not automatic and must be explicitly stated.julia> x = typemin(Int64)\n-9223372036854775808\n\njulia> x = x - 1\n9223372036854775807\n\njulia> typeof(x)\nInt64\n\njulia> y = BigInt(typemin(Int64))\n-9223372036854775808\n\njulia> y = y - 1\n-9223372036854775809\n\njulia> typeof(y)\nBigIntThe default precision (in number of bits of the significand) and rounding mode of BigFloat operations can be changed globally by calling setprecision() and setrounding(), and all further calculations will take these changes in account.  Alternatively, the precision or the rounding can be changed only within the execution of a particular block of code by using the same functions with a do block:julia> setrounding(BigFloat, RoundUp) do\n           BigFloat(1) + parse(BigFloat, \"0.1\")\n       end\n1.100000000000000000000000000000000000000000000000000000000000000000000000000003\n\njulia> setrounding(BigFloat, RoundDown) do\n           BigFloat(1) + parse(BigFloat, \"0.1\")\n       end\n1.099999999999999999999999999999999999999999999999999999999999999999999999999986\n\njulia> setprecision(40) do\n           BigFloat(1) + parse(BigFloat, \"0.1\")\n       end\n1.1000000000004"
},

{
    "location": "manual/integers-and-floating-point-numbers.html#man-numeric-literal-coefficients-1",
    "page": "Integers and Floating-Point Numbers",
    "title": "Numeric Literal Coefficients",
    "category": "section",
    "text": "To make common numeric formulas and expressions clearer, Julia allows variables to be immediately preceded by a numeric literal, implying multiplication. This makes writing polynomial expressions much cleaner:julia> x = 3\n3\n\njulia> 2x^2 - 3x + 1\n10\n\njulia> 1.5x^2 - .5x + 1\n13.0It also makes writing exponential functions more elegant:julia> 2^2x\n64The precedence of numeric literal coefficients is the same as that of unary operators such as negation. So 2^3x is parsed as 2^(3x), and 2x^3 is parsed as 2*(x^3).Numeric literals also work as coefficients to parenthesized expressions:julia> 2(x-1)^2 - 3(x-1) + 1\n3note: Note\nThe precedence of numeric literal coefficients used for implicit multiplication is higher than other binary operators such as multiplication (*), and division (/, \\, and //).  This means, for example, that 1 / 2im equals -0.5im and 6 // 2(2 + 1) equals 1 // 1.Additionally, parenthesized expressions can be used as coefficients to variables, implying multiplication of the expression by the variable:julia> (x-1)x\n6Neither juxtaposition of two parenthesized expressions, nor placing a variable before a parenthesized expression, however, can be used to imply multiplication:julia> (x-1)(x+1)\nERROR: MethodError: objects of type Int64 are not callable\n\njulia> x(x+1)\nERROR: MethodError: objects of type Int64 are not callableBoth expressions are interpreted as function application: any expression that is not a numeric literal, when immediately followed by a parenthetical, is interpreted as a function applied to the values in parentheses (see Functions for more about functions). Thus, in both of these cases, an error occurs since the left-hand value is not a function.The above syntactic enhancements significantly reduce the visual noise incurred when writing common mathematical formulae. Note that no whitespace may come between a numeric literal coefficient and the identifier or parenthesized expression which it multiplies."
},

{
    "location": "manual/integers-and-floating-point-numbers.html#Syntax-Conflicts-1",
    "page": "Integers and Floating-Point Numbers",
    "title": "Syntax Conflicts",
    "category": "section",
    "text": "Juxtaposed literal coefficient syntax may conflict with two numeric literal syntaxes: hexadecimal integer literals and engineering notation for floating-point literals. Here are some situations where syntactic conflicts arise:The hexadecimal integer literal expression 0xff could be interpreted as the numeric literal 0 multiplied by the variable xff.\nThe floating-point literal expression 1e10 could be interpreted as the numeric literal 1 multiplied by the variable e10, and similarly with the equivalent E form.In both cases, we resolve the ambiguity in favor of interpretation as a numeric literals:Expressions starting with 0x are always hexadecimal literals.\nExpressions starting with a numeric literal followed by e or E are always floating-point literals."
},

{
    "location": "manual/integers-and-floating-point-numbers.html#Literal-zero-and-one-1",
    "page": "Integers and Floating-Point Numbers",
    "title": "Literal zero and one",
    "category": "section",
    "text": "Julia provides functions which return literal 0 and 1 corresponding to a specified type or the type of a given variable.Function Description\nzero(x) Literal zero of type x or type of variable x\none(x) Literal one of type x or type of variable xThese functions are useful in Numeric Comparisons to avoid overhead from unnecessary type conversion.Examples:julia> zero(Float32)\n0.0f0\n\njulia> zero(1.0)\n0.0\n\njulia> one(Int32)\n1\n\njulia> one(BigFloat)\n1.000000000000000000000000000000000000000000000000000000000000000000000000000000"
},

{
    "location": "manual/mathematical-operations.html#",
    "page": "Mathematical Operations and Elementary Functions",
    "title": "Mathematical Operations and Elementary Functions",
    "category": "page",
    "text": ""
},

{
    "location": "manual/mathematical-operations.html#Mathematical-Operations-and-Elementary-Functions-1",
    "page": "Mathematical Operations and Elementary Functions",
    "title": "Mathematical Operations and Elementary Functions",
    "category": "section",
    "text": "Julia provides a complete collection of basic arithmetic and bitwise operators across all of its numeric primitive types, as well as providing portable, efficient implementations of a comprehensive collection of standard mathematical functions."
},

{
    "location": "manual/mathematical-operations.html#Arithmetic-Operators-1",
    "page": "Mathematical Operations and Elementary Functions",
    "title": "Arithmetic Operators",
    "category": "section",
    "text": "The following arithmetic operators are supported on all primitive numeric types:Expression Name Description\n+x unary plus the identity operation\n-x unary minus maps values to their additive inverses\nx + y binary plus performs addition\nx - y binary minus performs subtraction\nx * y times performs multiplication\nx / y divide performs division\nx \\ y inverse divide equivalent to y / x\nx ^ y power raises x to the yth power\nx % y remainder equivalent to rem(x,y)as well as the negation on Bool types:Expression Name Description\n!x negation changes true to false and vice versaJulia's promotion system makes arithmetic operations on mixtures of argument types \"just work\" naturally and automatically. See Conversion and Promotion for details of the promotion system.Here are some simple examples using arithmetic operators:julia> 1 + 2 + 3\n6\n\njulia> 1 - 2\n-1\n\njulia> 3*2/12\n0.5(By convention, we tend to space operators more tightly if they get applied before other nearby operators. For instance, we would generally write -x + 2 to reflect that first x gets negated, and then 2 is added to that result.)"
},

{
    "location": "manual/mathematical-operations.html#Bitwise-Operators-1",
    "page": "Mathematical Operations and Elementary Functions",
    "title": "Bitwise Operators",
    "category": "section",
    "text": "The following bitwise operators are supported on all primitive integer types:Expression Name\n~x bitwise not\nx & y bitwise and\nx | y bitwise or\nx ⊻ y bitwise xor (exclusive or)\nx >>> y logical shift right\nx >> y arithmetic shift right\nx << y logical/arithmetic shift leftHere are some examples with bitwise operators:julia> ~123\n-124\n\njulia> 123 & 234\n106\n\njulia> 123 | 234\n251\n\njulia> 123 ⊻ 234\n145\n\njulia> xor(123, 234)\n145\n\njulia> ~UInt32(123)\n0xffffff84\n\njulia> ~UInt8(123)\n0x84"
},

{
    "location": "manual/mathematical-operations.html#Updating-operators-1",
    "page": "Mathematical Operations and Elementary Functions",
    "title": "Updating operators",
    "category": "section",
    "text": "Every binary arithmetic and bitwise operator also has an updating version that assigns the result of the operation back into its left operand. The updating version of the binary operator is formed by placing a = immediately after the operator. For example, writing x += 3 is equivalent to writing x = x + 3:julia> x = 1\n1\n\njulia> x += 3\n4\n\njulia> x\n4The updating versions of all the binary arithmetic and bitwise operators are:+=  -=  *=  /=  \\=  ÷=  %=  ^=  &=  |=  ⊻=  >>>=  >>=  <<=note: Note\nAn updating operator rebinds the variable on the left-hand side. As a result, the type of the variable may change.julia> x = 0x01; typeof(x)\nUInt8\n\njulia> x *= 2 # Same as x = x * 2\n2\n\njulia> typeof(x)\nInt64"
},

{
    "location": "manual/mathematical-operations.html#man-dot-operators-1",
    "page": "Mathematical Operations and Elementary Functions",
    "title": "Vectorized \"dot\" operators",
    "category": "section",
    "text": "For every binary operation like ^, there is a corresponding \"dot\" operation .^ that is automatically defined to perform ^ element-by-element on arrays. For example, [1,2,3] ^ 3 is not defined, since there is no standard mathematical meaning to \"cubing\" an array, but [1,2,3] .^ 3 is defined as computing the elementwise (or \"vectorized\") result [1^3, 2^3, 3^3].  Similarly for unary operators like ! or √, there is a corresponding .√ that applies the operator elementwise.julia> [1,2,3] .^ 3\n3-element Array{Int64,1}:\n  1\n  8\n 27More specifically, a .^ b is parsed as the \"dot\" call (^).(a,b), which performs a broadcast operation: it can combine arrays and scalars, arrays of the same size (performing the operation elementwise), and even arrays of different shapes (e.g. combining row and column vectors to produce a matrix). Moreover, like all vectorized \"dot calls,\" these \"dot operators\" are fusing. For example, if you compute 2 .* A.^2 .+ sin.(A) (or equivalently @. 2A^2 + sin(A), using the @. macro) for an array A, it performs a single loop over A, computing 2a^2 + sin(a) for each element of A. In particular, nested dot calls like f.(g.(x)) are fused, and \"adjacent\" binary operators like x .+ 3 .* x.^2 are equivalent to nested dot calls (+).(x, (*).(3, (^).(x, 2))).Furthermore, \"dotted\" updating operators like a .+= b (or @. a += b) are parsed as a .= a .+ b, where .= is a fused in-place assignment operation (see the dot syntax documentation).Note the dot syntax is also applicable to user-defined operators. For example, if you define ⊗(A,B) = kron(A,B) to give a convenient infix syntax A ⊗ B for Kronecker products (kron), then [A,B] .⊗ [C,D] will compute [A⊗C, B⊗D] with no additional coding."
},

{
    "location": "manual/mathematical-operations.html#Numeric-Comparisons-1",
    "page": "Mathematical Operations and Elementary Functions",
    "title": "Numeric Comparisons",
    "category": "section",
    "text": "Standard comparison operations are defined for all the primitive numeric types:Operator Name\n== equality\n!=, ≠ inequality\n< less than\n<=, ≤ less than or equal to\n> greater than\n>=, ≥ greater than or equal toHere are some simple examples:julia> 1 == 1\ntrue\n\njulia> 1 == 2\nfalse\n\njulia> 1 != 2\ntrue\n\njulia> 1 == 1.0\ntrue\n\njulia> 1 < 2\ntrue\n\njulia> 1.0 > 3\nfalse\n\njulia> 1 >= 1.0\ntrue\n\njulia> -1 <= 1\ntrue\n\njulia> -1 <= -1\ntrue\n\njulia> -1 <= -2\nfalse\n\njulia> 3 < -0.5\nfalseIntegers are compared in the standard manner – by comparison of bits. Floating-point numbers are compared according to the IEEE 754 standard:Finite numbers are ordered in the usual manner.\nPositive zero is equal but not greater than negative zero.\nInf is equal to itself and greater than everything else except NaN.\n-Inf is equal to itself and less then everything else except NaN.\nNaN is not equal to, not less than, and not greater than anything, including itself.The last point is potentially surprising and thus worth noting:julia> NaN == NaN\nfalse\n\njulia> NaN != NaN\ntrue\n\njulia> NaN < NaN\nfalse\n\njulia> NaN > NaN\nfalseand can cause especial headaches with Arrays:julia> [1 NaN] == [1 NaN]\nfalseJulia provides additional functions to test numbers for special values, which can be useful in situations like hash key comparisons:Function Tests if\nisequal(x, y) x and y are identical\nisfinite(x) x is a finite number\nisinf(x) x is infinite\nisnan(x) x is not a numberisequal() considers NaNs equal to each other:julia> isequal(NaN, NaN)\ntrue\n\njulia> isequal([1 NaN], [1 NaN])\ntrue\n\njulia> isequal(NaN, NaN32)\ntrueisequal() can also be used to distinguish signed zeros:julia> -0.0 == 0.0\ntrue\n\njulia> isequal(-0.0, 0.0)\nfalseMixed-type comparisons between signed integers, unsigned integers, and floats can be tricky. A great deal of care has been taken to ensure that Julia does them correctly.For other types, isequal() defaults to calling ==(), so if you want to define equality for your own types then you only need to add a ==() method.  If you define your own equality function, you should probably define a corresponding hash() method to ensure that isequal(x,y) implies hash(x) == hash(y)."
},

{
    "location": "manual/mathematical-operations.html#Chaining-comparisons-1",
    "page": "Mathematical Operations and Elementary Functions",
    "title": "Chaining comparisons",
    "category": "section",
    "text": "Unlike most languages, with the notable exception of Python, comparisons can be arbitrarily chained:julia> 1 < 2 <= 2 < 3 == 3 > 2 >= 1 == 1 < 3 != 5\ntrueChaining comparisons is often quite convenient in numerical code. Chained comparisons use the && operator for scalar comparisons, and the & operator for elementwise comparisons, which allows them to work on arrays. For example, 0 .< A .< 1 gives a boolean array whose entries are true where the corresponding elements of A are between 0 and 1.Note the evaluation behavior of chained comparisons:julia> v(x) = (println(x); x)\nv (generic function with 1 method)\n\njulia> v(1) < v(2) <= v(3)\n2\n1\n3\ntrue\n\njulia> v(1) > v(2) <= v(3)\n2\n1\nfalseThe middle expression is only evaluated once, rather than twice as it would be if the expression were written as v(1) < v(2) && v(2) <= v(3). However, the order of evaluations in a chained comparison is undefined. It is strongly recommended not to use expressions with side effects (such as printing) in chained comparisons. If side effects are required, the short-circuit && operator should be used explicitly (see Short-Circuit Evaluation)."
},

{
    "location": "manual/mathematical-operations.html#Elementary-Functions-1",
    "page": "Mathematical Operations and Elementary Functions",
    "title": "Elementary Functions",
    "category": "section",
    "text": "Julia provides a comprehensive collection of mathematical functions and operators. These mathematical operations are defined over as broad a class of numerical values as permit sensible definitions, including integers, floating-point numbers, rationals, and complex numbers, wherever such definitions make sense.Moreover, these functions (like any Julia function) can be applied in \"vectorized\" fashion to arrays and other collections with the dot syntax f.(A), e.g. sin.(A) will compute the sine of each element of an array A."
},

{
    "location": "manual/mathematical-operations.html#Operator-Precedence-1",
    "page": "Mathematical Operations and Elementary Functions",
    "title": "Operator Precedence",
    "category": "section",
    "text": "Julia applies the following order of operations, from highest precedence to lowest:Category Operators\nSyntax . followed by ::\nExponentiation ^\nFractions //\nMultiplication * / % & \\\nBitshifts << >> >>>\nAddition + - | ⊻\nSyntax : .. followed by |>\nComparisons > < >= <= == === != !== <:\nControl flow && followed by || followed by ?\nAssignments = += -= *= /= //= \\= ^= ÷= %= |= &= ⊻= <<= >>= >>>=For a complete list of every Julia operator's precedence, see the top of this file: src/julia-parser.scmYou can also find the numerical precedence for any given operator via the built-in function Base.operator_precedence, where higher numbers take precedence:julia> Base.operator_precedence(:+), Base.operator_precedence(:*), Base.operator_precedence(:.)\n(9, 11, 15)\n\njulia> Base.operator_precedence(:+=), Base.operator_precedence(:(=))  # (Note the necessary parens on `:(=)`)\n(1, 1)"
},

{
    "location": "manual/mathematical-operations.html#Numerical-Conversions-1",
    "page": "Mathematical Operations and Elementary Functions",
    "title": "Numerical Conversions",
    "category": "section",
    "text": "Julia supports three forms of numerical conversion, which differ in their handling of inexact conversions.The notation T(x) or convert(T,x) converts x to a value of type T.\nIf T is a floating-point type, the result is the nearest representable value, which could be positive or negative infinity.\nIf T is an integer type, an InexactError is raised if x is not representable by T.\nx % T converts an integer x to a value of integer type T congruent to x modulo 2^n, where n is the number of bits in T. In other words, the binary representation is truncated to fit.\nThe Rounding functions take a type T as an optional argument. For example, round(Int,x) is a shorthand for Int(round(x)).The following examples show the different forms.julia> Int8(127)\n127\n\njulia> Int8(128)\nERROR: InexactError()\nStacktrace:\n [1] Int8(::Int64) at ./sysimg.jl:77\n\njulia> Int8(127.0)\n127\n\njulia> Int8(3.14)\nERROR: InexactError()\nStacktrace:\n [1] convert(::Type{Int8}, ::Float64) at ./float.jl:658\n [2] Int8(::Float64) at ./sysimg.jl:77\n\njulia> Int8(128.0)\nERROR: InexactError()\nStacktrace:\n [1] convert(::Type{Int8}, ::Float64) at ./float.jl:658\n [2] Int8(::Float64) at ./sysimg.jl:77\n\njulia> 127 % Int8\n127\n\njulia> 128 % Int8\n-128\n\njulia> round(Int8,127.4)\n127\n\njulia> round(Int8,127.6)\nERROR: InexactError()\nStacktrace:\n [1] trunc(::Type{Int8}, ::Float64) at ./float.jl:651\n [2] round(::Type{Int8}, ::Float64) at ./float.jl:337See Conversion and Promotion for how to define your own conversions and promotions."
},

{
    "location": "manual/mathematical-operations.html#Rounding-functions-1",
    "page": "Mathematical Operations and Elementary Functions",
    "title": "Rounding functions",
    "category": "section",
    "text": "Function Description Return type\nround(x) round x to the nearest integer typeof(x)\nround(T, x) round x to the nearest integer T\nfloor(x) round x towards -Inf typeof(x)\nfloor(T, x) round x towards -Inf T\nceil(x) round x towards +Inf typeof(x)\nceil(T, x) round x towards +Inf T\ntrunc(x) round x towards zero typeof(x)\ntrunc(T, x) round x towards zero T"
},

{
    "location": "manual/mathematical-operations.html#Division-functions-1",
    "page": "Mathematical Operations and Elementary Functions",
    "title": "Division functions",
    "category": "section",
    "text": "Function Description\ndiv(x,y) truncated division; quotient rounded towards zero\nfld(x,y) floored division; quotient rounded towards -Inf\ncld(x,y) ceiling division; quotient rounded towards +Inf\nrem(x,y) remainder; satisfies x == div(x,y)*y + rem(x,y); sign matches x\nmod(x,y) modulus; satisfies x == fld(x,y)*y + mod(x,y); sign matches y\nmod1(x,y) mod() with offset 1; returns r∈(0,y] for y>0 or r∈[y,0) for y<0, where mod(r, y) == mod(x, y)\nmod2pi(x) modulus with respect to 2pi;  0 <= mod2pi(x)    < 2pi\ndivrem(x,y) returns (div(x,y),rem(x,y))\nfldmod(x,y) returns (fld(x,y),mod(x,y))\ngcd(x,y...) greatest positive common divisor of x, y,...\nlcm(x,y...) least positive common multiple of x, y,..."
},

{
    "location": "manual/mathematical-operations.html#Sign-and-absolute-value-functions-1",
    "page": "Mathematical Operations and Elementary Functions",
    "title": "Sign and absolute value functions",
    "category": "section",
    "text": "Function Description\nabs(x) a positive value with the magnitude of x\nabs2(x) the squared magnitude of x\nsign(x) indicates the sign of x, returning -1, 0, or +1\nsignbit(x) indicates whether the sign bit is on (true) or off (false)\ncopysign(x,y) a value with the magnitude of x and the sign of y\nflipsign(x,y) a value with the magnitude of x and the sign of x*y"
},

{
    "location": "manual/mathematical-operations.html#Powers,-logs-and-roots-1",
    "page": "Mathematical Operations and Elementary Functions",
    "title": "Powers, logs and roots",
    "category": "section",
    "text": "Function Description\nsqrt(x), √x square root of x\ncbrt(x), ∛x cube root of x\nhypot(x,y) hypotenuse of right-angled triangle with other sides of length x and y\nexp(x) natural exponential function at x\nexpm1(x) accurate exp(x)-1 for x near zero\nldexp(x,n) x*2^n computed efficiently for integer values of n\nlog(x) natural logarithm of x\nlog(b,x) base b logarithm of x\nlog2(x) base 2 logarithm of x\nlog10(x) base 10 logarithm of x\nlog1p(x) accurate log(1+x) for x near zero\nexponent(x) binary exponent of x\nsignificand(x) binary significand (a.k.a. mantissa) of a floating-point number xFor an overview of why functions like hypot(), expm1(), and log1p() are necessary and useful, see John D. Cook's excellent pair of blog posts on the subject: expm1, log1p, erfc, and hypot."
},

{
    "location": "manual/mathematical-operations.html#Trigonometric-and-hyperbolic-functions-1",
    "page": "Mathematical Operations and Elementary Functions",
    "title": "Trigonometric and hyperbolic functions",
    "category": "section",
    "text": "All the standard trigonometric and hyperbolic functions are also defined:sin    cos    tan    cot    sec    csc\nsinh   cosh   tanh   coth   sech   csch\nasin   acos   atan   acot   asec   acsc\nasinh  acosh  atanh  acoth  asech  acsch\nsinc   cosc   atan2These are all single-argument functions, with the exception of atan2, which gives the angle in radians between the x-axis and the point specified by its arguments, interpreted as x and y coordinates.Additionally, sinpi(x) and cospi(x) are provided for more accurate computations of sin(pi*x) and cos(pi*x) respectively.In order to compute trigonometric functions with degrees instead of radians, suffix the function with d. For example, sind(x) computes the sine of x where x is specified in degrees. The complete list of trigonometric functions with degree variants is:sind   cosd   tand   cotd   secd   cscd\nasind  acosd  atand  acotd  asecd  acscd"
},

{
    "location": "manual/mathematical-operations.html#Special-functions-1",
    "page": "Mathematical Operations and Elementary Functions",
    "title": "Special functions",
    "category": "section",
    "text": "Function Description\ngamma(x) gamma function at x\nlgamma(x) accurate log(gamma(x)) for large x\nlfact(x) accurate log(factorial(x)) for large x; same as lgamma(x+1) for x > 1, zero otherwise\nbeta(x,y) beta function at x,y\nlbeta(x,y) accurate log(beta(x,y)) for large x or y"
},

{
    "location": "manual/complex-and-rational-numbers.html#",
    "page": "Complex and Rational Numbers",
    "title": "Complex and Rational Numbers",
    "category": "page",
    "text": ""
},

{
    "location": "manual/complex-and-rational-numbers.html#Complex-and-Rational-Numbers-1",
    "page": "Complex and Rational Numbers",
    "title": "Complex and Rational Numbers",
    "category": "section",
    "text": "Julia ships with predefined types representing both complex and rational numbers, and supports all standard Mathematical Operations and Elementary Functions on them. Conversion and Promotion are defined so that operations on any combination of predefined numeric types, whether primitive or composite, behave as expected."
},

{
    "location": "manual/complex-and-rational-numbers.html#Complex-Numbers-1",
    "page": "Complex and Rational Numbers",
    "title": "Complex Numbers",
    "category": "section",
    "text": "The global constant im is bound to the complex number i, representing the principal square root of -1. It was deemed harmful to co-opt the name i for a global constant, since it is such a popular index variable name. Since Julia allows numeric literals to be juxtaposed with identifiers as coefficients, this binding suffices to provide convenient syntax for complex numbers, similar to the traditional mathematical notation:julia> 1 + 2im\n1 + 2imYou can perform all the standard arithmetic operations with complex numbers:julia> (1 + 2im)*(2 - 3im)\n8 + 1im\n\njulia> (1 + 2im)/(1 - 2im)\n-0.6 + 0.8im\n\njulia> (1 + 2im) + (1 - 2im)\n2 + 0im\n\njulia> (-3 + 2im) - (5 - 1im)\n-8 + 3im\n\njulia> (-1 + 2im)^2\n-3 - 4im\n\njulia> (-1 + 2im)^2.5\n2.7296244647840084 - 6.960664459571898im\n\njulia> (-1 + 2im)^(1 + 1im)\n-0.27910381075826657 + 0.08708053414102428im\n\njulia> 3(2 - 5im)\n6 - 15im\n\njulia> 3(2 - 5im)^2\n-63 - 60im\n\njulia> 3(2 - 5im)^-1.0\n0.20689655172413796 + 0.5172413793103449imThe promotion mechanism ensures that combinations of operands of different types just work:julia> 2(1 - 1im)\n2 - 2im\n\njulia> (2 + 3im) - 1\n1 + 3im\n\njulia> (1 + 2im) + 0.5\n1.5 + 2.0im\n\njulia> (2 + 3im) - 0.5im\n2.0 + 2.5im\n\njulia> 0.75(1 + 2im)\n0.75 + 1.5im\n\njulia> (2 + 3im) / 2\n1.0 + 1.5im\n\njulia> (1 - 3im) / (2 + 2im)\n-0.5 - 1.0im\n\njulia> 2im^2\n-2 + 0im\n\njulia> 1 + 3/4im\n1.0 - 0.75imNote that 3/4im == 3/(4*im) == -(3/4*im), since a literal coefficient binds more tightly than division.Standard functions to manipulate complex values are provided:julia> z = 1 + 2im\n1 + 2im\n\njulia> real(1 + 2im) # real part of z\n1\n\njulia> imag(1 + 2im) # imaginary part of z\n2\n\njulia> conj(1 + 2im) # complex conjugate of z\n1 - 2im\n\njulia> abs(1 + 2im) # absolute value of z\n2.23606797749979\n\njulia> abs2(1 + 2im) # squared absolute value\n5\n\njulia> angle(1 + 2im) # phase angle in radians\n1.1071487177940904As usual, the absolute value (abs()) of a complex number is its distance from zero. abs2() gives the square of the absolute value, and is of particular use for complex numbers where it avoids taking a square root. angle() returns the phase angle in radians (also known as the argument or arg function). The full gamut of other Elementary Functions is also defined for complex numbers:julia> sqrt(1im)\n0.7071067811865476 + 0.7071067811865475im\n\njulia> sqrt(1 + 2im)\n1.272019649514069 + 0.7861513777574233im\n\njulia> cos(1 + 2im)\n2.0327230070196656 - 3.0518977991518im\n\njulia> exp(1 + 2im)\n-1.1312043837568135 + 2.4717266720048188im\n\njulia> sinh(1 + 2im)\n-0.4890562590412937 + 1.4031192506220405imNote that mathematical functions typically return real values when applied to real numbers and complex values when applied to complex numbers. For example, sqrt() behaves differently when applied to -1 versus -1 + 0im even though -1 == -1 + 0im:julia> sqrt(-1)\nERROR: DomainError:\nsqrt will only return a complex result if called with a complex argument. Try sqrt(complex(x)).\nStacktrace:\n [1] sqrt(::Int64) at ./math.jl:434\n\njulia> sqrt(-1 + 0im)\n0.0 + 1.0imThe literal numeric coefficient notation does not work when constructing a complex number from variables. Instead, the multiplication must be explicitly written out:julia> a = 1; b = 2; a + b*im\n1 + 2imHowever, this is not recommended; Use the complex() function instead to construct a complex value directly from its real and imaginary parts:julia> a = 1; b = 2; complex(a, b)\n1 + 2imThis construction avoids the multiplication and addition operations.Inf and NaN propagate through complex numbers in the real and imaginary parts of a complex number as described in the Special floating-point values section:julia> 1 + Inf*im\n1.0 + Inf*im\n\njulia> 1 + NaN*im\n1.0 + NaN*im"
},

{
    "location": "manual/complex-and-rational-numbers.html#Rational-Numbers-1",
    "page": "Complex and Rational Numbers",
    "title": "Rational Numbers",
    "category": "section",
    "text": "Julia has a rational number type to represent exact ratios of integers. Rationals are constructed using the // operator:julia> 2//3\n2//3If the numerator and denominator of a rational have common factors, they are reduced to lowest terms such that the denominator is non-negative:julia> 6//9\n2//3\n\njulia> -4//8\n-1//2\n\njulia> 5//-15\n-1//3\n\njulia> -4//-12\n1//3This normalized form for a ratio of integers is unique, so equality of rational values can be tested by checking for equality of the numerator and denominator. The standardized numerator and denominator of a rational value can be extracted using the numerator() and denominator() functions:julia> numerator(2//3)\n2\n\njulia> denominator(2//3)\n3Direct comparison of the numerator and denominator is generally not necessary, since the standard arithmetic and comparison operations are defined for rational values:julia> 2//3 == 6//9\ntrue\n\njulia> 2//3 == 9//27\nfalse\n\njulia> 3//7 < 1//2\ntrue\n\njulia> 3//4 > 2//3\ntrue\n\njulia> 2//4 + 1//6\n2//3\n\njulia> 5//12 - 1//4\n1//6\n\njulia> 5//8 * 3//12\n5//32\n\njulia> 6//5 / 10//7\n21//25Rationals can be easily converted to floating-point numbers:julia> float(3//4)\n0.75Conversion from rational to floating-point respects the following identity for any integral values of a and b, with the exception of the case a == 0 and b == 0:julia> a = 1; b = 2;\n\njulia> isequal(float(a//b), a/b)\ntrueConstructing infinite rational values is acceptable:julia> 5//0\n1//0\n\njulia> -3//0\n-1//0\n\njulia> typeof(ans)\nRational{Int64}Trying to construct a NaN rational value, however, is not:julia> 0//0\nERROR: ArgumentError: invalid rational: zero(Int64)//zero(Int64)\nStacktrace:\n [1] Rational{Int64}(::Int64, ::Int64) at ./rational.jl:13\n [2] //(::Int64, ::Int64) at ./rational.jl:40As usual, the promotion system makes interactions with other numeric types effortless:julia> 3//5 + 1\n8//5\n\njulia> 3//5 - 0.5\n0.09999999999999998\n\njulia> 2//7 * (1 + 2im)\n2//7 + 4//7*im\n\njulia> 2//7 * (1.5 + 2im)\n0.42857142857142855 + 0.5714285714285714im\n\njulia> 3//2 / (1 + 2im)\n3//10 - 3//5*im\n\njulia> 1//2 + 2im\n1//2 + 2//1*im\n\njulia> 1 + 2//3im\n1//1 - 2//3*im\n\njulia> 0.5 == 1//2\ntrue\n\njulia> 0.33 == 1//3\nfalse\n\njulia> 0.33 < 1//3\ntrue\n\njulia> 1//3 - 0.33\n0.0033333333333332993"
},

{
    "location": "manual/strings.html#",
    "page": "Strings",
    "title": "Strings",
    "category": "page",
    "text": ""
},

{
    "location": "manual/strings.html#man-strings-1",
    "page": "Strings",
    "title": "Strings",
    "category": "section",
    "text": "Strings are finite sequences of characters. Of course, the real trouble comes when one asks what a character is. The characters that English speakers are familiar with are the letters A, B, C, etc., together with numerals and common punctuation symbols. These characters are standardized together with a mapping to integer values between 0 and 127 by the ASCII standard. There are, of course, many other characters used in non-English languages, including variants of the ASCII characters with accents and other modifications, related scripts such as Cyrillic and Greek, and scripts completely unrelated to ASCII and English, including Arabic, Chinese, Hebrew, Hindi, Japanese, and Korean. The Unicode standard tackles the complexities of what exactly a character is, and is generally accepted as the definitive standard addressing this problem. Depending on your needs, you can either ignore these complexities entirely and just pretend that only ASCII characters exist, or you can write code that can handle any of the characters or encodings that one may encounter when handling non-ASCII text. Julia makes dealing with plain ASCII text simple and efficient, and handling Unicode is as simple and efficient as possible. In particular, you can write C-style string code to process ASCII strings, and they will work as expected, both in terms of performance and semantics. If such code encounters non-ASCII text, it will gracefully fail with a clear error message, rather than silently introducing corrupt results. When this happens, modifying the code to handle non-ASCII data is straightforward.There are a few noteworthy high-level features about Julia's strings:The built-in concrete type used for strings (and string literals) in Julia is String. This supports the full range of Unicode characters via the UTF-8 encoding. (A transcode() function is provided to convert to/from other Unicode encodings.)\nAll string types are subtypes of the abstract type AbstractString, and external packages define additional AbstractString subtypes (e.g. for other encodings).  If you define a function expecting a string argument, you should declare the type as AbstractString in order to accept any string type.\nLike C and Java, but unlike most dynamic languages, Julia has a first-class type representing a single character, called Char. This is just a special kind of 32-bit primitive type whose numeric value represents a Unicode code point.\nAs in Java, strings are immutable: the value of an AbstractString object cannot be changed. To construct a different string value, you construct a new string from parts of other strings.\nConceptually, a string is a partial function from indices to characters: for some index values, no character value is returned, and instead an exception is thrown. This allows for efficient indexing into strings by the byte index of an encoded representation rather than by a character index, which cannot be implemented both efficiently and simply for variable-width encodings of Unicode strings."
},

{
    "location": "manual/strings.html#man-characters-1",
    "page": "Strings",
    "title": "Characters",
    "category": "section",
    "text": "A Char value represents a single character: it is just a 32-bit primitive type with a special literal representation and appropriate arithmetic behaviors, whose numeric value is interpreted as a Unicode code point. Here is how Char values are input and shown:julia> 'x'\n'x': ASCII/Unicode U+0078 (category Ll: Letter, lowercase)\n\njulia> typeof(ans)\nCharYou can convert a Char to its integer value, i.e. code point, easily:julia> Int('x')\n120\n\njulia> typeof(ans)\nInt64On 32-bit architectures, typeof(ans) will be Int32. You can convert an integer value back to a Char just as easily:julia> Char(120)\n'x': ASCII/Unicode U+0078 (category Ll: Letter, lowercase)Not all integer values are valid Unicode code points, but for performance, the Char() conversion does not check that every character value is valid. If you want to check that each converted value is a valid code point, use the isvalid() function:julia> Char(0x110000)\n'\\U110000': Unicode U+110000 (category Cn: Other, not assigned)\n\njulia> isvalid(Char, 0x110000)\nfalseAs of this writing, the valid Unicode code points are U+00 through U+d7ff and U+e000 through U+10ffff. These have not all been assigned intelligible meanings yet, nor are they necessarily interpretable by applications, but all of these values are considered to be valid Unicode characters.You can input any Unicode character in single quotes using \\u followed by up to four hexadecimal digits or \\U followed by up to eight hexadecimal digits (the longest valid value only requires six):julia> '\\u0'\n'\\0': ASCII/Unicode U+0000 (category Cc: Other, control)\n\njulia> '\\u78'\n'x': ASCII/Unicode U+0078 (category Ll: Letter, lowercase)\n\njulia> '\\u2200'\n'∀': Unicode U+2200 (category Sm: Symbol, math)\n\njulia> '\\U10ffff'\n'\\U10ffff': Unicode U+10ffff (category Cn: Other, not assigned)Julia uses your system's locale and language settings to determine which characters can be printed as-is and which must be output using the generic, escaped \\u or \\U input forms. In addition to these Unicode escape forms, all of C's traditional escaped input forms can also be used:julia> Int('\\0')\n0\n\njulia> Int('\\t')\n9\n\njulia> Int('\\n')\n10\n\njulia> Int('\\e')\n27\n\njulia> Int('\\x7f')\n127\n\njulia> Int('\\177')\n127\n\njulia> Int('\\xff')\n255You can do comparisons and a limited amount of arithmetic with Char values:julia> 'A' < 'a'\ntrue\n\njulia> 'A' <= 'a' <= 'Z'\nfalse\n\njulia> 'A' <= 'X' <= 'Z'\ntrue\n\njulia> 'x' - 'a'\n23\n\njulia> 'A' + 1\n'B': ASCII/Unicode U+0042 (category Lu: Letter, uppercase)"
},

{
    "location": "manual/strings.html#String-Basics-1",
    "page": "Strings",
    "title": "String Basics",
    "category": "section",
    "text": "String literals are delimited by double quotes or triple double quotes:julia> str = \"Hello, world.\\n\"\n\"Hello, world.\\n\"\n\njulia> \"\"\"Contains \"quote\" characters\"\"\"\n\"Contains \\\"quote\\\" characters\"If you want to extract a character from a string, you index into it:julia> str[1]\n'H': ASCII/Unicode U+0048 (category Lu: Letter, uppercase)\n\njulia> str[6]\n',': ASCII/Unicode U+002c (category Po: Punctuation, other)\n\njulia> str[end]\n'\\n': ASCII/Unicode U+000a (category Cc: Other, control)All indexing in Julia is 1-based: the first element of any integer-indexed object is found at index 1. (As we will see below, this does not necessarily mean that the last element is found at index n, where n is the length of the string.)In any indexing expression, the keyword end can be used as a shorthand for the last index (computed by endof(str)). You can perform arithmetic and other operations with end, just like a normal value:julia> str[end-1]\n'.': ASCII/Unicode U+002e (category Po: Punctuation, other)\n\njulia> str[end÷2]\n' ': ASCII/Unicode U+0020 (category Zs: Separator, space)Using an index less than 1 or greater than end raises an error:julia> str[0]\nERROR: BoundsError: attempt to access \"Hello, world.\\n\"\n  at index [0]\n[...]\n\njulia> str[end+1]\nERROR: BoundsError: attempt to access \"Hello, world.\\n\"\n  at index [15]\n[...]You can also extract a substring using range indexing:julia> str[4:9]\n\"lo, wo\"Notice that the expressions str[k] and str[k:k] do not give the same result:julia> str[6]\n',': ASCII/Unicode U+002c (category Po: Punctuation, other)\n\njulia> str[6:6]\n\",\"The former is a single character value of type Char, while the latter is a string value that happens to contain only a single character. In Julia these are very different things."
},

{
    "location": "manual/strings.html#Unicode-and-UTF-8-1",
    "page": "Strings",
    "title": "Unicode and UTF-8",
    "category": "section",
    "text": "Julia fully supports Unicode characters and strings. As discussed above, in character literals, Unicode code points can be represented using Unicode \\u and \\U escape sequences, as well as all the standard C escape sequences. These can likewise be used to write string literals:julia> s = \"\\u2200 x \\u2203 y\"\n\"∀ x ∃ y\"Whether these Unicode characters are displayed as escapes or shown as special characters depends on your terminal's locale settings and its support for Unicode. String literals are encoded using the UTF-8 encoding. UTF-8 is a variable-width encoding, meaning that not all characters are encoded in the same number of bytes. In UTF-8, ASCII characters – i.e. those with code points less than 0x80 (128) – are encoded as they are in ASCII, using a single byte, while code points 0x80 and above are encoded using multiple bytes – up to four per character. This means that not every byte index into a UTF-8 string is necessarily a valid index for a character. If you index into a string at such an invalid byte index, an error is thrown:julia> s[1]\n'∀': Unicode U+2200 (category Sm: Symbol, math)\n\njulia> s[2]\nERROR: UnicodeError: invalid character index\n[...]\n\njulia> s[3]\nERROR: UnicodeError: invalid character index\n[...]\n\njulia> s[4]\n' ': ASCII/Unicode U+0020 (category Zs: Separator, space)In this case, the character ∀ is a three-byte character, so the indices 2 and 3 are invalid and the next character's index is 4; this next valid index can be computed by nextind(s,1), and the next index after that by nextind(s,4) and so on.Because of variable-length encodings, the number of characters in a string (given by length(s)) is not always the same as the last index. If you iterate through the indices 1 through endof(s) and index into s, the sequence of characters returned when errors aren't thrown is the sequence of characters comprising the string s. Thus we have the identity that length(s) <= endof(s), since each character in a string must have its own index. The following is an inefficient and verbose way to iterate through the characters of s:julia> for i = 1:endof(s)\n           try\n               println(s[i])\n           catch\n               # ignore the index error\n           end\n       end\n∀\n\nx\n\n∃\n\nyThe blank lines actually have spaces on them. Fortunately, the above awkward idiom is unnecessary for iterating through the characters in a string, since you can just use the string as an iterable object, no exception handling required:julia> for c in s\n           println(c)\n       end\n∀\n\nx\n\n∃\n\nyJulia uses the UTF-8 encoding by default, and support for new encodings can be added by packages. For example, the LegacyStrings.jl package implements UTF16String and UTF32String types. Additional discussion of other encodings and how to implement support for them is beyond the scope of this document for the time being. For further discussion of UTF-8 encoding issues, see the section below on byte array literals. The transcode() function is provided to convert data between the various UTF-xx encodings, primarily for working with external data and libraries."
},

{
    "location": "manual/strings.html#Concatenation-1",
    "page": "Strings",
    "title": "Concatenation",
    "category": "section",
    "text": "One of the most common and useful string operations is concatenation:julia> greet = \"Hello\"\n\"Hello\"\n\njulia> whom = \"world\"\n\"world\"\n\njulia> string(greet, \", \", whom, \".\\n\")\n\"Hello, world.\\n\"Julia also provides * for string concatenation:julia> greet * \", \" * whom * \".\\n\"\n\"Hello, world.\\n\"While * may seem like a surprising choice to users of languages that provide + for string concatenation, this use of * has precedent in mathematics, particularly in abstract algebra.In mathematics, + usually denotes a commutative operation, where the order of the operands does not matter. An example of this is matrix addition, where A + B == B + A for any matrices A and B that have the same shape. In contrast, * typically denotes a noncommutative operation, where the order of the operands does matter. An example of this is matrix multiplication, where in general A * B != B * A. As with matrix multiplication, string concatenation is noncommutative: greet * whom != whom * greet. As such, * is a more natural choice for an infix string concatenation operator, consistent with common mathematical use.More precisely, the set of all finite-length strings S together with the string concatenation operator * forms a free monoid (S, *). The identity element of this set is the empty string, \"\". Whenever a free monoid is not commutative, the operation is typically represented as \\cdot, *, or a similar symbol, rather than +, which as stated usually implies commutativity."
},

{
    "location": "manual/strings.html#string-interpolation-1",
    "page": "Strings",
    "title": "Interpolation",
    "category": "section",
    "text": "Constructing strings using concatenation can become a bit cumbersome, however. To reduce the need for these verbose calls to string() or repeated multiplications, Julia allows interpolation into string literals using $, as in Perl:julia> \"$greet, $whom.\\n\"\n\"Hello, world.\\n\"This is more readable and convenient and equivalent to the above string concatenation – the system rewrites this apparent single string literal into a concatenation of string literals with variables.The shortest complete expression after the $ is taken as the expression whose value is to be interpolated into the string. Thus, you can interpolate any expression into a string using parentheses:julia> \"1 + 2 = $(1 + 2)\"\n\"1 + 2 = 3\"Both concatenation and string interpolation call string() to convert objects into string form. Most non-AbstractString objects are converted to strings closely corresponding to how they are entered as literal expressions:julia> v = [1,2,3]\n3-element Array{Int64,1}:\n 1\n 2\n 3\n\njulia> \"v: $v\"\n\"v: [1, 2, 3]\"string() is the identity for AbstractString and Char values, so these are interpolated into strings as themselves, unquoted and unescaped:julia> c = 'x'\n'x': ASCII/Unicode U+0078 (category Ll: Letter, lowercase)\n\njulia> \"hi, $c\"\n\"hi, x\"To include a literal $ in a string literal, escape it with a backslash:julia> print(\"I have \\$100 in my account.\\n\")\nI have $100 in my account."
},

{
    "location": "manual/strings.html#Triple-Quoted-String-Literals-1",
    "page": "Strings",
    "title": "Triple-Quoted String Literals",
    "category": "section",
    "text": "When strings are created using triple-quotes (\"\"\"...\"\"\") they have some special behavior that can be useful for creating longer blocks of text. First, if the opening \"\"\" is followed by a newline, the newline is stripped from the resulting string.\"\"\"hello\"\"\"is equivalent to\"\"\"\nhello\"\"\"but\"\"\"\n\nhello\"\"\"will contain a literal newline at the beginning. Trailing whitespace is left unaltered. They can contain \" symbols without escaping. Triple-quoted strings are also dedented to the level of the least-indented line. This is useful for defining strings within code that is indented. For example:julia> str = \"\"\"\n           Hello,\n           world.\n         \"\"\"\n\"  Hello,\\n  world.\\n\"In this case the final (empty) line before the closing \"\"\" sets the indentation level.Note that line breaks in literal strings, whether single- or triple-quoted, result in a newline (LF) character \\n in the string, even if your editor uses a carriage return \\r (CR) or CRLF combination to end lines. To include a CR in a string, use an explicit escape \\r; for example, you can enter the literal string \"a CRLF line ending\\r\\n\"."
},

{
    "location": "manual/strings.html#Common-Operations-1",
    "page": "Strings",
    "title": "Common Operations",
    "category": "section",
    "text": "You can lexicographically compare strings using the standard comparison operators:julia> \"abracadabra\" < \"xylophone\"\ntrue\n\njulia> \"abracadabra\" == \"xylophone\"\nfalse\n\njulia> \"Hello, world.\" != \"Goodbye, world.\"\ntrue\n\njulia> \"1 + 2 = 3\" == \"1 + 2 = $(1 + 2)\"\ntrueYou can search for the index of a particular character using the search() function:julia> search(\"xylophone\", 'x')\n1\n\njulia> search(\"xylophone\", 'p')\n5\n\njulia> search(\"xylophone\", 'z')\n0You can start the search for a character at a given offset by providing a third argument:julia> search(\"xylophone\", 'o')\n4\n\njulia> search(\"xylophone\", 'o', 5)\n7\n\njulia> search(\"xylophone\", 'o', 8)\n0You can use the contains() function to check if a substring is contained in a string:julia> contains(\"Hello, world.\", \"world\")\ntrue\n\njulia> contains(\"Xylophon\", \"o\")\ntrue\n\njulia> contains(\"Xylophon\", \"a\")\nfalse\n\njulia> contains(\"Xylophon\", 'o')\nERROR: MethodError: no method matching contains(::String, ::Char)\nClosest candidates are:\n  contains(!Matched::Function, ::Any, !Matched::Any) at reduce.jl:664\n  contains(::AbstractString, !Matched::AbstractString) at strings/search.jl:378The last error is because 'o' is a character literal, and contains() is a generic function that looks for subsequences. To look for an element in a sequence, you must use in() instead.Two other handy string functions are repeat() and join():julia> repeat(\".:Z:.\", 10)\n\".:Z:..:Z:..:Z:..:Z:..:Z:..:Z:..:Z:..:Z:..:Z:..:Z:.\"\n\njulia> join([\"apples\", \"bananas\", \"pineapples\"], \", \", \" and \")\n\"apples, bananas and pineapples\"Some other useful functions include:endof(str) gives the maximal (byte) index that can be used to index into str.\nlength(str) the number of characters in str.\ni = start(str) gives the first valid index at which a character can be found in str (typically 1).\nc, j = next(str,i) returns next character at or after the index i and the next valid character index following that. With start() and endof(), can be used to iterate through the characters in str.\nind2chr(str,i) gives the number of characters in str up to and including any at index i.\nchr2ind(str,j) gives the index at which the jth character in str occurs."
},

{
    "location": "manual/strings.html#non-standard-string-literals-1",
    "page": "Strings",
    "title": "Non-Standard String Literals",
    "category": "section",
    "text": "There are situations when you want to construct a string or use string semantics, but the behavior of the standard string construct is not quite what is needed. For these kinds of situations, Julia provides non-standard string literals. A non-standard string literal looks like a regular double-quoted string literal, but is immediately prefixed by an identifier, and doesn't behave quite like a normal string literal.  Regular expressions, byte array literals and version number literals, as described below, are some examples of non-standard string literals. Other examples are given in the Metaprogramming section."
},

{
    "location": "manual/strings.html#Regular-Expressions-1",
    "page": "Strings",
    "title": "Regular Expressions",
    "category": "section",
    "text": "Julia has Perl-compatible regular expressions (regexes), as provided by the PCRE library. Regular expressions are related to strings in two ways: the obvious connection is that regular expressions are used to find regular patterns in strings; the other connection is that regular expressions are themselves input as strings, which are parsed into a state machine that can be used to efficiently search for patterns in strings. In Julia, regular expressions are input using non-standard string literals prefixed with various identifiers beginning with r. The most basic regular expression literal without any options turned on just uses r\"...\":julia> r\"^\\s*(?:#|$)\"\nr\"^\\s*(?:#|$)\"\n\njulia> typeof(ans)\nRegexTo check if a regex matches a string, use ismatch():julia> ismatch(r\"^\\s*(?:#|$)\", \"not a comment\")\nfalse\n\njulia> ismatch(r\"^\\s*(?:#|$)\", \"# a comment\")\ntrueAs one can see here, ismatch() simply returns true or false, indicating whether the given regex matches the string or not. Commonly, however, one wants to know not just whether a string matched, but also how it matched. To capture this information about a match, use the match() function instead:julia> match(r\"^\\s*(?:#|$)\", \"not a comment\")\n\njulia> match(r\"^\\s*(?:#|$)\", \"# a comment\")\nRegexMatch(\"#\")If the regular expression does not match the given string, match() returns nothing – a special value that does not print anything at the interactive prompt. Other than not printing, it is a completely normal value and you can test for it programmatically:m = match(r\"^\\s*(?:#|$)\", line)\nif m === nothing\n    println(\"not a comment\")\nelse\n    println(\"blank or comment\")\nendIf a regular expression does match, the value returned by match() is a RegexMatch object. These objects record how the expression matches, including the substring that the pattern matches and any captured substrings, if there are any. This example only captures the portion of the substring that matches, but perhaps we want to capture any non-blank text after the comment character. We could do the following:julia> m = match(r\"^\\s*(?:#\\s*(.*?)\\s*$|$)\", \"# a comment \")\nRegexMatch(\"# a comment \", 1=\"a comment\")When calling match(), you have the option to specify an index at which to start the search. For example:julia> m = match(r\"[0-9]\",\"aaaa1aaaa2aaaa3\",1)\nRegexMatch(\"1\")\n\njulia> m = match(r\"[0-9]\",\"aaaa1aaaa2aaaa3\",6)\nRegexMatch(\"2\")\n\njulia> m = match(r\"[0-9]\",\"aaaa1aaaa2aaaa3\",11)\nRegexMatch(\"3\")You can extract the following info from a RegexMatch object:the entire substring matched: m.match\nthe captured substrings as an array of strings: m.captures\nthe offset at which the whole match begins: m.offset\nthe offsets of the captured substrings as a vector: m.offsetsFor when a capture doesn't match, instead of a substring, m.captures contains nothing in that position, and m.offsets has a zero offset (recall that indices in Julia are 1-based, so a zero offset into a string is invalid). Here is a pair of somewhat contrived examples:julia> m = match(r\"(a|b)(c)?(d)\", \"acd\")\nRegexMatch(\"acd\", 1=\"a\", 2=\"c\", 3=\"d\")\n\njulia> m.match\n\"acd\"\n\njulia> m.captures\n3-element Array{Union{SubString{String}, Void},1}:\n \"a\"\n \"c\"\n \"d\"\n\njulia> m.offset\n1\n\njulia> m.offsets\n3-element Array{Int64,1}:\n 1\n 2\n 3\n\njulia> m = match(r\"(a|b)(c)?(d)\", \"ad\")\nRegexMatch(\"ad\", 1=\"a\", 2=nothing, 3=\"d\")\n\njulia> m.match\n\"ad\"\n\njulia> m.captures\n3-element Array{Union{SubString{String}, Void},1}:\n \"a\"\n nothing\n \"d\"\n\njulia> m.offset\n1\n\njulia> m.offsets\n3-element Array{Int64,1}:\n 1\n 0\n 2It is convenient to have captures returned as an array so that one can use destructuring syntax to bind them to local variables:julia> first, second, third = m.captures; first\n\"a\"Captures can also be accessed by indexing the RegexMatch object with the number or name of the capture group:julia> m=match(r\"(?<hour>\\d+):(?<minute>\\d+)\",\"12:45\")\nRegexMatch(\"12:45\", hour=\"12\", minute=\"45\")\n\njulia> m[:minute]\n\"45\"\n\njulia> m[2]\n\"45\"Captures can be referenced in a substitution string when using replace() by using \\n to refer to the nth capture group and prefixing the substitution string with s. Capture group 0 refers to the entire match object. Named capture groups can be referenced in the substitution with g<groupname>. For example:julia> replace(\"first second\", r\"(\\w+) (?<agroup>\\w+)\", s\"\\g<agroup> \\1\")\n\"second first\"Numbered capture groups can also be referenced as \\g<n> for disambiguation, as in:julia> replace(\"a\", r\".\", s\"\\g<0>1\")\n\"a1\"You can modify the behavior of regular expressions by some combination of the flags i, m, s, and x after the closing double quote mark. These flags have the same meaning as they do in Perl, as explained in this excerpt from the perlre manpage:i   Do case-insensitive pattern matching.\n\n    If locale matching rules are in effect, the case map is taken\n    from the current locale for code points less than 255, and\n    from Unicode rules for larger code points. However, matches\n    that would cross the Unicode rules/non-Unicode rules boundary\n    (ords 255/256) will not succeed.\n\nm   Treat string as multiple lines.  That is, change \"^\" and \"$\"\n    from matching the start or end of the string to matching the\n    start or end of any line anywhere within the string.\n\ns   Treat string as single line.  That is, change \".\" to match any\n    character whatsoever, even a newline, which normally it would\n    not match.\n\n    Used together, as r\"\"ms, they let the \".\" match any character\n    whatsoever, while still allowing \"^\" and \"$\" to match,\n    respectively, just after and just before newlines within the\n    string.\n\nx   Tells the regular expression parser to ignore most whitespace\n    that is neither backslashed nor within a character class. You\n    can use this to break up your regular expression into\n    (slightly) more readable parts. The '#' character is also\n    treated as a metacharacter introducing a comment, just as in\n    ordinary code.For example, the following regex has all three flags turned on:julia> r\"a+.*b+.*?d$\"ism\nr\"a+.*b+.*?d$\"ims\n\njulia> match(r\"a+.*b+.*?d$\"ism, \"Goodbye,\\nOh, angry,\\nBad world\\n\")\nRegexMatch(\"angry,\\nBad world\")Triple-quoted regex strings, of the form r\"\"\"...\"\"\", are also supported (and may be convenient for regular expressions containing quotation marks or newlines)."
},

{
    "location": "manual/strings.html#man-byte-array-literals-1",
    "page": "Strings",
    "title": "Byte Array Literals",
    "category": "section",
    "text": "Another useful non-standard string literal is the byte-array string literal: b\"...\". This form lets you use string notation to express literal byte arrays – i.e. arrays of UInt8 values. The rules for byte array literals are the following:ASCII characters and ASCII escapes produce a single byte.\n\\x and octal escape sequences produce the byte corresponding to the escape value.\nUnicode escape sequences produce a sequence of bytes encoding that code point in UTF-8.There is some overlap between these rules since the behavior of \\x and octal escapes less than 0x80 (128) are covered by both of the first two rules, but here these rules agree. Together, these rules allow one to easily use ASCII characters, arbitrary byte values, and UTF-8 sequences to produce arrays of bytes. Here is an example using all three:julia> b\"DATA\\xff\\u2200\"\n8-element Array{UInt8,1}:\n 0x44\n 0x41\n 0x54\n 0x41\n 0xff\n 0xe2\n 0x88\n 0x80The ASCII string \"DATA\" corresponds to the bytes 68, 65, 84, 65. \\xff produces the single byte 255. The Unicode escape \\u2200 is encoded in UTF-8 as the three bytes 226, 136, 128. Note that the resulting byte array does not correspond to a valid UTF-8 string – if you try to use this as a regular string literal, you will get a syntax error:julia> \"DATA\\xff\\u2200\"\nERROR: syntax: invalid UTF-8 sequenceAlso observe the significant distinction between \\xff and \\uff: the former escape sequence encodes the byte 255, whereas the latter escape sequence represents the code point 255, which is encoded as two bytes in UTF-8:julia> b\"\\xff\"\n1-element Array{UInt8,1}:\n 0xff\n\njulia> b\"\\uff\"\n2-element Array{UInt8,1}:\n 0xc3\n 0xbfIn character literals, this distinction is glossed over and \\xff is allowed to represent the code point 255, because characters always represent code points. In strings, however, \\x escapes always represent bytes, not code points, whereas \\u and \\U escapes always represent code points, which are encoded in one or more bytes. For code points less than \\u80, it happens that the UTF-8 encoding of each code point is just the single byte produced by the corresponding \\x escape, so the distinction can safely be ignored. For the escapes \\x80 through \\xff as compared to \\u80 through \\uff, however, there is a major difference: the former escapes all encode single bytes, which – unless followed by very specific continuation bytes – do not form valid UTF-8 data, whereas the latter escapes all represent Unicode code points with two-byte encodings.If this is all extremely confusing, try reading \"The Absolute Minimum Every Software Developer Absolutely, Positively Must Know About Unicode and Character Sets\". It's an excellent introduction to Unicode and UTF-8, and may help alleviate some confusion regarding the matter."
},

{
    "location": "manual/strings.html#man-version-number-literals-1",
    "page": "Strings",
    "title": "Version Number Literals",
    "category": "section",
    "text": "Version numbers can easily be expressed with non-standard string literals of the form v\"...\". Version number literals create VersionNumber objects which follow the specifications of semantic versioning, and therefore are composed of major, minor and patch numeric values, followed by pre-release and build alpha-numeric annotations. For example, v\"0.2.1-rc1+win64\" is broken into major version 0, minor version 2, patch version 1, pre-release rc1 and build win64. When entering a version literal, everything except the major version number is optional, therefore e.g.  v\"0.2\" is equivalent to v\"0.2.0\" (with empty pre-release/build annotations), v\"2\" is equivalent to v\"2.0.0\", and so on.VersionNumber objects are mostly useful to easily and correctly compare two (or more) versions. For example, the constant VERSION holds Julia version number as a VersionNumber object, and therefore one can define some version-specific behavior using simple statements as:if v\"0.2\" <= VERSION < v\"0.3-\"\n    # do something specific to 0.2 release series\nendNote that in the above example the non-standard version number v\"0.3-\" is used, with a trailing -: this notation is a Julia extension of the standard, and it's used to indicate a version which is lower than any 0.3 release, including all of its pre-releases. So in the above example the code would only run with stable 0.2 versions, and exclude such versions as v\"0.3.0-rc1\". In order to also allow for unstable (i.e. pre-release) 0.2 versions, the lower bound check should be modified like this: v\"0.2-\" <= VERSION.Another non-standard version specification extension allows one to use a trailing + to express an upper limit on build versions, e.g.  VERSION > v\"0.2-rc1+\" can be used to mean any version above 0.2-rc1 and any of its builds: it will return false for version v\"0.2-rc1+win64\" and true for v\"0.2-rc2\".It is good practice to use such special versions in comparisons (particularly, the trailing - should always be used on upper bounds unless there's a good reason not to), but they must not be used as the actual version number of anything, as they are invalid in the semantic versioning scheme.Besides being used for the VERSION constant, VersionNumber objects are widely used in the Pkg module, to specify packages versions and their dependencies."
},

{
    "location": "manual/strings.html#man-raw-string-literals-1",
    "page": "Strings",
    "title": "Raw String Literals",
    "category": "section",
    "text": "Raw strings without interpolation or unescaping can be expressed with non-standard string literals of the form raw\"...\". Raw string literals create ordinary String objects which contain the enclosed contents exactly as entered with no interpolation or unescaping. This is useful for strings which contain code or markup in other languages which use $ or \\ as special characters. The exception is quotation marks that still must be escaped, e.g. raw\"\\\"\" is equivalent to \"\\\"\"."
},

{
    "location": "manual/functions.html#",
    "page": "Functions",
    "title": "Functions",
    "category": "page",
    "text": ""
},

{
    "location": "manual/functions.html#man-functions-1",
    "page": "Functions",
    "title": "Functions",
    "category": "section",
    "text": "In Julia, a function is an object that maps a tuple of argument values to a return value. Julia functions are not pure mathematical functions, in the sense that functions can alter and be affected by the global state of the program. The basic syntax for defining functions in Julia is:julia> function f(x,y)\n           x + y\n       end\nf (generic function with 1 method)There is a second, more terse syntax for defining a function in Julia. The traditional function declaration syntax demonstrated above is equivalent to the following compact \"assignment form\":julia> f(x,y) = x + y\nf (generic function with 1 method)In the assignment form, the body of the function must be a single expression, although it can be a compound expression (see Compound Expressions). Short, simple function definitions are common in Julia. The short function syntax is accordingly quite idiomatic, considerably reducing both typing and visual noise.A function is called using the traditional parenthesis syntax:julia> f(2,3)\n5Without parentheses, the expression f refers to the function object, and can be passed around like any value:julia> g = f;\n\njulia> g(2,3)\n5As with variables, Unicode can also be used for function names:julia> ∑(x,y) = x + y\n∑ (generic function with 1 method)\n\njulia> ∑(2, 3)\n5"
},

{
    "location": "manual/functions.html#Argument-Passing-Behavior-1",
    "page": "Functions",
    "title": "Argument Passing Behavior",
    "category": "section",
    "text": "Julia function arguments follow a convention sometimes called \"pass-by-sharing\", which means that values are not copied when they are passed to functions. Function arguments themselves act as new variable bindings (new locations that can refer to values), but the values they refer to are identical to the passed values. Modifications to mutable values (such as Arrays) made within a function will be visible to the caller. This is the same behavior found in Scheme, most Lisps, Python, Ruby and Perl, among other dynamic languages."
},

{
    "location": "manual/functions.html#The-return-Keyword-1",
    "page": "Functions",
    "title": "The return Keyword",
    "category": "section",
    "text": "The value returned by a function is the value of the last expression evaluated, which, by default, is the last expression in the body of the function definition. In the example function, f, from the previous section this is the value of the expression x + y. As in C and most other imperative or functional languages, the return keyword causes a function to return immediately, providing an expression whose value is returned:function g(x,y)\n    return x * y\n    x + y\nendSince function definitions can be entered into interactive sessions, it is easy to compare these definitions:julia> f(x,y) = x + y\nf (generic function with 1 method)\n\njulia> function g(x,y)\n           return x * y\n           x + y\n       end\ng (generic function with 1 method)\n\njulia> f(2,3)\n5\n\njulia> g(2,3)\n6Of course, in a purely linear function body like g, the usage of return is pointless since the expression x + y is never evaluated and we could simply make x * y the last expression in the function and omit the return. In conjunction with other control flow, however, return is of real use. Here, for example, is a function that computes the hypotenuse length of a right triangle with sides of length x and y, avoiding overflow:julia> function hypot(x,y)\n           x = abs(x)\n           y = abs(y)\n           if x > y\n               r = y/x\n               return x*sqrt(1+r*r)\n           end\n           if y == 0\n               return zero(x)\n           end\n           r = x/y\n           return y*sqrt(1+r*r)\n       end\nhypot (generic function with 1 method)\n\njulia> hypot(3, 4)\n5.0There are three possible points of return from this function, returning the values of three different expressions, depending on the values of x and y. The return on the last line could be omitted since it is the last expression."
},

{
    "location": "manual/functions.html#Operators-Are-Functions-1",
    "page": "Functions",
    "title": "Operators Are Functions",
    "category": "section",
    "text": "In Julia, most operators are just functions with support for special syntax. (The exceptions are operators with special evaluation semantics like && and ||. These operators cannot be functions since Short-Circuit Evaluation requires that their operands are not evaluated before evaluation of the operator.) Accordingly, you can also apply them using parenthesized argument lists, just as you would any other function:julia> 1 + 2 + 3\n6\n\njulia> +(1,2,3)\n6The infix form is exactly equivalent to the function application form – in fact the former is parsed to produce the function call internally. This also means that you can assign and pass around operators such as +() and *() just like you would with other function values:julia> f = +;\n\njulia> f(1,2,3)\n6Under the name f, the function does not support infix notation, however."
},

{
    "location": "manual/functions.html#Operators-With-Special-Names-1",
    "page": "Functions",
    "title": "Operators With Special Names",
    "category": "section",
    "text": "A few special expressions correspond to calls to functions with non-obvious names. These are:Expression Calls\n[A B C ...] hcat()\n[A; B; C; ...] vcat()\n[A B; C D; ...] hvcat()\nA' ctranspose()\nA.' transpose()\n1:n colon()\nA[i] getindex()\nA[i]=x setindex!()These functions are included in the Base.Operators module even though they do not have operator-like names."
},

{
    "location": "manual/functions.html#man-anonymous-functions-1",
    "page": "Functions",
    "title": "Anonymous Functions",
    "category": "section",
    "text": "Functions in Julia are first-class objects: they can be assigned to variables, and called using the standard function call syntax from the variable they have been assigned to. They can be used as arguments, and they can be returned as values. They can also be created anonymously, without being given a name, using either of these syntaxes:julia> x -> x^2 + 2x - 1\n(::#1) (generic function with 1 method)\n\njulia> function (x)\n           x^2 + 2x - 1\n       end\n(::#3) (generic function with 1 method)This creates a function taking one argument x and returning the value of the polynomial x^2 + 2x - 1 at that value. Notice that the result is a generic function, but with a compiler-generated name based on consecutive numbering.The primary use for anonymous functions is passing them to functions which take other functions as arguments. A classic example is map(), which applies a function to each value of an array and returns a new array containing the resulting values:julia> map(round, [1.2,3.5,1.7])\n3-element Array{Float64,1}:\n 1.0\n 4.0\n 2.0This is fine if a named function effecting the transform one wants already exists to pass as the first argument to map(). Often, however, a ready-to-use, named function does not exist. In these situations, the anonymous function construct allows easy creation of a single-use function object without needing a name:julia> map(x -> x^2 + 2x - 1, [1,3,-1])\n3-element Array{Int64,1}:\n  2\n 14\n -2An anonymous function accepting multiple arguments can be written using the syntax (x,y,z)->2x+y-z. A zero-argument anonymous function is written as ()->3. The idea of a function with no arguments may seem strange, but is useful for \"delaying\" a computation. In this usage, a block of code is wrapped in a zero-argument function, which is later invoked by calling it as f()."
},

{
    "location": "manual/functions.html#Multiple-Return-Values-1",
    "page": "Functions",
    "title": "Multiple Return Values",
    "category": "section",
    "text": "In Julia, one returns a tuple of values to simulate returning multiple values. However, tuples can be created and destructured without needing parentheses, thereby providing an illusion that multiple values are being returned, rather than a single tuple value. For example, the following function returns a pair of values:julia> function foo(a,b)\n           a+b, a*b\n       end\nfoo (generic function with 1 method)If you call it in an interactive session without assigning the return value anywhere, you will see the tuple returned:julia> foo(2,3)\n(5, 6)A typical usage of such a pair of return values, however, extracts each value into a variable. Julia supports simple tuple \"destructuring\" that facilitates this:julia> x, y = foo(2,3)\n(5, 6)\n\njulia> x\n5\n\njulia> y\n6You can also return multiple values via an explicit usage of the return keyword:function foo(a,b)\n    return a+b, a*b\nendThis has the exact same effect as the previous definition of foo."
},

{
    "location": "manual/functions.html#Varargs-Functions-1",
    "page": "Functions",
    "title": "Varargs Functions",
    "category": "section",
    "text": "It is often convenient to be able to write functions taking an arbitrary number of arguments. Such functions are traditionally known as \"varargs\" functions, which is short for \"variable number of arguments\". You can define a varargs function by following the last argument with an ellipsis:julia> bar(a,b,x...) = (a,b,x)\nbar (generic function with 1 method)The variables a and b are bound to the first two argument values as usual, and the variable x is bound to an iterable collection of the zero or more values passed to bar after its first two arguments:julia> bar(1,2)\n(1, 2, ())\n\njulia> bar(1,2,3)\n(1, 2, (3,))\n\njulia> bar(1, 2, 3, 4)\n(1, 2, (3, 4))\n\njulia> bar(1,2,3,4,5,6)\n(1, 2, (3, 4, 5, 6))In all these cases, x is bound to a tuple of the trailing values passed to bar.It is possible to constrain the number of values passed as a variable argument; this will be discussed later in Parametrically-constrained Varargs methods.On the flip side, it is often handy to \"splice\" the values contained in an iterable collection into a function call as individual arguments. To do this, one also uses ... but in the function call instead:julia> x = (3, 4)\n(3, 4)\n\njulia> bar(1,2,x...)\n(1, 2, (3, 4))In this case a tuple of values is spliced into a varargs call precisely where the variable number of arguments go. This need not be the case, however:julia> x = (2, 3, 4)\n(2, 3, 4)\n\njulia> bar(1,x...)\n(1, 2, (3, 4))\n\njulia> x = (1, 2, 3, 4)\n(1, 2, 3, 4)\n\njulia> bar(x...)\n(1, 2, (3, 4))Furthermore, the iterable object spliced into a function call need not be a tuple:julia> x = [3,4]\n2-element Array{Int64,1}:\n 3\n 4\n\njulia> bar(1,2,x...)\n(1, 2, (3, 4))\n\njulia> x = [1,2,3,4]\n4-element Array{Int64,1}:\n 1\n 2\n 3\n 4\n\njulia> bar(x...)\n(1, 2, (3, 4))Also, the function that arguments are spliced into need not be a varargs function (although it often is):julia> baz(a,b) = a + b;\n\njulia> args = [1,2]\n2-element Array{Int64,1}:\n 1\n 2\n\njulia> baz(args...)\n3\n\njulia> args = [1,2,3]\n3-element Array{Int64,1}:\n 1\n 2\n 3\n\njulia> baz(args...)\nERROR: MethodError: no method matching baz(::Int64, ::Int64, ::Int64)\nClosest candidates are:\n  baz(::Any, ::Any) at none:1As you can see, if the wrong number of elements are in the spliced container, then the function call will fail, just as it would if too many arguments were given explicitly."
},

{
    "location": "manual/functions.html#Optional-Arguments-1",
    "page": "Functions",
    "title": "Optional Arguments",
    "category": "section",
    "text": "In many cases, function arguments have sensible default values and therefore might not need to be passed explicitly in every call. For example, the library function parse(T, num, base) interprets a string as a number in some base. The base argument defaults to 10. This behavior can be expressed concisely as:function parse(T, num, base=10)\n    ###\nendWith this definition, the function can be called with either two or three arguments, and 10 is automatically passed when a third argument is not specified:julia> parse(Int,\"12\",10)\n12\n\njulia> parse(Int,\"12\",3)\n5\n\njulia> parse(Int,\"12\")\n12Optional arguments are actually just a convenient syntax for writing multiple method definitions with different numbers of arguments (see Note on Optional and keyword Arguments)."
},

{
    "location": "manual/functions.html#Keyword-Arguments-1",
    "page": "Functions",
    "title": "Keyword Arguments",
    "category": "section",
    "text": "Some functions need a large number of arguments, or have a large number of behaviors. Remembering how to call such functions can be difficult. Keyword arguments can make these complex interfaces easier to use and extend by allowing arguments to be identified by name instead of only by position.For example, consider a function plot that plots a line. This function might have many options, for controlling line style, width, color, and so on. If it accepts keyword arguments, a possible call might look like plot(x, y, width=2), where we have chosen to specify only line width. Notice that this serves two purposes. The call is easier to read, since we can label an argument with its meaning. It also becomes possible to pass any subset of a large number of arguments, in any order.Functions with keyword arguments are defined using a semicolon in the signature:function plot(x, y; style=\"solid\", width=1, color=\"black\")\n    ###\nendWhen the function is called, the semicolon is optional: one can either call plot(x, y, width=2) or plot(x, y; width=2), but the former style is more common. An explicit semicolon is required only for passing varargs or computed keywords as described below.Keyword argument default values are evaluated only when necessary (when a corresponding keyword argument is not passed), and in left-to-right order. Therefore default expressions may refer to prior keyword arguments.The types of keyword arguments can be made explicit as follows:function f(;x::Int64=1)\n    ###\nendExtra keyword arguments can be collected using ..., as in varargs functions:function f(x; y=0, kwargs...)\n    ###\nendInside f, kwargs will be a collection of (key,value) tuples, where each key is a symbol. Such collections can be passed as keyword arguments using a semicolon in a call, e.g. f(x, z=1; kwargs...). Dictionaries can also be used for this purpose.One can also pass (key,value) tuples, or any iterable expression (such as a => pair) that can be assigned to such a tuple, explicitly after a semicolon. For example, plot(x, y; (:width,2)) and plot(x, y; :width => 2) are equivalent to plot(x, y, width=2). This is useful in situations where the keyword name is computed at runtime.The nature of keyword arguments makes it possible to specify the same argument more than once. For example, in the call plot(x, y; options..., width=2) it is possible that the options structure also contains a value for width. In such a case the rightmost occurrence takes precedence; in this example, width is certain to have the value 2."
},

{
    "location": "manual/functions.html#Evaluation-Scope-of-Default-Values-1",
    "page": "Functions",
    "title": "Evaluation Scope of Default Values",
    "category": "section",
    "text": "Optional and keyword arguments differ slightly in how their default values are evaluated. When optional argument default expressions are evaluated, only previous arguments are in scope. In contrast, all the arguments are in scope when keyword arguments default expressions are evaluated. For example, given this definition:function f(x, a=b, b=1)\n    ###\nendthe b in a=b refers to a b in an outer scope, not the subsequent argument b. However, if a and b were keyword arguments instead, then both would be created in the same scope and the b in a=b would refer to the subsequent argument b (shadowing any b in an outer scope), which would result in an undefined variable error (since the default expressions are evaluated left-to-right, and b has not been assigned yet)."
},

{
    "location": "manual/functions.html#Do-Block-Syntax-for-Function-Arguments-1",
    "page": "Functions",
    "title": "Do-Block Syntax for Function Arguments",
    "category": "section",
    "text": "Passing functions as arguments to other functions is a powerful technique, but the syntax for it is not always convenient. Such calls are especially awkward to write when the function argument requires multiple lines. As an example, consider calling map() on a function with several cases:map(x->begin\n           if x < 0 && iseven(x)\n               return 0\n           elseif x == 0\n               return 1\n           else\n               return x\n           end\n       end,\n    [A, B, C])Julia provides a reserved word do for rewriting this code more clearly:map([A, B, C]) do x\n    if x < 0 && iseven(x)\n        return 0\n    elseif x == 0\n        return 1\n    else\n        return x\n    end\nendThe do x syntax creates an anonymous function with argument x and passes it as the first argument to map(). Similarly, do a,b would create a two-argument anonymous function, and a plain do would declare that what follows is an anonymous function of the form () -> ....How these arguments are initialized depends on the \"outer\" function; here, map() will sequentially set x to A, B, C, calling the anonymous function on each, just as would happen in the syntax map(func, [A, B, C]).This syntax makes it easier to use functions to effectively extend the language, since calls look like normal code blocks. There are many possible uses quite different from map(), such as managing system state. For example, there is a version of open() that runs code ensuring that the opened file is eventually closed:open(\"outfile\", \"w\") do io\n    write(io, data)\nendThis is accomplished by the following definition:function open(f::Function, args...)\n    io = open(args...)\n    try\n        f(io)\n    finally\n        close(io)\n    end\nendHere, open() first opens the file for writing and then passes the resulting output stream to the anonymous function you defined in the do ... end block. After your function exits, open() will make sure that the stream is properly closed, regardless of whether your function exited normally or threw an exception. (The try/finally construct will be described in Control Flow.)With the do block syntax, it helps to check the documentation or implementation to know how the arguments of the user function are initialized."
},

{
    "location": "manual/functions.html#man-vectorized-1",
    "page": "Functions",
    "title": "Dot Syntax for Vectorizing Functions",
    "category": "section",
    "text": "In technical-computing languages, it is common to have \"vectorized\" versions of functions, which simply apply a given function f(x) to each element of an array A to yield a new array via f(A). This kind of syntax is convenient for data processing, but in other languages vectorization is also often required for performance: if loops are slow, the \"vectorized\" version of a function can call fast library code written in a low-level language. In Julia, vectorized functions are not required for performance, and indeed it is often beneficial to write your own loops (see Performance Tips), but they can still be convenient. Therefore, any Julia function f can be applied elementwise to any array (or other collection) with the syntax f.(A). For example sin can be applied to all elements in the vector A, like so:julia> A = [1.0, 2.0, 3.0]\n3-element Array{Float64,1}:\n 1.0\n 2.0\n 3.0\n\njulia> sin.(A)\n3-element Array{Float64,1}:\n 0.841471\n 0.909297\n 0.14112Of course, you can omit the dot if you write a specialized \"vector\" method of f, e.g. via f(A::AbstractArray) = map(f, A), and this is just as efficient as f.(A). But that approach requires you to decide in advance which functions you want to vectorize.More generally, f.(args...) is actually equivalent to broadcast(f, args...), which allows you to operate on multiple arrays (even of different shapes), or a mix of arrays and scalars (see Broadcasting). For example, if you have f(x,y) = 3x + 4y, then f.(pi,A) will return a new array consisting of f(pi,a) for each a in A, and f.(vector1,vector2) will return a new vector consisting of f(vector1[i],vector2[i]) for each index i (throwing an exception if the vectors have different length).julia> f(x,y) = 3x + 4y;\n\njulia> A = [1.0, 2.0, 3.0];\n\njulia> B = [4.0, 5.0, 6.0];\n\njulia> f.(pi, A)\n3-element Array{Float64,1}:\n 13.4248\n 17.4248\n 21.4248\n\njulia> f.(A, B)\n3-element Array{Float64,1}:\n 19.0\n 26.0\n 33.0Moreover, nested f.(args...) calls are fused into a single broadcast loop. For example, sin.(cos.(X)) is equivalent to broadcast(x -> sin(cos(x)), X), similar to [sin(cos(x)) for x in X]: there is only a single loop over X, and a single array is allocated for the result. [In contrast, sin(cos(X)) in a typical \"vectorized\" language would first allocate one temporary array for tmp=cos(X), and then compute sin(tmp) in a separate loop, allocating a second array.] This loop fusion is not a compiler optimization that may or may not occur, it is a syntactic guarantee whenever nested f.(args...) calls are encountered. Technically, the fusion stops as soon as a \"non-dot\" function call is encountered; for example, in sin.(sort(cos.(X))) the sin and cos loops cannot be merged because of the intervening sort function.Finally, the maximum efficiency is typically achieved when the output array of a vectorized operation is pre-allocated, so that repeated calls do not allocate new arrays over and over again for the results (Pre-allocating outputs:). A convenient syntax for this is X .= ..., which is equivalent to broadcast!(identity, X, ...) except that, as above, the broadcast! loop is fused with any nested \"dot\" calls. For example, X .= sin.(Y) is equivalent to broadcast!(sin, X, Y), overwriting X with sin.(Y) in-place. If the left-hand side is an array-indexing expression, e.g. X[2:end] .= sin.(Y), then it translates to broadcast! on a view, e.g. broadcast!(sin, view(X, 2:endof(X)), Y), so that the left-hand side is updated in-place.Since adding dots to many operations and function calls in an expression can be tedious and lead to code that is difficult to read, the macro @. is provided to convert every function call, operation, and assignment in an expression into the \"dotted\" version.julia> Y = [1.0, 2.0, 3.0, 4.0];\n\njulia> X = similar(Y); # pre-allocate output array\n\njulia> @. X = sin(cos(Y)) # equivalent to X .= sin.(cos.(Y))\n4-element Array{Float64,1}:\n  0.514395\n -0.404239\n -0.836022\n -0.608083Binary (or unary) operators like .+ are handled with the same mechanism: they are equivalent to broadcast calls and are fused with other nested \"dot\" calls.  X .+= Y etcetera is equivalent to X .= X .+ Y and results in a fused in-place assignment;  see also dot operators."
},

{
    "location": "manual/functions.html#Further-Reading-1",
    "page": "Functions",
    "title": "Further Reading",
    "category": "section",
    "text": "We should mention here that this is far from a complete picture of defining functions. Julia has a sophisticated type system and allows multiple dispatch on argument types. None of the examples given here provide any type annotations on their arguments, meaning that they are applicable to all types of arguments. The type system is described in Types and defining a function in terms of methods chosen by multiple dispatch on run-time argument types is described in Methods."
},

{
    "location": "manual/control-flow.html#",
    "page": "Control Flow",
    "title": "Control Flow",
    "category": "page",
    "text": ""
},

{
    "location": "manual/control-flow.html#Control-Flow-1",
    "page": "Control Flow",
    "title": "Control Flow",
    "category": "section",
    "text": "Julia provides a variety of control flow constructs:Compound Expressions: begin and (;).\nConditional Evaluation: if-elseif-else and ?: (ternary operator).\nShort-Circuit Evaluation: &&, || and chained comparisons.\nRepeated Evaluation: Loops: while and for.\nException Handling: try-catch, error() and throw().\nTasks (aka Coroutines): yieldto().The first five control flow mechanisms are standard to high-level programming languages. Tasks are not so standard: they provide non-local control flow, making it possible to switch between temporarily-suspended computations. This is a powerful construct: both exception handling and cooperative multitasking are implemented in Julia using tasks. Everyday programming requires no direct usage of tasks, but certain problems can be solved much more easily by using tasks."
},

{
    "location": "manual/control-flow.html#man-compound-expressions-1",
    "page": "Control Flow",
    "title": "Compound Expressions",
    "category": "section",
    "text": "Sometimes it is convenient to have a single expression which evaluates several subexpressions in order, returning the value of the last subexpression as its value. There are two Julia constructs that accomplish this: begin blocks and (;) chains. The value of both compound expression constructs is that of the last subexpression. Here's an example of a begin block:julia> z = begin\n           x = 1\n           y = 2\n           x + y\n       end\n3Since these are fairly small, simple expressions, they could easily be placed onto a single line, which is where the (;) chain syntax comes in handy:julia> z = (x = 1; y = 2; x + y)\n3This syntax is particularly useful with the terse single-line function definition form introduced in Functions. Although it is typical, there is no requirement that begin blocks be multiline or that (;) chains be single-line:julia> begin x = 1; y = 2; x + y end\n3\n\njulia> (x = 1;\n        y = 2;\n        x + y)\n3"
},

{
    "location": "manual/control-flow.html#man-conditional-evaluation-1",
    "page": "Control Flow",
    "title": "Conditional Evaluation",
    "category": "section",
    "text": "Conditional evaluation allows portions of code to be evaluated or not evaluated depending on the value of a boolean expression. Here is the anatomy of the if-elseif-else conditional syntax:if x < y\n    println(\"x is less than y\")\nelseif x > y\n    println(\"x is greater than y\")\nelse\n    println(\"x is equal to y\")\nendIf the condition expression x < y is true, then the corresponding block is evaluated; otherwise the condition expression x > y is evaluated, and if it is true, the corresponding block is evaluated; if neither expression is true, the else block is evaluated. Here it is in action:julia> function test(x, y)\n           if x < y\n               println(\"x is less than y\")\n           elseif x > y\n               println(\"x is greater than y\")\n           else\n               println(\"x is equal to y\")\n           end\n       end\ntest (generic function with 1 method)\n\njulia> test(1, 2)\nx is less than y\n\njulia> test(2, 1)\nx is greater than y\n\njulia> test(1, 1)\nx is equal to yThe elseif and else blocks are optional, and as many elseif blocks as desired can be used. The condition expressions in the if-elseif-else construct are evaluated until the first one evaluates to true, after which the associated block is evaluated, and no further condition expressions or blocks are evaluated.if blocks are \"leaky\", i.e. they do not introduce a local scope. This means that new variables defined inside the if clauses can be used after the if block, even if they weren't defined before. So, we could have defined the test function above asjulia> function test(x,y)\n           if x < y\n               relation = \"less than\"\n           elseif x == y\n               relation = \"equal to\"\n           else\n               relation = \"greater than\"\n           end\n           println(\"x is \", relation, \" y.\")\n       end\ntest (generic function with 1 method)\n\njulia> test(2, 1)\nx is greater than y.The variable relation is declared inside the if block, but used outside. However, when depending on this behavior, make sure all possible code paths define a value for the variable. The following change to the above function results in a runtime errorjulia> function test(x,y)\n           if x < y\n               relation = \"less than\"\n           elseif x == y\n               relation = \"equal to\"\n           end\n           println(\"x is \", relation, \" y.\")\n       end\ntest (generic function with 1 method)\n\njulia> test(1,2)\nx is less than y.\n\njulia> test(2,1)\nERROR: UndefVarError: relation not defined\nStacktrace:\n [1] test(::Int64, ::Int64) at ./none:7if blocks also return a value, which may seem unintuitive to users coming from many other languages. This value is simply the return value of the last executed statement in the branch that was chosen, sojulia> x = 3\n3\n\njulia> if x > 0\n           \"positive!\"\n       else\n           \"negative...\"\n       end\n\"positive!\"Note that very short conditional statements (one-liners) are frequently expressed using Short-Circuit Evaluation in Julia, as outlined in the next section.Unlike C, MATLAB, Perl, Python, and Ruby – but like Java, and a few other stricter, typed languages – it is an error if the value of a conditional expression is anything but true or false:julia> if 1\n           println(\"true\")\n       end\nERROR: TypeError: non-boolean (Int64) used in boolean contextThis error indicates that the conditional was of the wrong type: Int64 rather than the required Bool.The so-called \"ternary operator\", ?:, is closely related to the if-elseif-else syntax, but is used where a conditional choice between single expression values is required, as opposed to conditional execution of longer blocks of code. It gets its name from being the only operator in most languages taking three operands:a ? b : cThe expression a, before the ?, is a condition expression, and the ternary operation evaluates the expression b, before the :, if the condition a is true or the expression c, after the :, if it is false.The easiest way to understand this behavior is to see an example. In the previous example, the println call is shared by all three branches: the only real choice is which literal string to print. This could be written more concisely using the ternary operator. For the sake of clarity, let's try a two-way version first:julia> x = 1; y = 2;\n\njulia> println(x < y ? \"less than\" : \"not less than\")\nless than\n\njulia> x = 1; y = 0;\n\njulia> println(x < y ? \"less than\" : \"not less than\")\nnot less thanIf the expression x < y is true, the entire ternary operator expression evaluates to the string \"less than\" and otherwise it evaluates to the string \"not less than\". The original three-way example requires chaining multiple uses of the ternary operator together:julia> test(x, y) = println(x < y ? \"x is less than y\"    :\n                            x > y ? \"x is greater than y\" : \"x is equal to y\")\ntest (generic function with 1 method)\n\njulia> test(1, 2)\nx is less than y\n\njulia> test(2, 1)\nx is greater than y\n\njulia> test(1, 1)\nx is equal to yTo facilitate chaining, the operator associates from right to left.It is significant that like if-elseif-else, the expressions before and after the : are only evaluated if the condition expression evaluates to true or false, respectively:julia> v(x) = (println(x); x)\nv (generic function with 1 method)\n\njulia> 1 < 2 ? v(\"yes\") : v(\"no\")\nyes\n\"yes\"\n\njulia> 1 > 2 ? v(\"yes\") : v(\"no\")\nno\n\"no\""
},

{
    "location": "manual/control-flow.html#Short-Circuit-Evaluation-1",
    "page": "Control Flow",
    "title": "Short-Circuit Evaluation",
    "category": "section",
    "text": "Short-circuit evaluation is quite similar to conditional evaluation. The behavior is found in most imperative programming languages having the && and || boolean operators: in a series of boolean expressions connected by these operators, only the minimum number of expressions are evaluated as are necessary to determine the final boolean value of the entire chain. Explicitly, this means that:In the expression a && b, the subexpression b is only evaluated if a evaluates to true.\nIn the expression a || b, the subexpression b is only evaluated if a evaluates to false.The reasoning is that a && b must be false if a is false, regardless of the value of b, and likewise, the value of a || b must be true if a is true, regardless of the value of b. Both && and || associate to the right, but && has higher precedence than || does. It's easy to experiment with this behavior:julia> t(x) = (println(x); true)\nt (generic function with 1 method)\n\njulia> f(x) = (println(x); false)\nf (generic function with 1 method)\n\njulia> t(1) && t(2)\n1\n2\ntrue\n\njulia> t(1) && f(2)\n1\n2\nfalse\n\njulia> f(1) && t(2)\n1\nfalse\n\njulia> f(1) && f(2)\n1\nfalse\n\njulia> t(1) || t(2)\n1\ntrue\n\njulia> t(1) || f(2)\n1\ntrue\n\njulia> f(1) || t(2)\n1\n2\ntrue\n\njulia> f(1) || f(2)\n1\n2\nfalseYou can easily experiment in the same way with the associativity and precedence of various combinations of && and || operators.This behavior is frequently used in Julia to form an alternative to very short if statements. Instead of if <cond> <statement> end, one can write <cond> && <statement> (which could be read as: <cond> and then <statement>). Similarly, instead of if ! <cond> <statement> end, one can write <cond> || <statement> (which could be read as: <cond> or else <statement>).For example, a recursive factorial routine could be defined like this:julia> function fact(n::Int)\n           n >= 0 || error(\"n must be non-negative\")\n           n == 0 && return 1\n           n * fact(n-1)\n       end\nfact (generic function with 1 method)\n\njulia> fact(5)\n120\n\njulia> fact(0)\n1\n\njulia> fact(-1)\nERROR: n must be non-negative\nStacktrace:\n [1] fact(::Int64) at ./none:2Boolean operations without short-circuit evaluation can be done with the bitwise boolean operators introduced in Mathematical Operations and Elementary Functions: & and |. These are normal functions, which happen to support infix operator syntax, but always evaluate their arguments:julia> f(1) & t(2)\n1\n2\nfalse\n\njulia> t(1) | t(2)\n1\n2\ntrueJust like condition expressions used in if, elseif or the ternary operator, the operands of && or || must be boolean values (true or false). Using a non-boolean value anywhere except for the last entry in a conditional chain is an error:julia> 1 && true\nERROR: TypeError: non-boolean (Int64) used in boolean contextOn the other hand, any type of expression can be used at the end of a conditional chain. It will be evaluated and returned depending on the preceding conditionals:julia> true && (x = (1, 2, 3))\n(1, 2, 3)\n\njulia> false && (x = (1, 2, 3))\nfalse"
},

{
    "location": "manual/control-flow.html#man-loops-1",
    "page": "Control Flow",
    "title": "Repeated Evaluation: Loops",
    "category": "section",
    "text": "There are two constructs for repeated evaluation of expressions: the while loop and the for loop. Here is an example of a while loop:julia> i = 1;\n\njulia> while i <= 5\n           println(i)\n           i += 1\n       end\n1\n2\n3\n4\n5The while loop evaluates the condition expression (i <= 5 in this case), and as long it remains true, keeps also evaluating the body of the while loop. If the condition expression is false when the while loop is first reached, the body is never evaluated.The for loop makes common repeated evaluation idioms easier to write. Since counting up and down like the above while loop does is so common, it can be expressed more concisely with a for loop:julia> for i = 1:5\n           println(i)\n       end\n1\n2\n3\n4\n5Here the 1:5 is a Range object, representing the sequence of numbers 1, 2, 3, 4, 5. The for loop iterates through these values, assigning each one in turn to the variable i. One rather important distinction between the previous while loop form and the for loop form is the scope during which the variable is visible. If the variable i has not been introduced in an other scope, in the for loop form, it is visible only inside of the for loop, and not afterwards. You'll either need a new interactive session instance or a different variable name to test this:julia> for j = 1:5\n           println(j)\n       end\n1\n2\n3\n4\n5\n\njulia> j\nERROR: UndefVarError: j not definedSee Scope of Variables for a detailed explanation of variable scope and how it works in Julia.In general, the for loop construct can iterate over any container. In these cases, the alternative (but fully equivalent) keyword in or ∈ is typically used instead of =, since it makes the code read more clearly:julia> for i in [1,4,0]\n           println(i)\n       end\n1\n4\n0\n\njulia> for s ∈ [\"foo\",\"bar\",\"baz\"]\n           println(s)\n       end\nfoo\nbar\nbazVarious types of iterable containers will be introduced and discussed in later sections of the manual (see, e.g., Multi-dimensional Arrays).It is sometimes convenient to terminate the repetition of a while before the test condition is falsified or stop iterating in a for loop before the end of the iterable object is reached. This can be accomplished with the break keyword:julia> i = 1;\n\njulia> while true\n           println(i)\n           if i >= 5\n               break\n           end\n           i += 1\n       end\n1\n2\n3\n4\n5\n\njulia> for i = 1:1000\n           println(i)\n           if i >= 5\n               break\n           end\n       end\n1\n2\n3\n4\n5Without the break keyword, the above while loop would never terminate on its own, and the for loop would iterate up to 1000. These loops are both exited early by using break.In other circumstances, it is handy to be able to stop an iteration and move on to the next one immediately. The continue keyword accomplishes this:julia> for i = 1:10\n           if i % 3 != 0\n               continue\n           end\n           println(i)\n       end\n3\n6\n9This is a somewhat contrived example since we could produce the same behavior more clearly by negating the condition and placing the println call inside the if block. In realistic usage there is more code to be evaluated after the continue, and often there are multiple points from which one calls continue.Multiple nested for loops can be combined into a single outer loop, forming the cartesian product of its iterables:julia> for i = 1:2, j = 3:4\n           println((i, j))\n       end\n(1, 3)\n(1, 4)\n(2, 3)\n(2, 4)A break statement inside such a loop exits the entire nest of loops, not just the inner one."
},

{
    "location": "manual/control-flow.html#Exception-Handling-1",
    "page": "Control Flow",
    "title": "Exception Handling",
    "category": "section",
    "text": "When an unexpected condition occurs, a function may be unable to return a reasonable value to its caller. In such cases, it may be best for the exceptional condition to either terminate the program, printing a diagnostic error message, or if the programmer has provided code to handle such exceptional circumstances, allow that code to take the appropriate action."
},

{
    "location": "manual/control-flow.html#Built-in-Exceptions-1",
    "page": "Control Flow",
    "title": "Built-in Exceptions",
    "category": "section",
    "text": "Exceptions are thrown when an unexpected condition has occurred. The built-in Exceptions listed below all interrupt the normal flow of control.Exception\nArgumentError\nBoundsError\nCompositeException\nDivideError\nDomainError\nEOFError\nErrorException\nInexactError\nInitError\nInterruptException\nInvalidStateException\nKeyError\nLoadError\nOutOfMemoryError\nReadOnlyMemoryError\nRemoteException\nMethodError\nOverflowError\nParseError\nSystemError\nTypeError\nUndefRefError\nUndefVarError\nUnicodeErrorFor example, the sqrt() function throws a DomainError if applied to a negative real value:julia> sqrt(-1)\nERROR: DomainError:\nsqrt will only return a complex result if called with a complex argument. Try sqrt(complex(x)).\nStacktrace:\n [1] sqrt(::Int64) at ./math.jl:434You may define your own exceptions in the following way:julia> struct MyCustomException <: Exception end"
},

{
    "location": "manual/control-flow.html#The-[throw()](@ref)-function-1",
    "page": "Control Flow",
    "title": "The throw() function",
    "category": "section",
    "text": "Exceptions can be created explicitly with throw(). For example, a function defined only for nonnegative numbers could be written to throw() a DomainError if the argument is negative:julia> f(x) = x>=0 ? exp(-x) : throw(DomainError())\nf (generic function with 1 method)\n\njulia> f(1)\n0.36787944117144233\n\njulia> f(-1)\nERROR: DomainError:\nStacktrace:\n [1] f(::Int64) at ./none:1Note that DomainError without parentheses is not an exception, but a type of exception. It needs to be called to obtain an Exception object:julia> typeof(DomainError()) <: Exception\ntrue\n\njulia> typeof(DomainError) <: Exception\nfalseAdditionally, some exception types take one or more arguments that are used for error reporting:julia> throw(UndefVarError(:x))\nERROR: UndefVarError: x not definedThis mechanism can be implemented easily by custom exception types following the way UndefVarError is written:julia> struct MyUndefVarError <: Exception\n           var::Symbol\n       end\n\njulia> Base.showerror(io::IO, e::MyUndefVarError) = print(io, e.var, \" not defined\")note: Note\nWhen writing an error message, it is preferred to make the first word lowercase. For example, size(A) == size(B) || throw(DimensionMismatch(\"size of A not equal to size of B\"))is preferred oversize(A) == size(B) || throw(DimensionMismatch(\"Size of A not equal to size of B\")).However, sometimes it makes sense to keep the uppercase first letter, for instance if an argument to a function is a capital letter: size(A,1) == size(B,2) || throw(DimensionMismatch(\"A has first dimension...\"))."
},

{
    "location": "manual/control-flow.html#Errors-1",
    "page": "Control Flow",
    "title": "Errors",
    "category": "section",
    "text": "The error() function is used to produce an ErrorException that interrupts the normal flow of control.Suppose we want to stop execution immediately if the square root of a negative number is taken. To do this, we can define a fussy version of the sqrt() function that raises an error if its argument is negative:julia> fussy_sqrt(x) = x >= 0 ? sqrt(x) : error(\"negative x not allowed\")\nfussy_sqrt (generic function with 1 method)\n\njulia> fussy_sqrt(2)\n1.4142135623730951\n\njulia> fussy_sqrt(-1)\nERROR: negative x not allowed\nStacktrace:\n [1] fussy_sqrt(::Int64) at ./none:1If fussy_sqrt is called with a negative value from another function, instead of trying to continue execution of the calling function, it returns immediately, displaying the error message in the interactive session:julia> function verbose_fussy_sqrt(x)\n           println(\"before fussy_sqrt\")\n           r = fussy_sqrt(x)\n           println(\"after fussy_sqrt\")\n           return r\n       end\nverbose_fussy_sqrt (generic function with 1 method)\n\njulia> verbose_fussy_sqrt(2)\nbefore fussy_sqrt\nafter fussy_sqrt\n1.4142135623730951\n\njulia> verbose_fussy_sqrt(-1)\nbefore fussy_sqrt\nERROR: negative x not allowed\nStacktrace:\n [1] fussy_sqrt at ./none:1 [inlined]\n [2] verbose_fussy_sqrt(::Int64) at ./none:3"
},

{
    "location": "manual/control-flow.html#Warnings-and-informational-messages-1",
    "page": "Control Flow",
    "title": "Warnings and informational messages",
    "category": "section",
    "text": "Julia also provides other functions that write messages to the standard error I/O, but do not throw any Exceptions and hence do not interrupt execution:julia> info(\"Hi\"); 1+1\nINFO: Hi\n2\n\njulia> warn(\"Hi\"); 1+1\nWARNING: Hi\n2\n\njulia> error(\"Hi\"); 1+1\nERROR: Hi\nStacktrace:\n [1] error(::String) at ./error.jl:21"
},

{
    "location": "manual/control-flow.html#The-try/catch-statement-1",
    "page": "Control Flow",
    "title": "The try/catch statement",
    "category": "section",
    "text": "The try/catch statement allows for Exceptions to be tested for. For example, a customized square root function can be written to automatically call either the real or complex square root method on demand using Exceptions :julia> f(x) = try\n           sqrt(x)\n       catch\n           sqrt(complex(x, 0))\n       end\nf (generic function with 1 method)\n\njulia> f(1)\n1.0\n\njulia> f(-1)\n0.0 + 1.0imIt is important to note that in real code computing this function, one would compare x to zero instead of catching an exception. The exception is much slower than simply comparing and branching.try/catch statements also allow the Exception to be saved in a variable. In this contrived example, the following example calculates the square root of the second element of x if x is indexable, otherwise assumes x is a real number and returns its square root:julia> sqrt_second(x) = try\n           sqrt(x[2])\n       catch y\n           if isa(y, DomainError)\n               sqrt(complex(x[2], 0))\n           elseif isa(y, BoundsError)\n               sqrt(x)\n           end\n       end\nsqrt_second (generic function with 1 method)\n\njulia> sqrt_second([1 4])\n2.0\n\njulia> sqrt_second([1 -4])\n0.0 + 2.0im\n\njulia> sqrt_second(9)\n3.0\n\njulia> sqrt_second(-9)\nERROR: DomainError:\nStacktrace:\n [1] sqrt_second(::Int64) at ./none:7Note that the symbol following catch will always be interpreted as a name for the exception, so care is needed when writing try/catch expressions on a single line. The following code will not work to return the value of x in case of an error:try bad() catch x endInstead, use a semicolon or insert a line break after catch:try bad() catch; x end\n\ntry bad()\ncatch\n    x\nendThe catch clause is not strictly necessary; when omitted, the default return value is nothing.julia> try error() end # Returns nothingThe power of the try/catch construct lies in the ability to unwind a deeply nested computation immediately to a much higher level in the stack of calling functions. There are situations where no error has occurred, but the ability to unwind the stack and pass a value to a higher level is desirable. Julia provides the rethrow(), backtrace() and catch_backtrace() functions for more advanced error handling."
},

{
    "location": "manual/control-flow.html#finally-Clauses-1",
    "page": "Control Flow",
    "title": "finally Clauses",
    "category": "section",
    "text": "In code that performs state changes or uses resources like files, there is typically clean-up work (such as closing files) that needs to be done when the code is finished. Exceptions potentially complicate this task, since they can cause a block of code to exit before reaching its normal end. The finally keyword provides a way to run some code when a given block of code exits, regardless of how it exits.For example, here is how we can guarantee that an opened file is closed:f = open(\"file\")\ntry\n    # operate on file f\nfinally\n    close(f)\nendWhen control leaves the try block (for example due to a return, or just finishing normally), close(f) will be executed. If the try block exits due to an exception, the exception will continue propagating. A catch block may be combined with try and finally as well. In this case the finally block will run after catch has handled the error."
},

{
    "location": "manual/control-flow.html#man-tasks-1",
    "page": "Control Flow",
    "title": "Tasks (aka Coroutines)",
    "category": "section",
    "text": "Tasks are a control flow feature that allows computations to be suspended and resumed in a flexible manner. This feature is sometimes called by other names, such as symmetric coroutines, lightweight threads, cooperative multitasking, or one-shot continuations.When a piece of computing work (in practice, executing a particular function) is designated as a Task, it becomes possible to interrupt it by switching to another Task. The original Task can later be resumed, at which point it will pick up right where it left off. At first, this may seem similar to a function call. However there are two key differences. First, switching tasks does not use any space, so any number of task switches can occur without consuming the call stack. Second, switching among tasks can occur in any order, unlike function calls, where the called function must finish executing before control returns to the calling function.This kind of control flow can make it much easier to solve certain problems. In some problems, the various pieces of required work are not naturally related by function calls; there is no obvious \"caller\" or \"callee\" among the jobs that need to be done. An example is the producer-consumer problem, where one complex procedure is generating values and another complex procedure is consuming them. The consumer cannot simply call a producer function to get a value, because the producer may have more values to generate and so might not yet be ready to return. With tasks, the producer and consumer can both run as long as they need to, passing values back and forth as necessary.Julia provides a Channel mechanism for solving this problem. A Channel is a waitable first-in first-out queue which can have multiple tasks reading from and writing to it.Let's define a producer task, which produces values via the put! call. To consume values, we need to schedule the producer to run in a new task. A special Channel constructor which accepts a 1-arg function as an argument can be used to run a task bound to a channel. We can then take!() values repeatedly from the channel object:julia> function producer(c::Channel)\n           put!(c, \"start\")\n           for n=1:4\n               put!(c, 2n)\n           end\n           put!(c, \"stop\")\n       end;\n\njulia> chnl = Channel(producer);\n\njulia> take!(chnl)\n\"start\"\n\njulia> take!(chnl)\n2\n\njulia> take!(chnl)\n4\n\njulia> take!(chnl)\n6\n\njulia> take!(chnl)\n8\n\njulia> take!(chnl)\n\"stop\"One way to think of this behavior is that producer was able to return multiple times. Between calls to put!(), the producer's execution is suspended and the consumer has control.The returned Channel can be used as an iterable object in a for loop, in which case the loop variable takes on all the produced values. The loop is terminated when the channel is closed.julia> for x in Channel(producer)\n           println(x)\n       end\nstart\n2\n4\n6\n8\nstopNote that we did not have to explicitly close the channel in the producer. This is because the act of binding a Channel to a Task() associates the open lifetime of a channel with that of the bound task. The channel object is closed automatically when the task terminates. Multiple channels can be bound to a task, and vice-versa.While the Task() constructor expects a 0-argument function, the Channel() method which creates a channel bound task expects a function that accepts a single argument of type Channel. A common pattern is for the producer to be parameterized, in which case a partial function application is needed to create a 0 or 1 argument anonymous function.For Task() objects this can be done either directly or by use of a convenience macro:function mytask(myarg)\n    ...\nend\n\ntaskHdl = Task(() -> mytask(7))\n# or, equivalently\ntaskHdl = @task mytask(7)To orchestrate more advanced work distribution patterns, bind() and schedule() can be used in conjunction with Task() and Channel() constructors to explicitly link a set of channels with a set of producer/consumer tasks.Note that currently Julia tasks are not scheduled to run on separate CPU cores. True kernel threads are discussed under the topic of Parallel Computing."
},

{
    "location": "manual/control-flow.html#Core-task-operations-1",
    "page": "Control Flow",
    "title": "Core task operations",
    "category": "section",
    "text": "Let us explore the low level construct yieldto() to underestand how task switching works. yieldto(task,value) suspends the current task, switches to the specified task, and causes that task's last yieldto() call to return the specified value. Notice that yieldto() is the only operation required to use task-style control flow; instead of calling and returning we are always just switching to a different task. This is why this feature is also called \"symmetric coroutines\"; each task is switched to and from using the same mechanism.yieldto() is powerful, but most uses of tasks do not invoke it directly. Consider why this might be. If you switch away from the current task, you will probably want to switch back to it at some point, but knowing when to switch back, and knowing which task has the responsibility of switching back, can require considerable coordination. For example, put!() and take!() are blocking operations, which, when used in the context of channels maintain state to remember who the consumers are. Not needing to manually keep track of the consuming task is what makes put!() easier to use than the low-level yieldto().In addition to yieldto(), a few other basic functions are needed to use tasks effectively.current_task() gets a reference to the currently-running task.\nistaskdone() queries whether a task has exited.\nistaskstarted() queries whether a task has run yet.\ntask_local_storage() manipulates a key-value store specific to the current task."
},

{
    "location": "manual/control-flow.html#Tasks-and-events-1",
    "page": "Control Flow",
    "title": "Tasks and events",
    "category": "section",
    "text": "Most task switches occur as a result of waiting for events such as I/O requests, and are performed by a scheduler included in the standard library. The scheduler maintains a queue of runnable tasks, and executes an event loop that restarts tasks based on external events such as message arrival.The basic function for waiting for an event is wait(). Several objects implement wait(); for example, given a Process object, wait() will wait for it to exit. wait() is often implicit; for example, a wait() can happen inside a call to read() to wait for data to be available.In all of these cases, wait() ultimately operates on a Condition object, which is in charge of queueing and restarting tasks. When a task calls wait() on a Condition, the task is marked as non-runnable, added to the condition's queue, and switches to the scheduler. The scheduler will then pick another task to run, or block waiting for external events. If all goes well, eventually an event handler will call notify() on the condition, which causes tasks waiting for that condition to become runnable again.A task created explicitly by calling Task is initially not known to the scheduler. This allows you to manage tasks manually using yieldto() if you wish. However, when such a task waits for an event, it still gets restarted automatically when the event happens, as you would expect. It is also possible to make the scheduler run a task whenever it can, without necessarily waiting for any events. This is done by calling schedule(), or using the @schedule or @async macros (see Parallel Computing for more details)."
},

{
    "location": "manual/control-flow.html#Task-states-1",
    "page": "Control Flow",
    "title": "Task states",
    "category": "section",
    "text": "Tasks have a state field that describes their execution status. A Task state is one of the following symbols:Symbol Meaning\n:runnable Currently running, or available to be switched to\n:waiting Blocked waiting for a specific event\n:queued In the scheduler's run queue about to be restarted\n:done Successfully finished executing\n:failed Finished with an uncaught exception"
},

{
    "location": "manual/variables-and-scoping.html#",
    "page": "Scope of Variables",
    "title": "Scope of Variables",
    "category": "page",
    "text": ""
},

{
    "location": "manual/variables-and-scoping.html#scope-of-variables-1",
    "page": "Scope of Variables",
    "title": "Scope of Variables",
    "category": "section",
    "text": "The scope of a variable is the region of code within which a variable is visible. Variable scoping helps avoid variable naming conflicts. The concept is intuitive: two functions can both have arguments called x without the two x's referring to the same thing. Similarly there are many other cases where different blocks of code can use the same name without referring to the same thing. The rules for when the same variable name does or doesn't refer to the same thing are called scope rules; this section spells them out in detail.Certain constructs in the language introduce scope blocks, which are regions of code that are eligible to be the scope of some set of variables. The scope of a variable cannot be an arbitrary set of source lines; instead, it will always line up with one of these blocks. There are two main types of scopes in Julia, global scope and local scope, the latter can be nested. The constructs introducing scope blocks are:Scope name block/construct introducing this kind of scope\nGlobal Scope module, baremodule, at interactive prompt (REPL)\nLocal Scope Soft Local Scope: for, while, comprehensions, try-catch-finally, let\nLocal Scope Hard Local Scope: functions (either syntax, anonymous & do-blocks), struct, macroNotably missing from this table are begin blocks and if blocks, which do not introduce new scope blocks. All three types of scopes follow somewhat different rules which will be explained below as well as some extra rules for certain blocks.Julia uses lexical scoping, meaning that a function's scope does not inherit from its caller's scope, but from the scope in which the function was defined. For example, in the following code the x inside foo refers to the x in the global scope of its module Bar:julia> module Bar\n           x = 1\n           foo() = x\n       end;and not a x in the scope where foo is used:julia> import .Bar\n\njulia> x = -1;\n\njulia> Bar.foo()\n1Thus lexical scope means that the scope of variables can be inferred from the source code alone."
},

{
    "location": "manual/variables-and-scoping.html#Global-Scope-1",
    "page": "Scope of Variables",
    "title": "Global Scope",
    "category": "section",
    "text": "Each module introduces a new global scope, separate from the global scope of all other modules; there is no all-encompassing global scope. Modules can introduce variables of other modules into their scope through the using or import statements or through qualified access using the dot-notation, i.e. each module is a so-called namespace. Note that variable bindings can only be changed within their global scope and not from an outside module.julia> module A\n           a = 1 # a global in A's scope\n       end;\n\njulia> module B\n           module C\n               c = 2\n           end\n           b = C.c    # can access the namespace of a nested global scope\n                      # through a qualified access\n           import ..A # makes module A available\n           d = A.a\n       end;\n\njulia> module D\n           b = a # errors as D's global scope is separate from A's\n       end;\nERROR: UndefVarError: a not defined\n\njulia> module E\n           import ..A # make module A available\n           A.a = 2    # throws below error\n       end;\nERROR: cannot assign variables in other modulesNote that the interactive prompt (aka REPL) is in the global scope of the module Main."
},

{
    "location": "manual/variables-and-scoping.html#Local-Scope-1",
    "page": "Scope of Variables",
    "title": "Local Scope",
    "category": "section",
    "text": "A new local scope is introduced by most code-blocks, see above table for a complete list.  A local scope usually inherits all the variables from its parent scope, both for reading and writing. There are two subtypes of local scopes, hard and soft, with slightly different rules concerning what variables are inherited. Unlike global scopes, local scopes are not namespaces, thus variables in an inner scope cannot be retrieved from the parent scope through some sort of qualified access.The following rules and examples pertain to both hard and soft local scopes. A newly introduced variable in a local scope does not back-propagate to its parent scope. For example, here the z is not introduced into the top-level scope:julia> for i = 1:10\n           z = i\n       end\n\njulia> z\nERROR: UndefVarError: z not defined(Note, in this and all following examples it is assumed that their top-level is a global scope with a clean workspace, for instance a newly started REPL.)Inside a local scope a variable can be forced to be a local variable using the local keyword:julia> x = 0;\n\njulia> for i = 1:10\n           local x\n           x = i + 1\n       end\n\njulia> x\n0Inside a local scope a new global variable can be defined using the keyword global:julia> for i = 1:10\n           global z\n           z = i\n       end\n\njulia> z\n10The location of both the local and global keywords within the scope block is irrelevant. The following is equivalent to the last example (although stylistically worse):julia> for i = 1:10\n           z = i\n           global z\n       end\n\njulia> z\n10"
},

{
    "location": "manual/variables-and-scoping.html#Soft-Local-Scope-1",
    "page": "Scope of Variables",
    "title": "Soft Local Scope",
    "category": "section",
    "text": "In a soft local scope, all variables are inherited from its parent scope unless a variable is specifically marked with the keyword local.Soft local scopes are introduced by for-loops, while-loops, comprehensions, try-catch-finally-blocks, and let-blocks. There are some extra rules for Let Blocks and for For Loops and Comprehensions.In the following example the x and y refer always to the same variables as the soft local scope inherits both read and write variables:julia> x, y = 0, 1;\n\njulia> for i = 1:10\n           x = i + y + 1\n       end\n\njulia> x\n12Within soft scopes, the global keyword is never necessary, although allowed. The only case when it would change the semantics is (currently) a syntax error:julia> let\n           local j = 2\n           let\n               global j = 3\n           end\n       end\nERROR: syntax: `global j`: j is local variable in the enclosing scope"
},

{
    "location": "manual/variables-and-scoping.html#Hard-Local-Scope-1",
    "page": "Scope of Variables",
    "title": "Hard Local Scope",
    "category": "section",
    "text": "Hard local scopes are introduced by function definitions (in all their forms), struct type definition blocks, and macro-definitions.In a hard local scope, all variables are inherited from its parent scope unless:an assignment would result in a modified global variable, or\na variable is specifically marked with the keyword local.Thus global variables are only inherited for reading but not for writing:julia> x, y = 1, 2;\n\njulia> function foo()\n           x = 2        # assignment introduces a new local\n           return x + y # y refers to the global\n       end;\n\njulia> foo()\n4\n\njulia> x\n1An explicit global is needed to assign to a global variable:julia> x = 1;\n\njulia> function foobar()\n           global x = 2\n       end;\n\njulia> foobar();\n\njulia> x\n2Note that nested functions can behave differently to functions defined in the global scope as they can modify their parent scope's local variables:julia> x, y = 1, 2;\n\njulia> function baz()\n           x = 2 # introduces a new local\n           function bar()\n               x = 10       # modifies the parent's x\n               return x + y # y is global\n           end\n           return bar() + x # 12 + 10 (x is modified in call of bar())\n       end;\n\njulia> baz()\n22\n\njulia> x, y\n(1, 2)The distinction between inheriting global and local variables for assignment can lead to some slight differences between functions defined in local vs. global scopes. Consider the modification of the last example by moving bar to the global scope:julia> x, y = 1, 2;\n\njulia> function bar()\n           x = 10 # local\n           return x + y\n       end;\n\njulia> function quz()\n           x = 2 # local\n           return bar() + x # 12 + 2 (x is not modified)\n       end;\n\njulia> quz()\n14\n\njulia> x, y\n(1, 2)Note that above subtlety does not pertain to type and macro definitions as they can only appear at the global scope. There are special scoping rules concerning the evaluation of default and keyword function arguments which are described in the Function section.An assignment introducing a variable used inside a function, type or macro definition need not come before its inner usage:julia> f = y -> y + a\n(::#1) (generic function with 1 method)\n\njulia> f(3)\nERROR: UndefVarError: a not defined\nStacktrace:\n [1] (::##1#2)(::Int64) at ./none:1\n\njulia> a = 1\n1\n\njulia> f(3)\n4This behavior may seem slightly odd for a normal variable, but allows for named functions – which are just normal variables holding function objects – to be used before they are defined. This allows functions to be defined in whatever order is intuitive and convenient, rather than forcing bottom up ordering or requiring forward declarations, as long as they are defined by the time they are actually called. As an example, here is an inefficient, mutually recursive way to test if positive integers are even or odd:julia> even(n) = n == 0 ? true : odd(n-1);\n\njulia> odd(n) = n == 0 ? false : even(n-1);\n\njulia> even(3)\nfalse\n\njulia> odd(3)\ntrueJulia provides built-in, efficient functions to test for oddness and evenness called iseven() and isodd() so the above definitions should only be taken as examples."
},

{
    "location": "manual/variables-and-scoping.html#Hard-vs.-Soft-Local-Scope-1",
    "page": "Scope of Variables",
    "title": "Hard vs. Soft Local Scope",
    "category": "section",
    "text": "Blocks which introduce a soft local scope, such as loops, are generally used to manipulate the variables in their parent scope. Thus their default is to fully access all variables in their parent scope.Conversely, the code inside blocks which introduce a hard local scope (function, type, and macro definitions) can be executed at any place in a program. Remotely changing the state of global variables in other modules should be done with care and thus this is an opt-in feature requiring the global keyword.The reason to allow modifying local variables of parent scopes in nested functions is to allow constructing closures which have a private state, for instance the state variable in the following example:julia> let\n           state = 0\n           global counter\n           counter() = state += 1\n       end;\n\njulia> counter()\n1\n\njulia> counter()\n2See also the closures in the examples in the next two sections."
},

{
    "location": "manual/variables-and-scoping.html#Let-Blocks-1",
    "page": "Scope of Variables",
    "title": "Let Blocks",
    "category": "section",
    "text": "Unlike assignments to local variables, let statements allocate new variable bindings each time they run. An assignment modifies an existing value location, and let creates new locations. This difference is usually not important, and is only detectable in the case of variables that outlive their scope via closures. The let syntax accepts a comma-separated series of assignments and variable names:julia> x, y, z = -1, -1, -1;\n\njulia> let x = 1, z\n           println(\"x: $x, y: $y\") # x is local variable, y the global\n           println(\"z: $z\") # errors as z has not been assigned yet but is local\n       end\nx: 1, y: -1\nERROR: UndefVarError: z not definedThe assignments are evaluated in order, with each right-hand side evaluated in the scope before the new variable on the left-hand side has been introduced. Therefore it makes sense to write something like let x = x since the two x variables are distinct and have separate storage. Here is an example where the behavior of let is needed:julia> Fs = Array{Any}(2); i = 1;\n\njulia> while i <= 2\n           Fs[i] = ()->i\n           i += 1\n       end\n\njulia> Fs[1]()\n3\n\njulia> Fs[2]()\n3Here we create and store two closures that return variable i. However, it is always the same variable i, so the two closures behave identically. We can use let to create a new binding for i:julia> Fs = Array{Any}(2); i = 1;\n\njulia> while i <= 2\n           let i = i\n               Fs[i] = ()->i\n           end\n           i += 1\n       end\n\njulia> Fs[1]()\n1\n\njulia> Fs[2]()\n2Since the begin construct does not introduce a new scope, it can be useful to use a zero-argument let to just introduce a new scope block without creating any new bindings:julia> let\n           local x = 1\n           let\n               local x = 2\n           end\n           x\n       end\n1Since let introduces a new scope block, the inner local x is a different variable than the outer local x."
},

{
    "location": "manual/variables-and-scoping.html#For-Loops-and-Comprehensions-1",
    "page": "Scope of Variables",
    "title": "For Loops and Comprehensions",
    "category": "section",
    "text": "for loops and Comprehensions have the following behavior: any new variables introduced in their body scopes are freshly allocated for each loop iteration. This is in contrast to while loops which reuse the variables for all iterations. Therefore these constructs are similar to while loops with let blocks inside:julia> Fs = Array{Any}(2);\n\njulia> for j = 1:2\n           Fs[j] = ()->j\n       end\n\njulia> Fs[1]()\n1\n\njulia> Fs[2]()\n2for loops will reuse existing variables for its iteration variable:julia> i = 0;\n\njulia> for i = 1:3\n       end\n\njulia> i\n3However, comprehensions do not do this, and always freshly allocate their iteration variables:julia> x = 0;\n\njulia> [ x for x = 1:3 ];\n\njulia> x\n0"
},

{
    "location": "manual/variables-and-scoping.html#Constants-1",
    "page": "Scope of Variables",
    "title": "Constants",
    "category": "section",
    "text": "A common use of variables is giving names to specific, unchanging values. Such variables are only assigned once. This intent can be conveyed to the compiler using the const keyword:julia> const e  = 2.71828182845904523536;\n\njulia> const pi = 3.14159265358979323846;The const declaration is allowed on both global and local variables, but is especially useful for globals. It is difficult for the compiler to optimize code involving global variables, since their values (or even their types) might change at almost any time. If a global variable will not change, adding a const declaration solves this performance problem.Local constants are quite different. The compiler is able to determine automatically when a local variable is constant, so local constant declarations are not necessary for performance purposes.Special top-level assignments, such as those performed by the function and struct keywords, are constant by default.Note that const only affects the variable binding; the variable may be bound to a mutable object (such as an array), and that object may still be modified."
},

{
    "location": "manual/types.html#",
    "page": "Types",
    "title": "Types",
    "category": "page",
    "text": ""
},

{
    "location": "manual/types.html#man-types-1",
    "page": "Types",
    "title": "Types",
    "category": "section",
    "text": "Type systems have traditionally fallen into two quite different camps: static type systems, where every program expression must have a type computable before the execution of the program, and dynamic type systems, where nothing is known about types until run time, when the actual values manipulated by the program are available. Object orientation allows some flexibility in statically typed languages by letting code be written without the precise types of values being known at compile time. The ability to write code that can operate on different types is called polymorphism. All code in classic dynamically typed languages is polymorphic: only by explicitly checking types, or when objects fail to support operations at run-time, are the types of any values ever restricted.Julia's type system is dynamic, but gains some of the advantages of static type systems by making it possible to indicate that certain values are of specific types. This can be of great assistance in generating efficient code, but even more significantly, it allows method dispatch on the types of function arguments to be deeply integrated with the language. Method dispatch is explored in detail in Methods, but is rooted in the type system presented here.The default behavior in Julia when types are omitted is to allow values to be of any type. Thus, one can write many useful Julia programs without ever explicitly using types. When additional expressiveness is needed, however, it is easy to gradually introduce explicit type annotations into previously \"untyped\" code. Doing so will typically increase both the performance and robustness of these systems, and perhaps somewhat counterintuitively, often significantly simplify them.Describing Julia in the lingo of type systems, it is: dynamic, nominative and parametric. Generic types can be parameterized, and the hierarchical relationships between types are explicitly declared, rather than implied by compatible structure. One particularly distinctive feature of Julia's type system is that concrete types may not subtype each other: all concrete types are final and may only have abstract types as their supertypes. While this might at first seem unduly restrictive, it has many beneficial consequences with surprisingly few drawbacks. It turns out that being able to inherit behavior is much more important than being able to inherit structure, and inheriting both causes significant difficulties in traditional object-oriented languages. Other high-level aspects of Julia's type system that should be mentioned up front are:There is no division between object and non-object values: all values in Julia are true objects having a type that belongs to a single, fully connected type graph, all nodes of which are equally first-class as types.\nThere is no meaningful concept of a \"compile-time type\": the only type a value has is its actual type when the program is running. This is called a \"run-time type\" in object-oriented languages where the combination of static compilation with polymorphism makes this distinction significant.\nOnly values, not variables, have types – variables are simply names bound to values.\nBoth abstract and concrete types can be parameterized by other types. They can also be parameterized by symbols, by values of any type for which isbits() returns true (essentially, things like numbers and bools that are stored like C types or structs with no pointers to other objects), and also by tuples thereof. Type parameters may be omitted when they do not need to be referenced or restricted.Julia's type system is designed to be powerful and expressive, yet clear, intuitive and unobtrusive. Many Julia programmers may never feel the need to write code that explicitly uses types. Some kinds of programming, however, become clearer, simpler, faster and more robust with declared types."
},

{
    "location": "manual/types.html#Type-Declarations-1",
    "page": "Types",
    "title": "Type Declarations",
    "category": "section",
    "text": "The :: operator can be used to attach type annotations to expressions and variables in programs. There are two primary reasons to do this:As an assertion to help confirm that your program works the way you expect,\nTo provide extra type information to the compiler, which can then improve performance in some casesWhen appended to an expression computing a value, the :: operator is read as \"is an instance of\". It can be used anywhere to assert that the value of the expression on the left is an instance of the type on the right. When the type on the right is concrete, the value on the left must have that type as its implementation – recall that all concrete types are final, so no implementation is a subtype of any other. When the type is abstract, it suffices for the value to be implemented by a concrete type that is a subtype of the abstract type. If the type assertion is not true, an exception is thrown, otherwise, the left-hand value is returned:julia> (1+2)::AbstractFloat\nERROR: TypeError: typeassert: expected AbstractFloat, got Int64\n\njulia> (1+2)::Int\n3This allows a type assertion to be attached to any expression in-place.When appended to a variable on the left-hand side of an assignment, or as part of a local declaration, the :: operator means something a bit different: it declares the variable to always have the specified type, like a type declaration in a statically-typed language such as C. Every value assigned to the variable will be converted to the declared type using convert():julia> function foo()\n           x::Int8 = 100\n           x\n       end\nfoo (generic function with 1 method)\n\njulia> foo()\n100\n\njulia> typeof(ans)\nInt8This feature is useful for avoiding performance \"gotchas\" that could occur if one of the assignments to a variable changed its type unexpectedly.This \"declaration\" behavior only occurs in specific contexts:local x::Int8  # in a local declaration\nx::Int8 = 10   # as the left-hand side of an assignmentand applies to the whole current scope, even before the declaration. Currently, type declarations cannot be used in global scope, e.g. in the REPL, since Julia does not yet have constant-type globals.Declarations can also be attached to function definitions:function sinc(x)::Float64\n    if x == 0\n        return 1\n    end\n    return sin(pi*x)/(pi*x)\nendReturning from this function behaves just like an assignment to a variable with a declared type: the value is always converted to Float64."
},

{
    "location": "manual/types.html#Abstract-Types-1",
    "page": "Types",
    "title": "Abstract Types",
    "category": "section",
    "text": "Abstract types cannot be instantiated, and serve only as nodes in the type graph, thereby describing sets of related concrete types: those concrete types which are their descendants. We begin with abstract types even though they have no instantiation because they are the backbone of the type system: they form the conceptual hierarchy which makes Julia's type system more than just a collection of object implementations.Recall that in Integers and Floating-Point Numbers, we introduced a variety of concrete types of numeric values: Int8, UInt8, Int16, UInt16, Int32, UInt32, Int64, UInt64, Int128, UInt128, Float16, Float32, and Float64. Although they have different representation sizes, Int8, Int16, Int32, Int64 and Int128 all have in common that they are signed integer types. Likewise UInt8, UInt16, UInt32, UInt64 and UInt128 are all unsigned integer types, while Float16, Float32 and Float64 are distinct in being floating-point types rather than integers. It is common for a piece of code to make sense, for example, only if its arguments are some kind of integer, but not really depend on what particular kind of integer. For example, the greatest common denominator algorithm works for all kinds of integers, but will not work for floating-point numbers. Abstract types allow the construction of a hierarchy of types, providing a context into which concrete types can fit. This allows you, for example, to easily program to any type that is an integer, without restricting an algorithm to a specific type of integer.Abstract types are declared using the abstract type keyword. The general syntaxes for declaring an abstract type are:abstract type «name» end\nabstract type «name» <: «supertype» endThe abstract type keyword introduces a new abstract type, whose name is given by «name». This name can be optionally followed by <: and an already-existing type, indicating that the newly declared abstract type is a subtype of this \"parent\" type.When no supertype is given, the default supertype is Any – a predefined abstract type that all objects are instances of and all types are subtypes of. In type theory, Any is commonly called \"top\" because it is at the apex of the type graph. Julia also has a predefined abstract \"bottom\" type, at the nadir of the type graph, which is written as Union{}. It is the exact opposite of Any: no object is an instance of Union{} and all types are supertypes of Union{}.Let's consider some of the abstract types that make up Julia's numerical hierarchy:abstract type Number end\nabstract type Real     <: Number end\nabstract type AbstractFloat <: Real end\nabstract type Integer  <: Real end\nabstract type Signed   <: Integer end\nabstract type Unsigned <: Integer endThe Number type is a direct child type of Any, and Real is its child. In turn, Real has two children (it has more, but only two are shown here; we'll get to the others later): Integer and AbstractFloat, separating the world into representations of integers and representations of real numbers. Representations of real numbers include, of course, floating-point types, but also include other types, such as rationals. Hence, AbstractFloat is a proper subtype of Real, including only floating-point representations of real numbers. Integers are further subdivided into Signed and Unsigned varieties.The <: operator in general means \"is a subtype of\", and, used in declarations like this, declares the right-hand type to be an immediate supertype of the newly declared type. It can also be used in expressions as a subtype operator which returns true when its left operand is a subtype of its right operand:julia> Integer <: Number\ntrue\n\njulia> Integer <: AbstractFloat\nfalseAn important use of abstract types is to provide default implementations for concrete types. To give a simple example, consider:function myplus(x,y)\n    x+y\nendThe first thing to note is that the above argument declarations are equivalent to x::Any and y::Any. When this function is invoked, say as myplus(2,5), the dispatcher chooses the most specific method named myplus that matches the given arguments. (See Methods for more information on multiple dispatch.)Assuming no method more specific than the above is found, Julia next internally defines and compiles a method called myplus specifically for two Int arguments based on the generic function given above, i.e., it implicitly defines and compiles:function myplus(x::Int,y::Int)\n    x+y\nendand finally, it invokes this specific method.Thus, abstract types allow programmers to write generic functions that can later be used as the default method by many combinations of concrete types. Thanks to multiple dispatch, the programmer has full control over whether the default or more specific method is used.An important point to note is that there is no loss in performance if the programmer relies on a function whose arguments are abstract types, because it is recompiled for each tuple of argument concrete types with which it is invoked. (There may be a performance issue, however, in the case of function arguments that are containers of abstract types; see Performance Tips.)"
},

{
    "location": "manual/types.html#Primitive-Types-1",
    "page": "Types",
    "title": "Primitive Types",
    "category": "section",
    "text": "A primitive type is a concrete type whose data consists of plain old bits. Classic examples of primitive types are integers and floating-point values. Unlike most languages, Julia lets you declare your own primitive types, rather than providing only a fixed set of built-in ones. In fact, the standard primitive types are all defined in the language itself:primitive type Float16 <: AbstractFloat 16 end\nprimitive type Float32 <: AbstractFloat 32 end\nprimitive type Float64 <: AbstractFloat 64 end\n\nprimitive type Bool <: Integer 8 end\nprimitive type Char 32 end\n\nprimitive type Int8    <: Signed   8 end\nprimitive type UInt8   <: Unsigned 8 end\nprimitive type Int16   <: Signed   16 end\nprimitive type UInt16  <: Unsigned 16 end\nprimitive type Int32   <: Signed   32 end\nprimitive type UInt32  <: Unsigned 32 end\nprimitive type Int64   <: Signed   64 end\nprimitive type UInt64  <: Unsigned 64 end\nprimitive type Int128  <: Signed   128 end\nprimitive type UInt128 <: Unsigned 128 endThe general syntaxes for declaring a primitive type are:primitive type «name» «bits» end\nprimitive type «name» <: «supertype» «bits» endThe number of bits indicates how much storage the type requires and the name gives the new type a name. A primitive type can optionally be declared to be a subtype of some supertype. If a supertype is omitted, then the type defaults to having Any as its immediate supertype. The declaration of Bool above therefore means that a boolean value takes eight bits to store, and has Integer as its immediate supertype. Currently, only sizes that are multiples of 8 bits are supported. Therefore, boolean values, although they really need just a single bit, cannot be declared to be any smaller than eight bits.The types Bool, Int8 and UInt8 all have identical representations: they are eight-bit chunks of memory. Since Julia's type system is nominative, however, they are not interchangeable despite having identical structure. A fundamental difference between them is that they have different supertypes: Bool's direct supertype is Integer, Int8's is Signed, and UInt8's is Unsigned. All other differences between Bool, Int8, and UInt8 are matters of behavior – the way functions are defined to act when given objects of these types as arguments. This is why a nominative type system is necessary: if structure determined type, which in turn dictates behavior, then it would be impossible to make Bool behave any differently than Int8 or UInt8."
},

{
    "location": "manual/types.html#Composite-Types-1",
    "page": "Types",
    "title": "Composite Types",
    "category": "section",
    "text": "Composite types are called records, structs, or objects in various languages. A composite type is a collection of named fields, an instance of which can be treated as a single value. In many languages, composite types are the only kind of user-definable type, and they are by far the most commonly used user-defined type in Julia as well.In mainstream object oriented languages, such as C++, Java, Python and Ruby, composite types also have named functions associated with them, and the combination is called an \"object\". In purer object-oriented languages, such as Ruby or Smalltalk, all values are objects whether they are composites or not. In less pure object oriented languages, including C++ and Java, some values, such as integers and floating-point values, are not objects, while instances of user-defined composite types are true objects with associated methods. In Julia, all values are objects, but functions are not bundled with the objects they operate on. This is necessary since Julia chooses which method of a function to use by multiple dispatch, meaning that the types of all of a function's arguments are considered when selecting a method, rather than just the first one (see Methods for more information on methods and dispatch). Thus, it would be inappropriate for functions to \"belong\" to only their first argument. Organizing methods into function objects rather than having named bags of methods \"inside\" each object ends up being a highly beneficial aspect of the language design.Composite types are introduced with the struct keyword followed by a block of field names, optionally annotated with types using the :: operator:julia> struct Foo\n           bar\n           baz::Int\n           qux::Float64\n       endFields with no type annotation default to Any, and can accordingly hold any type of value.New objects of type Foo are created by applying the Foo type object like a function to values for its fields:julia> foo = Foo(\"Hello, world.\", 23, 1.5)\nFoo(\"Hello, world.\", 23, 1.5)\n\njulia> typeof(foo)\nFooWhen a type is applied like a function it is called a constructor. Two constructors are generated automatically (these are called default constructors). One accepts any arguments and calls convert() to convert them to the types of the fields, and the other accepts arguments that match the field types exactly. The reason both of these are generated is that this makes it easier to add new definitions without inadvertently replacing a default constructor.Since the bar field is unconstrained in type, any value will do. However, the value for baz must be convertible to Int:julia> Foo((), 23.5, 1)\nERROR: InexactError()\nStacktrace:\n [1] convert(::Type{Int64}, ::Float64) at ./float.jl:679\n [2] Foo(::Tuple{}, ::Float64, ::Int64) at ./none:2You may find a list of field names using the fieldnames function.julia> fieldnames(foo)\n3-element Array{Symbol,1}:\n :bar\n :baz\n :quxYou can access the field values of a composite object using the traditional foo.bar notation:julia> foo.bar\n\"Hello, world.\"\n\njulia> foo.baz\n23\n\njulia> foo.qux\n1.5Composite objects declared with struct are immutable; they cannot be modified after construction. This may seem odd at first, but it has several advantages:It can be more efficient. Some structs can be packed efficiently into arrays, and in some cases the compiler is able to avoid allocating immutable objects entirely.\nIt is not possible to violate the invariants provided by the type's constructors.\nCode using immutable objects can be easier to reason about.An immutable object might contain mutable objects, such as arrays, as fields. Those contained objects will remain mutable; only the fields of the immutable object itself cannot be changed to point to different objects.Where required, mutable composite objects can be declared with the keyword mutable struct, to be discussed in the next section.Composite types with no fields are singletons; there can be only one instance of such types:julia> struct NoFields\n       end\n\njulia> NoFields() === NoFields()\ntrueThe === function confirms that the \"two\" constructed instances of NoFields are actually one and the same. Singleton types are described in further detail below.There is much more to say about how instances of composite types are created, but that discussion depends on both Parametric Types and on Methods, and is sufficiently important to be addressed in its own section: Constructors."
},

{
    "location": "manual/types.html#Mutable-Composite-Types-1",
    "page": "Types",
    "title": "Mutable Composite Types",
    "category": "section",
    "text": "If a composite type is declared with mutable struct instead of struct, then instances of it can be modified:julia> mutable struct Bar\n           baz\n           qux::Float64\n       end\n\njulia> bar = Bar(\"Hello\", 1.5);\n\njulia> bar.qux = 2.0\n2.0\n\njulia> bar.baz = 1//2\n1//2In order to support mutation, such objects are generally allocated on the heap, and have stable memory addresses. A mutable object is like a little container that might hold different values over time, and so can only be reliably identified with its address. In contrast, an instance of an immutable type is associated with specific field values –- the field values alone tell you everything about the object. In deciding whether to make a type mutable, ask whether two instances with the same field values would be considered identical, or if they might need to change independently over time. If they would be considered identical, the type should probably be immutable.To recap, two essential properties define immutability in Julia:An object with an immutable type is passed around (both in assignment statements and in function calls) by copying, whereas a mutable type is passed around by reference.\nIt is not permitted to modify the fields of a composite immutable type.It is instructive, particularly for readers whose background is C/C++, to consider why these two properties go hand in hand.  If they were separated, i.e., if the fields of objects passed around by copying could be modified, then it would become more difficult to reason about certain instances of generic code.  For example, suppose x is a function argument of an abstract type, and suppose that the function changes a field: x.isprocessed = true.  Depending on whether x is passed by copying or by reference, this statement may or may not alter the actual argument in the calling routine.  Julia sidesteps the possibility of creating functions with unknown effects in this scenario by forbidding modification of fields of objects passed around by copying."
},

{
    "location": "manual/types.html#Declared-Types-1",
    "page": "Types",
    "title": "Declared Types",
    "category": "section",
    "text": "The three kinds of types discussed in the previous three sections are actually all closely related. They share the same key properties:They are explicitly declared.\nThey have names.\nThey have explicitly declared supertypes.\nThey may have parameters.Because of these shared properties, these types are internally represented as instances of the same concept, DataType, which is the type of any of these types:julia> typeof(Real)\nDataType\n\njulia> typeof(Int)\nDataTypeA DataType may be abstract or concrete. If it is concrete, it has a specified size, storage layout, and (optionally) field names. Thus a bits type is a DataType with nonzero size, but no field names. A composite type is a DataType that has field names or is empty (zero size).Every concrete value in the system is an instance of some DataType."
},

{
    "location": "manual/types.html#Type-Unions-1",
    "page": "Types",
    "title": "Type Unions",
    "category": "section",
    "text": "A type union is a special abstract type which includes as objects all instances of any of its argument types, constructed using the special Union function:julia> IntOrString = Union{Int,AbstractString}\nUnion{AbstractString, Int64}\n\njulia> 1 :: IntOrString\n1\n\njulia> \"Hello!\" :: IntOrString\n\"Hello!\"\n\njulia> 1.0 :: IntOrString\nERROR: TypeError: typeassert: expected Union{AbstractString, Int64}, got Float64The compilers for many languages have an internal union construct for reasoning about types; Julia simply exposes it to the programmer."
},

{
    "location": "manual/types.html#Parametric-Types-1",
    "page": "Types",
    "title": "Parametric Types",
    "category": "section",
    "text": "An important and powerful feature of Julia's type system is that it is parametric: types can take parameters, so that type declarations actually introduce a whole family of new types – one for each possible combination of parameter values. There are many languages that support some version of generic programming, wherein data structures and algorithms to manipulate them may be specified without specifying the exact types involved. For example, some form of generic programming exists in ML, Haskell, Ada, Eiffel, C++, Java, C#, F#, and Scala, just to name a few. Some of these languages support true parametric polymorphism (e.g. ML, Haskell, Scala), while others support ad-hoc, template-based styles of generic programming (e.g. C++, Java). With so many different varieties of generic programming and parametric types in various languages, we won't even attempt to compare Julia's parametric types to other languages, but will instead focus on explaining Julia's system in its own right. We will note, however, that because Julia is a dynamically typed language and doesn't need to make all type decisions at compile time, many traditional difficulties encountered in static parametric type systems can be relatively easily handled.All declared types (the DataType variety) can be parameterized, with the same syntax in each case. We will discuss them in the following order: first, parametric composite types, then parametric abstract types, and finally parametric bits types."
},

{
    "location": "manual/types.html#Parametric-Composite-Types-1",
    "page": "Types",
    "title": "Parametric Composite Types",
    "category": "section",
    "text": "Type parameters are introduced immediately after the type name, surrounded by curly braces:julia> struct Point{T}\n           x::T\n           y::T\n       endThis declaration defines a new parametric type, Point{T}, holding two \"coordinates\" of type T. What, one may ask, is T? Well, that's precisely the point of parametric types: it can be any type at all (or a value of any bits type, actually, although here it's clearly used as a type). Point{Float64} is a concrete type equivalent to the type defined by replacing T in the definition of Point with Float64. Thus, this single declaration actually declares an unlimited number of types: Point{Float64}, Point{AbstractString}, Point{Int64}, etc. Each of these is now a usable concrete type:julia> Point{Float64}\nPoint{Float64}\n\njulia> Point{AbstractString}\nPoint{AbstractString}The type Point{Float64} is a point whose coordinates are 64-bit floating-point values, while the type Point{AbstractString} is a \"point\" whose \"coordinates\" are string objects (see Strings).Point itself is also a valid type object, containing all instances Point{Float64}, Point{AbstractString}, etc. as subtypes:julia> Point{Float64} <: Point\ntrue\n\njulia> Point{AbstractString} <: Point\ntrueOther types, of course, are not subtypes of it:julia> Float64 <: Point\nfalse\n\njulia> AbstractString <: Point\nfalseConcrete Point types with different values of T are never subtypes of each other:julia> Point{Float64} <: Point{Int64}\nfalse\n\njulia> Point{Float64} <: Point{Real}\nfalsewarning: Warning\nThis last point is very important: even though Float64 <: Real we DO NOT have Point{Float64} <: Point{Real}.In other words, in the parlance of type theory, Julia's type parameters are invariant, rather than being covariant (or even contravariant). This is for practical reasons: while any instance of Point{Float64} may conceptually be like an instance of Point{Real} as well, the two types have different representations in memory:An instance of Point{Float64} can be represented compactly and efficiently as an immediate pair of 64-bit values;\nAn instance of Point{Real} must be able to hold any pair of instances of Real. Since objects that are instances of Real can be of arbitrary size and structure, in practice an instance of Point{Real} must be represented as a pair of pointers to individually allocated Real objects.The efficiency gained by being able to store Point{Float64} objects with immediate values is magnified enormously in the case of arrays: an Array{Float64} can be stored as a contiguous memory block of 64-bit floating-point values, whereas an Array{Real} must be an array of pointers to individually allocated Real objects – which may well be boxed 64-bit floating-point values, but also might be arbitrarily large, complex objects, which are declared to be implementations of the Real abstract type.Since Point{Float64} is not a subtype of Point{Real}, the following method can't be applied to arguments of type Point{Float64}:function norm(p::Point{Real})\n    sqrt(p.x^2 + p.y^2)\nendA correct way to define a method that accepts all arguments of type Point{T} where T is a subtype of Real is:function norm(p::Point{<:Real})\n    sqrt(p.x^2 + p.y^2)\nend(Equivalently, one could define function norm{T<:Real}(p::Point{T}) or function norm(p::Point{T} where T<:Real); see UnionAll Types.)More examples will be discussed later in Methods.How does one construct a Point object? It is possible to define custom constructors for composite types, which will be discussed in detail in Constructors, but in the absence of any special constructor declarations, there are two default ways of creating new composite objects, one in which the type parameters are explicitly given and the other in which they are implied by the arguments to the object constructor.Since the type Point{Float64} is a concrete type equivalent to Point declared with Float64 in place of T, it can be applied as a constructor accordingly:julia> Point{Float64}(1.0, 2.0)\nPoint{Float64}(1.0, 2.0)\n\njulia> typeof(ans)\nPoint{Float64}For the default constructor, exactly one argument must be supplied for each field:julia> Point{Float64}(1.0)\nERROR: MethodError: Cannot `convert` an object of type Float64 to an object of type Point{Float64}\nThis may have arisen from a call to the constructor Point{Float64}(...),\nsince type constructors fall back to convert methods.\nStacktrace:\n [1] Point{Float64}(::Float64) at ./sysimg.jl:77\n\njulia> Point{Float64}(1.0,2.0,3.0)\nERROR: MethodError: no method matching Point{Float64}(::Float64, ::Float64, ::Float64)Only one default constructor is generated for parametric types, since overriding it is not possible. This constructor accepts any arguments and converts them to the field types.In many cases, it is redundant to provide the type of Point object one wants to construct, since the types of arguments to the constructor call already implicitly provide type information. For that reason, you can also apply Point itself as a constructor, provided that the implied value of the parameter type T is unambiguous:julia> Point(1.0,2.0)\nPoint{Float64}(1.0, 2.0)\n\njulia> typeof(ans)\nPoint{Float64}\n\njulia> Point(1,2)\nPoint{Int64}(1, 2)\n\njulia> typeof(ans)\nPoint{Int64}In the case of Point, the type of T is unambiguously implied if and only if the two arguments to Point have the same type. When this isn't the case, the constructor will fail with a MethodError:julia> Point(1,2.5)\nERROR: MethodError: no method matching Point(::Int64, ::Float64)\nClosest candidates are:\n  Point(::T, !Matched::T) where T at none:2Constructor methods to appropriately handle such mixed cases can be defined, but that will not be discussed until later on in Constructors."
},

{
    "location": "manual/types.html#Parametric-Abstract-Types-1",
    "page": "Types",
    "title": "Parametric Abstract Types",
    "category": "section",
    "text": "Parametric abstract type declarations declare a collection of abstract types, in much the same way:julia> abstract type Pointy{T} endWith this declaration, Pointy{T} is a distinct abstract type for each type or integer value of T. As with parametric composite types, each such instance is a subtype of Pointy:julia> Pointy{Int64} <: Pointy\ntrue\n\njulia> Pointy{1} <: Pointy\ntrueParametric abstract types are invariant, much as parametric composite types are:julia> Pointy{Float64} <: Pointy{Real}\nfalse\n\njulia> Pointy{Real} <: Pointy{Float64}\nfalseThe notation Pointy{<:Real} can be used to express the Julia analogue of a covariant type, while Pointy{>:Int} the analogue of a contravariant type, but technically these represent sets of types (see UnionAll Types).julia> Pointy{Float64} <: Pointy{<:Real}\ntrue\n\njulia> Pointy{Real} <: Pointy{>:Int}\ntrueMuch as plain old abstract types serve to create a useful hierarchy of types over concrete types, parametric abstract types serve the same purpose with respect to parametric composite types. We could, for example, have declared Point{T} to be a subtype of Pointy{T} as follows:julia> struct Point{T} <: Pointy{T}\n           x::T\n           y::T\n       endGiven such a declaration, for each choice of T, we have Point{T} as a subtype of Pointy{T}:julia> Point{Float64} <: Pointy{Float64}\ntrue\n\njulia> Point{Real} <: Pointy{Real}\ntrue\n\njulia> Point{AbstractString} <: Pointy{AbstractString}\ntrueThis relationship is also invariant:julia> Point{Float64} <: Pointy{Real}\nfalse\n\njulia> Point{Float64} <: Pointy{<:Real}\ntrueWhat purpose do parametric abstract types like Pointy serve? Consider if we create a point-like implementation that only requires a single coordinate because the point is on the diagonal line x = y:julia> struct DiagPoint{T} <: Pointy{T}\n           x::T\n       endNow both Point{Float64} and DiagPoint{Float64} are implementations of the Pointy{Float64} abstraction, and similarly for every other possible choice of type T. This allows programming to a common interface shared by all Pointy objects, implemented for both Point and DiagPoint. This cannot be fully demonstrated, however, until we have introduced methods and dispatch in the next section, Methods.There are situations where it may not make sense for type parameters to range freely over all possible types. In such situations, one can constrain the range of T like so:julia> abstract type Pointy{T<:Real} endWith such a declaration, it is acceptable to use any type that is a subtype of Real in place of T, but not types that are not subtypes of Real:julia> Pointy{Float64}\nPointy{Float64}\n\njulia> Pointy{Real}\nPointy{Real}\n\njulia> Pointy{AbstractString}\nERROR: TypeError: Pointy: in T, expected T<:Real, got Type{AbstractString}\n\njulia> Pointy{1}\nERROR: TypeError: Pointy: in T, expected T<:Real, got Int64Type parameters for parametric composite types can be restricted in the same manner:struct Point{T<:Real} <: Pointy{T}\n    x::T\n    y::T\nendTo give a real-world example of how all this parametric type machinery can be useful, here is the actual definition of Julia's Rational immutable type (except that we omit the constructor here for simplicity), representing an exact ratio of integers:struct Rational{T<:Integer} <: Real\n    num::T\n    den::T\nendIt only makes sense to take ratios of integer values, so the parameter type T is restricted to being a subtype of Integer, and a ratio of integers represents a value on the real number line, so any Rational is an instance of the Real abstraction."
},

{
    "location": "manual/types.html#Tuple-Types-1",
    "page": "Types",
    "title": "Tuple Types",
    "category": "section",
    "text": "Tuples are an abstraction of the arguments of a function – without the function itself. The salient aspects of a function's arguments are their order and their types. Therefore a tuple type is similar to a parameterized immutable type where each parameter is the type of one field. For example, a 2-element tuple type resembles the following immutable type:struct Tuple2{A,B}\n    a::A\n    b::B\nendHowever, there are three key differences:Tuple types may have any number of parameters.\nTuple types are covariant in their parameters: Tuple{Int} is a subtype of Tuple{Any}. Therefore Tuple{Any} is considered an abstract type, and tuple types are only concrete if their parameters are.\nTuples do not have field names; fields are only accessed by index.Tuple values are written with parentheses and commas. When a tuple is constructed, an appropriate tuple type is generated on demand:julia> typeof((1,\"foo\",2.5))\nTuple{Int64,String,Float64}Note the implications of covariance:julia> Tuple{Int,AbstractString} <: Tuple{Real,Any}\ntrue\n\njulia> Tuple{Int,AbstractString} <: Tuple{Real,Real}\nfalse\n\njulia> Tuple{Int,AbstractString} <: Tuple{Real,}\nfalseIntuitively, this corresponds to the type of a function's arguments being a subtype of the function's signature (when the signature matches)."
},

{
    "location": "manual/types.html#Vararg-Tuple-Types-1",
    "page": "Types",
    "title": "Vararg Tuple Types",
    "category": "section",
    "text": "The last parameter of a tuple type can be the special type Vararg, which denotes any number of trailing elements:julia> mytupletype = Tuple{AbstractString,Vararg{Int}}\nTuple{AbstractString,Vararg{Int64,N} where N}\n\njulia> isa((\"1\",), mytupletype)\ntrue\n\njulia> isa((\"1\",1), mytupletype)\ntrue\n\njulia> isa((\"1\",1,2), mytupletype)\ntrue\n\njulia> isa((\"1\",1,2,3.0), mytupletype)\nfalseNotice that Vararg{T} corresponds to zero or more elements of type T. Vararg tuple types are used to represent the arguments accepted by varargs methods (see Varargs Functions).The type Vararg{T,N} corresponds to exactly N elements of type T.  NTuple{N,T} is a convenient alias for Tuple{Vararg{T,N}}, i.e. a tuple type containing exactly N elements of type T."
},

{
    "location": "manual/types.html#man-singleton-types-1",
    "page": "Types",
    "title": "Singleton Types",
    "category": "section",
    "text": "There is a special kind of abstract parametric type that must be mentioned here: singleton types. For each type, T, the \"singleton type\" Type{T} is an abstract type whose only instance is the object T. Since the definition is a little difficult to parse, let's look at some examples:julia> isa(Float64, Type{Float64})\ntrue\n\njulia> isa(Real, Type{Float64})\nfalse\n\njulia> isa(Real, Type{Real})\ntrue\n\njulia> isa(Float64, Type{Real})\nfalseIn other words, isa(A,Type{B}) is true if and only if A and B are the same object and that object is a type. Without the parameter, Type is simply an abstract type which has all type objects as its instances, including, of course, singleton types:julia> isa(Type{Float64}, Type)\ntrue\n\njulia> isa(Float64, Type)\ntrue\n\njulia> isa(Real, Type)\ntrueAny object that is not a type is not an instance of Type:julia> isa(1, Type)\nfalse\n\njulia> isa(\"foo\", Type)\nfalseUntil we discuss Parametric Methods and conversions, it is difficult to explain the utility of the singleton type construct, but in short, it allows one to specialize function behavior on specific type values. This is useful for writing methods (especially parametric ones) whose behavior depends on a type that is given as an explicit argument rather than implied by the type of one of its arguments.A few popular languages have singleton types, including Haskell, Scala and Ruby. In general usage, the term \"singleton type\" refers to a type whose only instance is a single value. This meaning applies to Julia's singleton types, but with that caveat that only type objects have singleton types."
},

{
    "location": "manual/types.html#Parametric-Primitive-Types-1",
    "page": "Types",
    "title": "Parametric Primitive Types",
    "category": "section",
    "text": "Primitive types can also be declared parametrically. For example, pointers are represented as primitive types which would be declared in Julia like this:# 32-bit system:\nprimitive type Ptr{T} 32 end\n\n# 64-bit system:\nprimitive type Ptr{T} 64 endThe slightly odd feature of these declarations as compared to typical parametric composite types, is that the type parameter T is not used in the definition of the type itself – it is just an abstract tag, essentially defining an entire family of types with identical structure, differentiated only by their type parameter. Thus, Ptr{Float64} and Ptr{Int64} are distinct types, even though they have identical representations. And of course, all specific pointer types are subtypes of the umbrella Ptr type:julia> Ptr{Float64} <: Ptr\ntrue\n\njulia> Ptr{Int64} <: Ptr\ntrue"
},

{
    "location": "manual/types.html#UnionAll-Types-1",
    "page": "Types",
    "title": "UnionAll Types",
    "category": "section",
    "text": "We have said that a parametric type like Ptr acts as a supertype of all its instances (Ptr{Int64} etc.). How does this work? Ptr itself cannot be a normal data type, since without knowing the type of the referenced data the type clearly cannot be used for memory operations. The answer is that Ptr (or other parametric types like Array) is a different kind of type called a UnionAll type. Such a type expresses the iterated union of types for all values of some parameter.UnionAll types are usually written using the keyword where. For example Ptr could be more accurately written as Ptr{T} where T, meaning all values whose type is Ptr{T} for some value of T. In this context, the parameter T is also often called a \"type variable\" since it is like a variable that ranges over types. Each where introduces a single type variable, so these expressions are nested for types with multiple parameters, for example Array{T,N} where N where T.The type application syntax A{B,C} requires A to be a UnionAll type, and first substitutes B for the outermost type variable in A. The result is expected to be another UnionAll type, into which C is then substituted. So A{B,C} is equivalent to A{B}{C}. This explains why it is possible to partially instantiate a type, as in Array{Float64}: the first parameter value has been fixed, but the second still ranges over all possible values. Using explicit where syntax, any subset of parameters can be fixed. For example, the type of all 1-dimensional arrays can be written as Array{T,1} where T.Type variables can be restricted with subtype relations. Array{T} where T<:Integer refers to all arrays whose element type is some kind of Integer. The syntax Array{<:Integer} is a convenient shorthand for Array{T} where T<:Integer. Type variables can have both lower and upper bounds. Array{T} where Int<:T<:Number refers to all arrays of Numbers that are able to contain Ints (since T must be at least as big as Int). The syntax where T>:Int also works to specify only the lower bound of a type variable, and Array{>:Int} is equivalent to Array{T} where T>:Int.Since where expressions nest, type variable bounds can refer to outer type variables. For example Tuple{T,Array{S}} where S<:AbstractArray{T} where T<:Real refers to 2-tuples whose first element is some Real, and whose second element is an Array of any kind of array whose element type contains the type of the first tuple element.The where keyword itself can be nested inside a more complex declaration. For example, consider the two types created by the following declarations:julia> const T1 = Array{Array{T,1} where T, 1}\nArray{Array{T,1} where T,1}\n\njulia> const T2 = Array{Array{T,1}, 1} where T\nArray{Array{T,1},1} where TType T1 defines a 1-dimensional array of 1-dimensional arrays; each of the inner arrays consists of objects of the same type, but this type may vary from one inner array to the next. On the other hand, type T2 defines a 1-dimensional array of 1-dimensional arrays all of whose inner arrays must have the same type.  Note that T2 is an abstract type, e.g., Array{Array{Int,1},1} <: T2, whereas T1 is a concrete type. As a consequence, T1 can be constructed with a zero-argument constructor a=T1() but T2 cannot.There is a convenient syntax for naming such types, similar to the short form of function definition syntax:Vector{T} = Array{T,1}This is equivalent to const Vector = Array{T,1} where T. Writing Vector{Float64} is equivalent to writing Array{Float64,1}, and the umbrella type Vector has as instances all Array objects where the second parameter – the number of array dimensions – is 1, regardless of what the element type is. In languages where parametric types must always be specified in full, this is not especially helpful, but in Julia, this allows one to write just Vector for the abstract type including all one-dimensional dense arrays of any element type."
},

{
    "location": "manual/types.html#Type-Aliases-1",
    "page": "Types",
    "title": "Type Aliases",
    "category": "section",
    "text": "Sometimes it is convenient to introduce a new name for an already expressible type. This can be done with a simple assignment statement. For example, UInt is aliased to either UInt32 or UInt64 as is appropriate for the size of pointers on the system:# 32-bit system:\njulia> UInt\nUInt32\n\n# 64-bit system:\njulia> UInt\nUInt64This is accomplished via the following code in base/boot.jl:if Int === Int64\n    const UInt = UInt64\nelse\n    const UInt = UInt32\nendOf course, this depends on what Int is aliased to – but that is predefined to be the correct type – either Int32 or Int64.(Note that unlike Int, Float does not exist as a type alias for a specific sized AbstractFloat. Unlike with integer registers, the floating point register sizes are specified by the IEEE-754 standard. Whereas the size of Int reflects the size of a native pointer on that machine.)"
},

{
    "location": "manual/types.html#Operations-on-Types-1",
    "page": "Types",
    "title": "Operations on Types",
    "category": "section",
    "text": "Since types in Julia are themselves objects, ordinary functions can operate on them. Some functions that are particularly useful for working with or exploring types have already been introduced, such as the <: operator, which indicates whether its left hand operand is a subtype of its right hand operand.The isa function tests if an object is of a given type and returns true or false:julia> isa(1, Int)\ntrue\n\njulia> isa(1, AbstractFloat)\nfalseThe typeof() function, already used throughout the manual in examples, returns the type of its argument. Since, as noted above, types are objects, they also have types, and we can ask what their types are:julia> typeof(Rational{Int})\nDataType\n\njulia> typeof(Union{Real,Float64,Rational})\nDataType\n\njulia> typeof(Union{Real,String})\nUnionWhat if we repeat the process? What is the type of a type of a type? As it happens, types are all composite values and thus all have a type of DataType:julia> typeof(DataType)\nDataType\n\njulia> typeof(Union)\nDataTypeDataType is its own type.Another operation that applies to some types is supertype(), which reveals a type's supertype. Only declared types (DataType) have unambiguous supertypes:julia> supertype(Float64)\nAbstractFloat\n\njulia> supertype(Number)\nAny\n\njulia> supertype(AbstractString)\nAny\n\njulia> supertype(Any)\nAnyIf you apply supertype() to other type objects (or non-type objects), a MethodError is raised:julia> supertype(Union{Float64,Int64})\nERROR: MethodError: no method matching supertype(::Type{Union{Float64, Int64}})\nClosest candidates are:\n  supertype(!Matched::DataType) at operators.jl:41\n  supertype(!Matched::UnionAll) at operators.jl:46"
},

{
    "location": "manual/types.html#Custom-pretty-printing-1",
    "page": "Types",
    "title": "Custom pretty-printing",
    "category": "section",
    "text": "Often, one wants to customize how instances of a type are displayed.  This is accomplished by overloading the show() function.  For example, suppose we define a type to represent complex numbers in polar form:julia> struct Polar{T<:Real} <: Number\n           r::T\n           Θ::T\n       end\n\njulia> Polar(r::Real,Θ::Real) = Polar(promote(r,Θ)...)\nPolarHere, we've added a custom constructor function so that it can take arguments of different Real types and promote them to a common type (see Constructors and Conversion and Promotion). (Of course, we would have to define lots of other methods, too, to make it act like a Number, e.g. +, *, one, zero, promotion rules and so on.) By default, instances of this type display rather simply, with information about the type name and the field values, as e.g. Polar{Float64}(3.0,4.0).If we want it to display instead as 3.0 * exp(4.0im), we would define the following method to print the object to a given output object io (representing a file, terminal, buffer, etcetera; see Networking and Streams):julia> Base.show(io::IO, z::Polar) = print(io, z.r, \" * exp(\", z.Θ, \"im)\")More fine-grained control over display of Polar objects is possible. In particular, sometimes one wants both a verbose multi-line printing format, used for displaying a single object in the REPL and other interactive environments, and also a more compact single-line format used for print() or for displaying the object as part of another object (e.g. in an array). Although by default the show(io, z) function is called in both cases, you can define a different multi-line format for displaying an object by overloading a three-argument form of show that takes the text/plain MIME type as its second argument (see Multimedia I/O), for example:julia> Base.show{T}(io::IO, ::MIME\"text/plain\", z::Polar{T}) =\n           print(io, \"Polar{$T} complex number:\\n   \", z)(Note that print(..., z) here will call the 2-argument show(io, z) method.) This results in:julia> Polar(3, 4.0)\nPolar{Float64} complex number:\n   3.0 * exp(4.0im)\n\njulia> [Polar(3, 4.0), Polar(4.0,5.3)]\n2-element Array{Polar{Float64},1}:\n 3.0 * exp(4.0im)\n 4.0 * exp(5.3im)where the single-line show(io, z) form is still used for an array of Polar values.   Technically, the REPL calls display(z) to display the result of executing a line, which defaults to show(STDOUT, MIME(\"text/plain\"), z), which in turn defaults to show(STDOUT, z), but you should not define new display() methods unless you are defining a new multimedia display handler (see Multimedia I/O).Moreover, you can also define show methods for other MIME types in order to enable richer display (HTML, images, etcetera) of objects in environments that support this (e.g. IJulia).   For example, we can define formatted HTML display of Polar objects, with superscripts and italics, via:julia> Base.show{T}(io::IO, ::MIME\"text/html\", z::Polar{T}) =\n           println(io, \"<code>Polar{$T}</code> complex number: \",\n                   z.r, \" <i>e</i><sup>\", z.Θ, \" <i>i</i></sup>\")A Polar object will then display automatically using HTML in an environment that supports HTML display, but you can call show manually to get HTML output if you want:julia> show(STDOUT, \"text/html\", Polar(3.0,4.0))\n<code>Polar{Float64}</code> complex number: 3.0 <i>e</i><sup>4.0 <i>i</i></sup><p>An HTML renderer would display this as: <code>Polar{Float64}</code> complex number: 3.0 <i>e</i><sup>4.0 <i>i</i></sup></p>"
},

{
    "location": "manual/types.html#\"Value-types\"-1",
    "page": "Types",
    "title": "\"Value types\"",
    "category": "section",
    "text": "In Julia, you can't dispatch on a value such as true or false. However, you can dispatch on parametric types, and Julia allows you to include \"plain bits\" values (Types, Symbols, Integers, floating-point numbers, tuples, etc.) as type parameters.  A common example is the dimensionality parameter in Array{T,N}, where T is a type (e.g., Float64) but N is just an Int.You can create your own custom types that take values as parameters, and use them to control dispatch of custom types. By way of illustration of this idea, let's introduce a parametric type, Val{T}, which serves as a customary way to exploit this technique for cases where you don't need a more elaborate hierarchy.Val is defined as:julia> struct Val{T}\n       endThere is no more to the implementation of Val than this.  Some functions in Julia's standard library accept Val types as arguments, and you can also use it to write your own functions.  For example:julia> firstlast(::Type{Val{true}}) = \"First\"\nfirstlast (generic function with 1 method)\n\njulia> firstlast(::Type{Val{false}}) = \"Last\"\nfirstlast (generic function with 2 methods)\n\njulia> firstlast(Val{true})\n\"First\"\n\njulia> firstlast(Val{false})\n\"Last\"For consistency across Julia, the call site should always pass a Valtype rather than creating an instance, i.e., use foo(Val{:bar}) rather than foo(Val{:bar}()).It's worth noting that it's extremely easy to mis-use parametric \"value\" types, including Val; in unfavorable cases, you can easily end up making the performance of your code much worse.  In particular, you would never want to write actual code as illustrated above.  For more information about the proper (and improper) uses of Val, please read the more extensive discussion in the performance tips."
},

{
    "location": "manual/types.html#man-nullable-types-1",
    "page": "Types",
    "title": "Nullable Types: Representing Missing Values",
    "category": "section",
    "text": "In many settings, you need to interact with a value of type T that may or may not exist. To handle these settings, Julia provides a parametric type called Nullable{T}, which can be thought of as a specialized container type that can contain either zero or one values. Nullable{T} provides a minimal interface designed to ensure that interactions with missing values are safe. At present, the interface consists of several possible interactions:Construct a Nullable object.\nCheck if a Nullable object has a missing value.\nAccess the value of a Nullable object with a guarantee that a NullException will be thrown if the object's value is missing.\nAccess the value of a Nullable object with a guarantee that a default value of type T will be returned if the object's value is missing.\nPerform an operation on the value (if it exists) of a Nullable object, getting a Nullable result. The result will be missing if the original value was missing.\nPerforming a test on the value (if it exists) of a Nullable object, getting a result that is missing if either the Nullable itself was missing, or the test failed.\nPerform general operations on single Nullable objects, propagating the missing data."
},

{
    "location": "manual/types.html#Constructing-[Nullable](@ref)-objects-1",
    "page": "Types",
    "title": "Constructing Nullable objects",
    "category": "section",
    "text": "To construct an object representing a missing value of type T, use the Nullable{T}() function:julia> x1 = Nullable{Int64}()\nNullable{Int64}()\n\njulia> x2 = Nullable{Float64}()\nNullable{Float64}()\n\njulia> x3 = Nullable{Vector{Int64}}()\nNullable{Array{Int64,1}}()To construct an object representing a non-missing value of type T, use the Nullable(x::T) function:julia> x1 = Nullable(1)\nNullable{Int64}(1)\n\njulia> x2 = Nullable(1.0)\nNullable{Float64}(1.0)\n\njulia> x3 = Nullable([1, 2, 3])\nNullable{Array{Int64,1}}([1, 2, 3])Note the core distinction between these two ways of constructing a Nullable object: in one style, you provide a type, T, as a function parameter; in the other style, you provide a single value of type T as an argument."
},

{
    "location": "manual/types.html#Checking-if-a-Nullable-object-has-a-value-1",
    "page": "Types",
    "title": "Checking if a Nullable object has a value",
    "category": "section",
    "text": "You can check if a Nullable object has any value using isnull():julia> isnull(Nullable{Float64}())\ntrue\n\njulia> isnull(Nullable(0.0))\nfalse"
},

{
    "location": "manual/types.html#Safely-accessing-the-value-of-a-Nullable-object-1",
    "page": "Types",
    "title": "Safely accessing the value of a Nullable object",
    "category": "section",
    "text": "You can safely access the value of a Nullable object using get():julia> get(Nullable{Float64}())\nERROR: NullException()\nStacktrace:\n [1] get(::Nullable{Float64}) at ./nullable.jl:92\n\njulia> get(Nullable(1.0))\n1.0If the value is not present, as it would be for Nullable{Float64}, a NullException error will be thrown. The error-throwing nature of the get() function ensures that any attempt to access a missing value immediately fails.In cases for which a reasonable default value exists that could be used when a Nullable object's value turns out to be missing, you can provide this default value as a second argument to get():julia> get(Nullable{Float64}(), 0.0)\n0.0\n\njulia> get(Nullable(1.0), 0.0)\n1.0tip: Tip\nMake sure the type of the default value passed to get() and that of the Nullable object match to avoid type instability, which could hurt performance. Use convert() manually if needed."
},

{
    "location": "manual/types.html#Performing-operations-on-Nullable-objects-1",
    "page": "Types",
    "title": "Performing operations on Nullable objects",
    "category": "section",
    "text": "Nullable objects represent values that are possibly missing, and it is possible to write all code using these objects by first testing to see if the value is missing with isnull(), and then doing an appropriate action. However, there are some common use cases where the code could be more concise or clear by using a higher-order function.The map function takes as arguments a function f and a Nullable value x. It produces a Nullable:If x is a missing value, then it produces a missing value;\nIf x has a value, then it produces a Nullable containing f(get(x)) as value.This is useful for performing simple operations on values that might be missing if the desired behaviour is to simply propagate the missing values forward.The filter function takes as arguments a predicate function p (that is, a function returning a boolean) and a Nullable value x. It produces a Nullable value:If x is a missing value, then it produces a missing value;\nIf p(get(x)) is true, then it produces the original value x;\nIf p(get(x)) is false, then it produces a missing value.In this way, filter can be thought of as selecting only allowable values, and converting non-allowable values to missing values.While map and filter are useful in specific cases, by far the most useful higher-order function is broadcast, which can handle a wide variety of cases, including making existing operations work and propagate Nullables. An example will motivate the need for broadcast. Suppose we have a function that computes the greater of two real roots of a quadratic equation, using the quadratic formula:julia> root(a::Real, b::Real, c::Real) = (-b + √(b^2 - 4a*c)) / 2a\nroot (generic function with 1 method)We may verify that the result of root(1, -9, 20) is 5.0, as we expect, since 5.0 is the greater of two real roots of the quadratic equation.Suppose now that we want to find the greatest real root of a quadratic equations where the coefficients might be missing values. Having missing values in datasets is a common occurrence in real-world data, and so it is important to be able to deal with them. But we cannot find the roots of an equation if we do not know all the coefficients. The best solution to this will depend on the particular use case; perhaps we should throw an error. However, for this example, we will assume that the best solution is to propagate the missing values forward; that is, if any input is missing, we simply produce a missing output.The broadcast() function makes this task easy; we can simply pass the root function we wrote to broadcast:julia> broadcast(root, Nullable(1), Nullable(-9), Nullable(20))\nNullable{Float64}(5.0)\n\njulia> broadcast(root, Nullable(1), Nullable{Int}(), Nullable{Int}())\nNullable{Float64}()\n\njulia> broadcast(root, Nullable{Int}(), Nullable(-9), Nullable(20))\nNullable{Float64}()If one or more of the inputs is missing, then the output of broadcast() will be missing.There exists special syntactic sugar for the broadcast() function using a dot notation:julia> root.(Nullable(1), Nullable(-9), Nullable(20))\nNullable{Float64}(5.0)In particular, the regular arithmetic operators can be broadcast() conveniently using .-prefixed operators:julia> Nullable(2) ./ Nullable(3) .+ Nullable(1.0)\nNullable{Float64}(1.66667)"
},

{
    "location": "manual/methods.html#",
    "page": "Methods",
    "title": "Methods",
    "category": "page",
    "text": ""
},

{
    "location": "manual/methods.html#Methods-1",
    "page": "Methods",
    "title": "Methods",
    "category": "section",
    "text": "Recall from Functions that a function is an object that maps a tuple of arguments to a return value, or throws an exception if no appropriate value can be returned. It is common for the same conceptual function or operation to be implemented quite differently for different types of arguments: adding two integers is very different from adding two floating-point numbers, both of which are distinct from adding an integer to a floating-point number. Despite their implementation differences, these operations all fall under the general concept of \"addition\". Accordingly, in Julia, these behaviors all belong to a single object: the + function.To facilitate using many different implementations of the same concept smoothly, functions need not be defined all at once, but can rather be defined piecewise by providing specific behaviors for certain combinations of argument types and counts. A definition of one possible behavior for a function is called a method. Thus far, we have presented only examples of functions defined with a single method, applicable to all types of arguments. However, the signatures of method definitions can be annotated to indicate the types of arguments in addition to their number, and more than a single method definition may be provided. When a function is applied to a particular tuple of arguments, the most specific method applicable to those arguments is applied. Thus, the overall behavior of a function is a patchwork of the behaviors of its various method definitions. If the patchwork is well designed, even though the implementations of the methods may be quite different, the outward behavior of the function will appear seamless and consistent.The choice of which method to execute when a function is applied is called dispatch. Julia allows the dispatch process to choose which of a function's methods to call based on the number of arguments given, and on the types of all of the function's arguments. This is different than traditional object-oriented languages, where dispatch occurs based only on the first argument, which often has a special argument syntax, and is sometimes implied rather than explicitly written as an argument. [1] Using all of a function's arguments to choose which method should be invoked, rather than just the first, is known as multiple dispatch. Multiple dispatch is particularly useful for mathematical code, where it makes little sense to artificially deem the operations to \"belong\" to one argument more than any of the others: does the addition operation in x + y belong to x any more than it does to y? The implementation of a mathematical operator generally depends on the types of all of its arguments. Even beyond mathematical operations, however, multiple dispatch ends up being a powerful and convenient paradigm for structuring and organizing programs.[1]: In C++ or Java, for example, in a method call like obj.meth(arg1,arg2), the object obj \"receives\" the method call and is implicitly passed to the method via the this keyword, rather than as an explicit method argument. When the current this object is the receiver of a method call, it can be omitted altogether, writing just meth(arg1,arg2), with this implied as the receiving object."
},

{
    "location": "manual/methods.html#Defining-Methods-1",
    "page": "Methods",
    "title": "Defining Methods",
    "category": "section",
    "text": "Until now, we have, in our examples, defined only functions with a single method having unconstrained argument types. Such functions behave just like they would in traditional dynamically typed languages. Nevertheless, we have used multiple dispatch and methods almost continually without being aware of it: all of Julia's standard functions and operators, like the aforementioned + function, have many methods defining their behavior over various possible combinations of argument type and count.When defining a function, one can optionally constrain the types of parameters it is applicable to, using the :: type-assertion operator, introduced in the section on Composite Types:julia> f(x::Float64, y::Float64) = 2x + y\nf (generic function with 1 method)This function definition applies only to calls where x and y are both values of type Float64:julia> f(2.0, 3.0)\n7.0Applying it to any other types of arguments will result in a MethodError:julia> f(2.0, 3)\nERROR: MethodError: no method matching f(::Float64, ::Int64)\nClosest candidates are:\n  f(::Float64, !Matched::Float64) at none:1\n\njulia> f(Float32(2.0), 3.0)\nERROR: MethodError: no method matching f(::Float32, ::Float64)\nClosest candidates are:\n  f(!Matched::Float64, ::Float64) at none:1\n\njulia> f(2.0, \"3.0\")\nERROR: MethodError: no method matching f(::Float64, ::String)\nClosest candidates are:\n  f(::Float64, !Matched::Float64) at none:1\n\njulia> f(\"2.0\", \"3.0\")\nERROR: MethodError: no method matching f(::String, ::String)As you can see, the arguments must be precisely of type Float64. Other numeric types, such as integers or 32-bit floating-point values, are not automatically converted to 64-bit floating-point, nor are strings parsed as numbers. Because Float64 is a concrete type and concrete types cannot be subclassed in Julia, such a definition can only be applied to arguments that are exactly of type Float64. It may often be useful, however, to write more general methods where the declared parameter types are abstract:julia> f(x::Number, y::Number) = 2x - y\nf (generic function with 2 methods)\n\njulia> f(2.0, 3)\n1.0This method definition applies to any pair of arguments that are instances of Number. They need not be of the same type, so long as they are each numeric values. The problem of handling disparate numeric types is delegated to the arithmetic operations in the expression 2x - y.To define a function with multiple methods, one simply defines the function multiple times, with different numbers and types of arguments. The first method definition for a function creates the function object, and subsequent method definitions add new methods to the existing function object. The most specific method definition matching the number and types of the arguments will be executed when the function is applied. Thus, the two method definitions above, taken together, define the behavior for f over all pairs of instances of the abstract type Number – but with a different behavior specific to pairs of Float64 values. If one of the arguments is a 64-bit float but the other one is not, then the f(Float64,Float64) method cannot be called and the more general f(Number,Number) method must be used:julia> f(2.0, 3.0)\n7.0\n\njulia> f(2, 3.0)\n1.0\n\njulia> f(2.0, 3)\n1.0\n\njulia> f(2, 3)\n1The 2x + y definition is only used in the first case, while the 2x - y definition is used in the others. No automatic casting or conversion of function arguments is ever performed: all conversion in Julia is non-magical and completely explicit. Conversion and Promotion, however, shows how clever application of sufficiently advanced technology can be indistinguishable from magic. [Clarke61]For non-numeric values, and for fewer or more than two arguments, the function f remains undefined, and applying it will still result in a MethodError:julia> f(\"foo\", 3)\nERROR: MethodError: no method matching f(::String, ::Int64)\nClosest candidates are:\n  f(!Matched::Number, ::Number) at none:1\n\njulia> f()\nERROR: MethodError: no method matching f()\nClosest candidates are:\n  f(!Matched::Float64, !Matched::Float64) at none:1\n  f(!Matched::Number, !Matched::Number) at none:1You can easily see which methods exist for a function by entering the function object itself in an interactive session:julia> f\nf (generic function with 2 methods)This output tells us that f is a function object with two methods. To find out what the signatures of those methods are, use the methods() function:julia> methods(f)\n# 2 methods for generic function \"f\":\nf(x::Float64, y::Float64) in Main at none:1\nf(x::Number, y::Number) in Main at none:1which shows that f has two methods, one taking two Float64 arguments and one taking arguments of type Number. It also indicates the file and line number where the methods were defined: because these methods were defined at the REPL, we get the apparent line number none:1.In the absence of a type declaration with ::, the type of a method parameter is Any by default, meaning that it is unconstrained since all values in Julia are instances of the abstract type Any. Thus, we can define a catch-all method for f like so:julia> f(x,y) = println(\"Whoa there, Nelly.\")\nf (generic function with 3 methods)\n\njulia> f(\"foo\", 1)\nWhoa there, Nelly.This catch-all is less specific than any other possible method definition for a pair of parameter values, so it will only be called on pairs of arguments to which no other method definition applies.Although it seems a simple concept, multiple dispatch on the types of values is perhaps the single most powerful and central feature of the Julia language. Core operations typically have dozens of methods:julia> methods(+)\n# 180 methods for generic function \"+\":\n+(x::Bool, z::Complex{Bool}) in Base at complex.jl:224\n+(x::Bool, y::Bool) in Base at bool.jl:89\n+(x::Bool) in Base at bool.jl:86\n+(x::Bool, y::T) where T<:AbstractFloat in Base at bool.jl:96\n+(x::Bool, z::Complex) in Base at complex.jl:231\n+(a::Float16, b::Float16) in Base at float.jl:372\n+(x::Float32, y::Float32) in Base at float.jl:374\n+(x::Float64, y::Float64) in Base at float.jl:375\n+(z::Complex{Bool}, x::Bool) in Base at complex.jl:225\n+(z::Complex{Bool}, x::Real) in Base at complex.jl:239\n+(x::Char, y::Integer) in Base at char.jl:40\n+(c::BigInt, x::BigFloat) in Base.MPFR at mpfr.jl:303\n+(a::BigInt, b::BigInt, c::BigInt, d::BigInt, e::BigInt) in Base.GMP at gmp.jl:303\n+(a::BigInt, b::BigInt, c::BigInt, d::BigInt) in Base.GMP at gmp.jl:296\n+(a::BigInt, b::BigInt, c::BigInt) in Base.GMP at gmp.jl:290\n+(x::BigInt, y::BigInt) in Base.GMP at gmp.jl:258\n+(x::BigInt, c::Union{UInt16, UInt32, UInt64, UInt8}) in Base.GMP at gmp.jl:315\n...\n+(a, b, c, xs...) at operators.jl:119Multiple dispatch together with the flexible parametric type system give Julia its ability to abstractly express high-level algorithms decoupled from implementation details, yet generate efficient, specialized code to handle each case at run time."
},

{
    "location": "manual/methods.html#man-ambiguities-1",
    "page": "Methods",
    "title": "Method Ambiguities",
    "category": "section",
    "text": "It is possible to define a set of function methods such that there is no unique most specific method applicable to some combinations of arguments:julia> g(x::Float64, y) = 2x + y\ng (generic function with 1 method)\n\njulia> g(x, y::Float64) = x + 2y\ng (generic function with 2 methods)\n\njulia> g(2.0, 3)\n7.0\n\njulia> g(2, 3.0)\n8.0\n\njulia> g(2.0, 3.0)\nERROR: MethodError: g(::Float64, ::Float64) is ambiguous.\n[...]Here the call g(2.0, 3.0) could be handled by either the g(Float64, Any) or the g(Any, Float64) method, and neither is more specific than the other. In such cases, Julia raises a MethodError rather than arbitrarily picking a method. You can avoid method ambiguities by specifying an appropriate method for the intersection case:julia> g(x::Float64, y::Float64) = 2x + 2y\ng (generic function with 3 methods)\n\njulia> g(2.0, 3)\n7.0\n\njulia> g(2, 3.0)\n8.0\n\njulia> g(2.0, 3.0)\n10.0It is recommended that the disambiguating method be defined first, since otherwise the ambiguity exists, if transiently, until the more specific method is defined.In more complex cases, resolving method ambiguities involves a certain element of design; this topic is explored further below."
},

{
    "location": "manual/methods.html#Parametric-Methods-1",
    "page": "Methods",
    "title": "Parametric Methods",
    "category": "section",
    "text": "Method definitions can optionally have type parameters qualifying the signature:julia> same_type(x::T, y::T) where {T} = true\nsame_type (generic function with 1 method)\n\njulia> same_type(x,y) = false\nsame_type (generic function with 2 methods)The first method applies whenever both arguments are of the same concrete type, regardless of what type that is, while the second method acts as a catch-all, covering all other cases. Thus, overall, this defines a boolean function that checks whether its two arguments are of the same type:julia> same_type(1, 2)\ntrue\n\njulia> same_type(1, 2.0)\nfalse\n\njulia> same_type(1.0, 2.0)\ntrue\n\njulia> same_type(\"foo\", 2.0)\nfalse\n\njulia> same_type(\"foo\", \"bar\")\ntrue\n\njulia> same_type(Int32(1), Int64(2))\nfalseSuch definitions correspond to methods whose type signatures are UnionAll types (see UnionAll Types).This kind of definition of function behavior by dispatch is quite common – idiomatic, even – in Julia. Method type parameters are not restricted to being used as the types of arguments: they can be used anywhere a value would be in the signature of the function or body of the function. Here's an example where the method type parameter T is used as the type parameter to the parametric type Vector{T} in the method signature:julia> myappend(v::Vector{T}, x::T) where {T} = [v..., x]\nmyappend (generic function with 1 method)\n\njulia> myappend([1,2,3],4)\n4-element Array{Int64,1}:\n 1\n 2\n 3\n 4\n\njulia> myappend([1,2,3],2.5)\nERROR: MethodError: no method matching myappend(::Array{Int64,1}, ::Float64)\nClosest candidates are:\n  myappend(::Array{T,1}, !Matched::T) where T at none:1\n\njulia> myappend([1.0,2.0,3.0],4.0)\n4-element Array{Float64,1}:\n 1.0\n 2.0\n 3.0\n 4.0\n\njulia> myappend([1.0,2.0,3.0],4)\nERROR: MethodError: no method matching myappend(::Array{Float64,1}, ::Int64)\nClosest candidates are:\n  myappend(::Array{T,1}, !Matched::T) where T at none:1As you can see, the type of the appended element must match the element type of the vector it is appended to, or else a MethodError is raised. In the following example, the method type parameter T is used as the return value:julia> mytypeof(x::T) where {T} = T\nmytypeof (generic function with 1 method)\n\njulia> mytypeof(1)\nInt64\n\njulia> mytypeof(1.0)\nFloat64Just as you can put subtype constraints on type parameters in type declarations (see Parametric Types), you can also constrain type parameters of methods:julia> same_type_numeric(x::T, y::T) where {T<:Number} = true\nsame_type_numeric (generic function with 1 method)\n\njulia> same_type_numeric(x::Number, y::Number) = false\nsame_type_numeric (generic function with 2 methods)\n\njulia> same_type_numeric(1, 2)\ntrue\n\njulia> same_type_numeric(1, 2.0)\nfalse\n\njulia> same_type_numeric(1.0, 2.0)\ntrue\n\njulia> same_type_numeric(\"foo\", 2.0)\nERROR: MethodError: no method matching same_type_numeric(::String, ::Float64)\nClosest candidates are:\n  same_type_numeric(!Matched::T<:Number, ::T<:Number) where T<:Number at none:1\n  same_type_numeric(!Matched::Number, ::Number) at none:1\n\njulia> same_type_numeric(\"foo\", \"bar\")\nERROR: MethodError: no method matching same_type_numeric(::String, ::String)\n\njulia> same_type_numeric(Int32(1), Int64(2))\nfalseThe same_type_numeric function behaves much like the same_type function defined above, but is only defined for pairs of numbers.Parametric methods allow the same syntax as where expressions used to write types (see UnionAll Types). If there is only a single parameter, the enclosing curly braces (in where {T}) can be omitted, but are often preferred for clarity. Multiple parameters can be separated with commas, e.g. where {T, S<:Real}, or written using nested where, e.g. where S<:Real where T."
},

{
    "location": "manual/methods.html#Redefining-Methods-1",
    "page": "Methods",
    "title": "Redefining Methods",
    "category": "section",
    "text": "When redefining a method or adding new methods, it is important to realize that these changes don't take effect immediately. This is key to Julia's ability to statically infer and compile code to run fast, without the usual JIT tricks and overhead. Indeed, any new method definition won't be visible to the current runtime environment, including Tasks and Threads (and any previously defined @generated functions). Let's start with an example to see what this means:julia> function tryeval()\n           @eval newfun() = 1\n           newfun()\n       end\ntryeval (generic function with 1 method)\n\njulia> tryeval()\nERROR: MethodError: no method matching newfun()\nThe applicable method may be too new: running in world age xxxx1, while current world is xxxx2.\nClosest candidates are:\n  newfun() at none:1 (method too new to be called from this world context.)\n in tryeval() at none:1\n ...\n\njulia> newfun()\n1In this example, observe that the new definition for newfun has been created, but can't be immediately called. The new global is immediately visible to the tryeval function, so you could write return newfun (without parentheses). But neither you, nor any of your callers, nor the functions they call, or etc. can call this new method definition!But there's an exception: future calls to newfun from the REPL work as expected, being able to both see and call the new definition of newfun.However, future calls to tryeval will continue to see the definition of newfun as it was at the previous statement at the REPL, and thus before that call to tryeval.You may want to try this for yourself to see how it works.The implementation of this behavior is a \"world age counter\". This monotonically increasing value tracks each method definition operation. This allows describing \"the set of method definitions visible to a given runtime environment\" as a single number, or \"world age\". It also allows comparing the methods available in two worlds just by comparing their ordinal value. In the example above, we see that the \"current world\" (in which the method newfun() exists), is one greater than the task-local \"runtime world\" that was fixed when the execution of tryeval started.Sometimes it is necessary to get around this (for example, if you are implementing the above REPL). Fortunately, there is an easy solution: call the function using Base.invokelatest:julia> function tryeval2()\n           @eval newfun2() = 2\n           Base.invokelatest(newfun2)\n       end\ntryeval2 (generic function with 1 method)\n\njulia> tryeval2()\n2Finally, let's take a look at some more complex examples where this rule comes into play. Define a function f(x), which initially has one method:julia> f(x) = \"original definition\"\nf (generic function with 1 method)Start some other operations that use f(x):julia> g(x) = f(x)\ng (generic function with 1 method)\n\njulia> t = @async f(wait()); yield();Now we add some new methods to f(x):julia> f(x::Int) = \"definition for Int\"\nf (generic function with 2 methods)\n\njulia> f(x::Type{Int}) = \"definition for Type{Int}\"\nf (generic function with 3 methods)Compare how these results differ:julia> f(1)\n\"definition for Int\"\n\njulia> g(1)\n\"definition for Int\"\n\njulia> wait(schedule(t, 1))\n\"original definition\"\n\njulia> t = @async f(wait()); yield();\n\njulia> wait(schedule(t, 1))\n\"definition for Int\""
},

{
    "location": "manual/methods.html#Parametrically-constrained-Varargs-methods-1",
    "page": "Methods",
    "title": "Parametrically-constrained Varargs methods",
    "category": "section",
    "text": "Function parameters can also be used to constrain the number of arguments that may be supplied to a \"varargs\" function (Varargs Functions).  The notation Vararg{T,N} is used to indicate such a constraint.  For example:julia> bar(a,b,x::Vararg{Any,2}) = (a,b,x)\nbar (generic function with 1 method)\n\njulia> bar(1,2,3)\nERROR: MethodError: no method matching bar(::Int64, ::Int64, ::Int64)\nClosest candidates are:\n  bar(::Any, ::Any, ::Any, !Matched::Any) at none:1\n\njulia> bar(1,2,3,4)\n(1, 2, (3, 4))\n\njulia> bar(1,2,3,4,5)\nERROR: MethodError: no method matching bar(::Int64, ::Int64, ::Int64, ::Int64, ::Int64)\nClosest candidates are:\n  bar(::Any, ::Any, ::Any, ::Any) at none:1More usefully, it is possible to constrain varargs methods by a parameter. For example:function getindex(A::AbstractArray{T,N}, indexes::Vararg{Number,N}) where {T,N}would be called only when the number of indexes matches the dimensionality of the array."
},

{
    "location": "manual/methods.html#Note-on-Optional-and-keyword-Arguments-1",
    "page": "Methods",
    "title": "Note on Optional and keyword Arguments",
    "category": "section",
    "text": "As mentioned briefly in Functions, optional arguments are implemented as syntax for multiple method definitions. For example, this definition:f(a=1,b=2) = a+2btranslates to the following three methods:f(a,b) = a+2b\nf(a) = f(a,2)\nf() = f(1,2)This means that calling f() is equivalent to calling f(1,2). In this case the result is 5, because f(1,2) invokes the first method of f above. However, this need not always be the case. If you define a fourth method that is more specialized for integers:f(a::Int,b::Int) = a-2bthen the result of both f() and f(1,2) is -3. In other words, optional arguments are tied to a function, not to any specific method of that function. It depends on the types of the optional arguments which method is invoked. When optional arguments are defined in terms of a global variable, the type of the optional argument may even change at run-time.Keyword arguments behave quite differently from ordinary positional arguments. In particular, they do not participate in method dispatch. Methods are dispatched based only on positional arguments, with keyword arguments processed after the matching method is identified."
},

{
    "location": "manual/methods.html#Function-like-objects-1",
    "page": "Methods",
    "title": "Function-like objects",
    "category": "section",
    "text": "Methods are associated with types, so it is possible to make any arbitrary Julia object \"callable\" by adding methods to its type. (Such \"callable\" objects are sometimes called \"functors.\")For example, you can define a type that stores the coefficients of a polynomial, but behaves like a function evaluating the polynomial:julia> struct Polynomial{R}\n           coeffs::Vector{R}\n       end\n\njulia> function (p::Polynomial)(x)\n           v = p.coeffs[end]\n           for i = (length(p.coeffs)-1):-1:1\n               v = v*x + p.coeffs[i]\n           end\n           return v\n       endNotice that the function is specified by type instead of by name. In the function body, p will refer to the object that was called. A Polynomial can be used as follows:julia> p = Polynomial([1,10,100])\nPolynomial{Int64}([1, 10, 100])\n\njulia> p(3)\n931This mechanism is also the key to how type constructors and closures (inner functions that refer to their surrounding environment) work in Julia, discussed later in the manual."
},

{
    "location": "manual/methods.html#Empty-generic-functions-1",
    "page": "Methods",
    "title": "Empty generic functions",
    "category": "section",
    "text": "Occasionally it is useful to introduce a generic function without yet adding methods. This can be used to separate interface definitions from implementations. It might also be done for the purpose of documentation or code readability. The syntax for this is an empty function block without a tuple of arguments:function emptyfunc\nend"
},

{
    "location": "manual/methods.html#man-method-design-ambiguities-1",
    "page": "Methods",
    "title": "Method design and the avoidance of ambiguities",
    "category": "section",
    "text": "Julia's method polymorphism is one of its most powerful features, yet exploiting this power can pose design challenges.  In particular, in more complex method hierarchies it is not uncommon for ambiguities to arise.Above, it was pointed out that one can resolve ambiguities likef(x, y::Int) = 1\nf(x::Int, y) = 2by defining a methodf(x::Int, y::Int) = 3This is often the right strategy; however, there are circumstances where following this advice blindly can be counterproductive. In particular, the more methods a generic function has, the more possibilities there are for ambiguities. When your method hierarchies get more complicated than this simple example, it can be worth your while to think carefully about alternative strategies.Below we discuss particular challenges and some alternative ways to resolve such issues."
},

{
    "location": "manual/methods.html#Tuple-and-NTuple-arguments-1",
    "page": "Methods",
    "title": "Tuple and NTuple arguments",
    "category": "section",
    "text": "Tuple (and NTuple) arguments present special challenges. For example,f(x::NTuple{N,Int}) where {N} = 1\nf(x::NTuple{N,Float64}) where {N} = 2are ambiguous because of the possibility that N == 0: there are no elements to determine whether the Int or Float64 variant should be called. To resolve the ambiguity, one approach is define a method for the empty tuple:f(x::Tuple{}) = 3Alternatively, for all methods but one you can insist that there is at least one element in the tuple:f(x::NTuple{N,Int}) where {N} = 1           # this is the fallback\nf(x::Tuple{Float64, Vararg{Float64}}) = 2   # this requires at least one Float64"
},

{
    "location": "manual/methods.html#man-methods-orthogonalize-1",
    "page": "Methods",
    "title": "Orthogonalize your design",
    "category": "section",
    "text": "When you might be tempted to dispatch on two or more arguments, consider whether a \"wrapper\" function might make for a simpler design. For example, instead of writing multiple variants:f(x::A, y::A) = ...\nf(x::A, y::B) = ...\nf(x::B, y::A) = ...\nf(x::B, y::B) = ...you might consider definingf(x::A, y::A) = ...\nf(x, y) = f(g(x), g(y))where g converts the argument to type A. This is a very specific example of the more general principle of orthogonal design, in which separate concepts are assigned to separate methods. Here, g will most likely need a fallback definitiong(x::A) = xA related strategy exploits promote to bring x and y to a common type:f(x::T, y::T) where {T} = ...\nf(x, y) = f(promote(x, y)...)One risk with this design is the possibility that if there is no suitable promotion method converting x and y to the same type, the second method will recurse on itself infinitely and trigger a stack overflow. The non-exported function Base.promote_noncircular can be used as an alternative; when promotion fails it will still throw an error, but one that fails faster with a more specific error message."
},

{
    "location": "manual/methods.html#Dispatch-on-one-argument-at-a-time-1",
    "page": "Methods",
    "title": "Dispatch on one argument at a time",
    "category": "section",
    "text": "If you need to dispatch on multiple arguments, and there are many fallbacks with too many combinations to make it practical to define all possible variants, then consider introducing a \"name cascade\" where (for example) you dispatch on the first argument and then call an internal method:f(x::A, y) = _fA(x, y)\nf(x::B, y) = _fB(x, y)Then the internal methods _fA and _fB can dispatch on y without concern about ambiguities with each other with respect to x.Be aware that this strategy has at least one major disadvantage: in many cases, it is not possible for users to further customize the behavior of f by defining further specializations of your exported function f. Instead, they have to define specializations for your internal methods _fA and _fB, and this blurs the lines between exported and internal methods."
},

{
    "location": "manual/methods.html#Abstract-containers-and-element-types-1",
    "page": "Methods",
    "title": "Abstract containers and element types",
    "category": "section",
    "text": "Where possible, try to avoid defining methods that dispatch on specific element types of abstract containers. For example,-(A::AbstractArray{T}, b::Date) where {T<:Date}generates ambiguities for anyone who defines a method-(A::MyArrayType{T}, b::T) where {T}The best approach is to avoid defining either of these methods: instead, rely on a generic method -(A::AbstractArray, b) and make sure this method is implemented with generic calls (like similar and -) that do the right thing for each container type and element type separately. This is just a more complex variant of the advice to orthogonalize your methods.When this approach is not possible, it may be worth starting a discussion with other developers about resolving the ambiguity; just because one method was defined first does not necessarily mean that it can't be modified or eliminated.  As a last resort, one developer can define the \"band-aid\" method-(A::MyArrayType{T}, b::Date) where {T<:Date} = ...that resolves the ambiguity by brute force."
},

{
    "location": "manual/methods.html#Complex-method-\"cascades\"-with-default-arguments-1",
    "page": "Methods",
    "title": "Complex method \"cascades\" with default arguments",
    "category": "section",
    "text": "If you are defining a method \"cascade\" that supplies defaults, be careful about dropping any arguments that correspond to potential defaults. For example, suppose you're writing a digital filtering algorithm and you have a method that handles the edges of the signal by applying padding:function myfilter(A, kernel, ::Replicate)\n    Apadded = replicate_edges(A, size(kernel))\n    myfilter(Apadded, kernel)  # now perform the \"real\" computation\nendThis will run afoul of a method that supplies default padding:myfilter(A, kernel) = myfilter(A, kernel, Replicate()) # replicate the edge by defaultTogether, these two methods generate an infinite recursion with A constantly growing bigger.The better design would be to define your call hierarchy like this:struct NoPad end  # indicate that no padding is desired, or that it's already applied\n\nmyfilter(A, kernel) = myfilter(A, kernel, Replicate())  # default boundary conditions\n\nfunction myfilter(A, kernel, ::Replicate)\n    Apadded = replicate_edges(A, size(kernel))\n    myfilter(Apadded, kernel, NoPad())  # indicate the new boundary conditions\nend\n\n# other padding methods go here\n\nfunction myfilter(A, kernel, ::NoPad)\n    # Here's the \"real\" implementation of the core computation\nendNoPad is supplied in the same argument position as any other kind of padding, so it keeps the dispatch hierarchy well organized and with reduced likelihood of ambiguities. Moreover, it extends the \"public\" myfilter interface: a user who wants to control the padding explicitly can call the NoPad variant directly.[Clarke61]: Arthur C. Clarke, Profiles of the Future (1961): Clarke's Third Law."
},

{
    "location": "manual/constructors.html#",
    "page": "Constructors",
    "title": "Constructors",
    "category": "page",
    "text": ""
},

{
    "location": "manual/constructors.html#man-constructors-1",
    "page": "Constructors",
    "title": "Constructors",
    "category": "section",
    "text": "Constructors [1] are functions that create new objects – specifically, instances of Composite Types. In Julia, type objects also serve as constructor functions: they create new instances of themselves when applied to an argument tuple as a function. This much was already mentioned briefly when composite types were introduced. For example:julia> struct Foo\n           bar\n           baz\n       end\n\njulia> foo = Foo(1, 2)\nFoo(1, 2)\n\njulia> foo.bar\n1\n\njulia> foo.baz\n2For many types, forming new objects by binding their field values together is all that is ever needed to create instances. There are, however, cases where more functionality is required when creating composite objects. Sometimes invariants must be enforced, either by checking arguments or by transforming them. Recursive data structures, especially those that may be self-referential, often cannot be constructed cleanly without first being created in an incomplete state and then altered programmatically to be made whole, as a separate step from object creation. Sometimes, it's just convenient to be able to construct objects with fewer or different types of parameters than they have fields. Julia's system for object construction addresses all of these cases and more.[1]: Nomenclature: while the term \"constructor\" generally refers to the entire function which constructs objects of a type, it is common to abuse terminology slightly and refer to specific constructor methods as \"constructors\". In such situations, it is generally clear from context that the term is used to mean \"constructor method\" rather than \"constructor function\", especially as it is often used in the sense of singling out a particular method of the constructor from all of the others."
},

{
    "location": "manual/constructors.html#Outer-Constructor-Methods-1",
    "page": "Constructors",
    "title": "Outer Constructor Methods",
    "category": "section",
    "text": "A constructor is just like any other function in Julia in that its overall behavior is defined by the combined behavior of its methods. Accordingly, you can add functionality to a constructor by simply defining new methods. For example, let's say you want to add a constructor method for Foo objects that takes only one argument and uses the given value for both the bar and baz fields. This is simple:julia> Foo(x) = Foo(x,x)\nFoo\n\njulia> Foo(1)\nFoo(1, 1)You could also add a zero-argument Foo constructor method that supplies default values for both of the bar and baz fields:julia> Foo() = Foo(0)\nFoo\n\njulia> Foo()\nFoo(0, 0)Here the zero-argument constructor method calls the single-argument constructor method, which in turn calls the automatically provided two-argument constructor method. For reasons that will become clear very shortly, additional constructor methods declared as normal methods like this are called outer constructor methods. Outer constructor methods can only ever create a new instance by calling another constructor method, such as the automatically provided default ones."
},

{
    "location": "manual/constructors.html#Inner-Constructor-Methods-1",
    "page": "Constructors",
    "title": "Inner Constructor Methods",
    "category": "section",
    "text": "While outer constructor methods succeed in addressing the problem of providing additional convenience methods for constructing objects, they fail to address the other two use cases mentioned in the introduction of this chapter: enforcing invariants, and allowing construction of self-referential objects. For these problems, one needs inner constructor methods. An inner constructor method is much like an outer constructor method, with two differences:It is declared inside the block of a type declaration, rather than outside of it like normal methods.\nIt has access to a special locally existent function called new that creates objects of the block's type.For example, suppose one wants to declare a type that holds a pair of real numbers, subject to the constraint that the first number is not greater than the second one. One could declare it like this:julia> struct OrderedPair\n           x::Real\n           y::Real\n           OrderedPair(x,y) = x > y ? error(\"out of order\") : new(x,y)\n       end\nNow OrderedPair objects can only be constructed such that x <= y:julia> OrderedPair(1, 2)\nOrderedPair(1, 2)\n\njulia> OrderedPair(2,1)\nERROR: out of order\nStacktrace:\n [1] OrderedPair(::Int64, ::Int64) at ./none:4If the type were declared mutable, you could reach in and directly change the field values to violate this invariant, but messing around with an object's internals uninvited is considered poor form. You (or someone else) can also provide additional outer constructor methods at any later point, but once a type is declared, there is no way to add more inner constructor methods. Since outer constructor methods can only create objects by calling other constructor methods, ultimately, some inner constructor must be called to create an object. This guarantees that all objects of the declared type must come into existence by a call to one of the inner constructor methods provided with the type, thereby giving some degree of enforcement of a type's invariants.If any inner constructor method is defined, no default constructor method is provided: it is presumed that you have supplied yourself with all the inner constructors you need. The default constructor is equivalent to writing your own inner constructor method that takes all of the object's fields as parameters (constrained to be of the correct type, if the corresponding field has a type), and passes them to new, returning the resulting object:julia> struct Foo\n           bar\n           baz\n           Foo(bar,baz) = new(bar,baz)\n       end\nThis declaration has the same effect as the earlier definition of the Foo type without an explicit inner constructor method. The following two types are equivalent – one with a default constructor, the other with an explicit constructor:julia> struct T1\n           x::Int64\n       end\n\njulia> struct T2\n           x::Int64\n           T2(x) = new(x)\n       end\n\njulia> T1(1)\nT1(1)\n\njulia> T2(1)\nT2(1)\n\njulia> T1(1.0)\nT1(1)\n\njulia> T2(1.0)\nT2(1)It is considered good form to provide as few inner constructor methods as possible: only those taking all arguments explicitly and enforcing essential error checking and transformation. Additional convenience constructor methods, supplying default values or auxiliary transformations, should be provided as outer constructors that call the inner constructors to do the heavy lifting. This separation is typically quite natural."
},

{
    "location": "manual/constructors.html#Incomplete-Initialization-1",
    "page": "Constructors",
    "title": "Incomplete Initialization",
    "category": "section",
    "text": "The final problem which has still not been addressed is construction of self-referential objects, or more generally, recursive data structures. Since the fundamental difficulty may not be immediately obvious, let us briefly explain it. Consider the following recursive type declaration:julia> mutable struct SelfReferential\n           obj::SelfReferential\n       end\nThis type may appear innocuous enough, until one considers how to construct an instance of it. If a is an instance of SelfReferential, then a second instance can be created by the call:julia> b = SelfReferential(a)But how does one construct the first instance when no instance exists to provide as a valid value for its obj field? The only solution is to allow creating an incompletely initialized instance of SelfReferential with an unassigned obj field, and using that incomplete instance as a valid value for the obj field of another instance, such as, for example, itself.To allow for the creation of incompletely initialized objects, Julia allows the new function to be called with fewer than the number of fields that the type has, returning an object with the unspecified fields uninitialized. The inner constructor method can then use the incomplete object, finishing its initialization before returning it. Here, for example, we take another crack at defining the SelfReferential type, with a zero-argument inner constructor returning instances having obj fields pointing to themselves:julia> mutable struct SelfReferential\n           obj::SelfReferential\n           SelfReferential() = (x = new(); x.obj = x)\n       end\nWe can verify that this constructor works and constructs objects that are, in fact, self-referential:julia> x = SelfReferential();\n\njulia> x === x\ntrue\n\njulia> x === x.obj\ntrue\n\njulia> x === x.obj.obj\ntrueAlthough it is generally a good idea to return a fully initialized object from an inner constructor, incompletely initialized objects can be returned:julia> mutable struct Incomplete\n           xx\n           Incomplete() = new()\n       end\n\njulia> z = Incomplete();While you are allowed to create objects with uninitialized fields, any access to an uninitialized reference is an immediate error:julia> z.xx\nERROR: UndefRefError: access to undefined referenceThis avoids the need to continually check for null values. However, not all object fields are references. Julia considers some types to be \"plain data\", meaning all of their data is self-contained and does not reference other objects. The plain data types consist of primitive types (e.g. Int) and immutable structs of other plain data types. The initial contents of a plain data type is undefined:julia> struct HasPlain\n           n::Int\n           HasPlain() = new()\n       end\n\njulia> HasPlain()\nHasPlain(438103441441)Arrays of plain data types exhibit the same behavior.You can pass incomplete objects to other functions from inner constructors to delegate their completion:julia> mutable struct Lazy\n           xx\n           Lazy(v) = complete_me(new(), v)\n       endAs with incomplete objects returned from constructors, if complete_me or any of its callees try to access the xx field of the Lazy object before it has been initialized, an error will be thrown immediately."
},

{
    "location": "manual/constructors.html#Parametric-Constructors-1",
    "page": "Constructors",
    "title": "Parametric Constructors",
    "category": "section",
    "text": "Parametric types add a few wrinkles to the constructor story. Recall from Parametric Types that, by default, instances of parametric composite types can be constructed either with explicitly given type parameters or with type parameters implied by the types of the arguments given to the constructor. Here are some examples:julia> struct Point{T<:Real}\n           x::T\n           y::T\n       end\n\njulia> Point(1,2) ## implicit T ##\nPoint{Int64}(1, 2)\n\njulia> Point(1.0,2.5) ## implicit T ##\nPoint{Float64}(1.0, 2.5)\n\njulia> Point(1,2.5) ## implicit T ##\nERROR: MethodError: no method matching Point(::Int64, ::Float64)\nClosest candidates are:\n  Point(::T<:Real, !Matched::T<:Real) where T<:Real at none:2\n\njulia> Point{Int64}(1, 2) ## explicit T ##\nPoint{Int64}(1, 2)\n\njulia> Point{Int64}(1.0,2.5) ## explicit T ##\nERROR: InexactError()\nStacktrace:\n [1] convert(::Type{Int64}, ::Float64) at ./float.jl:679\n [2] Point{Int64}(::Float64, ::Float64) at ./none:2\n\njulia> Point{Float64}(1.0, 2.5) ## explicit T ##\nPoint{Float64}(1.0, 2.5)\n\njulia> Point{Float64}(1,2) ## explicit T ##\nPoint{Float64}(1.0, 2.0)As you can see, for constructor calls with explicit type parameters, the arguments are converted to the implied field types: Point{Int64}(1,2) works, but Point{Int64}(1.0,2.5) raises an InexactError when converting 2.5 to Int64. When the type is implied by the arguments to the constructor call, as in Point(1,2), then the types of the arguments must agree – otherwise the T cannot be determined – but any pair of real arguments with matching type may be given to the generic Point constructor.What's really going on here is that Point, Point{Float64} and Point{Int64} are all different constructor functions. In fact, Point{T} is a distinct constructor function for each type T. Without any explicitly provided inner constructors, the declaration of the composite type Point{T<:Real} automatically provides an inner constructor, Point{T}, for each possible type T<:Real, that behaves just like non-parametric default inner constructors do. It also provides a single general outer Point constructor that takes pairs of real arguments, which must be of the same type. This automatic provision of constructors is equivalent to the following explicit declaration:julia> struct Point{T<:Real}\n           x::T\n           y::T\n           Point{T}(x,y) where {T<:Real} = new(x,y)\n       end\n\njulia> Point(x::T, y::T) where {T<:Real} = Point{T}(x,y);Notice that each definition looks like the form of constructor call that it handles. The call Point{Int64}(1,2) will invoke the definition Point{T}(x,y) inside the type block. The outer constructor declaration, on the other hand, defines a method for the general Point constructor which only applies to pairs of values of the same real type. This declaration makes constructor calls without explicit type parameters, like Point(1,2) and Point(1.0,2.5), work. Since the method declaration restricts the arguments to being of the same type, calls like Point(1,2.5), with arguments of different types, result in \"no method\" errors.Suppose we wanted to make the constructor call Point(1,2.5) work by \"promoting\" the integer value 1 to the floating-point value 1.0. The simplest way to achieve this is to define the following additional outer constructor method:julia> Point(x::Int64, y::Float64) = Point(convert(Float64,x),y);This method uses the convert() function to explicitly convert x to Float64 and then delegates construction to the general constructor for the case where both arguments are Float64. With this method definition what was previously a MethodError now successfully creates a point of type Point{Float64}:julia> Point(1,2.5)\nPoint{Float64}(1.0, 2.5)\n\njulia> typeof(ans)\nPoint{Float64}However, other similar calls still don't work:julia> Point(1.5,2)\nERROR: MethodError: no method matching Point(::Float64, ::Int64)\nClosest candidates are:\n  Point(::T<:Real, !Matched::T<:Real) where T<:Real at none:1For a more general way to make all such calls work sensibly, see Conversion and Promotion. At the risk of spoiling the suspense, we can reveal here that all it takes is the following outer method definition to make all calls to the general Point constructor work as one would expect:julia> Point(x::Real, y::Real) = Point(promote(x,y)...);The promote function converts all its arguments to a common type – in this case Float64. With this method definition, the Point constructor promotes its arguments the same way that numeric operators like + do, and works for all kinds of real numbers:julia> Point(1.5,2)\nPoint{Float64}(1.5, 2.0)\n\njulia> Point(1,1//2)\nPoint{Rational{Int64}}(1//1, 1//2)\n\njulia> Point(1.0,1//2)\nPoint{Float64}(1.0, 0.5)Thus, while the implicit type parameter constructors provided by default in Julia are fairly strict, it is possible to make them behave in a more relaxed but sensible manner quite easily. Moreover, since constructors can leverage all of the power of the type system, methods, and multiple dispatch, defining sophisticated behavior is typically quite simple."
},

{
    "location": "manual/constructors.html#Case-Study:-Rational-1",
    "page": "Constructors",
    "title": "Case Study: Rational",
    "category": "section",
    "text": "Perhaps the best way to tie all these pieces together is to present a real world example of a parametric composite type and its constructor methods. To that end, here is the (slightly modified) beginning of rational.jl, which implements Julia's Rational Numbers:julia> struct OurRational{T<:Integer} <: Real\n           num::T\n           den::T\n           function OurRational{T}(num::T, den::T) where T<:Integer\n               if num == 0 && den == 0\n                    error(\"invalid rational: 0//0\")\n               end\n               g = gcd(den, num)\n               num = div(num, g)\n               den = div(den, g)\n               new(num, den)\n           end\n       end\n\njulia> OurRational(n::T, d::T) where {T<:Integer} = OurRational{T}(n,d)\nOurRational\n\njulia> OurRational(n::Integer, d::Integer) = OurRational(promote(n,d)...)\nOurRational\n\njulia> OurRational(n::Integer) = OurRational(n,one(n))\nOurRational\n\njulia> //(n::Integer, d::Integer) = OurRational(n,d)\n// (generic function with 1 method)\n\njulia> //(x::OurRational, y::Integer) = x.num // (x.den*y)\n// (generic function with 2 methods)\n\njulia> //(x::Integer, y::OurRational) = (x*y.den) // y.num\n// (generic function with 3 methods)\n\njulia> //(x::Complex, y::Real) = complex(real(x)//y, imag(x)//y)\n// (generic function with 4 methods)\n\njulia> //(x::Real, y::Complex) = x*y'//real(y*y')\n// (generic function with 5 methods)\n\njulia> function //(x::Complex, y::Complex)\n           xy = x*y'\n           yy = real(y*y')\n           complex(real(xy)//yy, imag(xy)//yy)\n       end\n// (generic function with 6 methods)The first line – struct OurRational{T<:Integer} <: Real – declares that OurRational takes one type parameter of an integer type, and is itself a real type. The field declarations num::T and den::T indicate that the data held in a OurRational{T} object are a pair of integers of type T, one representing the rational value's numerator and the other representing its denominator.Now things get interesting. OurRational has a single inner constructor method which checks that both of num and den aren't zero and ensures that every rational is constructed in \"lowest terms\" with a non-negative denominator. This is accomplished by dividing the given numerator and denominator values by their greatest common divisor, computed using the gcd function. Since gcd returns the greatest common divisor of its arguments with sign matching the first argument (den here), after this division the new value of den is guaranteed to be non-negative. Because this is the only inner constructor for OurRational, we can be certain that OurRational objects are always constructed in this normalized form.OurRational also provides several outer constructor methods for convenience. The first is the \"standard\" general constructor that infers the type parameter T from the type of the numerator and denominator when they have the same type. The second applies when the given numerator and denominator values have different types: it promotes them to a common type and then delegates construction to the outer constructor for arguments of matching type. The third outer constructor turns integer values into rationals by supplying a value of 1 as the denominator.Following the outer constructor definitions, we have a number of methods for the // operator, which provides a syntax for writing rationals. Before these definitions, // is a completely undefined operator with only syntax and no meaning. Afterwards, it behaves just as described in Rational Numbers – its entire behavior is defined in these few lines. The first and most basic definition just makes a//b construct a OurRational by applying the OurRational constructor to a and b when they are integers. When one of the operands of // is already a rational number, we construct a new rational for the resulting ratio slightly differently; this behavior is actually identical to division of a rational with an integer. Finally, applying // to complex integral values creates an instance of Complex{OurRational} – a complex number whose real and imaginary parts are rationals:julia> ans = (1 + 2im)//(1 - 2im);\n\njulia> typeof(ans)\nComplex{OurRational{Int64}}\n\njulia> ans <: Complex{OurRational}\nfalseThus, although the // operator usually returns an instance of OurRational, if either of its arguments are complex integers, it will return an instance of Complex{OurRational} instead. The interested reader should consider perusing the rest of rational.jl: it is short, self-contained, and implements an entire basic Julia type."
},

{
    "location": "manual/constructors.html#constructors-and-conversion-1",
    "page": "Constructors",
    "title": "Constructors and Conversion",
    "category": "section",
    "text": "Constructors T(args...) in Julia are implemented like other callable objects: methods are added to their types. The type of a type is Type, so all constructor methods are stored in the method table for the Type type. This means that you can declare more flexible constructors, e.g. constructors for abstract types, by explicitly defining methods for the appropriate types.However, in some cases you could consider adding methods to Base.convert instead of defining a constructor, because Julia falls back to calling convert() if no matching constructor is found. For example, if no constructor T(args...) = ... exists Base.convert(::Type{T}, args...) = ... is called.convert is used extensively throughout Julia whenever one type needs to be converted to another (e.g. in assignment, ccall, etcetera), and should generally only be defined (or successful) if the conversion is lossless.  For example, convert(Int, 3.0) produces 3, but convert(Int, 3.2) throws an InexactError.  If you want to define a constructor for a lossless conversion from one type to another, you should probably define a convert method instead.On the other hand, if your constructor does not represent a lossless conversion, or doesn't represent \"conversion\" at all, it is better to leave it as a constructor rather than a convert method. For example, the Array{Int}() constructor creates a zero-dimensional Array of the type Int, but is not really a \"conversion\" from Int to an Array."
},

{
    "location": "manual/constructors.html#Outer-only-constructors-1",
    "page": "Constructors",
    "title": "Outer-only constructors",
    "category": "section",
    "text": "As we have seen, a typical parametric type has inner constructors that are called when type parameters are known; e.g. they apply to Point{Int} but not to Point. Optionally, outer constructors that determine type parameters automatically can be added, for example constructing a Point{Int} from the call Point(1,2). Outer constructors call inner constructors to do the core work of making an instance. However, in some cases one would rather not provide inner constructors, so that specific type parameters cannot be requested manually.For example, say we define a type that stores a vector along with an accurate representation of its sum:julia> struct SummedArray{T<:Number,S<:Number}\n           data::Vector{T}\n           sum::S\n       end\n\njulia> SummedArray(Int32[1; 2; 3], Int32(6))\nSummedArray{Int32,Int32}(Int32[1, 2, 3], 6)The problem is that we want S to be a larger type than T, so that we can sum many elements with less information loss. For example, when T is Int32, we would like S to be Int64. Therefore we want to avoid an interface that allows the user to construct instances of the type SummedArray{Int32,Int32}. One way to do this is to provide a constructor only for SummedArray, but inside the type definition block to suppress generation of default constructors:julia> struct SummedArray{T<:Number,S<:Number}\n           data::Vector{T}\n           sum::S\n           function SummedArray(a::Vector{T}) where T\n               S = widen(T)\n               new{T,S}(a, sum(S, a))\n           end\n       end\n\njulia> SummedArray(Int32[1; 2; 3], Int32(6))\nERROR: MethodError: no method matching SummedArray(::Array{Int32,1}, ::Int32)\nClosest candidates are:\n  SummedArray(::Array{T,1}) where T at none:5This constructor will be invoked by the syntax SummedArray(a). The syntax new{T,S} allows specifying parameters for the type to be constructed, i.e. this call will return a SummedArray{T,S}. new{T,S} can be used in any constructor definition, but for convenience the parameters to new{} are automatically derived from the type being constructed when possible."
},

{
    "location": "manual/conversion-and-promotion.html#",
    "page": "Conversion and Promotion",
    "title": "Conversion and Promotion",
    "category": "page",
    "text": ""
},

{
    "location": "manual/conversion-and-promotion.html#conversion-and-promotion-1",
    "page": "Conversion and Promotion",
    "title": "Conversion and Promotion",
    "category": "section",
    "text": "Julia has a system for promoting arguments of mathematical operators to a common type, which has been mentioned in various other sections, including Integers and Floating-Point Numbers, Mathematical Operations and Elementary Functions, Types, and Methods. In this section, we explain how this promotion system works, as well as how to extend it to new types and apply it to functions besides built-in mathematical operators. Traditionally, programming languages fall into two camps with respect to promotion of arithmetic arguments:Automatic promotion for built-in arithmetic types and operators. In most languages, built-in numeric types, when used as operands to arithmetic operators with infix syntax, such as +, -, *, and /, are automatically promoted to a common type to produce the expected results. C, Java, Perl, and Python, to name a few, all correctly compute the sum 1 + 1.5 as the floating-point value 2.5, even though one of the operands to + is an integer. These systems are convenient and designed carefully enough that they are generally all-but-invisible to the programmer: hardly anyone consciously thinks of this promotion taking place when writing such an expression, but compilers and interpreters must perform conversion before addition since integers and floating-point values cannot be added as-is. Complex rules for such automatic conversions are thus inevitably part of specifications and implementations for such languages.\nNo automatic promotion. This camp includes Ada and ML – very \"strict\" statically typed languages. In these languages, every conversion must be explicitly specified by the programmer. Thus, the example expression 1 + 1.5 would be a compilation error in both Ada and ML. Instead one must write real(1) + 1.5, explicitly converting the integer 1 to a floating-point value before performing addition. Explicit conversion everywhere is so inconvenient, however, that even Ada has some degree of automatic conversion: integer literals are promoted to the expected integer type automatically, and floating-point literals are similarly promoted to appropriate floating-point types.In a sense, Julia falls into the \"no automatic promotion\" category: mathematical operators are just functions with special syntax, and the arguments of functions are never automatically converted. However, one may observe that applying mathematical operations to a wide variety of mixed argument types is just an extreme case of polymorphic multiple dispatch – something which Julia's dispatch and type systems are particularly well-suited to handle. \"Automatic\" promotion of mathematical operands simply emerges as a special application: Julia comes with pre-defined catch-all dispatch rules for mathematical operators, invoked when no specific implementation exists for some combination of operand types. These catch-all rules first promote all operands to a common type using user-definable promotion rules, and then invoke a specialized implementation of the operator in question for the resulting values, now of the same type. User-defined types can easily participate in this promotion system by defining methods for conversion to and from other types, and providing a handful of promotion rules defining what types they should promote to when mixed with other types."
},

{
    "location": "manual/conversion-and-promotion.html#Conversion-1",
    "page": "Conversion and Promotion",
    "title": "Conversion",
    "category": "section",
    "text": "Conversion of values to various types is performed by the convert function. The convert function generally takes two arguments: the first is a type object while the second is a value to convert to that type; the returned value is the value converted to an instance of given type. The simplest way to understand this function is to see it in action:julia> x = 12\n12\n\njulia> typeof(x)\nInt64\n\njulia> convert(UInt8, x)\n0x0c\n\njulia> typeof(ans)\nUInt8\n\njulia> convert(AbstractFloat, x)\n12.0\n\njulia> typeof(ans)\nFloat64\n\njulia> a = Any[1 2 3; 4 5 6]\n2×3 Array{Any,2}:\n 1  2  3\n 4  5  6\n\njulia> convert(Array{Float64}, a)\n2×3 Array{Float64,2}:\n 1.0  2.0  3.0\n 4.0  5.0  6.0Conversion isn't always possible, in which case a no method error is thrown indicating that convert doesn't know how to perform the requested conversion:julia> convert(AbstractFloat, \"foo\")\nERROR: MethodError: Cannot `convert` an object of type String to an object of type AbstractFloat\nThis may have arisen from a call to the constructor AbstractFloat(...),\nsince type constructors fall back to convert methods.Some languages consider parsing strings as numbers or formatting numbers as strings to be conversions (many dynamic languages will even perform conversion for you automatically), however Julia does not: even though some strings can be parsed as numbers, most strings are not valid representations of numbers, and only a very limited subset of them are. Therefore in Julia the dedicated parse() function must be used to perform this operation, making it more explicit."
},

{
    "location": "manual/conversion-and-promotion.html#Defining-New-Conversions-1",
    "page": "Conversion and Promotion",
    "title": "Defining New Conversions",
    "category": "section",
    "text": "To define a new conversion, simply provide a new method for convert(). That's really all there is to it. For example, the method to convert a real number to a boolean is this:convert(::Type{Bool}, x::Real) = x==0 ? false : x==1 ? true : throw(InexactError())The type of the first argument of this method is a singleton type, Type{Bool}, the only instance of which is Bool. Thus, this method is only invoked when the first argument is the type value Bool. Notice the syntax used for the first argument: the argument name is omitted prior to the :: symbol, and only the type is given. This is the syntax in Julia for a function argument whose type is specified but whose value is never used in the function body. In this example, since the type is a singleton, there would never be any reason to use its value within the body. When invoked, the method determines whether a numeric value is true or false as a boolean, by comparing it to one and zero:julia> convert(Bool, 1)\ntrue\n\njulia> convert(Bool, 0)\nfalse\n\njulia> convert(Bool, 1im)\nERROR: InexactError()\nStacktrace:\n [1] convert(::Type{Bool}, ::Complex{Int64}) at ./complex.jl:31\n\njulia> convert(Bool, 0im)\nfalseThe method signatures for conversion methods are often quite a bit more involved than this example, especially for parametric types. The example above is meant to be pedagogical, and is not the actual Julia behaviour. This is the actual implementation in Julia:convert(::Type{T}, z::Complex) where {T<:Real} =\n    (imag(z) == 0 ? convert(T, real(z)) : throw(InexactError()))"
},

{
    "location": "manual/conversion-and-promotion.html#man-rational-conversion-1",
    "page": "Conversion and Promotion",
    "title": "Case Study: Rational Conversions",
    "category": "section",
    "text": "To continue our case study of Julia's Rational type, here are the conversions declared in rational.jl, right after the declaration of the type and its constructors:convert(::Type{Rational{T}}, x::Rational) where {T<:Integer} = Rational(convert(T,x.num),convert(T,x.den))\nconvert(::Type{Rational{T}}, x::Integer) where {T<:Integer} = Rational(convert(T,x), convert(T,1))\n\nfunction convert(::Type{Rational{T}}, x::AbstractFloat, tol::Real) where T<:Integer\n    if isnan(x); return zero(T)//zero(T); end\n    if isinf(x); return sign(x)//zero(T); end\n    y = x\n    a = d = one(T)\n    b = c = zero(T)\n    while true\n        f = convert(T,round(y)); y -= f\n        a, b, c, d = f*a+c, f*b+d, a, b\n        if y == 0 || abs(a/b-x) <= tol\n            return a//b\n        end\n        y = 1/y\n    end\nend\nconvert(rt::Type{Rational{T}}, x::AbstractFloat) where {T<:Integer} = convert(rt,x,eps(x))\n\nconvert(::Type{T}, x::Rational) where {T<:AbstractFloat} = convert(T,x.num)/convert(T,x.den)\nconvert(::Type{T}, x::Rational) where {T<:Integer} = div(convert(T,x.num),convert(T,x.den))The initial four convert methods provide conversions to rational types. The first method converts one type of rational to another type of rational by converting the numerator and denominator to the appropriate integer type. The second method does the same conversion for integers by taking the denominator to be 1. The third method implements a standard algorithm for approximating a floating-point number by a ratio of integers to within a given tolerance, and the fourth method applies it, using machine epsilon at the given value as the threshold. In general, one should have a//b == convert(Rational{Int64}, a/b).The last two convert methods provide conversions from rational types to floating-point and integer types. To convert to floating point, one simply converts both numerator and denominator to that floating point type and then divides. To convert to integer, one can use the div operator for truncated integer division (rounded towards zero)."
},

{
    "location": "manual/conversion-and-promotion.html#Promotion-1",
    "page": "Conversion and Promotion",
    "title": "Promotion",
    "category": "section",
    "text": "Promotion refers to converting values of mixed types to a single common type. Although it is not strictly necessary, it is generally implied that the common type to which the values are converted can faithfully represent all of the original values. In this sense, the term \"promotion\" is appropriate since the values are converted to a \"greater\" type – i.e. one which can represent all of the input values in a single common type. It is important, however, not to confuse this with object-oriented (structural) super-typing, or Julia's notion of abstract super-types: promotion has nothing to do with the type hierarchy, and everything to do with converting between alternate representations. For instance, although every Int32 value can also be represented as a Float64 value, Int32 is not a subtype of Float64.Promotion to a common \"greater\" type is performed in Julia by the promote function, which takes any number of arguments, and returns a tuple of the same number of values, converted to a common type, or throws an exception if promotion is not possible. The most common use case for promotion is to convert numeric arguments to a common type:julia> promote(1, 2.5)\n(1.0, 2.5)\n\njulia> promote(1, 2.5, 3)\n(1.0, 2.5, 3.0)\n\njulia> promote(2, 3//4)\n(2//1, 3//4)\n\njulia> promote(1, 2.5, 3, 3//4)\n(1.0, 2.5, 3.0, 0.75)\n\njulia> promote(1.5, im)\n(1.5 + 0.0im, 0.0 + 1.0im)\n\njulia> promote(1 + 2im, 3//4)\n(1//1 + 2//1*im, 3//4 + 0//1*im)Floating-point values are promoted to the largest of the floating-point argument types. Integer values are promoted to the larger of either the native machine word size or the largest integer argument type. Mixtures of integers and floating-point values are promoted to a floating-point type big enough to hold all the values. Integers mixed with rationals are promoted to rationals. Rationals mixed with floats are promoted to floats. Complex values mixed with real values are promoted to the appropriate kind of complex value.That is really all there is to using promotions. The rest is just a matter of clever application, the most typical \"clever\" application being the definition of catch-all methods for numeric operations like the arithmetic operators +, -, * and /. Here are some of the catch-all method definitions given in promotion.jl:+(x::Number, y::Number) = +(promote(x,y)...)\n-(x::Number, y::Number) = -(promote(x,y)...)\n*(x::Number, y::Number) = *(promote(x,y)...)\n/(x::Number, y::Number) = /(promote(x,y)...)These method definitions say that in the absence of more specific rules for adding, subtracting, multiplying and dividing pairs of numeric values, promote the values to a common type and then try again. That's all there is to it: nowhere else does one ever need to worry about promotion to a common numeric type for arithmetic operations – it just happens automatically. There are definitions of catch-all promotion methods for a number of other arithmetic and mathematical functions in promotion.jl, but beyond that, there are hardly any calls to promote required in the Julia standard library. The most common usages of promote occur in outer constructors methods, provided for convenience, to allow constructor calls with mixed types to delegate to an inner type with fields promoted to an appropriate common type. For example, recall that rational.jl provides the following outer constructor method:Rational(n::Integer, d::Integer) = Rational(promote(n,d)...)This allows calls like the following to work:julia> Rational(Int8(15),Int32(-5))\n-3//1\n\njulia> typeof(ans)\nRational{Int32}For most user-defined types, it is better practice to require programmers to supply the expected types to constructor functions explicitly, but sometimes, especially for numeric problems, it can be convenient to do promotion automatically."
},

{
    "location": "manual/conversion-and-promotion.html#Defining-Promotion-Rules-1",
    "page": "Conversion and Promotion",
    "title": "Defining Promotion Rules",
    "category": "section",
    "text": "Although one could, in principle, define methods for the promote function directly, this would require many redundant definitions for all possible permutations of argument types. Instead, the behavior of promote is defined in terms of an auxiliary function called promote_rule, which one can provide methods for. The promote_rule function takes a pair of type objects and returns another type object, such that instances of the argument types will be promoted to the returned type. Thus, by defining the rule:promote_rule(::Type{Float64}, ::Type{Float32}) = Float64one declares that when 64-bit and 32-bit floating-point values are promoted together, they should be promoted to 64-bit floating-point. The promotion type does not need to be one of the argument types, however; the following promotion rules both occur in Julia's standard library:promote_rule(::Type{UInt8}, ::Type{Int8}) = Int\npromote_rule(::Type{BigInt}, ::Type{Int8}) = BigIntIn the latter case, the result type is BigInt since BigInt is the only type large enough to hold integers for arbitrary-precision integer arithmetic. Also note that one does not need to define both promote_rule(::Type{A}, ::Type{B}) and promote_rule(::Type{B}, ::Type{A}) – the symmetry is implied by the way promote_rule is used in the promotion process.The promote_rule function is used as a building block to define a second function called promote_type, which, given any number of type objects, returns the common type to which those values, as arguments to promote should be promoted. Thus, if one wants to know, in absence of actual values, what type a collection of values of certain types would promote to, one can use promote_type:julia> promote_type(Int8, UInt16)\nInt64Internally, promote_type is used inside of promote to determine what type argument values should be converted to for promotion. It can, however, be useful in its own right. The curious reader can read the code in promotion.jl, which defines the complete promotion mechanism in about 35 lines."
},

{
    "location": "manual/conversion-and-promotion.html#Case-Study:-Rational-Promotions-1",
    "page": "Conversion and Promotion",
    "title": "Case Study: Rational Promotions",
    "category": "section",
    "text": "Finally, we finish off our ongoing case study of Julia's rational number type, which makes relatively sophisticated use of the promotion mechanism with the following promotion rules:promote_rule(::Type{Rational{T}}, ::Type{S}) where {T<:Integer,S<:Integer} = Rational{promote_type(T,S)}\npromote_rule(::Type{Rational{T}}, ::Type{Rational{S}}) where {T<:Integer,S<:Integer} = Rational{promote_type(T,S)}\npromote_rule(::Type{Rational{T}}, ::Type{S}) where {T<:Integer,S<:AbstractFloat} = promote_type(T,S)The first rule says that promoting a rational number with any other integer type promotes to a rational type whose numerator/denominator type is the result of promotion of its numerator/denominator type with the other integer type. The second rule applies the same logic to two different types of rational numbers, resulting in a rational of the promotion of their respective numerator/denominator types. The third and final rule dictates that promoting a rational with a float results in the same type as promoting the numerator/denominator type with the float.This small handful of promotion rules, together with the conversion methods discussed above, are sufficient to make rational numbers interoperate completely naturally with all of Julia's other numeric types – integers, floating-point numbers, and complex numbers. By providing appropriate conversion methods and promotion rules in the same manner, any user-defined numeric type can interoperate just as naturally with Julia's predefined numerics."
},

{
    "location": "manual/interfaces.html#",
    "page": "Interfaces",
    "title": "Interfaces",
    "category": "page",
    "text": ""
},

{
    "location": "manual/interfaces.html#Interfaces-1",
    "page": "Interfaces",
    "title": "Interfaces",
    "category": "section",
    "text": "A lot of the power and extensibility in Julia comes from a collection of informal interfaces.  By extending a few specific methods to work for a custom type, objects of that type not only receive those functionalities, but they are also able to be used in other methods that are written to generically build upon those behaviors."
},

{
    "location": "manual/interfaces.html#man-interface-iteration-1",
    "page": "Interfaces",
    "title": "Iteration",
    "category": "section",
    "text": "Required methods   Brief description\nstart(iter)   Returns the initial iteration state\nnext(iter, state)   Returns the current item and the next state\ndone(iter, state)   Tests if there are any items remaining\nImportant optional methods Default definition Brief description\niteratorsize(IterType) HasLength() One of HasLength(), HasShape(), IsInfinite(), or SizeUnknown() as appropriate\niteratoreltype(IterType) HasEltype() Either EltypeUnknown() or HasEltype() as appropriate\neltype(IterType) Any The type the items returned by next()\nlength(iter) (undefined) The number of items, if known\nsize(iter, [dim...]) (undefined) The number of items in each dimension, if knownValue returned by iteratorsize(IterType) Required Methods\nHasLength() length(iter)\nHasShape() length(iter)  and size(iter, [dim...])\nIsInfinite() (none)\nSizeUnknown() (none)Value returned by iteratoreltype(IterType) Required Methods\nHasEltype() eltype(IterType)\nEltypeUnknown() (none)Sequential iteration is implemented by the methods start(), done(), and next(). Instead of mutating objects as they are iterated over, Julia provides these three methods to keep track of the iteration state externally from the object. The start(iter) method returns the initial state for the iterable object iter. That state gets passed along to done(iter, state), which tests if there are any elements remaining, and next(iter, state), which returns a tuple containing the current element and an updated state. The state object can be anything, and is generally considered to be an implementation detail private to the iterable object.Any object defines these three methods is iterable and can be used in the many functions that rely upon iteration. It can also be used directly in a for loop since the syntax:for i in iter   # or  \"for i = iter\"\n    # body\nendis translated into:state = start(iter)\nwhile !done(iter, state)\n    (i, state) = next(iter, state)\n    # body\nendA simple example is an iterable sequence of square numbers with a defined length:julia> struct Squares\n           count::Int\n       end\n\njulia> Base.start(::Squares) = 1\n\njulia> Base.next(S::Squares, state) = (state*state, state+1)\n\njulia> Base.done(S::Squares, state) = state > S.count\n\njulia> Base.eltype(::Type{Squares}) = Int # Note that this is defined for the type\n\njulia> Base.length(S::Squares) = S.countWith only start, next, and done definitions, the Squares type is already pretty powerful. We can iterate over all the elements:julia> for i in Squares(7)\n           println(i)\n       end\n1\n4\n9\n16\n25\n36\n49We can use many of the builtin methods that work with iterables, like in(), mean() and std():julia> 25 in Squares(10)\ntrue\n\njulia> mean(Squares(100))\n3383.5\n\njulia> std(Squares(100))\n3024.355854282583There are a few more methods we can extend to give Julia more information about this iterable collection.  We know that the elements in a Squares sequence will always be Int. By extending the eltype() method, we can give that information to Julia and help it make more specialized code in the more complicated methods. We also know the number of elements in our sequence, so we can extend length(), too.Now, when we ask Julia to collect() all the elements into an array it can preallocate a Vector{Int} of the right size instead of blindly push!ing each element into a Vector{Any}:julia> collect(Squares(10))' # transposed to save space\n1×10 RowVector{Int64,Array{Int64,1}}:\n 1  4  9  16  25  36  49  64  81  100While we can rely upon generic implementations, we can also extend specific methods where we know there is a simpler algorithm. For example, there's a formula to compute the sum of squares, so we can override the generic iterative version with a more performant solution:julia> Base.sum(S::Squares) = (n = S.count; return n*(n+1)*(2n+1)÷6)\n\njulia> sum(Squares(1803))\n1955361914This is a very common pattern throughout the Julia standard library: a small set of required methods define an informal interface that enable many fancier behaviors. In some cases, types will want to additionally specialize those extra behaviors when they know a more efficient algorithm can be used in their specific case."
},

{
    "location": "manual/interfaces.html#Indexing-1",
    "page": "Interfaces",
    "title": "Indexing",
    "category": "section",
    "text": "Methods to implement Brief description\ngetindex(X, i) X[i], indexed element access\nsetindex!(X, v, i) X[i] = v, indexed assignment\nendof(X) The last index, used in X[end]For the Squares iterable above, we can easily compute the ith element of the sequence by squaring it.  We can expose this as an indexing expression S[i]. To opt into this behavior, Squares simply needs to define getindex():julia> function Base.getindex(S::Squares, i::Int)\n           1 <= i <= S.count || throw(BoundsError(S, i))\n           return i*i\n       end\n\njulia> Squares(100)[23]\n529Additionally, to support the syntax S[end], we must define endof() to specify the last valid index:julia> Base.endof(S::Squares) = length(S)\n\njulia> Squares(23)[end]\n529Note, though, that the above only defines getindex() with one integer index. Indexing with anything other than an Int will throw a MethodError saying that there was no matching method. In order to support indexing with ranges or vectors of Ints, separate methods must be written:julia> Base.getindex(S::Squares, i::Number) = S[convert(Int, i)]\n\njulia> Base.getindex(S::Squares, I) = [S[i] for i in I]\n\njulia> Squares(10)[[3,4.,5]]\n3-element Array{Int64,1}:\n  9\n 16\n 25While this is starting to support more of the indexing operations supported by some of the builtin types, there's still quite a number of behaviors missing. This Squares sequence is starting to look more and more like a vector as we've added behaviors to it. Instead of defining all these behaviors ourselves, we can officially define it as a subtype of an AbstractArray."
},

{
    "location": "manual/interfaces.html#man-interface-array-1",
    "page": "Interfaces",
    "title": "Abstract Arrays",
    "category": "section",
    "text": "Methods to implement   Brief description\nsize(A)   Returns a tuple containing the dimensions of A\ngetindex(A, i::Int)   (if IndexLinear) Linear scalar indexing\ngetindex(A, I::Vararg{Int, N})   (if IndexCartesian, where N = ndims(A)) N-dimensional scalar indexing\nsetindex!(A, v, i::Int)   (if IndexLinear) Scalar indexed assignment\nsetindex!(A, v, I::Vararg{Int, N})   (if IndexCartesian, where N = ndims(A)) N-dimensional scalar indexed assignment\nOptional methods Default definition Brief description\nIndexStyle(::Type) IndexCartesian() Returns either IndexLinear() or IndexCartesian(). See the description below.\ngetindex(A, I...) defined in terms of scalar getindex() Multidimensional and nonscalar indexing\nsetindex!(A, I...) defined in terms of scalar setindex!() Multidimensional and nonscalar indexed assignment\nstart()/next()/done() defined in terms of scalar getindex() Iteration\nlength(A) prod(size(A)) Number of elements\nsimilar(A) similar(A, eltype(A), size(A)) Return a mutable array with the same shape and element type\nsimilar(A, ::Type{S}) similar(A, S, size(A)) Return a mutable array with the same shape and the specified element type\nsimilar(A, dims::NTuple{Int}) similar(A, eltype(A), dims) Return a mutable array with the same element type and size dims\nsimilar(A, ::Type{S}, dims::NTuple{Int}) Array{S}(dims) Return a mutable array with the specified element type and size\nNon-traditional indices Default definition Brief description\nindices(A) map(OneTo, size(A)) Return the AbstractUnitRange of valid indices\nBase.similar(A, ::Type{S}, inds::NTuple{Ind}) similar(A, S, Base.to_shape(inds)) Return a mutable array with the specified indices inds (see below)\nBase.similar(T::Union{Type,Function}, inds) T(Base.to_shape(inds)) Return an array similar to T with the specified indices inds (see below)If a type is defined as a subtype of AbstractArray, it inherits a very large set of rich behaviors including iteration and multidimensional indexing built on top of single-element access.  See the arrays manual page and standard library section for more supported methods.A key part in defining an AbstractArray subtype is IndexStyle. Since indexing is such an important part of an array and often occurs in hot loops, it's important to make both indexing and indexed assignment as efficient as possible.  Array data structures are typically defined in one of two ways: either it most efficiently accesses its elements using just one index (linear indexing) or it intrinsically accesses the elements with indices specified for every dimension.  These two modalities are identified by Julia as IndexLinear() and IndexCartesian().  Converting a linear index to multiple indexing subscripts is typically very expensive, so this provides a traits-based mechanism to enable efficient generic code for all array types.This distinction determines which scalar indexing methods the type must define. IndexLinear() arrays are simple: just define getindex(A::ArrayType, i::Int).  When the array is subsequently indexed with a multidimensional set of indices, the fallback getindex(A::AbstractArray, I...)() efficiently converts the indices into one linear index and then calls the above method. IndexCartesian() arrays, on the other hand, require methods to be defined for each supported dimensionality with ndims(A) Int indices. For example, the built-in SparseMatrixCSC type only supports two dimensions, so it just defines getindex(A::SparseMatrixCSC, i::Int, j::Int). The same holds for setindex!().Returning to the sequence of squares from above, we could instead define it as a subtype of an AbstractArray{Int, 1}:julia> struct SquaresVector <: AbstractArray{Int, 1}\n           count::Int\n       end\n\njulia> Base.size(S::SquaresVector) = (S.count,)\n\njulia> Base.IndexStyle(::Type{<:SquaresVector}) = IndexLinear()\n\njulia> Base.getindex(S::SquaresVector, i::Int) = i*iNote that it's very important to specify the two parameters of the AbstractArray; the first defines the eltype(), and the second defines the ndims(). That supertype and those three methods are all it takes for SquaresVector to be an iterable, indexable, and completely functional array:julia> s = SquaresVector(7)\n7-element SquaresVector:\n  1\n  4\n  9\n 16\n 25\n 36\n 49\n\njulia> s[s .> 20]\n3-element Array{Int64,1}:\n 25\n 36\n 49\n\njulia> s \\ [1 2; 3 4; 5 6; 7 8; 9 10; 11 12; 13 14]\n1×2 Array{Float64,2}:\n 0.305389  0.335329\n\njulia> s ⋅ s # dot(s, s)\n4676As a more complicated example, let's define our own toy N-dimensional sparse-like array type built on top of Dict:julia> struct SparseArray{T,N} <: AbstractArray{T,N}\n           data::Dict{NTuple{N,Int}, T}\n           dims::NTuple{N,Int}\n       end\n\njulia> SparseArray{T}(::Type{T}, dims::Int...) = SparseArray(T, dims);\n\njulia> SparseArray{T,N}(::Type{T}, dims::NTuple{N,Int}) = SparseArray{T,N}(Dict{NTuple{N,Int}, T}(), dims);\n\njulia> Base.size(A::SparseArray) = A.dims\n\njulia> Base.similar(A::SparseArray, ::Type{T}, dims::Dims) where {T} = SparseArray(T, dims)\n\njulia> Base.getindex(A::SparseArray{T,N}, I::Vararg{Int,N}) where {T,N} = get(A.data, I, zero(T))\n\njulia> Base.setindex!(A::SparseArray{T,N}, v, I::Vararg{Int,N}) where {T,N} = (A.data[I] = v)Notice that this is an IndexCartesian array, so we must manually define getindex() and setindex!() at the dimensionality of the array. Unlike the SquaresVector, we are able to define setindex!(), and so we can mutate the array:julia> A = SparseArray(Float64, 3, 3)\n3×3 SparseArray{Float64,2}:\n 0.0  0.0  0.0\n 0.0  0.0  0.0\n 0.0  0.0  0.0\n\njulia> fill!(A, 2)\n3×3 SparseArray{Float64,2}:\n 2.0  2.0  2.0\n 2.0  2.0  2.0\n 2.0  2.0  2.0\n\njulia> A[:] = 1:length(A); A\n3×3 SparseArray{Float64,2}:\n 1.0  4.0  7.0\n 2.0  5.0  8.0\n 3.0  6.0  9.0The result of indexing an AbstractArray can itself be an array (for instance when indexing by a Range). The AbstractArray fallback methods use similar() to allocate an Array of the appropriate size and element type, which is filled in using the basic indexing method described above. However, when implementing an array wrapper you often want the result to be wrapped as well:julia> A[1:2,:]\n2×3 SparseArray{Float64,2}:\n 1.0  4.0  7.0\n 2.0  5.0  8.0In this example it is accomplished by defining Base.similar{T}(A::SparseArray, ::Type{T}, dims::Dims) to create the appropriate wrapped array. (Note that while similar supports 1- and 2-argument forms, in most case you only need to specialize the 3-argument form.) For this to work it's important that SparseArray is mutable (supports setindex!). Defining similar(), getindex() and setindex!() for SparseArray also makes it possible to copy() the array:julia> copy(A)\n3×3 SparseArray{Float64,2}:\n 1.0  4.0  7.0\n 2.0  5.0  8.0\n 3.0  6.0  9.0In addition to all the iterable and indexable methods from above, these types can also interact with each other and use most of the methods defined in the standard library for AbstractArrays:julia> A[SquaresVector(3)]\n3-element SparseArray{Float64,1}:\n 1.0\n 4.0\n 9.0\n\njulia> dot(A[:,1],A[:,2])\n32.0If you are defining an array type that allows non-traditional indexing (indices that start at something other than 1), you should specialize indices. You should also specialize similar so that the dims argument (ordinarily a Dims size-tuple) can accept AbstractUnitRange objects, perhaps range-types Ind of your own design. For more information, see Arrays with custom indices."
},

{
    "location": "manual/modules.html#",
    "page": "Modules",
    "title": "Modules",
    "category": "page",
    "text": ""
},

{
    "location": "manual/modules.html#modules-1",
    "page": "Modules",
    "title": "Modules",
    "category": "section",
    "text": "Modules in Julia are separate variable workspaces, i.e. they introduce a new global scope. They are delimited syntactically, inside module Name ... end. Modules allow you to create top-level definitions (aka global variables) without worrying about name conflicts when your code is used together with somebody else's. Within a module, you can control which names from other modules are visible (via importing), and specify which of your names are intended to be public (via exporting).The following example demonstrates the major features of modules. It is not meant to be run, but is shown for illustrative purposes:module MyModule\nusing Lib\n\nusing BigLib: thing1, thing2\n\nimport Base.show\n\nimportall OtherLib\n\nexport MyType, foo\n\nstruct MyType\n    x\nend\n\nbar(x) = 2x\nfoo(a::MyType) = bar(a.x) + 1\n\nshow(io::IO, a::MyType) = print(io, \"MyType $(a.x)\")\nendNote that the style is not to indent the body of the module, since that would typically lead to whole files being indented.This module defines a type MyType, and two functions. Function foo and type MyType are exported, and so will be available for importing into other modules.  Function bar is private to MyModule.The statement using Lib means that a module called Lib will be available for resolving names as needed. When a global variable is encountered that has no definition in the current module, the system will search for it among variables exported by Lib and import it if it is found there. This means that all uses of that global within the current module will resolve to the definition of that variable in Lib.The statement using BigLib: thing1, thing2 is a syntactic shortcut for using BigLib.thing1, BigLib.thing2.The import keyword supports all the same syntax as using, but only operates on a single name at a time. It does not add modules to be searched the way using does. import also differs from using in that functions must be imported using import to be extended with new methods.In MyModule above we wanted to add a method to the standard show function, so we had to write import Base.show. Functions whose names are only visible via using cannot be extended.The keyword importall explicitly imports all names exported by the specified module, as if import were individually used on all of them.Once a variable is made visible via using or import, a module may not create its own variable with the same name. Imported variables are read-only; assigning to a global variable always affects a variable owned by the current module, or else raises an error."
},

{
    "location": "manual/modules.html#Summary-of-module-usage-1",
    "page": "Modules",
    "title": "Summary of module usage",
    "category": "section",
    "text": "To load a module, two main keywords can be used: using and import. To understand their differences, consider the following example:module MyModule\n\nexport x, y\n\nx() = \"x\"\ny() = \"y\"\np() = \"p\"\n\nendIn this module we export the x and y functions (with the keyword export), and also have the non-exported function p. There are several different ways to load the Module and its inner functions into the current workspace:Import Command What is brought into scope Available for method extension\nusing MyModule All exported names (x and y), MyModule.x, MyModule.y and MyModule.p MyModule.x, MyModule.y and MyModule.p\nusing MyModule.x, MyModule.p x and p  \nusing MyModule: x, p x and p  \nimport MyModule MyModule.x, MyModule.y and MyModule.p MyModule.x, MyModule.y and MyModule.p\nimport MyModule.x, MyModule.p x and p x and p\nimport MyModule: x, p x and p x and p\nimportall MyModule All exported names (x and y) x and y"
},

{
    "location": "manual/modules.html#Modules-and-files-1",
    "page": "Modules",
    "title": "Modules and files",
    "category": "section",
    "text": "Files and file names are mostly unrelated to modules; modules are associated only with module expressions. One can have multiple files per module, and multiple modules per file:module Foo\n\ninclude(\"file1.jl\")\ninclude(\"file2.jl\")\n\nendIncluding the same code in different modules provides mixin-like behavior. One could use this to run the same code with different base definitions, for example testing code by running it with \"safe\" versions of some operators:module Normal\ninclude(\"mycode.jl\")\nend\n\nmodule Testing\ninclude(\"safe_operators.jl\")\ninclude(\"mycode.jl\")\nend"
},

{
    "location": "manual/modules.html#Standard-modules-1",
    "page": "Modules",
    "title": "Standard modules",
    "category": "section",
    "text": "There are three important standard modules: Main, Core, and Base.Main is the top-level module, and Julia starts with Main set as the current module.  Variables defined at the prompt go in Main, and whos() lists variables in Main.Core contains all identifiers considered \"built in\" to the language, i.e. part of the core language and not libraries. Every module implicitly specifies using Core, since you can't do anything without those definitions.Base is the standard library (the contents of base/). All modules implicitly contain using Base, since this is needed in the vast majority of cases."
},

{
    "location": "manual/modules.html#Default-top-level-definitions-and-bare-modules-1",
    "page": "Modules",
    "title": "Default top-level definitions and bare modules",
    "category": "section",
    "text": "In addition to using Base, modules also automatically contain a definition of the eval function, which evaluates expressions within the context of that module.If these default definitions are not wanted, modules can be defined using the keyword baremodule instead (note: Core is still imported, as per above). In terms of baremodule, a standard module looks like this:baremodule Mod\n\nusing Base\n\neval(x) = Core.eval(Mod, x)\neval(m,x) = Core.eval(m, x)\n\n...\n\nend"
},

{
    "location": "manual/modules.html#Relative-and-absolute-module-paths-1",
    "page": "Modules",
    "title": "Relative and absolute module paths",
    "category": "section",
    "text": "Given the statement using Foo, the system looks for Foo within Main. If the module does not exist, the system attempts to require(\"Foo\"), which typically results in loading code from an installed package.However, some modules contain submodules, which means you sometimes need to access a module that is not directly available in Main. There are two ways to do this. The first is to use an absolute path, for example using Base.Sort. The second is to use a relative path, which makes it easier to import submodules of the current module or any of its enclosing modules:module Parent\n\nmodule Utils\n...\nend\n\nusing .Utils\n\n...\nendHere module Parent contains a submodule Utils, and code in Parent wants the contents of Utils to be visible. This is done by starting the using path with a period. Adding more leading periods moves up additional levels in the module hierarchy. For example using ..Utils would look for Utils in Parent's enclosing module rather than in Parent itself.Note that relative-import qualifiers are only valid in using and import statements."
},

{
    "location": "manual/modules.html#Module-file-paths-1",
    "page": "Modules",
    "title": "Module file paths",
    "category": "section",
    "text": "The global variable LOAD_PATH contains the directories Julia searches for modules when calling require. It can be extended using push!:push!(LOAD_PATH, \"/Path/To/My/Module/\")Putting this statement in the file ~/.juliarc.jl will extend LOAD_PATH on every Julia startup. Alternatively, the module load path can be extended by defining the environment variable JULIA_LOAD_PATH."
},

{
    "location": "manual/modules.html#Namespace-miscellanea-1",
    "page": "Modules",
    "title": "Namespace miscellanea",
    "category": "section",
    "text": "If a name is qualified (e.g. Base.sin), then it can be accessed even if it is not exported. This is often useful when debugging. It can also have methods added to it by using the qualified name as the function name. However, due to syntactic ambiguities that arise, if you wish to add methods to a function in a different module whose name contains only symbols, such as an operator, Base.+ for example, you must use Base.:+ to refer to it. If the operator is more than one character in length you must surround it in brackets, such as: Base.:(==).Macro names are written with @ in import and export statements, e.g. import Mod.@mac. Macros in other modules can be invoked as Mod.@mac or @Mod.mac.The syntax M.x = y does not work to assign a global in another module; global assignment is always module-local.A variable can be \"reserved\" for the current module without assigning to it by declaring it as global x at the top level. This can be used to prevent name conflicts for globals initialized after load time."
},

{
    "location": "manual/modules.html#Module-initialization-and-precompilation-1",
    "page": "Modules",
    "title": "Module initialization and precompilation",
    "category": "section",
    "text": "Large modules can take several seconds to load because executing all of the statements in a module often involves compiling a large amount of code. Julia provides the ability to create precompiled versions of modules to reduce this time.To create an incremental precompiled module file, add __precompile__() at the top of your module file (before the module starts). This will cause it to be automatically compiled the first time it is imported. Alternatively, you can manually call Base.compilecache(modulename). The resulting cache files will be stored in Base.LOAD_CACHE_PATH[1]. Subsequently, the module is automatically recompiled upon import whenever any of its dependencies change; dependencies are modules it imports, the Julia build, files it includes, or explicit dependencies declared by include_dependency(path) in the module file(s).For file dependencies, a change is determined by examining whether the modification time (mtime) of each file loaded by include or added explicitly by include_dependency is unchanged, or equal to the modification time truncated to the nearest second (to accommodate systems that can't copy mtime with sub-second accuracy). It also takes into account whether the path to the file chosen by the search logic in require matches the path that had created the precompile file.It also takes into account the set of dependencies already loaded into the current process and won't recompile those modules, even if their files change or disappear, in order to avoid creating incompatibilities between the running system and the precompile cache. If you want to have changes to the source reflected in the running system, you should call reload(\"Module\") on the module you changed, and any module that depended on it in which you want to see the change reflected.Precompiling a module also recursively precompiles any modules that are imported therein. If you know that it is not safe to precompile your module (for the reasons described below), you should put __precompile__(false) in the module file to cause Base.compilecache to throw an error (and thereby prevent the module from being imported by any other precompiled module).__precompile__() should not be used in a module unless all of its dependencies are also using __precompile__(). Failure to do so can result in a runtime error when loading the module.In order to make your module work with precompilation, however, you may need to change your module to explicitly separate any initialization steps that must occur at runtime from steps that can occur at compile time.  For this purpose, Julia allows you to define an __init__() function in your module that executes any initialization steps that must occur at runtime. This function will not be called during compilation (--output-* or __precompile__()). You may, of course, call it manually if necessary, but the default is to assume this function deals with computing state for the local machine, which does not need to be – or even should not be – captured in the compiled image. It will be called after the module is loaded into a process, including if it is being loaded into an incremental compile (--output-incremental=yes), but not if it is being loaded into a full-compilation process.In particular, if you define a function __init__() in a module, then Julia will call __init__() immediately after the module is loaded (e.g., by import, using, or require) at runtime for the first time (i.e., __init__ is only called once, and only after all statements in the module have been executed). Because it is called after the module is fully imported, any submodules or other imported modules have their __init__ functions called before the __init__ of the enclosing module.Two typical uses of __init__ are calling runtime initialization functions of external C libraries and initializing global constants that involve pointers returned by external libraries.  For example, suppose that we are calling a C library libfoo that requires us to call a foo_init() initialization function at runtime. Suppose that we also want to define a global constant foo_data_ptr that holds the return value of a void *foo_data() function defined by libfoo – this constant must be initialized at runtime (not at compile time) because the pointer address will change from run to run.  You could accomplish this by defining the following __init__ function in your module:const foo_data_ptr = Ref{Ptr{Void}}(0)\nfunction __init__()\n    ccall((:foo_init, :libfoo), Void, ())\n    foo_data_ptr[] = ccall((:foo_data, :libfoo), Ptr{Void}, ())\nendNotice that it is perfectly possible to define a global inside a function like __init__; this is one of the advantages of using a dynamic language. But by making it a constant at global scope, we can ensure that the type is known to the compiler and allow it to generate better optimized code. Obviously, any other globals in your module that depends on foo_data_ptr would also have to be initialized in __init__.Constants involving most Julia objects that are not produced by ccall do not need to be placed in __init__: their definitions can be precompiled and loaded from the cached module image. This includes complicated heap-allocated objects like arrays. However, any routine that returns a raw pointer value must be called at runtime for precompilation to work (Ptr objects will turn into null pointers unless they are hidden inside an isbits object). This includes the return values of the Julia functions cfunction and pointer.Dictionary and set types, or in general anything that depends on the output of a hash(key) method, are a trickier case.  In the common case where the keys are numbers, strings, symbols, ranges, Expr, or compositions of these types (via arrays, tuples, sets, pairs, etc.) they are safe to precompile.  However, for a few other key types, such as Function or DataType and generic user-defined types where you haven't defined a hash method, the fallback hash method depends on the memory address of the object (via its object_id) and hence may change from run to run. If you have one of these key types, or if you aren't sure, to be safe you can initialize this dictionary from within your __init__ function. Alternatively, you can use the ObjectIdDict dictionary type, which is specially handled by precompilation so that it is safe to initialize at compile-time.When using precompilation, it is important to keep a clear sense of the distinction between the compilation phase and the execution phase. In this mode, it will often be much more clearly apparent that Julia is a compiler which allows execution of arbitrary Julia code, not a standalone interpreter that also generates compiled code.Other known potential failure scenarios include:Global counters (for example, for attempting to uniquely identify objects) Consider the following code snippet:\nmutable struct UniquedById\n    myid::Int\n    let counter = 0\n        UniquedById() = new(counter += 1)\n    end\nend\nwhile the intent of this code was to give every instance a unique id, the counter value is recorded at the end of compilation. All subsequent usages of this incrementally compiled module will start from that same counter value.\nNote that object_id (which works by hashing the memory pointer) has similar issues (see notes on Dict usage below).\nOne alternative is to store both current_module() and the current counter value, however, it may be better to redesign the code to not depend on this global state.\nAssociative collections (such as Dict and Set) need to be re-hashed in __init__. (In the future, a mechanism may be provided to register an initializer function.)\nDepending on compile-time side-effects persisting through load-time. Example include: modifying arrays or other variables in other Julia modules; maintaining handles to open files or devices; storing pointers to other system resources (including memory);\nCreating accidental \"copies\" of global state from another module, by referencing it directly instead of via its lookup path. For example, (in global scope):\n#mystdout = Base.STDOUT #= will not work correctly, since this will copy Base.STDOUT into this module =#\n# instead use accessor functions:\ngetstdout() = Base.STDOUT #= best option =#\n# or move the assignment into the runtime:\n__init__() = global mystdout = Base.STDOUT #= also works =#Several additional restrictions are placed on the operations that can be done while precompiling code to help the user avoid other wrong-behavior situations:Calling eval to cause a side-effect in another module. This will also cause a warning to be emitted when the incremental precompile flag is set.\nglobal const statements from local scope after __init__() has been started (see issue #12010 for plans to add an error for this)\nReplacing a module (or calling workspace()) is a runtime error while doing an incremental precompile.A few other points to be aware of:No code reload / cache invalidation is performed after changes are made to the source files themselves, (including by Pkg.update), and no cleanup is done after Pkg.rm\nThe memory sharing behavior of a reshaped array is disregarded by precompilation (each view gets its own copy)\nExpecting the filesystem to be unchanged between compile-time and runtime e.g. @__FILE__/source_path() to find resources at runtime, or the BinDeps @checked_lib macro. Sometimes this is unavoidable. However, when possible, it can be good practice to copy resources into the module at compile-time so they won't need to be found at runtime.\nWeakRef objects and finalizers are not currently handled properly by the serializer (this will be fixed in an upcoming release).\nIt is usually best to avoid capturing references to instances of internal metadata objects such as Method, MethodInstance, MethodTable, TypeMapLevel, TypeMapEntry and fields of those objects, as this can confuse the serializer and may not lead to the outcome you desire. It is not necessarily an error to do this, but you simply need to be prepared that the system will try to copy some of these and to create a single unique instance of others.It is sometimes helpful during module development to turn off incremental precompilation. The command line flag --compilecache={yes|no} enables you to toggle module precompilation on and off. When Julia is started with --compilecache=no the serialized modules in the compile cache are ignored when loading modules and module dependencies. Base.compilecache() can still be called manually and it will respect __precompile__() directives for the module. The state of this command line flag is passed to Pkg.build() to disable automatic precompilation triggering when installing, updating, and explicitly building packages."
},

{
    "location": "manual/documentation.html#",
    "page": "Documentation",
    "title": "Documentation",
    "category": "page",
    "text": ""
},

{
    "location": "manual/documentation.html#Documentation-1",
    "page": "Documentation",
    "title": "Documentation",
    "category": "section",
    "text": "Julia enables package developers and users to document functions, types and other objects easily via a built-in documentation system since Julia 0.4.The basic syntax is very simple: any string appearing at the top-level right before an object (function, macro, type or instance) will be interpreted as documenting it (these are called docstrings). Here is a very simple example:\"Tell whether there are too foo items in the array.\"\nfoo(xs::Array) = ...Documentation is interpreted as Markdown, so you can use indentation and code fences to delimit code examples from text. Technically, any object can be associated with any other as metadata; Markdown happens to be the default, but one can construct other string macros and pass them to the @doc macro just as well.Here is a more complex example, still using Markdown:\"\"\"\n    bar(x[, y])\n\nCompute the Bar index between `x` and `y`. If `y` is missing, compute\nthe Bar index between all pairs of columns of `x`.\n\n# Examples\n```julia-repl\njulia> bar([1, 2], [1, 2])\n1\n```\n\"\"\"\nfunction bar(x, y) ...As in the example above, we recommend following some simple conventions when writing documentation:Always show the signature of a function at the top of the documentation, with a four-space indent so that it is printed as Julia code.\nThis can be identical to the signature present in the Julia code (like mean(x::AbstractArray)), or a simplified form. Optional arguments should be represented with their default values (i.e. f(x, y=1)) when possible, following the actual Julia syntax. Optional arguments which do not have a default value should be put in brackets (i.e. f(x[, y]) and f(x[, y[, z]])). An alternative solution is to use several lines: one without optional arguments, the other(s) with them. This solution can also be used to document several related methods of a given function. When a function accepts many keyword arguments, only include a <keyword arguments> placeholder in the signature (i.e. f(x; <keyword arguments>)), and give the complete list under an # Arguments section (see point 4 below).\nInclude a single one-line sentence describing what the function does or what the object represents after the simplified signature block. If needed, provide more details in a second paragraph, after a blank line.\nThe one-line sentence should use the imperative form (\"Do this\", \"Return that\") instead of the third person (do not write \"Returns the length...\") when documenting functions. It should end with a period. If the meaning of a function cannot be summarized easily, splitting it into separate composable parts could be beneficial (this should not be taken as an absolute requirement for every single case though).\nDo not repeat yourself.\nSince the function name is given by the signature, there is no need to start the documentation with \"The function bar...\": go straight to the point. Similarly, if the signature specifies the types of the arguments, mentioning them in the description is redundant.\nOnly provide an argument list when really necessary.\nFor simple functions, it is often clearer to mention the role of the arguments directly in the description of the function's purpose. An argument list would only repeat information already provided elsewhere. However, providing an argument list can be a good idea for complex functions with many arguments (in particular keyword arguments). In that case, insert it after the general description of the function, under an # Arguments header, with one - bullet for each argument. The list should mention the types and default values (if any) of the arguments:\n\"\"\"\n...\n# Arguments\n- `n::Integer`: the number of elements to compute.\n- `dim::Integer=1`: the dimensions along which to perform the computation.\n...\n\"\"\"\nInclude any code examples in an # Examples section.\nExamples should, whenever possible, be written as doctests. A doctest is a fenced code block (see Code blocks) starting with ```jldoctest and contains any number of julia> prompts together with inputs and expected outputs that mimic the Julia REPL.\nFor example in the following docstring a variable a is defined and the expected result, as printed in a Julia REPL, appears afterwards:\n\"\"\"\nSome nice documentation here.\n\n# Examples\n\n```jldoctest\njulia> a = [1 2; 3 4]\n2×2 Array{Int64,2}:\n 1  2\n 3  4\n```\n\"\"\"\nwarning: Warning\nCalling rand and other RNG-related functions should be avoided in doctests since they will not produce consistent outputs during different Julia sessions.Operating system word size (Int32 or Int64) as well as path separator differences (/ or \\) will also affect the reproducibility of some doctests.Note that whitespace in your doctest is significant! The doctest will fail if you misalign the output of pretty-printing an array, for example.\nYou can then run make -C doc doctest to run all the doctests in the Julia Manual, which will ensure that your example works.\nExamples that are untestable should be written within fenced code blocks starting with ```julia so that they are highlighted correctly in the generated documentation.\ntip: Tip\nWherever possible examples should be self-contained and runnable so that readers are able to try them out without having to include any dependencies.\nUse backticks to identify code and equations.\nJulia identifiers and code excerpts should always appear between backticks ` to enable highlighting. Equations in the LaTeX syntax can be inserted between double backticks ``. Use Unicode characters rather than their LaTeX escape sequence, i.e. ``α = 1`` rather than ``\\\\alpha = 1``.\nPlace the starting and ending \"\"\" characters on lines by themselves.\nThat is, write:\n\"\"\"\n...\n\n...\n\"\"\"\nf(x, y) = ...\nrather than:\n\"\"\"...\n\n...\"\"\"\nf(x, y) = ...\nThis makes it more clear where docstrings start and end.\nRespect the line length limit used in the surrounding code.\nDocstrings are edited using the same tools as code. Therefore, the same conventions should apply. It it advised to add line breaks after 92 characters."
},

{
    "location": "manual/documentation.html#Accessing-Documentation-1",
    "page": "Documentation",
    "title": "Accessing Documentation",
    "category": "section",
    "text": "Documentation can be accessed at the REPL or in IJulia by typing ? followed by the name of a function or macro, and pressing Enter. For example,?fft\n?@time\n?r\"\"will bring up docs for the relevant function, macro or string macro respectively. In Juno using Ctrl-J, Ctrl-D will bring up documentation for the object under the cursor."
},

{
    "location": "manual/documentation.html#Functions-and-Methods-1",
    "page": "Documentation",
    "title": "Functions & Methods",
    "category": "section",
    "text": "Functions in Julia may have multiple implementations, known as methods. While it's good practice for generic functions to have a single purpose, Julia allows methods to be documented individually if necessary. In general, only the most generic method should be documented, or even the function itself (i.e. the object created without any methods by function bar end). Specific methods should only be documented if their behaviour differs from the more generic ones. In any case, they should not repeat the information provided elsewhere. For example:\"\"\"\n    *(x, y, z...)\n\nMultiplication operator. `x * y * z *...` calls this function with multiple\narguments, i.e. `*(x, y, z...)`.\n\"\"\"\nfunction *(x, y, z...)\n    # ... [implementation sold separately] ...\nend\n\n\"\"\"\n    *(x::AbstractString, y::AbstractString, z::AbstractString...)\n\nWhen applied to strings, concatenates them.\n\"\"\"\nfunction *(x::AbstractString, y::AbstractString, z::AbstractString...)\n    # ... [insert secret sauce here] ...\nend\n\nhelp?> *\nsearch: * .*\n\n  *(x, y, z...)\n\n  Multiplication operator. x * y * z *... calls this function with multiple\n  arguments, i.e. *(x,y,z...).\n\n  *(x::AbstractString, y::AbstractString, z::AbstractString...)\n\n  When applied to strings, concatenates them.When retrieving documentation for a generic function, the metadata for each method is concatenated with the catdoc function, which can of course be overridden for custom types."
},

{
    "location": "manual/documentation.html#Advanced-Usage-1",
    "page": "Documentation",
    "title": "Advanced Usage",
    "category": "section",
    "text": "The @doc macro associates its first argument with its second in a per-module dictionary called META. By default, documentation is expected to be written in Markdown, and the doc\"\" string macro simply creates an object representing the Markdown content. In the future it is likely to do more advanced things such as allowing for relative image or link paths.When used for retrieving documentation, the @doc macro (or equally, the doc function) will search all META dictionaries for metadata relevant to the given object and return it. The returned object (some Markdown content, for example) will by default display itself intelligently. This design also makes it easy to use the doc system in a programmatic way; for example, to re-use documentation between different versions of a function:@doc \"...\" foo!\n@doc (@doc foo!) fooOr for use with Julia's metaprogramming functionality:for (f, op) in ((:add, :+), (:subtract, :-), (:multiply, :*), (:divide, :/))\n    @eval begin\n        $f(a,b) = $op(a,b)\n    end\nend\n@doc \"`add(a,b)` adds `a` and `b` together\" add\n@doc \"`subtract(a,b)` subtracts `b` from `a`\" subtractDocumentation written in non-toplevel blocks, such as begin, if, for, and let, is added to the documentation system as blocks are evaluated. For example:if VERSION > v\"0.5\"\n    \"...\"\n    f(x) = x\nendwill add documentation to f(x) when the condition is true. Note that even if f(x) goes out of scope at the end of the block, its documentation will remain."
},

{
    "location": "manual/documentation.html#Dynamic-documentation-1",
    "page": "Documentation",
    "title": "Dynamic documentation",
    "category": "section",
    "text": "Sometimes the appropriate documentation for an instance of a type depends on the field values of that instance, rather than just on the type itself. In these cases, you can add a method to Docs.getdoc for your custom type that returns the documentation on a per-instance basis. For instance,struct MyType\n    value::String\nend\n\nDocs.getdoc(t::MyType) = \"Documentation for MyType with value $(t.value)\"\n\nx = MyType(\"x\")\ny = MyType(\"y\")?x will display \"Documentation for MyType with value x\" while ?y will display \"Documentation for MyType with value y\"."
},

{
    "location": "manual/documentation.html#Syntax-Guide-1",
    "page": "Documentation",
    "title": "Syntax Guide",
    "category": "section",
    "text": "A comprehensive overview of all documentable Julia syntax.In the following examples \"...\" is used to illustrate an arbitrary docstring which may be one of the follow four variants and contain arbitrary text:\"...\"\n\ndoc\"...\"\n\n\"\"\"\n...\n\"\"\"\n\ndoc\"\"\"\n...\n\"\"\"@doc_str should only be used when the docstring contains $ or \\ characters that should not be parsed by Julia such as LaTeX syntax or Julia source code examples containing interpolation."
},

{
    "location": "manual/documentation.html#Functions-and-Methods-2",
    "page": "Documentation",
    "title": "Functions and Methods",
    "category": "section",
    "text": "\"...\"\nfunction f end\n\n\"...\"\nfAdds docstring \"...\" to Functionf. The first version is the preferred syntax, however both are equivalent.\"...\"\nf(x) = x\n\n\"...\"\nfunction f(x)\n    x\nend\n\n\"...\"\nf(x)Adds docstring \"...\" to Methodf(::Any).\"...\"\nf(x, y = 1) = x + yAdds docstring \"...\" to two Methods, namely f(::Any) and f(::Any, ::Any)."
},

{
    "location": "manual/documentation.html#Macros-1",
    "page": "Documentation",
    "title": "Macros",
    "category": "section",
    "text": "\"...\"\nmacro m(x) endAdds docstring \"...\" to the @m(::Any) macro definition.\"...\"\n:(@m)Adds docstring \"...\" to the macro named @m."
},

{
    "location": "manual/documentation.html#Types-1",
    "page": "Documentation",
    "title": "Types",
    "category": "section",
    "text": "\"...\"\nabstract type T1 end\n\n\"...\"\nmutable struct T2\n    ...\nend\n\n\"...\"\nstruct T3\n    ...\nendAdds the docstring \"...\" to types T1, T2, and T3.\"...\"\nstruct T\n    \"x\"\n    x\n    \"y\"\n    y\nendAdds docstring \"...\" to type T, \"x\" to field T.x and \"y\" to field T.y. Also applicable to mutable struct types."
},

{
    "location": "manual/documentation.html#Modules-1",
    "page": "Documentation",
    "title": "Modules",
    "category": "section",
    "text": "\"...\"\nmodule M end\n\nmodule M\n\n\"...\"\nM\n\nendAdds docstring \"...\" to the ModuleM. Adding the docstring above the Module is the preferred syntax, however both are equivalent.\"...\"\nbaremodule M\n# ...\nend\n\nbaremodule M\n\nimport Base: @doc\n\n\"...\"\nf(x) = x\n\nendDocumenting a baremodule by placing a docstring above the expression automatically imports @doc into the module. These imports must be done manually when the module expression is not documented. Empty baremodules cannot be documented."
},

{
    "location": "manual/documentation.html#Global-Variables-1",
    "page": "Documentation",
    "title": "Global Variables",
    "category": "section",
    "text": "\"...\"\nconst a = 1\n\n\"...\"\nb = 2\n\n\"...\"\nglobal c = 3Adds docstring \"...\" to the Bindings a, b, and c.Bindings are used to store a reference to a particular Symbol in a Module without storing the referenced value itself.note: Note\nWhen a const definition is only used to define an alias of another definition, such as is the case with the function div and its alias ÷ in Base, do not document the alias and instead document the actual function.If the alias is documented and not the real definition then the docsystem (? mode) will not return the docstring attached to the alias when the real definition is searched for.For example you should write\"...\"\nf(x) = x + 1\nconst alias = frather thanf(x) = x + 1\n\"...\"\nconst alias = f\"...\"\nsymAdds docstring \"...\" to the value associated with sym. Users should prefer documenting sym at it's definition."
},

{
    "location": "manual/documentation.html#Multiple-Objects-1",
    "page": "Documentation",
    "title": "Multiple Objects",
    "category": "section",
    "text": "\"...\"\na, bAdds docstring \"...\" to a and b each of which should be a documentable expression. This syntax is equivalent to\"...\"\na\n\n\"...\"\nbAny number of expressions many be documented together in this way. This syntax can be useful when two functions are related, such as non-mutating and mutating versions f and f!."
},

{
    "location": "manual/documentation.html#Core.@__doc__",
    "page": "Documentation",
    "title": "Core.@__doc__",
    "category": "Macro",
    "text": "@__doc__(ex)\n\nLow-level macro used to mark expressions returned by a macro that should be documented. If more than one expression is marked then the same docstring is applied to each expression.\n\nmacro example(f)\n    quote\n        $(f)() = 0\n        @__doc__ $(f)(x) = 1\n        $(f)(x, y) = 2\n    end |> esc\nend\n\n@__doc__ has no effect when a macro that uses it is not documented.\n\n\n\n"
},

{
    "location": "manual/documentation.html#Macro-generated-code-1",
    "page": "Documentation",
    "title": "Macro-generated code",
    "category": "section",
    "text": "\"...\"\n@m expressionAdds docstring \"...\" to expression generated by expanding @m expression. This allows for expressions decorated with @inline, @noinline, @generated, or any other macro to be documented in the same way as undecorated expressions.Macro authors should take note that only macros that generate a single expression will automatically support docstrings. If a macro returns a block containing multiple subexpressions then the subexpression that should be documented must be marked using the @__doc__ macro.The @enum macro makes use of @__doc__ to allow for documenting Enums. Examining it's definition should serve as an example of how to use @__doc__ correctly.Core.@__doc__"
},

{
    "location": "manual/documentation.html#Markdown-syntax-1",
    "page": "Documentation",
    "title": "Markdown syntax",
    "category": "section",
    "text": "The following markdown syntax is supported in Julia."
},

{
    "location": "manual/documentation.html#Inline-elements-1",
    "page": "Documentation",
    "title": "Inline elements",
    "category": "section",
    "text": "Here \"inline\" refers to elements that can be found within blocks of text, i.e. paragraphs. These include the following elements."
},

{
    "location": "manual/documentation.html#Bold-1",
    "page": "Documentation",
    "title": "Bold",
    "category": "section",
    "text": "Surround words with two asterisks, **, to display the enclosed text in boldface.A paragraph containing a **bold** word."
},

{
    "location": "manual/documentation.html#Italics-1",
    "page": "Documentation",
    "title": "Italics",
    "category": "section",
    "text": "Surround words with one asterisk, *, to display the enclosed text in italics.A paragraph containing an *emphasised* word."
},

{
    "location": "manual/documentation.html#Literals-1",
    "page": "Documentation",
    "title": "Literals",
    "category": "section",
    "text": "Surround text that should be displayed exactly as written with single backticks, ` .A paragraph containing a `literal` word.Literals should be used when writing text that refers to names of variables, functions, or other parts of a Julia program.tip: Tip\nTo include a backtick character within literal text use three backticks rather than one to enclose the text.A paragraph containing a ``` `backtick` character ```.By extension any odd number of backticks may be used to enclose a lesser number of backticks."
},

{
    "location": "manual/documentation.html#\\LaTeX-1",
    "page": "Documentation",
    "title": "LaTeX",
    "category": "section",
    "text": "Surround text that should be displayed as mathematics using LaTeX syntax with double backticks, `` .A paragraph containing some ``\\LaTeX`` markup.tip: Tip\nAs with literals in the previous section, if literal backticks need to be written within double backticks use an even number greater than two. Note that if a single literal backtick needs to be included within LaTeX markup then two enclosing backticks is sufficient."
},

{
    "location": "manual/documentation.html#Links-1",
    "page": "Documentation",
    "title": "Links",
    "category": "section",
    "text": "Links to either external or internal addresses can be written using the following syntax, where the text enclosed in square brackets, [ ], is the name of the link and the text enclosed in parentheses, ( ), is the URL.A paragraph containing a link to [Julia](http://www.julialang.org).It's also possible to add cross-references to other documented functions/methods/variables within the Julia documentation itself. For example:\"\"\"\n    eigvals!(A,[irange,][vl,][vu]) -> values\n\nSame as [`eigvals`](@ref), but saves space by overwriting the input `A`, instead of creating a copy.\n\"\"\"This will create a link in the generated docs to the eigvals documentation (which has more information about what this function actually does). It's good to include cross references to mutating/non-mutating versions of a function, or to highlight a difference between two similar-seeming functions.note: Note\nThe above cross referencing is not a Markdown feature, and relies on Documenter.jl, which is used to build base Julia's documentation."
},

{
    "location": "manual/documentation.html#Footnote-references-1",
    "page": "Documentation",
    "title": "Footnote references",
    "category": "section",
    "text": "Named and numbered footnote references can be written using the following syntax. A footnote name must be a single alphanumeric word containing no punctuation.A paragraph containing a numbered footnote [^1] and a named one [^named].note: Note\nThe text associated with a footnote can be written anywhere within the same page as the footnote reference. The syntax used to define the footnote text is discussed in the Footnotes section below."
},

{
    "location": "manual/documentation.html#Toplevel-elements-1",
    "page": "Documentation",
    "title": "Toplevel elements",
    "category": "section",
    "text": "The following elements can be written either at the \"toplevel\" of a document or within another \"toplevel\" element."
},

{
    "location": "manual/documentation.html#Paragraphs-1",
    "page": "Documentation",
    "title": "Paragraphs",
    "category": "section",
    "text": "A paragraph is a block of plain text, possibly containing any number of inline elements defined in the Inline elements section above, with one or more blank lines above and below it.This is a paragraph.\n\nAnd this is *another* one containing some emphasised text.\nA new line, but still part of the same paragraph."
},

{
    "location": "manual/documentation.html#Headers-1",
    "page": "Documentation",
    "title": "Headers",
    "category": "section",
    "text": "A document can be split up into different sections using headers. Headers use the following syntax:# Level One\n## Level Two\n### Level Three\n#### Level Four\n##### Level Five\n###### Level SixA header line can contain any inline syntax in the same way as a paragraph can.tip: Tip\nTry to avoid using too many levels of header within a single document. A heavily nested document may be indicative of a need to restructure it or split it into several pages covering separate topics."
},

{
    "location": "manual/documentation.html#Code-blocks-1",
    "page": "Documentation",
    "title": "Code blocks",
    "category": "section",
    "text": "Source code can be displayed as a literal block using an indent of four spaces as shown in the following example.This is a paragraph.\n\n    function func(x)\n        # ...\n    end\n\nAnother paragraph.Additionally, code blocks can be enclosed using triple backticks with an optional \"language\" to specify how a block of code should be highlighted.A code block without a \"language\":\n\n```\nfunction func(x)\n    # ...\nend\n```\n\nand another one with the \"language\" specified as `julia`:\n\n```julia\nfunction func(x)\n    # ...\nend\n```note: Note\n\"Fenced\" code blocks, as shown in the last example, should be prefered over indented code blocks since there is no way to specify what language an indented code block is written in."
},

{
    "location": "manual/documentation.html#Block-quotes-1",
    "page": "Documentation",
    "title": "Block quotes",
    "category": "section",
    "text": "Text from external sources, such as quotations from books or websites, can be quoted using > characters prepended to each line of the quote as follows.Here's a quote:\n\n> Julia is a high-level, high-performance dynamic programming language for\n> technical computing, with syntax that is familiar to users of other\n> technical computing environments.Note that a single space must appear after the > character on each line. Quoted blocks may themselves contain other toplevel or inline elements."
},

{
    "location": "manual/documentation.html#Images-1",
    "page": "Documentation",
    "title": "Images",
    "category": "section",
    "text": "The syntax for images is similar to the link syntax mentioned above. Prepending a ! character to a link will display an image from the specified URL rather than a link to it.![alternative text](link/to/image.png)"
},

{
    "location": "manual/documentation.html#Lists-1",
    "page": "Documentation",
    "title": "Lists",
    "category": "section",
    "text": "Unordered lists can be written by prepending each item in a list with either *, +, or -.A list of items:\n\n  * item one\n  * item two\n  * item threeNote the two spaces before each * and the single space after each one.Lists can contain other nested toplevel elements such as lists, code blocks, or quoteblocks. A blank line should be left between each list item when including any toplevel elements within a list.Another list:\n\n  * item one\n\n  * item two\n\n    ```\n    f(x) = x\n    ```\n\n  * And a sublist:\n\n      + sub-item one\n      + sub-item twonote: Note\nThe contents of each item in the list must line up with the first line of the item. In the above example the fenced code block must be indented by four spaces to align with the i in item two.Ordered lists are written by replacing the \"bullet\" character, either *, +, or -, with a positive integer followed by either . or ).Two ordered lists:\n\n 1. item one\n 2. item two\n 3. item three\n\n 5) item five\n 6) item six\n 7) item sevenAn ordered list may start from a number other than one, as in the second list of the above example, where it is numbered from five. As with unordered lists, ordered lists can contain nested toplevel elements."
},

{
    "location": "manual/documentation.html#Display-equations-1",
    "page": "Documentation",
    "title": "Display equations",
    "category": "section",
    "text": "Large LaTeX equations that do not fit inline within a paragraph may be written as display equations using a fenced code block with the \"language\" math as in the example below.```math\nf(a) = \\frac{1}{2\\pi}\\int_{0}^{2\\pi} (\\alpha+R\\cos(\\theta))d\\theta\n```"
},

{
    "location": "manual/documentation.html#Footnotes-1",
    "page": "Documentation",
    "title": "Footnotes",
    "category": "section",
    "text": "This syntax is paired with the inline syntax for Footnote references. Make sure to read that section as well.Footnote text is defined using the following syntax, which is similar to footnote reference syntax, aside from the : character that is appended to the footnote label.[^1]: Numbered footnote text.\n\n[^note]:\n\n    Named footnote text containing several toplevel elements.\n\n      * item one\n      * item two\n      * item three\n\n    ```julia\n    function func(x)\n        # ...\n    end\n    ```note: Note\nNo checks are done during parsing to make sure that all footnote references have matching footnotes."
},

{
    "location": "manual/documentation.html#Horizontal-rules-1",
    "page": "Documentation",
    "title": "Horizontal rules",
    "category": "section",
    "text": "The equivalent of an <hr> HTML tag can be written using the following syntax:Text above the line.\n\n---\n\nAnd text below the line."
},

{
    "location": "manual/documentation.html#Tables-1",
    "page": "Documentation",
    "title": "Tables",
    "category": "section",
    "text": "Basic tables can be written using the syntax described below. Note that markdown tables have limited features and cannot contain nested toplevel elements unlike other elements discussed above – only inline elements are allowed. Tables must always contain a header row with column names. Cells cannot span multiple rows or columns of the table.| Column One | Column Two | Column Three |\n|:---------- | ---------- |:------------:|\n| Row `1`    | Column `2` |              |\n| *Row* 2    | **Row** 2  | Column ``3`` |note: Note\nAs illustrated in the above example each column of | characters must be aligned vertically.A : character on either end of a column's header separator (the row containing - characters) specifies whether the row is left-aligned, right-aligned, or (when : appears on both ends) center-aligned. Providing no : characters will default to right-aligning the column."
},

{
    "location": "manual/documentation.html#Admonitions-1",
    "page": "Documentation",
    "title": "Admonitions",
    "category": "section",
    "text": "Specially formatted blocks with titles such as \"Notes\", \"Warning\", or \"Tips\" are known as admonitions and are used when some part of a document needs special attention. They can be defined using the following !!! syntax:!!! note\n\n    This is the content of the note.\n\n!!! warning \"Beware!\"\n\n    And this is another one.\n\n    This warning admonition has a custom title: `\"Beware!\"`.Admonitions, like most other toplevel elements, can contain other toplevel elements. When no title text, specified after the admonition type in double quotes, is included then the title used will be the type of the block, i.e. \"Note\" in the case of the note admonition."
},

{
    "location": "manual/documentation.html#Markdown-Syntax-Extensions-1",
    "page": "Documentation",
    "title": "Markdown Syntax Extensions",
    "category": "section",
    "text": "Julia's markdown supports interpolation in a very similar way to basic string literals, with the difference that it will store the object itself in the Markdown tree (as opposed to converting it to a string). When the Markdown content is rendered the usual show methods will be called, and these can be overridden as usual. This design allows the Markdown to be extended with arbitrarily complex features (such as references) without cluttering the basic syntax.In principle, the Markdown parser itself can also be arbitrarily extended by packages, or an entirely custom flavour of Markdown can be used, but this should generally be unnecessary."
},

{
    "location": "manual/metaprogramming.html#",
    "page": "Metaprogramming",
    "title": "Metaprogramming",
    "category": "page",
    "text": ""
},

{
    "location": "manual/metaprogramming.html#Metaprogramming-1",
    "page": "Metaprogramming",
    "title": "Metaprogramming",
    "category": "section",
    "text": "The strongest legacy of Lisp in the Julia language is its metaprogramming support. Like Lisp, Julia represents its own code as a data structure of the language itself. Since code is represented by objects that can be created and manipulated from within the language, it is possible for a program to transform and generate its own code. This allows sophisticated code generation without extra build steps, and also allows true Lisp-style macros operating at the level of abstract syntax trees. In contrast, preprocessor \"macro\" systems, like that of C and C++, perform textual manipulation and substitution before any actual parsing or interpretation occurs. Because all data types and code in Julia are represented by Julia data structures, powerful reflection capabilities are available to explore the internals of a program and its types just like any other data."
},

{
    "location": "manual/metaprogramming.html#Program-representation-1",
    "page": "Metaprogramming",
    "title": "Program representation",
    "category": "section",
    "text": "Every Julia program starts life as a string:julia> prog = \"1 + 1\"\n\"1 + 1\"What happens next?The next step is to parse each string into an object called an expression, represented by the Julia type Expr:julia> ex1 = parse(prog)\n:(1 + 1)\n\njulia> typeof(ex1)\nExprExpr objects contain two parts:a Symbol identifying the kind of expression. A symbol is an interned string identifier (more discussion below).julia> ex1.head\n:callthe expression arguments, which may be symbols, other expressions, or literal values:julia> ex1.args\n3-element Array{Any,1}:\n  :+\n 1\n 1Expressions may also be constructed directly in prefix notation:julia> ex2 = Expr(:call, :+, 1, 1)\n:(1 + 1)The two expressions constructed above – by parsing and by direct construction – are equivalent:julia> ex1 == ex2\ntrueThe key point here is that Julia code is internally represented as a data structure that is accessible from the language itself.The dump() function provides indented and annotated display of Expr objects:julia> dump(ex2)\nExpr\n  head: Symbol call\n  args: Array{Any}((3,))\n    1: Symbol +\n    2: Int64 1\n    3: Int64 1\n  typ: AnyExpr objects may also be nested:julia> ex3 = parse(\"(4 + 4) / 2\")\n:((4 + 4) / 2)Another way to view expressions is with Meta.show_sexpr, which displays the S-expression form of a given Expr, which may look very familiar to users of Lisp. Here's an example illustrating the display on a nested Expr:julia> Meta.show_sexpr(ex3)\n(:call, :/, (:call, :+, 4, 4), 2)"
},

{
    "location": "manual/metaprogramming.html#Symbols-1",
    "page": "Metaprogramming",
    "title": "Symbols",
    "category": "section",
    "text": "The : character has two syntactic purposes in Julia. The first form creates a Symbol, an interned string used as one building-block of expressions:julia> :foo\n:foo\n\njulia> typeof(ans)\nSymbolThe Symbol constructor takes any number of arguments and creates a new symbol by concatenating their string representations together:julia> :foo == Symbol(\"foo\")\ntrue\n\njulia> Symbol(\"func\",10)\n:func10\n\njulia> Symbol(:var,'_',\"sym\")\n:var_symIn the context of an expression, symbols are used to indicate access to variables; when an expression is evaluated, a symbol is replaced with the value bound to that symbol in the appropriate scope.Sometimes extra parentheses around the argument to : are needed to avoid ambiguity in parsing.:julia> :(:)\n:(:)\n\njulia> :(::)\n:(::)"
},

{
    "location": "manual/metaprogramming.html#Expressions-and-evaluation-1",
    "page": "Metaprogramming",
    "title": "Expressions and evaluation",
    "category": "section",
    "text": ""
},

{
    "location": "manual/metaprogramming.html#Quoting-1",
    "page": "Metaprogramming",
    "title": "Quoting",
    "category": "section",
    "text": "The second syntactic purpose of the : character is to create expression objects without using the explicit Expr constructor. This is referred to as quoting. The : character, followed by paired parentheses around a single statement of Julia code, produces an Expr object based on the enclosed code. Here is example of the short form used to quote an arithmetic expression:julia> ex = :(a+b*c+1)\n:(a + b * c + 1)\n\njulia> typeof(ex)\nExpr(to view the structure of this expression, try ex.head and ex.args, or use dump() as above)Note that equivalent expressions may be constructed using parse() or the direct Expr form:julia>      :(a + b*c + 1)  ==\n       parse(\"a + b*c + 1\") ==\n       Expr(:call, :+, :a, Expr(:call, :*, :b, :c), 1)\ntrueExpressions provided by the parser generally only have symbols, other expressions, and literal values as their args, whereas expressions constructed by Julia code can have arbitrary run-time values without literal forms as args. In this specific example, + and a are symbols, *(b,c) is a subexpression, and 1 is a literal 64-bit signed integer.There is a second syntactic form of quoting for multiple expressions: blocks of code enclosed in quote ... end. Note that this form introduces QuoteNode elements to the expression tree, which must be considered when directly manipulating an expression tree generated from quote blocks. For other purposes, :( ... ) and quote .. end blocks are treated identically.julia> ex = quote\n           x = 1\n           y = 2\n           x + y\n       end\nquote  # none, line 2:\n    x = 1 # none, line 3:\n    y = 2 # none, line 4:\n    x + y\nend\n\njulia> typeof(ex)\nExpr"
},

{
    "location": "manual/metaprogramming.html#Interpolation-1",
    "page": "Metaprogramming",
    "title": "Interpolation",
    "category": "section",
    "text": "Direct construction of Expr objects with value arguments is powerful, but Expr constructors can be tedious compared to \"normal\" Julia syntax. As an alternative, Julia allows \"splicing\" or interpolation of literals or expressions into quoted expressions. Interpolation is indicated by the $ prefix.In this example, the literal value of a is interpolated:julia> a = 1;\n\njulia> ex = :($a + b)\n:(1 + b)Interpolating into an unquoted expression is not supported and will cause a compile-time error:julia> $a + b\nERROR: unsupported or misplaced expression $\n ...In this example, the tuple (1,2,3) is interpolated as an expression into a conditional test:julia> ex = :(a in $:((1,2,3)) )\n:(a in (1, 2, 3))Interpolating symbols into a nested expression requires enclosing each symbol in an enclosing quote block:julia> :( :a in $( :(:a + :b) ) )\n                   ^^^^^^^^^^\n                   quoted inner expressionThe use of $ for expression interpolation is intentionally reminiscent of string interpolation and command interpolation. Expression interpolation allows convenient, readable programmatic construction of complex Julia expressions."
},

{
    "location": "manual/metaprogramming.html#[eval()](@ref)-and-effects-1",
    "page": "Metaprogramming",
    "title": "eval() and effects",
    "category": "section",
    "text": "Given an expression object, one can cause Julia to evaluate (execute) it at global scope using eval():julia> :(1 + 2)\n:(1 + 2)\n\njulia> eval(ans)\n3\n\njulia> ex = :(a + b)\n:(a + b)\n\njulia> eval(ex)\nERROR: UndefVarError: b not defined\n[...]\n\njulia> a = 1; b = 2;\n\njulia> eval(ex)\n3Every module has its own eval() function that evaluates expressions in its global scope. Expressions passed to eval() are not limited to returning values – they can also have side-effects that alter the state of the enclosing module's environment:julia> ex = :(x = 1)\n:(x = 1)\n\njulia> x\nERROR: UndefVarError: x not defined\n\njulia> eval(ex)\n1\n\njulia> x\n1Here, the evaluation of an expression object causes a value to be assigned to the global variable x.Since expressions are just Expr objects which can be constructed programmatically and then evaluated, it is possible to dynamically generate arbitrary code which can then be run using eval(). Here is a simple example:julia> a = 1;\n\njulia> ex = Expr(:call, :+, a, :b)\n:(1 + b)\n\njulia> a = 0; b = 2;\n\njulia> eval(ex)\n3The value of a is used to construct the expression ex which applies the + function to the value 1 and the variable b. Note the important distinction between the way a and b are used:The value of the variablea at expression construction time is used as an immediate value in the expression. Thus, the value of a when the expression is evaluated no longer matters: the value in the expression is already 1, independent of whatever the value of a might be.\nOn the other hand, the symbol:b is used in the expression construction, so the value of the variable b at that time is irrelevant – :b is just a symbol and the variable b need not even be defined. At expression evaluation time, however, the value of the symbol :b is resolved by looking up the value of the variable b."
},

{
    "location": "manual/metaprogramming.html#Functions-on-Expressions-1",
    "page": "Metaprogramming",
    "title": "Functions on Expressions",
    "category": "section",
    "text": "As hinted above, one extremely useful feature of Julia is the capability to generate and manipulate Julia code within Julia itself. We have already seen one example of a function returning Expr objects: the parse() function, which takes a string of Julia code and returns the corresponding Expr. A function can also take one or more Expr objects as arguments, and return another Expr. Here is a simple, motivating example:julia> function math_expr(op, op1, op2)\n           expr = Expr(:call, op, op1, op2)\n           return expr\n       end\nmath_expr (generic function with 1 method)\n\njulia>  ex = math_expr(:+, 1, Expr(:call, :*, 4, 5))\n:(1 + 4 * 5)\n\njulia> eval(ex)\n21As another example, here is a function that doubles any numeric argument, but leaves expressions alone:julia> function make_expr2(op, opr1, opr2)\n           opr1f, opr2f = map(x -> isa(x, Number) ? 2*x : x, (opr1, opr2))\n           retexpr = Expr(:call, op, opr1f, opr2f)\n           return retexpr\n       end\nmake_expr2 (generic function with 1 method)\n\njulia> make_expr2(:+, 1, 2)\n:(2 + 4)\n\njulia> ex = make_expr2(:+, 1, Expr(:call, :*, 5, 8))\n:(2 + 5 * 8)\n\njulia> eval(ex)\n42"
},

{
    "location": "manual/metaprogramming.html#man-macros-1",
    "page": "Metaprogramming",
    "title": "Macros",
    "category": "section",
    "text": "Macros provide a method to include generated code in the final body of a program. A macro maps a tuple of arguments to a returned expression, and the resulting expression is compiled directly rather than requiring a runtime eval() call. Macro arguments may include expressions, literal values, and symbols."
},

{
    "location": "manual/metaprogramming.html#Basics-1",
    "page": "Metaprogramming",
    "title": "Basics",
    "category": "section",
    "text": "Here is an extraordinarily simple macro:julia> macro sayhello()\n           return :( println(\"Hello, world!\") )\n       end\n@sayhello (macro with 1 method)Macros have a dedicated character in Julia's syntax: the @ (at-sign), followed by the unique name declared in a macro NAME ... end block. In this example, the compiler will replace all instances of @sayhello with::( println(\"Hello, world!\") )When @sayhello is entered in the REPL, the expression executes immediately, thus we only see the evaluation result:julia> @sayhello()\nHello, world!Now, consider a slightly more complex macro:julia> macro sayhello(name)\n           return :( println(\"Hello, \", $name) )\n       end\n@sayhello (macro with 1 method)This macro takes one argument: name. When @sayhello is encountered, the quoted expression is expanded to interpolate the value of the argument into the final expression:julia> @sayhello(\"human\")\nHello, humanWe can view the quoted return expression using the function macroexpand() (important note: this is an extremely useful tool for debugging macros):julia> ex = macroexpand( :(@sayhello(\"human\")) )\n:((println)(\"Hello, \", \"human\"))\n\njulia> typeof(ex)\nExprWe can see that the \"human\" literal has been interpolated into the expression.There also exists a macro @macroexpand that is perhaps a bit more convenient than the macroexpand function:julia> @macroexpand @sayhello \"human\"\n:((println)(\"Hello, \", \"human\"))"
},

{
    "location": "manual/metaprogramming.html#Hold-up:-why-macros?-1",
    "page": "Metaprogramming",
    "title": "Hold up: why macros?",
    "category": "section",
    "text": "We have already seen a function f(::Expr...) -> Expr in a previous section. In fact, macroexpand() is also such a function. So, why do macros exist?Macros are necessary because they execute when code is parsed, therefore, macros allow the programmer to generate and include fragments of customized code before the full program is run. To illustrate the difference, consider the following example:julia> macro twostep(arg)\n           println(\"I execute at parse time. The argument is: \", arg)\n           return :(println(\"I execute at runtime. The argument is: \", $arg))\n       end\n@twostep (macro with 1 method)\n\njulia> ex = macroexpand( :(@twostep :(1, 2, 3)) );\nI execute at parse time. The argument is: $(Expr(:quote, :((1, 2, 3))))The first call to println() is executed when macroexpand() is called. The resulting expression contains only the second println:julia> typeof(ex)\nExpr\n\njulia> ex\n:((println)(\"I execute at runtime. The argument is: \", $(Expr(:copyast, :($(QuoteNode(:((1, 2, 3)))))))))\n\njulia> eval(ex)\nI execute at runtime. The argument is: (1, 2, 3)"
},

{
    "location": "manual/metaprogramming.html#Macro-invocation-1",
    "page": "Metaprogramming",
    "title": "Macro invocation",
    "category": "section",
    "text": "Macros are invoked with the following general syntax:@name expr1 expr2 ...\n@name(expr1, expr2, ...)Note the distinguishing @ before the macro name and the lack of commas between the argument expressions in the first form, and the lack of whitespace after @name in the second form. The two styles should not be mixed. For example, the following syntax is different from the examples above; it passes the tuple (expr1, expr2, ...) as one argument to the macro:@name (expr1, expr2, ...)It is important to emphasize that macros receive their arguments as expressions, literals, or symbols. One way to explore macro arguments is to call the show() function within the macro body:julia> macro showarg(x)\n           show(x)\n           # ... remainder of macro, returning an expression\n       end\n@showarg (macro with 1 method)\n\njulia> @showarg(a)\n:a\n\njulia> @showarg(1+1)\n:(1 + 1)\n\njulia> @showarg(println(\"Yo!\"))\n:(println(\"Yo!\"))"
},

{
    "location": "manual/metaprogramming.html#Building-an-advanced-macro-1",
    "page": "Metaprogramming",
    "title": "Building an advanced macro",
    "category": "section",
    "text": "Here is a simplified definition of Julia's @assert macro:julia> macro assert(ex)\n           return :( $ex ? nothing : throw(AssertionError($(string(ex)))) )\n       end\n@assert (macro with 1 method)This macro can be used like this:julia> @assert 1 == 1.0\n\njulia> @assert 1 == 0\nERROR: AssertionError: 1 == 0In place of the written syntax, the macro call is expanded at parse time to its returned result. This is equivalent to writing:1 == 1.0 ? nothing : throw(AssertionError(\"1 == 1.0\"))\n1 == 0 ? nothing : throw(AssertionError(\"1 == 0\"))That is, in the first call, the expression :(1 == 1.0) is spliced into the test condition slot, while the value of string(:(1 == 1.0)) is spliced into the assertion message slot. The entire expression, thus constructed, is placed into the syntax tree where the @assert macro call occurs. Then at execution time, if the test expression evaluates to true, then nothing is returned, whereas if the test is false, an error is raised indicating the asserted expression that was false. Notice that it would not be possible to write this as a function, since only the value of the condition is available and it would be impossible to display the expression that computed it in the error message.The actual definition of @assert in the standard library is more complicated. It allows the user to optionally specify their own error message, instead of just printing the failed expression. Just like in functions with a variable number of arguments, this is specified with an ellipses following the last argument:julia> macro assert(ex, msgs...)\n           msg_body = isempty(msgs) ? ex : msgs[1]\n           msg = string(msg_body)\n           return :($ex ? nothing : throw(AssertionError($msg)))\n       end\n@assert (macro with 1 method)Now @assert has two modes of operation, depending upon the number of arguments it receives! If there's only one argument, the tuple of expressions captured by msgs will be empty and it will behave the same as the simpler definition above. But now if the user specifies a second argument, it is printed in the message body instead of the failing expression. You can inspect the result of a macro expansion with the aptly named macroexpand() function:julia> macroexpand(:(@assert a == b))\n:(if a == b\n        nothing\n    else\n        (throw)((AssertionError)(\"a == b\"))\n    end)\n\njulia> macroexpand(:(@assert a==b \"a should equal b!\"))\n:(if a == b\n        nothing\n    else\n        (throw)((AssertionError)(\"a should equal b!\"))\n    end)There is yet another case that the actual @assert macro handles: what if, in addition to printing \"a should equal b,\" we wanted to print their values? One might naively try to use string interpolation in the custom message, e.g., @assert a==b \"a ($a) should equal b ($b)!\", but this won't work as expected with the above macro. Can you see why? Recall from string interpolation that an interpolated string is rewritten to a call to string(). Compare:julia> typeof(:(\"a should equal b\"))\nString\n\njulia> typeof(:(\"a ($a) should equal b ($b)!\"))\nExpr\n\njulia> dump(:(\"a ($a) should equal b ($b)!\"))\nExpr\n  head: Symbol string\n  args: Array{Any}((5,))\n    1: String \"a (\"\n    2: Symbol a\n    3: String \") should equal b (\"\n    4: Symbol b\n    5: String \")!\"\n  typ: AnySo now instead of getting a plain string in msg_body, the macro is receiving a full expression that will need to be evaluated in order to display as expected. This can be spliced directly into the returned expression as an argument to the string() call; see error.jl for the complete implementation.The @assert macro makes great use of splicing into quoted expressions to simplify the manipulation of expressions inside the macro body."
},

{
    "location": "manual/metaprogramming.html#Hygiene-1",
    "page": "Metaprogramming",
    "title": "Hygiene",
    "category": "section",
    "text": "An issue that arises in more complex macros is that of hygiene. In short, macros must ensure that the variables they introduce in their returned expressions do not accidentally clash with existing variables in the surrounding code they expand into. Conversely, the expressions that are passed into a macro as arguments are often expected to evaluate in the context of the surrounding code, interacting with and modifying the existing variables. Another concern arises from the fact that a macro may be called in a different module from where it was defined. In this case we need to ensure that all global variables are resolved to the correct module. Julia already has a major advantage over languages with textual macro expansion (like C) in that it only needs to consider the returned expression. All the other variables (such as msg in @assert above) follow the normal scoping block behavior.To demonstrate these issues, let us consider writing a @time macro that takes an expression as its argument, records the time, evaluates the expression, records the time again, prints the difference between the before and after times, and then has the value of the expression as its final value. The macro might look like this:macro time(ex)\n    return quote\n        local t0 = time()\n        local val = $ex\n        local t1 = time()\n        println(\"elapsed time: \", t1-t0, \" seconds\")\n        val\n    end\nendHere, we want t0, t1, and val to be private temporary variables, and we want time to refer to the time() function in the standard library, not to any time variable the user might have (the same applies to println). Imagine the problems that could occur if the user expression ex also contained assignments to a variable called t0, or defined its own time variable. We might get errors, or mysteriously incorrect behavior.Julia's macro expander solves these problems in the following way. First, variables within a macro result are classified as either local or global. A variable is considered local if it is assigned to (and not declared global), declared local, or used as a function argument name. Otherwise, it is considered global. Local variables are then renamed to be unique (using the gensym() function, which generates new symbols), and global variables are resolved within the macro definition environment. Therefore both of the above concerns are handled; the macro's locals will not conflict with any user variables, and time and println will refer to the standard library definitions.One problem remains however. Consider the following use of this macro:module MyModule\nimport Base.@time\n\ntime() = ... # compute something\n\n@time time()\nendHere the user expression ex is a call to time, but not the same time function that the macro uses. It clearly refers to MyModule.time. Therefore we must arrange for the code in ex to be resolved in the macro call environment. This is done by \"escaping\" the expression with esc():macro time(ex)\n    ...\n    local val = $(esc(ex))\n    ...\nendAn expression wrapped in this manner is left alone by the macro expander and simply pasted into the output verbatim. Therefore it will be resolved in the macro call environment.This escaping mechanism can be used to \"violate\" hygiene when necessary, in order to introduce or manipulate user variables. For example, the following macro sets x to zero in the call environment:julia> macro zerox()\n           return esc(:(x = 0))\n       end\n@zerox (macro with 1 method)\n\njulia> function foo()\n           x = 1\n           @zerox\n           return x # is zero\n       end\nfoo (generic function with 1 method)\n\njulia> foo()\n0This kind of manipulation of variables should be used judiciously, but is occasionally quite handy."
},

{
    "location": "manual/metaprogramming.html#Code-Generation-1",
    "page": "Metaprogramming",
    "title": "Code Generation",
    "category": "section",
    "text": "When a significant amount of repetitive boilerplate code is required, it is common to generate it programmatically to avoid redundancy. In most languages, this requires an extra build step, and a separate program to generate the repetitive code. In Julia, expression interpolation and eval() allow such code generation to take place in the normal course of program execution. For example, the following code defines a series of operators on three arguments in terms of their 2-argument forms:for op = (:+, :*, :&, :|, :$)\n    eval(quote\n        ($op)(a,b,c) = ($op)(($op)(a,b),c)\n    end)\nendIn this manner, Julia acts as its own preprocessor, and allows code generation from inside the language. The above code could be written slightly more tersely using the : prefix quoting form:for op = (:+, :*, :&, :|, :$)\n    eval(:(($op)(a,b,c) = ($op)(($op)(a,b),c)))\nendThis sort of in-language code generation, however, using the eval(quote(...)) pattern, is common enough that Julia comes with a macro to abbreviate this pattern:for op = (:+, :*, :&, :|, :$)\n    @eval ($op)(a,b,c) = ($op)(($op)(a,b),c)\nendThe @eval macro rewrites this call to be precisely equivalent to the above longer versions. For longer blocks of generated code, the expression argument given to @eval can be a block:@eval begin\n    # multiple lines\nend"
},

{
    "location": "manual/metaprogramming.html#Non-Standard-String-Literals-1",
    "page": "Metaprogramming",
    "title": "Non-Standard String Literals",
    "category": "section",
    "text": "Recall from Strings that string literals prefixed by an identifier are called non-standard string literals, and can have different semantics than un-prefixed string literals. For example:r\"^\\s*(?:#|$)\" produces a regular expression object rather than a string\nb\"DATA\\xff\\u2200\" is a byte array literal for [68,65,84,65,255,226,136,128].Perhaps surprisingly, these behaviors are not hard-coded into the Julia parser or compiler. Instead, they are custom behaviors provided by a general mechanism that anyone can use: prefixed string literals are parsed as calls to specially-named macros. For example, the regular expression macro is just the following:macro r_str(p)\n    Regex(p)\nendThat's all. This macro says that the literal contents of the string literal r\"^\\s*(?:#|$)\" should be passed to the @r_str macro and the result of that expansion should be placed in the syntax tree where the string literal occurs. In other words, the expression r\"^\\s*(?:#|$)\" is equivalent to placing the following object directly into the syntax tree:Regex(\"^\\\\s*(?:#|\\$)\")Not only is the string literal form shorter and far more convenient, but it is also more efficient: since the regular expression is compiled and the Regex object is actually created when the code is compiled, the compilation occurs only once, rather than every time the code is executed. Consider if the regular expression occurs in a loop:for line = lines\n    m = match(r\"^\\s*(?:#|$)\", line)\n    if m === nothing\n        # non-comment\n    else\n        # comment\n    end\nendSince the regular expression r\"^\\s*(?:#|$)\" is compiled and inserted into the syntax tree when this code is parsed, the expression is only compiled once instead of each time the loop is executed. In order to accomplish this without macros, one would have to write this loop like this:re = Regex(\"^\\\\s*(?:#|\\$)\")\nfor line = lines\n    m = match(re, line)\n    if m === nothing\n        # non-comment\n    else\n        # comment\n    end\nendMoreover, if the compiler could not determine that the regex object was constant over all loops, certain optimizations might not be possible, making this version still less efficient than the more convenient literal form above. Of course, there are still situations where the non-literal form is more convenient: if one needs to interpolate a variable into the regular expression, one must take this more verbose approach; in cases where the regular expression pattern itself is dynamic, potentially changing upon each loop iteration, a new regular expression object must be constructed on each iteration. In the vast majority of use cases, however, regular expressions are not constructed based on run-time data. In this majority of cases, the ability to write regular expressions as compile-time values is invaluable.Like non-standard string literals, non-standard command literals exist using a prefixed variant of the command literal syntax. The command literal custom`literal` is parsed as @custom_cmd \"literal\". Julia itself does not contain any non-standard command literals, but packages can make use of this syntax. Aside from the different syntax and the _cmd suffix instead of the _str suffix, non-standard command literals behave exactly like non-standard string literals.In the event that two modules provide non-standard string or command literals with the same name, it is possible to qualify the string or command literal with a module name. For instance, if both Foo and Bar provide non-standard string literal @x_str, then one can write Foo.x\"literal\" or Bar.x\"literal\" to disambiguate between the two.The mechanism for user-defined string literals is deeply, profoundly powerful. Not only are Julia's non-standard literals implemented using it, but also the command literal syntax (`echo \"Hello, $person\"`) is implemented with the following innocuous-looking macro:macro cmd(str)\n    :(cmd_gen($(shell_parse(str)[1])))\nendOf course, a large amount of complexity is hidden in the functions used in this macro definition, but they are just functions, written entirely in Julia. You can read their source and see precisely what they do – and all they do is construct expression objects to be inserted into your program's syntax tree."
},

{
    "location": "manual/metaprogramming.html#Generated-functions-1",
    "page": "Metaprogramming",
    "title": "Generated functions",
    "category": "section",
    "text": "A very special macro is @generated, which allows you to define so-called generated functions. These have the capability to generate specialized code depending on the types of their arguments with more flexibility and/or less code than what can be achieved with multiple dispatch. While macros work with expressions at parsing-time and cannot access the types of their inputs, a generated function gets expanded at a time when the types of the arguments are known, but the function is not yet compiled.Instead of performing some calculation or action, a generated function declaration returns a quoted expression which then forms the body for the method corresponding to the types of the arguments. When called, the body expression is first evaluated and compiled, then the returned expression is compiled and run. To make this efficient, the result is often cached. And to make this inferable, only a limited subset of the language is usable. Thus, generated functions provide a flexible framework to move work from run-time to compile-time, at the expense of greater restrictions on the allowable constructs.When defining generated functions, there are four main differences to ordinary functions:You annotate the function declaration with the @generated macro. This adds some information to the AST that lets the compiler know that this is a generated function.\nIn the body of the generated function you only have access to the types of the arguments – not their values – and any function that was defined before the definition of the generated function.\nInstead of calculating something or performing some action, you return a quoted expression which, when evaluated, does what you want.\nGenerated functions must not mutate or observe any non-constant global state (including, for example, IO, locks, non-local dictionaries, or using method_exists). This means they can only read global constants, and cannot have any side effects. In other words, they must be completely pure. Due to an implementation limitation, this also means that they currently cannot define a closure or untyped generator.It's easiest to illustrate this with an example. We can declare a generated function foo asjulia> @generated function foo(x)\n           Core.println(x)\n           return :(x * x)\n       end\nfoo (generic function with 1 method)Note that the body returns a quoted expression, namely :(x * x), rather than just the value of x * x.From the caller's perspective, they are very similar to regular functions; in fact, you don't have to know if you're calling a regular or generated function - the syntax and result of the call is just the same. Let's see how foo behaves:julia> x = foo(2); # note: output is from println() statement in the body\nInt64\n\njulia> x           # now we print x\n4\n\njulia> y = foo(\"bar\");\nString\n\njulia> y\n\"barbar\"So, we see that in the body of the generated function, x is the type of the passed argument, and the value returned by the generated function, is the result of evaluating the quoted expression we returned from the definition, now with the value of x.What happens if we evaluate foo again with a type that we have already used?julia> foo(4)\n16Note that there is no printout of Int64. We can see that the body of the generated function was only executed once here, for the specific set of argument types, and the result was cached. After that, for this example, the expression returned from the generated function on the first invocation was re-used as the method body. However, the actual caching behavior is an implementation-defined performance optimization, so it is invalid to depend too closely on this behavior.The number of times a generated function is generated might be only once, but it might also be more often, or appear to not happen at all. As a consequence, you should never write a generated function with side effects - when, and how often, the side effects occur is undefined. (This is true for macros too - and just like for macros, the use of eval() in a generated function is a sign that you're doing something the wrong way.) However, unlike macros, the runtime system cannot correctly handle a call to eval(), so it is disallowed.It is also important to see how @generated functions interact with method redefinition. Following the principle that a correct @generated function must not observe any mutable state or cause any mutation of global state, we see the following behavior. Observe that the generated function cannot call any method that was not defined prior to the definition of the generated function itself.Initially f(x) has one definitionjulia> f(x) = \"original definition\";Define other operations that use f(x):julia> g(x) = f(x);\n\njulia> @generated gen1(x) = f(x);\n\njulia> @generated gen2(x) = :(f(x));We now add some new definitions for f(x):julia> f(x::Int) = \"definition for Int\";\n\njulia> f(x::Type{Int}) = \"definition for Type{Int}\";and compare how these results differ:julia> f(1)\n\"definition for Int\"\n\njulia> g(1)\n\"definition for Int\"\n\njulia> gen1(1)\n\"original definition\"\n\njulia> gen2(1)\n\"definition for Int\"Each method of a generated function has its own view of defined functions:julia> @generated gen1(x::Real) = f(x);\n\njulia> gen1(1)\n\"definition for Type{Int}\"The example generated function foo above did not do anything a normal function foo(x) = x * x could not do (except printing the type on the first invocation, and incurring higher overhead). However, the power of a generated function lies in its ability to compute different quoted expressions depending on the types passed to it:julia> @generated function bar(x)\n           if x <: Integer\n               return :(x ^ 2)\n           else\n               return :(x)\n           end\n       end\nbar (generic function with 1 method)\n\njulia> bar(4)\n16\n\njulia> bar(\"baz\")\n\"baz\"(although of course this contrived example would be more easily implemented using multiple dispatch...)Abusing this will corrupt the runtime system and cause undefined behavior:julia> @generated function baz(x)\n           if rand() < .9\n               return :(x^2)\n           else\n               return :(\"boo!\")\n           end\n       end\nbaz (generic function with 1 method)Since the body of the generated function is non-deterministic, its behavior, and the behavior of all subsequent code is undefined.Don't copy these examples!These examples are hopefully helpful to illustrate how generated functions work, both in the definition end and at the call site; however, don't copy them, for the following reasons:the foo function has side-effects (the call to Core.println), and it is undefined exactly when, how often or how many times these side-effects will occur\nthe bar function solves a problem that is better solved with multiple dispatch - defining bar(x) = x and bar(x::Integer) = x ^ 2 will do the same thing, but it is both simpler and faster.\nthe baz function is pathologically insaneNote that the set of operations that should not be attempted in a generated function is unbounded, and the runtime system can currently only detect a subset of the invalid operations. There are many other operations that will simply corrupt the runtime system without notification, usually in subtle ways not obviously connected to the bad definition. Because the function generator is run during inference, it must respect all of the limitations of that code.Some operations that should not be attempted include:Caching of native pointers.\nInteracting with the contents or methods of Core.Inference in any way.\nObserving any mutable state.\nInference on the generated function may be run at any time, including while your code is attempting to observe or mutate this state.\nTaking any locks: C code you call out to may use locks internally, (for example, it is not problematic to call malloc, even though most implementations require locks internally) but don't attempt to hold or acquire any while executing Julia code.\nCalling any function that is defined after the body of the generated function. This condition is relaxed for incrementally-loaded precompiled modules to allow calling any function in the module.Alright, now that we have a better understanding of how generated functions work, let's use them to build some more advanced (and valid) functionality..."
},

{
    "location": "manual/metaprogramming.html#An-advanced-example-1",
    "page": "Metaprogramming",
    "title": "An advanced example",
    "category": "section",
    "text": "Julia's base library has a sub2ind() function to calculate a linear index into an n-dimensional array, based on a set of n multilinear indices - in other words, to calculate the index i that can be used to index into an array A using A[i], instead of A[x,y,z,...]. One possible implementation is the following:julia> function sub2ind_loop(dims::NTuple{N}, I::Integer...) where N\n           ind = I[N] - 1\n           for i = N-1:-1:1\n               ind = I[i]-1 + dims[i]*ind\n           end\n           return ind + 1\n       end\nsub2ind_loop (generic function with 1 method)\n\njulia> sub2ind_loop((3, 5), 1, 2)\n4The same thing can be done using recursion:julia> sub2ind_rec(dims::Tuple{}) = 1;\n\njulia> sub2ind_rec(dims::Tuple{}, i1::Integer, I::Integer...) =\n           i1 == 1 ? sub2ind_rec(dims, I...) : throw(BoundsError());\n\njulia> sub2ind_rec(dims::Tuple{Integer, Vararg{Integer}}, i1::Integer) = i1;\n\njulia> sub2ind_rec(dims::Tuple{Integer, Vararg{Integer}}, i1::Integer, I::Integer...) =\n           i1 + dims[1] * (sub2ind_rec(Base.tail(dims), I...) - 1);\n\njulia> sub2ind_rec((3, 5), 1, 2)\n4Both these implementations, although different, do essentially the same thing: a runtime loop over the dimensions of the array, collecting the offset in each dimension into the final index.However, all the information we need for the loop is embedded in the type information of the arguments. Thus, we can utilize generated functions to move the iteration to compile-time; in compiler parlance, we use generated functions to manually unroll the loop. The body becomes almost identical, but instead of calculating the linear index, we build up an expression that calculates the index:julia> @generated function sub2ind_gen(dims::NTuple{N}, I::Integer...) where N\n           ex = :(I[$N] - 1)\n           for i = (N - 1):-1:1\n               ex = :(I[$i] - 1 + dims[$i] * $ex)\n           end\n           return :($ex + 1)\n       end\nsub2ind_gen (generic function with 1 method)\n\njulia> sub2ind_gen((3, 5), 1, 2)\n4What code will this generate?An easy way to find out is to extract the body into another (regular) function:julia> @generated function sub2ind_gen(dims::NTuple{N}, I::Integer...) where N\n           return sub2ind_gen_impl(dims, I...)\n       end\nsub2ind_gen (generic function with 1 method)\n\njulia> function sub2ind_gen_impl(dims::Type{T}, I...) where T <: NTuple{N,Any} where N\n           length(I) == N || return :(error(\"partial indexing is unsupported\"))\n           ex = :(I[$N] - 1)\n           for i = (N - 1):-1:1\n               ex = :(I[$i] - 1 + dims[$i] * $ex)\n           end\n           return :($ex + 1)\n       end\nsub2ind_gen_impl (generic function with 1 method)We can now execute sub2ind_gen_impl and examine the expression it returns:julia> sub2ind_gen_impl(Tuple{Int,Int}, Int, Int)\n:(((I[1] - 1) + dims[1] * (I[2] - 1)) + 1)So, the method body that will be used here doesn't include a loop at all - just indexing into the two tuples, multiplication and addition/subtraction. All the looping is performed compile-time, and we avoid looping during execution entirely. Thus, we only loop once per type, in this case once per N (except in edge cases where the function is generated more than once - see disclaimer above)."
},

{
    "location": "manual/arrays.html#",
    "page": "Multi-dimensional Arrays",
    "title": "Multi-dimensional Arrays",
    "category": "page",
    "text": ""
},

{
    "location": "manual/arrays.html#man-multi-dim-arrays-1",
    "page": "Multi-dimensional Arrays",
    "title": "Multi-dimensional Arrays",
    "category": "section",
    "text": "Julia, like most technical computing languages, provides a first-class array implementation. Most technical computing languages pay a lot of attention to their array implementation at the expense of other containers. Julia does not treat arrays in any special way. The array library is implemented almost completely in Julia itself, and derives its performance from the compiler, just like any other code written in Julia. As such, it's also possible to define custom array types by inheriting from AbstractArray. See the manual section on the AbstractArray interface for more details on implementing a custom array type.An array is a collection of objects stored in a multi-dimensional grid. In the most general case, an array may contain objects of type Any. For most computational purposes, arrays should contain objects of a more specific type, such as Float64 or Int32.In general, unlike many other technical computing languages, Julia does not expect programs to be written in a vectorized style for performance. Julia's compiler uses type inference and generates optimized code for scalar array indexing, allowing programs to be written in a style that is convenient and readable, without sacrificing performance, and using less memory at times.In Julia, all arguments to functions are passed by reference. Some technical computing languages pass arrays by value, and this is convenient in many cases. In Julia, modifications made to input arrays within a function will be visible in the parent function. The entire Julia array library ensures that inputs are not modified by library functions. User code, if it needs to exhibit similar behavior, should take care to create a copy of inputs that it may modify."
},

{
    "location": "manual/arrays.html#Arrays-1",
    "page": "Multi-dimensional Arrays",
    "title": "Arrays",
    "category": "section",
    "text": ""
},

{
    "location": "manual/arrays.html#Basic-Functions-1",
    "page": "Multi-dimensional Arrays",
    "title": "Basic Functions",
    "category": "section",
    "text": "Function Description\neltype(A) the type of the elements contained in A\nlength(A) the number of elements in A\nndims(A) the number of dimensions of A\nsize(A) a tuple containing the dimensions of A\nsize(A,n) the size of A along dimension n\nindices(A) a tuple containing the valid indices of A\nindices(A,n) a range expressing the valid indices along dimension n\neachindex(A) an efficient iterator for visiting each position in A\nstride(A,k) the stride (linear index distance between adjacent elements) along dimension k\nstrides(A) a tuple of the strides in each dimension"
},

{
    "location": "manual/arrays.html#Construction-and-Initialization-1",
    "page": "Multi-dimensional Arrays",
    "title": "Construction and Initialization",
    "category": "section",
    "text": "Many functions for constructing and initializing arrays are provided. In the following list of such functions, calls with a dims... argument can either take a single tuple of dimension sizes or a series of dimension sizes passed as a variable number of arguments. Most of these functions also accept a first input T, which is the element type of the array. If the type T is omitted it will default to Float64.Function Description\nArray{T}(dims...) an uninitialized dense Array\nzeros(T, dims...) an Array of all zeros\nzeros(A) an array of all zeros with the same type, element type and shape as A\nones(T, dims...) an Array of all ones\nones(A) an array of all ones with the same type, element type and shape as A\ntrues(dims...) a BitArray with all values true\ntrues(A) a BitArray with all values true and the same shape as A\nfalses(dims...) a BitArray with all values false\nfalses(A) a BitArray with all values false and the same shape as A\nreshape(A, dims...) an array containing the same data as A, but with different dimensions\ncopy(A) copy A\ndeepcopy(A) copy A, recursively copying its elements\nsimilar(A, T, dims...) an uninitialized array of the same type as A (dense, sparse, etc.), but with the specified element type and dimensions. The second and third arguments are both optional, defaulting to the element type and dimensions of A if omitted.\nreinterpret(T, A) an array with the same binary data as A, but with element type T\nrand(T, dims...) an Array with random, iid [1] and uniformly distributed values in the half-open interval 0 1)\nrandn(T, dims...) an Array with random, iid and standard normally distributed values\neye(T, n) n-by-n identity matrix\neye(T, m, n) m-by-n identity matrix\nlinspace(start, stop, n) range of n linearly spaced elements from start to stop\nfill!(A, x) fill the array A with the value x\nfill(x, dims...) an Array filled with the value x[1]: iid, independently and identically distributed.The syntax [A, B, C, ...] constructs a 1-d array (vector) of its arguments. If all arguments have a common promotion type then they get converted to that type using convert()."
},

{
    "location": "manual/arrays.html#Concatenation-1",
    "page": "Multi-dimensional Arrays",
    "title": "Concatenation",
    "category": "section",
    "text": "Arrays can be constructed and also concatenated using the following functions:Function Description\ncat(k, A...) concatenate input n-d arrays along the dimension k\nvcat(A...) shorthand for cat(1, A...)\nhcat(A...) shorthand for cat(2, A...)Scalar values passed to these functions are treated as 1-element arrays.The concatenation functions are used so often that they have special syntax:Expression Calls\n[A; B; C; ...] vcat()\n[A B C ...] hcat()\n[A B; C D; ...] hvcat()hvcat() concatenates in both dimension 1 (with semicolons) and dimension 2 (with spaces)."
},

{
    "location": "manual/arrays.html#Typed-array-initializers-1",
    "page": "Multi-dimensional Arrays",
    "title": "Typed array initializers",
    "category": "section",
    "text": "An array with a specific element type can be constructed using the syntax T[A, B, C, ...]. This will construct a 1-d array with element type T, initialized to contain elements A, B, C, etc. For example Any[x, y, z] constructs a heterogeneous array that can contain any values.Concatenation syntax can similarly be prefixed with a type to specify the element type of the result.julia> [[1 2] [3 4]]\n1×4 Array{Int64,2}:\n 1  2  3  4\n\njulia> Int8[[1 2] [3 4]]\n1×4 Array{Int8,2}:\n 1  2  3  4"
},

{
    "location": "manual/arrays.html#Comprehensions-1",
    "page": "Multi-dimensional Arrays",
    "title": "Comprehensions",
    "category": "section",
    "text": "Comprehensions provide a general and powerful way to construct arrays. Comprehension syntax is similar to set construction notation in mathematics:A = [ F(x,y,...) for x=rx, y=ry, ... ]The meaning of this form is that F(x,y,...) is evaluated with the variables x, y, etc. taking on each value in their given list of values. Values can be specified as any iterable object, but will commonly be ranges like 1:n or 2:(n-1), or explicit arrays of values like [1.2, 3.4, 5.7]. The result is an N-d dense array with dimensions that are the concatenation of the dimensions of the variable ranges rx, ry, etc. and each F(x,y,...) evaluation returns a scalar.The following example computes a weighted average of the current element and its left and right neighbor along a 1-d grid. :julia> x = rand(8)\n8-element Array{Float64,1}:\n 0.843025\n 0.869052\n 0.365105\n 0.699456\n 0.977653\n 0.994953\n 0.41084\n 0.809411\n\njulia> [ 0.25*x[i-1] + 0.5*x[i] + 0.25*x[i+1] for i=2:length(x)-1 ]\n6-element Array{Float64,1}:\n 0.736559\n 0.57468\n 0.685417\n 0.912429\n 0.8446\n 0.656511The resulting array type depends on the types of the computed elements. In order to control the type explicitly, a type can be prepended to the comprehension. For example, we could have requested the result in single precision by writing:Float32[ 0.25*x[i-1] + 0.5*x[i] + 0.25*x[i+1] for i=2:length(x)-1 ]"
},

{
    "location": "manual/arrays.html#Generator-Expressions-1",
    "page": "Multi-dimensional Arrays",
    "title": "Generator Expressions",
    "category": "section",
    "text": "Comprehensions can also be written without the enclosing square brackets, producing an object known as a generator. This object can be iterated to produce values on demand, instead of allocating an array and storing them in advance (see Iteration). For example, the following expression sums a series without allocating memory:julia> sum(1/n^2 for n=1:1000)\n1.6439345666815615When writing a generator expression with multiple dimensions inside an argument list, parentheses are needed to separate the generator from subsequent arguments:julia> map(tuple, 1/(i+j) for i=1:2, j=1:2, [1:4;])\nERROR: syntax: invalid iteration specificationAll comma-separated expressions after for are interpreted as ranges. Adding parentheses lets us add a third argument to map:julia> map(tuple, (1/(i+j) for i=1:2, j=1:2), [1 3; 2 4])\n2×2 Array{Tuple{Float64,Int64},2}:\n (0.5, 1)       (0.333333, 3)\n (0.333333, 2)  (0.25, 4)Ranges in generators and comprehensions can depend on previous ranges by writing multiple for keywords:julia> [(i,j) for i=1:3 for j=1:i]\n6-element Array{Tuple{Int64,Int64},1}:\n (1, 1)\n (2, 1)\n (2, 2)\n (3, 1)\n (3, 2)\n (3, 3)In such cases, the result is always 1-d.Generated values can be filtered using the if keyword:julia> [(i,j) for i=1:3 for j=1:i if i+j == 4]\n2-element Array{Tuple{Int64,Int64},1}:\n (2, 2)\n (3, 1)"
},

{
    "location": "manual/arrays.html#man-array-indexing-1",
    "page": "Multi-dimensional Arrays",
    "title": "Indexing",
    "category": "section",
    "text": "The general syntax for indexing into an n-dimensional array A is:X = A[I_1, I_2, ..., I_n]where each I_k may be a scalar integer, an array of integers, or any other supported index. This includes Colon (:) to select all indices within the entire dimension, ranges of the form a:c or a:b:c to select contiguous or strided subsections, and arrays of booleans to select elements at their true indices.If all the indices are scalars, then the result X is a single element from the array A. Otherwise, X is an array with the same number of dimensions as the sum of the dimensionalities of all the indices.If all indices are vectors, for example, then the shape of X would be (length(I_1), length(I_2), ..., length(I_n)), with location (i_1, i_2, ..., i_n) of X containing the value A[I_1[i_1], I_2[i_2], ..., I_n[i_n]]. If I_1 is changed to a two-dimensional matrix, then X becomes an n+1-dimensional array of shape (size(I_1, 1), size(I_1, 2), length(I_2), ..., length(I_n)). The matrix adds a dimension. The location (i_1, i_2, i_3, ..., i_{n+1}) contains the value at A[I_1[i_1, i_2], I_2[i_3], ..., I_n[i_{n+1}]]. All dimensions indexed with scalars are dropped. For example, the result of A[2, I, 3] is an array with size size(I). Its ith element is populated by A[2, I[i], 3].As a special part of this syntax, the end keyword may be used to represent the last index of each dimension within the indexing brackets, as determined by the size of the innermost array being indexed. Indexing syntax without the end keyword is equivalent to a call to getindex:X = getindex(A, I_1, I_2, ..., I_n)Example:julia> x = reshape(1:16, 4, 4)\n4×4 Base.ReshapedArray{Int64,2,UnitRange{Int64},Tuple{}}:\n 1  5   9  13\n 2  6  10  14\n 3  7  11  15\n 4  8  12  16\n\njulia> x[2:3, 2:end-1]\n2×2 Array{Int64,2}:\n 6  10\n 7  11\n\njulia> x[1, [2 3; 4 1]]\n2×2 Array{Int64,2}:\n  5  9\n 13  1Empty ranges of the form n:n-1 are sometimes used to indicate the inter-index location between n-1 and n. For example, the searchsorted() function uses this convention to indicate the insertion point of a value not found in a sorted array:julia> a = [1,2,5,6,7];\n\njulia> searchsorted(a, 3)\n3:2"
},

{
    "location": "manual/arrays.html#Assignment-1",
    "page": "Multi-dimensional Arrays",
    "title": "Assignment",
    "category": "section",
    "text": "The general syntax for assigning values in an n-dimensional array A is:A[I_1, I_2, ..., I_n] = Xwhere each I_k may be a scalar integer, an array of integers, or any other supported index. This includes Colon (:) to select all indices within the entire dimension, ranges of the form a:c or a:b:c to select contiguous or strided subsections, and arrays of booleans to select elements at their true indices.If X is an array, it must have the same number of elements as the product of the lengths of the indices: prod(length(I_1), length(I_2), ..., length(I_n)). The value in location I_1[i_1], I_2[i_2], ..., I_n[i_n] of A is overwritten with the value X[i_1, i_2, ..., i_n]. If X is not an array, its value is written to all referenced locations of A.Just as in Indexing, the end keyword may be used to represent the last index of each dimension within the indexing brackets, as determined by the size of the array being assigned into. Indexed assignment syntax without the end keyword is equivalent to a call to setindex!():setindex!(A, X, I_1, I_2, ..., I_n)Example:julia> x = collect(reshape(1:9, 3, 3))\n3×3 Array{Int64,2}:\n 1  4  7\n 2  5  8\n 3  6  9\n\njulia> x[1:2, 2:3] = -1\n-1\n\njulia> x\n3×3 Array{Int64,2}:\n 1  -1  -1\n 2  -1  -1\n 3   6   9"
},

{
    "location": "manual/arrays.html#man-supported-index-types-1",
    "page": "Multi-dimensional Arrays",
    "title": "Supported index types",
    "category": "section",
    "text": "In the expression A[I_1, I_2, ..., I_n], each I_k may be a scalar index, an array of scalar indices, or an object that represents an array of scalar indices and can be converted to such by to_indices:A scalar index. By default this includes:\nNon-boolean integers\nCartesianIndex{N}s, which behave like an N-tuple of integers spanning multiple dimensions (see below for more details)\nAn array of scalar indices. This includes:\nVectors and multidimensional arrays of integers\nEmpty arrays like [], which select no elements\nRanges of the form a:c or a:b:c, which select contiguous or strided subsections from a to c (inclusive)\nAny custom array of scalar indices that is a subtype of AbstractArray\nArrays of CartesianIndex{N} (see below for more details)\nAn object that represents an array of scalar indices and can be converted to such by to_indices. By default this includes:\nColon() (:), which represents all indices within an entire dimension or across the entire array\nArrays of booleans, which select elements at their true indices (see below for more details)"
},

{
    "location": "manual/arrays.html#Cartesian-indices-1",
    "page": "Multi-dimensional Arrays",
    "title": "Cartesian indices",
    "category": "section",
    "text": "The special CartesianIndex{N} object represents a scalar index that behaves like an N-tuple of integers spanning multiple dimensions.  For example:julia> A = reshape(1:32, 4, 4, 2);\n\njulia> A[3, 2, 1]\n7\n\njulia> A[CartesianIndex(3, 2, 1)] == A[3, 2, 1] == 7\ntrueConsidered alone, this may seem relatively trivial; CartesianIndex simply gathers multiple integers together into one object that represents a single multidimensional index. When combined with other indexing forms and iterators that yield CartesianIndexes, however, this can lead directly to very elegant and efficient code. See Iteration below, and for some more advanced examples, see this blog post on multidimensional algorithms and iteration.Arrays of CartesianIndex{N} are also supported. They represent a collection of scalar indices that each span N dimensions, enabling a form of indexing that is sometimes referred to as pointwise indexing. For example, it enables accessing the diagonal elements from the first \"page\" of A from above:julia> page = A[:,:,1]\n4×4 Array{Int64,2}:\n 1  5   9  13\n 2  6  10  14\n 3  7  11  15\n 4  8  12  16\n\njulia> page[[CartesianIndex(1,1),\n             CartesianIndex(2,2),\n             CartesianIndex(3,3),\n             CartesianIndex(4,4)]]\n4-element Array{Int64,1}:\n  1\n  6\n 11\n 16This can be expressed much more simply with dot broadcasting and by combining it with a normal integer index (instead of extracting the first page from A as a separate step). It can even be combined with a : to extract both diagonals from the two pages at the same time:julia> A[CartesianIndex.(indices(A, 1), indices(A, 2)), 1]\n4-element Array{Int64,1}:\n  1\n  6\n 11\n 16\n\njulia> A[CartesianIndex.(indices(A, 1), indices(A, 2)), :]\n4×2 Array{Int64,2}:\n  1  17\n  6  22\n 11  27\n 16  32warning: Warning\nCartesianIndex and arrays of CartesianIndex are not compatible with the end keyword to represent the last index of a dimension. Do not use end in indexing expressions that may contain either CartesianIndex or arrays thereof."
},

{
    "location": "manual/arrays.html#Logical-indexing-1",
    "page": "Multi-dimensional Arrays",
    "title": "Logical indexing",
    "category": "section",
    "text": "Often referred to as logical indexing or indexing with a logical mask, indexing by a boolean array selects elements at the indices where its values are true. Indexing by a boolean vector B is effectively the same as indexing by the vector of integers that is returned by find(B). Similarly, indexing by a N-dimensional boolean array is effectively the same as indexing by the vector of CartesianIndex{N}s where its values are true. A logical index must be a vector of the same length as the dimension it indexes into, or it must be the only index provided and match the size and dimensionality of the array it indexes into. It is generally more efficient to use boolean arrays as indices directly instead of first calling find().julia> x = reshape(1:16, 4, 4)\n4×4 Base.ReshapedArray{Int64,2,UnitRange{Int64},Tuple{}}:\n 1  5   9  13\n 2  6  10  14\n 3  7  11  15\n 4  8  12  16\n\njulia> x[[false, true, true, false], :]\n2×4 Array{Int64,2}:\n 2  6  10  14\n 3  7  11  15\n\njulia> mask = map(ispow2, x)\n4×4 Array{Bool,2}:\n  true  false  false  false\n  true  false  false  false\n false  false  false  false\n  true   true  false   true\n\njulia> x[mask]\n5-element Array{Int64,1}:\n  1\n  2\n  4\n  8\n 16"
},

{
    "location": "manual/arrays.html#Iteration-1",
    "page": "Multi-dimensional Arrays",
    "title": "Iteration",
    "category": "section",
    "text": "The recommended ways to iterate over a whole array arefor a in A\n    # Do something with the element a\nend\n\nfor i in eachindex(A)\n    # Do something with i and/or A[i]\nendThe first construct is used when you need the value, but not index, of each element. In the second construct, i will be an Int if A is an array type with fast linear indexing; otherwise, it will be a CartesianIndex:julia> A = rand(4,3);\n\njulia> B = view(A, 1:3, 2:3);\n\njulia> for i in eachindex(B)\n           @show i\n       end\ni = CartesianIndex{2}((1, 1))\ni = CartesianIndex{2}((2, 1))\ni = CartesianIndex{2}((3, 1))\ni = CartesianIndex{2}((1, 2))\ni = CartesianIndex{2}((2, 2))\ni = CartesianIndex{2}((3, 2))In contrast with for i = 1:length(A), iterating with eachindex provides an efficient way to iterate over any array type."
},

{
    "location": "manual/arrays.html#Array-traits-1",
    "page": "Multi-dimensional Arrays",
    "title": "Array traits",
    "category": "section",
    "text": "If you write a custom AbstractArray type, you can specify that it has fast linear indexing usingBase.IndexStyle(::Type{<:MyArray}) = IndexLinear()This setting will cause eachindex iteration over a MyArray to use integers. If you don't specify this trait, the default value IndexCartesian() is used."
},

{
    "location": "manual/arrays.html#Array-and-Vectorized-Operators-and-Functions-1",
    "page": "Multi-dimensional Arrays",
    "title": "Array and Vectorized Operators and Functions",
    "category": "section",
    "text": "The following operators are supported for arrays:Unary arithmetic – -, +\nBinary arithmetic – -, +, *, /, \\, ^\nComparison – ==, !=, ≈ (isapprox), ≉Most of the binary arithmetic operators listed above also operate elementwise when one argument is scalar: -, +, and * when either argument is scalar, and / and \\ when the denominator is scalar. For example, [1, 2] + 3 == [4, 5] and [6, 4] / 2 == [3, 2].Additionally, to enable convenient vectorization of mathematical and other operations, Julia provides the dot syntax f.(args...), e.g. sin.(x) or min.(x,y), for elementwise operations over arrays or mixtures of arrays and scalars (a Broadcasting operation); these have the additional advantage of \"fusing\" into a single loop when combined with other dot calls, e.g. sin.(cos.(x)).Also, every binary operator supports a dot version that can be applied to arrays (and combinations of arrays and scalars) in such fused broadcasting operations, e.g. z .== sin.(x .* y).Note that comparisons such as == operate on whole arrays, giving a single boolean answer. Use dot operators like .== for elementwise comparisons. (For comparison operations like <, only the elementwise .< version is applicable to arrays.)Also notice the difference between max.(a,b), which broadcasts max() elementwise over a and b, and maximum(a), which finds the largest value within a. The same relationship holds for min.(a,b) and minimum(a)."
},

{
    "location": "manual/arrays.html#Broadcasting-1",
    "page": "Multi-dimensional Arrays",
    "title": "Broadcasting",
    "category": "section",
    "text": "It is sometimes useful to perform element-by-element binary operations on arrays of different sizes, such as adding a vector to each column of a matrix. An inefficient way to do this would be to replicate the vector to the size of the matrix:julia> a = rand(2,1); A = rand(2,3);\n\njulia> repmat(a,1,3)+A\n2×3 Array{Float64,2}:\n 1.20813  1.82068  1.25387\n 1.56851  1.86401  1.67846This is wasteful when dimensions get large, so Julia offers broadcast(), which expands singleton dimensions in array arguments to match the corresponding dimension in the other array without using extra memory, and applies the given function elementwise:julia> broadcast(+, a, A)\n2×3 Array{Float64,2}:\n 1.20813  1.82068  1.25387\n 1.56851  1.86401  1.67846\n\njulia> b = rand(1,2)\n1×2 Array{Float64,2}:\n 0.867535  0.00457906\n\njulia> broadcast(+, a, b)\n2×2 Array{Float64,2}:\n 1.71056  0.847604\n 1.73659  0.873631Dotted operators such as .+ and .* are equivalent to broadcast calls (except that they fuse, as described below). There is also a broadcast!() function to specify an explicit destination (which can also be accessed in a fusing fashion by .= assignment), and functions broadcast_getindex() and broadcast_setindex!() that broadcast the indices before indexing. Moreover, f.(args...) is equivalent to broadcast(f, args...), providing a convenient syntax to broadcast any function (dot syntax). Nested \"dot calls\" f.(...) (including calls to .+ etcetera) automatically fuse into a single broadcast call.Additionally, broadcast() is not limited to arrays (see the function documentation), it also handles tuples and treats any argument that is not an array, tuple or Ref (except for Ptr) as a \"scalar\".julia> convert.(Float32, [1, 2])\n2-element Array{Float32,1}:\n 1.0\n 2.0\n\njulia> ceil.((UInt8,), [1.2 3.4; 5.6 6.7])\n2×2 Array{UInt8,2}:\n 0x02  0x04\n 0x06  0x07\n\njulia> string.(1:3, \". \", [\"First\", \"Second\", \"Third\"])\n3-element Array{String,1}:\n \"1. First\"\n \"2. Second\"\n \"3. Third\""
},

{
    "location": "manual/arrays.html#Implementation-1",
    "page": "Multi-dimensional Arrays",
    "title": "Implementation",
    "category": "section",
    "text": "The base array type in Julia is the abstract type AbstractArray{T,N}. It is parametrized by the number of dimensions N and the element type T. AbstractVector and AbstractMatrix are aliases for the 1-d and 2-d cases. Operations on AbstractArray objects are defined using higher level operators and functions, in a way that is independent of the underlying storage. These operations generally work correctly as a fallback for any specific array implementation.The AbstractArray type includes anything vaguely array-like, and implementations of it might be quite different from conventional arrays. For example, elements might be computed on request rather than stored. However, any concrete AbstractArray{T,N} type should generally implement at least size(A) (returning an Int tuple), getindex(A,i) and getindex(A,i1,...,iN); mutable arrays should also implement setindex!(). It is recommended that these operations have nearly constant time complexity, or technically Õ(1) complexity, as otherwise some array functions may be unexpectedly slow. Concrete types should also typically provide a similar(A,T=eltype(A),dims=size(A)) method, which is used to allocate a similar array for copy() and other out-of-place operations. No matter how an AbstractArray{T,N} is represented internally, T is the type of object returned by integer indexing (A[1, ..., 1], when A is not empty) and N should be the length of the tuple returned by size().DenseArray is an abstract subtype of AbstractArray intended to include all arrays that are laid out at regular offsets in memory, and which can therefore be passed to external C and Fortran functions expecting this memory layout. Subtypes should provide a method stride(A,k) that returns the \"stride\" of dimension k: increasing the index of dimension k by 1 should increase the index i of getindex(A,i) by stride(A,k). If a pointer conversion method Base.unsafe_convert(Ptr{T}, A) is provided, the memory layout should correspond in the same way to these strides.The Array type is a specific instance of DenseArray where elements are stored in column-major order (see additional notes in Performance Tips). Vector and Matrix are aliases for the 1-d and 2-d cases. Specific operations such as scalar indexing, assignment, and a few other basic storage-specific operations are all that have to be implemented for Array, so that the rest of the array library can be implemented in a generic manner.SubArray is a specialization of AbstractArray that performs indexing by reference rather than by copying. A SubArray is created with the view() function, which is called the same way as getindex() (with an array and a series of index arguments). The result of view() looks the same as the result of getindex(), except the data is left in place. view() stores the input index vectors in a SubArray object, which can later be used to index the original array indirectly.  By putting the @views macro in front of an expression or block of code, any array[...] slice in that expression will be converted to create a SubArray view instead.StridedVector and StridedMatrix are convenient aliases defined to make it possible for Julia to call a wider range of BLAS and LAPACK functions by passing them either Array or SubArray objects, and thus saving inefficiencies from memory allocation and copying.The following example computes the QR decomposition of a small section of a larger array, without creating any temporaries, and by calling the appropriate LAPACK function with the right leading dimension size and stride parameters.julia> a = rand(10,10)\n10×10 Array{Float64,2}:\n 0.561255   0.226678   0.203391  0.308912   …  0.750307  0.235023   0.217964\n 0.718915   0.537192   0.556946  0.996234      0.666232  0.509423   0.660788\n 0.493501   0.0565622  0.118392  0.493498      0.262048  0.940693   0.252965\n 0.0470779  0.736979   0.264822  0.228787      0.161441  0.897023   0.567641\n 0.343935   0.32327    0.795673  0.452242      0.468819  0.628507   0.511528\n 0.935597   0.991511   0.571297  0.74485    …  0.84589   0.178834   0.284413\n 0.160706   0.672252   0.133158  0.65554       0.371826  0.770628   0.0531208\n 0.306617   0.836126   0.301198  0.0224702     0.39344   0.0370205  0.536062\n 0.890947   0.168877   0.32002   0.486136      0.096078  0.172048   0.77672\n 0.507762   0.573567   0.220124  0.165816      0.211049  0.433277   0.539476\n\njulia> b = view(a, 2:2:8,2:2:4)\n4×2 SubArray{Float64,2,Array{Float64,2},Tuple{StepRange{Int64,Int64},StepRange{Int64,Int64}},false}:\n 0.537192  0.996234\n 0.736979  0.228787\n 0.991511  0.74485\n 0.836126  0.0224702\n\njulia> (q,r) = qr(b);\n\njulia> q\n4×2 Array{Float64,2}:\n -0.338809   0.78934\n -0.464815  -0.230274\n -0.625349   0.194538\n -0.527347  -0.534856\n\njulia> r\n2×2 Array{Float64,2}:\n -1.58553  -0.921517\n  0.0       0.866567"
},

{
    "location": "manual/arrays.html#Sparse-Vectors-and-Matrices-1",
    "page": "Multi-dimensional Arrays",
    "title": "Sparse Vectors and Matrices",
    "category": "section",
    "text": "Julia has built-in support for sparse vectors and sparse matrices. Sparse arrays are arrays that contain enough zeros that storing them in a special data structure leads to savings in space and execution time, compared to dense arrays."
},

{
    "location": "manual/arrays.html#man-csc-1",
    "page": "Multi-dimensional Arrays",
    "title": "Compressed Sparse Column (CSC) Sparse Matrix Storage",
    "category": "section",
    "text": "In Julia, sparse matrices are stored in the Compressed Sparse Column (CSC) format. Julia sparse matrices have the type SparseMatrixCSC{Tv,Ti}, where Tv is the type of the stored values, and Ti is the integer type for storing column pointers and row indices. The internal representation of SparseMatrixCSC is as follows:struct SparseMatrixCSC{Tv,Ti<:Integer} <: AbstractSparseMatrix{Tv,Ti}\n    m::Int                  # Number of rows\n    n::Int                  # Number of columns\n    colptr::Vector{Ti}      # Column i is in colptr[i]:(colptr[i+1]-1)\n    rowval::Vector{Ti}      # Row indices of stored values\n    nzval::Vector{Tv}       # Stored values, typically nonzeros\nendThe compressed sparse column storage makes it easy and quick to access the elements in the column of a sparse matrix, whereas accessing the sparse matrix by rows is considerably slower. Operations such as insertion of previously unstored entries one at a time in the CSC structure tend to be slow. This is because all elements of the sparse matrix that are beyond the point of insertion have to be moved one place over.All operations on sparse matrices are carefully implemented to exploit the CSC data structure for performance, and to avoid expensive operations.If you have data in CSC format from a different application or library, and wish to import it in Julia, make sure that you use 1-based indexing. The row indices in every column need to be sorted. If your SparseMatrixCSC object contains unsorted row indices, one quick way to sort them is by doing a double transpose.In some applications, it is convenient to store explicit zero values in a SparseMatrixCSC. These are accepted by functions in Base (but there is no guarantee that they will be preserved in mutating operations). Such explicitly stored zeros are treated as structural nonzeros by many routines. The nnz() function returns the number of elements explicitly stored in the sparse data structure, including structural nonzeros. In order to count the exact number of numerical nonzeros, use countnz(), which inspects every stored element of a sparse matrix. dropzeros(), and the in-place dropzeros!(), can be used to remove stored zeros from the sparse matrix.julia> A = sparse([1, 2, 3], [1, 2, 3], [0, 2, 0])\n3×3 SparseMatrixCSC{Int64,Int64} with 3 stored entries:\n  [1, 1]  =  0\n  [2, 2]  =  2\n  [3, 3]  =  0\n\njulia> dropzeros(A)\n3×3 SparseMatrixCSC{Int64,Int64} with 1 stored entry:\n  [2, 2]  =  2"
},

{
    "location": "manual/arrays.html#Sparse-Vector-Storage-1",
    "page": "Multi-dimensional Arrays",
    "title": "Sparse Vector Storage",
    "category": "section",
    "text": "Sparse vectors are stored in a close analog to compressed sparse column format for sparse matrices. In Julia, sparse vectors have the type SparseVector{Tv,Ti} where Tv is the type of the stored values and Ti the integer type for the indices. The internal representation is as follows:struct SparseVector{Tv,Ti<:Integer} <: AbstractSparseVector{Tv,Ti}\n    n::Int              # Length of the sparse vector\n    nzind::Vector{Ti}   # Indices of stored values\n    nzval::Vector{Tv}   # Stored values, typically nonzeros\nendAs for SparseMatrixCSC, the SparseVector type can also contain explicitly stored zeros. (See Sparse Matrix Storage.)."
},

{
    "location": "manual/arrays.html#Sparse-Vector-and-Matrix-Constructors-1",
    "page": "Multi-dimensional Arrays",
    "title": "Sparse Vector and Matrix Constructors",
    "category": "section",
    "text": "The simplest way to create sparse arrays is to use functions equivalent to the zeros() and eye() functions that Julia provides for working with dense arrays. To produce sparse arrays instead, you can use the same names with an sp prefix:julia> spzeros(3)\n3-element SparseVector{Float64,Int64} with 0 stored entries\n\njulia> speye(3,5)\n3×5 SparseMatrixCSC{Float64,Int64} with 3 stored entries:\n  [1, 1]  =  1.0\n  [2, 2]  =  1.0\n  [3, 3]  =  1.0The sparse() function is often a handy way to construct sparse arrays. For example, to construct a sparse matrix we can input a vector I of row indices, a vector J of column indices, and a vector V of stored values (this is also known as the COO (coordinate) format). sparse(I,J,V) then constructs a sparse matrix such that S[I[k], J[k]] = V[k]. The equivalent sparse vector constructor is sparsevec, which takes the (row) index vector I and the vector V with the stored values and constructs a sparse vector R such that R[I[k]] = V[k].julia> I = [1, 4, 3, 5]; J = [4, 7, 18, 9]; V = [1, 2, -5, 3];\n\njulia> S = sparse(I,J,V)\n5×18 SparseMatrixCSC{Int64,Int64} with 4 stored entries:\n  [1 ,  4]  =  1\n  [4 ,  7]  =  2\n  [5 ,  9]  =  3\n  [3 , 18]  =  -5\n\njulia> R = sparsevec(I,V)\n5-element SparseVector{Int64,Int64} with 4 stored entries:\n  [1]  =  1\n  [3]  =  -5\n  [4]  =  2\n  [5]  =  3The inverse of the sparse() and sparsevec functions is findnz(), which retrieves the inputs used to create the sparse array. There is also a findn function which only returns the index vectors.julia> findnz(S)\n([1, 4, 5, 3], [4, 7, 9, 18], [1, 2, 3, -5])\n\njulia> findn(S)\n([1, 4, 5, 3], [4, 7, 9, 18])\n\njulia> findnz(R)\n([1, 3, 4, 5], [1, -5, 2, 3])\n\njulia> findn(R)\n4-element Array{Int64,1}:\n 1\n 3\n 4\n 5Another way to create a sparse array is to convert a dense array into a sparse array using the sparse() function:julia> sparse(eye(5))\n5×5 SparseMatrixCSC{Float64,Int64} with 5 stored entries:\n  [1, 1]  =  1.0\n  [2, 2]  =  1.0\n  [3, 3]  =  1.0\n  [4, 4]  =  1.0\n  [5, 5]  =  1.0\n\njulia> sparse([1.0, 0.0, 1.0])\n3-element SparseVector{Float64,Int64} with 2 stored entries:\n  [1]  =  1.0\n  [3]  =  1.0You can go in the other direction using the Array constructor. The issparse() function can be used to query if a matrix is sparse.julia> issparse(speye(5))\ntrue"
},

{
    "location": "manual/arrays.html#Sparse-matrix-operations-1",
    "page": "Multi-dimensional Arrays",
    "title": "Sparse matrix operations",
    "category": "section",
    "text": "Arithmetic operations on sparse matrices also work as they do on dense matrices. Indexing of, assignment into, and concatenation of sparse matrices work in the same way as dense matrices. Indexing operations, especially assignment, are expensive, when carried out one element at a time. In many cases it may be better to convert the sparse matrix into (I,J,V) format using findnz(), manipulate the values or the structure in the dense vectors (I,J,V), and then reconstruct the sparse matrix."
},

{
    "location": "manual/arrays.html#Correspondence-of-dense-and-sparse-methods-1",
    "page": "Multi-dimensional Arrays",
    "title": "Correspondence of dense and sparse methods",
    "category": "section",
    "text": "The following table gives a correspondence between built-in methods on sparse matrices and their corresponding methods on dense matrix types. In general, methods that generate sparse matrices differ from their dense counterparts in that the resulting matrix follows the same sparsity pattern as a given sparse matrix S, or that the resulting sparse matrix has density d, i.e. each matrix element has a probability d of being non-zero.Details can be found in the Sparse Vectors and Matrices section of the standard library reference.Sparse Dense Description\nspzeros(m,n) zeros(m,n) Creates a m-by-n matrix of zeros. (spzeros(m,n) is empty.)\nspones(S) ones(m,n) Creates a matrix filled with ones. Unlike the dense version, spones() has the same sparsity pattern as S.\nspeye(n) eye(n) Creates a n-by-n identity matrix.\nfull(S) sparse(A) Interconverts between dense and sparse formats.\nsprand(m,n,d) rand(m,n) Creates a m-by-n random matrix (of density d) with iid non-zero elements distributed uniformly on the half-open interval 0 1).\nsprandn(m,n,d) randn(m,n) Creates a m-by-n random matrix (of density d) with iid non-zero elements distributed according to the standard normal (Gaussian) distribution.\nsprandn(m,n,d,X) randn(m,n,X) Creates a m-by-n random matrix (of density d) with iid non-zero elements distributed according to the X distribution. (Requires the Distributions package.)"
},

{
    "location": "manual/linear-algebra.html#",
    "page": "Linear algebra",
    "title": "Linear algebra",
    "category": "page",
    "text": ""
},

{
    "location": "manual/linear-algebra.html#Linear-algebra-1",
    "page": "Linear algebra",
    "title": "Linear algebra",
    "category": "section",
    "text": "In addition to (and as part of) its support for multi-dimensional arrays, Julia provides native implementations of many common and useful linear algebra operations. Basic operations, such as trace, det, and inv are all supported:julia> A = [1 2 3; 4 1 6; 7 8 1]\n3×3 Array{Int64,2}:\n 1  2  3\n 4  1  6\n 7  8  1\n\njulia> trace(A)\n3\n\njulia> det(A)\n104.0\n\njulia> inv(A)\n3×3 Array{Float64,2}:\n -0.451923   0.211538    0.0865385\n  0.365385  -0.192308    0.0576923\n  0.240385   0.0576923  -0.0673077As well as other useful operations, such as finding eigenvalues or eigenvectors:julia> A = [1.5 2 -4; 3 -1 -6; -10 2.3 4]\n3×3 Array{Float64,2}:\n   1.5   2.0  -4.0\n   3.0  -1.0  -6.0\n -10.0   2.3   4.0\n\njulia> eigvals(A)\n3-element Array{Complex{Float64},1}:\n  9.31908+0.0im\n -2.40954+2.72095im\n -2.40954-2.72095im\n\njulia> eigvecs(A)\n3×3 Array{Complex{Float64},2}:\n -0.488645+0.0im  0.182546-0.39813im   0.182546+0.39813im\n -0.540358+0.0im  0.692926+0.0im       0.692926-0.0im\n   0.68501+0.0im  0.254058-0.513301im  0.254058+0.513301imIn addition, Julia provides many factorizations which can be used to speed up problems such as linear solve or matrix exponentiation by pre-factorizing a matrix into a form more amenable (for performance or memory reasons) to the problem. See the documentation on factorize for more information. As an example:julia> A = [1.5 2 -4; 3 -1 -6; -10 2.3 4]\n3×3 Array{Float64,2}:\n   1.5   2.0  -4.0\n   3.0  -1.0  -6.0\n -10.0   2.3   4.0\n\njulia> factorize(A)\nBase.LinAlg.LU{Float64,Array{Float64,2}} with factors L and U:\n[1.0 0.0 0.0; -0.15 1.0 0.0; -0.3 -0.132196 1.0]\n[-10.0 2.3 4.0; 0.0 2.345 -3.4; 0.0 0.0 -5.24947]Since A is not Hermitian, symmetric, triangular, tridiagonal, or bidiagonal, an LU factorization may be the best we can do. Compare with:julia> B = [1.5 2 -4; 2 -1 -3; -4 -3 5]\n3×3 Array{Float64,2}:\n  1.5   2.0  -4.0\n  2.0  -1.0  -3.0\n -4.0  -3.0   5.0\n\njulia> factorize(B)\nBase.LinAlg.BunchKaufman{Float64,Array{Float64,2}}([-1.64286 0.142857 -0.8; 2.0 -2.8 -0.6; -4.0 -3.0 5.0], [1, 2, 3], 'U', true, false, 0)Here, Julia was able to detect that B is in fact symmetric, and used a more appropriate factorization. Often it's possible to write more efficient code for a matrix that is known to have certain properties e.g. it is symmetric, or tridiagonal. Julia provides some special types so that you can \"tag\" matrices as having these properties. For instance:julia> B = [1.5 2 -4; 2 -1 -3; -4 -3 5]\n3×3 Array{Float64,2}:\n  1.5   2.0  -4.0\n  2.0  -1.0  -3.0\n -4.0  -3.0   5.0\n\njulia> sB = Symmetric(B)\n3×3 Symmetric{Float64,Array{Float64,2}}:\n  1.5   2.0  -4.0\n  2.0  -1.0  -3.0\n -4.0  -3.0   5.0sB has been tagged as a matrix that's (real) symmetric, so for later operations we might perform on it, such as eigenfactorization or computing matrix-vector products, efficiencies can be found by only referencing half of it. For example:julia> B = [1.5 2 -4; 2 -1 -3; -4 -3 5]\n3×3 Array{Float64,2}:\n  1.5   2.0  -4.0\n  2.0  -1.0  -3.0\n -4.0  -3.0   5.0\n\njulia> sB = Symmetric(B)\n3×3 Symmetric{Float64,Array{Float64,2}}:\n  1.5   2.0  -4.0\n  2.0  -1.0  -3.0\n -4.0  -3.0   5.0\n\njulia> x = [1; 2; 3]\n3-element Array{Int64,1}:\n 1\n 2\n 3\n\njulia> sB\\x\n3-element Array{Float64,1}:\n -1.73913\n -1.1087\n -1.45652The \\ operation here performs the linear solution. Julia's parser provides convenient dispatch to specialized methods for the transpose of a matrix left-divided by a vector, or for the various combinations of transpose operations in matrix-matrix solutions. Many of these are further specialized for certain special matrix types. For example, A\\B will end up calling Base.LinAlg.A_ldiv_B! while A'\\B will end up calling Base.LinAlg.Ac_ldiv_B, even though we used the same left-division operator. This works for matrices too: A.'\\B.' would call Base.LinAlg.At_ldiv_Bt. The left-division operator is pretty powerful and it's easy to write compact, readable code that is flexible enough to solve all sorts of systems of linear equations."
},

{
    "location": "manual/linear-algebra.html#Special-matrices-1",
    "page": "Linear algebra",
    "title": "Special matrices",
    "category": "section",
    "text": "Matrices with special symmetries and structures arise often in linear algebra and are frequently associated with various matrix factorizations. Julia features a rich collection of special matrix types, which allow for fast computation with specialized routines that are specially developed for particular matrix types.The following tables summarize the types of special matrices that have been implemented in Julia, as well as whether hooks to various optimized methods for them in LAPACK are available.Type Description\nHermitian Hermitian matrix\nUpperTriangular Upper triangular matrix\nLowerTriangular Lower triangular matrix\nTridiagonal Tridiagonal matrix\nSymTridiagonal Symmetric tridiagonal matrix\nBidiagonal Upper/lower bidiagonal matrix\nDiagonal Diagonal matrix\nUniformScaling Uniform scaling operator"
},

{
    "location": "manual/linear-algebra.html#Elementary-operations-1",
    "page": "Linear algebra",
    "title": "Elementary operations",
    "category": "section",
    "text": "Matrix type + - * \\ Other functions with optimized methods\nHermitian       MV inv(), sqrtm(), expm()\nUpperTriangular     MV MV inv(), det()\nLowerTriangular     MV MV inv(), det()\nSymTridiagonal M M MS MV eigmax(), eigmin()\nTridiagonal M M MS MV  \nBidiagonal M M MS MV  \nDiagonal M M MV MV inv(), det(), logdet(), /()\nUniformScaling M M MVS MVS /()Legend:Key Description\nM (matrix) An optimized method for matrix-matrix operations is available\nV (vector) An optimized method for matrix-vector operations is available\nS (scalar) An optimized method for matrix-scalar operations is available"
},

{
    "location": "manual/linear-algebra.html#Matrix-factorizations-1",
    "page": "Linear algebra",
    "title": "Matrix factorizations",
    "category": "section",
    "text": "Matrix type LAPACK eig() eigvals() eigvecs() svd() svdvals()\nHermitian HE   ARI      \nUpperTriangular TR A A A    \nLowerTriangular TR A A A    \nSymTridiagonal ST A ARI AV    \nTridiagonal GT          \nBidiagonal BD       A A\nDiagonal DI   A      Legend:Key Description Example\nA (all) An optimized method to find all the characteristic values and/or vectors is available e.g. eigvals(M)\nR (range) An optimized method to find the ilth through the ihth characteristic values are available eigvals(M, il, ih)\nI (interval) An optimized method to find the characteristic values in the interval [vl, vh] is available eigvals(M, vl, vh)\nV (vectors) An optimized method to find the characteristic vectors corresponding to the characteristic values x=[x1, x2,...] is available eigvecs(M, x)"
},

{
    "location": "manual/linear-algebra.html#The-uniform-scaling-operator-1",
    "page": "Linear algebra",
    "title": "The uniform scaling operator",
    "category": "section",
    "text": "A UniformScaling operator represents a scalar times the identity operator, λ*I. The identity operator  I is defined as a constant and is an instance of UniformScaling. The size of these operators are generic and match the other matrix in the binary operations +, -, * and \\. For A+I and A-I this means that A must be square. Multiplication with the identity operator I is a noop (except for checking that the scaling factor is one) and therefore almost without overhead."
},

{
    "location": "manual/linear-algebra.html#man-linalg-factorizations-1",
    "page": "Linear algebra",
    "title": "Matrix factorizations",
    "category": "section",
    "text": "Matrix factorizations (a.k.a. matrix decompositions) compute the factorization of a matrix into a product of matrices, and are one of the central concepts in linear algebra.The following table summarizes the types of matrix factorizations that have been implemented in Julia. Details of their associated methods can be found in the Linear Algebra section of the standard library documentation.Type Description\nCholesky Cholesky factorization\nCholeskyPivoted Pivoted Cholesky factorization\nLU LU factorization\nLUTridiagonal LU factorization for Tridiagonal matrices\nUmfpackLU LU factorization for sparse matrices (computed by UMFPack)\nQR QR factorization\nQRCompactWY Compact WY form of the QR factorization\nQRPivoted Pivoted QR factorization\nHessenberg Hessenberg decomposition\nEigen Spectral decomposition\nSVD Singular value decomposition\nGeneralizedSVD Generalized SVD"
},

{
    "location": "manual/networking-and-streams.html#",
    "page": "Networking and Streams",
    "title": "Networking and Streams",
    "category": "page",
    "text": ""
},

{
    "location": "manual/networking-and-streams.html#Networking-and-Streams-1",
    "page": "Networking and Streams",
    "title": "Networking and Streams",
    "category": "section",
    "text": "Julia provides a rich interface to deal with streaming I/O objects such as terminals, pipes and TCP sockets. This interface, though asynchronous at the system level, is presented in a synchronous manner to the programmer and it is usually unnecessary to think about the underlying asynchronous operation. This is achieved by making heavy use of Julia cooperative threading (coroutine) functionality."
},

{
    "location": "manual/networking-and-streams.html#Basic-Stream-I/O-1",
    "page": "Networking and Streams",
    "title": "Basic Stream I/O",
    "category": "section",
    "text": "All Julia streams expose at least a read() and a write() method, taking the stream as their first argument, e.g.:julia> write(STDOUT,\"Hello World\");  # suppress return value 11 with ;\nHello World\njulia> read(STDIN,Char)\n\n'\\n': ASCII/Unicode U+000a (category Cc: Other, control)Note that write() returns 11, the number of bytes (in \"Hello World\") written to STDOUT, but this return value is suppressed with the ;.Here Enter was pressed again so that Julia would read the newline. Now, as you can see from this example, write() takes the data to write as its second argument, while read() takes the type of the data to be read as the second argument.For example, to read a simple byte array, we could do:julia> x = zeros(UInt8, 4)\n4-element Array{UInt8,1}:\n 0x00\n 0x00\n 0x00\n 0x00\n\njulia> read!(STDIN, x)\nabcd\n4-element Array{UInt8,1}:\n 0x61\n 0x62\n 0x63\n 0x64However, since this is slightly cumbersome, there are several convenience methods provided. For example, we could have written the above as:julia> read(STDIN,4)\nabcd\n4-element Array{UInt8,1}:\n 0x61\n 0x62\n 0x63\n 0x64or if we had wanted to read the entire line instead:julia> readline(STDIN)\nabcd\n\"abcd\"Note that depending on your terminal settings, your TTY may be line buffered and might thus require an additional enter before the data is sent to Julia.To read every line from STDIN you can use eachline():for line in eachline(STDIN)\n    print(\"Found $line\")\nendor read() if you wanted to read by character instead:while !eof(STDIN)\n    x = read(STDIN, Char)\n    println(\"Found: $x\")\nend"
},

{
    "location": "manual/networking-and-streams.html#Text-I/O-1",
    "page": "Networking and Streams",
    "title": "Text I/O",
    "category": "section",
    "text": "Note that the write() method mentioned above operates on binary streams. In particular, values do not get converted to any canonical text representation but are written out as is:julia> write(STDOUT,0x61);  # suppress return value 1 with ;\naNote that a is written to STDOUT by the write() function and that the returned value is 1 (since 0x61 is one byte).For text I/O, use the print() or show() methods, depending on your needs (see the standard library reference for a detailed discussion of the difference between the two):julia> print(STDOUT, 0x61)\n97"
},

{
    "location": "manual/networking-and-streams.html#IO-Output-Contextual-Properties-1",
    "page": "Networking and Streams",
    "title": "IO Output Contextual Properties",
    "category": "section",
    "text": "Sometimes IO output can benefit from the ability to pass contextual information into show methods. The IOContext object provides this framework for associating arbitrary metadata with an IO object. For example, showcompact adds a hinting parameter to the IO object that the invoked show method should print a shorter output (if applicable)."
},

{
    "location": "manual/networking-and-streams.html#Working-with-Files-1",
    "page": "Networking and Streams",
    "title": "Working with Files",
    "category": "section",
    "text": "Like many other environments, Julia has an open() function, which takes a filename and returns an IOStream object that you can use to read and write things from the file. For example if we have a file, hello.txt, whose contents are Hello, World!:julia> f = open(\"hello.txt\")\nIOStream(<file hello.txt>)\n\njulia> readlines(f)\n1-element Array{String,1}:\n \"Hello, World!\"If you want to write to a file, you can open it with the write (\"w\") flag:julia> f = open(\"hello.txt\",\"w\")\nIOStream(<file hello.txt>)\n\njulia> write(f,\"Hello again.\")\n12If you examine the contents of hello.txt at this point, you will notice that it is empty; nothing has actually been written to disk yet. This is because the IOStream must be closed before the write is actually flushed to disk:julia> close(f)Examining hello.txt again will show its contents have been changed.Opening a file, doing something to its contents, and closing it again is a very common pattern. To make this easier, there exists another invocation of open() which takes a function as its first argument and filename as its second, opens the file, calls the function with the file as an argument, and then closes it again. For example, given a function:function read_and_capitalize(f::IOStream)\n    return uppercase(readstring(f))\nendYou can call:julia> open(read_and_capitalize, \"hello.txt\")\n\"HELLO AGAIN.\"to open hello.txt, call read_and_capitalize on it, close hello.txt and return the capitalized contents.To avoid even having to define a named function, you can use the do syntax, which creates an anonymous function on the fly:julia> open(\"hello.txt\") do f\n           uppercase(readstring(f))\n       end\n\"HELLO AGAIN.\""
},

{
    "location": "manual/networking-and-streams.html#A-simple-TCP-example-1",
    "page": "Networking and Streams",
    "title": "A simple TCP example",
    "category": "section",
    "text": "Let's jump right in with a simple example involving TCP sockets. Let's first create a simple server:julia> @async begin\n           server = listen(2000)\n           while true\n               sock = accept(server)\n               println(\"Hello World\\n\")\n           end\n       end\nTask (runnable) @0x00007fd31dc11ae0To those familiar with the Unix socket API, the method names will feel familiar, though their usage is somewhat simpler than the raw Unix socket API. The first call to listen() will create a server waiting for incoming connections on the specified port (2000) in this case. The same function may also be used to create various other kinds of servers:julia> listen(2000) # Listens on localhost:2000 (IPv4)\nTCPServer(active)\n\njulia> listen(ip\"127.0.0.1\",2000) # Equivalent to the first\nTCPServer(active)\n\njulia> listen(ip\"::1\",2000) # Listens on localhost:2000 (IPv6)\nTCPServer(active)\n\njulia> listen(IPv4(0),2001) # Listens on port 2001 on all IPv4 interfaces\nTCPServer(active)\n\njulia> listen(IPv6(0),2001) # Listens on port 2001 on all IPv6 interfaces\nTCPServer(active)\n\njulia> listen(\"testsocket\") # Listens on a UNIX domain socket/named pipe\nPipeServer(active)Note that the return type of the last invocation is different. This is because this server does not listen on TCP, but rather on a named pipe (Windows) or UNIX domain socket. The difference is subtle and has to do with the accept() and connect() methods. The accept() method retrieves a connection to the client that is connecting on the server we just created, while the connect() function connects to a server using the specified method. The connect() function takes the same arguments as listen(), so, assuming the environment (i.e. host, cwd, etc.) is the same you should be able to pass the same arguments to connect() as you did to listen to establish the connection. So let's try that out (after having created the server above):julia> connect(2000)\nTCPSocket(open, 0 bytes waiting)\n\njulia> Hello WorldAs expected we saw \"Hello World\" printed. So, let's actually analyze what happened behind the scenes. When we called connect(), we connect to the server we had just created. Meanwhile, the accept function returns a server-side connection to the newly created socket and prints \"Hello World\" to indicate that the connection was successful.A great strength of Julia is that since the API is exposed synchronously even though the I/O is actually happening asynchronously, we didn't have to worry callbacks or even making sure that the server gets to run. When we called connect() the current task waited for the connection to be established and only continued executing after that was done. In this pause, the server task resumed execution (because a connection request was now available), accepted the connection, printed the message and waited for the next client. Reading and writing works in the same way. To see this, consider the following simple echo server:julia> @async begin\n           server = listen(2001)\n           while true\n               sock = accept(server)\n               @async while isopen(sock)\n                   write(sock,readline(sock))\n               end\n           end\n       end\nTask (runnable) @0x00007fd31dc12e60\n\njulia> clientside = connect(2001)\nTCPSocket(RawFD(28) open, 0 bytes waiting)\n\njulia> @async while true\n           write(STDOUT,readline(clientside))\n       end\nTask (runnable) @0x00007fd31dc11870\n\njulia> println(clientside,\"Hello World from the Echo Server\")\nHello World from the Echo ServerAs with other streams, use close() to disconnect the socket:julia> close(clientside)"
},

{
    "location": "manual/networking-and-streams.html#Resolving-IP-Addresses-1",
    "page": "Networking and Streams",
    "title": "Resolving IP Addresses",
    "category": "section",
    "text": "One of the connect() methods that does not follow the listen() methods is connect(host::String,port), which will attempt to connect to the host given by the host parameter on the port given by the port parameter. It allows you to do things like:julia> connect(\"google.com\",80)\nTCPSocket(RawFD(30) open, 0 bytes waiting)At the base of this functionality is getaddrinfo(), which will do the appropriate address resolution:julia> getaddrinfo(\"google.com\")\nip\"74.125.226.225\""
},

{
    "location": "manual/parallel-computing.html#",
    "page": "Parallel Computing",
    "title": "Parallel Computing",
    "category": "page",
    "text": ""
},

{
    "location": "manual/parallel-computing.html#Parallel-Computing-1",
    "page": "Parallel Computing",
    "title": "Parallel Computing",
    "category": "section",
    "text": "Most modern computers possess more than one CPU, and several computers can be combined together in a cluster. Harnessing the power of these multiple CPUs allows many computations to be completed more quickly. There are two major factors that influence performance: the speed of the CPUs themselves, and the speed of their access to memory. In a cluster, it's fairly obvious that a given CPU will have fastest access to the RAM within the same computer (node). Perhaps more surprisingly, similar issues are relevant on a typical multicore laptop, due to differences in the speed of main memory and the cache. Consequently, a good multiprocessing environment should allow control over the \"ownership\" of a chunk of memory by a particular CPU. Julia provides a multiprocessing environment based on message passing to allow programs to run on multiple processes in separate memory domains at once.Julia's implementation of message passing is different from other environments such as MPI [1]. Communication in Julia is generally \"one-sided\", meaning that the programmer needs to explicitly manage only one process in a two-process operation. Furthermore, these operations typically do not look like \"message send\" and \"message receive\" but rather resemble higher-level operations like calls to user functions.Parallel programming in Julia is built on two primitives: remote references and remote calls. A remote reference is an object that can be used from any process to refer to an object stored on a particular process. A remote call is a request by one process to call a certain function on certain arguments on another (possibly the same) process.Remote references come in two flavors: Future and RemoteChannel.A remote call returns a Future to its result. Remote calls return immediately; the process that made the call proceeds to its next operation while the remote call happens somewhere else. You can wait for a remote call to finish by calling wait() on the returned Future, and you can obtain the full value of the result using fetch().On the other hand, RemoteChannel s are rewritable. For example, multiple processes can co-ordinate their processing by referencing the same remote Channel.Each process has an associated identifier. The process providing the interactive Julia prompt always has an id equal to 1. The processes used by default for parallel operations are referred to as \"workers\". When there is only one process, process 1 is considered a worker. Otherwise, workers are considered to be all processes other than process 1.Let's try this out. Starting with julia -p n provides n worker processes on the local machine. Generally it makes sense for n to equal the number of CPU cores on the machine.$ ./julia -p 2\n\njulia> r = remotecall(rand, 2, 2, 2)\nFuture(2, 1, 4, Nullable{Any}())\n\njulia> s = @spawnat 2 1 .+ fetch(r)\nFuture(2, 1, 5, Nullable{Any}())\n\njulia> fetch(s)\n2×2 Array{Float64,2}:\n 1.18526  1.50912\n 1.16296  1.60607The first argument to remotecall() is the function to call. Most parallel programming in Julia does not reference specific processes or the number of processes available, but remotecall() is considered a low-level interface providing finer control. The second argument to remotecall() is the id of the process that will do the work, and the remaining arguments will be passed to the function being called.As you can see, in the first line we asked process 2 to construct a 2-by-2 random matrix, and in the second line we asked it to add 1 to it. The result of both calculations is available in the two futures, r and s. The @spawnat macro evaluates the expression in the second argument on the process specified by the first argument.Occasionally you might want a remotely-computed value immediately. This typically happens when you read from a remote object to obtain data needed by the next local operation. The function remotecall_fetch() exists for this purpose. It is equivalent to fetch(remotecall(...)) but is more efficient.julia> remotecall_fetch(getindex, 2, r, 1, 1)\n0.18526337335308085Remember that getindex(r,1,1) is equivalent to r[1,1], so this call fetches the first element of the future r.The syntax of remotecall() is not especially convenient. The macro @spawn makes things easier. It operates on an expression rather than a function, and picks where to do the operation for you:julia> r = @spawn rand(2,2)\nFuture(2, 1, 4, Nullable{Any}())\n\njulia> s = @spawn 1 .+ fetch(r)\nFuture(3, 1, 5, Nullable{Any}())\n\njulia> fetch(s)\n2×2 Array{Float64,2}:\n 1.38854  1.9098\n 1.20939  1.57158Note that we used 1 .+ fetch(r) instead of 1 .+ r. This is because we do not know where the code will run, so in general a fetch() might be required to move r to the process doing the addition. In this case, @spawn is smart enough to perform the computation on the process that owns r, so the fetch() will be a no-op (no work is done).(It is worth noting that @spawn is not built-in but defined in Julia as a macro. It is possible to define your own such constructs.)An important thing to remember is that, once fetched, a Future will cache its value locally. Further fetch() calls do not entail a network hop. Once all referencing Futures have fetched, the remote stored value is deleted."
},

{
    "location": "manual/parallel-computing.html#Code-Availability-and-Loading-Packages-1",
    "page": "Parallel Computing",
    "title": "Code Availability and Loading Packages",
    "category": "section",
    "text": "Your code must be available on any process that runs it. For example, type the following into the Julia prompt:julia> function rand2(dims...)\n           return 2*rand(dims...)\n       end\n\njulia> rand2(2,2)\n2×2 Array{Float64,2}:\n 0.153756  0.368514\n 1.15119   0.918912\n\njulia> fetch(@spawn rand2(2,2))\nERROR: RemoteException(2, CapturedException(UndefVarError(Symbol(\"#rand2\"))\n[...]Process 1 knew about the function rand2, but process 2 did not.Most commonly you'll be loading code from files or packages, and you have a considerable amount of flexibility in controlling which processes load code. Consider a file, DummyModule.jl, containing the following code:module DummyModule\n\nexport MyType, f\n\nmutable struct MyType\n    a::Int\nend\n\nf(x) = x^2+1\n\nprintln(\"loaded\")\n\nendStarting Julia with julia -p 2, you can use this to verify the following:include(\"DummyModule.jl\") loads the file on just a single process (whichever one executes the statement).\nusing DummyModule causes the module to be loaded on all processes; however, the module is brought into scope only on the one executing the statement.\nAs long as DummyModule is loaded on process 2, commands like\nrr = RemoteChannel(2)\nput!(rr, MyType(7))\nallow you to store an object of type MyType on process 2 even if DummyModule is not in scope on process 2.You can force a command to run on all processes using the @everywhere macro. For example, @everywhere can also be used to directly define a function on all processes:julia> @everywhere id = myid()\n\njulia> remotecall_fetch(()->id, 2)\n2A file can also be preloaded on multiple processes at startup, and a driver script can be used to drive the computation:julia -p <n> -L file1.jl -L file2.jl driver.jlThe Julia process running the driver script in the example above has an id equal to 1, just like a process providing an interactive prompt.The base Julia installation has in-built support for two types of clusters:A local cluster specified with the -p option as shown above.\nA cluster spanning machines using the --machinefile option. This uses a passwordless ssh login to start Julia worker processes (from the same path as the current host) on the specified machines.Functions addprocs(), rmprocs(), workers(), and others are available as a programmatic means of adding, removing and querying the processes in a cluster.Note that workers do not run a .juliarc.jl startup script, nor do they synchronize their global state (such as global variables, new method definitions, and loaded modules) with any of the other running processes.Other types of clusters can be supported by writing your own custom ClusterManager, as described below in the ClusterManagers section."
},

{
    "location": "manual/parallel-computing.html#Data-Movement-1",
    "page": "Parallel Computing",
    "title": "Data Movement",
    "category": "section",
    "text": "Sending messages and moving data constitute most of the overhead in a parallel program. Reducing the number of messages and the amount of data sent is critical to achieving performance and scalability. To this end, it is important to understand the data movement performed by Julia's various parallel programming constructs.fetch() can be considered an explicit data movement operation, since it directly asks that an object be moved to the local machine. @spawn (and a few related constructs) also moves data, but this is not as obvious, hence it can be called an implicit data movement operation. Consider these two approaches to constructing and squaring a random matrix:Method 1:julia> A = rand(1000,1000);\n\njulia> Bref = @spawn A^2;\n\n[...]\n\njulia> fetch(Bref);Method 2:julia> Bref = @spawn rand(1000,1000)^2;\n\n[...]\n\njulia> fetch(Bref);The difference seems trivial, but in fact is quite significant due to the behavior of @spawn. In the first method, a random matrix is constructed locally, then sent to another process where it is squared. In the second method, a random matrix is both constructed and squared on another process. Therefore the second method sends much less data than the first.In this toy example, the two methods are easy to distinguish and choose from. However, in a real program designing data movement might require more thought and likely some measurement. For example, if the first process needs matrix A then the first method might be better. Or, if computing A is expensive and only the current process has it, then moving it to another process might be unavoidable. Or, if the current process has very little to do between the @spawn and fetch(Bref), it might be better to eliminate the parallelism altogether. Or imagine rand(1000,1000) is replaced with a more expensive operation. Then it might make sense to add another @spawn statement just for this step."
},

{
    "location": "manual/parallel-computing.html#Global-variables-1",
    "page": "Parallel Computing",
    "title": "Global variables",
    "category": "section",
    "text": "Expressions executed remotely via @spawn, or closures specified for remote execution using remotecall may refer to global variables. Global bindings under module Main are treated a little differently compared to global bindings in other modules. Consider the following code snippet:A = rand(10,10)\nremotecall_fetch(()->foo(A), 2)Note that A is a global variable defined in the local workspace. Worker 2 does not have a variable called A under Main. The act of shipping the closure ()->foo(A) to worker 2 results in Main.A being defined on 2. Main.A continues to exist on worker 2 even after the call remotecall_fetch returns. Remote calls with embedded global references (under Main module only) manage globals as follows:New global bindings are created on destination workers if they are referenced as part of a remote call.\nGlobal constants are declared as constants on remote nodes too.\nGlobals are re-sent to a destination worker only in the context of a remote call, and then only if its value has changed. Also, the cluster does not synchronize global bindings across nodes. For example:\nA = rand(10,10)\nremotecall_fetch(()->foo(A), 2) # worker 2\nA = rand(10,10)\nremotecall_fetch(()->foo(A), 3) # worker 3\nA = nothing\nExecuting the above snippet results in Main.A on worker 2 having a different value from Main.A on worker 3, while the value of Main.A on node 1 is set to nothing.As you may have realized, while memory associated with globals may be collected when they are reassigned on the master, no such action is taken on the workers as the bindings continue to be valid. clear! can be used to manually reassign specific globals on remote nodes to nothing once they are no longer required. This will release any memory associated with them as part of a regular garbage collection cycle.Thus programs should be careful referencing globals in remote calls. In fact, it is preferable to avoid them altogether if possible. If you must reference globals, consider using let blocks to localize global variables.For example:julia> A = rand(10,10);\n\njulia> remotecall_fetch(()->A, 2);\n\njulia> B = rand(10,10);\n\njulia> let B = B\n           remotecall_fetch(()->B, 2)\n       end;\n\njulia> @spawnat 2 whos();\n\njulia>  From worker 2:                               A    800 bytes  10×10 Array{Float64,2}\n        From worker 2:                            Base               Module\n        From worker 2:                            Core               Module\n        From worker 2:                            Main               ModuleAs can be seen, global variable A is defined on worker 2, but B is captured as a local variable and hence a binding for B does not exist on worker 2."
},

{
    "location": "manual/parallel-computing.html#Parallel-Map-and-Loops-1",
    "page": "Parallel Computing",
    "title": "Parallel Map and Loops",
    "category": "section",
    "text": "Fortunately, many useful parallel computations do not require data movement. A common example is a Monte Carlo simulation, where multiple processes can handle independent simulation trials simultaneously. We can use @spawn to flip coins on two processes. First, write the following function in count_heads.jl:function count_heads(n)\n    c::Int = 0\n    for i = 1:n\n        c += rand(Bool)\n    end\n    c\nendThe function count_heads simply adds together n random bits. Here is how we can perform some trials on two machines, and add together the results:julia> @everywhere include(\"count_heads.jl\")\n\njulia> a = @spawn count_heads(100000000)\nFuture(2, 1, 6, Nullable{Any}())\n\njulia> b = @spawn count_heads(100000000)\nFuture(3, 1, 7, Nullable{Any}())\n\njulia> fetch(a)+fetch(b)\n100001564This example demonstrates a powerful and often-used parallel programming pattern. Many iterations run independently over several processes, and then their results are combined using some function. The combination process is called a reduction, since it is generally tensor-rank-reducing: a vector of numbers is reduced to a single number, or a matrix is reduced to a single row or column, etc. In code, this typically looks like the pattern x = f(x,v[i]), where x is the accumulator, f is the reduction function, and the v[i] are the elements being reduced. It is desirable for f to be associative, so that it does not matter what order the operations are performed in.Notice that our use of this pattern with count_heads can be generalized. We used two explicit @spawn statements, which limits the parallelism to two processes. To run on any number of processes, we can use a parallel for loop, which can be written in Julia using @parallel like this:nheads = @parallel (+) for i = 1:200000000\n    Int(rand(Bool))\nendThis construct implements the pattern of assigning iterations to multiple processes, and combining them with a specified reduction (in this case (+)). The result of each iteration is taken as the value of the last expression inside the loop. The whole parallel loop expression itself evaluates to the final answer.Note that although parallel for loops look like serial for loops, their behavior is dramatically different. In particular, the iterations do not happen in a specified order, and writes to variables or arrays will not be globally visible since iterations run on different processes. Any variables used inside the parallel loop will be copied and broadcast to each process.For example, the following code will not work as intended:a = zeros(100000)\n@parallel for i = 1:100000\n    a[i] = i\nendThis code will not initialize all of a, since each process will have a separate copy of it. Parallel for loops like these must be avoided. Fortunately, Shared Arrays can be used to get around this limitation:a = SharedArray{Float64}(10)\n@parallel for i = 1:10\n    a[i] = i\nendUsing \"outside\" variables in parallel loops is perfectly reasonable if the variables are read-only:a = randn(1000)\n@parallel (+) for i = 1:100000\n    f(a[rand(1:end)])\nendHere each iteration applies f to a randomly-chosen sample from a vector a shared by all processes.As you could see, the reduction operator can be omitted if it is not needed. In that case, the loop executes asynchronously, i.e. it spawns independent tasks on all available workers and returns an array of Future immediately without waiting for completion. The caller can wait for the Future completions at a later point by calling fetch() on them, or wait for completion at the end of the loop by prefixing it with @sync, like @sync @parallel for.In some cases no reduction operator is needed, and we merely wish to apply a function to all integers in some range (or, more generally, to all elements in some collection). This is another useful operation called parallel map, implemented in Julia as the pmap() function. For example, we could compute the singular values of several large random matrices in parallel as follows:julia> M = Matrix{Float64}[rand(1000,1000) for i = 1:10];\n\njulia> pmap(svd, M);Julia's pmap() is designed for the case where each function call does a large amount of work. In contrast, @parallel for can handle situations where each iteration is tiny, perhaps merely summing two numbers. Only worker processes are used by both pmap() and @parallel for for the parallel computation. In case of @parallel for, the final reduction is done on the calling process."
},

{
    "location": "manual/parallel-computing.html#Synchronization-With-Remote-References-1",
    "page": "Parallel Computing",
    "title": "Synchronization With Remote References",
    "category": "section",
    "text": ""
},

{
    "location": "manual/parallel-computing.html#Scheduling-1",
    "page": "Parallel Computing",
    "title": "Scheduling",
    "category": "section",
    "text": "Julia's parallel programming platform uses Tasks (aka Coroutines) to switch among multiple computations. Whenever code performs a communication operation like fetch() or wait(), the current task is suspended and a scheduler picks another task to run. A task is restarted when the event it is waiting for completes.For many problems, it is not necessary to think about tasks directly. However, they can be used to wait for multiple events at the same time, which provides for dynamic scheduling. In dynamic scheduling, a program decides what to compute or where to compute it based on when other jobs finish. This is needed for unpredictable or unbalanced workloads, where we want to assign more work to processes only when they finish their current tasks.As an example, consider computing the singular values of matrices of different sizes:julia> M = Matrix{Float64}[rand(800,800), rand(600,600), rand(800,800), rand(600,600)];\n\njulia> pmap(svd, M);If one process handles both 800×800 matrices and another handles both 600×600 matrices, we will not get as much scalability as we could. The solution is to make a local task to \"feed\" work to each process when it completes its current task. For example, consider a simple pmap() implementation:function pmap(f, lst)\n    np = nprocs()  # determine the number of processes available\n    n = length(lst)\n    results = Vector{Any}(n)\n    i = 1\n    # function to produce the next work item from the queue.\n    # in this case it's just an index.\n    nextidx() = (idx=i; i+=1; idx)\n    @sync begin\n        for p=1:np\n            if p != myid() || np == 1\n                @async begin\n                    while true\n                        idx = nextidx()\n                        if idx > n\n                            break\n                        end\n                        results[idx] = remotecall_fetch(f, p, lst[idx])\n                    end\n                end\n            end\n        end\n    end\n    results\nend@async is similar to @spawn, but only runs tasks on the local process. We use it to create a \"feeder\" task for each process. Each task picks the next index that needs to be computed, then waits for its process to finish, then repeats until we run out of indexes. Note that the feeder tasks do not begin to execute until the main task reaches the end of the @sync block, at which point it surrenders control and waits for all the local tasks to complete before returning from the function. The feeder tasks are able to share state via nextidx() because they all run on the same process. No locking is required, since the threads are scheduled cooperatively and not preemptively. This means context switches only occur at well-defined points: in this case, when remotecall_fetch() is called."
},

{
    "location": "manual/parallel-computing.html#Channels-1",
    "page": "Parallel Computing",
    "title": "Channels",
    "category": "section",
    "text": "The section on Tasks in Control Flow discussed the execution of multiple functions in a co-operative manner. Channels can be quite useful to pass data between running tasks, particularly those involving I/O operations.Examples of operations involving I/O include reading/writing to files, accessing web services, executing external programs, etc. In all these cases, overall execution time can be improved if other tasks can be run while a file is being read, or while waiting for an external service/program to complete.A channel can be visualized as a pipe, i.e., it has a write end and read end.Multiple writers in different tasks can write to the same channel concurrently via put!() calls.\nMultiple readers in different tasks can read data concurrently via take!() calls.\nAs an example:\n# Given Channels c1 and c2,\nc1 = Channel(32)\nc2 = Channel(32)\n\n# and a function `foo()` which reads items from from c1, processes the item read\n# and writes a result to c2,\nfunction foo()\n    while true\n        data = take!(c1)\n        [...]               # process data\n        put!(c2, result)    # write out result\n    end\nend\n\n# we can schedule `n` instances of `foo()` to be active concurrently.\nfor _ in 1:n\n    @schedule foo()\nend\nChannels are created via the Channel{T}(sz) constructor. The channel will only hold objects of type T. If the type is not specified, the channel can hold objects of any type. sz refers to the maximum number of elements that can be held in the channel at any time. For example, Channel(32) creates a channel that can hold a maximum of 32 objects of any type. A Channel{MyType}(64) can hold up to 64 objects of MyType at any time.\nIf a Channel is empty, readers (on a take!() call) will block until data is available.\nIf a Channel is full, writers (on a put!() call) will block until space becomes available.\nisready() tests for the presence of any object in the channel, while wait() waits for an object to become available.\nA Channel is in an open state initially. This means that it can be read from and written to freely via take!() and put!() calls. close() closes a Channel. On a closed Channel, put!() will fail. For example:julia> c = Channel(2);\n\njulia> put!(c, 1) # `put!` on an open channel succeeds\n1\n\njulia> close(c);\n\njulia> put!(c, 2) # `put!` on a closed channel throws an exception.\nERROR: InvalidStateException(\"Channel is closed.\",:closed)\n[...]take!() and fetch() (which retrieves but does not remove the value) on a closed channel successfully return any existing values until it is emptied. Continuing the above example:julia> fetch(c) # Any number of `fetch` calls succeed.\n1\n\njulia> fetch(c)\n1\n\njulia> take!(c) # The first `take!` removes the value.\n1\n\njulia> take!(c) # No more data available on a closed channel.\nERROR: InvalidStateException(\"Channel is closed.\",:closed)\n[...]A Channel can be used as an iterable object in a for loop, in which case the loop runs as long as the Channel has data or is open. The loop variable takes on all values added to the Channel. The for loop is terminated once the Channel is closed and emptied.For example, the following would cause the for loop to wait for more data:julia> c = Channel{Int}(10);\n\njulia> foreach(i->put!(c, i), 1:3) # add a few entries\n\njulia> data = [i for i in c]while this will return after reading all data:julia> c = Channel{Int}(10);\n\njulia> foreach(i->put!(c, i), 1:3); # add a few entries\n\njulia> close(c);                    # `for` loops can exit\n\njulia> data = [i for i in c]\n3-element Array{Int64,1}:\n 1\n 2\n 3Consider a simple example using channels for inter-task communication. We start 4 tasks to process data from a single jobs channel. Jobs, identified by an id (job_id), are written to the channel. Each task in this simulation reads a job_id, waits for a random amout of time and writes back a tuple of job_id and the simulated time to the results channel. Finally all the results are printed out.julia> const jobs = Channel{Int}(32);\n\njulia> const results = Channel{Tuple}(32);\n\njulia> function do_work()\n           for job_id in jobs\n               exec_time = rand()\n               sleep(exec_time)                # simulates elapsed time doing actual work\n                                               # typically performed externally.\n               put!(results, (job_id, exec_time))\n           end\n       end;\n\njulia> function make_jobs(n)\n           for i in 1:n\n               put!(jobs, i)\n           end\n       end;\n\njulia> n = 12;\n\njulia> @schedule make_jobs(n); # feed the jobs channel with \"n\" jobs\n\njulia> for i in 1:4 # start 4 tasks to process requests in parallel\n           @schedule do_work()\n       end\n\njulia> @elapsed while n > 0 # print out results\n           job_id, exec_time = take!(results)\n           println(\"$job_id finished in $(round(exec_time,2)) seconds\")\n           n = n - 1\n       end\n4 finished in 0.22 seconds\n3 finished in 0.45 seconds\n1 finished in 0.5 seconds\n7 finished in 0.14 seconds\n2 finished in 0.78 seconds\n5 finished in 0.9 seconds\n9 finished in 0.36 seconds\n6 finished in 0.87 seconds\n8 finished in 0.79 seconds\n10 finished in 0.64 seconds\n12 finished in 0.5 seconds\n11 finished in 0.97 seconds\n0.029772311The current version of Julia multiplexes all tasks onto a single OS thread. Thus, while tasks involving I/O operations benefit from parallel execution, compute bound tasks are effectively executed sequentially on a single OS thread. Future versions of Julia may support scheduling of tasks on multiple threads, in which case compute bound tasks will see benefits of parallel execution too."
},

{
    "location": "manual/parallel-computing.html#Remote-References-and-AbstractChannels-1",
    "page": "Parallel Computing",
    "title": "Remote References and AbstractChannels",
    "category": "section",
    "text": "Remote references always refer to an implementation of an AbstractChannel.A concrete implementation of an AbstractChannel (like Channel), is required to implement put!(), take!(), fetch(), isready() and wait(). The remote object referred to by a Future is stored in a Channel{Any}(1), i.e., a Channel of size 1 capable of holding objects of Any type.RemoteChannel, which is rewritable, can point to any type and size of channels, or any other implementation of an AbstractChannel.The constructor RemoteChannel(f::Function, pid)() allows us to construct references to channels holding more than one value of a specific type. f() is a function executed on pid and it must return an AbstractChannel.For example, RemoteChannel(()->Channel{Int}(10), pid), will return a reference to a channel of type Int and size 10. The channel exists on worker pid.Methods put!(), take!(), fetch(), isready() and wait() on a RemoteChannel are proxied onto the backing store on the remote process.RemoteChannel can thus be used to refer to user implemented AbstractChannel objects. A simple example of this is provided in examples/dictchannel.jl which uses a dictionary as its remote store."
},

{
    "location": "manual/parallel-computing.html#Channels-and-RemoteChannels-1",
    "page": "Parallel Computing",
    "title": "Channels and RemoteChannels",
    "category": "section",
    "text": "A Channel is local to a process. Worker 2 cannot directly refer to a Channel on worker 3 and vice-versa. A RemoteChannel, however, can put and take values across workers.\nA RemoteChannel can be thought of as a handle to a Channel.\nThe process id, pid, associated with a RemoteChannel identifies the process where the backing store, i.e., the backing Channel exists.\nAny process with a reference to a RemoteChannel can put and take items from the channel. Data is automatically sent to (or retrieved from) the process a RemoteChannel is associated with.\nSerializing  a Channel also serializes any data present in the channel. Deserializing it therefore effectively makes a copy of the original object.\nOn the other hand, serializing a RemoteChannel only involves the serialization of an identifier that identifies the location and instance of Channel referred to by the handle. A deserialized RemoteChannel object (on any worker), therefore also points to the same backing store as the original.The channels example from above can be modified for interprocess communication, as shown below.We start 4 workers to process a single jobs remote channel. Jobs, identified by an id (job_id), are written to the channel. Each remotely executing task in this simulation reads a job_id, waits for a random amount of time and writes back a tuple of job_id, time taken and its own pid to the results channel. Finally all the results are printed out on the master process.julia> addprocs(4); # add worker processes\n\njulia> const jobs = RemoteChannel(()->Channel{Int}(32));\n\njulia> const results = RemoteChannel(()->Channel{Tuple}(32));\n\njulia> @everywhere function do_work(jobs, results) # define work function everywhere\n           while true\n               job_id = take!(jobs)\n               exec_time = rand()\n               sleep(exec_time) # simulates elapsed time doing actual work\n               put!(results, (job_id, exec_time, myid()))\n           end\n       end\n\njulia> function make_jobs(n)\n           for i in 1:n\n               put!(jobs, i)\n           end\n       end;\n\njulia> n = 12;\n\njulia> @schedule make_jobs(n); # feed the jobs channel with \"n\" jobs\n\njulia> for p in workers() # start tasks on the workers to process requests in parallel\n           @async remote_do(do_work, p, jobs, results)\n       end\n\njulia> @elapsed while n > 0 # print out results\n           job_id, exec_time, where = take!(results)\n           println(\"$job_id finished in $(round(exec_time,2)) seconds on worker $where\")\n           n = n - 1\n       end\n1 finished in 0.18 seconds on worker 4\n2 finished in 0.26 seconds on worker 5\n6 finished in 0.12 seconds on worker 4\n7 finished in 0.18 seconds on worker 4\n5 finished in 0.35 seconds on worker 5\n4 finished in 0.68 seconds on worker 2\n3 finished in 0.73 seconds on worker 3\n11 finished in 0.01 seconds on worker 3\n12 finished in 0.02 seconds on worker 3\n9 finished in 0.26 seconds on worker 5\n8 finished in 0.57 seconds on worker 4\n10 finished in 0.58 seconds on worker 2\n0.055971741"
},

{
    "location": "manual/parallel-computing.html#Remote-References-and-Distributed-Garbage-Collection-1",
    "page": "Parallel Computing",
    "title": "Remote References and Distributed Garbage Collection",
    "category": "section",
    "text": "Objects referred to by remote references can be freed only when all held references in the cluster are deleted.The node where the value is stored keeps track of which of the workers have a reference to it. Every time a RemoteChannel or a (unfetched) Future is serialized to a worker, the node pointed to by the reference is notified. And every time a RemoteChannel or a (unfetched) Future is garbage collected locally, the node owning the value is again notified.The notifications are done via sending of \"tracking\" messages–an \"add reference\" message when a reference is serialized to a different process and a \"delete reference\" message when a reference is locally garbage collected.Since Futures are write-once and cached locally, the act of fetch()ing a Future also updates reference tracking information on the node owning the value.The node which owns the value frees it once all references to it are cleared.With Futures, serializing an already fetched Future to a different node also sends the value since the original remote store may have collected the value by this time.It is important to note that when an object is locally garbage collected depends on the size of the object and the current memory pressure in the system.In case of remote references, the size of the local reference object is quite small, while the value stored on the remote node may be quite large. Since the local object may not be collected immediately, it is a good practice to explicitly call finalize() on local instances of a RemoteChannel, or on unfetched Futures. Since calling fetch() on a Future also removes its reference from the remote store, this is not required on fetched Futures. Explicitly calling finalize() results in an immediate message sent to the remote node to go ahead and remove its reference to the value.Once finalized, a reference becomes invalid and cannot be used in any further calls."
},

{
    "location": "manual/parallel-computing.html#man-shared-arrays-1",
    "page": "Parallel Computing",
    "title": "Shared Arrays",
    "category": "section",
    "text": "Shared Arrays use system shared memory to map the same array across many processes. While there are some similarities to a DArray, the behavior of a SharedArray is quite different. In a DArray, each process has local access to just a chunk of the data, and no two processes share the same chunk; in contrast, in a SharedArray each \"participating\" process has access to the entire array.  A SharedArray is a good choice when you want to have a large amount of data jointly accessible to two or more processes on the same machine.SharedArray indexing (assignment and accessing values) works just as with regular arrays, and is efficient because the underlying memory is available to the local process. Therefore, most algorithms work naturally on SharedArrays, albeit in single-process mode. In cases where an algorithm insists on an Array input, the underlying array can be retrieved from a SharedArray by calling sdata(). For other AbstractArray types, sdata() just returns the object itself, so it's safe to use sdata() on any Array-type object.The constructor for a shared array is of the form:SharedArray{T,N}(dims::NTuple; init=false, pids=Int[])which creates an N-dimensional shared array of a bits type T and size dims across the processes specified by pids. Unlike distributed arrays, a shared array is accessible only from those participating workers specified by the pids named argument (and the creating process too, if it is on the same host).If an init function, of signature initfn(S::SharedArray), is specified, it is called on all the participating workers. You can specify that each worker runs the init function on a distinct portion of the array, thereby parallelizing initialization.Here's a brief example:julia> addprocs(3)\n3-element Array{Int64,1}:\n 2\n 3\n 4\n\njulia> S = SharedArray{Int,2}((3,4), init = S -> S[Base.localindexes(S)] = myid())\n3×4 SharedArray{Int64,2}:\n 2  2  3  4\n 2  3  3  4\n 2  3  4  4\n\njulia> S[3,2] = 7\n7\n\njulia> S\n3×4 SharedArray{Int64,2}:\n 2  2  3  4\n 2  3  3  4\n 2  7  4  4Base.localindexes() provides disjoint one-dimensional ranges of indexes, and is sometimes convenient for splitting up tasks among processes. You can, of course, divide the work any way you wish:julia> S = SharedArray{Int,2}((3,4), init = S -> S[indexpids(S):length(procs(S)):length(S)] = myid())\n3×4 SharedArray{Int64,2}:\n 2  2  2  2\n 3  3  3  3\n 4  4  4  4Since all processes have access to the underlying data, you do have to be careful not to set up conflicts. For example:@sync begin\n    for p in procs(S)\n        @async begin\n            remotecall_wait(fill!, p, S, p)\n        end\n    end\nendwould result in undefined behavior. Because each process fills the entire array with its own pid, whichever process is the last to execute (for any particular element of S) will have its pid retained.As a more extended and complex example, consider running the following \"kernel\" in parallel:q[i,j,t+1] = q[i,j,t] + u[i,j,t]In this case, if we try to split up the work using a one-dimensional index, we are likely to run into trouble: if q[i,j,t] is near the end of the block assigned to one worker and q[i,j,t+1] is near the beginning of the block assigned to another, it's very likely that q[i,j,t] will not be ready at the time it's needed for computing q[i,j,t+1]. In such cases, one is better off chunking the array manually. Let's split along the second dimension. Define a function that returns the (irange, jrange) indexes assigned to this worker:julia> @everywhere function myrange(q::SharedArray)\n           idx = indexpids(q)\n           if idx == 0 # This worker is not assigned a piece\n               return 1:0, 1:0\n           end\n           nchunks = length(procs(q))\n           splits = [round(Int, s) for s in linspace(0,size(q,2),nchunks+1)]\n           1:size(q,1), splits[idx]+1:splits[idx+1]\n       endNext, define the kernel:julia> @everywhere function advection_chunk!(q, u, irange, jrange, trange)\n           @show (irange, jrange, trange)  # display so we can see what's happening\n           for t in trange, j in jrange, i in irange\n               q[i,j,t+1] = q[i,j,t] + u[i,j,t]\n           end\n           q\n       endWe also define a convenience wrapper for a SharedArray implementationjulia> @everywhere advection_shared_chunk!(q, u) =\n           advection_chunk!(q, u, myrange(q)..., 1:size(q,3)-1)Now let's compare three different versions, one that runs in a single process:julia> advection_serial!(q, u) = advection_chunk!(q, u, 1:size(q,1), 1:size(q,2), 1:size(q,3)-1);one that uses @parallel:julia> function advection_parallel!(q, u)\n           for t = 1:size(q,3)-1\n               @sync @parallel for j = 1:size(q,2)\n                   for i = 1:size(q,1)\n                       q[i,j,t+1]= q[i,j,t] + u[i,j,t]\n                   end\n               end\n           end\n           q\n       end;and one that delegates in chunks:julia> function advection_shared!(q, u)\n           @sync begin\n               for p in procs(q)\n                   @async remotecall_wait(advection_shared_chunk!, p, q, u)\n               end\n           end\n           q\n       end;If we create SharedArrays and time these functions, we get the following results (with julia -p 4):julia> q = SharedArray{Float64,3}((500,500,500));\n\njulia> u = SharedArray{Float64,3}((500,500,500));Run the functions once to JIT-compile and @time them on the second run:julia> @time advection_serial!(q, u);\n(irange,jrange,trange) = (1:500,1:500,1:499)\n 830.220 milliseconds (216 allocations: 13820 bytes)\n\njulia> @time advection_parallel!(q, u);\n   2.495 seconds      (3999 k allocations: 289 MB, 2.09% gc time)\n\njulia> @time advection_shared!(q,u);\n        From worker 2:       (irange,jrange,trange) = (1:500,1:125,1:499)\n        From worker 4:       (irange,jrange,trange) = (1:500,251:375,1:499)\n        From worker 3:       (irange,jrange,trange) = (1:500,126:250,1:499)\n        From worker 5:       (irange,jrange,trange) = (1:500,376:500,1:499)\n 238.119 milliseconds (2264 allocations: 169 KB)The biggest advantage of advection_shared! is that it minimizes traffic among the workers, allowing each to compute for an extended time on the assigned piece."
},

{
    "location": "manual/parallel-computing.html#Shared-Arrays-and-Distributed-Garbage-Collection-1",
    "page": "Parallel Computing",
    "title": "Shared Arrays and Distributed Garbage Collection",
    "category": "section",
    "text": "Like remote references, shared arrays are also dependent on garbage collection on the creating node to release references from all participating workers. Code which creates many short lived shared array objects would benefit from explicitly finalizing these objects as soon as possible. This results in both memory and file handles mapping the shared segment being released sooner."
},

{
    "location": "manual/parallel-computing.html#ClusterManagers-1",
    "page": "Parallel Computing",
    "title": "ClusterManagers",
    "category": "section",
    "text": "The launching, management and networking of Julia processes into a logical cluster is done via cluster managers. A ClusterManager is responsible forlaunching worker processes in a cluster environment\nmanaging events during the lifetime of each worker\noptionally, providing data transportA Julia cluster has the following characteristics:The initial Julia process, also called the master, is special and has an id of 1.\nOnly the master process can add or remove worker processes.\nAll processes can directly communicate with each other.Connections between workers (using the in-built TCP/IP transport) is established in the following manner:addprocs() is called on the master process with a ClusterManager object.\naddprocs() calls the appropriate launch() method which spawns required number of worker processes on appropriate machines.\nEach worker starts listening on a free port and writes out its host and port information to STDOUT.\nThe cluster manager captures the STDOUT of each worker and makes it available to the master process.\nThe master process parses this information and sets up TCP/IP connections to each worker.\nEvery worker is also notified of other workers in the cluster.\nEach worker connects to all workers whose id is less than the worker's own id.\nIn this way a mesh network is established, wherein every worker is directly connected with every other worker.While the default transport layer uses plain TCPSocket, it is possible for a Julia cluster to provide its own transport.Julia provides two in-built cluster managers:LocalManager, used when addprocs() or addprocs(np::Integer) are called\nSSHManager, used when addprocs(hostnames::Array) is called with a list of hostnamesLocalManager is used to launch additional workers on the same host, thereby leveraging multi-core and multi-processor hardware.Thus, a minimal cluster manager would need to:be a subtype of the abstract ClusterManager\nimplement launch(), a method responsible for launching new workers\nimplement manage(), which is called at various events during a worker's lifetime (for example, sending an interrupt signal)addprocs(manager::FooManager) requires FooManager to implement:function launch(manager::FooManager, params::Dict, launched::Array, c::Condition)\n    [...]\nend\n\nfunction manage(manager::FooManager, id::Integer, config::WorkerConfig, op::Symbol)\n    [...]\nendAs an example let us see how the LocalManager, the manager responsible for starting workers on the same host, is implemented:struct LocalManager <: ClusterManager\n    np::Integer\nend\n\nfunction launch(manager::LocalManager, params::Dict, launched::Array, c::Condition)\n    [...]\nend\n\nfunction manage(manager::LocalManager, id::Integer, config::WorkerConfig, op::Symbol)\n    [...]\nendThe launch() method takes the following arguments:manager::ClusterManager: the cluster manager that addprocs() is called with\nparams::Dict: all the keyword arguments passed to addprocs()\nlaunched::Array: the array to append one or more WorkerConfig objects to\nc::Condition: the condition variable to be notified as and when workers are launchedThe launch() method is called asynchronously in a separate task. The termination of this task signals that all requested workers have been launched. Hence the launch() function MUST exit as soon as all the requested workers have been launched.Newly launched workers are connected to each other, and the master process, in an all-to-all manner. Specifying the command argument --worker <cookie> results in the launched processes initializing themselves as workers and connections being set up via TCP/IP sockets. Optionally, --bind-to bind_addr[:port] may also be specified to enable other workers to connect to it at the specified bind_addr and port. This is useful for multi-homed hosts.As an example of a non-TCP/IP transport, an implementation may choose to use MPI, in which case --worker must NOT be specified. Instead, newly launched workers should call init_worker(cookie) before using any of the parallel constructs.For every worker launched, the launch() method must add a WorkerConfig object (with appropriate fields initialized) to launchedmutable struct WorkerConfig\n    # Common fields relevant to all cluster managers\n    io::Nullable{IO}\n    host::Nullable{AbstractString}\n    port::Nullable{Integer}\n\n    # Used when launching additional workers at a host\n    count::Nullable{Union{Int, Symbol}}\n    exename::Nullable{AbstractString}\n    exeflags::Nullable{Cmd}\n\n    # External cluster managers can use this to store information at a per-worker level\n    # Can be a dict if multiple fields need to be stored.\n    userdata::Nullable{Any}\n\n    # SSHManager / SSH tunnel connections to workers\n    tunnel::Nullable{Bool}\n    bind_addr::Nullable{AbstractString}\n    sshflags::Nullable{Cmd}\n    max_parallel::Nullable{Integer}\n\n    connect_at::Nullable{Any}\n\n    [...]\nendMost of the fields in WorkerConfig are used by the inbuilt managers. Custom cluster managers would typically specify only io or host / port:If io is specified, it is used to read host/port information. A Julia worker prints out its bind address and port at startup. This allows Julia workers to listen on any free port available instead of requiring worker ports to be configured manually.\nIf io is not specified, host and port are used to connect.\ncount, exename and exeflags are relevant for launching additional workers from a worker. For example, a cluster manager may launch a single worker per node, and use that to launch additional workers.\ncount with an integer value n will launch a total of n workers.\ncount with a value of :auto will launch as many workers as the number of cores on that machine.\nexename is the name of the julia executable including the full path.\nexeflags should be set to the required command line arguments for new workers.\ntunnel, bind_addr, sshflags and max_parallel are used when a ssh tunnel is required to connect to the workers from the master process.\nuserdata is provided for custom cluster managers to store their own worker-specific information.manage(manager::FooManager, id::Integer, config::WorkerConfig, op::Symbol) is called at different times during the worker's lifetime with appropriate op values:with :register/:deregister when a worker is added / removed from the Julia worker pool.\nwith :interrupt when interrupt(workers) is called. The ClusterManager should signal the appropriate worker with an interrupt signal.\nwith :finalize for cleanup purposes."
},

{
    "location": "manual/parallel-computing.html#Cluster-Managers-with-Custom-Transports-1",
    "page": "Parallel Computing",
    "title": "Cluster Managers with Custom Transports",
    "category": "section",
    "text": "Replacing the default TCP/IP all-to-all socket connections with a custom transport layer is a little more involved. Each Julia process has as many communication tasks as the workers it is connected to. For example, consider a Julia cluster of 32 processes in an all-to-all mesh network:Each Julia process thus has 31 communication tasks.\nEach task handles all incoming messages from a single remote worker in a message-processing loop.\nThe message-processing loop waits on an IO object (for example, a TCPSocket in the default implementation), reads an entire message, processes it and waits for the next one.\nSending messages to a process is done directly from any Julia task–not just communication tasks–again, via the appropriate IO object.Replacing the default transport requires the new implementation to set up connections to remote workers and to provide appropriate IO objects that the message-processing loops can wait on. The manager-specific callbacks to be implemented are:connect(manager::FooManager, pid::Integer, config::WorkerConfig)\nkill(manager::FooManager, pid::Int, config::WorkerConfig)The default implementation (which uses TCP/IP sockets) is implemented as connect(manager::ClusterManager, pid::Integer, config::WorkerConfig).connect should return a pair of IO objects, one for reading data sent from worker pid, and the other to write data that needs to be sent to worker pid. Custom cluster managers can use an in-memory BufferStream as the plumbing to proxy data between the custom, possibly non-IO transport and Julia's in-built parallel infrastructure.A BufferStream is an in-memory IOBuffer which behaves like an IO–it is a stream which can be handled asynchronously.Folder examples/clustermanager/0mq contains an example of using ZeroMQ to connect Julia workers in a star topology with a 0MQ broker in the middle. Note: The Julia processes are still all logically connected to each other–any worker can message any other worker directly without any awareness of 0MQ being used as the transport layer.When using custom transports:Julia workers must NOT be started with --worker. Starting with --worker will result in the newly launched workers defaulting to the TCP/IP socket transport implementation.\nFor every incoming logical connection with a worker, Base.process_messages(rd::IO, wr::IO)() must be called. This launches a new task that handles reading and writing of messages from/to the worker represented by the IO objects.\ninit_worker(cookie, manager::FooManager) MUST be called as part of worker process initialization.\nField connect_at::Any in WorkerConfig can be set by the cluster manager when launch() is called. The value of this field is passed in in all connect() callbacks. Typically, it carries information on how to connect to a worker. For example, the TCP/IP socket transport uses this field to specify the (host, port) tuple at which to connect to a worker.kill(manager, pid, config) is called to remove a worker from the cluster. On the master process, the corresponding IO objects must be closed by the implementation to ensure proper cleanup. The default implementation simply executes an exit() call on the specified remote worker.examples/clustermanager/simple is an example that shows a simple implementation using UNIX domain sockets for cluster setup."
},

{
    "location": "manual/parallel-computing.html#Network-Requirements-for-LocalManager-and-SSHManager-1",
    "page": "Parallel Computing",
    "title": "Network Requirements for LocalManager and SSHManager",
    "category": "section",
    "text": "Julia clusters are designed to be executed on already secured environments on infrastructure such as local laptops, departmental clusters, or even the cloud. This section covers network security requirements for the inbuilt LocalManager and SSHManager:The master process does not listen on any port. It only connects out to the workers.\nEach worker binds to only one of the local interfaces and listens on the first free port starting from 9009.\nLocalManager, used by addprocs(N), by default binds only to the loopback interface. This means that workers started later on remote hosts (or by anyone with malicious intentions) are unable to connect to the cluster. An addprocs(4) followed by an addprocs([\"remote_host\"]) will fail. Some users may need to create a cluster comprising their local system and a few remote systems. This can be done by explicitly requesting LocalManager to bind to an external network interface via the restrict keyword argument: addprocs(4; restrict=false).\nSSHManager, used by addprocs(list_of_remote_hosts), launches workers on remote hosts via SSH. By default SSH is only used to launch Julia workers. Subsequent master-worker and worker-worker connections use plain, unencrypted TCP/IP sockets. The remote hosts must have passwordless login enabled. Additional SSH flags or credentials may be specified via keyword argument sshflags.\naddprocs(list_of_remote_hosts; tunnel=true, sshflags=<ssh keys and other flags>) is useful when we wish to use SSH connections for master-worker too. A typical scenario for this is a local laptop running the Julia REPL (i.e., the master) with the rest of the cluster on the cloud, say on Amazon EC2. In this case only port 22 needs to be opened at the remote cluster coupled with SSH client authenticated via public key infrastructure (PKI). Authentication credentials can be supplied via sshflags, for example sshflags=`-e <keyfile>`.\nNote that worker-worker connections are still plain TCP and the local security policy on the remote cluster must allow for free connections between worker nodes, at least for ports 9009 and above.\nSecuring and encrypting all worker-worker traffic (via SSH) or encrypting individual messages can be done via a custom ClusterManager."
},

{
    "location": "manual/parallel-computing.html#Cluster-Cookie-1",
    "page": "Parallel Computing",
    "title": "Cluster Cookie",
    "category": "section",
    "text": "All processes in a cluster share the same cookie which, by default, is a randomly generated string on the master process:Base.cluster_cookie() returns the cookie, while Base.cluster_cookie(cookie)() sets it and returns the new cookie.\nAll connections are authenticated on both sides to ensure that only workers started by the master are allowed to connect to each other.\nThe cookie must be passed to the workers at startup via argument --worker <cookie>. Custom ClusterManagers can retrieve the cookie on the master by calling Base.cluster_cookie(). Cluster managers not using the default TCP/IP transport (and hence not specifying --worker) must call init_worker(cookie, manager) with the same cookie as on the master.Note that environments requiring higher levels of security can implement this via a custom ClusterManager. For example, cookies can be pre-shared and hence not specified as a startup argument."
},

{
    "location": "manual/parallel-computing.html#Specifying-Network-Topology-(Experimental)-1",
    "page": "Parallel Computing",
    "title": "Specifying Network Topology (Experimental)",
    "category": "section",
    "text": "The keyword argument topology passed to addprocs is used to specify how the workers must be connected to each other::all_to_all, the default: all workers are connected to each other.\n:master_slave: only the driver process, i.e. pid 1, has connections to the workers.\n:custom: the launch method of the cluster manager specifies the connection topology via the fields ident and connect_idents in WorkerConfig. A worker with a cluster-manager-provided identity ident will connect to all workers specified in connect_idents.Currently, sending a message between unconnected workers results in an error. This behaviour, as with the functionality and interface, should be considered experimental in nature and may change in future releases."
},

{
    "location": "manual/parallel-computing.html#Multi-Threading-(Experimental)-1",
    "page": "Parallel Computing",
    "title": "Multi-Threading (Experimental)",
    "category": "section",
    "text": "In addition to tasks, remote calls, and remote references, Julia from v0.5 forwards will natively support multi-threading. Note that this section is experimental and the interfaces may change in the future."
},

{
    "location": "manual/parallel-computing.html#Setup-1",
    "page": "Parallel Computing",
    "title": "Setup",
    "category": "section",
    "text": "By default, Julia starts up with a single thread of execution. This can be verified by using the command Threads.nthreads():julia> Threads.nthreads()\n1The number of threads Julia starts up with is controlled by an environment variable called JULIA_NUM_THREADS. Now, let's start up Julia with 4 threads:export JULIA_NUM_THREADS=4(The above command works on bourne shells on Linux and OSX. Note that if you're using a C shell on these platforms, you should use the keyword set instead of export. If you're on Windows, start up the command line in the location of julia.exe and use set instead of export.)Let's verify there are 4 threads at our disposal.julia> Threads.nthreads()\n4But we are currently on the master thread. To check, we use the command Threads.threadid()julia> Threads.threadid()\n1"
},

{
    "location": "manual/parallel-computing.html#The-@threads-Macro-1",
    "page": "Parallel Computing",
    "title": "The @threads Macro",
    "category": "section",
    "text": "Let's work a simple example using our native threads. Let us create an array of zeros:julia> a = zeros(10)\n10-element Array{Float64,1}:\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0Let us operate on this array simultaneously using 4 threads. We'll have each thread write its thread ID into each location.Julia supports parallel loops using the Threads.@threads macro. This macro is affixed in front of a for loop to indicate to Julia that the loop is a multi-threaded region:julia> Threads.@threads for i = 1:10\n           a[i] = Threads.threadid()\n       endThe iteration space is split amongst the threads, after which each thread writes its thread ID to its assigned locations:julia> a\n10-element Array{Float64,1}:\n 1.0\n 1.0\n 1.0\n 2.0\n 2.0\n 2.0\n 3.0\n 3.0\n 4.0\n 4.0Note that Threads.@threads does not have an optional reduction parameter like @parallel."
},

{
    "location": "manual/parallel-computing.html#@threadcall-(Experimental)-1",
    "page": "Parallel Computing",
    "title": "@threadcall (Experimental)",
    "category": "section",
    "text": "All I/O tasks, timers, REPL commands, etc are multiplexed onto a single OS thread via an event loop. A patched version of libuv (http://docs.libuv.org/en/v1.x/) provides this functionality. Yield points provide for co-operatively scheduling multiple tasks onto the same OS thread. I/O tasks and timers yield implicitly while waiting for the event to occur. Calling yield() explicitly allows for other tasks to be scheduled.Thus, a task executing a ccall effectively prevents the Julia scheduler from executing any other tasks till the call returns. This is true for all calls into external libraries. Exceptions are calls into custom C code that call back into Julia (which may then yield) or C code that calls jl_yield() (C equivalent of yield()).Note that while Julia code runs on a single thread (by default), libraries used by Julia may launch their own internal threads. For example, the BLAS library may start as many threads as there are cores on a machine.The @threadcall macro addresses scenarios where we do not want a ccall to block the main Julia event loop. It schedules a C function for execution in a separate thread. A threadpool with a default size of 4 is used for this. The size of the threadpool is controlled via environment variable UV_THREADPOOL_SIZE. While waiting for a free thread, and during function execution once a thread is available, the requesting task (on the main Julia event loop) yields to other tasks. Note that @threadcall does not return till the execution is complete. From a user point of view, it is therefore a blocking call like other Julia APIs.It is very important that the called function does not call back into Julia.@threadcall may be removed/changed in future versions of Julia.[1]: In this context, MPI refers to the MPI-1 standard. Beginning with MPI-2, the MPI standards committee introduced a new set of communication mechanisms, collectively referred to as Remote Memory Access (RMA). The motivation for adding RMA to the MPI standard was to facilitate one-sided communication patterns. For additional information on the latest MPI standard, see http://mpi-forum.org/docs."
},

{
    "location": "manual/dates.html#",
    "page": "Date and DateTime",
    "title": "Date and DateTime",
    "category": "page",
    "text": ""
},

{
    "location": "manual/dates.html#Date-and-DateTime-1",
    "page": "Date and DateTime",
    "title": "Date and DateTime",
    "category": "section",
    "text": "CurrentModule = Base.DatesThe Dates module provides two types for working with dates: Date and DateTime, representing day and millisecond precision, respectively; both are subtypes of the abstract TimeType. The motivation for distinct types is simple: some operations are much simpler, both in terms of code and mental reasoning, when the complexities of greater precision don't have to be dealt with. For example, since the Date type only resolves to the precision of a single date (i.e. no hours, minutes, or seconds), normal considerations for time zones, daylight savings/summer time, and leap seconds are unnecessary and avoided.Both Date and DateTime are basically immutable Int64 wrappers. The single instant field of either type is actually a UTInstant{P} type, which represents a continuously increasing machine timeline based on the UT second [1]. The DateTime type is not aware of time zones (naive, in Python parlance), analogous to a LocalDateTime in Java 8. Additional time zone functionality can be added through the TimeZones.jl package, which compiles the IANA time zone database. Both Date and DateTime are based on the ISO 8601 standard, which follows the proleptic Gregorian calendar. One note is that the ISO 8601 standard is particular about BC/BCE dates. In general, the last day of the BC/BCE era, 1-12-31 BC/BCE, was followed by 1-1-1 AD/CE, thus no year zero exists. The ISO standard, however, states that 1 BC/BCE is year zero, so 0000-12-31 is the day before 0001-01-01, and year -0001 (yes, negative one for the year) is 2 BC/BCE, year -0002 is 3 BC/BCE, etc.[1]: The notion of the UT second is actually quite fundamental. There are basically two different notions of time generally accepted, one based on the physical rotation of the earth (one full rotation = 1 day), the other based on the SI second (a fixed, constant value). These are radically different! Think about it, a \"UT second\", as defined relative to the rotation of the earth, may have a different absolute length depending on the day! Anyway, the fact that Date and DateTime are based on UT seconds is a simplifying, yet honest assumption so that things like leap seconds and all their complexity can be avoided. This basis of time is formally called UT or UT1. Basing types on the UT second basically means that every minute has 60 seconds and every day has 24 hours and leads to more natural calculations when working with calendar dates."
},

{
    "location": "manual/dates.html#Constructors-1",
    "page": "Date and DateTime",
    "title": "Constructors",
    "category": "section",
    "text": "Date and DateTime types can be constructed by integer or Period types, by parsing, or through adjusters (more on those later):julia> DateTime(2013)\n2013-01-01T00:00:00\n\njulia> DateTime(2013,7)\n2013-07-01T00:00:00\n\njulia> DateTime(2013,7,1)\n2013-07-01T00:00:00\n\njulia> DateTime(2013,7,1,12)\n2013-07-01T12:00:00\n\njulia> DateTime(2013,7,1,12,30)\n2013-07-01T12:30:00\n\njulia> DateTime(2013,7,1,12,30,59)\n2013-07-01T12:30:59\n\njulia> DateTime(2013,7,1,12,30,59,1)\n2013-07-01T12:30:59.001\n\njulia> Date(2013)\n2013-01-01\n\njulia> Date(2013,7)\n2013-07-01\n\njulia> Date(2013,7,1)\n2013-07-01\n\njulia> Date(Dates.Year(2013),Dates.Month(7),Dates.Day(1))\n2013-07-01\n\njulia> Date(Dates.Month(7),Dates.Year(2013))\n2013-07-01Date or DateTime parsing is accomplished by the use of format strings. Format strings work by the notion of defining delimited or fixed-width \"slots\" that contain a period to parse and passing the text to parse and format string to a Date or DateTime constructor, of the form Date(\"2015-01-01\",\"y-m-d\") or DateTime(\"20150101\",\"yyyymmdd\").Delimited slots are marked by specifying the delimiter the parser should expect between two subsequent periods; so \"y-m-d\" lets the parser know that between the first and second slots in a date string like \"2014-07-16\", it should find the - character. The y, m, and d characters let the parser know which periods to parse in each slot.Fixed-width slots are specified by repeating the period character the number of times corresponding to the width with no delimiter between characters. So \"yyyymmdd\" would correspond to a date string like \"20140716\". The parser distinguishes a fixed-width slot by the absence of a delimiter, noting the transition \"yyyymm\" from one period character to the next.Support for text-form month parsing is also supported through the u and U characters, for abbreviated and full-length month names, respectively. By default, only English month names are supported, so u corresponds to \"Jan\", \"Feb\", \"Mar\", etc. And U corresponds to \"January\", \"February\", \"March\", etc. Similar to other name=>value mapping functions dayname() and monthname(), custom locales can be loaded by passing in the locale=>Dict{String,Int} mapping to the MONTHTOVALUEABBR and MONTHTOVALUE dicts for abbreviated and full-name month names, respectively.One note on parsing performance: using the Date(date_string,format_string) function is fine if only called a few times. If there are many similarly formatted date strings to parse however, it is much more efficient to first create a Dates.DateFormat, and pass it instead of a raw format string.julia> df = DateFormat(\"y-m-d\");\n\njulia> dt = Date(\"2015-01-01\",df)\n2015-01-01\n\njulia> dt2 = Date(\"2015-01-02\",df)\n2015-01-02You can also use the dateformat\"\" string macro. This macro creates the DateFormat object once when the macro is expanded and uses the same DateFormat object even if a code snippet is run multiple times.julia> for i = 1:10^5\n           Date(\"2015-01-01\", dateformat\"y-m-d\")\n       endA full suite of parsing and formatting tests and examples is available in tests/dates/io.jl."
},

{
    "location": "manual/dates.html#Durations/Comparisons-1",
    "page": "Date and DateTime",
    "title": "Durations/Comparisons",
    "category": "section",
    "text": "Finding the length of time between two Date or DateTime is straightforward given their underlying representation as UTInstant{Day} and UTInstant{Millisecond}, respectively. The difference between Date is returned in the number of Day, and DateTime in the number of Millisecond. Similarly, comparing TimeType is a simple matter of comparing the underlying machine instants (which in turn compares the internal Int64 values).julia> dt = Date(2012,2,29)\n2012-02-29\n\njulia> dt2 = Date(2000,2,1)\n2000-02-01\n\njulia> dump(dt)\nDate\n  instant: Base.Dates.UTInstant{Base.Dates.Day}\n    periods: Base.Dates.Day\n      value: Int64 734562\n\njulia> dump(dt2)\nDate\n  instant: Base.Dates.UTInstant{Base.Dates.Day}\n    periods: Base.Dates.Day\n      value: Int64 730151\n\njulia> dt > dt2\ntrue\n\njulia> dt != dt2\ntrue\n\njulia> dt + dt2\nERROR: MethodError: no method matching +(::Date, ::Date)\n[...]\n\njulia> dt * dt2\nERROR: MethodError: no method matching *(::Date, ::Date)\n[...]\n\njulia> dt / dt2\nERROR: MethodError: no method matching /(::Date, ::Date)\n[...]\n\njulia> dt - dt2\n4411 days\n\njulia> dt2 - dt\n-4411 days\n\njulia> dt = DateTime(2012,2,29)\n2012-02-29T00:00:00\n\njulia> dt2 = DateTime(2000,2,1)\n2000-02-01T00:00:00\n\njulia> dt - dt2\n381110400000 milliseconds"
},

{
    "location": "manual/dates.html#Accessor-Functions-1",
    "page": "Date and DateTime",
    "title": "Accessor Functions",
    "category": "section",
    "text": "Because the Date and DateTime types are stored as single Int64 values, date parts or fields can be retrieved through accessor functions. The lowercase accessors return the field as an integer:julia> t = Date(2014, 1, 31)\n2014-01-31\n\njulia> Dates.year(t)\n2014\n\njulia> Dates.month(t)\n1\n\njulia> Dates.week(t)\n5\n\njulia> Dates.day(t)\n31While propercase return the same value in the corresponding Period type:julia> Dates.Year(t)\n2014 years\n\njulia> Dates.Day(t)\n31 daysCompound methods are provided, as they provide a measure of efficiency if multiple fields are needed at the same time:julia> Dates.yearmonth(t)\n(2014, 1)\n\njulia> Dates.monthday(t)\n(1, 31)\n\njulia> Dates.yearmonthday(t)\n(2014, 1, 31)One may also access the underlying UTInstant or integer value:julia> dump(t)\nDate\n  instant: Base.Dates.UTInstant{Base.Dates.Day}\n    periods: Base.Dates.Day\n      value: Int64 735264\n\njulia> t.instant\nBase.Dates.UTInstant{Base.Dates.Day}(735264 days)\n\njulia> Dates.value(t)\n735264"
},

{
    "location": "manual/dates.html#Query-Functions-1",
    "page": "Date and DateTime",
    "title": "Query Functions",
    "category": "section",
    "text": "Query functions provide calendrical information about a TimeType. They include information about the day of the week:julia> t = Date(2014, 1, 31)\n2014-01-31\n\njulia> Dates.dayofweek(t)\n5\n\njulia> Dates.dayname(t)\n\"Friday\"\n\njulia> Dates.dayofweekofmonth(t) # 5th Friday of January\n5Month of the year:julia> Dates.monthname(t)\n\"January\"\n\njulia> Dates.daysinmonth(t)\n31As well as information about the TimeType's year and quarter:julia> Dates.isleapyear(t)\nfalse\n\njulia> Dates.dayofyear(t)\n31\n\njulia> Dates.quarterofyear(t)\n1\n\njulia> Dates.dayofquarter(t)\n31The dayname() and monthname() methods can also take an optional locale keyword that can be used to return the name of the day or month of the year for other languages/locales. There are also versions of these functions returning the abbreviated names, namely dayabbr() and monthabbr(). First the mapping is loaded into the LOCALES variable:julia> french_months = [\"janvier\", \"février\", \"mars\", \"avril\", \"mai\", \"juin\",\n                        \"juillet\", \"août\", \"septembre\", \"octobre\", \"novembre\", \"décembre\"];\n\njulia> french_monts_abbrev = [\"janv\",\"févr\",\"mars\",\"avril\",\"mai\",\"juin\",\n                              \"juil\",\"août\",\"sept\",\"oct\",\"nov\",\"déc\"];\n\njulia> french_days = [\"lundi\",\"mardi\",\"mercredi\",\"jeudi\",\"vendredi\",\"samedi\",\"dimanche\"];\n\njulia> Dates.LOCALES[\"french\"] = Dates.DateLocale(french_months, french_monts_abbrev, french_days, [\"\"]);The above mentioned functions can then be used to perform the queries:julia> Dates.dayname(t;locale=\"french\")\n\"vendredi\"\n\njulia> Dates.monthname(t;locale=\"french\")\n\"janvier\"\n\njulia> Dates.monthabbr(t;locale=\"french\")\n\"janv\"Since the abbreviated versions of the days are not loaded, trying to use the function dayabbr() will error.julia> Dates.dayabbr(t;locale=\"french\")\nERROR: BoundsError: attempt to access 1-element Array{String,1} at index [5]\nStacktrace:\n [1] #dayabbr#6(::String, ::Function, ::Int64) at ./dates/query.jl:114\n [2] (::Base.Dates.#kw##dayabbr)(::Array{Any,1}, ::Base.Dates.#dayabbr, ::Int64) at ./<missing>:0 (repeats 2 times)"
},

{
    "location": "manual/dates.html#TimeType-Period-Arithmetic-1",
    "page": "Date and DateTime",
    "title": "TimeType-Period Arithmetic",
    "category": "section",
    "text": "It's good practice when using any language/date framework to be familiar with how date-period arithmetic is handled as there are some tricky issues to deal with (though much less so for day-precision types).The Dates module approach tries to follow the simple principle of trying to change as little as possible when doing Period arithmetic. This approach is also often known as calendrical arithmetic or what you would probably guess if someone were to ask you the same calculation in a conversation. Why all the fuss about this? Let's take a classic example: add 1 month to January 31st, 2014. What's the answer? Javascript will say March 3 (assumes 31 days). PHP says March 2 (assumes 30 days). The fact is, there is no right answer. In the Dates module, it gives the result of February 28th. How does it figure that out? I like to think of the classic 7-7-7 gambling game in casinos.Now just imagine that instead of 7-7-7, the slots are Year-Month-Day, or in our example, 2014-01-31. When you ask to add 1 month to this date, the month slot is incremented, so now we have 2014-02-31. Then the day number is checked if it is greater than the last valid day of the new month; if it is (as in the case above), the day number is adjusted down to the last valid day (28). What are the ramifications with this approach? Go ahead and add another month to our date, 2014-02-28 + Month(1) == 2014-03-28. What? Were you expecting the last day of March? Nope, sorry, remember the 7-7-7 slots. As few slots as possible are going to change, so we first increment the month slot by 1, 2014-03-28, and boom, we're done because that's a valid date. On the other hand, if we were to add 2 months to our original date, 2014-01-31, then we end up with 2014-03-31, as expected. The other ramification of this approach is a loss in associativity when a specific ordering is forced (i.e. adding things in different orders results in different outcomes). For example:julia> (Date(2014,1,29)+Dates.Day(1)) + Dates.Month(1)\n2014-02-28\n\njulia> (Date(2014,1,29)+Dates.Month(1)) + Dates.Day(1)\n2014-03-01What's going on there? In the first line, we're adding 1 day to January 29th, which results in 2014-01-30; then we add 1 month, so we get 2014-02-30, which then adjusts down to 2014-02-28. In the second example, we add 1 month first, where we get 2014-02-29, which adjusts down to 2014-02-28, and then add 1 day, which results in 2014-03-01. One design principle that helps in this case is that, in the presence of multiple Periods, the operations will be ordered by the Periods' types, not their value or positional order; this means Year will always be added first, then Month, then Week, etc. Hence the following does result in associativity and Just Works:julia> Date(2014,1,29) + Dates.Day(1) + Dates.Month(1)\n2014-03-01\n\njulia> Date(2014,1,29) + Dates.Month(1) + Dates.Day(1)\n2014-03-01Tricky? Perhaps. What is an innocent Dates user to do? The bottom line is to be aware that explicitly forcing a certain associativity, when dealing with months, may lead to some unexpected results, but otherwise, everything should work as expected. Thankfully, that's pretty much the extent of the odd cases in date-period arithmetic when dealing with time in UT (avoiding the \"joys\" of dealing with daylight savings, leap seconds, etc.).As a bonus, all period arithmetic objects work directly with ranges:julia> dr = Date(2014,1,29):Date(2014,2,3)\n2014-01-29:1 day:2014-02-03\n\njulia> collect(dr)\n6-element Array{Date,1}:\n 2014-01-29\n 2014-01-30\n 2014-01-31\n 2014-02-01\n 2014-02-02\n 2014-02-03\n\njulia> dr = Date(2014,1,29):Dates.Month(1):Date(2014,07,29)\n2014-01-29:1 month:2014-07-29\n\njulia> collect(dr)\n7-element Array{Date,1}:\n 2014-01-29\n 2014-02-28\n 2014-03-29\n 2014-04-29\n 2014-05-29\n 2014-06-29\n 2014-07-29"
},

{
    "location": "manual/dates.html#Adjuster-Functions-1",
    "page": "Date and DateTime",
    "title": "Adjuster Functions",
    "category": "section",
    "text": "As convenient as date-period arithmetics are, often the kinds of calculations needed on dates take on a calendrical or temporal nature rather than a fixed number of periods. Holidays are a perfect example; most follow rules such as \"Memorial Day = Last Monday of May\", or \"Thanksgiving = 4th Thursday of November\". These kinds of temporal expressions deal with rules relative to the calendar, like first or last of the month, next Tuesday, or the first and third Wednesdays, etc.The Dates module provides the adjuster API through several convenient methods that aid in simply and succinctly expressing temporal rules. The first group of adjuster methods deal with the first and last of weeks, months, quarters, and years. They each take a single TimeType as input and return or adjust to the first or last of the desired period relative to the input.julia> Dates.firstdayofweek(Date(2014,7,16)) # Adjusts the input to the Monday of the input's week\n2014-07-14\n\njulia> Dates.lastdayofmonth(Date(2014,7,16)) # Adjusts to the last day of the input's month\n2014-07-31\n\njulia> Dates.lastdayofquarter(Date(2014,7,16)) # Adjusts to the last day of the input's quarter\n2014-09-30The next two higher-order methods, tonext(), and toprev(), generalize working with temporal expressions by taking a DateFunction as first argument, along with a starting TimeType. A DateFunction is just a function, usually anonymous, that takes a single TimeType as input and returns a Bool, true indicating a satisfied adjustment criterion. For example:julia> istuesday = x->Dates.dayofweek(x) == Dates.Tuesday # Returns true if the day of the week of x is Tuesday\n(::#1) (generic function with 1 method)\n\njulia> Dates.tonext(istuesday, Date(2014,7,13)) # 2014-07-13 is a Sunday\n2014-07-15\n\njulia> Dates.tonext(Date(2014,7,13), Dates.Tuesday) # Convenience method provided for day of the week adjustments\n2014-07-15This is useful with the do-block syntax for more complex temporal expressions:julia> Dates.tonext(Date(2014,7,13)) do x\n           # Return true on the 4th Thursday of November (Thanksgiving)\n           Dates.dayofweek(x) == Dates.Thursday &&\n           Dates.dayofweekofmonth(x) == 4 &&\n           Dates.month(x) == Dates.November\n       end\n2014-11-27The Base.filter() method can be used to obtain all valid dates/moments in a specified range:# Pittsburgh street cleaning; Every 2nd Tuesday from April to November\n# Date range from January 1st, 2014 to January 1st, 2015\njulia> dr = Dates.Date(2014):Dates.Date(2015);\n\njulia> filter(dr) do x\n           Dates.dayofweek(x) == Dates.Tue &&\n           Dates.April <= Dates.month(x) <= Dates.Nov &&\n           Dates.dayofweekofmonth(x) == 2\n       end\n8-element Array{Date,1}:\n 2014-04-08\n 2014-05-13\n 2014-06-10\n 2014-07-08\n 2014-08-12\n 2014-09-09\n 2014-10-14\n 2014-11-11Additional examples and tests are available in test/dates/adjusters.jl."
},

{
    "location": "manual/dates.html#Period-Types-1",
    "page": "Date and DateTime",
    "title": "Period Types",
    "category": "section",
    "text": "Periods are a human view of discrete, sometimes irregular durations of time. Consider 1 month; it could represent, in days, a value of 28, 29, 30, or 31 depending on the year and month context. Or a year could represent 365 or 366 days in the case of a leap year. Period types are simple Int64 wrappers and are constructed by wrapping any Int64 convertible type, i.e. Year(1) or Month(3.0). Arithmetic between Period of the same type behave like integers, and limited Period-Real arithmetic is available.julia> y1 = Dates.Year(1)\n1 year\n\njulia> y2 = Dates.Year(2)\n2 years\n\njulia> y3 = Dates.Year(10)\n10 years\n\njulia> y1 + y2\n3 years\n\njulia> div(y3,y2)\n5\n\njulia> y3 - y2\n8 years\n\njulia> y3 % y2\n0 years\n\njulia> div(y3,3) # mirrors integer division\n3 years"
},

{
    "location": "manual/dates.html#Rounding-1",
    "page": "Date and DateTime",
    "title": "Rounding",
    "category": "section",
    "text": "Date and DateTime values can be rounded to a specified resolution (e.g., 1 month or 15 minutes) with floor(), ceil(), or round():julia> floor(Date(1985, 8, 16), Dates.Month)\n1985-08-01\n\njulia> ceil(DateTime(2013, 2, 13, 0, 31, 20), Dates.Minute(15))\n2013-02-13T00:45:00\n\njulia> round(DateTime(2016, 8, 6, 20, 15), Dates.Day)\n2016-08-07T00:00:00Unlike the numeric round() method, which breaks ties toward the even number by default, the TimeTyperound() method uses the RoundNearestTiesUp rounding mode. (It's difficult to guess what breaking ties to nearest \"even\" TimeType would entail.) Further details on the available RoundingMode s can be found in the API reference.Rounding should generally behave as expected, but there are a few cases in which the expected behaviour is not obvious."
},

{
    "location": "manual/dates.html#Rounding-Epoch-1",
    "page": "Date and DateTime",
    "title": "Rounding Epoch",
    "category": "section",
    "text": "In many cases, the resolution specified for rounding (e.g., Dates.Second(30)) divides evenly into the next largest period (in this case, Dates.Minute(1)). But rounding behaviour in cases in which this is not true may lead to confusion. What is the expected result of rounding a DateTime to the nearest 10 hours?julia> round(DateTime(2016, 7, 17, 11, 55), Dates.Hour(10))\n2016-07-17T12:00:00That may seem confusing, given that the hour (12) is not divisible by 10. The reason that 2016-07-17T12:00:00 was chosen is that it is 17,676,660 hours after 0000-01-01T00:00:00, and 17,676,660 is divisible by 10.As Julia Date and DateTime values are represented according to the ISO 8601 standard, 0000-01-01T00:00:00 was chosen as base (or \"rounding epoch\") from which to begin the count of days (and milliseconds) used in rounding calculations. (Note that this differs slightly from Julia's internal representation of Date s using Rata Die notation; but since the ISO 8601 standard is most visible to the end user, 0000-01-01T00:00:00 was chosen as the rounding epoch instead of the 0000-12-31T00:00:00 used internally to minimize confusion.)The only exception to the use of 0000-01-01T00:00:00 as the rounding epoch is when rounding to weeks. Rounding to the nearest week will always return a Monday (the first day of the week as specified by ISO 8601). For this reason, we use 0000-01-03T00:00:00 (the first day of the first week of year 0000, as defined by ISO 8601) as the base when rounding to a number of weeks.Here is a related case in which the expected behaviour is not necessarily obvious: What happens when we round to the nearest P(2), where P is a Period type? In some cases (specifically, when P <: Dates.TimePeriod) the answer is clear:julia> round(DateTime(2016, 7, 17, 8, 55, 30), Dates.Hour(2))\n2016-07-17T08:00:00\n\njulia> round(DateTime(2016, 7, 17, 8, 55, 30), Dates.Minute(2))\n2016-07-17T08:56:00This seems obvious, because two of each of these periods still divides evenly into the next larger order period. But in the case of two months (which still divides evenly into one year), the answer may be surprising:julia> round(DateTime(2016, 7, 17, 8, 55, 30), Dates.Month(2))\n2016-07-01T00:00:00Why round to the first day in July, even though it is month 7 (an odd number)? The key is that months are 1-indexed (the first month is assigned 1), unlike hours, minutes, seconds, and milliseconds (the first of which are assigned 0).This means that rounding a DateTime to an even multiple of seconds, minutes, hours, or years (because the ISO 8601 specification includes a year zero) will result in a DateTime with an even value in that field, while rounding a DateTime to an even multiple of months will result in the months field having an odd value. Because both months and years may contain an irregular number of days, whether rounding to an even number of days will result in an even value in the days field is uncertain.See the API reference for additional information on methods exported from the Dates module."
},

{
    "location": "manual/interacting-with-julia.html#",
    "page": "Interacting With Julia",
    "title": "Interacting With Julia",
    "category": "page",
    "text": ""
},

{
    "location": "manual/interacting-with-julia.html#Interacting-With-Julia-1",
    "page": "Interacting With Julia",
    "title": "Interacting With Julia",
    "category": "section",
    "text": "Julia comes with a full-featured interactive command-line REPL (read-eval-print loop) built into the julia executable. In addition to allowing quick and easy evaluation of Julia statements, it has a searchable history, tab-completion, many helpful keybindings, and dedicated help and shell modes. The REPL can be started by simply calling julia with no arguments or double-clicking on the executable:$ julia\n               _\n   _       _ _(_)_     |  A fresh approach to technical computing\n  (_)     | (_) (_)    |  Documentation: https://docs.julialang.org\n   _ _   _| |_  __ _   |  Type \"?help\" for help.\n  | | | | | | |/ _` |  |\n  | | |_| | | | (_| |  |  Version 0.6.0-dev.2493 (2017-01-31 18:53 UTC)\n _/ |\\__'_|_|_|\\__'_|  |  Commit c99e12c* (0 days old master)\n|__/                   |  x86_64-linux-gnu\n\njulia>To exit the interactive session, type ^D – the control key together with the d key on a blank line – or type quit() followed by the return or enter key. The REPL greets you with a banner and a julia> prompt."
},

{
    "location": "manual/interacting-with-julia.html#The-different-prompt-modes-1",
    "page": "Interacting With Julia",
    "title": "The different prompt modes",
    "category": "section",
    "text": ""
},

{
    "location": "manual/interacting-with-julia.html#The-Julian-mode-1",
    "page": "Interacting With Julia",
    "title": "The Julian mode",
    "category": "section",
    "text": "The REPL has four main modes of operation. The first and most common is the Julian prompt. It is the default mode of operation; each new line initially starts with julia>. It is here that you can enter Julia expressions. Hitting return or enter after a complete expression has been entered will evaluate the entry and show the result of the last expression.julia> string(1 + 2)\n\"3\"There are a number useful features unique to interactive work. In addition to showing the result, the REPL also binds the result to the variable ans. A trailing semicolon on the line can be used as a flag to suppress showing the result.julia> string(3 * 4);\n\njulia> ans\n\"12\"In Julia mode, the REPL supports something called prompt pasting. This activates when pasting text that starts with julia>  into the REPL. In that case, only expressions starting with julia>  are parsed, others are removed. This makes it is possible to paste a chunk of code that has been copied from a REPL session without having to scrub away prompts and outputs. This feature is enabled by default but can be disabled or enabled at will with Base.REPL.enable_promptpaste(::Bool). If it is enabled, you can try it out by pasting the code block above this paragraph straight into the REPL. This feature does not work on the standard Windows command prompt due to its limitation at detecting when a paste occurs."
},

{
    "location": "manual/interacting-with-julia.html#Help-mode-1",
    "page": "Interacting With Julia",
    "title": "Help mode",
    "category": "section",
    "text": "When the cursor is at the beginning of the line, the prompt can be changed to a help mode by typing ?. Julia will attempt to print help or documentation for anything entered in help mode:julia> ? # upon typing ?, the prompt changes (in place) to: help?>\n\nhelp?> string\nsearch: string String stringmime Cstring Cwstring RevString readstring randstring bytestring SubString\n\n  string(xs...)\n\n  Create a string from any values using the print function.Macros, types and variables can also be queried:help?> @time\n  @time\n\n  A macro to execute an expression, printing the time it took to execute, the number of allocations,\n  and the total number of bytes its execution caused to be allocated, before returning the value of the\n  expression.\n\n  See also @timev, @timed, @elapsed, and @allocated.\n\nhelp?> AbstractString\nsearch: AbstractString AbstractSparseMatrix AbstractSparseVector AbstractSet\n\n  No documentation found.\n\n  Summary:\n\n  abstract AbstractString <: Any\n\n  Subtypes:\n\n  Base.Test.GenericString\n  DirectIndexString\n  StringHelp mode can be exited by pressing backspace at the beginning of the line."
},

{
    "location": "manual/interacting-with-julia.html#man-shell-mode-1",
    "page": "Interacting With Julia",
    "title": "Shell mode",
    "category": "section",
    "text": "Just as help mode is useful for quick access to documentation, another common task is to use the system shell to execute system commands. Just as ? entered help mode when at the beginning of the line, a semicolon (;) will enter the shell mode. And it can be exited by pressing backspace at the beginning of the line.julia> ; # upon typing ;, the prompt changes (in place) to: shell>\n\nshell> echo hello\nhello"
},

{
    "location": "manual/interacting-with-julia.html#Search-modes-1",
    "page": "Interacting With Julia",
    "title": "Search modes",
    "category": "section",
    "text": "In all of the above modes, the executed lines get saved to a history file, which can be searched.  To initiate an incremental search through the previous history, type ^R – the control key together with the r key. The prompt will change to (reverse-i-search)`':, and as you type the search query will appear in the quotes. The most recent result that matches the query will dynamically update to the right of the colon as more is typed. To find an older result using the same query, simply type ^R again.Just as ^R is a reverse search, ^S is a forward search, with the prompt (i-search)`':.  The two may be used in conjunction with each other to move through the previous or next matching results, respectively."
},

{
    "location": "manual/interacting-with-julia.html#Key-bindings-1",
    "page": "Interacting With Julia",
    "title": "Key bindings",
    "category": "section",
    "text": "The Julia REPL makes great use of key bindings. Several control-key bindings were already introduced above (^D to exit, ^R and ^S for searching), but there are many more. In addition to the control-key, there are also meta-key bindings. These vary more by platform, but most terminals default to using alt- or option- held down with a key to send the meta-key (or can be configured to do so).Keybinding Description\nProgram control  \n^D Exit (when buffer is empty)\n^C Interrupt or cancel\n^L Clear console screen\nReturn/Enter, ^J New line, executing if it is complete\nmeta-Return/Enter Insert new line without executing it\n? or ; Enter help or shell mode (when at start of a line)\n^R, ^S Incremental history search, described above\nCursor movement  \nRight arrow, ^F Move right one character\nLeft arrow, ^B Move left one character\nHome, ^A Move to beginning of line\nEnd, ^E Move to end of line\n^P Change to the previous or next history entry\n^N Change to the next history entry\nUp arrow Move up one line (or to the previous history entry)\nDown arrow Move down one line (or to the next history entry)\nPage-up Change to the previous history entry that matches the text before the cursor\nPage-down Change to the next history entry that matches the text before the cursor\nmeta-F Move right one word\nmeta-B Move left one word\nEditing  \nBackspace, ^H Delete the previous character\nDelete, ^D Forward delete one character (when buffer has text)\nmeta-Backspace Delete the previous word\nmeta-D Forward delete the next word\n^W Delete previous text up to the nearest whitespace\n^K \"Kill\" to end of line, placing the text in a buffer\n^Y \"Yank\" insert the text from the kill buffer\n^T Transpose the characters about the cursor\n^Q Write a number in REPL and press ^Q to open editor at corresponding stackframe"
},

{
    "location": "manual/interacting-with-julia.html#Customizing-keybindings-1",
    "page": "Interacting With Julia",
    "title": "Customizing keybindings",
    "category": "section",
    "text": "Julia's REPL keybindings may be fully customized to a user's preferences by passing a dictionary to REPL.setup_interface(). The keys of this dictionary may be characters or strings. The key '*' refers to the default action. Control plus character x bindings are indicated with \"^x\". Meta plus x can be written \"\\\\Mx\". The values of the custom keymap must be nothing (indicating that the input should be ignored) or functions that accept the signature (PromptState, AbstractREPL, Char). The REPL.setup_interface() function must be called before the REPL is initialized, by registering the operation with atreplinit(). For example, to bind the up and down arrow keys to move through history without prefix search, one could put the following code in .juliarc.jl:import Base: LineEdit, REPL\n\nconst mykeys = Dict{Any,Any}(\n    # Up Arrow\n    \"\\e[A\" => (s,o...)->(LineEdit.edit_move_up(s) || LineEdit.history_prev(s, LineEdit.mode(s).hist)),\n    # Down Arrow\n    \"\\e[B\" => (s,o...)->(LineEdit.edit_move_up(s) || LineEdit.history_next(s, LineEdit.mode(s).hist))\n)\n\nfunction customize_keys(repl)\n    repl.interface = REPL.setup_interface(repl; extra_repl_keymap = mykeys)\nend\n\natreplinit(customize_keys)Users should refer to base/LineEdit.jl to discover the available actions on key input."
},

{
    "location": "manual/interacting-with-julia.html#Tab-completion-1",
    "page": "Interacting With Julia",
    "title": "Tab completion",
    "category": "section",
    "text": "In both the Julian and help modes of the REPL, one can enter the first few characters of a function or type and then press the tab key to get a list all matches:julia> stri[TAB]\nstride     strides     string      stringmime  strip\n\njulia> Stri[TAB]\nStridedArray    StridedMatrix    StridedVecOrMat  StridedVector    StringThe tab key can also be used to substitute LaTeX math symbols with their Unicode equivalents, and get a list of LaTeX matches as well:julia> \\pi[TAB]\njulia> π\nπ = 3.1415926535897...\n\njulia> e\\_1[TAB] = [1,0]\njulia> e₁ = [1,0]\n2-element Array{Int64,1}:\n 1\n 0\n\njulia> e\\^1[TAB] = [1 0]\njulia> e¹ = [1 0]\n1×2 Array{Int64,2}:\n 1  0\n\njulia> \\sqrt[TAB]2     # √ is equivalent to the sqrt() function\njulia> √2\n1.4142135623730951\n\njulia> \\hbar[TAB](h) = h / 2\\pi[TAB]\njulia> ħ(h) = h / 2π\nħ (generic function with 1 method)\n\njulia> \\h[TAB]\n\\hat              \\hermitconjmatrix  \\hkswarow          \\hrectangle\n\\hatapprox        \\hexagon           \\hookleftarrow     \\hrectangleblack\n\\hbar             \\hexagonblack      \\hookrightarrow    \\hslash\n\\heartsuit        \\hksearow          \\house             \\hspace\n\njulia> α=\"\\alpha[TAB]\"   # LaTeX completion also works in strings\njulia> α=\"α\"A full list of tab-completions can be found in the Unicode Input section of the manual.Completion of paths works for strings and julia's shell mode:julia> path=\"/[TAB]\"\n.dockerenv  .juliabox/   boot/        etc/         lib/         media/       opt/         root/        sbin/        sys/         usr/\n.dockerinit bin/         dev/         home/        lib64/       mnt/         proc/        run/         srv/         tmp/         var/\nshell> /[TAB]\n.dockerenv  .juliabox/   boot/        etc/         lib/         media/       opt/         root/        sbin/        sys/         usr/\n.dockerinit bin/         dev/         home/        lib64/       mnt/         proc/        run/         srv/         tmp/         var/Tab completion can help with investigation of the available methods matching the input arguments:julia> max([TAB] # All methods are displayed, not shown here due to size of the list\n\njulia> max([1, 2], [TAB] # All methods where `Vector{Int}` matches as first argument\nmax(x, y) in Base at operators.jl:215\nmax(a, b, c, xs...) in Base at operators.jl:281\n\njulia> max([1, 2], max(1, 2), [TAB] # All methods matching the arguments.\nmax(x, y) in Base at operators.jl:215\nmax(a, b, c, xs...) in Base at operators.jl:281Keywords are also displayed in the suggested methods, see second line after ; where limit and keep are keyword arguments:julia> split(\"1 1 1\", [TAB]\nsplit(str::AbstractString) in Base at strings/util.jl:278\nsplit{T<:AbstractString}(str::T, splitter; limit, keep) in Base at strings/util.jl:254The completion of the methods uses type inference and can therefore see if the arguments match even if the arguments are output from functions. The function needs to be type stable for the completion to be able to remove non-matching methods.Tab completion can also help completing fields:julia> Pkg.a[TAB]\nadd       availableFields for output from functions can also be completed:julia> split(\"\",\"\")[1].[TAB]\nendof  offset  stringThe completion of fields for output from functions uses type inference, and it can only suggest fields if the function is type stable."
},

{
    "location": "manual/interacting-with-julia.html#Customizing-Colors-1",
    "page": "Interacting With Julia",
    "title": "Customizing Colors",
    "category": "section",
    "text": "The colors used by Julia and the REPL can be customized, as well. To change the color of the Julia prompt you can add something like the following to your .juliarc.jl file, which is to be placed inside your home directory:function customize_colors(repl)\n    repl.prompt_color = Base.text_colors[:cyan]\nend\n\natreplinit(customize_colors)The available color keys can be seen by typing Base.text_colors in the help mode of the REPL. In addition, the integers 0 to 255 can be used as color keys for terminals with 256 color support.You can also change the colors for the help and shell prompts and input and answer text by setting the appropriate field of repl in the customize_colors function above (respectively, help_color, shell_color, input_color, and answer_color). For the latter two, be sure that the envcolors field is also set to false.It is also possible to apply boldface formatting by using Base.text_colors[:bold] as a color. For instance, to print answers in boldface font, one can use the following as a .juliarc.jl:function customize_colors(repl)\n    repl.envcolors = false\n    repl.answer_color = Base.text_colors[:bold]\nend\n\natreplinit(customize_colors)You can also customize the color used to render warning and informational messages by setting the appropriate environment variables. For instance, to render error, warning, and informational messages respectively in magenta, yellow, and cyan you can add the following to your .juliarc.jl file:ENV[\"JULIA_ERROR_COLOR\"] = :magenta\nENV[\"JULIA_WARN_COLOR\"] = :yellow\nENV[\"JULIA_INFO_COLOR\"] = :cyan"
},

{
    "location": "manual/running-external-programs.html#",
    "page": "Running External Programs",
    "title": "Running External Programs",
    "category": "page",
    "text": ""
},

{
    "location": "manual/running-external-programs.html#Running-External-Programs-1",
    "page": "Running External Programs",
    "title": "Running External Programs",
    "category": "section",
    "text": "Julia borrows backtick notation for commands from the shell, Perl, and Ruby. However, in Julia, writingjulia> `echo hello`\n`echo hello`differs in several aspects from the behavior in various shells, Perl, or Ruby:Instead of immediately running the command, backticks create a Cmd object to represent the command. You can use this object to connect the command to others via pipes, run it, and read or write to it.\nWhen the command is run, Julia does not capture its output unless you specifically arrange for it to. Instead, the output of the command by default goes to STDOUT as it would using libc's system call.\nThe command is never run with a shell. Instead, Julia parses the command syntax directly, appropriately interpolating variables and splitting on words as the shell would, respecting shell quoting syntax. The command is run as julia's immediate child process, using fork and exec calls.Here's a simple example of running an external program:julia> mycommand = `echo hello`\n`echo hello`\n\njulia> typeof(mycommand)\nCmd\n\njulia> run(mycommand)\nhelloThe hello is the output of the echo command, sent to STDOUT. The run method itself returns nothing, and throws an ErrorException if the external command fails to run successfully.If you want to read the output of the external command, readstring() can be used instead:julia> a = readstring(`echo hello`)\n\"hello\\n\"\n\njulia> chomp(a) == \"hello\"\ntrueMore generally, you can use open() to read from or write to an external command.julia> open(`less`, \"w\", STDOUT) do io\n           for i = 1:3\n               println(io, i)\n           end\n       end\n1\n2\n3"
},

{
    "location": "manual/running-external-programs.html#command-interpolation-1",
    "page": "Running External Programs",
    "title": "Interpolation",
    "category": "section",
    "text": "Suppose you want to do something a bit more complicated and use the name of a file in the variable file as an argument to a command. You can use $ for interpolation much as you would in a string literal (see Strings):julia> file = \"/etc/passwd\"\n\"/etc/passwd\"\n\njulia> `sort $file`\n`sort /etc/passwd`A common pitfall when running external programs via a shell is that if a file name contains characters that are special to the shell, they may cause undesirable behavior. Suppose, for example, rather than /etc/passwd, we wanted to sort the contents of the file /Volumes/External HD/data.csv. Let's try it:julia> file = \"/Volumes/External HD/data.csv\"\n\"/Volumes/External HD/data.csv\"\n\njulia> `sort $file`\n`sort '/Volumes/External HD/data.csv'`How did the file name get quoted? Julia knows that file is meant to be interpolated as a single argument, so it quotes the word for you. Actually, that is not quite accurate: the value of file is never interpreted by a shell, so there's no need for actual quoting; the quotes are inserted only for presentation to the user. This will even work if you interpolate a value as part of a shell word:julia> path = \"/Volumes/External HD\"\n\"/Volumes/External HD\"\n\njulia> name = \"data\"\n\"data\"\n\njulia> ext = \"csv\"\n\"csv\"\n\njulia> `sort $path/$name.$ext`\n`sort '/Volumes/External HD/data.csv'`As you can see, the space in the path variable is appropriately escaped. But what if you want to interpolate multiple words? In that case, just use an array (or any other iterable container):julia> files = [\"/etc/passwd\",\"/Volumes/External HD/data.csv\"]\n2-element Array{String,1}:\n \"/etc/passwd\"\n \"/Volumes/External HD/data.csv\"\n\njulia> `grep foo $files`\n`grep foo /etc/passwd '/Volumes/External HD/data.csv'`If you interpolate an array as part of a shell word, Julia emulates the shell's {a,b,c} argument generation:julia> names = [\"foo\",\"bar\",\"baz\"]\n3-element Array{String,1}:\n \"foo\"\n \"bar\"\n \"baz\"\n\njulia> `grep xylophone $names.txt`\n`grep xylophone foo.txt bar.txt baz.txt`Moreover, if you interpolate multiple arrays into the same word, the shell's Cartesian product generation behavior is emulated:julia> names = [\"foo\",\"bar\",\"baz\"]\n3-element Array{String,1}:\n \"foo\"\n \"bar\"\n \"baz\"\n\njulia> exts = [\"aux\",\"log\"]\n2-element Array{String,1}:\n \"aux\"\n \"log\"\n\njulia> `rm -f $names.$exts`\n`rm -f foo.aux foo.log bar.aux bar.log baz.aux baz.log`Since you can interpolate literal arrays, you can use this generative functionality without needing to create temporary array objects first:julia> `rm -rf $[\"foo\",\"bar\",\"baz\",\"qux\"].$[\"aux\",\"log\",\"pdf\"]`\n`rm -rf foo.aux foo.log foo.pdf bar.aux bar.log bar.pdf baz.aux baz.log baz.pdf qux.aux qux.log qux.pdf`"
},

{
    "location": "manual/running-external-programs.html#Quoting-1",
    "page": "Running External Programs",
    "title": "Quoting",
    "category": "section",
    "text": "Inevitably, one wants to write commands that aren't quite so simple, and it becomes necessary to use quotes. Here's a simple example of a Perl one-liner at a shell prompt:sh$ perl -le '$|=1; for (0..3) { print }'\n0\n1\n2\n3The Perl expression needs to be in single quotes for two reasons: so that spaces don't break the expression into multiple shell words, and so that uses of Perl variables like $| (yes, that's the name of a variable in Perl), don't cause interpolation. In other instances, you may want to use double quotes so that interpolation does occur:sh$ first=\"A\"\nsh$ second=\"B\"\nsh$ perl -le '$|=1; print for @ARGV' \"1: $first\" \"2: $second\"\n1: A\n2: BIn general, the Julia backtick syntax is carefully designed so that you can just cut-and-paste shell commands as is into backticks and they will work: the escaping, quoting, and interpolation behaviors are the same as the shell's. The only difference is that the interpolation is integrated and aware of Julia's notion of what is a single string value, and what is a container for multiple values. Let's try the above two examples in Julia:julia> A = `perl -le '$|=1; for (0..3) { print }'`\n`perl -le '$|=1; for (0..3) { print }'`\n\njulia> run(A)\n0\n1\n2\n3\n\njulia> first = \"A\"; second = \"B\";\n\njulia> B = `perl -le 'print for @ARGV' \"1: $first\" \"2: $second\"`\n`perl -le 'print for @ARGV' '1: A' '2: B'`\n\njulia> run(B)\n1: A\n2: BThe results are identical, and Julia's interpolation behavior mimics the shell's with some improvements due to the fact that Julia supports first-class iterable objects while most shells use strings split on spaces for this, which introduces ambiguities. When trying to port shell commands to Julia, try cut and pasting first. Since Julia shows commands to you before running them, you can easily and safely just examine its interpretation without doing any damage."
},

{
    "location": "manual/running-external-programs.html#Pipelines-1",
    "page": "Running External Programs",
    "title": "Pipelines",
    "category": "section",
    "text": "Shell metacharacters, such as |, &, and >, need to be quoted (or escaped) inside of Julia's backticks:julia> run(`echo hello '|' sort`)\nhello | sort\n\njulia> run(`echo hello \\| sort`)\nhello | sortThis expression invokes the echo command with three words as arguments: hello, |, and sort. The result is that a single line is printed: hello | sort. How, then, does one construct a pipeline? Instead of using '|' inside of backticks, one uses pipeline():julia> run(pipeline(`echo hello`, `sort`))\nhelloThis pipes the output of the echo command to the sort command. Of course, this isn't terribly interesting since there's only one line to sort, but we can certainly do much more interesting things:julia> run(pipeline(`cut -d: -f3 /etc/passwd`, `sort -n`, `tail -n5`))\n210\n211\n212\n213\n214This prints the highest five user IDs on a UNIX system. The cut, sort and tail commands are all spawned as immediate children of the current julia process, with no intervening shell process. Julia itself does the work to setup pipes and connect file descriptors that is normally done by the shell. Since Julia does this itself, it retains better control and can do some things that shells cannot.Julia can run multiple commands in parallel:julia> run(`echo hello` & `echo world`)\nworld\nhelloThe order of the output here is non-deterministic because the two echo processes are started nearly simultaneously, and race to make the first write to the STDOUT descriptor they share with each other and the julia parent process. Julia lets you pipe the output from both of these processes to another program:julia> run(pipeline(`echo world` & `echo hello`, `sort`))\nhello\nworldIn terms of UNIX plumbing, what's happening here is that a single UNIX pipe object is created and written to by both echo processes, and the other end of the pipe is read from by the sort command.IO redirection can be accomplished by passing keyword arguments stdin, stdout, and stderr to the pipeline function:pipeline(`do_work`, stdout=pipeline(`sort`, \"out.txt\"), stderr=\"errs.txt\")"
},

{
    "location": "manual/running-external-programs.html#Avoiding-Deadlock-in-Pipelines-1",
    "page": "Running External Programs",
    "title": "Avoiding Deadlock in Pipelines",
    "category": "section",
    "text": "When reading and writing to both ends of a pipeline from a single process, it is important to avoid forcing the kernel to buffer all of the data.For example, when reading all of the output from a command, call readstring(out), not wait(process), since the former will actively consume all of the data written by the process, whereas the latter will attempt to store the data in the kernel's buffers while waiting for a reader to be connected.Another common solution is to separate the reader and writer of the pipeline into separate Tasks:writer = @async writeall(process, \"data\")\nreader = @async do_compute(readstring(process))\nwait(process)\nfetch(reader)"
},

{
    "location": "manual/running-external-programs.html#Complex-Example-1",
    "page": "Running External Programs",
    "title": "Complex Example",
    "category": "section",
    "text": "The combination of a high-level programming language, a first-class command abstraction, and automatic setup of pipes between processes is a powerful one. To give some sense of the complex pipelines that can be created easily, here are some more sophisticated examples, with apologies for the excessive use of Perl one-liners:julia> prefixer(prefix, sleep) = `perl -nle '$|=1; print \"'$prefix' \", $_; sleep '$sleep';'`;\n\njulia> run(pipeline(`perl -le '$|=1; for(0..9){ print; sleep 1 }'`, prefixer(\"A\",2) & prefixer(\"B\",2)))\nA 0\nB 1\nA 2\nB 3\nA 4\nB 5\nA 6\nB 7\nA 8\nB 9This is a classic example of a single producer feeding two concurrent consumers: one perl process generates lines with the numbers 0 through 9 on them, while two parallel processes consume that output, one prefixing lines with the letter \"A\", the other with the letter \"B\". Which consumer gets the first line is non-deterministic, but once that race has been won, the lines are consumed alternately by one process and then the other. (Setting $|=1 in Perl causes each print statement to flush the STDOUT handle, which is necessary for this example to work. Otherwise all the output is buffered and printed to the pipe at once, to be read by just one consumer process.)Here is an even more complex multi-stage producer-consumer example:julia> run(pipeline(`perl -le '$|=1; for(0..9){ print; sleep 1 }'`,\n           prefixer(\"X\",3) & prefixer(\"Y\",3) & prefixer(\"Z\",3),\n           prefixer(\"A\",2) & prefixer(\"B\",2)))\nA X 0\nB Y 1\nA Z 2\nB X 3\nA Y 4\nB Z 5\nA X 6\nB Y 7\nA Z 8\nB X 9This example is similar to the previous one, except there are two stages of consumers, and the stages have different latency so they use a different number of parallel workers, to maintain saturated throughput.We strongly encourage you to try all these examples to see how they work."
},

{
    "location": "manual/calling-c-and-fortran-code.html#",
    "page": "Calling C and Fortran Code",
    "title": "Calling C and Fortran Code",
    "category": "page",
    "text": ""
},

{
    "location": "manual/calling-c-and-fortran-code.html#Calling-C-and-Fortran-Code-1",
    "page": "Calling C and Fortran Code",
    "title": "Calling C and Fortran Code",
    "category": "section",
    "text": "Though most code can be written in Julia, there are many high-quality, mature libraries for numerical computing already written in C and Fortran. To allow easy use of this existing code, Julia makes it simple and efficient to call C and Fortran functions. Julia has a \"no boilerplate\" philosophy: functions can be called directly from Julia without any \"glue\" code, code generation, or compilation – even from the interactive prompt. This is accomplished just by making an appropriate call with ccall syntax, which looks like an ordinary function call.The code to be called must be available as a shared library. Most C and Fortran libraries ship compiled as shared libraries already, but if you are compiling the code yourself using GCC (or Clang), you will need to use the -shared and -fPIC options. The machine instructions generated by Julia's JIT are the same as a native C call would be, so the resulting overhead is the same as calling a library function from C code. (Non-library function calls in both C and Julia can be inlined and thus may have even less overhead than calls to shared library functions. When both libraries and executables are generated by LLVM, it is possible to perform whole-program optimizations that can even optimize across this boundary, but Julia does not yet support that. In the future, however, it may do so, yielding even greater performance gains.)Shared libraries and functions are referenced by a tuple of the form (:function, \"library\") or (\"function\", \"library\") where function is the C-exported function name. library refers to the shared library name: shared libraries available in the (platform-specific) load path will be resolved by name, and if necessary a direct path may be specified.A function name may be used alone in place of the tuple (just :function or \"function\"). In this case the name is resolved within the current process. This form can be used to call C library functions, functions in the Julia runtime, or functions in an application linked to Julia.By default, Fortran compilers generate mangled names (for example, converting function names to lowercase or uppercase, often appending an underscore), and so to call a Fortran function via ccall you must pass the mangled identifier corresponding to the rule followed by your Fortran compiler.  Also, when calling a Fortran function, all inputs must be passed by reference.Finally, you can use ccall to actually generate a call to the library function. Arguments to ccall are as follows:A (:function, \"library\") pair, which must be written as a literal constant,\nOR\na function pointer (for example, from dlsym).\nReturn type (see below for mapping the declared C type to Julia)\nThis argument will be evaluated at compile-time, when the containing method is defined.\nA tuple of input types. The input types must be written as a literal tuple, not a tuple-valued variable or expression.\nThis argument will be evaluated at compile-time, when the containing method is defined.\nThe following arguments, if any, are the actual argument values passed to the function.As a complete but simple example, the following calls the clock function from the standard C library:julia> t = ccall((:clock, \"libc\"), Int32, ())\n2292761\n\njulia> t\n2292761\n\njulia> typeof(ans)\nInt32clock takes no arguments and returns an Int32. One common gotcha is that a 1-tuple must be written with a trailing comma. For example, to call the getenv function to get a pointer to the value of an environment variable, one makes a call like this:julia> path = ccall((:getenv, \"libc\"), Cstring, (Cstring,), \"SHELL\")\nCstring(@0x00007fff5fbffc45)\n\njulia> unsafe_string(path)\n\"/bin/bash\"Note that the argument type tuple must be written as (Cstring,), rather than (Cstring). This is because (Cstring) is just the expression Cstring surrounded by parentheses, rather than a 1-tuple containing Cstring:julia> (Cstring)\nCstring\n\njulia> (Cstring,)\n(Cstring,)In practice, especially when providing reusable functionality, one generally wraps ccall uses in Julia functions that set up arguments and then check for errors in whatever manner the C or Fortran function indicates them, propagating to the Julia caller as exceptions. This is especially important since C and Fortran APIs are notoriously inconsistent about how they indicate error conditions. For example, the getenv C library function is wrapped in the following Julia function, which is a simplified version of the actual definition from env.jl:function getenv(var::AbstractString)\n    val = ccall((:getenv, \"libc\"),\n                Cstring, (Cstring,), var)\n    if val == C_NULL\n        error(\"getenv: undefined variable: \", var)\n    end\n    unsafe_string(val)\nendThe C getenv function indicates an error by returning NULL, but other standard C functions indicate errors in various different ways, including by returning -1, 0, 1 and other special values. This wrapper throws an exception clearly indicating the problem if the caller tries to get a non-existent environment variable:julia> getenv(\"SHELL\")\n\"/bin/bash\"\n\njulia> getenv(\"FOOBAR\")\ngetenv: undefined variable: FOOBARHere is a slightly more complex example that discovers the local machine's hostname:function gethostname()\n    hostname = Vector{UInt8}(128)\n    ccall((:gethostname, \"libc\"), Int32,\n          (Ptr{UInt8}, Csize_t),\n          hostname, sizeof(hostname))\n    hostname[end] = 0; # ensure null-termination\n    return unsafe_string(pointer(hostname))\nendThis example first allocates an array of bytes, then calls the C library function gethostname to fill the array in with the hostname, takes a pointer to the hostname buffer, and converts the pointer to a Julia string, assuming that it is a NUL-terminated C string. It is common for C libraries to use this pattern of requiring the caller to allocate memory to be passed to the callee and filled in. Allocation of memory from Julia like this is generally accomplished by creating an uninitialized array and passing a pointer to its data to the C function. This is why we don't use the Cstring type here: as the array is uninitialized, it could contain NUL bytes. Converting to a Cstring as part of the ccall checks for contained NUL bytes and could therefore throw a conversion error."
},

{
    "location": "manual/calling-c-and-fortran-code.html#Creating-C-Compatible-Julia-Function-Pointers-1",
    "page": "Calling C and Fortran Code",
    "title": "Creating C-Compatible Julia Function Pointers",
    "category": "section",
    "text": "It is possible to pass Julia functions to native C functions that accept function pointer arguments. For example, to match C prototypes of the form:typedef returntype (*functiontype)(argumenttype,...)The function cfunction() generates the C-compatible function pointer for a call to a Julia library function. Arguments to cfunction() are as follows:A Julia Function\nReturn type\nA tuple of input typesA classic example is the standard C library qsort function, declared as:void qsort(void *base, size_t nmemb, size_t size,\n           int(*compare)(const void *a, const void *b));The base argument is a pointer to an array of length nmemb, with elements of size bytes each. compare is a callback function which takes pointers to two elements a and b and returns an integer less/greater than zero if a should appear before/after b (or zero if any order is permitted). Now, suppose that we have a 1d array A of values in Julia that we want to sort using the qsort function (rather than Julia's built-in sort function). Before we worry about calling qsort and passing arguments, we need to write a comparison function that works for some arbitrary type T:julia> function mycompare(a::T, b::T) where T\n           return convert(Cint, a < b ? -1 : a > b ? +1 : 0)::Cint\n       end\nmycompare (generic function with 1 method)Notice that we have to be careful about the return type: qsort expects a function returning a C int, so we must be sure to return Cint via a call to convert and a typeassert.In order to pass this function to C, we obtain its address using the function cfunction:julia> const mycompare_c = cfunction(mycompare, Cint, (Ref{Cdouble}, Ref{Cdouble}));cfunction() accepts three arguments: the Julia function (mycompare), the return type (Cint), and a tuple of the argument types, in this case to sort an array of Cdouble (Float64) elements.The final call to qsort looks like this:julia> A = [1.3, -2.7, 4.4, 3.1]\n4-element Array{Float64,1}:\n  1.3\n -2.7\n  4.4\n  3.1\n\njulia> ccall(:qsort, Void, (Ptr{Cdouble}, Csize_t, Csize_t, Ptr{Void}),\n             A, length(A), sizeof(eltype(A)), mycompare_c)\n\njulia> A\n4-element Array{Float64,1}:\n -2.7\n  1.3\n  3.1\n  4.4As can be seen, A is changed to the sorted array [-2.7, 1.3, 3.1, 4.4]. Note that Julia knows how to convert an array into a Ptr{Cdouble}, how to compute the size of a type in bytes (identical to C's sizeof operator), and so on. For fun, try inserting a println(\"mycompare($a,$b)\") line into mycompare, which will allow you to see the comparisons that qsort is performing (and to verify that it is really calling the Julia function that you passed to it)."
},

{
    "location": "manual/calling-c-and-fortran-code.html#Mapping-C-Types-to-Julia-1",
    "page": "Calling C and Fortran Code",
    "title": "Mapping C Types to Julia",
    "category": "section",
    "text": "It is critical to exactly match the declared C type with its declaration in Julia. Inconsistencies can cause code that works correctly on one system to fail or produce indeterminate results on a different system.Note that no C header files are used anywhere in the process of calling C functions: you are responsible for making sure that your Julia types and call signatures accurately reflect those in the C header file. (The Clang package can be used to auto-generate Julia code from a C header file.)"
},

{
    "location": "manual/calling-c-and-fortran-code.html#Auto-conversion:-1",
    "page": "Calling C and Fortran Code",
    "title": "Auto-conversion:",
    "category": "section",
    "text": "Julia automatically inserts calls to the Base.cconvert() function to convert each argument to the specified type. For example, the following call:ccall((:foo, \"libfoo\"), Void, (Int32, Float64), x, y)will behave as if the following were written:ccall((:foo, \"libfoo\"), Void, (Int32, Float64),\n      Base.unsafe_convert(Int32, Base.cconvert(Int32, x)),\n      Base.unsafe_convert(Float64, Base.cconvert(Float64, y)))Base.cconvert() normally just calls convert(), but can be defined to return an arbitrary new object more appropriate for passing to C. For example, this is used to convert an Array of objects (e.g. strings) to an array of pointers.Base.unsafe_convert() handles conversion to Ptr types. It is considered unsafe because converting an object to a native pointer can hide the object from the garbage collector, causing it to be freed prematurely."
},

{
    "location": "manual/calling-c-and-fortran-code.html#Type-Correspondences:-1",
    "page": "Calling C and Fortran Code",
    "title": "Type Correspondences:",
    "category": "section",
    "text": "First, a review of some relevant Julia type terminology:Syntax / Keyword Example Description\nmutable struct String \"Leaf Type\" :: A group of related data that includes a type-tag, is managed by the Julia GC, and is defined by object-identity. The type parameters of a leaf type must be fully defined (no TypeVars are allowed) in order for the instance to be constructed.\nabstract type Any, AbstractArray{T, N}, Complex{T} \"Super Type\" :: A super-type (not a leaf-type) that cannot be instantiated, but can be used to describe a group of types.\nT{A} Vector{Int} \"Type Parameter\" :: A specialization of a type (typically used for dispatch or storage optimization).\n  \"TypeVar\" :: The T in the type parameter declaration is referred to as a TypeVar (short for type variable).\nprimitive type Int, Float64 \"Primitive Type\" :: A type with no fields, but a size. It is stored and defined by-value.\nstruct Pair{Int, Int} \"Struct\" :: A type with all fields defined to be constant. It is defined by-value, and may be stored with a type-tag.\n Complex128 (isbits) \"Is-Bits\"   :: A primitive type, or a struct type where all fields are other isbits types. It is defined by-value, and is stored without a type-tag.\nstruct ...; end nothing \"Singleton\" :: a Leaf Type or Struct with no fields.\n(...) or tuple(...) (1, 2, 3) \"Tuple\" :: an immutable data-structure similar to an anonymous struct type, or a constant array. Represented as either an array or a struct."
},

{
    "location": "manual/calling-c-and-fortran-code.html#Bits-Types:-1",
    "page": "Calling C and Fortran Code",
    "title": "Bits Types:",
    "category": "section",
    "text": "There are several special types to be aware of, as no other type can be defined to behave the same:Float32\nExactly corresponds to the float type in C (or REAL*4 in Fortran).\nFloat64\nExactly corresponds to the double type in C (or REAL*8 in Fortran).\nComplex64\nExactly corresponds to the complex float type in C (or COMPLEX*8 in Fortran).\nComplex128\nExactly corresponds to the complex double type in C (or COMPLEX*16 in Fortran).\nSigned\nExactly corresponds to the signed type annotation in C (or any INTEGER type in Fortran). Any Julia type that is not a subtype of Signed is assumed to be unsigned.Ref{T}\nBehaves like a Ptr{T} that can manage its memory via the Julia GC.Array{T,N}\nWhen an array is passed to C as a Ptr{T} argument, it is not reinterpret-cast: Julia requires that the element type of the array matches T, and the address of the first element is passed.\nTherefore, if an Array contains data in the wrong format, it will have to be explicitly converted using a call such as trunc(Int32, a).\nTo pass an array A as a pointer of a different type without converting the data beforehand (for example, to pass a Float64 array to a function that operates on uninterpreted bytes), you can declare the argument as Ptr{Void}.\nIf an array of eltype Ptr{T} is passed as a Ptr{Ptr{T}} argument, Base.cconvert() will attempt to first make a null-terminated copy of the array with each element replaced by its Base.cconvert() version. This allows, for example, passing an argv pointer array of type Vector{String} to an argument of type Ptr{Ptr{Cchar}}.On all systems we currently support, basic C/C++ value types may be translated to Julia types as follows. Every C type also has a corresponding Julia type with the same name, prefixed by C. This can help for writing portable code (and remembering that an int in C is not the same as an Int in Julia).System Independent:C name Fortran name Standard Julia Alias Julia Base Type\nunsigned char CHARACTER Cuchar UInt8\nbool (only in C++)  Cuchar UInt8\nshort INTEGER*2, LOGICAL*2 Cshort Int16\nunsigned short   Cushort UInt16\nint, BOOL (C, typical) INTEGER*4, LOGICAL*4 Cint Int32\nunsigned int   Cuint UInt32\nlong long INTEGER*8, LOGICAL*8 Clonglong Int64\nunsigned long long   Culonglong UInt64\nintmax_t   Cintmax_t Int64\nuintmax_t   Cuintmax_t UInt64\nfloat REAL*4i Cfloat Float32\ndouble REAL*8 Cdouble Float64\ncomplex float COMPLEX*8 Complex64 Complex{Float32}\ncomplex double COMPLEX*16 Complex128 Complex{Float64}\nptrdiff_t   Cptrdiff_t Int\nssize_t   Cssize_t Int\nsize_t   Csize_t UInt\nvoid     Void\nvoid and [[noreturn]] or _Noreturn     Union{}\nvoid*     Ptr{Void}\nT* (where T represents an appropriately defined type)     Ref{T}\nchar* (or char[], e.g. a string) CHARACTER*N   Cstring if NUL-terminated, or Ptr{UInt8} if not\nchar** (or *char[])     Ptr{Ptr{UInt8}}\njl_value_t* (any Julia Type)     Any\njl_value_t** (a reference to a Julia Type)     Ref{Any}\nva_arg     Not supported\n... (variadic function specification)     T... (where T is one of the above types, variadic functions of different argument types are not supported)The Cstring type is essentially a synonym for Ptr{UInt8}, except the conversion to Cstring throws an error if the Julia string contains any embedded NUL characters (which would cause the string to be silently truncated if the C routine treats NUL as the terminator).  If you are passing a char* to a C routine that does not assume NUL termination (e.g. because you pass an explicit string length), or if you know for certain that your Julia string does not contain NUL and want to skip the check, you can use Ptr{UInt8} as the argument type. Cstring can also be used as the ccall return type, but in that case it obviously does not introduce any extra checks and is only meant to improve readability of the call.System-dependent:C name Standard Julia Alias Julia Base Type\nchar Cchar Int8 (x86, x86_64), UInt8 (powerpc, arm)\nlong Clong Int (UNIX), Int32 (Windows)\nunsigned long Culong UInt (UNIX), UInt32 (Windows)\nwchar_t Cwchar_t Int32 (UNIX), UInt16 (Windows)note: Note\nWhen calling a Fortran function, all inputs must be passed by reference, so all type correspondences above should contain an additional Ptr{..} or Ref{..} wrapper around their type specification.warning: Warning\nFor string arguments (char*) the Julia type should be Cstring (if NUL- terminated data is expected) or either Ptr{Cchar} or Ptr{UInt8} otherwise (these two pointer types have the same effect), as described above, not String. Similarly, for array arguments (T[] or T*), the Julia type should again be Ptr{T}, not Vector{T}.warning: Warning\nJulia's Char type is 32 bits, which is not the same as the wide character type (wchar_t or wint_t) on all platforms.warning: Warning\nA return type of Union{} means the function will not return i.e. C++11 [[noreturn]] or C11 _Noreturn (e.g. jl_throw or longjmp). Do not use this for functions that return no value (void) but do return, use Void instead.note: Note\nFor wchar_t* arguments, the Julia type should be Cwstring (if the C routine expects a NUL-terminated string) or Ptr{Cwchar_t} otherwise. Note also that UTF-8 string data in Julia is internally NUL-terminated, so it can be passed to C functions expecting NUL-terminated data without making a copy (but using the Cwstring type will cause an error to be thrown if the string itself contains NUL characters).note: Note\nC functions that take an argument of the type char** can be called by using a Ptr{Ptr{UInt8}} type within Julia. For example, C functions of the form:int main(int argc, char **argv);can be called via the following Julia code:argv = [ \"a.out\", \"arg1\", \"arg2\" ]\nccall(:main, Int32, (Int32, Ptr{Ptr{UInt8}}), length(argv), argv)note: Note\nA C function declared to return Void will return the value nothing in Julia."
},

{
    "location": "manual/calling-c-and-fortran-code.html#Struct-Type-correspondences-1",
    "page": "Calling C and Fortran Code",
    "title": "Struct Type correspondences",
    "category": "section",
    "text": "Composite types, aka struct in C or TYPE in Fortran90 (or STRUCTURE / RECORD in some variants of F77), can be mirrored in Julia by creating a struct definition with the same field layout.When used recursively, isbits types are stored inline. All other types are stored as a pointer to the data. When mirroring a struct used by-value inside another struct in C, it is imperative that you do not attempt to manually copy the fields over, as this will not preserve the correct field alignment. Instead, declare an isbits struct type and use that instead. Unnamed structs are not possible in the translation to Julia.Packed structs and union declarations are not supported by Julia.You can get a near approximation of a union if you know, a priori, the field that will have the greatest size (potentially including padding). When translating your fields to Julia, declare the Julia field to be only of that type.Arrays of parameters can be expressed with NTuple:in C:\nstruct B {\n    int A[3];\n};\nb_a_2 = B.A[2];\n\nin Julia:\nstruct B\n    A::NTuple{3, CInt}\nend\nb_a_2 = B.A[3]  # note the difference in indexing (1-based in Julia, 0-based in C)Arrays of unknown size (C99-compliant variable length structs specified by [] or [0]) are not directly supported. Often the best way to deal with these is to deal with the byte offsets directly. For example, if a C library declared a proper string type and returned a pointer to it:struct String {\n    int strlen;\n    char data[];\n};In Julia, we can access the parts independently to make a copy of that string:str = from_c::Ptr{Void}\nlen = unsafe_load(Ptr{Cint}(str))\nunsafe_string(str + Core.sizeof(Cint), len)"
},

{
    "location": "manual/calling-c-and-fortran-code.html#Type-Parameters-1",
    "page": "Calling C and Fortran Code",
    "title": "Type Parameters",
    "category": "section",
    "text": "The type arguments to ccall are evaluated statically, when the method containing the ccall is defined. They therefore must take the form of a literal tuple, not a variable, and cannot reference local variables.This may sound like a strange restriction, but remember that since C is not a dynamic language like Julia, its functions can only accept argument types with a statically-known, fixed signature.However, while the type layout must be known statically to compute the ccall ABI, the static parameters of the function are considered to be part of this static environment. The static parameters of the function may be used as type parameters in the ccall signature, as long as they don't affect the layout of the type. For example, f(x::T) where {T} = ccall(:valid, Ptr{T}, (Ptr{T},), x) is valid, since Ptr is always a word-size primitive type. But, g(x::T) where {T} = ccall(:notvalid, T, (T,), x) is not valid, since the type layout of T is not known statically."
},

{
    "location": "manual/calling-c-and-fortran-code.html#SIMD-Values-1",
    "page": "Calling C and Fortran Code",
    "title": "SIMD Values",
    "category": "section",
    "text": "Note: This feature is currently implemented on 64-bit x86 and AArch64 platforms only.If a C/C++ routine has an argument or return value that is a native SIMD type, the corresponding Julia type is a homogeneous tuple of VecElement that naturally maps to the SIMD type.  Specifically:The tuple must be the same size as the SIMD type. For example, a tuple representing an __m128 on x86 must have a size of 16 bytes.\nThe element type of the tuple must be an instance of VecElement{T} where T is a primitive type that is 1, 2, 4 or 8 bytes.For instance, consider this C routine that uses AVX intrinsics:#include <immintrin.h>\n\n__m256 dist( __m256 a, __m256 b ) {\n    return _mm256_sqrt_ps(_mm256_add_ps(_mm256_mul_ps(a, a),\n                                        _mm256_mul_ps(b, b)));\n}The following Julia code calls dist using ccall:const m256 = NTuple{8, VecElement{Float32}}\n\na = m256(ntuple(i -> VecElement(sin(Float32(i))), 8))\nb = m256(ntuple(i -> VecElement(cos(Float32(i))), 8))\n\nfunction call_dist(a::m256, b::m256)\n    ccall((:dist, \"libdist\"), m256, (m256, m256), a, b)\nend\n\nprintln(call_dist(a,b))The host machine must have the requisite SIMD registers.  For example, the code above will not work on hosts without AVX support."
},

{
    "location": "manual/calling-c-and-fortran-code.html#Memory-Ownership-1",
    "page": "Calling C and Fortran Code",
    "title": "Memory Ownership",
    "category": "section",
    "text": "malloc/freeMemory allocation and deallocation of such objects must be handled by calls to the appropriate cleanup routines in the libraries being used, just like in any C program. Do not try to free an object received from a C library with Libc.free in Julia, as this may result in the free function being called via the wrong libc library and cause Julia to crash. The reverse (passing an object allocated in Julia to be freed by an external library) is equally invalid."
},

{
    "location": "manual/calling-c-and-fortran-code.html#When-to-use-T,-Ptr{T}-and-Ref{T}-1",
    "page": "Calling C and Fortran Code",
    "title": "When to use T, Ptr{T} and Ref{T}",
    "category": "section",
    "text": "In Julia code wrapping calls to external C routines, ordinary (non-pointer) data should be declared to be of type T inside the ccall, as they are passed by value.  For C code accepting pointers, Ref{T} should generally be used for the types of input arguments, allowing the use of pointers to memory managed by either Julia or C through the implicit call to Base.cconvert().  In contrast, pointers returned by the C function called should be declared to be of output type Ptr{T}, reflecting that the memory pointed to is managed by C only. Pointers contained in C structs should be represented as fields of type Ptr{T} within the corresponding Julia struct types designed to mimic the internal structure of corresponding C structs.In Julia code wrapping calls to external Fortran routines, all input arguments should be declared as of type Ref{T}, as Fortran passes all variables by reference. The return type should either be Void for Fortran subroutines, or a T for Fortran functions returning the type T."
},

{
    "location": "manual/calling-c-and-fortran-code.html#Mapping-C-Functions-to-Julia-1",
    "page": "Calling C and Fortran Code",
    "title": "Mapping C Functions to Julia",
    "category": "section",
    "text": ""
},

{
    "location": "manual/calling-c-and-fortran-code.html#ccall/cfunction-argument-translation-guide-1",
    "page": "Calling C and Fortran Code",
    "title": "ccall/cfunction argument translation guide",
    "category": "section",
    "text": "For translating a C argument list to Julia:T, where T is one of the primitive types: char, int, long, short, float, double, complex, enum or any of their typedef equivalents\nT, where T is an equivalent Julia Bits Type (per the table above)\nif T is an enum, the argument type should be equivalent to Cint or Cuint\nargument value will be copied (passed by value)\nstruct T (including typedef to a struct)\nT, where T is a Julia leaf type\nargument value will be copied (passed by value)\nvoid*\ndepends on how this parameter is used, first translate this to the intended pointer type, then determine the Julia equivalent using the remaining rules in this list\nthis argument may be declared as Ptr{Void}, if it really is just an unknown pointer\njl_value_t*\nAny\nargument value must be a valid Julia object\ncurrently unsupported by cfunction()\njl_value_t**\nRef{Any}\nargument value must be a valid Julia object (or C_NULL)\ncurrently unsupported by cfunction()\nT*\nRef{T}, where T is the Julia type corresponding to T\nargument value will be copied if it is an isbits type otherwise, the value must be a valid Julia object\n(T*)(...) (e.g. a pointer to a function)\nPtr{Void} (you may need to use cfunction() explicitly to create this pointer)\n... (e.g. a vararg)\nT..., where T is the Julia type\nva_arg\nnot supported"
},

{
    "location": "manual/calling-c-and-fortran-code.html#ccall/cfunction-return-type-translation-guide-1",
    "page": "Calling C and Fortran Code",
    "title": "ccall/cfunction return type translation guide",
    "category": "section",
    "text": "For translating a C return type to Julia:void\nVoid (this will return the singleton instance nothing::Void)\nT, where T is one of the primitive types: char, int, long, short, float, double, complex, enum or any of their typedef equivalents\nT, where T is an equivalent Julia Bits Type (per the table above)\nif T is an enum, the argument type should be equivalent to Cint or Cuint\nargument value will be copied (returned by-value)\nstruct T (including typedef to a struct)\nT, where T is a Julia Leaf Type\nargument value will be copied (returned by-value)\nvoid*\ndepends on how this parameter is used, first translate this to the intended pointer type, then determine the Julia equivalent using the remaining rules in this list\nthis argument may be declared as Ptr{Void}, if it really is just an unknown pointer\njl_value_t*\nAny\nargument value must be a valid Julia object\njl_value_t**\nRef{Any}\nargument value must be a valid Julia object (or C_NULL)\nT*\nIf the memory is already owned by Julia, or is an isbits type, and is known to be non-null:\nRef{T}, where T is the Julia type corresponding to T\na return type of Ref{Any} is invalid, it should either be Any (corresponding to jl_value_t*) or Ptr{Any} (corresponding to Ptr{Any})\nC MUST NOT modify the memory returned via Ref{T} if T is an isbits type\nIf the memory is owned by C:\nPtr{T}, where T is the Julia type corresponding to T\n(T*)(...) (e.g. a pointer to a function)\nPtr{Void} (you may need to use cfunction() explicitly to create this pointer)"
},

{
    "location": "manual/calling-c-and-fortran-code.html#Passing-Pointers-for-Modifying-Inputs-1",
    "page": "Calling C and Fortran Code",
    "title": "Passing Pointers for Modifying Inputs",
    "category": "section",
    "text": "Because C doesn't support multiple return values, often C functions will take pointers to data that the function will modify. To accomplish this within a ccall, you need to first encapsulate the value inside an Ref{T} of the appropriate type. When you pass this Ref object as an argument, Julia will automatically pass a C pointer to the encapsulated data:width = Ref{Cint}(0)\nrange = Ref{Cfloat}(0)\nccall(:foo, Void, (Ref{Cint}, Ref{Cfloat}), width, range)Upon return, the contents of width and range can be retrieved (if they were changed by foo) by width[] and range[]; that is, they act like zero-dimensional arrays."
},

{
    "location": "manual/calling-c-and-fortran-code.html#Special-Reference-Syntax-for-ccall-(deprecated):-1",
    "page": "Calling C and Fortran Code",
    "title": "Special Reference Syntax for ccall (deprecated):",
    "category": "section",
    "text": "The & syntax is deprecated, use the Ref{T} argument type instead.A prefix & is used on an argument to ccall to indicate that a pointer to a scalar argument should be passed instead of the scalar value itself (required for all Fortran function arguments, as noted above). The following example computes a dot product using a BLAS function.function compute_dot(DX::Vector{Float64}, DY::Vector{Float64})\n    @assert length(DX) == length(DY)\n    n = length(DX)\n    incx = incy = 1\n    product = ccall((:ddot_, \"libLAPACK\"),\n                    Float64,\n                    (Ptr{Int32}, Ptr{Float64}, Ptr{Int32}, Ptr{Float64}, Ptr{Int32}),\n                    &n, DX, &incx, DY, &incy)\n    return product\nendThe meaning of prefix & is not quite the same as in C. In particular, any changes to the referenced variables will not be visible in Julia unless the type is mutable (declared via type). However, even for immutable structs it will not cause any harm for called functions to attempt such modifications (that is, writing through the passed pointers). Moreover, & may be used with any expression, such as &0 or &f(x).When a scalar value is passed with & as an argument of type Ptr{T}, the value will first be converted to type T."
},

{
    "location": "manual/calling-c-and-fortran-code.html#Some-Examples-of-C-Wrappers-1",
    "page": "Calling C and Fortran Code",
    "title": "Some Examples of C Wrappers",
    "category": "section",
    "text": "Here is a simple example of a C wrapper that returns a Ptr type:mutable struct gsl_permutation\nend\n\n# The corresponding C signature is\n#     gsl_permutation * gsl_permutation_alloc (size_t n);\nfunction permutation_alloc(n::Integer)\n    output_ptr = ccall(\n        (:gsl_permutation_alloc, :libgsl), # name of C function and library\n        Ptr{gsl_permutation},              # output type\n        (Csize_t,),                        # tuple of input types\n        n                                  # name of Julia variable to pass in\n    )\n    if output_ptr == C_NULL # Could not allocate memory\n        throw(OutOfMemoryError())\n    end\n    return output_ptr\nendThe GNU Scientific Library (here assumed to be accessible through :libgsl) defines an opaque pointer, gsl_permutation *, as the return type of the C function gsl_permutation_alloc(). As user code never has to look inside the gsl_permutation struct, the corresponding Julia wrapper simply needs a new type declaration, gsl_permutation, that has no internal fields and whose sole purpose is to be placed in the type parameter of a Ptr type.  The return type of the ccall is declared as Ptr{gsl_permutation}, since the memory allocated and pointed to by output_ptr is controlled by C (and not Julia).The input n is passed by value, and so the function's input signature is simply declared as (Csize_t,) without any Ref or Ptr necessary. (If the wrapper was calling a Fortran function instead, the corresponding function input signature should instead be (Ref{Csize_t},), since Fortran variables are passed by reference.) Furthermore, n can be any type that is convertable to a Csize_t integer; the ccall implicitly calls Base.cconvert(Csize_t, n).Here is a second example wrapping the corresponding destructor:# The corresponding C signature is\n#     void gsl_permutation_free (gsl_permutation * p);\nfunction permutation_free(p::Ref{gsl_permutation})\n    ccall(\n        (:gsl_permutation_free, :libgsl), # name of C function and library\n        Void,                             # output type\n        (Ref{gsl_permutation},),          # tuple of input types\n        p                                 # name of Julia variable to pass in\n    )\nendHere, the input p is declared to be of type Ref{gsl_permutation}, meaning that the memory that p points to may be managed by Julia or by C. A pointer to memory allocated by C should be of type Ptr{gsl_permutation}, but it is convertable using Base.cconvert() and therefore can be used in the same (covariant) context of the input argument to a ccall. A pointer to memory allocated by Julia must be of type Ref{gsl_permutation}, to ensure that the memory address pointed to is valid and that Julia's garbage collector manages the chunk of memory pointed to correctly. Therefore, the Ref{gsl_permutation} declaration allows pointers managed by C or Julia to be used.If the C wrapper never expects the user to pass pointers to memory managed by Julia, then using p::Ptr{gsl_permutation} for the method signature of the wrapper and similarly in the ccall is also acceptable.Here is a third example passing Julia arrays:# The corresponding C signature is\n#    int gsl_sf_bessel_Jn_array (int nmin, int nmax, double x,\n#                                double result_array[])\nfunction sf_bessel_Jn_array(nmin::Integer, nmax::Integer, x::Real)\n    if nmax < nmin\n        throw(DomainError())\n    end\n    result_array = Vector{Cdouble}(nmax - nmin + 1)\n    errorcode = ccall(\n        (:gsl_sf_bessel_Jn_array, :libgsl), # name of C function and library\n        Cint,                               # output type\n        (Cint, Cint, Cdouble, Ref{Cdouble}),# tuple of input types\n        nmin, nmax, x, result_array         # names of Julia variables to pass in\n    )\n    if errorcode != 0\n        error(\"GSL error code $errorcode\")\n    end\n    return result_array\nendThe C function wrapped returns an integer error code; the results of the actual evaluation of the Bessel J function populate the Julia array result_array. This variable can only be used with corresponding input type declaration Ref{Cdouble}, since its memory is allocated and managed by Julia, not C. The implicit call to Base.cconvert(Ref{Cdouble}, result_array) unpacks the Julia pointer to a Julia array data structure into a form understandable by C.Note that for this code to work correctly, result_array must be declared to be of type Ref{Cdouble} and not Ptr{Cdouble}. The memory is managed by Julia and the Ref signature alerts Julia's garbage collector to keep managing the memory for result_array while the ccall executes. If Ptr{Cdouble} were used instead, the ccall may still work, but Julia's garbage collector would not be aware that the memory declared for result_array is being used by the external C function. As a result, the code may produce a memory leak if result_array never gets freed by the garbage collector, or if the garbage collector prematurely frees result_array, the C function may end up throwing an invalid memory access exception."
},

{
    "location": "manual/calling-c-and-fortran-code.html#Garbage-Collection-Safety-1",
    "page": "Calling C and Fortran Code",
    "title": "Garbage Collection Safety",
    "category": "section",
    "text": "When passing data to a ccall, it is best to avoid using the pointer() function. Instead define a convert method and pass the variables directly to the ccall. ccall automatically arranges that all of its arguments will be preserved from garbage collection until the call returns. If a C API will store a reference to memory allocated by Julia, after the ccall returns, you must arrange that the object remains visible to the garbage collector. The suggested way to handle this is to make a global variable of type Array{Ref,1} to hold these values, until the C library notifies you that it is finished with them.Whenever you have created a pointer to Julia data, you must ensure the original data exists until you are done with using the pointer. Many methods in Julia such as unsafe_load() and String() make copies of data instead of taking ownership of the buffer, so that it is safe to free (or alter) the original data without affecting Julia. A notable exception is unsafe_wrap() which, for performance reasons, shares (or can be told to take ownership of) the underlying buffer.The garbage collector does not guarantee any order of finalization. That is, if a contained a reference to b and both a and b are due for garbage collection, there is no guarantee that b would be finalized after a. If proper finalization of a depends on b being valid, it must be handled in other ways."
},

{
    "location": "manual/calling-c-and-fortran-code.html#Non-constant-Function-Specifications-1",
    "page": "Calling C and Fortran Code",
    "title": "Non-constant Function Specifications",
    "category": "section",
    "text": "A (name, library) function specification must be a constant expression. However, it is possible to use computed values as function names by staging through eval as follows:@eval ccall(($(string(\"a\", \"b\")), \"lib\"), ...This expression constructs a name using string, then substitutes this name into a new ccall expression, which is then evaluated. Keep in mind that eval only operates at the top level, so within this expression local variables will not be available (unless their values are substituted with $). For this reason, eval is typically only used to form top-level definitions, for example when wrapping libraries that contain many similar functions.If your usage is more dynamic, use indirect calls as described in the next section."
},

{
    "location": "manual/calling-c-and-fortran-code.html#Indirect-Calls-1",
    "page": "Calling C and Fortran Code",
    "title": "Indirect Calls",
    "category": "section",
    "text": "The first argument to ccall can also be an expression evaluated at run time. In this case, the expression must evaluate to a Ptr, which will be used as the address of the native function to call. This behavior occurs when the first ccall argument contains references to non-constants, such as local variables, function arguments, or non-constant globals.For example, you might look up the function via dlsym, then cache it in a global variable for that session. For example:macro dlsym(func, lib)\n    z, zlocal = gensym(string(func)), gensym()\n    eval(current_module(), :(global $z = C_NULL))\n    z = esc(z)\n    quote\n        let $zlocal::Ptr{Void} = $z::Ptr{Void}\n            if $zlocal == C_NULL\n                $zlocal = dlsym($(esc(lib))::Ptr{Void}, $(esc(func)))\n                global $z = $zlocal\n            end\n            $zlocal\n        end\n    end\nend\n\nmylibvar = Libdl.dlopen(\"mylib\")\nccall(@dlsym(\"myfunc\", mylibvar), Void, ())"
},

{
    "location": "manual/calling-c-and-fortran-code.html#Calling-Convention-1",
    "page": "Calling C and Fortran Code",
    "title": "Calling Convention",
    "category": "section",
    "text": "The second argument to ccall can optionally be a calling convention specifier (immediately preceding return type). Without any specifier, the platform-default C calling convention is used. Other supported conventions are: stdcall, cdecl, fastcall, and thiscall. For example (from base/libc.jl) we see the same gethostnameccall as above, but with the correct signature for Windows:hn = Vector{UInt8}(256)\nerr = ccall(:gethostname, stdcall, Int32, (Ptr{UInt8}, UInt32), hn, length(hn))For more information, please see the LLVM Language Reference.There is one additional special calling convention llvmcall, which allows inserting calls to LLVM intrinsics directly. This can be especially useful when targeting unusual platforms such as GPGPUs. For example, for CUDA, we need to be able to read the thread index:ccall(\"llvm.nvvm.read.ptx.sreg.tid.x\", llvmcall, Int32, ())As with any ccall, it is essential to get the argument signature exactly correct. Also, note that there is no compatibility layer that ensures the intrinsic makes sense and works on the current target, unlike the equivalent Julia functions exposed by Core.Intrinsics."
},

{
    "location": "manual/calling-c-and-fortran-code.html#Accessing-Global-Variables-1",
    "page": "Calling C and Fortran Code",
    "title": "Accessing Global Variables",
    "category": "section",
    "text": "Global variables exported by native libraries can be accessed by name using the cglobal() function. The arguments to cglobal() are a symbol specification identical to that used by ccall, and a type describing the value stored in the variable:julia> cglobal((:errno, :libc), Int32)\nPtr{Int32} @0x00007f418d0816b8The result is a pointer giving the address of the value. The value can be manipulated through this pointer using unsafe_load() and unsafe_store!()."
},

{
    "location": "manual/calling-c-and-fortran-code.html#Accessing-Data-through-a-Pointer-1",
    "page": "Calling C and Fortran Code",
    "title": "Accessing Data through a Pointer",
    "category": "section",
    "text": "The following methods are described as \"unsafe\" because a bad pointer or type declaration can cause Julia to terminate abruptly.Given a Ptr{T}, the contents of type T can generally be copied from the referenced memory into a Julia object using unsafe_load(ptr, [index]). The index argument is optional (default is 1), and follows the Julia-convention of 1-based indexing. This function is intentionally similar to the behavior of getindex() and setindex!() (e.g. [] access syntax).The return value will be a new object initialized to contain a copy of the contents of the referenced memory. The referenced memory can safely be freed or released.If T is Any, then the memory is assumed to contain a reference to a Julia object (a jl_value_t*), the result will be a reference to this object, and the object will not be copied. You must be careful in this case to ensure that the object was always visible to the garbage collector (pointers do not count, but the new reference does) to ensure the memory is not prematurely freed. Note that if the object was not originally allocated by Julia, the new object will never be finalized by Julia's garbage collector.  If the Ptr itself is actually a jl_value_t*, it can be converted back to a Julia object reference by unsafe_pointer_to_objref(ptr). (Julia values v can be converted to jl_value_t* pointers, as Ptr{Void}, by calling pointer_from_objref(v).)The reverse operation (writing data to a Ptr{T}), can be performed using unsafe_store!(ptr, value, [index]). Currently, this is only supported for primitive types or other pointer-free (isbits) immutable struct types.Any operation that throws an error is probably currently unimplemented and should be posted as a bug so that it can be resolved.If the pointer of interest is a plain-data array (primitive type or immutable struct), the function unsafe_wrap(Array, ptr,dims,[own]) may be more useful. The final parameter should be true if Julia should \"take ownership\" of the underlying buffer and call free(ptr) when the returned Array object is finalized.  If the own parameter is omitted or false, the caller must ensure the buffer remains in existence until all access is complete.Arithmetic on the Ptr type in Julia (e.g. using +) does not behave the same as C's pointer arithmetic. Adding an integer to a Ptr in Julia always moves the pointer by some number of bytes, not elements. This way, the address values obtained from pointer arithmetic do not depend on the element types of pointers."
},

{
    "location": "manual/calling-c-and-fortran-code.html#Thread-safety-1",
    "page": "Calling C and Fortran Code",
    "title": "Thread-safety",
    "category": "section",
    "text": "Some C libraries execute their callbacks from a different thread, and since Julia isn't thread-safe you'll need to take some extra precautions. In particular, you'll need to set up a two-layered system: the C callback should only schedule (via Julia's event loop) the execution of your \"real\" callback. To do this, create a AsyncCondition object and wait on it:cond = Base.AsyncCondition()\nwait(cond)The callback you pass to C should only execute a ccall to :uv_async_send, passing cond.handle as the argument, taking care to avoid any allocations or other interactions with the Julia runtime.Note that events may be coalesced, so multiple calls to uv_async_send may result in a single wakeup notification to the condition."
},

{
    "location": "manual/calling-c-and-fortran-code.html#More-About-Callbacks-1",
    "page": "Calling C and Fortran Code",
    "title": "More About Callbacks",
    "category": "section",
    "text": "For more details on how to pass callbacks to C libraries, see this blog post."
},

{
    "location": "manual/calling-c-and-fortran-code.html#C-1",
    "page": "Calling C and Fortran Code",
    "title": "C++",
    "category": "section",
    "text": "For direct C++ interfacing, see the Cxx package. For tools to create C++ bindings, see the CxxWrap package."
},

{
    "location": "manual/handling-operating-system-variation.html#",
    "page": "Handling Operating System Variation",
    "title": "Handling Operating System Variation",
    "category": "page",
    "text": ""
},

{
    "location": "manual/handling-operating-system-variation.html#Handling-Operating-System-Variation-1",
    "page": "Handling Operating System Variation",
    "title": "Handling Operating System Variation",
    "category": "section",
    "text": "When dealing with platform libraries, it is often necessary to provide special cases for various platforms. The variable Sys.KERNEL can be used to write these special cases. There are several functions intended to make this easier: is_unix, is_linux, is_apple, is_bsd, and is_windows. These may be used as follows:if is_windows()\n    some_complicated_thing(a)\nendNote that is_linux and is_apple are mutually exclusive subsets of is_unix. Additionally, there is a macro @static which makes it possible to use these functions to conditionally hide invalid code, as demonstrated in the following examples.Simple blocks:ccall( (@static is_windows() ? :_fopen : :fopen), ...)Complex blocks:@static if is_linux()\n    some_complicated_thing(a)\nelse\n    some_different_thing(a)\nendWhen chaining conditionals (including if/elseif/end), the @static must be repeated for each level (parentheses optional, but recommended for readability):@static is_windows() ? :a : (@static is_apple() ? :b : :c)"
},

{
    "location": "manual/environment-variables.html#",
    "page": "Environment Variables",
    "title": "Environment Variables",
    "category": "page",
    "text": ""
},

{
    "location": "manual/environment-variables.html#Environment-Variables-1",
    "page": "Environment Variables",
    "title": "Environment Variables",
    "category": "section",
    "text": "Julia may be configured with a number of environment variables, either in the usual way of the operating system, or in a portable way from within Julia. Suppose you want to set the environment variable JULIA_EDITOR to vim, then either type ENV[\"JULIA_EDITOR\"] = \"vim\" for instance in the REPL to make this change on a case by case basis, or add the same to the user configuration file .juliarc.jl in the user's home directory to have a permanent effect. The current value of the same environment variable is determined by evaluating ENV[\"JULIA_EDITOR\"].The environment variables that Julia uses generally start with JULIA. If Base.versioninfo is called with verbose equal to true, then the output will list defined environment variables relevant for Julia, including those for which JULIA appears in the name."
},

{
    "location": "manual/environment-variables.html#File-locations-1",
    "page": "Environment Variables",
    "title": "File locations",
    "category": "section",
    "text": ""
},

{
    "location": "manual/environment-variables.html#JULIA_HOME-1",
    "page": "Environment Variables",
    "title": "JULIA_HOME",
    "category": "section",
    "text": "The absolute path of the directory containing the Julia executable, which sets the global variable Base.JULIA_HOME. If $JULIA_HOME is not set, then Julia determines the value Base.JULIA_HOME at run-time.The executable itself is one of$JULIA_HOME/julia\n$JULIA_HOME/julia-debugby default.The global variable Base.DATAROOTDIR determines a relative path from Base.JULIA_HOME to the data directory associated with Julia. Then the path$JULIA_HOME/$DATAROOTDIR/julia/basedetermines the directory in which Julia initially searches for source files (via Base.find_source_file()).Likewise, the global variable Base.SYSCONFDIR determines a relative path to the configuration file directory. Then Julia searches for a juliarc.jl file at$JULIA_HOME/$SYSCONFDIR/julia/juliarc.jl\n$JULIA_HOME/../etc/julia/juliarc.jlby default (via Base.load_juliarc()).For example, a Linux installation with a Julia executable located at /bin/julia, a DATAROOTDIR of ../share, and a SYSCONFDIR of ../etc will have JULIA_HOME set to /bin, a source-file search path of/share/julia/baseand a global configuration search path of/etc/julia/juliarc.jl"
},

{
    "location": "manual/environment-variables.html#JULIA_LOAD_PATH-1",
    "page": "Environment Variables",
    "title": "JULIA_LOAD_PATH",
    "category": "section",
    "text": "A separated list of absolute paths that are to be appended to the variable LOAD_PATH. (In Unix-like systems, the path separator is :; in Windows systems, the path separator is ;.) The LOAD_PATH variable is where Base.require and Base.load_in_path() look for code; it defaults to the absolute paths$JULIA_HOME/../local/share/julia/site/v$(VERSION.major).$(VERSION.minor)\n$JULIA_HOME/../share/julia/site/v$(VERSION.major).$(VERSION.minor)so that, e.g., version 0.6 of Julia on a Linux system with a Julia executable at /bin/julia will have a default LOAD_PATH of/local/share/julia/site/v0.6\n/share/julia/site/v0.6"
},

{
    "location": "manual/environment-variables.html#JULIA_PKGDIR-1",
    "page": "Environment Variables",
    "title": "JULIA_PKGDIR",
    "category": "section",
    "text": "The path of the parent directory Pkg.Dir._pkgroot() for the version-specific Julia package repositories. If the path is relative, then it is taken with respect to the working directory. If $JULIA_PKGDIR is not set, then Pkg.Dir._pkgroot() defaults to$HOME/.juliaThen the repository location Pkg.dir for a given Julia version is$JULIA_PKGDIR/v$(VERSION.major).$(VERSION.minor)For example, for a Linux user whose home directory is /home/alice, the directory containing the package repositories would by default be/home/alice/.juliaand the package repository for version 0.6 of Julia would be/home/alice/.julia/v0.6"
},

{
    "location": "manual/environment-variables.html#JULIA_HISTORY-1",
    "page": "Environment Variables",
    "title": "JULIA_HISTORY",
    "category": "section",
    "text": "The absolute path Base.REPL.find_hist_file() of the REPL's history file. If $JULIA_HISTORY is not set, then Base.REPL.find_hist_file() defaults to$HOME/.julia_history"
},

{
    "location": "manual/environment-variables.html#JULIA_PKGRESOLVE_ACCURACY-1",
    "page": "Environment Variables",
    "title": "JULIA_PKGRESOLVE_ACCURACY",
    "category": "section",
    "text": "A positive Int that determines how much time the max-sum subroutine MaxSum.maxsum() of the package dependency resolver Base.Pkg.resolve will devote to attempting satisfying constraints before giving up: this value is by default 1, and larger values correspond to larger amounts of time.Suppose the value of $JULIA_PKGRESOLVE_ACCURACY is n. Thenthe number of pre-decimation iterations is 20*n,\nthe number of iterations between decimation steps is 10*n, and\nat decimation steps, at most one in every 20*n packages is decimated."
},

{
    "location": "manual/environment-variables.html#External-applications-1",
    "page": "Environment Variables",
    "title": "External applications",
    "category": "section",
    "text": ""
},

{
    "location": "manual/environment-variables.html#JULIA_SHELL-1",
    "page": "Environment Variables",
    "title": "JULIA_SHELL",
    "category": "section",
    "text": "The absolute path of the shell with which Julia should execute external commands (via Base.repl_cmd()). Defaults to the environment variable $SHELL, and falls back to /bin/sh if $SHELL is unset.note: Note\nOn Windows, this environment variable is ignored, and external commands are executed directly."
},

{
    "location": "manual/environment-variables.html#JULIA_EDITOR-1",
    "page": "Environment Variables",
    "title": "JULIA_EDITOR",
    "category": "section",
    "text": "The editor returned by Base.editor() and used in, e.g., Base.edit, referring to the command of the preferred editor, for instance vim.$JULIA_EDITOR takes precedence over $VISUAL, which in turn takes precedence over $EDITOR. If none of these environment variables is set, then the editor is taken to be open on Windows and OS X, or /etc/alternatives/editor if it exists, or emacs otherwise.note: Note\n$JULIA_EDITOR is not used in the determination of the editor for Base.Pkg.edit: this function checks $VISUAL and $EDITOR alone."
},

{
    "location": "manual/environment-variables.html#Parallelization-1",
    "page": "Environment Variables",
    "title": "Parallelization",
    "category": "section",
    "text": ""
},

{
    "location": "manual/environment-variables.html#JULIA_CPU_CORES-1",
    "page": "Environment Variables",
    "title": "JULIA_CPU_CORES",
    "category": "section",
    "text": "Overrides the global variable Base.Sys.CPU_CORES, the number of logical CPU cores available."
},

{
    "location": "manual/environment-variables.html#JULIA_WORKER_TIMEOUT-1",
    "page": "Environment Variables",
    "title": "JULIA_WORKER_TIMEOUT",
    "category": "section",
    "text": "A Float64 that sets the value of Base.worker_timeout() (default: 60.0). This function gives the number of seconds a worker process will wait for a master process to establish a connection before dying."
},

{
    "location": "manual/environment-variables.html#JULIA_NUM_THREADS-1",
    "page": "Environment Variables",
    "title": "JULIA_NUM_THREADS",
    "category": "section",
    "text": "An unsigned 64-bit integer (uint64_t) that sets the maximum number of threads available to Julia. If $JULIA_NUM_THREADS exceeds the number of available physical CPU cores, then the number of threads is set to the number of cores. If $JULIA_NUM_THREADS is not positive or is not set, or if the number of CPU cores cannot be determined through system calls, then the number of threads is set to 1."
},

{
    "location": "manual/environment-variables.html#JULIA_THREAD_SLEEP_THRESHOLD-1",
    "page": "Environment Variables",
    "title": "JULIA_THREAD_SLEEP_THRESHOLD",
    "category": "section",
    "text": "If set to a string that starts with the case-insensitive substring \"infinite\", then spinning threads never sleep. Otherwise, $JULIA_THREAD_SLEEP_THRESHOLD is interpreted as an unsigned 64-bit integer (uint64_t) and gives, in nanoseconds, the amount of time after which spinning threads should sleep."
},

{
    "location": "manual/environment-variables.html#JULIA_EXCLUSIVE-1",
    "page": "Environment Variables",
    "title": "JULIA_EXCLUSIVE",
    "category": "section",
    "text": "If set to anything besides 0, then Julia's thread policy is consistent with running on a dedicated machine: the master thread is on proc 0, and threads are affinitized. Otherwise, Julia lets the operating system handle thread policy."
},

{
    "location": "manual/environment-variables.html#REPL-formatting-1",
    "page": "Environment Variables",
    "title": "REPL formatting",
    "category": "section",
    "text": "Environment variables that determine how REPL output should be formatted at the terminal. Generally, these variables should be set to ANSI terminal escape sequences. Julia provides a high-level interface with much of the same functionality: see the section on Interacting With Julia."
},

{
    "location": "manual/environment-variables.html#JULIA_ERROR_COLOR-1",
    "page": "Environment Variables",
    "title": "JULIA_ERROR_COLOR",
    "category": "section",
    "text": "The formatting Base.error_color() (default: light red, \"\\033[91m\") that errors should have at the terminal."
},

{
    "location": "manual/environment-variables.html#JULIA_WARN_COLOR-1",
    "page": "Environment Variables",
    "title": "JULIA_WARN_COLOR",
    "category": "section",
    "text": "The formatting Base.warn_color() (default: yellow, \"\\033[93m\") that warnings should have at the terminal."
},

{
    "location": "manual/environment-variables.html#JULIA_INFO_COLOR-1",
    "page": "Environment Variables",
    "title": "JULIA_INFO_COLOR",
    "category": "section",
    "text": "The formatting Base.info_color() (default: cyan, \"\\033[36m\") that info should have at the terminal."
},

{
    "location": "manual/environment-variables.html#JULIA_INPUT_COLOR-1",
    "page": "Environment Variables",
    "title": "JULIA_INPUT_COLOR",
    "category": "section",
    "text": "The formatting Base.input_color() (default: normal, \"\\033[0m\") that input should have at the terminal."
},

{
    "location": "manual/environment-variables.html#JULIA_ANSWER_COLOR-1",
    "page": "Environment Variables",
    "title": "JULIA_ANSWER_COLOR",
    "category": "section",
    "text": "The formatting Base.answer_color() (default: normal, \"\\033[0m\") that output should have at the terminal."
},

{
    "location": "manual/environment-variables.html#JULIA_STACKFRAME_LINEINFO_COLOR-1",
    "page": "Environment Variables",
    "title": "JULIA_STACKFRAME_LINEINFO_COLOR",
    "category": "section",
    "text": "The formatting Base.stackframe_lineinfo_color() (default: bold, \"\\033[1m\") that line info should have during a stack trace at the terminal."
},

{
    "location": "manual/environment-variables.html#JULIA_STACKFRAME_FUNCTION_COLOR-1",
    "page": "Environment Variables",
    "title": "JULIA_STACKFRAME_FUNCTION_COLOR",
    "category": "section",
    "text": "The formatting Base.stackframe_function_color() (default: bold, \"\\033[1m\") that function calls should have during a stack trace at the terminal."
},

{
    "location": "manual/environment-variables.html#Debugging-and-profiling-1",
    "page": "Environment Variables",
    "title": "Debugging and profiling",
    "category": "section",
    "text": ""
},

{
    "location": "manual/environment-variables.html#JULIA_GC_ALLOC_POOL,-JULIA_GC_ALLOC_OTHER,-JULIA_GC_ALLOC_PRINT-1",
    "page": "Environment Variables",
    "title": "JULIA_GC_ALLOC_POOL, JULIA_GC_ALLOC_OTHER, JULIA_GC_ALLOC_PRINT",
    "category": "section",
    "text": "If set, these environment variables take strings that optionally start with the character 'r', followed by a string interpolation of a colon-separated list of three signed 64-bit integers (int64_t). This triple of integers a:b:c represents the arithmetic sequence a, a + b, a + 2*b, ... c.If it's the nth time that jl_gc_pool_alloc() has been called, and n   belongs to the arithmetic sequence represented by $JULIA_GC_ALLOC_POOL,   then garbage collection is forced.\nIf it's the nth time that maybe_collect() has been called, and n belongs   to the arithmetic sequence represented by $JULIA_GC_ALLOC_OTHER, then garbage   collection is forced.\nIf it's the nth time that jl_gc_collect() has been called, and n belongs   to the arithmetic sequence represented by $JULIA_GC_ALLOC_PRINT, then counts   for the number of calls to jl_gc_pool_alloc() and maybe_collect() are   printed.If the value of the environment variable begins with the character 'r', then the interval between garbage collection events is randomized.note: Note\nThese environment variables only have an effect if Julia was compiled with garbage-collection debugging (that is, if WITH_GC_DEBUG_ENV is set to 1 in the build configuration)."
},

{
    "location": "manual/environment-variables.html#JULIA_GC_NO_GENERATIONAL-1",
    "page": "Environment Variables",
    "title": "JULIA_GC_NO_GENERATIONAL",
    "category": "section",
    "text": "If set to anything besides 0, then the Julia garbage collector never performs \"quick sweeps\" of memory.note: Note\nThis environment variable only has an effect if Julia was compiled with garbage-collection debugging (that is, if WITH_GC_DEBUG_ENV is set to 1 in the build configuration)."
},

{
    "location": "manual/environment-variables.html#JULIA_GC_WAIT_FOR_DEBUGGER-1",
    "page": "Environment Variables",
    "title": "JULIA_GC_WAIT_FOR_DEBUGGER",
    "category": "section",
    "text": "If set to anything besides 0, then the Julia garbage collector will wait for a debugger to attach instead of aborting whenever there's a critical error.note: Note\nThis environment variable only has an effect if Julia was compiled with garbage-collection debugging (that is, if WITH_GC_DEBUG_ENV is set to 1 in the build configuration)."
},

{
    "location": "manual/environment-variables.html#ENABLE_JITPROFILING-1",
    "page": "Environment Variables",
    "title": "ENABLE_JITPROFILING",
    "category": "section",
    "text": "If set to anything besides 0, then the compiler will create and register an event listener for just-in-time (JIT) profiling.note: Note\nThis environment variable only has an effect if Julia was compiled with JIT profiling support, using eitherIntel's VTune™ Amplifier   (USE_INTEL_JITEVENTS set to 1 in the build configuration), or\nOProfile (USE_OPROFILE_JITEVENTS set to 1   in the build configuration)."
},

{
    "location": "manual/environment-variables.html#JULIA_LLVM_ARGS-1",
    "page": "Environment Variables",
    "title": "JULIA_LLVM_ARGS",
    "category": "section",
    "text": "Arguments to be passed to the LLVM backend.note: Note\nThis environment variable has an effect only if Julia was compiled with JL_DEBUG_BUILD set — in particular, the julia-debug executable is always compiled with this build variable."
},

{
    "location": "manual/environment-variables.html#JULIA_DEBUG_LOADING-1",
    "page": "Environment Variables",
    "title": "JULIA_DEBUG_LOADING",
    "category": "section",
    "text": "If set, then Julia prints detailed information about the cache in the loading process of Base.require."
},

{
    "location": "manual/embedding.html#",
    "page": "Embedding Julia",
    "title": "Embedding Julia",
    "category": "page",
    "text": ""
},

{
    "location": "manual/embedding.html#Embedding-Julia-1",
    "page": "Embedding Julia",
    "title": "Embedding Julia",
    "category": "section",
    "text": "As we have seen in Calling C and Fortran Code, Julia has a simple and efficient way to call functions written in C. But there are situations where the opposite is needed: calling Julia function from C code. This can be used to integrate Julia code into a larger C/C++ project, without the need to rewrite everything in C/C++. Julia has a C API to make this possible. As almost all programming languages have some way to call C functions, the Julia C API can also be used to build further language bridges (e.g. calling Julia from Python or C#)."
},

{
    "location": "manual/embedding.html#High-Level-Embedding-1",
    "page": "Embedding Julia",
    "title": "High-Level Embedding",
    "category": "section",
    "text": "We start with a simple C program that initializes Julia and calls some Julia code:#include <julia.h>\n\nint main(int argc, char *argv[])\n{\n    /* required: setup the Julia context */\n    jl_init();\n\n    /* run Julia commands */\n    jl_eval_string(\"print(sqrt(2.0))\");\n\n    /* strongly recommended: notify Julia that the\n         program is about to terminate. this allows\n         Julia time to cleanup pending write requests\n         and run all finalizers\n    */\n    jl_atexit_hook(0);\n    return 0;\n}In order to build this program you have to put the path to the Julia header into the include path and link against libjulia. For instance, when Julia is installed to $JULIA_DIR, one can compile the above test program test.c with gcc using:gcc -o test -fPIC -I$JULIA_DIR/include/julia -L$JULIA_DIR/lib test.c -ljulia $JULIA_DIR/lib/julia/libstdc++.so.6Then if the environment variable JULIA_HOME is set to $JULIA_DIR/bin, the output test program can be executed.Alternatively, look at the embedding.c program in the Julia source tree in the examples/ folder. The file ui/repl.c program is another simple example of how to set jl_options options while linking against libjulia.The first thing that has to be done before calling any other Julia C function is to initialize Julia. This is done by calling jl_init, which tries to automatically determine Julia's install location. If you need to specify a custom location, or specify which system image to load, use jl_init_with_image instead.The second statement in the test program evaluates a Julia statement using a call to jl_eval_string.Before the program terminates, it is strongly recommended to call jl_atexit_hook.  The above example program calls this before returning from main.note: Note\nCurrently, dynamically linking with the libjulia shared library requires passing the RTLD_GLOBAL option. In Python, this looks like:>>> julia=CDLL('./libjulia.dylib',RTLD_GLOBAL)\n>>> julia.jl_init.argtypes = []\n>>> julia.jl_init()\n250593296note: Note\nIf the julia program needs to access symbols from the main executable, it may be necessary to add -Wl,--export-dynamic linker flag at compile time on Linux in addition to the ones generated by julia-config.jl described below. This is not necessary when compiling a shared library."
},

{
    "location": "manual/embedding.html#Using-julia-config-to-automatically-determine-build-parameters-1",
    "page": "Embedding Julia",
    "title": "Using julia-config to automatically determine build parameters",
    "category": "section",
    "text": "The script julia-config.jl was created to aid in determining what build parameters are required by a program that uses embedded Julia.  This script uses the build parameters and system configuration of the particular Julia distribution it is invoked by to export the necessary compiler flags for an embedding program to interact with that distribution.  This script is located in the Julia shared data directory."
},

{
    "location": "manual/embedding.html#Example-1",
    "page": "Embedding Julia",
    "title": "Example",
    "category": "section",
    "text": "#include <julia.h>\n\nint main(int argc, char *argv[])\n{\n    jl_init();\n    (void)jl_eval_string(\"println(sqrt(2.0))\");\n    jl_atexit_hook(0);\n    return 0;\n}"
},

{
    "location": "manual/embedding.html#On-the-command-line-1",
    "page": "Embedding Julia",
    "title": "On the command line",
    "category": "section",
    "text": "A simple use of this script is from the command line.  Assuming that julia-config.jl is located in /usr/local/julia/share/julia, it can be invoked on the command line directly and takes any combination of 3 flags:/usr/local/julia/share/julia/julia-config.jl\nUsage: julia-config [--cflags|--ldflags|--ldlibs]If the above example source is saved in the file embed_example.c, then the following command will compile it into a running program on Linux and Windows (MSYS2 environment), or if on OS/X, then substitute clang for gcc.:/usr/local/julia/share/julia/julia-config.jl --cflags --ldflags --ldlibs | xargs gcc embed_example.c"
},

{
    "location": "manual/embedding.html#Use-in-Makefiles-1",
    "page": "Embedding Julia",
    "title": "Use in Makefiles",
    "category": "section",
    "text": "But in general, embedding projects will be more complicated than the above, and so the following allows general makefile support as well – assuming GNU make because of the use of the shell macro expansions.  Additionally, though many times julia-config.jl may be found in the directory /usr/local, this is not necessarily the case, but Julia can be used to locate julia-config.jl too, and the makefile can be used to take advantage of that.  The above example is extended to use a Makefile:JL_SHARE = $(shell julia -e 'print(joinpath(JULIA_HOME,Base.DATAROOTDIR,\"julia\"))')\nCFLAGS   += $(shell $(JL_SHARE)/julia-config.jl --cflags)\nCXXFLAGS += $(shell $(JL_SHARE)/julia-config.jl --cflags)\nLDFLAGS  += $(shell $(JL_SHARE)/julia-config.jl --ldflags)\nLDLIBS   += $(shell $(JL_SHARE)/julia-config.jl --ldlibs)\n\nall: embed_exampleNow the build command is simply make."
},

{
    "location": "manual/embedding.html#Converting-Types-1",
    "page": "Embedding Julia",
    "title": "Converting Types",
    "category": "section",
    "text": "Real applications will not just need to execute expressions, but also return their values to the host program. jl_eval_string returns a jl_value_t*, which is a pointer to a heap-allocated Julia object. Storing simple data types like Float64 in this way is called boxing, and extracting the stored primitive data is called unboxing. Our improved sample program that calculates the square root of 2 in Julia and reads back the result in C looks as follows:jl_value_t *ret = jl_eval_string(\"sqrt(2.0)\");\n\nif (jl_typeis(ret, jl_float64_type)) {\n    double ret_unboxed = jl_unbox_float64(ret);\n    printf(\"sqrt(2.0) in C: %e \\n\", ret_unboxed);\n}\nelse {\n    printf(\"ERROR: unexpected return type from sqrt(::Float64)\\n\");\n}In order to check whether ret is of a specific Julia type, we can use the jl_isa, jl_typeis, or jl_is_... functions. By typing typeof(sqrt(2.0)) into the Julia shell we can see that the return type is Float64 (double in C). To convert the boxed Julia value into a C double the jl_unbox_float64 function is used in the above code snippet.Corresponding jl_box_... functions are used to convert the other way:jl_value_t *a = jl_box_float64(3.0);\njl_value_t *b = jl_box_float32(3.0f);\njl_value_t *c = jl_box_int32(3);As we will see next, boxing is required to call Julia functions with specific arguments."
},

{
    "location": "manual/embedding.html#Calling-Julia-Functions-1",
    "page": "Embedding Julia",
    "title": "Calling Julia Functions",
    "category": "section",
    "text": "While jl_eval_string allows C to obtain the result of a Julia expression, it does not allow passing arguments computed in C to Julia. For this you will need to invoke Julia functions directly, using jl_call:jl_function_t *func = jl_get_function(jl_base_module, \"sqrt\");\njl_value_t *argument = jl_box_float64(2.0);\njl_value_t *ret = jl_call1(func, argument);In the first step, a handle to the Julia function sqrt is retrieved by calling jl_get_function. The first argument passed to jl_get_function is a pointer to the Base module in which sqrt is defined. Then, the double value is boxed using jl_box_float64. Finally, in the last step, the function is called using jl_call1. jl_call0, jl_call2, and jl_call3 functions also exist, to conveniently handle different numbers of arguments. To pass more arguments, use jl_call:jl_value_t *jl_call(jl_function_t *f, jl_value_t **args, int32_t nargs)Its second argument args is an array of jl_value_t* arguments and nargs is the number of arguments."
},

{
    "location": "manual/embedding.html#Memory-Management-1",
    "page": "Embedding Julia",
    "title": "Memory Management",
    "category": "section",
    "text": "As we have seen, Julia objects are represented in C as pointers. This raises the question of who is responsible for freeing these objects.Typically, Julia objects are freed by a garbage collector (GC), but the GC does not automatically know that we are holding a reference to a Julia value from C. This means the GC can free objects out from under you, rendering pointers invalid.The GC can only run when Julia objects are allocated. Calls like jl_box_float64 perform allocation, and allocation might also happen at any point in running Julia code. However, it is generally safe to use pointers in between jl_... calls. But in order to make sure that values can survive jl_... calls, we have to tell Julia that we hold a reference to a Julia value. This can be done using the JL_GC_PUSH macros:jl_value_t *ret = jl_eval_string(\"sqrt(2.0)\");\nJL_GC_PUSH1(&ret);\n// Do something with ret\nJL_GC_POP();The JL_GC_POP call releases the references established by the previous JL_GC_PUSH. Note that JL_GC_PUSH  is working on the stack, so it must be exactly paired with a JL_GC_POP before the stack frame is destroyed.Several Julia values can be pushed at once using the JL_GC_PUSH2 , JL_GC_PUSH3 , and JL_GC_PUSH4 macros. To push an array of Julia values one can use the  JL_GC_PUSHARGS macro, which can be used as follows:jl_value_t **args;\nJL_GC_PUSHARGS(args, 2); // args can now hold 2 `jl_value_t*` objects\nargs[0] = some_value;\nargs[1] = some_other_value;\n// Do something with args (e.g. call jl_... functions)\nJL_GC_POP();The garbage collector also operates under the assumption that it is aware of every old-generation object pointing to a young-generation one. Any time a pointer is updated breaking that assumption, it must be signaled to the collector with the jl_gc_wb (write barrier) function like so:jl_value_t *parent = some_old_value, *child = some_young_value;\n((some_specific_type*)parent)->field = child;\njl_gc_wb(parent, child);It is in general impossible to predict which values will be old at runtime, so the write barrier must be inserted after all explicit stores. One notable exception is if the parent object was just allocated and garbage collection was not run since then. Remember that most jl_... functions can sometimes invoke garbage collection.The write barrier is also necessary for arrays of pointers when updating their data directly. For example:jl_array_t *some_array = ...; // e.g. a Vector{Any}\nvoid **data = (void**)jl_array_data(some_array);\njl_value_t *some_value = ...;\ndata[0] = some_value;\njl_gc_wb(some_array, some_value);"
},

{
    "location": "manual/embedding.html#Manipulating-the-Garbage-Collector-1",
    "page": "Embedding Julia",
    "title": "Manipulating the Garbage Collector",
    "category": "section",
    "text": "There are some functions to control the GC. In normal use cases, these should not be necessary.Function Description\njl_gc_collect() Force a GC run\njl_gc_enable(0) Disable the GC, return previous state as int\njl_gc_enable(1) Enable the GC,  return previous state as int\njl_gc_is_enabled() Return current state as int"
},

{
    "location": "manual/embedding.html#Working-with-Arrays-1",
    "page": "Embedding Julia",
    "title": "Working with Arrays",
    "category": "section",
    "text": "Julia and C can share array data without copying. The next example will show how this works.Julia arrays are represented in C by the datatype jl_array_t*. Basically, jl_array_t is a struct that contains:Information about the datatype\nA pointer to the data block\nInformation about the sizes of the arrayTo keep things simple, we start with a 1D array. Creating an array containing Float64 elements of length 10 is done by:jl_value_t* array_type = jl_apply_array_type(jl_float64_type, 1);\njl_array_t* x          = jl_alloc_array_1d(array_type, 10);Alternatively, if you have already allocated the array you can generate a thin wrapper around its data:double *existingArray = (double*)malloc(sizeof(double)*10);\njl_array_t *x = jl_ptr_to_array_1d(array_type, existingArray, 10, 0);The last argument is a boolean indicating whether Julia should take ownership of the data. If this argument is non-zero, the GC will call free on the data pointer when the array is no longer referenced.In order to access the data of x, we can use jl_array_data:double *xData = (double*)jl_array_data(x);Now we can fill the array:for(size_t i=0; i<jl_array_len(x); i++)\n    xData[i] = i;Now let us call a Julia function that performs an in-place operation on x:jl_function_t *func = jl_get_function(jl_base_module, \"reverse!\");\njl_call1(func, (jl_value_t*)x);By printing the array, one can verify that the elements of x are now reversed."
},

{
    "location": "manual/embedding.html#Accessing-Returned-Arrays-1",
    "page": "Embedding Julia",
    "title": "Accessing Returned Arrays",
    "category": "section",
    "text": "If a Julia function returns an array, the return value of jl_eval_string and jl_call can be cast to a jl_array_t*:jl_function_t *func  = jl_get_function(jl_base_module, \"reverse\");\njl_array_t *y = (jl_array_t*)jl_call1(func, (jl_value_t*)x);Now the content of y can be accessed as before using jl_array_data. As always, be sure to keep a reference to the array while it is in use."
},

{
    "location": "manual/embedding.html#Multidimensional-Arrays-1",
    "page": "Embedding Julia",
    "title": "Multidimensional Arrays",
    "category": "section",
    "text": "Julia's multidimensional arrays are stored in memory in column-major order. Here is some code that creates a 2D array and accesses its properties:// Create 2D array of float64 type\njl_value_t *array_type = jl_apply_array_type(jl_float64_type, 2);\njl_array_t *x  = jl_alloc_array_2d(array_type, 10, 5);\n\n// Get array pointer\ndouble *p = (double*)jl_array_data(x);\n// Get number of dimensions\nint ndims = jl_array_ndims(x);\n// Get the size of the i-th dim\nsize_t size0 = jl_array_dim(x,0);\nsize_t size1 = jl_array_dim(x,1);\n\n// Fill array with data\nfor(size_t i=0; i<size1; i++)\n    for(size_t j=0; j<size0; j++)\n        p[j + size0*i] = i + j;Notice that while Julia arrays use 1-based indexing, the C API uses 0-based indexing (for example in calling jl_array_dim) in order to read as idiomatic C code."
},

{
    "location": "manual/embedding.html#Exceptions-1",
    "page": "Embedding Julia",
    "title": "Exceptions",
    "category": "section",
    "text": "Julia code can throw exceptions. For example, consider:jl_eval_string(\"this_function_does_not_exist()\");This call will appear to do nothing. However, it is possible to check whether an exception was thrown:if (jl_exception_occurred())\n    printf(\"%s \\n\", jl_typeof_str(jl_exception_occurred()));If you are using the Julia C API from a language that supports exceptions (e.g. Python, C#, C++), it makes sense to wrap each call into libjulia with a function that checks whether an exception was thrown, and then rethrows the exception in the host language."
},

{
    "location": "manual/embedding.html#Throwing-Julia-Exceptions-1",
    "page": "Embedding Julia",
    "title": "Throwing Julia Exceptions",
    "category": "section",
    "text": "When writing Julia callable functions, it might be necessary to validate arguments and throw exceptions to indicate errors. A typical type check looks like:if (!jl_typeis(val, jl_float64_type)) {\n    jl_type_error(function_name, (jl_value_t*)jl_float64_type, val);\n}General exceptions can be raised using the functions:void jl_error(const char *str);\nvoid jl_errorf(const char *fmt, ...);jl_error takes a C string, and jl_errorf is called like printf:jl_errorf(\"argument x = %d is too large\", x);where in this example x is assumed to be an integer."
},

{
    "location": "manual/packages.html#",
    "page": "Packages",
    "title": "Packages",
    "category": "page",
    "text": ""
},

{
    "location": "manual/packages.html#Packages-1",
    "page": "Packages",
    "title": "Packages",
    "category": "section",
    "text": "Julia has a built-in package manager for installing add-on functionality written in Julia. It can also install external libraries using your operating system's standard system for doing so, or by compiling from source. The list of registered Julia packages can be found at http://pkg.julialang.org. All package manager commands are found in the Pkg module, included in Julia's Base install.First we'll go over the mechanics of the Pkg family of commands and then we'll provide some guidance on how to get your package registered. Be sure to read the section below on package naming conventions, tagging versions and the importance of a REQUIRE file for when you're ready to add your code to the curated METADATA repository."
},

{
    "location": "manual/packages.html#Package-Status-1",
    "page": "Packages",
    "title": "Package Status",
    "category": "section",
    "text": "The Pkg.status() function prints out a summary of the state of packages you have installed. Initially, you'll have no packages installed:julia> Pkg.status()\nINFO: Initializing package repository /Users/stefan/.julia/v0.6\nINFO: Cloning METADATA from git://github.com/JuliaLang/METADATA.jl\nNo packages installed.Your package directory is automatically initialized the first time you run a Pkg command that expects it to exist – which includes Pkg.status(). Here's an example non-trivial set of required and additional packages:julia> Pkg.status()\nRequired packages:\n - Distributions                 0.2.8\n - SHA                           0.3.2\nAdditional packages:\n - NumericExtensions             0.2.17\n - Stats                         0.2.6These packages are all on registered versions, managed by Pkg. Packages can be in more complicated states, indicated by annotations to the right of the installed package version; we will explain these states and annotations as we encounter them. For programmatic usage, Pkg.installed() returns a dictionary, mapping installed package names to the version of that package which is installed:julia> Pkg.installed()\nDict{String,VersionNumber} with 4 entries:\n\"Distributions\"     => v\"0.2.8\"\n\"Stats\"             => v\"0.2.6\"\n\"SHA\"               => v\"0.3.2\"\n\"NumericExtensions\" => v\"0.2.17\""
},

{
    "location": "manual/packages.html#Adding-and-Removing-Packages-1",
    "page": "Packages",
    "title": "Adding and Removing Packages",
    "category": "section",
    "text": "Julia's package manager is a little unusual in that it is declarative rather than imperative. This means that you tell it what you want and it figures out what versions to install (or remove) to satisfy those requirements optimally – and minimally. So rather than installing a package, you just add it to the list of requirements and then \"resolve\" what needs to be installed. In particular, this means that if some package had been installed because it was needed by a previous version of something you wanted, and a newer version doesn't have that requirement anymore, updating will actually remove that package.Your package requirements are in the file ~/.julia/v0.6/REQUIRE. You can edit this file by hand and then call Pkg.resolve() to install, upgrade or remove packages to optimally satisfy the requirements, or you can do Pkg.edit(), which will open REQUIRE in your editor (configured via the EDITOR or VISUAL environment variables), and then automatically call Pkg.resolve() afterwards if necessary. If you only want to add or remove the requirement for a single package, you can also use the non-interactive Pkg.add() and Pkg.rm() commands, which add or remove a single requirement to REQUIRE and then call Pkg.resolve().You can add a package to the list of requirements with the Pkg.add() function, and the package and all the packages that it depends on will be installed:julia> Pkg.status()\nNo packages installed.\n\njulia> Pkg.add(\"Distributions\")\nINFO: Cloning cache of Distributions from git://github.com/JuliaStats/Distributions.jl.git\nINFO: Cloning cache of NumericExtensions from git://github.com/lindahua/NumericExtensions.jl.git\nINFO: Cloning cache of Stats from git://github.com/JuliaStats/Stats.jl.git\nINFO: Installing Distributions v0.2.7\nINFO: Installing NumericExtensions v0.2.17\nINFO: Installing Stats v0.2.6\nINFO: REQUIRE updated.\n\njulia> Pkg.status()\nRequired packages:\n - Distributions                 0.2.7\nAdditional packages:\n - NumericExtensions             0.2.17\n - Stats                         0.2.6What this is doing is first adding Distributions to your ~/.julia/v0.6/REQUIRE file:$ cat ~/.julia/v0.6/REQUIRE\nDistributionsIt then runs Pkg.resolve() using these new requirements, which leads to the conclusion that the Distributions package should be installed since it is required but not installed. As stated before, you can accomplish the same thing by editing your ~/.julia/v0.6/REQUIRE file by hand and then running Pkg.resolve() yourself:$ echo SHA >> ~/.julia/v0.6/REQUIRE\n\njulia> Pkg.resolve()\nINFO: Cloning cache of SHA from git://github.com/staticfloat/SHA.jl.git\nINFO: Installing SHA v0.3.2\n\njulia> Pkg.status()\nRequired packages:\n - Distributions                 0.2.7\n - SHA                           0.3.2\nAdditional packages:\n - NumericExtensions             0.2.17\n - Stats                         0.2.6This is functionally equivalent to calling Pkg.add(\"SHA\"), except that Pkg.add() doesn't change REQUIRE until after installation has completed, so if there are problems, REQUIRE will be left as it was before calling Pkg.add(). The format of the REQUIRE file is described in Requirements Specification; it allows, among other things, requiring specific ranges of versions of packages.When you decide that you don't want to have a package around any more, you can use Pkg.rm() to remove the requirement for it from the REQUIRE file:julia> Pkg.rm(\"Distributions\")\nINFO: Removing Distributions v0.2.7\nINFO: Removing Stats v0.2.6\nINFO: Removing NumericExtensions v0.2.17\nINFO: REQUIRE updated.\n\njulia> Pkg.status()\nRequired packages:\n - SHA                           0.3.2\n\njulia> Pkg.rm(\"SHA\")\nINFO: Removing SHA v0.3.2\nINFO: REQUIRE updated.\n\njulia> Pkg.status()\nNo packages installed.Once again, this is equivalent to editing the REQUIRE file to remove the line with each package name on it then running Pkg.resolve() to update the set of installed packages to match. While Pkg.add() and Pkg.rm() are convenient for adding and removing requirements for a single package, when you want to add or remove multiple packages, you can call Pkg.edit() to manually change the contents of REQUIRE and then update your packages accordingly. Pkg.edit() does not roll back the contents of REQUIRE if Pkg.resolve() fails – rather, you have to run Pkg.edit() again to fix the files contents yourself.Because the package manager uses libgit2 internally to manage the package git repositories, users may run into protocol issues (if behind a firewall, for example), when running Pkg.add(). By default, all GitHub-hosted packages wil be accessed via 'https'; this default can be modified by calling Pkg.setprotocol!(). The following command can be run from the command line in order to tell git to use 'https' instead of the 'git' protocol when cloning all repositories, wherever they are hosted:git config --global url.\"https://\".insteadOf git://However, this change will be system-wide and thus the use of Pkg.setprotocol!() is preferable.note: Note\nThe package manager functions also accept the .jl suffix on package names, though the suffix is stripped internally. For example:Pkg.add(\"Distributions.jl\")\nPkg.rm(\"Distributions.jl\")"
},

{
    "location": "manual/packages.html#Offline-Installation-of-Packages-1",
    "page": "Packages",
    "title": "Offline Installation of Packages",
    "category": "section",
    "text": "For machines with no Internet connection, packages may be installed by copying the package root directory (given by Pkg.dir()) from a machine with the same operating system and environment.Pkg.add() does the following within the package root directory:Adds the name of the package to REQUIRE.\nDownloads the package to .cache, then copies the package to the package root directory.\nRecursively performs step 2 against all the packages listed in the package's REQUIRE file.\nRuns Pkg.build()warning: Warning\nCopying installed packages from a different machine is brittle for packages requiring binary external dependencies. Such packages may break due to differences in operating system versions, build environments, and/or absolute path dependencies."
},

{
    "location": "manual/packages.html#Installing-Unregistered-Packages-1",
    "page": "Packages",
    "title": "Installing Unregistered Packages",
    "category": "section",
    "text": "Julia packages are simply git repositories, clonable via any of the protocols that git supports, and containing Julia code that follows certain layout conventions. Official Julia packages are registered in the METADATA.jl repository, available at a well-known location [1]. The Pkg.add() and Pkg.rm() commands in the previous section interact with registered packages, but the package manager can install and work with unregistered packages too. To install an unregistered package, use Pkg.clone(url), where url is a git URL from which the package can be cloned:julia> Pkg.clone(\"git://example.com/path/to/Package.jl.git\")\nINFO: Cloning Package from git://example.com/path/to/Package.jl.git\nCloning into 'Package'...\nremote: Counting objects: 22, done.\nremote: Compressing objects: 100% (10/10), done.\nremote: Total 22 (delta 8), reused 22 (delta 8)\nReceiving objects: 100% (22/22), 2.64 KiB, done.\nResolving deltas: 100% (8/8), done.By convention, Julia repository names end with .jl (the additional .git indicates a \"bare\" git repository), which keeps them from colliding with repositories for other languages, and also makes Julia packages easy to find in search engines. When packages are installed in your .julia/v0.6 directory, however, the extension is redundant so we leave it off.If unregistered packages contain a REQUIRE file at the top of their source tree, that file will be used to determine which registered packages the unregistered package depends on, and they will automatically be installed. Unregistered packages participate in the same version resolution logic as registered packages, so installed package versions will be adjusted as necessary to satisfy the requirements of both registered and unregistered packages.[1]: The official set of packages is at https://github.com/JuliaLang/METADATA.jl, but individuals and organizations can easily use a different metadata repository. This allows control which packages are available for automatic installation. One can allow only audited and approved package versions, and make private packages or forks available. See Custom METADATA Repository for details."
},

{
    "location": "manual/packages.html#Updating-Packages-1",
    "page": "Packages",
    "title": "Updating Packages",
    "category": "section",
    "text": "When package developers publish new registered versions of packages that you're using, you will, of course, want the new shiny versions. To get the latest and greatest versions of all your packages, just do Pkg.update():julia> Pkg.update()\nINFO: Updating METADATA...\nINFO: Computing changes...\nINFO: Upgrading Distributions: v0.2.8 => v0.2.10\nINFO: Upgrading Stats: v0.2.7 => v0.2.8The first step of updating packages is to pull new changes to ~/.julia/v0.6/METADATA and see if any new registered package versions have been published. After this, Pkg.update() attempts to update packages that are checked out on a branch and not dirty (i.e. no changes have been made to files tracked by git) by pulling changes from the package's upstream repository. Upstream changes will only be applied if no merging or rebasing is necessary – i.e. if the branch can be \"fast-forwarded\". If the branch cannot be fast-forwarded, it is assumed that you're working on it and will update the repository yourself.Finally, the update process recomputes an optimal set of package versions to have installed to satisfy your top-level requirements and the requirements of \"fixed\" packages. A package is considered fixed if it is one of the following:Unregistered: the package is not in METADATA – you installed it with Pkg.clone().\nChecked out: the package repo is on a development branch.\nDirty: changes have been made to files in the repo.If any of these are the case, the package manager cannot freely change the installed version of the package, so its requirements must be satisfied by whatever other package versions it picks. The combination of top-level requirements in ~/.julia/v0.6/REQUIRE and the requirement of fixed packages are used to determine what should be installed.You can also update only a subset of the installed packages, by providing arguments to the Pkg.update function. In that case, only the packages provided as arguments and their dependencies will be updated:julia> Pkg.update(\"Example\")\nINFO: Updating METADATA...\nINFO: Computing changes...\nINFO: Upgrading Example: v0.4.0 => 0.4.1This partial update process still computes the new set of package versions according to top-level requirements and \"fixed\" packages, but it additionally considers all other packages except those explicitly provided, and their dependencies, as fixed."
},

{
    "location": "manual/packages.html#Checkout,-Pin-and-Free-1",
    "page": "Packages",
    "title": "Checkout, Pin and Free",
    "category": "section",
    "text": "You may want to use the master version of a package rather than one of its registered versions. There might be fixes or functionality on master that you need that aren't yet published in any registered versions, or you may be a developer of the package and need to make changes on master or some other development branch. In such cases, you can do Pkg.checkout(pkg) to checkout the master branch of pkg or Pkg.checkout(pkg,branch) to checkout some other branch:julia> Pkg.add(\"Distributions\")\nINFO: Installing Distributions v0.2.9\nINFO: Installing NumericExtensions v0.2.17\nINFO: Installing Stats v0.2.7\nINFO: REQUIRE updated.\n\njulia> Pkg.status()\nRequired packages:\n - Distributions                 0.2.9\nAdditional packages:\n - NumericExtensions             0.2.17\n - Stats                         0.2.7\n\njulia> Pkg.checkout(\"Distributions\")\nINFO: Checking out Distributions master...\nINFO: No packages to install, update or remove.\n\njulia> Pkg.status()\nRequired packages:\n - Distributions                 0.2.9+             master\nAdditional packages:\n - NumericExtensions             0.2.17\n - Stats                         0.2.7Immediately after installing Distributions with Pkg.add() it is on the current most recent registered version – 0.2.9 at the time of writing this. Then after running Pkg.checkout(\"Distributions\"), you can see from the output of Pkg.status() that Distributions is on an unregistered version greater than 0.2.9, indicated by the \"pseudo-version\" number 0.2.9+.When you checkout an unregistered version of a package, the copy of the REQUIRE file in the package repo takes precedence over any requirements registered in METADATA, so it is important that developers keep this file accurate and up-to-date, reflecting the actual requirements of the current version of the package. If the REQUIRE file in the package repo is incorrect or missing, dependencies may be removed when the package is checked out. This file is also used to populate newly published versions of the package if you use the API that Pkg provides for this (described below).When you decide that you no longer want to have a package checked out on a branch, you can \"free\" it back to the control of the package manager with Pkg.free(pkg):julia> Pkg.free(\"Distributions\")\nINFO: Freeing Distributions...\nINFO: No packages to install, update or remove.\n\njulia> Pkg.status()\nRequired packages:\n - Distributions                 0.2.9\nAdditional packages:\n - NumericExtensions             0.2.17\n - Stats                         0.2.7After this, since the package is on a registered version and not on a branch, its version will be updated as new registered versions of the package are published.If you want to pin a package at a specific version so that calling Pkg.update() won't change the version the package is on, you can use the Pkg.pin() function:julia> Pkg.pin(\"Stats\")\nINFO: Creating Stats branch pinned.47c198b1.tmp\n\njulia> Pkg.status()\nRequired packages:\n - Distributions                 0.2.9\nAdditional packages:\n - NumericExtensions             0.2.17\n - Stats                         0.2.7              pinned.47c198b1.tmpAfter this, the Stats package will remain pinned at version 0.2.7 – or more specifically, at commit 47c198b1, but since versions are permanently associated a given git hash, this is the same thing. Pkg.pin() works by creating a throw-away branch for the commit you want to pin the package at and then checking that branch out. By default, it pins a package at the current commit, but you can choose a different version by passing a second argument:julia> Pkg.pin(\"Stats\",v\"0.2.5\")\nINFO: Creating Stats branch pinned.1fd0983b.tmp\nINFO: No packages to install, update or remove.\n\njulia> Pkg.status()\nRequired packages:\n - Distributions                 0.2.9\nAdditional packages:\n - NumericExtensions             0.2.17\n - Stats                         0.2.5              pinned.1fd0983b.tmpNow the Stats package is pinned at commit 1fd0983b, which corresponds to version 0.2.5. When you decide to \"unpin\" a package and let the package manager update it again, you can use Pkg.free() like you would to move off of any branch:julia> Pkg.free(\"Stats\")\nINFO: Freeing Stats...\nINFO: No packages to install, update or remove.\n\njulia> Pkg.status()\nRequired packages:\n - Distributions                 0.2.9\nAdditional packages:\n - NumericExtensions             0.2.17\n - Stats                         0.2.7After this, the Stats package is managed by the package manager again, and future calls to Pkg.update() will upgrade it to newer versions when they are published. The throw-away pinned.1fd0983b.tmp branch remains in your local Stats repo, but since git branches are extremely lightweight, this doesn't really matter; if you feel like cleaning them up, you can go into the repo and delete those branches [2].[2]: Packages that aren't on branches will also be marked as dirty if you make changes in the repo, but that's a less common thing to do."
},

{
    "location": "manual/packages.html#Custom-METADATA-Repository-1",
    "page": "Packages",
    "title": "Custom METADATA Repository",
    "category": "section",
    "text": "By default, Julia assumes you will be using the official METADATA.jl repository for downloading and installing packages. You can also provide a different metadata repository location. A common approach is to keep your metadata-v2 branch up to date with the Julia official branch and add another branch with your custom packages. You can initialize your local metadata repository using that custom location and branch and then periodically rebase your custom branch with the official metadata-v2 branch. In order to use a custom repository and branch, issue the following command:julia> Pkg.init(\"https://me.example.com/METADATA.jl.git\", \"branch\")The branch argument is optional and defaults to metadata-v2. Once initialized, a file named META_BRANCH in your ~/.julia/vX.Y/ path will track the branch that your METADATA repository was initialized with. If you want to change branches, you will need to either modify the META_BRANCH file directly (be careful!) or remove the vX.Y directory and re-initialize your METADATA repository using the Pkg.init command."
},

{
    "location": "manual/packages.html#Package-Development-1",
    "page": "Packages",
    "title": "Package Development",
    "category": "section",
    "text": "Julia's package manager is designed so that when you have a package installed, you are already in a position to look at its source code and full development history. You are also able to make changes to packages, commit them using git, and easily contribute fixes and enhancements upstream. Similarly, the system is designed so that if you want to create a new package, the simplest way to do so is within the infrastructure provided by the package manager."
},

{
    "location": "manual/packages.html#man-initial-setup-1",
    "page": "Packages",
    "title": "Initial Setup",
    "category": "section",
    "text": "Since packages are git repositories, before doing any package development you should setup the following standard global git configuration settings:$ git config --global user.name \"FULL NAME\"\n$ git config --global user.email \"EMAIL\"where FULL NAME is your actual full name (spaces are allowed between the double quotes) and EMAIL is your actual email address. Although it isn't necessary to use GitHub to create or publish Julia packages, most Julia packages as of writing this are hosted on GitHub and the package manager knows how to format origin URLs correctly and otherwise work with the service smoothly. We recommend that you create a free account on GitHub and then do:$ git config --global github.user \"USERNAME\"where USERNAME is your actual GitHub user name. Once you do this, the package manager knows your GitHub user name and can configure things accordingly. You should also upload your public SSH key to GitHub and set up an SSH agent on your development machine so that you can push changes with minimal hassle. In the future, we will make this system extensible and support other common git hosting options like BitBucket and allow developers to choose their favorite. Since the package development functions has been moved to the PkgDev package, you need to run Pkg.add(\"PkgDev\"); import PkgDev to access the functions starting with PkgDev. in the document below."
},

{
    "location": "manual/packages.html#Making-changes-to-an-existing-package-1",
    "page": "Packages",
    "title": "Making changes to an existing package",
    "category": "section",
    "text": ""
},

{
    "location": "manual/packages.html#Documentation-changes-1",
    "page": "Packages",
    "title": "Documentation changes",
    "category": "section",
    "text": "If you want to improve the online documentation of a package, the easiest approach (at least for small changes) is to use GitHub's online editing functionality. First, navigate to the repository's GitHub \"home page,\" find the file (e.g., README.md) within the repository's folder structure, and click on it. You'll see the contents displayed, along with a small \"pencil\" icon in the upper right hand corner. Clicking that icon opens the file in edit mode. Make your changes, write a brief summary describing the changes you want to make (this is your commit message), and then hit \"Propose file change.\" Your changes will be submitted for consideration by the package owner(s) and collaborators.For larger documentation changes–and especially ones that you expect to have to update in response to feedback–you might find it easier to use the procedure for code changes described below."
},

{
    "location": "manual/packages.html#Code-changes-1",
    "page": "Packages",
    "title": "Code changes",
    "category": "section",
    "text": ""
},

{
    "location": "manual/packages.html#Executive-summary-1",
    "page": "Packages",
    "title": "Executive summary",
    "category": "section",
    "text": "Here we assume you've already set up git on your local machine and have a GitHub account (see above). Let's imagine you're fixing a bug in the Images package:Pkg.checkout(\"Images\")           # check out the master branch\n<here, make sure your bug is still a bug and hasn't been fixed already>\ncd(Pkg.dir(\"Images\"))\n;git checkout -b myfixes         # create a branch for your changes\n<edit code>                      # be sure to add a test for your bug\nPkg.test(\"Images\")               # make sure everything works now\n;git commit -a -m \"Fix foo by calling bar\"   # write a descriptive message\nusing PkgDev\nPkgDev.submit(\"Images\")The last line will present you with a link to submit a pull request to incorporate your changes."
},

{
    "location": "manual/packages.html#Detailed-description-1",
    "page": "Packages",
    "title": "Detailed description",
    "category": "section",
    "text": "If you want to fix a bug or add new functionality, you want to be able to test your changes before you submit them for consideration. You also need to have an easy way to update your proposal in response to the package owner's feedback. Consequently, in this case the strategy is to work locally on your own machine; once you are satisfied with your changes, you submit them for consideration.  This process is called a pull request because you are asking to \"pull\" your changes into the project's main repository. Because the online repository can't see the code on your private machine, you first push your changes to a publicly-visible location, your own online fork of the package (hosted on your own personal GitHub account).Let's assume you already have the Foo package installed. In the description below, anything starting with Pkg. or PkgDev. is meant to be typed at the Julia prompt; anything starting with git is meant to be typed in julia's shell mode (or using the shell that comes with your operating system). Within Julia, you can combine these two modes:julia> cd(Pkg.dir(\"Foo\"))          # go to Foo's folder\n\nshell> git command arguments...    # command will apply to FooNow suppose you're ready to make some changes to Foo. While there are several possible approaches, here is one that is widely used:From the Julia prompt, type Pkg.checkout(\"Foo\"). This ensures you're running the latest code (the master branch), rather than just whatever \"official release\" version you have installed. (If you're planning to fix a bug, at this point it's a good idea to check again whether the bug has already been fixed by someone else. If it has, you can request that a new official release be tagged so that the fix gets distributed to the rest of the community.) If you receive an error Foo is dirty, bailing, see Dirty packages below.\nCreate a branch for your changes: navigate to the package folder (the one that Julia reports from Pkg.dir(\"Foo\")) and (in shell mode) create a new branch using git checkout -b <newbranch>, where <newbranch> might be some descriptive name (e.g., fixbar). By creating a branch, you ensure that you can easily go back and forth between your new work and the current master branch (see https://git-scm.com/book/en/v2/Git-Branching-Branches-in-a-Nutshell).\nIf you forget to do this step until after you've already made some changes, don't worry: see more detail about branching below.\nMake your changes. Whether it's fixing a bug or adding new functionality, in most cases your change should include updates to both the src/ and test/ folders. If you're fixing a bug, add your minimal example demonstrating the bug (on the current code) to the test suite; by contributing a test for the bug, you ensure that the bug won't accidentally reappear at some later time due to other changes. If you're adding new functionality, creating tests demonstrates to the package owner that you've made sure your code works as intended.\nRun the package's tests and make sure they pass. There are several ways to run the tests:\nFrom Julia, run Pkg.test(\"Foo\"): this will run your tests in a separate (new) julia process.\nFrom Julia, include(\"runtests.jl\") from the package's test/ folder (it's possible the file has a different name, look for one that runs all the tests): this allows you to run the tests repeatedly in the same session without reloading all the package code; for packages that take a while to load, this can be much faster. With this approach, you do have to do some extra work to make changes in the package code.\nFrom the shell, run julia ../test/runtests.jl from within the package's src/ folder.\nCommit your changes: see https://git-scm.com/book/en/v2/Git-Basics-Recording-Changes-to-the-Repository.\nSubmit your changes: From the Julia prompt, type PkgDev.submit(\"Foo\"). This will push your changes to your GitHub fork, creating it if it doesn't already exist. (If you encounter an error, make sure you've set up your SSH keys.) Julia will then give you a hyperlink; open that link, edit the message, and then click \"submit.\" At that point, the package owner will be notified of your changes and may initiate discussion. (If you are comfortable with git, you can also do these steps manually from the shell.)\nThe package owner may suggest additional improvements. To respond to those suggestions, you can easily update the pull request (this only works for changes that have not already been merged; for merged pull requests, make new changes by starting a new branch):\nIf you've changed branches in the meantime, make sure you go back to the same branch with git checkout fixbar (from shell mode) or Pkg.checkout(\"Foo\", \"fixbar\") (from the Julia prompt).\nAs above, make your changes, run the tests, and commit your changes.\nFrom the shell, type git push.  This will add your new commit(s) to the same pull request; you should see them appear automatically on the page holding the discussion of your pull request.\nOne potential type of change the owner may request is that you squash your commits. See Squashing below."
},

{
    "location": "manual/packages.html#Dirty-packages-1",
    "page": "Packages",
    "title": "Dirty packages",
    "category": "section",
    "text": "If you can't change branches because the package manager complains that your package is dirty, it means you have some changes that have not been committed. From the shell, use git diff to see what these changes are; you can either discard them (git checkout changedfile.jl) or commit them before switching branches.  If you can't easily resolve the problems manually, as a last resort you can delete the entire \"Foo\" folder and reinstall a fresh copy with Pkg.add(\"Foo\"). Naturally, this deletes any changes you've made."
},

{
    "location": "manual/packages.html#man-branch-post-hoc-1",
    "page": "Packages",
    "title": "Making a branch post hoc",
    "category": "section",
    "text": "Especially for newcomers to git, one often forgets to create a new branch until after some changes have already been made. If you haven't yet staged or committed your changes, you can create a new branch with git checkout -b <newbranch> just as usual–git will kindly show you that some files have been modified and create the new branch for you. Your changes have not yet been committed to this new branch, so the normal work rules still apply.However, if you've already made a commit to master but wish to go back to the official master (called origin/master), use the following procedure:Create a new branch. This branch will hold your changes.\nMake sure everything is committed to this branch.\ngit checkout master. If this fails, do not proceed further until you have resolved the problems, or you may lose your changes.\nResetmaster (your current branch) back to an earlier state with git reset --hard origin/master (see https://git-scm.com/blog/2011/07/11/reset.html).This requires a bit more familiarity with git, so it's much better to get in the habit of creating a branch at the outset."
},

{
    "location": "manual/packages.html#man-squashing-and-rebasing-1",
    "page": "Packages",
    "title": "Squashing and rebasing",
    "category": "section",
    "text": "Depending on the tastes of the package owner (s)he may ask you to \"squash\" your commits. This is especially likely if your change is quite simple but your commit history looks like this:WIP: add new 1-line whizbang function (currently breaks package)\nFinish whizbang function\nFix typo in variable name\nOops, don't forget to supply default argument\nSplit into two 1-line functions\nRats, forgot to export the second function\n...This gets into the territory of more advanced git usage, and you're encouraged to do some reading (https://git-scm.com/book/en/v2/Git-Branching-Rebasing).  However, a brief summary of the procedure is as follows:To protect yourself from error, start from your fixbar branch and create a new branch with git checkout -b fixbar_backup.  Since you started from fixbar, this will be a copy. Now go back to the one you intend to modify with git checkout fixbar.\nFrom the shell, type git rebase -i origin/master.\nTo combine commits, change pick to squash (for additional options, consult other sources). Save the file and close the editor window.\nEdit the combined commit message.If the rebase goes badly, you can go back to the beginning to try again like this:git checkout fixbar\ngit reset --hard fixbar_backupNow let's assume you've rebased successfully. Since your fixbar repository has now diverged from the one in your GitHub fork, you're going to have to do a force push:To make it easy to refer to your GitHub fork, create a \"handle\" for it with git remote add myfork https://github.com/myaccount/Foo.jl.git, where the URL comes from the \"clone URL\" on your GitHub fork's page.\nForce-push to your fork with git push myfork +fixbar. The + indicates that this should replace the fixbar branch found at myfork."
},

{
    "location": "manual/packages.html#Creating-a-new-Package-1",
    "page": "Packages",
    "title": "Creating a new Package",
    "category": "section",
    "text": ""
},

{
    "location": "manual/packages.html#REQUIRE-speaks-for-itself-1",
    "page": "Packages",
    "title": "REQUIRE speaks for itself",
    "category": "section",
    "text": "You should have a REQUIRE file in your package repository, with a bare minimum directive of what Julia version you expect your users to be running for the package to work. Putting a floor on what Julia version your package supports is done by simply adding julia 0.x in this file. While this line is partly informational, it also has the consequence of whether Pkg.update() will update code found in .julia version directories. It will not update code found in version directories beneath the floor of what's specified in your REQUIRE.As the development version 0.y matures, you may find yourself using it more frequently, and wanting your package to support it. Be warned, the development branch of Julia is the land of breakage, and you can expect things to break. When you go about fixing whatever broke your package in the development 0.y branch, you will likely find that you just broke your package on the stable version.There is a mechanism found in the Compat package that will enable you to support both the stable version and breaking changes found in the development version. Should you decide to use this solution, you will need to add Compat to your REQUIRE file. In this case, you will still have julia 0.x in your REQUIRE. The x is the floor version of what your package supports.You might also have no interest in supporting the development version of Julia. Just as you can add a floor to the version you expect your users to be on, you can set an upper bound. In this case, you would put julia 0.x 0.y- in your REQUIRE file. The - at the end of the version number means pre-release versions of that specific version from the very first commit. By setting it as the ceiling, you mean the code supports everything up to but not including the ceiling version.Another scenario is that you are writing the bulk of the code for your package with Julia 0.y and do not want to support the current stable version of Julia. If you choose to do this, simply add julia 0.y- to your REQUIRE. Just remember to change the julia 0.y- to julia 0.y in your REQUIRE file once 0.y is officially released. If you don't edit the dash cruft you are suggesting that you support both the development and stable versions of the same version number! That would be madness. See the Requirements Specification for the full format of REQUIRE.Lastly, in many cases you may need extra packages for testing. Additional packages which are only required for tests should be specified in the test/REQUIRE file. This REQUIRE file has the same specification as the standard REQUIRE file."
},

{
    "location": "manual/packages.html#Guidelines-for-naming-a-package-1",
    "page": "Packages",
    "title": "Guidelines for naming a package",
    "category": "section",
    "text": "Package names should be sensible to most Julia users, even to those who are not domain experts. When you submit your package to METADATA, you can expect a little back and forth about the package name with collaborators, especially if it's ambiguous or can be confused with something other than what it is. During this bike-shedding, it's not uncommon to get a range of different name suggestions. These are only suggestions though, with the intent being to keep a tidy namespace in the curated METADATA repository. Since this repository belongs to the entire community, there will likely be a few collaborators who care about your package name. Here are some guidelines to follow in naming your package:Avoid jargon. In particular, avoid acronyms unless there is minimal possibility of confusion.\nIt's ok to say USA if you're talking about the USA.\nIt's not ok to say PMA, even if you're talking about positive mental attitude.\nAvoid using Julia in your package name.\nIt is usually clear from context and to your users that the package is a Julia package.\nHaving Julia in the name can imply that the package is connected to, or endorsed by, contributors to the Julia language itself.\nPackages that provide most of their functionality in association with a new type should have pluralized names.\nDataFrames provides the DataFrame type.\nBloomFilters provides the BloomFilter type.\nIn contrast, JuliaParser provides no new type, but instead new functionality in the JuliaParser.parse() function.\nErr on the side of clarity, even if clarity seems long-winded to you.\nRandomMatrices is a less ambiguous name than RndMat or RMT, even though the latter are shorter.\nA less systematic name may suit a package that implements one of several possible approaches to its domain.\nJulia does not have a single comprehensive plotting package. Instead, Gadfly, PyPlot, Winston and other packages each implement a unique approach based on a particular design philosophy.\nIn contrast, SortingAlgorithms provides a consistent interface to use many well-established sorting algorithms.\nPackages that wrap external libraries or programs should be named after those libraries or programs.\nCPLEX.jl wraps the CPLEX library, which can be identified easily in a web search.\nMATLAB.jl provides an interface to call the MATLAB engine from within Julia."
},

{
    "location": "manual/packages.html#Generating-the-package-1",
    "page": "Packages",
    "title": "Generating the package",
    "category": "section",
    "text": "Suppose you want to create a new Julia package called FooBar. To get started, do PkgDev.generate(pkg,license) where pkg is the new package name and license is the name of a license that the package generator knows about:julia> PkgDev.generate(\"FooBar\",\"MIT\")\nINFO: Initializing FooBar repo: /Users/stefan/.julia/v0.6/FooBar\nINFO: Origin: git://github.com/StefanKarpinski/FooBar.jl.git\nINFO: Generating LICENSE.md\nINFO: Generating README.md\nINFO: Generating src/FooBar.jl\nINFO: Generating test/runtests.jl\nINFO: Generating REQUIRE\nINFO: Generating .travis.yml\nINFO: Generating appveyor.yml\nINFO: Generating .gitignore\nINFO: Committing FooBar generated filesThis creates the directory ~/.julia/v0.6/FooBar, initializes it as a git repository, generates a bunch of files that all packages should have, and commits them to the repository:$ cd ~/.julia/v0.6/FooBar && git show --stat\n\ncommit 84b8e266dae6de30ab9703150b3bf771ec7b6285\nAuthor: Stefan Karpinski <stefan@karpinski.org>\nDate:   Wed Oct 16 17:57:58 2013 -0400\n\n    FooBar.jl generated files.\n\n        license: MIT\n        authors: Stefan Karpinski\n        years:   2013\n        user:    StefanKarpinski\n\n    Julia Version 0.3.0-prerelease+3217 [5fcfb13*]\n\n .gitignore       |  2 ++\n .travis.yml      | 13 +++++++++++++\n LICENSE.md       | 22 +++++++++++++++++++++++\n README.md        |  3 +++\n REQUIRE          |  1 +\n appveyor.yml     | 34 ++++++++++++++++++++++++++++++++++\n src/FooBar.jl    |  5 +++++\n test/runtests.jl |  5 +++++\n 8 files changed, 85 insertions(+)At the moment, the package manager knows about the MIT \"Expat\" License, indicated by \"MIT\", the Simplified BSD License, indicated by \"BSD\", and version 2.0 of the Apache Software License, indicated by \"ASL\". If you want to use a different license, you can ask us to add it to the package generator, or just pick one of these three and then modify the ~/.julia/v0.6/PACKAGE/LICENSE.md file after it has been generated.If you created a GitHub account and configured git to know about it, PkgDev.generate() will set an appropriate origin URL for you. It will also automatically generate a .travis.yml file for using the Travis automated testing service, and an appveyor.yml file for using AppVeyor. You will have to enable testing on the Travis and AppVeyor websites for your package repository, but once you've done that, it will already have working tests. Of course, all the default testing does is verify that using FooBar in Julia works."
},

{
    "location": "manual/packages.html#Loading-Static-Non-Julia-Files-1",
    "page": "Packages",
    "title": "Loading Static Non-Julia Files",
    "category": "section",
    "text": "If your package code needs to load static files which are not Julia code, e.g. an external library or data files, and are located within the package directory, use the @__DIR__ macro to determine the directory of the current source file. For example if FooBar/src/FooBar.jl needs to load FooBar/data/foo.csv, use the following code:datapath = joinpath(@__DIR__, \"..\", \"data\")\nfoo = readcsv(joinpath(datapath, \"foo.csv\"))"
},

{
    "location": "manual/packages.html#Making-Your-Package-Available-1",
    "page": "Packages",
    "title": "Making Your Package Available",
    "category": "section",
    "text": "Once you've made some commits and you're happy with how FooBar is working, you may want to get some other people to try it out. First you'll need to create the remote repository and push your code to it; we don't yet automatically do this for you, but we will in the future and it's not too hard to figure out [3]. Once you've done this, letting people try out your code is as simple as sending them the URL of the published repo – in this case:git://github.com/StefanKarpinski/FooBar.jl.gitFor your package, it will be your GitHub user name and the name of your package, but you get the idea. People you send this URL to can use Pkg.clone() to install the package and try it out:julia> Pkg.clone(\"git://github.com/StefanKarpinski/FooBar.jl.git\")\nINFO: Cloning FooBar from git@github.com:StefanKarpinski/FooBar.jl.git[3]: Installing and using GitHub's \"hub\" tool is highly recommended. It allows you to do things like run hub create in the package repo and have it automatically created via GitHub's API."
},

{
    "location": "manual/packages.html#Tagging-and-Publishing-Your-Package-1",
    "page": "Packages",
    "title": "Tagging and Publishing Your Package",
    "category": "section",
    "text": "tip: Tip\nIf you are hosting your package on GitHub, you can use the attobot integration to handle package registration, tagging and publishing.Once you've decided that FooBar is ready to be registered as an official package, you can add it to your local copy of METADATA using PkgDev.register():julia> PkgDev.register(\"FooBar\")\nINFO: Registering FooBar at git://github.com/StefanKarpinski/FooBar.jl.git\nINFO: Committing METADATA for FooBarThis creates a commit in the ~/.julia/v0.6/METADATA repo:$ cd ~/.julia/v0.6/METADATA && git show\n\ncommit 9f71f4becb05cadacb983c54a72eed744e5c019d\nAuthor: Stefan Karpinski <stefan@karpinski.org>\nDate:   Wed Oct 16 18:46:02 2013 -0400\n\n    Register FooBar\n\ndiff --git a/FooBar/url b/FooBar/url\nnew file mode 100644\nindex 0000000..30e525e\n--- /dev/null\n+++ b/FooBar/url\n@@ -0,0 +1 @@\n+git://github.com/StefanKarpinski/FooBar.jl.gitThis commit is only locally visible, however. To make it visible to the Julia community, you need to merge your local METADATA upstream into the official repo. The PkgDev.publish() command will fork the METADATA repository on GitHub, push your changes to your fork, and open a pull request:julia> PkgDev.publish()\nINFO: Validating METADATA\nINFO: No new package versions to publish\nINFO: Submitting METADATA changes\nINFO: Forking JuliaLang/METADATA.jl to StefanKarpinski\nINFO: Pushing changes as branch pull-request/ef45f54b\nINFO: To create a pull-request open:\n\n  https://github.com/StefanKarpinski/METADATA.jl/compare/pull-request/ef45f54btip: Tip\nIf PkgDev.publish() fails with error:ERROR: key not found: \"token\"then you may have encountered an issue from using the GitHub API on multiple systems. The solution is to delete the \"Julia Package Manager\" personal access token from your Github account and try again.Other failures may require you to circumvent PkgDev.publish() by creating a pull request on GitHub. See: Publishing METADATA manually below.Once the package URL for FooBar is registered in the official METADATA repo, people know where to clone the package from, but there still aren't any registered versions available. You can tag and register it with the PkgDev.tag() command:julia> PkgDev.tag(\"FooBar\")\nINFO: Tagging FooBar v0.0.1\nINFO: Committing METADATA for FooBarThis tags v0.0.1 in the FooBar repo:$ cd ~/.julia/v0.6/FooBar && git tag\nv0.0.1It also creates a new version entry in your local METADATA repo for FooBar:$ cd ~/.julia/v0.6/FooBar && git show\ncommit de77ee4dc0689b12c5e8b574aef7f70e8b311b0e\nAuthor: Stefan Karpinski <stefan@karpinski.org>\nDate:   Wed Oct 16 23:06:18 2013 -0400\n\n    Tag FooBar v0.0.1\n\ndiff --git a/FooBar/versions/0.0.1/sha1 b/FooBar/versions/0.0.1/sha1\nnew file mode 100644\nindex 0000000..c1cb1c1\n--- /dev/null\n+++ b/FooBar/versions/0.0.1/sha1\n@@ -0,0 +1 @@\n+84b8e266dae6de30ab9703150b3bf771ec7b6285The PkgDev.tag() command takes an optional second argument that is either an explicit version number object like v\"0.0.1\" or one of the symbols :patch, :minor or :major. These increment the patch, minor or major version number of your package intelligently.Adding a tagged version of your package will expedite the official registration into METADATA.jl by collaborators. It is strongly recommended that you complete this process, regardless if your package is completely ready for an official release.As a general rule, packages should be tagged 0.0.1 first. Since Julia itself hasn't achieved 1.0 status, it's best to be conservative in your package's tagged versions.As with PkgDev.register(), these changes to METADATA aren't available to anyone else until they've been included upstream. Again, use the PkgDev.publish() command, which first makes sure that individual package repos have been tagged, pushes them if they haven't already been, and then opens a pull request to METADATA:julia> PkgDev.publish()\nINFO: Validating METADATA\nINFO: Pushing FooBar permanent tags: v0.0.1\nINFO: Submitting METADATA changes\nINFO: Forking JuliaLang/METADATA.jl to StefanKarpinski\nINFO: Pushing changes as branch pull-request/3ef4f5c4\nINFO: To create a pull-request open:\n\n  https://github.com/StefanKarpinski/METADATA.jl/compare/pull-request/3ef4f5c4"
},

{
    "location": "manual/packages.html#Publishing-METADATA-manually-1",
    "page": "Packages",
    "title": "Publishing METADATA manually",
    "category": "section",
    "text": "If PkgDev.publish() fails you can follow these instructions to manually publish your package.By \"forking\" the main METADATA repository, you can create a personal copy (of METADATA.jl) under your GitHub account. Once that copy exists, you can push your local changes to your copy (just like any other GitHub project).Create a fork of METADATA.jl.\nAdd your fork as a remote repository for the METADATA repository on your local computer (in the terminal where USERNAME is your github username):\ncd ~/.julia/v0.6/METADATA\ngit remote add USERNAME https://github.com/USERNAME/METADATA.jl.git\nPush your changes to your fork:\ngit push USERNAME metadata-v2\nIf all of that works, then go back to the GitHub page for your fork, and click the \"pull request\" link."
},

{
    "location": "manual/packages.html#Fixing-Package-Requirements-1",
    "page": "Packages",
    "title": "Fixing Package Requirements",
    "category": "section",
    "text": "If you need to fix the registered requirements of an already-published package version, you can do so just by editing the metadata for that version, which will still have the same commit hash – the hash associated with a version is permanent:$ cd ~/.julia/v0.6/METADATA/FooBar/versions/0.0.1 && cat requires\njulia 0.3-\n$ vi requiresSince the commit hash stays the same, the contents of the REQUIRE file that will be checked out in the repo will not match the requirements in METADATA after such a change; this is unavoidable. When you fix the requirements in METADATA for a previous version of a package, however, you should also fix the REQUIRE file in the current version of the package."
},

{
    "location": "manual/packages.html#Requirements-Specification-1",
    "page": "Packages",
    "title": "Requirements Specification",
    "category": "section",
    "text": "The ~/.julia/v0.6/REQUIRE file, the REQUIRE file inside packages, and the METADATA package requires files use a simple line-based format to express the ranges of package versions which need to be installed. Package REQUIRE and METADATA requires files should also include the range of versions of julia the package is expected to work with. Additionally, packages can include a test/REQUIRE file to specify additional packages which are only required for testing.Here's how these files are parsed and interpreted.Everything after a # mark is stripped from each line as a comment.\nIf nothing but whitespace is left, the line is ignored.\nIf there are non-whitespace characters remaining, the line is a requirement and the is split on whitespace into words.The simplest possible requirement is just the name of a package name on a line by itself:DistributionsThis requirement is satisfied by any version of the Distributions package. The package name can be followed by zero or more version numbers in ascending order, indicating acceptable intervals of versions of that package. One version opens an interval, while the next closes it, and the next opens a new interval, and so on; if an odd number of version numbers are given, then arbitrarily large versions will satisfy; if an even number of version numbers are given, the last one is an upper limit on acceptable version numbers. For example, the line:Distributions 0.1is satisfied by any version of Distributions greater than or equal to 0.1.0. Suffixing a version with - allows any pre-release versions as well. For example:Distributions 0.1-is satisfied by pre-release versions such as 0.1-dev or 0.1-rc1, or by any version greater than or equal to 0.1.0.This requirement entry:Distributions 0.1 0.2.5is satisfied by versions from 0.1.0 up to, but not including 0.2.5. If you want to indicate that any 0.1.x version will do, you will want to write:Distributions 0.1 0.2-If you want to start accepting versions after 0.2.7, you can write:Distributions 0.1 0.2- 0.2.7If a requirement line has leading words that begin with @, it is a system-dependent requirement. If your system matches these system conditionals, the requirement is included, if not, the requirement is ignored. For example:@osx Homebrewwill require the Homebrew package only on systems where the operating system is OS X. The system conditions that are currently supported are (hierarchically):@unix\n@linux\n@bsd\n@osx\n@windowsThe @unix condition is satisfied on all UNIX systems, including Linux and BSD. Negated system conditionals are also supported by adding a ! after the leading @. Examples:@!windows\n@unix @!osxThe first condition applies to any system but Windows and the second condition applies to any UNIX system besides OS X.Runtime checks for the current version of Julia can be made using the built-in VERSION variable, which is of type VersionNumber. Such code is occasionally necessary to keep track of new or deprecated functionality between various releases of Julia. Examples of runtime checks:VERSION < v\"0.3-\" #exclude all pre-release versions of 0.3\n\nv\"0.2-\" <= VERSION < v\"0.3-\" #get all 0.2 versions, including pre-releases, up to the above\n\nv\"0.2\" <= VERSION < v\"0.3-\" #To get only stable 0.2 versions (Note v\"0.2\" == v\"0.2.0\")\n\nVERSION >= v\"0.2.1\" #get at least version 0.2.1See the section on version number literals for a more complete description."
},

{
    "location": "manual/profile.html#",
    "page": "Profiling",
    "title": "Profiling",
    "category": "page",
    "text": ""
},

{
    "location": "manual/profile.html#Profiling-1",
    "page": "Profiling",
    "title": "Profiling",
    "category": "section",
    "text": "The Profile module provides tools to help developers improve the performance of their code. When used, it takes measurements on running code, and produces output that helps you understand how much time is spent on individual line(s). The most common usage is to identify \"bottlenecks\" as targets for optimization.Profile implements what is known as a \"sampling\" or statistical profiler.  It works by periodically taking a backtrace during the execution of any task. Each backtrace captures the currently-running function and line number, plus the complete chain of function calls that led to this line, and hence is a \"snapshot\" of the current state of execution.If much of your run time is spent executing a particular line of code, this line will show up frequently in the set of all backtraces. In other words, the \"cost\" of a given line–or really, the cost of the sequence of function calls up to and including this line–is proportional to how often it appears in the set of all backtraces.A sampling profiler does not provide complete line-by-line coverage, because the backtraces occur at intervals (by default, 1 ms on Unix systems and 10 ms on Windows, although the actual scheduling is subject to operating system load). Moreover, as discussed further below, because samples are collected at a sparse subset of all execution points, the data collected by a sampling profiler is subject to statistical noise.Despite these limitations, sampling profilers have substantial strengths:You do not have to make any modifications to your code to take timing measurements (in contrast to the alternative instrumenting profiler).\nIt can profile into Julia's core code and even (optionally) into C and Fortran libraries.\nBy running \"infrequently\" there is very little performance overhead; while profiling, your code can run at nearly native speed.For these reasons, it's recommended that you try using the built-in sampling profiler before considering any alternatives."
},

{
    "location": "manual/profile.html#Basic-usage-1",
    "page": "Profiling",
    "title": "Basic usage",
    "category": "section",
    "text": "Let's work with a simple test case:julia> function myfunc()\n           A = rand(200, 200, 400)\n           maximum(A)\n       endIt's a good idea to first run the code you intend to profile at least once (unless you want to profile Julia's JIT-compiler):julia> myfunc() # run once to force compilationNow we're ready to profile this function:julia> @profile myfunc()To see the profiling results, there is a graphical browser available, but here we'll use the text-based display that comes with the standard library:julia> Profile.print()\n80 ./event.jl:73; (::Base.REPL.##1#2{Base.REPL.REPLBackend})()\n 80 ./REPL.jl:97; macro expansion\n  80 ./REPL.jl:66; eval_user_input(::Any, ::Base.REPL.REPLBackend)\n   80 ./boot.jl:235; eval(::Module, ::Any)\n    80 ./<missing>:?; anonymous\n     80 ./profile.jl:23; macro expansion\n      52 ./REPL[1]:2; myfunc()\n       38 ./random.jl:431; rand!(::MersenneTwister, ::Array{Float64,3}, ::Int64, ::Type{B...\n        38 ./dSFMT.jl:84; dsfmt_fill_array_close_open!(::Base.dSFMT.DSFMT_state, ::Ptr{F...\n       14 ./random.jl:278; rand\n        14 ./random.jl:277; rand\n         14 ./random.jl:366; rand\n          14 ./random.jl:369; rand\n      28 ./REPL[1]:3; myfunc()\n       28 ./reduce.jl:270; _mapreduce(::Base.#identity, ::Base.#scalarmax, ::IndexLinear,...\n        3  ./reduce.jl:426; mapreduce_impl(::Base.#identity, ::Base.#scalarmax, ::Array{F...\n        25 ./reduce.jl:428; mapreduce_impl(::Base.#identity, ::Base.#scalarmax, ::Array{F...Each line of this display represents a particular spot (line number) in the code. Indentation is used to indicate the nested sequence of function calls, with more-indented lines being deeper in the sequence of calls. In each line, the first \"field\" is the number of backtraces (samples) taken at this line or in any functions executed by this line. The second field is the file name and line number and the third field is the function name. Note that the specific line numbers may change as Julia's code changes; if you want to follow along, it's best to run this example yourself.In this example, we can see that the top level function called is in the file event.jl. This is the function that runs the REPL when you launch Julia. If you examine line 97 of REPL.jl, you'll see this is where the function eval_user_input() is called. This is the function that evaluates what you type at the REPL, and since we're working interactively these functions were invoked when we entered @profile myfunc(). The next line reflects actions taken in the @profile macro.The first line shows that 80 backtraces were taken at line 73 of event.jl, but it's not that this line was \"expensive\" on its own: the third line reveals that all 80 of these backtraces were actually triggered inside its call to eval_user_input, and so on. To find out which operations are actually taking the time, we need to look deeper in the call chain.The first \"important\" line in this output is this one:52 ./REPL[1]:2; myfunc()REPL refers to the fact that we defined myfunc in the REPL, rather than putting it in a file; if we had used a file, this would show the file name. The [1] shows that the function myfunc was the first expression evaluated in this REPL session. Line 2 of myfunc() contains the call to rand, and there were 52 (out of 80) backtraces that occurred at this line. Below that, you can see a call to dsfmt_fill_array_close_open! inside dSFMT.jl.A little further down, you see:28 ./REPL[1]:3; myfunc()Line 3 of myfunc contains the call to maximum, and there were 28 (out of 80) backtraces taken here. Below that, you can see the specific places in base/reduce.jl that carry out the time-consuming operations in the maximum function for this type of input data.Overall, we can tentatively conclude that generating the random numbers is approximately twice as expensive as finding the maximum element. We could increase our confidence in this result by collecting more samples:julia> @profile (for i = 1:100; myfunc(); end)\n\njulia> Profile.print()\n[....]\n 3821 ./REPL[1]:2; myfunc()\n  3511 ./random.jl:431; rand!(::MersenneTwister, ::Array{Float64,3}, ::Int64, ::Type...\n   3511 ./dSFMT.jl:84; dsfmt_fill_array_close_open!(::Base.dSFMT.DSFMT_state, ::Ptr...\n  310  ./random.jl:278; rand\n   [....]\n 2893 ./REPL[1]:3; myfunc()\n  2893 ./reduce.jl:270; _mapreduce(::Base.#identity, ::Base.#scalarmax, ::IndexLinea...\n   [....]In general, if you have N samples collected at a line, you can expect an uncertainty on the order of sqrt(N) (barring other sources of noise, like how busy the computer is with other tasks). The major exception to this rule is garbage collection, which runs infrequently but tends to be quite expensive. (Since Julia's garbage collector is written in C, such events can be detected using the C=true output mode described below, or by using ProfileView.jl.)This illustrates the default \"tree\" dump; an alternative is the \"flat\" dump, which accumulates counts independent of their nesting:julia> Profile.print(format=:flat)\n Count File          Line Function\n  6714 ./<missing>     -1 anonymous\n  6714 ./REPL.jl       66 eval_user_input(::Any, ::Base.REPL.REPLBackend)\n  6714 ./REPL.jl       97 macro expansion\n  3821 ./REPL[1]        2 myfunc()\n  2893 ./REPL[1]        3 myfunc()\n  6714 ./REPL[7]        1 macro expansion\n  6714 ./boot.jl      235 eval(::Module, ::Any)\n  3511 ./dSFMT.jl      84 dsfmt_fill_array_close_open!(::Base.dSFMT.DSFMT_s...\n  6714 ./event.jl      73 (::Base.REPL.##1#2{Base.REPL.REPLBackend})()\n  6714 ./profile.jl    23 macro expansion\n  3511 ./random.jl    431 rand!(::MersenneTwister, ::Array{Float64,3}, ::In...\n   310 ./random.jl    277 rand\n   310 ./random.jl    278 rand\n   310 ./random.jl    366 rand\n   310 ./random.jl    369 rand\n  2893 ./reduce.jl    270 _mapreduce(::Base.#identity, ::Base.#scalarmax, :...\n     5 ./reduce.jl    420 mapreduce_impl(::Base.#identity, ::Base.#scalarma...\n   253 ./reduce.jl    426 mapreduce_impl(::Base.#identity, ::Base.#scalarma...\n  2592 ./reduce.jl    428 mapreduce_impl(::Base.#identity, ::Base.#scalarma...\n    43 ./reduce.jl    429 mapreduce_impl(::Base.#identity, ::Base.#scalarma...If your code has recursion, one potentially-confusing point is that a line in a \"child\" function can accumulate more counts than there are total backtraces. Consider the following function definitions:dumbsum(n::Integer) = n == 1 ? 1 : 1 + dumbsum(n-1)\ndumbsum3() = dumbsum(3)If you were to profile dumbsum3, and a backtrace was taken while it was executing dumbsum(1), the backtrace would look like this:dumbsum3\n    dumbsum(3)\n        dumbsum(2)\n            dumbsum(1)Consequently, this child function gets 3 counts, even though the parent only gets one. The \"tree\" representation makes this much clearer, and for this reason (among others) is probably the most useful way to view the results."
},

{
    "location": "manual/profile.html#Accumulation-and-clearing-1",
    "page": "Profiling",
    "title": "Accumulation and clearing",
    "category": "section",
    "text": "Results from @profile accumulate in a buffer; if you run multiple pieces of code under @profile, then Profile.print() will show you the combined results. This can be very useful, but sometimes you want to start fresh; you can do so with Profile.clear()."
},

{
    "location": "manual/profile.html#Options-for-controlling-the-display-of-profile-results-1",
    "page": "Profiling",
    "title": "Options for controlling the display of profile results",
    "category": "section",
    "text": "Profile.print() has more options than we've described so far. Let's see the full declaration:function print(io::IO = STDOUT, data = fetch(); kwargs...)Let's first discuss the two positional arguments, and later the keyword arguments:io – Allows you to save the results to a buffer, e.g. a file, but the default is to print to STDOUT (the console).\ndata – Contains the data you want to analyze; by default that is obtained from Profile.fetch(), which pulls out the backtraces from a pre-allocated buffer. For example, if you want to profile the profiler, you could say:\ndata = copy(Profile.fetch())\nProfile.clear()\n@profile Profile.print(STDOUT, data) # Prints the previous results\nProfile.print()                      # Prints results from Profile.print()The keyword arguments can be any combination of:format – Introduced above, determines whether backtraces are printed  with (default, :tree) or without (:flat) indentation indicating tree  structure.\nC – If true, backtraces from C and Fortran code are shown (normally they are excluded). Try running the introductory example with Profile.print(C = true). This can be extremely helpful in deciding whether it's Julia code or C code that is causing a bottleneck; setting C = true also improves the interpretability of the nesting, at the cost of longer profile dumps.\ncombine – Some lines of code contain multiple operations; for example, s += A[i] contains both an array reference (A[i]) and a sum operation. These correspond to different lines in the generated machine code, and hence there may be two or more different addresses captured during backtraces on this line. combine = true lumps them together, and is probably what you typically want, but you can generate an output separately for each unique instruction pointer with combine = false.\nmaxdepth – Limits frames at a depth higher than maxdepth in the :tree format.\nsortedby – Controls the order in :flat format. :filefuncline (default) sorts by the source line, whereas :count sorts in order of number of collected samples.\nnoisefloor – Limits frames that are below the heuristic noise floor of the sample (only applies to format :tree). A suggested value to try for this is 2.0 (the default is 0). This parameter hides samples for which n <= noisefloor * √N, where n is the number of samples on this line, and N is the number of samples for the callee.\nmincount – Limits frames with less than mincount occurrences.File/function names are sometimes truncated (with ...), and indentation is truncated with a +n at the beginning, where n is the number of extra spaces that would have been inserted, had there been room. If you want a complete profile of deeply-nested code, often a good idea is to save to a file using a wide displaysize in an IOContext:open(\"/tmp/prof.txt\", \"w\") do s\n    Profile.print(IOContext(s, :displaysize => (24, 500)))\nend"
},

{
    "location": "manual/profile.html#Configuration-1",
    "page": "Profiling",
    "title": "Configuration",
    "category": "section",
    "text": "@profile just accumulates backtraces, and the analysis happens when you call Profile.print(). For a long-running computation, it's entirely possible that the pre-allocated buffer for storing backtraces will be filled. If that happens, the backtraces stop but your computation continues. As a consequence, you may miss some important profiling data (you will get a warning when that happens).You can obtain and configure the relevant parameters this way:Profile.init() # returns the current settings\nProfile.init(n = 10^7, delay = 0.01)n is the total number of instruction pointers you can store, with a default value of 10^6. If your typical backtrace is 20 instruction pointers, then you can collect 50000 backtraces, which suggests a statistical uncertainty of less than 1%. This may be good enough for most applications.Consequently, you are more likely to need to modify delay, expressed in seconds, which sets the amount of time that Julia gets between snapshots to perform the requested computations. A very long-running job might not need frequent backtraces. The default setting is delay = 0.001. Of course, you can decrease the delay as well as increase it; however, the overhead of profiling grows once the delay becomes similar to the amount of time needed to take a backtrace (~30 microseconds on the author's laptop)."
},

{
    "location": "manual/profile.html#Memory-allocation-analysis-1",
    "page": "Profiling",
    "title": "Memory allocation analysis",
    "category": "section",
    "text": "One of the most common techniques to improve performance is to reduce memory allocation. The total amount of allocation can be measured with @time and @allocated, and specific lines triggering allocation can often be inferred from profiling via the cost of garbage collection that these lines incur. However, sometimes it is more efficient to directly measure the amount of memory allocated by each line of code.To measure allocation line-by-line, start Julia with the --track-allocation=<setting> command-line option, for which you can choose none (the default, do not measure allocation), user (measure memory allocation everywhere except Julia's core code), or all (measure memory allocation at each line of Julia code). Allocation gets measured for each line of compiled code. When you quit Julia, the cumulative results are written to text files with .mem appended after the file name, residing in the same directory as the source file. Each line lists the total number of bytes allocated. The Coverage package contains some elementary analysis tools, for example to sort the lines in order of number of bytes allocated.In interpreting the results, there are a few important details. Under the user setting, the first line of any function directly called from the REPL will exhibit allocation due to events that happen in the REPL code itself. More significantly, JIT-compilation also adds to allocation counts, because much of Julia's compiler is written in Julia (and compilation usually requires memory allocation). The recommended procedure is to force compilation by executing all the commands you want to analyze, then call Profile.clear_malloc_data() to reset all allocation counters.  Finally, execute the desired commands and quit Julia to trigger the generation of the .mem files."
},

{
    "location": "manual/stacktraces.html#",
    "page": "Stack Traces",
    "title": "Stack Traces",
    "category": "page",
    "text": ""
},

{
    "location": "manual/stacktraces.html#Stack-Traces-1",
    "page": "Stack Traces",
    "title": "Stack Traces",
    "category": "section",
    "text": "The StackTraces module provides simple stack traces that are both human readable and easy to use programmatically."
},

{
    "location": "manual/stacktraces.html#Viewing-a-stack-trace-1",
    "page": "Stack Traces",
    "title": "Viewing a stack trace",
    "category": "section",
    "text": "The primary function used to obtain a stack trace is stacktrace():julia> stacktrace()\n4-element Array{StackFrame,1}:\n eval(::Module, ::Any) at boot.jl:236\n eval_user_input(::Any, ::Base.REPL.REPLBackend) at REPL.jl:66\n macro expansion at REPL.jl:97 [inlined]\n (::Base.REPL.##1#2{Base.REPL.REPLBackend})() at event.jl:73Calling stacktrace() returns a vector of StackFrame s. For ease of use, the alias StackTrace can be used in place of Vector{StackFrame}. (Examples with [...] indicate that output may vary depending on how the code is run.)julia> example() = stacktrace()\nexample (generic function with 1 method)\n\njulia> example()\n5-element Array{StackFrame,1}:\n example() at REPL[1]:1\n eval(::Module, ::Any) at boot.jl:236\n[...]\n\njulia> @noinline child() = stacktrace()\nchild (generic function with 1 method)\n\njulia> @noinline parent() = child()\nparent (generic function with 1 method)\n\njulia> grandparent() = parent()\ngrandparent (generic function with 1 method)\n\njulia> grandparent()\n7-element Array{StackFrame,1}:\n child() at REPL[3]:1\n parent() at REPL[4]:1\n grandparent() at REPL[5]:1\n[...]Note that when calling stacktrace() you'll typically see a frame with eval(...) at boot.jl. When calling stacktrace() from the REPL you'll also have a few extra frames in the stack from REPL.jl, usually looking something like this:julia> example() = stacktrace()\nexample (generic function with 1 method)\n\njulia> example()\n5-element Array{StackFrame,1}:\n example() at REPL[1]:1\n eval(::Module, ::Any) at boot.jl:236\n eval_user_input(::Any, ::Base.REPL.REPLBackend) at REPL.jl:66\n macro expansion at REPL.jl:97 [inlined]\n (::Base.REPL.##1#2{Base.REPL.REPLBackend})() at event.jl:73"
},

{
    "location": "manual/stacktraces.html#Extracting-useful-information-1",
    "page": "Stack Traces",
    "title": "Extracting useful information",
    "category": "section",
    "text": "Each StackFrame contains the function name, file name, line number, lambda info, a flag indicating whether the frame has been inlined, a flag indicating whether it is a C function (by default C functions do not appear in the stack trace), and an integer representation of the pointer returned by backtrace():julia> top_frame = stacktrace()[1]\neval(::Module, ::Any) at boot.jl:236\n\njulia> top_frame.func\n:eval\n\njulia> top_frame.file\nSymbol(\"./boot.jl\")\n\njulia> top_frame.line\n236\n\njulia> top_frame.linfo\nNullable{Core.MethodInstance}(MethodInstance for eval(::Module, ::Any))\n\njulia> top_frame.inlined\nfalse\n\njulia> top_frame.from_c\nfalsejulia> top_frame.pointer\n0x00007f390d152a59This makes stack trace information available programmatically for logging, error handling, and more."
},

{
    "location": "manual/stacktraces.html#Error-handling-1",
    "page": "Stack Traces",
    "title": "Error handling",
    "category": "section",
    "text": "While having easy access to information about the current state of the callstack can be helpful in many places, the most obvious application is in error handling and debugging.julia> @noinline bad_function() = undeclared_variable\nbad_function (generic function with 1 method)\n\njulia> @noinline example() = try\n           bad_function()\n       catch\n           stacktrace()\n       end\nexample (generic function with 1 method)\n\njulia> example()\n5-element Array{StackFrame,1}:\n example() at REPL[2]:4\n eval(::Module, ::Any) at boot.jl:236\n[...]You may notice that in the example above the first stack frame points points at line 4, where stacktrace() is called, rather than line 2, where bad_function is called, and bad_function's frame is missing entirely. This is understandable, given that stacktrace() is called from the context of the catch. While in this example it's fairly easy to find the actual source of the error, in complex cases tracking down the source of the error becomes nontrivial.This can be remedied by calling catch_stacktrace() instead of stacktrace(). Instead of returning callstack information for the current context, catch_stacktrace() returns stack information for the context of the most recent exception:julia> @noinline bad_function() = undeclared_variable\nbad_function (generic function with 1 method)\n\njulia> @noinline example() = try\n           bad_function()\n       catch\n           catch_stacktrace()\n       end\nexample (generic function with 1 method)\n\njulia> example()\n6-element Array{StackFrame,1}:\n bad_function() at REPL[1]:1\n example() at REPL[2]:2\n[...]Notice that the stack trace now indicates the appropriate line number and the missing frame.julia> @noinline child() = error(\"Whoops!\")\nchild (generic function with 1 method)\n\njulia> @noinline parent() = child()\nparent (generic function with 1 method)\n\njulia> @noinline function grandparent()\n           try\n               parent()\n           catch err\n               println(\"ERROR: \", err.msg)\n               catch_stacktrace()\n           end\n       end\ngrandparent (generic function with 1 method)\n\njulia> grandparent()\nERROR: Whoops!\n7-element Array{StackFrame,1}:\n child() at REPL[1]:1\n parent() at REPL[2]:1\n grandparent() at REPL[3]:3\n[...]"
},

{
    "location": "manual/stacktraces.html#Comparison-with-[backtrace()](@ref)-1",
    "page": "Stack Traces",
    "title": "Comparison with backtrace()",
    "category": "section",
    "text": "A call to backtrace() returns a vector of Ptr{Void}, which may then be passed into stacktrace() for translation:julia> trace = backtrace()\n21-element Array{Ptr{Void},1}:\n Ptr{Void} @0x00007f10049d5b2f\n Ptr{Void} @0x00007f0ffeb4d29c\n Ptr{Void} @0x00007f0ffeb4d2a9\n Ptr{Void} @0x00007f1004993fe7\n Ptr{Void} @0x00007f10049a92be\n Ptr{Void} @0x00007f10049a823a\n Ptr{Void} @0x00007f10049a9fb0\n Ptr{Void} @0x00007f10049aa718\n Ptr{Void} @0x00007f10049c0d5e\n Ptr{Void} @0x00007f10049a3286\n Ptr{Void} @0x00007f0ffe9ba3ba\n Ptr{Void} @0x00007f0ffe9ba3d0\n Ptr{Void} @0x00007f1004993fe7\n Ptr{Void} @0x00007f0ded34583d\n Ptr{Void} @0x00007f0ded345a87\n Ptr{Void} @0x00007f1004993fe7\n Ptr{Void} @0x00007f0ded34308f\n Ptr{Void} @0x00007f0ded343320\n Ptr{Void} @0x00007f1004993fe7\n Ptr{Void} @0x00007f10049aeb67\n Ptr{Void} @0x0000000000000000\n\njulia> stacktrace(trace)\n5-element Array{StackFrame,1}:\n backtrace() at error.jl:46\n eval(::Module, ::Any) at boot.jl:236\n eval_user_input(::Any, ::Base.REPL.REPLBackend) at REPL.jl:66\n macro expansion at REPL.jl:97 [inlined]\n (::Base.REPL.##1#2{Base.REPL.REPLBackend})() at event.jl:73Notice that the vector returned by backtrace() had 21 pointers, while the vector returned by stacktrace() only has 5. This is because, by default, stacktrace() removes any lower-level C functions from the stack. If you want to include stack frames from C calls, you can do it like this:julia> stacktrace(trace, true)\n27-element Array{StackFrame,1}:\n jl_backtrace_from_here at stackwalk.c:103\n backtrace() at error.jl:46\n backtrace() at sys.so:?\n jl_call_method_internal at julia_internal.h:248 [inlined]\n jl_apply_generic at gf.c:2215\n do_call at interpreter.c:75\n eval at interpreter.c:215\n eval_body at interpreter.c:519\n jl_interpret_toplevel_thunk at interpreter.c:664\n jl_toplevel_eval_flex at toplevel.c:592\n jl_toplevel_eval_in at builtins.c:614\n eval(::Module, ::Any) at boot.jl:236\n eval(::Module, ::Any) at sys.so:?\n jl_call_method_internal at julia_internal.h:248 [inlined]\n jl_apply_generic at gf.c:2215\n eval_user_input(::Any, ::Base.REPL.REPLBackend) at REPL.jl:66\n ip:0x7f1c707f1846\n jl_call_method_internal at julia_internal.h:248 [inlined]\n jl_apply_generic at gf.c:2215\n macro expansion at REPL.jl:97 [inlined]\n (::Base.REPL.##1#2{Base.REPL.REPLBackend})() at event.jl:73\n ip:0x7f1c707ea1ef\n jl_call_method_internal at julia_internal.h:248 [inlined]\n jl_apply_generic at gf.c:2215\n jl_apply at julia.h:1411 [inlined]\n start_task at task.c:261\n ip:0xffffffffffffffffIndividual pointers returned by backtrace() can be translated into StackFrame s by passing them into StackTraces.lookup():julia> pointer = backtrace()[1];\n\njulia> frame = StackTraces.lookup(pointer)\n1-element Array{StackFrame,1}:\n jl_backtrace_from_here at stackwalk.c:103\n\njulia> println(\"The top frame is from $(frame[1].func)!\")\nThe top frame is from jl_backtrace_from_here!"
},

{
    "location": "manual/performance-tips.html#",
    "page": "Performance Tips",
    "title": "Performance Tips",
    "category": "page",
    "text": ""
},

{
    "location": "manual/performance-tips.html#man-performance-tips-1",
    "page": "Performance Tips",
    "title": "Performance Tips",
    "category": "section",
    "text": "In the following sections, we briefly go through a few techniques that can help make your Julia code run as fast as possible."
},

{
    "location": "manual/performance-tips.html#Avoid-global-variables-1",
    "page": "Performance Tips",
    "title": "Avoid global variables",
    "category": "section",
    "text": "A global variable might have its value, and therefore its type, change at any point. This makes it difficult for the compiler to optimize code using global variables. Variables should be local, or passed as arguments to functions, whenever possible.Any code that is performance critical or being benchmarked should be inside a function.We find that global names are frequently constants, and declaring them as such greatly improves performance:const DEFAULT_VAL = 0Uses of non-constant globals can be optimized by annotating their types at the point of use:global x\ny = f(x::Int + 1)Writing functions is better style. It leads to more reusable code and clarifies what steps are being done, and what their inputs and outputs are.note: Note\nAll code in the REPL is evaluated in global scope, so a variable defined and assigned at toplevel will be a global variable.In the following REPL session:julia> x = 1.0is equivalent to:julia> global x = 1.0so all the performance issues discussed previously apply."
},

{
    "location": "manual/performance-tips.html#Measure-performance-with-[@time](@ref)-and-pay-attention-to-memory-allocation-1",
    "page": "Performance Tips",
    "title": "Measure performance with @time and pay attention to memory allocation",
    "category": "section",
    "text": "A useful tool for measuring performance is the @time macro. The following example illustrates good working style:julia> function f(n)\n           s = 0\n           for i = 1:n\n               s += i/2\n           end\n           s\n       end\nf (generic function with 1 method)\n\njulia> @time f(1)\n  0.012686 seconds (2.09 k allocations: 103.421 KiB)\n0.5\n\njulia> @time f(10^6)\n  0.021061 seconds (3.00 M allocations: 45.777 MiB, 11.69% gc time)\n2.5000025e11On the first call (@time f(1)), f gets compiled.  (If you've not yet used @time in this session, it will also compile functions needed for timing.)  You should not take the results of this run seriously. For the second run, note that in addition to reporting the time, it also indicated that a large amount of memory was allocated. This is the single biggest advantage of @time vs. functions like tic() and toc(), which only report time.Unexpected memory allocation is almost always a sign of some problem with your code, usually a problem with type-stability. Consequently, in addition to the allocation itself, it's very likely that the code generated for your function is far from optimal. Take such indications seriously and follow the advice below.For more serious benchmarking, consider the BenchmarkTools.jl package which evaluates the function multiple times in order to reduce noise.As a teaser, an improved version of this function allocates no memory (the allocation reported below is due to running the @time macro in global scope) and has an order of magnitude faster execution after the first call:julia> @time f_improved(1)\n  0.007008 seconds (1.32 k allocations: 63.640 KiB)\n0.5\n\njulia> @time f_improved(10^6)\n  0.002997 seconds (6 allocations: 192 bytes)\n2.5000025e11Below you'll learn how to spot the problem with f and how to fix it.In some situations, your function may need to allocate memory as part of its operation, and this can complicate the simple picture above. In such cases, consider using one of the tools below to diagnose problems, or write a version of your function that separates allocation from its algorithmic aspects (see Pre-allocating outputs)."
},

{
    "location": "manual/performance-tips.html#tools-1",
    "page": "Performance Tips",
    "title": "Tools",
    "category": "section",
    "text": "Julia and its package ecosystem includes tools that may help you diagnose problems and improve the performance of your code:Profiling allows you to measure the performance of your running code and identify lines that serve as bottlenecks.  For complex projects, the ProfileView package can help you visualize your profiling results.\nUnexpectedly-large memory allocations–as reported by @time, @allocated, or the profiler (through calls to the garbage-collection routines)–hint that there might be issues with your code.  If you don't see another reason for the allocations, suspect a type problem.  You can also start Julia with the --track-allocation=user option and examine the resulting *.mem files to see information about where those allocations occur.  See Memory allocation analysis.\n@code_warntype generates a representation of your code that can be helpful in finding expressions that result in type uncertainty. See @code_warntype below.\nThe Lint package can also warn you of certain types of programming errors."
},

{
    "location": "manual/performance-tips.html#Avoid-containers-with-abstract-type-parameters-1",
    "page": "Performance Tips",
    "title": "Avoid containers with abstract type parameters",
    "category": "section",
    "text": "When working with parameterized types, including arrays, it is best to avoid parameterizing with abstract types where possible.Consider the following:a = Real[]    # typeof(a) = Array{Real,1}\nif (f = rand()) < .8\n    push!(a, f)\nendBecause a is a an array of abstract type Real, it must be able to hold any Real value.  Since Real objects can be of arbitrary size and structure, a must be represented as an array of pointers to individually allocated Real objects. Because f will always be a Float64, we should instead, use:a = Float64[] # typeof(a) = Array{Float64,1}which will create a contiguous block of 64-bit floating-point values that can be manipulated efficiently.See also the discussion under Parametric Types."
},

{
    "location": "manual/performance-tips.html#Type-declarations-1",
    "page": "Performance Tips",
    "title": "Type declarations",
    "category": "section",
    "text": "In many languages with optional type declarations, adding declarations is the principal way to make code run faster. This is not the case in Julia. In Julia, the compiler generally knows the types of all function arguments, local variables, and expressions. However, there are a few specific instances where declarations are helpful."
},

{
    "location": "manual/performance-tips.html#Avoid-fields-with-abstract-type-1",
    "page": "Performance Tips",
    "title": "Avoid fields with abstract type",
    "category": "section",
    "text": "Types can be declared without specifying the types of their fields:julia> struct MyAmbiguousType\n           a\n       endThis allows a to be of any type. This can often be useful, but it does have a downside: for objects of type MyAmbiguousType, the compiler will not be able to generate high-performance code.  The reason is that the compiler uses the types of objects, not their values, to determine how to build code. Unfortunately, very little can be inferred about an object of type MyAmbiguousType:julia> b = MyAmbiguousType(\"Hello\")\nMyAmbiguousType(\"Hello\")\n\njulia> c = MyAmbiguousType(17)\nMyAmbiguousType(17)\n\njulia> typeof(b)\nMyAmbiguousType\n\njulia> typeof(c)\nMyAmbiguousTypeb and c have the same type, yet their underlying representation of data in memory is very different. Even if you stored just numeric values in field a, the fact that the memory representation of a UInt8 differs from a Float64 also means that the CPU needs to handle them using two different kinds of instructions. Since the required information is not available in the type, such decisions have to be made at run-time. This slows performance.You can do better by declaring the type of a. Here, we are focused on the case where a might be any one of several types, in which case the natural solution is to use parameters. For example:julia> mutable struct MyType{T<:AbstractFloat}\n           a::T\n       endThis is a better choice thanjulia> mutable struct MyStillAmbiguousType\n           a::AbstractFloat\n       endbecause the first version specifies the type of a from the type of the wrapper object.  For example:julia> m = MyType(3.2)\nMyType{Float64}(3.2)\n\njulia> t = MyStillAmbiguousType(3.2)\nMyStillAmbiguousType(3.2)\n\njulia> typeof(m)\nMyType{Float64}\n\njulia> typeof(t)\nMyStillAmbiguousTypeThe type of field a can be readily determined from the type of m, but not from the type of t.  Indeed, in t it's possible to change the type of field a:julia> typeof(t.a)\nFloat64\n\njulia> t.a = 4.5f0\n4.5f0\n\njulia> typeof(t.a)\nFloat32In contrast, once m is constructed, the type of m.a cannot change:julia> m.a = 4.5f0\n4.5f0\n\njulia> typeof(m.a)\nFloat64The fact that the type of m.a is known from m's type–coupled with the fact that its type cannot change mid-function–allows the compiler to generate highly-optimized code for objects like m but not for objects like t.Of course, all of this is true only if we construct m with a concrete type.  We can break this by explicitly constructing it with an abstract type:julia> m = MyType{AbstractFloat}(3.2)\nMyType{AbstractFloat}(3.2)\n\njulia> typeof(m.a)\nFloat64\n\njulia> m.a = 4.5f0\n4.5f0\n\njulia> typeof(m.a)\nFloat32For all practical purposes, such objects behave identically to those of MyStillAmbiguousType.It's quite instructive to compare the sheer amount code generated for a simple functionfunc(m::MyType) = m.a+1usingcode_llvm(func,Tuple{MyType{Float64}})\ncode_llvm(func,Tuple{MyType{AbstractFloat}})\ncode_llvm(func,Tuple{MyType})For reasons of length the results are not shown here, but you may wish to try this yourself. Because the type is fully-specified in the first case, the compiler doesn't need to generate any code to resolve the type at run-time. This results in shorter and faster code."
},

{
    "location": "manual/performance-tips.html#Avoid-fields-with-abstract-containers-1",
    "page": "Performance Tips",
    "title": "Avoid fields with abstract containers",
    "category": "section",
    "text": "The same best practices also work for container types:julia> mutable struct MySimpleContainer{A<:AbstractVector}\n           a::A\n       end\n\njulia> mutable struct MyAmbiguousContainer{T}\n           a::AbstractVector{T}\n       endFor example:julia> c = MySimpleContainer(1:3);\n\njulia> typeof(c)\nMySimpleContainer{UnitRange{Int64}}\n\njulia> c = MySimpleContainer([1:3;]);\n\njulia> typeof(c)\nMySimpleContainer{Array{Int64,1}}\n\njulia> b = MyAmbiguousContainer(1:3);\n\njulia> typeof(b)\nMyAmbiguousContainer{Int64}\n\njulia> b = MyAmbiguousContainer([1:3;]);\n\njulia> typeof(b)\nMyAmbiguousContainer{Int64}For MySimpleContainer, the object is fully-specified by its type and parameters, so the compiler can generate optimized functions. In most instances, this will probably suffice.While the compiler can now do its job perfectly well, there are cases where you might wish that your code could do different things depending on the element type of a.  Usually the best way to achieve this is to wrap your specific operation (here, foo) in a separate function:julia> function sumfoo(c::MySimpleContainer)\n           s = 0\n           for x in c.a\n               s += foo(x)\n           end\n           s\n       end\nsumfoo (generic function with 1 method)\n\njulia> foo(x::Integer) = x\nfoo (generic function with 1 method)\n\njulia> foo(x::AbstractFloat) = round(x)\nfoo (generic function with 2 methods)This keeps things simple, while allowing the compiler to generate optimized code in all cases.However, there are cases where you may need to declare different versions of the outer function for different element types of a. You could do it like this:function myfun(c::MySimpleContainer{Vector{T}}) where T<:AbstractFloat\n    ...\nend\nfunction myfun(c::MySimpleContainer{Vector{T}}) where T<:Integer\n    ...\nendThis works fine for Vector{T}, but we'd also have to write explicit versions for UnitRange{T} or other abstract types. To prevent such tedium, you can use two parameters in the declaration of MyContainer:julia> mutable struct MyContainer{T, A<:AbstractVector}\n           a::A\n       end\n\njulia> MyContainer(v::AbstractVector) = MyContainer{eltype(v), typeof(v)}(v)\nMyContainer\n\njulia> b = MyContainer(1:5);\n\njulia> typeof(b)\nMyContainer{Int64,UnitRange{Int64}}Note the somewhat surprising fact that T doesn't appear in the declaration of field a, a point that we'll return to in a moment. With this approach, one can write functions such as:julia> function myfunc(c::MyContainer{<:Integer, <:AbstractArray})\n           return c.a[1]+1\n       end\nmyfunc (generic function with 1 method)\n\njulia> function myfunc(c::MyContainer{<:AbstractFloat})\n           return c.a[1]+2\n       end\nmyfunc (generic function with 2 methods)\n\njulia> function myfunc(c::MyContainer{T,Vector{T}}) where T<:Integer\n           return c.a[1]+3\n       end\nmyfunc (generic function with 3 methods)note: Note\nBecause we can only define MyContainer for A<:AbstractArray, and any unspecified parameters are arbitrary, the first function above could have been written more succinctly as function myfunc{T<:Integer}(c::MyContainer{T})julia> myfunc(MyContainer(1:3))\n2\n\njulia> myfunc(MyContainer(1.0:3))\n3.0\n\njulia> myfunc(MyContainer([1:3;]))\n4As you can see, with this approach it's possible to specialize on both the element type T and the array type A.However, there's one remaining hole: we haven't enforced that A has element type T, so it's perfectly possible to construct an object like this:julia> b = MyContainer{Int64, UnitRange{Float64}}(UnitRange(1.3, 5.0));\n\njulia> typeof(b)\nMyContainer{Int64,UnitRange{Float64}}To prevent this, we can add an inner constructor:julia> mutable struct MyBetterContainer{T<:Real, A<:AbstractVector}\n           a::A\n           MyBetterContainer{T,A}(v::AbstractVector{T}) where {T,A} = new(v)\n       end\n\njulia> MyBetterContainer(v::AbstractVector) = MyBetterContainer{eltype(v),typeof(v)}(v)\nMyBetterContainer\n\njulia> b = MyBetterContainer(UnitRange(1.3, 5.0));\n\njulia> typeof(b)\nMyBetterContainer{Float64,UnitRange{Float64}}\n\njulia> b = MyBetterContainer{Int64, UnitRange{Float64}}(UnitRange(1.3, 5.0));\nERROR: MethodError: Cannot `convert` an object of type UnitRange{Float64} to an object of type MyBetterContainer{Int64,UnitRange{Float64}}\n[...]The inner constructor requires that the element type of A be T."
},

{
    "location": "manual/performance-tips.html#Annotate-values-taken-from-untyped-locations-1",
    "page": "Performance Tips",
    "title": "Annotate values taken from untyped locations",
    "category": "section",
    "text": "It is often convenient to work with data structures that may contain values of any type (arrays of type Array{Any}). But, if you're using one of these structures and happen to know the type of an element, it helps to share this knowledge with the compiler:function foo(a::Array{Any,1})\n    x = a[1]::Int32\n    b = x+1\n    ...\nendHere, we happened to know that the first element of a would be an Int32. Making an annotation like this has the added benefit that it will raise a run-time error if the value is not of the expected type, potentially catching certain bugs earlier."
},

{
    "location": "manual/performance-tips.html#Declare-types-of-keyword-arguments-1",
    "page": "Performance Tips",
    "title": "Declare types of keyword arguments",
    "category": "section",
    "text": "Keyword arguments can have declared types:function with_keyword(x; name::Int = 1)\n    ...\nendFunctions are specialized on the types of keyword arguments, so these declarations will not affect performance of code inside the function. However, they will reduce the overhead of calls to the function that include keyword arguments.Functions with keyword arguments have near-zero overhead for call sites that pass only positional arguments.Passing dynamic lists of keyword arguments, as in f(x; keywords...), can be slow and should be avoided in performance-sensitive code."
},

{
    "location": "manual/performance-tips.html#Break-functions-into-multiple-definitions-1",
    "page": "Performance Tips",
    "title": "Break functions into multiple definitions",
    "category": "section",
    "text": "Writing a function as many small definitions allows the compiler to directly call the most applicable code, or even inline it.Here is an example of a \"compound function\" that should really be written as multiple definitions:function norm(A)\n    if isa(A, Vector)\n        return sqrt(real(dot(A,A)))\n    elseif isa(A, Matrix)\n        return maximum(svd(A)[2])\n    else\n        error(\"norm: invalid argument\")\n    end\nendThis can be written more concisely and efficiently as:norm(x::Vector) = sqrt(real(dot(x,x)))\nnorm(A::Matrix) = maximum(svd(A)[2])"
},

{
    "location": "manual/performance-tips.html#Write-\"type-stable\"-functions-1",
    "page": "Performance Tips",
    "title": "Write \"type-stable\" functions",
    "category": "section",
    "text": "When possible, it helps to ensure that a function always returns a value of the same type. Consider the following definition:pos(x) = x < 0 ? 0 : xAlthough this seems innocent enough, the problem is that 0 is an integer (of type Int) and x might be of any type. Thus, depending on the value of x, this function might return a value of either of two types. This behavior is allowed, and may be desirable in some cases. But it can easily be fixed as follows:pos(x) = x < 0 ? zero(x) : xThere is also a one() function, and a more general oftype(x, y) function, which returns y converted to the type of x."
},

{
    "location": "manual/performance-tips.html#Avoid-changing-the-type-of-a-variable-1",
    "page": "Performance Tips",
    "title": "Avoid changing the type of a variable",
    "category": "section",
    "text": "An analogous \"type-stability\" problem exists for variables used repeatedly within a function:function foo()\n    x = 1\n    for i = 1:10\n        x = x/bar()\n    end\n    return x\nendLocal variable x starts as an integer, and after one loop iteration becomes a floating-point number (the result of / operator). This makes it more difficult for the compiler to optimize the body of the loop. There are several possible fixes:Initialize x with x = 1.0\nDeclare the type of x: x::Float64 = 1\nUse an explicit conversion: x = oneunit(T)\nInitialize with the first loop iteration, to x = 1/bar(), then loop for i = 2:10"
},

{
    "location": "manual/performance-tips.html#kernal-functions-1",
    "page": "Performance Tips",
    "title": "Separate kernel functions (aka, function barriers)",
    "category": "section",
    "text": "Many functions follow a pattern of performing some set-up work, and then running many iterations to perform a core computation. Where possible, it is a good idea to put these core computations in separate functions. For example, the following contrived function returns an array of a randomly-chosen type:DocTestSetup = quote\n    srand(1234)\nendjulia> function strange_twos(n)\n           a = Vector{rand(Bool) ? Int64 : Float64}(n)\n           for i = 1:n\n               a[i] = 2\n           end\n           return a\n       end\nstrange_twos (generic function with 1 method)\n\njulia> strange_twos(3)\n3-element Array{Float64,1}:\n 2.0\n 2.0\n 2.0This should be written as:julia> function fill_twos!(a)\n           for i=1:length(a)\n               a[i] = 2\n           end\n       end\nfill_twos! (generic function with 1 method)\n\njulia> function strange_twos(n)\n           a = Array{rand(Bool) ? Int64 : Float64}(n)\n           fill_twos!(a)\n           return a\n       end\nstrange_twos (generic function with 1 method)\n\njulia> strange_twos(3)\n3-element Array{Float64,1}:\n 2.0\n 2.0\n 2.0Julia's compiler specializes code for argument types at function boundaries, so in the original implementation it does not know the type of a during the loop (since it is chosen randomly). Therefore the second version is generally faster since the inner loop can be recompiled as part of fill_twos! for different types of a.The second form is also often better style and can lead to more code reuse.This pattern is used in several places in the standard library. For example, see hvcat_fill in abstractarray.jl, or the fill! function, which we could have used instead of writing our own fill_twos!.Functions like strange_twos occur when dealing with data of uncertain type, for example data loaded from an input file that might contain either integers, floats, strings, or something else."
},

{
    "location": "manual/performance-tips.html#Types-with-values-as-parameters-1",
    "page": "Performance Tips",
    "title": "Types with values-as-parameters",
    "category": "section",
    "text": "Let's say you want to create an N-dimensional array that has size 3 along each axis.  Such arrays can be created like this:julia> A = fill(5.0, (3, 3))\n3×3 Array{Float64,2}:\n 5.0  5.0  5.0\n 5.0  5.0  5.0\n 5.0  5.0  5.0This approach works very well: the compiler can figure out that A is an Array{Float64,2} because it knows the type of the fill value (5.0::Float64) and the dimensionality ((3, 3)::NTuple{2,Int}). This implies that the compiler can generate very efficient code for any future usage of A in the same function.But now let's say you want to write a function that creates a 3×3×... array in arbitrary dimensions; you might be tempted to write a functionjulia> function array3(fillval, N)\n           fill(fillval, ntuple(d->3, N))\n       end\narray3 (generic function with 1 method)\n\njulia> array3(5.0, 2)\n3×3 Array{Float64,2}:\n 5.0  5.0  5.0\n 5.0  5.0  5.0\n 5.0  5.0  5.0This works, but (as you can verify for yourself using @code_warntype array3(5.0, 2)) the problem is that the output type cannot be inferred: the argument N is a value of type Int, and type-inference does not (and cannot) predict its value in advance. This means that code using the output of this function has to be conservative, checking the type on each access of A; such code will be very slow.Now, one very good way to solve such problems is by using the function-barrier technique. However, in some cases you might want to eliminate the type-instability altogether.  In such cases, one approach is to pass the dimensionality as a parameter, for example through Val{T} (see \"Value types\"):julia> function array3(fillval, ::Type{Val{N}}) where N\n           fill(fillval, ntuple(d->3, Val{N}))\n       end\narray3 (generic function with 1 method)\n\njulia> array3(5.0, Val{2})\n3×3 Array{Float64,2}:\n 5.0  5.0  5.0\n 5.0  5.0  5.0\n 5.0  5.0  5.0Julia has a specialized version of ntuple that accepts a Val{::Int} as the second parameter; by passing N as a type-parameter, you make its \"value\" known to the compiler. Consequently, this version of array3 allows the compiler to predict the return type.However, making use of such techniques can be surprisingly subtle. For example, it would be of no help if you called array3 from a function like this:function call_array3(fillval, n)\n    A = array3(fillval, Val{n})\nendHere, you've created the same problem all over again: the compiler can't guess the type of n, so it doesn't know the type of Val{n}.  Attempting to use Val, but doing so incorrectly, can easily make performance worse in many situations.  (Only in situations where you're effectively combining Val with the function-barrier trick, to make the kernel function more efficient, should code like the above be used.)An example of correct usage of Val would be:function filter3(A::AbstractArray{T,N}) where {T,N}\n    kernel = array3(1, Val{N})\n    filter(A, kernel)\nendIn this example, N is passed as a parameter, so its \"value\" is known to the compiler.  Essentially, Val{T} works only when T is either hard-coded (Val{3}) or already specified in the type-domain."
},

{
    "location": "manual/performance-tips.html#The-dangers-of-abusing-multiple-dispatch-(aka,-more-on-types-with-values-as-parameters)-1",
    "page": "Performance Tips",
    "title": "The dangers of abusing multiple dispatch (aka, more on types with values-as-parameters)",
    "category": "section",
    "text": "Once one learns to appreciate multiple dispatch, there's an understandable tendency to go crazy and try to use it for everything. For example, you might imagine using it to store information, e.g.struct Car{Make,Model}\n    year::Int\n    ...more fields...\nendand then dispatch on objects like Car{:Honda,:Accord}(year, args...).This might be worthwhile when the following are true:You require CPU-intensive processing on each Car, and it becomes vastly more efficient if you know the Make and Model at compile time.\nYou have homogenous lists of the same type of Car to process, so that you can store them all in an Array{Car{:Honda,:Accord},N}.When the latter holds, a function processing such a homogenous array can be productively specialized: Julia knows the type of each element in advance (all objects in the container have the same concrete type), so Julia can \"look up\" the correct method calls when the function is being compiled (obviating the need to check at run-time) and thereby emit efficient code for processing the whole list.When these do not hold, then it's likely that you'll get no benefit; worse, the resulting \"combinatorial explosion of types\" will be counterproductive.  If items[i+1] has a different type than item[i], Julia has to look up the type at run-time, search for the appropriate method in method tables, decide (via type intersection) which one matches, determine whether it has been JIT-compiled yet (and do so if not), and then make the call. In essence, you're asking the full type- system and JIT-compilation machinery to basically execute the equivalent of a switch statement or dictionary lookup in your own code.Some run-time benchmarks comparing (1) type dispatch, (2) dictionary lookup, and (3) a \"switch\" statement can be found on the mailing list.Perhaps even worse than the run-time impact is the compile-time impact: Julia will compile specialized functions for each different Car{Make, Model}; if you have hundreds or thousands of such types, then every function that accepts such an object as a parameter (from a custom get_year function you might write yourself, to the generic push! function in the standard library) will have hundreds or thousands of variants compiled for it.  Each of these increases the size of the cache of compiled code, the length of internal lists of methods, etc.  Excess enthusiasm for values-as-parameters can easily waste enormous resources."
},

{
    "location": "manual/performance-tips.html#Access-arrays-in-memory-order,-along-columns-1",
    "page": "Performance Tips",
    "title": "Access arrays in memory order, along columns",
    "category": "section",
    "text": "Multidimensional arrays in Julia are stored in column-major order. This means that arrays are stacked one column at a time. This can be verified using the vec function or the syntax [:] as shown below (notice that the array is ordered [1 3 2 4], not [1 2 3 4]):julia> x = [1 2; 3 4]\n2×2 Array{Int64,2}:\n 1  2\n 3  4\n\njulia> x[:]\n4-element Array{Int64,1}:\n 1\n 3\n 2\n 4This convention for ordering arrays is common in many languages like Fortran, Matlab, and R (to name a few). The alternative to column-major ordering is row-major ordering, which is the convention adopted by C and Python (numpy) among other languages. Remembering the ordering of arrays can have significant performance effects when looping over arrays. A rule of thumb to keep in mind is that with column-major arrays, the first index changes most rapidly. Essentially this means that looping will be faster if the inner-most loop index is the first to appear in a slice expression.Consider the following contrived example. Imagine we wanted to write a function that accepts a Vector and returns a square Matrix with either the rows or the columns filled with copies of the input vector. Assume that it is not important whether rows or columns are filled with these copies (perhaps the rest of the code can be easily adapted accordingly). We could conceivably do this in at least four ways (in addition to the recommended call to the built-in repmat()):function copy_cols(x::Vector{T}) where T\n    n = size(x, 1)\n    out = Array{T}(n, n)\n    for i = 1:n\n        out[:, i] = x\n    end\n    out\nend\n\nfunction copy_rows(x::Vector{T}) where T\n    n = size(x, 1)\n    out = Array{T}(n, n)\n    for i = 1:n\n        out[i, :] = x\n    end\n    out\nend\n\nfunction copy_col_row(x::Vector{T}) where T\n    n = size(x, 1)\n    out = Array{T}(n, n)\n    for col = 1:n, row = 1:n\n        out[row, col] = x[row]\n    end\n    out\nend\n\nfunction copy_row_col(x::Vector{T}) where T\n    n = size(x, 1)\n    out = Array{T}(n, n)\n    for row = 1:n, col = 1:n\n        out[row, col] = x[col]\n    end\n    out\nendNow we will time each of these functions using the same random 10000 by 1 input vector:julia> x = randn(10000);\n\njulia> fmt(f) = println(rpad(string(f)*\": \", 14, ' '), @elapsed f(x))\n\njulia> map(fmt, Any[copy_cols, copy_rows, copy_col_row, copy_row_col]);\ncopy_cols:    0.331706323\ncopy_rows:    1.799009911\ncopy_col_row: 0.415630047\ncopy_row_col: 1.721531501Notice that copy_cols is much faster than copy_rows. This is expected because copy_cols respects the column-based memory layout of the Matrix and fills it one column at a time. Additionally, copy_col_row is much faster than copy_row_col because it follows our rule of thumb that the first element to appear in a slice expression should be coupled with the inner-most loop."
},

{
    "location": "manual/performance-tips.html#Pre-allocating-outputs-1",
    "page": "Performance Tips",
    "title": "Pre-allocating outputs",
    "category": "section",
    "text": "If your function returns an Array or some other complex type, it may have to allocate memory.  Unfortunately, oftentimes allocation and its converse, garbage collection, are substantial bottlenecks.Sometimes you can circumvent the need to allocate memory on each function call by preallocating the output.  As a trivial example, comparefunction xinc(x)\n    return [x, x+1, x+2]\nend\n\nfunction loopinc()\n    y = 0\n    for i = 1:10^7\n        ret = xinc(i)\n        y += ret[2]\n    end\n    y\nendwithfunction xinc!(ret::AbstractVector{T}, x::T) where T\n    ret[1] = x\n    ret[2] = x+1\n    ret[3] = x+2\n    nothing\nend\n\nfunction loopinc_prealloc()\n    ret = Array{Int}(3)\n    y = 0\n    for i = 1:10^7\n        xinc!(ret, i)\n        y += ret[2]\n    end\n    y\nendTiming results:julia> @time loopinc()\n  0.529894 seconds (40.00 M allocations: 1.490 GiB, 12.14% gc time)\n50000015000000\n\njulia> @time loopinc_prealloc()\n  0.030850 seconds (6 allocations: 288 bytes)\n50000015000000Preallocation has other advantages, for example by allowing the caller to control the \"output\" type from an algorithm.  In the example above, we could have passed a SubArray rather than an Array, had we so desired.Taken to its extreme, pre-allocation can make your code uglier, so performance measurements and some judgment may be required. However, for \"vectorized\" (element-wise) functions, the convenient syntax x .= f.(y) can be used for in-place operations with fused loops and no temporary arrays (see the dot syntax for vectorizing functions)."
},

{
    "location": "manual/performance-tips.html#More-dots:-Fuse-vectorized-operations-1",
    "page": "Performance Tips",
    "title": "More dots: Fuse vectorized operations",
    "category": "section",
    "text": "Julia has a special dot syntax that converts any scalar function into a \"vectorized\" function call, and any operator into a \"vectorized\" operator, with the special property that nested \"dot calls\" are fusing: they are combined at the syntax level into a single loop, without allocating temporary arrays. If you use .= and similar assignment operators, the result can also be stored in-place in a pre-allocated array (see above).In a linear-algebra context, this means that even though operations like vector + vector and vector * scalar are defined, it can be advantageous to instead use vector .+ vector and vector .* scalar because the resulting loops can be fused with surrounding computations. For example, consider the two functions:f(x) = 3x.^2 + 4x + 7x.^3\n\nfdot(x) = @. 3x^2 + 4x + 7x^3 # equivalent to 3 .* x.^2 .+ 4 .* x .+ 7 .* x.^3Both f and fdot compute the same thing.  However, fdot (defined with the help of the @. macro) is significantly faster when applied to an array:julia> x = rand(10^6);\n\njulia> @time f(x);\n  0.010986 seconds (18 allocations: 53.406 MiB, 11.45% gc time)\n\njulia> @time fdot(x);\n  0.003470 seconds (6 allocations: 7.630 MiB)\n\njulia> @time f.(x);\n  0.003297 seconds (30 allocations: 7.631 MiB)That is, fdot(x) is three times faster and allocates 1/7 the memory of f(x), because each * and + operation in f(x) allocates a new temporary array and executes in a separate loop. (Of course, if you just do f.(x) then it is as fast as fdot(x) in this example, but in many contexts it is more convenient to just sprinkle some dots in your expressions rather than defining a separate function for each vectorized operation.)"
},

{
    "location": "manual/performance-tips.html#Consider-using-views-for-slices-1",
    "page": "Performance Tips",
    "title": "Consider using views for slices",
    "category": "section",
    "text": "In Julia, an array \"slice\" expression like array[1:5, :] creates a copy of that data (except on the left-hand side of an assignment, where array[1:5, :] = ... assigns in-place to that portion of array). If you are doing many operations on the slice, this can be good for performance because it is more efficient to work with a smaller contiguous copy than it would be to index into the original array. On the other hand, if you are just doing a few simple operations on the slice, the cost of the allocation and copy operations can be substantial.An alternative is to create a \"view\" of the array, which is an array object (a SubArray) that actually references the data of the original array in-place, without making a copy.  (If you write to a view, it modifies the original array's data as well.) This can be done for individual slices by calling view(), or more simply for a whole expression or block of code by putting @views in front of that expression.  For example:julia> fcopy(x) = sum(x[2:end-1])\n\njulia> @views fview(x) = sum(x[2:end-1])\n\njulia> x = rand(10^6);\n\njulia> @time fcopy(x);\n  0.003051 seconds (7 allocations: 7.630 MB)\n\njulia> @time fview(x);\n  0.001020 seconds (6 allocations: 224 bytes)Notice both the 3× speedup and the decreased memory allocation of the fview version of the function."
},

{
    "location": "manual/performance-tips.html#Avoid-string-interpolation-for-I/O-1",
    "page": "Performance Tips",
    "title": "Avoid string interpolation for I/O",
    "category": "section",
    "text": "When writing data to a file (or other I/O device), forming extra intermediate strings is a source of overhead. Instead of:println(file, \"$a $b\")use:println(file, a, \" \", b)The first version of the code forms a string, then writes it to the file, while the second version writes values directly to the file. Also notice that in some cases string interpolation can be harder to read. Consider:println(file, \"$(f(a))$(f(b))\")versus:println(file, f(a), f(b))"
},

{
    "location": "manual/performance-tips.html#Optimize-network-I/O-during-parallel-execution-1",
    "page": "Performance Tips",
    "title": "Optimize network I/O during parallel execution",
    "category": "section",
    "text": "When executing a remote function in parallel:responses = Vector{Any}(nworkers())\n@sync begin\n    for (idx, pid) in enumerate(workers())\n        @async responses[idx] = remotecall_fetch(pid, foo, args...)\n    end\nendis faster than:refs = Vector{Any}(nworkers())\nfor (idx, pid) in enumerate(workers())\n    refs[idx] = @spawnat pid foo(args...)\nend\nresponses = [fetch(r) for r in refs]The former results in a single network round-trip to every worker, while the latter results in two network calls - first by the @spawnat and the second due to the fetch (or even a wait). The fetch/wait is also being executed serially resulting in an overall poorer performance."
},

{
    "location": "manual/performance-tips.html#Fix-deprecation-warnings-1",
    "page": "Performance Tips",
    "title": "Fix deprecation warnings",
    "category": "section",
    "text": "A deprecated function internally performs a lookup in order to print a relevant warning only once. This extra lookup can cause a significant slowdown, so all uses of deprecated functions should be modified as suggested by the warnings."
},

{
    "location": "manual/performance-tips.html#Tweaks-1",
    "page": "Performance Tips",
    "title": "Tweaks",
    "category": "section",
    "text": "These are some minor points that might help in tight inner loops.Avoid unnecessary arrays. For example, instead of sum([x,y,z]) use x+y+z.\nUse abs2(z) instead of abs(z)^2 for complex z. In general, try to rewrite code to use abs2() instead of abs() for complex arguments.\nUse div(x,y) for truncating division of integers instead of trunc(x/y), fld(x,y) instead of floor(x/y), and cld(x,y) instead of ceil(x/y)."
},

{
    "location": "manual/performance-tips.html#Performance-Annotations-1",
    "page": "Performance Tips",
    "title": "Performance Annotations",
    "category": "section",
    "text": "Sometimes you can enable better optimization by promising certain program properties.Use @inbounds to eliminate array bounds checking within expressions. Be certain before doing this. If the subscripts are ever out of bounds, you may suffer crashes or silent corruption.\nUse @fastmath to allow floating point optimizations that are correct for real numbers, but lead to differences for IEEE numbers. Be careful when doing this, as this may change numerical results. This corresponds to the -ffast-math option of clang.\nWrite @simd in front of for loops that are amenable to vectorization. This feature is experimental and could change or disappear in future versions of Julia.Note: While @simd needs to be placed directly in front of a loop, both @inbounds and @fastmath can be applied to several statements at once, e.g. using begin ... end, or even to a whole function.Here is an example with both @inbounds and @simd markup:function inner(x, y)\n    s = zero(eltype(x))\n    for i=1:length(x)\n        @inbounds s += x[i]*y[i]\n    end\n    s\nend\n\nfunction innersimd(x, y)\n    s = zero(eltype(x))\n    @simd for i=1:length(x)\n        @inbounds s += x[i]*y[i]\n    end\n    s\nend\n\nfunction timeit(n, reps)\n    x = rand(Float32,n)\n    y = rand(Float32,n)\n    s = zero(Float64)\n    time = @elapsed for j in 1:reps\n        s+=inner(x,y)\n    end\n    println(\"GFlop/sec        = \",2.0*n*reps/time*1E-9)\n    time = @elapsed for j in 1:reps\n        s+=innersimd(x,y)\n    end\n    println(\"GFlop/sec (SIMD) = \",2.0*n*reps/time*1E-9)\nend\n\ntimeit(1000,1000)On a computer with a 2.4GHz Intel Core i5 processor, this produces:GFlop/sec        = 1.9467069505224963\nGFlop/sec (SIMD) = 17.578554163920018(GFlop/sec measures the performance, and larger numbers are better.) The range for a @simd for loop should be a one-dimensional range. A variable used for accumulating, such as s in the example, is called a reduction variable. By using @simd, you are asserting several properties of the loop:It is safe to execute iterations in arbitrary or overlapping order, with special consideration for reduction variables.\nFloating-point operations on reduction variables can be reordered, possibly causing different results than without @simd.\nNo iteration ever waits on another iteration to make forward progress.A loop containing break, continue, or @goto will cause a compile-time error.Using @simd merely gives the compiler license to vectorize. Whether it actually does so depends on the compiler. To actually benefit from the current implementation, your loop should have the following additional properties:The loop must be an innermost loop.\nThe loop body must be straight-line code. This is why @inbounds is currently needed for all array accesses. The compiler can sometimes turn short &&, ||, and ?: expressions into straight-line code, if it is safe to evaluate all operands unconditionally. Consider using ifelse() instead of ?: in the loop if it is safe to do so.\nAccesses must have a stride pattern and cannot be \"gathers\" (random-index reads) or \"scatters\" (random-index writes).\nThe stride should be unit stride.\nIn some simple cases, for example with 2-3 arrays accessed in a loop, the LLVM auto-vectorization may kick in automatically, leading to no further speedup with @simd.Here is an example with all three kinds of markup. This program first calculates the finite difference of a one-dimensional array, and then evaluates the L2-norm of the result:function init!(u)\n    n = length(u)\n    dx = 1.0 / (n-1)\n    @fastmath @inbounds @simd for i in 1:n\n        u[i] = sin(2pi*dx*i)\n    end\nend\n\nfunction deriv!(u, du)\n    n = length(u)\n    dx = 1.0 / (n-1)\n    @fastmath @inbounds du[1] = (u[2] - u[1]) / dx\n    @fastmath @inbounds @simd for i in 2:n-1\n        du[i] = (u[i+1] - u[i-1]) / (2*dx)\n    end\n    @fastmath @inbounds du[n] = (u[n] - u[n-1]) / dx\nend\n\nfunction norm(u)\n    n = length(u)\n    T = eltype(u)\n    s = zero(T)\n    @fastmath @inbounds @simd for i in 1:n\n        s += u[i]^2\n    end\n    @fastmath @inbounds return sqrt(s/n)\nend\n\nfunction main()\n    n = 2000\n    u = Array{Float64}(n)\n    init!(u)\n    du = similar(u)\n\n    deriv!(u, du)\n    nu = norm(du)\n\n    @time for i in 1:10^6\n        deriv!(u, du)\n        nu = norm(du)\n    end\n\n    println(nu)\nend\n\nmain()On a computer with a 2.7 GHz Intel Core i7 processor, this produces:$ julia wave.jl;\nelapsed time: 1.207814709 seconds (0 bytes allocated)\n\n$ julia --math-mode=ieee wave.jl;\nelapsed time: 4.487083643 seconds (0 bytes allocated)Here, the option --math-mode=ieee disables the @fastmath macro, so that we can compare results.In this case, the speedup due to @fastmath is a factor of about 3.7. This is unusually large – in general, the speedup will be smaller. (In this particular example, the working set of the benchmark is small enough to fit into the L1 cache of the processor, so that memory access latency does not play a role, and computing time is dominated by CPU usage. In many real world programs this is not the case.) Also, in this case this optimization does not change the result – in general, the result will be slightly different. In some cases, especially for numerically unstable algorithms, the result can be very different.The annotation @fastmath re-arranges floating point expressions, e.g. changing the order of evaluation, or assuming that certain special cases (inf, nan) cannot occur. In this case (and on this particular computer), the main difference is that the expression 1 / (2*dx) in the function deriv is hoisted out of the loop (i.e. calculated outside the loop), as if one had written idx = 1 / (2*dx). In the loop, the expression ... / (2*dx) then becomes ... * idx, which is much faster to evaluate. Of course, both the actual optimization that is applied by the compiler as well as the resulting speedup depend very much on the hardware. You can examine the change in generated code by using Julia's code_native() function."
},

{
    "location": "manual/performance-tips.html#Treat-Subnormal-Numbers-as-Zeros-1",
    "page": "Performance Tips",
    "title": "Treat Subnormal Numbers as Zeros",
    "category": "section",
    "text": "Subnormal numbers, formerly called denormal numbers, are useful in many contexts, but incur a performance penalty on some hardware. A call set_zero_subnormals(true) grants permission for floating-point operations to treat subnormal inputs or outputs as zeros, which may improve performance on some hardware. A call set_zero_subnormals(false) enforces strict IEEE behavior for subnormal numbers.Below is an example where subnormals noticeably impact performance on some hardware:function timestep(b::Vector{T}, a::Vector{T}, Δt::T) where T\n    @assert length(a)==length(b)\n    n = length(b)\n    b[1] = 1                            # Boundary condition\n    for i=2:n-1\n        b[i] = a[i] + (a[i-1] - T(2)*a[i] + a[i+1]) * Δt\n    end\n    b[n] = 0                            # Boundary condition\nend\n\nfunction heatflow(a::Vector{T}, nstep::Integer) where T\n    b = similar(a)\n    for t=1:div(nstep,2)                # Assume nstep is even\n        timestep(b,a,T(0.1))\n        timestep(a,b,T(0.1))\n    end\nend\n\nheatflow(zeros(Float32,10),2)           # Force compilation\nfor trial=1:6\n    a = zeros(Float32,1000)\n    set_zero_subnormals(iseven(trial))  # Odd trials use strict IEEE arithmetic\n    @time heatflow(a,1000)\nendThis example generates many subnormal numbers because the values in a become an exponentially decreasing curve, which slowly flattens out over time.Treating subnormals as zeros should be used with caution, because doing so breaks some identities, such as x-y == 0 implies x == y:julia> x = 3f-38; y = 2f-38;\n\njulia> set_zero_subnormals(true); (x - y, x == y)\n(0.0f0, false)\n\njulia> set_zero_subnormals(false); (x - y, x == y)\n(1.0000001f-38, false)In some applications, an alternative to zeroing subnormal numbers is to inject a tiny bit of noise.  For example, instead of initializing a with zeros, initialize it with:a = rand(Float32,1000) * 1.f-9"
},

{
    "location": "manual/performance-tips.html#man-code-warntype-1",
    "page": "Performance Tips",
    "title": "@code_warntype",
    "category": "section",
    "text": "The macro @code_warntype (or its function variant code_warntype()) can sometimes be helpful in diagnosing type-related problems. Here's an example:pos(x) = x < 0 ? 0 : x\n\nfunction f(x)\n    y = pos(x)\n    sin(y*x+1)\nend\n\njulia> @code_warntype f(3.2)\nVariables:\n  #self#::#f\n  x::Float64\n  y::UNION{FLOAT64,INT64}\n  fy::Float64\n  #temp#@_5::UNION{FLOAT64,INT64}\n  #temp#@_6::Core.MethodInstance\n  #temp#@_7::Float64\n\nBody:\n  begin\n      $(Expr(:inbounds, false))\n      # meta: location REPL[1] pos 1\n      # meta: location float.jl < 487\n      fy::Float64 = (Core.typeassert)((Base.sitofp)(Float64,0)::Float64,Float64)::Float64\n      # meta: pop location\n      unless (Base.or_int)((Base.lt_float)(x::Float64,fy::Float64)::Bool,(Base.and_int)((Base.and_int)((Base.eq_float)(x::Float64,fy::Float64)::Bool,(Base.lt_float)(fy::Float64,9.223372036854776e18)::Bool)::Bool,(Base.slt_int)((Base.fptosi)(Int64,fy::Float64)::Int64,0)::Bool)::Bool)::Bool goto 9\n      #temp#@_5::UNION{FLOAT64,INT64} = 0\n      goto 11\n      9:\n      #temp#@_5::UNION{FLOAT64,INT64} = x::Float64\n      11:\n      # meta: pop location\n      $(Expr(:inbounds, :pop))\n      y::UNION{FLOAT64,INT64} = #temp#@_5::UNION{FLOAT64,INT64} # line 3:\n      unless (y::UNION{FLOAT64,INT64} isa Int64)::ANY goto 19\n      #temp#@_6::Core.MethodInstance = MethodInstance for *(::Int64, ::Float64)\n      goto 28\n      19:\n      unless (y::UNION{FLOAT64,INT64} isa Float64)::ANY goto 23\n      #temp#@_6::Core.MethodInstance = MethodInstance for *(::Float64, ::Float64)\n      goto 28\n      23:\n      goto 25\n      25:\n      #temp#@_7::Float64 = (y::UNION{FLOAT64,INT64} * x::Float64)::Float64\n      goto 30\n      28:\n      #temp#@_7::Float64 = $(Expr(:invoke, :(#temp#@_6), :(Main.*), :(y), :(x)))\n      30:\n      return $(Expr(:invoke, MethodInstance for sin(::Float64), :(Main.sin), :((Base.add_float)(#temp#@_7,(Base.sitofp)(Float64,1)::Float64)::Float64)))\n  end::Float64Interpreting the output of @code_warntype, like that of its cousins @code_lowered, @code_typed, @code_llvm, and @code_native, takes a little practice. Your code is being presented in form that has been partially digested on its way to generating compiled machine code.  Most of the expressions are annotated by a type, indicated by the ::T (where T might be Float64, for example). The most important characteristic of @code_warntype is that non-concrete types are displayed in red; in the above example, such output is shown in all-caps.The top part of the output summarizes the type information for the different variables internal to the function. You can see that y, one of the variables you created, is a Union{Int64,Float64}, due to the type-instability of pos.  There is another variable, _var4, which you can see also has the same type.The next lines represent the body of f. The lines starting with a number followed by a colon (1:, 2:) are labels, and represent targets for jumps (via goto) in your code.  Looking at the body, you can see that pos has been inlined into f–everything before 2: comes from code defined in pos.Starting at 2:, the variable y is defined, and again annotated as a Union type.  Next, we see that the compiler created the temporary variable _var1 to hold the result of y*x. Because a Float64 times either an Int64 or Float64 yields a Float64, all type-instability ends here. The net result is that f(x::Float64) will not be type-unstable in its output, even if some of the intermediate computations are type-unstable.How you use this information is up to you. Obviously, it would be far and away best to fix pos to be type-stable: if you did so, all of the variables in f would be concrete, and its performance would be optimal.  However, there are circumstances where this kind of ephemeral type instability might not matter too much: for example, if pos is never used in isolation, the fact that f's output is type-stable (for Float64 inputs) will shield later code from the propagating effects of type instability.  This is particularly relevant in cases where fixing the type instability is difficult or impossible: for example, currently it's not possible to infer the return type of an anonymous function.  In such cases, the tips above (e.g., adding type annotations and/or breaking up functions) are your best tools to contain the \"damage\" from type instability.The following examples may help you interpret expressions marked as containing non-leaf types:Function body ending in end::Union{T1,T2})\nInterpretation: function with unstable return type\nSuggestion: make the return value type-stable, even if you have to annotate it\nf(x::T)::Union{T1,T2}\nInterpretation: call to a type-unstable function\nSuggestion: fix the function, or if necessary annotate the return value\n(top(arrayref))(A::Array{Any,1},1)::Any\nInterpretation: accessing elements of poorly-typed arrays\nSuggestion: use arrays with better-defined types, or if necessary annotate the type of individual element accesses\n(top(getfield))(A::ArrayContainer{Float64},:data)::Array{Float64,N}\nInterpretation: getting a field that is of non-leaf type. In this case, ArrayContainer had a field data::Array{T}. But Array needs the dimension N, too, to be a concrete type.\nSuggestion: use concrete types like Array{T,3} or Array{T,N}, where N is now a parameter of ArrayContainer"
},

{
    "location": "manual/workflow-tips.html#",
    "page": "Workflow Tips",
    "title": "Workflow Tips",
    "category": "page",
    "text": ""
},

{
    "location": "manual/workflow-tips.html#man-workflow-tips-1",
    "page": "Workflow Tips",
    "title": "Workflow Tips",
    "category": "section",
    "text": "Here are some tips for working with Julia efficiently."
},

{
    "location": "manual/workflow-tips.html#REPL-based-workflow-1",
    "page": "Workflow Tips",
    "title": "REPL-based workflow",
    "category": "section",
    "text": "As already elaborated in Interacting With Julia, Julia's REPL provides rich functionality that facilitates an efficient interactive workflow. Here are some tips that might further enhance your experience at the command line."
},

{
    "location": "manual/workflow-tips.html#A-basic-editor/REPL-workflow-1",
    "page": "Workflow Tips",
    "title": "A basic editor/REPL workflow",
    "category": "section",
    "text": "The most basic Julia workflows involve using a text editor in conjunction with the julia command line. A common pattern includes the following elements:Put code under development in a temporary module. Create a file, say Tmp.jl, and include within it\nmodule Tmp\n\n<your definitions here>\n\nend\nPut your test code in another file. Create another file, say tst.jl, which begins with\nimport Tmp\nand includes tests for the contents of Tmp. The value of using import versus using is that you can call reload(\"Tmp\") instead of having to restart the REPL when your definitions change. Of course, the cost is the need to prepend Tmp. to uses of names defined in your module. (You can lower that cost by keeping your module name short.)\nAlternatively, you can wrap the contents of your test file in a module, as\nmodule Tst\n    using Tmp\n\n    <scratch work>\n\nend\nThe advantage is that you can now do using Tmp in your test code and can therefore avoid prepending Tmp. everywhere. The disadvantage is that code can no longer be selectively copied to the REPL without some tweaking.\nLather. Rinse. Repeat. Explore ideas at the julia command prompt. Save good ideas in tst.jl. Occasionally restart the REPL, issuing\nreload(\"Tmp\")\ninclude(\"tst.jl\")"
},

{
    "location": "manual/workflow-tips.html#Simplify-initialization-1",
    "page": "Workflow Tips",
    "title": "Simplify initialization",
    "category": "section",
    "text": "To simplify restarting the REPL, put project-specific initialization code in a file, say _init.jl, which you can run on startup by issuing the command:julia -L _init.jlIf you further add the following to your .juliarc.jl fileisfile(\"_init.jl\") && include(joinpath(pwd(), \"_init.jl\"))then calling julia from that directory will run the initialization code without the additional command line argument."
},

{
    "location": "manual/workflow-tips.html#Browser-based-workflow-1",
    "page": "Workflow Tips",
    "title": "Browser-based workflow",
    "category": "section",
    "text": "It is also possible to interact with a Julia REPL in the browser via IJulia. See the package home for details."
},

{
    "location": "manual/style-guide.html#",
    "page": "Style Guide",
    "title": "Style Guide",
    "category": "page",
    "text": ""
},

{
    "location": "manual/style-guide.html#Style-Guide-1",
    "page": "Style Guide",
    "title": "Style Guide",
    "category": "section",
    "text": "The following sections explain a few aspects of idiomatic Julia coding style. None of these rules are absolute; they are only suggestions to help familiarize you with the language and to help you choose among alternative designs."
},

{
    "location": "manual/style-guide.html#Write-functions,-not-just-scripts-1",
    "page": "Style Guide",
    "title": "Write functions, not just scripts",
    "category": "section",
    "text": "Writing code as a series of steps at the top level is a quick way to get started solving a problem, but you should try to divide a program into functions as soon as possible. Functions are more reusable and testable, and clarify what steps are being done and what their inputs and outputs are. Furthermore, code inside functions tends to run much faster than top level code, due to how Julia's compiler works.It is also worth emphasizing that functions should take arguments, instead of operating directly on global variables (aside from constants like pi)."
},

{
    "location": "manual/style-guide.html#Avoid-writing-overly-specific-types-1",
    "page": "Style Guide",
    "title": "Avoid writing overly-specific types",
    "category": "section",
    "text": "Code should be as generic as possible. Instead of writing:convert(Complex{Float64}, x)it's better to use available generic functions:complex(float(x))The second version will convert x to an appropriate type, instead of always the same type.This style point is especially relevant to function arguments. For example, don't declare an argument to be of type Int or Int32 if it really could be any integer, expressed with the abstract type Integer. In fact, in many cases you can omit the argument type altogether, unless it is needed to disambiguate from other method definitions, since a MethodError will be thrown anyway if a type is passed that does not support any of the requisite operations. (This is known as duck typing.)For example, consider the following definitions of a function addone that returns one plus its argument:addone(x::Int) = x + 1                 # works only for Int\naddone(x::Integer) = x + oneunit(x)    # any integer type\naddone(x::Number) = x + oneunit(x)     # any numeric type\naddone(x) = x + oneunit(x)             # any type supporting + and oneunitThe last definition of addone handles any type supporting oneunit (which returns 1 in the same type as x, which avoids unwanted type promotion) and the + function with those arguments. The key thing to realize is that there is no performance penalty to defining only the general addone(x) = x + oneunit(x), because Julia will automatically compile specialized versions as needed. For example, the first time you call addone(12), Julia will automatically compile a specialized addone function for x::Int arguments, with the call to oneunit replaced by its inlined value 1. Therefore, the first three definitions of addone above are completely redundant with the fourth definition."
},

{
    "location": "manual/style-guide.html#Handle-excess-argument-diversity-in-the-caller-1",
    "page": "Style Guide",
    "title": "Handle excess argument diversity in the caller",
    "category": "section",
    "text": "Instead of:function foo(x, y)\n    x = Int(x); y = Int(y)\n    ...\nend\nfoo(x, y)use:function foo(x::Int, y::Int)\n    ...\nend\nfoo(Int(x), Int(y))This is better style because foo does not really accept numbers of all types; it really needs Int s.One issue here is that if a function inherently requires integers, it might be better to force the caller to decide how non-integers should be converted (e.g. floor or ceiling). Another issue is that declaring more specific types leaves more \"space\" for future method definitions."
},

{
    "location": "manual/style-guide.html#Append-!-to-names-of-functions-that-modify-their-arguments-1",
    "page": "Style Guide",
    "title": "Append ! to names of functions that modify their arguments",
    "category": "section",
    "text": "Instead of:function double(a::AbstractArray{<:Number})\n    for i = 1:endof(a)\n        a[i] *= 2\n    end\n    return a\nenduse:function double!(a::AbstractArray{<:Number})\n    for i = 1:endof(a)\n        a[i] *= 2\n    end\n    return a\nendThe Julia standard library uses this convention throughout and contains examples of functions with both copying and modifying forms (e.g., sort() and sort!()), and others which are just modifying (e.g., push!(), pop!(), splice!()).  It is typical for such functions to also return the modified array for convenience."
},

{
    "location": "manual/style-guide.html#Avoid-strange-type-Unions-1",
    "page": "Style Guide",
    "title": "Avoid strange type Unions",
    "category": "section",
    "text": "Types such as Union{Function,AbstractString} are often a sign that some design could be cleaner."
},

{
    "location": "manual/style-guide.html#Avoid-type-Unions-in-fields-1",
    "page": "Style Guide",
    "title": "Avoid type Unions in fields",
    "category": "section",
    "text": "When creating a type such as:mutable struct MyType\n    ...\n    x::Union{Void,T}\nendask whether the option for x to be nothing (of type Void) is really necessary. Here are some alternatives to consider:Find a safe default value to initialize x with\nIntroduce another type that lacks x\nIf there are many fields like x, store them in a dictionary\nDetermine whether there is a simple rule for when x is nothing. For example, often the field will start as nothing but get initialized at some well-defined point. In that case, consider leaving it undefined at first.\nIf x really needs to hold no value at some times, define it as ::Nullable{T} instead, as this guarantees type-stability in the code accessing this field (see Nullable types)."
},

{
    "location": "manual/style-guide.html#Avoid-elaborate-container-types-1",
    "page": "Style Guide",
    "title": "Avoid elaborate container types",
    "category": "section",
    "text": "It is usually not much help to construct arrays like the following:a = Array{Union{Int,AbstractString,Tuple,Array}}(n)In this case Array{Any}(n) is better. It is also more helpful to the compiler to annotate specific uses (e.g. a[i]::Int) than to try to pack many alternatives into one type."
},

{
    "location": "manual/style-guide.html#Use-naming-conventions-consistent-with-Julia's-base/-1",
    "page": "Style Guide",
    "title": "Use naming conventions consistent with Julia's base/",
    "category": "section",
    "text": "modules and type names use capitalization and camel case: module SparseArrays, struct UnitRange.\nfunctions are lowercase (maximum(), convert()) and, when readable, with multiple words squashed together (isequal(), haskey()). When necessary, use underscores as word separators. Underscores are also used to indicate a combination of concepts (remotecall_fetch() as a more efficient implementation of fetch(remotecall(...))) or as modifiers (sum_kbn()).\nconciseness is valued, but avoid abbreviation (indexin() rather than indxin()) as it becomes difficult to remember whether and how particular words are abbreviated.If a function name requires multiple words, consider whether it might represent more than one concept and might be better split into pieces."
},

{
    "location": "manual/style-guide.html#Don't-overuse-try-catch-1",
    "page": "Style Guide",
    "title": "Don't overuse try-catch",
    "category": "section",
    "text": "It is better to avoid errors than to rely on catching them."
},

{
    "location": "manual/style-guide.html#Don't-parenthesize-conditions-1",
    "page": "Style Guide",
    "title": "Don't parenthesize conditions",
    "category": "section",
    "text": "Julia doesn't require parens around conditions in if and while. Write:if a == binstead of:if (a == b)"
},

{
    "location": "manual/style-guide.html#Don't-overuse-...-1",
    "page": "Style Guide",
    "title": "Don't overuse ...",
    "category": "section",
    "text": "Splicing function arguments can be addictive. Instead of [a..., b...], use simply [a; b], which already concatenates arrays. collect(a) is better than [a...], but since a is already iterable it is often even better to leave it alone, and not convert it to an array."
},

{
    "location": "manual/style-guide.html#Don't-use-unnecessary-static-parameters-1",
    "page": "Style Guide",
    "title": "Don't use unnecessary static parameters",
    "category": "section",
    "text": "A function signature:foo(x::T) where {T<:Real} = ...should be written as:foo(x::Real) = ...instead, especially if T is not used in the function body. Even if T is used, it can be replaced with typeof(x) if convenient. There is no performance difference. Note that this is not a general caution against static parameters, just against uses where they are not needed.Note also that container types, specifically may need type parameters in function calls. See the FAQ Avoid fields with abstract containers for more information."
},

{
    "location": "manual/style-guide.html#Avoid-confusion-about-whether-something-is-an-instance-or-a-type-1",
    "page": "Style Guide",
    "title": "Avoid confusion about whether something is an instance or a type",
    "category": "section",
    "text": "Sets of definitions like the following are confusing:foo(::Type{MyType}) = ...\nfoo(::MyType) = foo(MyType)Decide whether the concept in question will be written as MyType or MyType(), and stick to it.The preferred style is to use instances by default, and only add methods involving Type{MyType} later if they become necessary to solve some problem.If a type is effectively an enumeration, it should be defined as a single (ideally immutable struct or primitive) type, with the enumeration values being instances of it. Constructors and conversions can check whether values are valid. This design is preferred over making the enumeration an abstract type, with the \"values\" as subtypes."
},

{
    "location": "manual/style-guide.html#Don't-overuse-macros-1",
    "page": "Style Guide",
    "title": "Don't overuse macros",
    "category": "section",
    "text": "Be aware of when a macro could really be a function instead.Calling eval() inside a macro is a particularly dangerous warning sign; it means the macro will only work when called at the top level. If such a macro is written as a function instead, it will naturally have access to the run-time values it needs."
},

{
    "location": "manual/style-guide.html#Don't-expose-unsafe-operations-at-the-interface-level-1",
    "page": "Style Guide",
    "title": "Don't expose unsafe operations at the interface level",
    "category": "section",
    "text": "If you have a type that uses a native pointer:mutable struct NativeType\n    p::Ptr{UInt8}\n    ...\nenddon't write definitions like the following:getindex(x::NativeType, i) = unsafe_load(x.p, i)The problem is that users of this type can write x[i] without realizing that the operation is unsafe, and then be susceptible to memory bugs.Such a function should either check the operation to ensure it is safe, or have unsafe somewhere in its name to alert callers."
},

{
    "location": "manual/style-guide.html#Don't-overload-methods-of-base-container-types-1",
    "page": "Style Guide",
    "title": "Don't overload methods of base container types",
    "category": "section",
    "text": "It is possible to write definitions like the following:show(io::IO, v::Vector{MyType}) = ...This would provide custom showing of vectors with a specific new element type. While tempting, this should be avoided. The trouble is that users will expect a well-known type like Vector() to behave in a certain way, and overly customizing its behavior can make it harder to work with."
},

{
    "location": "manual/style-guide.html#Avoid-type-piracy-1",
    "page": "Style Guide",
    "title": "Avoid type piracy",
    "category": "section",
    "text": "\"Type piracy\" refers to the practice of extending or redefining methods in Base or other packages on types that you have not defined. In some cases, you can get away with type piracy with little ill effect. In extreme cases, however, you can even crash Julia (e.g. if your method extension or redefinition causes invalid input to be passed to a ccall). Type piracy can complicate reasoning about code, and may introduce incompatibilities that are hard to predict and diagnose.As an example, suppose you wanted to define multiplication on symbols in a module:module A\nimport Base.*\n*(x::Symbol, y::Symbol) = Symbol(x,y)\nendThe problem is that now any other module that uses Base.* will also see this definition. Since Symbol is defined in Base and is used by other modules, this can change the behavior of unrelated code unexpectedly. There are several alternatives here, including using a different function name, or wrapping the Symbols in another type that you define.Sometimes, coupled packages may engage in type piracy to separate features from definitions, especially when the packages were designed by collaborating authors, and when the definitions are reusable. For example, one package might provide some types useful for working with colors; another package could define methods for those types that enable conversions between color spaces. Another example might be a package that acts as a thin wrapper for some C code, which another package might then pirate to implement a higher-level, Julia-friendly API."
},

{
    "location": "manual/style-guide.html#Be-careful-with-type-equality-1",
    "page": "Style Guide",
    "title": "Be careful with type equality",
    "category": "section",
    "text": "You generally want to use isa() and <: (issubtype()) for testing types, not ==. Checking types for exact equality typically only makes sense when comparing to a known concrete type (e.g. T == Float64), or if you really, really know what you're doing."
},

{
    "location": "manual/style-guide.html#Do-not-write-x-f(x)-1",
    "page": "Style Guide",
    "title": "Do not write x->f(x)",
    "category": "section",
    "text": "Since higher-order functions are often called with anonymous functions, it is easy to conclude that this is desirable or even necessary. But any function can be passed directly, without being \"wrapped\" in an anonymous function. Instead of writing map(x->f(x), a), write map(f, a)."
},

{
    "location": "manual/style-guide.html#Avoid-using-floats-for-numeric-literals-in-generic-code-when-possible-1",
    "page": "Style Guide",
    "title": "Avoid using floats for numeric literals in generic code when possible",
    "category": "section",
    "text": "If you write generic code which handles numbers, and which can be expected to run with many different numeric type arguments, try using literals of a numeric type that will affect the arguments as little as possible through promotion.For example,julia> f(x) = 2.0 * x\nf (generic function with 1 method)\n\njulia> f(1//2)\n1.0\n\njulia> f(1/2)\n1.0\n\njulia> f(1)\n2.0whilejulia> g(x) = 2 * x\ng (generic function with 1 method)\n\njulia> g(1//2)\n1//1\n\njulia> g(1/2)\n1.0\n\njulia> g(1)\n2As you can see, the second version, where we used an Int literal, preserved the type of the input argument, while the first didn't. This is because e.g. promote_type(Int, Float64) == Float64, and promotion happens with the multiplication. Similarly, Rational literals are less type disruptive than Float64 literals, but more disruptive than Ints:julia> h(x) = 2//1 * x\nh (generic function with 1 method)\n\njulia> h(1//2)\n1//1\n\njulia> h(1/2)\n1.0\n\njulia> h(1)\n2//1Thus, use Int literals when possible, with Rational{Int} for literal non-integer numbers, in order to make it easier to use your code."
},

{
    "location": "manual/faq.html#",
    "page": "Frequently Asked Questions",
    "title": "Frequently Asked Questions",
    "category": "page",
    "text": ""
},

{
    "location": "manual/faq.html#Frequently-Asked-Questions-1",
    "page": "Frequently Asked Questions",
    "title": "Frequently Asked Questions",
    "category": "section",
    "text": ""
},

{
    "location": "manual/faq.html#Sessions-and-the-REPL-1",
    "page": "Frequently Asked Questions",
    "title": "Sessions and the REPL",
    "category": "section",
    "text": ""
},

{
    "location": "manual/faq.html#How-do-I-delete-an-object-in-memory?-1",
    "page": "Frequently Asked Questions",
    "title": "How do I delete an object in memory?",
    "category": "section",
    "text": "Julia does not have an analog of MATLAB's clear function; once a name is defined in a Julia session (technically, in module Main), it is always present.If memory usage is your concern, you can always replace objects with ones that consume less memory.  For example, if A is a gigabyte-sized array that you no longer need, you can free the memory with A = 0.  The memory will be released the next time the garbage collector runs; you can force this to happen with gc()."
},

{
    "location": "manual/faq.html#How-can-I-modify-the-declaration-of-a-type-in-my-session?-1",
    "page": "Frequently Asked Questions",
    "title": "How can I modify the declaration of a type in my session?",
    "category": "section",
    "text": "Perhaps you've defined a type and then realize you need to add a new field.  If you try this at the REPL, you get the error:ERROR: invalid redefinition of constant MyTypeTypes in module Main cannot be redefined.While this can be inconvenient when you are developing new code, there's an excellent workaround.  Modules can be replaced by redefining them, and so if you wrap all your new code inside a module you can redefine types and constants.  You can't import the type names into Main and then expect to be able to redefine them there, but you can use the module name to resolve the scope.  In other words, while developing you might use a workflow something like this:include(\"mynewcode.jl\")              # this defines a module MyModule\nobj1 = MyModule.ObjConstructor(a, b)\nobj2 = MyModule.somefunction(obj1)\n# Got an error. Change something in \"mynewcode.jl\"\ninclude(\"mynewcode.jl\")              # reload the module\nobj1 = MyModule.ObjConstructor(a, b) # old objects are no longer valid, must reconstruct\nobj2 = MyModule.somefunction(obj1)   # this time it worked!\nobj3 = MyModule.someotherfunction(obj2, c)\n..."
},

{
    "location": "manual/faq.html#Functions-1",
    "page": "Frequently Asked Questions",
    "title": "Functions",
    "category": "section",
    "text": ""
},

{
    "location": "manual/faq.html#I-passed-an-argument-x-to-a-function,-modified-it-inside-that-function,-but-on-the-outside,-1",
    "page": "Frequently Asked Questions",
    "title": "I passed an argument x to a function, modified it inside that function, but on the outside,",
    "category": "section",
    "text": "the variable x is still unchanged. Why?Suppose you call a function like this:julia> x = 10\n10\n\njulia> function change_value!(y)\n           y = 17\n       end\nchange_value! (generic function with 1 method)\n\njulia> change_value!(x)\n17\n\njulia> x # x is unchanged!\n10In Julia, the binding of a variable x cannot be changed by passing x as an argument to a function. When calling change_value!(x) in the above example, y is a newly created variable, bound initially to the value of x, i.e. 10; then y is rebound to the constant 17, while the variable x of the outer scope is left untouched.But here is a thing you should pay attention to: suppose x is bound to an object of type Array (or any other mutable type). From within the function, you cannot \"unbind\" x from this Array, but you can change its content. For example:julia> x = [1,2,3]\n3-element Array{Int64,1}:\n 1\n 2\n 3\n\njulia> function change_array!(A)\n           A[1] = 5\n       end\nchange_array! (generic function with 1 method)\n\njulia> change_array!(x)\n5\n\njulia> x\n3-element Array{Int64,1}:\n 5\n 2\n 3Here we created a function change_array!(), that assigns 5 to the first element of the passed array (bound to x at the call site, and bound to A within the function). Notice that, after the function call, x is still bound to the same array, but the content of that array changed: the variables A and x were distinct bindings refering to the same mutable Array object."
},

{
    "location": "manual/faq.html#Can-I-use-using-or-import-inside-a-function?-1",
    "page": "Frequently Asked Questions",
    "title": "Can I use using or import inside a function?",
    "category": "section",
    "text": "No, you are not allowed to have a using or import statement inside a function.  If you want to import a module but only use its symbols inside a specific function or set of functions, you have two options:Use import:\nimport Foo\nfunction bar(...)\n    # ... refer to Foo symbols via Foo.baz ...\nend\nThis loads the module Foo and defines a variable Foo that refers to the module, but does not import any of the other symbols from the module into the current namespace.  You refer to the Foo symbols by their qualified names Foo.bar etc.\nWrap your function in a module:\nmodule Bar\nexport bar\nusing Foo\nfunction bar(...)\n    # ... refer to Foo.baz as simply baz ....\nend\nend\nusing Bar\nThis imports all the symbols from Foo, but only inside the module Bar."
},

{
    "location": "manual/faq.html#What-does-the-...-operator-do?-1",
    "page": "Frequently Asked Questions",
    "title": "What does the ... operator do?",
    "category": "section",
    "text": ""
},

{
    "location": "manual/faq.html#The-two-uses-of-the-...-operator:-slurping-and-splatting-1",
    "page": "Frequently Asked Questions",
    "title": "The two uses of the ... operator: slurping and splatting",
    "category": "section",
    "text": "Many newcomers to Julia find the use of ... operator confusing. Part of what makes the ... operator confusing is that it means two different things depending on context."
},

{
    "location": "manual/faq.html#...-combines-many-arguments-into-one-argument-in-function-definitions-1",
    "page": "Frequently Asked Questions",
    "title": "... combines many arguments into one argument in function definitions",
    "category": "section",
    "text": "In the context of function definitions, the ... operator is used to combine many different arguments into a single argument. This use of ... for combining many different arguments into a single argument is called slurping:julia> function printargs(args...)\n           @printf(\"%s\\n\", typeof(args))\n           for (i, arg) in enumerate(args)\n               @printf(\"Arg %d = %s\\n\", i, arg)\n           end\n       end\nprintargs (generic function with 1 method)\n\njulia> printargs(1, 2, 3)\nTuple{Int64,Int64,Int64}\nArg 1 = 1\nArg 2 = 2\nArg 3 = 3If Julia were a language that made more liberal use of ASCII characters, the slurping operator might have been written as <-... instead of ...."
},

{
    "location": "manual/faq.html#...-splits-one-argument-into-many-different-arguments-in-function-calls-1",
    "page": "Frequently Asked Questions",
    "title": "... splits one argument into many different arguments in function calls",
    "category": "section",
    "text": "In contrast to the use of the ... operator to denote slurping many different arguments into one argument when defining a function, the ... operator is also used to cause a single function argument to be split apart into many different arguments when used in the context of a function call. This use of ... is called splatting:julia> function threeargs(a, b, c)\n           @printf(\"a = %s::%s\\n\", a, typeof(a))\n           @printf(\"b = %s::%s\\n\", b, typeof(b))\n           @printf(\"c = %s::%s\\n\", c, typeof(c))\n       end\nthreeargs (generic function with 1 method)\n\njulia> vec = [1, 2, 3]\n3-element Array{Int64,1}:\n 1\n 2\n 3\n\njulia> threeargs(vec...)\na = 1::Int64\nb = 2::Int64\nc = 3::Int64If Julia were a language that made more liberal use of ASCII characters, the splatting operator might have been written as ...-> instead of ...."
},

{
    "location": "manual/faq.html#Types,-type-declarations,-and-constructors-1",
    "page": "Frequently Asked Questions",
    "title": "Types, type declarations, and constructors",
    "category": "section",
    "text": ""
},

{
    "location": "manual/faq.html#man-type-stability-1",
    "page": "Frequently Asked Questions",
    "title": "What does \"type-stable\" mean?",
    "category": "section",
    "text": "It means that the type of the output is predictable from the types of the inputs.  In particular, it means that the type of the output cannot vary depending on the values of the inputs. The following code is not type-stable:julia> function unstable(flag::Bool)\n           if flag\n               return 1\n           else\n               return 1.0\n           end\n       end\nunstable (generic function with 1 method)It returns either an Int or a Float64 depending on the value of its argument. Since Julia can't predict the return type of this function at compile-time, any computation that uses it will have to guard against both types possibly occurring, making generation of fast machine code difficult."
},

{
    "location": "manual/faq.html#faq-domain-errors-1",
    "page": "Frequently Asked Questions",
    "title": "Why does Julia give a DomainError for certain seemingly-sensible operations?",
    "category": "section",
    "text": "Certain operations make mathematical sense but result in errors:julia> sqrt(-2.0)\nERROR: DomainError:\nsqrt will only return a complex result if called with a complex argument. Try sqrt(complex(x)).\nStacktrace:\n [1] sqrt(::Float64) at ./math.jl:425\n\njulia> 2^-5\nERROR: DomainError:\nCannot raise an integer x to a negative power -n.\nMake x a float by adding a zero decimal (e.g. 2.0^-n instead of 2^-n), or write 1/x^n, float(x)^-n, or (x//1)^-n.\nStacktrace:\n [1] power_by_squaring(::Int64, ::Int64) at ./intfuncs.jl:173\n [2] literal_pow(::Base.#^, ::Int64, ::Type{Val{-5}}) at ./intfuncs.jl:208This behavior is an inconvenient consequence of the requirement for type-stability.  In the case of sqrt(), most users want sqrt(2.0) to give a real number, and would be unhappy if it produced the complex number 1.4142135623730951 + 0.0im.  One could write the sqrt() function to switch to a complex-valued output only when passed a negative number (which is what sqrt() does in some other languages), but then the result would not be type-stable and the sqrt() function would have poor performance.In these and other cases, you can get the result you want by choosing an input type that conveys your willingness to accept an output type in which the result can be represented:julia> sqrt(-2.0+0im)\n0.0 + 1.4142135623730951im\n\njulia> 2.0^-5\n0.03125"
},

{
    "location": "manual/faq.html#Why-does-Julia-use-native-machine-integer-arithmetic?-1",
    "page": "Frequently Asked Questions",
    "title": "Why does Julia use native machine integer arithmetic?",
    "category": "section",
    "text": "Julia uses machine arithmetic for integer computations. This means that the range of Int values is bounded and wraps around at either end so that adding, subtracting and multiplying integers can overflow or underflow, leading to some results that can be unsettling at first:julia> typemax(Int)\n9223372036854775807\n\njulia> ans+1\n-9223372036854775808\n\njulia> -ans\n-9223372036854775808\n\njulia> 2*ans\n0Clearly, this is far from the way mathematical integers behave, and you might think it less than ideal for a high-level programming language to expose this to the user. For numerical work where efficiency and transparency are at a premium, however, the alternatives are worse.One alternative to consider would be to check each integer operation for overflow and promote results to bigger integer types such as Int128 or BigInt in the case of overflow. Unfortunately, this introduces major overhead on every integer operation (think incrementing a loop counter) – it requires emitting code to perform run-time overflow checks after arithmetic instructions and branches to handle potential overflows. Worse still, this would cause every computation involving integers to be type-unstable. As we mentioned above, type-stability is crucial for effective generation of efficient code. If you can't count on the results of integer operations being integers, it's impossible to generate fast, simple code the way C and Fortran compilers do.A variation on this approach, which avoids the appearance of type instability is to merge the Int and BigInt types into a single hybrid integer type, that internally changes representation when a result no longer fits into the size of a machine integer. While this superficially avoids type-instability at the level of Julia code, it just sweeps the problem under the rug by foisting all of the same difficulties onto the C code implementing this hybrid integer type. This approach can be made to work and can even be made quite fast in many cases, but has several drawbacks. One problem is that the in-memory representation of integers and arrays of integers no longer match the natural representation used by C, Fortran and other languages with native machine integers. Thus, to interoperate with those languages, we would ultimately need to introduce native integer types anyway. Any unbounded representation of integers cannot have a fixed number of bits, and thus cannot be stored inline in an array with fixed-size slots – large integer values will always require separate heap-allocated storage. And of course, no matter how clever a hybrid integer implementation one uses, there are always performance traps – situations where performance degrades unexpectedly. Complex representation, lack of interoperability with C and Fortran, the inability to represent integer arrays without additional heap storage, and unpredictable performance characteristics make even the cleverest hybrid integer implementations a poor choice for high-performance numerical work.An alternative to using hybrid integers or promoting to BigInts is to use saturating integer arithmetic, where adding to the largest integer value leaves it unchanged and likewise for subtracting from the smallest integer value. This is precisely what Matlab™ does:>> int64(9223372036854775807)\n\nans =\n\n  9223372036854775807\n\n>> int64(9223372036854775807) + 1\n\nans =\n\n  9223372036854775807\n\n>> int64(-9223372036854775808)\n\nans =\n\n -9223372036854775808\n\n>> int64(-9223372036854775808) - 1\n\nans =\n\n -9223372036854775808At first blush, this seems reasonable enough since 9223372036854775807 is much closer to 9223372036854775808 than -9223372036854775808 is and integers are still represented with a fixed size in a natural way that is compatible with C and Fortran. Saturated integer arithmetic, however, is deeply problematic. The first and most obvious issue is that this is not the way machine integer arithmetic works, so implementing saturated operations requires emitting instructions after each machine integer operation to check for underflow or overflow and replace the result with typemin(Int) or typemax(Int) as appropriate. This alone expands each integer operation from a single, fast instruction into half a dozen instructions, probably including branches. Ouch. But it gets worse – saturating integer arithmetic isn't associative. Consider this Matlab computation:>> n = int64(2)^62\n4611686018427387904\n\n>> n + (n - 1)\n9223372036854775807\n\n>> (n + n) - 1\n9223372036854775806This makes it hard to write many basic integer algorithms since a lot of common techniques depend on the fact that machine addition with overflow is associative. Consider finding the midpoint between integer values lo and hi in Julia using the expression (lo + hi) >>> 1:julia> n = 2^62\n4611686018427387904\n\njulia> (n + 2n) >>> 1\n6917529027641081856See? No problem. That's the correct midpoint between 2^62 and 2^63, despite the fact that n + 2n is -4611686018427387904. Now try it in Matlab:>> (n + 2*n)/2\n\nans =\n\n  4611686018427387904Oops. Adding a >>> operator to Matlab wouldn't help, because saturation that occurs when adding n and 2n has already destroyed the information necessary to compute the correct midpoint.Not only is lack of associativity unfortunate for programmers who cannot rely it for techniques like this, but it also defeats almost anything compilers might want to do to optimize integer arithmetic. For example, since Julia integers use normal machine integer arithmetic, LLVM is free to aggressively optimize simple little functions like f(k) = 5k-1. The machine code for this function is just this:julia> code_native(f, Tuple{Int})\n  .text\nFilename: none\n  pushq %rbp\n  movq  %rsp, %rbp\nSource line: 1\n  leaq  -1(%rdi,%rdi,4), %rax\n  popq  %rbp\n  retq\n  nopl  (%rax,%rax)The actual body of the function is a single leaq instruction, which computes the integer multiply and add at once. This is even more beneficial when f gets inlined into another function:julia> function g(k, n)\n           for i = 1:n\n               k = f(k)\n           end\n           return k\n       end\ng (generic function with 1 methods)\n\njulia> code_native(g, Tuple{Int,Int})\n  .text\nFilename: none\n  pushq %rbp\n  movq  %rsp, %rbp\nSource line: 2\n  testq %rsi, %rsi\n  jle L26\n  nopl  (%rax)\nSource line: 3\nL16:\n  leaq  -1(%rdi,%rdi,4), %rdi\nSource line: 2\n  decq  %rsi\n  jne L16\nSource line: 5\nL26:\n  movq  %rdi, %rax\n  popq  %rbp\n  retq\n  nopSince the call to f gets inlined, the loop body ends up being just a single leaq instruction. Next, consider what happens if we make the number of loop iterations fixed:julia> function g(k)\n           for i = 1:10\n               k = f(k)\n           end\n           return k\n       end\ng (generic function with 2 methods)\n\njulia> code_native(g,(Int,))\n  .text\nFilename: none\n  pushq %rbp\n  movq  %rsp, %rbp\nSource line: 3\n  imulq $9765625, %rdi, %rax    # imm = 0x9502F9\n  addq  $-2441406, %rax         # imm = 0xFFDABF42\nSource line: 5\n  popq  %rbp\n  retq\n  nopw  %cs:(%rax,%rax)Because the compiler knows that integer addition and multiplication are associative and that multiplication distributes over addition – neither of which is true of saturating arithmetic – it can optimize the entire loop down to just a multiply and an add. Saturated arithmetic completely defeats this kind of optimization since associativity and distributivity can fail at each loop iteration, causing different outcomes depending on which iteration the failure occurs in. The compiler can unroll the loop, but it cannot algebraically reduce multiple operations into fewer equivalent operations.The most reasonable alternative to having integer arithmetic silently overflow is to do checked arithmetic everywhere, raising errors when adds, subtracts, and multiplies overflow, producing values that are not value-correct. In this blog post, Dan Luu analyzes this and finds that rather than the trivial cost that this approach should in theory have, it ends up having a substantial cost due to compilers (LLVM and GCC) not gracefully optimizing around the added overflow checks. If this improves in the future, we could consider defaulting to checked integer arithmetic in Julia, but for now, we have to live with the possibility of overflow."
},

{
    "location": "manual/faq.html#What-are-the-possible-causes-of-an-UndefVarError-during-remote-execution?-1",
    "page": "Frequently Asked Questions",
    "title": "What are the possible causes of an UndefVarError during remote execution?",
    "category": "section",
    "text": "As the error states, an immediate cause of an UndefVarError on a remote node is that a binding by that name does not exist. Let us explore some of the possible causes.julia> module Foo\n           foo() = remotecall_fetch(x->x, 2, \"Hello\")\n       end\n\njulia> Foo.foo()\nERROR: On worker 2:\nUndefVarError: Foo not defined\n[...]The closure x->x carries a reference to Foo, and since Foo is unavailable on node 2, an UndefVarError is thrown.Globals under modules other than Main are not serialized by value to the remote node. Only a reference is sent. Functions which create global bindings (except under Main) may cause an UndefVarError to be thrown later.julia> @everywhere module Foo\n           function foo()\n               global gvar = \"Hello\"\n               remotecall_fetch(()->gvar, 2)\n           end\n       end\n\njulia> Foo.foo()\nERROR: On worker 2:\nUndefVarError: gvar not defined\n[...]In the above example, @everywhere module Foo defined Foo on all nodes. However the call to Foo.foo() created a new global binding gvar on the local node, but this was not found on node 2 resulting in an UndefVarError error.Note that this does not apply to globals created under module Main. Globals under module Main are serialized and new bindings created under Main on the remote node.julia> gvar_self = \"Node1\"\n\"Node1\"\n\njulia> remotecall_fetch(()->gvar_self, 2)\n\"Node1\"\n\njulia> remotecall_fetch(whos, 2)\n	From worker 2:	                          Base  41762 KB     Module\n	From worker 2:	                          Core  27337 KB     Module\n	From worker 2:	                           Foo   2477 bytes  Module\n	From worker 2:	                          Main  46191 KB     Module\n	From worker 2:	                     gvar_self     13 bytes  StringThis does not apply to function or type declarations. However, anonymous functions bound to global variables are serialized as can be seen below.julia> bar() = 1\nbar (generic function with 1 method)\n\njulia> remotecall_fetch(bar, 2)\nERROR: On worker 2:\nUndefVarError: #bar not defined\n[...]\n\njulia> anon_bar  = ()->1\n(::#21) (generic function with 1 method)\n\njulia> remotecall_fetch(anon_bar, 2)\n1"
},

{
    "location": "manual/faq.html#Packages-and-Modules-1",
    "page": "Frequently Asked Questions",
    "title": "Packages and Modules",
    "category": "section",
    "text": ""
},

{
    "location": "manual/faq.html#What-is-the-difference-between-\"using\"-and-\"importall\"?-1",
    "page": "Frequently Asked Questions",
    "title": "What is the difference between \"using\" and \"importall\"?",
    "category": "section",
    "text": "There is only one difference, and on the surface (syntax-wise) it may seem very minor. The difference between using and importall is that with using you need to say function Foo.bar(.. to extend module Foo's function bar with a new method, but with importall or import Foo.bar, you only need to say function bar(... and it automatically extends module Foo's function bar.If you use importall, then function Foo.bar(... and function bar(... become equivalent. If you use using, then they are different.The reason this is important enough to have been given separate syntax is that you don't want to accidentally extend a function that you didn't know existed, because that could easily cause a bug. This is most likely to happen with a method that takes a common type like a string or integer, because both you and the other module could define a method to handle such a common type. If you use importall, then you'll replace the other module's implementation of bar(s::AbstractString) with your new implementation, which could easily do something completely different (and break all/many future usages of the other functions in module Foo that depend on calling bar)."
},

{
    "location": "manual/faq.html#Nothingness-and-missing-values-1",
    "page": "Frequently Asked Questions",
    "title": "Nothingness and missing values",
    "category": "section",
    "text": ""
},

{
    "location": "manual/faq.html#How-does-\"null\"-or-\"nothingness\"-work-in-Julia?-1",
    "page": "Frequently Asked Questions",
    "title": "How does \"null\" or \"nothingness\" work in Julia?",
    "category": "section",
    "text": "Unlike many languages (for example, C and Java), Julia does not have a \"null\" value. When a reference (variable, object field, or array element) is uninitialized, accessing it will immediately throw an error. This situation can be detected using the isdefined function.Some functions are used only for their side effects, and do not need to return a value. In these cases, the convention is to return the value nothing, which is just a singleton object of type Void. This is an ordinary type with no fields; there is nothing special about it except for this convention, and that the REPL does not print anything for it. Some language constructs that would not otherwise have a value also yield nothing, for example if false; end.For situations where a value exists only sometimes (for example, missing statistical data), it is best to use the Nullable{T} type, which allows specifying the type of a missing value.The empty tuple (()) is another form of nothingness. But, it should not really be thought of as nothing but rather a tuple of zero values.In code written for Julia prior to version 0.4 you may occasionally see None, which is quite different. It is the empty (or \"bottom\") type, a type with no values and no subtypes (except itself). This is now written as Union{} (an empty union type). You will generally not need to use this type."
},

{
    "location": "manual/faq.html#Memory-1",
    "page": "Frequently Asked Questions",
    "title": "Memory",
    "category": "section",
    "text": ""
},

{
    "location": "manual/faq.html#Why-does-x-y-allocate-memory-when-x-and-y-are-arrays?-1",
    "page": "Frequently Asked Questions",
    "title": "Why does x += y allocate memory when x and y are arrays?",
    "category": "section",
    "text": "In Julia, x += y gets replaced during parsing by x = x + y. For arrays, this has the consequence that, rather than storing the result in the same location in memory as x, it allocates a new array to store the result.While this behavior might surprise some, the choice is deliberate. The main reason is the presence of immutable objects within Julia, which cannot change their value once created.  Indeed, a number is an immutable object; the statements x = 5; x += 1 do not modify the meaning of 5, they modify the value bound to x. For an immutable, the only way to change the value is to reassign it.To amplify a bit further, consider the following function:function power_by_squaring(x, n::Int)\n    ispow2(n) || error(\"This implementation only works for powers of 2\")\n    while n >= 2\n        x *= x\n        n >>= 1\n    end\n    x\nendAfter a call like x = 5; y = power_by_squaring(x, 4), you would get the expected result: x == 5 && y == 625.  However, now suppose that *=, when used with matrices, instead mutated the left hand side.  There would be two problems:For general square matrices, A = A*B cannot be implemented without temporary storage: A[1,1] gets computed and stored on the left hand side before you're done using it on the right hand side.\nSuppose you were willing to allocate a temporary for the computation (which would eliminate most of the point of making *= work in-place); if you took advantage of the mutability of x, then this function would behave differently for mutable vs. immutable inputs. In particular, for immutable x, after the call you'd have (in general) y != x, but for mutable x you'd have y == x.Because supporting generic programming is deemed more important than potential performance optimizations that can be achieved by other means (e.g., using explicit loops), operators like += and *= work by rebinding new values."
},

{
    "location": "manual/faq.html#Asynchronous-IO-and-concurrent-synchronous-writes-1",
    "page": "Frequently Asked Questions",
    "title": "Asynchronous IO and concurrent synchronous writes",
    "category": "section",
    "text": ""
},

{
    "location": "manual/faq.html#Why-do-concurrent-writes-to-the-same-stream-result-in-inter-mixed-output?-1",
    "page": "Frequently Asked Questions",
    "title": "Why do concurrent writes to the same stream result in inter-mixed output?",
    "category": "section",
    "text": "While the streaming I/O API is synchronous, the underlying implementation is fully asynchronous.Consider the printed output from the following:julia> @sync for i in 1:3\n           @async write(STDOUT, string(i), \" Foo \", \" Bar \")\n       end\n123 Foo  Foo  Foo  Bar  Bar  BarThis is happening because, while the write call is synchronous, the writing of each argument yields to other tasks while waiting for that part of the I/O to complete.print and println \"lock\" the stream during a call. Consequently changing write to println in the above example results in:julia> @sync for i in 1:3\n           @async println(STDOUT, string(i), \" Foo \", \" Bar \")\n       end\n1 Foo  Bar\n2 Foo  Bar\n3 Foo  BarYou can lock your writes with a ReentrantLock like this:julia> l = ReentrantLock()\nReentrantLock(Nullable{Task}(), Condition(Any[]), 0)\n\njulia> @sync for i in 1:3\n           @async begin\n               lock(l)\n               try\n                   write(STDOUT, string(i), \" Foo \", \" Bar \")\n               finally\n                   unlock(l)\n               end\n           end\n       end\n1 Foo  Bar 2 Foo  Bar 3 Foo  Bar"
},

{
    "location": "manual/faq.html#Julia-Releases-1",
    "page": "Frequently Asked Questions",
    "title": "Julia Releases",
    "category": "section",
    "text": ""
},

{
    "location": "manual/faq.html#Do-I-want-to-use-a-release,-beta,-or-nightly-version-of-Julia?-1",
    "page": "Frequently Asked Questions",
    "title": "Do I want to use a release, beta, or nightly version of Julia?",
    "category": "section",
    "text": "You may prefer the release version of Julia if you are looking for a stable code base. Releases generally occur every 6 months, giving you a stable platform for writing code.You may prefer the beta version of Julia if you don't mind being slightly behind the latest bugfixes and changes, but find the slightly faster rate of changes more appealing. Additionally, these binaries are tested before they are published to ensure they are fully functional.You may prefer the nightly version of Julia if you want to take advantage of the latest updates to the language, and don't mind if the version available today occasionally doesn't actually work.Finally, you may also consider building Julia from source for yourself. This option is mainly for those individuals who are comfortable at the command line, or interested in learning. If this describes you, you may also be interested in reading our guidelines for contributing.Links to each of these download types can be found on the download page at https://julialang.org/downloads/. Note that not all versions of Julia are available for all platforms."
},

{
    "location": "manual/faq.html#When-are-deprecated-functions-removed?-1",
    "page": "Frequently Asked Questions",
    "title": "When are deprecated functions removed?",
    "category": "section",
    "text": "Deprecated functions are removed after the subsequent release. For example, functions marked as deprecated in the 0.1 release will not be available starting with the 0.2 release."
},

{
    "location": "manual/noteworthy-differences.html#",
    "page": "Noteworthy Differences from other Languages",
    "title": "Noteworthy Differences from other Languages",
    "category": "page",
    "text": ""
},

{
    "location": "manual/noteworthy-differences.html#Noteworthy-Differences-from-other-Languages-1",
    "page": "Noteworthy Differences from other Languages",
    "title": "Noteworthy Differences from other Languages",
    "category": "section",
    "text": ""
},

{
    "location": "manual/noteworthy-differences.html#Noteworthy-differences-from-MATLAB-1",
    "page": "Noteworthy Differences from other Languages",
    "title": "Noteworthy differences from MATLAB",
    "category": "section",
    "text": "Although MATLAB users may find Julia's syntax familiar, Julia is not a MATLAB clone. There are major syntactic and functional differences. The following are some noteworthy differences that may trip up Julia users accustomed to MATLAB:Julia arrays are indexed with square brackets, A[i,j].\nJulia arrays are assigned by reference. After A=B, changing elements of B will modify A as well.\nJulia values are passed and assigned by reference. If a function modifies an array, the changes will be visible in the caller.\nJulia does not automatically grow arrays in an assignment statement. Whereas in MATLAB a(4) = 3.2 can create the array a = [0 0 0 3.2] and a(5) = 7 can grow it into a = [0 0 0 3.2 7], the corresponding Julia statement a[5] = 7 throws an error if the length of a is less than 5 or if this statement is the first use of the identifier a. Julia has push!() and append!(), which grow Vectors much more efficiently than MATLAB's a(end+1) = val.\nThe imaginary unit sqrt(-1) is represented in Julia as im, not i or j as in MATLAB.\nIn Julia, literal numbers without a decimal point (such as 42) create integers instead of floating point numbers. Arbitrarily large integer literals are supported. As a result, some operations such as 2^-1 will throw a domain error as the result is not an integer (see the FAQ entry on domain errors for details).\nIn Julia, multiple values are returned and assigned as tuples, e.g. (a, b) = (1, 2) or a, b = 1, 2. MATLAB's nargout, which is often used in MATLAB to do optional work based on the number of returned values, does not exist in Julia. Instead, users can use optional and keyword arguments to achieve similar capabilities.\nJulia has true one-dimensional arrays. Column vectors are of size N, not Nx1. For example, rand(N) makes a 1-dimensional array.\nIn Julia, [x,y,z] will always construct a 3-element array containing x, y and z.\nTo concatenate in the first (\"vertical\") dimension use either vcat(x,y,z) or separate with semicolons ([x; y; z]).\nTo concatenate in the second (\"horizontal\") dimension use either hcat(x,y,z) or separate with spaces ([x y z]).\nTo construct block matrices (concatenating in the first two dimensions), use either hvcat() or combine spaces and semicolons ([a b; c d]).\nIn Julia, a:b and a:b:c construct Range objects. To construct a full vector like in MATLAB, use collect(a:b). Generally, there is no need to call collect though. Range will act like a normal array in most cases but is more efficient because it lazily computes its values. This pattern of creating specialized objects instead of full arrays is used frequently, and is also seen in functions such as linspace, or with iterators such as enumerate, and zip. The special objects can mostly be used as if they were normal arrays.\nFunctions in Julia return values from their last expression or the return keyword instead of listing the names of variables to return in the function definition (see The return Keyword for details).\nA Julia script may contain any number of functions, and all definitions will be externally visible when the file is loaded. Function definitions can be loaded from files outside the current working directory.\nIn Julia, reductions such as sum(), prod(), and max() are performed over every element of an array when called with a single argument, as in sum(A), even if A has more than one dimension.\nIn Julia, functions such as sort() that operate column-wise by default (sort(A) is equivalent to sort(A,1)) do not have special behavior for 1xN arrays; the argument is returned unmodified since it still performs sort(A,1). To sort a 1xN matrix like a vector, use sort(A,2).\nIn Julia, if A is a 2-dimensional array, fft(A) computes a 2D FFT. In particular, it is not equivalent to fft(A,1), which computes a 1D FFT acting column-wise.\nIn Julia, parentheses must be used to call a function with zero arguments, like in tic() and toc().\nJulia discourages the used of semicolons to end statements. The results of statements are not automatically printed (except at the interactive prompt), and lines of code do not need to end with semicolons. println() or @printf() can be used to print specific output.\nIn Julia, if A and B are arrays, logical comparison operations like A == B do not return an array of booleans. Instead, use A .== B, and similarly for the other boolean operators like <, > and =.\nIn Julia, the operators &, |, and ⊻ (xor) perform the bitwise operations equivalent to and, or, and xor respectively in MATLAB, and have precedence similar to Python's bitwise operators (unlike C). They can operate on scalars or element-wise across arrays and can be used to combine logical arrays, but note the difference in order of operations: parentheses may be required (e.g., to select elements of A equal to 1 or 2 use (A .== 1) | (A .== 2)).\nIn Julia, the elements of a collection can be passed as arguments to a function using the splat operator ..., as in xs=[1,2]; f(xs...).\nJulia's svd() returns singular values as a vector instead of as a dense diagonal matrix.\nIn Julia, ... is not used to continue lines of code. Instead, incomplete expressions automatically continue onto the next line.\nIn both Julia and MATLAB, the variable ans is set to the value of the last expression issued in an interactive session. In Julia, unlike MATLAB, ans is not set when Julia code is run in non-interactive mode.\nJulia's types do not support dynamically adding fields at runtime, unlike MATLAB's classes. Instead, use a Dict.\nIn Julia each module has its own global scope/namespace, whereas in MATLAB there is just one global scope.\nIn MATLAB, an idiomatic way to remove unwanted values is to use logical indexing, like in the expression x(x>3) or in the statement x(x>3) = [] to modify x in-place. In contrast, Julia provides the higher order functions filter() and filter!(), allowing users to write filter(z->z>3, x) and filter!(z->z>3, x) as alternatives to the corresponding transliterations x[x.>3] and x = x[x.>3]. Using filter!() reduces the use of temporary arrays.\nThe analogue of extracting (or \"dereferencing\") all elements of a cell array, e.g. in vertcat(A{:}) in MATLAB, is written using the splat operator in Julia, e.g. as vcat(A...)."
},

{
    "location": "manual/noteworthy-differences.html#Noteworthy-differences-from-R-1",
    "page": "Noteworthy Differences from other Languages",
    "title": "Noteworthy differences from R",
    "category": "section",
    "text": "One of Julia's goals is to provide an effective language for data analysis and statistical programming. For users coming to Julia from R, these are some noteworthy differences:Julia's single quotes enclose characters, not strings.\nJulia can create substrings by indexing into strings. In R, strings must be converted into character vectors before creating substrings.\nIn Julia, like Python but unlike R, strings can be created with triple quotes \"\"\" ... \"\"\". This syntax is convenient for constructing strings that contain line breaks.\nIn Julia, varargs are specified using the splat operator ..., which always follows the name of a specific variable, unlike R, for which ... can occur in isolation.\nIn Julia, modulus is mod(a, b), not a %% b. % in Julia is the remainder operator.\nIn Julia, not all data structures support logical indexing. Furthermore, logical indexing in Julia is supported only with vectors of length equal to the object being indexed. For example:\nIn R, c(1, 2, 3, 4)[c(TRUE, FALSE)] is equivalent to c(1, 3).\nIn R, c(1, 2, 3, 4)[c(TRUE, FALSE, TRUE, FALSE)] is equivalent to c(1, 3).\nIn Julia, [1, 2, 3, 4][[true, false]] throws a BoundsError.\nIn Julia, [1, 2, 3, 4][[true, false, true, false]] produces [1, 3].\nLike many languages, Julia does not always allow operations on vectors of different lengths, unlike R where the vectors only need to share a common index range.  For example, c(1, 2, 3, 4) + c(1, 2) is valid R but the equivalent [1, 2, 3, 4] + [1, 2] will throw an error in Julia.\nJulia's map() takes the function first, then its arguments, unlike lapply(<structure>, function, ...) in R. Similarly Julia's equivalent of apply(X, MARGIN, FUN, ...) in R is mapslices() where the function is the first argument.\nMultivariate apply in R, e.g. mapply(choose, 11:13, 1:3), can be written as broadcast(binomial, 11:13, 1:3) in Julia. Equivalently Julia offers a shorter dot syntax for vectorizing functions binomial.(11:13, 1:3).\nJulia uses end to denote the end of conditional blocks, like if, loop blocks, like while/ for, and functions. In lieu of the one-line if ( cond ) statement, Julia allows statements of the form if cond; statement; end, cond && statement and !cond || statement. Assignment statements in the latter two syntaxes must be explicitly wrapped in parentheses, e.g. cond && (x = value).\nIn Julia, <-, <<- and -> are not assignment operators.\nJulia's -> creates an anonymous function, like Python.\nJulia constructs vectors using brackets. Julia's [1, 2, 3] is the equivalent of R's c(1, 2, 3).\nJulia's * operator can perform matrix multiplication, unlike in R. If A and B are matrices, then A * B denotes a matrix multiplication in Julia, equivalent to R's A %*% B. In R, this same notation would perform an element-wise (Hadamard) product. To get the element-wise multiplication operation, you need to write A .* B in Julia.\nJulia performs matrix transposition using the .' operator and conjugated transposition using the ' operator. Julia's A.' is therefore equivalent to R's t(A).\nJulia does not require parentheses when writing if statements or for/while loops: use for i in [1, 2, 3] instead of for (i in c(1, 2, 3)) and if i == 1 instead of if (i == 1).\nJulia does not treat the numbers 0 and 1 as Booleans. You cannot write if (1) in Julia, because if statements accept only booleans. Instead, you can write if true, if Bool(1), or if 1==1.\nJulia does not provide nrow and ncol. Instead, use size(M, 1) for nrow(M) and size(M, 2) for ncol(M).\nJulia is careful to distinguish scalars, vectors and matrices.  In R, 1 and c(1) are the same. In Julia, they cannot be used interchangeably.\nJulia's diag and diagm are not like R's.\nJulia cannot assign to the results of function calls on the left hand side of an assignment operation: you cannot write diag(M) = ones(n).\nJulia discourages populating the main namespace with functions. Most statistical functionality for Julia is found in packages under the JuliaStats organization. For example:\nFunctions pertaining to probability distributions are provided by the Distributions package.\nThe DataFrames package provides data frames.\nGeneralized linear models are provided by the GLM package.\nJulia provides tuples and real hash tables, but not R-style lists. When returning multiple items, you should typically use a tuple: instead of list(a = 1, b = 2), use (1, 2).\nJulia encourages users to write their own types, which are easier to use than S3 or S4 objects in R. Julia's multiple dispatch system means that table(x::TypeA) and table(x::TypeB) act like R's table.TypeA(x) and table.TypeB(x).\nIn Julia, values are passed and assigned by reference. If a function modifies an array, the changes will be visible in the caller. This is very different from R and allows new functions to operate on large data structures much more efficiently.\nIn Julia, vectors and matrices are concatenated using hcat(), vcat() and hvcat(), not c, rbind and cbind like in R.\nIn Julia, a range like a:b is not shorthand for a vector like in R, but is a specialized Range that is used for iteration without high memory overhead. To convert a range into a vector, use collect(a:b).\nJulia's max() and min() are the equivalent of pmax and pmin respectively in R, but both arguments need to have the same dimensions.  While maximum() and minimum() replace max and min in R, there are important differences.\nJulia's sum(), prod(), maximum(), and minimum() are different from their counterparts in R. They all accept one or two arguments. The first argument is an iterable collection such as an array.  If there is a second argument, then this argument indicates the dimensions, over which the operation is carried out.  For instance, let A=[[1 2],[3 4]] in Julia and B=rbind(c(1,2),c(3,4)) be the same matrix in R.  Then sum(A) gives the same result as sum(B), but sum(A, 1) is a row vector containing the sum over each column and sum(A, 2) is a column vector containing the sum over each row.  This contrasts to the behavior of R, where sum(B,1)=11 and sum(B,2)=12.  If the second argument is a vector, then it specifies all the dimensions over which the sum is performed, e.g., sum(A,[1,2])=10.  It should be noted that there is no error checking regarding the second argument.\nJulia has several functions that can mutate their arguments. For example, it has both sort() and sort!().\nIn R, performance requires vectorization. In Julia, almost the opposite is true: the best performing code is often achieved by using devectorized loops.\nJulia is eagerly evaluated and does not support R-style lazy evaluation. For most users, this means that there are very few unquoted expressions or column names.\nJulia does not support the NULL type.\nJulia lacks the equivalent of R's assign or get.\nIn Julia, return does not require parentheses.\nIn R, an idiomatic way to remove unwanted values is to use logical indexing, like in the expression x[x>3] or in the statement x = x[x>3] to modify x in-place. In contrast, Julia provides the higher order functions filter() and filter!(), allowing users to write filter(z->z>3, x) and filter!(z->z>3, x) as alternatives to the corresponding transliterations x[x.>3] and x = x[x.>3]. Using filter!() reduces the use of temporary arrays."
},

{
    "location": "manual/noteworthy-differences.html#Noteworthy-differences-from-Python-1",
    "page": "Noteworthy Differences from other Languages",
    "title": "Noteworthy differences from Python",
    "category": "section",
    "text": "Julia requires end to end a block. Unlike Python, Julia has no pass keyword.\nIn Julia, indexing of arrays, strings, etc. is 1-based not 0-based.\nJulia's slice indexing includes the last element, unlike in Python. a[2:3] in Julia is a[1:3] in Python.\nJulia does not support negative indexes. In particular, the last element of a list or array is indexed with end in Julia, not -1 as in Python.\nJulia's for, if, while, etc. blocks are terminated by the end keyword. Indentation level is not significant as it is in Python.\nJulia has no line continuation syntax: if, at the end of a line, the input so far is a complete expression, it is considered done; otherwise the input continues. One way to force an expression to continue is to wrap it in parentheses.\nJulia arrays are column major (Fortran ordered) whereas NumPy arrays are row major (C-ordered) by default. To get optimal performance when looping over arrays, the order of the loops should be reversed in Julia relative to NumPy (see relevant section of Performance Tips).\nJulia's updating operators (e.g. +=, -=, ...) are not in-place whereas NumPy's are. This means A = ones(4); B = A; B += 3 doesn't change values in A, it rather rebinds the name B to the result of the right- hand side B = B + 3, which is a new array. Use B[:] += 3, explicit loops, or InplaceOps.jl.\nJulia evaluates default values of function arguments every time the method is invoked, unlike in Python where the default values are evaluated only once when the function is defined. For example, the function f(x=rand()) = x returns a new random number every time it is invoked without argument. On the other hand, the function g(x=[1,2]) = push!(x,3) returns [1,2,3] every time it is called as g().\nIn Julia % is the remainder operator, whereas in Python it is the modulus."
},

{
    "location": "manual/noteworthy-differences.html#Noteworthy-differences-from-C/C-1",
    "page": "Noteworthy Differences from other Languages",
    "title": "Noteworthy differences from C/C++",
    "category": "section",
    "text": "Julia arrays are indexed with square brackets, and can have more than one dimension A[i,j]. This syntax is not just syntactic sugar for a reference to a pointer or address as in C/C++. See the Julia documentation for the syntax for array construction (it has changed between versions).\nIn Julia, indexing of arrays, strings, etc. is 1-based not 0-based.\nJulia arrays are assigned by reference. After A=B, changing elements of B will modify A as well. Updating operators like += do not operate in-place, they are equivalent to A = A + B which rebinds the left-hand side to the result of the right-hand side expression.\nJulia arrays are column major (Fortran ordered) whereas C/C++ arrays are row major ordered by default. To get optimal performance when looping over arrays, the order of the loops should be reversed in Julia relative to C/C++ (see relevant section of Performance Tips).\nJulia values are passed and assigned by reference. If a function modifies an array, the changes will be visible in the caller.\nIn Julia, whitespace is significant, unlike C/C++, so care must be taken when adding/removing whitespace from a Julia program.\nIn Julia, literal numbers without a decimal point (such as 42) create signed integers, of type Int, but literals too large to fit in the machine word size will automatically be promoted to a larger size type, such as Int64 (if Int is Int32), Int128, or the arbitrarily large BigInt type. There are no numeric literal suffixes, such as L, LL, U, UL, ULL to indicate unsigned and/or signed vs. unsigned. Decimal literals are always signed, and hexadecimal literals (which start with 0x like C/C++), are unsigned. Hexadecimal literals also, unlike C/C++/Java and unlike decimal literals in Julia, have a type based on the length of the literal, including leading 0s. For example, 0x0 and 0x00 have type UInt8, 0x000 and 0x0000 have type UInt16, then literals with 5 to 8 hex digits have type UInt32, 9 to 16 hex digits type UInt64 and 17 to 32 hex digits type UInt128. This needs to be taken into account when defining hexadecimal masks, for example ~0xf == 0xf0 is very different from ~0x000f == 0xfff0. 64 bit Float64 and 32 bit Float32 bit literals are expressed as 1.0 and 1.0f0 respectively. Floating point literals are rounded (and not promoted to the BigFloat type) if they can not be exactly represented.  Floating point literals are closer in behavior to C/C++. Octal (prefixed with 0o) and binary (prefixed with 0b) literals are also treated as unsigned.\nString literals can be delimited with either \"  or \"\"\", \"\"\" delimited literals can contain \" characters without quoting it like \"\\\"\" String literals can have values of other variables or expressions interpolated into them, indicated by $variablename or $(expression), which evaluates the variable name or the expression in the context of the function.\n// indicates a Rational number, and not a single-line comment (which is # in Julia)\n#= indicates the start of a multiline comment, and =# ends it.\nFunctions in Julia return values from their last expression(s) or the return keyword.  Multiple values can be returned from functions and assigned as tuples, e.g. (a, b) = myfunction() or a, b = myfunction(), instead of having to pass pointers to values as one would have to do in C/C++ (i.e. a = myfunction(&b).\nJulia does not require the use of semicolons to end statements. The results of expressions are not automatically printed (except at the interactive prompt, i.e. the REPL), and lines of code do not need to end with semicolons. println() or @printf() can be used to print specific output. In the REPL, ; can be used to suppress output. ; also has a different meaning within [ ], something to watch out for. ; can be used to separate expressions on a single line, but are not strictly necessary in many cases, and are more an aid to readability.\nIn Julia, the operator ⊻ (xor) performs the bitwise XOR operation, i.e. ^ in C/C++.  Also, the bitwise operators do not have the same precedence as C/++, so parenthesis may be required.\nJulia's ^ is exponentiation (pow), not bitwise XOR as in C/C++ (use ⊻, or xor, in Julia)\nJulia has two right-shift operators, >> and >>>.  >>> performs an arithmetic shift, >> always performs a logical shift, unlike C/C++, where the meaning of >> depends on the type of the value being shifted.\nJulia's -> creates an anonymous function, it does not access a member via a pointer.\nJulia does not require parentheses when writing if statements or for/while loops: use for i in [1, 2, 3] instead of for (int i=1; i <= 3; i++) and if i == 1 instead of if (i == 1).\nJulia does not treat the numbers 0 and 1 as Booleans. You cannot write if (1) in Julia, because if statements accept only booleans. Instead, you can write if true, if Bool(1), or if 1==1.\nJulia uses end to denote the end of conditional blocks, like if, loop blocks, like while/ for, and functions. In lieu of the one-line if ( cond ) statement, Julia allows statements of the form if cond; statement; end, cond && statement and !cond || statement. Assignment statements in the latter two syntaxes must be explicitly wrapped in parentheses, e.g. cond && (x = value), because of the operator precedence.\nJulia has no line continuation syntax: if, at the end of a line, the input so far is a complete expression, it is considered done; otherwise the input continues. One way to force an expression to continue is to wrap it in parentheses.\nJulia macros operate on parsed expressions, rather than the text of the program, which allows them to perform sophisticated transformations of Julia code. Macro names start with the @ character, and have both a function-like syntax, @mymacro(arg1, arg2, arg3), and a statement-like syntax, @mymacro arg1 arg2 arg3. The forms are interchangable; the function-like form is particularly useful if the macro appears within another expression, and is often clearest. The statement-like form is often used to annotate blocks, as in the parallel for construct: @parallel for i in 1:n; #= body =#; end. Where the end of the macro construct may be unclear, use the function-like form.\nJulia now has an enumeration type, expressed using the macro @enum(name, value1, value2, ...) For example: @enum(Fruit, banana=1, apple, pear)\nBy convention, functions that modify their arguments have a ! at the end of the name, for example push!.\nIn C++, by default, you have static dispatch, i.e. you need to annotate a function as virtual, in order to have dynamic dispatch. On the other hand, in Julia every method is \"virtual\" (although it's more general than that since methods are dispatched on every argument type, not only this, using the most-specific-declaration rule)."
},

{
    "location": "manual/unicode-input.html#",
    "page": "Unicode Input",
    "title": "Unicode Input",
    "category": "page",
    "text": ""
},

{
    "location": "manual/unicode-input.html#Unicode-Input-1",
    "page": "Unicode Input",
    "title": "Unicode Input",
    "category": "section",
    "text": "The following table lists Unicode characters that can be entered via tab completion of LaTeX-like abbreviations in the Julia REPL (and in various other editing environments).  You can also get information on how to type a symbol by entering it in the REPL help, i.e. by typing ? and then entering the symbol in the REPL (e.g., by copy-paste from somewhere you saw the symbol).warning: Warning\nThis table may appear to contain missing characters in the second column, or even show characters that are inconsistent with the characters as they are rendered in the Julia REPL. In these cases, users are strongly advised to check their choice of fonts in their browser and REPL environment, as there are known issues with glyphs in many fonts.#\n# Generate a table containing all LaTeX and Emoji tab completions available in the REPL.\n#\n\nfunction tab_completions(symbols...)\n    completions = Dict{String, Vector{String}}()\n    for each in symbols, (k, v) in each\n        completions[v] = push!(get!(completions, v, String[]), k)\n    end\n    return completions\nend\n\nfunction unicode_data()\n    file = normpath(JULIA_HOME, \"..\", \"..\", \"doc\", \"UnicodeData.txt\")\n    names = Dict{UInt32, String}()\n    open(file) do unidata\n        for line in readlines(unidata)\n            id, name, desc = split(line, \";\")[[1, 2, 11]]\n            codepoint = parse(UInt32, \"0x$id\")\n            names[codepoint] = titlecase(lowercase(name == \"\" ? desc : desc == \"\" ? name : \"$name / $desc\"))\n        end\n    end\n    return names\nend\n\n# Prepend a dotted circle ('◌' i.e. '\\u25CC') to combining characters\nfunction fix_combining_chars(char)\n    cat = Base.UTF8proc.category_code(char)\n    return string(cat == 6 || cat == 8 ? \"◌\" : \"\", char)\nend\n\n\nfunction table_entries(completions, unicode_dict)\n    entries = [[\n        \"Code point(s)\", \"Character(s)\",\n        \"Tab completion sequence(s)\", \"Unicode name(s)\"\n    ]]\n    for (chars, inputs) in sort!(collect(completions), by = first)\n        code_points, unicode_names, characters = String[], String[], String[]\n        for char in chars\n            push!(code_points, \"U+$(uppercase(hex(char, 5)))\")\n            push!(unicode_names, get(unicode_dict, UInt32(char), \"(No Unicode name)\"))\n            push!(characters, isempty(characters) ? fix_combining_chars(char) : \"$char\")\n        end\n        push!(entries, [\n            join(code_points, \" + \"), join(characters),\n            join(inputs, \", \"), join(unicode_names, \" + \")\n        ])\n    end\n    return Markdown.Table(entries, [:l, :l, :l, :l])\nend\n\ntable_entries(\n    tab_completions(\n        Base.REPLCompletions.latex_symbols,\n        Base.REPLCompletions.emoji_symbols\n    ),\n    unicode_data()\n)"
},

{
    "location": "stdlib/base.html#",
    "page": "Essentials",
    "title": "Essentials",
    "category": "page",
    "text": ""
},

{
    "location": "stdlib/base.html#Essentials-1",
    "page": "Essentials",
    "title": "Essentials",
    "category": "section",
    "text": ""
},

{
    "location": "stdlib/base.html#Introduction-1",
    "page": "Essentials",
    "title": "Introduction",
    "category": "section",
    "text": "The Julia standard library contains a range of functions and macros appropriate for performing scientific and numerical computing, but is also as broad as those of many general purpose programming languages.  Additional functionality is available from a growing collection of available packages. Functions are grouped by topic below.Some general notes:Except for functions in built-in modules (Pkg, Collections, Test and Profile), all functions documented here are directly available for use in programs.\nTo use module functions, use import Module to import the module, and Module.fn(x) to use the functions.\nAlternatively, using Module will import all exported Module functions into the current namespace.\nBy convention, function names ending with an exclamation point (!) modify their arguments. Some functions have both modifying (e.g., sort!) and non-modifying (sort) versions."
},

{
    "location": "stdlib/base.html#Base.exit",
    "page": "Essentials",
    "title": "Base.exit",
    "category": "Function",
    "text": "exit([code])\n\nQuit (or control-D at the prompt). The default exit code is zero, indicating that the processes completed successfully.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.quit",
    "page": "Essentials",
    "title": "Base.quit",
    "category": "Function",
    "text": "quit()\n\nQuit the program indicating that the processes completed successfully. This function calls exit(0) (see exit).\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.atexit",
    "page": "Essentials",
    "title": "Base.atexit",
    "category": "Function",
    "text": "atexit(f)\n\nRegister a zero-argument function f() to be called at process exit. atexit() hooks are called in last in first out (LIFO) order and run before object finalizers.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.atreplinit",
    "page": "Essentials",
    "title": "Base.atreplinit",
    "category": "Function",
    "text": "atreplinit(f)\n\nRegister a one-argument function to be called before the REPL interface is initialized in interactive sessions; this is useful to customize the interface. The argument of f is the REPL object. This function should be called from within the .juliarc.jl initialization file.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.isinteractive",
    "page": "Essentials",
    "title": "Base.isinteractive",
    "category": "Function",
    "text": "isinteractive() -> Bool\n\nDetermine whether Julia is running an interactive session.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.whos",
    "page": "Essentials",
    "title": "Base.whos",
    "category": "Function",
    "text": "whos(io::IO=STDOUT, m::Module=current_module(), pattern::Regex=r\"\")\n\nPrint information about exported global variables in a module, optionally restricted to those matching pattern.\n\nThe memory consumption estimate is an approximate lower bound on the size of the internal structure of the object.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.summarysize",
    "page": "Essentials",
    "title": "Base.summarysize",
    "category": "Function",
    "text": "Base.summarysize(obj; exclude=Union{...}, chargeall=Union{...}) -> Int\n\nCompute the amount of memory used by all unique objects reachable from the argument.\n\nKeyword Arguments\n\nexclude: specifies the types of objects to exclude from the traversal.\nchargeall: specifies the types of objects to always charge the size of all of their fields, even if those fields would normally be excluded.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.edit-Tuple{AbstractString,Integer}",
    "page": "Essentials",
    "title": "Base.edit",
    "category": "Method",
    "text": "edit(path::AbstractString, line::Integer=0)\n\nEdit a file or directory optionally providing a line number to edit the file at. Returns to the julia prompt when you quit the editor. The editor can be changed by setting JULIA_EDITOR, VISUAL or EDITOR as an environment variable.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.edit-Tuple{Any}",
    "page": "Essentials",
    "title": "Base.edit",
    "category": "Method",
    "text": "edit(function, [types])\n\nEdit the definition of a function, optionally specifying a tuple of types to indicate which method to edit. The editor can be changed by setting JULIA_EDITOR, VISUAL or EDITOR as an environment variable.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@edit",
    "page": "Essentials",
    "title": "Base.@edit",
    "category": "Macro",
    "text": "@edit\n\nEvaluates the arguments to the function or macro call, determines their types, and calls the edit function on the resulting expression.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.less-Tuple{AbstractString}",
    "page": "Essentials",
    "title": "Base.less",
    "category": "Method",
    "text": "less(file::AbstractString, [line::Integer])\n\nShow a file using the default pager, optionally providing a starting line number. Returns to the julia prompt when you quit the pager.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.less-Tuple{Any}",
    "page": "Essentials",
    "title": "Base.less",
    "category": "Method",
    "text": "less(function, [types])\n\nShow the definition of a function using the default pager, optionally specifying a tuple of types to indicate which method to see.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@less",
    "page": "Essentials",
    "title": "Base.@less",
    "category": "Macro",
    "text": "@less\n\nEvaluates the arguments to the function or macro call, determines their types, and calls the less function on the resulting expression.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.clipboard-Tuple{Any}",
    "page": "Essentials",
    "title": "Base.clipboard",
    "category": "Method",
    "text": "clipboard(x)\n\nSend a printed form of x to the operating system clipboard (\"copy\").\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.clipboard-Tuple{}",
    "page": "Essentials",
    "title": "Base.clipboard",
    "category": "Method",
    "text": "clipboard() -> AbstractString\n\nReturn a string with the contents of the operating system clipboard (\"paste\").\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.reload",
    "page": "Essentials",
    "title": "Base.reload",
    "category": "Function",
    "text": "reload(name::AbstractString)\n\nForce reloading of a package, even if it has been loaded before. This is intended for use during package development as code is modified.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.require",
    "page": "Essentials",
    "title": "Base.require",
    "category": "Function",
    "text": "require(module::Symbol)\n\nThis function is part of the implementation of using / import, if a module is not already defined in Main. It can also be called directly to force reloading a module, regardless of whether it has been loaded before (for example, when interactively developing libraries).\n\nLoads a source file, in the context of the Main module, on every active node, searching standard locations for files. require is considered a top-level operation, so it sets the current include path but does not use it to search for files (see help for include). This function is typically used to load library code, and is implicitly called by using to load packages.\n\nWhen searching for files, require first looks for package code under Pkg.dir(), then tries paths in the global array LOAD_PATH. require is case-sensitive on all platforms, including those with case-insensitive filesystems like macOS and Windows.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.compilecache",
    "page": "Essentials",
    "title": "Base.compilecache",
    "category": "Function",
    "text": "Base.compilecache(module::String)\n\nCreates a precompiled cache file for a module and all of its dependencies. This can be used to reduce package load times. Cache files are stored in LOAD_CACHE_PATH[1], which defaults to ~/.julia/lib/VERSION. See Module initialization and precompilation for important notes.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.__precompile__",
    "page": "Essentials",
    "title": "Base.__precompile__",
    "category": "Function",
    "text": "__precompile__(isprecompilable::Bool=true)\n\nSpecify whether the file calling this function is precompilable. If isprecompilable is true, then __precompile__ throws an exception when the file is loaded by using/import/require unless the file is being precompiled, and in a module file it causes the module to be automatically precompiled when it is imported. Typically, __precompile__() should occur before the module declaration in the file, or better yet VERSION >= v\"0.4\" && __precompile__() in order to be backward-compatible with Julia 0.3.\n\nIf a module or file is not safely precompilable, it should call __precompile__(false) in order to throw an error if Julia attempts to precompile it.\n\n__precompile__() should not be used in a module unless all of its dependencies are also using __precompile__(). Failure to do so can result in a runtime error when loading the module.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.include",
    "page": "Essentials",
    "title": "Base.include",
    "category": "Function",
    "text": "include(path::AbstractString)\n\nEvaluate the contents of the input source file in the current context. Returns the result of the last evaluated expression of the input file. During including, a task-local include path is set to the directory containing the file. Nested calls to include will search relative to that path. All paths refer to files on node 1 when running in parallel, and files will be fetched from node 1. This function is typically used to load source interactively, or to combine files in packages that are broken into multiple source files.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.include_string",
    "page": "Essentials",
    "title": "Base.include_string",
    "category": "Function",
    "text": "include_string(code::AbstractString, filename::AbstractString=\"string\")\n\nLike include, except reads code from the given string rather than from a file. Since there is no file path involved, no path processing or fetching from node 1 is done.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.include_dependency",
    "page": "Essentials",
    "title": "Base.include_dependency",
    "category": "Function",
    "text": "include_dependency(path::AbstractString)\n\nIn a module, declare that the file specified by path (relative or absolute) is a dependency for precompilation; that is, the module will need to be recompiled if this file changes.\n\nThis is only needed if your module depends on a file that is not used via include. It has no effect outside of compilation.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.Docs.apropos",
    "page": "Essentials",
    "title": "Base.Docs.apropos",
    "category": "Function",
    "text": "apropos(string)\n\nSearch through all documentation for a string, ignoring case.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.which-Tuple{Any,Any}",
    "page": "Essentials",
    "title": "Base.which",
    "category": "Method",
    "text": "which(f, types)\n\nReturns the method of f (a Method object) that would be called for arguments of the given types.\n\nIf types is an abstract type, then the method that would be called by invoke is returned.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.which-Tuple{Symbol}",
    "page": "Essentials",
    "title": "Base.which",
    "category": "Method",
    "text": "which(symbol)\n\nReturn the module in which the binding for the variable referenced by symbol was created.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@which",
    "page": "Essentials",
    "title": "Base.@which",
    "category": "Macro",
    "text": "@which\n\nApplied to a function or macro call, it evaluates the arguments to the specified call, and returns the Method object for the method that would be called for those arguments. Applied to a variable, it returns the module in which the variable was bound. It calls out to the which function.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.methods",
    "page": "Essentials",
    "title": "Base.methods",
    "category": "Function",
    "text": "methods(f, [types])\n\nReturns the method table for f.\n\nIf types is specified, returns an array of methods whose types match.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.methodswith",
    "page": "Essentials",
    "title": "Base.methodswith",
    "category": "Function",
    "text": "methodswith(typ[, module or function][, showparents::Bool=false])\n\nReturn an array of methods with an argument of type typ.\n\nThe optional second argument restricts the search to a particular module or function (the default is all modules, starting from Main).\n\nIf optional showparents is true, also return arguments with a parent type of typ, excluding type Any.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@show",
    "page": "Essentials",
    "title": "Base.@show",
    "category": "Macro",
    "text": "@show\n\nShow an expression and result, returning the result.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.versioninfo",
    "page": "Essentials",
    "title": "Base.versioninfo",
    "category": "Function",
    "text": "versioninfo(io::IO=STDOUT, verbose::Bool=false)\n\nPrint information about the version of Julia in use. If the verbose argument is true, detailed system information is shown as well.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.workspace",
    "page": "Essentials",
    "title": "Base.workspace",
    "category": "Function",
    "text": "workspace()\n\nReplace the top-level module (Main) with a new one, providing a clean workspace. The previous Main module is made available as LastMain. A previously-loaded package can be accessed using a statement such as using LastMain.Package.\n\nThis function should only be used interactively.\n\n\n\n"
},

{
    "location": "stdlib/base.html#ans",
    "page": "Essentials",
    "title": "ans",
    "category": "Keyword",
    "text": "ans\n\nA variable referring to the last computed value, automatically set at the interactive prompt.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Getting-Around-1",
    "page": "Essentials",
    "title": "Getting Around",
    "category": "section",
    "text": "Base.exit\nBase.quit\nBase.atexit\nBase.atreplinit\nBase.isinteractive\nBase.whos\nBase.summarysize\nBase.edit(::AbstractString, ::Integer)\nBase.edit(::Any)\nBase.@edit\nBase.less(::AbstractString)\nBase.less(::Any)\nBase.@less\nBase.clipboard(::Any)\nBase.clipboard()\nBase.reload\nBase.require\nBase.compilecache\nBase.__precompile__\nBase.include\nBase.include_string\nBase.include_dependency\nBase.Docs.apropos\nBase.which(::Any, ::Any)\nBase.which(::Symbol)\nBase.@which\nBase.methods\nBase.methodswith\nBase.@show\nBase.versioninfo\nBase.workspace\nans"
},

{
    "location": "stdlib/base.html#Core.:===",
    "page": "Essentials",
    "title": "Core.:===",
    "category": "Function",
    "text": "===(x,y) -> Bool\n≡(x,y) -> Bool\n\nDetermine whether x and y are identical, in the sense that no program could distinguish them. Compares mutable objects by address in memory, and compares immutable objects (such as numbers) by contents at the bit level. This function is sometimes called egal.\n\njulia> a = [1, 2]; b = [1, 2];\n\njulia> a == b\ntrue\n\njulia> a === b\nfalse\n\njulia> a === a\ntrue\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.isa",
    "page": "Essentials",
    "title": "Core.isa",
    "category": "Function",
    "text": "isa(x, type) -> Bool\n\nDetermine whether x is of the given type. Can also be used as an infix operator, e.g. x isa type.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.isequal-Tuple{Any,Any}",
    "page": "Essentials",
    "title": "Base.isequal",
    "category": "Method",
    "text": "isequal(x, y)\n\nSimilar to ==, except treats all floating-point NaN values as equal to each other, and treats -0.0 as unequal to 0.0. The default implementation of isequal calls ==, so if you have a type that doesn't have these floating-point subtleties then you probably only need to define ==.\n\nisequal is the comparison function used by hash tables (Dict). isequal(x,y) must imply that hash(x) == hash(y).\n\nThis typically means that if you define your own == function then you must define a corresponding hash (and vice versa). Collections typically implement isequal by calling isequal recursively on all contents.\n\nScalar types generally do not need to implement isequal separate from ==, unless they represent floating-point numbers amenable to a more efficient implementation than that provided as a generic fallback (based on isnan, signbit, and ==).\n\njulia> isequal([1., NaN], [1., NaN])\ntrue\n\njulia> [1., NaN] == [1., NaN]\nfalse\n\njulia> 0.0 == -0.0\ntrue\n\njulia> isequal(0.0, -0.0)\nfalse\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.isequal-Tuple{Nullable,Nullable}",
    "page": "Essentials",
    "title": "Base.isequal",
    "category": "Method",
    "text": "isequal(x, y)\n\nSimilar to ==, except treats all floating-point NaN values as equal to each other, and treats -0.0 as unequal to 0.0. The default implementation of isequal calls ==, so if you have a type that doesn't have these floating-point subtleties then you probably only need to define ==.\n\nisequal is the comparison function used by hash tables (Dict). isequal(x,y) must imply that hash(x) == hash(y).\n\nThis typically means that if you define your own == function then you must define a corresponding hash (and vice versa). Collections typically implement isequal by calling isequal recursively on all contents.\n\nScalar types generally do not need to implement isequal separate from ==, unless they represent floating-point numbers amenable to a more efficient implementation than that provided as a generic fallback (based on isnan, signbit, and ==).\n\njulia> isequal([1., NaN], [1., NaN])\ntrue\n\njulia> [1., NaN] == [1., NaN]\nfalse\n\njulia> 0.0 == -0.0\ntrue\n\njulia> isequal(0.0, -0.0)\nfalse\n\n\n\nisequal(x::Nullable, y::Nullable)\n\nIf neither x nor y is null, compare them according to their values (i.e. isequal(get(x), get(y))). Else, return true if both arguments are null, and false if one is null but not the other: nulls are considered equal.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.isless",
    "page": "Essentials",
    "title": "Base.isless",
    "category": "Function",
    "text": "isless(x, y)\n\nTest whether x is less than y, according to a canonical total order. Values that are normally unordered, such as NaN, are ordered in an arbitrary but consistent fashion. This is the default comparison used by sort. Non-numeric types with a canonical total order should implement this function. Numeric types only need to implement it if they have special values such as NaN.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.isless-Tuple{Nullable,Nullable}",
    "page": "Essentials",
    "title": "Base.isless",
    "category": "Method",
    "text": "isless(x::Nullable, y::Nullable)\n\nIf neither x nor y is null, compare them according to their values (i.e. isless(get(x), get(y))). Else, return true if only y is null, and false otherwise: nulls are always considered greater than non-nulls, but not greater than another null.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.ifelse",
    "page": "Essentials",
    "title": "Base.ifelse",
    "category": "Function",
    "text": "ifelse(condition::Bool, x, y)\n\nReturn x if condition is true, otherwise return y. This differs from ? or if in that it is an ordinary function, so all the arguments are evaluated first. In some cases, using ifelse instead of an if statement can eliminate the branch in generated code and provide higher performance in tight loops.\n\njulia> ifelse(1 > 2, 1, 2)\n2\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.lexcmp",
    "page": "Essentials",
    "title": "Base.lexcmp",
    "category": "Function",
    "text": "lexcmp(x, y)\n\nCompare x and y lexicographically and return -1, 0, or 1 depending on whether x is less than, equal to, or greater than y, respectively. This function should be defined for lexicographically comparable types, and lexless will call lexcmp by default.\n\njulia> lexcmp(\"abc\", \"abd\")\n-1\n\njulia> lexcmp(\"abc\", \"abc\")\n0\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.lexless",
    "page": "Essentials",
    "title": "Base.lexless",
    "category": "Function",
    "text": "lexless(x, y)\n\nDetermine whether x is lexicographically less than y.\n\njulia> lexless(\"abc\", \"abd\")\ntrue\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.typeof",
    "page": "Essentials",
    "title": "Core.typeof",
    "category": "Function",
    "text": "typeof(x)\n\nGet the concrete type of x.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.tuple",
    "page": "Essentials",
    "title": "Core.tuple",
    "category": "Function",
    "text": "tuple(xs...)\n\nConstruct a tuple of the given objects.\n\nExample\n\njulia> tuple(1, 'a', pi)\n(1, 'a', π = 3.1415926535897...)\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.ntuple",
    "page": "Essentials",
    "title": "Base.ntuple",
    "category": "Function",
    "text": "ntuple(f::Function, n::Integer)\n\nCreate a tuple of length n, computing each element as f(i), where i is the index of the element.\n\njulia> ntuple(i -> 2*i, 4)\n(2, 4, 6, 8)\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.object_id",
    "page": "Essentials",
    "title": "Base.object_id",
    "category": "Function",
    "text": "object_id(x)\n\nGet a hash value for x based on object identity. object_id(x)==object_id(y) if x === y.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.hash",
    "page": "Essentials",
    "title": "Base.hash",
    "category": "Function",
    "text": "hash(x[, h::UInt])\n\nCompute an integer hash code such that isequal(x,y) implies hash(x)==hash(y). The optional second argument h is a hash code to be mixed with the result.\n\nNew types should implement the 2-argument form, typically by calling the 2-argument hash method recursively in order to mix hashes of the contents with each other (and with h). Typically, any type that implements hash should also implement its own == (hence isequal) to guarantee the property mentioned above.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.finalizer",
    "page": "Essentials",
    "title": "Base.finalizer",
    "category": "Function",
    "text": "finalizer(x, f)\n\nRegister a function f(x) to be called when there are no program-accessible references to x. The type of x must be a mutable struct, otherwise the behavior of this function is unpredictable.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.finalize",
    "page": "Essentials",
    "title": "Base.finalize",
    "category": "Function",
    "text": "finalize(x)\n\nImmediately run finalizers registered for object x.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.copy",
    "page": "Essentials",
    "title": "Base.copy",
    "category": "Function",
    "text": "copy(x)\n\nCreate a shallow copy of x: the outer structure is copied, but not all internal values. For example, copying an array produces a new array with identically-same elements as the original.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.deepcopy",
    "page": "Essentials",
    "title": "Base.deepcopy",
    "category": "Function",
    "text": "deepcopy(x)\n\nCreate a deep copy of x: everything is copied recursively, resulting in a fully independent object. For example, deep-copying an array produces a new array whose elements are deep copies of the original elements. Calling deepcopy on an object should generally have the same effect as serializing and then deserializing it.\n\nAs a special case, functions can only be actually deep-copied if they are anonymous, otherwise they are just copied. The difference is only relevant in the case of closures, i.e. functions which may contain hidden internal references.\n\nWhile it isn't normally necessary, user-defined types can override the default deepcopy behavior by defining a specialized version of the function deepcopy_internal(x::T, dict::ObjectIdDict) (which shouldn't otherwise be used), where T is the type to be specialized for, and dict keeps track of objects copied so far within the recursion. Within the definition, deepcopy_internal should be used in place of deepcopy, and the dict variable should be updated as appropriate before returning.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.isdefined",
    "page": "Essentials",
    "title": "Core.isdefined",
    "category": "Function",
    "text": "isdefined([m::Module,] s::Symbol)\nisdefined(object, s::Symbol)\nisdefined(object, index::Int)\n\nTests whether an assignable location is defined. The arguments can be a module and a symbol or a composite object and field name (as a symbol) or index. With a single symbol argument, tests whether a global variable with that name is defined in current_module().\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.convert",
    "page": "Essentials",
    "title": "Base.convert",
    "category": "Function",
    "text": "convert(T, x)\n\nConvert x to a value of type T.\n\nIf T is an Integer type, an InexactError will be raised if x is not representable by T, for example if x is not integer-valued, or is outside the range supported by T.\n\nExamples\n\njulia> convert(Int, 3.0)\n3\n\njulia> convert(Int, 3.5)\nERROR: InexactError()\nStacktrace:\n [1] convert(::Type{Int64}, ::Float64) at ./float.jl:679\n\nIf T is a AbstractFloat or Rational type, then it will return the closest value to x representable by T.\n\njulia> x = 1/3\n0.3333333333333333\n\njulia> convert(Float32, x)\n0.33333334f0\n\njulia> convert(Rational{Int32}, x)\n1//3\n\njulia> convert(Rational{Int64}, x)\n6004799503160661//18014398509481984\n\nIf T is a collection type and x a collection, the result of convert(T, x) may alias x.\n\njulia> x = Int[1,2,3];\n\njulia> y = convert(Vector{Int}, x);\n\njulia> y === x\ntrue\n\nSimilarly, if T is a composite type and x a related instance, the result of convert(T, x) may alias part or all of x.\n\njulia> x = speye(5);\n\njulia> typeof(x)\nSparseMatrixCSC{Float64,Int64}\n\njulia> y = convert(SparseMatrixCSC{Float64,Int64}, x);\n\njulia> z = convert(SparseMatrixCSC{Float32,Int64}, y);\n\njulia> y === x\ntrue\n\njulia> z === x\nfalse\n\njulia> z.colptr === x.colptr\ntrue\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.promote",
    "page": "Essentials",
    "title": "Base.promote",
    "category": "Function",
    "text": "promote(xs...)\n\nConvert all arguments to their common promotion type (if any), and return them all (as a tuple).\n\nExample\n\njulia> promote(Int8(1), Float16(4.5), Float32(4.1))\n(1.0f0, 4.5f0, 4.1f0)\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.oftype",
    "page": "Essentials",
    "title": "Base.oftype",
    "category": "Function",
    "text": "oftype(x, y)\n\nConvert y to the type of x (convert(typeof(x), y)).\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.widen",
    "page": "Essentials",
    "title": "Base.widen",
    "category": "Function",
    "text": "widen(x)\n\nIf x is a type, return a \"larger\" type (for numeric types, this will be a type with at least as much range and precision as the argument, and usually more). Otherwise x is converted to widen(typeof(x)).\n\nExamples\n\njulia> widen(Int32)\nInt64\n\njulia> widen(1.5f0)\n1.5\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.identity",
    "page": "Essentials",
    "title": "Base.identity",
    "category": "Function",
    "text": "identity(x)\n\nThe identity function. Returns its argument.\n\njulia> identity(\"Well, what did you expect?\")\n\"Well, what did you expect?\"\n\n\n\n"
},

{
    "location": "stdlib/base.html#All-Objects-1",
    "page": "Essentials",
    "title": "All Objects",
    "category": "section",
    "text": "Core.:(===)\nCore.isa\nBase.isequal(::Any, ::Any)\nBase.isequal(::Nullable, ::Nullable)\nBase.isless\nBase.isless(::Nullable, ::Nullable)\nBase.ifelse\nBase.lexcmp\nBase.lexless\nCore.typeof\nCore.tuple\nBase.ntuple\nBase.object_id\nBase.hash\nBase.finalizer\nBase.finalize\nBase.copy\nBase.deepcopy\nCore.isdefined\nBase.convert\nBase.promote\nBase.oftype\nBase.widen\nBase.identity"
},

{
    "location": "stdlib/base.html#Base.supertype",
    "page": "Essentials",
    "title": "Base.supertype",
    "category": "Function",
    "text": "supertype(T::DataType)\n\nReturn the supertype of DataType T.\n\njulia> supertype(Int32)\nSigned\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.issubtype",
    "page": "Essentials",
    "title": "Core.issubtype",
    "category": "Function",
    "text": "issubtype(type1, type2)\n\nReturn true if and only if all values of type1 are also of type2. Can also be written using the <: infix operator as type1 <: type2.\n\nExamples\n\njulia> issubtype(Int8, Int32)\nfalse\n\njulia> Int8 <: Integer\ntrue\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.:<:",
    "page": "Essentials",
    "title": "Base.:<:",
    "category": "Function",
    "text": "<:(T1, T2)\n\nSubtype operator, equivalent to issubtype(T1, T2).\n\njulia> Float64 <: AbstractFloat\ntrue\n\njulia> Vector{Int} <: AbstractArray\ntrue\n\njulia> Matrix{Float64} <: Matrix{AbstractFloat}\nfalse\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.:>:",
    "page": "Essentials",
    "title": "Base.:>:",
    "category": "Function",
    "text": ">:(T1, T2)\n\nSupertype operator, equivalent to issubtype(T2, T1).\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.subtypes",
    "page": "Essentials",
    "title": "Base.subtypes",
    "category": "Function",
    "text": "subtypes(T::DataType)\n\nReturn a list of immediate subtypes of DataType T. Note that all currently loaded subtypes are included, including those not visible in the current module.\n\njulia> subtypes(Integer)\n4-element Array{Union{DataType, UnionAll},1}:\n BigInt\n Bool\n Signed\n Unsigned\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.typemin",
    "page": "Essentials",
    "title": "Base.typemin",
    "category": "Function",
    "text": "typemin(T)\n\nThe lowest value representable by the given (real) numeric DataType T.\n\nExamples\n\njulia> typemin(Float16)\n-Inf16\n\njulia> typemin(Float32)\n-Inf32\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.typemax",
    "page": "Essentials",
    "title": "Base.typemax",
    "category": "Function",
    "text": "typemax(T)\n\nThe highest value representable by the given (real) numeric DataType.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.realmin",
    "page": "Essentials",
    "title": "Base.realmin",
    "category": "Function",
    "text": "realmin(T)\n\nThe smallest in absolute value non-subnormal value representable by the given floating-point DataType T.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.realmax",
    "page": "Essentials",
    "title": "Base.realmax",
    "category": "Function",
    "text": "realmax(T)\n\nThe highest finite value representable by the given floating-point DataType T.\n\nExamples\n\njulia> realmax(Float16)\nFloat16(6.55e4)\n\njulia> realmax(Float32)\n3.4028235f38\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.maxintfloat",
    "page": "Essentials",
    "title": "Base.maxintfloat",
    "category": "Function",
    "text": "maxintfloat(T)\n\nThe largest integer losslessly representable by the given floating-point DataType T.\n\n\n\nmaxintfloat(T, S)\n\nThe largest integer losslessly representable by the given floating-point DataType T that also does not exceed the maximum integer representable by the integer DataType S.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.sizeof-Tuple{Type}",
    "page": "Essentials",
    "title": "Base.sizeof",
    "category": "Method",
    "text": "sizeof(T)\n\nSize, in bytes, of the canonical binary representation of the given DataType T, if any.\n\nExamples\n\njulia> sizeof(Float32)\n4\n\njulia> sizeof(Complex128)\n16\n\nIf T does not have a specific size, an error is thrown.\n\njulia> sizeof(Base.LinAlg.LU)\nERROR: argument is an abstract type; size is indeterminate\nStacktrace:\n [1] sizeof(::Type{T} where T) at ./essentials.jl:159\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.eps-Tuple{Type{#s27} where #s27<:AbstractFloat}",
    "page": "Essentials",
    "title": "Base.eps",
    "category": "Method",
    "text": "eps(::Type{T}) where T<:AbstractFloat\neps()\n\nReturns the machine epsilon of the floating point type T (T = Float64 by default). This is defined as the gap between 1 and the next largest value representable by T, and is equivalent to eps(one(T)).\n\njulia> eps()\n2.220446049250313e-16\n\njulia> eps(Float32)\n1.1920929f-7\n\njulia> 1.0 + eps()\n1.0000000000000002\n\njulia> 1.0 + eps()/2\n1.0\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.eps-Tuple{AbstractFloat}",
    "page": "Essentials",
    "title": "Base.eps",
    "category": "Method",
    "text": "eps(x::AbstractFloat)\n\nReturns the unit in last place (ulp) of x. This is the distance between consecutive representable floating point values at x. In most cases, if the distance on either side of x is different, then the larger of the two is taken, that is\n\neps(x) == max(x-prevfloat(x), nextfloat(x)-x)\n\nThe exceptions to this rule are the smallest and largest finite values (e.g. nextfloat(-Inf) and prevfloat(Inf) for Float64), which round to the smaller of the values.\n\nThe rationale for this behavior is that eps bounds the floating point rounding error. Under the default RoundNearest rounding mode, if y is a real number and x is the nearest floating point number to y, then\n\ny-x leq operatornameeps(x)2\n\njulia> eps(1.0)\n2.220446049250313e-16\n\njulia> eps(prevfloat(2.0))\n2.220446049250313e-16\n\njulia> eps(2.0)\n4.440892098500626e-16\n\njulia> x = prevfloat(Inf)      # largest finite Float64\n1.7976931348623157e308\n\njulia> x + eps(x)/2            # rounds up\nInf\n\njulia> x + prevfloat(eps(x)/2) # rounds down\n1.7976931348623157e308\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.promote_type",
    "page": "Essentials",
    "title": "Base.promote_type",
    "category": "Function",
    "text": "promote_type(type1, type2)\n\nDetermine a type big enough to hold values of each argument type without loss, whenever possible. In some cases, where no type exists to which both types can be promoted losslessly, some loss is tolerated; for example, promote_type(Int64, Float64) returns Float64 even though strictly, not all Int64 values can be represented exactly as Float64 values.\n\njulia> promote_type(Int64, Float64)\nFloat64\n\njulia> promote_type(Int32, Int64)\nInt64\n\njulia> promote_type(Float32, BigInt)\nBigFloat\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.promote_rule",
    "page": "Essentials",
    "title": "Base.promote_rule",
    "category": "Function",
    "text": "promote_rule(type1, type2)\n\nSpecifies what type should be used by promote when given values of types type1 and type2. This function should not be called directly, but should have definitions added to it for new types as appropriate.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.getfield",
    "page": "Essentials",
    "title": "Core.getfield",
    "category": "Function",
    "text": "getfield(value, name::Symbol)\n\nExtract a named field from a value of composite type. The syntax a.b calls getfield(a, :b).\n\nExample\n\njulia> a = 1//2\n1//2\n\njulia> getfield(a, :num)\n1\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.setfield!",
    "page": "Essentials",
    "title": "Core.setfield!",
    "category": "Function",
    "text": "setfield!(value, name::Symbol, x)\n\nAssign x to a named field in value of composite type. The syntax a.b = c calls setfield!(a, :b, c).\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.fieldoffset",
    "page": "Essentials",
    "title": "Base.fieldoffset",
    "category": "Function",
    "text": "fieldoffset(type, i)\n\nThe byte offset of field i of a type relative to the data start. For example, we could use it in the following manner to summarize information about a struct:\n\njulia> structinfo(T) = [(fieldoffset(T,i), fieldname(T,i), fieldtype(T,i)) for i = 1:nfields(T)];\n\njulia> structinfo(Base.Filesystem.StatStruct)\n12-element Array{Tuple{UInt64,Symbol,DataType},1}:\n (0x0000000000000000, :device, UInt64)\n (0x0000000000000008, :inode, UInt64)\n (0x0000000000000010, :mode, UInt64)\n (0x0000000000000018, :nlink, Int64)\n (0x0000000000000020, :uid, UInt64)\n (0x0000000000000028, :gid, UInt64)\n (0x0000000000000030, :rdev, UInt64)\n (0x0000000000000038, :size, Int64)\n (0x0000000000000040, :blksize, Int64)\n (0x0000000000000048, :blocks, Int64)\n (0x0000000000000050, :mtime, Float64)\n (0x0000000000000058, :ctime, Float64)\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.fieldtype",
    "page": "Essentials",
    "title": "Core.fieldtype",
    "category": "Function",
    "text": "fieldtype(T, name::Symbol | index::Int)\n\nDetermine the declared type of a field (specified by name or index) in a composite DataType T.\n\njulia> struct Foo\n           x::Int64\n           y::String\n       end\n\njulia> fieldtype(Foo, :x)\nInt64\n\njulia> fieldtype(Foo, 2)\nString\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.isimmutable",
    "page": "Essentials",
    "title": "Base.isimmutable",
    "category": "Function",
    "text": "isimmutable(v)\n\nReturn true iff value v is immutable.  See Mutable Composite Types for a discussion of immutability. Note that this function works on values, so if you give it a type, it will tell you that a value of DataType is mutable.\n\njulia> isimmutable(1)\ntrue\n\njulia> isimmutable([1,2])\nfalse\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.isbits",
    "page": "Essentials",
    "title": "Base.isbits",
    "category": "Function",
    "text": "isbits(T)\n\nReturn true if T is a \"plain data\" type, meaning it is immutable and contains no references to other values. Typical examples are numeric types such as UInt8, Float64, and Complex{Float64}.\n\njulia> isbits(Complex{Float64})\ntrue\n\njulia> isbits(Complex)\nfalse\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.isleaftype",
    "page": "Essentials",
    "title": "Base.isleaftype",
    "category": "Function",
    "text": "isleaftype(T)\n\nDetermine whether T's only subtypes are itself and Union{}. This means T is a concrete type that can have instances.\n\njulia> isleaftype(Complex)\nfalse\n\njulia> isleaftype(Complex{Float32})\ntrue\n\njulia> isleaftype(Vector{Complex})\ntrue\n\njulia> isleaftype(Vector{Complex{Float32}})\ntrue\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.typejoin",
    "page": "Essentials",
    "title": "Base.typejoin",
    "category": "Function",
    "text": "typejoin(T, S)\n\nCompute a type that contains both T and S.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.typeintersect",
    "page": "Essentials",
    "title": "Base.typeintersect",
    "category": "Function",
    "text": "typeintersect(T, S)\n\nCompute a type that contains the intersection of T and S. Usually this will be the smallest such type or one close to it.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.Val",
    "page": "Essentials",
    "title": "Base.Val",
    "category": "Type",
    "text": "Val{c}\n\nCreate a \"value type\" out of c, which must be an isbits value. The intent of this construct is to be able to dispatch on constants, e.g., f(Val{false}) allows you to dispatch directly (at compile-time) to an implementation f(::Type{Val{false}}), without having to test the boolean value at runtime.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.Enums.@enum",
    "page": "Essentials",
    "title": "Base.Enums.@enum",
    "category": "Macro",
    "text": "@enum EnumName[::BaseType] value1[=x] value2[=y]\n\nCreate an Enum{BaseType} subtype with name EnumName and enum member values of value1 and value2 with optional assigned values of x and y, respectively. EnumName can be used just like other types and enum member values as regular values, such as\n\njulia> @enum Fruit apple=1 orange=2 kiwi=3\n\njulia> f(x::Fruit) = \"I'm a Fruit with value: $(Int(x))\"\nf (generic function with 1 method)\n\njulia> f(apple)\n\"I'm a Fruit with value: 1\"\n\nBaseType, which defaults to Int32, must be a primitive subtype of Integer. Member values can be converted between the enum type and BaseType. read and write perform these conversions automatically.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.instances",
    "page": "Essentials",
    "title": "Base.instances",
    "category": "Function",
    "text": "instances(T::Type)\n\nReturn a collection of all instances of the given type, if applicable. Mostly used for enumerated types (see @enum).\n\njulia> @enum Color red blue green\n\njulia> instances(Color)\n(red::Color = 0, blue::Color = 1, green::Color = 2)\n\n\n\n"
},

{
    "location": "stdlib/base.html#Types-1",
    "page": "Essentials",
    "title": "Types",
    "category": "section",
    "text": "Base.supertype\nCore.issubtype\nBase.:(<:)\nBase.:(>:)\nBase.subtypes\nBase.typemin\nBase.typemax\nBase.realmin\nBase.realmax\nBase.maxintfloat\nBase.sizeof(::Type)\nBase.eps(::Type{<:AbstractFloat})\nBase.eps(::AbstractFloat)\nBase.promote_type\nBase.promote_rule\nCore.getfield\nCore.setfield!\nBase.fieldoffset\nCore.fieldtype\nBase.isimmutable\nBase.isbits\nBase.isleaftype\nBase.typejoin\nBase.typeintersect\nBase.Val\nBase.Enums.@enum\nBase.instances"
},

{
    "location": "stdlib/base.html#Core.Function",
    "page": "Essentials",
    "title": "Core.Function",
    "category": "Type",
    "text": "Function\n\nAbstract type of all functions.\n\njulia> isa(+, Function)\ntrue\n\njulia> typeof(sin)\nBase.#sin\n\njulia> ans <: Function\ntrue\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.method_exists",
    "page": "Essentials",
    "title": "Base.method_exists",
    "category": "Function",
    "text": "method_exists(f, Tuple type, world=typemax(UInt)) -> Bool\n\nDetermine whether the given generic function has a method matching the given Tuple of argument types with the upper bound of world age given by world.\n\njulia> method_exists(length, Tuple{Array})\ntrue\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.applicable",
    "page": "Essentials",
    "title": "Core.applicable",
    "category": "Function",
    "text": "applicable(f, args...) -> Bool\n\nDetermine whether the given generic function has a method applicable to the given arguments.\n\nExamples\n\njulia> function f(x, y)\n           x + y\n       end;\n\njulia> applicable(f, 1)\nfalse\n\njulia> applicable(f, 1, 2)\ntrue\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.invoke",
    "page": "Essentials",
    "title": "Core.invoke",
    "category": "Function",
    "text": "invoke(f, types <: Tuple, args...)\n\nInvoke a method for the given generic function matching the specified types, on the specified arguments. The arguments must be compatible with the specified types. This allows invoking a method other than the most specific matching method, which is useful when the behavior of a more general definition is explicitly needed (often as part of the implementation of a more specific method of the same function).\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.invokelatest",
    "page": "Essentials",
    "title": "Base.invokelatest",
    "category": "Function",
    "text": "invokelatest(f, args...)\n\nCalls f(args...), but guarantees that the most recent method of f will be executed.   This is useful in specialized circumstances, e.g. long-running event loops or callback functions that may call obsolete versions of a function f. (The drawback is that invokelatest is somewhat slower than calling f directly, and the type of the result cannot be inferred by the compiler.)\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.:|>",
    "page": "Essentials",
    "title": "Base.:|>",
    "category": "Function",
    "text": "|>(x, f)\n\nApplies a function to the preceding argument. This allows for easy function chaining.\n\njulia> [1:5;] |> x->x.^2 |> sum |> inv\n0.01818181818181818\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.:∘",
    "page": "Essentials",
    "title": "Base.:∘",
    "category": "Function",
    "text": "f ∘ g\n\nCompose functions: i.e. (f ∘ g)(args...) means f(g(args...)). The ∘ symbol can be entered in the Julia REPL (and most editors, appropriately configured) by typing \\circ<tab>. Example:\n\njulia> map(uppercase∘hex, 250:255)\n6-element Array{String,1}:\n \"FA\"\n \"FB\"\n \"FC\"\n \"FD\"\n \"FE\"\n \"FF\"\n\n\n\n"
},

{
    "location": "stdlib/base.html#Generic-Functions-1",
    "page": "Essentials",
    "title": "Generic Functions",
    "category": "section",
    "text": "Core.Function\nBase.method_exists\nCore.applicable\nCore.invoke\nBase.invokelatest\nBase.:(|>)\nBase.:(∘)"
},

{
    "location": "stdlib/base.html#Core.eval",
    "page": "Essentials",
    "title": "Core.eval",
    "category": "Function",
    "text": "eval([m::Module], expr::Expr)\n\nEvaluate an expression in the given module and return the result. Every Module (except those defined with baremodule) has its own 1-argument definition of eval, which evaluates expressions in that module.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@eval",
    "page": "Essentials",
    "title": "Base.@eval",
    "category": "Macro",
    "text": "@eval [mod,] ex\n\nEvaluate an expression with values interpolated into it using eval. If two arguments are provided, the first is the module to evaluate in.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.evalfile",
    "page": "Essentials",
    "title": "Base.evalfile",
    "category": "Function",
    "text": "evalfile(path::AbstractString, args::Vector{String}=String[])\n\nLoad the file using include, evaluate all expressions, and return the value of the last one.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.esc",
    "page": "Essentials",
    "title": "Base.esc",
    "category": "Function",
    "text": "esc(e::ANY)\n\nOnly valid in the context of an Expr returned from a macro. Prevents the macro hygiene pass from turning embedded variables into gensym variables. See the Macros section of the Metaprogramming chapter of the manual for more details and examples.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@inbounds",
    "page": "Essentials",
    "title": "Base.@inbounds",
    "category": "Macro",
    "text": "@inbounds(blk)\n\nEliminates array bounds checking within expressions.\n\nIn the example below the bound check of array A is skipped to improve performance.\n\nfunction sum(A::AbstractArray)\n    r = zero(eltype(A))\n    for i = 1:length(A)\n        @inbounds r += A[i]\n    end\n    return r\nend\n\nwarning: Warning\nUsing @inbounds may return incorrect results/crashes/corruption for out-of-bounds indices. The user is responsible for checking it manually.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@inline",
    "page": "Essentials",
    "title": "Base.@inline",
    "category": "Macro",
    "text": "@inline\n\nGive a hint to the compiler that this function is worth inlining.\n\nSmall functions typically do not need the @inline annotation, as the compiler does it automatically. By using @inline on bigger functions, an extra nudge can be given to the compiler to inline it. This is shown in the following example:\n\n@inline function bigfunction(x)\n    #=\n        Function Definition\n    =#\nend\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@noinline",
    "page": "Essentials",
    "title": "Base.@noinline",
    "category": "Macro",
    "text": "@noinline\n\nPrevents the compiler from inlining a function.\n\nSmall functions are typically inlined automatically. By using @noinline on small functions, auto-inlining can be prevented. This is shown in the following example:\n\n@noinline function smallfunction(x)\n    #=\n        Function Definition\n    =#\nend\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.gensym",
    "page": "Essentials",
    "title": "Base.gensym",
    "category": "Function",
    "text": "gensym([tag])\n\nGenerates a symbol which will not conflict with other variable names.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@gensym",
    "page": "Essentials",
    "title": "Base.@gensym",
    "category": "Macro",
    "text": "@gensym\n\nGenerates a gensym symbol for a variable. For example, @gensym x y is transformed into x = gensym(\"x\"); y = gensym(\"y\").\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@polly",
    "page": "Essentials",
    "title": "Base.@polly",
    "category": "Macro",
    "text": "@polly\n\nTells the compiler to apply the polyhedral optimizer Polly to a function.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.parse-Tuple{Any,Any}",
    "page": "Essentials",
    "title": "Base.parse",
    "category": "Method",
    "text": "parse(str, start; greedy=true, raise=true)\n\nParse the expression string and return an expression (which could later be passed to eval for execution). start is the index of the first character to start parsing. If greedy is true (default), parse will try to consume as much input as it can; otherwise, it will stop as soon as it has parsed a valid expression. Incomplete but otherwise syntactically valid expressions will return Expr(:incomplete, \"(error message)\"). If raise is true (default), syntax errors other than incomplete expressions will raise an error. If raise is false, parse will return an expression that will raise an error upon evaluation.\n\njulia> parse(\"x = 3, y = 5\", 7)\n(:(y = 5), 13)\n\njulia> parse(\"x = 3, y = 5\", 5)\n(:((3, y) = 5), 13)\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.parse-Tuple{Any}",
    "page": "Essentials",
    "title": "Base.parse",
    "category": "Method",
    "text": "parse(str; raise=true)\n\nParse the expression string greedily, returning a single expression. An error is thrown if there are additional characters after the first expression. If raise is true (default), syntax errors will raise an error; otherwise, parse will return an expression that will raise an error upon evaluation.\n\njulia> parse(\"x = 3\")\n:(x = 3)\n\njulia> parse(\"x = \")\n:($(Expr(:incomplete, \"incomplete: premature end of input\")))\n\njulia> parse(\"1.0.2\")\nERROR: ParseError(\"invalid numeric constant \\\"1.0.\\\"\")\nStacktrace:\n[...]\n\njulia> parse(\"1.0.2\"; raise = false)\n:($(Expr(:error, \"invalid numeric constant \\\"1.0.\\\"\")))\n\n\n\n"
},

{
    "location": "stdlib/base.html#Syntax-1",
    "page": "Essentials",
    "title": "Syntax",
    "category": "section",
    "text": "Core.eval\nBase.@eval\nBase.evalfile\nBase.esc\nBase.@inbounds\nBase.@inline\nBase.@noinline\nBase.gensym\nBase.@gensym\nBase.@polly\nBase.parse(::Any, ::Any)\nBase.parse(::Any)"
},

{
    "location": "stdlib/base.html#Base.Nullable",
    "page": "Essentials",
    "title": "Base.Nullable",
    "category": "Type",
    "text": "Nullable(x, hasvalue::Bool=true)\n\nWrap value x in an object of type Nullable, which indicates whether a value is present. Nullable(x) yields a non-empty wrapper and Nullable{T}() yields an empty instance of a wrapper that might contain a value of type T.\n\nNullable(x, false) yields Nullable{typeof(x)}() with x stored in the result's value field.\n\nExamples\n\njulia> Nullable(1)\nNullable{Int64}(1)\n\njulia> Nullable{Int64}()\nNullable{Int64}()\n\njulia> Nullable(1, false)\nNullable{Int64}()\n\njulia> dump(Nullable(1, false))\nNullable{Int64}\n  hasvalue: Bool false\n  value: Int64 1\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.get-Tuple{Nullable,Any}",
    "page": "Essentials",
    "title": "Base.get",
    "category": "Method",
    "text": "get(x::Nullable[, y])\n\nAttempt to access the value of x. Returns the value if it is present; otherwise, returns y if provided, or throws a NullException if not.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.isnull",
    "page": "Essentials",
    "title": "Base.isnull",
    "category": "Function",
    "text": "isnull(x)\n\nReturn whether or not x is null for Nullable x; return false for all other x.\n\nExamples\n\njulia> x = Nullable(1, false)\nNullable{Int64}()\n\njulia> isnull(x)\ntrue\n\njulia> x = Nullable(1, true)\nNullable{Int64}(1)\n\njulia> isnull(x)\nfalse\n\njulia> x = 1\n1\n\njulia> isnull(x)\nfalse\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.unsafe_get",
    "page": "Essentials",
    "title": "Base.unsafe_get",
    "category": "Function",
    "text": "unsafe_get(x)\n\nReturn the value of x for Nullable x; return x for all other x.\n\nThis method does not check whether or not x is null before attempting to access the value of x for x::Nullable (hence \"unsafe\").\n\njulia> x = Nullable(1)\nNullable{Int64}(1)\n\njulia> unsafe_get(x)\n1\n\njulia> x = Nullable{String}()\nNullable{String}()\n\njulia> unsafe_get(x)\nERROR: UndefRefError: access to undefined reference\nStacktrace:\n [1] unsafe_get(::Nullable{String}) at ./nullable.jl:125\n\njulia> x = 1\n1\n\njulia> unsafe_get(x)\n1\n\n\n\n"
},

{
    "location": "stdlib/base.html#Nullables-1",
    "page": "Essentials",
    "title": "Nullables",
    "category": "section",
    "text": "Base.Nullable\nBase.get(::Nullable, ::Any)\nBase.isnull\nBase.unsafe_get"
},

{
    "location": "stdlib/base.html#Base.run",
    "page": "Essentials",
    "title": "Base.run",
    "category": "Function",
    "text": "run(command, args...)\n\nRun a command object, constructed with backticks. Throws an error if anything goes wrong, including the process exiting with a non-zero status.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.spawn",
    "page": "Essentials",
    "title": "Base.spawn",
    "category": "Function",
    "text": "spawn(command)\n\nRun a command object asynchronously, returning the resulting Process object.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.DevNull",
    "page": "Essentials",
    "title": "Base.DevNull",
    "category": "Constant",
    "text": "DevNull\n\nUsed in a stream redirect to discard all data written to it. Essentially equivalent to /dev/null on Unix or NUL on Windows. Usage:\n\nrun(pipeline(`cat test.txt`, DevNull))\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.success",
    "page": "Essentials",
    "title": "Base.success",
    "category": "Function",
    "text": "success(command)\n\nRun a command object, constructed with backticks, and tell whether it was successful (exited with a code of 0). An exception is raised if the process cannot be started.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.process_running",
    "page": "Essentials",
    "title": "Base.process_running",
    "category": "Function",
    "text": "process_running(p::Process)\n\nDetermine whether a process is currently running.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.process_exited",
    "page": "Essentials",
    "title": "Base.process_exited",
    "category": "Function",
    "text": "process_exited(p::Process)\n\nDetermine whether a process has exited.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.kill-Tuple{Base.Process,Integer}",
    "page": "Essentials",
    "title": "Base.kill",
    "category": "Method",
    "text": "kill(p::Process, signum=SIGTERM)\n\nSend a signal to a process. The default is to terminate the process.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.Sys.set_process_title",
    "page": "Essentials",
    "title": "Base.Sys.set_process_title",
    "category": "Function",
    "text": "Sys.set_process_title(title::AbstractString)\n\nSet the process title. No-op on some operating systems.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.Sys.get_process_title",
    "page": "Essentials",
    "title": "Base.Sys.get_process_title",
    "category": "Function",
    "text": "Sys.get_process_title()\n\nGet the process title. On some systems, will always return an empty string.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.readandwrite",
    "page": "Essentials",
    "title": "Base.readandwrite",
    "category": "Function",
    "text": "readandwrite(command)\n\nStarts running a command asynchronously, and returns a tuple (stdout,stdin,process) of the output stream and input stream of the process, and the process object itself.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.ignorestatus",
    "page": "Essentials",
    "title": "Base.ignorestatus",
    "category": "Function",
    "text": "ignorestatus(command)\n\nMark a command object so that running it will not throw an error if the result code is non-zero.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.detach",
    "page": "Essentials",
    "title": "Base.detach",
    "category": "Function",
    "text": "detach(command)\n\nMark a command object so that it will be run in a new process group, allowing it to outlive the julia process, and not have Ctrl-C interrupts passed to it.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.Cmd",
    "page": "Essentials",
    "title": "Base.Cmd",
    "category": "Type",
    "text": "Cmd(cmd::Cmd; ignorestatus, detach, windows_verbatim, windows_hide, env, dir)\n\nConstruct a new Cmd object, representing an external program and arguments, from cmd, while changing the settings of the optional keyword arguments:\n\nignorestatus::Bool: If true (defaults to false), then the Cmd will not throw an error if the return code is nonzero.\ndetach::Bool: If true (defaults to false), then the Cmd will be run in a new process group, allowing it to outlive the julia process and not have Ctrl-C passed to it.\nwindows_verbatim::Bool: If true (defaults to false), then on Windows the Cmd will send a command-line string to the process with no quoting or escaping of arguments, even arguments containing spaces. (On Windows, arguments are sent to a program as a single \"command-line\" string, and programs are responsible for parsing it into arguments. By default, empty arguments and arguments with spaces or tabs are quoted with double quotes \" in the command line, and \\ or \" are preceded by backslashes. windows_verbatim=true is useful for launching programs that parse their command line in nonstandard ways.) Has no effect on non-Windows systems.\nwindows_hide::Bool: If true (defaults to false), then on Windows no new console window is displayed when the Cmd is executed. This has no effect if a console is already open or on non-Windows systems.\nenv: Set environment variables to use when running the Cmd. env is either a dictionary mapping strings to strings, an array of strings of the form \"var=val\", an array or tuple of \"var\"=>val pairs, or nothing. In order to modify (rather than replace) the existing environment, create env by copy(ENV) and then set env[\"var\"]=val as desired.\ndir::AbstractString: Specify a working directory for the command (instead of the current directory).\n\nFor any keywords that are not specified, the current settings from cmd are used. Normally, to create a Cmd object in the first place, one uses backticks, e.g.\n\nCmd(`echo \"Hello world\"`, ignorestatus=true, detach=false)\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.setenv",
    "page": "Essentials",
    "title": "Base.setenv",
    "category": "Function",
    "text": "setenv(command::Cmd, env; dir=\"\")\n\nSet environment variables to use when running the given command. env is either a dictionary mapping strings to strings, an array of strings of the form \"var=val\", or zero or more \"var\"=>val pair arguments. In order to modify (rather than replace) the existing environment, create env by copy(ENV) and then setting env[\"var\"]=val as desired, or use withenv.\n\nThe dir keyword argument can be used to specify a working directory for the command.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.withenv",
    "page": "Essentials",
    "title": "Base.withenv",
    "category": "Function",
    "text": "withenv(f::Function, kv::Pair...)\n\nExecute f() in an environment that is temporarily modified (not replaced as in setenv) by zero or more \"var\"=>val arguments kv. withenv is generally used via the withenv(kv...) do ... end syntax. A value of nothing can be used to temporarily unset an environment variable (if it is set). When withenv returns, the original environment has been restored.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.pipeline-Tuple{Any,Any,Any,Vararg{Any,N} where N}",
    "page": "Essentials",
    "title": "Base.pipeline",
    "category": "Method",
    "text": "pipeline(from, to, ...)\n\nCreate a pipeline from a data source to a destination. The source and destination can be commands, I/O streams, strings, or results of other pipeline calls. At least one argument must be a command. Strings refer to filenames. When called with more than two arguments, they are chained together from left to right. For example pipeline(a,b,c) is equivalent to pipeline(pipeline(a,b),c). This provides a more concise way to specify multi-stage pipelines.\n\nExamples:\n\nrun(pipeline(`ls`, `grep xyz`))\nrun(pipeline(`ls`, \"out.txt\"))\nrun(pipeline(\"out.txt\", `grep xyz`))\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.pipeline-Tuple{Base.AbstractCmd}",
    "page": "Essentials",
    "title": "Base.pipeline",
    "category": "Method",
    "text": "pipeline(command; stdin, stdout, stderr, append=false)\n\nRedirect I/O to or from the given command. Keyword arguments specify which of the command's streams should be redirected. append controls whether file output appends to the file. This is a more general version of the 2-argument pipeline function. pipeline(from, to) is equivalent to pipeline(from, stdout=to) when from is a command, and to pipeline(to, stdin=from) when from is another kind of data source.\n\nExamples:\n\nrun(pipeline(`dothings`, stdout=\"out.txt\", stderr=\"errs.txt\"))\nrun(pipeline(`update`, stdout=\"log.txt\", append=true))\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.Libc.gethostname",
    "page": "Essentials",
    "title": "Base.Libc.gethostname",
    "category": "Function",
    "text": "gethostname() -> AbstractString\n\nGet the local machine's host name.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.getipaddr",
    "page": "Essentials",
    "title": "Base.getipaddr",
    "category": "Function",
    "text": "getipaddr() -> IPAddr\n\nGet the IP address of the local machine.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.Libc.getpid",
    "page": "Essentials",
    "title": "Base.Libc.getpid",
    "category": "Function",
    "text": "getpid() -> Int32\n\nGet Julia's process ID.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.Libc.time-Tuple{}",
    "page": "Essentials",
    "title": "Base.Libc.time",
    "category": "Method",
    "text": "time()\n\nGet the system time in seconds since the epoch, with fairly high (typically, microsecond) resolution.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.time_ns",
    "page": "Essentials",
    "title": "Base.time_ns",
    "category": "Function",
    "text": "time_ns()\n\nGet the time in nanoseconds. The time corresponding to 0 is undefined, and wraps every 5.8 years.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.tic",
    "page": "Essentials",
    "title": "Base.tic",
    "category": "Function",
    "text": "tic()\n\nSet a timer to be read by the next call to toc or toq. The macro call @time expr can also be used to time evaluation.\n\njulia> tic()\n0x0000c45bc7abac95\n\njulia> sleep(0.3)\n\njulia> toc()\nelapsed time: 0.302745944 seconds\n0.302745944\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.toc",
    "page": "Essentials",
    "title": "Base.toc",
    "category": "Function",
    "text": "toc()\n\nPrint and return the time elapsed since the last tic. The macro call @time expr can also be used to time evaluation.\n\njulia> tic()\n0x0000c45bc7abac95\n\njulia> sleep(0.3)\n\njulia> toc()\nelapsed time: 0.302745944 seconds\n0.302745944\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.toq",
    "page": "Essentials",
    "title": "Base.toq",
    "category": "Function",
    "text": "toq()\n\nReturn, but do not print, the time elapsed since the last tic. The macro calls @timed expr and @elapsed expr also return evaluation time.\n\njulia> tic()\n0x0000c46477a9675d\n\njulia> sleep(0.3)\n\njulia> toq()\n0.302251004\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@time",
    "page": "Essentials",
    "title": "Base.@time",
    "category": "Macro",
    "text": "@time\n\nA macro to execute an expression, printing the time it took to execute, the number of allocations, and the total number of bytes its execution caused to be allocated, before returning the value of the expression.\n\nSee also @timev, @timed, @elapsed, and @allocated.\n\njulia> @time rand(10^6);\n  0.001525 seconds (7 allocations: 7.630 MiB)\n\njulia> @time begin\n           sleep(0.3)\n           1+1\n       end\n  0.301395 seconds (8 allocations: 336 bytes)\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@timev",
    "page": "Essentials",
    "title": "Base.@timev",
    "category": "Macro",
    "text": "@timev\n\nThis is a verbose version of the @time macro. It first prints the same information as @time, then any non-zero memory allocation counters, and then returns the value of the expression.\n\nSee also @time, @timed, @elapsed, and @allocated.\n\njulia> @timev rand(10^6);\n  0.001006 seconds (7 allocations: 7.630 MiB)\nelapsed time (ns): 1005567\nbytes allocated:   8000256\npool allocs:       6\nmalloc() calls:    1\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@timed",
    "page": "Essentials",
    "title": "Base.@timed",
    "category": "Macro",
    "text": "@timed\n\nA macro to execute an expression, and return the value of the expression, elapsed time, total bytes allocated, garbage collection time, and an object with various memory allocation counters.\n\nSee also @time, @timev, @elapsed, and @allocated.\n\njulia> val, t, bytes, gctime, memallocs = @timed rand(10^6);\n\njulia> t\n0.006634834\n\njulia> bytes\n8000256\n\njulia> gctime\n0.0055765\n\njulia> fieldnames(typeof(memallocs))\n9-element Array{Symbol,1}:\n :allocd\n :malloc\n :realloc\n :poolalloc\n :bigalloc\n :freecall\n :total_time\n :pause\n :full_sweep\n\njulia> memallocs.total_time\n5576500\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@elapsed",
    "page": "Essentials",
    "title": "Base.@elapsed",
    "category": "Macro",
    "text": "@elapsed\n\nA macro to evaluate an expression, discarding the resulting value, instead returning the number of seconds it took to execute as a floating-point number.\n\nSee also @time, @timev, @timed, and @allocated.\n\njulia> @elapsed sleep(0.3)\n0.301391426\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@allocated",
    "page": "Essentials",
    "title": "Base.@allocated",
    "category": "Macro",
    "text": "@allocated\n\nA macro to evaluate an expression, discarding the resulting value, instead returning the total number of bytes allocated during evaluation of the expression. Note: the expression is evaluated inside a local function, instead of the current context, in order to eliminate the effects of compilation, however, there still may be some allocations due to JIT compilation. This also makes the results inconsistent with the @time macros, which do not try to adjust for the effects of compilation.\n\nSee also @time, @timev, @timed, and @elapsed.\n\njulia> @allocated rand(10^6)\n8000080\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.EnvHash",
    "page": "Essentials",
    "title": "Base.EnvHash",
    "category": "Type",
    "text": "EnvHash() -> EnvHash\n\nA singleton of this type provides a hash table interface to environment variables.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.ENV",
    "page": "Essentials",
    "title": "Base.ENV",
    "category": "Constant",
    "text": "ENV\n\nReference to the singleton EnvHash, providing a dictionary interface to system environment variables.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.is_unix",
    "page": "Essentials",
    "title": "Base.is_unix",
    "category": "Function",
    "text": "is_unix([os])\n\nPredicate for testing if the OS provides a Unix-like interface. See documentation in Handling Operating System Variation.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.is_apple",
    "page": "Essentials",
    "title": "Base.is_apple",
    "category": "Function",
    "text": "is_apple([os])\n\nPredicate for testing if the OS is a derivative of Apple Macintosh OS X or Darwin. See documentation in Handling Operating System Variation.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.is_linux",
    "page": "Essentials",
    "title": "Base.is_linux",
    "category": "Function",
    "text": "is_linux([os])\n\nPredicate for testing if the OS is a derivative of Linux. See documentation in Handling Operating System Variation.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.is_bsd",
    "page": "Essentials",
    "title": "Base.is_bsd",
    "category": "Function",
    "text": "is_bsd([os])\n\nPredicate for testing if the OS is a derivative of BSD. See documentation in Handling Operating System Variation.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.is_windows",
    "page": "Essentials",
    "title": "Base.is_windows",
    "category": "Function",
    "text": "is_windows([os])\n\nPredicate for testing if the OS is a derivative of Microsoft Windows NT. See documentation in Handling Operating System Variation.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.Sys.windows_version",
    "page": "Essentials",
    "title": "Base.Sys.windows_version",
    "category": "Function",
    "text": "Sys.windows_version()\n\nReturns the version number for the Windows NT Kernel as a (major, minor) pair, or (0, 0) if this is not running on Windows.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@static",
    "page": "Essentials",
    "title": "Base.@static",
    "category": "Macro",
    "text": "@static\n\nPartially evaluates an expression at parse time.\n\nFor example, @static is_windows() ? foo : bar will evaluate is_windows() and insert either foo or bar into the expression. This is useful in cases where a construct would be invalid on other platforms, such as a ccall to a non-existent function. @static if is_apple() foo end and @static foo <&&,||> bar are also valid syntax.\n\n\n\n"
},

{
    "location": "stdlib/base.html#System-1",
    "page": "Essentials",
    "title": "System",
    "category": "section",
    "text": "Base.run\nBase.spawn\nBase.DevNull\nBase.success\nBase.process_running\nBase.process_exited\nBase.kill(::Base.Process, ::Integer)\nBase.Sys.set_process_title\nBase.Sys.get_process_title\nBase.readandwrite\nBase.ignorestatus\nBase.detach\nBase.Cmd\nBase.setenv\nBase.withenv\nBase.pipeline(::Any, ::Any, ::Any, ::Any...)\nBase.pipeline(::Base.AbstractCmd)\nBase.Libc.gethostname\nBase.getipaddr\nBase.Libc.getpid\nBase.Libc.time()\nBase.time_ns\nBase.tic\nBase.toc\nBase.toq\nBase.@time\nBase.@timev\nBase.@timed\nBase.@elapsed\nBase.@allocated\nBase.EnvHash\nBase.ENV\nBase.is_unix\nBase.is_apple\nBase.is_linux\nBase.is_bsd\nBase.is_windows\nBase.Sys.windows_version\nBase.@static"
},

{
    "location": "stdlib/base.html#Base.error",
    "page": "Essentials",
    "title": "Base.error",
    "category": "Function",
    "text": "error(message::AbstractString)\n\nRaise an ErrorException with the given message.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.throw",
    "page": "Essentials",
    "title": "Core.throw",
    "category": "Function",
    "text": "throw(e)\n\nThrow an object as an exception.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.rethrow",
    "page": "Essentials",
    "title": "Base.rethrow",
    "category": "Function",
    "text": "rethrow([e])\n\nThrow an object without changing the current exception backtrace. The default argument is the current exception (if called within a catch block).\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.backtrace",
    "page": "Essentials",
    "title": "Base.backtrace",
    "category": "Function",
    "text": "backtrace()\n\nGet a backtrace object for the current program point.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.catch_backtrace",
    "page": "Essentials",
    "title": "Base.catch_backtrace",
    "category": "Function",
    "text": "catch_backtrace()\n\nGet the backtrace of the current exception, for use within catch blocks.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.assert",
    "page": "Essentials",
    "title": "Base.assert",
    "category": "Function",
    "text": "assert(cond)\n\nThrow an AssertionError if cond is false. Also available as the macro @assert expr.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@assert",
    "page": "Essentials",
    "title": "Base.@assert",
    "category": "Macro",
    "text": "@assert cond [text]\n\nThrow an AssertionError if cond is false. Preferred syntax for writing assertions. Message text is optionally displayed upon assertion failure.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.ArgumentError",
    "page": "Essentials",
    "title": "Base.ArgumentError",
    "category": "Type",
    "text": "ArgumentError(msg)\n\nThe parameters to a function call do not match a valid signature. Argument msg is a descriptive error string.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.AssertionError",
    "page": "Essentials",
    "title": "Base.AssertionError",
    "category": "Type",
    "text": "AssertionError([msg])\n\nThe asserted condition did not evaluate to true. Optional argument msg is a descriptive error string.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.BoundsError",
    "page": "Essentials",
    "title": "Core.BoundsError",
    "category": "Type",
    "text": "BoundsError([a],[i])\n\nAn indexing operation into an array, a, tried to access an out-of-bounds element, i.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.DimensionMismatch",
    "page": "Essentials",
    "title": "Base.DimensionMismatch",
    "category": "Type",
    "text": "DimensionMismatch([msg])\n\nThe objects called do not have matching dimensionality. Optional argument msg is a descriptive error string.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.DivideError",
    "page": "Essentials",
    "title": "Core.DivideError",
    "category": "Type",
    "text": "DivideError()\n\nInteger division was attempted with a denominator value of 0.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.DomainError",
    "page": "Essentials",
    "title": "Core.DomainError",
    "category": "Type",
    "text": "DomainError()\n\nThe arguments to a function or constructor are outside the valid domain.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.EOFError",
    "page": "Essentials",
    "title": "Base.EOFError",
    "category": "Type",
    "text": "EOFError()\n\nNo more data was available to read from a file or stream.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.ErrorException",
    "page": "Essentials",
    "title": "Core.ErrorException",
    "category": "Type",
    "text": "ErrorException(msg)\n\nGeneric error type. The error message, in the .msg field, may provide more specific details.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.InexactError",
    "page": "Essentials",
    "title": "Core.InexactError",
    "category": "Type",
    "text": "InexactError()\n\nType conversion cannot be done exactly.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.InterruptException",
    "page": "Essentials",
    "title": "Core.InterruptException",
    "category": "Type",
    "text": "InterruptException()\n\nThe process was stopped by a terminal interrupt (CTRL+C).\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.KeyError",
    "page": "Essentials",
    "title": "Base.KeyError",
    "category": "Type",
    "text": "KeyError(key)\n\nAn indexing operation into an Associative (Dict) or Set like object tried to access or delete a non-existent element.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.LoadError",
    "page": "Essentials",
    "title": "Base.LoadError",
    "category": "Type",
    "text": "LoadError(file::AbstractString, line::Int, error)\n\nAn error occurred while includeing, requireing, or using a file. The error specifics should be available in the .error field.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.MethodError",
    "page": "Essentials",
    "title": "Base.MethodError",
    "category": "Type",
    "text": "MethodError(f, args)\n\nA method with the required type signature does not exist in the given generic function. Alternatively, there is no unique most-specific method.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.NullException",
    "page": "Essentials",
    "title": "Base.NullException",
    "category": "Type",
    "text": "NullException()\n\nAn attempted access to a Nullable with no defined value.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.OutOfMemoryError",
    "page": "Essentials",
    "title": "Core.OutOfMemoryError",
    "category": "Type",
    "text": "OutOfMemoryError()\n\nAn operation allocated too much memory for either the system or the garbage collector to handle properly.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.ReadOnlyMemoryError",
    "page": "Essentials",
    "title": "Core.ReadOnlyMemoryError",
    "category": "Type",
    "text": "ReadOnlyMemoryError()\n\nAn operation tried to write to memory that is read-only.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.OverflowError",
    "page": "Essentials",
    "title": "Core.OverflowError",
    "category": "Type",
    "text": "OverflowError()\n\nThe result of an expression is too large for the specified type and will cause a wraparound.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.ParseError",
    "page": "Essentials",
    "title": "Base.ParseError",
    "category": "Type",
    "text": "ParseError(msg)\n\nThe expression passed to the parse function could not be interpreted as a valid Julia expression.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.Distributed.ProcessExitedException",
    "page": "Essentials",
    "title": "Base.Distributed.ProcessExitedException",
    "category": "Type",
    "text": "ProcessExitedException()\n\nAfter a client Julia process has exited, further attempts to reference the dead child will throw this exception.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.StackOverflowError",
    "page": "Essentials",
    "title": "Core.StackOverflowError",
    "category": "Type",
    "text": "StackOverflowError()\n\nThe function call grew beyond the size of the call stack. This usually happens when a call recurses infinitely.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.SystemError",
    "page": "Essentials",
    "title": "Base.SystemError",
    "category": "Type",
    "text": "SystemError(prefix::AbstractString, [errno::Int32])\n\nA system call failed with an error code (in the errno global variable).\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.TypeError",
    "page": "Essentials",
    "title": "Core.TypeError",
    "category": "Type",
    "text": "TypeError(func::Symbol, context::AbstractString, expected::Type, got)\n\nA type assertion failure, or calling an intrinsic function with an incorrect argument type.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.UndefRefError",
    "page": "Essentials",
    "title": "Core.UndefRefError",
    "category": "Type",
    "text": "UndefRefError()\n\nThe item or field is not defined for the given object.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.UndefVarError",
    "page": "Essentials",
    "title": "Core.UndefVarError",
    "category": "Type",
    "text": "UndefVarError(var::Symbol)\n\nA symbol in the current scope is not defined.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.InitError",
    "page": "Essentials",
    "title": "Base.InitError",
    "category": "Type",
    "text": "InitError(mod::Symbol, error)\n\nAn error occurred when running a module's __init__ function. The actual error thrown is available in the .error field.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.retry",
    "page": "Essentials",
    "title": "Base.retry",
    "category": "Function",
    "text": "retry(f::Function;  delays=ExponentialBackOff(), check=nothing) -> Function\n\nReturns an anonymous function that calls function f.  If an exception arises, f is repeatedly called again, each time check returns true, after waiting the number of seconds specified in delays.  check should input delays's current state and the Exception.\n\nExamples\n\nretry(f, delays=fill(5.0, 3))\nretry(f, delays=rand(5:10, 2))\nretry(f, delays=Base.ExponentialBackOff(n=3, first_delay=5, max_delay=1000))\nretry(http_get, check=(s,e)->e.status == \"503\")(url)\nretry(read, check=(s,e)->isa(e, UVError))(io, 128; all=false)\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.ExponentialBackOff",
    "page": "Essentials",
    "title": "Base.ExponentialBackOff",
    "category": "Type",
    "text": "ExponentialBackOff(; n=1, first_delay=0.05, max_delay=10.0, factor=5.0, jitter=0.1)\n\nA Float64 iterator of length n whose elements exponentially increase at a rate in the interval factor * (1 ± jitter).  The first element is first_delay and all elements are clamped to max_delay.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Errors-1",
    "page": "Essentials",
    "title": "Errors",
    "category": "section",
    "text": "Base.error\nCore.throw\nBase.rethrow\nBase.backtrace\nBase.catch_backtrace\nBase.assert\nBase.@assert\nBase.ArgumentError\nBase.AssertionError\nCore.BoundsError\nBase.DimensionMismatch\nCore.DivideError\nCore.DomainError\nBase.EOFError\nCore.ErrorException\nCore.InexactError\nCore.InterruptException\nBase.KeyError\nBase.LoadError\nBase.MethodError\nBase.NullException\nCore.OutOfMemoryError\nCore.ReadOnlyMemoryError\nCore.OverflowError\nBase.ParseError\nBase.ProcessExitedException\nCore.StackOverflowError\nBase.SystemError\nCore.TypeError\nCore.UndefRefError\nCore.UndefVarError\nBase.InitError\nBase.retry\nBase.ExponentialBackOff"
},

{
    "location": "stdlib/base.html#Base.Timer-Tuple{Function,Real,Real}",
    "page": "Essentials",
    "title": "Base.Timer",
    "category": "Method",
    "text": "Timer(callback::Function, delay, repeat=0)\n\nCreate a timer to call the given callback function. The callback is passed one argument, the timer object itself. The callback will be invoked after the specified initial delay, and then repeating with the given repeat interval. If repeat is 0, the timer is only triggered once. Times are in seconds. A timer is stopped and has its resources freed by calling close on it.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.Timer",
    "page": "Essentials",
    "title": "Base.Timer",
    "category": "Type",
    "text": "Timer(delay, repeat=0)\n\nCreate a timer that wakes up tasks waiting for it (by calling wait on the timer object) at a specified interval.  Times are in seconds.  Waiting tasks are woken with an error when the timer is closed (by close. Use isopen to check whether a timer is still active.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.AsyncCondition",
    "page": "Essentials",
    "title": "Base.AsyncCondition",
    "category": "Type",
    "text": "AsyncCondition()\n\nCreate a async condition that wakes up tasks waiting for it (by calling wait on the object) when notified from C by a call to uv_async_send. Waiting tasks are woken with an error when the object is closed (by close. Use isopen to check whether it is still active.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.AsyncCondition-Tuple{Function}",
    "page": "Essentials",
    "title": "Base.AsyncCondition",
    "category": "Method",
    "text": "AsyncCondition(callback::Function)\n\nCreate a async condition that calls the given callback function. The callback is passed one argument, the async condition object itself.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Events-1",
    "page": "Essentials",
    "title": "Events",
    "category": "section",
    "text": "Base.Timer(::Function, ::Real, ::Real)\nBase.Timer\nBase.AsyncCondition\nBase.AsyncCondition(::Function)"
},

{
    "location": "stdlib/base.html#Base.module_name",
    "page": "Essentials",
    "title": "Base.module_name",
    "category": "Function",
    "text": "module_name(m::Module) -> Symbol\n\nGet the name of a Module as a Symbol.\n\njulia> module_name(Base.LinAlg)\n:LinAlg\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.module_parent",
    "page": "Essentials",
    "title": "Base.module_parent",
    "category": "Function",
    "text": "module_parent(m::Module) -> Module\n\nGet a module's enclosing Module. Main is its own parent, as is LastMain after workspace().\n\njulia> module_parent(Main)\nMain\n\njulia> module_parent(Base.LinAlg.BLAS)\nBase.LinAlg\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.current_module",
    "page": "Essentials",
    "title": "Base.current_module",
    "category": "Function",
    "text": "current_module() -> Module\n\nGet the dynamically current Module, which is the Module code is currently being read from. In general, this is not the same as the module containing the call to this function.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.fullname",
    "page": "Essentials",
    "title": "Base.fullname",
    "category": "Function",
    "text": "fullname(m::Module)\n\nGet the fully-qualified name of a module as a tuple of symbols. For example,\n\njulia> fullname(Base.Pkg)\n(:Base, :Pkg)\n\njulia> fullname(Main)\n()\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.names",
    "page": "Essentials",
    "title": "Base.names",
    "category": "Function",
    "text": "names(x::Module, all::Bool=false, imported::Bool=false)\n\nGet an array of the names exported by a Module, excluding deprecated names. If all is true, then the list also includes non-exported names defined in the module, deprecated names, and compiler-generated names. If imported is true, then names explicitly imported from other modules are also included.\n\nAs a special case, all names defined in Main are considered \"exported\", since it is not idiomatic to explicitly export names from Main.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Core.nfields",
    "page": "Essentials",
    "title": "Core.nfields",
    "category": "Function",
    "text": "nfields(x::DataType) -> Int\n\nGet the number of fields of a DataType.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.fieldnames",
    "page": "Essentials",
    "title": "Base.fieldnames",
    "category": "Function",
    "text": "fieldnames(x::DataType)\n\nGet an array of the fields of a DataType.\n\njulia> fieldnames(Hermitian)\n2-element Array{Symbol,1}:\n :data\n :uplo\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.fieldname",
    "page": "Essentials",
    "title": "Base.fieldname",
    "category": "Function",
    "text": "fieldname(x::DataType, i::Integer)\n\nGet the name of field i of a DataType.\n\njulia> fieldname(SparseMatrixCSC,1)\n:m\n\njulia> fieldname(SparseMatrixCSC,5)\n:nzval\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.datatype_module",
    "page": "Essentials",
    "title": "Base.datatype_module",
    "category": "Function",
    "text": "Base.datatype_module(t::DataType) -> Module\n\nDetermine the module containing the definition of a DataType.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.datatype_name",
    "page": "Essentials",
    "title": "Base.datatype_name",
    "category": "Function",
    "text": "Base.datatype_name(t) -> Symbol\n\nGet the name of a (potentially UnionAll-wrapped) DataType (without its parent module) as a symbol.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.isconst",
    "page": "Essentials",
    "title": "Base.isconst",
    "category": "Function",
    "text": "isconst([m::Module], s::Symbol) -> Bool\n\nDetermine whether a global is declared const in a given Module. The default Module argument is current_module().\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.function_name",
    "page": "Essentials",
    "title": "Base.function_name",
    "category": "Function",
    "text": "Base.function_name(f::Function) -> Symbol\n\nGet the name of a generic Function as a symbol, or :anonymous.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.function_module-Tuple{Function}",
    "page": "Essentials",
    "title": "Base.function_module",
    "category": "Method",
    "text": "Base.function_module(f::Function) -> Module\n\nDetermine the module containing the (first) definition of a generic function.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.function_module-Tuple{Any,Any}",
    "page": "Essentials",
    "title": "Base.function_module",
    "category": "Method",
    "text": "Base.function_module(f::Function, types) -> Module\n\nDetermine the module containing a given definition of a generic function.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.functionloc-Tuple{Any,Any}",
    "page": "Essentials",
    "title": "Base.functionloc",
    "category": "Method",
    "text": "functionloc(f::Function, types)\n\nReturns a tuple (filename,line) giving the location of a generic Function definition.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.functionloc-Tuple{Method}",
    "page": "Essentials",
    "title": "Base.functionloc",
    "category": "Method",
    "text": "functionloc(m::Method)\n\nReturns a tuple (filename,line) giving the location of a Method definition.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@functionloc",
    "page": "Essentials",
    "title": "Base.@functionloc",
    "category": "Macro",
    "text": "@functionloc\n\nApplied to a function or macro call, it evaluates the arguments to the specified call, and returns a tuple (filename,line) giving the location for the method that would be called for those arguments. It calls out to the functionloc function.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Reflection-1",
    "page": "Essentials",
    "title": "Reflection",
    "category": "section",
    "text": "Base.module_name\nBase.module_parent\nBase.current_module\nBase.fullname\nBase.names\nCore.nfields\nBase.fieldnames\nBase.fieldname\nBase.datatype_module\nBase.datatype_name\nBase.isconst\nBase.function_name\nBase.function_module(::Function)\nBase.function_module(::Any, ::Any)\nBase.functionloc(::Any, ::Any)\nBase.functionloc(::Method)\nBase.@functionloc"
},

{
    "location": "stdlib/base.html#Base.gc",
    "page": "Essentials",
    "title": "Base.gc",
    "category": "Function",
    "text": "gc()\n\nPerform garbage collection. This should not generally be used.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.gc_enable",
    "page": "Essentials",
    "title": "Base.gc_enable",
    "category": "Function",
    "text": "gc_enable(on::Bool)\n\nControl whether garbage collection is enabled using a boolean argument (true for enabled, false for disabled). Returns previous GC state. Disabling garbage collection should be used only with extreme caution, as it can cause memory use to grow without bound.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.macroexpand",
    "page": "Essentials",
    "title": "Base.macroexpand",
    "category": "Function",
    "text": "macroexpand(x)\n\nTakes the expression x and returns an equivalent expression with all macros removed (expanded).\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@macroexpand",
    "page": "Essentials",
    "title": "Base.@macroexpand",
    "category": "Macro",
    "text": "@macroexpand\n\nReturn equivalent expression with all macros removed (expanded).\n\nThere is a subtle difference between @macroexpand and macroexpand in that expansion takes place in different contexts. This is best seen in the following example:\n\njulia> module M\n           macro m()\n               1\n           end\n           function f()\n               (@macroexpand(@m), macroexpand(:(@m)))\n           end\n       end\nM\n\njulia> macro m()\n           2\n       end\n@m (macro with 1 method)\n\njulia> M.f()\n(1, 2)\n\nWith @macroexpand the expression expands where @macroexpand appears in the code (module M in the example). With macroexpand the expression expands in the current module where the code was finally called (REPL in the example). Note that when calling macroexpand or @macroexpand directly from the REPL, both of these contexts coincide, hence there is no difference.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.expand",
    "page": "Essentials",
    "title": "Base.expand",
    "category": "Function",
    "text": "expand(x)\n\nTakes the expression x and returns an equivalent expression in lowered form. See also code_lowered.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.code_lowered",
    "page": "Essentials",
    "title": "Base.code_lowered",
    "category": "Function",
    "text": "code_lowered(f, types)\n\nReturns an array of lowered ASTs for the methods matching the given generic function and type signature.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@code_lowered",
    "page": "Essentials",
    "title": "Base.@code_lowered",
    "category": "Macro",
    "text": "@code_lowered\n\nEvaluates the arguments to the function or macro call, determines their types, and calls code_lowered on the resulting expression.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.code_typed",
    "page": "Essentials",
    "title": "Base.code_typed",
    "category": "Function",
    "text": "code_typed(f, types; optimize=true)\n\nReturns an array of lowered and type-inferred ASTs for the methods matching the given generic function and type signature. The keyword argument optimize controls whether additional optimizations, such as inlining, are also applied.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@code_typed",
    "page": "Essentials",
    "title": "Base.@code_typed",
    "category": "Macro",
    "text": "@code_typed\n\nEvaluates the arguments to the function or macro call, determines their types, and calls code_typed on the resulting expression.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.code_warntype",
    "page": "Essentials",
    "title": "Base.code_warntype",
    "category": "Function",
    "text": "code_warntype([io::IO], f, types)\n\nPrints lowered and type-inferred ASTs for the methods matching the given generic function and type signature to io which defaults to STDOUT. The ASTs are annotated in such a way as to cause \"non-leaf\" types to be emphasized (if color is available, displayed in red). This serves as a warning of potential type instability. Not all non-leaf types are particularly problematic for performance, so the results need to be used judiciously. See @code_warntype for more information.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@code_warntype",
    "page": "Essentials",
    "title": "Base.@code_warntype",
    "category": "Macro",
    "text": "@code_warntype\n\nEvaluates the arguments to the function or macro call, determines their types, and calls code_warntype on the resulting expression.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.code_llvm",
    "page": "Essentials",
    "title": "Base.code_llvm",
    "category": "Function",
    "text": "code_llvm([io], f, types)\n\nPrints the LLVM bitcodes generated for running the method matching the given generic function and type signature to io which defaults to STDOUT.\n\nAll metadata and dbg.* calls are removed from the printed bitcode. Use code_llvm_raw for the full IR.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@code_llvm",
    "page": "Essentials",
    "title": "Base.@code_llvm",
    "category": "Macro",
    "text": "@code_llvm\n\nEvaluates the arguments to the function or macro call, determines their types, and calls code_llvm on the resulting expression.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.code_native",
    "page": "Essentials",
    "title": "Base.code_native",
    "category": "Function",
    "text": "code_native([io], f, types, [syntax])\n\nPrints the native assembly instructions generated for running the method matching the given generic function and type signature to io which defaults to STDOUT. Switch assembly syntax using syntax symbol parameter set to :att for AT&T syntax or :intel for Intel syntax. Output is AT&T syntax by default.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.@code_native",
    "page": "Essentials",
    "title": "Base.@code_native",
    "category": "Macro",
    "text": "@code_native\n\nEvaluates the arguments to the function or macro call, determines their types, and calls code_native on the resulting expression.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Base.precompile",
    "page": "Essentials",
    "title": "Base.precompile",
    "category": "Function",
    "text": "precompile(f,args::Tuple{Vararg{Any}})\n\nCompile the given function f for the argument tuple (of types) args, but do not execute it.\n\n\n\n"
},

{
    "location": "stdlib/base.html#Internals-1",
    "page": "Essentials",
    "title": "Internals",
    "category": "section",
    "text": "Base.gc\nBase.gc_enable\nBase.macroexpand\nBase.@macroexpand\nBase.expand\nBase.code_lowered\nBase.@code_lowered\nBase.code_typed\nBase.@code_typed\nBase.code_warntype\nBase.@code_warntype\nBase.code_llvm\nBase.@code_llvm\nBase.code_native\nBase.@code_native\nBase.precompile"
},

{
    "location": "stdlib/collections.html#",
    "page": "Collections and Data Structures",
    "title": "Collections and Data Structures",
    "category": "page",
    "text": ""
},

{
    "location": "stdlib/collections.html#Collections-and-Data-Structures-1",
    "page": "Collections and Data Structures",
    "title": "Collections and Data Structures",
    "category": "section",
    "text": ""
},

{
    "location": "stdlib/collections.html#Base.start",
    "page": "Collections and Data Structures",
    "title": "Base.start",
    "category": "Function",
    "text": "start(iter) -> state\n\nGet initial iteration state for an iterable object.\n\nExamples\n\njulia> start(1:5)\n1\n\njulia> start([1;2;3])\n1\n\njulia> start([4;2;3])\n1\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.done",
    "page": "Collections and Data Structures",
    "title": "Base.done",
    "category": "Function",
    "text": "done(iter, state) -> Bool\n\nTest whether we are done iterating.\n\nExamples\n\njulia> done(1:5, 3)\nfalse\n\njulia> done(1:5, 5)\nfalse\n\njulia> done(1:5, 6)\ntrue\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.next",
    "page": "Collections and Data Structures",
    "title": "Base.next",
    "category": "Function",
    "text": "next(iter, state) -> item, state\n\nFor a given iterable object and iteration state, return the current item and the next iteration state.\n\nExamples\n\njulia> next(1:5, 3)\n(3, 4)\n\njulia> next(1:5, 5)\n(5, 6)\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.iteratorsize",
    "page": "Collections and Data Structures",
    "title": "Base.iteratorsize",
    "category": "Function",
    "text": "iteratorsize(itertype::Type) -> IteratorSize\n\nGiven the type of an iterator, returns one of the following values:\n\nSizeUnknown() if the length (number of elements) cannot be determined in advance.\nHasLength() if there is a fixed, finite length.\nHasShape() if there is a known length plus a notion of multidimensional shape (as for an array).  In this case the size function is valid for the iterator.\nIsInfinite() if the iterator yields values forever.\n\nThe default value (for iterators that do not define this function) is HasLength(). This means that most iterators are assumed to implement length.\n\nThis trait is generally used to select between algorithms that pre-allocate space for their result, and algorithms that resize their result incrementally.\n\njulia> Base.iteratorsize(1:5)\nBase.HasShape()\n\njulia> Base.iteratorsize((2,3))\nBase.HasLength()\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.iteratoreltype",
    "page": "Collections and Data Structures",
    "title": "Base.iteratoreltype",
    "category": "Function",
    "text": "iteratoreltype(itertype::Type) -> IteratorEltype\n\nGiven the type of an iterator, returns one of the following values:\n\nEltypeUnknown() if the type of elements yielded by the iterator is not known in advance.\nHasEltype() if the element type is known, and eltype would return a meaningful value.\n\nHasEltype() is the default, since iterators are assumed to implement eltype.\n\nThis trait is generally used to select between algorithms that pre-allocate a specific type of result, and algorithms that pick a result type based on the types of yielded values.\n\njulia> Base.iteratoreltype(1:5)\nBase.HasEltype()\n\n\n\n"
},

{
    "location": "stdlib/collections.html#lib-collections-iteration-1",
    "page": "Collections and Data Structures",
    "title": "Iteration",
    "category": "section",
    "text": "Sequential iteration is implemented by the methods start(), done(), and next(). The general for loop:for i = I   # or  \"for i in I\"\n    # body\nendis translated into:state = start(I)\nwhile !done(I, state)\n    (i, state) = next(I, state)\n    # body\nendThe state object may be anything, and should be chosen appropriately for each iterable type. See the manual section on the iteration interface for more details about defining a custom iterable type.Base.start\nBase.done\nBase.next\nBase.iteratorsize\nBase.iteratoreltypeFully implemented by:Range\nUnitRange\nTuple\nNumber\nAbstractArray\nIntSet\nObjectIdDict\nDict\nWeakKeyDict\nEachLine\nAbstractString\nSet"
},

{
    "location": "stdlib/collections.html#Base.isempty",
    "page": "Collections and Data Structures",
    "title": "Base.isempty",
    "category": "Function",
    "text": "isempty(collection) -> Bool\n\nDetermine whether a collection is empty (has no elements).\n\nExamples\n\njulia> isempty([])\ntrue\n\njulia> isempty([1 2 3])\nfalse\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.empty!",
    "page": "Collections and Data Structures",
    "title": "Base.empty!",
    "category": "Function",
    "text": "empty!(collection) -> collection\n\nRemove all elements from a collection.\n\njulia> A = Dict(\"a\" => 1, \"b\" => 2)\nDict{String,Int64} with 2 entries:\n  \"b\" => 2\n  \"a\" => 1\n\njulia> empty!(A);\n\njulia> A\nDict{String,Int64} with 0 entries\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.length-Tuple{Any}",
    "page": "Collections and Data Structures",
    "title": "Base.length",
    "category": "Method",
    "text": "length(collection) -> Integer\n\nFor ordered, indexable collections, returns the maximum index i for which getindex(collection, i) is valid. For unordered collections, returns the number of elements.\n\nExamples\n\njulia> length(1:5)\n5\n\njulia> length([1; 2; 3; 4])\n4\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.endof",
    "page": "Collections and Data Structures",
    "title": "Base.endof",
    "category": "Function",
    "text": "endof(collection) -> Integer\n\nReturns the last index of the collection.\n\nExample\n\njulia> endof([1,2,4])\n3\n\n\n\n"
},

{
    "location": "stdlib/collections.html#General-Collections-1",
    "page": "Collections and Data Structures",
    "title": "General Collections",
    "category": "section",
    "text": "Base.isempty\nBase.empty!\nBase.length(::Any)\nBase.endofFully implemented by:Range\nUnitRange\nTuple\nNumber\nAbstractArray\nIntSet\nObjectIdDict\nDict\nWeakKeyDict\nAbstractString\nSet"
},

{
    "location": "stdlib/collections.html#Base.in",
    "page": "Collections and Data Structures",
    "title": "Base.in",
    "category": "Function",
    "text": "in(item, collection) -> Bool\n∈(item,collection) -> Bool\n∋(collection,item) -> Bool\n∉(item,collection) -> Bool\n∌(collection,item) -> Bool\n\nDetermine whether an item is in the given collection, in the sense that it is == to one of the values generated by iterating over the collection. Some collections need a slightly different definition; for example Sets check whether the item isequal to one of the elements. Dicts look for (key,value) pairs, and the key is compared using isequal. To test for the presence of a key in a dictionary, use haskey or k in keys(dict).\n\njulia> a = 1:3:20\n1:3:19\n\njulia> 4 in a\ntrue\n\njulia> 5 in a\nfalse\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.eltype",
    "page": "Collections and Data Structures",
    "title": "Base.eltype",
    "category": "Function",
    "text": "eltype(type)\n\nDetermine the type of the elements generated by iterating a collection of the given type. For associative collection types, this will be a Pair{KeyType,ValType}. The definition eltype(x) = eltype(typeof(x)) is provided for convenience so that instances can be passed instead of types. However the form that accepts a type argument should be defined for new types.\n\njulia> eltype(ones(Float32,2,2))\nFloat32\n\njulia> eltype(ones(Int8,2,2))\nInt8\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.indexin",
    "page": "Collections and Data Structures",
    "title": "Base.indexin",
    "category": "Function",
    "text": "indexin(a, b)\n\nReturns a vector containing the highest index in b for each value in a that is a member of b . The output vector contains 0 wherever a is not a member of b.\n\nExamples\n\njulia> a = ['a', 'b', 'c', 'b', 'd', 'a'];\n\njulia> b = ['a','b','c'];\n\njulia> indexin(a,b)\n6-element Array{Int64,1}:\n 1\n 2\n 3\n 2\n 0\n 1\n\njulia> indexin(b,a)\n3-element Array{Int64,1}:\n 6\n 4\n 3\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.findin",
    "page": "Collections and Data Structures",
    "title": "Base.findin",
    "category": "Function",
    "text": "findin(a, b)\n\nReturns the indices of elements in collection a that appear in collection b.\n\nExamples\n\njulia> a = collect(1:3:15)\n5-element Array{Int64,1}:\n  1\n  4\n  7\n 10\n 13\n\njulia> b = collect(2:4:10)\n3-element Array{Int64,1}:\n  2\n  6\n 10\n\njulia> findin(a,b) # 10 is the only common element\n1-element Array{Int64,1}:\n 4\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.unique",
    "page": "Collections and Data Structures",
    "title": "Base.unique",
    "category": "Function",
    "text": "unique(itr)\n\nReturns an array containing one value from itr for each unique value, as determined by isequal.\n\njulia> unique([1; 2; 2; 6])\n3-element Array{Int64,1}:\n 1\n 2\n 6\n\n\n\nunique(f, itr)\n\nReturns an array containing one value from itr for each unique value produced by f applied to elements of itr.\n\njulia> unique(isodd, [1; 2; 2; 6])\n2-element Array{Int64,1}:\n 1\n 2\n\n\n\nunique(itr[, dim])\n\nReturns an array containing only the unique elements of the iterable itr, in the order that the first of each set of equivalent elements originally appears. If dim is specified, returns unique regions of the array itr along dim.\n\njulia> A = map(isodd, reshape(collect(1:8), (2,2,2)))\n2×2×2 Array{Bool,3}:\n[:, :, 1] =\n  true   true\n false  false\n\n[:, :, 2] =\n  true   true\n false  false\n\njulia> unique(A)\n2-element Array{Bool,1}:\n  true\n false\n\njulia> unique(A, 2)\n2×1×2 Array{Bool,3}:\n[:, :, 1] =\n  true\n false\n\n[:, :, 2] =\n  true\n false\n\njulia> unique(A, 3)\n2×2×1 Array{Bool,3}:\n[:, :, 1] =\n  true   true\n false  false\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.allunique",
    "page": "Collections and Data Structures",
    "title": "Base.allunique",
    "category": "Function",
    "text": "allunique(itr) -> Bool\n\nReturn true if all values from itr are distinct when compared with isequal.\n\njulia> a = [1; 2; 3]\n3-element Array{Int64,1}:\n 1\n 2\n 3\n\njulia> allunique([a, a])\nfalse\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.reduce-Tuple{Any,Any,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.reduce",
    "category": "Method",
    "text": "reduce(op, v0, itr)\n\nReduce the given collection ìtr with the given binary operator op. v0 must be a neutral element for op that will be returned for empty collections. It is unspecified whether v0 is used for non-empty collections.\n\nReductions for certain commonly-used operators have special implementations which should be used instead: maximum(itr), minimum(itr), sum(itr), prod(itr), any(itr), all(itr).\n\nThe associativity of the reduction is implementation dependent. This means that you can't use non-associative operations like - because it is undefined whether reduce(-,[1,2,3]) should be evaluated as (1-2)-3 or 1-(2-3). Use foldl or foldr instead for guaranteed left or right associativity.\n\nSome operations accumulate error, and parallelism will also be easier if the reduction can be executed in groups. Future versions of Julia might change the algorithm. Note that the elements are not reordered if you use an ordered collection.\n\nExamples\n\njulia> reduce(*, 1, [2; 3; 4])\n24\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.reduce-Tuple{Any,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.reduce",
    "category": "Method",
    "text": "reduce(op, itr)\n\nLike reduce(op, v0, itr). This cannot be used with empty collections, except for some special cases (e.g. when op is one of +, *, max, min, &, |) when Julia can determine the neutral element of op.\n\njulia> reduce(*, [2; 3; 4])\n24\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.foldl-Tuple{Any,Any,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.foldl",
    "category": "Method",
    "text": "foldl(op, v0, itr)\n\nLike reduce, but with guaranteed left associativity. v0 will be used exactly once.\n\njulia> foldl(-, 1, 2:5)\n-13\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.foldl-Tuple{Any,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.foldl",
    "category": "Method",
    "text": "foldl(op, itr)\n\nLike foldl(op, v0, itr), but using the first element of itr as v0. In general, this cannot be used with empty collections (see reduce(op, itr)).\n\njulia> foldl(-, 2:5)\n-10\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.foldr-Tuple{Any,Any,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.foldr",
    "category": "Method",
    "text": "foldr(op, v0, itr)\n\nLike reduce, but with guaranteed right associativity. v0 will be used exactly once.\n\njulia> foldr(-, 1, 2:5)\n-1\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.foldr-Tuple{Any,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.foldr",
    "category": "Method",
    "text": "foldr(op, itr)\n\nLike foldr(op, v0, itr), but using the last element of itr as v0. In general, this cannot be used with empty collections (see reduce(op, itr)).\n\njulia> foldr(-, 2:5)\n-2\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.maximum-Tuple{Any}",
    "page": "Collections and Data Structures",
    "title": "Base.maximum",
    "category": "Method",
    "text": "maximum(itr)\n\nReturns the largest element in a collection.\n\njulia> maximum(-20.5:10)\n9.5\n\njulia> maximum([1,2,3])\n3\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.maximum-Tuple{Any,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.maximum",
    "category": "Method",
    "text": "maximum(A, dims)\n\nCompute the maximum value of an array over the given dimensions. See also the max(a,b) function to take the maximum of two or more arguments, which can be applied elementwise to arrays via max.(a,b).\n\njulia> A = [1 2; 3 4]\n2×2 Array{Int64,2}:\n 1  2\n 3  4\n\njulia> maximum(A, 1)\n1×2 Array{Int64,2}:\n 3  4\n\njulia> maximum(A, 2)\n2×1 Array{Int64,2}:\n 2\n 4\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.maximum!",
    "page": "Collections and Data Structures",
    "title": "Base.maximum!",
    "category": "Function",
    "text": "maximum!(r, A)\n\nCompute the maximum value of A over the singleton dimensions of r, and write results to r.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Array{Int64,2}:\n 1  2\n 3  4\n\njulia> maximum!([1; 1], A)\n2-element Array{Int64,1}:\n 2\n 4\n\njulia> maximum!([1 1], A)\n1×2 Array{Int64,2}:\n 3  4\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.minimum-Tuple{Any}",
    "page": "Collections and Data Structures",
    "title": "Base.minimum",
    "category": "Method",
    "text": "minimum(itr)\n\nReturns the smallest element in a collection.\n\njulia> minimum(-20.5:10)\n-20.5\n\njulia> minimum([1,2,3])\n1\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.minimum-Tuple{Any,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.minimum",
    "category": "Method",
    "text": "minimum(A, dims)\n\nCompute the minimum value of an array over the given dimensions. See also the min(a,b) function to take the minimum of two or more arguments, which can be applied elementwise to arrays via min.(a,b).\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Array{Int64,2}:\n 1  2\n 3  4\n\njulia> minimum(A, 1)\n1×2 Array{Int64,2}:\n 1  2\n\njulia> minimum(A, 2)\n2×1 Array{Int64,2}:\n 1\n 3\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.minimum!",
    "page": "Collections and Data Structures",
    "title": "Base.minimum!",
    "category": "Function",
    "text": "minimum!(r, A)\n\nCompute the minimum value of A over the singleton dimensions of r, and write results to r.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Array{Int64,2}:\n 1  2\n 3  4\n\njulia> minimum!([1; 1], A)\n2-element Array{Int64,1}:\n 1\n 3\n\njulia> minimum!([1 1], A)\n1×2 Array{Int64,2}:\n 1  2\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.extrema-Tuple{Any}",
    "page": "Collections and Data Structures",
    "title": "Base.extrema",
    "category": "Method",
    "text": "extrema(itr) -> Tuple\n\nCompute both the minimum and maximum element in a single pass, and return them as a 2-tuple.\n\njulia> extrema(2:10)\n(2, 10)\n\njulia> extrema([9,pi,4.5])\n(3.141592653589793, 9.0)\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.extrema-Tuple{AbstractArray,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.extrema",
    "category": "Method",
    "text": "extrema(A, dims) -> Array{Tuple}\n\nCompute the minimum and maximum elements of an array over the given dimensions.\n\nExample\n\njulia> A = reshape(collect(1:2:16), (2,2,2))\n2×2×2 Array{Int64,3}:\n[:, :, 1] =\n 1  5\n 3  7\n\n[:, :, 2] =\n  9  13\n 11  15\n\njulia> extrema(A, (1,2))\n1×1×2 Array{Tuple{Int64,Int64},3}:\n[:, :, 1] =\n (1, 7)\n\n[:, :, 2] =\n (9, 15)\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.indmax",
    "page": "Collections and Data Structures",
    "title": "Base.indmax",
    "category": "Function",
    "text": "indmax(itr) -> Integer\n\nReturns the index of the maximum element in a collection. If there are multiple maximal elements, then the first one will be returned. NaN values are ignored, unless all elements are NaN.\n\nThe collection must not be empty.\n\nExamples\n\njulia> indmax([8,0.1,-9,pi])\n1\n\njulia> indmax([1,7,7,6])\n2\n\njulia> indmax([1,7,7,NaN])\n2\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.indmin",
    "page": "Collections and Data Structures",
    "title": "Base.indmin",
    "category": "Function",
    "text": "indmin(itr) -> Integer\n\nReturns the index of the minimum element in a collection. If there are multiple minimal elements, then the first one will be returned. NaN values are ignored, unless all elements are NaN.\n\nThe collection must not be empty.\n\nExamples\n\njulia> indmin([8,0.1,-9,pi])\n3\n\njulia> indmin([7,1,1,6])\n2\n\njulia> indmin([7,1,1,NaN])\n2\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.findmax-Tuple{Any}",
    "page": "Collections and Data Structures",
    "title": "Base.findmax",
    "category": "Method",
    "text": "findmax(itr) -> (x, index)\n\nReturns the maximum element of the collection itr and its index. If there are multiple maximal elements, then the first one will be returned. NaN values are ignored, unless all elements are NaN.\n\nThe collection must not be empty.\n\nExamples\n\njulia> findmax([8,0.1,-9,pi])\n(8.0, 1)\n\njulia> findmax([1,7,7,6])\n(7, 2)\n\njulia> findmax([1,7,7,NaN])\n(7.0, 2)\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.findmax-Tuple{AbstractArray,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.findmax",
    "category": "Method",
    "text": "findmax(A, region) -> (maxval, index)\n\nFor an array input, returns the value and index of the maximum over the given region.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Array{Int64,2}:\n 1  2\n 3  4\n\njulia> findmax(A,1)\n([3 4], [2 4])\n\njulia> findmax(A,2)\n([2; 4], [3; 4])\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.findmin-Tuple{Any}",
    "page": "Collections and Data Structures",
    "title": "Base.findmin",
    "category": "Method",
    "text": "findmin(itr) -> (x, index)\n\nReturns the minimum element of the collection itr and its index. If there are multiple minimal elements, then the first one will be returned. NaN values are ignored, unless all elements are NaN.\n\nThe collection must not be empty.\n\nExamples\n\njulia> findmin([8,0.1,-9,pi])\n(-9.0, 3)\n\njulia> findmin([7,1,1,6])\n(1, 2)\n\njulia> findmin([7,1,1,NaN])\n(1.0, 2)\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.findmin-Tuple{AbstractArray,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.findmin",
    "category": "Method",
    "text": "findmin(A, region) -> (minval, index)\n\nFor an array input, returns the value and index of the minimum over the given region.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Array{Int64,2}:\n 1  2\n 3  4\n\njulia> findmin(A, 1)\n([1 2], [1 3])\n\njulia> findmin(A, 2)\n([1; 3], [1; 2])\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.findmax!",
    "page": "Collections and Data Structures",
    "title": "Base.findmax!",
    "category": "Function",
    "text": "findmax!(rval, rind, A, [init=true]) -> (maxval, index)\n\nFind the maximum of A and the corresponding linear index along singleton dimensions of rval and rind, and store the results in rval and rind.\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.findmin!",
    "page": "Collections and Data Structures",
    "title": "Base.findmin!",
    "category": "Function",
    "text": "findmin!(rval, rind, A, [init=true]) -> (minval, index)\n\nFind the minimum of A and the corresponding linear index along singleton dimensions of rval and rind, and store the results in rval and rind.\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.sum",
    "page": "Collections and Data Structures",
    "title": "Base.sum",
    "category": "Function",
    "text": "sum(f, itr)\n\nSum the results of calling function f on each element of itr.\n\njulia> sum(abs2, [2; 3; 4])\n29\n\n\n\nsum(itr)\n\nReturns the sum of all elements in a collection.\n\njulia> sum(1:20)\n210\n\n\n\nsum(A, dims)\n\nSum elements of an array over the given dimensions.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Array{Int64,2}:\n 1  2\n 3  4\n\njulia> sum(A, 1)\n1×2 Array{Int64,2}:\n 4  6\n\njulia> sum(A, 2)\n2×1 Array{Int64,2}:\n 3\n 7\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.sum!",
    "page": "Collections and Data Structures",
    "title": "Base.sum!",
    "category": "Function",
    "text": "sum!(r, A)\n\nSum elements of A over the singleton dimensions of r, and write results to r.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Array{Int64,2}:\n 1  2\n 3  4\n\njulia> sum!([1; 1], A)\n2-element Array{Int64,1}:\n 3\n 7\n\njulia> sum!([1 1], A)\n1×2 Array{Int64,2}:\n 4  6\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.prod",
    "page": "Collections and Data Structures",
    "title": "Base.prod",
    "category": "Function",
    "text": "prod(f, itr)\n\nReturns the product of f applied to each element of itr.\n\njulia> prod(abs2, [2; 3; 4])\n576\n\n\n\nprod(itr)\n\nReturns the product of all elements of a collection.\n\njulia> prod(1:20)\n2432902008176640000\n\n\n\nprod(A, dims)\n\nMultiply elements of an array over the given dimensions.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Array{Int64,2}:\n 1  2\n 3  4\n\njulia> prod(A, 1)\n1×2 Array{Int64,2}:\n 3  8\n\njulia> prod(A, 2)\n2×1 Array{Int64,2}:\n  2\n 12\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.prod!",
    "page": "Collections and Data Structures",
    "title": "Base.prod!",
    "category": "Function",
    "text": "prod!(r, A)\n\nMultiply elements of A over the singleton dimensions of r, and write results to r.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Array{Int64,2}:\n 1  2\n 3  4\n\njulia> prod!([1; 1], A)\n2-element Array{Int64,1}:\n  2\n 12\n\njulia> prod!([1 1], A)\n1×2 Array{Int64,2}:\n 3  8\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.any-Tuple{Any}",
    "page": "Collections and Data Structures",
    "title": "Base.any",
    "category": "Method",
    "text": "any(itr) -> Bool\n\nTest whether any elements of a boolean collection are true, returning true as soon as the first true value in itr is encountered (short-circuiting).\n\njulia> a = [true,false,false,true]\n4-element Array{Bool,1}:\n  true\n false\n false\n  true\n\njulia> any(a)\ntrue\n\njulia> any((println(i); v) for (i, v) in enumerate(a))\n1\ntrue\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.any-Tuple{AbstractArray,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.any",
    "category": "Method",
    "text": "any(A, dims)\n\nTest whether any values along the given dimensions of an array are true.\n\nExamples\n\njulia> A = [true false; true false]\n2×2 Array{Bool,2}:\n true  false\n true  false\n\njulia> any(A, 1)\n1×2 Array{Bool,2}:\n true  false\n\njulia> any(A, 2)\n2×1 Array{Bool,2}:\n true\n true\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.any!",
    "page": "Collections and Data Structures",
    "title": "Base.any!",
    "category": "Function",
    "text": "any!(r, A)\n\nTest whether any values in A along the singleton dimensions of r are true, and write results to r.\n\nExamples\n\njulia> A = [true false; true false]\n2×2 Array{Bool,2}:\n true  false\n true  false\n\njulia> any!([1; 1], A)\n2-element Array{Int64,1}:\n 1\n 1\n\njulia> any!([1 1], A)\n1×2 Array{Int64,2}:\n 1  0\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.all-Tuple{Any}",
    "page": "Collections and Data Structures",
    "title": "Base.all",
    "category": "Method",
    "text": "all(itr) -> Bool\n\nTest whether all elements of a boolean collection are true, returning false as soon as the first false value in itr is encountered (short-circuiting).\n\njulia> a = [true,false,false,true]\n4-element Array{Bool,1}:\n  true\n false\n false\n  true\n\njulia> all(a)\nfalse\n\njulia> all((println(i); v) for (i, v) in enumerate(a))\n1\n2\nfalse\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.all-Tuple{AbstractArray,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.all",
    "category": "Method",
    "text": "all(A, dims)\n\nTest whether all values along the given dimensions of an array are true.\n\nExamples\n\njulia> A = [true false; true true]\n2×2 Array{Bool,2}:\n true  false\n true   true\n\njulia> all(A, 1)\n1×2 Array{Bool,2}:\n true  false\n\njulia> all(A, 2)\n2×1 Array{Bool,2}:\n false\n  true\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.all!",
    "page": "Collections and Data Structures",
    "title": "Base.all!",
    "category": "Function",
    "text": "all!(r, A)\n\nTest whether all values in A along the singleton dimensions of r are true, and write results to r.\n\nExamples\n\njulia> A = [true false; true false]\n2×2 Array{Bool,2}:\n true  false\n true  false\n\njulia> all!([1; 1], A)\n2-element Array{Int64,1}:\n 0\n 0\n\njulia> all!([1 1], A)\n1×2 Array{Int64,2}:\n 1  0\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.count",
    "page": "Collections and Data Structures",
    "title": "Base.count",
    "category": "Function",
    "text": "count(p, itr) -> Integer\ncount(itr) -> Integer\n\nCount the number of elements in itr for which predicate p returns true. If p is omitted, counts the number of true elements in itr (which should be a collection of boolean values).\n\njulia> count(i->(4<=i<=6), [2,3,4,5,6])\n3\n\njulia> count([true, false, true, true])\n3\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.any-Tuple{Any,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.any",
    "category": "Method",
    "text": "any(p, itr) -> Bool\n\nDetermine whether predicate p returns true for any elements of itr, returning true as soon as the first item in itr for which p returns true is encountered (short-circuiting).\n\njulia> any(i->(4<=i<=6), [3,5,7])\ntrue\n\njulia> any(i -> (println(i); i > 3), 1:10)\n1\n2\n3\n4\ntrue\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.all-Tuple{Any,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.all",
    "category": "Method",
    "text": "all(p, itr) -> Bool\n\nDetermine whether predicate p returns true for all elements of itr, returning false as soon as the first item in itr for which p returns false is encountered (short-circuiting).\n\njulia> all(i->(4<=i<=6), [4,5,6])\ntrue\n\njulia> all(i -> (println(i); i < 3), 1:10)\n1\n2\n3\nfalse\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.foreach",
    "page": "Collections and Data Structures",
    "title": "Base.foreach",
    "category": "Function",
    "text": "foreach(f, c...) -> Void\n\nCall function f on each element of iterable c. For multiple iterable arguments, f is called elementwise. foreach should be used instead of map when the results of f are not needed, for example in foreach(println, array).\n\nExample\n\njulia> a = 1:3:7;\n\njulia> foreach(x -> println(x^2), a)\n1\n16\n49\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.map",
    "page": "Collections and Data Structures",
    "title": "Base.map",
    "category": "Function",
    "text": "map(f, c...) -> collection\n\nTransform collection c by applying f to each element. For multiple collection arguments, apply f elementwise.\n\nExamples\n\njulia> map(x -> x * 2, [1, 2, 3])\n3-element Array{Int64,1}:\n 2\n 4\n 6\n\njulia> map(+, [1, 2, 3], [10, 20, 30])\n3-element Array{Int64,1}:\n 11\n 22\n 33\n\n\n\nmap(f, x::Nullable)\n\nReturn f applied to the value of x if it has one, as a Nullable. If x is null, then return a null value of type Nullable{S}. S is guaranteed to be either Union{} or a concrete type. Whichever of these is chosen is an implementation detail, but typically the choice that maximizes performance would be used. If x has a value, then the return type is guaranteed to be of type Nullable{typeof(f(x))}.\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.map!",
    "page": "Collections and Data Structures",
    "title": "Base.map!",
    "category": "Function",
    "text": "map!(function, destination, collection...)\n\nLike map, but stores the result in destination rather than a new collection. destination must be at least as large as the first collection.\n\nExample\n\njulia> x = zeros(3);\n\njulia> map!(x -> x * 2, x, [1, 2, 3]);\n\njulia> x\n3-element Array{Float64,1}:\n 2.0\n 4.0\n 6.0\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.mapreduce-NTuple{4,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.mapreduce",
    "category": "Method",
    "text": "mapreduce(f, op, v0, itr)\n\nApply function f to each element in itr, and then reduce the result using the binary function op. v0 must be a neutral element for op that will be returned for empty collections. It is unspecified whether v0 is used for non-empty collections.\n\nmapreduce is functionally equivalent to calling reduce(op, v0, map(f, itr)), but will in general execute faster since no intermediate collection needs to be created. See documentation for reduce and map.\n\njulia> mapreduce(x->x^2, +, [1:3;]) # == 1 + 4 + 9\n14\n\nThe associativity of the reduction is implementation-dependent. Additionally, some implementations may reuse the return value of f for elements that appear multiple times in itr. Use mapfoldl or mapfoldr instead for guaranteed left or right associativity and invocation of f for every value.\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.mapreduce-Tuple{Any,Any,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.mapreduce",
    "category": "Method",
    "text": "mapreduce(f, op, itr)\n\nLike mapreduce(f, op, v0, itr). In general, this cannot be used with empty collections (see reduce(op, itr)).\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.mapfoldl-NTuple{4,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.mapfoldl",
    "category": "Method",
    "text": "mapfoldl(f, op, v0, itr)\n\nLike mapreduce, but with guaranteed left associativity, as in foldl. v0 will be used exactly once.\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.mapfoldl-Tuple{Any,Any,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.mapfoldl",
    "category": "Method",
    "text": "mapfoldl(f, op, itr)\n\nLike mapfoldl(f, op, v0, itr), but using the first element of itr as v0. In general, this cannot be used with empty collections (see reduce(op, itr)).\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.mapfoldr-NTuple{4,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.mapfoldr",
    "category": "Method",
    "text": "mapfoldr(f, op, v0, itr)\n\nLike mapreduce, but with guaranteed right associativity, as in foldr. v0 will be used exactly once.\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.mapfoldr-Tuple{Any,Any,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.mapfoldr",
    "category": "Method",
    "text": "mapfoldr(f, op, itr)\n\nLike mapfoldr(f, op, v0, itr), but using the first element of itr as v0. In general, this cannot be used with empty collections (see reduce(op, itr)).\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.first",
    "page": "Collections and Data Structures",
    "title": "Base.first",
    "category": "Function",
    "text": "first(coll)\n\nGet the first element of an iterable collection. Returns the start point of a Range even if it is empty.\n\njulia> first(2:2:10)\n2\n\njulia> first([1; 2; 3; 4])\n1\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.last",
    "page": "Collections and Data Structures",
    "title": "Base.last",
    "category": "Function",
    "text": "last(coll)\n\nGet the last element of an ordered collection, if it can be computed in O(1) time. This is accomplished by calling endof to get the last index. Returns the end point of a Range even if it is empty.\n\njulia> last(1:2:10)\n9\n\njulia> last([1; 2; 3; 4])\n4\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.step",
    "page": "Collections and Data Structures",
    "title": "Base.step",
    "category": "Function",
    "text": "step(r)\n\nGet the step size of a Range object.\n\njulia> step(1:10)\n1\n\njulia> step(1:2:10)\n2\n\njulia> step(2.5:0.3:10.9)\n0.3\n\njulia> step(linspace(2.5,10.9,85))\n0.1\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.collect-Tuple{Any}",
    "page": "Collections and Data Structures",
    "title": "Base.collect",
    "category": "Method",
    "text": "collect(collection)\n\nReturn an Array of all items in a collection or iterator. For associative collections, returns Pair{KeyType, ValType}. If the argument is array-like or is an iterator with the HasShape() trait, the result will have the same shape and number of dimensions as the argument.\n\nExample\n\njulia> collect(1:2:13)\n7-element Array{Int64,1}:\n  1\n  3\n  5\n  7\n  9\n 11\n 13\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.collect-Tuple{Type,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.collect",
    "category": "Method",
    "text": "collect(element_type, collection)\n\nReturn an Array with the given element type of all items in a collection or iterable. The result has the same shape and number of dimensions as collection.\n\njulia> collect(Float64, 1:2:5)\n3-element Array{Float64,1}:\n 1.0\n 3.0\n 5.0\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.issubset-Tuple{Any,Any}",
    "page": "Collections and Data Structures",
    "title": "Base.issubset",
    "category": "Method",
    "text": "issubset(a, b)\n⊆(a,b) -> Bool\n⊈(a,b) -> Bool\n⊊(a,b) -> Bool\n\nDetermine whether every element of a is also in b, using in.\n\nExamples\n\njulia> issubset([1, 2], [1, 2, 3])\ntrue\n\njulia> issubset([1, 2, 3], [1, 2])\nfalse\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.filter",
    "page": "Collections and Data Structures",
    "title": "Base.filter",
    "category": "Function",
    "text": "filter(function, collection)\n\nReturn a copy of collection, removing elements for which function is false. For associative collections, the function is passed two arguments (key and value).\n\nExamples\n\njulia> a = 1:10\n1:10\n\njulia> filter(isodd, a)\n5-element Array{Int64,1}:\n 1\n 3\n 5\n 7\n 9\n\njulia> d = Dict(1=>\"a\", 2=>\"b\")\nDict{Int64,String} with 2 entries:\n  2 => \"b\"\n  1 => \"a\"\n\njulia> filter((x,y)->isodd(x), d)\nDict{Int64,String} with 1 entry:\n  1 => \"a\"\n\n\n\nfilter(p, x::Nullable)\n\nReturn null if either x is null or p(get(x)) is false, and x otherwise.\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Base.filter!",
    "page": "Collections and Data Structures",
    "title": "Base.filter!",
    "category": "Function",
    "text": "filter!(function, collection)\n\nUpdate collection, removing elements for which function is false. For associative collections, the function is passed two arguments (key and value).\n\nExample\n\njulia> filter!(isodd, collect(1:10))\n5-element Array{Int64,1}:\n 1\n 3\n 5\n 7\n 9\n\n\n\n"
},

{
    "location": "stdlib/collections.html#Iterable-Collections-1",
    "page": "Collections and Data Structures",
    "title": "Iterable Collections",
    "category": "section",
    "text": "Base.in\nBase.eltype\nBase.indexin\nBase.findin\nBase.unique\nBase.allunique\nBase.reduce(::Any, ::Any, ::Any)\nBase.reduce(::Any, ::Any)\nBas