/*
 * This software is
 * Copyright (c) 2012, 2013 Lukas Odzioba <ukasz at openwall dot net>
 * Copyright (c) 2014 JimF
 * Copyright (c) 2014 magnum
 * and it is hereby released to the general public under the following terms:
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted.
 */
#ifdef HAVE_OPENCL

#if FMT_EXTERNS_H
extern struct fmt_main fmt_opencl_pbkdf2_hmac_sha512;
#elif FMT_REGISTERS_H
john_register_one(&fmt_opencl_pbkdf2_hmac_sha512);
#else

#include <ctype.h>
#include <string.h>
#include <assert.h>
#include "misc.h"
#include "arch.h"
#include "common.h"
#include "formats.h"
#include "options.h"
#include "common-opencl.h"
#include "stdint.h"
#include "johnswap.h"

#define NUUL NULL
#define FORMAT_LABEL		"pbkdf2-hmac-sha512-opencl"
#define FORMAT_NAME		    "GRUB2 / OS X 10.8+"
#define ALGORITHM_NAME		"PBKDF2-SHA512 OpenCL"

#define BENCHMARK_COMMENT	", rounds=10000"
#define BENCHMARK_LENGTH	-1

#define BINARY_ALIGN		8
#define SALT_ALIGN		8

#define MAX_CIPHERTEXT_LENGTH   1024
#define MAX_BINARY_SIZE         (4*64)
#define MAX_SALT_SIZE           107 /* NOTE, we also add 0001 and 0x80 */
#define PLAINTEXT_LENGTH	110
#define BINARY_SIZE		64
#define	SALT_SIZE		sizeof(salt_t)

#define FORMAT_TAG              "$pbkdf2-hmac-sha512$"
#define FORMAT_TAG2             "$ml$"
#define FORMAT_TAG3             "grub.pbkdf2.sha512."

#define KERNEL_NAME             "pbkdf2_sha512_kernel"
#define SPLIT_KERNEL_NAME	"pbkdf2_sha512_loop"
#define CONFIG_NAME             "pbkdf2_sha512"

#define MIN(a,b)		(((a)<(b))?(a):(b))
#define HASH_LOOPS		250
#define ITERATIONS		10000

typedef struct {
    // for plaintext, we must make sure it is a full uint64 width.
	uint64_t v[(PLAINTEXT_LENGTH+7)/8]; // v must be kept aligned(8)
	uint64_t length; // keep 64 bit aligned. length is overkill, but easiest way to stay aligned.
} pass_t;

typedef struct {
	uint64_t hash[8];
} crack_t;

typedef struct {
	// for salt, we append \x00\x00\x00\x01\x80 and must make sure it is a full uint64 width
	uint64_t salt[(MAX_SALT_SIZE+1+4+7)/8]; // salt must be kept aligned(8)
	uint32_t length;
	uint32_t rounds;
} salt_t;

typedef struct {
	cl_ulong ipad[8];
	cl_ulong opad[8];
	cl_ulong hash[8];
	cl_ulong W[8];
	cl_uint rounds;
} state_t;

/** Testcases generated by python passlib **/
static struct fmt_tests tests[] = {
	/* GRUB hash, GRUB format */
	{"grub.pbkdf2.sha512.10000.4483972AD2C52E1F590B3E2260795FDA9CA0B07B96FF492814CA9775F08C4B59CD1707F10B269E09B61B1E2D11729BCA8D62B7827B25B093EC58C4C1EAC23137.DF4FCB5DD91340D6D31E33423E4210AD47C7A4DF9FA16F401663BF288C20BF973530866178FE6D134256E4DBEFBD984B652332EED3ACAED834FEA7B73CAE851D", "password"},
	/* Canonical format */
	{"$pbkdf2-hmac-sha512$10000.82dbab96e072834d1f725db9aded51e703f1d449e77d01f6be65147a765c997d8865a1f1ab50856aa3b08d25a5602fe538ad0757b8ca2933fe8ca9e7601b3fbf.859d65960e6d05c6858f3d63fa35c6adfc456637b4a0a90f6afa7d8e217ce2d3dfdc56c8deaca217f0864ae1efb4a09b00eb84cf9e4a2723534f34e26a279193", "openwall"},
	/* max length password (and longer salt) made by pass_gen.pl */
	/*110*/
	{"$pbkdf2-hmac-sha512$10000.78783334353637383930313233343536373839303132333435363738393031323334353637383930313233343536373839303132333435363738393031323334.33cfd654a173d39fcae804ba07744d276d184ca85f1e422a623aa7eec66b6bc372074551f9ded2bdff225a9afc22f4f0565d32fab2a0a639f81dddd8f347523e","12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"},
	/* 80 */
	{"$pbkdf2-hmac-sha512$10000.78783334353637383930313233343536373839303132333435363738393031323334353637383930313233343536373839303132333435363738393031323334.302c2f7f2a6366f4cc854aca7c74be3b5b7c2110c05d3a0700e51740b4fbb929a233e9fad8e36c6b7f8d80a3ede00dbef057a95ffef96a998f234d44c84608fd","12345678901234567890123456789012345678901234567890123456789012345678901234567890"},
	/* 107 salt, 110 pw */
	{"$pbkdf2-hmac-sha512$10000.7878333435363738393031323334353637383930313233343536373839303132333435363738393031323334353637383930313233343536373839303132333435363738393031323334353637383930313233343536373839303132333435363738393031323334353637.8e3b24e8dd572201c5294edb2605ce14acbd9645f616bb9b3be8e558d2ec2018fbb52df026fef71854cf0277e2a5adb3162e93c9e897e21368e5091f3a581598","12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"},
	/* OS X 10.8 Mountain Lion hashes, "dave" format, large #iterations */
	{"$ml$23923$c3fa2e153466f7619286024fe7d812d0a8ae836295f84b9133ccc65456519fc3$ccb903ee691ade6d5dee9b3c6931ebed6ddbb1348f1b26c21add8ba0d45f27e61e97c0b80d9a18020944bb78f1ebda6fdd79c5cf08a12c80522caf987c287b6d", "openwall"},
	{"$ml$37174$ef768765ba15907760b71789fe62436e3584dfadbbf1eb8bf98673b60ff4e12b$294d42f6e0c3a93d598340bfb256efd630b53f32173c2b0d278eafab3753c10ec57b7d66e0fa79be3b80b3693e515cdd06e9e9d26d665b830159dcae152ad156", "m\xC3\xBCller"},
	{"$ml$24213$db9168b655339be3ff8936d2cf3cb573bdf7d40afd9a17fca439a0fae1375960$471a868524d66d995c6a8b7a0d27bbbc1af0c203f1ac31e7ceb2fde92f94997b887b38131ac2b543d285674dce639560997136c9af91916a2865ba960762196f", "applecrap"},
	{"$ml$37313$189dce2ede21e297a8884d0a33e4431107e3e40866f3c493e5f9506c2bd2fe44$948010870e110a6d185494799552d8cf30b0203c6706ab06e6270bf0ac17d496d820c5a75c12caf9070051f34acd2a2911bb38b202eebd4413e571e4fbff883e75f35c36c88a2b42a4fb521a97953438c72b2182fd9c5bba902395766e703b52b9aaa3895770d3cebffbee05076d9110ebb9f0342692a238174655b1acdce1c0", "crackable4us"},
	{NULL}
};

// #define DEBUG
static pass_t *host_pass;			      /** plain ciphertexts **/
static salt_t *host_salt;			      /** salt **/
static crack_t *host_crack;			      /** cracked or no **/
static cl_mem mem_in, mem_out, mem_salt, mem_state;
static cl_kernel split_kernel;
static cl_int cl_error;

#define STEP			0
#define SEED			256
#define OCL_CONFIG		"pbkdf2-hmac-sha512"

static const char * warn[] = {
        "xfer: ",  ", init: " , ", crypt: ", ", res xfer: "
};

static int split_events[] = { 2, -1, -1 };

//This file contains auto-tuning routine(s). Has to be included after formats definitions.
#include "opencl-autotune.h"
#include "memdbg.h"

/* ------- Helper functions ------- */
static size_t get_task_max_work_group_size()
{
	return autotune_get_task_max_work_group_size(FALSE, 0, crypt_kernel);
}

static size_t get_task_max_size()
{
	return 0;
}

static size_t get_default_workgroup()
{
	if (cpu(device_info[gpu_id]))
		return get_platform_vendor_id(platform_id) == DEV_INTEL ?
			8 : 1;
	else
		return 64;
}

static void create_clobj(size_t kpc, struct fmt_main *self)
{
	host_pass = mem_calloc(kpc * sizeof(pass_t));
	host_crack = mem_calloc(kpc * sizeof(crack_t));
	host_salt = mem_calloc(sizeof(salt_t));
#define CL_RO CL_MEM_READ_ONLY
#define CL_WO CL_MEM_WRITE_ONLY
#define CL_RW CL_MEM_READ_WRITE

#define CLCREATEBUFFER(_flags, _size, _string)	  \
	clCreateBuffer(context[gpu_id], _flags, _size, NULL, &cl_error); \
	HANDLE_CLERROR(cl_error, _string);

#define CLKERNELARG(kernel, id, arg, msg)	  \
	HANDLE_CLERROR(clSetKernelArg(kernel, id, sizeof(arg), &arg), msg);

	mem_in = CLCREATEBUFFER(CL_RO, kpc * sizeof(pass_t),
	                        "Cannot allocate mem in");
	mem_salt = CLCREATEBUFFER(CL_RO, sizeof(salt_t),
	                          "Cannot allocate mem salt");
	mem_out = CLCREATEBUFFER(CL_WO, kpc * sizeof(crack_t),
	                         "Cannot allocate mem out");
	mem_state = CLCREATEBUFFER(CL_RW, kpc * sizeof(state_t),
	                           "Cannot allocate mem state");

	CLKERNELARG(crypt_kernel, 0, mem_in, "Error while setting mem_in");
	CLKERNELARG(crypt_kernel, 1, mem_salt, "Error while setting mem_salt");
	CLKERNELARG(crypt_kernel, 2, mem_state, "Error while setting mem_state");

	CLKERNELARG(split_kernel, 0, mem_state, "Error while setting mem_state");
	CLKERNELARG(split_kernel, 1 ,mem_out, "Error while setting mem_out");
}

static int crypt_all(int *pcount, struct db_salt *_salt);
static int crypt_all_benchmark(int *pcount, struct db_salt *_salt);

static void init(struct fmt_main *self)
{
	char build_opts[128];

	snprintf(build_opts, sizeof(build_opts),
	         "-DHASH_LOOPS=%u -DPLAINTEXT_LENGTH=%d -DMAX_SALT_SIZE=%d",
	         HASH_LOOPS, PLAINTEXT_LENGTH, MAX_SALT_SIZE);

	opencl_init("$JOHN/kernels/pbkdf2_hmac_sha512_kernel.cl",
	            gpu_id, build_opts);

	crypt_kernel = clCreateKernel(program[gpu_id], KERNEL_NAME, &cl_error);
	HANDLE_CLERROR(cl_error, "Error creating kernel");

	split_kernel =
	    clCreateKernel(program[gpu_id], SPLIT_KERNEL_NAME, &cl_error);
	HANDLE_CLERROR(cl_error, "Error creating split kernel");

	//Initialize openCL tuning (library) for this format.
	opencl_init_auto_setup(SEED, HASH_LOOPS, split_events,
		warn, 2, self, create_clobj, release_clobj,
		sizeof(state_t), 0);

	//Auto tune execution from shared/included code.
	self->methods.crypt_all = crypt_all_benchmark;
	autotune_run(self, ITERATIONS, 0,
	             (cpu(device_info[gpu_id]) ? 1000000000 : 10000000000ULL));
	self->methods.crypt_all = crypt_all;
}

static void release_clobj(void)
{
	MEM_FREE(host_pass);
	MEM_FREE(host_salt);
	MEM_FREE(host_crack);

	HANDLE_CLERROR(clReleaseMemObject(mem_in), "Release mem in");
	HANDLE_CLERROR(clReleaseMemObject(mem_salt), "Release mem salt");
	HANDLE_CLERROR(clReleaseMemObject(mem_out), "Release mem out");
	HANDLE_CLERROR(clReleaseMemObject(mem_state), "Release mem state");
}

static int ishex(char *q)
{
	while (atoi16[ARCH_INDEX(*q)] != 0x7F)
		q++;
	return !*q;
}

static void done(void)
{
	release_clobj();
	HANDLE_CLERROR(clReleaseKernel(crypt_kernel), "Release kernel");
	HANDLE_CLERROR(clReleaseProgram(program[gpu_id]), "Release Program");
}

static int isdecu(char *q)
{
	char buf[24];
	uint32_t x = atoi(q); /* this is how it is 'used', atoi() to unsigned */
	sprintf(buf, "%u", x);
	return !strcmp(q,buf);
}

static int valid(char *ciphertext, struct fmt_main *self)
{
	char *ptr, *ctcopy, *keeptr;
	size_t len;

	if (strncmp(ciphertext, FORMAT_TAG, sizeof(FORMAT_TAG) - 1))
		return 0;
	if (strlen(ciphertext) > MAX_CIPHERTEXT_LENGTH)
		return 0;
	ciphertext += sizeof(FORMAT_TAG) - 1;
	if (!(ctcopy = strdup(ciphertext)))
		return 0;
	keeptr = ctcopy;
	if (!(ptr = strtok(ctcopy, ".")))
		goto error;
	if (strlen(ptr) >= 10)
		goto error;
	if (!isdecu(ptr))
		goto error;
	if (!(ptr = strtok(NULL, ".")))
		goto error;
	len = strlen(ptr); // salt length
	if (len > 2 * MAX_SALT_SIZE || len & 1)
		goto error;
	if (!ishex(ptr))
		goto error;
	if (!(ptr = strtok(NULL, ".")))
		goto error;
	len = strlen(ptr); // binary length
	if (len < BINARY_SIZE || len > MAX_BINARY_SIZE || len & 1)
		goto error;
	if (!ishex(ptr))
		goto error;
	MEM_FREE(keeptr);
	return 1;
error:
	MEM_FREE(keeptr);
	return 0;
}

/* This converts any input format to the canonical $pbkdf2-hmac-sha512$ */
static char *prepare(char *split_fields[10], struct fmt_main *self)
{
	static char out[MAX_CIPHERTEXT_LENGTH + 1];
	int i;

	if (!*split_fields[1])
		return split_fields[1];

	/* Unify format */
	if (!strncmp(split_fields[1], FORMAT_TAG, sizeof(FORMAT_TAG)-1))
		i = sizeof(FORMAT_TAG) - 1;
	else if (!strncmp(split_fields[1], FORMAT_TAG2, sizeof(FORMAT_TAG2)-1))
		i = sizeof(FORMAT_TAG2) - 1;
	else if (!strncmp(split_fields[1], FORMAT_TAG3, sizeof(FORMAT_TAG3)-1))
		i = sizeof(FORMAT_TAG3) - 1;
	else
		return split_fields[1];

	strcpy(out, FORMAT_TAG);
	strncat(out, &split_fields[1][i], sizeof(out) - 1);

	if (!strncmp(split_fields[1], FORMAT_TAG2, sizeof(FORMAT_TAG2) - 1))
		for (i = sizeof(FORMAT_TAG); out[i]; i++)
			if (out[i] == '$')
				out[i] = '.';

	if (valid(out, self))
		return out;
	else
		return split_fields[1];
}

static char *split(char *ciphertext, int index, struct fmt_main *self)
{
	static char out[MAX_CIPHERTEXT_LENGTH + 1];

	strnzcpy(out, ciphertext, sizeof(out));
	strlwr(out);
	return out;
}

static void *binary(char *ciphertext)
{
	static union {
		uint64_t swp[BINARY_SIZE/8];
		uint8_t ret[BINARY_SIZE];
	}u;
	int i = 0;
	//int len;
	char *p,delim;

	delim = strchr(ciphertext, '.') ? '.' : '$';
	p = strrchr(ciphertext, delim) + 1;
	//len = strlen(p) / 2;
	//for (i = 0; i < len && *p; i++) {
	for (i = 0; i < BINARY_SIZE && *p; i++) {
		u.ret[i] =
			(atoi16[ARCH_INDEX(*p)] << 4) |
			atoi16[ARCH_INDEX(p[1])];
		p += 2;
	}
	// swap here, so we do not have to swap at end of GPU code.
	for (i = 0; i < BINARY_SIZE/8; ++i)
		u.swp[i] = JOHNSWAP64(u.swp[i]);
	return u.ret;
}

static void *get_salt(char *ciphertext)
{
	static salt_t cs;
	uint8_t salt[MAX_SALT_SIZE+1+4+1];
	char *p;
	int saltlen;
	char delim;

	if (!strncmp(ciphertext, FORMAT_TAG, sizeof(FORMAT_TAG) - 1))
		ciphertext += sizeof(FORMAT_TAG) - 1;
	else if (!strncmp(ciphertext, FORMAT_TAG2, sizeof(FORMAT_TAG2) - 1))
		ciphertext += sizeof(FORMAT_TAG2) - 1;
	else if (!strncmp(ciphertext, FORMAT_TAG3, sizeof(FORMAT_TAG3) - 1))
		ciphertext += sizeof(FORMAT_TAG3) - 1;
	else
		error(); /* Can't happen - caught in valid() */
	memset(&cs, 0, sizeof(cs));
	cs.rounds = atoi(ciphertext);
	delim = strchr(ciphertext, '.') ? '.' : '$';
	ciphertext = strchr(ciphertext, delim) + 1;
	p = strchr(ciphertext, delim);
	saltlen = 0;
	while (ciphertext < p) {        /** extract salt **/
		salt[saltlen++] =
			atoi16[ARCH_INDEX(ciphertext[0])] * 16 +
			atoi16[ARCH_INDEX(ciphertext[1])];
		ciphertext += 2;
	}
	// we append the count and EOM here, one time.
	memcpy(&salt[saltlen], "\x0\x0\x0\x1\x80", 5);
	memcpy(cs.salt, salt, saltlen+5);
	cs.length = saltlen+5; // we include the x80 byte in our saltlen, but the .cl kernel knows to reduce saltlen by 1

	return (void *)&cs;
}


static void set_salt(void *salt)
{
	memcpy(host_salt, salt, SALT_SIZE);
	HANDLE_CLERROR(clEnqueueWriteBuffer(queue[gpu_id], mem_salt,
		CL_FALSE, 0, sizeof(salt_t), host_salt, 0, NUUL, NULL),
		"Copy salt to gpu");
}


static void opencl_limit_gws(int count)
{
	global_work_size =
	    (count + local_work_size - 1) / local_work_size * local_work_size;
}

static int crypt_all(int *pcount, struct db_salt *salt)
{
	int i, count = *pcount;
	int loops = (host_salt->rounds + HASH_LOOPS - 1) / HASH_LOOPS;

	opencl_limit_gws(count);

#if 0
	printf("crypt_all(%d)\n", count);
	printf("LWS = %d, GWS = %d, loops=%d\n",(int)local_work_size, (int)global_work_size, loops);
#endif

	/// Copy data to gpu
	HANDLE_CLERROR(clEnqueueWriteBuffer(queue[gpu_id], mem_in, CL_FALSE, 0,
		global_work_size * sizeof(pass_t), host_pass, 0, NUUL,
		NULL), "Copy data to gpu");

	/// Run kernel
	HANDLE_CLERROR(clEnqueueNDRangeKernel(queue[gpu_id], crypt_kernel, 1,
		NUUL, &global_work_size, &local_work_size, 0, NULL,
		NULL), "Run kernel");

	for(i = 0; i < loops; i++) {
		HANDLE_CLERROR(clEnqueueNDRangeKernel(queue[gpu_id],
		        split_kernel,
			1, NULL, &global_work_size, &local_work_size, 0, NULL,
			NULL), "Run split kernel");
		HANDLE_CLERROR(clFinish(queue[gpu_id]), "clFinish");
		opencl_process_event();
	}
	/// Read the result back
	HANDLE_CLERROR(clEnqueueReadBuffer(queue[gpu_id], mem_out, CL_TRUE, 0,
		global_work_size * sizeof(crack_t), host_crack, 0, NUUL,
		 NULL), "Copy result back");

	return count;
}

static int crypt_all_benchmark(int *pcount, struct db_salt *salt)
{
	int count = *pcount;

	opencl_limit_gws(count);

#if 0
	printf("crypt_all_benchmark(%d)\n", count);
	printf("LWS = %d, GWS = %d\n",(int)local_work_size, (int)global_work_size);
#endif

	/// Copy data to gpu
	BENCH_CLERROR(clEnqueueWriteBuffer(queue[gpu_id], mem_in, CL_FALSE, 0,
		global_work_size * sizeof(pass_t), host_pass, 0, NUUL,
		multi_profilingEvent[0]), "Copy data to gpu");

	/// Run kernel
	BENCH_CLERROR(clEnqueueNDRangeKernel(queue[gpu_id], crypt_kernel, 1,
		NUUL, &global_work_size, &local_work_size, 0, NULL,
		multi_profilingEvent[1]), "Run kernel");
	BENCH_CLERROR(clFinish(queue[gpu_id]), "clFinish");


	BENCH_CLERROR(clEnqueueNDRangeKernel(queue[gpu_id], split_kernel,
		1, NULL, &global_work_size, &local_work_size, 0, NULL,
		NULL), "Run split kernel");
	BENCH_CLERROR(clFinish(queue[gpu_id]), "clFinish");

	BENCH_CLERROR(clEnqueueNDRangeKernel(queue[gpu_id], split_kernel,
		1, NULL, &global_work_size, &local_work_size, 0, NULL,
		multi_profilingEvent[2]), "Run split kernel");
	BENCH_CLERROR(clFinish(queue[gpu_id]), "clFinish");

	/// Read the result back
	BENCH_CLERROR(clEnqueueReadBuffer(queue[gpu_id], mem_out, CL_FALSE, 0,
		global_work_size * sizeof(crack_t), host_crack, 0, NUUL,
		 multi_profilingEvent[3]), "Copy result back");

	/// Await completion of all the above
	BENCH_CLERROR(clFinish(queue[gpu_id]), "clFinish");

	return count;
}

static int cmp_all(void *binary, int count)
{
	int i;
	for (i = 0; i < count; i++) {
		if (host_crack[i].hash[0] == ((uint64_t*)binary)[0])
			return 1;
	}
	return 0;
}

static int cmp_one(void *binary, int index)
{
	return host_crack[index].hash[0] == ((uint64_t*)binary)[0];
}

static int cmp_exact(char *source, int index)
{
	int i;
	void *bin = binary(source);
	for (i = 0; i < 8; i++)
		if (host_crack[index].hash[i] != ((uint64_t *) bin)[i])
			return 0;
	return 1;
}

static void set_key(char *key, int index)
{
	int saved_key_length = MIN(strlen(key), PLAINTEXT_LENGTH);
	// make sure LAST uint64 that has any key in it gets null, since we simply
	// ^= the whole uint64 with the ipad/opad mask
	strncpy((char*)host_pass[index].v, key, PLAINTEXT_LENGTH);
	host_pass[index].length = saved_key_length;
}

static char *get_key(int index)
{
	static char ret[PLAINTEXT_LENGTH + 1];
	memcpy(ret, host_pass[index].v, PLAINTEXT_LENGTH);
	ret[host_pass[index].length] = 0;
	return ret;
}

static int binary_hash_0(void *binary)
{
#if 0
	uint32_t i, *b = binary;
	puts("binary");
	for (i = 0; i < 16; i++)
		printf("%08x ", b[i]);
	puts("");
#endif
	return (((uint32_t *) binary)[0] & 0xf);
}
static int binary_hash_1(void *binary) { return (((uint32_t *) binary)[0] & 0xff); }
static int binary_hash_2(void *binary) { return (((uint32_t *) binary)[0] & 0xfff); }
static int binary_hash_3(void *binary) { return (((uint32_t *) binary)[0] & 0xffff); }
static int binary_hash_4(void *binary) { return (((uint32_t *) binary)[0] & 0xfffff); }
static int binary_hash_5(void *binary) { return (((uint32_t *) binary)[0] & 0xffffff); }
static int binary_hash_6(void *binary) { return (((uint32_t *) binary)[0] & 0x7ffffff); }

static int get_hash_0(int index)
{
#if 0
	uint32_t i;
	puts("get_hash");
	for (i = 0; i < 16; i++)
		printf("%08x ", ((uint32_t*)host_crack[index].hash)[i]);
	puts("");
#endif
	return host_crack[index].hash[0] & 0xf;
}

static int get_hash_1(int index) { return host_crack[index].hash[0] & 0xff; }
static int get_hash_2(int index) { return host_crack[index].hash[0] & 0xfff; }
static int get_hash_3(int index) { return host_crack[index].hash[0] & 0xffff; }
static int get_hash_4(int index) { return host_crack[index].hash[0] & 0xfffff; }
static int get_hash_5(int index) { return host_crack[index].hash[0] & 0xffffff; }
static int get_hash_6(int index) { return host_crack[index].hash[0] & 0x7ffffff; }

#if FMT_MAIN_VERSION > 11
static unsigned int iteration_count(void *salt)
{
	salt_t *my_salt;

	my_salt = salt;
	return (unsigned int) my_salt->rounds;
}
#endif

struct fmt_main fmt_opencl_pbkdf2_hmac_sha512 = {
	{
		    FORMAT_LABEL,
		    FORMAT_NAME,
		    ALGORITHM_NAME,
		    BENCHMARK_COMMENT,
		    BENCHMARK_LENGTH,
		    PLAINTEXT_LENGTH,
		    BINARY_SIZE,
		    BINARY_ALIGN,
		    SALT_SIZE,
		    SALT_ALIGN,
		    1,
		    1,
		    FMT_CASE | FMT_8_BIT | FMT_SPLIT_UNIFIES_CASE,
#if FMT_MAIN_VERSION > 11
		{
			"iteration count",
		},
#endif
	            tests}, {
		    init,
		    done,
		    fmt_default_reset,
		    prepare,
		    valid,
		    split,
		    binary,
		    get_salt,
#if FMT_MAIN_VERSION > 11
		{
			iteration_count,
		},
#endif
		    fmt_default_source,
		    {
				binary_hash_0,
				binary_hash_1,
				binary_hash_2,
				binary_hash_3,
				binary_hash_4,
				binary_hash_5,
				binary_hash_6
		    },
		    fmt_default_salt_hash,
		    set_salt,
		    set_key,
		    get_key,
		    fmt_default_clear_keys,
		    crypt_all,
		    {
				get_hash_0,
				get_hash_1,
				get_hash_2,
				get_hash_3,
				get_hash_4,
				get_hash_5,
				get_hash_6
		    },
		    cmp_all,
		    cmp_one,
	    cmp_exact}
};

#endif /* plugin stanza */

#endif /* HAVE_OPENCL */
