
open Hashtbl
open Char

exception InternalError
exception Todo
exception Failure of string

let stop_here () = () 		(* called from debug code - a trap for the debugger to stop on *)

let trace name f x = 
	print_string (name ^ "[") ; let r = f x in print_string "]"; r  

let interr s = print_endline ("internal warning: "^s)
		
let fail () = raise InternalError

let rec lookup env x =
  match env with
    [] -> None
  | ((n,v)::rest) ->
      if x=n then Some v else lookup rest x

let rec remove env x =
  match env with 
    [] -> []
  | ((y,v)::ys) ->
      if x=y then ys else (y,v)::(remove ys x)

let rec subtract env s =
  match s with
    [] -> env
  | (x::xs) ->
      let new_env = remove env x
      in subtract new_env xs

let rec substFirst env x y =
  match env with
      [] -> []
  | ((a,b)::rest) ->
      if x=a then (a,y)::rest
      else (a,b)::(substFirst rest x y)

let rec remove_first x l =
  match l with
    [] -> []
  | (y::ys) -> if x=y then ys else y::(remove_first x ys)

let valof x = match x with 
	| Some y -> y 
	| None -> raise InternalError

let rec member x l =
  match l with
    [] -> false
  | (y::ys) -> (x=y) || (member x ys)

let rec intersect s1 s2 =
  match (s1,s2) with
    ([],x) -> []
  | (x,[]) -> []
  | (x::xs,ys) ->
      if (member x ys) then x::(intersect xs ys)
      else intersect xs ys

let rec subset s1 s2 =
  match s1 with
    [] -> true
  | (x::xs) -> (member x s2) && (subset xs s2)

let rec equiv s1 s2 =
  (subset s1 s2) && (subset s2 s1)

let ($) f x = f x;;

let rec iter_from_int i f list = match list with
	| [] -> ()
	| x::xs -> f i x; iter_from_int (i+1) f xs

let rec map_from_int i f xs = match xs with
		  [] -> []
		| (x::xs) -> 
				let y = f i x in
				y :: map_from_int (i+1) f xs

let count_map (f : int -> 'a -> 'b) (xs : 'a list) = map_from_int 0 f xs

let rec fold_from_int i f list init = match list with
	| [] -> init
	| (x::xs) -> fold_from_int (i+1) f xs (f i x init)

let rec map_int_range f first last = if first > last then []
	else f first :: (map_int_range f (first + 1) last)

let concat_map f l = List.concat $ List.map f l   	
let concat_map2 f l1 l2 = List.concat $ List.map2 f l1 l2

let identity x = x 

let is_alpha c = (code c <= code 'z' && code c >= code 'a') ||
			(code c <= code 'Z' && code c >= code 'A')

let rec zip l1 l2 =
  match (l1,l2) with
  | ([],[]) -> []
  | (x::xs,y::ys) -> (x,y)::(zip xs ys)
  | _ -> interr "cannot zip lists of different lengths"; []

let rec unzip list = match list with
	| [] -> ([],[])
	| (x,y)::rest -> 
		let xs,ys = unzip rest in 
		(x::xs,y::ys)
  
let rec zip3 l1 l2 l3 =
  match (l1,l2,l3) with
  | ([],[],[]) -> []
  | (x::xs,y::ys,z::zs) -> (x,y,z)::(zip3 xs ys zs)
  | _ -> interr "cannot zip3 lists of different lengths"; []

let rec unzipwith f list = match list with
	| [] -> ([],[])
 	| (x::xs) -> 
 		let (y,z) = f x in
 		let (ys,zs) = unzipwith f xs in
 		(y::ys,z::zs)		

exception ListNotSingular
 		
let onlyelement xs = match xs with
	| [x] -> x
	| _ -> interr "List not singular"; List.hd xs
	
let rec replace_one ~cmp ~list ~oldelem ~newelem = match list with
	| [] -> interr "replacement target not found"; list
	| (x::xs) when cmp x oldelem -> newelem :: xs
	| (x::xs) -> x :: (replace_one ~cmp ~list:xs ~oldelem ~newelem)


let is_op n = not ((is_alpha $ String.get n 0) || (String.get n 0 = '_'))

let option_apply f x = match x with
	| None -> None
  	| Some y -> Some (f y)

let option_iter f x = match x with
	| None -> ()
	| Some y -> f y

let  rec option_map f list = match list with
	| [] -> []
	| x::xs -> begin match f x with
		| Some r -> r::option_map f xs
		| None -> option_map f xs
		end 
		
let option_list_collapse l = option_map (fun x -> x) l
		
let list_of_option op = match op with
	| None -> []
	| Some x -> [x]		
	
let rec list_first_option f list = match list with
	| [] -> None
	| x::xs -> begin match f x with
		| Some y -> Some y
		| None -> list_first_option f xs
		end
		
let rec list_first_match f list = match list with
	| [] -> None
	| x::xs when f x -> Some x
	| x::xs -> list_first_match f xs		

let bool_to_option x = match x with true -> Some () | false -> None
let option_to_bool x = match x with Some _ -> true | None -> false
				
let iter_until_succeeds f list = 
	option_to_bool (list_first_option (fun x -> bool_to_option (f x)) list)
			
let rec list_iter3 f l1 l2 l3 = match l1, l2, l3 with
	| [],[],[]	-> ()
	| x::xs,y::ys,z::zs	-> f x y z; list_iter3 f xs ys zs
	| _ -> interr "iter3 lists different lengths"; ()
			
let rec list_acmap f init list = match list with
	| [] -> []
	| x::xs -> 
		let y,i2 = f x init in 
		y :: (list_acmap f i2 xs)	
	
let rec intersperse sep list = match list with
	| [] -> []
	| [x] -> [x]
	| x::xs -> x::sep::intersperse sep xs
	
let rec all_equal cmp list = match list with
	| [] -> true
	| [x] -> true
	| x::y::xs -> cmp x y && all_equal cmp (y::xs)
	 
let check_all_equal cmp list = ignore (all_equal (fun x y -> cmp; true)	list) 
	 
let one_of_equals cmp dflt list = match list with
	| [] -> dflt
	| _ -> assert (all_equal cmp list);
		 List.hd list

let seq x =
    assert(x >= 0);
    let rec count_up n limit =
        if n < (limit - 1)
        then
            n::(count_up (n + 1) limit)
        else
            [n]
    in
        match x with
            0 -> []
        |   _ -> count_up 0 x


module type EQTYPE = sig
	type t
	val equal : t -> t -> bool
end

let word_align_down i = i - (i mod 4)
let word_align_up i = word_align_down (i+3)

let split_long l = 
		let high_bits = Int64.to_int32 (Int64.shift_right_logical l 32) in
		let low_bits = Int64.to_int32 l in
		(high_bits, low_bits)

let rec index_of f list = match list with
	| (x::xs) when f x -> 0
	| (x::xs) -> 1 + index_of f xs
	| [] -> interr "no element in list matches"; 0
		
let rec list_replace inner index outer = match outer,index with
	| (x::xs,0) -> inner @ xs
	| (x::xs,n) -> x :: (list_replace inner (index - 1) xs)
	| ([],_) -> interr "list index out of bounds"; []
	
let list_remove index list = list_replace [] index list	

let rec list_set item index list = match list,index with
	| (x::xs,0) -> item::xs
	| ([],0) -> [item]
	| ([],_) -> interr "list index out of bounds"; []
	| (x::xs,_) -> x :: (list_set item (index - 1) xs)

let rec list_maybe_find f list = match list with
	| (x::xs) when f x -> Some x
	| (x::xs) -> list_maybe_find f xs
	| [] -> None
	
let rec list_repeat item i = if i = 0 then [] else item :: (list_repeat item (i-1))

let remove_duplicates l =
    let rec iter_dups l =
        match l with
            x::(y::rest) ->
                if(x = y) then iter_dups(y::rest)
                          else x::(iter_dups(y::rest))
        |   x::[] -> [x]
        |   [] -> []
    in
        iter_dups(List.sort (fun x y -> if x < y then -1 else 1) l)

(* these three funcs are rather similar - it might be nice to abstract this *)        
        
let rec list_equal f list1 list2 = match list1, list2 with
	| [], [] -> true
	| (x::xs), (y::ys) when f x y -> list_equal f xs ys
	| _ -> false
	
let rec list_prefix list1 list2 = match list1, list2 with
	| [], _ -> true
	| (x::xs), (y::ys) when x = y -> list_prefix xs ys
	| _ -> false
	
let list_suffix list1 list2 = list_prefix (List.rev list1) (List.rev list2)

let rec list_common_prefix list1 list2 = match list1, list2 with	 
	| (x::xs), (y::ys) when x = y -> x::(list_common_prefix xs ys)
	| _ -> [] 
	
let rec list_nth_dflt list dflt i = match list, i with
	| [],_ -> dflt
	| x::xs,0 -> x
	| x::xs,n -> list_nth_dflt xs dflt (n-1)

let rec list_last list = match list with
	| [x] -> x
	| x::xs -> list_last xs
	| [] -> raise (Failure "List empty")	

let rec all_but_last list = match list with
	| [x] -> []
	| x::xs -> x:: all_but_last xs
	| [] -> interr "List empty"; []	
					
let rec list_divide n list = match list with
	| _ when n = 0 -> [],list
	| x::xs -> 
		let ys,zs = list_divide (n-1) xs in	
		(x::ys,zs)
	| [] -> interr "List empty"; [],[]
	
let list_n_tail n list = snd (list_divide n list)
let list_n_head n list = fst (list_divide n list)

let list_strip_suffix list1 list2 = 
	list_n_head (List.length list2 - List.length list1) list2
				
(*Count the size of a hashtable*)
let hash_size h =
    let count = ref 0 in
        Hashtbl.iter (fun x y -> count := !count + 1) h;
        !count

(*Directory containing our libraries*)
let libdir = "stdlib"

(* operators for Int32 operations *)
let (<<*) x y = Int32.shift_left x y
let (|*) x y = Int32.logor x y
let (!*) x = Int32.of_int x 
let (+*) x y = Int32.add x y
let ($**) x y = Int32.mul x y
let (-*) x y = Int32.sub x y

let list_sum = List.fold_left (+) 0 
let list_max list = List.fold_left (max) (List.hd list) (List.tl list)


(* take the log of an int32, to base 2 *)
let rec logbase2 x = if x = Int32.one then 0 else 1 + logbase2 (Int32.shift_right x 1)

(* compute base to the power of x *)
let rec pow base x = if x = 0 then 1 else base * pow base (x-1)

(* generate a function that returns sequential integers *)
let intcounter i = let r = ref i in (fun () -> r := !r + 1; !r)


let rec string_to_list_from s i = 
	if i >= String.length s then [] else
	String.get s i :: string_to_list_from s (i+1)
let string_to_list s = string_to_list_from s 0

let rec fill_string_from l i s = match l with
	| [] -> ()
	| x::xs -> String.set s i x; fill_string_from xs (i+1) s
let list_to_string l = 
	let s = String.create (List.length l) in
	fill_string_from l 0 s;
	s

let rec collapse_spaces_list l = match l with
	| ' '::' '::xs -> collapse_spaces_list (' '::xs)
	| x::xs -> x::collapse_spaces_list xs
	| [] -> []

let collapse_spaces s = 
	list_to_string (collapse_spaces_list (string_to_list s))

let string_map s f = 
	let new_s = String.copy s in
	for i = 0 to (String.length s - 1) do
		String.set new_s i (f (String.get new_s i))
	done;
	new_s
	
let strip_tabs s = string_map s (fun c -> if c = '\t' then ' ' else c)
let strip_comments s = 
	string_map s (fun c -> if c = '\t' || c = ' ' || c = '\n' || c = '\r' then c else ' ')

let contains_comment s = strip_comments s <> s

let is_prefix prefix s = 
	prefix = String.sub s 0 (min (String.length prefix) (String.length s))

let strip_prefix prefix s =
	String.sub s (String.length prefix) (String.length s - String.length prefix)

let is_suffix suffix s =
	if String.length suffix > String.length s then false
	else
	suffix = String.sub s (String.length s - String.length suffix) 
				(String.length suffix)

let strip_suffix suffix s =
	if not (is_suffix suffix s) then
		(interr ("name "^s^" should have suffix "^suffix);s)
	else	
		String.sub s 0 (String.length s - String.length suffix)

let rec list_extract x l = match l with
	| y::ys when x == y -> Some ys
	| y::ys -> 
		(match list_extract x ys with
		 | Some zs -> Some (y ::zs)
		 | None -> None)		
	| [] -> None

let rec same_elements l1 l2 = match l1 with
	| x::xs ->
		(match list_extract x l2 with
			| Some ys -> same_elements xs ys
			| None -> false)
	| [] when l2 = [] -> true
	| _ -> false
	
	
let rec opt_equal f x y = match x,y with
	| None,None -> true
	| Some a, Some b -> f a b
	| _ -> false

let rec exception_iter f l = match l with
	| [x] -> f x
	| x::xs -> 
		(try f x
		with _ -> exception_iter f xs)
	| [] -> raise InternalError

let rec tailiter f l = match l with
	| [] -> ()
	| x::xs -> f x xs; tailiter f xs

let string_longest x y = if String.length x > String.length y then x else y
