//===========================================================================
//  CVS Information:                                                         
//                                                                           
//     $RCSfile: driven_cavity.cpp,v $  $Revision: 1.2 $  $State: Exp $ 
//     $Author: llee $  $Date: 2001/10/23 16:15:01 $ 
//     $Locker:  $ 
//---------------------------------------------------------------------------
//                                                                           
// DESCRIPTION                                                               
//    This problem is modeled by the partial differential equation system
//  
//	- Lap(U) - Grad_y(Omega) = 0
//	- Lap(V) + Grad_x(Omega) = 0
//	- Lap(Omega) + Div([U*Omega,V*Omega]) - GR*Grad_x(T) = 0
//	- Lap(T) + PR*Div([U*T,V*T]) = 0
//
//    in the unit square, which is uniformly discretized in each of x and
//    y in this simple encoding.
//
//    No-slip, rigid-wall Dirichlet conditions are used for [U,V].
//    Dirichlet conditions are used for Omega, based on the definition of
//    vorticity: Omega = - Grad_y(U) + Grad_x(V), where along each
//    constant coordinate boundary, the tangential derivative is zero.
//    Dirichlet conditions are used for T on the left and right walls,
//    and insulation homogeneous Neumann conditions are used for T on
//    the top and bottom walls. 
//
//    A finite difference approximation with the usual 5-point stencil 
//    is used to discretize the boundary value problem to obtain a 
//    nonlinear system of equations.  Upwinding is used for the divergence
//    (convective) terms and central for the gradient (source) terms.
// 
//---------------------------------------------------------------------------
//                                                                           
// LICENSE AGREEMENT                                                         
//=======================================================================
// Copyright (C) 1997-2001
// Authors: Andrew Lumsdaine <lums@osl.iu.edu> 
//          Lie-Quan Lee     <llee@osl.iu.edu>
//
// This file is part of the Iterative Template Library
//
// You should have received a copy of the License Agreement for the
// Iterative Template Library along with the software;  see the
// file LICENSE.  
//
// Permission to modify the code and to distribute modified code is
// granted, provided the text of this NOTICE is retained, a notice that
// the code was modified is included with the above COPYRIGHT NOTICE and
// with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
// file is distributed with the modified code.
//
// LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
// By way of example, but not limitation, Licensor MAKES NO
// REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
// PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
// OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
// OR OTHER RIGHTS.
//=======================================================================
//---------------------------------------------------------------------------
//                                                                           
// REVISION HISTORY:                                                         
//                                                                           
// $Log: driven_cavity.cpp,v $
// Revision 1.2  2001/10/23 16:15:01  llee
// *** empty log message ***
//
// Revision 1.1  2001/10/19 16:14:08  llee
// thermally driven cavity problem: done
//
//                                                                           
//===========================================================================
//#define MTL_BLAS_GROT 

#include <fstream>
#include <iostream>
#include <vector>
#include <iterator>

#include <mtl/dense1D.h>
#include <mtl/matrix.h>
#include <mtl/mtl.h>

//#include "parallel_interface.h"
#if defined USE_MTL_PARAPPEL
#include <itl/interface/mtl_parallel.h>
#else
#include <itl/interface/blas_parallel.h>
#endif

#include <itl/krylov/gmres.h>
#include <itl/preconditioner/block_ilu.h>
#include <itl/matrix_free_operator.h>

#include "initial_guess.h"
#include "evaluate_function.h"
#include "driven_cavity.h"


using namespace std;
using namespace itl;

typedef mtl::matrix<double, mtl::rectangle<>,
		    mtl::compressed<>,
		    mtl::row_major>::type Matrix;
typedef mtl::matrix<double, mtl::rectangle<>,
		    mtl::array<mtl::compressed<> >,
		    mtl::row_major>::type InitMatrix;

typedef mtl::dense1D<double> Vector;

int 
main(int argc, char *argv[])
{
  Manager::init(argc, argv);
  bool monitor = false;
  int kmax;
  int iter_max;
  int restart;

  double ksp_rtol, ksp_atol, newton_rtol, newton_atol, newton_stol;

  int nx, ny;
  if ( argc > 1 ) {
    nx = atoi(argv[1]);
    if ( nx <= 0 )
      nx = Manager::size()*8;
  } else 
    nx = 16;

  if ( argc > 2 ) {   
    ny = atoi(argv[2]);
    if ( ny <= 0 )
      ny = nx;
  } else 
    ny = nx;

  if ( argc > 3 )
    newton_rtol = atof(argv[3]);
  else
    newton_rtol = 1.0e-8;  

  newton_stol = 1.0e-8;
  newton_atol = 1.0e-50;

  if ( argc > 4 )
    ksp_rtol = atof(argv[4]);
  else
    ksp_rtol = 1.0e-5;

  if ( argc > 5 )
    ksp_atol = atof(argv[5]);
  else
    ksp_atol = 1.0e-50;
  
  if ( argc > 6 )
    kmax = atoi(argv[6]);
  else 
    kmax = 60;

  if ( argc > 7 )
    iter_max = atoi(argv[7]);
  else 
    iter_max = 60;

  if ( argc > 8 )
    restart = atoi(argv[8]);
  else 
    restart = 60;

  if ( argc > 9 )
    monitor = true;

  if ( !Manager::rank() ) {
    cout << "Relative tolerance for Newton iteration: " << newton_rtol << endl
	 << "Absolute tolerance for Newton iteration: " << newton_atol << endl
	 << "Secussive tolerance for Newton iteration: " << newton_stol << endl
	 << "Relative tolerance for KSP iteraion: " << ksp_rtol << endl
	 << "Absolute tolerance for KSP iteraion: " << ksp_atol << endl;
  }

  int msize = nx * ny;
  int pos = msize / Manager::size();
  
  int local_nrows = pos;
  if ( Manager::rank() == Manager::size() - 1 ) //last processor
    local_nrows = msize - Manager::rank() * pos;

  local_nrows *= NUM_OF_SV; //system

  Vector X(local_nrows, 0.0);
  Vector F(local_nrows, 0.0);
  Vector DX(local_nrows, 0.0);

  Info info;
  info.nx = nx;
  info.ny = ny;
  info.grashof = 1.0;
  info.prandtl = 1.0;
  info.lidvelocity = 1.0/(nx*ny);

  //currently only row partitioning is supported
  info.local_nx_begin = 0;
  info.local_nx_end   = nx;

  int ny_pos = ny / Manager::size();
  info.local_ny_begin = Manager::rank()*ny_pos;
  info.local_ny_end   = info.local_ny_begin + ny_pos;
  if ( Manager::rank() == Manager::size() - 1 ) //last processor
    info.local_ny_end = ny;

  if ( !Manager::rank() )
    cout << "("<< Manager::size() 
	 << " CPU(s)) for driven cavity problem: " << nx << " x " << ny << endl
	 << " grashof = " << info.grashof 
	 << " prandtl = " << info.prandtl
	 << " lidvelocity = " << info.lidvelocity << endl;

  {
    initial_guess(X, info);
  } //initialize X

  eval _f(info);
  typedef  evaluate_function<double, eval> EvalFunc;

  EvalFunc f(msize*NUM_OF_SV, pos*NUM_OF_SV, _f);

  f(X, F);
  
  double norm0 = itl::two_norm(F);
  double two_norm_F = norm0, old_two_norm_F = 0.0;

  if ( ! Manager::rank() ) 
    cout << "norm0 = " << norm0 << endl;

  // Start Newton solver
  int k;
  for (k = 0; k < kmax; k++) {

    //update Jacobian here. 
    matrix_free_operator<Vector, EvalFunc> A(f, X, F);

    identity_preconditioner p;

    double two_norm_f = two_norm_F;

    noisy_iteration<double> iter(two_norm_f, iter_max, ksp_rtol, ksp_atol);

    //  Solve J dx = -f  with restarted GMRES (identity precond)

    //initialize DX
    for (int i=0; i<DX.size(); i++)
      DX[i] = 0.0;

    itl::scale(F, -1.0);

    itl::modified_gram_schmidt<Vector> orth(restart, DX.size());

    if ( monitor )
      gmres(A, DX, F, p(), restart, iter, orth);
    else
      gmres(A, DX, F, p(), restart, 
	    static_cast<basic_iteration<double>&>(iter), orth);

    itl::add(DX, X);

    //  Compute F
    f(X, F);

    old_two_norm_F = two_norm_F;
    two_norm_F     = itl::two_norm(F);

    if ( Manager::rank() == 0 ) {
      cout << "Number of iteration used in GMRES: " << iter.iterations() 
	   << "     Residual: " << iter.resid() << endl;

      cout << "Newton iteration:                  " << k 
	   << "     |F| = " << two_norm_F << endl;
    }

    if ( two_norm_F < newton_rtol * norm0 
	 || std::abs( old_two_norm_F - two_norm_F ) < newton_stol
	 || two_norm_F < newton_atol )
      break;

  }

  if ( Manager::rank() == 0 ) {
    cout << "Number of Newton iteration: " << k+1 << endl;
  }

  f.cleanup();
  
  Manager::finish();
  return 0;
}
