"use strict";

var _interopRequire = function (obj) { return obj && obj.__esModule ? obj["default"] : obj; };

var _createClass = (function () { function defineProperties(target, props) { for (var key in props) { var prop = props[key]; prop.configurable = true; if (prop.value) prop.writable = true; } Object.defineProperties(target, props); } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; })();

var _get = function get(object, property, receiver) { var desc = Object.getOwnPropertyDescriptor(object, property); if (desc === undefined) { var parent = Object.getPrototypeOf(object); if (parent === null) { return undefined; } else { return get(parent, property, receiver); } } else if ("value" in desc && desc.writable) { return desc.value; } else { var getter = desc.get; if (getter === undefined) { return undefined; } return getter.call(receiver); } };

var _inherits = function (subClass, superClass) { if (typeof superClass !== "function" && superClass !== null) { throw new TypeError("Super expression must either be null or a function, not " + typeof superClass); } subClass.prototype = Object.create(superClass && superClass.prototype, { constructor: { value: subClass, enumerable: false, writable: true, configurable: true } }); if (superClass) subClass.__proto__ = superClass; };

var _classCallCheck = function (instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } };

/*
  Copyright (C) 2015 Yusuke Suzuki <utatane.tea@gmail.com>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

var Syntax = require("estraverse").Syntax;

var esrecurse = _interopRequire(require("esrecurse"));

var Reference = _interopRequire(require("./reference"));

var Variable = _interopRequire(require("./variable"));

var _definition = require("./definition");

var ParameterDefinition = _definition.ParameterDefinition;
var Definition = _definition.Definition;

var assert = _interopRequire(require("assert"));

var PatternVisitor = (function (_esrecurse$Visitor) {
    function PatternVisitor(rootPattern, referencer, callback) {
        _classCallCheck(this, PatternVisitor);

        _get(Object.getPrototypeOf(PatternVisitor.prototype), "constructor", this).call(this);
        this.referencer = referencer;
        this.callback = callback;
    }

    _inherits(PatternVisitor, _esrecurse$Visitor);

    _createClass(PatternVisitor, {
        perform: {
            value: function perform(pattern) {
                if (pattern.type === Syntax.Identifier) {
                    this.callback(pattern, true);
                    return;
                }
                this.visit(pattern);
            }
        },
        Identifier: {
            value: function Identifier(pattern) {
                this.callback(pattern, false);
            }
        },
        ObjectPattern: {
            value: function ObjectPattern(pattern) {
                var i, iz, property;
                for (i = 0, iz = pattern.properties.length; i < iz; ++i) {
                    property = pattern.properties[i];
                    if (property.shorthand) {
                        this.visit(property.key);
                        continue;
                    }
                    this.visit(property.value);
                }
            }
        },
        ArrayPattern: {
            value: function ArrayPattern(pattern) {
                var i, iz, element;
                for (i = 0, iz = pattern.elements.length; i < iz; ++i) {
                    element = pattern.elements[i];
                    if (element) {
                        this.visit(element);
                    }
                }
            }
        },
        AssignmentPattern: {
            value: function AssignmentPattern(pattern) {
                this.visit(pattern.left);
                // FIXME: Condier TDZ scope.
                this.referencer.visit(pattern.right);
            }
        }
    });

    return PatternVisitor;
})(esrecurse.Visitor);

function traverseIdentifierInPattern(rootPattern, referencer, callback) {
    var visitor = new PatternVisitor(rootPattern, referencer, callback);
    visitor.perform(rootPattern);
}

function isPattern(node) {
    var nodeType = node.type;
    return nodeType === Syntax.Identifier || nodeType === Syntax.ObjectPattern || nodeType === Syntax.ArrayPattern || nodeType === Syntax.SpreadElement || nodeType === Syntax.RestElement || nodeType === Syntax.AssignmentPattern;
}

// Importing ImportDeclaration.
// http://people.mozilla.org/~jorendorff/es6-draft.html#sec-moduledeclarationinstantiation
// https://github.com/estree/estree/blob/master/es6.md#importdeclaration
// FIXME: Now, we don't create module environment, because the context is
// implementation dependent.

var Importer = (function (_esrecurse$Visitor2) {
    function Importer(declaration, referencer) {
        _classCallCheck(this, Importer);

        _get(Object.getPrototypeOf(Importer.prototype), "constructor", this).call(this);
        this.declaration = declaration;
        this.referencer = referencer;
    }

    _inherits(Importer, _esrecurse$Visitor2);

    _createClass(Importer, {
        visitImport: {
            value: function visitImport(id, specifier) {
                var _this = this;

                this.referencer.visitPattern(id, function (pattern) {
                    _this.referencer.currentScope().__define(pattern, new Definition(Variable.ImportBinding, pattern, specifier, _this.declaration, null, null));
                });
            }
        },
        ImportNamespaceSpecifier: {
            value: function ImportNamespaceSpecifier(node) {
                var local = node.local || node.id;
                if (local) {
                    this.visitImport(local, node);
                }
            }
        },
        ImportDefaultSpecifier: {
            value: function ImportDefaultSpecifier(node) {
                var local = node.local || node.id;
                this.visitImport(local, node);
            }
        },
        ImportSpecifier: {
            value: function ImportSpecifier(node) {
                var local = node.local || node.id;
                if (node.name) {
                    this.visitImport(node.name, node);
                } else {
                    this.visitImport(local, node);
                }
            }
        }
    });

    return Importer;
})(esrecurse.Visitor);

// Referencing variables and creating bindings.

var Referencer = (function (_esrecurse$Visitor3) {
    function Referencer(scopeManager) {
        _classCallCheck(this, Referencer);

        _get(Object.getPrototypeOf(Referencer.prototype), "constructor", this).call(this);
        this.scopeManager = scopeManager;
        this.parent = null;
        this.isInnerMethodDefinition = false;
    }

    _inherits(Referencer, _esrecurse$Visitor3);

    _createClass(Referencer, {
        currentScope: {
            value: function currentScope() {
                return this.scopeManager.__currentScope;
            }
        },
        close: {
            value: function close(node) {
                while (this.currentScope() && node === this.currentScope().block) {
                    this.scopeManager.__currentScope = this.currentScope().__close(this.scopeManager);
                }
            }
        },
        pushInnerMethodDefinition: {
            value: function pushInnerMethodDefinition(isInnerMethodDefinition) {
                var previous = this.isInnerMethodDefinition;
                this.isInnerMethodDefinition = isInnerMethodDefinition;
                return previous;
            }
        },
        popInnerMethodDefinition: {
            value: function popInnerMethodDefinition(isInnerMethodDefinition) {
                this.isInnerMethodDefinition = isInnerMethodDefinition;
            }
        },
        materializeTDZScope: {
            value: function materializeTDZScope(node, iterationNode) {
                // http://people.mozilla.org/~jorendorff/es6-draft.html#sec-runtime-semantics-forin-div-ofexpressionevaluation-abstract-operation
                // TDZ scope hides the declaration's names.
                this.scopeManager.__nestTDZScope(node, iterationNode);
                this.visitVariableDeclaration(this.currentScope(), Variable.TDZ, iterationNode.left, 0);
            }
        },
        materializeIterationScope: {
            value: function materializeIterationScope(node) {
                var _this = this;

                // Generate iteration scope for upper ForIn/ForOf Statements.
                var letOrConstDecl;
                this.scopeManager.__nestForScope(node);
                letOrConstDecl = node.left;
                this.visitVariableDeclaration(this.currentScope(), Variable.Variable, letOrConstDecl, 0);
                this.visitPattern(letOrConstDecl.declarations[0].id, function (pattern) {
                    _this.currentScope().__referencing(pattern, Reference.WRITE, node.right, null, true);
                });
            }
        },
        visitPattern: {
            value: function visitPattern(node, callback) {
                traverseIdentifierInPattern(node, this, callback);
            }
        },
        visitFunction: {
            value: function visitFunction(node) {
                var _this = this;

                var i, iz;
                // FunctionDeclaration name is defined in upper scope
                // NOTE: Not referring variableScope. It is intended.
                // Since
                //  in ES5, FunctionDeclaration should be in FunctionBody.
                //  in ES6, FunctionDeclaration should be block scoped.
                if (node.type === Syntax.FunctionDeclaration) {
                    // id is defined in upper scope
                    this.currentScope().__define(node.id, new Definition(Variable.FunctionName, node.id, node, null, null, null));
                }

                // FunctionExpression with name creates its special scope;
                // FunctionExpressionNameScope.
                if (node.type === Syntax.FunctionExpression && node.id) {
                    this.scopeManager.__nestFunctionExpressionNameScope(node);
                }

                // Consider this function is in the MethodDefinition.
                this.scopeManager.__nestFunctionScope(node, this.isInnerMethodDefinition);

                for (i = 0, iz = node.params.length; i < iz; ++i) {
                    this.visitPattern(node.params[i], function (pattern) {
                        _this.currentScope().__define(pattern, new ParameterDefinition(pattern, node, i, false));
                    });
                }

                // if there's a rest argument, add that
                if (node.rest) {
                    this.visitPattern({
                        type: "RestElement",
                        argument: node.rest
                    }, function (pattern) {
                        _this.currentScope().__define(pattern, new ParameterDefinition(pattern, node, node.params.length, true));
                    });
                }

                // Skip BlockStatement to prevent creating BlockStatement scope.
                if (node.body.type === Syntax.BlockStatement) {
                    this.visitChildren(node.body);
                } else {
                    this.visit(node.body);
                }

                this.close(node);
            }
        },
        visitClass: {
            value: function visitClass(node) {
                if (node.type === Syntax.ClassDeclaration) {
                    this.currentScope().__define(node.id, new Definition(Variable.ClassName, node.id, node, null, null, null));
                }

                // FIXME: Maybe consider TDZ.
                this.visit(node.superClass);

                this.scopeManager.__nestClassScope(node);

                if (node.id) {
                    this.currentScope().__define(node.id, new Definition(Variable.ClassName, node.id, node));
                }
                this.visit(node.body);

                this.close(node);
            }
        },
        visitProperty: {
            value: function visitProperty(node) {
                var previous, isMethodDefinition;
                if (node.computed) {
                    this.visit(node.key);
                }

                isMethodDefinition = node.type === Syntax.MethodDefinition || node.method;
                if (isMethodDefinition) {
                    previous = this.pushInnerMethodDefinition(true);
                }
                this.visit(node.value);
                if (isMethodDefinition) {
                    this.popInnerMethodDefinition(previous);
                }
            }
        },
        visitForIn: {
            value: function visitForIn(node) {
                var _this = this;

                if (node.left.type === Syntax.VariableDeclaration && node.left.kind !== "var") {
                    this.materializeTDZScope(node.right, node);
                    this.visit(node.right);
                    this.close(node.right);

                    this.materializeIterationScope(node);
                    this.visit(node.body);
                    this.close(node);
                } else {
                    if (node.left.type === Syntax.VariableDeclaration) {
                        this.visit(node.left);
                        this.visitPattern(node.left.declarations[0].id, function (pattern) {
                            _this.currentScope().__referencing(pattern, Reference.WRITE, node.right, null, true);
                        });
                    } else {
                        if (!isPattern(node.left)) {
                            this.visit(node.left);
                        }
                        this.visitPattern(node.left, function (pattern) {
                            var maybeImplicitGlobal = null;
                            if (!_this.currentScope().isStrict) {
                                maybeImplicitGlobal = {
                                    pattern: pattern,
                                    node: node
                                };
                            }
                            _this.currentScope().__referencing(pattern, Reference.WRITE, node.right, maybeImplicitGlobal, true);
                        });
                    }
                    this.visit(node.right);
                    this.visit(node.body);
                }
            }
        },
        visitVariableDeclaration: {
            value: function visitVariableDeclaration(variableTargetScope, type, node, index) {
                var _this = this;

                var decl, init;

                decl = node.declarations[index];
                init = decl.init;
                this.visitPattern(decl.id, function (pattern, toplevel) {
                    variableTargetScope.__define(pattern, new Definition(type, pattern, decl, node, index, node.kind));

                    if (init) {
                        _this.currentScope().__referencing(pattern, Reference.WRITE, init, null, !toplevel);
                    }
                });
            }
        },
        AssignmentExpression: {
            value: function AssignmentExpression(node) {
                var _this = this;

                if (isPattern(node.left)) {
                    if (node.operator === "=") {
                        this.visitPattern(node.left, function (pattern, toplevel) {
                            var maybeImplicitGlobal = null;
                            if (!_this.currentScope().isStrict) {
                                maybeImplicitGlobal = {
                                    pattern: pattern,
                                    node: node
                                };
                            }
                            _this.currentScope().__referencing(pattern, Reference.WRITE, node.right, maybeImplicitGlobal, !toplevel);
                        });
                    } else {
                        this.currentScope().__referencing(node.left, Reference.RW, node.right);
                    }
                } else {
                    this.visit(node.left);
                }
                this.visit(node.right);
            }
        },
        CatchClause: {
            value: function CatchClause(node) {
                var _this = this;

                this.scopeManager.__nestCatchScope(node);

                this.visitPattern(node.param, function (pattern) {
                    _this.currentScope().__define(pattern, new Definition(Variable.CatchClause, node.param, node, null, null, null));
                });
                this.visit(node.body);

                this.close(node);
            }
        },
        Program: {
            value: function Program(node) {
                this.scopeManager.__nestGlobalScope(node);

                if (this.scopeManager.__isNodejsScope()) {
                    // Force strictness of GlobalScope to false when using node.js scope.
                    this.currentScope().isStrict = false;
                    this.scopeManager.__nestFunctionScope(node, false);
                }

                if (this.scopeManager.__isES6() && this.scopeManager.isModule()) {
                    this.scopeManager.__nestModuleScope(node);
                }

                this.visitChildren(node);
                this.close(node);
            }
        },
        Identifier: {
            value: function Identifier(node) {
                this.currentScope().__referencing(node);
            }
        },
        UpdateExpression: {
            value: function UpdateExpression(node) {
                if (isPattern(node.argument)) {
                    this.currentScope().__referencing(node.argument, Reference.RW, null);
                } else {
                    this.visitChildren(node);
                }
            }
        },
        MemberExpression: {
            value: function MemberExpression(node) {
                this.visit(node.object);
                if (node.computed) {
                    this.visit(node.property);
                }
            }
        },
        Property: {
            value: function Property(node) {
                this.visitProperty(node);
            }
        },
        MethodDefinition: {
            value: function MethodDefinition(node) {
                this.visitProperty(node);
            }
        },
        BreakStatement: {
            value: function BreakStatement() {}
        },
        ContinueStatement: {
            value: function ContinueStatement() {}
        },
        LabeledStatement: {
            value: function LabeledStatement(node) {
                this.visit(node.body);
            }
        },
        ForStatement: {
            value: function ForStatement(node) {
                // Create ForStatement declaration.
                // NOTE: In ES6, ForStatement dynamically generates
                // per iteration environment. However, escope is
                // a static analyzer, we only generate one scope for ForStatement.
                if (node.init && node.init.type === Syntax.VariableDeclaration && node.init.kind !== "var") {
                    this.scopeManager.__nestForScope(node);
                }

                this.visitChildren(node);

                this.close(node);
            }
        },
        ClassExpression: {
            value: function ClassExpression(node) {
                this.visitClass(node);
            }
        },
        ClassDeclaration: {
            value: function ClassDeclaration(node) {
                this.visitClass(node);
            }
        },
        CallExpression: {
            value: function CallExpression(node) {
                // Check this is direct call to eval
                if (!this.scopeManager.__ignoreEval() && node.callee.type === Syntax.Identifier && node.callee.name === "eval") {
                    // NOTE: This should be `variableScope`. Since direct eval call always creates Lexical environment and
                    // let / const should be enclosed into it. Only VariableDeclaration affects on the caller's environment.
                    this.currentScope().variableScope.__detectEval();
                }
                this.visitChildren(node);
            }
        },
        BlockStatement: {
            value: function BlockStatement(node) {
                if (this.scopeManager.__isES6()) {
                    this.scopeManager.__nestBlockScope(node);
                }

                this.visitChildren(node);

                this.close(node);
            }
        },
        ThisExpression: {
            value: function ThisExpression() {
                this.currentScope().variableScope.__detectThis();
            }
        },
        WithStatement: {
            value: function WithStatement(node) {
                this.visit(node.object);
                // Then nest scope for WithStatement.
                this.scopeManager.__nestWithScope(node);

                this.visit(node.body);

                this.close(node);
            }
        },
        VariableDeclaration: {
            value: function VariableDeclaration(node) {
                var variableTargetScope, i, iz, decl;
                variableTargetScope = node.kind === "var" ? this.currentScope().variableScope : this.currentScope();
                for (i = 0, iz = node.declarations.length; i < iz; ++i) {
                    decl = node.declarations[i];
                    this.visitVariableDeclaration(variableTargetScope, Variable.Variable, node, i);
                    if (decl.init) {
                        this.visit(decl.init);
                    }
                }
            }
        },
        SwitchStatement: {

            // sec 13.11.8

            value: function SwitchStatement(node) {
                var i, iz;

                this.visit(node.discriminant);

                if (this.scopeManager.__isES6()) {
                    this.scopeManager.__nestSwitchScope(node);
                }

                for (i = 0, iz = node.cases.length; i < iz; ++i) {
                    this.visit(node.cases[i]);
                }

                this.close(node);
            }
        },
        FunctionDeclaration: {
            value: function FunctionDeclaration(node) {
                this.visitFunction(node);
            }
        },
        FunctionExpression: {
            value: function FunctionExpression(node) {
                this.visitFunction(node);
            }
        },
        ForOfStatement: {
            value: function ForOfStatement(node) {
                this.visitForIn(node);
            }
        },
        ForInStatement: {
            value: function ForInStatement(node) {
                this.visitForIn(node);
            }
        },
        ArrowFunctionExpression: {
            value: function ArrowFunctionExpression(node) {
                this.visitFunction(node);
            }
        },
        ImportDeclaration: {
            value: function ImportDeclaration(node) {
                var importer;

                assert(this.scopeManager.__isES6() && this.scopeManager.isModule(), "ImportDeclaration should appear when the mode is ES6 and in the module context.");

                importer = new Importer(node, this);
                importer.visit(node);
            }
        },
        visitExportDeclaration: {
            value: function visitExportDeclaration(node) {
                if (node.source) {
                    return;
                }
                if (node.declaration) {
                    this.visit(node.declaration);
                    return;
                }

                this.visitChildren(node);
            }
        },
        ExportDeclaration: {
            value: function ExportDeclaration(node) {
                this.visitExportDeclaration(node);
            }
        },
        ExportNamedDeclaration: {
            value: function ExportNamedDeclaration(node) {
                this.visitExportDeclaration(node);
            }
        },
        ExportSpecifier: {
            value: function ExportSpecifier(node) {
                var local = node.id || node.local;
                this.visit(local);
            }
        }
    });

    return Referencer;
})(esrecurse.Visitor);

module.exports = Referencer;

/* vim: set sw=4 ts=4 et tw=80 : */
//# sourceMappingURL=data:application/json;base64,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