subroutine uv_removes_clean(nv,duv,ouv,nc,mic,dcct,freq,first,last)
  use image_def
  use gbl_message
  use imager_interfaces, only : ompget_inner_threads
  !$ use omp_lib
  !-----------------------------------------------------------------
  ! @ private-mandatory
  !
  ! IMAGER   Support for UV_RESTORE
  !     Compute visibilities for a UV data set according to a
  !     set of Clean Components , and remove them from the original
  !     UV table.
  !       Semi-Slow version using interpolation from pre-tabulated
  !       Sin/Cos, which could still be optimized
  !-----------------------------------------------------------------
  integer, intent(in) :: nv         ! Number of visibilities
  real, intent(in) :: duv(:,:)      ! UV data set
  integer, intent(in) :: nc         ! Number of channels
  integer, intent(in) :: mic(:)     ! Number of Clean Components
  real, intent(out) :: ouv(:,:)     ! Extracted UV data set
  real, intent(in) :: dcct(:,:,:)   ! Clean component
  real(8), intent(in) :: freq       ! Apparent observing frequency
  integer, intent(in) :: first      ! First channel
  integer, intent(in) :: last       ! Last channel
  !
  real(8), parameter :: pi=3.14159265358979323846d0
  real(8), parameter :: clight = 299792458.d-6   ! in meter and MHz units
  real(8), parameter :: f_to_k=2.d0*pi/clight    !
  !
  real(8) :: cst
  real(8) :: pidsur            ! 2*pi/Lambda  to convert D (in m) to wave numbers
  real(8) :: phase, cp, sp
  real(4) :: rvis, ivis, value
  integer ii, ir, iw
  integer oi, or, ow
  integer ic,iv,jc,kc
  !
  real :: cc_size(20), cc_count(20), cc_flux(20)
  real :: gauss, q2
  logical :: do_size
  integer :: no, icmax, scc, ncc_size
  integer :: iplane, j, jsize
  integer :: nthread
  !
  scc = size(dcct,1)
  pidsur = f_to_k * freq
  !
  if (scc.ne.3) then
    ncc_size = 0
    do iplane=1,nc  ! jc = first, last
      do ic=1,mic(iplane)
        if (dcct(3,ic,iplane).eq.0) exit
        jsize = 0
        do j=1,ncc_size
          if (dcct(4,ic,iplane).eq.cc_size(j)) then
            jsize = j
            exit
          endif
        enddo
        if (jsize.eq.0) then
          ncc_size = ncc_size+1
          cc_size(ncc_size) =  dcct(4,ic,iplane)
        endif
        if (ncc_size.gt.10) exit
      enddo
    enddo
    cc_count = 0
    cc_flux = 0.
  else
    ncc_size = 1
  endif
  !
  no = ubound(mic,1)
  if (no.lt.last-first+1 .or. no.gt.ubound(dcct,3)) then
    Print *,'Remove Clean Slow dimension error ',no,last-first+1,ubound(dcct,3)
  endif
  icmax = maxval(mic)
  if (icmax.gt.ubound(dcct,2))  then
    Print *,'Remove Clean Slow -- too many Clean Comp.',icmax,ubound(dcct,2)
  endif
  !!Print *,'CCT Bounds ',ubound(dcct),' FIRST ',first,' LAST ',last
  !
  ! Remove clean component from UV data set
  !
  ! G = flux * exp(-2i*pi(u*x0+v*y0)) * g(u,v)
  ! and for the 1-D Gaussian
  !     model = flux*4*ln2/(pi*b**2) * exp (-4*ln2*(r/b)**2)
  !     g(u,v) = exp(- pi**2/4/ln(2)*(b*q)**2)
  !     where q**2 = u**2+v**2
  !      and  b is the beam size in Radian (FWHM)
  !
  cst = pi**2/4d0/log(2D0) * (freq/clight)**2   ! Scale factor from u^2+v^2 in m^2 to radian^-2
  !
  do_size = scc.eq.4    ! Check size if needed
  !
  !$ nthread = ompget_inner_threads() 
  !$OMP PARALLEL DEFAULT(none) NUM_THREADS(nthread) &
  !$OMP    & SHARED(duv,ouv,dcct)  SHARED(pidsur,nv,first,last,mic) &
  !$OMP    & PRIVATE(iv,jc,ic,kc,ir,ii,iw,or,oi,ow,phase,rvis,ivis,cp,sp) &
  !$OMP    & PRIVATE(value, q2, gauss) SHARED(do_size, cst)
  !
  !$OMP DO SCHEDULE(DYNAMIC,32)
  do iv=1,nv            ! Visibilies
    ouv(1:7,iv) = duv(1:7,iv)
    if (do_size) then
      ! Load (U,V) in radian**-1
      q2 = cst * (ouv(1,iv)**2 + ouv(2,iv)**2)
    endif
    !
    do jc = first,last  ! Channels
      ir = 5+3*jc
      ii = 6+3*jc
      iw = 7+3*jc
      !
      kc = jc-first+1
      or = 5+3*kc      
      oi = 6+3*kc
      ow = 7+3*kc
      !
      ouv(or,iv) = duv(ir,iv)
      ouv(oi,iv) = duv(ii,iv)
      do ic = 1,mic(kc) ! Clean components
        if (dcct(3,ic,kc).ne.0) then
          value = dcct(3,ic,kc)
          !
          ! Apply the Gauss factor if any
          if (do_size) then
            if (dcct(4,ic,kc).ne.0) then
              gauss = q2 * dcct(4,ic,kc)**2
              if (gauss.lt.40) then
                gauss = exp(-gauss)
                value = value*gauss
              else
                cycle  ! Too faint, no signal
              endif
            endif
          endif
          !
          phase = (ouv(1,iv)*dcct(1,ic,kc) + ouv(2,iv)*dcct(2,ic,kc))*pidsur
          cp = cos(phase)
          sp = sin(phase)
          !!call cossin(phase,cp,sp)
          rvis = value*cp
          ivis = value*sp
          ouv(or,iv) = ouv(or,iv) - rvis   ! Subtract
          ouv(oi,iv) = ouv(oi,iv) - ivis
        else if (iv.eq.1) then
          Print *,'Premature end of work for channel ',jc, kc, mic(kc)
          exit ! End of work, jump to next channel
        endif
      enddo             ! Clean components
      ouv(ow,iv) = duv(iw,iv)
    enddo               ! Channels
  enddo                 ! Visibilities
  !$OMP END DO
  !$OMP END PARALLEL
end subroutine uv_removes_clean
!
subroutine uv_squeeze_clean(nc,ccin,ccou, mic, first, last)
  use image_def
  !$ use omp_lib
  !-----------------------------------------------------------------
  ! @ private-mandatory
  !
  ! IMAGER   Support for UV_RESTORE
  !   Compact the component list by summing up all values at the
  !   same position in a range of channels.
  !   The output list has a transposed order
  !-----------------------------------------------------------------
  integer, intent(in) :: nc         ! Number of channels
  real, intent(in) :: ccin(:,:,:)   ! Initial Clean Component List
  real, intent(out) :: ccou(:,:,:)  ! Resulting list
  integer, intent(inout) :: mic(:)  ! Number of Clean component per channel
  integer, intent(in) :: first      ! First channel
  integer, intent(in) :: last       ! Last channel
  !
  integer :: ii,jj,ic,jc,kc, scc
  integer :: ki    ! Number of different components per channel
  integer :: nin   ! Number of input channels
  logical :: doit, do_size
  integer :: ithread
  !
  ccou = 0
  !
  scc = size(ccin,1)    ! Size of a Clean Component (3 or 4)
  nin = size(ccin,2)    ! That is the number of channels
  do_size = scc.eq.4    ! Check size if needed
  !
  if (nin.eq.1) then
    ki = 0
    do ii=1,mic(1)
      if (ccin(3,1,ii).eq.0) then
        exit
      else
        doit = .true.
        do jj=1,ki
          if (ccou(1,jj,1).eq.ccin(1,1,ii)) then
            if (ccou(2,jj,1).eq.ccin(2,1,ii)) then
              if (do_size) then
                if (ccou(4,jj,1).eq.ccin(4,1,ii)) then
                  doit = .false.
                  ccou(3,jj,1) = ccou(3,jj,1) + ccin(3,1,ii)
                  exit
                endif
              else
                doit = .false.
                ccou(3,jj,1) = ccou(3,jj,1) + ccin(3,1,ii)
                exit
              endif
            endif
          endif
        enddo
        if (doit) then
          ki = ki+1
          ccou(1:scc,ki,1) = ccin(1:scc,1,ii)
        endif
      endif
    enddo
    mic(1) = ki
    !
    ! Then duplicate 
    do ic=2,last-first+1
      ccou(:,:,ic) = ccou(:,:,1)
    enddo
  else
    !
    !$OMP PARALLEL DEFAULT(none) SHARED(ccin,ccou) &
    !$OMP    & SHARED(first,last, mic, nin, do_size) &
    !$OMP    & PRIVATE(ic,jc,kc,ki, ii, doit, ithread)   
    !
    ithread = 1
    !$ ithread = omp_get_thread_num()+1
    !$OMP DO
    do ic=first,last
      jc = ic-first+1
      ki = 0
      do ii=1,mic(jc)
        kc = min(ic,nin)
        if (ccin(3,kc,ii).eq.0) then
          exit
        else
          doit = .true.
          do jj=1,ki
            if (ccou(1,jj,jc).eq.ccin(1,kc,ii)) then
              if (ccou(2,jj,jc).eq.ccin(2,kc,ii)) then
                if (do_size) then
                  if (ccou(4,jj,1).eq.ccin(4,1,ii)) then
                    doit = .false.
                    ccou(3,jj,jc) = ccou(3,jj,jc) + ccin(3,kc,ii)
                    exit
                  endif
                else
                  doit = .false.
                  ccou(3,jj,jc) = ccou(3,jj,jc) + ccin(3,kc,ii)
                  exit
                endif
              endif
            endif
          enddo
          if (doit) then
            ki = ki+1
            ccou(:,ki,jc) = ccin(:,ic,ii)
          endif
        endif
      enddo
      mic(jc) = ki
    enddo
    !$OMP END DO
    !$OMP END PARALLEL
  endif
end subroutine uv_squeeze_clean
!
subroutine uv_extract_clean(duv,ouv,nc,first,last)
  !-----------------------------------------------------------------
  ! @ private-mandatory
  !
  ! IMAGER   Support for UV_MAP  &  UV_RESTORE
  !     Extract a subset of visibilities
  !-----------------------------------------------------------------
  integer, intent(in) :: nc         ! Number of channels
  real, intent(in) :: duv(:,:)      ! Input visbilities
  real, intent(out) :: ouv(:,:)     ! Output visibilities
  integer, intent(in) :: first      ! First channel
  integer, intent(in) :: last       ! Last channel
  !
  integer :: iv, nv
  !
  nv = ubound(duv,2)                ! Number of Visibilities
  !
  do iv=1,nv
    ouv(1:7,iv) = duv(1:7,iv)
    ouv(8:,iv) = duv(5+3*first:7+3*last,iv)
  enddo
end subroutine uv_extract_clean
!
subroutine uv_clean_sizes(hcct,ccin, mic, first, last)
  use image_def
  !-----------------------------------------------------------------
  ! @ private-mandatory
  !
  ! IMAGER   Support for UV_RESTORE
  !   Compute the actual number of components
  !-----------------------------------------------------------------
  type(gildas), intent(in) :: hcct  ! header of CCT data set
  real, intent(in) :: ccin(:,:,:)   ! Clean component table
  integer, intent(out) :: mic(:)    ! Number of iterations per channel
  integer, intent(in) :: first      ! First
  integer, intent(in) :: last       ! and last channel
  ! Local
  integer :: nc
  integer :: ni, ki
  integer :: i,ic,jc, mi
  !
  nc = hcct%gil%dim(2)  ! Number of channels
  ni = hcct%gil%dim(3)  ! Maximum number of Clean Components
  ! 
  !!Print *,'Number of channels in CCT ',nc
  !!Print *,'First & Last ',first,last
  !
  if (nc.eq.1) then
    mi = ni
    do i=1,ni
      if (ccin(3,1,i).eq.0) then
        mi = i-1
        exit
      endif
    enddo
    mic(:) = mi
  else
    !
    mic(:) = ni ! Up to the last one
    do ic=first,last
      jc = ic-first+1
      ki = 0
      do i=1,ni
        if (ccin(3,ic,i).eq.0) then
          mic(jc) = i-1
          exit
        endif
      enddo
    enddo
  endif
end subroutine uv_clean_sizes
!
subroutine do_smodel (visi,nc,nv,a,nx,ny,nf,freq,xinc,yinc,factor,mthread)
  !$ use omp_lib
  !-----------------------------------------------------------------
  ! @ private-mandatory
  !
  ! IMAGER   Support for UV_RESTORE
  !
  !     Remove model visibilities from UV data set.
  !     Interpolate from a grid of values obtained by FFT before.
  !-----------------------------------------------------------------
  integer, intent(in) :: nc ! Visibility size
  integer, intent(in) :: nv ! Number of visibilities
  integer, intent(in) :: nx ! X size
  integer, intent(in) :: ny ! Y size
  integer, intent(in) :: nf ! Number of frequencies
  real, intent(inout) :: visi(:,:)    ! Computed visibilities
  real(8), intent(in) :: freq         ! Effective frequency
  real, intent(in)  :: factor         ! Flux factor
  complex, intent(in) ::  a(:,:,:)    ! FFT
  real(8), intent(in) :: xinc,yinc    ! Pixel sizes
  integer, intent(in) :: mthread
  !
  real(8), parameter :: clight=299792458d0
  real(8) :: kwx,kwy,stepx,stepy,lambda,xr,yr
  complex(8) :: aplus,amoin,azero,afin
  integer i,if,ia,ja
  !
  lambda = clight/(freq*1d6)
  stepx = 1.d0/(nx*xinc)*lambda
  stepy = 1.d0/(ny*yinc)*lambda
  !
  ! Loop on visibility
  !$OMP PARALLEL DEFAULT(none) NUM_THREADS(mthread) &
  !$OMP   SHARED(visi,a, factor,nc,nv,nf,nx,ny,stepx,stepy) &
  !$OMP   PRIVATE(kwx,ia,kwy,ja,xr,yr,aplus,azero,amoin,afin,if) 
  !$OMP DO
  do i = 1, nv
    kwx =  visi(1,i) / stepx + dble(nx/2 + 1)
    ia = int(kwx)
    if (ia.le.1) cycle
    if (ia.ge.nx) cycle
    kwy =  visi(2,i) / stepy + dble(ny/2 + 1)
    ja = int(kwy)
    if (ja.le.1) cycle
    if (ja.ge.ny) cycle
    !
    xr = kwx - ia
    yr = kwy - ja
    do if=1,nf
      !
      ! Interpolate (X or Y first, does not matter in this case)
      aplus = ( (a(ia+1,ja+1,if)+a(ia-1,ja+1,if)   &
   &          - 2.d0*a(ia,ja+1,if) )*xr   &
   &          + a(ia+1,ja+1,if)-a(ia-1,ja+1,if) )*xr*0.5d0   &
   &          + a(ia,ja+1,if)
      azero = ( (a(ia+1,ja,if)+a(ia-1,ja,if)   &
   &          - 2.d0*a(ia,ja,if) )*xr   &
   &          + a(ia+1,ja,if)-a(ia-1,ja,if) )*xr*0.5d0   &
   &          + a(ia,ja,if)
      amoin = ( (a(ia+1,ja-1,if)+a(ia-1,ja-1,if)   &
   &          - 2.d0*a(ia,ja-1,if) )*xr   &
   &          + a(ia+1,ja-1,if)-a(ia-1,ja-1,if) )*xr*0.5d0   &
   &          + a(ia,ja-1,if)
      ! Then Y (or X)
      afin = ( (aplus+amoin-2.d0*azero)   &
   &          *yr + aplus-amoin )*yr*0.5d0 + azero
      !
      visi(5+3*if,i) =  visi(5+3*if,i) - real(afin)*factor
      visi(6+3*if,i) =  visi(6+3*if,i) - imag(afin)*factor
    enddo
  enddo
  !$OMP ENDDO
  !$OMP END PARALLEL
end subroutine do_smodel
! 
subroutine cube_flag_extrema(nchan,varname,mcol,hvar)
  use clean_arrays
  use gbl_message
  use imager_interfaces, only : map_message
  ! @ private
  integer, intent(in) :: nchan
  integer, intent(in) :: mcol(2)
  character(len=*), intent(in) :: varname
  type(gildas), intent(inout) :: hvar
  !
  integer :: ic
  logical :: error
  !
  ! 1 channel only means Continuum data
  if (nchan.ne.1) then
    if (any(dchanflag.eq.0)) then
      !!Print *,'MCOL ',mcol,' Nchan ',nchan
      !!Print *,'Nullifying Filtered Channels ',dchanflag(mcol(1):mcol(2)) 
      call map_message(seve%i,'UV_MAP','Nullifying Filtered Channels ')
      do ic=1,hvar%gil%dim(3)
        if (dchanflag(ic+mcol(1)-1).eq.0) then
          hvar%r3d(:,:,ic) = 0
        endif
      enddo
    endif
  endif
  !
  ! Compute extrema
  error = .false.
  call cube_minmax(varname,hvar,error)
end subroutine cube_flag_extrema

subroutine cct_fft_size(mx,my,nx,ny)
  integer, intent(out) :: nx,ny
  integer, intent(in) :: mx,my
  !
  integer :: kx,ky
  real :: rx,ry
  !
  ! Define the image size
  rx = log(float(mx))/log(2.0)
  kx = nint(rx)
  if (kx.lt.rx) kx = kx+1
  nx = 2**kx
  ry = log(float(my))/log(2.0)
  ky = nint(ry)
  if (ky.lt.ry) ky = ky+1
  ny = 2**ky
  kx = max(nx,ny)
  kx = min(4*kx,4096)
  !
  ! If the image is already fine enough, use it "as is"
  !    even if it is not a power of two...
  nx = max(kx,mx)  ! max(kx,nx) for power of two
  ny = max(kx,my)  ! max(kx,ny) for power of two
end subroutine cct_fft_size
!
