subroutine uv_line2cont(line,error)
  use gkernel_interfaces
  use imager_interfaces, except_this=>uv_line2cont
  use clean_def
  use clean_default
  use clean_types
  use clean_arrays
  use gbl_message
  !---------------------------------------------------------------------
  ! @ private
  !
  ! IMAGER  Support routine for command 
  !     UV_CONT [Step] [/RANGE Start End TYPE] /INDEX Alpha [Frequency]
  !
  ! Create a continuum UV table from a Line one
  !---------------------------------------------------------------------
  character(len=*), intent(inout) :: line  ! Command line
  logical, intent(out) :: error            ! Error flag
  !
  character(len=*), parameter :: rname='UV_CONT'
  integer, parameter :: o_naver=0
  integer, parameter :: o_index=1
  integer, parameter :: o_range=2
  !
  real, pointer :: duv_previous(:,:), duv_next(:,:)
  integer :: nu, nv
  type (gildas) :: hcuv
  integer :: channels(3)
  !
  ! Define the Header
  call sub_uvcont_header(line,error,hcuv,channels,o_naver,o_index,o_range)
  if (error) return
  !
  ! Define the Data
  nu = hcuv%gil%dim(1)
  nv = hcuv%gil%dim(2)
  nullify (duv_previous, duv_next)
  call uv_find_buffers (rname,nu,nv,duv_previous,duv_next,error)
  if (error) return
  !
  hcuv%r2d => duv_next
  huv%r2d => duv_previous
  call sub_uvcont_data(line,hcuv,huv,channels,dchanflag,o_index,error)
  !
  call uv_clean_buffers (duv_previous, duv_next,error)
  if (error) return
  !
  ! Copy back to UV data set
  call gdf_copy_header(hcuv,huv,error)  
  !
  ! Indicate UV data has changed, and weight must be computed
  call uv_new_data(weight=.true.)
end subroutine uv_line2cont
!
subroutine sub_uvcont_header(line,error,hcuv,channels,o_step,o_index,o_range)
  use gkernel_interfaces
  use imager_interfaces, except_this=>sub_uvcont_header
  use clean_def
  use clean_default
  use clean_types
  use clean_arrays
  use gbl_message
  !---------------------------------------------------------------------
  ! @ private
  !
  ! IMAGER  Support routine for commands 
  !   UV_CONT [Step] /INDEX Alpha [Frequency] [/RANGE Min Max Type]
  ! and
  !   UV_MAP [Map Center Args] /CONT [Step] /INDEX Alpha [Frequency] [/RANGE Min Max Type]
  !
  ! Create a continuum UV table from the Line one
  !---------------------------------------------------------------------
  character(len=*), intent(inout) :: line  ! Command line
  logical, intent(out) :: error            ! Error flag
  type (gildas), intent(inout) :: hcuv     ! UV header
  integer, intent(out) :: channels(3)      ! Channels (first, last, step)
  integer, intent(in) :: o_step            ! Location of channel step
  integer, intent(in) :: o_index           ! Location of /INDEX option
  integer, intent(in) :: o_range           ! Location of /RANGE option
  !
  character(len=*), parameter :: rname='UV_CONT'
  real(kind=8), parameter :: pi=3.14159265358979323846d0
  real(kind=8), parameter :: f_to_k = 2.d0*pi/299792458.d-6
  !
  integer :: nchan
  character(len=80) :: mess
  real :: uvmax, uvmin
  integer :: msize
  type (uvmap_par) :: map
  !
  real(8) :: drange(2), freq
  character(len=12) :: csort
  integer :: isort, n
  integer :: nc(2)
  integer(kind=4), parameter :: msort=3
  character(len=12) :: types(msort),mysort
  data types /'CHANNEL','VELOCITY','FREQUENCY'/
  !
  !
  if (huv%loca%size.eq.0) then
    call map_message(seve%e,rname,'No UV data loaded')
    error = .true.
    return
  endif
  if (huv%gil%nstokes.gt.1) then
    call map_message(seve%e,rname,'Only single polarization data supported',1)
    error = .true.
    return
  endif
  !
  channels = 0
  !
  msize = maxval(default_map%size)
  if (msize.eq.0) then
    map = default_map
    freq = gdf_uv_frequency(huv)
    call uvgmax (huv,duv,uvmax,uvmin)
    ! Now transform UVMAX in kiloWavelength (including 2 pi factor)
    uvmax = uvmax*freq*f_to_k
    uvmin = uvmin*freq*f_to_k
    error = .false.
    call map_parameters(rname,map,huv,freq,uvmax,uvmin,error)
    if (error) return
    msize = maxval(map%size)
  endif
  channels = 0
  call t_channel_sampling (rname,huv,channels(3),msize)
  !
  if (sic_present(o_step,1)) then
    call sic_i4(line,o_step,1,channels(3),.false.,error)
  else
    write(mess,'(A,I0,A)') 'Averaging by ',channels(3),' original channels'
    call map_message(seve%i,rname,mess)
  endif
  !
  if (sic_present(o_range,0)) then
    ! Get channel range from /RANGE option
    call sic_r8 (line,o_range,1,drange(1),.true.,error)
    if (error) return
    call sic_r8 (line,o_range,2,drange(2),.true.,error)
    if (error) return
    call sic_ke (line,o_range,3,csort,n,.true.,error)
    if (error) return
    error = .false.
    call sic_ambigs(rname,csort,mysort,isort,types,msort,error)
    if (error)  return    
    ! Define the input range
    call out_range(rname,mysort,drange,nc,huv,error)
    if (error) return
    channels(1:2) = nc
  else
    !
    nchan = huv%gil%nchan
    channels(1) = 1
    channels(2) = nchan
  endif
  !
  call uv_cont_header(rname,huv,hcuv,channels,error)
  !
end subroutine sub_uvcont_header
!
subroutine uv_cont_header(rname,hluv,hcuv,channels,error)
  use gbl_message
  use gkernel_interfaces
  use image_def
  use imager_interfaces, only : map_message
  !-----------------------------------------------------------------------------
  ! @ private
  !   Derive the Continuum UV header from the Line UV header
  !   once the channel range has been specified
  !-----------------------------------------------------------------------------
  character(len=*), intent(in) :: rname ! Caller name
  type(gildas), intent(in) :: hluv      ! Line UV header
  type(gildas), intent(inout) :: hcuv   ! Continuum UV header
  integer, intent(in) :: channels(3)    ! Channel range
  logical, intent(out) :: error
  !
  integer :: i, mc, nchan
  character(len=80) :: mess
  logical :: add_if
  !
  nchan = channels(2)-channels(1)+1
  !
  ! Make sure things remain tractable for "random" Frequency axis...
  if (hluv%gil%nfreq.ne.0 .and. nchan.ne.1) then
    if (channels(3).ne.1) then
      !
      ! Ultimately, one may check here pseudo-linear axes...
      call map_message(seve%e,rname,'UV tables with random Frequency axis ' &
      & //' can only be converted with channel step = 1',1)
      error = .true.
      return
    endif
  endif
  !
  error = .false.
  call gildas_null(hcuv, type = 'UVT')
  call gdf_copy_header(hluv,hcuv,error)
  !
  mc = nchan/channels(3)
  if (mc*channels(3).ne.nchan) mc = mc+1
  write(mess,'(A,I0,A)') 'Generating visibilities for ',mc,' frequencies' 
  call map_message(seve%i,rname,mess)
  !
  ! Add the IF id if not already present
  hcuv%gil%dim(2) = hluv%gil%dim(2)*mc
  hcuv%gil%dim(1) = 10+hluv%gil%ntrail
  if (mc.eq.1) then
    add_if = .false.
  else if ( (hluv%gil%column_pointer(code_uvt_if).gt.0).and. &
    & (hluv%gil%column_pointer(code_uvt_if).le.hluv%gil%dim(1)) ) then
    add_if = .false.
  else
    add_if = .true.
    hcuv%gil%dim(1) = hcuv%gil%dim(1)+1
  endif
  hcuv%gil%nchan = 1
  hcuv%gil%nvisi = hcuv%gil%dim(2)
  hcuv%gil%inc(1) = hluv%gil%inc(1)*mc
  !
  ! The effective bandwidth does not account for blanked channels
  ! in the range
  hcuv%gil%fres = (channels(2)-channels(1)+1)*hcuv%gil%fres
  hcuv%gil%vres = (channels(2)-channels(1)+1)*hcuv%gil%vres
  !
  hcuv%gil%nfreq = 0 ! Reset this
  !
  ! Shift trailing columns to handle Mosaicing
  hcuv%gil%lcol = 10
  do i=1,code_uvt_last
    if (hluv%gil%column_pointer(i).gt.hluv%gil%lcol) then
      hcuv%gil%column_pointer(i) = hluv%gil%column_pointer(i)-hluv%gil%lcol+hcuv%gil%lcol
    endif
  enddo
  if (add_if) then
    hcuv%gil%column_pointer(code_uvt_if) = hcuv%gil%dim(1)
    hcuv%gil%column_size(code_uvt_if) = 1
    hcuv%gil%ntrail = hcuv%gil%ntrail + 1
  endif
  !
  error = .false.
  call gdf_setuv(hcuv,error) ! Test
end subroutine uv_cont_header
!
subroutine sub_uvcont_data(line,hcuv,hiuv,channels,chflag,o_index,error)
  use image_def
  use gkernel_interfaces
  use imager_interfaces, except_this=>sub_uvcont_data
  !---------------------------------------------------------------------
  ! @ private-mandatory
  !
  !   IMAGER
  !     Transform the Line UV data into a Continuum UV data
  !---------------------------------------------------------------------  
  character(len=*), intent(in) :: line
  type(gildas), intent(inout) :: hcuv
  type(gildas), intent(inout) :: hiuv
  integer, intent(in) :: channels(3)
  integer, intent(in) :: chflag(:)
  integer, intent(in) :: o_index
  logical, intent(inout) :: error
  !
  real :: alpha
  real(8) :: ofreq
  integer :: uvcode=1
  !
  if (sic_present(o_index,0)) then
    call sic_r4(line,o_index,1,alpha,.true.,error)
    if (error) return
    ofreq = 0.D0
    call sic_r8(line,o_index,2,ofreq,.false.,error)
    if (error) return
    call t_continuum(hiuv,hcuv,channels,chflag,uvcode,error,alpha,ofreq)
  else
    call t_continuum(hiuv,hcuv,channels,chflag,uvcode,error)
  endif
  !
end subroutine sub_uvcont_data
!
subroutine t_continuum(hluv,hcuv,channels,chflag,uvcode,error,spindex,ofreq)
  use gkernel_interfaces
  use gildas_def
  use image_def
  use uvmap_def
  use gbl_message
  use imager_interfaces, only : map_message
  !$ use omp_lib
  !---------------------------------------------------------------------
  ! @ public-mandatory
  !
  ! IMAGER
  !   Support routine for UV_CONTINUUM 
  !
  !   Create a continuum UV table from a Line one
  !---------------------------------------------------------------------
  !
  type (gildas), intent(in) :: hluv       ! Line UV header
  type (gildas), intent(inout) :: hcuv    ! Continuum UV header
  integer, intent(in) :: channels(3)      ! Channel selection
  integer, intent(in) :: chflag(:)        ! Channel Flags
  integer, intent(in) :: uvcode           ! Type of UV data
  logical, intent(out) :: error
  real, intent(in), optional :: spindex   ! Spectral index
  real(8), intent(in), optional :: ofreq  ! Reference Frequency
  !
  character(len=*), parameter :: rname='UV_CONT'
  character(len=80) :: mess
  real(8) :: freq,fval,scale_uv,scale_flux,scale_w
  real, allocatable :: are(:), aim(:), awe(:), adw(:)
  real :: re, im, we, dw, alpha
  integer :: ifi,ila,nv,ic,jc,kc,ier
  integer :: nt,ft
  integer :: if_add, kif ! Pointer to IF column, and IF pointer
  integer(kind=index_length) :: iv,ov
  integer :: othread, nthread, ov_num, env
  real(8), allocatable :: uv_scale(:)
  real(8), parameter :: clight_kms = 299792.458d0   ! [km/s] Light velocity
  !
  error = .false.
  !
  ! Define number of Visibilities and Channels...
  nv = hluv%gil%nvisi
  nt = hluv%gil%ntrail
  ft = hluv%gil%lcol+1
  if ( (hluv%gil%column_pointer(code_uvt_if).ge.ft) .and. &
    & (hluv%gil%column_pointer(code_uvt_if).le.hluv%gil%dim(1)) ) then
    if_add = 0
  else
    if_add = hcuv%gil%column_pointer(code_uvt_if)
  endif
  !
  ov_num =  (channels(2)-channels(1))/channels(3)+1
  !
  allocate(uv_scale(channels(2)),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,rname,'Memory allocation error')
    error = .true.
    return
  endif
  !
  ! The problem is to define the effective "mean" frequency of the observation.  
  !
  !! fval = hluv%gil%val(hluv%gil%faxi)    ! This is not correct... 
  ! For linear sampling this is
  !! fval = fval + ((channels(1)+channels(2))*0.5d0-hluv%gil%ref(hluv%gil%faxi)) * hluv%gil%inc(hluv%gil%faxi)  
  ! or more precisely
  !! fval = gdf_uv_frequency(hluv, dble (channels(1)+channels(2))/2.d0 )
  !
  ! In general, including irregular sampling
  fval = 0d0
  do ic=channels(1),channels(2)
    fval = fval+gdf_uv_frequency(hluv, dble (ic))
  enddo
  fval = fval/dble(channels(2)-channels(1)+1)
  if (present(spindex)) then
    alpha = spindex    
    if (ofreq.ne.0D0) then
      if (abs(fval-ofreq)/fval.gt.0.3) then
        write (mess,'(A,F10.0,A)') 'Specified frequency is too far from mean frequency ',fval,' MHz'
        call map_message(seve%e,rname,mess,3)
        error = .true.
        return
      endif
      fval = ofreq
    endif
  else
    alpha = 0.
  endif
  !
  ! All these assume similar weights for all channels
  !
  do ic=channels(1),channels(2),channels(3)
    freq = gdf_uv_frequency(hluv, dble (2*ic+channels(3)-1)/2.d0 )
    !
    ! The angular scale goes as Lambda/B, i.e. 1/(B nu)
    ! so to preserve a constant angular scale, the "apparent"
    ! B must scale as \nu
    !
    ! so
    uv_scale(ic) = freq/fval ! This is the right formula...
    scale_uv = uv_scale(ic)
    scale_flux = scale_uv**(-alpha)
    scale_w = scale_uv**(2*alpha)
    !! if (alpha.ne.0) Print *,'Scale UV ',ic, (2*ic+channels(3)-1.0)/2.0, scale_uv, fval, freq, scale_flux, scale_w
  enddo
  !  
  if (alpha.ne.0.) then
    !
    ! Spectral index code
    if (uvcode.eq.code_uvt) then  ! UVT order
      !
      !$OMP PARALLEL DEFAULT(none) &
      !$OMP   & SHARED(nv, ov_num, uv_scale, hluv, hcuv, channels, if_add) &
      !$OMP   & SHARED(chflag, alpha, nt, ft) &
      !$OMP   & PRIVATE(iv,ov,fval,ic,freq,re,im,we,kc,jc,dw,kif) &
      !$OMP   & PRIVATE(scale_uv,scale_flux,scale_w)
      !
      !$OMP DO
      do iv=1,nv
        ov = (iv-1)*ov_num+1
        !
        ! Fill in, channel after channel
        kif = 0
        do ic=channels(1),channels(2),channels(3)
          scale_uv = uv_scale(ic)
          scale_flux = scale_uv**(-alpha)
          scale_w = scale_uv**(2*alpha)
          !
          hcuv%r2d(1:3,ov) = hluv%r2d(1:3,iv)*scale_uv
          hcuv%r2d(4:7,ov) = hluv%r2d(4:7,iv)
          !
          ! Compact the channels first
          if (channels(3).gt.1) then
            re = 0
            im = 0
            we = 0
            kc = 5+3*ic
            do jc = ic,min(ic+channels(3)-1,channels(2))
              if (chflag(jc).eq.1) then
                ! Here we use the Channel mask defined by UV_FILTER in addition
                ! to any previous flag.   UV_FILTER /RESET would then work in all cases
                dw = max(0.,hluv%r2d(kc+2,iv))
                !
                re = re + hluv%r2d(kc,iv)*dw
                kc = kc+1
                im = im + hluv%r2d(kc,iv)*dw
                we = we+dw
                kc = kc+2
              else
                kc = kc+3   ! Skip channel
              endif
            enddo
            if (we.ne.0.0) then
              re = re/we
              im = im/we
            end if
            hcuv%r2d(8,ov) = re*scale_flux
            hcuv%r2d(9,ov) = im*scale_flux
            hcuv%r2d(10,ov) = we*scale_w
          else
            if (chflag(ic).eq.1) then
              ! Here we use the Channel mask defined by UV_FILTER in addition
              ! to any previous flag.   UV_FILTER /RESET would then work in all cases
              hcuv%r2d(8:9,ov)  = hluv%r2d(5+3*ic:6+3*ic,iv)*scale_flux
              hcuv%r2d(10,ov)  = hluv%r2d(7+3*ic,iv)*scale_w
            else
              hcuv%r2d(8:10,ov) = 0.0
            endif
          endif
          if (nt.gt.0) then
            hcuv%r2d(11:11+nt-1,ov) = hluv%r2d(ft:ft+nt-1,iv)
          endif
          if (if_add.ne.0) then
            kif = kif+1
            hcuv%r2d(if_add,ov) = kif
          endif
          ov = ov+1
        enddo
        if (ov.ne.iv*ov_num+1) print *,'Programming error ',iv,ov,ov_num
      enddo
      !$OMP END DO
      !$OMP END PARALLEL
      !
    else if (uvcode.eq.code_tuv) then ! TUV order
      !
      Print *,'TUV order '
      if (channels(3).gt.1) then
        env = nv  ! Work arrays per visibility
      else
        env = 1   ! Dummies
      endif
      allocate (are(env),aim(env),awe(env),adw(env),stat=ier)
      if (ier.ne.0) then
        call map_message(seve%e,rname,'Channels allocation error')
        error = .true.
        return
      endif
      !
      ! Fill in, channel after channel
      othread = 1
      nthread = 1
      !$ othread = omp_get_max_threads()
      !$ nthread = min(othread,ov_num)
      !$ call omp_set_num_threads(nthread)
      !
      !$OMP PARALLEL DEFAULT(none) &
      !$OMP   & SHARED(nv,uv_scale,hluv,hcuv,channels,alpha,nt,ft,chflag,if_add) &
      !$OMP   & PRIVATE(ic,ifi,ila,freq) &
      !$OMP   & PRIVATE(are,aim,awe,jc,adw,kif) &
      !$OMP   & PRIVATE(scale_uv,scale_flux,scale_w)
      !
      !$OMP DO
      do ic=channels(1),channels(2),channels(3)
        ! Block counter
        kif = (ic-channels(1)+channels(3)-1)/channels(3) + 1
        ifi = (ic-1)*nv+1
        ila = ifi+nv-1
        scale_uv = uv_scale(ic)
        scale_flux = scale_uv**(-alpha)
        scale_w = scale_uv**(2*alpha)
        !
        hcuv%r2d(ifi:ila,1:3) = hluv%r2d(1:nv,1:3)*scale_uv
        hcuv%r2d(ifi:ila,4:7) = hluv%r2d(1:nv,4:7)
        !
        ! Compact the channels first
        if (channels(3).gt.1) then
          are = 0
          aim = 0
          awe = 0
          do jc = ic,min(ic+channels(3)-1,channels(2))
            if (chflag(jc).eq.1) then
              ! Here we use the Channel mask defined by UV_FILTER in addition
              ! to any previous flag.   UV_FILTER /RESET would then work in all cases
              adw(:) = max(0.,hluv%r2d(1:nv,7+3*jc))
              are(:) = are + hluv%r2d(1:nv,5+3*jc)*adw
              aim(:) = aim + hluv%r2d(1:nv,6+3*jc)*adw
              awe(:) = awe+adw
            endif
          enddo
          where (awe.ne.0.)
            are(:) = are/awe
            aim(:) = aim/awe
          end where
          hcuv%r2d(ifi:ila,8) = are*scale_flux
          hcuv%r2d(ifi:ila,9) = aim*scale_flux
          hcuv%r2d(ifi:ila,10) = awe*scale_w
        else
          if (chflag(ic).eq.1) then
            hcuv%r2d(ifi:ila,8:9) = hluv%r2d(1:nv,5+3*ic:6+3*ic)*scale_flux
            ! Here we use the Channel mask defined by UV_FILTER in addition
            ! to any previous flag.   UV_FILTER /RESET would then work in all cases
            hcuv%r2d(ifi:ila,10) = hluv%r2d(1:nv,7+3*ic)*scale_w
          else
            hcuv%r2d(ifi:ila,8:10) = 0.
          endif
        endif
        if (nt.gt.0) then
          hcuv%r2d(ifi:ila,11:11+nt-1) = hluv%r2d(1:nv,ft:ft+nt-1)
        endif
        if (if_add.ne.0) then
          hcuv%r2d(ifi:ila,if_add) = kif
        endif
      enddo
      !$OMP END DO
      !$OMP END PARALLEL
      !$ call omp_set_num_threads(othread)
      deallocate (are,aim,awe,stat=ier)
    endif
    !
  else
    ! Simple code with no spectral index
    if (uvcode.eq.code_uvt) then  ! UVT order
      !
      !$OMP PARALLEL DEFAULT(none) &
      !$OMP   & SHARED(nv, ov_num, uv_scale, hluv, hcuv, channels, nt, ft, chflag, if_add) &
      !$OMP   & PRIVATE(iv,ov,fval,ic,freq,scale_uv,re,im,we,kc,jc,dw, kif)
      !
      !$OMP DO
      do iv=1,nv
        ov = (iv-1)*ov_num+1
        !
        ! Fill in, channel after channel
        kif = 0
        do ic=channels(1),channels(2),channels(3)
          scale_uv = uv_scale(ic)
          !
          hcuv%r2d(1:3,ov) = hluv%r2d(1:3,iv)*scale_uv
          hcuv%r2d(4:7,ov) = hluv%r2d(4:7,iv)
          !
          ! Compact the channels first
          if (channels(3).gt.1) then
            re = 0
            im = 0
            we = 0
            kc = 5+3*ic
            do jc = ic,min(ic+channels(3)-1,channels(2))
              if (chflag(jc).eq.1) then
                ! Here we use the Channel mask defined by UV_FILTER in addition
                ! to any previous flag.   UV_FILTER /RESET would then work in all cases
                dw = max(0.,hluv%r2d(kc+2,iv))
                re = re + hluv%r2d(kc,iv)*dw
                kc = kc+1
                im = im + hluv%r2d(kc,iv)*dw
                we = we+dw
                kc = kc+2
              else
                kc = kc+3
              endif
            enddo
            if (we.ne.0.0) then
              re = re/we
              im = im/we
            end if
            hcuv%r2d(8,ov) = re
            hcuv%r2d(9,ov) = im
            hcuv%r2d(10,ov) = we
          else
            if (chflag(ic).eq.1) then
              ! Here we use the Channel mask defined by UV_FILTER in addition
              ! to any previous flag.   UV_FILTER /RESET would then work in all cases
              hcuv%r2d(8:10,ov)  = hluv%r2d(5+3*ic:7+3*ic,iv)
            else
              hcuv%r2d(8:10,ov)  = 0.
            endif            
          endif
          if (nt.gt.0) then
            hcuv%r2d(11:11+nt-1,ov) = hluv%r2d(ft:ft+nt-1,iv)
          endif
          if (if_add.ne.0) then
            kif = kif+1
            hcuv%r2d(if_add,ov) = kif
          endif
          ov = ov+1
        enddo
        if (ov.ne.iv*ov_num+1) print *,'Programming error ',iv,ov,ov_num
      enddo
      !$OMP END DO
      !$OMP END PARALLEL
      !
    else if (uvcode.eq.code_tuv) then ! TUV order
      Print *,'TUV order '
      if (channels(3).gt.1) then
        env = nv
      else 
        env = 1
      endif
      allocate (are(env),aim(env),awe(env),adw(env),stat=ier)
      if (ier.ne.0) then
        call map_message(seve%e,rname,'Channels allocation error')
        error = .true.
        return
      endif
      !
      ! Fill in, channel after channel
      othread = 1
      nthread = 1
      !$ othread = omp_get_max_threads()
      !$ nthread = min(othread,ov_num)
      !$ call omp_set_num_threads(nthread)
      !
      !$OMP PARALLEL DEFAULT(none) &
      !$OMP   & SHARED(nv,uv_scale,hluv,hcuv,channels,nt,ft,chflag,if_add) &
      !$OMP   & PRIVATE(ic,ifi,ila,freq,scale_uv) &
      !$OMP   & PRIVATE(are,aim,awe,jc,adw,kif)
      !
      !$OMP DO
      do ic=channels(1),channels(2),channels(3)
        ! Block counter
        kif = (ic-channels(1)+channels(3)-1)/channels(3) + 1
        !
        ifi = (ic-1)*nv+1
        ila = ifi+nv-1
        scale_uv = uv_scale(ic)
        !
        hcuv%r2d(ifi:ila,1:3) = hluv%r2d(1:nv,1:3)*scale_uv
        hcuv%r2d(ifi:ila,4:7) = hluv%r2d(1:nv,4:7)
        !
        ! Compact the channels first
        if (channels(3).gt.1) then
          are = 0
          aim = 0
          awe = 0
          do jc = ic,min(ic+channels(3)-1,channels(2))
            if (chflag(jc).eq.1) then
              ! Here we use the Channel mask defined by UV_FILTER in addition
              ! to any previous flag.   UV_FILTER /RESET would then work in all cases
              adw(:) = max(0.,hluv%r2d(1:nv,7+3*jc))
              are(:) = are + hluv%r2d(1:nv,5+3*jc)*adw
              aim(:) = aim + hluv%r2d(1:nv,6+3*jc)*adw
              awe(:) = awe+adw
            endif
          enddo
          where (awe.ne.0.)
            are(:) = are/awe
            aim(:) = aim/awe
          end where
          hcuv%r2d(ifi:ila,8) = are
          hcuv%r2d(ifi:ila,9) = aim
          hcuv%r2d(ifi:ila,10) = awe
        else
          if (chflag(ic).eq.1) then
            ! Here we use the Channel mask defined by UV_FILTER in addition
            ! to any previous flag.   UV_FILTER /RESET would then work in all cases
            hcuv%r2d(ifi:ila,8:10) = hluv%r2d(1:nv,5+3*ic:7+3*ic)
          else
            hcuv%r2d(ifi:ila,8:10) = 0.
          endif
        endif
        if (nt.gt.0) then
          hcuv%r2d(ifi:ila,11:11+nt-1) = hluv%r2d(1:nv,ft:ft+nt-1)
        endif
        if (if_add.ne.0) then
          hcuv%r2d(ifi:ila,if_add) = kif
        endif
      enddo
      !$OMP END DO
      !$OMP END PARALLEL
      !$ call omp_set_num_threads(othread)
      deallocate (are,aim,awe,stat=ier)
    endif
    !
  endif
  !
  ! FVAL should be the new Rest Frequency
  fval = fval / (1.d0 + hcuv%gil%dopp - hcuv%gil%voff/clight_kms)
  call gdf_modify(hcuv,hcuv%gil%voff,fval,error=error)
  hcuv%gil%ref(1) = 1.D0 
  !
  ! Update number of visibilities
  hcuv%gil%nvisi = nv*ov_num
end subroutine t_continuum
!
subroutine t_channel_sampling(rname,huv,nident,msize)
  use image_def
  use clean_default
  use gbl_message
  use imager_interfaces, only : map_message
  !---------------------------------------------------------------------
  ! @ public-mandatory
  !
  ! IMAGER
  !   Check how many channels can share the same beam with a user
  !   specified accuracy
  !---------------------------------------------------------------------
  character(len=*), intent(in) :: rname      ! Caller name
  type(gildas), intent(in) :: huv            ! UV data header
  integer, intent(out) :: nident             ! returned step
  integer, intent(in) :: msize               ! Current map_size
  !
  real :: df, dc, r
  integer :: q
  character(len=80) :: chain
  !
  !
  ! Allow Cell_Precis (default 10 %) pixel difference at map edge
  !
  ! Relative Delta Frequency per channel is
  df = abs(huv%gil%fres/huv%gil%freq)
  !
  ! so scale error at map edge is, in pixel units
  dc = df * msize / 2
  !
  ! so allowed number of channels with same beam is
  nident = max(nint(default_map%precis/dc),1)
  nident = min(nident,huv%gil%nchan)
  !
  write(chain,'(A,I6,A,F10.1)') 'Maximum number of channels '// &
    & 'for same beam ',nident,' Bandwidth ',abs(nident*huv%gil%fres)
  call map_message(seve%i,rname,chain)
  !
  ! With random Frequency axis, use only 1 channel
  if (huv%gil%nfreq.gt.1) then
    nident = 1
    return
  endif
  !
  ! Equalize the number of averaged channels to get channels of similar widths. 
  r = real(huv%gil%nchan)/real(nident)
  if (r.lt.10.0) then
    q = int(r)
    if (real(q).ne.r) q = q+1
    nident = int((huv%gil%nchan+q-1.)/real(q))
  endif
  !  
end subroutine t_channel_sampling
!
