subroutine uv_fields_comm(line,comm,error)
  use gildas_def
  use image_def
  use gkernel_types
  use gbl_message
  use clean_arrays
  !
  use gkernel_interfaces
  use imager_interfaces, except_this => uv_fields_comm
  use iso_c_binding
  !---------------------------------------------------------------------
  ! @ private
  !
  ! IMAGE
  !   Support command for @ fits_to_uvt.ima,
  !   the UVFITS to UV Table conversion script.
  !
  !   UV_FIELDS RaVar DecVar [Unit] [/FILE FileIn FileOut]
  !       [/CENTER Ra Dec]
  !
  !     Replace fields ID by Fields offsets.
  !
  ! For files imported from UVFITS, the Fields offsets are found in 
  !   the AIPS SU Table which is not decoded by the FITS command, 
  !   but only through the DEFINE FITS command.
  !---------------------------------------------------------------------
  character(len=*), intent(in) :: line
  character(len=*), intent(in) :: comm
  logical, intent(out) :: error
  !
  real(8), parameter :: pi=3.14159265358979323846d0
  real(4), parameter :: rad_to_sec=180.*3600./pi
  integer, parameter :: o_center=1
  integer, parameter :: o_file=2
  character(len=*), parameter :: rname='UV_FIELDS'
  !
  character(len=filename_length) :: uvdata
  character(len=64) :: coff_x, coff_y
  character(len=16) :: chain
  integer :: nf, ier, i, n
  real(kind=8), allocatable :: raoff(:), deoff(:), rapos(:), depos(:)
  real(kind=8), allocatable :: offs(:,:)
  real(kind=4), allocatable :: tmp(:)
  logical :: found
  type(sic_descriptor_t) :: desc_x,desc_y
  !
  type(gildas) :: hinuv
  real(kind=8) :: px, py
  character(len=32) :: cx, cy
  character(len=20) :: c20
  character(len=80) :: c80
  integer :: next
  type(projection_t) :: uv_proj
  type(c_ptr) :: cptr
  real(4), pointer :: r1ptr(:)
  real(8), pointer :: d1ptr(:)
  integer(kind=address_length) :: addrx, addry
  !
  ! Code:
  error = .false.
  !
  call sic_ch(line,0,1,coff_x,n,.true.,error)
  if (error) return
  call sic_descriptor(coff_x,desc_x,found)  
  if (.not.found) then
    call map_message(seve%e,rname,'X pointing variable '//trim(coff_x)//' does not exist')
    error = .true.
    return
  endif
  !
  call sic_ch(line,0,2,coff_y,n,.true.,error)
  if (error) return
  call sic_descriptor(coff_y,desc_y,found)  
  if (.not.found) then
    call map_message(seve%e,rname,'Y pointing variable '//trim(coff_y)//' does not exist')
    error = .true.
    return
  endif
  !
  ! Unit is either ABS or Angle Unit (R, D, M, S) - Not fully coded yet...
  if (desc_x%type.gt.0) then
    if (desc_y%type.gt.0) then
      chain = 'ABS'
    else
      call map_message(seve%e,rname,'Invalid mixture of X and Y pointing variable types')
      error = .true.
      return
    endif
  else
    chain = 'RAD'
  endif
  call sic_ke(line,0,3,chain,n,.false.,error)
  chain = chain(1:3)
  !
  if (any(desc_x%dims.ne.desc_y%dims)) then
    call map_message(seve%e,rname,'Pointing variables '//trim(coff_x)// &
      & ' and '// trim(coff_y)//' do not match')
    error = .true.
  else if (desc_x%ndim.ne.1) then
    call map_message(seve%e,rname,'Pointing variables '//trim(coff_x)// &
      & ' and '// trim(coff_y)//' must be of Rank 1')
    error = .true.
  endif
  if (error) return
  !
  nf = desc_x%dims(1)
  !
  allocate (raoff(nf),deoff(nf),offs(2,nf),rapos(nf),depos(nf),tmp(nf),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,rname,'Memory allocation error')
    error = .true.
    return
  endif
  !
  ! We must read the HEADER for the /FILE stuff, in case there is an angle...
  call gildas_null (hinuv, type = 'UVT')     ! Define a UVTable gildas header
  if (sic_present(o_file,0)) then
    call sic_ch(line,o_file,1,uvdata,n,.true.,error)
    if (error) return
    !
    call gdf_read_gildas (hinuv, uvdata, '.uvt', error, data=.false.)
    if (error) then
      call map_message(seve%e,rname,'Cannot read input UV header')
      return
    endif
  else
    call gdf_copy_header(huv,hinuv,error)
  endif
  !
  ! OK, get the Fields coordinates from the Variable Descriptors
  if (.not.sic_present(o_center,0)) then
    !
    ! no /CENTER option: these are offsets, as specified by user
    call adtoad(desc_x%addr,cptr,1)
    if (desc_x%type.eq.fmt_r4) then
      call c_f_pointer(cptr,r1ptr,[nf])
      raoff(:) = r1ptr
    else if (desc_x%type.eq.fmt_r8) then
      call c_f_pointer(cptr,d1ptr,[nf])
      raoff(:) = d1ptr
    else
      call map_message(seve%e,rname,'Offset Pointing variables must be Real or Double')
      error = .true.
      return
    endif
    !
    call adtoad(desc_y%addr,cptr,1)
    if (desc_x%type.eq.fmt_r4) then
      call c_f_pointer(cptr,r1ptr,[nf])
      deoff(:) = r1ptr
    else if (desc_x%type.eq.fmt_r8) then
      call c_f_pointer(cptr,d1ptr,[nf])
      deoff(:) = d1ptr
    else
      call map_message(seve%e,rname,'Offset Pointing variables must be Real or Double')
      error = .true.
      return
    endif
    !
    ! Scale with appropriate angular unit.
    select case (chain)
    case ('RAD')
      offs(1,:) = raoff
      offs(2,:) = deoff
    case ('DEG')
      offs(1,:) = raoff*pi/180.d0
      offs(2,:) = deoff*pi/180.d0
    case ('MIN')
      offs(1,:) = raoff*pi/180.d0/60.d0
      offs(2,:) = deoff*pi/180.d0/60.d0
    case ('SEC')
      offs(1,:) = raoff*pi/180.d0/3600.d0
      offs(2,:) = deoff*pi/180.d0/3600.d0
    case default
      call map_message(seve%e,rname,'Unknown offset units '//trim(chain))
      error = .true.
      return
    end select     
  else
    ! /CENTER option
    !   These are absolute coordinates, either as Characters
    !   or with the specified unit. 
    !
    ! The pointing center is specified, but the Angle is inherited
    !   from the UV data.
    !
    call sic_ch(line,o_center,1,cx,n,.true.,error)
    if (error) return
    call gwcs_sexa(cx,px,error)
    if (error) return
    px = px*pi/12.d0
    hinuv%gil%a0 = px
    !
    call sic_ch(line,o_center,2,cy,n,.true.,error)
    if (error) return
    call gwcs_sexa(cy,py,error)
    if (error) return
    py = py*pi/180.d0
    hinuv%gil%d0 = py
    !
    ! Set the Projection
    call gwcs_projec(hinuv%gil%a0,hinuv%gil%d0,hinuv%gil%pang,hinuv%gil%ptyp,uv_proj,error)
    if (error) return
    !
    call map_message(seve%i,rname,'Centering Mosaic on R.A. '//trim(cx)//'   Dec. '//cy)
    !      
    if (desc_x%type.le.0 .or. desc_y%type.le.0) then
      !
      ! no ABSOLUTE keyword: these are centers, in the specified unit.
      call adtoad(desc_x%addr,cptr,1)
      if (desc_x%type.eq.fmt_r4) then
        call c_f_pointer(cptr,r1ptr,[nf])
        rapos(:) = r1ptr
      else if (desc_x%type.eq.fmt_r8) then
        call c_f_pointer(cptr,d1ptr,[nf])
        rapos(:) = d1ptr
      else
        call map_message(seve%e,rname,'Pointing variables must be Real or Double')
        error = .true.
        return
      endif
      !
      call adtoad(desc_y%addr,cptr,1)
      if (desc_y%type.eq.fmt_r4) then
        call c_f_pointer(cptr,r1ptr,[nf])
        depos(:) = r1ptr
      else if (desc_y%type.eq.fmt_r8) then
        call c_f_pointer(cptr,d1ptr,[nf])
        depos(:) = d1ptr
      else
        call map_message(seve%e,rname,'Pointing variables must be Real or Double')
        error = .true.
        return
      endif
      !
      select case (chain)
      case ('RAD')
        continue
      case ('DEG')
        rapos = rapos*pi/180.d0
        depos = depos*pi/180.d0
      case ('MIN')
        rapos = rapos*pi/180.d0/60.d0
        depos = depos*pi/180.d0/60.d0
      case ('SEC')
        rapos = rapos*pi/180.d0/60.d0
        depos = depos*pi/180.d0/60.d0
      case default
        call map_message(seve%e,rname,'Unknown pointing units '//trim(chain))
        error = .true.
        return
      end select     
      !
      ! Convert to Offsets
      call abs_to_rel(uv_proj,rapos,depos,raoff,deoff,nf)
      offs(1,:) = raoff
      offs(2,:) = deoff
    else
      ! Character case
      addrx = desc_x%addr
      addry = desc_y%addr
      do i=1,nf
        call destoc(desc_x%type,addrx,cx)
        addrx = addrx + desc_x%type
        call destoc(desc_y%type,addry,cy)
        addry = addry + desc_y%type
        call gwcs_sexa(cx,px,error)
        px = px*pi/12.d0
        call gwcs_sexa(cy,py,error)
        py = py*pi/180.d0
        !
        call abs_to_rel(uv_proj,px,py,offs(1,i),offs(2,i),1)
      enddo
    endif
    !
    ! List the Offsets
    call map_message(seve%i,rname,'Offsets : ')
    next = 1
    do i=1,nf
      write(c20,'(A,F8.3,A,F8.3,A)') "(",offs(1,i)*rad_to_sec,',',offs(2,i)*rad_to_sec,')'
      c80(next:) = c20
      next = next+20
      if (next.ge.80) then
        write(*,'(A)') c80
        next = 1
      endif
    enddo
    if (next.gt.1) write(*,'(A)') c80(1:next-1)
  endif
  !
  if (sic_present(o_file,0)) then
    !
    call sub_file_fields(hinuv,nf,offs,error)
    if (error) return
  else
    call sub_memory_fields(huv,duv,nf,offs,error)
    if (error) return
    if (sic_present(o_center,0)) then
      huv%gil%a0 = hinuv%gil%a0
      huv%gil%d0 = hinuv%gil%d0
      huv%gil%ra = hinuv%gil%a0
      huv%gil%dec = hinuv%gil%d0
    endif
  endif
end subroutine uv_fields_comm
!
subroutine sub_file_fields (hinuv, nf, offs, error)
  use gkernel_interfaces
  use image_def
  use gbl_message
  use imager_interfaces, only : map_message
  !---------------------------------------------------------------------
  ! @ private
  !
  ! IMAGER
  !   Support for @ fits_to_uvt.map
  !     Replace fields ID by Fields offsets
  !---------------------------------------------------------------------
  type (gildas), intent(inout) :: hinuv    ! Input Header
  integer, intent(in) :: nf                ! Number of fields
  real(kind=8), intent(in) :: offs(2,nf)   ! Offset values
  logical, intent(out) :: error            ! Error flag
  !
  ! Local variables
  character(len=*), parameter :: rname='UV_FIELDS'
  character(len=80)  :: mess
  integer :: ier, nblock, ib, invisi, iv, ixoff, iyoff, isou, icode
  real :: rsou
  !
  ! Update the proper columns
  if (hinuv%gil%column_size(code_uvt_id).ne.0) then
    call map_message(seve%i,rname,'Replacing Field numbers by PHASE offsets')
    ixoff = hinuv%gil%column_pointer(code_uvt_id)
    iyoff = ixoff+1
    hinuv%gil%column_pointer(code_uvt_loff) = ixoff
    hinuv%gil%column_pointer(code_uvt_moff) = iyoff
    hinuv%gil%column_size(code_uvt_loff) = 1
    hinuv%gil%column_size(code_uvt_moff) = 1
    hinuv%gil%column_pointer(code_uvt_id) = 0
    hinuv%gil%column_size(code_uvt_id) = 0
    icode = 0
  else if (hinuv%gil%column_size(code_uvt_loff).ne.0) then
    ixoff =     hinuv%gil%column_pointer(code_uvt_loff)
    iyoff =     hinuv%gil%column_pointer(code_uvt_moff)
    call map_message(seve%w,rname,'UV Table is already of type PHASE, checking Field numbers')
    icode = 1
  else if (hinuv%gil%column_size(code_uvt_xoff).ne.0) then
    ixoff =     hinuv%gil%column_pointer(code_uvt_xoff)
    iyoff =     hinuv%gil%column_pointer(code_uvt_yoff)
    call map_message(seve%w,rname,'UV Table is already of type POINT, checking Field numbers')
    icode = -1
  else
    call map_message(seve%e,rname,'UV table is not a mosaic')
    error = .true.
    return
  endif
  !
  ! Define blocking factor, on largest data file, usually the input one
  ! but not always...
  call gdf_nitems('SPACE_GILDAS',nblock,hinuv%gil%dim(1)) ! Visibilities at once
  nblock = min(nblock,hinuv%gil%dim(2))
  ! Allocate respective space for each file
  allocate (hinuv%r2d(hinuv%gil%dim(1),nblock), stat=ier)
  if (ier.ne.0) then
    write(mess,*) 'Memory allocation error ',hinuv%gil%dim(1), nblock
    call map_message(seve%e,rname,mess)
    error = .true.
    return
  endif
  !
  ! Loop over line table by block
  hinuv%blc = 0
  hinuv%trc = 0
  !
  do ib = 1,hinuv%gil%dim(2),nblock
    write(mess,*) ib,' / ',hinuv%gil%dim(2),nblock
    call map_message(seve%i,rname,mess)
    hinuv%blc(2) = ib
    hinuv%trc(2) = min(hinuv%gil%dim(2),ib-1+nblock)
    invisi = hinuv%trc(2)-hinuv%blc(2)+1
    call gdf_read_data(hinuv,hinuv%r2d,error)
    if (error) return
    !
    if (icode.eq.0) then
      do iv=1,invisi
        isou = hinuv%r2D(ixoff,iv)
        if (isou.gt.0 .and. isou.le.nf) then
          hinuv%r2d(ixoff,iv) = offs(1,isou)
          hinuv%r2d(iyoff,iv) = offs(2,isou)
        else
          write(mess,'(A,I0,A,I0)') 'Invalid source ID ',isou,' for visibility ',iv
          call map_message(seve%w,rname,mess)
        endif
      enddo
    else
      do iv=1,invisi
        isou = hinuv%r2D(ixoff,iv)
        rsou = isou
        if (rsou.ne.hinuv%r2D(ixoff,iv)) then  
          mess = 'Non integer field number'
          error = .true.
        else if (isou.lt.1 .or. isou.gt.nf) then
          mess = 'Field number out of range'
          error = .true.
        endif
        if (error) then
          call map_message(seve%e,rname,mess)
          error = .true.
          return
        endif
        hinuv%r2d(ixoff,iv) = offs(1,isou)
        hinuv%r2d(iyoff,iv) = offs(2,isou)
      enddo
    endif
    !
    call gdf_write_data (hinuv,hinuv%r2d,error)
    if (error) return
  enddo
  !
  ! Finalize the image
  call gdf_update_header(hinuv,error)
  if (error) return
  call gdf_close_image(hinuv,error)
end subroutine sub_file_fields

subroutine sub_memory_fields (hinuv, duv, nf, offs, error)
  use gkernel_interfaces
  use image_def
  use gbl_message
  use imager_interfaces, only : map_message
  !---------------------------------------------------------------------
  ! @ private
  !
  ! IMAGER
  !   Support for @ fits_to_uvt.map and READ UV ...uvfits
  !     Replace fields ID by Fields offsets
  !---------------------------------------------------------------------
  type(gildas), intent(inout) :: hinuv     ! Input / Output file UV header
  real, intent(inout) :: duv(:,:)          ! UV data
  integer, intent(in) :: nf                ! Number of fields
  real(kind=8), intent(in) :: offs(2,nf)   ! Offset values
  logical, intent(out) :: error            ! Error flag
  !
  ! Local variables
  character(len=*), parameter :: rname='UV_FIELDS'
  character(len=80)  :: mess
  integer :: iv, ixoff, iyoff, isou, icode
  real :: rsou
  !
  if (hinuv%loca%size.eq.0) then
    call map_message(seve%e,rname,'No UV data')
    error = .true.
    return
  endif  
  !
  ! Update the proper columns
  if (hinuv%gil%column_size(code_uvt_id).ne.0) then
    call map_message(seve%i,rname,'Replacing Field numbers by PHASE offsets')
    ixoff = hinuv%gil%column_pointer(code_uvt_id)
    iyoff = ixoff+1
    hinuv%gil%column_pointer(code_uvt_loff) = ixoff
    hinuv%gil%column_pointer(code_uvt_moff) = iyoff
    hinuv%gil%column_size(code_uvt_loff) = 1
    hinuv%gil%column_size(code_uvt_moff) = 1
    hinuv%gil%column_pointer(code_uvt_id) = 0
    hinuv%gil%column_size(code_uvt_id) = 0
    icode = 0
  else if (hinuv%gil%column_size(code_uvt_loff).ne.0) then
    ixoff =     hinuv%gil%column_pointer(code_uvt_loff)
    iyoff =     hinuv%gil%column_pointer(code_uvt_moff)
    icode = 1
    call map_message(seve%e,rname,'UV Table is already of type PHASE') 
    error = .true.
  else if (hinuv%gil%column_size(code_uvt_xoff).ne.0) then
    ixoff =     hinuv%gil%column_pointer(code_uvt_xoff)
    iyoff =     hinuv%gil%column_pointer(code_uvt_yoff)
    icode = -1
    call map_message(seve%e,rname,'UV Table is already of type POINT') 
    error = .true.
  else
    call map_message(seve%e,rname,'UV table is not a mosaic')
    error = .true.
  endif
  if (error) return
  !
  if (icode.eq.0) then
    do iv=1,hinuv%gil%nvisi
      isou = duv(ixoff,iv)
      if (isou.gt.0 .and. isou.le.nf) then
        duv(ixoff,iv) = offs(1,isou)
        duv(iyoff,iv) = offs(2,isou)
      else
        write(mess,'(A,I0,A,I0)') 'Invalid source ID ',isou,' for visibility ',iv
        call map_message(seve%w,rname,mess)
        error = .true.
        return
      endif
    enddo
  else
    do iv=1,hinuv%gil%nvisi
      isou = duv(ixoff,iv)
      rsou = isou
      if (rsou.ne.duv(ixoff,iv)) then  
        mess = 'Non integer field number'
        error = .true.
      else if (isou.lt.1 .or. isou.gt.nf) then
        mess = 'Field number out of range'
        error = .true.
      endif
      if (error) then
        call map_message(seve%e,rname,mess)
        return
      endif
      duv(ixoff,iv) = offs(1,isou)
      duv(iyoff,iv) = offs(2,isou)
    enddo
  endif
end subroutine sub_memory_fields

subroutine sub_mosaic_fields (hinuv, hmos, duv, nf, offs, error)
  use gkernel_types
  use image_def
  use clean_def
  use iso_c_binding
  use gkernel_interfaces
  use gbl_message
  use imager_interfaces, only : map_message
  !---------------------------------------------------------------------
  ! @ private
  !
  ! IMAGER
  !   Support for READ FITS 
  !   Define the HMOSAIC structure from a UVFITS file
  !---------------------------------------------------------------------
  type(gildas), intent(inout) :: hinuv     ! Input / Output file UV header
  real, intent(inout) :: duv(:,:)          ! UV data
  type(mosaic_par), intent(out) :: hmos(:) ! Mosaic Header
  integer, intent(in) :: nf                ! Number of fields
  real(kind=8), intent(in) :: offs(2,nf)   ! Offset values
  logical, intent(out) :: error            ! Error flag
  !
  ! Constants
  character(len=*), parameter :: rname='UV_FIELDS'
  real(8), parameter :: pi=acos(1.d0)
  real, parameter :: tole=1e-2*pi/180/3600 ! 10 mas, more than enough 
  !
  ! Local variables
  character(len=32)  :: name
  character(len=80)  :: mess
  real(8), pointer :: d1ptr(:)
  type(c_ptr) :: cptr
  logical :: found
  type(sic_descriptor_t) :: desc
  type(projection_t) :: uv_proj
  integer :: iv, ixoff, iyoff, idoff, isou, icode, jsou, ier, nnn
  real :: rsou, xoff, yoff
  real(8), allocatable :: raval(:),deval(:),raoff(:),deoff(:)
  real(8) :: dm,dp,ra,dec
  !
  error = .false.
  if (hinuv%loca%size.eq.0) then
    call map_message(seve%e,rname,'No UV data')
    error = .true.
    return
  endif  
  !
  ! Update the proper columns
  allocate(raval(nf),deval(nf),raoff(nf),deoff(nf),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,rname,'Memory allocation error')
    error = .true.
    return
  endif
  !
  if (hinuv%gil%column_size(code_uvt_id).ne.0) then
    call map_message(seve%i,rname,'Using Field numbers')
    idoff = hinuv%gil%column_pointer(code_uvt_id)
    icode = 0
    name = 'ATMPF%AIPS_SU%COL%RAEPO'
    nnn = len_trim(name)
    if (.not.(sic_varexist(name))) then
      call map_message(seve%e,rname,'Missing AIPS_SU column variable')
      error = .true.
      return
    endif
    !
    ! Convert the Descriptors to Fortran arrays
    call sic_descriptor(name(1:nnn),desc,found)
    call adtoad(desc%addr,cptr,1)
    call c_f_pointer(cptr,d1ptr,[nf])
    raval(:) = d1ptr * pi/180.d0
    dm = minval(raval)
    dp = maxval(raval)
    if (dm.lt.0) then
      dm = dm+2*pi
      dp = dp+2*pi
    endif
    ra = (dm+dp)/2
    !
    name = 'ATMPF%AIPS_SU%COL%DECEPO'
    nnn = len_trim(name)
    call sic_descriptor(name(1:nnn),desc,found)
    call adtoad(desc%addr,cptr,1)
    call c_f_pointer(cptr,d1ptr,[nf])
    deval(:) = d1ptr * pi/180d0
    dm = minval(deval)
    dp = maxval(deval)
    dec = (dm+dp)/2
    !
    hinuv%gil%a0 = ra
    hinuv%gil%d0 = dec
    !
    ! Set the Projection
    call gwcs_projec(hinuv%gil%a0,hinuv%gil%d0,hinuv%gil%pang,hinuv%gil%ptyp,uv_proj,error)
    !
    ! Convert to Offsets
    call abs_to_rel(uv_proj,raval,deval,raoff,deoff,nf)
    !
    ! Out into HMOS values
    do isou=1,nf
      hmos(isou)%apoint = [raval(isou),deval(isou)]
      hmos(isou)%aphase = [raval(isou),deval(isou)]
      hmos(isou)%opoint = [raoff(isou),deoff(isou)]
      hmos(isou)%ophase = [raoff(isou),deoff(isou)]
    enddo
  else if (hinuv%gil%column_size(code_uvt_loff).ne.0) then
    ixoff =     hinuv%gil%column_pointer(code_uvt_loff)
    iyoff =     hinuv%gil%column_pointer(code_uvt_moff)
    icode = 1
    call map_message(seve%i,rname,'UV Table is of type PHASE') 
  else if (hinuv%gil%column_size(code_uvt_xoff).ne.0) then
    ixoff =     hinuv%gil%column_pointer(code_uvt_xoff)
    iyoff =     hinuv%gil%column_pointer(code_uvt_yoff)
    icode = -1
    call map_message(seve%e,rname,'UV Table is of type POINT') 
  else
    call map_message(seve%e,rname,'UV table is not a mosaic')
    error = .true.
  endif
  if (error) return
  !
  if (icode.eq.0) then
    !
    ! Check field ID range
    do iv=1,hinuv%gil%nvisi
      isou = duv(idoff,iv)
      rsou = isou
      if (rsou.ne.duv(idoff,iv)) then  
        write(mess,'(A,I0)') 'Non integer field number at visi # ',iv
        error = .true.
      else if (isou.lt.1 .or. isou.gt.nf) then
        write(mess,'(A,I0)') 'Field number out of range at visi # ',iv
        error = .true.
      endif
      if (error) then
        call map_message(seve%e,rname,mess)
        return
      endif
    enddo
    call map_message(seve%i,rname,'Using Sault et al method for Mosaics')
  else
    !
    ! This in principle is done in "load_fields" ???
    call map_message(seve%i,rname,'Use command MOSAIC later to use the Sault method',3)
    return
    !
    ! Replace Offsets by Field ID
    do iv=1,hinuv%gil%nvisi
      xoff = duv(ixoff,iv)
      yoff = duv(iyoff,iv)
      !
      jsou = 0
      do isou=1,nf
        if (abs(xoff-offs(1,isou)).gt.tole) cycle
        if (abs(yoff-offs(2,isou)).gt.tole) cycle
        jsou = isou
        exit
      enddo
      !
      if (jsou.eq.0) then
        write(mess,'(A,I0)') 'Field number out of range at visi # ',iv
        error = .true.
        call map_message(seve%e,rname,mess)
        return
      endif
      duv(ixoff,iv) = jsou
      duv(iyoff,iv) = 0      ! Unset
    enddo
    !
    ! Set the Projection
    call gwcs_projec(hinuv%gil%a0,hinuv%gil%d0,hinuv%gil%pang,hinuv%gil%ptyp,uv_proj,error)
    do isou=1,nf
      raoff(isou) = offs(1,isou)
      deoff(isou) = offs(2,isou)
    enddo
    call rel_to_abs(uv_proj,raoff,deoff,raval,deval,nf)
    !
    if (icode.eq.1) then 
      ! PHASE Table, common Phase center for all fields
      do isou=1,nf
        hmos(isou)%apoint = [raval(isou),deval(isou)]
        hmos(isou)%aphase = [hinuv%gil%a0,hinuv%gil%d0]
        hmos(isou)%opoint = [raoff(isou),deoff(isou)]
        hmos(isou)%ophase = [0.,0.]
      enddo
    else
      ! POINTING Table, identical Pointing & Phase center for each field
      do isou=1,nf
        hmos(isou)%apoint = [raval(isou),deval(isou)]
        hmos(isou)%aphase = [raval(isou),deval(isou)]
        hmos(isou)%opoint = [raoff(isou),deoff(isou)]
        hmos(isou)%ophase = [raoff(isou),deoff(isou)]
      enddo
    endif
  endif
end subroutine sub_mosaic_fields
