module clean_arrays
  use clean_def
  use image_def
  !!
  !!  Define all Static variables first, then allocatable next.
  !!
  type (gildas), save, target :: hview  !! VIEWER DATA header
  type (uvmap_par), save :: themap      !! UV_MAP parameters
  type (clean_par), target :: the_method    !! CLEAN parameters
  type (gildas), save, target :: hbeam  !! Synthesized beam Header
  type (gildas), save, target :: hdirty !! Dirty image Header
  type (gildas), save, target :: hclean !! Clean image Header
  type (gildas), save, target :: hresid !! Residual image Header
  type (gildas), save, target :: hatten !! Field weights header
  type (gildas), save, target :: hprim  !! Primary beams Header
  type (gildas), save :: hfields        !! Fields Header
  type (gildas), save, target :: hcct   !! Clean component table Header
  type (gildas), save, target :: hmask  !! Mask Header
  type (gildas), save, target :: hsky   !! Sky brightness image Header
  type (gildas), save, target :: hcont  !! Continuum image (Clean or Sky)
  type (gildas), save :: hpv            !! Position-Velocity data
  type (gildas), save :: hrobust        !! Beams as a function of Robust parameter
  !
  !! Self-calibration headers
  type (gildas), save :: hraw           !! Raw (non self-calibrated) UV header
  type (gildas), save :: hcalib         !! Self-calibrated UV header 
  type (gildas), save :: hself          !! Self-calibrated UV header for Display
  type (gildas), save :: hagain         !! Antenna-based gain Header
  type (gildas), save :: hbgain         !! Baseline-based gain Header
  !
  !! Mosaics & Short spacings
  type (gildas), save :: hsingle        !! Single Dish Cube or Table
  type (gildas), save :: hshort         !! Short spacing Cube 
  type (gildas), save :: hpoint
  !
  !! UV data like headers
  type (gildas), save, target :: huvi   !! Initial UV data Header
  type (gildas), save, target :: huv    !! Current UV data Header 
  type (gildas), save, target :: huvt   !! Transposed UV data Header
  type (gildas), save, target :: huvc   !! Continuum UV data Header
  type (gildas), save :: huvm           !! Model UV data Header
  type (gildas), save :: huvf           !! Fit residual UV data Header
  !
  type (clean_par), target :: c_method, cuse_method
  !! For the Multi-Primary beams Mosaic deconvolution
  !! C_ stands for "Compact Array"
  type (gildas), save :: c_hbeam, c_hdirty, c_hresid, c_hclean
  type (gildas), save :: c_hprim
  !
  !! UV_MAP optimization arrays: Weigts and V values
  real, allocatable, save :: g_weight(:)  !! UV weights
  real, allocatable, save :: g_v(:)       !! V values
  logical, save :: do_weig                !! Optimization of weight computing
  !
  !! Viewer data 
  real, allocatable, target, save :: dview(:,:,:)     !! VIEWER DATA area 
  !! Imaging and deconvolution arrays
  real, allocatable, target, save :: dbeam(:,:,:,:)   !! 4-D         (lm,m,v,p)
  real, allocatable, target, save :: ddirty(:,:,:)    !! Dirty 3-D   (l,m,v)
  real, allocatable, target, save :: dclean(:,:,:)    !! Clean 3-D   (l,m,v)
  real, allocatable, target, save :: dresid(:,:,:)    !! Residual 3-D  (l,m,v)
  real, allocatable, target, save :: dprim(:,:,:,:)   !! Primary beam 4-D    (p,x,y,v)
  real, allocatable, target, save :: datten(:,:,:)    !! Mosaic weights 3-D  (lm,m,v)
  real, allocatable, target, save :: dsky(:,:,:)      !! Primary beam corrected 3-D  (l,m,v)
  real, allocatable, target, save :: dfields(:,:,:,:) !! Mosaic fields 4-D   (l,m,v,p)
  real, allocatable, target, save :: dcont(:,:)       !! Continuum image 2-D (l,m)  (and SKY ?)
  real, allocatable, target, save :: dsmooth(:,:,:,:) !! Spectrally Smoothed array UNDER TEST ONLY
  real, allocatable, target, save :: drobust(:,:,:)   !! Beams as a function of Robust parameter
  !
  !! Deconvolution arrays
  real, allocatable, target, save :: dmask(:,:,:)     !! Input 3-D 0, #0 mask  (lm,m,v)
  integer, allocatable, target, save :: d_list(:)     !! List of selected pixels
  logical, allocatable, target, save :: d_mask(:,:)   !! Logical mask for pixel selection
  real, allocatable, target, save :: dcct(:,:,:)      !! Clean component values (cct_size,v,n)
  type(cct_lst), allocatable, save :: dcclist(:)      !! Clean component list (v) all channels
  !
  !! Channel flag list
  integer, allocatable, target, save :: dchanflag(:)  !! Per channel flag status
  !
  !! UV data like arrays
  real, pointer :: duv(:,:)                        !! Current UV data
  real, allocatable, target, save :: duvi(:,:)     !! Original UV data
  real, pointer :: duvr(:,:)                       !! "Resampled" UV data
  real, pointer :: duvs(:,:)                       !! "Sorted" UV data
  real, allocatable, target, save :: duvc(:,:)     !! Sorted or Unsorted Continuum UV data
  real, allocatable, target, save :: duvt(:,:)     !! Time-Baseline sorted UV data, for UV_FLAG
  real, allocatable, target, save :: duvm(:,:)     !! Model UV data
  real, allocatable, target, save :: duvf(:,:)     !! Fit residual UV data  
  real, allocatable, target, save :: duvbg(:,:)    !! Baseline Gains (UV data like)
  real, allocatable, target, save :: duvraw(:,:)   !! Raw UV data for Self calibration
  real, allocatable, target, save :: duvcalib(:,:) !! Self calibrated UV data
  real, pointer, save :: duvself(:,:)              !! Self calibrated UV data, continuum version
  real, allocatable, target, save :: dagain(:,:)   !! Antenna based gain
  !
  !! Short spacing data array
  real, allocatable, target, save :: dsingle(:,:,:)   !! Short spacing data (Table or Cube)
  real, allocatable, target, save :: dshort(:,:,:)    !! Short spacing 3-D cube (x,y,v)
  !! The above one was by hshort%r3d only before 26-Apr-2023
  !
  !! For the Multi-Primary beams Mosaic deconvolution
  !! C_ stands for "Compact Array"
  real, allocatable, target, save :: c_dbeam(:,:,:)
  real, allocatable, target, save :: c_ddirty(:,:,:)
  real, allocatable, target, save :: c_dclean(:,:,:)
  real, allocatable, target, save :: c_dresid(:,:,:)
  real, allocatable, target, save :: c_dprim(:,:,:)
  logical, allocatable, target, save :: c_mask(:,:)
  integer, allocatable, target, save :: c_list(:)
  real, allocatable, target, save :: c_atten(:,:,:)  !! 3-D
  !
  integer :: niter_listsize=0
  integer, allocatable :: niter_list(:)  !! Per channel CLEAN Niter - little used
  integer :: ares_listsize=0
  real, allocatable :: ares_list(:)  !! Per channel CLEAN Ares - little used
  !
  integer :: selected_fieldsize=0
  integer, allocatable :: selected_fields(:)
  !! List of selected fields in UV_MAP (used for UV_RESTORE and also for SHOW FIELDS)
end module clean_arrays
!!
module moment_arrays
  use image_def
  !! Moment Images
  type(gildas), save :: hpeak, hmean, hvelo, hwidth
  real, allocatable, target, save :: dmean(:,:), dpeak(:,:), dvelo(:,:), dwidth(:,:)
  !! Flux spectra
  integer, target, save :: flux_nf=0
  integer, target, save :: flux_nc=0
  real(4), target, save, allocatable :: flux_velo(:)
  real(8), target, save, allocatable :: flux_freq(:)
  real(4), target, save, allocatable :: flux_values(:,:)
  real(4), target, save, allocatable :: flux_errors(:,:)
end module moment_arrays
!!
module uvsplit_mod
  use image_def
  !! Module for the UV_SPLIT command
  integer, parameter :: opt_chan=1
  integer, parameter :: opt_file=2
  integer, parameter :: opt_freq=3
  integer, parameter :: opt_range=4
  integer, parameter :: opt_velo=5
  integer, parameter :: opt_width=6
  integer, parameter :: opt_reset=7
  integer, parameter :: opt_zero=8
  integer, parameter :: code_baseline=-1
  integer, parameter :: code_split=0
  integer, parameter :: code_filter=1
  !!
  character(len=filename_length) :: in_file
  character(len=filename_length) :: ou_file
  !!
  type(gildas), target, save :: hiuv
  type(gildas), target, save :: houv
  type(gildas), target, save :: hfuv
  integer :: code_routine
  integer :: nblock        !! Blocking factor
  integer :: ncompr        !! Channel compression factor
  logical :: use_file      !! /FILE option
  real :: edge_drop=0.05   !! Fraction of bandwidth dropped at edges for continuum
  !!
end module uvsplit_mod
!!
module clean_support
  use gkernel_types
  !! CLEAN\SUPPORT polygon, mapped on SUPPORT%
  integer, parameter :: support_mask = 1 
  integer, parameter :: support_poly = -1
  integer, parameter :: support_none = 0
  character(len=varname_length), parameter :: supportvar='SUPPORT'
  !!
  type(polygon_t), save :: supportpol
  integer, save :: support_type         !! Current type of support
  type (gildas), save :: hmask_temp     !! Mask Template
  character(len=64) :: mask_template    !! Name of Mask Template Variable
end module clean_support
!!
module clean_types
  use gkernel_types
  !!
  !! Definitions of known Buffers names and status
  !!
  integer, parameter :: mtype=34
  integer, parameter :: ptype=11
  integer :: ftype(1:mtype) = (/0,0,1,1,1,1,1,1,0,0,1,0,0,0,-1,1,1,-1,-1,-1,-1,-1,-1,-1,-1,-1,1,-1,-1,-1, 0,-1,-1,1/)
  !!  0 -- No or Special buffer (e.g. POLYGON)
  !!  1 -- Normal FITS or Gildas buffers
  !! -1 -- No buffer - special display action
  integer :: vwrite(1:22) = (/2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,20,23,26,27,31,34/)
  !! List of Writeable entities
  integer :: vshow(1:32) = (/1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,32,34/)
  !! List of Entities that can be Shown or Viewed
  integer :: vread(1:20) = (/2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,23,26,27,31/)
  !! List of Readable quantities
  character(len=12) :: vtype(1:mtype) = (/ &
  &  'COVERAGE    ',  &     !! 1
  &  'UV_DATA     ',  &     !! 2
  &  'BEAM        ',  &     !! 3
  &  'DIRTY       ',  &     !! 4
  &  'CLEAN       ',  &     !! 5
  &  'PRIMARY     ',  &     !! 6
  &  'RESIDUAL    ',  &     !! 7
  &  'MASK        ',  &     !! 8
  &  'SUPPORT     ',  &     !! 9
  &  'CCT         ',  &     !! 10
  &  'SKY         ',  &     !! 11      !! Up to PTYPE, allowed Images (FITS support allowed)
  &  'MODEL       ',  &     !! 12
  &  'FIELDS      ',  &     !! 13
  &  'CGAINS      ',  &     !! 14
  &  'SELFCAL     ',  &     !! 15
  &  'SHORT       ',  &     !! 16
  &  'SINGLEDISH  ',  &     !! 17
  &  'NOISE       ',  &     !! 18
  &  'SNR         ',  &     !! NOISE and SNR are special, just for SHOW NOISE  
  &  'MOMENTS     ',  &     !! special, just for SHOW MOMENTS      20
  &  'COMPOSITE   ',  &     !! special, Moments + Flux on one plot
  &  'KEPLER      ',  &     !! special, for SHOW KEPLER
  &  'FLUX        ',  &     !! special, for SHOW FLUX and WRITE FLUX and READ FLUX    
  &  'SOURCES     ',  &     !! special, for SHOW SOURCES
  &  'SPECTRA     ',  &     !! special, for SHOW SPECTRA           25
  &  'UV_FIT      ',  &
  &  'CONTINUUM   ',  &     !! Continuum Image (if any)            27
  &  'SED         ',  &     !! Special, for SHOW SED
  &  'PV          ',  &     !! special, for SHOW PV
  &  'SIMULATION  ',  &     !! Special, for SHOW SIMU              30
  &  'POLYGON     ',  &     !! To avoid issues with WRITE POLY     31
  &  'INTENSITY   ',  &     !! Like NOISE but Logarithm
  &  'STATISTIC   ',  &     !! Like NOISE but Logarithm, and fit the RMS noise
  &  'DATA        '   /)    !! DATA, must be the last one 
  !!
  character(len=12) :: etype(1:mtype) = (/ &
  & '            ',  &
  & '.uvt        ',  &
  & '.beam       ',  &
  & '.lmv        ',  &
  & '.lmv-clean  ',  &
  & '.lobe       ',  &
  & '.lmv-res    ',  &
  & '.msk        ',  &
  & '.pol        ',  &
  & '.cct        ',  &
  & '.lmv-sky    ',  &
  & '.uvt        ',  &
  & '.fields     ',  &
  & '.uvt        ',  &      
  & '.tab        ',  &
  & '.lmv-sky    ',  &
  & '.tab        ',  &
  & '            ',  &
  & '            ',  &
  & '            ',  &
  & '            ',  &
  & '            ',  &
  & '.tab        ',  &
  & '            ',  &
  & '            ',  &
  & '.uvfit      ',  &
  & '.lmv-clean  ',  &
  & '.txt        ',  &
  & '            ',  &
  & '            ',  &
  & '.pol        ',  &
  & '            ',  &
  & '            ',  &
  & '            ' /)    !! DATA, no genering extension       
  !!
  integer, parameter :: mbuffer=27+3
  character(len=12) :: cbuffer(1:mbuffer) = (/ &
  &  'AGAINS      ',  &
  &  'BEAM        ',  &
  &  'CCT         ',  &
  &  'CGAINS      ',  &
  &  'CLEAN       ',  &
  &  'CLIPPED     ',  &  
  &  'CONTINUUM   ',  &
  &  'DATA        ',  &
  &  'DIRTY       ',  &
  &  'EXTRACTED   ',  &
  &  'MASK        ',  &
  &  'M_AREA      ',  &
  &  'M_PEAK      ',  &
  &  'M_VELO      ',  &
  &  'M_WIDTH     ',  &
  &  'MODEL       ',  &
  &  'PRIMARY     ',  &
  &  'RESIDUAL    ',  &
  &  'WEIGHT      ',  &
  &  'FIELDS      ',  &
  &  'SHORT       ',  &
  &  'SINGLE      ',  &
  &  'SKY         ',  &
  &  'SLICE       ',  &
  &  'SPECTRUM    ',  &
  &  'UV          ',  &
  &  'UV_FIT      ',  &
  &  'UV_RESIDUAL ',  &
  &  'UVCONT      ',  &
  &  'UVRADIAL    '  /)  
  character(len=36) :: sbuffer(1:mbuffer) = (/ &
!     123456789 12345678 123456789 123456
  &  'Self-Cal Antenna Gains             ',  &
  &  'Synthesized Dirty Beam             ',  &
  &  'Clean Component Table              ',  &
  &  'Self-Calibration Complex Gains     ',  &
  &  'Clean-ed data cube                 ',  &
  &  'Clipped Spectrum from UV_PREVIEW   ',  &  
  &  'Continuum image from MAP_CONTINUUM ',  &
  &  'Generic VIEWER data area           ',  &        
  &  'Dirty Image                        ',  &
  &  'Image extracted by EXTRACT         ',  &
  &  'Current Mask                       ',  &
  &  'Integrated Area from MOMENTS       ',  &
  &  'Peak flux from MOMENTS             ',  &
  &  'Mean Velocity from MOMENTS         ',  &
  &  'Line Width from MOMENTS            ',  &
  &  'UV Model visibilities              ',  &
  &  'Primary Beam                       ',  &
  &  'Residual Image                     ',  &  
  &  'Mosaic Weight Image                ',  &
  &  'Mosaic Fields                      ',  &
  &  'Short Spacing Image                ',  &
  &  'Single-Dish Data                   ',  &
  &  'Sky Brightness (PB corrected)      ',  &
  &  'Image extracted by SLICE           ',  &
  &  'Initial Spectrum from UV_PREVIEW   ',  &
  &  'UV data                            ',  &
  &  'UV Fit results binary table        ',  &
  &  'Residual visibilities              ',  &
  &  'Continuum UV data                  ',  &
  &  'Azimutally averaged UV data        '  /)  
  !!
  character(len=12) display_type
  !!
  integer, parameter :: code_save_none=0
  integer, parameter :: code_save_uv=2
  integer, parameter :: code_save_beam=3
  integer, parameter :: code_save_dirty=4
  integer, parameter :: code_save_clean=5
  integer, parameter :: code_save_primary=6
  integer, parameter :: code_save_resid=7
  integer, parameter :: code_save_mask=8
  integer, parameter :: code_save_support=9
  integer, parameter :: code_save_cct=10
  integer, parameter :: code_save_sky=11
  integer, parameter :: code_save_model=12
  integer, parameter :: code_save_fields=13
  integer, parameter :: code_save_gains=14
  integer, parameter :: code_save_short=16
  integer, parameter :: code_save_single=17
  integer, parameter :: code_save_cont=27 
  !!
  !! Output data format
  integer, parameter :: write_gildas=0
  integer, parameter :: write_fits=1
  integer, parameter :: write_fits_velo=2
  integer, parameter :: write_fits_freq=3
  integer, parameter :: write_csv=-1
  integer, save :: write_default = write_gildas
  integer, save :: write_format  = write_gildas
  !!
  !! Input read option
  logical, save :: read_trail=.true.
  !!
  !! Read / Write optimization
  type readop_t
    type(mfile_t) :: modif   !! From kernel / gsys_types
    integer :: lastnc(2) = 0
    integer :: change = 0
  end type
  !!
  logical :: save_data(mtype)
  type (readop_t), save :: optimize(mtype)
  !!
  integer :: rw_optimize
end module clean_types
!!
module clean_default
  use gildas_def
  use clean_def
  !!
  !! Default values for IMAGER behaviours
  real d_min,d_max       !! min max of dirty image, used by MRC plots only
  real beam_size(3)      !! Major, Minor (in ") and Angle (in °)
  !!
  type(clean_par), save, target :: user_method !! CLEAN parameters
  type (uvmap_par), save :: default_map, old_map, save_map
  character(len=16), save :: current_uvdata = 'DATA_UV'
  character(len=32), save :: gtv_root = '<GREG'
  logical, save :: uv_model_updated
  logical, save :: uv_resid_updated
  logical, save :: uv_data_updated
  !!
  real, save :: field_size = 4.0           !! Size of 1 field in Mosaics (FWHP) could be in Method%
  real, save :: guard_size = 2.5           !! Guard size around Mosaics (FWHP)
  integer, save :: mosaic_enlarge = 1      !! Enlarge Beam Mode
  integer, save :: multi_print = 0         !! Step between Printing in MULTI mode
  character(len=8) :: mosaic_mode='NONE'   !! Mosaic Mode (NONE, GUETH or SAULT)
  real, save :: mosaic_tolerance=1.0       !! In arcsec, Mosaic tolerance
  !
  logical, save :: call_debug = .false.    !! Call Tree Debugging
  !
  integer, save :: map_version = 0         !! Imaging tool version (deprecated)
  !! Non zero means old non-optimized code.
  !!
  logical, save :: cct_dynamic = .true.    !! Parameter for dynamic Clean Component List
  !
  ! Parameters for determining image size
  integer, save :: map_power = 2           !! Max exponent of powers of 3 and 5
  real, save :: map_rounding = 0.05        !! Tolerance for rounding to floor
  !
  logical, save :: ImagerError = .false.  !! Error status from scripts
  character(len=16), save :: last_shown=' '       !! CLEAN, DATA, DIRTY or SKY
  character(len=16), save :: last_resid=' '       !! CCT or UV_FIT
  character(len=filename_length), save :: last_temporary=' '  !! Last TEMPORARY data file
  character(len=16), save :: clean_stop(2) !! CLEAN_STOP criterium
  logical, save :: view_do_loop            !! Interactive mode for VIEW
  logical, save :: show_side               !! Side view for SHOW 
  logical, save :: do_cont                 !! Save the Continuum image ?
  integer, save :: expand_cont             !! Continuum expansion factor
  integer, save :: iangle_unit=-3          !! Angle Unit code, default Second
  !!
  integer, save :: sys_ramsize             !! RAM size in MBytes
  !!
  real(8), save :: area_size(2)            !! SIZE[2]
  real(8), save :: area_center(2)          !! CENTER[2]
  !!
  character(len=64) :: defmap_center       !! MAP_CENTER string
  !!      
  real(4), save :: recmap_field(2)         !! Recommended map field (radians)
  integer, save :: recmap_size(2)          !! recommended map size  (pixels)
  real(4), save :: recmap_xycell(2)        !! recommended map cell  (radians)
  real(4), save :: recmap_uvcell           !! recommended UV cell (m)
  !!
  !! Active values
  type(channel_par), save  :: saved_chan
  type(channel_par), save  :: uvmap_chan
  !!
  !! SHOW NOISE issues
  integer, save :: noise_type
  logical, save :: noise_fit
  logical, save :: noise_one
  logical, save :: noise_plot
  !! UV_REWEIGHT output
  real, save :: uv_weight_scale             !! Weight scale factor found by UV_REWEIGHT
end module clean_default
!!
module clean_beams
  !! Dirty beam control parameters
  real, save, target :: beam_tolerance=0.01 !! 1 % loss is not much an issue usually
  real, allocatable, save :: beam_ranges(:,:) !! Edges of beam ranges
  integer, save :: nbeam_ranges           !! Number of Beam ranges
  integer, save :: beam_gap               !! Size of allowed "gaps" in channels  
  logical, save :: beam_defined=.false.   !! 
  integer, save, target :: beams_number=0 !! Number of Frequency dependent beams
  integer, save, target :: beams_field=0  !! Number of Fields
  real, save, target :: beam_fitted(4)    !! Beam size in "convenient" units (user-readable)
  !! Major & Minor axis (arcsec), Position Angle (deg) and JvM factor
  real, allocatable, save, target :: beams_param(:,:,:) 
  !! Array (4,beams_number,beams_field), in same units: Major, Minor, Angle, JvM
end module clean_beams
!!
module uvfit_data
  !! Support module for command UV_FIT
  use image_def
  integer, parameter :: mpar=70         !! Max number of parameters
  integer, parameter :: mf=10           !! Max number of functions
  integer, parameter :: mpin=7          !! Max number of parameters per function
  real(kind=8) :: pars(mpar)            !! Parameter values
  real(kind=8) :: sw
  real, allocatable :: uvriw(:,:)       !! Work space for visibilities
  integer :: nf                         !! number of functions
  integer :: ifunc(mf)                  !! Function codes
  integer :: npfunc(mf)                 !! number of parameters per function
  logical :: savef(mf)                  !! Subtract function
  integer :: npuvfit                    !! number of points
  integer :: npar                       !! Number of parameters
  integer :: nstart(mpar)               !! Number of starts per parameter
  integer :: ncall                      !! Number of function calls
  !!
  real(kind=4) :: uvf_results(mpin,mf)  !! Results for each function
  real(kind=4) :: uvf_errors(mpin,mf)   !! Errors 
  !!
  type (gildas) :: huvfit               !! Table of fitted parameters
  real, allocatable :: duvfit(:,:)      !! UV_FIT results
end module uvfit_data
!!
module preview_mod
  use image_def
  !! UV_PREVIEW control parameters and associated data structures
  type(gildas), save :: htab
  type(gildas), save :: hclip 
  integer, allocatable, save :: final_lines(:)
  real(kind=8), allocatable, save :: edge_freq(:,:)   !! Edge Frequencies
  integer, allocatable, save :: edge_chan(:,:)        !! Edge channels
  integer, save :: number_of_channels = 0
  integer, save :: browse_last = -1
  !!
  !! Control parameters
  integer, save :: utaper=3
  real,    save :: uclip=3.0
  integer, save :: uhist=0
  integer, save :: usmooth=3
  !!
end module preview_mod
!!
module omp_control
  !! OPEN-MP control parameters
  logical, save :: omp_compiled = .false.
  integer, save :: omp_grid = 0
  integer, save :: omp_single_beam = 0
  logical, save :: omp_debug = .false.
  integer, save :: omp_outer_thread = 1   !! Number of Threads at outer stage
  integer, save :: omp_inner_thread = 1   !! Number of Threads at inner stage
  integer, save :: omp_outer_threadgoal = 4 !!
end module omp_control
!!
module grid_control
  !! Gridding method control parameters (speed issues)
  integer, save :: grid_bigvisi=10000000  !! Threshold for Gridding weights
  integer, save :: grid_subcell=8         !! Sub-cell per Uniform cell 
  integer, save :: grid_weighting=0       !! Weighting mode
end module grid_control
!!
module gclean_mod
  !! GCLEAN  Smoothed Dirty beams and Scale factors
  integer, save :: n_scales               !! ms*(ms+1)/2
  integer, allocatable :: beamindex(:,:)
  real, allocatable ::  gbeam(:,:,:,:)    !! (nx,ny,n_scales,np)
  real, allocatable :: scales(:,:)        !! (n_scales,np) Scales factor for each beam
  real, allocatable :: scale(:)           !! (n_scales)    Mean Scale factor 
  real, allocatable :: loss(:)            !! (n_scales)    Noise degradation factor...
end module gclean_mod
!!
