/*
Copyright (c) 2008 Regents of The University of Michigan.
All Rights Reserved.

Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appears in all copies and
that both that copyright notice and this permission notice appear
in supporting documentation, and that the name of The University
of Michigan not be used in advertising or publicity pertaining to
distribution of the software without specific, written prior
permission. This software is supplied as is without expressed or
implied warranties of any kind.

Research Systems Unix Group
The University of Michigan
c/o Wesley Craig
4251 Plymouth Road B1F2, #2600
Ann Arbor, MI 48105-2785

http://rsug.itd.umich.edu/software/ihook
ihook@umich.edu
*/

#import "LHController.h"
#import "HookOperation.h"
#import "LHWindow.h"
#import "LHString.h"
#import "NSColor(ColorForName).h"

#include <Carbon/Carbon.h>
#include <CoreFoundation/CoreFoundation.h>
#include <SystemConfiguration/SystemConfiguration.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <assert.h>
#include <ctype.h>
#include <fcntl.h>
#include <libgen.h>
#include <stdlib.h>
#include <unistd.h>
#include "argcargv.h"

#define DEFAULT_WIDTH	450.0
#define DEFAULT_HEIGHT	150.0

@implementation LHController

pid_t			childpgid;
int			input_queue_count = 0;

- ( id )init
{
    self = [ super init ];
    
    ihookHadDirective = NO;
    ihookElapsedTime = 0;
    ihookTimer = nil;
    ihookInputQueue = [[ NSMutableArray alloc ] init ];
    ihookOperationQueue = [[ NSOperationQueue alloc ] init ];
    logfd = -1;
    
    self.debug = NO;
    self.showTimer = YES;
    self.scriptPath = nil;
    self.isExecuting = [ NSNumber numberWithBool: NO ];
    self.exitStatus = [ NSNumber numberWithInt: 0 ];

    [ NSApp setDelegate: self ];
    return( self );
}

#define HookWindowTitleBarMask (NSTitledWindowMask | NSMiniaturizableWindowMask)
- ( void )awakeFromNib
{
    NSArray		*switchArgs;
    NSString		*arg;
    NSRect		contentRect = [ mainView bounds ];
    unsigned int	mask = HookWindowTitleBarMask;
    
    /* check for console user */
    [ self setConsoleUser ];
    
    if ( self.consoleUser ) {
	if ( [[ NSUserDefaults standardUserDefaults ]
		    boolForKey: @"NoWindowTitleBar" ] ) {
	    mask = NSBorderlessWindowMask;
	}
    }
    
    for ( arg in [[ NSProcessInfo processInfo ] arguments ] ) {
	if ( [ arg beginsWithString: @"--script=" ] ) {
            switchArgs = [ arg componentsSeparatedByString: @"=" ];

            if ( [ switchArgs count ] == 1 ) {
                NSLog( @"Missing value for --script, using default path." );
            } else if ( [ switchArgs count ] == 2 ) {
                self.scriptPath = [[ switchArgs lastObject ]
					stringByExpandingTildeInPath ];
            }
        } else if ( [ arg isEqualToString: @"--no-titlebar" ] ) {
            mask = NSBorderlessWindowMask;
        }
    }
    
    if ( self.scriptPath == nil ) {
        if ( self.consoleUser == nil ) {
	    self.scriptPath = HOOK_DEFAULT_PATH;
    
	    if ( access( [ self.scriptPath UTF8String ], F_OK | X_OK ) < 0 ) {
		NSLog( @"%@: %s", HOOK_DEFAULT_PATH, strerror( errno ));
                exit( 2 );
            }
        }
    }
    
    mainWindow = [[ LHWindow alloc ] initWithContentRect: contentRect
                                    styleMask: mask
                                    backing: NSBackingStoreBuffered
                                    defer: NO ];
    [ mainWindow setWindowZoomEnabled: YES ];
    
    /* 10.5 is much stricter about who can run over the loginwindow */
    if ( [ mainWindow respondsToSelector:
	    @selector( setCanBecomeVisibleWithoutLogin: ) ] ) {
	[ mainWindow setCanBecomeVisibleWithoutLogin: YES ];
    }
    [ mainWindow setContentView: mainView ];
    [ mainWindow setLevel: ( self.consoleUser
            ? NSNormalWindowLevel : ( NSScreenSaverWindowLevel - 1 ))];
    [ mainWindow setHasShadow: YES ];
    [ progBar retain ];
    [ progBar removeFromSuperview ];

    [ defaultButton retain ];
    [ defaultButton removeFromSuperview ];
    [ cancelButton setAction: @selector( cancel: ) ];
    [ cancelButton retain ];
    [ cancelButton removeFromSuperview ];
    [ stdoutField setStringValue: @"" ];
    [ titleField setStringValue: @"" ];
    
    [ stderrDrawer setParentWindow: mainWindow ];
    [ stderrDrawer setDelegate: self ];
    [ stderrDrawer setLeadingOffset: 15.0 ];
    
    /* hide input UI elements */
    self.inputEnabled = NO;
}

- ( BOOL )application: ( NSApplication * )theApplication
            openFile: ( NSString * )filename
{
    if ( self.scriptPath ) {
        NSLog( @"%@ already running. Not executing %@.",
		self.scriptPath, filename );
        return( NO );
    }
    
    self.scriptPath = [ filename stringByExpandingTildeInPath ];

    return( YES );
}

- ( void )applicationDidFinishLaunching: ( NSNotification * )aNotification
{
    NSRect		mw;
    
    if ( self.scriptPath == nil ) {
        int		rc;
        NSOpenPanel	*op = [ NSOpenPanel openPanel ];
        
        [ op setTitle: @"Choose a hook to run" ];
        [ op setPrompt: @"Run" ];
        [ op setCanChooseDirectories: NO ];
        [ op setAllowsMultipleSelection: NO ];
        
        rc = [ op runModalForDirectory: nil file: nil types: nil ];
        switch ( rc ) {
        case NSOKButton:
            self.scriptPath = [[ op filenames ] objectAtIndex: 0 ];
            break;
            
        default:
        case NSCancelButton:
            NSLog( @"No script selected, exiting..." );
            [ NSApp terminate: self ];
        }
    }
    
    [ self start ];
    
    mw = [ mainWindow frame ];
    [ mainWindow setFrameOrigin:
	    [ mainWindow centeredOriginForWindowSize: mw.size ]];

    [ mainWindow makeKeyAndOrderFront: nil ];
    [ mainWindow orderFrontRegardless ];
}

/*
 * this is only something like an accessor method pair.
 * we only want to check once for a console user and
 * save the result. this way seemed most idiomatic.
 */
- ( void )setConsoleUser
{
    CFStringRef		cfuser = NULL;
    
    ihookConsoleUser = nil;
    
    cfuser = SCDynamicStoreCopyConsoleUser( NULL, NULL, NULL );
    /* See Technical QA1133: "loginwindow" means no one's logged in */
    if ( cfuser == NULL ||
		[ ( NSString * )cfuser isEqualToString: @"loginwindow" ] ) {
        NSLog( @"No user logged in" );
    } else {
	if ( setenv( "CONSOLE_USER",
		[ ( NSString * )cfuser UTF8String ], 1 ) != 0 ) {
	    NSLog( @"setenv: %s", strerror( errno ));
	}
    }
    
    if ( cfuser != NULL ) {
	ihookConsoleUser = [ ( NSString * )cfuser retain ];
	CFRelease( cfuser );
    }
}

- ( int )logFileDescriptor
{
    return( logfd );
}

- ( void )setLogFileDescriptor: ( int )fd
{
    if ( fd == -1 ) {
	if ( close( logfd ) != 0 ) {
	    [ self addToStderrField:
		[ NSString stringWithFormat: @"Warning: close log file "
		 @"descriptor failed: %s.\n", strerror( errno ) ]];
	}
    }
    logfd = fd;
}

- ( void )logFileAppendText: ( NSString * )text
{
    const char          *string = [ text UTF8String ];
    
    if ( self.logFileDescriptor < 0 || string == NULL ) {
	return;
    }

    if ( write( self.logFileDescriptor, string, strlen( string ))
	    != strlen( string )) {
        NSLog( @"write error: %s", strerror( errno ));
        return;
    }
}

- ( void )setInputEnabled: ( BOOL )enabled
{
    switch ( enabled ) {
    case NO:
	if ( [ inputBox superview ] ) {
	    [ inputBox retain ];
	    [ inputBox removeFromSuperview ];
	}
	break;
	
    case YES:
	if ( ! [ inputBox superview ] ) {
	    [ superBox addSubview: inputBox ];
	}
	break;
    }
    
    ihookInputEnabled = enabled;
}

- ( BOOL )inputEnabled
{
    return( ihookInputEnabled );
}

- ( void )inputQueueAddInput: ( NSString * )input
{
    NSLock	    *lock = [[ NSLock alloc ] init ];

    if ( [ lock tryLock ] ) {
	[ ihookInputQueue addObject: input ];
	input_queue_count++;
	[ lock unlock ];
    }
    [ lock release ];
}

- ( void )centerProgBar
{
    NSRect		superRect, progBarRect, textFieldRect;
    float		newx, newy, ratio = ( 4.0 / 9.0 ); /* ratio of progBar width to view width */
    
    progBarRect = [ progBar frame ];
    superRect = [ progBarView frame ];
    textFieldRect = [ progMessageField frame ];
        
    progBarRect.size.width = ( superRect.size.width * ratio );
    textFieldRect.size.width = ( superRect.size.width * ratio );

    newx = (( superRect.size.width - progBarRect.size.width ) / 2 );
    newy = 5.0;

    progBarRect = NSMakeRect( newx, newy, progBarRect.size.width,
                            progBarRect.size.height );
    [ progBar setFrame: progBarRect ];
    
    newy = (( 2 * newy ) + progBarRect.size.height );

    textFieldRect = NSMakeRect( newx, newy, textFieldRect.size.width,
                                textFieldRect.size.height );
    [ progMessageField setFrame: textFieldRect ];
}

- ( void )basicProgressFeedback: ( id )context
{
    NSImage	*backgroundImage;
    NSString	*imagePath;
    
    /* 
     * If script doesn't give us any directive within 3 secs of start,
     * start an indeterminate progress bar to give a sense
     * of progress being made.
     */
    
    if ( ihookHadDirective || ihookTimer == nil ||
	    ![ self.isExecuting boolValue ] ) {
	return;
    }
    
    imagePath = [[ NSBundle mainBundle ] pathForResource:
                            @"ihook-trans" ofType: @"png" ];
    backgroundImage = [[ NSImage alloc ] initWithContentsOfFile: imagePath ];
    if ( [ backgroundImage isValid ] ) {
	[ mainImage setImage: backgroundImage ];
    }
    [ backgroundImage release ];
    [ progBarView addSubview: progBar ];
    [ self centerProgBar ];
    [ progBar setIndeterminate: YES ];
    [ progBar startAnimation: nil ];
}

- ( void )start
{
    HookOperation	*hook = nil;
    NSString		*scriptDirectory = nil;
    NSArray		*procArgs = [[ NSProcessInfo processInfo ] arguments ];
    NSMutableArray 	*args = nil;
    int			i;

    NSAssert( self.scriptPath, @"Tried to execute a nil script" );
            
    for ( i = 1; i < [ procArgs count ]; i++ ) {
	if ( args == nil ) {
	    args = [[ NSMutableArray alloc ] init ];
	}
	
	if ( [[ procArgs objectAtIndex: i ] beginsWithString: @"--script=" ] ||
		[[ procArgs objectAtIndex: i ] isEqualToString: @"--no-titlebar" ] ||
		[[ procArgs objectAtIndex: i ] isEqualToString: self.scriptPath ] ) {
	    continue;
	}
	
	[ args addObject: [ procArgs objectAtIndex: i ]];
    }
		
    scriptDirectory = [ self.scriptPath stringByDeletingLastPathComponent ];
    if ( [ scriptDirectory length ] == 0 ) {
	scriptDirectory = @".";
    }
    
    /*
     * we could use NSFileManager's -changeCurrentDirectoryPath:, but
     * chdir is superior because it at least tells us why it failed.
     */
    if ( chdir( [ scriptDirectory UTF8String ] ) < 0 ) {
	NSLog( @"chdir to %s: %s",
		[ scriptDirectory UTF8String ], strerror( errno ));
    }

    hook = [[ HookOperation alloc ] initWithController: self
	    executable: self.scriptPath
	    arguments: args
	    inputQueue: ihookInputQueue ];
    [ args release ];
    [ self.operationQueue addOperation: hook ];
    [ hook release ];

    [ mainWindow setTitle: [ NSString stringWithFormat: @"iHook: %@",
			    [ self.scriptPath lastPathComponent ]]];
    
    if ( ihookTimer == nil ) {
	ihookTimer = [ NSTimer scheduledTimerWithTimeInterval: 5.0 target: self
			selector: @selector( elapsedTimeUpdate ) userInfo: nil
			repeats: YES ];
    }
    [ self performSelector: @selector( basicProgressFeedback: )
	    withObject: nil afterDelay: 3.0 ];
}

- ( void )hideProgressBar
{
    [ progBar retain ];
    [ progBar removeFromSuperview ];
}

- ( oneway void )setStdoutMessage: ( NSString * )msg
{
    NSAutoreleasePool	*pool = nil;
    int			i, tac;
    char		*p, *output = NULL, **targv;

    if ( ![ msg length ] ) return;

    if (( output = strdup( [ msg UTF8String ] )) == NULL ) {
        NSLog( @"strdup %@: %s\n", msg, strerror( errno ));
        exit( 2 );
    }
    
    [ self logFileAppendText: msg ];
    
    if (( p = strchr( output, '%' )) != NULL ) {
        if ( !ihookHadDirective ) {
	    ihookHadDirective = YES;
	}
        if ( self.debug ) {
            [ self addToStderrField: @"DEBUG: " ];
            [ self addToStderrField: msg ];
        }
        
        p++;
        if ( p == NULL ) { NSLog( @"Nothing following %%." ); return; }
        
        tac = argcargv( p, &targv );
        if ( tac <= 0 ) return;
        
	pool = [[ NSAutoreleasePool alloc ] init ];
	
        switch ( *targv[ 0 ] ) {
        case 'B':
            if ( strcmp( targv[ 0 ], "BACKGROUND" ) == 0 && tac > 1 ) {
                NSImage		*bg;
                NSString	*imgpath = nil;
                
                imgpath = [ NSString stringWithUTF8String: targv[ 1 ]];
                
                if ( [ imgpath isEqualToString: @"IHOOKDEFAULT" ] ) {
                    bg = [[ NSImage imageNamed: @"ihook-trans.png" ] retain ];
                } else {
                    for ( i = 2; i < tac; i++ ) {
                        imgpath = [ imgpath stringByAppendingFormat: @" %s", targv[ i ]];
                    }
        
                    bg = [[ NSImage alloc ] initWithContentsOfFile: imgpath ];
                    if ( ! [ bg isValid ] ) {
                        [ self addToStderrField:
                            @"iHook Warning: %BACKGROUND parameter is not a valid image.\n" ];
                    }
                }
                
                [ mainImage setImage: bg ];
                [ bg release ];
            } else if ( strcmp( targv[ 0 ], "BACKGROUNDSCALING" ) == 0 && tac == 2 ) {
                if ( strcmp( targv[ 1 ], "TOFIT" ) == 0 ) {
                    [ mainImage setImageScaling: NSScaleToFit ];
                } else if ( strcmp( targv[ 1 ], "PROPORTIONALLY" ) == 0 ) {
                    [ mainImage setImageScaling: NSScaleProportionally ];
                } else if ( strcmp( targv[ 1 ], "NONE" ) == 0 ) {
                    [ mainImage setImageScaling: NSScaleNone ];
                }
            }  else if ( strcmp( targv[ 0 ], "BECOMEKEY" ) == 0 ) {
                [ NSApp activateIgnoringOtherApps: YES ];
            } else if ( strcmp( targv[ 0 ], "BEEP" ) == 0 ) {
                NSBeep();
            } else if ( strcmp( targv[ 0 ], "BEGINPOLE" ) == 0 ) {
                [ NSObject cancelPreviousPerformRequestsWithTarget: self       
                                selector: @selector( hideProgressBar )
                                object: nil ];
                if ( [ progBar superview ] == nil ) {
                    [ progBarView addSubview: progBar ];
                    [ self centerProgBar ];
                }
                [ progBar setIndeterminate: YES ];
                [ progBar startAnimation: nil ];
            }
            break;
            
        case 'C':
            if ( strcmp( targv[ 0 ], "CANCEL" ) == 0 && tac == 2 ) {
                if ( strcmp( targv[ 1 ], "DISABLE" ) == 0 ) {
                    if ( [ cancelButton superview ] == nil ) {
                        [[ stdoutField superview ] addSubview: cancelButton ];
                        [ cancelButton setFrameOrigin: NSMakePoint( 15.0, 15.0 ) ];
                    }
                    [ cancelButton setEnabled: NO ];
                } else if ( strcmp( targv[ 1 ], "ENABLE" ) == 0 ) {
                    if ( [ cancelButton superview ] == nil ) {
                        [[ stdoutField superview ] addSubview: cancelButton ];
                        [ cancelButton setFrameOrigin: NSMakePoint( 15.0, 15.0 ) ];
                    }
                    [ cancelButton setEnabled: YES ];
                } else if ( strcmp( targv[ 1 ], "REMOVE" ) == 0 ) {
                    if ( [ cancelButton superview ] != nil ) {
                        [ cancelButton retain ];
                        [ cancelButton removeFromSuperview ];
                    }
                }
            } else if ( strcmp( targv[ 0 ], "CLOSEDRAWER" ) == 0 ) {
                if ( [ stderrDrawer state ] == NSDrawerOpenState
                        || [ stderrDrawer state ] == NSDrawerOpeningState ) {
                    [ stderrDrawer close ];
                }
            }
            break;
            
        case 'D':
            if ( strcmp( targv[ 0 ], "DEBUG" ) == 0 ) {
                self.debug = YES;
            }
            break;
            
        case 'E':
            if ( strcmp( targv[ 0 ], "ENDPOLE" ) == 0 ) {
                if ( [ progBar superview ] != nil ) {
                    [ progBar setIndeterminate: YES ];
                    [ progBar stopAnimation: nil ];
                    [ progBar retain ];
                    [ progBar removeFromSuperview ];
                }
            }
            break;
            
        case 'H':
            if ( strcmp( targv[ 0 ], "HIDETIMER" ) == 0 ) {
                [ eTimeField setStringValue: @"" ];
                self.showTimer = NO;
            } else if ( strcmp( targv[ 0 ], "HIDE" ) == 0 ) {
		[ NSApp hide: self ];
	    }
            break;
	    
	case 'I':
	    if ( strcmp( targv[ 0 ], "INPUT" ) == 0 ) {
		if ( tac == 1 ) {
		    NSLog( @"%%%s requires additional parameters.", targv[ 0 ] );
		    break;
		}
            
		if ( strcmp( targv[ 1 ], "ENABLE" ) == 0 ) {
		    NSString	*title = @"";
		    
		    for ( i = 2; i < tac; i++ ) {
			title = [ title stringByAppendingFormat: @"%s ", targv[ i ]];
		    }
		    [ inputTitle setStringValue: title ];
		    [ inputField setEnabled: YES ];
		    [ inputButton setEnabled: YES ];
		    self.inputEnabled = YES;
		    [[ inputField window ] makeFirstResponder: inputField ];
		} else if ( strcmp( targv[ 1 ], "DISABLE" ) == 0 ) {
		    NSString	*title = @"";
		    
		    for ( i = 2; i < tac; i++ ) {
			title = [ title stringByAppendingFormat: @"%s ", targv[ i ]];
		    }
		    [ inputTitle setStringValue: title ];
		    [ inputField setEnabled: NO ];
		    [ inputButton setEnabled: NO ];
		} else if ( strcmp( targv[ 1 ], "REMOVE" ) == 0 ) {
		    self.inputEnabled = NO;
		}
	    }
	    break;
		    
        case 'L':
            if ( tac == 3 && strcmp( targv[ 0 ], "LOG" ) == 0 ) {
                int             fd;
                int             flags = ( O_CREAT | O_WRONLY | O_NONBLOCK );
                char            *logopt = targv[ 1 ];
                char            *logpath = targv[ 2 ];
                
		if ( self.logFileDescriptor >= 0 ) {
		    self.logFileDescriptor = -1;
                }
                
                if ( strcmp( logopt, "CLOSE" ) == 0 ) {
                    break;
                }
                
                if ( strcmp( logopt, "APPEND" ) == 0 ) {
                    flags |= O_APPEND;
                } else if ( strcmp( logopt, "OPEN" ) == 0 ) {
                    flags |= O_TRUNC;
                } else {
                    NSLog( @"%s: unrecognized iHook %%LOG option.", logopt );
                    break;
                }
                
                if (( fd = open( logpath, flags, 0600 )) < 0 ) {
                    NSLog( @"open %s: %s", logpath, strerror( errno ));
                    break;
                }
                self.logFileDescriptor = fd;
            }
            break;
            
        case 'O':
            if ( strcmp( targv[ 0 ], "OK" ) == 0 && tac == 2 ) {
                if ( strcmp( targv[ 1 ], "DISABLE" ) == 0 ) {
                    if ( [ defaultButton superview ] == nil ) {
                        [[ stdoutField superview ] addSubview: defaultButton ];
                        [ defaultButton setTitle: @"OK" ];
                        [ defaultButton setFrameOrigin:
                                NSMakePoint(( [ mainWindow frame ].size.width
                                        - [ defaultButton frame ].size.width - 15.0 ), 15.0 ) ];
                        [ defaultButton setAction: @selector( normalQuit ) ];
                    }
                    [ defaultButton setEnabled: NO ];
                } else if ( strcmp( targv[ 1 ], "ENABLE" ) == 0 ) {
                    if ( [ defaultButton superview ] == nil ) {
                        [[ stdoutField superview ] addSubview: defaultButton ];
                        [ defaultButton setTitle: @"OK" ];
                        [ defaultButton setFrameOrigin:
                                NSMakePoint(( [ mainWindow frame ].size.width
                                        - [ defaultButton frame ].size.width - 15.0 ), 15.0 ) ];
                        [ defaultButton setAction: @selector( normalQuit ) ];
                    }
                    [ defaultButton setEnabled: YES ];
                } else if ( strcmp( targv[ 1 ], "REMOVE" ) == 0 ) {
                    if ( [ defaultButton superview ] != nil ) {
                        [ defaultButton retain ];
                        [ defaultButton removeFromSuperview ];
                    }
                }
            } else if ( strcmp( targv[ 0 ], "OPENDRAWER" ) == 0 ) {
                if ( [ stderrDrawer state ] == NSDrawerClosedState
                        || [ stderrDrawer state ] == NSDrawerClosingState ) {
                    [ stderrDrawer openOnEdge: NSMinYEdge ];
                }
            }
            break;
            
        case 'R':
            if ( strcmp( targv[ 0 ], "RESIGNKEY" ) == 0 ) {
                [ NSApp deactivate ];
            }
            break;
            
        case 'S':
            if ( strcmp( targv[ 0 ], "SHOWTIMER" ) == 0 ) {
                self.showTimer = YES;
                [ eTimeField setStringValue:
			[ NSString clockStringFromInteger: ihookElapsedTime ]];
            } else if ( strcmp( targv[ 0 ], "SOUND" ) == 0 && tac > 1 ) {
                NSSound		*snd = nil;
                NSString	*sndpath = nil;
                
                sndpath = [ NSString stringWithUTF8String: targv[ 1 ]];
                
                for ( i = 2; i < tac; i++ ) {
                    sndpath = [ NSString stringWithFormat: @"%@ %s", sndpath, targv[ i ]];
                }
                
                /* check to make sure file exists */
                if ( access( [ sndpath UTF8String ], F_OK ) < 0 ) {
                    [ self addToStderrField: [ NSString stringWithFormat:
                            @"iHook Warning: %@: %s\n", sndpath, strerror( errno ) ]];
                    break;
                }
                
                snd = [[[ NSSound alloc ] initWithContentsOfFile: sndpath
                                byReference: YES ] autorelease ];
                                
                if ( snd == nil ) {
                    [ self addToStderrField:
                        [ NSString stringWithFormat: @"iHook Warning: %@ is not a valid sound file\n",
                                                    sndpath ]];
                    break;
                }
                if ( ! [ snd play ] ) {
                    [ self addToStderrField: @"iHook Error: failed to play sound.\n" ];
                }
            }
                
            break;
            
        case 'T':
            if ( strcmp( targv[ 0 ], "TEXTCOLOR" ) == 0 ) {
                NSColor		*textColor = nil;
                
                if ( tac == 1 ) {
                    textColor = [ NSColor blackColor ];
                } else if ( tac == 2 ) {
                    textColor = [ NSColor colorForName: [ NSString stringWithUTF8String: targv[ 1 ]]];
                    
                    if ( textColor == nil ) {
                        [ self addToStderrField:
                                [ NSString stringWithFormat:
                                @"iHook Warning: %s: invalid color\n", targv[ 1 ]]];
                        break;
                    }
                } else {
                    [ self addToStderrField: @"iHook Warning: too many arguments to %TEXTCOLOR\n" ];
                    break;
                }
                
                [ titleField setTextColor: textColor ];
                [ stdoutField setTextColor: textColor ];
            } else if ( strcmp( targv[ 0 ], "TITLE" ) == 0 ) {
                NSString	*title = nil;
                
                if ( tac == 1 ) {
                    [ titleField setStringValue: @"" ];
                } else if ( tac > 1 ) {
                    title = [ NSString stringWithUTF8String: targv[ 1 ]];
                    
                    for ( i = 2; i < tac; i++ ) {
                        title = [ NSString stringWithFormat: @"%@ %s", title, targv[ i ]];
                    }
                    [ titleField setStringValue: title ];
                }
            }
            break;
            
        case 'U':
            if ( strcmp( targv[ 0 ], "UIMODE" ) == 0 && tac == 2 ) {
                SystemUIMode		mode = kUIModeNormal;
                SystemUIOptions		options = 0;
                OSStatus		status;
                
                if ( strcmp( targv[ 1 ], "AUTOCRATIC" ) == 0 ) {
                    mode = kUIModeAllHidden;
                    options = ( kUIOptionDisableAppleMenu | kUIOptionDisableProcessSwitch |
                                kUIOptionDisableForceQuit | kUIOptionDisableSessionTerminate );
                } else if ( strcmp( targv[ 1 ], "SELFISH" ) == 0 ) {
                    mode = kUIModeContentHidden;
                    options = ( kUIOptionDisableForceQuit | kUIOptionDisableSessionTerminate );
                } else if ( strcmp( targv[ 1 ], "NORMAL" ) != 0 ) {
                    [ self addToStderrField: [ NSString stringWithFormat:
                            @"Warning: %s: not a valid iHook mode.", targv[ 1 ]]];
                }
                
                if (( status = SetSystemUIMode( mode, options )) != noErr ) {
                    [ self addToStderrField: [ NSString stringWithFormat:
                            @"SetSystemUIMode failed: error %d", status ]];
                }
            } else if ( strcmp( targv[ 0 ], "UNHIDE" ) == 0 ) {
		[ NSApp unhide: self ];
		[ NSApp activateIgnoringOtherApps: YES ];
	    }
	    break;
            
        case 'W':
            if ( strcmp( targv[ 0 ], "WINDOWDEMINIATURIZE" ) == 0 ) {
                if ( [ mainWindow isMiniaturized ] ) {
                    [ mainWindow deminiaturize: nil ];
                }
            } else if ( strcmp( targv[ 0 ], "WINDOWLEVEL" ) == 0 && tac == 2 ) {
                if ( strcmp( targv[ 1 ], "HIGH" ) == 0 ) {
                    [ mainWindow setLevel: ( NSScreenSaverWindowLevel - 1 ) ];
                } else if ( strcmp( targv[ 1 ], "NORMAL" ) == 0 ) {
                    [ mainWindow setLevel: NSNormalWindowLevel ];
                }
            } else if ( strcmp( targv[ 0 ], "WINDOWMINIATURIZE" ) == 0 ) {
                if ( self.consoleUser && ! [ mainWindow isMiniaturized ] ) {
                    [ mainWindow miniaturize: nil ];
                }
            } else if ( strcmp( targv[ 0 ], "WINDOWPOSITION" ) == 0 && tac > 1 ) {
                int		top = 0, bottom = 0, left = 0, right = 0;
                NSPoint		newOrigin = { 0, 0 };
                NSRect		newframe, curframe;
            
                if ( strcmp( targv[ 1 ], "CENTER" ) == 0 ) {
                    curframe = [ mainWindow frame ];
                    newOrigin = [ mainWindow centeredOriginForWindowSize: curframe.size ];
                    newframe = NSMakeRect( newOrigin.x, newOrigin.y,
                                            curframe.size.width, curframe.size.height );
                    [ mainWindow setFrame: newframe display: YES animate: [ mainWindow windowZoom ]];
                    break;
                }
                
                if ( strcmp( targv[ 1 ], "TOP" ) == 0 ) {
                    top = 1;
                } else if ( strcmp( targv[ 1 ], "BOTTOM" ) == 0 ) {
                    bottom = 1;
                } else {
                    [ self addToStderrField:
                        @"iHook Warning: malformed %WINDOWPOSITION command.\n" ];
                    break;
                }
                if ( strcmp( targv[ 2 ], "LEFT" ) == 0 ) {
                    left = 1;
                } else if ( strcmp( targv[ 2 ], "RIGHT" ) == 0 ) {
                    right = 1;
                } else {
                    [ self addToStderrField:
                        @"iHook Warning: malformed %WINDOWPOSITION command.\n" ];
                    break;
                }
                
                curframe = [ mainWindow frame ];
                if ( top ) {
                    if ( left ) {
                        newOrigin = [ mainWindow quadrantTwoOriginForWindowSize: curframe.size ];
                    } else if ( right ) {
                        newOrigin = [ mainWindow quadrantOneOriginForWindowSize: curframe.size ];
                    }
                } else if ( bottom ) {
                    if ( left ) {
                        newOrigin = [ mainWindow quadrantThreeOriginForWindowSize: curframe.size ];
                    } else if ( right ) {
                        newOrigin = [ mainWindow quadrantFourOriginForWindowSize: curframe.size ];
                    }
                }
                newframe = NSMakeRect( newOrigin.x, newOrigin.y,
                                        curframe.size.width, curframe.size.height );
                [ mainWindow setFrame: newframe display: YES animate: [ mainWindow windowZoom ]];
                [ progBar setNeedsDisplay: YES ];
            } else if ( strcmp( targv[ 0 ], "WINDOWSIZE" ) == 0 && tac > 1 ) {
                NSPoint		cOrigin;
                NSRect		newframe;
                BOOL		animate = [ mainWindow windowZoom ];
                float		w, h;
                
                if ( strcmp( targv[ 1 ], "MAX" ) == 0 ) {
                    NSRect	screensize = [[ NSScreen mainScreen ] frame ];
                    
                    [ mainWindow setFrame: screensize display: YES animate: animate ];
                    [ progBar setNeedsDisplay: YES ];
                    break;
                } else if ( strcmp( targv[ 1 ], "MIN" ) == 0 ) {
                    cOrigin = [ mainWindow centeredOriginForWindowSize: NSMakeSize( 200.0, 200.0 ) ];
                    newframe = NSMakeRect( cOrigin.x, cOrigin.y, 200.0, 200.0 );
                    [ mainWindow setFrame: newframe display: YES animate: animate ];
                    [ progBar setNeedsDisplay: YES ];
                    break;
                }
                
                if ( tac != 3 || ! isdigit( *targv[ 1 ] ) || ! isdigit( *targv[ 2 ] )) {
                    [ self addToStderrField: @"iHook Warning: malformed %WINDOWSIZE directive.\n" ];
                    break;
                }
                w = strtod( targv[ 1 ], NULL );
                h = strtod( targv[ 2 ], NULL );
                
                if ( w < 200 || h < 200 || w > 10000 || h > 10000 ) {
                    [ self addToStderrField: @"iHook Warning: invalid window size.\n" ];
                    break;
                }
                
                cOrigin = [ mainWindow centeredOriginForWindowSize: NSMakeSize( w, h ) ];
                newframe = NSMakeRect( cOrigin.x, cOrigin.y, w, h );
                [ mainWindow setFrame: newframe display: YES animate: animate ];
                [ progBar setNeedsDisplay: YES ];
            } else if ( strcmp( targv[ 0 ], "WINDOWZOOM" ) == 0 && tac == 2 ) {
                if ( strcmp( targv[ 1 ], "ENABLE" ) == 0 ) {
                    [ mainWindow setWindowZoomEnabled: YES ];
                } else if ( strcmp( targv[ 1 ], "DISABLE" ) == 0 ) {
                    [ mainWindow setWindowZoomEnabled: NO ];
                }
            }
            break;
            
        default:
            if ( isdigit( *targv[ 0 ] )) {
                double		pctdone = 0.0;
                char		*endptr = NULL;
                NSString	*progMessage = @"";
                
                if ( strlen( targv[ 0 ] ) > 3 ) {
                    [ self addToStderrField: @"iHook Warning: %%done value exceeds bounds.\n" ];
                    break;
                }
                
                pctdone = strtod( targv[ 0 ], &endptr );
                if ( pctdone == 0.0 && strcmp( targv[ 0 ], endptr ) == 0 ) {
                    [ self addToStderrField: @"iHook Warning: invalid %%done value.\n" ];
                    break;
                }
                
                [ NSObject cancelPreviousPerformRequestsWithTarget: self       
                                selector: @selector( hideProgressBar )
                                object: nil ];
                
                if ( [ progBar superview ] == nil ) {
                    [ progBarView addSubview: progBar ];
                    [ self centerProgBar ];
                }
    
                if ( [ progBar isIndeterminate ] ) {
                    [ progBar setIndeterminate: NO ];
                }
                
                [ progBar setDoubleValue: pctdone ];
                
                if ( pctdone == 100.0 ) {
                    [ self performSelector: @selector( hideProgressBar )
                                withObject: nil
                                afterDelay: 1.0 ];
                }
                
                if ( tac > 1 ) {
                    int		i;
                    
                    progMessage = [ NSString stringWithUTF8String: targv[ 1 ]];
                    
                    for ( i = 2; i < tac; i++ ) {
                        progMessage = [ progMessage stringByAppendingFormat:
                                        @" %s", targv[ i ]];
                    }
                }
                [ progMessageField setStringValue: progMessage ];
            }
            break;
        }
	
	[ pool drain ];
    } else {
        [ stdoutField setStringValue: msg ];
    }
    
    free( output );
}

- ( oneway void )addToStderrField: ( NSString * )msg
{
    [ msg retain ];
    [ self logFileAppendText: msg ];
    
    [ stderrTextView setEditable: YES ];
    [ stderrTextView insertText: msg ];
    [ stderrTextView setEditable: NO ];
    [ msg release ];
}

- ( void )error: ( NSString * )error
{
    [ ihookTimer invalidate ];
    ihookTimer = nil;
    
    if ( [ mainWindow isMiniaturized ] ) {
        [ mainWindow deminiaturize: nil ];
    }
    
    if ( [ stderrDrawer state ] == NSDrawerClosedState
            || [ stderrDrawer state ] == NSDrawerClosingState ) {
        [ stderrDrawer openOnEdge: NSMinYEdge ];
    }
    [ self performSelector: @selector( errorQuit: ) withObject: nil
	    afterDelay: 120.0 ];
    
    if ( [ progBar superview ] != nil ) [ progBar removeFromSuperview ];
    if ( [ cancelButton superview ] != nil ) [ cancelButton removeFromSuperview ];
    [ self addToStderrField: error ];
    if ( ![[ stdoutField stringValue ] length ] ) {
        [ stdoutField setStringValue: @"An error occurred." ];
    }
    [[ stdoutField superview ] addSubview: defaultButton ];
    [ defaultButton setTitle: @"Exit" ];
    [ defaultButton setFrameOrigin:
            NSMakePoint(( [ mainWindow frame ].size.width
                    - [ defaultButton frame ].size.width - 15.0 ), 15.0 ) ];
    [ defaultButton setAction: @selector( errorQuit: ) ];
}

- ( void )elapsedTimeUpdate
{
    if ( ![ self.isExecuting boolValue ] ) {
        return;
    }
    ihookElapsedTime += 5;
    if ( self.showTimer ) {
        [ eTimeField setStringValue:
		[ NSString clockStringFromInteger: ihookElapsedTime ]];
    } else {
        if ( [[ eTimeField stringValue ] length ] ) {
            [ eTimeField setStringValue: @"" ];
        }
    }
}

/*
 * We don't actually write to the script's stdin.
 * Instead, we add the input to an input queue
 * that the script thread will handle.
 */
- ( IBAction )writeToStdin: ( id )sender
{
    NSString		    *s = [ inputField stringValue ];
    
    if ( ! [ s length ] ) {
	s = @"";
    }
    s = [ s stringByAppendingString: @"\n" ];
    
    [ self inputQueueAddInput: s ];
}

- ( IBAction )errorQuit: ( id )sender
{
    if ( self.logFileDescriptor >= 0 ) {
	self.logFileDescriptor = -1;
    }
    
    exit( [ self.exitStatus intValue ] );
}

- ( void )normalQuit: ( id )context
{
    if ( self.scriptPath ) {
        [[ NSUserDefaults standardUserDefaults ]
                setObject: [ self.scriptPath lastPathComponent ]
                forKey: @"IHLastHookRun" ];
    }
    if ( self.logFileDescriptor >= 0 ) {
	self.logFileDescriptor = -1;
    }
    
    [ NSApp terminate: self ];
}

- ( IBAction )cancel: ( id )sender
{
    HookOperation   *cancelOperation = nil;

    if ( [ cancelButton superview ] == nil ) {
	return;
    }
    
    if ( [[ self.scriptPath pathExtension ] isEqualToString: @"shook" ] ) {
	cancelOperation = [[ HookOperation alloc ]
			    initAuthorizedCancellationOfTaskWithPID: childpgid
			    controller: self ];
	[ self.operationQueue addOperation: cancelOperation ];
	[ cancelOperation release ];
    } else if ( killpg( childpgid, SIGTERM ) < 0 ) {
        [ stderrTextView insertText:
            [ NSString stringWithFormat:
                @"killpg %d failed: %s", childpgid, strerror( errno ) ]];
    }
}

- ( IBAction )toggleDrawer: ( id )sender
{
    if ( [ stderrDrawer state ] == NSDrawerOpeningState ||
            [ stderrDrawer state ] == NSDrawerOpenState ) {
        [ stderrDrawer close ];
    } else {
        [ stderrDrawer openOnEdge: NSMinYEdge ];
    }
}

- ( void )drawerWillOpen: ( NSNotification * )aNotification
{
    float		y = [ mainWindow frame ].origin.y;
    float		drawerHeight = DEFAULT_HEIGHT;
    
    while ( drawerHeight > y && drawerHeight > 30.0 ) {
        drawerHeight -= 10.0;
    }
    
    [ stderrDrawer setContentSize:
                NSMakeSize( [ stderrDrawer contentSize ].width, drawerHeight ) ];
}

- ( void )dealloc
{
    [ ihookInputQueue release ];
    [ ihookOperationQueue release ];
    [ super dealloc ];
}

/* synthesized properties */
@synthesize	    consoleUser = ihookConsoleUser;
@synthesize	    scriptPath = ihookScriptPath;
@synthesize	    debug = ihookDebug;
@synthesize	    inputEnabled = ihookInputEnabled;
@synthesize	    showTimer = ihookShowTimer;
@synthesize	    isExecuting = ihookIsExecuting;
@synthesize	    exitStatus = ihookExitStatus;
@synthesize	    logFileDescriptor = logfd;
@synthesize	    operationQueue = ihookOperationQueue;
@end
