/**
 * HyperEstraierWrapper.cpp - C++ wrapper for Hyper Estraier
 */
#include <estraier.h>
#include <estmtdb.h>
#include <cabin.h>
#include <cstdlib>
#include <string>
#include <vector>
#include <map>
#include <cassert>
#include <stdexcept>

namespace estraier {

	class IOError : public std::runtime_error {
	public:
		explicit IOError (const std::string& w) : std::runtime_error(w) {}
	};

	class Condition {
	public:
		enum {								// enumeration for options 
			SURE	= ESTCONDSURE,		// check every N-gram key
			USUAL	= ESTCONDUSUAL,		// check N-gram keys skipping by one
			FAST	= ESTCONDFAST,		// check N-gram keys skipping by two
			AGITO	= ESTCONDAGITO,		// check N-gram keys skipping by three
			NOIDF	= ESTCONDNOIDF,		// without TF-IDF tuning
			SIMPLE	= ESTCONDSIMPLE,	// with the simplefied phrase
		};
		ESTCOND * cond;
		Condition() {
			/**
			 * constructor
			 */
			cond = est_cond_new();
		}
		~Condition() {
			/**
			 * destructor
			 */
			est_cond_delete(cond);
		}
		void set_phrase(const char *phrase) {
			/**
			 * set the search phrase
			 */
			est_cond_set_phrase(cond, phrase);
		}
		void add_attr(const char *expr) {
			/**
			 * set the attribute expression
			 */
			est_cond_add_attr(cond, expr);
		}
		void set_order(const char *expr) {
			/**
			 * set the order of a condition object
			 */
			est_cond_set_order(cond, expr);
		}
		void set_max(int _max) {
			/**
			 * set the maximum number of retrieval of a condition object
			 */
			est_cond_set_max(cond, _max);
		}
		void set_options(int options) {
			/**
			 * set options of retrieval of a condition object
			 */
			est_cond_set_options(cond, options);
		}
	};

	class Document {
	private:
		std::string text_buf;
	public:
		ESTDOC *doc;
		Document() {
			/**
			 * constructor
			 */
			doc = est_doc_new();
		}
		Document(const char* draft) {
			/**
			 * constructor
			 */
			doc = est_doc_new_from_draft(draft);
		}
		Document(ESTDOC *_doc) {
			/**
			 * constructor
			 */
			doc = _doc;
		}
		~Document() {
			/**
			 * destructor
			 */
			est_doc_delete(doc);
		}
		void add_attr(const char * name, const char*value) {
			/**
			 * add an attribute to a document object
			 */
			est_doc_add_attr(doc, name, value);
		}
		void add_text(const char *text) {
			/**
			 * add a sentence of text to a document object
			 */
			est_doc_add_text(doc, text);
		}
		void add_hidden_text(const char * text) {
			/**
			 * add a hidden sentence to a document object
			 */
			est_doc_add_hidden_text(doc, text);
		}
		int id() {
			/**
			 * get the ID number of a document object
			 */
			return est_doc_id(doc);
		}
		std::vector<std::string> * attr_names() {
			/**
			 * get a list of attribute names of a document object
			 */
			std::vector<std::string> * vs = new std::vector<std::string>;
			CBLIST * attr_names = est_doc_attr_names(doc);
			for (int i=0; i < cblistnum(attr_names); i++) {
				vs->push_back(cblistval(attr_names, i, NULL));
			}
			cblistclose(attr_names);
			return vs;
		}
		const char * attr(const char *name) {
			/**
			 * get the value of an attribute of a document object
			 */
			return est_doc_attr(doc, name);
		}
		const char * cat_texts() {
			/**
			 * get a list of sentences of the text of a document object
			 */
			return est_doc_cat_texts(doc);
		}
		std::vector<std::string>* texts() {
			/**
			 * get a list of sentences of the text of a document object
			 */
			std::vector<std::string> * vs = new std::vector<std::string>;
			const CBLIST *texts;
			texts = est_doc_texts(doc);
			for(int i = 0; i < cblistnum(texts); i++) {
				vs->push_back(cblistval(texts, i, NULL));
			}
			return vs;
		}
		const char * dump_draft() {
			/**
			 * dump draft data of a document object
			 */
			return est_doc_dump_draft(doc);
		}
		const char * make_snippet(std::vector<std::string> _words, int wwidth, int hwidth, int awidth) {
			/**
			 * make a snippet of the body text of a document object
			 */
			CBLIST * words;
			std::vector<std::string>::iterator iter;
			words = cblistopen();
			for (iter = _words.begin(); _words.end() != iter; iter++) {
				cblistpush(words, iter->c_str(), -1);
			}
			const char *result = est_doc_make_snippet(doc, words, wwidth, hwidth, awidth); 
			cblistclose(words);
			return result;
		}
		const char * hidden_texts() {
			/**
			 * get the hidden texts of a document object.
			 */
			return est_doc_hidden_texts(doc);
		}
	};

	class Database {
	private:
		ESTMTDB *db;
		int ecode;
	public:
		enum {								// enumeration for error codes
			ERRNOERR	= ESTENOERR,		// no error
			ERRINVAL	= ESTEINVAL,		// invalid argument
			ERRACCES	= ESTEACCES,		// access forbidden
			ERRLOCK		= ESTELOCK,			// lock failure
			ERRDB		= ESTEDB,			// database problem
			ERRIO		= ESTEIO,			// I/O problem
			ERRNOITEM	= ESTENOITEM,		// no item
			ERRMISC		= ESTEMISC			// miscellaneous
		};
		enum {								// enumeration for open modes
			DBREADER	= ESTDBREADER,		// open as a reader
			DBWRITER	= ESTDBWRITER,		// open as a writer
			DBCREAT		= ESTDBCREAT,		// a writer creating
			DBTRUNC		= ESTDBTRUNC,		// a writer truncating
			DBNOLCK		= ESTDBNOLCK,		// open without locking
			DBLCKNB		= ESTDBLCKNB,		// lock without blocking
			DBPERFNG	= ESTDBPERFNG 		// use perfect N-gram analyzer
		};
		enum {								// enumeration for options of document registration
			PDCLEAN		= ESTPDCLEAN		// clean up dispensable regions
		};
		enum {								// enumeration for options of document deletion
			ODCLEAN		= ESTODCLEAN		// clean up dispensable regions
		};
		enum {								// enumeration for options of optimization
			OPTNOPURGE	= ESTOPTNOPURGE,	// omit purging dispensable region of deleted
			OPTNODBOPT	= ESTOPTNODBOPT		// omit optimizization of the database files
		};
		enum {								// enumeration for options of document retrieval
			GDNOATTR	= ESTGDNOATTR,		// no attributes
			GDNOTEXT	= ESTGDNOTEXT 		// no text
		};
		Database() {
			/**
			 * constructor(dummy)
			 */
			db = NULL;
			ecode = ERRNOERR;
		}
		~Database() {
			if (db) close();
		}
		bool open(const char * dbname, int mode) {
			/**
			 * open the database
			 */
			if (db) close();
			int ec;
			db = est_mtdb_open(dbname, mode, &ec);
			if (!db) ecode = ec;
			return db;
		}
		bool close() {
			/**
			 * close the database
			 */
			if (!db) throw IOError("closed database");
			int ec;
			bool result = est_mtdb_close(db, &ec);
			if (!result) ecode = ec;
			db = NULL;
			return result;
		}
		bool put_doc(Document *doc, int options) {
			/**
			 * add a document to a database
			 */
			if (!db) throw IOError("closed database");
			bool result = est_mtdb_put_doc(db, doc->doc, options);
			if (!result) ecode = est_mtdb_error(db);
			return result;
		}
		std::vector<int> * search(Condition * cond, int options) {
			/**
			 * search documents corresponding a condition for a database
			 */
			if (!db) throw IOError("closed database");
			int resnum;
			int * result = est_mtdb_search(db, cond->cond, &resnum, NULL);
			std::vector<int> *numbers = new std::vector<int>;	
			for (int i=0; i<resnum; i++) {
				numbers->push_back(result[i]);
			}
			return numbers;
		}
		static const char * err_msg(int ecode) {
			/**
			 * get the string of an error
			 */
			return est_err_msg(ecode);
		}
		int error() {
			/**
			 * get the last happended error code of a database
			 */
			return ecode;
		}
		bool fatal() {
			/**
			 * check whether a database has a fatal error
			 */
			if (!db) throw IOError("closed database");
			return est_mtdb_fatal(db);
		}
		bool flush(int _max) {
			/**
			 * flush index words in the cache of a database
			 */
			if (!db) throw IOError("closed database");
			bool result = est_mtdb_flush(db, _max);
			if (!result) ecode = est_mtdb_error(db);
			return result;
		}
		bool sync() {
			/**
			 * synchronize updating contents of a database
			 */
			if (!db) throw IOError("closed database");
			bool result = est_mtdb_sync(db);
			if (!result) ecode = est_mtdb_error(db);
			return result;
		}
		bool optimize(int options) {
			/**
			 * optimize a database
			 */
			if (!db) throw IOError("closed database");
			bool result = est_mtdb_optimize(db, options);
			if (!result) ecode = est_mtdb_error(db);
			return result;
		}
		bool out_doc(int id, int options) {
			/**
			 * remove a document from a database
			 */
			if (!db) throw IOError("closed database");
			bool result = est_mtdb_out_doc(db, id, options);
			if (!result) ecode = est_mtdb_error(db);
			return result;
		}
		bool edit_doc(Document *doc) {
			/**
			 * edit an attribute of a document in a database
			 */
			if (!db) throw IOError("closed database");
			bool result = est_mtdb_edit_doc(db, doc->doc);
			if (!result) ecode = est_mtdb_error(db);
			return result;
		}
		Document * get_doc(int id, int options) {
			/**
			 * retrieve a document in a database
			 */
			if (!db) throw IOError("closed database");
			ESTDOC *doc = est_mtdb_get_doc(db, id, options);
			if (!doc) {
				ecode = est_mtdb_error(db);
				throw est_err_msg(est_mtdb_error(db));
			} else {
				return new Document(doc);
			}
		}
		int uri_to_id(const char *uri) {
			/**
			 * get the ID of a document spacified by URI
			 */
			if (!db) throw IOError("closed database");
			int result = est_mtdb_uri_to_id(db, uri);
			if(result == -1) ecode = est_mtdb_error(db);
			return result;
		}
		std::map<std::string, std::string> * etch_doc(Document * doc, int max) {
			/**
			 * extract keywords of a document object
			 */
			if (!db) throw IOError("closed database");
			std::map<std::string, std::string> * mss = new std::map<std::string, std::string>;
			CBMAP * keys = est_mtdb_etch_doc(db, doc->doc, max);
			cbmapiterinit(keys);
			int ksiz;
			while (const char *key = cbmapiternext(keys, &ksiz)) {
				mss->insert(std::make_pair(key, cbmapget(keys, key, ksiz, NULL)));
			}
			return mss;
		}
		const char * name() {
			/**
			 * get the name of a database
			 */
			if (!db) throw IOError("closed database");
			return est_mtdb_name(db);
		}
		int doc_num() {
			/**
			 * get the number of documents in a database
			 */
			if (!db) throw IOError("closed database");
			return est_mtdb_doc_num(db);
		}
		int word_num() {
			/**
			 * get the number of unique words in a database
			 */
			if (!db) throw IOError("closed database");
			return est_mtdb_word_num(db);
		}
		double size() {
			/**
			 * get the size of a database
			 */
			if (!db) throw IOError("closed database");
			return est_mtdb_size(db);
		}
		void set_cache_size(size_t size, int anum, int tnum, int rnum) {
			/**
			 * set the maximum size of the cache memory of a database
			 */
			if (!db) throw IOError("closed database");
			est_mtdb_set_cache_size(db, size, anum, tnum, rnum);
		}
		void set_special_cache(const char *name, int num) {
			/**
			 * Set the special cache for narrowing and sorting
			 * with document attributes
			 */
			est_mtdb_set_special_cache(db, name, num);
		}
	};

	static std::vector<std::string> * break_text(const char *text, bool norm, bool tail) {
		std::vector<std::string> * vs = new std::vector<std::string>;
		CBLIST *list;
		list = cblistopen();
		est_break_text(text, list, norm, tail);
		for (int i=0; i < cblistnum(list); i++) {
			vs->push_back(cblistval(list, i, NULL));
		}
		cblistclose(list);
		return vs;
	}

	static std::vector<std::string> * break_text_perfng(const char *text, bool norm, bool tail) {
		std::vector<std::string> * vs = new std::vector<std::string>;
		CBLIST *list;
		list = cblistopen();
		est_break_text_perfng(text, list, norm, tail);
		for (int i=0; i < cblistnum(list); i++) {
			vs->push_back(cblistval(list, i, NULL));
		}
		cblistclose(list);
		return vs;
	}

};
