/*
 * Hrsh2csv is a command line utility for converting PDB desktop files created by
 * the Hours (http://hours.sourceforge.net) Palm application on a handheld into
 * files in the CSV (comma separated value) format.
 *
 * Copyright (C) 2002 Peter Novotnik
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdlib.h>	/* because of free */
#include <string.h>	/* because of memset */
#include <strings.h>/* because of index */
#include <ctype.h>
#include <time.h>		/* because of strftime */
#include "hrsh2csv.h"
#include "globals.h"


#define IS_HOURS_DB(headerP) (strncmp( headerP->creator, HoursCreator, 4 ) == 0 && strncmp( headerP->type, HoursDBType, 4 ) == 0)


typedef struct
{
	unsigned short	hours;
	unsigned short	minutes;
}
HrshTimeType;

static char * getTotalTimeString( char * buffer, int len, int hours, int minutes );
static char * getTimeString( char * buffer, int len, unsigned short palm_time );
static char * getDateString( char * buffer, int len, unsigned short date );
static void getTotalHoursMinusMinutes( HrshTimeType * totalP, unsigned short begin, unsigned short end, unsigned short lminutes );
static char * clearText( char *s );
static PDBHeaderType * getHeader( FILE * dbFile );


/*
 * FUNCTION:		getTotalTimeString
 * DESCRIPTION:	this routine converts the given time values
 * 							into a string format described by gTotalTimePattern
 * PARAMETERS:	buffer - buffer to recieve the string
 * 							len - length of the buffer
 * 							hours - hours >= 0
 * 							minutes - 0 <= minutes < 60
 * RETURNS:			returns buffer again
 */
static char * getTotalTimeString( char * buffer, int len, int hours, int minutes )
{
	char * 	pattern = gTotalTimePattern;
	int 		i = 0;

	while( *pattern && i < len )
	{
		if( *pattern == '%' )
		{
			pattern++;
			if( *pattern == '%' )
				buffer[i++] = '%';
			/* we do not need to check for the chars as
			 * we already validated the pattern at app startup */
			else /* if( *pattern == 'H' || *pattern == 'M' ) */
			{
				int * val = &hours;
				if( *pattern == 'M' )
					val = &minutes;
				i += snprintf( &buffer[i], len - i, "%02d", *val );
			}
		} 
		else
			buffer[i++] = *pattern;

		pattern++;
	}

	buffer[i] = '\0';
	return (buffer);
}

/*
 * FUNCTION:		getHeader
 * DESCRIPTION:	this routine make some checks and retrieves the header
 * PARAMETERS:	dbFile - the database file
 * RETURNS:			NULL on failure, otherwise a pointer to 
 * 							the header struct (needs to be freed with free)
 */
static PDBHeaderType * getHeader( FILE * dbFile )
{
	PDBHeaderType * headerP;

	if( !dbFile )
	{
		fprintf( stderr, ERROR_PREFIX " Invalid param.\n" );
		return NULL;
	}
	
	if( fseek( dbFile, 0, SEEK_SET ) )
	{
		fprintf( stderr, ERROR_PREFIX " fseek() failed.\n" );
		return NULL;
	}

	if( (headerP = pdbLoadDBHeader( dbFile )) == NULL )
	{
		fprintf( stderr, ERROR_PREFIX " Not a PDB database.\n" );
		return NULL;
	}

	return headerP;
}


/*
 * FUNCTION:		getTimeString
 * DESCRIPTION:	this routine converts the given time
 * 							into the string format described by gTimePattern and writes
 * 							it to the passed buffer
 * PARAMETERS:	buffer - buffer to recieve the string
 * 							len - length of the buffer
 * 							palm_time - the time
 * RETURNS:			returns buffer again
 */
static char * getTimeString( char * buffer, int len, unsigned short palm_time )
{
	struct tm time;

	if( palm_time == TimeNoTime )
	{
		*buffer = '\0';
		return (buffer);
	}

	memset( &time, 0, sizeof( time ) );
	time.tm_hour = TimeGetHours( palm_time );
	time.tm_min = TimeGetMinutes( palm_time );
	time.tm_isdst = -1;

	strftime( buffer, len, gTimePattern, &time );

	return (buffer);
}

/*
 * FUNCTION:		getDateString
 * DESCRIPTION:	this routine converts the given date
 * 							into string format described by gDatePattern and writes it
 * 							to the passed buffer
 * PARAMETERS:	buffer - output buffer
 * 							len - length of the buffer
 * 							date - to be converted
 * RETURNS:			returns buffer again
 */
static char * getDateString( char * buffer, int len, unsigned short date )
{
	struct tm time;

	memset( &time, 0, sizeof( time ) );
	time.tm_mday = DateGetDay( date );
	time.tm_mon = DateGetMonth( date ) - 1;
	time.tm_year = DateGetYear( date ) + firstYear - 1900;
	time.tm_isdst = -1;

	strftime( buffer, len, gDatePattern, &time );

	return (buffer);
}

/*
 * FUNCTION:		clearText
 * DESCRIPTION:	replaces all '\n' in s thour ' '
 * PARAMETERS:	s - pointer to the text
 * RETURNS:			pointer to the begining of s
 */
static char * clearText( char *s )
{
	char * t = s;

	while( *s != '\0' )
	{
		if( ispunct( *s ) )
			;
		else if( *s == '\"' )
			*s = '\'';
		else if( !gDontConvert0x95 && *s == '\x95' )	/* this is the bold dot in symbol font on palm */
			*s = '-';
		else if( !gDontRemoveNL && *s == '\n' )
			*s = ' ';
		else if( gRemoveNonASCII && !isalnum( *s ) )
			*s = ' ';
		s++;
	}

	return (t);
}

/*
 * FUNCTION:				getTotalHoursMinusMinutes
 * DESCRIPTION:			this routine computes the hours between b and e
 * 									and substracts the lminutes parameter from the result
 */
static void getTotalHoursMinusMinutes( HrshTimeType * totalP, unsigned short begin, unsigned short end, unsigned short lminutes )
{
	long totalminutes;
	short hours;
	short minutes;

	memset( totalP, 0, sizeof( HrshTimeType ) );

	/* either begin or end time is not specified */
	/* this seems to be an open record */
	if( begin == TimeNoTime || end == TimeNoTime )
		return;

	hours = ((signed short)TimeGetHours( end ) - (signed short)TimeGetHours( begin ));
	minutes = ((signed short)TimeGetMinutes( end ) - (signed short)TimeGetMinutes( begin ));

	if( hours < 0 )		// end time is earlier as begin time
		hours += 24;

	totalP->hours = hours;
	if( minutes < 0 )
	{
		if( totalP->hours > 0 )
		{
			totalP->hours--;
			minutes = 60 + minutes;		// minutes is negative
		}
		else
			minutes = -minutes;
	}
	totalP->minutes = minutes;

	if( lminutes )
	{
		totalminutes = (totalP->hours * 60) + totalP->minutes;
		if( totalminutes > lminutes )
			totalminutes -= lminutes;
		else
			totalminutes = 0;
		totalP->hours = totalminutes / 60;
		totalP->minutes = totalminutes % 60;
	}
}

/*
 * FUNCTION:				hrshGetSizeOfDBInfoType
 * DESCRIPTION:			this function returns the number
 * 									of bytes of the expected appInfoType
 * 									depending on the database version
 * PARAMETERS:			version ...
 * RETURNS:					sizeof( the appropriate db app info type )
 */
size_t hrshGetSizeOfDBInfoType( unsigned short version )
{
	size_t size;

	if( version == HoursDBVersionV18 )
		size = sizeof( HoursDBInfoTypeV18 );
	else
		size = sizeof( HoursDBInfoTypeV16 );

	return (size);
}



/*
 * FUNCTION:				hrshBeforeIterationCallback
 * DESCRIPTION:			this function is called by the pdb-library before it starts
 * 									to iterate over the records. we use this possibility to print
 * 									a header of columns.
 */
int hrshBeforeIterationCallback( FILE * outfile, int numRecords, PDBHeaderType * headerP, void * appInfoP )
{
	if( !headerP )
		return (pdbErrGeneralError);

	if( !IS_HOURS_DB(headerP) )
	{
		fprintf( stderr, ERROR_PREFIX " Not an Hours Database.\n" );
		return (pdbErrCreatorDifferent);
	}

	if( headerP->version != HoursDBVersionV15 && headerP->version != HoursDBVersionV16 && headerP->version != HoursDBVersionV18 )
	{
		fprintf( stderr, ERROR_PREFIX " Cannot handle database version of %d.\nCheck for newer version of this converter.\n", headerP->version );
		return (pdbErrGeneralError);
	}

	if( gPrintHeader )
	{
		char * enclose = gDontEncloseValues ? "" : ENCLOSE_VALUE_STRING;
		fprintf( outfile, "%sCategory%s"
											"%s"
											"%sBilled%s"
											"%s"
											"%sDate%s"
											"%s"
											"%sBegin%s"
											"%s"
											"%sEnd%s"
											"%s"
											"%sTotal hours%s",
												enclose,
												enclose,
												gSeparatorString,
												enclose,
												enclose,
												gSeparatorString,
												enclose,
												enclose,
												gSeparatorString,
												enclose,
												enclose,
												gSeparatorString,
												enclose,
												enclose,
												gSeparatorString,
												enclose,
												enclose
										);
		if( headerP->version >= HoursDBVersionV16 )
			fprintf( outfile, "%s"
												"%sLunch break%s"
												"%s"
												"%sTotal hours [minus Break]%s",
													gSeparatorString,
													enclose,
													enclose,
													gSeparatorString,
													enclose,
													enclose
											);
		if( headerP->version >= HoursDBVersionV18 )
			fprintf( outfile, "%s"
												"%sRate (%s)%s"
												"%s"
												"%sExtra (%s)%s",
													gSeparatorString,
													enclose,
													appInfoP ? ((HoursDBInfoTypeV18 *)appInfoP)->currency_symbol : "",
													enclose,
													gSeparatorString,
													enclose,
													appInfoP ? ((HoursDBInfoTypeV18 *)appInfoP)->currency_symbol : "",
													enclose
											);
		fprintf( outfile, "%s"
											"%sDescription%s\n",
												gSeparatorString,
												enclose,
												enclose
										);
	}

	return (pdbErrNone);
}

/*
 * FUNCTION:					hrshConvertRecord
 * DESCRIPTION:				this function is called each time the pdb-library finds a record.
 * 										it prints the values of the records.
 */
void hrshConvertRecord( FILE * outfile, int recIndex, HoursRecordTypeV15 * recordP, PDBRecordType * recInfoP, PDBHeaderType * headerP, HoursDBInfoTypeV16 * appInfoP )
{
	if( headerP && recordP )
	{
		HrshTimeType ttime;
		char date_buffer[DATE_STRING_BUF_LEN];
		char begin_buffer[TIME_STRING_BUF_LEN];
		char end_buffer[TIME_STRING_BUF_LEN];
		char total_buffer[TOTAL_TIME_STRING_BUF_LEN];
		char * s;
		char * enclose;

		enclose = gDontEncloseValues ? "" : ENCLOSE_VALUE_STRING;

#ifdef __MAC2PC_CONVERSION__
		recordP->begin = MAC2PC_SHORT( recordP->begin );
		recordP->end = MAC2PC_SHORT( recordP->end );
		recordP->date = MAC2PC_SHORT( recordP->date );
#endif /* __MAC2PC_CONVERSION__ */

		getTotalHoursMinusMinutes( &ttime, recordP->begin, recordP->end, 0 );

		fprintf( outfile, "%s%s%s%s"					/* category */
											"%s%s%s%s"					/* billed */
											"%s%s%s%s"					/* date */
											"%s%s%s%s"					/* begin */
											"%s%s%s%s"					/* end */
											"%s%s%s%s",					/* total hours */
			enclose,
			(recInfoP != NULL && appInfoP != NULL) ?  appInfoP->categoryLabels[recInfoP->attributes & dmRecAttrCategoryMask] : "",
			enclose,
			gSeparatorString,

			enclose,
			(recInfoP != NULL && (recInfoP->attributes & dmRecAttrSecret) == dmRecAttrSecret) ? "yes" : "no",
			enclose,
			gSeparatorString,

			enclose,
			getDateString( date_buffer, DATE_STRING_BUF_LEN, recordP->date ),
			enclose,
			gSeparatorString,

			enclose,
			getTimeString( begin_buffer, TIME_STRING_BUF_LEN, recordP->begin ),
			enclose,
			gSeparatorString,

			enclose,
			getTimeString( end_buffer, TIME_STRING_BUF_LEN, recordP->end ),
			enclose,
			gSeparatorString,

			enclose,
			(recordP->begin == TimeNoTime || recordP->end == TimeNoTime) ? "" : getTotalTimeString( total_buffer, TOTAL_TIME_STRING_BUF_LEN, ttime.hours, ttime.minutes ),
			enclose,
			gSeparatorString
		);

		if( headerP->version >= HoursDBVersionV16 )
		{
			char break_buffer[TOTAL_TIME_STRING_BUF_LEN];
#ifdef __MAC2PC_CONVERSION__
			((HoursRecordTypeV16 *)recordP)->lunchminutes = MAC2PC_SHORT( ((HoursRecordTypeV16 *)recordP)->lunchminutes );
#endif /* __MAC2PC_CONVERSION__ */

			getTotalHoursMinusMinutes( &ttime, recordP->begin, recordP->end, ((HoursRecordTypeV16 *)recordP)->lunchminutes );

			fprintf( outfile, "%s%s%s%s"				/* lunch break */
												"%s%s%s%s",				/* total hours minus lunch break */
				enclose,
				getTotalTimeString( break_buffer, TOTAL_TIME_STRING_BUF_LEN, ((HoursRecordTypeV16 *)recordP)->lunchminutes / 60, ((HoursRecordTypeV16 *)recordP)->lunchminutes % 60 ),
				enclose,
				gSeparatorString,

				enclose,
				(recordP->begin == TimeNoTime || recordP->end == TimeNoTime) ? "" : getTotalTimeString( total_buffer, TOTAL_TIME_STRING_BUF_LEN, ttime.hours, ttime.minutes ),
				enclose,
				gSeparatorString
			);

			if( headerP->version >= HoursDBVersionV18 )
			{
#ifdef __MAC2PC_CONVERSION__
				((HoursRecordTypeV18 *)recordP)->extra.dollars = MAC2PC_SHORT( ((HoursRecordTypeV18 *)recordP)->extra.dollars );
				((HoursRecordTypeV18 *)recordP)->rate.dollars = MAC2PC_SHORT( ((HoursRecordTypeV18 *)recordP)->rate.dollars );
#endif /* __MAC2PC_CONVERSION__ */

				fprintf( outfile, "%s%d,%02d%s%s"			/* rate */
												  "%s%d,%02d%s%s",		/* extra */
					enclose,
					((HoursRecordTypeV18 *)recordP)->rate.dollars, ((HoursRecordTypeV18 *)recordP)->rate.cents,
					enclose,
					gSeparatorString,

					enclose,
					((HoursRecordTypeV18 *)recordP)->extra.dollars, ((HoursRecordTypeV18 *)recordP)->extra.cents,
					enclose,
					gSeparatorString
				);

				s = &((HoursRecordTypeV18 *)recordP)->s;
			}
			else
				s = &((HoursRecordTypeV16 *)recordP)->s;
		}
		else
			s = &recordP->s;

		fprintf( outfile, "%s%s%s\n", enclose, clearText( s ), enclose );
	}
	else if( gVerbose )
		fprintf( stderr, "warning: header or record is NULL\n" );
}

/*
 * FUNCTION:					hrshAfterIterationCallback
 * DESCRIPTION:				this function is called after the itaration was successfully closed.
 * 										it is not neccarily needed.
 */
void hrshAfterIterationCallback( FILE * outfile, int numRecords, PDBHeaderType * headerP, void * appInfoP )
{
	char * ending;
	int i;
	char temp_char = 0;
	char * enclose;

	if( gPrintBanner )
	{
		ending = strstr( headerP->name, "-HRSH" );
		if( ending )
		{
			temp_char = *ending;
			*ending = '\0';
		}

		enclose = gDontEncloseValues ? "" : ENCLOSE_VALUE_STRING;
		fprintf( outfile, "\n%sThis CSV file was generated out of the Hours Database \'%s\'.%s\n", enclose, headerP->name, enclose );
		fprintf( outfile, "%sWritten by Peter Novotnik <peternov1@gmx.de>.%s\n", enclose, enclose );
		fprintf( outfile, "%sCommand line: %s%s%s", enclose, enclose, gSeparatorString, enclose );
		for( i = 0; i < gARGC; i++ )
		{
			char * s = (index( gARGV[i], ' ' ) == NULL) ? "" : "\'";
				fprintf( outfile, "%s%s%s%s", s, gARGV[i], s, (i == (gARGC-1)) ? "" : " " );
		}
		fprintf( outfile, "%s\n", enclose );

		if( ending )
			*ending = temp_char;
	}
}

/*
 * FUNCTION:		hrshPrintDBVersion
 * DESCRIPTION:	this routine prints the database version
 * PARAMETERS:	dbFile - the database file
 * RETURNS:			0 on success, non-zero on failure
 */
int hrshPrintDBVersion( FILE * dbFile )
{
  PDBHeaderType * headerP;

	if( !(headerP = getHeader( dbFile )) )
		return 1;

	if( !IS_HOURS_DB(headerP) )
	{
		fprintf( stderr, ERROR_PREFIX " Not an Hours Database.\n" );
		free( headerP );
		return 1;
	}

	printf( "%d\n", headerP->version );
	free( headerP );
	return 0;
}


/*
 * FUNCTION:			hrshPrintDBInfo
 * DESCRIPTION:		this routine prints some standard
 * 								information about the passed palm database
 * 								this routine does work even on not HRSH databases
 * PARAMETERS:		dbFile - file of the database
 * RETURNS:				zero on success, otherwise non-zero
 */
int hrshPrintDBInfo( FILE * dbFile )
{
	PDBHeaderType * headerP;
	int nextRecordListID;
	unsigned short numRecords;

	if( !(headerP = getHeader( dbFile )) )
		return 1;

	if( fread( &nextRecordListID, sizeof( int ), 1, dbFile ) == 0 )
	{
		free( headerP );
		return (1);
	}
	if( fread( &numRecords, sizeof( unsigned short ), 1, dbFile ) == 0 )
	{
		free( headerP );
		return (1);
	}
#ifdef __MAC2PC_CONVERSION__
	numRecords = (unsigned short)MAC2PC_SHORT( numRecords ); /* WARNING: Intel specific */
#endif //__MAC2PC_CONVERSION__

	printf( "\
name:      \'%s\'\n\
type:      \'%c%c%c%c\'\n\
creator:   \'%c%c%c%c\'\n\
numRecs:   %d\n\
dbVersion: %d\n",
		headerP->name, headerP->type[0], headerP->type[1], 
		headerP->type[2], headerP->type[3], 
		headerP->creator[0], headerP->creator[1], 
		headerP->creator[2], headerP->creator[3],
		(unsigned int)numRecords, (unsigned int)headerP->version );

	free( headerP );

	return 0;
}
