/********************************************************************************************
 * HFFzip - Little File Compressor for Linux and FreeBSD.
 * By Claudio Scordino , Linda Martorini, Francesco Lelli
 *
 * ******************************************************************************************
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ********************************************************************************************/


/*main.c*/

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>

#include "bitIO.h"
#include "albero.h"


/*****************************************************************************************/

int compressore(FILE* ingresso, struct bitIO* uscita)
/*ingresso e uscita sono files aperti nella funzione main*/
/*si crea la tabella che viene distrutta automaticamente alla fine*/

{
	struct nodo* tab= crea_tabella(2*N+1);
	int posizione,i,ritorno;
	unsigned char byte;
 	calcola_frequenza(tab,ingresso);		/*calcolo la frequenza dei simboli*/
	inizializzazione(tab);				/*inizializzo l'albero*/
	codifica(tab,2*N);				/*codifico i simboli*/
	posizione=fseek(ingresso,0,SEEK_SET);		/*mi sposto a inizio file*/
	if (posizione!=0)
		goto errore;

	/*scriviamo all'inizio del file compresso le frequenze di apparizione di
	tutti i simboli (per i simboli a frequenza di apparizione nulla poniamo  un
	intero pari a 0)*/

	for (i=0;i<=N;i++)				/*scrivo le frequenze di apparizione dei simboli*/
	{
		int fr=ritorno_freq(tab, i);
		int ritorno=bit_write(uscita,&fr,8*sizeof (int));
	}

	ritorno=fread(&byte,sizeof (char),1,ingresso);
	while (ritorno!=0)
	{
		bit_write(uscita, ritorno_cod(tab,(unsigned)byte),ritorno_lung(tab,(unsigned)byte));
		ritorno=fread(&byte,sizeof (char),1,ingresso);
	}
	/*scrivo il simbolo di finefile*/
	bit_write(uscita,ritorno_cod(tab,N),ritorno_lung(tab,N));
	distruggi_tabella(tab);
	return 1;

errore:
	printf("errore nella funzione compressore\n");
	fflush(NULL);
	distruggi_tabella(tab);
	return (int) NULL;
}


/*****************************************************************************************/

int decompressore(FILE* uscita, struct bitIO* ingresso)
/*i file sono gia' aperti dal main*/

{
	struct nodo* tab =crea_tabella(2*N +1);
	int finito=0;
	leggi_frequenze_da_file(tab,ingresso);
	inizializzazione(tab);
	while(!finito)
	{
		int trovato=0;
		unsigned int successivo;
		unsigned int nod=2*N;
		while(!trovato)
		{
			char byte= (char) 0;
			if (nod>N)
			{
				bit_read(ingresso,&byte,1);
				visita_dec(tab,nod,byte,&successivo);
				nod=successivo;
			}
			else
			{
				successivo=nod;
				trovato=1;
				if (successivo!=N)		/*non ho trovato il simbolo di finefile*/
				{
					char simbolo=(char) successivo;
					fwrite(&simbolo,sizeof (char),1,uscita);
				}
				else
					finito=1;
			}
		}
	}
	distruggi_tabella(tab);
	return 1;
}


/*****************************************************************************************/

int main(int argc, char* argv[])
{
	int ch;
	int comprimi=0;
	int decomprimi=0;
	int flag_ingresso=0;		/*Indica se e' stato specificato un file di ingresso*/
	int flag_uscita=0;		/*Indica se e' stato specificato un file di uscita*/
	int lunghezza;
	char* file_ingresso;
	char* file_uscita;
	while ((ch=getopt(argc,argv,"c d i: o:")) !=-1)
		switch(ch)
		{
			case 'c':
				comprimi=1;
				break;
			case 'd':
				decomprimi=1;
				break;
			case 'i':
				lunghezza=strlen(optarg);
				flag_ingresso=1;
				lunghezza++;
				file_ingresso=malloc(lunghezza*sizeof(char));
				strcpy (file_ingresso, optarg);
				break;
			case 'o':
				lunghezza=strlen(optarg);
				flag_uscita=1;
				lunghezza++;
				file_uscita=malloc(lunghezza*sizeof(char));
				strcpy (file_uscita, optarg);
				break;
			default:
				printf("\nHFFzip 1.1 - Little File Compressor for Linux and FreeBSD\n");
				printf("By Claudio Scordino , Linda Martorini, Francesco Lelli (2002)\n\n");
				printf("Usage: hffzip -c|-d [-i input_file] [-o output_file]\n\n");
				printf("Options:\n\n");
				printf("-c		compress\n");
				printf("-d		decompress\n");
				printf("input_file	if not defined standard input will be used\n");
				printf("output_file	if not defined standard output will be used\n\n");
				exit(-1);
		}
	if ((comprimi==1&&decomprimi==1) || (comprimi==0&&decomprimi==0))
	{
		printf("\nHFFzip 1.1 - Little File Compressor for Linux and FreeBSD\n");
		printf("By Claudio Scordino , Linda Martorini, Francesco Lelli (2002)\n\n");
		printf("Usage: hffzip -c|-d [-i input] [-o output]\n\n");
		printf("Options:\n\n");
		printf("-c		compress\n");
		printf("-d		decompress\n");
		printf("input_file	if not defined standard input will be used\n");
		printf("output_file	if not defined standard output will be used\n\n");
		exit(-1);
	}

	else if (comprimi==1)
	/*chiamata al compressore*/
	{
		struct bitIO* uscita;
		FILE* ingresso;
		if (flag_uscita==1)
			uscita=bit_open(file_uscita , O_WRONLY|O_CREAT ,(S_IRWXU | S_IRWXG | S_IRWXO) );
		else
			uscita=bit_open("#$%^STDOUTPUT",O_WRONLY|O_CREAT,(S_IRWXU | S_IRWXG | S_IRWXO));
		if (flag_ingresso==1)
			ingresso=fopen(file_ingresso,"r");
		else
			ingresso=fdopen(STDIN_FILENO,"r");
		compressore(ingresso, uscita);
		bit_close(uscita);
		fclose(ingresso);
	}


	else if (decomprimi==1)/*chiamata al decompressore*/
	{
		struct bitIO* ingresso;
		FILE* uscita;
		if (flag_ingresso==1)
			ingresso=bit_open(file_ingresso,O_RDONLY,0);
		else
			ingresso=bit_open("#$%^STDINPUT",O_RDONLY,0);
		if (flag_uscita==1)
			uscita=fopen(file_uscita,"w");
		else
			uscita=fdopen(STDOUT_FILENO,"w");
		decompressore(uscita, ingresso);
		bit_close(ingresso);
		fclose(uscita);
	}
	return 1;
}
