/********************************************************************************************
 * Little File Compressor for *nix systems.
 *
 * Copyright (C) 2002 Claudio Scordino , Linda Martorini, Francesco Lelli
 *
 * ******************************************************************************************
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ********************************************************************************************/

/*bitIO.c*/

#include <unistd.h>
#include <fcntl.h>

#define MAXBUFF 4096

struct bitIO
{
	int filedes;
	char buf[MAXBUFF];
	int primo,ultimo; /*le dimensioni sono in bit, ultimo e' il primo bit libero */
	                  /*primo e' il primo bit occupato;se c'e' un solo bit primo=0*/
	int modo;
};


/*********************************************************************************************/

struct bitIO* bit_open (const char* nome, int modo, mode_t diritti)
{
	struct bitIO *b;
	b= (struct bitIO* ) malloc (sizeof (struct bitIO));
	if (b==NULL)
		return NULL;
	if (!strcmp(nome,"#$%^STDINPUT"))	/*Nome di comodo per indicare lo standard input*/
	{
		b->modo=O_RDONLY;
		b->filedes=STDIN_FILENO;
	}
	else if (!strcmp(nome,"#$%^STDOUTPUT")) /*Nome di comodo per indicare lo standard output*/
	{
		b->modo=O_WRONLY;
		b->filedes=STDOUT_FILENO;
	}
	else
	{
		if ( (modo !=O_RDONLY) && (modo !=O_WRONLY) && (modo !=(O_WRONLY|O_CREAT)))
    		goto errore;
		b->modo= modo;
		(b->filedes)= open (nome,modo,diritti);
		if (b->filedes< 0)
			goto errore;
	}
	b->ultimo=b->primo=0;
	return b;

errore:
	free(b);
	printf("errore nella bit_open\n");
	fflush(NULL);
	return NULL;
}


/*********************************************************************************************/

int bit_close (struct bitIO* struttura)
{
	if ((struttura->modo!=O_RDONLY) && (struttura->ultimo!=struttura->primo))
	{
		int lunghezza=((struttura->ultimo-struttura->primo)/8);
		int ritorno;
		if ((struttura->ultimo-struttura->primo)%8 != 0)
			lunghezza++;
		ritorno=write(struttura->filedes,struttura->buf,lunghezza);
		if (ritorno<0)
			goto errore;
    	}
	close (struttura->filedes);
	free(struttura); 					/* struttura e' stata allocata con una malloc*/
	return 1;

errore:
	free(struttura);
	printf("errore nella bit_close\n");
	fflush(NULL);
	return (int) NULL;
}


/*********************************************************************************************/

int bit_write (struct bitIO* struttura ,void*  buffer, int lung)
{
	int i;
	char byte;				/*byte che vogliamo analizzare*/
	for (i=0;i<lung;i++)
	{
		byte=((char*)buffer)[i/8];	/*prendo la parte intera della divisione*/
		byte=(byte>>(i%8))&1;		/*trasla byte e mette nella posizione piu' bassa se e' 0 o 1*/
						/*il bit meno significativo lo supponiamo a destra*/
		if (byte==0)			/*dobbiamo scrivere uno 0*/
			struttura->buf[(struttura->ultimo)/8]&=~(1<<(struttura->ultimo%8));
						/*con ultimo/8 mi posiziono correttamente nel byte con il %8 mi posiziono al bit*/
		else				/* scrivo un 1*/
			struttura->buf[(struttura->ultimo)/8]|=(1<<(struttura->ultimo%8));
						/*il byte viene caricato da destra verso sinistra  <------*/
		struttura->ultimo++;
		if (struttura->ultimo==MAXBUFF*8)	/*si scrive*/
	 	{
	    		int ritorno=write(struttura->filedes,struttura->buf,MAXBUFF);
	    		struttura->ultimo=struttura->primo=0;
		}
	}
	return lung;
}


/*********************************************************************************************/

int bit_read (struct bitIO* struttura ,void*  buffer, int lung)
/*la bit_read deve leggere in struttura->buff e scrivere in buffer; se struttura->buffer e' vuoto lo
riempiamo con la read*/
{
	int i;
	char byte;
	char appoggio;
	for (i=0;i<lung;i++)
	{
		if (struttura->primo==struttura->ultimo)/*il buffer di struttura e' vuoto*/
		{
												/*riempimento del buffer*/
			int ritorno=read(struttura->filedes,struttura->buf,MAXBUFF);
			struttura->ultimo=ritorno*8;		/*potrebbe leggere meno byte di tutto il buffer*/
 			struttura->primo=0;
		}
												/*leggo un bit da struttura->buf e lo metto in buffer*/
		byte=struttura->buf[struttura->primo/8];
		byte=(byte>>(struttura->primo%8))&1;	/*vedi sopra nella bit_write*/
		appoggio=((char*) buffer)[i/8];
		if (byte==0)							/*dobbiamo scrivere uno 0*/
			appoggio&=~(1<<(i%8));
												/*con i/8 mi posiziono correttamente nel byte con il %8 mi posiziono al bit*/
		else /* scrivo un 1*/
			appoggio|=(1<<(i%8));
												/*il byte viene caricato da destra verso sinistra  <------*/
		((char*)buffer)[i/8]=appoggio;
		struttura->primo++;
	}
    return 1;

}

/*********************************************************************************************/

int descrittore(struct bitIO* nod)
/*Ritorna il descrittore del file contenuto nella struttura*/

{
	return nod->filedes;
}
